% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/groupedHolmMin.R
\name{groupedHolmMin}
\alias{groupedHolmMin}
\title{Combine grouped p-values with the minimum Holm approach}
\usage{
groupedHolmMin(
  p.values,
  grouping,
  weights = NULL,
  log.p = FALSE,
  min.n = 1,
  min.prop = 0.5
)
}
\arguments{
\item{p.values}{A numeric vector containing p-values for individual tests.}

\item{grouping}{A vector or factor of length equal to \code{p.values}, specifying the group to which each test is assigned.

Alternatively, an \link{rle} object where each run corresponds to a group and specifies the entries of \code{p.values} belonging to that group.
This assumes that \code{p.values} is ordered such that all entries in the same group are adjacent to each other.}

\item{weights}{A numeric vector of length equal to \code{p.values}, containing a positive weight for each test.
Alternatively \code{NULL}, in which case equal weights are assigned to all tests.}

\item{log.p}{Logical scalar indicating whether the p-values in \code{p.values} are log-transformed.}

\item{min.n}{Integer scalar specifying the minimum number of individual nulls to reject when testing the joint null.}

\item{min.prop}{Numeric scalar in [0, 1], specifying the minimum proportion of individual nulls to reject when testing the joint null.}
}
\value{
A list containing:
\itemize{
\item \code{p.value}, a named numeric vector of length equal to the number of unique levels in \code{grouping}.
This contains the minimum Holm p-value for each group, log-transformed if \code{log.p=TRUE}.
Each entry is named according to the group.
\item \code{representative}, a named integer scalar specifying the representative test for each group.
Each index refers to an entry of \code{p.values} and is named according to its group.
\item \code{influential}, a logical vector of length equal to \code{p.values}.
Entries are \code{TRUE} for any p-value that is deemed \dQuote{influential} to the final combined p-value for its group.
}
}
\description{
Combine p-values from grouped tests with the minimum Holm approach.
Groups are defined according to unique levels of a grouping factor.
}
\details{
Here, the joint null hypothesis for each group is that fewer than \eqn{N} individual null hypotheses are false.
The joint null is only rejected if \eqn{N} or more individual nulls are rejected; hence the \dQuote{minimum} in the function name.

\eqn{N} is defined as the larger of \code{min.n} and the product of \code{min.prop} with the number of tests in the group (rounded up).
This allows users to scale rejection of the joint null with the size of the group, while avoiding a too-low \eqn{N} when the group is small.
Note that \eqn{N} is always capped at the total size of the group.

To compute the combined p-value, we apply the Holm-Bonferroni correction to all p-values in the set and take the \eqn{N}-th smallest value.
This effectively recapitulates the step-down procedure where we reject individual nulls until we reach the \eqn{N}-th test.
This method works correctly in the presence of dependencies between p-values.

If non-equal weights are provided, they are used to effectively downscale the p-values. 
This aims to redistribute the error rate across the individual tests,
i.e., tests with lower weights are given fewer opportunities to drive acceptance of the joint null.

The representative test for each group is defined as that with the \eqn{N}-th smallest p-value, as this is directly used as the combined p-value.
The influential tests for each group are defined as those with p-values no greater than the representative test's p-value.
This is based on the fact that increasing them (e.g., by setting them to unity) would result in a larger combined p-value.
}
\examples{
p1 <- rbeta(100, 0.8, 1)
g <- sample(10, length(p1), replace=TRUE)

# Standard application:
out <- groupedHolmMin(p1, g)
str(out)

# With weights:
out <- groupedHolmMin(p1, g, weights=rexp(length(p1)))
str(out)

# With log p-values. 
out <- groupedHolmMin(log(p1), g, log.p=TRUE)
str(out)

}
\references{
Holm S (1979).
A simple sequentially rejective multiple test procedure.
\emph{Scand. J. Stat.} 6, 65-70.
}
\seealso{
\code{\link{parallelHolmMin}}, for a version that operates on parallel vectors of p-values.

\code{\link{groupedWilkinson}}, for a more relaxed version of this test when hypotheses are independent.
}
\author{
Aaron Lun
}
