/****************************************************************************
 * Header file for obi errnos                                               *
 ****************************************************************************/

/**
 * @file obierrno.h
 * @author Eric Coissac (eric.coissac@metabarcoding.org)
 * @date 23 May 2015
 * @brief Header file for obi errnos.
 */


#ifndef OBIERRNO_H_
#define OBIERRNO_H_


/**
 * @brief The declaration of the external variable `obi_errno`.
 *
 * `obi_errno` is an equivalent of `errno` for the system level error,
 * but for errors generated by the C layer of the OBITools3 framework.
 *
 * @TODO We have to look into defining this variable as thread specific.
 */
extern int obi_errno;


/**
 * @brief Sets the global `obi_errno` variable with
 *        the specified `err` code
 *
 * This function is defined as a macro to reduce the risk
 * of increasing the problem by generating another error by calling
 * another function.
 *
 * @param err The error code as an integer value.
 *
 * @since May 2015
 * @author Eric Coissac (eric.coissac@metabarcoding.org)
 */
#define obi_set_errno(err) (obi_errno = (err))


/**
 * @defgroup OBIDMSErrors OBIDMS related error codes
 *
 * Error codes set in errno following an error related
 * to the manipulation of an OBIDMS.
 *
 * @TODO
 *
 * @{
 */
#define OBIDMS_EXIST_ERROR     		(1) 	/**< Trying to create an OBIDMS with a name
                                    		 *   that corresponds to an existing one
                                    		 */
#define OBIDMS_NOT_EXIST_ERROR 		(2) 	/**< Trying to open a non-existing OBIDMS
                                    		 */
#define OBIDMS_LONG_NAME_ERROR 		(3) 	/**< The specified OBIDMS name is too long
                                    		 */
#define OBIDMS_MEMORY_ERROR    		(4) 	/**< A memory error occurred during allocation while handling
											 *   an OBIDMS
                                    		 */
#define OBIDMS_UNKNOWN_ERROR   		(5) 	/**< Undetermined error while handling an OBIDMS
                                    		 */
#define OBIDMS_ACCESS_ERROR    		(6) 	/**< Permission error trying to access the database
                                    		 */
#define OBIDMS_BAD_ENDIAN_ERROR   	(7) 	/**< The opened database does not correspond
                                    		 *   to the endianness of the platform.
                                    		 */

#define OBICOLDIR_EXIST_ERROR     	(8) 	/**< Trying to create an OBIDMS column directory with a name
                                    		 *   that corresponds to an existing one
                                    		 */
#define OBICOLDIR_NOT_EXIST_ERROR 	(9) 	/**< Trying to open a non-existing OBIDMS column directory
                                    		 */
#define OBICOLDIR_LONG_NAME_ERROR 	(10) 	/**< The specified OBIDMS column directory name is too long
                                    		 */
#define OBICOLDIR_MEMORY_ERROR    	(11) 	/**< A memory error occurred during allocation while handling
											 *	 an OBIDMS column directory
                                    	 	 */
#define OBICOLDIR_UNKNOWN_ERROR   	(12) 	/**< Undetermined error while handling an OBIDMS column directory
                                    	 	 */
#define OBICOLDIR_ACCESS_ERROR    	(13) 	/**< Permission error trying to access an OBIDSM column directory
                                    		 */

#define OBICOL_EXIST_ERROR     		(14) 	/**< Trying to create an OBIDMS column with a name
                                    		 *   that corresponds to an existing one
                                    		 */
#define OBICOL_NOT_EXIST_ERROR  	(15) 	/**< Trying to open a non-existing OBIDMS column directory
                                    		 */
#define OBICOL_LONG_NAME_ERROR 	    (16) 	/**< The specified OBIDMS column directory name is too long
                                    		 */
#define OBICOL_MEMORY_ERROR    		(17) 	/**< A memory error occurred during allocation while handling
											 *	 an OBIDMS column directory
                                    	 	 */
#define OBICOL_UNKNOWN_ERROR   		(18) 	/**< Undetermined error while handling an OBIDMS column directory
                                    	 	 */
#define OBICOL_ACCESS_ERROR    		(19) 	/**< Permission error trying to access an OBIDSM column directory
                                    		 */
#define OBI_AVL_ERROR				(20)    /** Error while handling an AVL tree	// TODO delete and just keep OBI_INDEXER_ERROR?
											 */
#define OBIVIEW_ERROR				(21)    /** Error while handling an OBIView
											 */
#define OBI_TAXONOMY_ERROR			(22)    /** Error while handling binary taxonomy files
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBI_MALLOC_ERROR			(23)    /** Error while allocating memory
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBI_ENCODE_ERROR			(24)    /** Error while encoding a value
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBI_DECODE_ERROR			(25)    /** Error while decoding a value
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBI_UTILS_ERROR				(26)    /** Error in an utils function
	 	 	 	 	 	 	 	 	 	 	 */
#define OBI_INDEXER_ERROR			(27)    /** Error handling a blob indexer
											 */
#define OBI_READ_ONLY_INDEXER_ERROR (28)    /** Error trying to modify a read-only blob indexer
											 */
#define OBI_ALIGN_ERROR				(29)    /** Error while aligning sequences
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBI_LINE_IDX_ERROR          (30)    /** Error setting or getting a value at a forbidden line index (greater than the line count of the view or the column)
											 */
#define OBI_ELT_IDX_ERROR           (31)    /** Error setting or getting a value at a non-existent element index or with a non-existent element name
											 */
#define OBI_CLEAN_ERROR				(32)    /** Error while cleaning sequences
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBI_ECOPCR_ERROR    		(33)    /** Error while performing an in silico PCR
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBI_JSON_ERROR       		(34)    /** Error related to JSON operations.
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBIVIEW_ALREADY_EXISTS_ERROR (35)   /** Tried to create a new view with a name already existing in the DMS.
											 */
#define OBI_ECOTAG_ERROR (36)   			/** Error while running ecotag.
											 */
#define OBIDMS_NOT_CLEAN (37)				/** OBIDMS has unfinished views or columns.
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBIDMS_WORKING (38) 				/** OBIDMS is tagged as being currently used by a process.
 	 	 	 	 	 	 	 	 	 	 	 */
#define OBI_SIGNAL_CAUGHT (39)				/** Caught an interrupting signal.
	 	 	 	 	 	 	 	 	 	 	 */
/**@}*/


#endif /* OBIERRNO_H_ */
