/*
 * Copyright (C) 2022-2024 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#if ENABLE(WK_WEB_EXTENSIONS)

#include "JSWebExtensionAPITest.h"
#include "WebExtensionAPIEvent.h"
#include "WebExtensionAPIObject.h"
#include "WebExtensionAPIWebNavigationEvent.h"
#include <wtf/Deque.h>

OBJC_CLASS NSString;

namespace WebKit {

class WebExtensionAPITest : public WebExtensionAPIObject, public JSWebExtensionWrappable {
    WEB_EXTENSION_DECLARE_JS_WRAPPER_CLASS(WebExtensionAPITest, test, test);

public:
#if PLATFORM(COCOA)
    void notifyFail(JSContextRef, NSString *message);
    void notifyPass(JSContextRef, NSString *message);

    void sendMessage(JSContextRef, NSString *message, JSValue *argument);
    WebExtensionAPIEvent& onMessage();

    JSValue *runWithUserGesture(WebFrame&, JSValue *function);
    bool isProcessingUserGesture();

    void log(JSContextRef, JSValue *);

    void fail(JSContextRef, NSString *message);
    void succeed(JSContextRef, NSString *message);

    void assertTrue(JSContextRef, bool testValue, NSString *message);
    void assertFalse(JSContextRef, bool testValue, NSString *message);

    void assertDeepEq(JSContextRef, JSValue *actualValue, JSValue *expectedValue, NSString *message);
    void assertEq(JSContextRef, JSValue *actualValue, JSValue *expectedValue, NSString *message);

    JSValue *assertRejects(JSContextRef, JSValue *promise, JSValue *expectedError, NSString *message);
    JSValue *assertResolves(JSContextRef, JSValue *promise, NSString *message);

    void assertThrows(JSContextRef, JSValue *function, JSValue *expectedError, NSString *message);
    JSValue *assertSafe(JSContextRef, JSValue *function, NSString *message);

    JSValue *assertSafeResolve(JSContextRef, JSValue *function, NSString *message);

    JSValue *addTest(JSContextRef, JSValue *testFunction);

private:
    RefPtr<WebExtensionAPIEvent> m_onMessage;

    struct Test {
        String testName;
        std::pair<String, unsigned> location;
        WebExtensionControllerIdentifier webExtensionControllerIdentifier;
        RetainPtr<JSValue> testFunction;
        RetainPtr<JSValue> resolveCallback;
        RetainPtr<JSValue> rejectCallback;
    };

    Deque<Test> m_testQueue;
    bool m_runningTest { false };
    bool m_hitAssertion { false };

    void assertEquals(JSContextRef, bool result, NSString *expectedString, NSString *actualString, NSString *message);
    void startNextTest();
    void recordAssertionIfNeeded(bool result)
    {
        if (m_runningTest && !m_hitAssertion && !result)
            m_hitAssertion = true;
    }
#endif
};

} // namespace WebKit

#endif // ENABLE(WK_WEB_EXTENSIONS)
