#include "dekkodserviceplugin.h"
#include <QCoreApplication>
#include <QDir>
#include <QFile>
#include <QProcess>
#include <QRegExp>
#include <QSettings>
#include <SnapStandardPaths.h>

DekkodService::DekkodService(QObject *parent): ServicePlugin(parent)
{
    m_service = QStringLiteral("dekkod");
    m_serviceFile = QString("%1/.config/systemd/user/%2.service").arg(QDir::homePath(), m_service);
}

QString DekkodService::pluginId() const
{
    return QStringLiteral("dekkod-service");
}

QString DekkodService::location() const
{
    return QStringLiteral("Dekko::Service");
}

QString DekkodService::i18n() const
{
    return QString();
}

void DekkodService::start()
{
    if (newVersion()) {
        qDebug() << "[DekkodService] Stopping service for version upgrade";
        stopService();
    }

    if (!serviceRunning()) {
        // The service is started on login, so actually it should already be running.
        // Overwrite service file to make sure thats not causing the problem.
        // see also issue #114
        qDebug() << "[DekkodService] Installing service file";
        installServiceFile();

        qDebug() << "[DekkodService] Starting dekkod service";
        startService();
    }
}

void DekkodService::stop()
{
//    if (serviceRunning()) {
//        stopService();
//    }

//    if (serviceFileInstalled()) {
//        removeServiceFile();
//    }
}

bool DekkodService::serviceFileInstalled() const
{
    return QFile(m_serviceFile).exists();
}

bool DekkodService::installServiceFile() const
{
    QFile f(m_serviceFile);
    QDir parent = QFileInfo(f).dir();

    if (!parent.mkpath(".")) {
        qDebug() << "[DekkodService] Cannot create systemd user service directory";
        return false;
    }

    if (!f.open(QFile::WriteOnly | QFile::Truncate)) {
        qDebug() << "[DekkodService] Cannot create service file";
        return false;
    }

    QString appDir = QCoreApplication::applicationDirPath();
    appDir.replace(QRegExp("dekko2.dekkoproject/[^/]+/"), "dekko2.dekkoproject/current/");

    f.write("[Unit]\n");
    f.write("Description=Dekkod Service\n");

    f.write("\n[Service]\n");
    f.write("ExecStart=" + appDir.toUtf8() + "/" + m_service.toUtf8() + "\n");
    f.write("WorkingDirectory=" + appDir.toUtf8() + "\n");
    f.write("Environment=\"LD_LIBRARY_PATH=" + appDir.toUtf8() + "/../:$LD_LIBRARY_PATH\"\n");
    f.write("Environment=\"DEKKO_PLUGINS=" + appDir.toUtf8() + "/../Dekko/plugins\"\n");
    f.write("Environment=\"QMF_PLUGINS=" + appDir.toUtf8() + "/../qmf/plugins5\"\n");
    f.write("Environment=\"QMF_DATA=" + QDir::homePath().toUtf8() + "/.cache/dekko2.dekkoproject\"\n");
    f.write("Restart=on-failure\n");

    f.write("\n[Install]\n");
    f.write("WantedBy=graphical-session.target\n");
    
    f.close();

    qDebug() << "[DekkodService] should enable service";
    int ret1 = QProcess::execute("systemctl", {"--user", "daemon-reload"});
    int ret2 = QProcess::execute("systemctl", {"--user", "enable", m_service});

    return ret1 == 0 && ret2 == 0;
}

bool DekkodService::removeServiceFile() const
{
    if (serviceFileInstalled()) {
        int ret = QProcess::execute("systemctl", {"--user", "disable", "--now", m_service});
        return QFile(m_serviceFile).remove() && ret == 0;
    }
    return true;
}

bool DekkodService::serviceRunning() const
{
    QProcess p;
    p.start("systemctl", {"--user", "status", m_service});
    p.waitForFinished();
    QByteArray output = p.readAll();
    qDebug() << output;
    return output.contains("active (running)");
}

bool DekkodService::startService()
{
    qDebug() << "[DekkodService] should start service";
    int ret = QProcess::execute("systemctl", {"--user", "start", m_service});
    return ret == 0;
}

bool DekkodService::restartService()
{
    qDebug() << "[DekkodService] should restart service";
    int ret = QProcess::execute("systemctl", {"--user", "restart", m_service});
    return ret == 0;
}

bool DekkodService::stopService()
{
    qDebug() << "[DekkodService] should stop service";
    int ret = QProcess::execute("systemctl", {"--user", "stop", m_service});
    return ret == 0;
}

bool DekkodService::newVersion()
{
    static const QString path = SnapStandardPaths::writableLocation(SnapStandardPaths::AppConfigLocation) + QStringLiteral("/dekkod/settings.ini");
    QSettings settings(path, QSettings::IniFormat);
    if (!settings.contains(QStringLiteral("version"))) {
        settings.setValue(QStringLiteral("version"), QStringLiteral(DEKKO_VERSION));
        // Dekkod may already be running as we previously didn't version it.
        // So we still need to stop it if running.
        return serviceRunning();
    }

    // We also want to support downgrades so just check the version doesn't match DEKKO_VERSION
    const bool result = settings.value(QStringLiteral("version")).toString() != QStringLiteral(DEKKO_VERSION);
    if (result) {
        settings.setValue(QStringLiteral("version"), QStringLiteral(DEKKO_VERSION));
    }
    settings.sync();
    return result;
}

QVariantMap DekkodService::documentation() const
{
    return QVariantMap();
}

QString DekkodServicePlugin::name() const
{
    return QStringLiteral("dekkod-service");
}

QString DekkodServicePlugin::description() const
{
    return QStringLiteral("Dekko's messaging server");
}

PluginInfo *DekkodServicePlugin::create(QObject *parent) const
{
    return new DekkodService(parent);
}
