# -------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for
# license information.
# --------------------------------------------------------------------------

import os
import pytest

from azure.storage.extensions import crc64

def test_compute_none():
    with pytest.raises(TypeError):
        crc64.compute(None, 0)

    with pytest.raises(TypeError):
        crc64.compute(b'', None)

@pytest.mark.parametrize("data, initial, expected", [
    (b'', 0, 0),
    (b'Hello World!', 0, 208604604655264165),
    (b'123456789!@#$%^&*()', 0, 2153758901452455624),
    (b'This is a test where the data is longer than 64 characters so that we can test that code path.', 0, 2736107658526394369),
    (b'\xC8\xE1\x1B\x40\xD7\x93\xD1\x52\x60\x18', 0, 3386042136331673945),
    (b'6QYd.$\xd9n\xf5\xc0\xec\x17Y\xcap\xc9D\xf7\x03fn}]\xfc\x18\xb4\xdd\xf0oK\xb6W~\xffs{\xbf\x98\xf9Xs\xb6\xcd\x10\x07Fmbz\x0e\xbbH\x02\xaa\xec\xecr\x12\x95\xf24\xb2\xc3;\x81\xd3\xa5\xdc\xfeM(\xdaVs\x99\x1a\x915q\xc1#\xc2\xe8\xe7\xf6eE\xc7J\xe1dC\x91@\x89\t1\x84\xdeG\xb3\xd6G\x11\xf6\x13\xd6\x85j\xc9q@\xce\xbd\xfc{0\xc9\xbe*\x90\x02U&&F\xe0\xde\xd3 \x02\x1e\xad\xae\xfd5\xea\xb0\xe1XW\x1f\xb0\x13N\xd3\x8c\xae\x07\x928\xf0\x98\x91\xac\xfa\xd1\x9cO\x990\x85\xc3\x91\xbbh\xac\x83\xf59{\xa5\xca\x07\r]\xb1\x80W\xde\xbd\xa6\x08\n!\xe7\x02\xfeK\xe9B\x95\x08l\xa6~\xad\xf1J\x02\r\x01\x1f)&.\x95\x1c\x11\x0b\x15\xb0a\x81V\xa9]\xa0\xed\x01\xc8\xd0%\x02\x14"\x7f\xddI\xd2\x08\xd5"6\xe2y\xf3\xe8\xf6\x0f\xedn\x8e\x96\x8b\xf3{~\xbaf"T\xfc\x17\xd7lP\xab\xbb\xb2\xf5\xf0\xfaF\x13\xaaV>\x896\xc9\x8dBM\xb3\x0eO\x9b\xf9\x7f\xe9LJ\xc6\x8bQ\x81cA\xac\xe1\x82\x10\x9bT\r\xb9\x8au\xbdN\xbd\xa3\x00\x16nz1ZF8R\xa4\x18\xf3\x85\x17\xfa;\xd7k,\to\xebc\xd0\x1f\xd5N\xfbX\x88\xb6\x9e\xf3\x85\x83\xb4\xd9\xd9\xfc<\xe5\x94I\xc8\x19\xfb>\xe2\x97A\xbc\xaf\xa2\xee\xb7\xd7\x93\x08\xd8b\x9b\x9b\x83\xb9\xd0\x93\xbf\xb5\x9d\xbb,\x10\x04|\xfc\xf0\xad\xc7\xb3\xd1X6\xdc\xf53\x0e\xb96I\x1ajj`\xdd\xac:\x06/H\xe3/}\x8e4\xbae\xf8\x0e\x88\xd5\x99\xdf/\x15\xb9C\xb4Yu\xba\x19\xdd;4,\x81l\xb0S\xd7D\x02\x8e\xd5zX\\\x88\xa7\xcb\xe3QX(\xe5\x08\xc9\xf9\xaa\xcc\x85\xbc\xb4l\x84"\x1a\xe4\xc0!\xaa\x1b\x15\xcc%\xe6\xa1\xee\x11V\x8cu\xa6\xd3Xe\x9d\xa4\xd3\xbb\xbaN7\x8b\xf3\x98\xd2\xcd\x99\xfc\xddv<\xad{\xd8\xc5\xc0R\n\x11\xe8L\xab\x19\xbf\xf6w\x88\x18\xb3bg\x15\\0:q\x95\x8b%\xb9;\xadr\x96\x98\xe6\xea?\xb4\xb7\x04c<\t:\x87G\x97%\xde\xd2\x83\xbb\xa7\xe7n\xc6\'WY\xc7yP\xad\x0bC\x7f\xf8\x02c#\xc0\xc8$\x81\xa1\xb6\xbb\xa9\xcb\xcf\xae?Z\x1aP\x1e?"\xa5\xc0\xb8J\xea=\xa7\xc39\xb9}M\x84\xd9mKd\x05\x9f\x1a\x10\x99q\xefji\xbf\xf4\xbc\xdf\xd7\xb2e\xcd\xbc>\x05uL\x05\xac\xbd\xfa\xfc\xbd`R0\t\x19eu\n\xc9Y\xa2e\xfd\x84 >\xb0\xf7\xfb\xbe!\xb0\x80\x89W\nO\x8f\x14)\x00\xc1E\xf5[\xd4\xae\x7f\xbfn\xba\x81}\'\x14M\x9b\xc6\x13\xd6\xcd\x1aph\xd6!-\xe5`\xd2h\x1e\xe6\xa20\xf1\x1eH\xe3\x80\x0e\xde\x06\x93\xdb\x10B\xaa:\xb0N\x81F\x07\xde8z\xb9\xd1KQB_\xf3\xb9\xae\x88\\q%\xc0\x07\xf8;-V\xc0\x9f\x1aX\xc7\xee8\x9a\x1dO\x18\xb3X\x8c\x85\xcf\xf15eg\xa3\xaft:\xbc\x1f\x1bA\xfa/T\xf1\x1c\xe4\xe7\xe7Ot\x06\xcc\xaf\xfe\x166C\xd5x\xf9d%\x9c~q"\x16\xf5\xb8\xb4~UN\x00\x88\xad\xa6]\xb2\x18\x7f7\xa0\xca\x83)QL\xe1\xe1jP\xd2\x7f(\x1fj\xed1W\xe9\xf1\x8c\xa6\xe4f\xbd\xab\x80\xd0\x88\xb7\x9d\xa9\x1e\x9c\xaf"~\x0b\\\x99<\x02r\x15\xfco\x00\xa2,\xa9\xc5Lv\x0ez\x1b\xc8Y\xbe,\xe2%QTUf!\x9b\xee\x11\xa4\xe6\x18\x87\xe8\xb9\xfb\x14\xfb/\xfa9\xd7Ag\x95\x035\xc7\x15EA\x06\x15n+T\xdb\xf6\x05,-;\x80s(\xca}\xd6\xbf{\x87\xe0 XB\xf3r.z\xf512)\xc8\x07\xc7v\xd7S\x10\x80\xc2\xd1\x1e`\xfb\xf2\xf8O=\x08\xc3\xc7\x8fe\xea31HKF\xa6\xc6\x1a\xc75\xbc\xa7\xac\xb5\xc4\xf4\xd3]\xb8\xd85Z\x97\xba\xee\xf9\xf6\xa32-{\xdb\xe6?q\x13:\x93\x18\x91\x00(\x06\x91\xda=\x98E5r\x17\xf5z\x8b\xabt\xe4\xaax\x17C\x14\xf5\xa7\x10 \xa5\x81\x9d\xc0\xe5\x86', 0, 10493919289952869355),

    (b'', 208604604655264165, 208604604655264165),
    (b'Hello World!', 208604604655264165, 8707751150827052565),
    (b'123456789!@#$%^&*()', 208604604655264165, 16914450898118618261),
    (b'This is a test where the data is longer than 64 characters so that we can test that code path.', 208604604655264165, 13786689105496759098),
    (b'\xC8\xE1\x1B\x40\xD7\x93\xD1\x52\x60\x18', 208604604655264165, 4570059697646401418),
    (b'6QYd.$\xd9n\xf5\xc0\xec\x17Y\xcap\xc9D\xf7\x03fn}]\xfc\x18\xb4\xdd\xf0oK\xb6W~\xffs{\xbf\x98\xf9Xs\xb6\xcd\x10\x07Fmbz\x0e\xbbH\x02\xaa\xec\xecr\x12\x95\xf24\xb2\xc3;\x81\xd3\xa5\xdc\xfeM(\xdaVs\x99\x1a\x915q\xc1#\xc2\xe8\xe7\xf6eE\xc7J\xe1dC\x91@\x89\t1\x84\xdeG\xb3\xd6G\x11\xf6\x13\xd6\x85j\xc9q@\xce\xbd\xfc{0\xc9\xbe*\x90\x02U&&F\xe0\xde\xd3 \x02\x1e\xad\xae\xfd5\xea\xb0\xe1XW\x1f\xb0\x13N\xd3\x8c\xae\x07\x928\xf0\x98\x91\xac\xfa\xd1\x9cO\x990\x85\xc3\x91\xbbh\xac\x83\xf59{\xa5\xca\x07\r]\xb1\x80W\xde\xbd\xa6\x08\n!\xe7\x02\xfeK\xe9B\x95\x08l\xa6~\xad\xf1J\x02\r\x01\x1f)&.\x95\x1c\x11\x0b\x15\xb0a\x81V\xa9]\xa0\xed\x01\xc8\xd0%\x02\x14"\x7f\xddI\xd2\x08\xd5"6\xe2y\xf3\xe8\xf6\x0f\xedn\x8e\x96\x8b\xf3{~\xbaf"T\xfc\x17\xd7lP\xab\xbb\xb2\xf5\xf0\xfaF\x13\xaaV>\x896\xc9\x8dBM\xb3\x0eO\x9b\xf9\x7f\xe9LJ\xc6\x8bQ\x81cA\xac\xe1\x82\x10\x9bT\r\xb9\x8au\xbdN\xbd\xa3\x00\x16nz1ZF8R\xa4\x18\xf3\x85\x17\xfa;\xd7k,\to\xebc\xd0\x1f\xd5N\xfbX\x88\xb6\x9e\xf3\x85\x83\xb4\xd9\xd9\xfc<\xe5\x94I\xc8\x19\xfb>\xe2\x97A\xbc\xaf\xa2\xee\xb7\xd7\x93\x08\xd8b\x9b\x9b\x83\xb9\xd0\x93\xbf\xb5\x9d\xbb,\x10\x04|\xfc\xf0\xad\xc7\xb3\xd1X6\xdc\xf53\x0e\xb96I\x1ajj`\xdd\xac:\x06/H\xe3/}\x8e4\xbae\xf8\x0e\x88\xd5\x99\xdf/\x15\xb9C\xb4Yu\xba\x19\xdd;4,\x81l\xb0S\xd7D\x02\x8e\xd5zX\\\x88\xa7\xcb\xe3QX(\xe5\x08\xc9\xf9\xaa\xcc\x85\xbc\xb4l\x84"\x1a\xe4\xc0!\xaa\x1b\x15\xcc%\xe6\xa1\xee\x11V\x8cu\xa6\xd3Xe\x9d\xa4\xd3\xbb\xbaN7\x8b\xf3\x98\xd2\xcd\x99\xfc\xddv<\xad{\xd8\xc5\xc0R\n\x11\xe8L\xab\x19\xbf\xf6w\x88\x18\xb3bg\x15\\0:q\x95\x8b%\xb9;\xadr\x96\x98\xe6\xea?\xb4\xb7\x04c<\t:\x87G\x97%\xde\xd2\x83\xbb\xa7\xe7n\xc6\'WY\xc7yP\xad\x0bC\x7f\xf8\x02c#\xc0\xc8$\x81\xa1\xb6\xbb\xa9\xcb\xcf\xae?Z\x1aP\x1e?"\xa5\xc0\xb8J\xea=\xa7\xc39\xb9}M\x84\xd9mKd\x05\x9f\x1a\x10\x99q\xefji\xbf\xf4\xbc\xdf\xd7\xb2e\xcd\xbc>\x05uL\x05\xac\xbd\xfa\xfc\xbd`R0\t\x19eu\n\xc9Y\xa2e\xfd\x84 >\xb0\xf7\xfb\xbe!\xb0\x80\x89W\nO\x8f\x14)\x00\xc1E\xf5[\xd4\xae\x7f\xbfn\xba\x81}\'\x14M\x9b\xc6\x13\xd6\xcd\x1aph\xd6!-\xe5`\xd2h\x1e\xe6\xa20\xf1\x1eH\xe3\x80\x0e\xde\x06\x93\xdb\x10B\xaa:\xb0N\x81F\x07\xde8z\xb9\xd1KQB_\xf3\xb9\xae\x88\\q%\xc0\x07\xf8;-V\xc0\x9f\x1aX\xc7\xee8\x9a\x1dO\x18\xb3X\x8c\x85\xcf\xf15eg\xa3\xaft:\xbc\x1f\x1bA\xfa/T\xf1\x1c\xe4\xe7\xe7Ot\x06\xcc\xaf\xfe\x166C\xd5x\xf9d%\x9c~q"\x16\xf5\xb8\xb4~UN\x00\x88\xad\xa6]\xb2\x18\x7f7\xa0\xca\x83)QL\xe1\xe1jP\xd2\x7f(\x1fj\xed1W\xe9\xf1\x8c\xa6\xe4f\xbd\xab\x80\xd0\x88\xb7\x9d\xa9\x1e\x9c\xaf"~\x0b\\\x99<\x02r\x15\xfco\x00\xa2,\xa9\xc5Lv\x0ez\x1b\xc8Y\xbe,\xe2%QTUf!\x9b\xee\x11\xa4\xe6\x18\x87\xe8\xb9\xfb\x14\xfb/\xfa9\xd7Ag\x95\x035\xc7\x15EA\x06\x15n+T\xdb\xf6\x05,-;\x80s(\xca}\xd6\xbf{\x87\xe0 XB\xf3r.z\xf512)\xc8\x07\xc7v\xd7S\x10\x80\xc2\xd1\x1e`\xfb\xf2\xf8O=\x08\xc3\xc7\x8fe\xea31HKF\xa6\xc6\x1a\xc75\xbc\xa7\xac\xb5\xc4\xf4\xd3]\xb8\xd85Z\x97\xba\xee\xf9\xf6\xa32-{\xdb\xe6?q\x13:\x93\x18\x91\x00(\x06\x91\xda=\x98E5r\x17\xf5z\x8b\xabt\xe4\xaax\x17C\x14\xf5\xa7\x10 \xa5\x81\x9d\xc0\xe5\x86', 208604604655264165, 9508443584543844346),

    (b'', 2153758901452455624, 2153758901452455624),
    (b'Hello World!', 2153758901452455624, 11658374609342321938),
    (b'123456789!@#$%^&*()', 2153758901452455624, 17932421111440067515),
    (b'This is a test where the data is longer than 64 characters so that we can test that code path.', 2153758901452455624, 4524234450386137392),
    (b'\xC8\xE1\x1B\x40\xD7\x93\xD1\x52\x60\x18', 2153758901452455624, 13366433516720813220),
    (b'6QYd.$\xd9n\xf5\xc0\xec\x17Y\xcap\xc9D\xf7\x03fn}]\xfc\x18\xb4\xdd\xf0oK\xb6W~\xffs{\xbf\x98\xf9Xs\xb6\xcd\x10\x07Fmbz\x0e\xbbH\x02\xaa\xec\xecr\x12\x95\xf24\xb2\xc3;\x81\xd3\xa5\xdc\xfeM(\xdaVs\x99\x1a\x915q\xc1#\xc2\xe8\xe7\xf6eE\xc7J\xe1dC\x91@\x89\t1\x84\xdeG\xb3\xd6G\x11\xf6\x13\xd6\x85j\xc9q@\xce\xbd\xfc{0\xc9\xbe*\x90\x02U&&F\xe0\xde\xd3 \x02\x1e\xad\xae\xfd5\xea\xb0\xe1XW\x1f\xb0\x13N\xd3\x8c\xae\x07\x928\xf0\x98\x91\xac\xfa\xd1\x9cO\x990\x85\xc3\x91\xbbh\xac\x83\xf59{\xa5\xca\x07\r]\xb1\x80W\xde\xbd\xa6\x08\n!\xe7\x02\xfeK\xe9B\x95\x08l\xa6~\xad\xf1J\x02\r\x01\x1f)&.\x95\x1c\x11\x0b\x15\xb0a\x81V\xa9]\xa0\xed\x01\xc8\xd0%\x02\x14"\x7f\xddI\xd2\x08\xd5"6\xe2y\xf3\xe8\xf6\x0f\xedn\x8e\x96\x8b\xf3{~\xbaf"T\xfc\x17\xd7lP\xab\xbb\xb2\xf5\xf0\xfaF\x13\xaaV>\x896\xc9\x8dBM\xb3\x0eO\x9b\xf9\x7f\xe9LJ\xc6\x8bQ\x81cA\xac\xe1\x82\x10\x9bT\r\xb9\x8au\xbdN\xbd\xa3\x00\x16nz1ZF8R\xa4\x18\xf3\x85\x17\xfa;\xd7k,\to\xebc\xd0\x1f\xd5N\xfbX\x88\xb6\x9e\xf3\x85\x83\xb4\xd9\xd9\xfc<\xe5\x94I\xc8\x19\xfb>\xe2\x97A\xbc\xaf\xa2\xee\xb7\xd7\x93\x08\xd8b\x9b\x9b\x83\xb9\xd0\x93\xbf\xb5\x9d\xbb,\x10\x04|\xfc\xf0\xad\xc7\xb3\xd1X6\xdc\xf53\x0e\xb96I\x1ajj`\xdd\xac:\x06/H\xe3/}\x8e4\xbae\xf8\x0e\x88\xd5\x99\xdf/\x15\xb9C\xb4Yu\xba\x19\xdd;4,\x81l\xb0S\xd7D\x02\x8e\xd5zX\\\x88\xa7\xcb\xe3QX(\xe5\x08\xc9\xf9\xaa\xcc\x85\xbc\xb4l\x84"\x1a\xe4\xc0!\xaa\x1b\x15\xcc%\xe6\xa1\xee\x11V\x8cu\xa6\xd3Xe\x9d\xa4\xd3\xbb\xbaN7\x8b\xf3\x98\xd2\xcd\x99\xfc\xddv<\xad{\xd8\xc5\xc0R\n\x11\xe8L\xab\x19\xbf\xf6w\x88\x18\xb3bg\x15\\0:q\x95\x8b%\xb9;\xadr\x96\x98\xe6\xea?\xb4\xb7\x04c<\t:\x87G\x97%\xde\xd2\x83\xbb\xa7\xe7n\xc6\'WY\xc7yP\xad\x0bC\x7f\xf8\x02c#\xc0\xc8$\x81\xa1\xb6\xbb\xa9\xcb\xcf\xae?Z\x1aP\x1e?"\xa5\xc0\xb8J\xea=\xa7\xc39\xb9}M\x84\xd9mKd\x05\x9f\x1a\x10\x99q\xefji\xbf\xf4\xbc\xdf\xd7\xb2e\xcd\xbc>\x05uL\x05\xac\xbd\xfa\xfc\xbd`R0\t\x19eu\n\xc9Y\xa2e\xfd\x84 >\xb0\xf7\xfb\xbe!\xb0\x80\x89W\nO\x8f\x14)\x00\xc1E\xf5[\xd4\xae\x7f\xbfn\xba\x81}\'\x14M\x9b\xc6\x13\xd6\xcd\x1aph\xd6!-\xe5`\xd2h\x1e\xe6\xa20\xf1\x1eH\xe3\x80\x0e\xde\x06\x93\xdb\x10B\xaa:\xb0N\x81F\x07\xde8z\xb9\xd1KQB_\xf3\xb9\xae\x88\\q%\xc0\x07\xf8;-V\xc0\x9f\x1aX\xc7\xee8\x9a\x1dO\x18\xb3X\x8c\x85\xcf\xf15eg\xa3\xaft:\xbc\x1f\x1bA\xfa/T\xf1\x1c\xe4\xe7\xe7Ot\x06\xcc\xaf\xfe\x166C\xd5x\xf9d%\x9c~q"\x16\xf5\xb8\xb4~UN\x00\x88\xad\xa6]\xb2\x18\x7f7\xa0\xca\x83)QL\xe1\xe1jP\xd2\x7f(\x1fj\xed1W\xe9\xf1\x8c\xa6\xe4f\xbd\xab\x80\xd0\x88\xb7\x9d\xa9\x1e\x9c\xaf"~\x0b\\\x99<\x02r\x15\xfco\x00\xa2,\xa9\xc5Lv\x0ez\x1b\xc8Y\xbe,\xe2%QTUf!\x9b\xee\x11\xa4\xe6\x18\x87\xe8\xb9\xfb\x14\xfb/\xfa9\xd7Ag\x95\x035\xc7\x15EA\x06\x15n+T\xdb\xf6\x05,-;\x80s(\xca}\xd6\xbf{\x87\xe0 XB\xf3r.z\xf512)\xc8\x07\xc7v\xd7S\x10\x80\xc2\xd1\x1e`\xfb\xf2\xf8O=\x08\xc3\xc7\x8fe\xea31HKF\xa6\xc6\x1a\xc75\xbc\xa7\xac\xb5\xc4\xf4\xd3]\xb8\xd85Z\x97\xba\xee\xf9\xf6\xa32-{\xdb\xe6?q\x13:\x93\x18\x91\x00(\x06\x91\xda=\x98E5r\x17\xf5z\x8b\xabt\xe4\xaax\x17C\x14\xf5\xa7\x10 \xa5\x81\x9d\xc0\xe5\x86', 2153758901452455624, 7867920104196670365),

    (b'', 12345, 12345),
    (b'Hello World!', 12345, 4200897209132841242),
    (b'123456789!@#$%^&*()', 12345, 1823927556695515382),
    (b'This is a test where the data is longer than 64 characters so that we can test that code path.', 12345, 16182239408178475978),
    (b'\xC8\xE1\x1B\x40\xD7\x93\xD1\x52\x60\x18', 12345, 5139183895903464380),
    (b'6QYd.$\xd9n\xf5\xc0\xec\x17Y\xcap\xc9D\xf7\x03fn}]\xfc\x18\xb4\xdd\xf0oK\xb6W~\xffs{\xbf\x98\xf9Xs\xb6\xcd\x10\x07Fmbz\x0e\xbbH\x02\xaa\xec\xecr\x12\x95\xf24\xb2\xc3;\x81\xd3\xa5\xdc\xfeM(\xdaVs\x99\x1a\x915q\xc1#\xc2\xe8\xe7\xf6eE\xc7J\xe1dC\x91@\x89\t1\x84\xdeG\xb3\xd6G\x11\xf6\x13\xd6\x85j\xc9q@\xce\xbd\xfc{0\xc9\xbe*\x90\x02U&&F\xe0\xde\xd3 \x02\x1e\xad\xae\xfd5\xea\xb0\xe1XW\x1f\xb0\x13N\xd3\x8c\xae\x07\x928\xf0\x98\x91\xac\xfa\xd1\x9cO\x990\x85\xc3\x91\xbbh\xac\x83\xf59{\xa5\xca\x07\r]\xb1\x80W\xde\xbd\xa6\x08\n!\xe7\x02\xfeK\xe9B\x95\x08l\xa6~\xad\xf1J\x02\r\x01\x1f)&.\x95\x1c\x11\x0b\x15\xb0a\x81V\xa9]\xa0\xed\x01\xc8\xd0%\x02\x14"\x7f\xddI\xd2\x08\xd5"6\xe2y\xf3\xe8\xf6\x0f\xedn\x8e\x96\x8b\xf3{~\xbaf"T\xfc\x17\xd7lP\xab\xbb\xb2\xf5\xf0\xfaF\x13\xaaV>\x896\xc9\x8dBM\xb3\x0eO\x9b\xf9\x7f\xe9LJ\xc6\x8bQ\x81cA\xac\xe1\x82\x10\x9bT\r\xb9\x8au\xbdN\xbd\xa3\x00\x16nz1ZF8R\xa4\x18\xf3\x85\x17\xfa;\xd7k,\to\xebc\xd0\x1f\xd5N\xfbX\x88\xb6\x9e\xf3\x85\x83\xb4\xd9\xd9\xfc<\xe5\x94I\xc8\x19\xfb>\xe2\x97A\xbc\xaf\xa2\xee\xb7\xd7\x93\x08\xd8b\x9b\x9b\x83\xb9\xd0\x93\xbf\xb5\x9d\xbb,\x10\x04|\xfc\xf0\xad\xc7\xb3\xd1X6\xdc\xf53\x0e\xb96I\x1ajj`\xdd\xac:\x06/H\xe3/}\x8e4\xbae\xf8\x0e\x88\xd5\x99\xdf/\x15\xb9C\xb4Yu\xba\x19\xdd;4,\x81l\xb0S\xd7D\x02\x8e\xd5zX\\\x88\xa7\xcb\xe3QX(\xe5\x08\xc9\xf9\xaa\xcc\x85\xbc\xb4l\x84"\x1a\xe4\xc0!\xaa\x1b\x15\xcc%\xe6\xa1\xee\x11V\x8cu\xa6\xd3Xe\x9d\xa4\xd3\xbb\xbaN7\x8b\xf3\x98\xd2\xcd\x99\xfc\xddv<\xad{\xd8\xc5\xc0R\n\x11\xe8L\xab\x19\xbf\xf6w\x88\x18\xb3bg\x15\\0:q\x95\x8b%\xb9;\xadr\x96\x98\xe6\xea?\xb4\xb7\x04c<\t:\x87G\x97%\xde\xd2\x83\xbb\xa7\xe7n\xc6\'WY\xc7yP\xad\x0bC\x7f\xf8\x02c#\xc0\xc8$\x81\xa1\xb6\xbb\xa9\xcb\xcf\xae?Z\x1aP\x1e?"\xa5\xc0\xb8J\xea=\xa7\xc39\xb9}M\x84\xd9mKd\x05\x9f\x1a\x10\x99q\xefji\xbf\xf4\xbc\xdf\xd7\xb2e\xcd\xbc>\x05uL\x05\xac\xbd\xfa\xfc\xbd`R0\t\x19eu\n\xc9Y\xa2e\xfd\x84 >\xb0\xf7\xfb\xbe!\xb0\x80\x89W\nO\x8f\x14)\x00\xc1E\xf5[\xd4\xae\x7f\xbfn\xba\x81}\'\x14M\x9b\xc6\x13\xd6\xcd\x1aph\xd6!-\xe5`\xd2h\x1e\xe6\xa20\xf1\x1eH\xe3\x80\x0e\xde\x06\x93\xdb\x10B\xaa:\xb0N\x81F\x07\xde8z\xb9\xd1KQB_\xf3\xb9\xae\x88\\q%\xc0\x07\xf8;-V\xc0\x9f\x1aX\xc7\xee8\x9a\x1dO\x18\xb3X\x8c\x85\xcf\xf15eg\xa3\xaft:\xbc\x1f\x1bA\xfa/T\xf1\x1c\xe4\xe7\xe7Ot\x06\xcc\xaf\xfe\x166C\xd5x\xf9d%\x9c~q"\x16\xf5\xb8\xb4~UN\x00\x88\xad\xa6]\xb2\x18\x7f7\xa0\xca\x83)QL\xe1\xe1jP\xd2\x7f(\x1fj\xed1W\xe9\xf1\x8c\xa6\xe4f\xbd\xab\x80\xd0\x88\xb7\x9d\xa9\x1e\x9c\xaf"~\x0b\\\x99<\x02r\x15\xfco\x00\xa2,\xa9\xc5Lv\x0ez\x1b\xc8Y\xbe,\xe2%QTUf!\x9b\xee\x11\xa4\xe6\x18\x87\xe8\xb9\xfb\x14\xfb/\xfa9\xd7Ag\x95\x035\xc7\x15EA\x06\x15n+T\xdb\xf6\x05,-;\x80s(\xca}\xd6\xbf{\x87\xe0 XB\xf3r.z\xf512)\xc8\x07\xc7v\xd7S\x10\x80\xc2\xd1\x1e`\xfb\xf2\xf8O=\x08\xc3\xc7\x8fe\xea31HKF\xa6\xc6\x1a\xc75\xbc\xa7\xac\xb5\xc4\xf4\xd3]\xb8\xd85Z\x97\xba\xee\xf9\xf6\xa32-{\xdb\xe6?q\x13:\x93\x18\x91\x00(\x06\x91\xda=\x98E5r\x17\xf5z\x8b\xabt\xe4\xaax\x17C\x14\xf5\xa7\x10 \xa5\x81\x9d\xc0\xe5\x86', 12345, 6065674385535310669),
])
def test_compute(data, initial, expected):
    actual = crc64.compute(data, initial)
    assert actual == expected

def test_compute_chunks():
    data = os.urandom(4 * 1024 * 1024)
    chunk_size = 1024 * 1024
    full_crc = crc64.compute(data, 0)

    index = 0
    crc = 0
    while index < len(data):
        end = index + chunk_size
        crc = crc64.compute(data[index:end], crc)
        index = end

    assert crc == full_crc

@pytest.mark.parametrize("crc1, size1, crc2, size2, expected", [
    (0, 0, 0, 0, 0),
    (17360427831495520774, 949533, 13068224794440996385, 99043, 2942932174470096852),
    (11788770130477425887, 505156, 11825964890373840515, 543420, 11679439596881108042),
    (3295333047304801182, 732633, 15304759627474960884, 315943, 31840984168952505),
    (16590039424904606984, 550299, 6063316096266934453, 498277, 4430932446378441680),
    (4505532069077416052, 852279, 4910763717047934640, 196297, 15119506491662968913),
    (390777554329396866, 834642, 2639871931800812330, 213934, 11705441749781302341),
    (3373070654000205532, 282713, 998330282635826126, 765863, 9830625244855600085),
    (2124306943908111903, 737293, 11017351202683543503, 311283, 8163771928713973931),
    (15994440356403990005, 85861, 13803536430055425947, 962715, 1941624554903785510),
    (3705122932895036835, 444701, 17573219681510991482, 603875, 4421337306620606216),
])
def test_concat(crc1, size1, crc2, size2, expected):
    actaul = crc64.concat(0, 0, crc1, size1, 0, crc2, size2)
    assert actaul == expected

@pytest.mark.parametrize("initial, initial1, crc1, size1, initial2, crc2, size2, expected", [
    (0, 0, 0, 0, 0, 0, 0, 0),
    (556425425686929588, 346224202686926702, 16342296696377857982, 332915, 1153230192133190692, 12153371329672466699, 715661, 1441822370130745021),
    (6707243468313456313, 572263087298867634, 16994544883182326144, 75745, 9131338361339398429, 10182915179976307502, 972831, 14966971284513070994),
    (5753644013440131291, 5049702011265556767, 17549647897932809624, 255140, 7204171574261853450, 1993084328138883374, 793436, 6041621697050742380),
    (6856094926385348025, 5380840211500611709, 9696539459657763690, 537777, 4787042077805010903, 13660128687379374948, 510799, 17784586126519415898),
    (7768574238870932405, 97145001356670685, 607054043350981298, 706788, 667444555190985522, 10677778047180339455, 341788, 5763961866513573791),
    (3302120679354661969, 7763531798276712053, 8827557196489825944, 490442, 8582969104890206846, 6702182603435500761, 558134, 4787302867829109706),
    (7553023568245626261, 9093436341919279996, 10815569438302788871, 785480, 8305342016037017917, 6633140726058569127, 263096, 14625483825524673467),
    (8894905328920043035, 9101951045389247372, 10098427678135105249, 782758, 8101576936188464286, 8318237935995533450, 265818, 5983082645903611588),
    (1084935736425738155, 5378644106529179816, 13762475631325587388, 1014816, 8473418370223760471, 10401355811619715622, 33760, 3692020299859515126),
    (889000539881195835, 2971048229276949174, 5346315327374690144, 307387, 1407121768110541356, 10535852615249992663, 741189, 3634018251978804152),
])
def test_concat__with_initials(initial, initial1, crc1, size1, initial2, crc2, size2, expected):
    actaul = crc64.concat(initial, initial1, crc1, size1, initial2, crc2, size2)
    assert actaul == expected

def test_concat_chunks():
    data = os.urandom(4 * 1024 * 1024)
    chunk_size = 1024 * 1024
    full_crc = crc64.compute(data, 0)

    index = 0
    crc = 0
    while index < len(data):
        end = index + chunk_size
        chunk_crc = crc64.compute(data[index:end], 0)
        crc = crc64.concat(0, 0, crc, index, 0, chunk_crc, chunk_size)
        index = end

    assert crc == full_crc
