/* SPDX-License-Identifier: BSD-3-Clause
 * Copyright(c) 2022 Corigine, Inc.
 * All rights reserved.
 */

#ifndef __NFP_CMSG_H__
#define __NFP_CMSG_H__

#include "nfp_flower.h"
#include "nfp_flower_flow.h"

struct nfp_flower_cmsg_hdr {
	rte_be16_t pad;
	uint8_t type;
	uint8_t version;
};

/* Types defined for control messages */
enum nfp_flower_cmsg_type {
	NFP_FLOWER_CMSG_TYPE_FLOW_ADD       = 0,
	NFP_FLOWER_CMSG_TYPE_FLOW_MOD       = 1,
	NFP_FLOWER_CMSG_TYPE_FLOW_DEL       = 2,
	NFP_FLOWER_CMSG_TYPE_LAG_CONFIG     = 4,
	NFP_FLOWER_CMSG_TYPE_PORT_REIFY     = 6,
	NFP_FLOWER_CMSG_TYPE_MAC_REPR       = 7,
	NFP_FLOWER_CMSG_TYPE_PORT_MOD       = 8,
	NFP_FLOWER_CMSG_TYPE_MERGE_HINT     = 9,
	NFP_FLOWER_CMSG_TYPE_NO_NEIGH       = 10,
	NFP_FLOWER_CMSG_TYPE_TUN_MAC        = 11,
	NFP_FLOWER_CMSG_TYPE_ACTIVE_TUNS    = 12,
	NFP_FLOWER_CMSG_TYPE_TUN_NEIGH      = 13,
	NFP_FLOWER_CMSG_TYPE_TUN_IPS        = 14,
	NFP_FLOWER_CMSG_TYPE_FLOW_STATS     = 15,
	NFP_FLOWER_CMSG_TYPE_PORT_ECHO      = 16,
	NFP_FLOWER_CMSG_TYPE_QOS_MOD        = 18,
	NFP_FLOWER_CMSG_TYPE_QOS_DEL        = 19,
	NFP_FLOWER_CMSG_TYPE_QOS_STATS      = 20,
	NFP_FLOWER_CMSG_TYPE_PRE_TUN_RULE   = 21,
	NFP_FLOWER_CMSG_TYPE_TUN_IPS_V6     = 22,
	NFP_FLOWER_CMSG_TYPE_NO_NEIGH_V6    = 23,
	NFP_FLOWER_CMSG_TYPE_TUN_NEIGH_V6   = 24,
	NFP_FLOWER_CMSG_TYPE_ACTIVE_TUNS_V6 = 25,
	NFP_FLOWER_CMSG_TYPE_MAX            = 32,
};

/*
 * NFP_FLOWER_CMSG_TYPE_MAC_REPR
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *     Word +---------------+-----------+---+---------------+---------------+
 *       0  |                  spare                        |Number of ports|
 *          +---------------+-----------+---+---------------+---------------+
 *       1  |    Index      |   spare   |NBI|  Port on NBI  | Chip-wide port|
 *          +---------------+-----------+---+---------------+---------------+
 *                                        ....
 *          +---------------+-----------+---+---------------+---------------+
 *     N-1  |    Index      |   spare   |NBI|  Port on NBI  | Chip-wide port|
 *          +---------------+-----------+---+---------------+---------------+
 *     N    |    Index      |   spare   |NBI|  Port on NBI  | Chip-wide port|
 *          +---------------+-----------+---+---------------+---------------+
 *
 *          Index: index into the eth table
 *          NBI (bits 17-16): NBI number (0-3)
 *          Port on NBI (bits 15-8): “base” in the driver
 *            this forms NBIX.PortY notation as the NSP eth table.
 *          "Chip-wide" port (bits 7-0):
 */
struct nfp_flower_cmsg_mac_repr {
	uint8_t reserved[3];
	uint8_t num_ports;
	struct {
		uint8_t idx;
		uint8_t info;
		uint8_t nbi_port;
		uint8_t phys_port;
	} ports[];
};

/*
 * NFP_FLOWER_CMSG_TYPE_PORT_REIFY
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *    Word  +-------+-------+---+---+-------+---+---+-----------+-----------+
 *       0  |Port Ty|Sys ID |NIC|Rsv| Spare |PCI|typ|    vNIC   |  queue    |
 *          +-------+-----+-+---+---+-------+---+---+-----------+---------+-+
 *       1  |                             Spare                           |E|
 *          +-------------------------------------------------------------+-+
 *          E: 1 = Representor exists, 0 = Representor does not exist
 */
struct nfp_flower_cmsg_port_reify {
	rte_be32_t portnum;
	rte_be16_t reserved;
	rte_be16_t info;
};

/*
 * NFP_FLOWER_CMSG_TYPE_PORT_MOD
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *    Word  +-------+-------+---+---+-------+---+---+-------+---+-----------+
 *       0  |Port Ty|Sys ID |NIC|Rsv|       Reserved        |    Port       |
 *          +-------+-------+---+---+-----+-+---+---+-------+---+-----------+
 *       1  |            Spare            |L|              MTU              |
 *          +-----------------------------+-+-------------------------------+
 *        L: Link or Admin state bit. When message is generated by host, this
 *           bit indicates the admin state (0=down, 1=up). When generated by
 *           NFP, it indicates the link state (0=down, 1=up)
 *
 *        Port Type (word 1, bits 31 to 28) = 1 (Physical Network)
 *        Port: “Chip-wide number” as assigned by BSP
 *
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *    Word  +-------+-------+---+---+-------+---+---+-------+---+-----------+
 *       0  |Port Ty|Sys ID |NIC|Rsv| Spare |PCI|typ|    vNIC   |  queue    |
 *          +-------+-----+-+---+---+---+-+-+---+---+-------+---+-----------+
 *       1  |            Spare            |L|              MTU              |
 *          +-----------------------------+-+-------------------------------+
 *        L: Link or Admin state bit. When message is generated by host, this
 *           bit indicates the admin state (0=down, 1=up). When generated by
 *           NFP, it indicates the link state (0=down, 1=up)
 *
 *        Port Type (word 1, bits 31 to 28) = 2 (PCIE)
 */
struct nfp_flower_cmsg_port_mod {
	rte_be32_t portnum;
	uint8_t reserved;
	uint8_t info;
	rte_be16_t mtu;
};

#define NFP_FLOWER_CMSG_PORT_MOD_INFO_LINK              RTE_BIT32(0)
#define NFP_FLOWER_CMSG_PORT_MOD_MTU_CHANGE_ONLY        RTE_BIT32(1)

struct nfp_flower_tun_neigh {
	uint8_t dst_mac[RTE_ETHER_ADDR_LEN];
	uint8_t src_mac[RTE_ETHER_ADDR_LEN];
	rte_be32_t port_id;
};

/*
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *          +---------------------------------------------------------------+
 *        0 |         VLAN_TPID             |            VLAN_ID            |
 *          +---------------------------------------------------------------+
 *        1 |                           HOST_CTX                            |
 *          +---------------------------------------------------------------+
 */
struct nfp_flower_tun_neigh_ext {
	rte_be16_t vlan_tpid;
	rte_be16_t vlan_tci;
	rte_be32_t host_ctx;
};

/*
 * NFP_FLOWER_CMSG_TYPE_TUN_NEIGH_V4
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *          +---------------------------------------------------------------+
 *        0 |                           DST_IPV4                            |
 *          +---------------------------------------------------------------+
 *        1 |                           SRC_IPV4                            |
 *          +---------------------------------------------------------------+
 *        2 |                      DST_MAC_B5_B4_B3_B2                      |
 *          +-------------------------------+-------------------------------+
 *        3 |             DST_MAC_B1_B0     |        SRC_MAC_B5_B4          |
 *          +-------------------------------+-------------------------------+
 *        4 |                       SRC_MAC_B3_B2_B1_B0                     |
 *          +---------------------------------------------------------------+
 *        5 |                    Egress Port (NFP internal)                 |
 *          +---------------------------------------------------------------+
 */
struct nfp_flower_cmsg_tun_neigh_v4 {
	rte_be32_t dst_ipv4;
	rte_be32_t src_ipv4;
	struct nfp_flower_tun_neigh common;
	struct nfp_flower_tun_neigh_ext ext;
};

/*
 * NFP_FLOWER_CMSG_TYPE_TUN_NEIGH_V6
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *          +---------------------------------------------------------------+
 *        0 |                          DST_IPV6 [0]                         |
 *          +---------------------------------------------------------------+
 *        1 |                          DST_IPV6 [1]                         |
 *          +---------------------------------------------------------------+
 *        2 |                          DST_IPV6 [2]                         |
 *          +---------------------------------------------------------------+
 *        3 |                          DST_IPV6 [3]                         |
 *          +---------------------------------------------------------------+
 *        4 |                          SRC_IPV6 [0]                         |
 *          +---------------------------------------------------------------+
 *        5 |                          SRC_IPV6 [1]                         |
 *          +---------------------------------------------------------------+
 *        6 |                          SRC_IPV6 [2]                         |
 *          +---------------------------------------------------------------+
 *        7 |                          SRC_IPV6 [3]                         |
 *          +---------------------------------------------------------------+
 *        8 |                       DST_MAC_B5_B4_B3_B2                     |
 *          +-------------------------------+-------------------------------+
 *        9 |             DST_MAC_B1_B0     |        SRC_MAC_B5_B4          |
 *          +-------------------------------+-------------------------------+
 *       10 |                       SRC_MAC_B3_B2_B1_B0                     |
 *          +---------------+---------------+---------------+---------------+
 *       11 |                    Egress Port (NFP internal)                 |
 *          +---------------------------------------------------------------+
 */
struct nfp_flower_cmsg_tun_neigh_v6 {
	uint8_t dst_ipv6[16];
	uint8_t src_ipv6[16];
	struct nfp_flower_tun_neigh common;
	struct nfp_flower_tun_neigh_ext ext;
};

#define NFP_TUN_PRE_TUN_RULE_DEL    (1 << 0)
#define NFP_TUN_PRE_TUN_IDX_BIT     (1 << 3)
#define NFP_TUN_PRE_TUN_IPV6_BIT    (1 << 7)

/*
 * NFP_FLOWER_CMSG_TYPE_PRE_TUN_RULE
 * Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 * -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *       +---------------------------------------------------------------+
 *     0 |                             FLAGS                             |
 *       +---------------------------------------------------------------+
 *     1 |         MAC_IDX               |            VLAN_ID            |
 *       +---------------------------------------------------------------+
 *     2 |                           HOST_CTX                            |
 *       +---------------------------------------------------------------+
 */
struct nfp_flower_cmsg_pre_tun_rule {
	rte_be32_t flags;
	rte_be16_t port_idx;
	rte_be16_t vlan_tci;
	rte_be32_t host_ctx_id;
};

#define NFP_TUN_MAC_OFFLOAD_DEL_FLAG  0x2

/*
 * NFP_FLOWER_CMSG_TYPE_TUN_MAC
 *     Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *    Word  +-----------------------+---+-+-+---------------+---------------+
 *       0  |              spare    |NBI|D|F| Amount of MAC’s in this msg   |
 *          +---------------+-------+---+-+-+---------------+---------------+
 *       1  |            Index 0            |     MAC[0]    |     MAC[1]    |
 *          +---------------+---------------+---------------+---------------+
 *       2  |     MAC[2]    |     MAC[3]    |     MAC[4]    |     MAC[5]    |
 *          +---------------+---------------+---------------+---------------+
 *       3  |            Index 1            |     MAC[0]    |     MAC[1]    |
 *          +---------------+---------------+---------------+---------------+
 *       4  |     MAC[2]    |     MAC[3]    |     MAC[4]    |     MAC[5]    |
 *          +---------------+---------------+---------------+---------------+
 *                                        ...
 *          +---------------+---------------+---------------+---------------+
 *     2N-1 |            Index N            |     MAC[0]    |     MAC[1]    |
 *          +---------------+---------------+---------------+---------------+
 *     2N   |     MAC[2]    |     MAC[3]    |     MAC[4]    |     MAC[5]    |
 *          +---------------+---------------+---------------+---------------+
 *
 *    F:   Flush bit. Set if entire table must be flushed. Rest of info in cmsg
 *        will be ignored. Not implemented.
 *    D:   Delete bit. Set if entry must be deleted instead of added
 *    NBI: Network Block Interface. Set to 0
 *    The amount of MAC’s per control message is limited only by the packet
 *    buffer size. A 2048B buffer can fit 253 MAC address and a 10240B buffer
 *    1277 MAC addresses.
 */
struct nfp_flower_cmsg_tun_mac {
	rte_be16_t flags;
	rte_be16_t count;           /**< Should always be 1 */
	rte_be16_t index;
	struct rte_ether_addr addr;
};

#define NFP_FL_IPV4_ADDRS_MAX        32

/*
 * NFP_FLOWER_CMSG_TYPE_TUN_IPS
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *          +---------------------------------------------------------------+
 *        0 |                    Number of IP Addresses                     |
 *          +---------------------------------------------------------------+
 *        1 |                        IP Address #1                          |
 *          +---------------------------------------------------------------+
 *        2 |                        IP Address #2                          |
 *          +---------------------------------------------------------------+
 *          |                             ...                               |
 *          +---------------------------------------------------------------+
 *       32 |                        IP Address #32                         |
 *          +---------------------------------------------------------------+
 */
struct nfp_flower_cmsg_tun_ipv4_addr {
	rte_be32_t count;
	rte_be32_t ipv4_addr[NFP_FL_IPV4_ADDRS_MAX];
};

#define NFP_FL_IPV6_ADDRS_MAX        4

/*
 * NFP_FLOWER_CMSG_TYPE_TUN_IP_V6
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *          +---------------------------------------------------------------+
 *        0 |                    Number of IP Addresses                     |
 *          +---------------------------------------------------------------+
 *        1 |                        IP Address1 #1                         |
 *          +---------------------------------------------------------------+
 *        2 |                        IP Address1 #2                         |
 *          +---------------------------------------------------------------+
 *          |                             ...                               |
 *          +---------------------------------------------------------------+
 *       16 |                        IP Address4 #4                         |
 *          +---------------------------------------------------------------+
 */
struct nfp_flower_cmsg_tun_ipv6_addr {
	rte_be32_t count;
	uint8_t ipv6_addr[NFP_FL_IPV6_ADDRS_MAX * 16];
};

/*
 * NFP_FLOWER_CMSG_TYPE_FLOW_STATS
 *    Bit    3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0
 *    -----\ 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 *    Word  +---------------+-----------------------------------------------+
 *       0  |    Reserved   |               Host Context                    |
 *          +---------------+-----------------------------------------------+
 *       1  |                          Packet Count                         |
 *          +---------------------------------------------------------------+
 *       2  |                          Byte Count                           |
 *          +---------------------------------------------------------------+
 *       2  |                          Byte Count                           |
 *          +---------------------------------------------------------------+
 *       3  |                          Host Cookie                          |
 *          +---------------------------------------------------------------+
 *       4  |                          Host Cookie                          |
 *          +---------------------------------------------------------------+
 */
struct nfp_flower_stats_frame {
	rte_be32_t stats_con_id;
	rte_be32_t pkt_count;
	rte_be64_t byte_count;
	rte_be64_t stats_cookie;
};

/*
 * See RFC 2698 for more details.
 * Word[0](Flag options):
 * [15] p(pps) 1 for pps, 0 for bps
 *
 * Meter control message
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-------------------------------+-+---+-----+-+---------+-+---+-+
 * |            Reserved           |p| Y |TYPE |E|  TSHFV  |P| PC|R|
 * +-------------------------------+-+---+-----+-+---------+-+---+-+
 * |                           Profile ID                          |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                        Token Bucket Peak                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     Token Bucket Committed                    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         Peak Burst Size                       |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                      Committed Burst Size                     |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                      Peak Information Rate                    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                    Committed Information Rate                 |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_cfg_head {
	rte_be32_t flags_opts;
	rte_be32_t profile_id;
};

/* Profile config, offload to NIC */
struct nfp_profile_conf {
	struct nfp_cfg_head head;    /**< Config head information */
	rte_be32_t bkt_tkn_p;        /**< Token bucket peak */
	rte_be32_t bkt_tkn_c;        /**< Token bucket committed */
	rte_be32_t pbs;              /**< Peak burst size */
	rte_be32_t cbs;              /**< Committed burst size */
	rte_be32_t pir;              /**< Peak information rate */
	rte_be32_t cir;              /**< Committed information rate */
};

/* Meter stats, read from firmware */
struct nfp_mtr_stats_reply {
	struct nfp_cfg_head head;    /**< Config head information */
	rte_be64_t pass_bytes;       /**< Count of passed bytes */
	rte_be64_t pass_pkts;        /**< Count of passed packets */
	rte_be64_t drop_bytes;       /**< Count of dropped bytes */
	rte_be64_t drop_pkts;        /**< Count of dropped packets */
};

enum nfp_flower_cmsg_port_type {
	NFP_FLOWER_CMSG_PORT_TYPE_UNSPEC,
	NFP_FLOWER_CMSG_PORT_TYPE_PHYS_PORT,
	NFP_FLOWER_CMSG_PORT_TYPE_PCIE_PORT,
	NFP_FLOWER_CMSG_PORT_TYPE_OTHER_PORT,
};

enum nfp_flower_cmsg_port_vnic_type {
	NFP_FLOWER_CMSG_PORT_VNIC_TYPE_VF,
	NFP_FLOWER_CMSG_PORT_VNIC_TYPE_PF,
	NFP_FLOWER_CMSG_PORT_VNIC_TYPE_CTRL,
};

#define NFP_FLOWER_CMSG_MAC_REPR_NBI            (0x3)

#define NFP_FLOWER_CMSG_HLEN            sizeof(struct nfp_flower_cmsg_hdr)
#define NFP_FLOWER_CMSG_VER1            1

#define NFP_FLOWER_CMSG_PORT_TYPE(x)            (((x) >> 28) & 0xf)  /* [31,28] */
#define NFP_FLOWER_CMSG_PORT_SYS_ID(x)          (((x) >> 24) & 0xf)  /* [24,27] */
#define NFP_FLOWER_CMSG_PORT_NFP_ID(x)          (((x) >> 22) & 0x3)  /* [22,23] */
#define NFP_FLOWER_CMSG_PORT_PCI(x)             (((x) >> 14) & 0x3)  /* [14,15] */
#define NFP_FLOWER_CMSG_PORT_VNIC_TYPE(x)       (((x) >> 12) & 0x3)  /* [12,13] */
#define NFP_FLOWER_CMSG_PORT_VNIC(x)            (((x) >> 6) & 0x3f)  /* [6,11] */
#define NFP_FLOWER_CMSG_PORT_VNIC_OFFSET(x, offset)    (NFP_FLOWER_CMSG_PORT_VNIC(x) - (offset))
#define NFP_FLOWER_CMSG_PORT_PCIE_Q(x)          ((x) & 0x3f)         /* [0,5] */
#define NFP_FLOWER_CMSG_PORT_PHYS_PORT_NUM(x)   ((x) & 0xff)         /* [0,7] */

/*
 * Metadata with L2 (1W/4B)
 * ----------------------------------------------------------------
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |    key_type   |    mask_id    | PCP |p|   vlan outermost VID  |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *                                 ^                               ^
 *                           NOTE: |             TCI               |
 *                                 +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_meta_tci {
	uint8_t nfp_flow_key_layer;
	uint8_t mask_id;
	rte_be16_t tci;
};

/*
 * Extended metadata for additional key_layers (1W/4B)
 * ----------------------------------------------------------------
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                      nfp_flow_key_layer2                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_ext_meta {
	rte_be32_t nfp_flow_key_layer2;
};

/*
 * L1 Port details (1W/4B)
 * ----------------------------------------------------------------
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         port_ingress                          |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_in_port {
	rte_be32_t in_port;
};

/*
 * L2 details (4W/16B)
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     mac_addr_dst, 31 - 0                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |      mac_addr_dst, 47 - 32    |     mac_addr_src, 15 - 0      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     mac_addr_src, 47 - 16                     |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |       mpls outermost label            |  TC |B|   reserved  |q|
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_mac_mpls {
	uint8_t mac_dst[6];
	uint8_t mac_src[6];
	rte_be32_t mpls_lse;
};

/*
 * L4 ports (for UDP, TCP, SCTP) (1W/4B)
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |            port_src           |           port_dst            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_tp_ports {
	rte_be16_t port_src;
	rte_be16_t port_dst;
};

struct nfp_flower_ip_ext {
	uint8_t tos;
	uint8_t proto;
	uint8_t ttl;
	uint8_t flags;
};

/*
 * L3 IPv4 details (3W/12B)
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |    DSCP   |ECN|   protocol    |      ttl      |     flags     |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                        ipv4_addr_src                          |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                        ipv4_addr_dst                          |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_ipv4 {
	struct nfp_flower_ip_ext ip_ext;
	rte_be32_t ipv4_src;
	rte_be32_t ipv4_dst;
};

/*
 * L3 IPv6 details (10W/40B)
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |    DSCP   |ECN|   protocol    |      ttl      |     flags     |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |   ipv6_exthdr   | res |            ipv6_flow_label            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src,   31 - 0                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src,  63 - 32                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src,  95 - 64                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src, 127 - 96                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst,   31 - 0                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst,  63 - 32                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst,  95 - 64                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst, 127 - 96                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_ipv6 {
	struct nfp_flower_ip_ext ip_ext;
	rte_be32_t ipv6_flow_label_exthdr;
	uint8_t ipv6_src[16];
	uint8_t ipv6_dst[16];
};

struct nfp_flower_tun_ipv4 {
	rte_be32_t src;
	rte_be32_t dst;
};

struct nfp_flower_tun_ipv6 {
	uint8_t ipv6_src[16];
	uint8_t ipv6_dst[16];
};

struct nfp_flower_tun_ip_ext {
	uint8_t tos;
	uint8_t ttl;
};

/*
 * Flow Frame IPv4 UDP TUNNEL --> Tunnel details (5W/20B)
 * -----------------------------------------------------------------
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         ipv4_addr_src                         |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         ipv4_addr_dst                         |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |           Reserved            |      tos      |      ttl      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                            Reserved                           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     VNI                       |   Reserved    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_ipv4_udp_tun {
	struct nfp_flower_tun_ipv4 ipv4;
	rte_be16_t reserved1;
	struct nfp_flower_tun_ip_ext ip_ext;
	rte_be32_t reserved2;
	rte_be32_t tun_id;
};

/*
 * Flow Frame IPv6 UDP TUNNEL --> Tunnel details (11W/44B)
 * -----------------------------------------------------------------
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src,   31 - 0                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src,  63 - 32                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src,  95 - 64                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src, 127 - 96                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst,   31 - 0                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst,  63 - 32                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst,  95 - 64                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst, 127 - 96                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |           Reserved            |      tos      |      ttl      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                            Reserved                           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     VNI                       |   Reserved    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_ipv6_udp_tun {
	struct nfp_flower_tun_ipv6 ipv6;
	rte_be16_t reserved1;
	struct nfp_flower_tun_ip_ext ip_ext;
	rte_be32_t reserved2;
	rte_be32_t tun_id;
};

/*
 * Flow Frame GRE TUNNEL --> Tunnel details (6W/24B)
 * -----------------------------------------------------------------
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         ipv4_addr_src                         |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         ipv4_addr_dst                         |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |           tun_flags           |       tos     |       ttl     |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |            Reserved           |           Ethertype           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                              Key                              |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                           Reserved                            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_ipv4_gre_tun {
	struct nfp_flower_tun_ipv4 ipv4;
	rte_be16_t tun_flags;
	struct nfp_flower_tun_ip_ext ip_ext;
	rte_be16_t reserved1;
	rte_be16_t ethertype;
	rte_be32_t tun_key;
	rte_be32_t reserved2;
};

/*
 * Flow Frame GRE TUNNEL V6 --> Tunnel details (12W/48B)
 * -----------------------------------------------------------------
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src,   31 - 0                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src,  63 - 32                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src,  95 - 64                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_src, 127 - 96                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst,   31 - 0                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst,  63 - 32                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst,  95 - 64                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                  ipv6_addr_dst, 127 - 96                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |           tun_flags           |       tos     |       ttl     |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |            Reserved           |           Ethertype           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                              Key                              |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                           Reserved                            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_flower_ipv6_gre_tun {
	struct nfp_flower_tun_ipv6 ipv6;
	rte_be16_t tun_flags;
	struct nfp_flower_tun_ip_ext ip_ext;
	rte_be16_t reserved1;
	rte_be16_t ethertype;
	rte_be32_t tun_key;
	rte_be32_t reserved2;
};

/*
 * L3 other (1W/4B)
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |            reserved           |           ethertype           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * Note: This is only used when no specific L3 header available.
 */
struct nfp_flower_l3_other {
	rte_be16_t reserved;
	rte_be16_t ethertype;
};

struct nfp_fl_act_head {
	uint8_t jump_id;
	uint8_t len_lw;
};

struct nfp_fl_act_output {
	struct nfp_fl_act_head head;
	rte_be16_t flags;
	rte_be32_t port;
};

/*
 * ETH
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|               -               |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     eth_dst_47_16_mask                        |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |      eth_dst_15_0_mask        |      eth_src_47_32_mask       |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     eth_src_31_0_mask                         |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     eth_dst_47_16                             |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |      eth_dst_15_0             |      eth_src_47_32            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     eth_src_31_0                              |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_set_eth {
	struct nfp_fl_act_head head;
	rte_be16_t reserved;
	uint8_t eth_addr_mask[RTE_ETHER_ADDR_LEN * 2];
	uint8_t eth_addr[RTE_ETHER_ADDR_LEN * 2];
};

struct nfp_fl_act_pop_vlan {
	struct nfp_fl_act_head head;
	rte_be16_t reserved;
};

struct nfp_fl_act_push_vlan {
	struct nfp_fl_act_head head;
	rte_be16_t reserved;
	rte_be16_t vlan_tpid;
	rte_be16_t vlan_tci;
};

/*
 * IPv4 addrs
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|               -               |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv4_saddr_mask                           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv4_saddr                                |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv4_daddr_mask                           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv4_daddr                                |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_set_ip4_addrs {
	struct nfp_fl_act_head head;
	rte_be16_t reserved;
	rte_be32_t ipv4_src_mask;
	rte_be32_t ipv4_src;
	rte_be32_t ipv4_dst_mask;
	rte_be32_t ipv4_dst;
};

/*
 * IPv4 ttl tos
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|    ttl_mask   |   tos_mask    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |       ttl     |      tos      |               0               |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_set_ip4_ttl_tos {
	struct nfp_fl_act_head head;
	uint8_t ipv4_ttl_mask;
	uint8_t ipv4_tos_mask;
	uint8_t ipv4_ttl;
	uint8_t ipv4_tos;
	rte_be16_t reserved;
};

/*
 * IPv6 addr
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|               -               |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_addr_127_96_mask                     |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_addr_127_96                          |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_addr_95_64_mask                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_addr_95_64                           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_addr_63_32_mask                      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_addr_63_32                           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_addr_31_0_mask                       |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_addr_31_0                            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_set_ipv6_addr {
	struct nfp_fl_act_head head;
	rte_be16_t reserved;
	struct {
		rte_be32_t mask;
		rte_be32_t exact;
	} ipv6[4];
};

/*
 * Ipv6 tc hl fl
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|  tclass_mask  |  hlimit_mask  |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |               0               |  tclass       |  hlimit       |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |           0           |             flabel_mask               |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |           0           |             flabel                    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_set_ipv6_tc_hl_fl {
	struct nfp_fl_act_head head;
	uint8_t ipv6_tc_mask;
	uint8_t ipv6_hop_limit_mask;
	rte_be16_t reserved;
	uint8_t ipv6_tc;
	uint8_t ipv6_hop_limit;
	rte_be32_t ipv6_label_mask;
	rte_be32_t ipv6_label;
};

/*
 * TCP/UDP/SCTP
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|               -               |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |          src_mask             |         dst_mask              |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |          src                  |         dst                   |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_set_tport {
	struct nfp_fl_act_head head;
	rte_be16_t reserved;
	rte_be16_t src_port_mask;
	rte_be16_t dst_port_mask;
	rte_be16_t src_port;
	rte_be16_t dst_port;
};

/*
 * Pre-tunnel
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|              -      |M|   - |V|
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |         ipv6_daddr_127_96     /     ipv4_daddr                |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_daddr_95_64                          |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_daddr_63_32                          |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                     ipv6_daddr_31_0                           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_pre_tun {
	struct nfp_fl_act_head head;
	rte_be16_t flags;
	union {
		rte_be32_t ipv4_dst;
		uint8_t ipv6_dst[16];
	};
};

#define NFP_FL_PRE_TUN_IPV6    (1 << 0)

/*
 * Set tunnel
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|           reserved            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                            tun_id0                            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                            tun_id1                            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                      reserved                 |  type |r| idx |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |           ipv4_flags          |      ttl      |      tos      |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         reserved_cvs1                         |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |          reserved_cvs2        |        reserved_cvs3          |
 * |            var_flags          |            var_np             |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_set_tun {
	struct nfp_fl_act_head head;
	rte_be16_t reserved;
	rte_be64_t tun_id;
	rte_be32_t tun_type_index;
	rte_be16_t tun_flags;
	uint8_t    ttl;
	uint8_t    tos;
	rte_be16_t outer_vlan_tpid;
	rte_be16_t outer_vlan_tci;
	uint8_t    tun_len;      /**< Only valid for NFP_FL_TUNNEL_GENEVE */
	uint8_t    reserved2;
	rte_be16_t tun_proto;    /**< Only valid for NFP_FL_TUNNEL_GENEVE */
} __rte_packed;

/*
 * Meter
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|            reserved           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         Profile ID                            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_meter {
	struct nfp_fl_act_head head;
	rte_be16_t reserved;
	rte_be32_t profile_id;
};

/*
 * Mark
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|            reserved           |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         Mark                                  |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
struct nfp_fl_act_mark {
	struct nfp_fl_act_head head;
	rte_be16_t reserved;
	rte_be32_t mark;
};

/*
 * Partial
 *    3                   2                   1
 *  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | res |  opcode |  res  | len_lw|     flag       |     resv1    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                         Mark                                  |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |           queue_id            |              resv2            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * flag = 0x0: only mark action
 * flag = 0x1: only queue action
 * flag = 0x2: mark and queue action
 */
struct nfp_fl_act_partial {
	struct nfp_fl_act_head head;
	uint8_t flag;
	uint8_t resv1;
	rte_be32_t mark;
	rte_be16_t queue_id;
	rte_be16_t resv2;
};

int nfp_flower_cmsg_mac_repr(struct nfp_app_fw_flower *app_fw_flower,
		struct nfp_pf_dev *pf_dev);
int nfp_flower_cmsg_repr_reify(struct nfp_app_fw_flower *app_fw_flower,
		struct nfp_flower_representor *repr);
int nfp_flower_cmsg_port_mod(struct nfp_app_fw_flower *app_fw_flower,
		uint32_t port_id, bool carrier_ok);
int nfp_flower_cmsg_flow_delete(struct nfp_app_fw_flower *app_fw_flower,
		struct rte_flow *flow);
int nfp_flower_cmsg_flow_add(struct nfp_app_fw_flower *app_fw_flower,
		struct rte_flow *flow);
int nfp_flower_cmsg_tun_neigh_v4_rule(struct nfp_app_fw_flower *app_fw_flower,
		struct nfp_flower_cmsg_tun_neigh_v4 *payload);
int nfp_flower_cmsg_tun_neigh_v6_rule(struct nfp_app_fw_flower *app_fw_flower,
		struct nfp_flower_cmsg_tun_neigh_v6 *payload);
int nfp_flower_cmsg_tun_off_v4(struct nfp_app_fw_flower *app_fw_flower);
int nfp_flower_cmsg_tun_off_v6(struct nfp_app_fw_flower *app_fw_flower);
int nfp_flower_cmsg_pre_tunnel_rule(struct nfp_app_fw_flower *app_fw_flower,
		struct nfp_fl_rule_metadata *nfp_flow_meta,
		uint16_t mac_idx,
		bool is_del);
int nfp_flower_cmsg_tun_mac_rule(struct nfp_app_fw_flower *app_fw_flower,
		struct rte_ether_addr *mac,
		uint16_t mac_idx,
		bool is_del);
int nfp_flower_cmsg_qos_add(struct nfp_app_fw_flower *app_fw_flower,
		struct nfp_profile_conf *conf);
int nfp_flower_cmsg_qos_delete(struct nfp_app_fw_flower *app_fw_flower,
		struct nfp_profile_conf *conf);
int nfp_flower_cmsg_qos_stats(struct nfp_app_fw_flower *app_fw_flower,
		struct nfp_cfg_head *head);

static inline bool
nfp_flower_port_is_phy_port(uint32_t port_id)
{
	return (NFP_FLOWER_CMSG_PORT_TYPE(port_id) ==
			NFP_FLOWER_CMSG_PORT_TYPE_PHYS_PORT);
}

#endif /* __NFP_CMSG_H__ */
