/* -*- Mode: Vala; indent-tabs-mode: nil; tab-width: 2 -*-
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 * SPDX-FileCopyrightText: Michael Terry
 */

using GLib;

[GtkTemplate (ui = "/org/gnome/DejaDup/MainWindow.ui")]
public class MainWindow : Adw.ApplicationWindow
{
  public void add_toast(string message)
  {
    toast_overlay.add_toast(new Adw.Toast(message));
  }

  [GtkChild]
  unowned Adw.ToastOverlay toast_overlay;
  [GtkChild]
  unowned Adw.NavigationView nav_view;
  [GtkChild]
  unowned SnapshotsPage snapshots_page;
  [GtkChild]
  unowned BrowserPage browser_page;

  DejaDup.FilteredSettings settings;
  OperationWrapper wrapper;

  construct {
    if (Config.PROFILE == "Devel")
      add_css_class("devel"); // changes look of headerbars usually

    settings = DejaDup.get_settings();
    settings.bind(DejaDup.WINDOW_WIDTH_KEY, this, "default-width", SettingsBindFlags.DEFAULT);
    settings.bind(DejaDup.WINDOW_HEIGHT_KEY, this, "default-height", SettingsBindFlags.DEFAULT);
    settings.bind(DejaDup.WINDOW_MAXIMIZED_KEY, this, "maximized", SettingsBindFlags.DEFAULT);
    settings.bind(DejaDup.WINDOW_FULLSCREENED_KEY, this, "fullscreened", SettingsBindFlags.DEFAULT);

    settings.changed[DejaDup.LAST_RUN_KEY].connect(update_visible_child);
    update_visible_child();

    var watcher = DejaDup.OperationWatcher.get_instance();
    watcher.bind_property("active", this, "hide-on-close", BindingFlags.SYNC_CREATE);

    // Use a wrapper to tell whether to close when hidden, because it has the
    // extra logic to keep being active during an error - which we want if the
    // user was notified via desktop notifications and clicks it to see error.
    wrapper = new OperationWrapper(watcher, this);
    wrapper.notify["active"].connect(maybe_destroy);
  }

  ~MainWindow()
  {
    debug("Finalizing MainWindow\n");
  }

  void update_visible_child()
  {
    if (settings.get_string(DejaDup.LAST_RUN_KEY) == "")
      nav_view.replace_with_tags({"welcome"});
    else
      nav_view.replace_with_tags({"overview"});
  }

  void maybe_destroy()
  {
    var open_mounts = !DejaDup.MountManager.get_instance().has_mounts();
    if (!wrapper.active && !visible && !open_mounts)
      destroy();
  }

  public override bool close_request()
  {
    if (!DejaDup.MountManager.get_instance().unmount_all()) {
      // Couldn't unmount - the mounts must still be in use.
      // Let's merely hide, so that the mounts can still be active.
      hide();
      return true;
    }
    return false;
  }

  [GtkCallback]
  void on_welcome_backup()
  {
    DejaDupApp.get_instance().lookup_action("backup").activate(null);
    nav_view.replace_with_tags({"overview"});
  }

  [GtkCallback]
  void on_welcome_browse(DejaDup.Backend backend)
  {
    snapshots_page.scan(backend);
    nav_view.push_by_tag("snapshots");
  }

  [GtkCallback]
  void on_browse_snapshots(bool all)
  {
    if (all) {
      snapshots_page.scan();
      nav_view.push_by_tag("snapshots");
    } else {
      browser_page.scan("");
      nav_view.push_by_tag("browser");
    }
  }

  [GtkCallback]
  void on_browse_files(DejaDup.Operation.State state, string tag)
  {
    browser_page.scan(tag, state);
    nav_view.push_by_tag("browser");
  }

  [GtkCallback]
  void on_show_schedule()
  {
    nav_view.push_by_tag("schedule");
  }

  [GtkCallback]
  void on_show_folders()
  {
    nav_view.push_by_tag("folders");
  }

  [GtkCallback]
  void on_show_storage()
  {
    nav_view.push_by_tag("storage");
  }
}
