# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Tests for tabular representation of querysets."""

import abc
from urllib.parse import urlencode

from django.core.exceptions import ImproperlyConfigured
from django.db.models import QuerySet
from django.test import RequestFactory

from debusine.db.models import User
from debusine.test.django import TestCase
from debusine.web.views.table import Ordering, Table
from debusine.web.views.table.columns import ColumnState
from debusine.web.views.tests.utils import ViewTestMixin


class TableTestCase(ViewTestMixin, TestCase, abc.ABC):
    """Common test infrastructure."""

    @abc.abstractmethod
    def get_table_class(self) -> type[Table[User]]:
        """Get the table class to use for tests."""

    def _table(
        self,
        queryset: QuerySet[User, User] | None = None,
        prefix: str = "",
        default_column: str | None = None,
        table_class: type[Table[User]] | None = None,
        **kwargs: str,
    ) -> Table[User]:
        """Instantiate a table."""
        url = "/"
        if kwargs:
            url += "?" + urlencode(kwargs)
        request = RequestFactory().get(url)
        if queryset is None:
            queryset = User.objects.all()
        if table_class is None:
            table_class = self.get_table_class()
        return table_class(
            request=request,
            object_list=queryset,
            prefix=prefix,
            default_column=default_column,
        )


class TableTests(TableTestCase):
    """Tests for :py:class:`Table`."""

    @classmethod
    def setUpTestData(cls) -> None:
        """Initialize class data."""
        super().setUpTestData()
        for idx in range(10):
            cls.playground.create_user(f"user{idx:02d}")

    def get_table_class(self) -> type[Table[User]]:
        class _Table(Table[User]):
            default_column = "user"

            def init(self) -> None:
                super().init()
                self.add_column("user", "User", Ordering("username"))
                self.add_column("email", "Email", Ordering("email"))

        return _Table

    def test_default_column(self) -> None:
        """Test with default ordering."""
        t = self._table()
        self.assertEqual(t.default_column, "user")
        self.assertEqual(t.current_column.name, "user")
        self.assertEqual(t.current_column.state, ColumnState.ASCENDING)
        self.assertQuerySetEqual(
            t.rows, list(User.objects.order_by("username"))
        )

    def test_default_column_in_constructor(self) -> None:
        """Test with default ordering set in constructor."""
        t = self._table(default_column="email")
        self.assertEqual(t.default_column, "email")
        self.assertEqual(t.current_column.name, "email")
        self.assertEqual(t.current_column.state, ColumnState.ASCENDING)
        self.assertQuerySetEqual(t.rows, list(User.objects.order_by("email")))

    def test_default_column_in_constructor_desc(self) -> None:
        """Test with default ordering set in constructor, descending."""
        t = self._table(default_column="-email")
        self.assertEqual(t.default_column, "-email")
        self.assertEqual(t.current_column.name, "email")
        self.assertEqual(t.current_column.state, ColumnState.DESCENDING)
        self.assertQuerySetEqual(t.rows, list(User.objects.order_by("-email")))

    def test_column_invalid_in_args(self) -> None:
        """An invalid column in query string defaults to default ordering."""
        t = self._table(order="does-not-exist")
        self.assertIs(t.current_column.name, "user")
        self.assertEqual(t.current_column.state, ColumnState.ASCENDING)
        self.assertQuerySetEqual(
            t.rows, list(User.objects.order_by("username"))
        )

    def test_column_current(self) -> None:
        """Test selecting the current column via query string."""
        t = self._table(order="user")
        self.assertIs(t.current_column.name, "user")
        self.assertEqual(t.current_column.state, ColumnState.ASCENDING)
        self.assertQuerySetEqual(
            t.rows, list(User.objects.order_by("username"))
        )

    def test_column_not_current(self) -> None:
        """The columns that are not selected have status INACTIVE."""
        t = self._table(order="user")
        col = t.columns["email"]
        self.assertEqual(col.state, ColumnState.INACTIVE)

    def test_column_current_desc(self) -> None:
        """Test with no columns defined."""
        t = self._table(order="user", asc="0")
        self.assertIs(t.current_column.name, "user")
        self.assertEqual(t.current_column.state, ColumnState.DESCENDING)
        self.assertQuerySetEqual(
            t.rows, list(User.objects.order_by("-username"))
        )

    def test_order_by_no_ordering_column(self) -> None:
        """Test selecting a non-orderable column."""

        class _Table(self.get_table_class()):  # type: ignore[misc]
            """Table with default ordering not set."""

            def init(self) -> None:
                """Add a non-orderable column."""
                super().init()
                self.add_column("test", "Test")
                self.default_column = "email"

        t = self._table(
            table_class=_Table,
            queryset=User.objects.order_by("username"),
            order="test",
        )
        self.assertIs(t.current_column.name, "email")
        self.assertEqual(t.current_column.state, ColumnState.ASCENDING)
        self.assertQuerySetEqual(t.rows, list(User.objects.order_by("email")))

    def test_missing_default_column(self) -> None:
        """Test with no default ordering."""

        class _Table(self.get_table_class()):  # type: ignore[misc]
            """Table with default ordering not set."""

            default_column = None

        with self.assertRaisesRegex(
            ImproperlyConfigured,
            r"default_column not set in class or constructor arguments,"
            r" and not set in init\(\)",
        ):
            self._table(table_class=_Table, queryset=User.objects.all())

    def test_invalid_default_column(self) -> None:
        """Test with no default ordering."""
        with self.assertRaisesRegex(
            ImproperlyConfigured,
            r"default_column 'does-not-exist' is not a valid column name",
        ):
            self._table(
                queryset=User.objects.all(), default_column="does-not-exist"
            )
