# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Pagination support widgets."""

from enum import StrEnum
from functools import cached_property
from typing import Any, TYPE_CHECKING, TypeVar, Union, assert_never

from django.db.models import Model
from django.template.context import BaseContext
from django.utils.html import format_html

from debusine.web.views.base import Widget

if TYPE_CHECKING:
    from django.core.paginator import _SupportsPagination

    from debusine.web.views.table.ordering import Ordering
    from debusine.web.views.table.table import Table

    _SupportsPagination  # fake usage for vulture


M = TypeVar("M", bound=Model)


class ColumnState(StrEnum):
    """Activity state of a column."""

    INACTIVE = "inactive"
    ASCENDING = "ascending"
    DESCENDING = "descending"


class Column(Widget):
    """A sortable column."""

    #: Column name
    name: str
    #: Column title
    title: str
    #: Ordering
    ordering: Union["Ordering", None]

    def __init__(
        self,
        table: "Table[Any]",
        name: str,
        title: str,
        ordering: Union["Ordering", None] = None,
    ) -> None:
        """Store arguments in the Column object for rendering."""
        self.table = table
        self.name = name
        self.title = title
        self.ordering = ordering
        self.state: ColumnState = ColumnState.INACTIVE

    @cached_property
    def query_asc(self) -> str:
        """Return the query string for sorting by this column (ascending)."""
        query = self.table.request.GET.copy()
        query[f"{self.table.prefix}order"] = self.name
        query[f"{self.table.prefix}asc"] = "1"
        return query.urlencode()

    @cached_property
    def query_desc(self) -> str:
        """Return the query string for sorting by this column (descending)."""
        query = self.table.request.GET.copy()
        query[f"{self.table.prefix}order"] = self.name
        query[f"{self.table.prefix}asc"] = "0"
        return query.urlencode()

    def render(self, context: BaseContext) -> str:  # noqa: U100
        """Render the column header."""
        if self.ordering is None:
            return self.title
        match self.state:
            case ColumnState.INACTIVE:
                return format_html(
                    "{title} "
                    "<a href='?{query_asc}' class='order-inactive'>&#9650;</a>"
                    "<a href='?{query_desc}'"
                    " class='order-inactive'>&#9660;</a>",
                    title=self.title,
                    query_asc=self.query_asc,
                    query_desc=self.query_desc,
                )
            case ColumnState.ASCENDING:
                return format_html(
                    "{title} "
                    "<span class='order-active'>&#9650;</span>"
                    "<a href='?{query}' class='order-inactive'>"
                    "<span class='arrow'>&#9660;</span></a>",
                    title=self.title,
                    query=self.query_desc,
                )
            case ColumnState.DESCENDING:
                return format_html(
                    "{title} "
                    "<a href='?{query}' class='order-inactive'>"
                    "<span class='arrow'>&#9650;</span></a>"
                    "<span class='order-active'>&#9660;</span>",
                    title=self.title,
                    query=self.query_asc,
                )
            case _ as unreachable:
                assert_never(unreachable)
