# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Tests for the debusine-client setup code."""

import errno
import json
import tempfile
import textwrap
from unittest import mock

import responses
from configobj import ConfigObj

from debusine.client.models import (
    EnrollConfirmPayload,
    EnrollOutcome,
    EnrollPayload,
)
from debusine.client.setup import (
    KNOWN_SERVERS,
    ServerConfig,
    ServerConfigEditor,
    ServerInfo,
    ServerSelector,
    setup_server,
)
from debusine.client.tests.utils import TestConsole
from debusine.test import TestCase

FEEDBACK_CONFIG_SAVED = "👍 Configuration saved"
FEEDBACK_TOKEN_ACQUIRED = "✅ Confirmation confirmed, token acquired"


class ServerInfoTests(TestCase):
    """Tests for :py:class:`ServerInfo`."""

    def test_from_config(self) -> None:
        """Test from_config."""
        workdir = self.create_temporary_directory()
        path = workdir / "debusine.ini"
        desc = f"Configured in {path}"
        for name, cfg, info in (
            (
                "example",
                {"api-url": "http://example.org/api/", "scope": "debusine"},
                ServerInfo(
                    name="example",
                    api_url="http://example.org/api/",
                    desc=desc,
                    scope="debusine",
                ),
            ),
            (
                "debian",
                {"scope": "debusine"},
                ServerInfo(
                    name="debian",
                    api_url="https://debusine.debian.net/api/",
                    desc=desc,
                    scope="debusine",
                ),
            ),
            (
                "empty",
                {},
                ServerInfo(
                    name="empty",
                    api_url="https://empty/api/",
                    desc=desc,
                    scope="debusine",
                ),
            ),
        ):
            initial_cfg = ConfigObj({f"server:{name}": cfg})
            initial_cfg.filename = path.as_posix()
            initial_cfg.write()
            loaded_cfg = ConfigObj(path.as_posix())
            loaded_info = ServerInfo.from_config(
                name, loaded_cfg[f"server:{name}"]
            )
            self.assertEqual(loaded_info, info)

    def test_from_string_known(self) -> None:
        """Test from_string with a known name or hostname."""
        for arg, name in (
            # Match by name
            ("debian", "debian"),
            ("localhost", "localhost"),
            # Match by hostname
            ("debusine.debian.net", "debian"),
            ("localhost:8000", "localhost"),
            (":8000", "localhost"),
            # Match by URL hostname
            ("https://debusine.debian.net/", "debian"),
            ("http://localhost:8000/api/", "localhost"),
        ):
            with self.subTest(arg=arg):
                self.assertIs(
                    ServerInfo.from_string(arg, desc="test"),
                    KNOWN_SERVERS[name],
                )

    def test_from_string(self) -> None:
        """Test building a ServerInfo from a string argument."""
        for server, server_name, api_url in (
            # Bare name
            ("example", "example", "https://example/api/"),
            # Domain name
            (
                "example.org",
                "example.org",
                "https://example.org/api/",
            ),
            # Domain name and port
            (
                "example.org:8000",
                "example.org",
                "https://example.org:8000/api/",
            ),
            # Base URL
            (
                "https://example.org",
                "example.org",
                "https://example.org/api/",
            ),
            # API URL without trailing slash
            (
                "https://example.org/api",
                "example.org",
                "https://example.org/api/",
            ),
            # API URL with trailing slash
            (
                "https://example.org/api/",
                "example.org",
                "https://example.org/api/",
            ),
            # Subdir URL without trailing slash
            (
                "https://example.org/debusine",
                "example.org",
                "https://example.org/debusine/api/",
            ),
            # Subdir URL with trailing slash
            (
                "https://example.org/debusine/",
                "example.org",
                "https://example.org/debusine/api/",
            ),
        ):
            with self.subTest(server=server):
                self.assertEqual(
                    ServerInfo.from_string(server, "desc"),
                    ServerInfo(
                        name=server_name,
                        api_url=api_url,
                        scope="debusine",
                        desc="desc",
                    ),
                )


class ServerConfigTests(TestCase):
    """Tests for :py:class:`ServerConfig`."""

    def test_save(self) -> None:
        """Test saving the configuration."""

        def body(text: str) -> str:
            return textwrap.dedent(text).lstrip()

        for initial, server, changed, expected in (
            # Fill with defaults
            (
                "",
                "localhost",
                {},
                body(
                    """
                    [server:localhost]
                    api-url = http://localhost:8000/api/
                    scope = debusine
                    """
                ),
            ),
            # Add a section from defaults
            (
                body(
                    """
                    # Debian account
                    [server:debian]
                    api-url=http://debusine.debian.net/api/
                    scope=debian
                    # Test comment
                    token=12345678
                    """
                ),
                "localhost",
                {},
                body(
                    """
                    # Debian account
                    [server:debian]
                    api-url = http://debusine.debian.net/api/
                    scope = debian
                    # Test comment
                    token = 12345678
                    [server:localhost]
                    api-url = http://localhost:8000/api/
                    scope = debusine
                    """
                ),
            ),
            # Fetch a token for an existing section
            (
                body(
                    """
                    # Debian account
                    [server:debian]
                    api-url=http://debusine.debian.net/api/
                    scope=debian
                    """
                ),
                "debian",
                {"api_token": "12345678"},
                body(
                    """
                    # Debian account
                    [server:debian]
                    api-url = http://debusine.debian.net/api/
                    scope = debian
                    token = 12345678
                    """
                ),
            ),
            # Change API URL, preserving comments
            (
                body(
                    """
                    # Debian account
                    [server:localhost]
                    # Misses a port?
                    api-url=http://localhost/api/
                    scope=debian
                    """
                ),
                "localhost",
                {"api_url": "http://localhost:8000/api/"},
                body(
                    """
                    # Debian account
                    [server:localhost]
                    # Misses a port?
                    api-url = http://localhost:8000/api/
                    scope = debian
                    """
                ),
            ),
            # Clear API token
            (
                body(
                    """
                    [server:localhost]
                    api-url=http://localhost/api/
                    scope=debian
                    # Token value
                    token=12345678
                    """
                ),
                "localhost",
                {"api_token": None},
                body(
                    """
                    [server:localhost]
                    api-url = http://localhost/api/
                    scope = debian
                    """
                ),
            ),
        ):
            with (
                self.subTest(initial=initial, server=server, changed=changed),
                tempfile.NamedTemporaryFile("w+t") as cfg_file,
            ):
                cfg_file.write(initial)
                cfg_file.flush()
                cfg = ServerConfig(
                    ConfigObj(cfg_file.name),
                    ServerInfo.from_string(server, desc="desc"),
                )
                for k, v in changed.items():
                    setattr(cfg, k, v)
                cfg.save()
                cfg_file.seek(0)
                self.assertEqual(cfg_file.read(), expected)

    def test_save_parent_dir_missing(self) -> None:
        """Test saving the configuration in a missing path."""
        workdir = self.create_temporary_directory()
        configdir = workdir / "client"
        configfile = configdir / "config.ini"
        self.assertFalse(configdir.exists())
        self.assertFalse(configfile.exists())
        cfg = ServerConfig(
            ConfigObj(configfile.as_posix()), KNOWN_SERVERS["localhost"]
        )
        cfg.save()
        self.assertTrue(configdir.is_dir())
        self.assertTrue(configfile.is_file())

    def test_lint_api_url(self) -> None:
        """Test linting of API URLs."""
        for config, messages in (
            (
                "api-url = https://localhost/api/",
                [
                    "Server url is an [bold]https[/] url,"
                    " should it be [bold]http[/]?"
                ],
            ),
            (
                "api-url = http://example.org/api/",
                [
                    "Server url is an [bold]http[/] url,"
                    " should it be [bold]https[/]?"
                ],
            ),
            ("api-url = https://example.org/api", []),
            (
                "api-url = https://example.org/other/",
                ["API URL does not end in /api or /api/"],
            ),
            (
                "api-url = http://example.org/",
                [
                    "Server url is an [bold]http[/] url,"
                    " should it be [bold]https[/]?",
                    "API URL does not end in /api or /api/",
                ],
            ),
        ):
            with self.subTest(config=config):
                cfg = ServerConfig(
                    ConfigObj(
                        ["[server:name]"] + textwrap.dedent(config).splitlines()
                    ),
                    ServerInfo.from_string("name", "test"),
                )
                self.assertEqual(cfg.lint(), messages)

    def _fetch_token(
        self,
        cfg: ServerConfig,
        nonce: str = "12345678",
        challenge: str = "correct horse battery staple",
    ) -> tuple[bool, list[str]]:
        console = TestConsole()
        with (
            mock.patch(
                "debusine.client.setup.secrets.token_urlsafe",
                return_value=nonce,
            ),
            mock.patch(
                "debusine.client.setup.xkcd_password.generate_xkcdpassword",
                return_value=challenge,
            ),
            mock.patch(
                "debusine.client.setup.platform.node",
                return_value="hostname",
            ),
        ):
            result = cfg.fetch_token(console)

        # A proper payload was sent
        self.assertEqual(len(responses.calls), 1)
        call = responses.calls[0]
        assert call.request.body is not None
        self.assertEqual(
            call.request.headers["Content-Type"], "application/json"
        )
        self.assertEqual(
            json.loads(call.request.body),
            EnrollPayload(
                nonce=nonce,
                challenge=challenge,
                hostname="hostname",
                scope=cfg.scope,
            ).dict(),
        )

        output_lines = console.output.getvalue().splitlines()

        confirm_url = f"https://debusine.debian.net/-/enroll/confirm/{nonce}"
        self.assertEqual(
            output_lines[0],
            f"👉 Please visit {confirm_url} to confirm registration",
        )

        self.assertEqual(
            output_lines[1],
            f'👉 Make sure the page mentions the passphrase: "{challenge}"',
        )

        return result, output_lines[2:]

    @responses.activate
    def test_fetch_token_confirm(self) -> None:
        """Test getting a token from the server."""
        nonce = "12345678"
        challenge = "correct horse battery staple"
        token = "a" * 32
        responses.add(
            responses.POST,
            "https://debusine.debian.net/api/enroll/",
            json=EnrollConfirmPayload(
                outcome=EnrollOutcome.CONFIRM, token=token
            ).dict(),
        )

        # Fetch the token
        cfg = ServerConfig(ConfigObj(), KNOWN_SERVERS["debian"])
        result, feedback = self._fetch_token(cfg, nonce, challenge)
        self.assertTrue(result)

        # Token was fetched and recorded
        self.assertEqual(cfg.api_token, token)

        # URL and challenge were printed in console
        self.assertEqual(
            feedback, ["✅ Confirmation confirmed, token acquired"]
        )

    @responses.activate
    def test_fetch_token_cancel(self) -> None:
        """Test getting a token from the server."""
        nonce = "12345678"
        challenge = "correct horse battery staple"
        responses.add(
            responses.POST,
            "https://debusine.debian.net/api/enroll/",
            json=EnrollConfirmPayload(outcome=EnrollOutcome.CANCEL).dict(),
        )

        # Fetch the token
        cfg = ServerConfig(ConfigObj(), KNOWN_SERVERS["debian"])
        result, feedback = self._fetch_token(cfg, nonce, challenge)
        self.assertTrue(result)

        # Token was fetched and recorded
        self.assertIsNone(cfg.api_token)

        # URL and challenge were printed in console
        self.assertEqual(feedback, ["❗ Confirmation cancelled"])

    @responses.activate
    def test_fetch_token_server_not_found(self) -> None:
        """Test fetch_token encountering a network error."""
        responses.add(
            responses.POST,
            "https://debusine.debian.net/api/enroll/",
            body=OSError(errno.ESRCH),
        )

        # Fetch the token
        cfg = ServerConfig(ConfigObj(), KNOWN_SERVERS["debian"])
        result, feedback = self._fetch_token(cfg)
        self.assertFalse(result)
        self.assertEqual(feedback, ["❗ request to server failed: 3"])

    @responses.activate
    def test_fetch_token_server_problemresponse(self) -> None:
        """Test fetch_token getting a ProblemResponse."""
        responses.add(
            responses.POST,
            "https://debusine.debian.net/api/enroll/",
            status=404,
            json={"title": "fake server error"},
        )

        cfg = ServerConfig(ConfigObj(), KNOWN_SERVERS["debian"])
        result, feedback = self._fetch_token(cfg)
        self.assertFalse(result)
        self.assertEqual(
            feedback,
            [
                "❗ The server returned status code 404",
                "❗ fake server error",
            ],
        )

    @responses.activate
    def test_fetch_token_non_problemresponse_json(self) -> None:
        """Test a JSON server error but not a ProblemResponse."""
        responses.add(
            responses.POST,
            "https://debusine.debian.net/api/enroll/",
            status=500,
            json={"mischief": True},
        )

        cfg = ServerConfig(ConfigObj(), KNOWN_SERVERS["debian"])
        result, feedback = self._fetch_token(cfg)
        self.assertFalse(result)
        self.assertEqual(
            feedback,
            [
                "❗ The server returned status code 500",
                '❗ {"mischief": true}',
            ],
        )

    @responses.activate
    def test_fetch_token_server_error(self) -> None:
        """Test fetch_token getting a server error."""
        responses.add(
            responses.POST,
            "https://debusine.debian.net/api/enroll/",
            status=500,
            body="fake internal server error",
        )

        cfg = ServerConfig(ConfigObj(), KNOWN_SERVERS["debian"])
        result, feedback = self._fetch_token(cfg)
        self.assertFalse(result)
        self.assertEqual(
            feedback,
            [
                "❗ The server returned status code 500",
                "❗ fake internal server error",
            ],
        )

    @responses.activate
    def test_fetch_token_server_badjson(self) -> None:
        """Test fetch_token getting invalid json."""
        responses.add(
            responses.POST,
            "https://debusine.debian.net/api/enroll/",
            body="{",
            content_type="application/json",
        )

        cfg = ServerConfig(ConfigObj(), KNOWN_SERVERS["debian"])
        result, feedback = self._fetch_token(cfg)
        self.assertFalse(result)
        self.assertEqual(feedback, ["❗ Invalid JSON in response"])

    @responses.activate
    def test_fetch_token_server_badpayload(self) -> None:
        """Test fetch_token getting a bad payload."""
        responses.add(
            responses.POST,
            "https://debusine.debian.net/api/enroll/",
            json={"mischief": True},
        )

        cfg = ServerConfig(ConfigObj(), KNOWN_SERVERS["debian"])
        result, feedback = self._fetch_token(cfg)
        self.assertFalse(result)
        self.assertEqual(feedback, ["❗ Invalid response payload"])


class ServerSelectorTests(TestCase):
    """Tests for :py:class:`ServerSelector`."""

    def test_cmdloop_no_entries(self) -> None:
        """Test cmdloop behaviour with no entries."""
        with (
            mock.patch(
                "debusine.client.dataentry.DataEntry.cmdloop"
            ) as cmdloop,
            mock.patch("debusine.client.setup.ServerSelector.do_new") as do_new,
        ):
            console = TestConsole()
            selector = ServerSelector([], console=console)
            selector.cmdloop()
        cmdloop.assert_not_called()
        do_new.assert_called()

    def test_cmdloop_with_entries(self) -> None:
        """Test cmdloop behaviour with entries."""
        with (
            mock.patch(
                "debusine.client.dataentry.DataEntry.cmdloop"
            ) as cmdloop,
            mock.patch("debusine.client.setup.ServerSelector.do_new") as do_new,
        ):
            selector = ServerSelector(
                list(KNOWN_SERVERS.values()), console=TestConsole()
            )
            selector.cmdloop()
        cmdloop.assert_called()
        do_new.assert_not_called()

    def test_do_new(self) -> None:
        """Test do_new."""
        for entered, stored in (
            ("", None),
            ("localhost", KNOWN_SERVERS["localhost"]),
            (
                "https://example.org",
                ServerInfo(
                    "example.org",
                    desc="Manually entered",
                    api_url="https://example.org/api/",
                    scope="debusine",
                ),
            ),
        ):
            with self.subTest(entered=entered):
                selector = ServerSelector(
                    [KNOWN_SERVERS["localhost"]], console=TestConsole()
                )
                with mock.patch(
                    "debusine.client.dataentry.DataEntry.input_line",
                    return_value=entered,
                ):
                    selector.onecmd("new")
                self.assertEqual(selector.selected, stored)

    def test_menu_prompt(self) -> None:
        """Test menu prompt."""
        for entries, expected in (
            ([], "new, quit:"),
            ([ServerInfo.from_string("a", "a")], "1, new, quit:"),
            (
                [ServerInfo.from_string(x, x) for x in ("a", "b", "c")],
                "1…3, new, quit:",
            ),
        ):
            with self.subTest(entries=entries):
                console = TestConsole()
                selector = ServerSelector(entries, console=console)
                selector.menu()
                self.assertEqual(
                    console.output.getvalue().splitlines()[-1], expected
                )

    def test_enter_number(self) -> None:
        """Test entering a number to pick an entry."""
        selector = ServerSelector(
            list(KNOWN_SERVERS.values()), console=TestConsole()
        )
        self.assertTrue(selector.onecmd("2"))
        self.assertIs(selector.selected, KNOWN_SERVERS["freexian"])

    def test_enter_number_out_of_range(self) -> None:
        """Test entering a number out of range."""
        console = TestConsole()
        selector = ServerSelector(list(KNOWN_SERVERS.values()), console=console)
        for num in 0, len(KNOWN_SERVERS) + 1, len(KNOWN_SERVERS) + 10:
            with self.subTest(num=num):
                console.reset_output()
                self.assertFalse(selector.onecmd(str(num)))
                self.assertEqual(
                    console.output.getvalue(),
                    "Entry number must be between"
                    f" 1 and {len(KNOWN_SERVERS)}\n",
                )
                self.assertIsNone(selector.selected)

    def test_enter_invalid_command(self) -> None:
        """Test entering an invalid command."""
        console = TestConsole()
        selector = ServerSelector(list(KNOWN_SERVERS.values()), console=console)
        for cmd in "-1", "14.3", "mischief":
            with self.subTest(cmd=cmd):
                console.reset_output()
                self.assertFalse(selector.onecmd(cmd))
                self.assertEqual(
                    console.output.getvalue(),
                    f"command not recognized: {cmd!r}\n",
                )
                self.assertIsNone(selector.selected)


class ServerConfigEditorTests(TestCase):
    """Tests for :py:class:`ServerConfigEditor`."""

    def _config(self) -> ServerConfig:
        return ServerConfig(config=ConfigObj(), server=KNOWN_SERVERS["debian"])

    def test_menu(self) -> None:
        """Test formatting the menu."""
        cfg = self._config()
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        editor.menu()
        output = console.output.getvalue()
        self.assertEqual(
            output.splitlines()[-1], "Commands: url, token, scope, save, quit:"
        )
        self.assertIn("Configuration for debian", output)
        self.assertIn("[server:debian]", output)
        self.assertIn("scope = debian", output)
        self.assertIn("token = acquired on save", output)
        self.assertIn("api-url = https://debusine.debian.net/api/", output)

    def test_menu_with_token(self) -> None:
        """Test formatting the menu when the token is set."""
        cfg = self._config()
        cfg.api_token = "12345678"
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        editor.menu()
        output = console.output.getvalue()
        self.assertIn("token = present", output)

    def test_menu_linter(self) -> None:
        """Ensure linter messages appear in the menu."""
        cfg = self._config()
        cfg.api_url = "http://debusine.debian.net/api/"
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        editor.menu()
        output = console.output.getvalue()
        self.assertIn(
            "👉 Server url is an http url, should it be https?", output
        )

    def test_edit_scope(self) -> None:
        """Edit the scope."""
        cfg = self._config()
        self.assertEqual(cfg.scope, "debian")
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        with mock.patch(
            "debusine.client.dataentry.DataEntry.input_line",
            return_value="debusine",
        ) as input_line:
            editor.onecmd("scope")
        self.assertEqual(cfg.scope, "debusine")
        input_line.assert_called_with(
            "Enter scope", initial="debian", required=True
        )

    def test_edit_url(self) -> None:
        """Edit the api url."""
        old_url = "https://debusine.debian.net/api/"
        new_url = "https://debusine.debian.org/api/"
        cfg = self._config()
        self.assertEqual(cfg.api_url, old_url)
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        with mock.patch(
            "debusine.client.dataentry.DataEntry.input_line",
            return_value=new_url,
        ) as input_line:
            editor.onecmd("url")
        self.assertEqual(cfg.api_url, new_url)
        input_line.assert_called_with(
            "Enter API URL", initial=old_url, required=True
        )

    def test_edit_token(self) -> None:
        """Edit the token."""
        new_token = "12345678"
        cfg = self._config()
        self.assertIsNone(cfg.api_token)
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        with mock.patch(
            "debusine.client.dataentry.DataEntry.input_line",
            return_value=new_token,
        ) as input_line:
            editor.onecmd("token")
        self.assertEqual(cfg.api_token, new_token)
        input_line.assert_called_with(
            "Enter (or clear) API token", initial=None
        )

    def test_clear_token(self) -> None:
        """Clear the token."""
        old_token = "12345678"
        cfg = self._config()
        cfg.api_token = old_token
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        with mock.patch(
            "debusine.client.dataentry.DataEntry.input_line",
            return_value="",
        ) as input_line:
            editor.onecmd("token")
        self.assertIsNone(cfg.api_token)
        input_line.assert_called_with(
            "Enter (or clear) API token", initial=old_token
        )

    def test_quit(self) -> None:
        """Test quitting the editor."""
        cfg = self._config()
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        self.assertTrue(editor.onecmd("quit"))
        output = console.output.getvalue()
        self.assertEqual(
            output.splitlines()[-1], "🖐 Configuration left unchanged"
        )

    def test_save_with_token(self) -> None:
        """Test the save command with a token already present."""
        cfg = self._config()
        cfg.api_token = "12345678"
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        with mock.patch("debusine.client.setup.ServerConfig.save") as save:
            self.assertTrue(editor.onecmd("save"))
        save.assert_called()
        self.assertEqual(
            console.output.getvalue().splitlines(), [FEEDBACK_CONFIG_SAVED]
        )

    def test_save_fetch_token_confirm(self) -> None:
        """Test the save command with a token, user confirmed."""
        cfg = self._config()

        def _mock_fetch_token(console: TestConsole) -> bool:  # noqa: U100
            cfg.api_token = "12345678"
            return True

        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        with (
            mock.patch.object(cfg, "save") as save,
            mock.patch.object(
                cfg, "fetch_token", side_effect=_mock_fetch_token
            ) as fetch_token,
        ):
            self.assertTrue(editor.onecmd("save"))
        fetch_token.assert_called_with(console)
        save.assert_called()
        self.assertEqual(cfg.api_token, "12345678")
        self.assertEqual(
            console.output.getvalue().splitlines(), [FEEDBACK_CONFIG_SAVED]
        )

    def test_save_fetch_token_cancel(self) -> None:
        """Test the save command with a token, user cancelled."""
        cfg = self._config()

        def _mock_fetch_token(console: TestConsole) -> bool:  # noqa: U100
            cfg.api_token = None
            return True

        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        with (
            mock.patch.object(cfg, "save") as save,
            mock.patch.object(
                cfg, "fetch_token", side_effect=_mock_fetch_token
            ) as fetch_token,
        ):
            self.assertTrue(editor.onecmd("save"))
        fetch_token.assert_called_with(console)
        save.assert_called()
        self.assertIsNone(cfg.api_token)
        self.assertEqual(
            console.output.getvalue().splitlines(), [FEEDBACK_CONFIG_SAVED]
        )

    def test_save_retry(self) -> None:
        """Test the save command with a token."""
        cfg = self._config()

        call_count = 0

        def _mock_fetch_token(console: TestConsole) -> bool:  # noqa: U100
            nonlocal call_count
            try:
                match call_count:
                    case 0:
                        return False
                    case _:
                        cfg.api_token = "12345678"
                        return True
            finally:
                call_count += 1

        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        with (
            mock.patch.object(cfg, "save") as save,
            mock.patch.object(
                cfg, "fetch_token", side_effect=_mock_fetch_token
            ) as fetch_token,
            mock.patch(
                "debusine.client.setup.Confirm.ask", return_value=True
            ) as ask,
        ):
            self.assertTrue(editor.onecmd("save"))
        fetch_token.assert_called_with(console)
        ask.assert_called_with("Fetch failed: retry?", console=console)
        save.assert_called()
        self.assertEqual(cfg.api_token, "12345678")
        self.assertEqual(
            console.output.getvalue().splitlines(), [FEEDBACK_CONFIG_SAVED]
        )

    def test_save_abort(self) -> None:
        """Test the save command with a token."""
        cfg = self._config()
        console = TestConsole()
        editor = ServerConfigEditor(cfg, console=console)
        with (
            mock.patch.object(cfg, "save") as save,
            mock.patch.object(
                cfg, "fetch_token", return_value=False
            ) as fetch_token,
            mock.patch(
                "debusine.client.setup.Confirm.ask", return_value=False
            ) as ask,
        ):
            self.assertTrue(editor.onecmd("save"))
        fetch_token.assert_called_with(console)
        ask.assert_called_with("Fetch failed: retry?", console=console)
        save.assert_called()
        self.assertIsNone(cfg.api_token)
        self.assertEqual(
            console.output.getvalue().splitlines(), [FEEDBACK_CONFIG_SAVED]
        )


class SetupServerTests(TestCase):
    """Tests for :py:func:`setup_server`."""

    def test_no_args(self) -> None:
        """Invoke with no arguments."""
        workdir = self.create_temporary_directory()
        console = TestConsole()

        def _select_server(self: ServerSelector) -> None:
            self.selected = ServerInfo.from_string("name", desc="desc")

        with (
            mock.patch(
                "debusine.client.setup.ServerSelector.cmdloop",
                autospec=True,
                side_effect=_select_server,
            ) as selector_cmdloop,
            mock.patch(
                "debusine.client.setup.ServerConfigEditor.cmdloop",
                autospec=True,
            ) as editor_cmdloop,
        ):
            setup_server(
                config_file_path=workdir / "config.ini", console=console
            )

        selector_cmdloop.assert_called()
        selector = selector_cmdloop.call_args.args[0]
        self.assertEqual(selector.entries, list(KNOWN_SERVERS.values()))
        editor_cmdloop.assert_called()
        editor = editor_cmdloop.call_args.args[0]
        self.assertEqual(editor.config.server.name, "name")
        self.assertEqual(editor.config.scope, "debusine")

    def test_no_server_selected(self) -> None:
        """Invoke with no arguments."""
        workdir = self.create_temporary_directory()
        console = TestConsole()

        with (
            mock.patch(
                "debusine.client.setup.ServerSelector.cmdloop",
            ) as selector_cmdloop,
            mock.patch(
                "debusine.client.setup.ServerConfigEditor.cmdloop",
                autospec=True,
            ) as editor_cmdloop,
        ):
            setup_server(
                config_file_path=workdir / "config.ini", console=console
            )

        selector_cmdloop.assert_called()
        editor_cmdloop.assert_not_called()

    def test_server_provided(self) -> None:
        """Invoke with server set."""
        workdir = self.create_temporary_directory()
        console = TestConsole()

        with (
            mock.patch(
                "debusine.client.setup.ServerSelector.cmdloop",
            ) as selector_cmdloop,
            mock.patch(
                "debusine.client.setup.ServerConfigEditor.cmdloop",
                autospec=True,
            ) as editor_cmdloop,
        ):
            setup_server(
                config_file_path=workdir / "config.ini",
                server="name",
                console=console,
            )

        selector_cmdloop.assert_not_called()
        editor_cmdloop.assert_called()
        editor = editor_cmdloop.call_args.args[0]
        self.assertEqual(editor.config.server.name, "name")
        self.assertEqual(editor.config.scope, "debusine")

    def test_scope_provided(self) -> None:
        """Invoke with scope set."""
        workdir = self.create_temporary_directory()
        console = TestConsole()

        def _select_server(self: ServerSelector) -> None:
            self.selected = ServerInfo.from_string("name", desc="desc")

        with (
            mock.patch(
                "debusine.client.setup.ServerSelector.cmdloop",
                autospec=True,
                side_effect=_select_server,
            ) as selector_cmdloop,
            mock.patch(
                "debusine.client.setup.ServerConfigEditor.cmdloop",
                autospec=True,
            ) as editor_cmdloop,
        ):
            setup_server(
                config_file_path=workdir / "config.ini",
                scope="scope",
                console=console,
            )

        selector_cmdloop.assert_called()
        selector = selector_cmdloop.call_args.args[0]
        self.assertEqual(selector.entries, list(KNOWN_SERVERS.values()))
        editor_cmdloop.assert_called()
        editor = editor_cmdloop.call_args.args[0]
        self.assertEqual(editor.config.server.name, "name")
        self.assertEqual(editor.config.scope, "scope")

    def test_existing_config(self) -> None:
        """Invoke with an existing configuration."""
        workdir = self.create_temporary_directory()
        config_path = workdir / "config.ini"
        config_path.write_text(
            textwrap.dedent(
                """
                [server:localhost]
                scope = debusine
                api-url = http://localhost:8000/api/
                [server:debian]
                scope = debian
                api-url = http://debusine.debian.net/api/
                [server:example]
                """
            )
        )
        console = TestConsole()

        def _select_server(self: ServerSelector) -> None:
            self.selected = KNOWN_SERVERS["debian"]

        with (
            mock.patch(
                "debusine.client.setup.ServerSelector.cmdloop",
                autospec=True,
                side_effect=_select_server,
            ) as selector_cmdloop,
            mock.patch(
                "debusine.client.setup.ServerConfigEditor.cmdloop",
                autospec=True,
            ) as editor_cmdloop,
        ):
            setup_server(
                config_file_path=config_path,
                console=console,
            )

        selector_cmdloop.assert_called()
        selector = selector_cmdloop.call_args.args[0]
        self.assertEqual(
            [x.name for x in selector.entries],
            ["localhost", "debian", "example", "freexian"],
        )
        editor_cmdloop.assert_called()
        editor = editor_cmdloop.call_args.args[0]
        self.assertEqual(editor.config.server.name, "debian")
        self.assertEqual(editor.config.scope, "debian")

    def test_existing_config_override_scope(self) -> None:
        """Invoke with an existing configuration, setting a scope."""
        workdir = self.create_temporary_directory()
        config_path = workdir / "config.ini"
        config_path.write_text(
            textwrap.dedent(
                """
                [server:localhost]
                scope = debusine
                api-url = http://localhost:8000/api/
                """
            )
        )
        console = TestConsole()

        with (
            mock.patch(
                "debusine.client.setup.ServerConfigEditor.cmdloop",
                autospec=True,
            ) as editor_cmdloop,
        ):
            setup_server(
                config_file_path=config_path,
                server="localhost",
                scope="debian",
                console=console,
            )

        editor_cmdloop.assert_called()
        editor = editor_cmdloop.call_args.args[0]
        self.assertEqual(editor.config.server.name, "localhost")
        self.assertEqual(editor.config.scope, "debian")
