/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.backend.store.ram;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import org.apache.commons.io.FileUtils;
import org.apache.hugegraph.HugeException;
import org.apache.hugegraph.HugeGraph;
import org.apache.hugegraph.backend.id.Id;
import org.apache.hugegraph.backend.id.IdGenerator;
import org.apache.hugegraph.backend.query.Condition;
import org.apache.hugegraph.backend.query.ConditionQuery;
import org.apache.hugegraph.backend.query.ConditionQueryFlatten;
import org.apache.hugegraph.backend.query.Query;
import org.apache.hugegraph.backend.store.ram.IntIntMap;
import org.apache.hugegraph.backend.store.ram.IntLongMap;
import org.apache.hugegraph.iterator.FlatMapperIterator;
import org.apache.hugegraph.perf.PerfUtil;
import org.apache.hugegraph.schema.EdgeLabel;
import org.apache.hugegraph.schema.VertexLabel;
import org.apache.hugegraph.structure.HugeEdge;
import org.apache.hugegraph.structure.HugeVertex;
import org.apache.hugegraph.type.HugeType;
import org.apache.hugegraph.type.define.Directions;
import org.apache.hugegraph.type.define.HugeKeys;
import org.apache.hugegraph.util.Consumers;
import org.apache.hugegraph.util.Log;
import org.apache.tinkerpop.gremlin.structure.Edge;
import org.apache.tinkerpop.gremlin.structure.Vertex;
import org.apache.tinkerpop.gremlin.structure.util.CloseableIterator;
import org.apache.tinkerpop.gremlin.util.iterator.IteratorUtils;
import org.slf4j.Logger;

public final class RamTable {
    public static final String USER_DIR = System.getProperty("user.dir");
    public static final String EXPORT_PATH = USER_DIR + "/export";
    private static final Logger LOG = Log.logger(RamTable.class);
    private static final long VERTICES_CAPACITY = 2400000000L;
    private static final int EDGES_CAPACITY = 2100000000;
    private static final int NULL = 0;
    private static final Condition BOTH_COND = Condition.or(Condition.eq(HugeKeys.DIRECTION, (Object)Directions.OUT), Condition.eq(HugeKeys.DIRECTION, (Object)Directions.IN));
    private final HugeGraph graph;
    private final long verticesCapacity;
    private final int verticesCapacityHalf;
    private final int edgesCapacity;
    private IntIntMap verticesLow;
    private IntIntMap verticesHigh;
    private IntLongMap edges;
    private volatile boolean loading = false;

    public RamTable(HugeGraph graph) {
        this(graph, 2400000000L, 2100000000);
    }

    public RamTable(HugeGraph graph, long maxVertices, int maxEdges) {
        this.graph = graph;
        this.verticesCapacity = maxVertices + 2L;
        this.verticesCapacityHalf = (int)(this.verticesCapacity / 2L);
        this.edgesCapacity = maxEdges + 1;
        this.reset();
    }

    private void reset() {
        this.verticesLow = null;
        this.verticesHigh = null;
        this.edges = null;
        this.verticesLow = new IntIntMap(this.verticesCapacityHalf);
        this.verticesHigh = new IntIntMap(this.verticesCapacityHalf);
        this.edges = new IntLongMap(this.edgesCapacity);
        this.edges.add(0L);
    }

    public void reload(boolean loadFromFile, String file) {
        if (this.loading) {
            throw new HugeException("There is one loading task, please wait for it to complete");
        }
        this.loading = true;
        try {
            this.reset();
            if (loadFromFile) {
                this.loadFromFile(file);
            } else {
                this.loadFromDB();
                if (file != null) {
                    LOG.info("Export graph to file '{}'", (Object)file);
                    if (!this.exportToFile(file)) {
                        LOG.warn("Can't export graph to file '{}'", (Object)file);
                    }
                }
            }
            LOG.info("Loaded {} edges", (Object)this.edgesSize());
        }
        catch (Throwable e) {
            this.reset();
            throw new HugeException("Failed to load ramtable", e);
        }
        finally {
            this.loading = false;
        }
    }

    private void loadFromFile(String fileName) throws Exception {
        File file = Paths.get(EXPORT_PATH, fileName).toFile();
        if (!(file.exists() && file.isFile() && file.canRead())) {
            throw new IllegalArgumentException(String.format("File '%s' does not existed or readable", fileName));
        }
        try (FileInputStream fis = new FileInputStream(file);
             BufferedInputStream bis = new BufferedInputStream(fis);
             DataInputStream input = new DataInputStream(bis);){
            this.verticesLow.readFrom(input);
            this.verticesHigh.readFrom(input);
            this.edges.readFrom(input);
        }
    }

    private boolean exportToFile(String fileName) throws Exception {
        File file = Paths.get(EXPORT_PATH, fileName).toFile();
        if (!file.exists()) {
            FileUtils.forceMkdir((File)file.getParentFile());
            if (!file.createNewFile()) {
                return false;
            }
        }
        try (FileOutputStream fos = new FileOutputStream(file);
             BufferedOutputStream bos = new BufferedOutputStream(fos);
             DataOutputStream output = new DataOutputStream(bos);){
            this.verticesLow.writeTo(output);
            this.verticesHigh.writeTo(output);
            this.edges.writeTo(output);
        }
        return true;
    }

    private void loadFromDB() throws Exception {
        Query query = new Query(HugeType.VERTEX);
        query.capacity((long)this.verticesCapacityHalf * 2L);
        query.limit(Long.MAX_VALUE);
        Iterator<Vertex> vertices = this.graph.vertices(query);
        boolean concurrent = true;
        if (concurrent) {
            try (LoadTraverser traverser = new LoadTraverser();){
                traverser.load(vertices);
            }
            return;
        }
        Id lastId = IdGenerator.ZERO;
        while (vertices.hasNext()) {
            HugeEdge edge;
            Id vertex = (Id)vertices.next().id();
            if (vertex.compareTo(lastId) < 0) {
                throw new HugeException("The ramtable feature is not supported by %s backend", this.graph.backend());
            }
            RamTable.ensureNumberId(vertex);
            lastId = vertex;
            Iterator<Edge> adjEdges = this.graph.adjacentEdges(vertex);
            if (adjEdges.hasNext()) {
                edge = (HugeEdge)adjEdges.next();
                this.addEdge(true, edge);
            }
            while (adjEdges.hasNext()) {
                edge = (HugeEdge)adjEdges.next();
                this.addEdge(false, edge);
            }
        }
    }

    public void addEdge(boolean newVertex, HugeEdge edge) {
        if (edge.schemaLabel().existSortKeys()) {
            throw new HugeException("Only edge label without sortkey is supported by ramtable, but got '%s'", edge.schemaLabel());
        }
        RamTable.ensureNumberId(edge.id().ownerVertexId());
        RamTable.ensureNumberId(edge.id().otherVertexId());
        this.addEdge(newVertex, edge.id().ownerVertexId().asLong(), edge.id().otherVertexId().asLong(), edge.direction(), (int)edge.schemaLabel().id().asLong());
    }

    public void addEdge(boolean newVertex, long owner, long target, Directions direction, int label) {
        long value = RamTable.encode(target, direction, label);
        this.addEdge(newVertex, owner, value);
    }

    public void addEdge(boolean newVertex, long owner, long value) {
        int position = this.edges.add(value);
        if (newVertex) {
            assert (this.vertexAdjPosition(owner) <= 0) : owner;
            this.vertexAdjPosition(owner, position);
        }
        this.vertexAdjPosition(owner + 1L, -position);
    }

    public long edgesSize() {
        return this.edges.size() - 1L;
    }

    @PerfUtil.Watched
    public boolean matched(Query query) {
        if (this.edgesSize() == 0L || this.loading) {
            return false;
        }
        if (!query.resultType().isEdge() || !(query instanceof ConditionQuery)) {
            return false;
        }
        ConditionQuery cq = (ConditionQuery)query;
        int conditionsSize = cq.conditionsSize();
        Object owner = cq.condition((Object)HugeKeys.OWNER_VERTEX);
        Directions direction = (Directions)cq.condition((Object)HugeKeys.DIRECTION);
        Id label = (Id)cq.condition((Object)HugeKeys.LABEL);
        if (direction == null && conditionsSize > 1) {
            for (Condition cond : cq.conditions()) {
                if (!cond.equals(BOTH_COND)) continue;
                direction = Directions.BOTH;
                break;
            }
        }
        int matchedConds = 0;
        if (owner != null) {
            ++matchedConds;
        } else {
            return false;
        }
        if (direction != null) {
            ++matchedConds;
        }
        if (label != null) {
            ++matchedConds;
        }
        return matchedConds == cq.conditionsSize();
    }

    @PerfUtil.Watched
    public Iterator<HugeEdge> query(Query query) {
        assert (this.matched(query));
        assert (this.edgesSize() > 0L);
        List<ConditionQuery> cqs = ConditionQueryFlatten.flatten((ConditionQuery)query);
        if (cqs.size() == 1) {
            ConditionQuery cq2 = cqs.get(0);
            return this.query(cq2);
        }
        return new FlatMapperIterator(cqs.iterator(), cq -> this.query((ConditionQuery)cq));
    }

    private Iterator<HugeEdge> query(ConditionQuery query) {
        Id label;
        Id owner = (Id)query.condition((Object)HugeKeys.OWNER_VERTEX);
        assert (owner != null);
        Directions dir = (Directions)query.condition((Object)HugeKeys.DIRECTION);
        if (dir == null) {
            dir = Directions.BOTH;
        }
        if ((label = (Id)query.condition((Object)HugeKeys.LABEL)) == null) {
            label = IdGenerator.ZERO;
        }
        return this.query(owner.asLong(), dir, (int)label.asLong());
    }

    @PerfUtil.Watched
    public Iterator<HugeEdge> query(long owner, Directions dir, int label) {
        if (this.loading) {
            return Collections.emptyIterator();
        }
        int start = this.vertexAdjPosition(owner);
        if (start <= 0) {
            return Collections.emptyIterator();
        }
        int end = this.vertexAdjPosition(owner + 1L);
        assert (start != 0);
        if (end < 0) {
            end = 1 - end;
        }
        return new EdgeRangeIterator(start, end, dir, label, owner);
    }

    private void vertexAdjPosition(long vertex, int position) {
        if (vertex < (long)this.verticesCapacityHalf) {
            this.verticesLow.put(vertex, position);
        } else if (vertex < this.verticesCapacity) {
            assert ((vertex -= (long)this.verticesCapacityHalf) < Integer.MAX_VALUE);
            this.verticesHigh.put(vertex, position);
        } else {
            throw new HugeException("Out of vertices capacity %s", this.verticesCapacity);
        }
    }

    private int vertexAdjPosition(long vertex) {
        if (vertex < (long)this.verticesCapacityHalf) {
            return this.verticesLow.get(vertex);
        }
        if (vertex < this.verticesCapacity) {
            assert ((vertex -= (long)this.verticesCapacityHalf) < Integer.MAX_VALUE);
            return this.verticesHigh.get(vertex);
        }
        throw new HugeException("Out of vertices capacity %s: %s", this.verticesCapacity, vertex);
    }

    private static void ensureNumberId(Id id) {
        if (!id.number()) {
            throw new HugeException("Only number id is supported by ramtable, but got %s id '%s'", id.type().name().toLowerCase(), id);
        }
    }

    private static long encode(long target, Directions direction, int label) {
        assert ((label & 0xFFFFFFF) == label);
        assert (target < 0xFFFFFFFEL) : target;
        long value = target & 0xFFFFFFFFFFFFFFFFL;
        long dir = direction == Directions.OUT ? 0L : 0x80000000L;
        value = value << 32 | (dir | (long)label);
        return value;
    }

    private class LoadTraverser
    implements AutoCloseable {
        private final HugeGraph graph;
        private final ExecutorService executor;
        private final List<Id> vertices;
        private final Map<Id, List<Edge>> edges;
        private static final int ADD_BATCH = 1000;

        public LoadTraverser() {
            this.graph = RamTable.this.graph;
            this.executor = Consumers.newThreadPool("ramtable-load", Consumers.THREADS);
            this.vertices = new ArrayList<Id>(1000);
            this.edges = new ConcurrentHashMap<Id, List<Edge>>();
        }

        @Override
        public void close() throws Exception {
            if (this.executor != null) {
                for (int i = 0; i < Consumers.THREADS; ++i) {
                    this.executor.execute(() -> this.graph.tx().commit());
                }
                this.executor.shutdown();
            }
        }

        protected long load(Iterator<Vertex> vertices) {
            Consumers<Id> consumers = new Consumers<Id>(this.executor, vertex -> {
                Iterator<Edge> adjEdges = this.graph.adjacentEdges((Id)vertex);
                this.edges.put((Id)vertex, IteratorUtils.list(adjEdges));
            }, null);
            consumers.start("ramtable-loading");
            long total = 0L;
            try {
                while (vertices.hasNext()) {
                    if (++total % 10000000L == 0L) {
                        LOG.info("Loaded {} vertices", (Object)total);
                    }
                    Id vertex2 = (Id)vertices.next().id();
                    this.addVertex(vertex2);
                    consumers.provide(vertex2);
                }
            }
            catch (Consumers.StopExecution e) {
                try {
                    consumers.await();
                }
                catch (Throwable e2) {
                    throw Consumers.wrapException(e2);
                }
                finally {
                    CloseableIterator.closeIterator(vertices);
                }
            }
            catch (Throwable e) {
                throw Consumers.wrapException(e);
            }
            finally {
                try {
                    consumers.await();
                }
                catch (Throwable e) {
                    throw Consumers.wrapException(e);
                }
                finally {
                    CloseableIterator.closeIterator(vertices);
                }
            }
            this.addEdgesByBatch();
            return total;
        }

        private void addVertex(Id vertex) {
            Id lastId = IdGenerator.ZERO;
            if (this.vertices.size() > 0) {
                lastId = this.vertices.get(this.vertices.size() - 1);
            }
            LOG.info("scan from hbase source {} lastId value: {} compare {} size {}", new Object[]{vertex, lastId, vertex.compareTo(lastId), this.vertices.size()});
            if (vertex.compareTo(lastId) < 0) {
                throw new HugeException("The ramtable feature is not supported by %s backend", this.graph.backend());
            }
            if (!vertex.number()) {
                throw new HugeException("Only number id is supported by ramtable, but got %s id '%s'", vertex.type().name().toLowerCase(), vertex);
            }
            if (this.vertices.size() >= 1000) {
                this.addEdgesByBatch();
            }
            this.vertices.add(vertex);
        }

        private void addEdgesByBatch() {
            int waitTimes = 0;
            for (Id vertex : this.vertices) {
                List<Edge> adjEdges = this.edges.remove(vertex);
                while (adjEdges == null) {
                    ++waitTimes;
                    try {
                        Thread.sleep(1L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                    adjEdges = this.edges.remove(vertex);
                }
                for (int i = 0; i < adjEdges.size(); ++i) {
                    HugeEdge edge = (HugeEdge)adjEdges.get(i);
                    assert (edge.id().ownerVertexId().equals(vertex));
                    RamTable.this.addEdge(i == 0, edge);
                }
            }
            if (waitTimes > this.vertices.size()) {
                LOG.info("Loading wait times is {}", (Object)waitTimes);
            }
            this.vertices.clear();
        }
    }

    private class EdgeRangeIterator
    implements Iterator<HugeEdge> {
        private final int end;
        private final Directions dir;
        private final int label;
        private final HugeVertex owner;
        private int current;
        private HugeEdge currentEdge;

        public EdgeRangeIterator(int start, int end, Directions dir, int label, long owner) {
            assert (0 < start && start < end);
            this.end = end;
            this.dir = dir;
            this.label = label;
            this.owner = new HugeVertex(RamTable.this.graph, IdGenerator.of(owner), VertexLabel.NONE);
            this.current = start;
            this.currentEdge = null;
        }

        @Override
        public boolean hasNext() {
            if (this.currentEdge != null) {
                return true;
            }
            while (this.current < this.end) {
                this.currentEdge = this.fetch();
                if (this.currentEdge == null) continue;
                return true;
            }
            return false;
        }

        @Override
        public HugeEdge next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            assert (this.currentEdge != null);
            HugeEdge edge = this.currentEdge;
            this.currentEdge = null;
            return edge;
        }

        private HugeEdge fetch() {
            if (this.current >= this.end) {
                return null;
            }
            long value = RamTable.this.edges.get(this.current++);
            long otherV = value >>> 32;
            assert (otherV >= 0L) : otherV;
            Directions actualDir = (value & 0x80000000L) == 0L ? Directions.OUT : Directions.IN;
            int label = (int)value & Integer.MAX_VALUE;
            assert (label >= 0);
            if (this.dir != actualDir && this.dir != Directions.BOTH) {
                return null;
            }
            if (this.label != label && this.label != 0) {
                return null;
            }
            HugeGraph graph = RamTable.this.graph;
            this.owner.correctVertexLabel(VertexLabel.NONE);
            boolean direction = actualDir == Directions.OUT;
            Id labelId = IdGenerator.of(label);
            Id otherVertexId = IdGenerator.of(otherV);
            String sortValues = "";
            EdgeLabel edgeLabel = graph.edgeLabel(labelId);
            HugeEdge edge = HugeEdge.constructEdge(this.owner, direction, edgeLabel, sortValues, otherVertexId);
            edge.propNotLoaded();
            return edge;
        }
    }
}

