/* 
 * Unix privileges layer for Cygwin
 * Copyright (C) 2014 Daniel Boland
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* #include <stdio.h> */
#include <sys/errno.h>
#include <string.h>
#include <pwd.h>
#include <sys/syslog.h>
#include <sys/stat.h>
#include <grp.h>
#include <unistd.h>

#define FACILITY LOG_MAIL	// man syslog

static uid_t root_uid = 18;
static gid_t root_gid = 544;

static uid_t my_uid = -1;
static gid_t my_gid = -1;

struct passwd my_pwd;
struct group my_grp;

char str255[255];

int SU_LOG_LEVEL = 0;

/****************************************************/

uid_t uid_get(uid_t uid){
	if (uid == root_uid){
		return 0;
	}else{
		return uid;
	}
}
uid_t su_getuid(void){
	return uid_get(getuid());
}
uid_t su_geteuid(void){
	return uid_get(geteuid());
}
gid_t gid_get(gid_t gid){
	if (gid == root_gid){
		return 0;
	}else{
		return gid;
	}
}
gid_t su_getgid(void){
	return gid_get(getgid());
}
gid_t su_getegid(void){
	return gid_get(getegid());
}

/****************************************************/

int uid_set(uid_t uid){
	int result = -1;
	if (su_getuid() == uid){
		result = 0;
	}else if (!su_getuid() || !su_geteuid()){
		result = 0;
	}else{
		errno = EPERM;
	}
	if (!uid){
		my_uid = root_uid;
	}else{
		my_uid = uid;
	}
	return result;
}
int su_setuid(uid_t uid){
	if (uid_set(uid)){
		return -1;
	}else{
		return setuid(my_uid);
	}
}
int su_seteuid(uid_t uid){
	if (uid_set(uid)){
		return -1;
	}else{
		return seteuid(my_uid);
	}
}
int su_setreuid(uid_t ruid, uid_t euid){
	if (uid_set(ruid)){
		return -1;
	}else if (!euid){
		return setreuid(my_uid, root_uid);
	}else{
		return setreuid(my_uid, euid);
	}
}
int su_setfsuid(uid_t fsuid){
	return su_seteuid(fsuid);
}

/****************************************************/

int gid_set(gid_t gid){
	int result = -1;
	if (su_getgid() == gid){
		result = 0;
	}else if (!su_getuid() || !su_geteuid()){
		result = 0;
	}else{
		errno = EPERM;
	}
	if (!gid){
		my_gid = root_gid;
	}else{
		my_gid = gid;
	}
	return result;
}
int su_setgid(gid_t gid){
	if (gid_set(gid)){
		return -1;
	}else{
		return setgid(my_gid);
	}
}
int su_setegid(gid_t gid){
	if (gid_set(gid)){
		return -1;
	}else{
		return setegid(my_gid);
	}
}
int su_setregid(gid_t rgid, gid_t egid){
	if (gid_set(rgid)){
		return -1;
	}else if (!egid){
		return setregid(my_gid, root_gid);
	}else{
		return setregid(my_gid, egid);
	}
}
int su_setfsgid(uid_t fsgid){
	return su_setegid(fsgid);
}

/****************************************************/

int su_getgroups(int size, gid_t list[]){
	int result = getgroups(size, list);
	while (size--){
		if (list[size] == root_gid){
			list[size] = 0;
		}
	}
	return result;
}
int su_setgroups(int size, const gid_t *list){
	size_t s = size;
	gid_t l[s];
	memcpy(l, list, sizeof(gid_t) * s);
	while (s--){
		if (!l[s]){
			l[s] = root_gid;
		}
	}
	return setgroups(size, l);
}

/****************************************************/

int su_getpwnam_r(const char *name, struct passwd *pwd, char *buf, size_t buflen, struct passwd **result){
	int r = getpwnam_r(name, pwd, buf, buflen, result);
	if (pwd->pw_uid == root_uid){
		pwd->pw_uid = 0;
	}
	if (pwd->pw_gid == root_gid){
		pwd->pw_gid = 0;
	}
	return r;
}
int su_getpwuid_r(uid_t uid, struct passwd *pwd, char *buf, size_t buflen, struct passwd **result){
	if (!uid) uid = root_uid;
	int r = getpwuid_r(uid, pwd, buf, buflen, result);
	if (pwd->pw_uid == root_uid){
		pwd->pw_uid = 0;
	}
	if (pwd->pw_gid == root_gid){
		pwd->pw_gid = 0;
	}
	return r;
}
int su_getgrnam_r(const char *name, struct group *grp, char *buf, size_t buflen, struct group **result){
	int r = getgrnam_r(name, grp, buf, buflen, result);
	if (grp->gr_gid == root_gid){
		grp->gr_gid = 0;
	}
	return r;
}
int su_getgrgid_r(gid_t gid, struct group *grp, char *buf, size_t buflen, struct group **result){
	if (!gid) gid = root_gid;
	int r = getgrgid_r(gid, grp, buf, buflen, result);
	if (grp->gr_gid == root_gid){
		grp->gr_gid = 0;
	}
	return r;
}

/****************************************************/

struct passwd *su_getpwnam(const char *name){
	struct passwd *result = NULL;
	memset(str255, 0, 255);
	su_getpwnam_r(name, &my_pwd, str255, 255, &result);
	return result;
}
struct passwd *su_getpwuid(uid_t uid){
	struct passwd *result = NULL;
	memset(str255, 0, 255);
	su_getpwuid_r(uid, &my_pwd, str255, 255, &result);
	return result;
}
struct group *su_getgrnam(const char *name){
	struct group *result = NULL;
	memset(str255, 0, 255);
	su_getgrnam_r(name, &my_grp, str255, 255, &result);
	return result;
}
struct group *su_getgrgid(gid_t gid){
	struct group *result = NULL;
	memset(str255, 0, 255);
	su_getgrgid_r(gid, &my_grp, str255, 255, &result);
	return result;
}

/****************************************************/

int su_stat(const char *path, struct stat *buf){
	int result = stat(path, buf);
	if (!result){
		if (buf->st_uid == root_uid){
			buf->st_uid = 0;
		}
		if (buf->st_gid == root_gid){
			buf->st_gid = 0;
		}
	}
	return result;
}
int su_fstat(int fd, struct stat *buf){
	int result = fstat(fd, buf);
	if (!result){
		if (buf->st_uid == root_uid){
			buf->st_uid = 0;
		}
		if (buf->st_gid == root_gid){
			buf->st_gid = 0;
		}
	}
	return result;
}
int su_lstat(const char *path, struct stat *buf){
	int result = lstat(path, buf);
	if (!result){
		if (buf->st_uid == root_uid){
			buf->st_uid = 0;
		}
		if (buf->st_gid == root_gid){
			buf->st_gid = 0;
		}
	}
	return result;
}

/****************************************************/

int su_chown(const char *path, uid_t owner, gid_t group){
	if (!owner) owner = root_uid;
	if (!group) group = root_gid;
	return chown(path, owner, group);
}
int su_fchown(int fd, uid_t owner, gid_t group){
	if (!owner) owner = root_uid;
	if (!group) group = root_gid;
	return fchown(fd, owner, group);
}
int su_lchown(const char *path, uid_t owner, gid_t group){
	if (!owner) owner = root_uid;
	if (!group) group = root_gid;
	return lchown(path, owner, group);
}

/****************************************************/

int is_admin(void){
	int size = getgroups(0, NULL);
	uid_t list[size];
	getgroups(size, list);
	while (size--){
		if (list[size] == root_gid){
			return 1;
		}
	}
	return 0;
}
void fakesu(const char *name){
	struct stat info;
	if (is_admin()){
		root_uid = getuid();
		root_gid = getgid();
	}
	if (!strstr(name, "/")){
		readlink("/proc/self/exe", str255, 255);
		name = str255;
	}
	if (stat(name, &info)){
		syslog(FACILITY | LOG_ERR, "cygfakesu.c: stat(%s) failed: %s", name, strerror(errno));
	}else{
		if ((info.st_mode & 04000) == 04000){
			if (seteuid(info.st_uid) && SU_LOG_LEVEL){
				syslog(FACILITY | LOG_WARNING, "%s: cygfakesu.c: seteuid(%d) failed: %s", name, info.st_uid, strerror(errno));
			}
		}
		if ((info.st_mode & 02000) == 02000){
			if (setegid(info.st_gid) && SU_LOG_LEVEL){
				syslog(FACILITY | LOG_WARNING, "%s: cygfakesu.c: setegid(%d) failed: %s", name, info.st_gid, strerror(errno));
			}
		}
	}
	if (SU_LOG_LEVEL){
		syslog(FACILITY | LOG_INFO, "%s: cygfakesu.c: Started with r/euid(%d/%d) r/egid(%d/%d)", name, su_getuid(), su_geteuid(), su_getgid(), su_getegid());
	}
}
