/*
 * Copyright (c) Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "error_map.h"
#include "ras_decode_constants.h"
#include <string.h>

#define AFID_VERSION "0.7"

static const error_map_entry_t error_map[] = {
    {1, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_FW_LOAD, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {2, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_HBM_BIST_TEST, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {3, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_HBM_MEMORY_TEST, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {4, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_HBM_TRAINING, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {5, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_UNHANDLED, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {6, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_UNKNOWN_ERROR, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {7, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_USR_CP_LINK_TRAINING, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {8, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_USR_DP_LINK_TRAINING, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {9, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_WAFL_LINK_TRAINING, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {10, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_XGMI_LINK_TRAINING, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {11, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_BOOT_CONTROLLER_DATA_ABORT, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {12, RAS_DECODE_CATEGORY_BOOT_TIME_ERRORS, RAS_DECODE_ERROR_TYPE_BOOT_CONTROLLER_GENERIC, RAS_DECODE_PROTOCOL_CPER_WITH_SPACE, RAS_DECODE_SEVERITY_FAIL_TO_INIT},
    {13, RAS_DECODE_CATEGORY_OFF_PACKAGE_LINK_ERRORS, RAS_DECODE_ERROR_TYPE_PCIE_AER, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_CORRECTED},
    {14, RAS_DECODE_CATEGORY_OFF_PACKAGE_LINK_ERRORS, RAS_DECODE_ERROR_TYPE_PCIE_AER, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {15, RAS_DECODE_CATEGORY_OFF_PACKAGE_LINK_ERRORS, RAS_DECODE_ERROR_TYPE_WAFL, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_CORRECTED},
    {16, RAS_DECODE_CATEGORY_OFF_PACKAGE_LINK_ERRORS, RAS_DECODE_ERROR_TYPE_WAFL, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {17, RAS_DECODE_CATEGORY_OFF_PACKAGE_LINK_ERRORS, RAS_DECODE_ERROR_TYPE_XGMI, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_CORRECTED},
    {18, RAS_DECODE_CATEGORY_OFF_PACKAGE_LINK_ERRORS, RAS_DECODE_ERROR_TYPE_XGMI, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {19, RAS_DECODE_CATEGORY_HBM_ERRORS, RAS_DECODE_ERROR_TYPE_BAD_PAGE_RETIREMENT_THRESHOLD, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {20, RAS_DECODE_CATEGORY_HBM_ERRORS, RAS_DECODE_ERROR_TYPE_ON_DIE_ECC, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {21, RAS_DECODE_CATEGORY_HBM_ERRORS, RAS_DECODE_ERROR_TYPE_END_TO_END_CRC, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {22, RAS_DECODE_CATEGORY_HBM_ERRORS, RAS_DECODE_ERROR_TYPE_ON_DIE_ECC, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_UNCORRECTED_NON_FATAL},
    {23, RAS_DECODE_CATEGORY_HBM_ERRORS, RAS_DECODE_ERROR_TYPE_END_TO_END_CRC, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_UNCORRECTED_NON_FATAL},
    {24, RAS_DECODE_CATEGORY_HBM_ERRORS, RAS_DECODE_ERROR_TYPE_ALL, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_CORRECTED},
    {25, RAS_DECODE_CATEGORY_HBM_ERRORS, RAS_DECODE_ERROR_TYPE_ALL_OTHERS, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {26, RAS_DECODE_CATEGORY_DEVICE_INTERNAL_ERRORS, RAS_DECODE_ERROR_TYPE_HARDWARE_ASSERTION, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {27, RAS_DECODE_CATEGORY_DEVICE_INTERNAL_ERRORS, RAS_DECODE_ERROR_TYPE_WATCHDOG_TIMEOUT, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {28, RAS_DECODE_CATEGORY_DEVICE_INTERNAL_ERRORS, RAS_DECODE_ERROR_TYPE_ALL_OTHERS, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_UNCORRECTED_NON_FATAL},
    {29, RAS_DECODE_CATEGORY_DEVICE_INTERNAL_ERRORS, RAS_DECODE_ERROR_TYPE_ALL_OTHERS, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_CORRECTED},
    {30, RAS_DECODE_CATEGORY_DEVICE_INTERNAL_ERRORS, RAS_DECODE_ERROR_TYPE_ALL_OTHERS, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_FATAL},
    {31, RAS_DECODE_CATEGORY_CPER_FORMAT, RAS_DECODE_ERROR_TYPE_MALFORMED_CPER, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_ALL_CAPS},
    {32, RAS_DECODE_CATEGORY_CPER_FORMAT, RAS_DECODE_ERROR_TYPE_INCOMPLETE_RAS_DECODE_DATA, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_ALL_CAPS},
    {33, RAS_DECODE_CATEGORY_CPER_FORMAT, RAS_DECODE_ERROR_TYPE_INVALID_RAS_DECODE_DATA, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_ALL_CAPS},
    {34, RAS_DECODE_CATEGORY_UNIDENTIFIED_ERRORS, RAS_DECODE_ERROR_TYPE_UNIDENTIFIED_ERROR, RAS_DECODE_PROTOCOL_CPER, RAS_DECODE_SEVERITY_ALL_CAPS}};

static const size_t NUM_ERROR_ENTRIES = sizeof(error_map) / sizeof(error_map[0]);

int get_error_id(const char *error_category, const char *error_type, const char *error_severity)
{
    if (!error_category || !error_type || !error_severity ||
        strcmp(error_category, RAS_DECODE_SEVERITY_UNKNOWN) == 0 ||
        strcmp(error_type, RAS_DECODE_SEVERITY_UNKNOWN) == 0 ||
        strcmp(error_severity, RAS_DECODE_SEVERITY_UNKNOWN) == 0)
    {
        return RAS_DECODE_ERROR_INVALID_RAS_DECODE_DATA_ID; // Return ID for "Invalid Error" if any input is "UNKNOWN" or NULL
    }

    for (size_t i = 0; i < NUM_ERROR_ENTRIES; i++)
    {
        if (strcmp(error_map[i].error_category, error_category) == 0 &&
            strcmp(error_map[i].error_type, error_type) == 0 &&
            strcmp(error_map[i].error_severity, error_severity) == 0)
        {
            return (int)error_map[i].id;
        }
    }

    return RAS_DECODE_ERROR_UNIDENTIFIED_ERROR_ID; // Return ID for "Unidentified Errors" if no match found
}
