/*
 * Copyright (C) 2022, KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Authors: Nicole <buxiaoqing@kylinos.cn>
 *
 */

#include "kbadge.h"
#include <QPainter>
#include <QRect>
#include <QFont>
#include <QFontMetrics>
#include <QApplication>
#include <QDebug>

class KBadgePrivate:public QObject
{
    Q_OBJECT
    Q_DECLARE_PUBLIC(KBadge)

public:
    KBadgePrivate(KBadge* parent);

private:
    KBadge* q_ptr;
    QColor m_color;
    int m_value;
    int m_fontSize;
    bool m_isShowValue;
    QRect m_rect;
    QSize m_size;
};

KBadgePrivate::KBadgePrivate(KBadge *parent)
    :q_ptr(parent)
{
    Q_Q(KBadge);
    m_value = -1;
    m_color = q->palette().color(QPalette::Highlight);
    m_fontSize = 8;
    m_isShowValue = true;
    m_rect = QRect(0,0,40,30);
}

KBadge::KBadge(QWidget *parent)
    :QToolButton(parent),
      d_ptr(new KBadgePrivate(this))

{
    Q_D(KBadge);

}

QRect KBadge::geometry()
{
    Q_D(KBadge);
    return d->m_rect;
}

void KBadge::setGeometry(QRect rect)
{
    Q_D(KBadge);
    d->m_rect = rect;
}

void KBadge::setValue(int value)
{
    Q_D(KBadge);
    d->m_value = value;
}

void KBadge::setValueVisiable(bool flag)
{
    Q_D(KBadge);
    d->m_isShowValue = flag;
}

bool KBadge::isValueVisiable() const
{
    Q_D(const KBadge);
    return d->m_isShowValue;
}

int KBadge::value()
{
    Q_D(KBadge);
    return d->m_value;
}

QColor KBadge::color()
{
    Q_D(KBadge);
    return d->m_color;
}

void KBadge::setColor(const QColor &color)
{
    Q_D(KBadge);
    d->m_color = color;
}

int KBadge::fontSize()
{
    Q_D(KBadge);
    return d->m_fontSize;
}

void KBadge::setFontSize(int size)
{
    Q_D(KBadge);
    if(size<1 ||size >100)
        return;
    d->m_fontSize = size;
    updateSize();
}

QSize KBadge::updateSize()
{
    Q_D(KBadge);
    QFont font(QApplication::font());
    font.setPixelSize(d->m_fontSize);
    QFontMetrics fm(font);
    int height = fm.height();
    if (height < 14 ) {
        height = 14;
    }
    int width;
    if(d->m_value <1 ||!d->m_isShowValue)
    {
        this->setVisible(false);
    }
    else if(d->m_value >= 1 && d->m_value < 100)
    {
        width = fm.width(QString::number(d->m_value)) + 10;
        width = width > height ? width:height;
    }
    else
    {
        width = fm.width(QString::number(999)) + 10;
        width = width > height ? width:height;
    }
    this->setFixedSize(QSize(width,height));
    //qDebug()<<"=======update kbadge size"<<width<<height;

    return QSize(width, height);

}
void KBadge::paintEvent(QPaintEvent *event)
{
    Q_D(KBadge);
    QSize size = updateSize();
    int height = size.height();
    int width = size.width();

    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing);
    painter.setPen(Qt::NoPen);
    painter.setBrush(d->m_color);
    QRect tmpRect(0, 0, width, height);
    painter.drawRoundedRect(tmpRect, height/2, height/2);
    //文字颜色固定
    painter.setPen(QToolButton::palette().color(QPalette::Light));
    if(d->m_value >= 1 && d->m_value<100 && d->m_isShowValue)
    {
        QFont font(QApplication::font());
        font.setPixelSize(d->m_fontSize);
        if (font.family() != QString("Noto Sans CJK SC")) {
            font.setBold(true);
        }
        painter.setFont(font);
        painter.drawText(tmpRect,Qt::AlignCenter,QString::number(d->m_value));
    }

    if(d->m_value >= 100 && d->m_value < INTMAX_MAX && d->m_isShowValue)
    {
        QFont font(QApplication::font());
        font.setPixelSize(d->m_fontSize);
        if (font.family() != QString("Noto Sans CJK SC")) {
            font.setBold(true);
        }
        painter.setFont(font);
        painter.drawText(tmpRect,Qt::AlignCenter,QString("99+"));
    }
}

void KBadge::resizeEvent(QResizeEvent *event)
{
    QToolButton::resizeEvent(event);
    repaint();
}

#include "kbadge.moc"
