// Copyright 2012-2015 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

// NOTE: The following code was generated by "scripts/unicode.py", do not edit directly

#![allow(missing_docs, non_upper_case_globals, non_snake_case)]

/// The version of [Unicode](http://www.unicode.org/)
/// that this version of unicode-security is based on.
pub const UNICODE_VERSION: (u64, u64, u64) = (12, 1, 0);


pub mod util {
    use core::result::Result::{Ok, Err};
    #[inline]
    pub fn bsearch_range_table(c: char, r: &'static [(char,char)]) -> bool {
        use core::cmp::Ordering::{Equal, Less, Greater};
        r.binary_search_by(|&(lo,hi)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }).is_ok()
    }
    
    pub fn bsearch_range_value_table<T: Copy>(c: char, r: &'static [(char, char, T)]) -> Option<T> {
        use core::cmp::Ordering::{Equal, Less, Greater};
        match r.binary_search_by(|&(lo, hi, _)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }) {
            Ok(idx) => {
                let (_, _, cat) = r[idx];
                Some(cat)
            }
            Err(_) => None
        }
    }

}

pub mod identifier {

    #[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd, Debug)]
    #[allow(non_camel_case_types)]
    /// https://www.unicode.org/reports/tr39/#Identifier_Status_and_Type
    pub enum IdentifierType {
        // Restricted
        Not_Character,
        Deprecated,
        Default_Ignorable,
        Not_NFKC,
        Not_XID,
        Exclusion,
        Obsolete,
        Technical,
        Uncommon_Use,
        Limited_Use,

        // Allowed
        Inclusion,
        Recommended
    }
    #[inline]
    pub fn identifier_status_allowed(c: char) -> bool {
        // FIXME: do we want to special case ASCII here?
        match c as usize {
            _ => super::util::bsearch_range_table(c, IDENTIFIER_STATUS)
        }
    }

    #[inline]
    pub fn identifier_type(c: char) -> Option<IdentifierType> {
        // FIXME: do we want to special case ASCII here?
        match c as usize {
            _ => super::util::bsearch_range_value_table(c, IDENTIFIER_TYPE)
        }
    }
    // Identifier status table:
    const IDENTIFIER_STATUS: &'static [(char, char)] = &[
        ('\u{27}', '\u{27}'), ('\u{2d}', '\u{2e}'), ('\u{30}', '\u{3a}'), ('\u{41}', '\u{5a}'),
        ('\u{5f}', '\u{5f}'), ('\u{61}', '\u{7a}'), ('\u{b7}', '\u{b7}'), ('\u{c0}', '\u{d6}'),
        ('\u{d8}', '\u{f6}'), ('\u{f8}', '\u{131}'), ('\u{134}', '\u{13e}'), ('\u{141}', '\u{148}'),
        ('\u{14a}', '\u{17e}'), ('\u{18f}', '\u{18f}'), ('\u{1a0}', '\u{1a1}'), ('\u{1af}',
        '\u{1b0}'), ('\u{1cd}', '\u{1dc}'), ('\u{1de}', '\u{1e3}'), ('\u{1e6}', '\u{1f0}'),
        ('\u{1f4}', '\u{1f5}'), ('\u{1f8}', '\u{1f9}'), ('\u{1fa}', '\u{217}'), ('\u{218}',
        '\u{21b}'), ('\u{21e}', '\u{21f}'), ('\u{226}', '\u{233}'), ('\u{259}', '\u{259}'),
        ('\u{2bb}', '\u{2bc}'), ('\u{2ec}', '\u{2ec}'), ('\u{300}', '\u{304}'), ('\u{306}',
        '\u{30c}'), ('\u{30f}', '\u{311}'), ('\u{313}', '\u{314}'), ('\u{31b}', '\u{31b}'),
        ('\u{323}', '\u{328}'), ('\u{32d}', '\u{32e}'), ('\u{330}', '\u{331}'), ('\u{335}',
        '\u{335}'), ('\u{338}', '\u{339}'), ('\u{342}', '\u{342}'), ('\u{345}', '\u{345}'),
        ('\u{375}', '\u{375}'), ('\u{37b}', '\u{37d}'), ('\u{386}', '\u{386}'), ('\u{388}',
        '\u{38a}'), ('\u{38c}', '\u{38c}'), ('\u{38e}', '\u{3a1}'), ('\u{3a3}', '\u{3ce}'),
        ('\u{3fc}', '\u{3ff}'), ('\u{400}', '\u{400}'), ('\u{401}', '\u{40c}'), ('\u{40d}',
        '\u{40d}'), ('\u{40e}', '\u{44f}'), ('\u{450}', '\u{450}'), ('\u{451}', '\u{45c}'),
        ('\u{45d}', '\u{45d}'), ('\u{45e}', '\u{45f}'), ('\u{48a}', '\u{48b}'), ('\u{48c}',
        '\u{48f}'), ('\u{490}', '\u{4c4}'), ('\u{4c5}', '\u{4c6}'), ('\u{4c7}', '\u{4c8}'),
        ('\u{4c9}', '\u{4ca}'), ('\u{4cb}', '\u{4cc}'), ('\u{4cd}', '\u{4ce}'), ('\u{4cf}',
        '\u{4cf}'), ('\u{4d0}', '\u{4eb}'), ('\u{4ec}', '\u{4ed}'), ('\u{4ee}', '\u{4f5}'),
        ('\u{4f6}', '\u{4f7}'), ('\u{4f8}', '\u{4f9}'), ('\u{4fa}', '\u{4ff}'), ('\u{500}',
        '\u{50f}'), ('\u{510}', '\u{513}'), ('\u{514}', '\u{523}'), ('\u{524}', '\u{525}'),
        ('\u{526}', '\u{527}'), ('\u{528}', '\u{529}'), ('\u{52e}', '\u{52f}'), ('\u{531}',
        '\u{556}'), ('\u{559}', '\u{559}'), ('\u{560}', '\u{560}'), ('\u{561}', '\u{586}'),
        ('\u{588}', '\u{588}'), ('\u{58a}', '\u{58a}'), ('\u{5b4}', '\u{5b4}'), ('\u{5d0}',
        '\u{5ea}'), ('\u{5ef}', '\u{5ef}'), ('\u{5f0}', '\u{5f4}'), ('\u{620}', '\u{620}'),
        ('\u{621}', '\u{63a}'), ('\u{63b}', '\u{63f}'), ('\u{641}', '\u{652}'), ('\u{653}',
        '\u{655}'), ('\u{660}', '\u{669}'), ('\u{670}', '\u{672}'), ('\u{674}', '\u{674}'),
        ('\u{679}', '\u{68d}'), ('\u{68f}', '\u{6b7}'), ('\u{6b8}', '\u{6b9}'), ('\u{6ba}',
        '\u{6be}'), ('\u{6bf}', '\u{6bf}'), ('\u{6c0}', '\u{6ce}'), ('\u{6cf}', '\u{6cf}'),
        ('\u{6d0}', '\u{6d3}'), ('\u{6d5}', '\u{6d5}'), ('\u{6e5}', '\u{6e6}'), ('\u{6ee}',
        '\u{6ef}'), ('\u{6f0}', '\u{6f9}'), ('\u{6fa}', '\u{6fe}'), ('\u{6ff}', '\u{6ff}'),
        ('\u{750}', '\u{76d}'), ('\u{76e}', '\u{77f}'), ('\u{780}', '\u{7b0}'), ('\u{7b1}',
        '\u{7b1}'), ('\u{8a0}', '\u{8a0}'), ('\u{8a1}', '\u{8a1}'), ('\u{8a2}', '\u{8ac}'),
        ('\u{8b2}', '\u{8b2}'), ('\u{8b6}', '\u{8bd}'), ('\u{901}', '\u{903}'), ('\u{904}',
        '\u{904}'), ('\u{905}', '\u{939}'), ('\u{93a}', '\u{93b}'), ('\u{93c}', '\u{94d}'),
        ('\u{94f}', '\u{94f}'), ('\u{950}', '\u{950}'), ('\u{956}', '\u{957}'), ('\u{960}',
        '\u{963}'), ('\u{966}', '\u{96f}'), ('\u{971}', '\u{972}'), ('\u{973}', '\u{977}'),
        ('\u{979}', '\u{97a}'), ('\u{97b}', '\u{97c}'), ('\u{97d}', '\u{97d}'), ('\u{97e}',
        '\u{97f}'), ('\u{981}', '\u{983}'), ('\u{985}', '\u{98c}'), ('\u{98f}', '\u{990}'),
        ('\u{993}', '\u{9a8}'), ('\u{9aa}', '\u{9b0}'), ('\u{9b2}', '\u{9b2}'), ('\u{9b6}',
        '\u{9b9}'), ('\u{9bc}', '\u{9bc}'), ('\u{9bd}', '\u{9bd}'), ('\u{9be}', '\u{9c4}'),
        ('\u{9c7}', '\u{9c8}'), ('\u{9cb}', '\u{9cd}'), ('\u{9ce}', '\u{9ce}'), ('\u{9d7}',
        '\u{9d7}'), ('\u{9e0}', '\u{9e3}'), ('\u{9e6}', '\u{9f1}'), ('\u{9fc}', '\u{9fc}'),
        ('\u{9fe}', '\u{9fe}'), ('\u{a01}', '\u{a01}'), ('\u{a02}', '\u{a02}'), ('\u{a03}',
        '\u{a03}'), ('\u{a05}', '\u{a0a}'), ('\u{a0f}', '\u{a10}'), ('\u{a13}', '\u{a28}'),
        ('\u{a2a}', '\u{a30}'), ('\u{a32}', '\u{a32}'), ('\u{a35}', '\u{a35}'), ('\u{a38}',
        '\u{a39}'), ('\u{a3c}', '\u{a3c}'), ('\u{a3e}', '\u{a42}'), ('\u{a47}', '\u{a48}'),
        ('\u{a4b}', '\u{a4d}'), ('\u{a5c}', '\u{a5c}'), ('\u{a66}', '\u{a74}'), ('\u{a81}',
        '\u{a83}'), ('\u{a85}', '\u{a8b}'), ('\u{a8c}', '\u{a8c}'), ('\u{a8d}', '\u{a8d}'),
        ('\u{a8f}', '\u{a91}'), ('\u{a93}', '\u{aa8}'), ('\u{aaa}', '\u{ab0}'), ('\u{ab2}',
        '\u{ab3}'), ('\u{ab5}', '\u{ab9}'), ('\u{abc}', '\u{ac5}'), ('\u{ac7}', '\u{ac9}'),
        ('\u{acb}', '\u{acd}'), ('\u{ad0}', '\u{ad0}'), ('\u{ae0}', '\u{ae0}'), ('\u{ae1}',
        '\u{ae3}'), ('\u{ae6}', '\u{aef}'), ('\u{afa}', '\u{aff}'), ('\u{b01}', '\u{b03}'),
        ('\u{b05}', '\u{b0c}'), ('\u{b0f}', '\u{b10}'), ('\u{b13}', '\u{b28}'), ('\u{b2a}',
        '\u{b30}'), ('\u{b32}', '\u{b33}'), ('\u{b35}', '\u{b35}'), ('\u{b36}', '\u{b39}'),
        ('\u{b3c}', '\u{b43}'), ('\u{b47}', '\u{b48}'), ('\u{b4b}', '\u{b4d}'), ('\u{b56}',
        '\u{b57}'), ('\u{b5f}', '\u{b61}'), ('\u{b66}', '\u{b6f}'), ('\u{b71}', '\u{b71}'),
        ('\u{b82}', '\u{b83}'), ('\u{b85}', '\u{b8a}'), ('\u{b8e}', '\u{b90}'), ('\u{b92}',
        '\u{b95}'), ('\u{b99}', '\u{b9a}'), ('\u{b9c}', '\u{b9c}'), ('\u{b9e}', '\u{b9f}'),
        ('\u{ba3}', '\u{ba4}'), ('\u{ba8}', '\u{baa}'), ('\u{bae}', '\u{bb5}'), ('\u{bb6}',
        '\u{bb6}'), ('\u{bb7}', '\u{bb9}'), ('\u{bbe}', '\u{bc2}'), ('\u{bc6}', '\u{bc8}'),
        ('\u{bca}', '\u{bcd}'), ('\u{bd0}', '\u{bd0}'), ('\u{bd7}', '\u{bd7}'), ('\u{be6}',
        '\u{be6}'), ('\u{be7}', '\u{bef}'), ('\u{c01}', '\u{c03}'), ('\u{c04}', '\u{c04}'),
        ('\u{c05}', '\u{c0c}'), ('\u{c0e}', '\u{c10}'), ('\u{c12}', '\u{c28}'), ('\u{c2a}',
        '\u{c33}'), ('\u{c35}', '\u{c39}'), ('\u{c3d}', '\u{c3d}'), ('\u{c3e}', '\u{c44}'),
        ('\u{c46}', '\u{c48}'), ('\u{c4a}', '\u{c4d}'), ('\u{c55}', '\u{c56}'), ('\u{c60}',
        '\u{c61}'), ('\u{c66}', '\u{c6f}'), ('\u{c80}', '\u{c80}'), ('\u{c82}', '\u{c83}'),
        ('\u{c85}', '\u{c8c}'), ('\u{c8e}', '\u{c90}'), ('\u{c92}', '\u{ca8}'), ('\u{caa}',
        '\u{cb3}'), ('\u{cb5}', '\u{cb9}'), ('\u{cbc}', '\u{cbd}'), ('\u{cbe}', '\u{cc4}'),
        ('\u{cc6}', '\u{cc8}'), ('\u{cca}', '\u{ccd}'), ('\u{cd5}', '\u{cd6}'), ('\u{ce0}',
        '\u{ce1}'), ('\u{ce2}', '\u{ce3}'), ('\u{ce6}', '\u{cef}'), ('\u{cf1}', '\u{cf2}'),
        ('\u{d00}', '\u{d00}'), ('\u{d02}', '\u{d03}'), ('\u{d05}', '\u{d0c}'), ('\u{d0e}',
        '\u{d10}'), ('\u{d12}', '\u{d28}'), ('\u{d29}', '\u{d29}'), ('\u{d2a}', '\u{d39}'),
        ('\u{d3a}', '\u{d3a}'), ('\u{d3b}', '\u{d3c}'), ('\u{d3d}', '\u{d3d}'), ('\u{d3e}',
        '\u{d43}'), ('\u{d46}', '\u{d48}'), ('\u{d4a}', '\u{d4d}'), ('\u{d4e}', '\u{d4e}'),
        ('\u{d54}', '\u{d56}'), ('\u{d57}', '\u{d57}'), ('\u{d60}', '\u{d61}'), ('\u{d66}',
        '\u{d6f}'), ('\u{d7a}', '\u{d7f}'), ('\u{d82}', '\u{d83}'), ('\u{d85}', '\u{d8e}'),
        ('\u{d91}', '\u{d96}'), ('\u{d9a}', '\u{da5}'), ('\u{da7}', '\u{db1}'), ('\u{db3}',
        '\u{dbb}'), ('\u{dbd}', '\u{dbd}'), ('\u{dc0}', '\u{dc6}'), ('\u{dca}', '\u{dca}'),
        ('\u{dcf}', '\u{dd4}'), ('\u{dd6}', '\u{dd6}'), ('\u{dd8}', '\u{dde}'), ('\u{df2}',
        '\u{df2}'), ('\u{e01}', '\u{e32}'), ('\u{e34}', '\u{e3a}'), ('\u{e40}', '\u{e4e}'),
        ('\u{e50}', '\u{e59}'), ('\u{e81}', '\u{e82}'), ('\u{e84}', '\u{e84}'), ('\u{e86}',
        '\u{e86}'), ('\u{e87}', '\u{e88}'), ('\u{e89}', '\u{e89}'), ('\u{e8a}', '\u{e8a}'),
        ('\u{e8c}', '\u{e8c}'), ('\u{e8d}', '\u{e8d}'), ('\u{e8e}', '\u{e93}'), ('\u{e94}',
        '\u{e97}'), ('\u{e98}', '\u{e98}'), ('\u{e99}', '\u{e9f}'), ('\u{ea0}', '\u{ea0}'),
        ('\u{ea1}', '\u{ea3}'), ('\u{ea5}', '\u{ea5}'), ('\u{ea7}', '\u{ea7}'), ('\u{ea8}',
        '\u{ea9}'), ('\u{eaa}', '\u{eab}'), ('\u{eac}', '\u{eac}'), ('\u{ead}', '\u{eb2}'),
        ('\u{eb4}', '\u{eb9}'), ('\u{eba}', '\u{eba}'), ('\u{ebb}', '\u{ebd}'), ('\u{ec0}',
        '\u{ec4}'), ('\u{ec6}', '\u{ec6}'), ('\u{ec8}', '\u{ecd}'), ('\u{ed0}', '\u{ed9}'),
        ('\u{ede}', '\u{edf}'), ('\u{f00}', '\u{f00}'), ('\u{f0b}', '\u{f0b}'), ('\u{f20}',
        '\u{f29}'), ('\u{f35}', '\u{f35}'), ('\u{f37}', '\u{f37}'), ('\u{f3e}', '\u{f42}'),
        ('\u{f44}', '\u{f47}'), ('\u{f49}', '\u{f4c}'), ('\u{f4e}', '\u{f51}'), ('\u{f53}',
        '\u{f56}'), ('\u{f58}', '\u{f5b}'), ('\u{f5d}', '\u{f68}'), ('\u{f6a}', '\u{f6a}'),
        ('\u{f6b}', '\u{f6c}'), ('\u{f71}', '\u{f72}'), ('\u{f74}', '\u{f74}'), ('\u{f7a}',
        '\u{f80}'), ('\u{f82}', '\u{f84}'), ('\u{f86}', '\u{f8b}'), ('\u{f8c}', '\u{f8f}'),
        ('\u{f90}', '\u{f92}'), ('\u{f94}', '\u{f95}'), ('\u{f96}', '\u{f96}'), ('\u{f97}',
        '\u{f97}'), ('\u{f99}', '\u{f9c}'), ('\u{f9e}', '\u{fa1}'), ('\u{fa3}', '\u{fa6}'),
        ('\u{fa8}', '\u{fab}'), ('\u{fad}', '\u{fad}'), ('\u{fae}', '\u{fb0}'), ('\u{fb1}',
        '\u{fb7}'), ('\u{fb8}', '\u{fb8}'), ('\u{fba}', '\u{fbc}'), ('\u{fc6}', '\u{fc6}'),
        ('\u{1000}', '\u{1021}'), ('\u{1022}', '\u{1022}'), ('\u{1023}', '\u{1027}'), ('\u{1028}',
        '\u{1028}'), ('\u{1029}', '\u{102a}'), ('\u{102b}', '\u{102b}'), ('\u{102c}', '\u{1032}'),
        ('\u{1033}', '\u{1035}'), ('\u{1036}', '\u{1039}'), ('\u{103a}', '\u{103f}'), ('\u{1040}',
        '\u{1049}'), ('\u{1050}', '\u{1059}'), ('\u{105a}', '\u{1099}'), ('\u{109a}', '\u{109d}'),
        ('\u{10c7}', '\u{10c7}'), ('\u{10cd}', '\u{10cd}'), ('\u{10d0}', '\u{10f0}'), ('\u{10f7}',
        '\u{10f8}'), ('\u{10f9}', '\u{10fa}'), ('\u{10fd}', '\u{10ff}'), ('\u{1200}', '\u{1206}'),
        ('\u{1207}', '\u{1207}'), ('\u{1208}', '\u{1246}'), ('\u{1247}', '\u{1247}'), ('\u{1248}',
        '\u{1248}'), ('\u{124a}', '\u{124d}'), ('\u{1250}', '\u{1256}'), ('\u{1258}', '\u{1258}'),
        ('\u{125a}', '\u{125d}'), ('\u{1260}', '\u{1286}'), ('\u{1287}', '\u{1287}'), ('\u{1288}',
        '\u{1288}'), ('\u{128a}', '\u{128d}'), ('\u{1290}', '\u{12ae}'), ('\u{12af}', '\u{12af}'),
        ('\u{12b0}', '\u{12b0}'), ('\u{12b2}', '\u{12b5}'), ('\u{12b8}', '\u{12be}'), ('\u{12c0}',
        '\u{12c0}'), ('\u{12c2}', '\u{12c5}'), ('\u{12c8}', '\u{12ce}'), ('\u{12cf}', '\u{12cf}'),
        ('\u{12d0}', '\u{12d6}'), ('\u{12d8}', '\u{12ee}'), ('\u{12ef}', '\u{12ef}'), ('\u{12f0}',
        '\u{130e}'), ('\u{130f}', '\u{130f}'), ('\u{1310}', '\u{1310}'), ('\u{1312}', '\u{1315}'),
        ('\u{1318}', '\u{131e}'), ('\u{131f}', '\u{131f}'), ('\u{1320}', '\u{1346}'), ('\u{1347}',
        '\u{1347}'), ('\u{1348}', '\u{135a}'), ('\u{135d}', '\u{135e}'), ('\u{135f}', '\u{135f}'),
        ('\u{1380}', '\u{138f}'), ('\u{1780}', '\u{17a2}'), ('\u{17a5}', '\u{17a7}'), ('\u{17a9}',
        '\u{17b3}'), ('\u{17b6}', '\u{17ca}'), ('\u{17d2}', '\u{17d2}'), ('\u{17d7}', '\u{17d7}'),
        ('\u{17dc}', '\u{17dc}'), ('\u{17e0}', '\u{17e9}'), ('\u{1c80}', '\u{1c88}'), ('\u{1c90}',
        '\u{1cba}'), ('\u{1cbd}', '\u{1cbf}'), ('\u{1e00}', '\u{1e99}'), ('\u{1e9e}', '\u{1e9e}'),
        ('\u{1ea0}', '\u{1ef9}'), ('\u{1f00}', '\u{1f15}'), ('\u{1f18}', '\u{1f1d}'), ('\u{1f20}',
        '\u{1f45}'), ('\u{1f48}', '\u{1f4d}'), ('\u{1f50}', '\u{1f57}'), ('\u{1f59}', '\u{1f59}'),
        ('\u{1f5b}', '\u{1f5b}'), ('\u{1f5d}', '\u{1f5d}'), ('\u{1f5f}', '\u{1f70}'), ('\u{1f72}',
        '\u{1f72}'), ('\u{1f74}', '\u{1f74}'), ('\u{1f76}', '\u{1f76}'), ('\u{1f78}', '\u{1f78}'),
        ('\u{1f7a}', '\u{1f7a}'), ('\u{1f7c}', '\u{1f7c}'), ('\u{1f80}', '\u{1fb4}'), ('\u{1fb6}',
        '\u{1fba}'), ('\u{1fbc}', '\u{1fbc}'), ('\u{1fc2}', '\u{1fc4}'), ('\u{1fc6}', '\u{1fc8}'),
        ('\u{1fca}', '\u{1fca}'), ('\u{1fcc}', '\u{1fcc}'), ('\u{1fd0}', '\u{1fd2}'), ('\u{1fd6}',
        '\u{1fda}'), ('\u{1fe0}', '\u{1fe2}'), ('\u{1fe4}', '\u{1fea}'), ('\u{1fec}', '\u{1fec}'),
        ('\u{1ff2}', '\u{1ff4}'), ('\u{1ff6}', '\u{1ff8}'), ('\u{1ffa}', '\u{1ffa}'), ('\u{1ffc}',
        '\u{1ffc}'), ('\u{200c}', '\u{200d}'), ('\u{2010}', '\u{2010}'), ('\u{2019}', '\u{2019}'),
        ('\u{2027}', '\u{2027}'), ('\u{2d27}', '\u{2d27}'), ('\u{2d2d}', '\u{2d2d}'), ('\u{2d80}',
        '\u{2d96}'), ('\u{2da0}', '\u{2da6}'), ('\u{2da8}', '\u{2dae}'), ('\u{2db0}', '\u{2db6}'),
        ('\u{2db8}', '\u{2dbe}'), ('\u{2dc0}', '\u{2dc6}'), ('\u{2dc8}', '\u{2dce}'), ('\u{2dd0}',
        '\u{2dd6}'), ('\u{2dd8}', '\u{2dde}'), ('\u{3005}', '\u{3007}'), ('\u{3041}', '\u{3094}'),
        ('\u{3095}', '\u{3096}'), ('\u{3099}', '\u{309a}'), ('\u{309d}', '\u{309e}'), ('\u{30a0}',
        '\u{30a0}'), ('\u{30a1}', '\u{30fe}'), ('\u{3105}', '\u{312c}'), ('\u{312d}', '\u{312d}'),
        ('\u{312e}', '\u{312e}'), ('\u{312f}', '\u{312f}'), ('\u{31a0}', '\u{31b7}'), ('\u{31b8}',
        '\u{31ba}'), ('\u{3400}', '\u{4db5}'), ('\u{4e00}', '\u{9fa5}'), ('\u{9fa6}', '\u{9fbb}'),
        ('\u{9fbc}', '\u{9fc3}'), ('\u{9fc4}', '\u{9fcb}'), ('\u{9fcc}', '\u{9fcc}'), ('\u{9fcd}',
        '\u{9fd5}'), ('\u{9fd6}', '\u{9fea}'), ('\u{9feb}', '\u{9fef}'), ('\u{a660}', '\u{a661}'),
        ('\u{a674}', '\u{a67b}'), ('\u{a67f}', '\u{a67f}'), ('\u{a69f}', '\u{a69f}'), ('\u{a717}',
        '\u{a71a}'), ('\u{a71b}', '\u{a71f}'), ('\u{a788}', '\u{a788}'), ('\u{a78d}', '\u{a78e}'),
        ('\u{a790}', '\u{a791}'), ('\u{a792}', '\u{a793}'), ('\u{a7a0}', '\u{a7a9}'), ('\u{a7aa}',
        '\u{a7aa}'), ('\u{a7ae}', '\u{a7ae}'), ('\u{a7af}', '\u{a7af}'), ('\u{a7b8}', '\u{a7b9}'),
        ('\u{a7ba}', '\u{a7bf}'), ('\u{a7c2}', '\u{a7c6}'), ('\u{a7fa}', '\u{a7fa}'), ('\u{a9e7}',
        '\u{a9fe}'), ('\u{aa60}', '\u{aa76}'), ('\u{aa7a}', '\u{aa7b}'), ('\u{aa7c}', '\u{aa7f}'),
        ('\u{ab01}', '\u{ab06}'), ('\u{ab09}', '\u{ab0e}'), ('\u{ab11}', '\u{ab16}'), ('\u{ab20}',
        '\u{ab26}'), ('\u{ab28}', '\u{ab2e}'), ('\u{ab66}', '\u{ab67}'), ('\u{ac00}', '\u{d7a3}'),
        ('\u{fa0e}', '\u{fa0f}'), ('\u{fa11}', '\u{fa11}'), ('\u{fa13}', '\u{fa14}'), ('\u{fa1f}',
        '\u{fa1f}'), ('\u{fa21}', '\u{fa21}'), ('\u{fa23}', '\u{fa24}'), ('\u{fa27}', '\u{fa29}'),
        ('\u{1133b}', '\u{1133b}'), ('\u{1b150}', '\u{1b152}'), ('\u{1b164}', '\u{1b167}'),
        ('\u{20000}', '\u{2a6d6}'), ('\u{2a700}', '\u{2b734}'), ('\u{2b740}', '\u{2b81d}'),
        ('\u{2b820}', '\u{2cea1}'), ('\u{2ceb0}', '\u{2ebe0}')
    ];

    const IDENTIFIER_TYPE: &'static [(char, char, IdentifierType)] = &[
        ('\u{9}', '\u{d}',  IdentifierType::Not_XID), ('\u{20}', '\u{26}',
         IdentifierType::Not_XID), ('\u{27}', '\u{27}',  IdentifierType::Inclusion), ('\u{28}',
        '\u{2c}',  IdentifierType::Not_XID), ('\u{2d}', '\u{2e}',  IdentifierType::Inclusion),
        ('\u{2f}', '\u{2f}',  IdentifierType::Not_XID), ('\u{30}', '\u{39}',
         IdentifierType::Recommended), ('\u{3a}', '\u{3a}',  IdentifierType::Inclusion), ('\u{3b}',
        '\u{40}',  IdentifierType::Not_XID), ('\u{41}', '\u{5a}',  IdentifierType::Recommended),
        ('\u{5b}', '\u{5e}',  IdentifierType::Not_XID), ('\u{5f}', '\u{5f}',
         IdentifierType::Recommended), ('\u{60}', '\u{60}',  IdentifierType::Not_XID), ('\u{61}',
        '\u{7a}',  IdentifierType::Recommended), ('\u{7b}', '\u{7e}',  IdentifierType::Not_XID),
        ('\u{85}', '\u{85}',  IdentifierType::Not_XID), ('\u{a0}', '\u{a0}',
         IdentifierType::Not_XID), ('\u{a1}', '\u{a7}',  IdentifierType::Not_XID), ('\u{a8}',
        '\u{a8}',  IdentifierType::Not_XID), ('\u{a9}', '\u{a9}',  IdentifierType::Not_XID),
        ('\u{aa}', '\u{aa}',  IdentifierType::Not_NFKC), ('\u{ab}', '\u{ac}',
         IdentifierType::Not_XID), ('\u{ad}', '\u{ad}',  IdentifierType::Exclusion), ('\u{ae}',
        '\u{ae}',  IdentifierType::Not_XID), ('\u{af}', '\u{af}',  IdentifierType::Not_XID),
        ('\u{b0}', '\u{b1}',  IdentifierType::Not_XID), ('\u{b2}', '\u{b4}',
         IdentifierType::Not_XID), ('\u{b5}', '\u{b5}',  IdentifierType::Not_NFKC), ('\u{b6}',
        '\u{b6}',  IdentifierType::Not_XID), ('\u{b7}', '\u{b7}',  IdentifierType::Inclusion),
        ('\u{b8}', '\u{b9}',  IdentifierType::Not_XID), ('\u{ba}', '\u{ba}',
         IdentifierType::Not_NFKC), ('\u{bb}', '\u{bb}',  IdentifierType::Not_XID), ('\u{bc}',
        '\u{be}',  IdentifierType::Not_XID), ('\u{bf}', '\u{bf}',  IdentifierType::Not_XID),
        ('\u{c0}', '\u{d6}',  IdentifierType::Recommended), ('\u{d7}', '\u{d7}',
         IdentifierType::Not_XID), ('\u{d8}', '\u{f6}',  IdentifierType::Recommended), ('\u{f7}',
        '\u{f7}',  IdentifierType::Not_XID), ('\u{f8}', '\u{131}',  IdentifierType::Recommended),
        ('\u{132}', '\u{133}',  IdentifierType::Not_NFKC), ('\u{134}', '\u{13e}',
         IdentifierType::Recommended), ('\u{13f}', '\u{140}',  IdentifierType::Not_NFKC),
        ('\u{141}', '\u{148}',  IdentifierType::Recommended), ('\u{149}', '\u{149}',
         IdentifierType::Obsolete), ('\u{14a}', '\u{17e}',  IdentifierType::Recommended),
        ('\u{17f}', '\u{17f}',  IdentifierType::Not_NFKC), ('\u{180}', '\u{180}',
         IdentifierType::Technical), ('\u{181}', '\u{18c}',  IdentifierType::Uncommon_Use),
        ('\u{18d}', '\u{18d}',  IdentifierType::Technical), ('\u{18e}', '\u{18e}',
         IdentifierType::Uncommon_Use), ('\u{18f}', '\u{18f}',  IdentifierType::Recommended),
        ('\u{190}', '\u{19f}',  IdentifierType::Uncommon_Use), ('\u{1a0}', '\u{1a1}',
         IdentifierType::Recommended), ('\u{1a2}', '\u{1a9}',  IdentifierType::Uncommon_Use),
        ('\u{1aa}', '\u{1ab}',  IdentifierType::Technical), ('\u{1ac}', '\u{1ae}',
         IdentifierType::Uncommon_Use), ('\u{1af}', '\u{1b0}',  IdentifierType::Recommended),
        ('\u{1b1}', '\u{1b8}',  IdentifierType::Uncommon_Use), ('\u{1b9}', '\u{1b9}',
         IdentifierType::Obsolete), ('\u{1ba}', '\u{1bb}',  IdentifierType::Technical), ('\u{1bc}',
        '\u{1bd}',  IdentifierType::Uncommon_Use), ('\u{1be}', '\u{1be}',
         IdentifierType::Technical), ('\u{1bf}', '\u{1bf}',  IdentifierType::Obsolete), ('\u{1c0}',
        '\u{1c3}',  IdentifierType::Technical), ('\u{1c4}', '\u{1cc}',  IdentifierType::Not_NFKC),
        ('\u{1cd}', '\u{1dc}',  IdentifierType::Recommended), ('\u{1dd}', '\u{1dd}',
         IdentifierType::Uncommon_Use), ('\u{1de}', '\u{1e3}',  IdentifierType::Recommended),
        ('\u{1e4}', '\u{1e5}',  IdentifierType::Uncommon_Use), ('\u{1e6}', '\u{1f0}',
         IdentifierType::Recommended), ('\u{1f1}', '\u{1f3}',  IdentifierType::Not_NFKC),
        ('\u{1f4}', '\u{1f5}',  IdentifierType::Recommended), ('\u{1f6}', '\u{1f7}',
         IdentifierType::Obsolete), ('\u{1f8}', '\u{1f9}',  IdentifierType::Recommended),
        ('\u{1fa}', '\u{217}',  IdentifierType::Recommended), ('\u{218}', '\u{21b}',
         IdentifierType::Recommended), ('\u{21c}', '\u{21d}',  IdentifierType::Obsolete),
        ('\u{21e}', '\u{21f}',  IdentifierType::Recommended), ('\u{220}', '\u{220}',
         IdentifierType::Uncommon_Use), ('\u{221}', '\u{221}',  IdentifierType::Uncommon_Use),
        ('\u{222}', '\u{225}',  IdentifierType::Uncommon_Use), ('\u{226}', '\u{233}',
         IdentifierType::Recommended), ('\u{234}', '\u{236}',  IdentifierType::Technical),
        ('\u{237}', '\u{241}',  IdentifierType::Uncommon_Use), ('\u{242}', '\u{24f}',
         IdentifierType::Uncommon_Use), ('\u{250}', '\u{252}',  IdentifierType::Technical),
        ('\u{253}', '\u{254}',  IdentifierType::Uncommon_Use), ('\u{255}', '\u{255}',
         IdentifierType::Technical), ('\u{256}', '\u{257}',  IdentifierType::Uncommon_Use),
        ('\u{258}', '\u{258}',  IdentifierType::Technical), ('\u{259}', '\u{259}',
         IdentifierType::Recommended), ('\u{25a}', '\u{25a}',  IdentifierType::Technical),
        ('\u{25b}', '\u{25b}',  IdentifierType::Uncommon_Use), ('\u{25c}', '\u{25f}',
         IdentifierType::Technical), ('\u{260}', '\u{260}',  IdentifierType::Uncommon_Use),
        ('\u{261}', '\u{262}',  IdentifierType::Technical), ('\u{263}', '\u{263}',
         IdentifierType::Uncommon_Use), ('\u{264}', '\u{267}',  IdentifierType::Technical),
        ('\u{268}', '\u{269}',  IdentifierType::Uncommon_Use), ('\u{26a}', '\u{26e}',
         IdentifierType::Technical), ('\u{26f}', '\u{26f}',  IdentifierType::Uncommon_Use),
        ('\u{270}', '\u{271}',  IdentifierType::Technical), ('\u{272}', '\u{272}',
         IdentifierType::Uncommon_Use), ('\u{273}', '\u{274}',  IdentifierType::Technical),
        ('\u{275}', '\u{275}',  IdentifierType::Uncommon_Use), ('\u{276}', '\u{276}',
         IdentifierType::Technical), ('\u{277}', '\u{277}',  IdentifierType::Technical), ('\u{278}',
        '\u{27b}',  IdentifierType::Technical), ('\u{27c}', '\u{27c}',  IdentifierType::Technical),
        ('\u{27d}', '\u{27f}',  IdentifierType::Technical), ('\u{280}', '\u{280}',
         IdentifierType::Uncommon_Use), ('\u{281}', '\u{282}',  IdentifierType::Technical),
        ('\u{283}', '\u{283}',  IdentifierType::Uncommon_Use), ('\u{284}', '\u{287}',
         IdentifierType::Technical), ('\u{288}', '\u{288}',  IdentifierType::Uncommon_Use),
        ('\u{289}', '\u{289}',  IdentifierType::Uncommon_Use), ('\u{28a}', '\u{28b}',
         IdentifierType::Uncommon_Use), ('\u{28c}', '\u{291}',  IdentifierType::Technical),
        ('\u{292}', '\u{292}',  IdentifierType::Uncommon_Use), ('\u{293}', '\u{293}',
         IdentifierType::Technical), ('\u{294}', '\u{294}',  IdentifierType::Uncommon_Use),
        ('\u{295}', '\u{29d}',  IdentifierType::Technical), ('\u{29e}', '\u{29e}',
         IdentifierType::Technical), ('\u{29f}', '\u{2a8}',  IdentifierType::Technical), ('\u{2a9}',
        '\u{2ad}',  IdentifierType::Technical), ('\u{2ae}', '\u{2af}',  IdentifierType::Technical),
        ('\u{2b0}', '\u{2b8}',  IdentifierType::Not_NFKC), ('\u{2b9}', '\u{2ba}',
         IdentifierType::Technical), ('\u{2bb}', '\u{2bc}',  IdentifierType::Recommended),
        ('\u{2bd}', '\u{2c1}',  IdentifierType::Technical), ('\u{2c2}', '\u{2c5}',
         IdentifierType::Not_XID), ('\u{2c6}', '\u{2d1}',  IdentifierType::Technical), ('\u{2d2}',
        '\u{2d7}',  IdentifierType::Not_XID), ('\u{2d8}', '\u{2dd}',  IdentifierType::Not_XID),
        ('\u{2de}', '\u{2de}',  IdentifierType::Not_XID), ('\u{2df}', '\u{2df}',
         IdentifierType::Not_XID), ('\u{2e0}', '\u{2e4}',  IdentifierType::Not_NFKC), ('\u{2e5}',
        '\u{2e9}',  IdentifierType::Not_XID), ('\u{2ea}', '\u{2eb}',  IdentifierType::Not_XID),
        ('\u{2ec}', '\u{2ec}',  IdentifierType::Recommended), ('\u{2ed}', '\u{2ed}',
         IdentifierType::Not_XID), ('\u{2ee}', '\u{2ee}',  IdentifierType::Technical), ('\u{2ef}',
        '\u{2ff}',  IdentifierType::Not_XID), ('\u{300}', '\u{304}',  IdentifierType::Recommended),
        ('\u{305}', '\u{305}',  IdentifierType::Uncommon_Use), ('\u{306}', '\u{30c}',
         IdentifierType::Recommended), ('\u{30d}', '\u{30d}',  IdentifierType::Uncommon_Use),
        ('\u{30e}', '\u{30e}',  IdentifierType::Technical), ('\u{30f}', '\u{311}',
         IdentifierType::Recommended), ('\u{312}', '\u{312}',  IdentifierType::Technical),
        ('\u{313}', '\u{314}',  IdentifierType::Recommended), ('\u{315}', '\u{315}',
         IdentifierType::Technical), ('\u{316}', '\u{316}',  IdentifierType::Uncommon_Use),
        ('\u{317}', '\u{31a}',  IdentifierType::Technical), ('\u{31b}', '\u{31b}',
         IdentifierType::Recommended), ('\u{31c}', '\u{320}',  IdentifierType::Technical),
        ('\u{321}', '\u{322}',  IdentifierType::Uncommon_Use), ('\u{323}', '\u{328}',
         IdentifierType::Recommended), ('\u{329}', '\u{32c}',  IdentifierType::Technical),
        ('\u{32d}', '\u{32e}',  IdentifierType::Recommended), ('\u{32f}', '\u{32f}',
         IdentifierType::Technical), ('\u{330}', '\u{331}',  IdentifierType::Recommended),
        ('\u{332}', '\u{332}',  IdentifierType::Uncommon_Use), ('\u{333}', '\u{333}',
         IdentifierType::Technical), ('\u{334}', '\u{334}',  IdentifierType::Uncommon_Use),
        ('\u{335}', '\u{335}',  IdentifierType::Recommended), ('\u{336}', '\u{336}',
         IdentifierType::Uncommon_Use), ('\u{337}', '\u{337}',  IdentifierType::Technical),
        ('\u{338}', '\u{339}',  IdentifierType::Recommended), ('\u{33a}', '\u{33f}',
         IdentifierType::Technical), ('\u{340}', '\u{341}',  IdentifierType::Technical), ('\u{342}',
        '\u{342}',  IdentifierType::Recommended), ('\u{343}', '\u{343}',
         IdentifierType::Technical), ('\u{344}', '\u{344}',  IdentifierType::Obsolete), ('\u{345}',
        '\u{345}',  IdentifierType::Recommended), ('\u{346}', '\u{34e}',
         IdentifierType::Technical), ('\u{34f}', '\u{34f}',  IdentifierType::Exclusion), ('\u{350}',
        '\u{357}',  IdentifierType::Technical), ('\u{358}', '\u{358}',
         IdentifierType::Uncommon_Use), ('\u{359}', '\u{35c}',  IdentifierType::Technical),
        ('\u{35d}', '\u{35f}',  IdentifierType::Technical), ('\u{360}', '\u{361}',
         IdentifierType::Technical), ('\u{362}', '\u{362}',  IdentifierType::Technical), ('\u{363}',
        '\u{36f}',  IdentifierType::Obsolete), ('\u{370}', '\u{373}',  IdentifierType::Obsolete),
        ('\u{374}', '\u{374}',  IdentifierType::Not_NFKC), ('\u{375}', '\u{375}',
         IdentifierType::Inclusion), ('\u{376}', '\u{377}',  IdentifierType::Obsolete), ('\u{37a}',
        '\u{37a}',  IdentifierType::Not_XID), ('\u{37b}', '\u{37d}',  IdentifierType::Recommended),
        ('\u{37e}', '\u{37e}',  IdentifierType::Not_XID), ('\u{37f}', '\u{37f}',
         IdentifierType::Obsolete), ('\u{384}', '\u{385}',  IdentifierType::Not_XID), ('\u{386}',
        '\u{386}',  IdentifierType::Recommended), ('\u{387}', '\u{387}',  IdentifierType::Not_NFKC),
        ('\u{388}', '\u{38a}',  IdentifierType::Recommended), ('\u{38c}', '\u{38c}',
         IdentifierType::Recommended), ('\u{38e}', '\u{3a1}',  IdentifierType::Recommended),
        ('\u{3a3}', '\u{3ce}',  IdentifierType::Recommended), ('\u{3cf}', '\u{3cf}',
         IdentifierType::Technical), ('\u{3d0}', '\u{3d1}',  IdentifierType::Obsolete), ('\u{3d2}',
        '\u{3d4}',  IdentifierType::Not_NFKC), ('\u{3d5}', '\u{3d6}',  IdentifierType::Obsolete),
        ('\u{3d7}', '\u{3d7}',  IdentifierType::Technical), ('\u{3d8}', '\u{3d9}',
         IdentifierType::Obsolete), ('\u{3da}', '\u{3da}',  IdentifierType::Obsolete), ('\u{3db}',
        '\u{3db}',  IdentifierType::Obsolete), ('\u{3dc}', '\u{3dc}',  IdentifierType::Obsolete),
        ('\u{3dd}', '\u{3dd}',  IdentifierType::Obsolete), ('\u{3de}', '\u{3de}',
         IdentifierType::Obsolete), ('\u{3df}', '\u{3df}',  IdentifierType::Obsolete), ('\u{3e0}',
        '\u{3e0}',  IdentifierType::Obsolete), ('\u{3e1}', '\u{3e1}',  IdentifierType::Obsolete),
        ('\u{3e2}', '\u{3ef}',  IdentifierType::Exclusion), ('\u{3f0}', '\u{3f2}',
         IdentifierType::Not_NFKC), ('\u{3f3}', '\u{3f3}',  IdentifierType::Technical), ('\u{3f4}',
        '\u{3f5}',  IdentifierType::Not_NFKC), ('\u{3f6}', '\u{3f6}',  IdentifierType::Not_XID),
        ('\u{3f7}', '\u{3f8}',  IdentifierType::Obsolete), ('\u{3f9}', '\u{3f9}',
         IdentifierType::Obsolete), ('\u{3fa}', '\u{3fb}',  IdentifierType::Obsolete), ('\u{3fc}',
        '\u{3ff}',  IdentifierType::Recommended), ('\u{400}', '\u{400}',
         IdentifierType::Recommended), ('\u{401}', '\u{40c}',  IdentifierType::Recommended),
        ('\u{40d}', '\u{40d}',  IdentifierType::Recommended), ('\u{40e}', '\u{44f}',
         IdentifierType::Recommended), ('\u{450}', '\u{450}',  IdentifierType::Recommended),
        ('\u{451}', '\u{45c}',  IdentifierType::Recommended), ('\u{45d}', '\u{45d}',
         IdentifierType::Recommended), ('\u{45e}', '\u{45f}',  IdentifierType::Recommended),
        ('\u{460}', '\u{481}',  IdentifierType::Obsolete), ('\u{482}', '\u{482}',
         IdentifierType::Obsolete), ('\u{483}', '\u{483}',  IdentifierType::Obsolete), ('\u{484}',
        '\u{486}',  IdentifierType::Technical), ('\u{487}', '\u{487}',  IdentifierType::Technical),
        ('\u{488}', '\u{489}',  IdentifierType::Obsolete), ('\u{48a}', '\u{48b}',
         IdentifierType::Recommended), ('\u{48c}', '\u{48f}',  IdentifierType::Recommended),
        ('\u{490}', '\u{4c4}',  IdentifierType::Recommended), ('\u{4c5}', '\u{4c6}',
         IdentifierType::Recommended), ('\u{4c7}', '\u{4c8}',  IdentifierType::Recommended),
        ('\u{4c9}', '\u{4ca}',  IdentifierType::Recommended), ('\u{4cb}', '\u{4cc}',
         IdentifierType::Recommended), ('\u{4cd}', '\u{4ce}',  IdentifierType::Recommended),
        ('\u{4cf}', '\u{4cf}',  IdentifierType::Recommended), ('\u{4d0}', '\u{4eb}',
         IdentifierType::Recommended), ('\u{4ec}', '\u{4ed}',  IdentifierType::Recommended),
        ('\u{4ee}', '\u{4f5}',  IdentifierType::Recommended), ('\u{4f6}', '\u{4f7}',
         IdentifierType::Recommended), ('\u{4f8}', '\u{4f9}',  IdentifierType::Recommended),
        ('\u{4fa}', '\u{4ff}',  IdentifierType::Recommended), ('\u{500}', '\u{50f}',
         IdentifierType::Recommended), ('\u{510}', '\u{513}',  IdentifierType::Recommended),
        ('\u{514}', '\u{523}',  IdentifierType::Recommended), ('\u{524}', '\u{525}',
         IdentifierType::Recommended), ('\u{526}', '\u{527}',  IdentifierType::Recommended),
        ('\u{528}', '\u{529}',  IdentifierType::Recommended), ('\u{52a}', '\u{52d}',
         IdentifierType::Obsolete), ('\u{52e}', '\u{52f}',  IdentifierType::Recommended),
        ('\u{531}', '\u{556}',  IdentifierType::Recommended), ('\u{559}', '\u{559}',
         IdentifierType::Recommended), ('\u{55a}', '\u{55f}',  IdentifierType::Not_XID), ('\u{560}',
        '\u{560}',  IdentifierType::Recommended), ('\u{561}', '\u{586}',
         IdentifierType::Recommended), ('\u{587}', '\u{587}',  IdentifierType::Not_NFKC),
        ('\u{588}', '\u{588}',  IdentifierType::Recommended), ('\u{589}', '\u{589}',
         IdentifierType::Not_XID), ('\u{58a}', '\u{58a}',  IdentifierType::Inclusion), ('\u{58d}',
        '\u{58e}',  IdentifierType::Not_XID), ('\u{58f}', '\u{58f}',  IdentifierType::Not_XID),
        ('\u{591}', '\u{5a1}',  IdentifierType::Uncommon_Use), ('\u{5a2}', '\u{5a2}',
         IdentifierType::Uncommon_Use), ('\u{5a3}', '\u{5af}',  IdentifierType::Uncommon_Use),
        ('\u{5b0}', '\u{5b3}',  IdentifierType::Uncommon_Use), ('\u{5b4}', '\u{5b4}',
         IdentifierType::Recommended), ('\u{5b5}', '\u{5b9}',  IdentifierType::Uncommon_Use),
        ('\u{5ba}', '\u{5ba}',  IdentifierType::Uncommon_Use), ('\u{5bb}', '\u{5bd}',
         IdentifierType::Uncommon_Use), ('\u{5be}', '\u{5be}',  IdentifierType::Not_XID),
        ('\u{5bf}', '\u{5bf}',  IdentifierType::Uncommon_Use), ('\u{5c0}', '\u{5c0}',
         IdentifierType::Not_XID), ('\u{5c1}', '\u{5c2}',  IdentifierType::Uncommon_Use),
        ('\u{5c3}', '\u{5c3}',  IdentifierType::Not_XID), ('\u{5c4}', '\u{5c4}',
         IdentifierType::Uncommon_Use), ('\u{5c5}', '\u{5c5}',  IdentifierType::Uncommon_Use),
        ('\u{5c6}', '\u{5c6}',  IdentifierType::Obsolete), ('\u{5c7}', '\u{5c7}',
         IdentifierType::Uncommon_Use), ('\u{5d0}', '\u{5ea}',  IdentifierType::Recommended),
        ('\u{5ef}', '\u{5ef}',  IdentifierType::Recommended), ('\u{5f0}', '\u{5f2}',
         IdentifierType::Recommended), ('\u{5f3}', '\u{5f4}',  IdentifierType::Inclusion),
        ('\u{600}', '\u{603}',  IdentifierType::Not_XID), ('\u{604}', '\u{604}',
         IdentifierType::Not_XID), ('\u{605}', '\u{605}',  IdentifierType::Not_XID), ('\u{606}',
        '\u{60a}',  IdentifierType::Not_XID), ('\u{60b}', '\u{60b}',  IdentifierType::Not_XID),
        ('\u{60c}', '\u{60c}',  IdentifierType::Not_XID), ('\u{60d}', '\u{60f}',
         IdentifierType::Not_XID), ('\u{610}', '\u{615}',  IdentifierType::Uncommon_Use),
        ('\u{616}', '\u{61a}',  IdentifierType::Uncommon_Use), ('\u{61b}', '\u{61b}',
         IdentifierType::Not_XID), ('\u{61c}', '\u{61c}',  IdentifierType::Exclusion), ('\u{61e}',
        '\u{61e}',  IdentifierType::Not_XID), ('\u{61f}', '\u{61f}',  IdentifierType::Not_XID),
        ('\u{620}', '\u{620}',  IdentifierType::Recommended), ('\u{621}', '\u{63a}',
         IdentifierType::Recommended), ('\u{63b}', '\u{63f}',  IdentifierType::Recommended),
        ('\u{640}', '\u{640}',  IdentifierType::Exclusion), ('\u{641}', '\u{652}',
         IdentifierType::Recommended), ('\u{653}', '\u{655}',  IdentifierType::Recommended),
        ('\u{656}', '\u{658}',  IdentifierType::Uncommon_Use), ('\u{659}', '\u{65e}',
         IdentifierType::Uncommon_Use), ('\u{65f}', '\u{65f}',  IdentifierType::Uncommon_Use),
        ('\u{660}', '\u{669}',  IdentifierType::Recommended), ('\u{66a}', '\u{66d}',
         IdentifierType::Not_XID), ('\u{66e}', '\u{66f}',  IdentifierType::Obsolete), ('\u{670}',
        '\u{672}',  IdentifierType::Recommended), ('\u{673}', '\u{673}',  IdentifierType::Obsolete),
        ('\u{674}', '\u{674}',  IdentifierType::Recommended), ('\u{675}', '\u{678}',
         IdentifierType::Not_NFKC), ('\u{679}', '\u{68d}',  IdentifierType::Recommended),
        ('\u{68e}', '\u{68e}',  IdentifierType::Obsolete), ('\u{68f}', '\u{6b7}',
         IdentifierType::Recommended), ('\u{6b8}', '\u{6b9}',  IdentifierType::Recommended),
        ('\u{6ba}', '\u{6be}',  IdentifierType::Recommended), ('\u{6bf}', '\u{6bf}',
         IdentifierType::Recommended), ('\u{6c0}', '\u{6ce}',  IdentifierType::Recommended),
        ('\u{6cf}', '\u{6cf}',  IdentifierType::Recommended), ('\u{6d0}', '\u{6d3}',
         IdentifierType::Recommended), ('\u{6d4}', '\u{6d4}',  IdentifierType::Not_XID), ('\u{6d5}',
        '\u{6d5}',  IdentifierType::Recommended), ('\u{6d6}', '\u{6dc}',
         IdentifierType::Uncommon_Use), ('\u{6dd}', '\u{6dd}',  IdentifierType::Not_XID),
        ('\u{6de}', '\u{6de}',  IdentifierType::Not_XID), ('\u{6df}', '\u{6e4}',
         IdentifierType::Uncommon_Use), ('\u{6e5}', '\u{6e6}',  IdentifierType::Recommended),
        ('\u{6e7}', '\u{6e8}',  IdentifierType::Uncommon_Use), ('\u{6e9}', '\u{6e9}',
         IdentifierType::Not_XID), ('\u{6ea}', '\u{6ed}',  IdentifierType::Uncommon_Use),
        ('\u{6ee}', '\u{6ef}',  IdentifierType::Recommended), ('\u{6f0}', '\u{6f9}',
         IdentifierType::Recommended), ('\u{6fa}', '\u{6fc}',  IdentifierType::Recommended),
        ('\u{6fd}', '\u{6fe}',  IdentifierType::Inclusion), ('\u{6ff}', '\u{6ff}',
         IdentifierType::Recommended), ('\u{700}', '\u{70d}',  IdentifierType::Limited_Use),
        ('\u{70f}', '\u{70f}',  IdentifierType::Limited_Use), ('\u{710}', '\u{72c}',
         IdentifierType::Limited_Use), ('\u{72d}', '\u{72f}',  IdentifierType::Limited_Use),
        ('\u{730}', '\u{73f}',  IdentifierType::Limited_Use), ('\u{740}', '\u{74a}',
         IdentifierType::Limited_Use), ('\u{74d}', '\u{74f}',  IdentifierType::Limited_Use),
        ('\u{750}', '\u{76d}',  IdentifierType::Recommended), ('\u{76e}', '\u{77f}',
         IdentifierType::Recommended), ('\u{780}', '\u{7b0}',  IdentifierType::Recommended),
        ('\u{7b1}', '\u{7b1}',  IdentifierType::Recommended), ('\u{7c0}', '\u{7e7}',
         IdentifierType::Limited_Use), ('\u{7e8}', '\u{7ea}',  IdentifierType::Limited_Use),
        ('\u{7eb}', '\u{7f5}',  IdentifierType::Limited_Use), ('\u{7f6}', '\u{7f9}',
         IdentifierType::Limited_Use), ('\u{7fa}', '\u{7fa}',  IdentifierType::Limited_Use),
        ('\u{7fd}', '\u{7fd}',  IdentifierType::Limited_Use), ('\u{7fe}', '\u{7ff}',
         IdentifierType::Limited_Use), ('\u{800}', '\u{82d}',  IdentifierType::Exclusion),
        ('\u{830}', '\u{83e}',  IdentifierType::Exclusion), ('\u{840}', '\u{85b}',
         IdentifierType::Limited_Use), ('\u{85e}', '\u{85e}',  IdentifierType::Limited_Use),
        ('\u{860}', '\u{86a}',  IdentifierType::Limited_Use), ('\u{8a0}', '\u{8a0}',
         IdentifierType::Recommended), ('\u{8a1}', '\u{8a1}',  IdentifierType::Recommended),
        ('\u{8a2}', '\u{8ac}',  IdentifierType::Recommended), ('\u{8ad}', '\u{8b1}',
         IdentifierType::Obsolete), ('\u{8b2}', '\u{8b2}',  IdentifierType::Recommended),
        ('\u{8b3}', '\u{8b4}',  IdentifierType::Uncommon_Use), ('\u{8b6}', '\u{8bd}',
         IdentifierType::Recommended), ('\u{8d3}', '\u{8d3}',  IdentifierType::Uncommon_Use),
        ('\u{8d4}', '\u{8e1}',  IdentifierType::Uncommon_Use), ('\u{8e2}', '\u{8e2}',
         IdentifierType::Not_XID), ('\u{8e3}', '\u{8e3}',  IdentifierType::Uncommon_Use),
        ('\u{8e4}', '\u{8fe}',  IdentifierType::Uncommon_Use), ('\u{8ff}', '\u{8ff}',
         IdentifierType::Uncommon_Use), ('\u{900}', '\u{900}',  IdentifierType::Uncommon_Use),
        ('\u{901}', '\u{903}',  IdentifierType::Recommended), ('\u{904}', '\u{904}',
         IdentifierType::Recommended), ('\u{905}', '\u{939}',  IdentifierType::Recommended),
        ('\u{93a}', '\u{93b}',  IdentifierType::Recommended), ('\u{93c}', '\u{94d}',
         IdentifierType::Recommended), ('\u{94e}', '\u{94e}',  IdentifierType::Obsolete),
        ('\u{94f}', '\u{94f}',  IdentifierType::Recommended), ('\u{950}', '\u{950}',
         IdentifierType::Recommended), ('\u{951}', '\u{952}',  IdentifierType::Obsolete),
        ('\u{953}', '\u{954}',  IdentifierType::Technical), ('\u{955}', '\u{955}',
         IdentifierType::Uncommon_Use), ('\u{956}', '\u{957}',  IdentifierType::Recommended),
        ('\u{958}', '\u{95f}',  IdentifierType::Not_NFKC), ('\u{960}', '\u{963}',
         IdentifierType::Recommended), ('\u{964}', '\u{965}',  IdentifierType::Not_XID), ('\u{966}',
        '\u{96f}',  IdentifierType::Recommended), ('\u{970}', '\u{970}',  IdentifierType::Not_XID),
        ('\u{971}', '\u{972}',  IdentifierType::Recommended), ('\u{973}', '\u{977}',
         IdentifierType::Recommended), ('\u{978}', '\u{978}',  IdentifierType::Obsolete),
        ('\u{979}', '\u{97a}',  IdentifierType::Recommended), ('\u{97b}', '\u{97c}',
         IdentifierType::Recommended), ('\u{97d}', '\u{97d}',  IdentifierType::Recommended),
        ('\u{97e}', '\u{97f}',  IdentifierType::Recommended), ('\u{980}', '\u{980}',
         IdentifierType::Obsolete), ('\u{981}', '\u{983}',  IdentifierType::Recommended),
        ('\u{985}', '\u{98c}',  IdentifierType::Recommended), ('\u{98f}', '\u{990}',
         IdentifierType::Recommended), ('\u{993}', '\u{9a8}',  IdentifierType::Recommended),
        ('\u{9aa}', '\u{9b0}',  IdentifierType::Recommended), ('\u{9b2}', '\u{9b2}',
         IdentifierType::Recommended), ('\u{9b6}', '\u{9b9}',  IdentifierType::Recommended),
        ('\u{9bc}', '\u{9bc}',  IdentifierType::Recommended), ('\u{9bd}', '\u{9bd}',
         IdentifierType::Recommended), ('\u{9be}', '\u{9c4}',  IdentifierType::Recommended),
        ('\u{9c7}', '\u{9c8}',  IdentifierType::Recommended), ('\u{9cb}', '\u{9cd}',
         IdentifierType::Recommended), ('\u{9ce}', '\u{9ce}',  IdentifierType::Recommended),
        ('\u{9d7}', '\u{9d7}',  IdentifierType::Recommended), ('\u{9dc}', '\u{9dd}',
         IdentifierType::Not_NFKC), ('\u{9df}', '\u{9df}',  IdentifierType::Not_NFKC), ('\u{9e0}',
        '\u{9e3}',  IdentifierType::Recommended), ('\u{9e6}', '\u{9f1}',
         IdentifierType::Recommended), ('\u{9f2}', '\u{9fa}',  IdentifierType::Not_XID), ('\u{9fb}',
        '\u{9fb}',  IdentifierType::Not_XID), ('\u{9fc}', '\u{9fc}',  IdentifierType::Recommended),
        ('\u{9fd}', '\u{9fd}',  IdentifierType::Not_XID), ('\u{9fe}', '\u{9fe}',
         IdentifierType::Recommended), ('\u{a01}', '\u{a01}',  IdentifierType::Recommended),
        ('\u{a02}', '\u{a02}',  IdentifierType::Recommended), ('\u{a03}', '\u{a03}',
         IdentifierType::Recommended), ('\u{a05}', '\u{a0a}',  IdentifierType::Recommended),
        ('\u{a0f}', '\u{a10}',  IdentifierType::Recommended), ('\u{a13}', '\u{a28}',
         IdentifierType::Recommended), ('\u{a2a}', '\u{a30}',  IdentifierType::Recommended),
        ('\u{a32}', '\u{a32}',  IdentifierType::Recommended), ('\u{a33}', '\u{a33}',
         IdentifierType::Not_NFKC), ('\u{a35}', '\u{a35}',  IdentifierType::Recommended),
        ('\u{a36}', '\u{a36}',  IdentifierType::Not_NFKC), ('\u{a38}', '\u{a39}',
         IdentifierType::Recommended), ('\u{a3c}', '\u{a3c}',  IdentifierType::Recommended),
        ('\u{a3e}', '\u{a42}',  IdentifierType::Recommended), ('\u{a47}', '\u{a48}',
         IdentifierType::Recommended), ('\u{a4b}', '\u{a4d}',  IdentifierType::Recommended),
        ('\u{a51}', '\u{a51}',  IdentifierType::Uncommon_Use), ('\u{a59}', '\u{a5b}',
         IdentifierType::Not_NFKC), ('\u{a5c}', '\u{a5c}',  IdentifierType::Recommended),
        ('\u{a5e}', '\u{a5e}',  IdentifierType::Not_NFKC), ('\u{a66}', '\u{a74}',
         IdentifierType::Recommended), ('\u{a75}', '\u{a75}',  IdentifierType::Uncommon_Use),
        ('\u{a76}', '\u{a76}',  IdentifierType::Not_XID), ('\u{a81}', '\u{a83}',
         IdentifierType::Recommended), ('\u{a85}', '\u{a8b}',  IdentifierType::Recommended),
        ('\u{a8c}', '\u{a8c}',  IdentifierType::Recommended), ('\u{a8d}', '\u{a8d}',
         IdentifierType::Recommended), ('\u{a8f}', '\u{a91}',  IdentifierType::Recommended),
        ('\u{a93}', '\u{aa8}',  IdentifierType::Recommended), ('\u{aaa}', '\u{ab0}',
         IdentifierType::Recommended), ('\u{ab2}', '\u{ab3}',  IdentifierType::Recommended),
        ('\u{ab5}', '\u{ab9}',  IdentifierType::Recommended), ('\u{abc}', '\u{ac5}',
         IdentifierType::Recommended), ('\u{ac7}', '\u{ac9}',  IdentifierType::Recommended),
        ('\u{acb}', '\u{acd}',  IdentifierType::Recommended), ('\u{ad0}', '\u{ad0}',
         IdentifierType::Recommended), ('\u{ae0}', '\u{ae0}',  IdentifierType::Recommended),
        ('\u{ae1}', '\u{ae3}',  IdentifierType::Recommended), ('\u{ae6}', '\u{aef}',
         IdentifierType::Recommended), ('\u{af0}', '\u{af0}',  IdentifierType::Not_XID), ('\u{af1}',
        '\u{af1}',  IdentifierType::Not_XID), ('\u{af9}', '\u{af9}',  IdentifierType::Uncommon_Use),
        ('\u{afa}', '\u{aff}',  IdentifierType::Recommended), ('\u{b01}', '\u{b03}',
         IdentifierType::Recommended), ('\u{b05}', '\u{b0c}',  IdentifierType::Recommended),
        ('\u{b0f}', '\u{b10}',  IdentifierType::Recommended), ('\u{b13}', '\u{b28}',
         IdentifierType::Recommended), ('\u{b2a}', '\u{b30}',  IdentifierType::Recommended),
        ('\u{b32}', '\u{b33}',  IdentifierType::Recommended), ('\u{b35}', '\u{b35}',
         IdentifierType::Recommended), ('\u{b36}', '\u{b39}',  IdentifierType::Recommended),
        ('\u{b3c}', '\u{b43}',  IdentifierType::Recommended), ('\u{b44}', '\u{b44}',
         IdentifierType::Uncommon_Use), ('\u{b47}', '\u{b48}',  IdentifierType::Recommended),
        ('\u{b4b}', '\u{b4d}',  IdentifierType::Recommended), ('\u{b56}', '\u{b57}',
         IdentifierType::Recommended), ('\u{b5c}', '\u{b5d}',  IdentifierType::Not_NFKC),
        ('\u{b5f}', '\u{b61}',  IdentifierType::Recommended), ('\u{b62}', '\u{b63}',
         IdentifierType::Uncommon_Use), ('\u{b66}', '\u{b6f}',  IdentifierType::Recommended),
        ('\u{b70}', '\u{b70}',  IdentifierType::Not_XID), ('\u{b71}', '\u{b71}',
         IdentifierType::Recommended), ('\u{b72}', '\u{b77}',  IdentifierType::Not_XID), ('\u{b82}',
        '\u{b83}',  IdentifierType::Recommended), ('\u{b85}', '\u{b8a}',
         IdentifierType::Recommended), ('\u{b8e}', '\u{b90}',  IdentifierType::Recommended),
        ('\u{b92}', '\u{b95}',  IdentifierType::Recommended), ('\u{b99}', '\u{b9a}',
         IdentifierType::Recommended), ('\u{b9c}', '\u{b9c}',  IdentifierType::Recommended),
        ('\u{b9e}', '\u{b9f}',  IdentifierType::Recommended), ('\u{ba3}', '\u{ba4}',
         IdentifierType::Recommended), ('\u{ba8}', '\u{baa}',  IdentifierType::Recommended),
        ('\u{bae}', '\u{bb5}',  IdentifierType::Recommended), ('\u{bb6}', '\u{bb6}',
         IdentifierType::Recommended), ('\u{bb7}', '\u{bb9}',  IdentifierType::Recommended),
        ('\u{bbe}', '\u{bc2}',  IdentifierType::Recommended), ('\u{bc6}', '\u{bc8}',
         IdentifierType::Recommended), ('\u{bca}', '\u{bcd}',  IdentifierType::Recommended),
        ('\u{bd0}', '\u{bd0}',  IdentifierType::Recommended), ('\u{bd7}', '\u{bd7}',
         IdentifierType::Recommended), ('\u{be6}', '\u{be6}',  IdentifierType::Recommended),
        ('\u{be7}', '\u{bef}',  IdentifierType::Recommended), ('\u{bf0}', '\u{bf2}',
         IdentifierType::Not_XID), ('\u{bf3}', '\u{bfa}',  IdentifierType::Not_XID), ('\u{c00}',
        '\u{c00}',  IdentifierType::Obsolete), ('\u{c01}', '\u{c03}',  IdentifierType::Recommended),
        ('\u{c04}', '\u{c04}',  IdentifierType::Recommended), ('\u{c05}', '\u{c0c}',
         IdentifierType::Recommended), ('\u{c0e}', '\u{c10}',  IdentifierType::Recommended),
        ('\u{c12}', '\u{c28}',  IdentifierType::Recommended), ('\u{c2a}', '\u{c33}',
         IdentifierType::Recommended), ('\u{c34}', '\u{c34}',  IdentifierType::Obsolete),
        ('\u{c35}', '\u{c39}',  IdentifierType::Recommended), ('\u{c3d}', '\u{c3d}',
         IdentifierType::Recommended), ('\u{c3e}', '\u{c44}',  IdentifierType::Recommended),
        ('\u{c46}', '\u{c48}',  IdentifierType::Recommended), ('\u{c4a}', '\u{c4d}',
         IdentifierType::Recommended), ('\u{c55}', '\u{c56}',  IdentifierType::Recommended),
        ('\u{c58}', '\u{c59}',  IdentifierType::Obsolete), ('\u{c5a}', '\u{c5a}',
         IdentifierType::Uncommon_Use), ('\u{c60}', '\u{c61}',  IdentifierType::Recommended),
        ('\u{c62}', '\u{c63}',  IdentifierType::Uncommon_Use), ('\u{c66}', '\u{c6f}',
         IdentifierType::Recommended), ('\u{c77}', '\u{c77}',  IdentifierType::Not_XID), ('\u{c78}',
        '\u{c7f}',  IdentifierType::Not_XID), ('\u{c80}', '\u{c80}',  IdentifierType::Recommended),
        ('\u{c81}', '\u{c81}',  IdentifierType::Obsolete), ('\u{c82}', '\u{c83}',
         IdentifierType::Recommended), ('\u{c84}', '\u{c84}',  IdentifierType::Not_XID), ('\u{c85}',
        '\u{c8c}',  IdentifierType::Recommended), ('\u{c8e}', '\u{c90}',
         IdentifierType::Recommended), ('\u{c92}', '\u{ca8}',  IdentifierType::Recommended),
        ('\u{caa}', '\u{cb3}',  IdentifierType::Recommended), ('\u{cb5}', '\u{cb9}',
         IdentifierType::Recommended), ('\u{cbc}', '\u{cbd}',  IdentifierType::Recommended),
        ('\u{cbe}', '\u{cc4}',  IdentifierType::Recommended), ('\u{cc6}', '\u{cc8}',
         IdentifierType::Recommended), ('\u{cca}', '\u{ccd}',  IdentifierType::Recommended),
        ('\u{cd5}', '\u{cd6}',  IdentifierType::Recommended), ('\u{cde}', '\u{cde}',
         IdentifierType::Obsolete), ('\u{ce0}', '\u{ce1}',  IdentifierType::Recommended),
        ('\u{ce2}', '\u{ce3}',  IdentifierType::Recommended), ('\u{ce6}', '\u{cef}',
         IdentifierType::Recommended), ('\u{cf1}', '\u{cf2}',  IdentifierType::Recommended),
        ('\u{d00}', '\u{d00}',  IdentifierType::Recommended), ('\u{d01}', '\u{d01}',
         IdentifierType::Obsolete), ('\u{d02}', '\u{d03}',  IdentifierType::Recommended),
        ('\u{d05}', '\u{d0c}',  IdentifierType::Recommended), ('\u{d0e}', '\u{d10}',
         IdentifierType::Recommended), ('\u{d12}', '\u{d28}',  IdentifierType::Recommended),
        ('\u{d29}', '\u{d29}',  IdentifierType::Recommended), ('\u{d2a}', '\u{d39}',
         IdentifierType::Recommended), ('\u{d3a}', '\u{d3a}',  IdentifierType::Recommended),
        ('\u{d3b}', '\u{d3c}',  IdentifierType::Recommended), ('\u{d3d}', '\u{d3d}',
         IdentifierType::Recommended), ('\u{d3e}', '\u{d43}',  IdentifierType::Recommended),
        ('\u{d44}', '\u{d44}',  IdentifierType::Uncommon_Use), ('\u{d46}', '\u{d48}',
         IdentifierType::Recommended), ('\u{d4a}', '\u{d4d}',  IdentifierType::Recommended),
        ('\u{d4e}', '\u{d4e}',  IdentifierType::Recommended), ('\u{d4f}', '\u{d4f}',
         IdentifierType::Not_XID), ('\u{d54}', '\u{d56}',  IdentifierType::Recommended), ('\u{d57}',
        '\u{d57}',  IdentifierType::Recommended), ('\u{d58}', '\u{d5e}',  IdentifierType::Not_XID),
        ('\u{d5f}', '\u{d5f}',  IdentifierType::Obsolete), ('\u{d60}', '\u{d61}',
         IdentifierType::Recommended), ('\u{d62}', '\u{d63}',  IdentifierType::Uncommon_Use),
        ('\u{d66}', '\u{d6f}',  IdentifierType::Recommended), ('\u{d70}', '\u{d75}',
         IdentifierType::Not_XID), ('\u{d76}', '\u{d78}',  IdentifierType::Not_XID), ('\u{d79}',
        '\u{d79}',  IdentifierType::Not_XID), ('\u{d7a}', '\u{d7f}',  IdentifierType::Recommended),
        ('\u{d82}', '\u{d83}',  IdentifierType::Recommended), ('\u{d85}', '\u{d8e}',
         IdentifierType::Recommended), ('\u{d8f}', '\u{d90}',  IdentifierType::Uncommon_Use),
        ('\u{d91}', '\u{d96}',  IdentifierType::Recommended), ('\u{d9a}', '\u{da5}',
         IdentifierType::Recommended), ('\u{da6}', '\u{da6}',  IdentifierType::Uncommon_Use),
        ('\u{da7}', '\u{db1}',  IdentifierType::Recommended), ('\u{db3}', '\u{dbb}',
         IdentifierType::Recommended), ('\u{dbd}', '\u{dbd}',  IdentifierType::Recommended),
        ('\u{dc0}', '\u{dc6}',  IdentifierType::Recommended), ('\u{dca}', '\u{dca}',
         IdentifierType::Recommended), ('\u{dcf}', '\u{dd4}',  IdentifierType::Recommended),
        ('\u{dd6}', '\u{dd6}',  IdentifierType::Recommended), ('\u{dd8}', '\u{dde}',
         IdentifierType::Recommended), ('\u{ddf}', '\u{ddf}',  IdentifierType::Uncommon_Use),
        ('\u{de6}', '\u{def}',  IdentifierType::Obsolete), ('\u{df2}', '\u{df2}',
         IdentifierType::Recommended), ('\u{df3}', '\u{df3}',  IdentifierType::Uncommon_Use),
        ('\u{df4}', '\u{df4}',  IdentifierType::Not_XID), ('\u{e01}', '\u{e32}',
         IdentifierType::Recommended), ('\u{e33}', '\u{e33}',  IdentifierType::Not_NFKC),
        ('\u{e34}', '\u{e3a}',  IdentifierType::Recommended), ('\u{e3f}', '\u{e3f}',
         IdentifierType::Not_XID), ('\u{e40}', '\u{e4e}',  IdentifierType::Recommended), ('\u{e4f}',
        '\u{e4f}',  IdentifierType::Not_XID), ('\u{e50}', '\u{e59}',  IdentifierType::Recommended),
        ('\u{e5a}', '\u{e5b}',  IdentifierType::Not_XID), ('\u{e81}', '\u{e82}',
         IdentifierType::Recommended), ('\u{e84}', '\u{e84}',  IdentifierType::Recommended),
        ('\u{e86}', '\u{e86}',  IdentifierType::Recommended), ('\u{e87}', '\u{e88}',
         IdentifierType::Recommended), ('\u{e89}', '\u{e89}',  IdentifierType::Recommended),
        ('\u{e8a}', '\u{e8a}',  IdentifierType::Recommended), ('\u{e8c}', '\u{e8c}',
         IdentifierType::Recommended), ('\u{e8d}', '\u{e8d}',  IdentifierType::Recommended),
        ('\u{e8e}', '\u{e93}',  IdentifierType::Recommended), ('\u{e94}', '\u{e97}',
         IdentifierType::Recommended), ('\u{e98}', '\u{e98}',  IdentifierType::Recommended),
        ('\u{e99}', '\u{e9f}',  IdentifierType::Recommended), ('\u{ea0}', '\u{ea0}',
         IdentifierType::Recommended), ('\u{ea1}', '\u{ea3}',  IdentifierType::Recommended),
        ('\u{ea5}', '\u{ea5}',  IdentifierType::Recommended), ('\u{ea7}', '\u{ea7}',
         IdentifierType::Recommended), ('\u{ea8}', '\u{ea9}',  IdentifierType::Recommended),
        ('\u{eaa}', '\u{eab}',  IdentifierType::Recommended), ('\u{eac}', '\u{eac}',
         IdentifierType::Recommended), ('\u{ead}', '\u{eb2}',  IdentifierType::Recommended),
        ('\u{eb3}', '\u{eb3}',  IdentifierType::Not_NFKC), ('\u{eb4}', '\u{eb9}',
         IdentifierType::Recommended), ('\u{eba}', '\u{eba}',  IdentifierType::Recommended),
        ('\u{ebb}', '\u{ebd}',  IdentifierType::Recommended), ('\u{ec0}', '\u{ec4}',
         IdentifierType::Recommended), ('\u{ec6}', '\u{ec6}',  IdentifierType::Recommended),
        ('\u{ec8}', '\u{ecd}',  IdentifierType::Recommended), ('\u{ed0}', '\u{ed9}',
         IdentifierType::Recommended), ('\u{edc}', '\u{edd}',  IdentifierType::Not_NFKC),
        ('\u{ede}', '\u{edf}',  IdentifierType::Recommended), ('\u{f00}', '\u{f00}',
         IdentifierType::Recommended), ('\u{f01}', '\u{f0a}',  IdentifierType::Not_XID), ('\u{f0b}',
        '\u{f0b}',  IdentifierType::Inclusion), ('\u{f0c}', '\u{f0c}',  IdentifierType::Not_XID),
        ('\u{f0d}', '\u{f17}',  IdentifierType::Not_XID), ('\u{f18}', '\u{f19}',
         IdentifierType::Technical), ('\u{f1a}', '\u{f1f}',  IdentifierType::Not_XID), ('\u{f20}',
        '\u{f29}',  IdentifierType::Recommended), ('\u{f2a}', '\u{f34}',  IdentifierType::Not_XID),
        ('\u{f35}', '\u{f35}',  IdentifierType::Recommended), ('\u{f36}', '\u{f36}',
         IdentifierType::Not_XID), ('\u{f37}', '\u{f37}',  IdentifierType::Recommended), ('\u{f38}',
        '\u{f38}',  IdentifierType::Not_XID), ('\u{f39}', '\u{f39}',  IdentifierType::Uncommon_Use),
        ('\u{f3a}', '\u{f3d}',  IdentifierType::Not_XID), ('\u{f3e}', '\u{f42}',
         IdentifierType::Recommended), ('\u{f43}', '\u{f43}',  IdentifierType::Not_NFKC),
        ('\u{f44}', '\u{f47}',  IdentifierType::Recommended), ('\u{f49}', '\u{f4c}',
         IdentifierType::Recommended), ('\u{f4d}', '\u{f4d}',  IdentifierType::Not_NFKC),
        ('\u{f4e}', '\u{f51}',  IdentifierType::Recommended), ('\u{f52}', '\u{f52}',
         IdentifierType::Not_NFKC), ('\u{f53}', '\u{f56}',  IdentifierType::Recommended),
        ('\u{f57}', '\u{f57}',  IdentifierType::Not_NFKC), ('\u{f58}', '\u{f5b}',
         IdentifierType::Recommended), ('\u{f5c}', '\u{f5c}',  IdentifierType::Not_NFKC),
        ('\u{f5d}', '\u{f68}',  IdentifierType::Recommended), ('\u{f69}', '\u{f69}',
         IdentifierType::Not_NFKC), ('\u{f6a}', '\u{f6a}',  IdentifierType::Recommended),
        ('\u{f6b}', '\u{f6c}',  IdentifierType::Recommended), ('\u{f71}', '\u{f72}',
         IdentifierType::Recommended), ('\u{f73}', '\u{f73}',  IdentifierType::Obsolete),
        ('\u{f74}', '\u{f74}',  IdentifierType::Recommended), ('\u{f75}', '\u{f75}',
         IdentifierType::Obsolete), ('\u{f76}', '\u{f76}',  IdentifierType::Not_NFKC), ('\u{f77}',
        '\u{f77}',  IdentifierType::Obsolete), ('\u{f78}', '\u{f78}',  IdentifierType::Not_NFKC),
        ('\u{f79}', '\u{f79}',  IdentifierType::Obsolete), ('\u{f7a}', '\u{f80}',
         IdentifierType::Recommended), ('\u{f81}', '\u{f81}',  IdentifierType::Obsolete),
        ('\u{f82}', '\u{f84}',  IdentifierType::Recommended), ('\u{f85}', '\u{f85}',
         IdentifierType::Not_XID), ('\u{f86}', '\u{f8b}',  IdentifierType::Recommended), ('\u{f8c}',
        '\u{f8f}',  IdentifierType::Recommended), ('\u{f90}', '\u{f92}',
         IdentifierType::Recommended), ('\u{f93}', '\u{f93}',  IdentifierType::Not_NFKC),
        ('\u{f94}', '\u{f95}',  IdentifierType::Recommended), ('\u{f96}', '\u{f96}',
         IdentifierType::Recommended), ('\u{f97}', '\u{f97}',  IdentifierType::Recommended),
        ('\u{f99}', '\u{f9c}',  IdentifierType::Recommended), ('\u{f9d}', '\u{f9d}',
         IdentifierType::Not_NFKC), ('\u{f9e}', '\u{fa1}',  IdentifierType::Recommended),
        ('\u{fa2}', '\u{fa2}',  IdentifierType::Not_NFKC), ('\u{fa3}', '\u{fa6}',
         IdentifierType::Recommended), ('\u{fa7}', '\u{fa7}',  IdentifierType::Not_NFKC),
        ('\u{fa8}', '\u{fab}',  IdentifierType::Recommended), ('\u{fac}', '\u{fac}',
         IdentifierType::Not_NFKC), ('\u{fad}', '\u{fad}',  IdentifierType::Recommended),
        ('\u{fae}', '\u{fb0}',  IdentifierType::Recommended), ('\u{fb1}', '\u{fb7}',
         IdentifierType::Recommended), ('\u{fb8}', '\u{fb8}',  IdentifierType::Recommended),
        ('\u{fb9}', '\u{fb9}',  IdentifierType::Not_NFKC), ('\u{fba}', '\u{fbc}',
         IdentifierType::Recommended), ('\u{fbe}', '\u{fc5}',  IdentifierType::Not_XID), ('\u{fc6}',
        '\u{fc6}',  IdentifierType::Recommended), ('\u{fc7}', '\u{fcc}',  IdentifierType::Not_XID),
        ('\u{fce}', '\u{fce}',  IdentifierType::Not_XID), ('\u{fcf}', '\u{fcf}',
         IdentifierType::Not_XID), ('\u{fd0}', '\u{fd1}',  IdentifierType::Not_XID), ('\u{fd2}',
        '\u{fd4}',  IdentifierType::Not_XID), ('\u{fd5}', '\u{fd8}',  IdentifierType::Not_XID),
        ('\u{fd9}', '\u{fda}',  IdentifierType::Not_XID), ('\u{1000}', '\u{1021}',
         IdentifierType::Recommended), ('\u{1022}', '\u{1022}',  IdentifierType::Recommended),
        ('\u{1023}', '\u{1027}',  IdentifierType::Recommended), ('\u{1028}', '\u{1028}',
         IdentifierType::Recommended), ('\u{1029}', '\u{102a}',  IdentifierType::Recommended),
        ('\u{102b}', '\u{102b}',  IdentifierType::Recommended), ('\u{102c}', '\u{1032}',
         IdentifierType::Recommended), ('\u{1033}', '\u{1035}',  IdentifierType::Recommended),
        ('\u{1036}', '\u{1039}',  IdentifierType::Recommended), ('\u{103a}', '\u{103f}',
         IdentifierType::Recommended), ('\u{1040}', '\u{1049}',  IdentifierType::Recommended),
        ('\u{104a}', '\u{104f}',  IdentifierType::Not_XID), ('\u{1050}', '\u{1059}',
         IdentifierType::Recommended), ('\u{105a}', '\u{1099}',  IdentifierType::Recommended),
        ('\u{109a}', '\u{109d}',  IdentifierType::Recommended), ('\u{109e}', '\u{109f}',
         IdentifierType::Not_XID), ('\u{10a0}', '\u{10c5}',  IdentifierType::Obsolete), ('\u{10c7}',
        '\u{10c7}',  IdentifierType::Recommended), ('\u{10cd}', '\u{10cd}',
         IdentifierType::Recommended), ('\u{10d0}', '\u{10f0}',  IdentifierType::Recommended),
        ('\u{10f1}', '\u{10f6}',  IdentifierType::Obsolete), ('\u{10f7}', '\u{10f8}',
         IdentifierType::Recommended), ('\u{10f9}', '\u{10fa}',  IdentifierType::Recommended),
        ('\u{10fb}', '\u{10fb}',  IdentifierType::Not_XID), ('\u{10fc}', '\u{10fc}',
         IdentifierType::Not_NFKC), ('\u{10fd}', '\u{10ff}',  IdentifierType::Recommended),
        ('\u{1100}', '\u{1159}',  IdentifierType::Obsolete), ('\u{115a}', '\u{115e}',
         IdentifierType::Obsolete), ('\u{115f}', '\u{1160}',  IdentifierType::Exclusion),
        ('\u{1161}', '\u{11a2}',  IdentifierType::Obsolete), ('\u{11a3}', '\u{11a7}',
         IdentifierType::Obsolete), ('\u{11a8}', '\u{11f9}',  IdentifierType::Obsolete),
        ('\u{11fa}', '\u{11ff}',  IdentifierType::Obsolete), ('\u{1200}', '\u{1206}',
         IdentifierType::Recommended), ('\u{1207}', '\u{1207}',  IdentifierType::Recommended),
        ('\u{1208}', '\u{1246}',  IdentifierType::Recommended), ('\u{1247}', '\u{1247}',
         IdentifierType::Recommended), ('\u{1248}', '\u{1248}',  IdentifierType::Recommended),
        ('\u{124a}', '\u{124d}',  IdentifierType::Recommended), ('\u{1250}', '\u{1256}',
         IdentifierType::Recommended), ('\u{1258}', '\u{1258}',  IdentifierType::Recommended),
        ('\u{125a}', '\u{125d}',  IdentifierType::Recommended), ('\u{1260}', '\u{1286}',
         IdentifierType::Recommended), ('\u{1287}', '\u{1287}',  IdentifierType::Recommended),
        ('\u{1288}', '\u{1288}',  IdentifierType::Recommended), ('\u{128a}', '\u{128d}',
         IdentifierType::Recommended), ('\u{1290}', '\u{12ae}',  IdentifierType::Recommended),
        ('\u{12af}', '\u{12af}',  IdentifierType::Recommended), ('\u{12b0}', '\u{12b0}',
         IdentifierType::Recommended), ('\u{12b2}', '\u{12b5}',  IdentifierType::Recommended),
        ('\u{12b8}', '\u{12be}',  IdentifierType::Recommended), ('\u{12c0}', '\u{12c0}',
         IdentifierType::Recommended), ('\u{12c2}', '\u{12c5}',  IdentifierType::Recommended),
        ('\u{12c8}', '\u{12ce}',  IdentifierType::Recommended), ('\u{12cf}', '\u{12cf}',
         IdentifierType::Recommended), ('\u{12d0}', '\u{12d6}',  IdentifierType::Recommended),
        ('\u{12d8}', '\u{12ee}',  IdentifierType::Recommended), ('\u{12ef}', '\u{12ef}',
         IdentifierType::Recommended), ('\u{12f0}', '\u{130e}',  IdentifierType::Recommended),
        ('\u{130f}', '\u{130f}',  IdentifierType::Recommended), ('\u{1310}', '\u{1310}',
         IdentifierType::Recommended), ('\u{1312}', '\u{1315}',  IdentifierType::Recommended),
        ('\u{1318}', '\u{131e}',  IdentifierType::Recommended), ('\u{131f}', '\u{131f}',
         IdentifierType::Recommended), ('\u{1320}', '\u{1346}',  IdentifierType::Recommended),
        ('\u{1347}', '\u{1347}',  IdentifierType::Recommended), ('\u{1348}', '\u{135a}',
         IdentifierType::Recommended), ('\u{135d}', '\u{135e}',  IdentifierType::Recommended),
        ('\u{135f}', '\u{135f}',  IdentifierType::Recommended), ('\u{1360}', '\u{1360}',
         IdentifierType::Not_XID), ('\u{1361}', '\u{1368}',  IdentifierType::Not_XID), ('\u{1369}',
        '\u{1371}',  IdentifierType::Obsolete), ('\u{1372}', '\u{137c}',  IdentifierType::Not_XID),
        ('\u{1380}', '\u{138f}',  IdentifierType::Recommended), ('\u{1390}', '\u{1399}',
         IdentifierType::Not_XID), ('\u{13a0}', '\u{13f4}',  IdentifierType::Limited_Use),
        ('\u{13f5}', '\u{13f5}',  IdentifierType::Limited_Use), ('\u{13f8}', '\u{13fd}',
         IdentifierType::Limited_Use), ('\u{1400}', '\u{1400}',  IdentifierType::Limited_Use),
        ('\u{1401}', '\u{166c}',  IdentifierType::Limited_Use), ('\u{166d}', '\u{166e}',
         IdentifierType::Limited_Use), ('\u{166f}', '\u{1676}',  IdentifierType::Limited_Use),
        ('\u{1677}', '\u{167f}',  IdentifierType::Limited_Use), ('\u{1680}', '\u{1680}',
         IdentifierType::Exclusion), ('\u{1681}', '\u{169a}',  IdentifierType::Exclusion),
        ('\u{169b}', '\u{169c}',  IdentifierType::Exclusion), ('\u{16a0}', '\u{16ea}',
         IdentifierType::Exclusion), ('\u{16eb}', '\u{16ed}',  IdentifierType::Not_XID),
        ('\u{16ee}', '\u{16f0}',  IdentifierType::Exclusion), ('\u{16f1}', '\u{16f8}',
         IdentifierType::Exclusion), ('\u{1700}', '\u{170c}',  IdentifierType::Exclusion),
        ('\u{170e}', '\u{1714}',  IdentifierType::Exclusion), ('\u{1720}', '\u{1734}',
         IdentifierType::Exclusion), ('\u{1735}', '\u{1736}',  IdentifierType::Exclusion),
        ('\u{1740}', '\u{1753}',  IdentifierType::Exclusion), ('\u{1760}', '\u{176c}',
         IdentifierType::Exclusion), ('\u{176e}', '\u{1770}',  IdentifierType::Exclusion),
        ('\u{1772}', '\u{1773}',  IdentifierType::Exclusion), ('\u{1780}', '\u{17a2}',
         IdentifierType::Recommended), ('\u{17a3}', '\u{17a4}',  IdentifierType::Obsolete),
        ('\u{17a5}', '\u{17a7}',  IdentifierType::Recommended), ('\u{17a8}', '\u{17a8}',
         IdentifierType::Obsolete), ('\u{17a9}', '\u{17b3}',  IdentifierType::Recommended),
        ('\u{17b4}', '\u{17b5}',  IdentifierType::Exclusion), ('\u{17b6}', '\u{17ca}',
         IdentifierType::Recommended), ('\u{17cb}', '\u{17d0}',  IdentifierType::Technical),
        ('\u{17d1}', '\u{17d1}',  IdentifierType::Technical), ('\u{17d2}', '\u{17d2}',
         IdentifierType::Recommended), ('\u{17d3}', '\u{17d3}',  IdentifierType::Obsolete),
        ('\u{17d4}', '\u{17d6}',  IdentifierType::Not_XID), ('\u{17d7}', '\u{17d7}',
         IdentifierType::Recommended), ('\u{17d8}', '\u{17d8}',  IdentifierType::Obsolete),
        ('\u{17d9}', '\u{17db}',  IdentifierType::Not_XID), ('\u{17dc}', '\u{17dc}',
         IdentifierType::Recommended), ('\u{17dd}', '\u{17dd}',  IdentifierType::Technical),
        ('\u{17e0}', '\u{17e9}',  IdentifierType::Recommended), ('\u{17f0}', '\u{17f9}',
         IdentifierType::Not_XID), ('\u{1800}', '\u{1801}',  IdentifierType::Limited_Use),
        ('\u{1802}', '\u{1803}',  IdentifierType::Exclusion), ('\u{1804}', '\u{1804}',
         IdentifierType::Limited_Use), ('\u{1805}', '\u{1805}',  IdentifierType::Exclusion),
        ('\u{1806}', '\u{180a}',  IdentifierType::Limited_Use), ('\u{180b}', '\u{180d}',
         IdentifierType::Limited_Use), ('\u{180e}', '\u{180e}',  IdentifierType::Limited_Use),
        ('\u{1810}', '\u{1819}',  IdentifierType::Limited_Use), ('\u{1820}', '\u{1877}',
         IdentifierType::Limited_Use), ('\u{1878}', '\u{1878}',  IdentifierType::Limited_Use),
        ('\u{1880}', '\u{18a8}',  IdentifierType::Limited_Use), ('\u{18a9}', '\u{18a9}',
         IdentifierType::Limited_Use), ('\u{18aa}', '\u{18aa}',  IdentifierType::Limited_Use),
        ('\u{18b0}', '\u{18f5}',  IdentifierType::Limited_Use), ('\u{1900}', '\u{191c}',
         IdentifierType::Limited_Use), ('\u{191d}', '\u{191e}',  IdentifierType::Limited_Use),
        ('\u{1920}', '\u{192b}',  IdentifierType::Limited_Use), ('\u{1930}', '\u{193b}',
         IdentifierType::Limited_Use), ('\u{1940}', '\u{1940}',  IdentifierType::Limited_Use),
        ('\u{1944}', '\u{1945}',  IdentifierType::Limited_Use), ('\u{1946}', '\u{196d}',
         IdentifierType::Limited_Use), ('\u{1970}', '\u{1974}',  IdentifierType::Limited_Use),
        ('\u{1980}', '\u{19a9}',  IdentifierType::Limited_Use), ('\u{19aa}', '\u{19ab}',
         IdentifierType::Limited_Use), ('\u{19b0}', '\u{19c9}',  IdentifierType::Limited_Use),
        ('\u{19d0}', '\u{19d9}',  IdentifierType::Limited_Use), ('\u{19da}', '\u{19da}',
         IdentifierType::Limited_Use), ('\u{19de}', '\u{19df}',  IdentifierType::Limited_Use),
        ('\u{19e0}', '\u{19ff}',  IdentifierType::Not_XID), ('\u{1a00}', '\u{1a1b}',
         IdentifierType::Exclusion), ('\u{1a1e}', '\u{1a1f}',  IdentifierType::Exclusion),
        ('\u{1a20}', '\u{1a5e}',  IdentifierType::Limited_Use), ('\u{1a60}', '\u{1a7c}',
         IdentifierType::Limited_Use), ('\u{1a7f}', '\u{1a89}',  IdentifierType::Limited_Use),
        ('\u{1a90}', '\u{1a99}',  IdentifierType::Limited_Use), ('\u{1aa0}', '\u{1aa6}',
         IdentifierType::Limited_Use), ('\u{1aa7}', '\u{1aa7}',  IdentifierType::Limited_Use),
        ('\u{1aa8}', '\u{1aad}',  IdentifierType::Limited_Use), ('\u{1ab0}', '\u{1abd}',
         IdentifierType::Obsolete), ('\u{1abe}', '\u{1abe}',  IdentifierType::Not_XID), ('\u{1b00}',
        '\u{1b4b}',  IdentifierType::Limited_Use), ('\u{1b50}', '\u{1b59}',
         IdentifierType::Limited_Use), ('\u{1b5a}', '\u{1b6a}',  IdentifierType::Limited_Use),
        ('\u{1b6b}', '\u{1b73}',  IdentifierType::Limited_Use), ('\u{1b74}', '\u{1b7c}',
         IdentifierType::Limited_Use), ('\u{1b80}', '\u{1baa}',  IdentifierType::Limited_Use),
        ('\u{1bab}', '\u{1bad}',  IdentifierType::Limited_Use), ('\u{1bae}', '\u{1bb9}',
         IdentifierType::Limited_Use), ('\u{1bba}', '\u{1bbf}',  IdentifierType::Limited_Use),
        ('\u{1bc0}', '\u{1bf3}',  IdentifierType::Limited_Use), ('\u{1bfc}', '\u{1bff}',
         IdentifierType::Limited_Use), ('\u{1c00}', '\u{1c37}',  IdentifierType::Limited_Use),
        ('\u{1c3b}', '\u{1c3f}',  IdentifierType::Limited_Use), ('\u{1c40}', '\u{1c49}',
         IdentifierType::Limited_Use), ('\u{1c4d}', '\u{1c7d}',  IdentifierType::Limited_Use),
        ('\u{1c7e}', '\u{1c7f}',  IdentifierType::Limited_Use), ('\u{1c80}', '\u{1c88}',
         IdentifierType::Recommended), ('\u{1c90}', '\u{1cba}',  IdentifierType::Recommended),
        ('\u{1cbd}', '\u{1cbf}',  IdentifierType::Recommended), ('\u{1cc0}', '\u{1cc7}',
         IdentifierType::Limited_Use), ('\u{1cd0}', '\u{1cd2}',  IdentifierType::Obsolete),
        ('\u{1cd3}', '\u{1cd3}',  IdentifierType::Obsolete), ('\u{1cd4}', '\u{1cf2}',
         IdentifierType::Obsolete), ('\u{1cf3}', '\u{1cf6}',  IdentifierType::Obsolete),
        ('\u{1cf7}', '\u{1cf7}',  IdentifierType::Obsolete), ('\u{1cf8}', '\u{1cf9}',
         IdentifierType::Obsolete), ('\u{1cfa}', '\u{1cfa}',  IdentifierType::Exclusion),
        ('\u{1d00}', '\u{1d2b}',  IdentifierType::Technical), ('\u{1d2c}', '\u{1d2e}',
         IdentifierType::Technical), ('\u{1d2f}', '\u{1d2f}',  IdentifierType::Technical),
        ('\u{1d30}', '\u{1d3a}',  IdentifierType::Technical), ('\u{1d3b}', '\u{1d3b}',
         IdentifierType::Technical), ('\u{1d3c}', '\u{1d4d}',  IdentifierType::Technical),
        ('\u{1d4e}', '\u{1d4e}',  IdentifierType::Technical), ('\u{1d4f}', '\u{1d6a}',
         IdentifierType::Technical), ('\u{1d6b}', '\u{1d6b}',  IdentifierType::Technical),
        ('\u{1d6c}', '\u{1d77}',  IdentifierType::Technical), ('\u{1d78}', '\u{1d78}',
         IdentifierType::Technical), ('\u{1d79}', '\u{1d9a}',  IdentifierType::Technical),
        ('\u{1d9b}', '\u{1dbf}',  IdentifierType::Technical), ('\u{1dc0}', '\u{1dc3}',
         IdentifierType::Technical), ('\u{1dc4}', '\u{1dca}',  IdentifierType::Technical),
        ('\u{1dcb}', '\u{1dcd}',  IdentifierType::Technical), ('\u{1dce}', '\u{1dce}',
         IdentifierType::Technical), ('\u{1dcf}', '\u{1dd0}',  IdentifierType::Technical),
        ('\u{1dd1}', '\u{1de6}',  IdentifierType::Technical), ('\u{1de7}', '\u{1df5}',
         IdentifierType::Technical), ('\u{1df6}', '\u{1df9}',  IdentifierType::Technical),
        ('\u{1dfb}', '\u{1dfb}',  IdentifierType::Technical), ('\u{1dfc}', '\u{1dfc}',
         IdentifierType::Technical), ('\u{1dfd}', '\u{1dfd}',  IdentifierType::Technical),
        ('\u{1dfe}', '\u{1dff}',  IdentifierType::Technical), ('\u{1e00}', '\u{1e99}',
         IdentifierType::Recommended), ('\u{1e9a}', '\u{1e9a}',  IdentifierType::Not_NFKC),
        ('\u{1e9b}', '\u{1e9b}',  IdentifierType::Not_NFKC), ('\u{1e9c}', '\u{1e9d}',
         IdentifierType::Technical), ('\u{1e9e}', '\u{1e9e}',  IdentifierType::Recommended),
        ('\u{1e9f}', '\u{1e9f}',  IdentifierType::Technical), ('\u{1ea0}', '\u{1ef9}',
         IdentifierType::Recommended), ('\u{1efa}', '\u{1eff}',  IdentifierType::Technical),
        ('\u{1f00}', '\u{1f15}',  IdentifierType::Recommended), ('\u{1f18}', '\u{1f1d}',
         IdentifierType::Recommended), ('\u{1f20}', '\u{1f45}',  IdentifierType::Recommended),
        ('\u{1f48}', '\u{1f4d}',  IdentifierType::Recommended), ('\u{1f50}', '\u{1f57}',
         IdentifierType::Recommended), ('\u{1f59}', '\u{1f59}',  IdentifierType::Recommended),
        ('\u{1f5b}', '\u{1f5b}',  IdentifierType::Recommended), ('\u{1f5d}', '\u{1f5d}',
         IdentifierType::Recommended), ('\u{1f5f}', '\u{1f70}',  IdentifierType::Recommended),
        ('\u{1f71}', '\u{1f71}',  IdentifierType::Not_NFKC), ('\u{1f72}', '\u{1f72}',
         IdentifierType::Recommended), ('\u{1f73}', '\u{1f73}',  IdentifierType::Not_NFKC),
        ('\u{1f74}', '\u{1f74}',  IdentifierType::Recommended), ('\u{1f75}', '\u{1f75}',
         IdentifierType::Not_NFKC), ('\u{1f76}', '\u{1f76}',  IdentifierType::Recommended),
        ('\u{1f77}', '\u{1f77}',  IdentifierType::Not_NFKC), ('\u{1f78}', '\u{1f78}',
         IdentifierType::Recommended), ('\u{1f79}', '\u{1f79}',  IdentifierType::Not_NFKC),
        ('\u{1f7a}', '\u{1f7a}',  IdentifierType::Recommended), ('\u{1f7b}', '\u{1f7b}',
         IdentifierType::Not_NFKC), ('\u{1f7c}', '\u{1f7c}',  IdentifierType::Recommended),
        ('\u{1f7d}', '\u{1f7d}',  IdentifierType::Not_NFKC), ('\u{1f80}', '\u{1fb4}',
         IdentifierType::Recommended), ('\u{1fb6}', '\u{1fba}',  IdentifierType::Recommended),
        ('\u{1fbb}', '\u{1fbb}',  IdentifierType::Not_NFKC), ('\u{1fbc}', '\u{1fbc}',
         IdentifierType::Recommended), ('\u{1fbd}', '\u{1fbd}',  IdentifierType::Not_XID),
        ('\u{1fbe}', '\u{1fbe}',  IdentifierType::Not_NFKC), ('\u{1fbf}', '\u{1fc1}',
         IdentifierType::Not_XID), ('\u{1fc2}', '\u{1fc4}',  IdentifierType::Recommended),
        ('\u{1fc6}', '\u{1fc8}',  IdentifierType::Recommended), ('\u{1fc9}', '\u{1fc9}',
         IdentifierType::Not_NFKC), ('\u{1fca}', '\u{1fca}',  IdentifierType::Recommended),
        ('\u{1fcb}', '\u{1fcb}',  IdentifierType::Not_NFKC), ('\u{1fcc}', '\u{1fcc}',
         IdentifierType::Recommended), ('\u{1fcd}', '\u{1fcf}',  IdentifierType::Not_XID),
        ('\u{1fd0}', '\u{1fd2}',  IdentifierType::Recommended), ('\u{1fd3}', '\u{1fd3}',
         IdentifierType::Not_NFKC), ('\u{1fd6}', '\u{1fda}',  IdentifierType::Recommended),
        ('\u{1fdb}', '\u{1fdb}',  IdentifierType::Not_NFKC), ('\u{1fdd}', '\u{1fdf}',
         IdentifierType::Not_XID), ('\u{1fe0}', '\u{1fe2}',  IdentifierType::Recommended),
        ('\u{1fe3}', '\u{1fe3}',  IdentifierType::Not_NFKC), ('\u{1fe4}', '\u{1fea}',
         IdentifierType::Recommended), ('\u{1feb}', '\u{1feb}',  IdentifierType::Not_NFKC),
        ('\u{1fec}', '\u{1fec}',  IdentifierType::Recommended), ('\u{1fed}', '\u{1fef}',
         IdentifierType::Not_XID), ('\u{1ff2}', '\u{1ff4}',  IdentifierType::Recommended),
        ('\u{1ff6}', '\u{1ff8}',  IdentifierType::Recommended), ('\u{1ff9}', '\u{1ff9}',
         IdentifierType::Not_NFKC), ('\u{1ffa}', '\u{1ffa}',  IdentifierType::Recommended),
        ('\u{1ffb}', '\u{1ffb}',  IdentifierType::Not_NFKC), ('\u{1ffc}', '\u{1ffc}',
         IdentifierType::Recommended), ('\u{1ffd}', '\u{1ffe}',  IdentifierType::Not_XID),
        ('\u{2000}', '\u{200a}',  IdentifierType::Not_XID), ('\u{200b}', '\u{200b}',
         IdentifierType::Exclusion), ('\u{200c}', '\u{200d}',  IdentifierType::Inclusion),
        ('\u{200e}', '\u{200f}',  IdentifierType::Exclusion), ('\u{2010}', '\u{2010}',
         IdentifierType::Inclusion), ('\u{2011}', '\u{2011}',  IdentifierType::Not_XID),
        ('\u{2012}', '\u{2016}',  IdentifierType::Not_XID), ('\u{2017}', '\u{2017}',
         IdentifierType::Not_XID), ('\u{2018}', '\u{2018}',  IdentifierType::Not_XID), ('\u{2019}',
        '\u{2019}',  IdentifierType::Inclusion), ('\u{201a}', '\u{2023}',  IdentifierType::Not_XID),
        ('\u{2024}', '\u{2026}',  IdentifierType::Not_XID), ('\u{2027}', '\u{2027}',
         IdentifierType::Inclusion), ('\u{2028}', '\u{2029}',  IdentifierType::Not_XID),
        ('\u{202a}', '\u{202e}',  IdentifierType::Exclusion), ('\u{202f}', '\u{202f}',
         IdentifierType::Not_XID), ('\u{2030}', '\u{2032}',  IdentifierType::Not_XID), ('\u{2033}',
        '\u{2034}',  IdentifierType::Not_XID), ('\u{2035}', '\u{2035}',  IdentifierType::Not_XID),
        ('\u{2036}', '\u{2037}',  IdentifierType::Not_XID), ('\u{2038}', '\u{203b}',
         IdentifierType::Not_XID), ('\u{203c}', '\u{203c}',  IdentifierType::Not_XID), ('\u{203d}',
        '\u{203d}',  IdentifierType::Not_XID), ('\u{203e}', '\u{203e}',  IdentifierType::Not_XID),
        ('\u{203f}', '\u{2040}',  IdentifierType::Technical), ('\u{2041}', '\u{2046}',
         IdentifierType::Not_XID), ('\u{2047}', '\u{2047}',  IdentifierType::Not_XID), ('\u{2048}',
        '\u{2049}',  IdentifierType::Not_XID), ('\u{204a}', '\u{204d}',  IdentifierType::Not_XID),
        ('\u{204e}', '\u{2052}',  IdentifierType::Not_XID), ('\u{2053}', '\u{2053}',
         IdentifierType::Not_XID), ('\u{2054}', '\u{2054}',  IdentifierType::Uncommon_Use),
        ('\u{2055}', '\u{2055}',  IdentifierType::Not_XID), ('\u{2056}', '\u{2056}',
         IdentifierType::Obsolete), ('\u{2057}', '\u{2057}',  IdentifierType::Not_XID), ('\u{2058}',
        '\u{205e}',  IdentifierType::Obsolete), ('\u{205f}', '\u{205f}',  IdentifierType::Not_XID),
        ('\u{2060}', '\u{2063}',  IdentifierType::Exclusion), ('\u{2064}', '\u{2064}',
         IdentifierType::Exclusion), ('\u{2066}', '\u{2069}',  IdentifierType::Exclusion),
        ('\u{206a}', '\u{206f}',  IdentifierType::Exclusion), ('\u{2070}', '\u{2070}',
         IdentifierType::Not_XID), ('\u{2071}', '\u{2071}',  IdentifierType::Not_NFKC), ('\u{2074}',
        '\u{207e}',  IdentifierType::Not_XID), ('\u{207f}', '\u{207f}',  IdentifierType::Not_NFKC),
        ('\u{2080}', '\u{208e}',  IdentifierType::Not_XID), ('\u{2090}', '\u{2094}',
         IdentifierType::Not_NFKC), ('\u{2095}', '\u{209c}',  IdentifierType::Not_NFKC),
        ('\u{20a0}', '\u{20a7}',  IdentifierType::Not_XID), ('\u{20a8}', '\u{20a8}',
         IdentifierType::Not_XID), ('\u{20a9}', '\u{20aa}',  IdentifierType::Not_XID), ('\u{20ab}',
        '\u{20ab}',  IdentifierType::Not_XID), ('\u{20ac}', '\u{20ac}',  IdentifierType::Not_XID),
        ('\u{20ad}', '\u{20af}',  IdentifierType::Not_XID), ('\u{20b0}', '\u{20b1}',
         IdentifierType::Not_XID), ('\u{20b2}', '\u{20b5}',  IdentifierType::Not_XID), ('\u{20b6}',
        '\u{20b8}',  IdentifierType::Not_XID), ('\u{20b9}', '\u{20b9}',  IdentifierType::Not_XID),
        ('\u{20ba}', '\u{20ba}',  IdentifierType::Not_XID), ('\u{20bb}', '\u{20bd}',
         IdentifierType::Not_XID), ('\u{20be}', '\u{20be}',  IdentifierType::Not_XID), ('\u{20bf}',
        '\u{20bf}',  IdentifierType::Not_XID), ('\u{20d0}', '\u{20dc}',  IdentifierType::Technical),
        ('\u{20dd}', '\u{20dd}',  IdentifierType::Exclusion), ('\u{20de}', '\u{20e0}',
         IdentifierType::Technical), ('\u{20e1}', '\u{20e1}',  IdentifierType::Technical),
        ('\u{20e2}', '\u{20e3}',  IdentifierType::Technical), ('\u{20e4}', '\u{20e4}',
         IdentifierType::Technical), ('\u{20e5}', '\u{20ea}',  IdentifierType::Technical),
        ('\u{20eb}', '\u{20eb}',  IdentifierType::Technical), ('\u{20ec}', '\u{20ef}',
         IdentifierType::Technical), ('\u{20f0}', '\u{20f0}',  IdentifierType::Exclusion),
        ('\u{2100}', '\u{2101}',  IdentifierType::Not_XID), ('\u{2102}', '\u{2102}',
         IdentifierType::Not_NFKC), ('\u{2103}', '\u{2103}',  IdentifierType::Not_XID), ('\u{2104}',
        '\u{2104}',  IdentifierType::Not_XID), ('\u{2105}', '\u{2106}',  IdentifierType::Not_XID),
        ('\u{2107}', '\u{2107}',  IdentifierType::Not_NFKC), ('\u{2108}', '\u{2108}',
         IdentifierType::Not_XID), ('\u{2109}', '\u{2109}',  IdentifierType::Not_XID), ('\u{210a}',
        '\u{2113}',  IdentifierType::Not_NFKC), ('\u{2114}', '\u{2114}',  IdentifierType::Not_XID),
        ('\u{2115}', '\u{2115}',  IdentifierType::Not_NFKC), ('\u{2116}', '\u{2116}',
         IdentifierType::Not_XID), ('\u{2117}', '\u{2117}',  IdentifierType::Not_XID), ('\u{2118}',
        '\u{2118}',  IdentifierType::Technical), ('\u{2119}', '\u{211d}',
         IdentifierType::Not_NFKC), ('\u{211e}', '\u{211f}',  IdentifierType::Not_XID), ('\u{2120}',
        '\u{2122}',  IdentifierType::Not_XID), ('\u{2123}', '\u{2123}',  IdentifierType::Not_XID),
        ('\u{2124}', '\u{2124}',  IdentifierType::Not_NFKC), ('\u{2125}', '\u{2125}',
         IdentifierType::Not_XID), ('\u{2126}', '\u{2126}',  IdentifierType::Not_NFKC), ('\u{2127}',
        '\u{2127}',  IdentifierType::Obsolete), ('\u{2128}', '\u{2128}',  IdentifierType::Not_NFKC),
        ('\u{2129}', '\u{2129}',  IdentifierType::Not_XID), ('\u{212a}', '\u{212d}',
         IdentifierType::Not_NFKC), ('\u{212e}', '\u{212e}',  IdentifierType::Technical),
        ('\u{212f}', '\u{2131}',  IdentifierType::Not_NFKC), ('\u{2132}', '\u{2132}',
         IdentifierType::Obsolete), ('\u{2133}', '\u{2138}',  IdentifierType::Not_NFKC),
        ('\u{2139}', '\u{2139}',  IdentifierType::Not_NFKC), ('\u{213a}', '\u{213a}',
         IdentifierType::Not_XID), ('\u{213b}', '\u{213b}',  IdentifierType::Not_XID), ('\u{213c}',
        '\u{213c}',  IdentifierType::Not_NFKC), ('\u{213d}', '\u{213f}',  IdentifierType::Not_NFKC),
        ('\u{2140}', '\u{2140}',  IdentifierType::Not_XID), ('\u{2141}', '\u{2144}',
         IdentifierType::Not_XID), ('\u{2145}', '\u{2149}',  IdentifierType::Not_NFKC), ('\u{214a}',
        '\u{214b}',  IdentifierType::Not_XID), ('\u{214c}', '\u{214c}',  IdentifierType::Not_XID),
        ('\u{214d}', '\u{214d}',  IdentifierType::Not_XID), ('\u{214e}', '\u{214e}',
         IdentifierType::Obsolete), ('\u{214f}', '\u{214f}',  IdentifierType::Obsolete),
        ('\u{2150}', '\u{2152}',  IdentifierType::Not_XID), ('\u{2153}', '\u{215f}',
         IdentifierType::Not_XID), ('\u{2160}', '\u{217f}',  IdentifierType::Not_NFKC), ('\u{2180}',
        '\u{2182}',  IdentifierType::Technical), ('\u{2183}', '\u{2183}',
         IdentifierType::Technical), ('\u{2184}', '\u{2184}',  IdentifierType::Obsolete),
        ('\u{2185}', '\u{2188}',  IdentifierType::Obsolete), ('\u{2189}', '\u{2189}',
         IdentifierType::Not_XID), ('\u{218a}', '\u{218b}',  IdentifierType::Uncommon_Use),
        ('\u{2190}', '\u{21ea}',  IdentifierType::Not_XID), ('\u{21eb}', '\u{21f3}',
         IdentifierType::Not_XID), ('\u{21f4}', '\u{21ff}',  IdentifierType::Not_XID), ('\u{2200}',
        '\u{222b}',  IdentifierType::Not_XID), ('\u{222c}', '\u{222d}',  IdentifierType::Not_XID),
        ('\u{222e}', '\u{222e}',  IdentifierType::Not_XID), ('\u{222f}', '\u{2230}',
         IdentifierType::Not_XID), ('\u{2231}', '\u{22f1}',  IdentifierType::Not_XID), ('\u{22f2}',
        '\u{22ff}',  IdentifierType::Not_XID), ('\u{2300}', '\u{2300}',  IdentifierType::Not_XID),
        ('\u{2301}', '\u{2301}',  IdentifierType::Not_XID), ('\u{2302}', '\u{2328}',
         IdentifierType::Not_XID), ('\u{2329}', '\u{232a}',  IdentifierType::Obsolete), ('\u{232b}',
        '\u{237a}',  IdentifierType::Not_XID), ('\u{237b}', '\u{237b}',  IdentifierType::Not_XID),
        ('\u{237c}', '\u{237c}',  IdentifierType::Not_XID), ('\u{237d}', '\u{239a}',
         IdentifierType::Not_XID), ('\u{239b}', '\u{23ce}',  IdentifierType::Not_XID), ('\u{23cf}',
        '\u{23d0}',  IdentifierType::Not_XID), ('\u{23d1}', '\u{23db}',  IdentifierType::Not_XID),
        ('\u{23dc}', '\u{23e7}',  IdentifierType::Not_XID), ('\u{23e8}', '\u{23e8}',
         IdentifierType::Not_XID), ('\u{23e9}', '\u{23f3}',  IdentifierType::Not_XID), ('\u{23f4}',
        '\u{23fa}',  IdentifierType::Not_XID), ('\u{23fb}', '\u{23fe}',  IdentifierType::Not_XID),
        ('\u{23ff}', '\u{23ff}',  IdentifierType::Not_XID), ('\u{2400}', '\u{2424}',
         IdentifierType::Not_XID), ('\u{2425}', '\u{2426}',  IdentifierType::Not_XID), ('\u{2440}',
        '\u{244a}',  IdentifierType::Not_XID), ('\u{2460}', '\u{24ea}',  IdentifierType::Technical),
        ('\u{24eb}', '\u{24fe}',  IdentifierType::Technical), ('\u{24ff}', '\u{24ff}',
         IdentifierType::Technical), ('\u{2500}', '\u{2595}',  IdentifierType::Not_XID),
        ('\u{2596}', '\u{259f}',  IdentifierType::Not_XID), ('\u{25a0}', '\u{25ef}',
         IdentifierType::Not_XID), ('\u{25f0}', '\u{25f7}',  IdentifierType::Not_XID), ('\u{25f8}',
        '\u{25ff}',  IdentifierType::Not_XID), ('\u{2600}', '\u{2613}',  IdentifierType::Not_XID),
        ('\u{2614}', '\u{2615}',  IdentifierType::Not_XID), ('\u{2616}', '\u{2617}',
         IdentifierType::Not_XID), ('\u{2618}', '\u{2618}',  IdentifierType::Not_XID), ('\u{2619}',
        '\u{2619}',  IdentifierType::Not_XID), ('\u{261a}', '\u{266f}',  IdentifierType::Not_XID),
        ('\u{2670}', '\u{2671}',  IdentifierType::Not_XID), ('\u{2672}', '\u{267d}',
         IdentifierType::Not_XID), ('\u{267e}', '\u{267f}',  IdentifierType::Not_XID), ('\u{2680}',
        '\u{2689}',  IdentifierType::Not_XID), ('\u{268a}', '\u{2691}',  IdentifierType::Not_XID),
        ('\u{2692}', '\u{269c}',  IdentifierType::Not_XID), ('\u{269d}', '\u{269d}',
         IdentifierType::Not_XID), ('\u{269e}', '\u{269f}',  IdentifierType::Not_XID), ('\u{26a0}',
        '\u{26a1}',  IdentifierType::Not_XID), ('\u{26a2}', '\u{26b1}',  IdentifierType::Not_XID),
        ('\u{26b2}', '\u{26b2}',  IdentifierType::Not_XID), ('\u{26b3}', '\u{26bc}',
         IdentifierType::Not_XID), ('\u{26bd}', '\u{26bf}',  IdentifierType::Not_XID), ('\u{26c0}',
        '\u{26c3}',  IdentifierType::Not_XID), ('\u{26c4}', '\u{26cd}',  IdentifierType::Not_XID),
        ('\u{26ce}', '\u{26ce}',  IdentifierType::Not_XID), ('\u{26cf}', '\u{26e1}',
         IdentifierType::Not_XID), ('\u{26e2}', '\u{26e2}',  IdentifierType::Not_XID), ('\u{26e3}',
        '\u{26e3}',  IdentifierType::Not_XID), ('\u{26e4}', '\u{26e7}',  IdentifierType::Not_XID),
        ('\u{26e8}', '\u{26ff}',  IdentifierType::Not_XID), ('\u{2700}', '\u{2700}',
         IdentifierType::Not_XID), ('\u{2701}', '\u{2704}',  IdentifierType::Not_XID), ('\u{2705}',
        '\u{2705}',  IdentifierType::Not_XID), ('\u{2706}', '\u{2709}',  IdentifierType::Not_XID),
        ('\u{270a}', '\u{270b}',  IdentifierType::Not_XID), ('\u{270c}', '\u{2727}',
         IdentifierType::Not_XID), ('\u{2728}', '\u{2728}',  IdentifierType::Not_XID), ('\u{2729}',
        '\u{274b}',  IdentifierType::Not_XID), ('\u{274c}', '\u{274c}',  IdentifierType::Not_XID),
        ('\u{274d}', '\u{274d}',  IdentifierType::Not_XID), ('\u{274e}', '\u{274e}',
         IdentifierType::Not_XID), ('\u{274f}', '\u{2752}',  IdentifierType::Not_XID), ('\u{2753}',
        '\u{2755}',  IdentifierType::Not_XID), ('\u{2756}', '\u{2756}',  IdentifierType::Not_XID),
        ('\u{2757}', '\u{2757}',  IdentifierType::Not_XID), ('\u{2758}', '\u{275e}',
         IdentifierType::Not_XID), ('\u{275f}', '\u{2760}',  IdentifierType::Not_XID), ('\u{2761}',
        '\u{2767}',  IdentifierType::Not_XID), ('\u{2768}', '\u{2775}',  IdentifierType::Not_XID),
        ('\u{2776}', '\u{2794}',  IdentifierType::Not_XID), ('\u{2795}', '\u{2797}',
         IdentifierType::Not_XID), ('\u{2798}', '\u{27af}',  IdentifierType::Not_XID), ('\u{27b0}',
        '\u{27b0}',  IdentifierType::Not_XID), ('\u{27b1}', '\u{27be}',  IdentifierType::Not_XID),
        ('\u{27bf}', '\u{27bf}',  IdentifierType::Not_XID), ('\u{27c0}', '\u{27c6}',
         IdentifierType::Not_XID), ('\u{27c7}', '\u{27ca}',  IdentifierType::Not_XID), ('\u{27cb}',
        '\u{27cb}',  IdentifierType::Not_XID), ('\u{27cc}', '\u{27cc}',  IdentifierType::Not_XID),
        ('\u{27cd}', '\u{27cd}',  IdentifierType::Not_XID), ('\u{27ce}', '\u{27cf}',
         IdentifierType::Not_XID), ('\u{27d0}', '\u{27eb}',  IdentifierType::Not_XID), ('\u{27ec}',
        '\u{27ef}',  IdentifierType::Not_XID), ('\u{27f0}', '\u{27ff}',  IdentifierType::Not_XID),
        ('\u{2800}', '\u{28ff}',  IdentifierType::Technical), ('\u{2900}', '\u{2a0b}',
         IdentifierType::Not_XID), ('\u{2a0c}', '\u{2a0c}',  IdentifierType::Not_XID), ('\u{2a0d}',
        '\u{2a73}',  IdentifierType::Not_XID), ('\u{2a74}', '\u{2a76}',  IdentifierType::Not_XID),
        ('\u{2a77}', '\u{2adb}',  IdentifierType::Not_XID), ('\u{2adc}', '\u{2adc}',
         IdentifierType::Not_XID), ('\u{2add}', '\u{2aff}',  IdentifierType::Not_XID), ('\u{2b00}',
        '\u{2b0d}',  IdentifierType::Not_XID), ('\u{2b0e}', '\u{2b13}',  IdentifierType::Not_XID),
        ('\u{2b14}', '\u{2b1a}',  IdentifierType::Not_XID), ('\u{2b1b}', '\u{2b1f}',
         IdentifierType::Not_XID), ('\u{2b20}', '\u{2b23}',  IdentifierType::Not_XID), ('\u{2b24}',
        '\u{2b4c}',  IdentifierType::Not_XID), ('\u{2b4d}', '\u{2b4f}',  IdentifierType::Not_XID),
        ('\u{2b50}', '\u{2b54}',  IdentifierType::Not_XID), ('\u{2b55}', '\u{2b59}',
         IdentifierType::Not_XID), ('\u{2b5a}', '\u{2b73}',  IdentifierType::Not_XID), ('\u{2b76}',
        '\u{2b95}',  IdentifierType::Not_XID), ('\u{2b98}', '\u{2bb9}',  IdentifierType::Not_XID),
        ('\u{2bba}', '\u{2bbc}',  IdentifierType::Not_XID), ('\u{2bbd}', '\u{2bc8}',
         IdentifierType::Not_XID), ('\u{2bc9}', '\u{2bc9}',  IdentifierType::Not_XID), ('\u{2bca}',
        '\u{2bd1}',  IdentifierType::Not_XID), ('\u{2bd2}', '\u{2bd2}',  IdentifierType::Not_XID),
        ('\u{2bd3}', '\u{2beb}',  IdentifierType::Not_XID), ('\u{2bec}', '\u{2bef}',
         IdentifierType::Uncommon_Use), ('\u{2bf0}', '\u{2bfe}',  IdentifierType::Not_XID),
        ('\u{2bff}', '\u{2bff}',  IdentifierType::Not_XID), ('\u{2c00}', '\u{2c2e}',
         IdentifierType::Exclusion), ('\u{2c30}', '\u{2c5e}',  IdentifierType::Exclusion),
        ('\u{2c60}', '\u{2c66}',  IdentifierType::Technical), ('\u{2c67}', '\u{2c6c}',
         IdentifierType::Uncommon_Use), ('\u{2c6d}', '\u{2c6f}',  IdentifierType::Obsolete),
        ('\u{2c70}', '\u{2c70}',  IdentifierType::Obsolete), ('\u{2c71}', '\u{2c73}',
         IdentifierType::Obsolete), ('\u{2c74}', '\u{2c76}',  IdentifierType::Obsolete),
        ('\u{2c77}', '\u{2c77}',  IdentifierType::Technical), ('\u{2c78}', '\u{2c7b}',
         IdentifierType::Technical), ('\u{2c7c}', '\u{2c7d}',  IdentifierType::Not_NFKC),
        ('\u{2c7e}', '\u{2c7f}',  IdentifierType::Obsolete), ('\u{2c80}', '\u{2cb1}',
         IdentifierType::Exclusion), ('\u{2cb2}', '\u{2ce3}',  IdentifierType::Exclusion),
        ('\u{2ce4}', '\u{2ce4}',  IdentifierType::Exclusion), ('\u{2ce5}', '\u{2cea}',
         IdentifierType::Exclusion), ('\u{2ceb}', '\u{2cef}',  IdentifierType::Exclusion),
        ('\u{2cf0}', '\u{2cf1}',  IdentifierType::Technical), ('\u{2cf2}', '\u{2cf3}',
         IdentifierType::Exclusion), ('\u{2cf9}', '\u{2cfc}',  IdentifierType::Exclusion),
        ('\u{2cfd}', '\u{2cff}',  IdentifierType::Exclusion), ('\u{2d00}', '\u{2d25}',
         IdentifierType::Obsolete), ('\u{2d27}', '\u{2d27}',  IdentifierType::Recommended),
        ('\u{2d2d}', '\u{2d2d}',  IdentifierType::Recommended), ('\u{2d30}', '\u{2d65}',
         IdentifierType::Limited_Use), ('\u{2d66}', '\u{2d67}',  IdentifierType::Limited_Use),
        ('\u{2d6f}', '\u{2d6f}',  IdentifierType::Limited_Use), ('\u{2d70}', '\u{2d70}',
         IdentifierType::Limited_Use), ('\u{2d7f}', '\u{2d7f}',  IdentifierType::Limited_Use),
        ('\u{2d80}', '\u{2d96}',  IdentifierType::Recommended), ('\u{2da0}', '\u{2da6}',
         IdentifierType::Recommended), ('\u{2da8}', '\u{2dae}',  IdentifierType::Recommended),
        ('\u{2db0}', '\u{2db6}',  IdentifierType::Recommended), ('\u{2db8}', '\u{2dbe}',
         IdentifierType::Recommended), ('\u{2dc0}', '\u{2dc6}',  IdentifierType::Recommended),
        ('\u{2dc8}', '\u{2dce}',  IdentifierType::Recommended), ('\u{2dd0}', '\u{2dd6}',
         IdentifierType::Recommended), ('\u{2dd8}', '\u{2dde}',  IdentifierType::Recommended),
        ('\u{2de0}', '\u{2dff}',  IdentifierType::Obsolete), ('\u{2e00}', '\u{2e0d}',
         IdentifierType::Technical), ('\u{2e0e}', '\u{2e16}',  IdentifierType::Obsolete),
        ('\u{2e17}', '\u{2e17}',  IdentifierType::Not_XID), ('\u{2e18}', '\u{2e1b}',
         IdentifierType::Not_XID), ('\u{2e1c}', '\u{2e1d}',  IdentifierType::Not_XID), ('\u{2e1e}',
        '\u{2e29}',  IdentifierType::Not_XID), ('\u{2e2a}', '\u{2e30}',  IdentifierType::Obsolete),
        ('\u{2e31}', '\u{2e31}',  IdentifierType::Obsolete), ('\u{2e32}', '\u{2e32}',
         IdentifierType::Obsolete), ('\u{2e33}', '\u{2e34}',  IdentifierType::Not_XID), ('\u{2e35}',
        '\u{2e35}',  IdentifierType::Obsolete), ('\u{2e36}', '\u{2e38}',  IdentifierType::Not_XID),
        ('\u{2e39}', '\u{2e39}',  IdentifierType::Obsolete), ('\u{2e3a}', '\u{2e3b}',
         IdentifierType::Not_XID), ('\u{2e3c}', '\u{2e42}',  IdentifierType::Not_XID), ('\u{2e43}',
        '\u{2e44}',  IdentifierType::Not_XID), ('\u{2e45}', '\u{2e49}',  IdentifierType::Not_XID),
        ('\u{2e4a}', '\u{2e4e}',  IdentifierType::Not_XID), ('\u{2e4f}', '\u{2e4f}',
         IdentifierType::Not_XID), ('\u{2e80}', '\u{2e99}',  IdentifierType::Not_XID), ('\u{2e9b}',
        '\u{2e9e}',  IdentifierType::Not_XID), ('\u{2e9f}', '\u{2e9f}',  IdentifierType::Not_XID),
        ('\u{2ea0}', '\u{2ef2}',  IdentifierType::Not_XID), ('\u{2ef3}', '\u{2ef3}',
         IdentifierType::Not_XID), ('\u{2f00}', '\u{2fd5}',  IdentifierType::Not_XID), ('\u{2ff0}',
        '\u{2ffb}',  IdentifierType::Not_XID), ('\u{3000}', '\u{3000}',  IdentifierType::Not_XID),
        ('\u{3001}', '\u{3004}',  IdentifierType::Not_XID), ('\u{3005}', '\u{3007}',
         IdentifierType::Recommended), ('\u{3008}', '\u{301d}',  IdentifierType::Not_XID),
        ('\u{301e}', '\u{301e}',  IdentifierType::Obsolete), ('\u{301f}', '\u{3020}',
         IdentifierType::Not_XID), ('\u{3021}', '\u{302d}',  IdentifierType::Technical),
        ('\u{302e}', '\u{302f}',  IdentifierType::Technical), ('\u{3030}', '\u{3030}',
         IdentifierType::Not_XID), ('\u{3031}', '\u{3035}',  IdentifierType::Technical),
        ('\u{3036}', '\u{3036}',  IdentifierType::Not_XID), ('\u{3037}', '\u{3037}',
         IdentifierType::Not_XID), ('\u{3038}', '\u{303a}',  IdentifierType::Not_NFKC), ('\u{303b}',
        '\u{303c}',  IdentifierType::Technical), ('\u{303d}', '\u{303d}',  IdentifierType::Not_XID),
        ('\u{303e}', '\u{303e}',  IdentifierType::Not_XID), ('\u{303f}', '\u{303f}',
         IdentifierType::Not_XID), ('\u{3041}', '\u{3094}',  IdentifierType::Recommended),
        ('\u{3095}', '\u{3096}',  IdentifierType::Recommended), ('\u{3099}', '\u{309a}',
         IdentifierType::Recommended), ('\u{309b}', '\u{309c}',  IdentifierType::Not_XID),
        ('\u{309d}', '\u{309e}',  IdentifierType::Recommended), ('\u{309f}', '\u{309f}',
         IdentifierType::Not_NFKC), ('\u{30a0}', '\u{30a0}',  IdentifierType::Inclusion),
        ('\u{30a1}', '\u{30fa}',  IdentifierType::Recommended), ('\u{30fb}', '\u{30fb}',
         IdentifierType::Inclusion), ('\u{30fc}', '\u{30fe}',  IdentifierType::Recommended),
        ('\u{30ff}', '\u{30ff}',  IdentifierType::Not_NFKC), ('\u{3105}', '\u{312c}',
         IdentifierType::Recommended), ('\u{312d}', '\u{312d}',  IdentifierType::Recommended),
        ('\u{312e}', '\u{312e}',  IdentifierType::Recommended), ('\u{312f}', '\u{312f}',
         IdentifierType::Recommended), ('\u{3131}', '\u{3163}',  IdentifierType::Obsolete),
        ('\u{3164}', '\u{3164}',  IdentifierType::Exclusion), ('\u{3165}', '\u{318e}',
         IdentifierType::Obsolete), ('\u{3190}', '\u{3191}',  IdentifierType::Not_XID), ('\u{3192}',
        '\u{319f}',  IdentifierType::Not_XID), ('\u{31a0}', '\u{31b7}',
         IdentifierType::Recommended), ('\u{31b8}', '\u{31ba}',  IdentifierType::Recommended),
        ('\u{31c0}', '\u{31cf}',  IdentifierType::Not_XID), ('\u{31d0}', '\u{31e3}',
         IdentifierType::Not_XID), ('\u{31f0}', '\u{31ff}',  IdentifierType::Technical),
        ('\u{3200}', '\u{321c}',  IdentifierType::Obsolete), ('\u{321d}', '\u{321e}',
         IdentifierType::Obsolete), ('\u{3220}', '\u{3243}',  IdentifierType::Not_XID), ('\u{3244}',
        '\u{3247}',  IdentifierType::Not_XID), ('\u{3248}', '\u{324f}',  IdentifierType::Not_XID),
        ('\u{3250}', '\u{3250}',  IdentifierType::Not_XID), ('\u{3251}', '\u{325f}',
         IdentifierType::Technical), ('\u{3260}', '\u{327b}',  IdentifierType::Technical),
        ('\u{327c}', '\u{327d}',  IdentifierType::Technical), ('\u{327e}', '\u{327e}',
         IdentifierType::Technical), ('\u{327f}', '\u{327f}',  IdentifierType::Technical),
        ('\u{3280}', '\u{32b0}',  IdentifierType::Technical), ('\u{32b1}', '\u{32bf}',
         IdentifierType::Technical), ('\u{32c0}', '\u{32cb}',  IdentifierType::Technical),
        ('\u{32cc}', '\u{32cf}',  IdentifierType::Technical), ('\u{32d0}', '\u{32fe}',
         IdentifierType::Technical), ('\u{32ff}', '\u{32ff}',  IdentifierType::Technical),
        ('\u{3300}', '\u{3376}',  IdentifierType::Technical), ('\u{3377}', '\u{337a}',
         IdentifierType::Technical), ('\u{337b}', '\u{33dd}',  IdentifierType::Technical),
        ('\u{33de}', '\u{33df}',  IdentifierType::Technical), ('\u{33e0}', '\u{33fe}',
         IdentifierType::Technical), ('\u{33ff}', '\u{33ff}',  IdentifierType::Technical),
        ('\u{3400}', '\u{4db5}',  IdentifierType::Recommended), ('\u{4dc0}', '\u{4dff}',
         IdentifierType::Technical), ('\u{4e00}', '\u{9fa5}',  IdentifierType::Recommended),
        ('\u{9fa6}', '\u{9fbb}',  IdentifierType::Recommended), ('\u{9fbc}', '\u{9fc3}',
         IdentifierType::Recommended), ('\u{9fc4}', '\u{9fcb}',  IdentifierType::Recommended),
        ('\u{9fcc}', '\u{9fcc}',  IdentifierType::Recommended), ('\u{9fcd}', '\u{9fd5}',
         IdentifierType::Recommended), ('\u{9fd6}', '\u{9fea}',  IdentifierType::Recommended),
        ('\u{9feb}', '\u{9fef}',  IdentifierType::Recommended), ('\u{a000}', '\u{a48c}',
         IdentifierType::Limited_Use), ('\u{a490}', '\u{a4a1}',  IdentifierType::Limited_Use),
        ('\u{a4a2}', '\u{a4a3}',  IdentifierType::Limited_Use), ('\u{a4a4}', '\u{a4b3}',
         IdentifierType::Limited_Use), ('\u{a4b4}', '\u{a4b4}',  IdentifierType::Limited_Use),
        ('\u{a4b5}', '\u{a4c0}',  IdentifierType::Limited_Use), ('\u{a4c1}', '\u{a4c1}',
         IdentifierType::Limited_Use), ('\u{a4c2}', '\u{a4c4}',  IdentifierType::Limited_Use),
        ('\u{a4c5}', '\u{a4c5}',  IdentifierType::Limited_Use), ('\u{a4c6}', '\u{a4c6}',
         IdentifierType::Limited_Use), ('\u{a4d0}', '\u{a4fd}',  IdentifierType::Limited_Use),
        ('\u{a4fe}', '\u{a4ff}',  IdentifierType::Limited_Use), ('\u{a500}', '\u{a60c}',
         IdentifierType::Limited_Use), ('\u{a60d}', '\u{a60f}',  IdentifierType::Limited_Use),
        ('\u{a610}', '\u{a612}',  IdentifierType::Limited_Use), ('\u{a613}', '\u{a629}',
         IdentifierType::Limited_Use), ('\u{a62a}', '\u{a62b}',  IdentifierType::Limited_Use),
        ('\u{a640}', '\u{a65f}',  IdentifierType::Obsolete), ('\u{a660}', '\u{a661}',
         IdentifierType::Recommended), ('\u{a662}', '\u{a66e}',  IdentifierType::Obsolete),
        ('\u{a66f}', '\u{a66f}',  IdentifierType::Uncommon_Use), ('\u{a670}', '\u{a673}',
         IdentifierType::Obsolete), ('\u{a674}', '\u{a67b}',  IdentifierType::Recommended),
        ('\u{a67c}', '\u{a67d}',  IdentifierType::Uncommon_Use), ('\u{a67e}', '\u{a67e}',
         IdentifierType::Not_XID), ('\u{a67f}', '\u{a67f}',  IdentifierType::Recommended),
        ('\u{a680}', '\u{a697}',  IdentifierType::Obsolete), ('\u{a698}', '\u{a69b}',
         IdentifierType::Obsolete), ('\u{a69c}', '\u{a69d}',  IdentifierType::Not_NFKC),
        ('\u{a69e}', '\u{a69e}',  IdentifierType::Uncommon_Use), ('\u{a69f}', '\u{a69f}',
         IdentifierType::Recommended), ('\u{a6a0}', '\u{a6f1}',  IdentifierType::Limited_Use),
        ('\u{a6f2}', '\u{a6f7}',  IdentifierType::Limited_Use), ('\u{a700}', '\u{a707}',
         IdentifierType::Obsolete), ('\u{a708}', '\u{a716}',  IdentifierType::Technical),
        ('\u{a717}', '\u{a71a}',  IdentifierType::Recommended), ('\u{a71b}', '\u{a71f}',
         IdentifierType::Recommended), ('\u{a720}', '\u{a721}',  IdentifierType::Not_XID),
        ('\u{a722}', '\u{a72f}',  IdentifierType::Technical), ('\u{a730}', '\u{a76f}',
         IdentifierType::Obsolete), ('\u{a770}', '\u{a770}',  IdentifierType::Obsolete),
        ('\u{a771}', '\u{a787}',  IdentifierType::Obsolete), ('\u{a788}', '\u{a788}',
         IdentifierType::Recommended), ('\u{a789}', '\u{a78a}',  IdentifierType::Not_XID),
        ('\u{a78b}', '\u{a78c}',  IdentifierType::Uncommon_Use), ('\u{a78d}', '\u{a78e}',
         IdentifierType::Recommended), ('\u{a78f}', '\u{a78f}',  IdentifierType::Uncommon_Use),
        ('\u{a790}', '\u{a791}',  IdentifierType::Recommended), ('\u{a792}', '\u{a793}',
         IdentifierType::Recommended), ('\u{a794}', '\u{a79f}',  IdentifierType::Obsolete),
        ('\u{a7a0}', '\u{a7a9}',  IdentifierType::Recommended), ('\u{a7aa}', '\u{a7aa}',
         IdentifierType::Recommended), ('\u{a7ab}', '\u{a7ad}',  IdentifierType::Obsolete),
        ('\u{a7ae}', '\u{a7ae}',  IdentifierType::Recommended), ('\u{a7af}', '\u{a7af}',
         IdentifierType::Recommended), ('\u{a7b0}', '\u{a7b1}',  IdentifierType::Obsolete),
        ('\u{a7b2}', '\u{a7b7}',  IdentifierType::Uncommon_Use), ('\u{a7b8}', '\u{a7b9}',
         IdentifierType::Recommended), ('\u{a7ba}', '\u{a7bf}',  IdentifierType::Recommended),
        ('\u{a7c2}', '\u{a7c6}',  IdentifierType::Recommended), ('\u{a7f7}', '\u{a7f7}',
         IdentifierType::Obsolete), ('\u{a7f8}', '\u{a7f9}',  IdentifierType::Not_NFKC),
        ('\u{a7fa}', '\u{a7fa}',  IdentifierType::Recommended), ('\u{a7fb}', '\u{a7ff}',
         IdentifierType::Obsolete), ('\u{a800}', '\u{a827}',  IdentifierType::Limited_Use),
        ('\u{a828}', '\u{a82b}',  IdentifierType::Limited_Use), ('\u{a830}', '\u{a839}',
         IdentifierType::Not_XID), ('\u{a840}', '\u{a873}',  IdentifierType::Exclusion),
        ('\u{a874}', '\u{a877}',  IdentifierType::Exclusion), ('\u{a880}', '\u{a8c4}',
         IdentifierType::Limited_Use), ('\u{a8c5}', '\u{a8c5}',  IdentifierType::Limited_Use),
        ('\u{a8ce}', '\u{a8cf}',  IdentifierType::Limited_Use), ('\u{a8d0}', '\u{a8d9}',
         IdentifierType::Limited_Use), ('\u{a8e0}', '\u{a8f7}',  IdentifierType::Obsolete),
        ('\u{a8f8}', '\u{a8fa}',  IdentifierType::Obsolete), ('\u{a8fb}', '\u{a8fb}',
         IdentifierType::Obsolete), ('\u{a8fc}', '\u{a8fc}',  IdentifierType::Uncommon_Use),
        ('\u{a8fd}', '\u{a8fd}',  IdentifierType::Uncommon_Use), ('\u{a8fe}', '\u{a8ff}',
         IdentifierType::Obsolete), ('\u{a900}', '\u{a92d}',  IdentifierType::Limited_Use),
        ('\u{a92e}', '\u{a92e}',  IdentifierType::Not_XID), ('\u{a92f}', '\u{a92f}',
         IdentifierType::Limited_Use), ('\u{a930}', '\u{a953}',  IdentifierType::Exclusion),
        ('\u{a95f}', '\u{a95f}',  IdentifierType::Exclusion), ('\u{a960}', '\u{a97c}',
         IdentifierType::Obsolete), ('\u{a980}', '\u{a9c0}',  IdentifierType::Limited_Use),
        ('\u{a9c1}', '\u{a9cd}',  IdentifierType::Limited_Use), ('\u{a9cf}', '\u{a9cf}',
         IdentifierType::Limited_Use), ('\u{a9d0}', '\u{a9d9}',  IdentifierType::Limited_Use),
        ('\u{a9de}', '\u{a9df}',  IdentifierType::Limited_Use), ('\u{a9e0}', '\u{a9e6}',
         IdentifierType::Obsolete), ('\u{a9e7}', '\u{a9fe}',  IdentifierType::Recommended),
        ('\u{aa00}', '\u{aa36}',  IdentifierType::Limited_Use), ('\u{aa40}', '\u{aa4d}',
         IdentifierType::Limited_Use), ('\u{aa50}', '\u{aa59}',  IdentifierType::Limited_Use),
        ('\u{aa5c}', '\u{aa5f}',  IdentifierType::Limited_Use), ('\u{aa60}', '\u{aa76}',
         IdentifierType::Recommended), ('\u{aa77}', '\u{aa79}',  IdentifierType::Not_XID),
        ('\u{aa7a}', '\u{aa7b}',  IdentifierType::Recommended), ('\u{aa7c}', '\u{aa7f}',
         IdentifierType::Recommended), ('\u{aa80}', '\u{aac2}',  IdentifierType::Limited_Use),
        ('\u{aadb}', '\u{aadd}',  IdentifierType::Limited_Use), ('\u{aade}', '\u{aadf}',
         IdentifierType::Limited_Use), ('\u{aae0}', '\u{aaef}',  IdentifierType::Limited_Use),
        ('\u{aaf0}', '\u{aaf1}',  IdentifierType::Limited_Use), ('\u{aaf2}', '\u{aaf6}',
         IdentifierType::Limited_Use), ('\u{ab01}', '\u{ab06}',  IdentifierType::Recommended),
        ('\u{ab09}', '\u{ab0e}',  IdentifierType::Recommended), ('\u{ab11}', '\u{ab16}',
         IdentifierType::Recommended), ('\u{ab20}', '\u{ab26}',  IdentifierType::Recommended),
        ('\u{ab28}', '\u{ab2e}',  IdentifierType::Recommended), ('\u{ab30}', '\u{ab5a}',
         IdentifierType::Obsolete), ('\u{ab5b}', '\u{ab5b}',  IdentifierType::Not_XID), ('\u{ab5c}',
        '\u{ab5f}',  IdentifierType::Not_NFKC), ('\u{ab60}', '\u{ab63}',
         IdentifierType::Uncommon_Use), ('\u{ab64}', '\u{ab65}',  IdentifierType::Obsolete),
        ('\u{ab66}', '\u{ab67}',  IdentifierType::Recommended), ('\u{ab70}', '\u{abbf}',
         IdentifierType::Limited_Use), ('\u{abc0}', '\u{abea}',  IdentifierType::Limited_Use),
        ('\u{abeb}', '\u{abeb}',  IdentifierType::Limited_Use), ('\u{abec}', '\u{abed}',
         IdentifierType::Limited_Use), ('\u{abf0}', '\u{abf9}',  IdentifierType::Limited_Use),
        ('\u{ac00}', '\u{d7a3}',  IdentifierType::Recommended), ('\u{d7b0}', '\u{d7c6}',
         IdentifierType::Obsolete), ('\u{d7cb}', '\u{d7fb}',  IdentifierType::Obsolete),
        ('\u{f900}', '\u{fa0d}',  IdentifierType::Not_NFKC), ('\u{fa0e}', '\u{fa0f}',
         IdentifierType::Recommended), ('\u{fa10}', '\u{fa10}',  IdentifierType::Not_NFKC),
        ('\u{fa11}', '\u{fa11}',  IdentifierType::Recommended), ('\u{fa12}', '\u{fa12}',
         IdentifierType::Not_NFKC), ('\u{fa13}', '\u{fa14}',  IdentifierType::Recommended),
        ('\u{fa15}', '\u{fa1e}',  IdentifierType::Not_NFKC), ('\u{fa1f}', '\u{fa1f}',
         IdentifierType::Recommended), ('\u{fa20}', '\u{fa20}',  IdentifierType::Not_NFKC),
        ('\u{fa21}', '\u{fa21}',  IdentifierType::Recommended), ('\u{fa22}', '\u{fa22}',
         IdentifierType::Not_NFKC), ('\u{fa23}', '\u{fa24}',  IdentifierType::Recommended),
        ('\u{fa25}', '\u{fa26}',  IdentifierType::Not_NFKC), ('\u{fa27}', '\u{fa29}',
         IdentifierType::Recommended), ('\u{fa2a}', '\u{fa2d}',  IdentifierType::Not_NFKC),
        ('\u{fa2e}', '\u{fa2f}',  IdentifierType::Not_NFKC), ('\u{fa30}', '\u{fa6a}',
         IdentifierType::Not_NFKC), ('\u{fa6b}', '\u{fa6d}',  IdentifierType::Not_NFKC),
        ('\u{fa70}', '\u{fad9}',  IdentifierType::Not_NFKC), ('\u{fb00}', '\u{fb06}',
         IdentifierType::Technical), ('\u{fb13}', '\u{fb17}',  IdentifierType::Technical),
        ('\u{fb1d}', '\u{fb1d}',  IdentifierType::Technical), ('\u{fb1e}', '\u{fb1e}',
         IdentifierType::Uncommon_Use), ('\u{fb1f}', '\u{fb28}',  IdentifierType::Technical),
        ('\u{fb29}', '\u{fb29}',  IdentifierType::Technical), ('\u{fb2a}', '\u{fb36}',
         IdentifierType::Technical), ('\u{fb38}', '\u{fb3c}',  IdentifierType::Technical),
        ('\u{fb3e}', '\u{fb3e}',  IdentifierType::Technical), ('\u{fb40}', '\u{fb41}',
         IdentifierType::Technical), ('\u{fb43}', '\u{fb44}',  IdentifierType::Technical),
        ('\u{fb46}', '\u{fbb1}',  IdentifierType::Technical), ('\u{fbb2}', '\u{fbc1}',
         IdentifierType::Technical), ('\u{fbd3}', '\u{fc5d}',  IdentifierType::Technical),
        ('\u{fc5e}', '\u{fc63}',  IdentifierType::Technical), ('\u{fc64}', '\u{fd3d}',
         IdentifierType::Technical), ('\u{fd3e}', '\u{fd3f}',  IdentifierType::Technical),
        ('\u{fd50}', '\u{fd8f}',  IdentifierType::Technical), ('\u{fd92}', '\u{fdc7}',
         IdentifierType::Technical), ('\u{fdf0}', '\u{fdf9}',  IdentifierType::Technical),
        ('\u{fdfa}', '\u{fdfb}',  IdentifierType::Technical), ('\u{fdfc}', '\u{fdfc}',
         IdentifierType::Technical), ('\u{fdfd}', '\u{fdfd}',  IdentifierType::Technical),
        ('\u{fe00}', '\u{fe0f}',  IdentifierType::Technical), ('\u{fe10}', '\u{fe19}',
         IdentifierType::Technical), ('\u{fe20}', '\u{fe23}',  IdentifierType::Technical),
        ('\u{fe24}', '\u{fe26}',  IdentifierType::Technical), ('\u{fe27}', '\u{fe2d}',
         IdentifierType::Technical), ('\u{fe2e}', '\u{fe2f}',  IdentifierType::Uncommon_Use),
        ('\u{fe30}', '\u{fe32}',  IdentifierType::Technical), ('\u{fe33}', '\u{fe34}',
         IdentifierType::Technical), ('\u{fe35}', '\u{fe44}',  IdentifierType::Technical),
        ('\u{fe45}', '\u{fe46}',  IdentifierType::Technical), ('\u{fe47}', '\u{fe48}',
         IdentifierType::Technical), ('\u{fe49}', '\u{fe4c}',  IdentifierType::Technical),
        ('\u{fe4d}', '\u{fe4f}',  IdentifierType::Technical), ('\u{fe50}', '\u{fe52}',
         IdentifierType::Technical), ('\u{fe54}', '\u{fe66}',  IdentifierType::Technical),
        ('\u{fe68}', '\u{fe6b}',  IdentifierType::Technical), ('\u{fe70}', '\u{fe70}',
         IdentifierType::Technical), ('\u{fe71}', '\u{fe71}',  IdentifierType::Technical),
        ('\u{fe72}', '\u{fe72}',  IdentifierType::Technical), ('\u{fe73}', '\u{fe73}',
         IdentifierType::Technical), ('\u{fe74}', '\u{fe74}',  IdentifierType::Technical),
        ('\u{fe76}', '\u{fe76}',  IdentifierType::Technical), ('\u{fe77}', '\u{fe77}',
         IdentifierType::Technical), ('\u{fe78}', '\u{fe78}',  IdentifierType::Technical),
        ('\u{fe79}', '\u{fe79}',  IdentifierType::Technical), ('\u{fe7a}', '\u{fe7a}',
         IdentifierType::Technical), ('\u{fe7b}', '\u{fe7b}',  IdentifierType::Technical),
        ('\u{fe7c}', '\u{fe7c}',  IdentifierType::Technical), ('\u{fe7d}', '\u{fe7d}',
         IdentifierType::Technical), ('\u{fe7e}', '\u{fe7e}',  IdentifierType::Technical),
        ('\u{fe7f}', '\u{fefc}',  IdentifierType::Technical), ('\u{feff}', '\u{feff}',
         IdentifierType::Exclusion), ('\u{ff01}', '\u{ff0f}',  IdentifierType::Not_XID),
        ('\u{ff10}', '\u{ff19}',  IdentifierType::Not_NFKC), ('\u{ff1a}', '\u{ff20}',
         IdentifierType::Not_XID), ('\u{ff21}', '\u{ff3a}',  IdentifierType::Not_NFKC), ('\u{ff3b}',
        '\u{ff3e}',  IdentifierType::Not_XID), ('\u{ff3f}', '\u{ff3f}',  IdentifierType::Not_NFKC),
        ('\u{ff40}', '\u{ff40}',  IdentifierType::Not_XID), ('\u{ff41}', '\u{ff5a}',
         IdentifierType::Not_NFKC), ('\u{ff5b}', '\u{ff5e}',  IdentifierType::Not_XID), ('\u{ff5f}',
        '\u{ff60}',  IdentifierType::Not_XID), ('\u{ff61}', '\u{ff65}',  IdentifierType::Not_XID),
        ('\u{ff66}', '\u{ff9f}',  IdentifierType::Not_NFKC), ('\u{ffa0}', '\u{ffa0}',
         IdentifierType::Exclusion), ('\u{ffa1}', '\u{ffbe}',  IdentifierType::Obsolete),
        ('\u{ffc2}', '\u{ffc7}',  IdentifierType::Obsolete), ('\u{ffca}', '\u{ffcf}',
         IdentifierType::Obsolete), ('\u{ffd2}', '\u{ffd7}',  IdentifierType::Obsolete),
        ('\u{ffda}', '\u{ffdc}',  IdentifierType::Obsolete), ('\u{ffe0}', '\u{ffe6}',
         IdentifierType::Not_XID), ('\u{ffe8}', '\u{ffee}',  IdentifierType::Not_XID), ('\u{fff9}',
        '\u{fffb}',  IdentifierType::Not_XID), ('\u{fffc}', '\u{fffc}',  IdentifierType::Not_XID),
        ('\u{fffd}', '\u{fffd}',  IdentifierType::Not_XID), ('\u{10000}', '\u{1000b}',
         IdentifierType::Exclusion), ('\u{1000d}', '\u{10026}',  IdentifierType::Exclusion),
        ('\u{10028}', '\u{1003a}',  IdentifierType::Exclusion), ('\u{1003c}', '\u{1003d}',
         IdentifierType::Exclusion), ('\u{1003f}', '\u{1004d}',  IdentifierType::Exclusion),
        ('\u{10050}', '\u{1005d}',  IdentifierType::Exclusion), ('\u{10080}', '\u{100fa}',
         IdentifierType::Exclusion), ('\u{10100}', '\u{10102}',  IdentifierType::Exclusion),
        ('\u{10107}', '\u{10133}',  IdentifierType::Exclusion), ('\u{10137}', '\u{1013f}',
         IdentifierType::Exclusion), ('\u{10140}', '\u{10174}',  IdentifierType::Obsolete),
        ('\u{10175}', '\u{1018a}',  IdentifierType::Not_XID), ('\u{1018b}', '\u{1018c}',
         IdentifierType::Not_XID), ('\u{1018d}', '\u{1018e}',  IdentifierType::Not_XID),
        ('\u{10190}', '\u{1019b}',  IdentifierType::Not_XID), ('\u{101a0}', '\u{101a0}',
         IdentifierType::Not_XID), ('\u{101d0}', '\u{101fc}',  IdentifierType::Exclusion),
        ('\u{101fd}', '\u{101fd}',  IdentifierType::Exclusion), ('\u{10280}', '\u{1029c}',
         IdentifierType::Exclusion), ('\u{102a0}', '\u{102d0}',  IdentifierType::Exclusion),
        ('\u{102e0}', '\u{102e0}',  IdentifierType::Obsolete), ('\u{102e1}', '\u{102fb}',
         IdentifierType::Not_XID), ('\u{10300}', '\u{1031e}',  IdentifierType::Exclusion),
        ('\u{1031f}', '\u{1031f}',  IdentifierType::Exclusion), ('\u{10320}', '\u{10323}',
         IdentifierType::Exclusion), ('\u{1032d}', '\u{1032f}',  IdentifierType::Exclusion),
        ('\u{10330}', '\u{1034a}',  IdentifierType::Exclusion), ('\u{10350}', '\u{1037a}',
         IdentifierType::Exclusion), ('\u{10380}', '\u{1039d}',  IdentifierType::Exclusion),
        ('\u{1039f}', '\u{1039f}',  IdentifierType::Exclusion), ('\u{103a0}', '\u{103c3}',
         IdentifierType::Exclusion), ('\u{103c8}', '\u{103cf}',  IdentifierType::Exclusion),
        ('\u{103d0}', '\u{103d0}',  IdentifierType::Exclusion), ('\u{103d1}', '\u{103d5}',
         IdentifierType::Exclusion), ('\u{10400}', '\u{10425}',  IdentifierType::Exclusion),
        ('\u{10426}', '\u{10427}',  IdentifierType::Exclusion), ('\u{10428}', '\u{1044d}',
         IdentifierType::Exclusion), ('\u{1044e}', '\u{1049d}',  IdentifierType::Exclusion),
        ('\u{104a0}', '\u{104a9}',  IdentifierType::Exclusion), ('\u{104b0}', '\u{104d3}',
         IdentifierType::Limited_Use), ('\u{104d8}', '\u{104fb}',  IdentifierType::Limited_Use),
        ('\u{10500}', '\u{10527}',  IdentifierType::Exclusion), ('\u{10530}', '\u{10563}',
         IdentifierType::Exclusion), ('\u{1056f}', '\u{1056f}',  IdentifierType::Exclusion),
        ('\u{10600}', '\u{10736}',  IdentifierType::Exclusion), ('\u{10740}', '\u{10755}',
         IdentifierType::Exclusion), ('\u{10760}', '\u{10767}',  IdentifierType::Exclusion),
        ('\u{10800}', '\u{10805}',  IdentifierType::Exclusion), ('\u{10808}', '\u{10808}',
         IdentifierType::Exclusion), ('\u{1080a}', '\u{10835}',  IdentifierType::Exclusion),
        ('\u{10837}', '\u{10838}',  IdentifierType::Exclusion), ('\u{1083c}', '\u{1083c}',
         IdentifierType::Exclusion), ('\u{1083f}', '\u{1083f}',  IdentifierType::Exclusion),
        ('\u{10840}', '\u{10855}',  IdentifierType::Exclusion), ('\u{10857}', '\u{1085f}',
         IdentifierType::Exclusion), ('\u{10860}', '\u{10876}',  IdentifierType::Exclusion),
        ('\u{10877}', '\u{1087f}',  IdentifierType::Exclusion), ('\u{10880}', '\u{1089e}',
         IdentifierType::Exclusion), ('\u{108a7}', '\u{108af}',  IdentifierType::Exclusion),
        ('\u{108e0}', '\u{108f2}',  IdentifierType::Exclusion), ('\u{108f4}', '\u{108f5}',
         IdentifierType::Exclusion), ('\u{108fb}', '\u{108ff}',  IdentifierType::Exclusion),
        ('\u{10900}', '\u{10915}',  IdentifierType::Exclusion), ('\u{10916}', '\u{10919}',
         IdentifierType::Exclusion), ('\u{1091a}', '\u{1091b}',  IdentifierType::Exclusion),
        ('\u{1091f}', '\u{1091f}',  IdentifierType::Exclusion), ('\u{10920}', '\u{10939}',
         IdentifierType::Exclusion), ('\u{1093f}', '\u{1093f}',  IdentifierType::Exclusion),
        ('\u{10980}', '\u{109b7}',  IdentifierType::Exclusion), ('\u{109bc}', '\u{109bd}',
         IdentifierType::Exclusion), ('\u{109be}', '\u{109bf}',  IdentifierType::Exclusion),
        ('\u{109c0}', '\u{109cf}',  IdentifierType::Exclusion), ('\u{109d2}', '\u{109ff}',
         IdentifierType::Exclusion), ('\u{10a00}', '\u{10a03}',  IdentifierType::Exclusion),
        ('\u{10a05}', '\u{10a06}',  IdentifierType::Exclusion), ('\u{10a0c}', '\u{10a0c}',
         IdentifierType::Exclusion), ('\u{10a0d}', '\u{10a0d}',  IdentifierType::Technical),
        ('\u{10a0e}', '\u{10a13}',  IdentifierType::Exclusion), ('\u{10a15}', '\u{10a17}',
         IdentifierType::Exclusion), ('\u{10a19}', '\u{10a33}',  IdentifierType::Exclusion),
        ('\u{10a34}', '\u{10a35}',  IdentifierType::Exclusion), ('\u{10a38}', '\u{10a3a}',
         IdentifierType::Exclusion), ('\u{10a3f}', '\u{10a3f}',  IdentifierType::Exclusion),
        ('\u{10a40}', '\u{10a47}',  IdentifierType::Exclusion), ('\u{10a48}', '\u{10a48}',
         IdentifierType::Exclusion), ('\u{10a50}', '\u{10a58}',  IdentifierType::Exclusion),
        ('\u{10a60}', '\u{10a7c}',  IdentifierType::Exclusion), ('\u{10a7d}', '\u{10a7f}',
         IdentifierType::Exclusion), ('\u{10a80}', '\u{10a9c}',  IdentifierType::Exclusion),
        ('\u{10a9d}', '\u{10a9f}',  IdentifierType::Exclusion), ('\u{10ac0}', '\u{10ac7}',
         IdentifierType::Exclusion), ('\u{10ac8}', '\u{10ac8}',  IdentifierType::Exclusion),
        ('\u{10ac9}', '\u{10ae6}',  IdentifierType::Exclusion), ('\u{10aeb}', '\u{10af6}',
         IdentifierType::Exclusion), ('\u{10b00}', '\u{10b35}',  IdentifierType::Exclusion),
        ('\u{10b39}', '\u{10b3f}',  IdentifierType::Exclusion), ('\u{10b40}', '\u{10b55}',
         IdentifierType::Exclusion), ('\u{10b58}', '\u{10b5f}',  IdentifierType::Exclusion),
        ('\u{10b60}', '\u{10b72}',  IdentifierType::Exclusion), ('\u{10b78}', '\u{10b7f}',
         IdentifierType::Exclusion), ('\u{10b80}', '\u{10b91}',  IdentifierType::Exclusion),
        ('\u{10b99}', '\u{10b9c}',  IdentifierType::Exclusion), ('\u{10ba9}', '\u{10baf}',
         IdentifierType::Exclusion), ('\u{10c00}', '\u{10c48}',  IdentifierType::Exclusion),
        ('\u{10c80}', '\u{10cb2}',  IdentifierType::Exclusion), ('\u{10cc0}', '\u{10cf2}',
         IdentifierType::Exclusion), ('\u{10cfa}', '\u{10cff}',  IdentifierType::Exclusion),
        ('\u{10d00}', '\u{10d27}',  IdentifierType::Limited_Use), ('\u{10d30}', '\u{10d39}',
         IdentifierType::Limited_Use), ('\u{10e60}', '\u{10e7e}',  IdentifierType::Not_XID),
        ('\u{10f00}', '\u{10f1c}',  IdentifierType::Exclusion), ('\u{10f1d}', '\u{10f26}',
         IdentifierType::Exclusion), ('\u{10f27}', '\u{10f27}',  IdentifierType::Exclusion),
        ('\u{10f30}', '\u{10f50}',  IdentifierType::Exclusion), ('\u{10f51}', '\u{10f59}',
         IdentifierType::Exclusion), ('\u{10fe0}', '\u{10ff6}',  IdentifierType::Exclusion),
        ('\u{11000}', '\u{11046}',  IdentifierType::Exclusion), ('\u{11047}', '\u{1104d}',
         IdentifierType::Exclusion), ('\u{11052}', '\u{11065}',  IdentifierType::Exclusion),
        ('\u{11066}', '\u{1106f}',  IdentifierType::Exclusion), ('\u{1107f}', '\u{1107f}',
         IdentifierType::Exclusion), ('\u{11080}', '\u{110ba}',  IdentifierType::Exclusion),
        ('\u{110bb}', '\u{110bc}',  IdentifierType::Exclusion), ('\u{110bd}', '\u{110bd}',
         IdentifierType::Exclusion), ('\u{110be}', '\u{110c1}',  IdentifierType::Exclusion),
        ('\u{110cd}', '\u{110cd}',  IdentifierType::Exclusion), ('\u{110d0}', '\u{110e8}',
         IdentifierType::Exclusion), ('\u{110f0}', '\u{110f9}',  IdentifierType::Exclusion),
        ('\u{11100}', '\u{11134}',  IdentifierType::Limited_Use), ('\u{11136}', '\u{1113f}',
         IdentifierType::Limited_Use), ('\u{11140}', '\u{11143}',  IdentifierType::Limited_Use),
        ('\u{11144}', '\u{11146}',  IdentifierType::Limited_Use), ('\u{11150}', '\u{11173}',
         IdentifierType::Exclusion), ('\u{11174}', '\u{11175}',  IdentifierType::Exclusion),
        ('\u{11176}', '\u{11176}',  IdentifierType::Exclusion), ('\u{11180}', '\u{111c4}',
         IdentifierType::Exclusion), ('\u{111c5}', '\u{111c8}',  IdentifierType::Exclusion),
        ('\u{111c9}', '\u{111cc}',  IdentifierType::Exclusion), ('\u{111cd}', '\u{111cd}',
         IdentifierType::Exclusion), ('\u{111d0}', '\u{111d9}',  IdentifierType::Exclusion),
        ('\u{111da}', '\u{111da}',  IdentifierType::Exclusion), ('\u{111db}', '\u{111db}',
         IdentifierType::Exclusion), ('\u{111dc}', '\u{111dc}',  IdentifierType::Exclusion),
        ('\u{111dd}', '\u{111df}',  IdentifierType::Exclusion), ('\u{111e1}', '\u{111f4}',
         IdentifierType::Not_XID), ('\u{11200}', '\u{11211}',  IdentifierType::Exclusion),
        ('\u{11213}', '\u{11237}',  IdentifierType::Exclusion), ('\u{11238}', '\u{1123d}',
         IdentifierType::Exclusion), ('\u{1123e}', '\u{1123e}',  IdentifierType::Exclusion),
        ('\u{11280}', '\u{11286}',  IdentifierType::Exclusion), ('\u{11288}', '\u{11288}',
         IdentifierType::Exclusion), ('\u{1128a}', '\u{1128d}',  IdentifierType::Exclusion),
        ('\u{1128f}', '\u{1129d}',  IdentifierType::Exclusion), ('\u{1129f}', '\u{112a8}',
         IdentifierType::Exclusion), ('\u{112a9}', '\u{112a9}',  IdentifierType::Exclusion),
        ('\u{112b0}', '\u{112ea}',  IdentifierType::Exclusion), ('\u{112f0}', '\u{112f9}',
         IdentifierType::Exclusion), ('\u{11300}', '\u{11300}',  IdentifierType::Exclusion),
        ('\u{11301}', '\u{11303}',  IdentifierType::Exclusion), ('\u{11305}', '\u{1130c}',
         IdentifierType::Exclusion), ('\u{1130f}', '\u{11310}',  IdentifierType::Exclusion),
        ('\u{11313}', '\u{11328}',  IdentifierType::Exclusion), ('\u{1132a}', '\u{11330}',
         IdentifierType::Exclusion), ('\u{11332}', '\u{11333}',  IdentifierType::Exclusion),
        ('\u{11335}', '\u{11339}',  IdentifierType::Exclusion), ('\u{1133b}', '\u{1133b}',
         IdentifierType::Recommended), ('\u{1133c}', '\u{11344}',  IdentifierType::Exclusion),
        ('\u{11347}', '\u{11348}',  IdentifierType::Exclusion), ('\u{1134b}', '\u{1134d}',
         IdentifierType::Exclusion), ('\u{11350}', '\u{11350}',  IdentifierType::Exclusion),
        ('\u{11357}', '\u{11357}',  IdentifierType::Exclusion), ('\u{1135d}', '\u{11363}',
         IdentifierType::Exclusion), ('\u{11366}', '\u{1136c}',  IdentifierType::Exclusion),
        ('\u{11370}', '\u{11374}',  IdentifierType::Exclusion), ('\u{11400}', '\u{1144a}',
         IdentifierType::Limited_Use), ('\u{1144b}', '\u{1144f}',  IdentifierType::Limited_Use),
        ('\u{11450}', '\u{11459}',  IdentifierType::Limited_Use), ('\u{1145b}', '\u{1145b}',
         IdentifierType::Limited_Use), ('\u{1145d}', '\u{1145d}',  IdentifierType::Limited_Use),
        ('\u{1145e}', '\u{1145e}',  IdentifierType::Limited_Use), ('\u{1145f}', '\u{1145f}',
         IdentifierType::Limited_Use), ('\u{11480}', '\u{114c5}',  IdentifierType::Exclusion),
        ('\u{114c6}', '\u{114c6}',  IdentifierType::Exclusion), ('\u{114c7}', '\u{114c7}',
         IdentifierType::Exclusion), ('\u{114d0}', '\u{114d9}',  IdentifierType::Exclusion),
        ('\u{11580}', '\u{115b5}',  IdentifierType::Exclusion), ('\u{115b8}', '\u{115c0}',
         IdentifierType::Exclusion), ('\u{115c1}', '\u{115c9}',  IdentifierType::Exclusion),
        ('\u{115ca}', '\u{115d7}',  IdentifierType::Exclusion), ('\u{115d8}', '\u{115dd}',
         IdentifierType::Exclusion), ('\u{11600}', '\u{11640}',  IdentifierType::Exclusion),
        ('\u{11641}', '\u{11643}',  IdentifierType::Exclusion), ('\u{11644}', '\u{11644}',
         IdentifierType::Exclusion), ('\u{11650}', '\u{11659}',  IdentifierType::Exclusion),
        ('\u{11660}', '\u{1166c}',  IdentifierType::Limited_Use), ('\u{11680}', '\u{116b7}',
         IdentifierType::Exclusion), ('\u{116b8}', '\u{116b8}',  IdentifierType::Exclusion),
        ('\u{116c0}', '\u{116c9}',  IdentifierType::Exclusion), ('\u{11700}', '\u{11719}',
         IdentifierType::Exclusion), ('\u{1171a}', '\u{1171a}',  IdentifierType::Exclusion),
        ('\u{1171d}', '\u{1172b}',  IdentifierType::Exclusion), ('\u{11730}', '\u{11739}',
         IdentifierType::Exclusion), ('\u{1173a}', '\u{1173f}',  IdentifierType::Exclusion),
        ('\u{11800}', '\u{1183a}',  IdentifierType::Exclusion), ('\u{1183b}', '\u{1183b}',
         IdentifierType::Exclusion), ('\u{118a0}', '\u{118e9}',  IdentifierType::Exclusion),
        ('\u{118ea}', '\u{118f2}',  IdentifierType::Exclusion), ('\u{118ff}', '\u{118ff}',
         IdentifierType::Exclusion), ('\u{119a0}', '\u{119a7}',  IdentifierType::Exclusion),
        ('\u{119aa}', '\u{119d7}',  IdentifierType::Exclusion), ('\u{119da}', '\u{119e1}',
         IdentifierType::Exclusion), ('\u{119e2}', '\u{119e2}',  IdentifierType::Exclusion),
        ('\u{119e3}', '\u{119e4}',  IdentifierType::Exclusion), ('\u{11a00}', '\u{11a3e}',
         IdentifierType::Exclusion), ('\u{11a3f}', '\u{11a46}',  IdentifierType::Exclusion),
        ('\u{11a47}', '\u{11a47}',  IdentifierType::Exclusion), ('\u{11a50}', '\u{11a83}',
         IdentifierType::Exclusion), ('\u{11a84}', '\u{11a85}',  IdentifierType::Exclusion),
        ('\u{11a86}', '\u{11a99}',  IdentifierType::Exclusion), ('\u{11a9a}', '\u{11a9c}',
         IdentifierType::Exclusion), ('\u{11a9d}', '\u{11a9d}',  IdentifierType::Exclusion),
        ('\u{11a9e}', '\u{11aa2}',  IdentifierType::Exclusion), ('\u{11ac0}', '\u{11af8}',
         IdentifierType::Exclusion), ('\u{11c00}', '\u{11c08}',  IdentifierType::Exclusion),
        ('\u{11c0a}', '\u{11c36}',  IdentifierType::Exclusion), ('\u{11c38}', '\u{11c40}',
         IdentifierType::Exclusion), ('\u{11c41}', '\u{11c45}',  IdentifierType::Exclusion),
        ('\u{11c50}', '\u{11c59}',  IdentifierType::Exclusion), ('\u{11c5a}', '\u{11c6c}',
         IdentifierType::Exclusion), ('\u{11c70}', '\u{11c71}',  IdentifierType::Exclusion),
        ('\u{11c72}', '\u{11c8f}',  IdentifierType::Exclusion), ('\u{11c92}', '\u{11ca7}',
         IdentifierType::Exclusion), ('\u{11ca9}', '\u{11cb6}',  IdentifierType::Exclusion),
        ('\u{11d00}', '\u{11d06}',  IdentifierType::Exclusion), ('\u{11d08}', '\u{11d09}',
         IdentifierType::Exclusion), ('\u{11d0b}', '\u{11d36}',  IdentifierType::Exclusion),
        ('\u{11d3a}', '\u{11d3a}',  IdentifierType::Exclusion), ('\u{11d3c}', '\u{11d3d}',
         IdentifierType::Exclusion), ('\u{11d3f}', '\u{11d47}',  IdentifierType::Exclusion),
        ('\u{11d50}', '\u{11d59}',  IdentifierType::Exclusion), ('\u{11d60}', '\u{11d65}',
         IdentifierType::Limited_Use), ('\u{11d67}', '\u{11d68}',  IdentifierType::Limited_Use),
        ('\u{11d6a}', '\u{11d8e}',  IdentifierType::Limited_Use), ('\u{11d90}', '\u{11d91}',
         IdentifierType::Limited_Use), ('\u{11d93}', '\u{11d98}',  IdentifierType::Limited_Use),
        ('\u{11da0}', '\u{11da9}',  IdentifierType::Limited_Use), ('\u{11ee0}', '\u{11ef6}',
         IdentifierType::Exclusion), ('\u{11ef7}', '\u{11ef8}',  IdentifierType::Exclusion),
        ('\u{11fc0}', '\u{11ff1}',  IdentifierType::Not_XID), ('\u{11fff}', '\u{11fff}',
         IdentifierType::Not_XID), ('\u{12000}', '\u{1236e}',  IdentifierType::Exclusion),
        ('\u{1236f}', '\u{12398}',  IdentifierType::Exclusion), ('\u{12399}', '\u{12399}',
         IdentifierType::Exclusion), ('\u{12400}', '\u{12462}',  IdentifierType::Exclusion),
        ('\u{12463}', '\u{1246e}',  IdentifierType::Exclusion), ('\u{12470}', '\u{12473}',
         IdentifierType::Exclusion), ('\u{12474}', '\u{12474}',  IdentifierType::Exclusion),
        ('\u{12480}', '\u{12543}',  IdentifierType::Exclusion), ('\u{13000}', '\u{1342e}',
         IdentifierType::Exclusion), ('\u{13430}', '\u{13438}',  IdentifierType::Exclusion),
        ('\u{14400}', '\u{14646}',  IdentifierType::Exclusion), ('\u{16800}', '\u{16a38}',
         IdentifierType::Limited_Use), ('\u{16a40}', '\u{16a5e}',  IdentifierType::Uncommon_Use),
        ('\u{16a60}', '\u{16a69}',  IdentifierType::Uncommon_Use), ('\u{16a6e}', '\u{16a6f}',
         IdentifierType::Exclusion), ('\u{16ad0}', '\u{16aed}',  IdentifierType::Exclusion),
        ('\u{16af0}', '\u{16af4}',  IdentifierType::Exclusion), ('\u{16af5}', '\u{16af5}',
         IdentifierType::Exclusion), ('\u{16b00}', '\u{16b36}',  IdentifierType::Exclusion),
        ('\u{16b37}', '\u{16b3f}',  IdentifierType::Exclusion), ('\u{16b40}', '\u{16b43}',
         IdentifierType::Exclusion), ('\u{16b44}', '\u{16b45}',  IdentifierType::Exclusion),
        ('\u{16b50}', '\u{16b59}',  IdentifierType::Exclusion), ('\u{16b5b}', '\u{16b61}',
         IdentifierType::Exclusion), ('\u{16b63}', '\u{16b77}',  IdentifierType::Exclusion),
        ('\u{16b7d}', '\u{16b8f}',  IdentifierType::Exclusion), ('\u{16e40}', '\u{16e7f}',
         IdentifierType::Exclusion), ('\u{16e80}', '\u{16e9a}',  IdentifierType::Exclusion),
        ('\u{16f00}', '\u{16f44}',  IdentifierType::Limited_Use), ('\u{16f45}', '\u{16f4a}',
         IdentifierType::Limited_Use), ('\u{16f4f}', '\u{16f4f}',  IdentifierType::Limited_Use),
        ('\u{16f50}', '\u{16f7e}',  IdentifierType::Limited_Use), ('\u{16f7f}', '\u{16f87}',
         IdentifierType::Limited_Use), ('\u{16f8f}', '\u{16f9f}',  IdentifierType::Limited_Use),
        ('\u{16fe0}', '\u{16fe0}',  IdentifierType::Exclusion), ('\u{16fe1}', '\u{16fe1}',
         IdentifierType::Exclusion), ('\u{16fe2}', '\u{16fe2}',  IdentifierType::Not_XID),
        ('\u{16fe3}', '\u{16fe3}',  IdentifierType::Obsolete), ('\u{17000}', '\u{187ec}',
         IdentifierType::Exclusion), ('\u{187ed}', '\u{187f1}',  IdentifierType::Exclusion),
        ('\u{187f2}', '\u{187f7}',  IdentifierType::Exclusion), ('\u{18800}', '\u{18af2}',
         IdentifierType::Exclusion), ('\u{1b000}', '\u{1b001}',  IdentifierType::Obsolete),
        ('\u{1b002}', '\u{1b11e}',  IdentifierType::Obsolete), ('\u{1b150}', '\u{1b152}',
         IdentifierType::Recommended), ('\u{1b164}', '\u{1b167}',  IdentifierType::Recommended),
        ('\u{1b170}', '\u{1b2fb}',  IdentifierType::Exclusion), ('\u{1bc00}', '\u{1bc6a}',
         IdentifierType::Exclusion), ('\u{1bc70}', '\u{1bc7c}',  IdentifierType::Exclusion),
        ('\u{1bc80}', '\u{1bc88}',  IdentifierType::Exclusion), ('\u{1bc90}', '\u{1bc99}',
         IdentifierType::Exclusion), ('\u{1bc9c}', '\u{1bc9c}',  IdentifierType::Exclusion),
        ('\u{1bc9d}', '\u{1bc9e}',  IdentifierType::Exclusion), ('\u{1bc9f}', '\u{1bc9f}',
         IdentifierType::Exclusion), ('\u{1bca0}', '\u{1bca3}',  IdentifierType::Exclusion),
        ('\u{1d000}', '\u{1d0f5}',  IdentifierType::Technical), ('\u{1d100}', '\u{1d126}',
         IdentifierType::Technical), ('\u{1d129}', '\u{1d129}',  IdentifierType::Technical),
        ('\u{1d12a}', '\u{1d15d}',  IdentifierType::Technical), ('\u{1d15e}', '\u{1d164}',
         IdentifierType::Technical), ('\u{1d165}', '\u{1d169}',  IdentifierType::Technical),
        ('\u{1d16a}', '\u{1d16c}',  IdentifierType::Technical), ('\u{1d16d}', '\u{1d172}',
         IdentifierType::Technical), ('\u{1d173}', '\u{1d17a}',  IdentifierType::Technical),
        ('\u{1d17b}', '\u{1d182}',  IdentifierType::Technical), ('\u{1d183}', '\u{1d184}',
         IdentifierType::Technical), ('\u{1d185}', '\u{1d18b}',  IdentifierType::Technical),
        ('\u{1d18c}', '\u{1d1a9}',  IdentifierType::Technical), ('\u{1d1aa}', '\u{1d1ad}',
         IdentifierType::Technical), ('\u{1d1ae}', '\u{1d1ba}',  IdentifierType::Technical),
        ('\u{1d1bb}', '\u{1d1c0}',  IdentifierType::Technical), ('\u{1d1c1}', '\u{1d1dd}',
         IdentifierType::Technical), ('\u{1d1de}', '\u{1d1e8}',  IdentifierType::Uncommon_Use),
        ('\u{1d200}', '\u{1d241}',  IdentifierType::Exclusion), ('\u{1d242}', '\u{1d244}',
         IdentifierType::Technical), ('\u{1d245}', '\u{1d245}',  IdentifierType::Exclusion),
        ('\u{1d2e0}', '\u{1d2f3}',  IdentifierType::Not_XID), ('\u{1d300}', '\u{1d356}',
         IdentifierType::Technical), ('\u{1d360}', '\u{1d371}',  IdentifierType::Not_XID),
        ('\u{1d372}', '\u{1d378}',  IdentifierType::Not_XID), ('\u{1d400}', '\u{1d454}',
         IdentifierType::Technical), ('\u{1d456}', '\u{1d49c}',  IdentifierType::Technical),
        ('\u{1d49e}', '\u{1d49f}',  IdentifierType::Technical), ('\u{1d4a2}', '\u{1d4a2}',
         IdentifierType::Technical), ('\u{1d4a5}', '\u{1d4a6}',  IdentifierType::Technical),
        ('\u{1d4a9}', '\u{1d4ac}',  IdentifierType::Technical), ('\u{1d4ae}', '\u{1d4b9}',
         IdentifierType::Technical), ('\u{1d4bb}', '\u{1d4bb}',  IdentifierType::Technical),
        ('\u{1d4bd}', '\u{1d4c0}',  IdentifierType::Technical), ('\u{1d4c1}', '\u{1d4c1}',
         IdentifierType::Technical), ('\u{1d4c2}', '\u{1d4c3}',  IdentifierType::Technical),
        ('\u{1d4c5}', '\u{1d505}',  IdentifierType::Technical), ('\u{1d507}', '\u{1d50a}',
         IdentifierType::Technical), ('\u{1d50d}', '\u{1d514}',  IdentifierType::Technical),
        ('\u{1d516}', '\u{1d51c}',  IdentifierType::Technical), ('\u{1d51e}', '\u{1d539}',
         IdentifierType::Technical), ('\u{1d53b}', '\u{1d53e}',  IdentifierType::Technical),
        ('\u{1d540}', '\u{1d544}',  IdentifierType::Technical), ('\u{1d546}', '\u{1d546}',
         IdentifierType::Technical), ('\u{1d54a}', '\u{1d550}',  IdentifierType::Technical),
        ('\u{1d552}', '\u{1d6a3}',  IdentifierType::Technical), ('\u{1d6a4}', '\u{1d6a5}',
         IdentifierType::Technical), ('\u{1d6a8}', '\u{1d6c0}',  IdentifierType::Technical),
        ('\u{1d6c1}', '\u{1d6c1}',  IdentifierType::Technical), ('\u{1d6c2}', '\u{1d6da}',
         IdentifierType::Technical), ('\u{1d6db}', '\u{1d6db}',  IdentifierType::Technical),
        ('\u{1d6dc}', '\u{1d6fa}',  IdentifierType::Technical), ('\u{1d6fb}', '\u{1d6fb}',
         IdentifierType::Technical), ('\u{1d6fc}', '\u{1d714}',  IdentifierType::Technical),
        ('\u{1d715}', '\u{1d715}',  IdentifierType::Technical), ('\u{1d716}', '\u{1d734}',
         IdentifierType::Technical), ('\u{1d735}', '\u{1d735}',  IdentifierType::Technical),
        ('\u{1d736}', '\u{1d74e}',  IdentifierType::Technical), ('\u{1d74f}', '\u{1d74f}',
         IdentifierType::Technical), ('\u{1d750}', '\u{1d76e}',  IdentifierType::Technical),
        ('\u{1d76f}', '\u{1d76f}',  IdentifierType::Technical), ('\u{1d770}', '\u{1d788}',
         IdentifierType::Technical), ('\u{1d789}', '\u{1d789}',  IdentifierType::Technical),
        ('\u{1d78a}', '\u{1d7a8}',  IdentifierType::Technical), ('\u{1d7a9}', '\u{1d7a9}',
         IdentifierType::Technical), ('\u{1d7aa}', '\u{1d7c2}',  IdentifierType::Technical),
        ('\u{1d7c3}', '\u{1d7c3}',  IdentifierType::Technical), ('\u{1d7c4}', '\u{1d7c9}',
         IdentifierType::Technical), ('\u{1d7ca}', '\u{1d7cb}',  IdentifierType::Technical),
        ('\u{1d7ce}', '\u{1d7ff}',  IdentifierType::Technical), ('\u{1d800}', '\u{1d9ff}',
         IdentifierType::Exclusion), ('\u{1da00}', '\u{1da36}',  IdentifierType::Exclusion),
        ('\u{1da37}', '\u{1da3a}',  IdentifierType::Exclusion), ('\u{1da3b}', '\u{1da6c}',
         IdentifierType::Exclusion), ('\u{1da6d}', '\u{1da74}',  IdentifierType::Exclusion),
        ('\u{1da75}', '\u{1da75}',  IdentifierType::Exclusion), ('\u{1da76}', '\u{1da83}',
         IdentifierType::Exclusion), ('\u{1da84}', '\u{1da84}',  IdentifierType::Exclusion),
        ('\u{1da85}', '\u{1da8b}',  IdentifierType::Exclusion), ('\u{1da9b}', '\u{1da9f}',
         IdentifierType::Exclusion), ('\u{1daa1}', '\u{1daaf}',  IdentifierType::Exclusion),
        ('\u{1e000}', '\u{1e006}',  IdentifierType::Exclusion), ('\u{1e008}', '\u{1e018}',
         IdentifierType::Exclusion), ('\u{1e01b}', '\u{1e021}',  IdentifierType::Exclusion),
        ('\u{1e023}', '\u{1e024}',  IdentifierType::Exclusion), ('\u{1e026}', '\u{1e02a}',
         IdentifierType::Exclusion), ('\u{1e100}', '\u{1e12c}',  IdentifierType::Limited_Use),
        ('\u{1e130}', '\u{1e13d}',  IdentifierType::Limited_Use), ('\u{1e140}', '\u{1e149}',
         IdentifierType::Limited_Use), ('\u{1e14e}', '\u{1e14e}',  IdentifierType::Limited_Use),
        ('\u{1e14f}', '\u{1e14f}',  IdentifierType::Limited_Use), ('\u{1e2c0}', '\u{1e2f9}',
         IdentifierType::Limited_Use), ('\u{1e2ff}', '\u{1e2ff}',  IdentifierType::Limited_Use),
        ('\u{1e800}', '\u{1e8c4}',  IdentifierType::Exclusion), ('\u{1e8c7}', '\u{1e8cf}',
         IdentifierType::Exclusion), ('\u{1e8d0}', '\u{1e8d6}',  IdentifierType::Exclusion),
        ('\u{1e900}', '\u{1e94a}',  IdentifierType::Limited_Use), ('\u{1e94b}', '\u{1e94b}',
         IdentifierType::Limited_Use), ('\u{1e950}', '\u{1e959}',  IdentifierType::Limited_Use),
        ('\u{1e95e}', '\u{1e95f}',  IdentifierType::Limited_Use), ('\u{1ec71}', '\u{1ecb4}',
         IdentifierType::Not_XID), ('\u{1ed01}', '\u{1ed3d}',  IdentifierType::Not_XID),
        ('\u{1ee00}', '\u{1ee03}',  IdentifierType::Not_NFKC), ('\u{1ee05}', '\u{1ee1f}',
         IdentifierType::Not_NFKC), ('\u{1ee21}', '\u{1ee22}',  IdentifierType::Not_NFKC),
        ('\u{1ee24}', '\u{1ee24}',  IdentifierType::Not_NFKC), ('\u{1ee27}', '\u{1ee27}',
         IdentifierType::Not_NFKC), ('\u{1ee29}', '\u{1ee32}',  IdentifierType::Not_NFKC),
        ('\u{1ee34}', '\u{1ee37}',  IdentifierType::Not_NFKC), ('\u{1ee39}', '\u{1ee39}',
         IdentifierType::Not_NFKC), ('\u{1ee3b}', '\u{1ee3b}',  IdentifierType::Not_NFKC),
        ('\u{1ee42}', '\u{1ee42}',  IdentifierType::Not_NFKC), ('\u{1ee47}', '\u{1ee47}',
         IdentifierType::Not_NFKC), ('\u{1ee49}', '\u{1ee49}',  IdentifierType::Not_NFKC),
        ('\u{1ee4b}', '\u{1ee4b}',  IdentifierType::Not_NFKC), ('\u{1ee4d}', '\u{1ee4f}',
         IdentifierType::Not_NFKC), ('\u{1ee51}', '\u{1ee52}',  IdentifierType::Not_NFKC),
        ('\u{1ee54}', '\u{1ee54}',  IdentifierType::Not_NFKC), ('\u{1ee57}', '\u{1ee57}',
         IdentifierType::Not_NFKC), ('\u{1ee59}', '\u{1ee59}',  IdentifierType::Not_NFKC),
        ('\u{1ee5b}', '\u{1ee5b}',  IdentifierType::Not_NFKC), ('\u{1ee5d}', '\u{1ee5d}',
         IdentifierType::Not_NFKC), ('\u{1ee5f}', '\u{1ee5f}',  IdentifierType::Not_NFKC),
        ('\u{1ee61}', '\u{1ee62}',  IdentifierType::Not_NFKC), ('\u{1ee64}', '\u{1ee64}',
         IdentifierType::Not_NFKC), ('\u{1ee67}', '\u{1ee6a}',  IdentifierType::Not_NFKC),
        ('\u{1ee6c}', '\u{1ee72}',  IdentifierType::Not_NFKC), ('\u{1ee74}', '\u{1ee77}',
         IdentifierType::Not_NFKC), ('\u{1ee79}', '\u{1ee7c}',  IdentifierType::Not_NFKC),
        ('\u{1ee7e}', '\u{1ee7e}',  IdentifierType::Not_NFKC), ('\u{1ee80}', '\u{1ee89}',
         IdentifierType::Not_NFKC), ('\u{1ee8b}', '\u{1ee9b}',  IdentifierType::Not_NFKC),
        ('\u{1eea1}', '\u{1eea3}',  IdentifierType::Not_NFKC), ('\u{1eea5}', '\u{1eea9}',
         IdentifierType::Not_NFKC), ('\u{1eeab}', '\u{1eebb}',  IdentifierType::Not_NFKC),
        ('\u{1eef0}', '\u{1eef1}',  IdentifierType::Not_XID), ('\u{1f000}', '\u{1f02b}',
         IdentifierType::Not_XID), ('\u{1f030}', '\u{1f093}',  IdentifierType::Not_XID),
        ('\u{1f0a0}', '\u{1f0ae}',  IdentifierType::Not_XID), ('\u{1f0b1}', '\u{1f0be}',
         IdentifierType::Not_XID), ('\u{1f0bf}', '\u{1f0bf}',  IdentifierType::Not_XID),
        ('\u{1f0c1}', '\u{1f0cf}',  IdentifierType::Not_XID), ('\u{1f0d1}', '\u{1f0df}',
         IdentifierType::Not_XID), ('\u{1f0e0}', '\u{1f0f5}',  IdentifierType::Not_XID),
        ('\u{1f100}', '\u{1f10a}',  IdentifierType::Not_XID), ('\u{1f10b}', '\u{1f10c}',
         IdentifierType::Not_XID), ('\u{1f110}', '\u{1f12e}',  IdentifierType::Not_XID),
        ('\u{1f12f}', '\u{1f12f}',  IdentifierType::Not_XID), ('\u{1f130}', '\u{1f130}',
         IdentifierType::Not_XID), ('\u{1f131}', '\u{1f131}',  IdentifierType::Not_XID),
        ('\u{1f132}', '\u{1f13c}',  IdentifierType::Not_XID), ('\u{1f13d}', '\u{1f13d}',
         IdentifierType::Not_XID), ('\u{1f13e}', '\u{1f13e}',  IdentifierType::Not_XID),
        ('\u{1f13f}', '\u{1f13f}',  IdentifierType::Not_XID), ('\u{1f140}', '\u{1f141}',
         IdentifierType::Not_XID), ('\u{1f142}', '\u{1f142}',  IdentifierType::Not_XID),
        ('\u{1f143}', '\u{1f145}',  IdentifierType::Not_XID), ('\u{1f146}', '\u{1f146}',
         IdentifierType::Not_XID), ('\u{1f147}', '\u{1f149}',  IdentifierType::Not_XID),
        ('\u{1f14a}', '\u{1f14e}',  IdentifierType::Not_XID), ('\u{1f14f}', '\u{1f14f}',
         IdentifierType::Not_XID), ('\u{1f150}', '\u{1f156}',  IdentifierType::Not_XID),
        ('\u{1f157}', '\u{1f157}',  IdentifierType::Not_XID), ('\u{1f158}', '\u{1f15e}',
         IdentifierType::Not_XID), ('\u{1f15f}', '\u{1f15f}',  IdentifierType::Not_XID),
        ('\u{1f160}', '\u{1f169}',  IdentifierType::Not_XID), ('\u{1f16a}', '\u{1f16b}',
         IdentifierType::Not_XID), ('\u{1f16c}', '\u{1f16c}',  IdentifierType::Not_XID),
        ('\u{1f170}', '\u{1f178}',  IdentifierType::Not_XID), ('\u{1f179}', '\u{1f179}',
         IdentifierType::Not_XID), ('\u{1f17a}', '\u{1f17a}',  IdentifierType::Not_XID),
        ('\u{1f17b}', '\u{1f17c}',  IdentifierType::Not_XID), ('\u{1f17d}', '\u{1f17e}',
         IdentifierType::Not_XID), ('\u{1f17f}', '\u{1f17f}',  IdentifierType::Not_XID),
        ('\u{1f180}', '\u{1f189}',  IdentifierType::Not_XID), ('\u{1f18a}', '\u{1f18d}',
         IdentifierType::Not_XID), ('\u{1f18e}', '\u{1f18f}',  IdentifierType::Not_XID),
        ('\u{1f190}', '\u{1f190}',  IdentifierType::Not_XID), ('\u{1f191}', '\u{1f19a}',
         IdentifierType::Not_XID), ('\u{1f19b}', '\u{1f1ac}',  IdentifierType::Not_XID),
        ('\u{1f1e6}', '\u{1f1ff}',  IdentifierType::Not_XID), ('\u{1f200}', '\u{1f200}',
         IdentifierType::Not_XID), ('\u{1f201}', '\u{1f202}',  IdentifierType::Not_XID),
        ('\u{1f210}', '\u{1f231}',  IdentifierType::Not_XID), ('\u{1f232}', '\u{1f23a}',
         IdentifierType::Not_XID), ('\u{1f23b}', '\u{1f23b}',  IdentifierType::Not_XID),
        ('\u{1f240}', '\u{1f248}',  IdentifierType::Not_XID), ('\u{1f250}', '\u{1f251}',
         IdentifierType::Not_XID), ('\u{1f260}', '\u{1f265}',  IdentifierType::Not_XID),
        ('\u{1f300}', '\u{1f320}',  IdentifierType::Not_XID), ('\u{1f321}', '\u{1f32c}',
         IdentifierType::Not_XID), ('\u{1f32d}', '\u{1f32f}',  IdentifierType::Not_XID),
        ('\u{1f330}', '\u{1f335}',  IdentifierType::Not_XID), ('\u{1f336}', '\u{1f336}',
         IdentifierType::Not_XID), ('\u{1f337}', '\u{1f37c}',  IdentifierType::Not_XID),
        ('\u{1f37d}', '\u{1f37d}',  IdentifierType::Not_XID), ('\u{1f37e}', '\u{1f37f}',
         IdentifierType::Not_XID), ('\u{1f380}', '\u{1f393}',  IdentifierType::Not_XID),
        ('\u{1f394}', '\u{1f39f}',  IdentifierType::Not_XID), ('\u{1f3a0}', '\u{1f3c4}',
         IdentifierType::Not_XID), ('\u{1f3c5}', '\u{1f3c5}',  IdentifierType::Not_XID),
        ('\u{1f3c6}', '\u{1f3ca}',  IdentifierType::Not_XID), ('\u{1f3cb}', '\u{1f3ce}',
         IdentifierType::Not_XID), ('\u{1f3cf}', '\u{1f3d3}',  IdentifierType::Not_XID),
        ('\u{1f3d4}', '\u{1f3df}',  IdentifierType::Not_XID), ('\u{1f3e0}', '\u{1f3f0}',
         IdentifierType::Not_XID), ('\u{1f3f1}', '\u{1f3f7}',  IdentifierType::Not_XID),
        ('\u{1f3f8}', '\u{1f3ff}',  IdentifierType::Not_XID), ('\u{1f400}', '\u{1f43e}',
         IdentifierType::Not_XID), ('\u{1f43f}', '\u{1f43f}',  IdentifierType::Not_XID),
        ('\u{1f440}', '\u{1f440}',  IdentifierType::Not_XID), ('\u{1f441}', '\u{1f441}',
         IdentifierType::Not_XID), ('\u{1f442}', '\u{1f4f7}',  IdentifierType::Not_XID),
        ('\u{1f4f8}', '\u{1f4f8}',  IdentifierType::Not_XID), ('\u{1f4f9}', '\u{1f4fc}',
         IdentifierType::Not_XID), ('\u{1f4fd}', '\u{1f4fe}',  IdentifierType::Not_XID),
        ('\u{1f4ff}', '\u{1f4ff}',  IdentifierType::Not_XID), ('\u{1f500}', '\u{1f53d}',
         IdentifierType::Not_XID), ('\u{1f53e}', '\u{1f53f}',  IdentifierType::Not_XID),
        ('\u{1f540}', '\u{1f543}',  IdentifierType::Not_XID), ('\u{1f544}', '\u{1f54a}',
         IdentifierType::Not_XID), ('\u{1f54b}', '\u{1f54e}',  IdentifierType::Not_XID),
        ('\u{1f54f}', '\u{1f54f}',  IdentifierType::Uncommon_Use), ('\u{1f550}', '\u{1f567}',
         IdentifierType::Not_XID), ('\u{1f568}', '\u{1f579}',  IdentifierType::Not_XID),
        ('\u{1f57a}', '\u{1f57a}',  IdentifierType::Not_XID), ('\u{1f57b}', '\u{1f5a3}',
         IdentifierType::Not_XID), ('\u{1f5a4}', '\u{1f5a4}',  IdentifierType::Not_XID),
        ('\u{1f5a5}', '\u{1f5fa}',  IdentifierType::Not_XID), ('\u{1f5fb}', '\u{1f5ff}',
         IdentifierType::Not_XID), ('\u{1f600}', '\u{1f600}',  IdentifierType::Not_XID),
        ('\u{1f601}', '\u{1f610}',  IdentifierType::Not_XID), ('\u{1f611}', '\u{1f611}',
         IdentifierType::Not_XID), ('\u{1f612}', '\u{1f614}',  IdentifierType::Not_XID),
        ('\u{1f615}', '\u{1f615}',  IdentifierType::Not_XID), ('\u{1f616}', '\u{1f616}',
         IdentifierType::Not_XID), ('\u{1f617}', '\u{1f617}',  IdentifierType::Not_XID),
        ('\u{1f618}', '\u{1f618}',  IdentifierType::Not_XID), ('\u{1f619}', '\u{1f619}',
         IdentifierType::Not_XID), ('\u{1f61a}', '\u{1f61a}',  IdentifierType::Not_XID),
        ('\u{1f61b}', '\u{1f61b}',  IdentifierType::Not_XID), ('\u{1f61c}', '\u{1f61e}',
         IdentifierType::Not_XID), ('\u{1f61f}', '\u{1f61f}',  IdentifierType::Not_XID),
        ('\u{1f620}', '\u{1f625}',  IdentifierType::Not_XID), ('\u{1f626}', '\u{1f627}',
         IdentifierType::Not_XID), ('\u{1f628}', '\u{1f62b}',  IdentifierType::Not_XID),
        ('\u{1f62c}', '\u{1f62c}',  IdentifierType::Not_XID), ('\u{1f62d}', '\u{1f62d}',
         IdentifierType::Not_XID), ('\u{1f62e}', '\u{1f62f}',  IdentifierType::Not_XID),
        ('\u{1f630}', '\u{1f633}',  IdentifierType::Not_XID), ('\u{1f634}', '\u{1f634}',
         IdentifierType::Not_XID), ('\u{1f635}', '\u{1f640}',  IdentifierType::Not_XID),
        ('\u{1f641}', '\u{1f642}',  IdentifierType::Not_XID), ('\u{1f643}', '\u{1f644}',
         IdentifierType::Not_XID), ('\u{1f645}', '\u{1f64f}',  IdentifierType::Not_XID),
        ('\u{1f650}', '\u{1f67f}',  IdentifierType::Not_XID), ('\u{1f680}', '\u{1f6c5}',
         IdentifierType::Not_XID), ('\u{1f6c6}', '\u{1f6cf}',  IdentifierType::Not_XID),
        ('\u{1f6d0}', '\u{1f6d0}',  IdentifierType::Not_XID), ('\u{1f6d1}', '\u{1f6d2}',
         IdentifierType::Not_XID), ('\u{1f6d3}', '\u{1f6d4}',  IdentifierType::Not_XID),
        ('\u{1f6d5}', '\u{1f6d5}',  IdentifierType::Not_XID), ('\u{1f6e0}', '\u{1f6ec}',
         IdentifierType::Not_XID), ('\u{1f6f0}', '\u{1f6f3}',  IdentifierType::Not_XID),
        ('\u{1f6f4}', '\u{1f6f6}',  IdentifierType::Not_XID), ('\u{1f6f7}', '\u{1f6f8}',
         IdentifierType::Not_XID), ('\u{1f6f9}', '\u{1f6f9}',  IdentifierType::Not_XID),
        ('\u{1f6fa}', '\u{1f6fa}',  IdentifierType::Not_XID), ('\u{1f700}', '\u{1f773}',
         IdentifierType::Not_XID), ('\u{1f780}', '\u{1f7d4}',  IdentifierType::Not_XID),
        ('\u{1f7d5}', '\u{1f7d8}',  IdentifierType::Not_XID), ('\u{1f7e0}', '\u{1f7eb}',
         IdentifierType::Not_XID), ('\u{1f800}', '\u{1f80b}',  IdentifierType::Not_XID),
        ('\u{1f810}', '\u{1f847}',  IdentifierType::Not_XID), ('\u{1f850}', '\u{1f859}',
         IdentifierType::Not_XID), ('\u{1f860}', '\u{1f887}',  IdentifierType::Not_XID),
        ('\u{1f890}', '\u{1f8ad}',  IdentifierType::Not_XID), ('\u{1f900}', '\u{1f90b}',
         IdentifierType::Not_XID), ('\u{1f90d}', '\u{1f90f}',  IdentifierType::Not_XID),
        ('\u{1f910}', '\u{1f918}',  IdentifierType::Not_XID), ('\u{1f919}', '\u{1f91e}',
         IdentifierType::Not_XID), ('\u{1f91f}', '\u{1f91f}',  IdentifierType::Not_XID),
        ('\u{1f920}', '\u{1f927}',  IdentifierType::Not_XID), ('\u{1f928}', '\u{1f92f}',
         IdentifierType::Not_XID), ('\u{1f930}', '\u{1f930}',  IdentifierType::Not_XID),
        ('\u{1f931}', '\u{1f932}',  IdentifierType::Not_XID), ('\u{1f933}', '\u{1f93e}',
         IdentifierType::Not_XID), ('\u{1f93f}', '\u{1f93f}',  IdentifierType::Not_XID),
        ('\u{1f940}', '\u{1f94b}',  IdentifierType::Not_XID), ('\u{1f94c}', '\u{1f94c}',
         IdentifierType::Not_XID), ('\u{1f94d}', '\u{1f94f}',  IdentifierType::Not_XID),
        ('\u{1f950}', '\u{1f95e}',  IdentifierType::Not_XID), ('\u{1f95f}', '\u{1f96b}',
         IdentifierType::Not_XID), ('\u{1f96c}', '\u{1f970}',  IdentifierType::Not_XID),
        ('\u{1f971}', '\u{1f971}',  IdentifierType::Not_XID), ('\u{1f973}', '\u{1f976}',
         IdentifierType::Not_XID), ('\u{1f97a}', '\u{1f97a}',  IdentifierType::Not_XID),
        ('\u{1f97b}', '\u{1f97b}',  IdentifierType::Not_XID), ('\u{1f97c}', '\u{1f97f}',
         IdentifierType::Not_XID), ('\u{1f980}', '\u{1f984}',  IdentifierType::Not_XID),
        ('\u{1f985}', '\u{1f991}',  IdentifierType::Not_XID), ('\u{1f992}', '\u{1f997}',
         IdentifierType::Not_XID), ('\u{1f998}', '\u{1f9a2}',  IdentifierType::Not_XID),
        ('\u{1f9a5}', '\u{1f9aa}',  IdentifierType::Not_XID), ('\u{1f9ae}', '\u{1f9af}',
         IdentifierType::Not_XID), ('\u{1f9b0}', '\u{1f9b9}',  IdentifierType::Not_XID),
        ('\u{1f9ba}', '\u{1f9bf}',  IdentifierType::Not_XID), ('\u{1f9c0}', '\u{1f9c0}',
         IdentifierType::Not_XID), ('\u{1f9c1}', '\u{1f9c2}',  IdentifierType::Not_XID),
        ('\u{1f9c3}', '\u{1f9ca}',  IdentifierType::Not_XID), ('\u{1f9cd}', '\u{1f9cf}',
         IdentifierType::Not_XID), ('\u{1f9d0}', '\u{1f9e6}',  IdentifierType::Not_XID),
        ('\u{1f9e7}', '\u{1f9ff}',  IdentifierType::Not_XID), ('\u{1fa00}', '\u{1fa53}',
         IdentifierType::Not_XID), ('\u{1fa60}', '\u{1fa6d}',  IdentifierType::Not_XID),
        ('\u{1fa70}', '\u{1fa73}',  IdentifierType::Not_XID), ('\u{1fa78}', '\u{1fa7a}',
         IdentifierType::Not_XID), ('\u{1fa80}', '\u{1fa82}',  IdentifierType::Not_XID),
        ('\u{1fa90}', '\u{1fa95}',  IdentifierType::Not_XID), ('\u{20000}', '\u{2a6d6}',
         IdentifierType::Recommended), ('\u{2a700}', '\u{2b734}',  IdentifierType::Recommended),
        ('\u{2b740}', '\u{2b81d}',  IdentifierType::Recommended), ('\u{2b820}', '\u{2cea1}',
         IdentifierType::Recommended), ('\u{2ceb0}', '\u{2ebe0}',  IdentifierType::Recommended),
        ('\u{2f800}', '\u{2fa1d}',  IdentifierType::Not_NFKC), ('\u{e0001}', '\u{e0001}',
         IdentifierType::Exclusion), ('\u{e0020}', '\u{e007f}',  IdentifierType::Exclusion),
        ('\u{e0100}', '\u{e01ef}',  IdentifierType::Exclusion)
    ];

}

