\docType{methods}
\name{traceplot}
\alias{traceplot} 
\alias{traceplot,stanfit-method}

\title{Markov chain traceplots}
\description{Draw the traceplot corresponding to one or more
 Markov chains, providing a visual way to inspect sampling behavior
 and assess mixing across chains and convergence.}

\usage{
  %% traceplot(object, \dots)  
  \S4method{traceplot}{stanfit}(object, pars, include = TRUE, unconstrain = FALSE, 
          inc_warmup = FALSE, window = NULL, nrow = NULL, ncol = NULL, ...) 
} 

\section{Methods}{
  \describe{
    \item{traceplot}{\code{signature(object = "stanfit")} Plot the sampling paths for all chains.}
  }
}

\arguments{
  \item{object}{An instance of class \code{\linkS4class{stanfit}}.}
  \item{pars}{A character vector of parameter names. Defaults to all parameters 
    or the first 10 parameters (if there are more than 10).}
  \item{include}{Should the parameters given by the \code{pars} argument be
    included (the default) or excluded from the plot? Only relevant if 
    \code{pars} is not missing.}
  \item{inc_warmup}{\code{TRUE} or \code{FALSE}, indicating whether 
    the warmup sample are included in the trace plot; defaults to \code{FALSE}.}
  \item{window}{A vector of length 2. Iterations between \code{window[1]}
    and \code{window[2]} will be shown in the plot. The default is to 
    show all iterations if \code{inc_warmup} is \code{TRUE} and all iterations
    from the sampling period only if \code{inc_warmup} is \code{FALSE}. If 
    \code{inc_warmup} is \code{FALSE} the iterations specified in \code{window}
    should not include iterations from the warmup period.}
  \item{unconstrain}{Should parameters be plotted on the unconstrained space?
    Defaults to \code{FALSE}.}  
  \item{nrow,ncol}{Passed to \code{\link[ggplot2]{facet_wrap}}.} 
  \item{\dots}{Optional arguments to pass to \code{\link[ggplot2]{geom_path}} 
    (e.g. \code{size}, \code{linetype}, \code{alpha}, etc.).} 
}
\value{A \code{\link[ggplot2]{ggplot}} object that can be further customized
  using the \pkg{ggplot2} package.} 

\seealso{
\code{\link[=rstan-plotting-functions]{List of RStan plotting functions}},
\code{\link[=rstan_gg_options]{Plot options}}
}

\examples{\dontrun{
# Create a stanfit object from reading CSV files of samples (saved in rstan
# package) generated by funtion stan for demonstration purpose from model as follows. 
# 
excode <- '
  transformed data {
    array[20] real y;
    y[1] <- 0.5796;  y[2]  <- 0.2276;   y[3] <- -0.2959; 
    y[4] <- -0.3742; y[5]  <- 0.3885;   y[6] <- -2.1585;
    y[7] <- 0.7111;  y[8]  <- 1.4424;   y[9] <- 2.5430; 
    y[10] <- 0.3746; y[11] <- 0.4773;   y[12] <- 0.1803; 
    y[13] <- 0.5215; y[14] <- -1.6044;  y[15] <- -0.6703; 
    y[16] <- 0.9459; y[17] <- -0.382;   y[18] <- 0.7619;
    y[19] <- 0.1006; y[20] <- -1.7461;
  }
  parameters {
    real mu;
    real<lower=0, upper=10> sigma;
    vector[2] z[3];
    real<lower=0> alpha;
  } 
  model {
    y ~ normal(mu, sigma);
    for (i in 1:3) 
      z[i] ~ normal(0, 1);
    alpha ~ exponential(2);
  } 
'
# exfit <- stan(model_code = excode, save_dso = FALSE, iter = 200, 
#               sample_file = "rstan_doc_ex.csv")
# 
exfit <- read_stan_csv(dir(system.file('misc', package = 'rstan'),
                       pattern='rstan_doc_ex_[[:digit:]].csv',
                       full.names = TRUE))

print(exfit)
traceplot(exfit)
traceplot(exfit, size = 0.25)
traceplot(exfit, pars = "sigma", inc_warmup = TRUE)

trace <- traceplot(exfit, pars = c("z[1,1]", "z[3,1]"))
trace + scale_color_discrete() + theme(legend.position = "top")
}
} 
\keyword{methods}
