% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transform.R
\name{transformCIBP}
\alias{transformCIBP}
\title{Compute Confidence Intervals/Bands and P-values After a Transformation}
\usage{
transformCIBP(
  estimate,
  se,
  iid,
  null,
  conf.level,
  alternative,
  ci,
  type,
  min.value,
  max.value,
  band,
  method.band,
  n.sim,
  seed,
  p.value,
  df = NULL
)
}
\arguments{
\item{estimate}{[numeric matrix] the estimate value before transformation.}

\item{se}{[numeric matrix] the standard error before transformation.}

\item{iid}{[numeric array] the iid decomposition before transformation.}

\item{null}{[numeric] the value of the estimate (before transformation) under the null hypothesis.}

\item{conf.level}{[numeric, 0-1] Level of confidence.}

\item{alternative}{[character] a character string specifying the alternative hypothesis,
must be one of \code{"two.sided"} (default), \code{"greater"} or \code{"less"}.}

\item{ci}{[logical] should confidence intervals be computed.}

\item{type}{[character] the transforamtion.
Can be \code{"log"}, \code{"loglog"}, \code{"cloglog"}, or \code{"atanh"} (Fisher transform), or \code{"atanh2"} (modified Fisher transform for [0-1] variable).}

\item{min.value}{[numeric] if not \code{NULL} and the lower bound of the confidence interval is below \code{min},
it will be set at \code{min}.}

\item{max.value}{[numeric] if not \code{NULL} and the lower bound of the confidence interval is below \code{max},
it will be set at \code{max}.}

\item{band}{[integer 0,1,2] When non-0, the confidence bands are computed for each contrasts (\code{band=1}) or over all contrasts (\code{band=2}).}

\item{method.band}{[character] method used to adjust for multiple comparisons.
Can be any element of \code{p.adjust.methods} (e.g. \code{"holm"}), \code{"maxT-integration"}, or \code{"maxT-simulation"}.}

\item{n.sim}{[integer, >0] the number of simulations used to compute the quantiles for the confidence bands.}

\item{seed}{[integer, >0] seed number set before performing simulations for the confidence bands.}

\item{p.value}{[logical] should p-values and adjusted p-values be computed. Only active if \code{ci=TRUE} or \code{band>0}.}

\item{df}{[integer, >0] optional. Degrees of freedom used for the student distribution of the test statistic. If not specified, use a normal distribution instead.}
}
\description{
Compute confidence intervals/bands and p-values after a transformation
}
\details{
The iid decomposition must have dimensions [n.obs,time,n.prediction]
while estimate and se must have dimensions [n.prediction,time].

Single step max adjustment for multiple comparisons, i.e. accounting for the correlation between the test statistics but not for the ordering of the tests, can be performed setting the arguemnt \code{method.band} to \code{"maxT-integration"} or \code{"maxT-simulation"}. The former uses numerical integration (\code{pmvnorm} and \code{qmvnorm} to perform the adjustment while the latter using simulation. Both assume that the test statistics are jointly normally distributed.
}
\examples{
set.seed(10)
n <- 100
X <- rnorm(n)

res2sided <- transformCIBP(estimate = mean(X), se = cbind(sd(X)/sqrt(n)), null = 0,
              type = "none", ci = TRUE, conf.level = 0.95, alternative = "two.sided",
              min.value = NULL, max.value = NULL, band = FALSE,
              p.value = TRUE, seed = 10, df = n-1)

resLess <- transformCIBP(estimate = mean(X), se = cbind(sd(X)/sqrt(n)), null = 0,
              type = "none", ci = TRUE, conf.level = 0.95, alternative = "less",
              min.value = NULL, max.value = NULL, band = FALSE,
              p.value = TRUE, seed = 10, df = n-1)

resGreater <- transformCIBP(estimate = mean(X), se = cbind(sd(X)/sqrt(n)), null = 0,
              type = "none", ci = TRUE, conf.level = 0.95, alternative = "greater",
              min.value = NULL, max.value = NULL, band = FALSE,
              p.value = TRUE, seed = 10, df = n-1)


## comparison with t-test
GS <- t.test(X, alternative = "two.sided")
res2sided$p.value - GS$p.value
unlist(res2sided[c("lower","upper")]) - GS$conf.int

GS <- t.test(X, alternative = "less")
resLess$p.value - GS$p.value
unlist(resLess[c("lower","upper")]) - GS$conf.int

GS <- t.test(X, alternative = "greater")
resGreater$p.value - GS$p.value
unlist(resGreater[c("lower","upper")]) - GS$conf.int

}
