/* --- BEGIN COPYRIGHT BLOCK ---
 * Copyright (C) 2015  Red Hat
 * see files 'COPYING' and 'COPYING.openssl' for use and warranty
 * information
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Additional permission under GPLv3 section 7:
 * 
 * If you modify this Program, or any covered work, by linking or
 * combining it with OpenSSL, or a modified version of OpenSSL licensed
 * under the OpenSSL license
 * (https://www.openssl.org/source/license.html), the licensors of this
 * Program grant you additional permission to convey the resulting
 * work. Corresponding Source for a non-source form of such a
 * combination shall include the source code for the parts that are
 * licensed under the OpenSSL license as well as that of the covered
 * work.
 * --- END COPYRIGHT BLOCK ---
 */
#include <string.h>
#include <unistd.h>
#include <plstr.h>
#include <syslog.h>

#include "test-protocol.h"

#define STATE_READING 1
#define STATE_WRITING 2

typedef struct ns_job_t ns_job_t;

struct conn_ctx {
    size_t offset; /* current offset into buffer for reading or writing */
    size_t len; /* size of buffer */
    size_t needed; /* content-length + start of body */
    size_t body; /* when reading, offset from buffer of beginning of http body */
    size_t cl; /* http content-length when reading */
#define CONN_BUFFER_SIZE BUFSIZ /* default buffer size */
    char *buffer;
    PRBool need_sec_layer_before_next_read;
    struct ns_sec_ctx_t *sc;
    PRBool state;
};

static void conn_read(ns_job_t *job);
static void conn_write(ns_job_t *job);
static const char header1[] = {"HTTP/1.1 200 OK\r\nDate: Fri, 31 Dec 1969 23:59:59 GMT\r\nContent-type:text/plain\r\nContent-Length:"};
static const char header2[] = {"\r\n\r\n"};
static size_t header1len = sizeof(header1) - 1; /* for null */
static size_t header2len = sizeof(header2) - 1; /* for null */
/* flags to apply to io events */
static int USE_POLL = 0;
/* test with poll() - see if poll and epoll can co-exist */
static ns_job_type_t readtype = NS_JOB_READ|NS_JOB_PERSIST;
static ns_job_type_t writetype = NS_JOB_WRITE|NS_JOB_PERSIST;
#define NS_POLL_READ_TIMEOUT 10000 /* for USE_POLL - 10 seconds */
#define NS_POLL_WRITE_TIMEOUT 10000 /* for USE_POLL - 10 seconds */
#define PR_WOULD_BLOCK(iii) (iii == PR_PENDING_INTERRUPT_ERROR) || (iii == PR_WOULD_BLOCK_ERROR)


static struct conn_ctx *
conn_ctx_new()
{
    struct conn_ctx *connctx = PR_NEW(struct conn_ctx);
    connctx->offset = 0;
    connctx->len = 0;
    connctx->buffer = NULL;
    connctx->needed = 0;
    connctx->body = 0;
    connctx->cl = 0;
    connctx->need_sec_layer_before_next_read = PR_FALSE;
    connctx->sc = NULL;
    connctx->state = STATE_READING; /* start off reading */
    return connctx;
}

static void
conn_ctx_reset(struct conn_ctx *connctx)
{
    connctx->offset = 0;
    connctx->needed = 0;
    connctx->body = 0;
    connctx->cl = 0;
    connctx->state = STATE_READING;
}

static void
conn_ctx_done(struct conn_ctx *connctx)
{
    free(connctx->buffer);
    connctx->buffer = NULL;
    memset(connctx, 0, sizeof(*connctx));
}

static void
conn_ctx_free(struct conn_ctx **connctx)
{
    conn_ctx_done(*connctx);
    PR_DELETE(*connctx);
}

static void
long_time_consuming_job(ns_job_t *job)
{
    struct conn_ctx *connctx;

    PR_ASSERT(job);
    PR_ASSERT(ns_job_get_data(job));
    connctx = (struct conn_ctx *)ns_job_get_data(job);

#ifdef TEST_WRITE_DELAY
    fprintf(stderr, "long_time_consuming_job: sleeping for 10 seconds . . .\n");
    sleep(10);
    fprintf(stderr, "long_time_consuming_job: finished - ready to write\n");
#endif
    /* we are ready to write - make the job persistent so we don't have
       to re-arm it - remove the thread flag so the io will occur
       in the event thread 
    */
    connctx->state = STATE_WRITING;
    ns_job_modify(job, writetype); /* execute write job */
}

static int
dopoll(ns_job_t *job, int isread)
{
    int ret = 0;
    struct PRPollDesc pr_pd;
    int msec = isread ? NS_POLL_READ_TIMEOUT : NS_POLL_WRITE_TIMEOUT;
    PRIntervalTime timeout = PR_MillisecondsToInterval(msec);

    pr_pd.fd = ns_job_get_fd(job);
    pr_pd.in_flags = isread ? PR_POLL_READ : PR_POLL_WRITE;
    pr_pd.out_flags = 0;
    ret = PR_Poll(&pr_pd, 1, timeout);
    if (ret == 0) { /* timeout */
        ret = -1;
        do_logging(LOG_WARNING, "dopoll: poll timeout job [%p]\n", job);
   } else if (ret < 0) { /* error */
        ret = -1;
        do_logging(LOG_ERR, "dopoll: poll error for job [%p] %d: %s\n", job,
                   PR_GetError(), PR_ErrorToString(PR_GetError(), PR_LANGUAGE_I_DEFAULT));
    } else { /* success */
        ret = 0;
    }
    return ret;
}

static void
conn_write(ns_job_t *job)
{
    struct conn_ctx *connctx;
    PRInt32 len;
    
    PR_ASSERT(job);
    PR_ASSERT(ns_job_get_data(job));
    connctx = (struct conn_ctx *)ns_job_get_data(job);
    if (NS_JOB_IS_TIMER(ns_job_get_output_type(job))) {
        /* we timed out waiting for the client - close */
        do_logging(LOG_INFO, "conn_write: job [%p] timeout\n", job);
        conn_ctx_free(&connctx);
        ns_job_done(job);
        return;
    }

retry:
    len = PR_Write(ns_job_get_fd(job), connctx->buffer+connctx->offset, connctx->needed);
    if (len < 0) {
        PRErrorCode prerr = PR_GetError();
        if (PR_WOULD_BLOCK(prerr)) {
            /* ok */
            /* reschedule write */
            if (!NS_JOB_IS_PERSIST(ns_job_get_type(job))) {
                ns_job_rearm(job);
            }
            if (USE_POLL) {
                if (0 == dopoll(job, 0)) {
                    goto retry;
                } else {
                    /* error or timeout */
                    conn_ctx_free(&connctx);
                    ns_job_done(job);
                }
            }
        } else {
            /* error */
            do_logging(LOG_ERR, "conn_write: write error for job [%p] %d: %s\n", job,
                       PR_GetError(), PR_ErrorToString(PR_GetError(), PR_LANGUAGE_I_DEFAULT));
            conn_ctx_free(&connctx);
            ns_job_done(job);
        }
    } else if (len == 0) {
        /* closed */
        do_logging(LOG_INFO, "conn_write: job [%p] closed\n", job);
        conn_ctx_free(&connctx);
        ns_job_done(job);
    } else {
        connctx->offset += len;
        connctx->needed -= len;
        if (connctx->needed) {
            /* reschedule write */
            if (!NS_JOB_IS_PERSIST(ns_job_get_type(job))) {
                ns_job_rearm(job);
            }
            if (USE_POLL) {
                if (0 == dopoll(job, 0)) {
                    goto retry;
                } else {
                    /* error or timeout */
                    conn_ctx_free(&connctx);
                    ns_job_done(job);
                }
            }
        } else if (0 == connctx->needed) {
            /* done - schedule read */
            do_logging(LOG_DEBUG, "conn_write: job [%p] wrote [%ld] bytes, schedule read\n",
                       job, connctx->offset);
            conn_ctx_reset(connctx);
            ns_job_modify(job, readtype);
        }
    }
}

static size_t
get_content_length_and_body_offset(struct conn_ctx *connctx)
{
    /* Parse the input, starting at the beginning of the buffer.
     * Stop when we detect two consecutive \n's (or \r\n's)
     * as this signifies the end of the headers.  Then, look
     * for the Content-Length in the headers and return the value.
     */
    size_t cl = 0;
    char *ptr;

    if (connctx->cl) {
        return connctx->cl; /* already have it */
    }

    if ((ptr = PL_strnstr(connctx->buffer, "\n\n", connctx->offset))) {
        ptr += 2;
    } else if ((ptr = PL_strnstr(connctx->buffer, "\r\n\r\n", connctx->offset))) {
        ptr += 4;
    } else {
        return cl; /* no body yet */
    }
    /* pos is now the offset of the beginning of the body */
    connctx->body = ptr - connctx->buffer;
    /* end of headers found - look for content-length */
#define CL_STRING "content-length:"
    if ((ptr = PL_strncasestr(connctx->buffer, CL_STRING, connctx->body))) {
        ptr += sizeof(CL_STRING) - 1; /* skip to first char after colon */
        while(*ptr == ' ') ++ptr; /* skip whitespace */
        cl = strtoul(ptr, NULL, 10); /* should be a number here */
        connctx->cl = cl;
    } else {
        do_logging(LOG_ERR, "get_content_length_and_body_offset: connctx [%p]: no content-length\n", connctx);
    }
    return cl;
}

/* it is assumed the reads are persistent, so
   no need to re-arm */
static void
conn_read(ns_job_t *job)
{
    struct conn_ctx *connctx;
    PRInt32 len, nbytes;
    
    PR_ASSERT(job);
    PR_ASSERT(ns_job_get_data(job));

    connctx = (struct conn_ctx *)ns_job_get_data(job);
    if (NS_JOB_IS_TIMER(ns_job_get_output_type(job))) {
        /* we timed out waiting for the client - close */
        do_logging(LOG_INFO, "conn_read: job [%p] timeout\n", job);
        conn_ctx_free(&connctx);
        ns_job_done(job);
        return;
    }

    /* do we need to add a security layer before the next read? */
    if (connctx->need_sec_layer_before_next_read) {
        connctx->need_sec_layer_before_next_read = PR_FALSE;
        PRErrorCode prerr = ns_add_sec_layer(ns_job_get_fd(job), connctx->sc);
        if (prerr != PR_SUCCESS) {
            /* handle error */
            conn_ctx_free(&connctx);
            ns_job_done(job);
            return;
        }
    }

retry:
    /* figure out how many bytes we need to read */
    if (connctx->needed) {
        nbytes = connctx->needed - connctx->offset;
    } else {
        nbytes = CONN_BUFFER_SIZE;
    }
    /* increase size of buffer if necessary */
    if ((nbytes + connctx->offset) > connctx->len) {
        connctx->len = nbytes + connctx->offset;
        connctx->buffer = (char *)realloc(connctx->buffer, connctx->len * sizeof(char));
    }
    len = PR_Read(ns_job_get_fd(job), connctx->buffer+connctx->offset, nbytes);
    if (len < 0) {
        PRErrorCode prerr = PR_GetError();
        if (PR_WOULD_BLOCK(prerr)) {
            /* need more data in socket */
            if (USE_POLL) {
                if (0 == dopoll(job, 1)) {
                    goto retry;
                } else {
                    /* error or timeout */
                    conn_ctx_free(&connctx);
                    ns_job_done(job);
                    return;
                }
            }
            /* reschedule read */
            if (!NS_JOB_IS_PERSIST(ns_job_get_type(job))) {
                ns_job_rearm(job);
            }
            return;
        } else {
            /* error */
            do_logging(LOG_ERR, "conn_read: read error for job [%p] %d: %s\n", job,
                       PR_GetError(), PR_ErrorToString(PR_GetError(), PR_LANGUAGE_I_DEFAULT));
            conn_ctx_free(&connctx);
            ns_job_done(job);
            return;
        }
    } else if (len == 0) {
        /* closed */
        do_logging(LOG_INFO, "conn_read: job [%p] closed\n", job);
        conn_ctx_free(&connctx);
        ns_job_done(job);
        return;
    } else {
        connctx->offset += len;
        if (get_content_length_and_body_offset(connctx)) {
            /* we have the content-length and the headers */
            if (!connctx->needed) {
                connctx->needed = connctx->body + connctx->cl;
            }
            /* see if we have all of the content as well */
            if (connctx->offset >= connctx->needed) {
                /* done - generate and write response */
                int offsetdiff;
                size_t numdigits;

                /* number of bytes in string rep of content length */
                numdigits = (connctx->cl >= 1000000000) ? 10 : (connctx->cl >= 100000000) ? 9 :
                    (connctx->cl >= 10000000) ? 8 : (connctx->cl >= 1000000) ? 7 :
                    (connctx->cl >= 100000) ? 6 : (connctx->cl >= 10000) ? 5 :
                    (connctx->cl >= 1000) ? 4 : (connctx->cl >= 100) ? 3 :
                    (connctx->cl >= 10) ? 2 : 1;
                /* number of bytes we need to write */
                connctx->needed = header1len + numdigits + header2len + connctx->cl;
                /* start at the body offset - work backwards - need space for header1 + numdigits + header2 */
                offsetdiff = connctx->body - (header1len + numdigits + header2len);
                if (offsetdiff < 0) {
                    /* may need more room - response headers are larger than request headers */
                    if ((header1len + numdigits + header2len + connctx->cl) > connctx->len) {
                        connctx->len = header1len + numdigits + header2len + connctx->cl;
                        connctx->buffer = (char *)realloc(connctx->buffer, connctx->len);
                    }
                    /* move the body up enough to accommodate the headers */
                    memmove(connctx->buffer + connctx->body - offsetdiff,
                            connctx->buffer + connctx->body, connctx->cl);
                    connctx->offset = 0;
                } else { /* new header size <= old header size */
                    connctx->offset = offsetdiff;
                }
                memcpy(connctx->buffer + connctx->offset, header1, header1len);
                sprintf(connctx->buffer + connctx->offset + header1len, "%lu", connctx->cl);
                memcpy(connctx->buffer + connctx->offset + header1len + numdigits, header2, header2len);
                connctx->state = 0; /* not reading any more */
                ns_job_modify(job, NS_JOB_THREAD); /* execute as threaded job for time consuming task */
                return;
            }
        }
        goto retry; /* keep reading until we get all data OR EAGAIN OR error */
    }
}

static void
conn_handler(ns_job_t *job)
{
    struct conn_ctx *connctx;
    
    PR_ASSERT(job);
    PR_ASSERT(ns_job_get_data(job));

    connctx = (struct conn_ctx *)ns_job_get_data(job);
    if (connctx->state == STATE_READING) {
        PR_ASSERT(NS_JOB_IS_READ(ns_job_get_output_type(job)));
        conn_read(job);
    } else if (connctx->state == STATE_WRITING) {
        PR_ASSERT(NS_JOB_IS_WRITE(ns_job_get_output_type(job)));
        conn_write(job);
    } else {
        long_time_consuming_job(job);
    }
}

static void
listen_accept(ns_job_t *job)
{
    PRFileDesc *connfd = NULL;
    struct conn_ctx *connctx;
    PRSocketOptionData prsod = {PR_SockOpt_Nonblocking, {PR_TRUE}};

    PR_ASSERT(job);

    /* accept and add read event for new connection */
    connfd = PR_Accept(ns_job_get_fd(job), NULL, PR_INTERVAL_NO_WAIT);
    if (connfd) {
        PR_SetSocketOption(connfd, &prsod);
        connctx = conn_ctx_new();
        connctx->sc = (struct ns_sec_ctx_t *)ns_job_get_data(job);
        ns_add_io_job(ns_job_get_tp(job), connfd, readtype, conn_handler, connctx, NULL);
    } else {
        PRErrorCode prerr = PR_GetError();
        if (PR_WOULD_BLOCK(prerr)) {
            /* ok - fall through */
        } else {
            do_logging(LOG_ERR, "listen_accept: accept error for job [%p] %d: %s\n", job,
                       prerr, PR_ErrorToString(prerr, PR_LANGUAGE_I_DEFAULT));
            /* error - bail out */
            return;
        }
    }
    /* re-arm listen_accept */
    /* ns_job_rearm(job); not needed if persistent */

    return;
}

ns_job_t *
test_protocol_init(ns_thrpool_t *tp, PRFileDesc *listenfd, struct ns_sec_ctx_t *sc)
{
    ns_job_t *listen_job = NULL;

    readtype = NS_JOB_READ;
    writetype = NS_JOB_WRITE;
    if (getenv("USE_POLL") && (getenv("USE_POLL")[0] != '0')) {
        USE_POLL = 1;
    } else {
        USE_POLL = 0;
    }

    /* add_sec_layer(listenfd, sc); */
    ns_add_io_job(tp, listenfd, NS_JOB_ACCEPT|NS_JOB_PERSIST, listen_accept, sc, &listen_job);

    return listen_job;
}
