/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "functs.h"
#include "process_int.h"

static void
vfarcmulcplx(Field &field, const double rconst[2])
{
  const auto grid = field.grid;
  const auto missval1 = field.missval;
  const auto missval2 = field.missval;
  auto &array = field.vec_d;

  const auto gridsize = gridInqSize(grid);

  // z1 x z2 = (x1x2 - y1y2) + i(x1y2 + x2y1)
  for (size_t i = 0; i < gridsize; i++)
    {
      const auto a1r = array[2 * i];
      const auto a1i = array[2 * i + 1];
      array[2 * i] = SUBMN(MULMN(a1r, rconst[0]), MULMN(a1i, rconst[1]));
      array[2 * i + 1] = ADDMN(MULMN(a1r, rconst[1]), MULMN(a1i, rconst[0]));
    }
}

static void
vfarcdivcplx(Field &field, const double rconst[2])
{
  const auto grid = field.grid;
  const auto missval1 = field.missval;
  const auto missval2 = field.missval;
  auto &array = field.vec_d;

  const auto gridsize = gridInqSize(grid);

  // z1 / z2 = (x1x2 + y1y2) / (x2x2 + y2y2) + i (y1x2 - x1y2) / (x2x2 + y2y2)
  for (size_t i = 0; i < gridsize; i++)
    {
      const auto a1r = array[2 * i];
      const auto a1i = array[2 * i + 1];
      const auto denominator = ADDMN(MULMN(rconst[0], rconst[0]), MULMN(rconst[1], rconst[1]));
      array[2 * i] = DIVMN(ADDMN(MULMN(a1r, rconst[0]), MULMN(a1i, rconst[1])), denominator);
      array[2 * i + 1] = DIVMN(SUBMN(MULMN(a1i, rconst[0]), MULMN(a1r, rconst[1])), denominator);
    }
}

static void
vfarcaddcplx(Field &field, const double rconst[2])
{
  const auto grid = field.grid;
  const auto missval1 = field.missval;
  const auto missval2 = field.missval;
  auto &array = field.vec_d;

  const auto gridsize = gridInqSize(grid);

  for (size_t i = 0; i < gridsize; i++)
    {
      array[2 * i] = ADDMN(array[2 * i], rconst[0]);
      array[2 * i + 1] = ADDMN(array[2 * i + 1], rconst[1]);
    }
}

static void
vfarcsubcplx(Field &field, const double rconst[2])
{
  const auto grid = field.grid;
  const auto missval1 = field.missval;
  const auto missval2 = field.missval;
  auto &array = field.vec_d;

  const auto gridsize = gridInqSize(grid);

  for (size_t i = 0; i < gridsize; i++)
    {
      array[2 * i] = SUBMN(array[2 * i], rconst[0]);
      array[2 * i + 1] = SUBMN(array[2 * i + 1], rconst[1]);
    }
}

void
vfarcfuncplx(Field &field, const double rconst[2], int function)
{
  switch (function)
    {
    case func_add: vfarcaddcplx(field, rconst); break;
    case func_sub: vfarcsubcplx(field, rconst); break;
    case func_mul: vfarcmulcplx(field, rconst); break;
    case func_div: vfarcdivcplx(field, rconst); break;
    default: cdoAbort("%s: function %d not implemented!", __func__, function);
    }
}
