package aho_corasick

import (
	"sync"
	"testing"
)

var benchmarkReplacerDFA []Replacer

func init() {
	benchmarkReplacerDFA = make([]Replacer, len(testCasesReplace))
	for i, t2 := range testCasesReplace {
		builder := NewAhoCorasickBuilder(Opts{
			AsciiCaseInsensitive: true,
			MatchOnlyWholeWords:  true,
			MatchKind:            LeftMostLongestMatch,
			DFA:                  true,
		})
		ac := builder.Build(t2.patterns)
		benchmarkReplacerDFA[i] = NewReplacer(ac)
	}
}

func BenchmarkAhoCorasick_ReplaceAllDFA(b *testing.B) {
	for i := 0; i < b.N; i++ {
		for i, ac := range benchmarkReplacerDFA {
			_ = ac.ReplaceAll(testCasesReplace[i].haystack, testCasesReplace[i].replaceWith)
		}
	}
}

func TestAhoCorasick_ReplaceAllFuncStopN(t *testing.T) {
	for _, i2 := range testCasesReplaceN {
		builder := NewAhoCorasickBuilder(Opts{
			AsciiCaseInsensitive: true,
			MatchOnlyWholeWords:  true,
			MatchKind:            LeftMostLongestMatch,
			DFA:                  true,
		})

		ac := builder.Build(i2.patterns)
		r := NewReplacer(ac)
		i := -1
		replaced := r.ReplaceAllFunc(i2.haystack, func(match Match) (string, bool) {
			i += 1
			return i2.replaceWith[match.pattern], i2.stopAt != i
		})
		if replaced != i2.replaced {
			t.Errorf("expected `%v` matches \n\n got `%v`", i2.replaced, replaced)
		}
	}
}

type testCaseReplaceN struct {
	patterns    []string
	haystack    string
	replaceWith []string
	replaced    string
	stopAt      int
}

var testCasesReplaceN = []testCaseReplaceN{
	{
		patterns:    []string{"bear", "masha"},
		replaceWith: []string{"robocop", "jinx"},
		haystack:    "The Bear and Masha",
		replaced:    "The robocop and Masha",
		stopAt:      1,
	},
	{
		patterns:    []string{"bear", "masha"},
		replaceWith: []string{"robocop", "jinx"},
		haystack:    "Bear and Masha",
		replaced:    "Bear and Masha",
		stopAt:      0,
	},
	{
		patterns:    []string{"bear", "masha"},
		replaceWith: []string{"robocop", "jinx"},
		haystack:    "The Bear is annoyed at Masha very much Bear",
		replaced:    "The robocop is annoyed at Masha very much Bear",
		stopAt:      1,
	},
	{
		patterns:    []string{"hinweise und vorsichtsmaßnahmen", "hinweisen"},
		replaceWith: []string{`<a>hinweise und vorsichtsmaßnahmen<\a>`, `<a>hinweisen<\a>`},
		haystack:    "<h2>Hinweise und Vorsichtsmaßnahmen, Wirkstoffprofil <br />(kumulative Wirkstoffangaben ohne Berücksichtigung des individuellen Zulassungsstatus)</h2> Erythropoietin - invasiv<br /><br /><p>Epoetin alfa</p><ul><li>allgemein<ul><li>Hypertonie<ul><li>bei allen Patienten sollte der Blutdruck engmaschig überwacht und ggf. behandelt werden</li><li>mit Vorsicht anwenden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte antihypertensive Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar, sollte die Behandlung beendet werden</li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurfte<ul><li>traten unter der Behandlung mit Epoetin alfa auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie<ul><li>bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS-Infektionen und Hirnmetastasen, vorsichtig anwenden</li></ul></li><li>chronische Leberinsuffizienz<ul><li>bei Patienten mit chronischer Leberinsuffizienz mit Vorsicht anwenden</li><li>Sicherheit von Epoetin alfa ist bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignissen (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden<ul><li>venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken)</li></ul></li><li>berichtetes Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin alfa abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li><li>bei allen Patienten sollten die Hämoglobinwerte engmaschig überwacht werden, da ein potentiell erhöhtes Risiko für thromboembolische Ereignisse und Mortalität besteht, wenn der Hämoglobin-Wert der Patienten oberhalb des für die jeweilige Indikation gültigen Zielbereichs liegt</li><li>während der Behandlung kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>dieser ist bei fortgesetzter Behandlung rückläufig</li></ul></li><li>es wird empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Anämieursachen sollten abgeklärt und müssen vor Beginn einer Behandlung mit Epoetin alfa sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden, z.B.<ul><li>Eisenmangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Folsäuremangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Hämolyse</li><li>Blutverlust</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen wird der Serumferritinwert simultan zum Anstieg des Hämatokritwertes abfallen; um ein optimales Ansprechen zu erzielen, müssen ausreichende Eisenspeicher gewährleistet sein, falls nötig substituieren<ul><li>orale Eisensubstitution für Patienten mit chronischer Niereninsuffizienz empfohlen, deren Serumferritinspiegel unter 100 ng/ml liegt, z.B.<ul><li>200 - 300 mg Fe2+ / Tag für Erwachsene</li><li>100 - 200 mg Fe2+ / Tag für Kinder und Jugendliche</li></ul></li><li>orale Eisensubstitution für alle Tumorpatienten, deren Transferrinsättigung unter 20% liegt<ul><li>200 - 300 mg Fe2+ / Tag</li></ul></li><li>orale Eisensubstitution für Patienten in einem autologen Blutspendeprogramm bereits einige Wochen vor der Eigenblutspende beginnen, um vor der Einleitung der Behandlung mit Epoetin alfa sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten<ul><li>200 mg Fe2+ / Tag</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin alfa eine orale Eisen-II-Substitution erhalten<ul><li>200 mg Fe2+ / Tag</li><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin alfa begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>in sehr seltenen Fällen eine Porphyrie oder Verschlimmerung einer Porphyrie beobachtet</li><li>bei diesen Patienten mit Vorsicht anwenden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen, einschließlich Stevens Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet</li><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li><li>zum Zeitpunkt der Verschreibung sollten Patienten auf Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, Behandlung unverzüglich beenden und alternative Behandlung in Betracht ziehen</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>Behandlung darf zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Substanzen (ESAs) zu verbessern, soll der Handelsname des verordneten ESAs deutlich in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung <ul><li>Patienten sollen nur unter geeigneter Überwachung von einem ESA auf einen anderen umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>über das Auftreten einer antikörpervermittelten PRCA nach monate- bis jahrelanger Behandlung mit subkutanem Erythropoetin berichtet<ul><li>v.a. bei Patienten mit chronischer Niereninsuffizienz</li><li>Fallberichte auch für Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten<ul><li>für diese Indikation besteht keine Zulassung</li></ul></li></ul></li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl bzw. 0,62 - 1,25 mmol/l pro Monat) mit erhöhtem Bedarf an Transfusionen, sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen untersucht werden, z.B.<ul><li>Eisenmangel</li><li>Folsäuremangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>bei paradoxer Abnahme des Hämoglobinwerts und Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozytenzahlen Absetzen der Behandlung und Durchführung einer Untersuchung auf Anti-Erythropoetin-Antikörper; auch Knochenmarkuntersuchung zur Diagnose einer PRCA erwägen</li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere Erythropoetin-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anämie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>regelmäßig Hämoglobin-Bestimmungen durchführen, bis ein stabiler Hämoglobin-Wert erreicht wird<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und sollte 2 g/dl (1,25mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko einer Zunahme des Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von &gt;12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative<br />Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, die mit subkutaner Gabe von Epoetin alfa behandelt werden, müssen regelmäßig auf Wirkverlust hin überwacht werden, der als keine oder verringerte Reaktion auf die Epoetin alfa-Behandlung bei Patienten definiert wird, die zuvor auf solch eine Therapie angesprochen haben<ul><li>zeigt sich durch andauernden Hämoglobinabfall trotz einer Erhöhung der Dosis von Epoetin alfa</li></ul></li><li>bei Verabreichung von Epoetin alfa in längeren Dosierungsintervallen (länger als einmal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-alfa-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shunt-Thrombosen sind bei Hämodialysepatienten, insbesondere bei Patienten mit Neigung zu Hypotonie oder bei Komplikationen an den arteriovenösen Fisteln aufgetreten (z.B. Stenosen, Aneurysmen etc.)<ul><li>frühzeitige Shuntrevision und Thromboseprophylaxe durch Gabe von z. B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden<ul><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder steigendem Kaliumwert sollte erwogen werden, die Epoetin alfa-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>aufgrund des Anstiegs des Hämatokrits im Rahmen der Epoetin-alfa-Behandlung ist häufig eine Erhöhung der Heparin-Dosis während der Dialyse notwendig<ul><li>ein Verschluss des Dialysesystems ist bei nicht optimaler Heparinisierung möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie<ul><li>regelmäßig Hämoglobin-Bestimmungen durchühren, bis ein stabiler Hämoglobin-Wert erreicht ist<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Erythropoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>es sollte bedacht werden, dass Epoetine das Wachstum von Tumoren stimulieren können</li><li>Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war Anwendung von Epoetin alfa und anderen ESAs mit verkürzter Zeit bis zur Tumorprogression oder mit verkürzter Gesamtüberlebenszeit verbunden<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürzte Gesamtüberlebenszeit und erhöhte Zahl an Todesfällen, zurückzuführen<br />auf eine Tumorprogression nach 4 Monaten, bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen Hämoglobinwerte von 12 bis 14 g/dl (7,5 bis 8,7 mmol/l) angestrebt wurden</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l), angestrebt wurde<ul><li>ESAs sind für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-alfa plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15% erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 und 12 g/dl (6,2 bis 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Tumorpatienten bevorzugt durch Bluttransfusionen behandelt werden</li></ul></li><li>Entscheidung für die Anwendung rekombinanter Erythropoetine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des jeweiligen Patienten beruhen, die auch den spezifischen klinischen Kontext berücksichtigen sollte<ul><li>in diese Beurteilung einzubeziehende Faktoren sind<ul><li>Tumortyp und -stadium</li><li>Anämiegrad</li><li>Lebenserwartung</li><li>Umgebung, in der der Patient behandelt wird</li><li>Präferenzen des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen sollte für die Beurteilung der Therapieeffizienz von eine 2 - 3-wöchige Verzögerung zwischen Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle spezifischen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbes. routinemäßiger Volumenersatz, sind zu berücksichtigen</li></ul></li><li>Patienten, bei denen ein großer elektiver orthopädischer Eingriff vorgesehen ist <ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrundeliegenden kardiovaskulären Erkrankung<ul><li>sollten eine angemessene Thromboseprophylaxe erhalten</li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit einer Prädisposition für tiefe Venenthrombosen (TVT) ergriffen werden</li></ul></li><li>darüber hinaus kann nicht ausgeschlossen werden, dass bei Behandlung mit Epoetin alfa für Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl ein erhöhtes Risiko für postoperative thrombotische/vaskuläre Ereignisse bestehen kann</li><li>aus diesem Grund sollte es bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl nicht angewendet werden</li></ul></li><li>Anwendung kann bei Dopingkontrollen zu positiven Ergebnissen führen<ul><li>Missbrauch durch Gesunde (z.B. für Dopingzwecke) kann zu übermäßigem Anstieg des Hämoglobinwertes führen; mit Risiko lebensbedrohlicher Komplikationen des Herz-Kreislauf-Systems (Thromboserisiko durch Hämokonzentration bei Polyglobulie) verbunden</li></ul></li></ul><p>Epoetin beta</p><ul><li>allgemein<ul><li>mit Vorsicht anwenden bei <ul><li>refraktärer Anämie mit Blastenexzess in Transformation</li><li>epileptiformen Erkrankungen</li><li>Thrombozytose</li><li>chronischer Leberinsuffizienz</li></ul></li><li>Folsäure- und Vitamin-B12-Mangel sollten ausgeschlossen sein, da hierdurch die Wirksamkeit vermindert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>um eine wirksame Erythropoiese sicherzustellen, sollte bei allen Patienten vor und während der Behandlung der Eisenwert bestimmt werden und ggf. eine Eisenersatztherapie gemäß den therapeutischen Richtlinien durchgeführt werden</li><li>Wirksamkeit von Epoetin beta kann durch eine Aluminiumüberladung infolge der Therapie der chronischen Niereninsuffizienz abgeschwächt werden</li><li>Prädialyse-Patienten mit Nephrosklerose <ul><li>über die Anwendung von Epoetin beta sollte bei Prädialyse-Patienten mit Nephrosklerose individuell entschieden werden, da eine beschleunigte Progression der Niereninsuffizienz bei diesen Patienten nicht mit Sicherheit ausgeschlossen werden kann</li></ul></li></ul></li><li>Aregenerative Anämie (pure red cell aplasia, PRCA)<ul><li>in Verbindung mit einer Erythropoietin-Therapie wurde über eine durch neutralisierende Antikörper gegen Erythropoietin hervorgerufene aregenerative Anämie (pure red cell aplasia) berichtet</li><li>für diese Antikörper konnte eine Kreuzreaktivität mit allen erythropoietischen Proteinen gezeigt werden</li><li>Patienten, bei denen der Verdacht auf neutralisierende Antikörper gegen Erythropoietin besteht oder bei denen diese nachgewiesen wurden, sollen deshalb nicht auf Epoetin beta umgestellt werden</li></ul></li><li>PRCA bei Patienten mit Hepatitis C<ul><li>eine paradoxe Hämoglobin-Abnahme und die Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozyten-Zahlen sollten zum Abbruch der Behandlung mit Epoetin und zur Durchführung von Tests auf Antikörper gegen Erythropoietin veranlassen</li><li>über Fälle berichtet bei Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten</li><li>Epoetine sind nicht zugelassen für die Behandlung einer Anämie in Verbindung mit Hepatitis C</li></ul></li><li>Hypertonie<ul><li>bei Patienten mit chronischer Nierenerkrankung kann eine Erhöhung des Blutdrucks erfolgen bzw. die Verstärkung einer bereits bestehenden Hypertension auftreten, insbes. in Fällen eines schnellen Hämatokrit-Anstiegs<ul><li>diese Blutdruckerhöhungen lassen sich medikamentös behandeln</li></ul></li><li>wenn der Blutdruckanstieg nicht durch medikamentöse Therapie kontrolliert werden kann, wird eine vorübergehende Unterbrechung der Therapie mit Epoetin beta empfohlen</li><li>insbes. zu Beginn der Behandlung wird darüber hinaus eine regelmäßige Blutdruckkontrolle empfohlen, auch zwischen den Dialysen</li><li>Blutdruckentgleisungen mit Enzephalopathie-ähnlicher Symptomatik können auftreten, die eine sofortige ärztliche Untersuchung und intensivmedizinische Maßnahmen erfordern<ul><li>plötzlich auftretende stechende migräneartige Kopfschmerzen sind als mögliches Warnzeichen besonders zu beachten</li></ul></li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung mit Epoetin unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>chronische Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung kann es insbesondere bei intravenöser Anwendung zu einem mäßiggradigen dosisabhängigen Anstieg der Thrombozytenzahl innerhalb des Normbereiches kommen, der im Verlauf der Therapie wieder zurückgeht</li><li>empfohlen, die Thrombozytenzahl in den ersten 8 Wochen der Therapie regelmäßig zu kontrollieren</li></ul></li><li>Hämoglobinkonzentration<ul><li>bei Patienten mit chronischer Nierenerkrankung sollte in der Erhaltungsphase die Hämoglobinkonzentration die empfohlene Obergrenze der angestrebten Hämoglobinkonzentration nicht überschreiten</li><li>in klinischen Studien wurde ein erhöhtes Sterberisiko und ein erhöhtes Risiko für schwerwiegende kardiovaskuläre Ereignisse einschl. Hirnschlag beobachtet, wenn Erythropoiese-stimulierende Substanzen (ESA) verabreicht wurden, um ein Hämoglobin &gt; 12 g/dl (7,5 mmol/l) zu erzielen</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobinkonzentration über den für die Kontrolle der Symptome einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li><li>bei Frühgeborenen kann die Thrombozytenzahl insbes. in den ersten zwei Lebenswochen leicht ansteigen, deshalb sollte die Thrombozytenzahl regelmäßig bestimmt werden</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, welche in erster Linie die Produktion von roten Blutkörperchen stimulieren</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedenster Tumorzellen gebildet werden<ul><li>wie bei allen Wachstumsfaktoren besteht der Verdacht, dass auch Epoetine das Wachstum von Tumoren stimulieren könnten</li></ul></li><li>in mehreren kontrollierten Studien mit Patienten, die eine Anämie im Zusammenhang mit einer Krebserkrankung aufwiesen, wurde keine Verbesserung des Gesamtüberlebens bzw. keine Verringerung des Risikos der Tumorprogression durch Epoetine gezeigt</li><li>in kontrollierten klinischen Studien hat die Anwendung von Epoetin beta und anderen Erythropoiese-stimulierenden Substanzen (ESA) Folgendes gezeigt:<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobinzielwert &gt; 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobinzielbereich von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko, wenn bei Patienten mit aktiven malignen Erkrankungen, die weder eine Chemotherapie noch eine Strahlentherapie erhielten, ein Hämoglobinzielwert von 12 g/dl (7,5 mmol/l) angestrebt wurde</li><li>ESA sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden</li><li>einer Entscheidung zur Gabe von rekombinantem Erythropoietin sollte eine Nutzen-Risiko-Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko-Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Prognose der Erkrankung</li><li>Umfeld, in dem der Patient behandelt</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>Laborwerte<ul><li>es kann zu einem Anstieg des Blutdrucks kommen, der medikamentös behandelt werden kann<ul><li>insbes. zu Beginn der Behandlung sollte deshalb der Blutdruck bei Krebspatienten kontrolliert werden</li></ul></li><li>Thrombozytenzahl und Hämoglobinkonzentration sollten bei Krebspatienten in regelmäßigen Abständen überprüft werden</li><li>bei Patienten in einem Eigenblutspendeprogramm kann es zu einem Anstieg der Thrombozytenzahl, in der Regel innerhalb des Normbereiches, kommen<ul><li>daher empfohlen, die Thrombozytenzahl mind. 1mal / Woche zu kontrollieren</li><li>falls Steigerung &gt; 150 x 10<sup>9</sup>/l oder liegt die Thrombozytenzahl über dem Normbereich, sollte die Therapie mit Epoetin beta abgebrochen werden</li></ul></li><li>Serum-Kalium- und -Phosphat-Spiegel sollten während der Behandlung mit Epoetin beta regelmäßig überprüft werden<ul><li>bei einigen urämischen Patienten wurde unter einer Behandlung mit Epoetin beta über erhöhte Kaliumspiegel berichtet, ohne dass ein kausaler Zusammenhang gezeigt wurde</li><li>sollten erhöhte oder ansteigende Kaliumspiegel beobachtet werden, dann sollte eine Unterbrechung der Behandlung mit Epoetin beta in Betracht gezogen werden, bis die Hyperkalämie korrigiert ist</li></ul></li></ul></li><li>Frühgeborene, Retinopathie<ul><li>bei Frühgeborenen konnte das potentielle Risiko, dass Erythropoietin eine Retinopathie auslöst, nicht ausgeschlossen werden<ul><li>deshalb Vorsicht bei der Anwendung</li></ul></li><li>bei der Entscheidung, ob ein Frühgeborenes behandelt wird, sollten die möglichen Nutzen und Risiken dieser Behandlung und verfügbarer alternativer Therapiemöglichkeiten gegeneinander abgewogen werden</li></ul></li><li>Patienten mit chronischer Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung ist im Verlauf der Therapie mit Epoetin beta aufgrund des gesteigerten Hämatokritwertes häufig eine Erhöhung der Heparin-Dosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialyse-Systems möglich</li></ul></li><li>frühzeitige Shunt-Revision und eine Thrombose-Prophylaxe, z.B. durch Gabe von Acetylsalicylsäure, sollte bei Patienten mit chronischer Nierenerkrankung und mit dem Risiko einer Shunt-Thrombose in Betracht gezogen werden</li></ul></li><li>Eigenblutspende<ul><li>für den Einsatz von Epoetin beta im Rahmen der Eigenblutspende müssen die offiziellen Richtlinien zur Blutspende berücksichtigt werden, insbesondere:<ul><li>nur Patienten mit einem Hämatokritwert &gt;= 33 % (Hämoglobinwert &gt;= 11 g/dl [6,83 mmol/l]) dürfen Blut spenden</li><li>Patienten mit einem Körpergewicht &lt; 50 kg sollten besonders sorgfältig beobachtet werden</li><li>das Volumen einer einzelnen Blutspende sollte nicht größer als ca. 12% des Blutvolumens des Patienten sein</li></ul></li><li>Behandlung sollte auf Patienten beschränkt werden, bei denen die Vermeidung von Fremdbluttransfusionen nach deren Nutzen-Risiko-Abschätzung als besonders wichtig angesehen wird</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Verwendung bei Gesunden kann zu einer übermäßigen Steigerung des Hämatokritwertes führen</li><li>in diesem Zusammenhang wurde über lebensbedrohliche Komplikationen des Herz-Kreislauf-Systems berichtet</li></ul></li></ul><p>Epoetin theta</p><ul><li>allgemein<ul><li>Eisensupplementierung wird bei allen Patienten mit Ferritin-Serumwerten &lt; 100 µg/l oder einer Transferrin-Sättigung &lt; 20% empfohlen<ul><li>um eine effektive Erythropoiese zu gewährleisten, muss bei allen Patienten vor und während der Behandlung der Eisenstatus bestimmt werden</li></ul></li><li>Nicht-Ansprechen auf die Behandlung mit Epoetin theta sollte Anlass für eine Suche nach ursächlichen Faktoren sein<ul><li>Eisen-, Folsäure- und Vitamin-B12-Mangelzustände verringern die Wirksamkeit von Epoetinen und müssen daher korrigiert werden</li><li>die folgenden Faktoren können eine Ansprechen der Erythropoiese beeinträchtigen <ul><li>interkurrente Infektionen</li><li>entzündliche oder traumatische Vorfälle</li><li>okkulter Blutverlust</li><li>Hämolyse</li><li>Aluminium-Intoxikation</li><li>hämatologische Grunderkrankung</li><li>Knochenmarkfibrose</li></ul></li><li>Beurteilung sollte auch die Retikulozytenzahlen umfassen</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>wenn typische Ursachen für ein Nicht-Ansprechen ausgeschlossen wurden und bei dem Patienten ein plötzlicher Abfall des Hämoglobins zusammen mit einer Retikulopenie auftritt, sind Untersuchungen auf Anti-Erythropoietin-Antikörper und eine Knochenmarkuntersuchung zur Diagnosestellung einer Erythroblastopenie in Betracht zu ziehen</li><li>Unterbrechung der Behandlung mit Epoetin theta ist in Betracht zu ziehen</li><li>unter Erythropoietin-Therapie wurden Fälle von PRCA beschrieben, die durch neutralisierende Anti-Erythropoietin-Antikörper verursacht wurden</li><li>da für diese Antikörper Kreuzreaktivität mit allen Epoetinen gezeigt wurde, sollten Patienten mit Verdacht oder Nachweis auf neutralisierende Antikörper gegen Erythropoietin nicht auf Epoetin theta umgestellt werden</li><li>bei einer paradoxen Abnahme des Hämoglobinwerts und Ausbildung einer schweren Anämie in Zusammenhang mit einer niedrigen Retikulozytenzahl ist die Behandlung mit Epoetin abzubrechen und einen Test auf Anti-Erythropoetin-Antikörper durchzuführen<ul><li>entsprechende Fälle sind bei Hepatitis C-Patienten berichtet worden, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetin erhielten</li><li>Epoetine sind nicht für die Behandlung einer Hepatitis C-assoziierten Anämie bestimmt</li></ul></li></ul></li><li>Hypertonie<ul><li>während einer Epoetin-theta-Therapie kann es zu einer Erhöhung des Blutdrucks bzw. einer Verstärkung einer bereits bestehenden Hypertonie kommen, insbes. zu Beginn der Behandlung</li><li>auf eine engmaschige Überwachung und Blutdruckkontrollen achten<ul><li>Blutdruck muss vor Beginn der Behandlung und während der Behandlung adäquat eingestellt werden, um akute Komplikationen wie eine hypertensive Krise mit Enzephalopathie-ähnlichen Symptomen (z.B. Kopfschmerzen, Verwirrtheit, Sprachstörungen, Gangstörungen) und assoziierten Komplikationen (Krampfanfälle, Schlaganfall) zu vermeiden, die auch vereinzelt bei Patienten mit ansonsten normalem oder niedrigem Blutdruck auftreten kann</li><li>beim Auftreten dieser Reaktionen ist eine umgehende ärztliche Behandlung und intensivmedizinische Überwachung erforderlich</li></ul></li><li>auf plötzlich auftretende, heftige, migräneartige Kopfschmerzen als mögliches Warnsignal sollte besonders geachtet werden</li><li>Blutdruckerhöhungen <ul><li>können eine Behandlung mit blutdrucksenkenden Arzneimitteln oder eine Dosiserhöhung einer bereits bestehenden antihypertensiven Behandlung notwendig machen</li><li>zusätzlich Reduktion der angewendeten Epoetin theta-Dosis erwägen</li><li>falls die Blutdruckwerte weiterhin erhöht bleiben, kann eine vorübergehende Unterbrechung der Therapie mit Epoetin theta erforderlich sein</li></ul></li><li>nach erfolgreicher Blutdruckeinstellung durch eine Intensivierung der Behandlung sollte die Epoetin-theta-Therapie in reduzierter Dosierung wieder aufgenommen werden</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Anwendung von Epoetin theta bei gesunden Personen kann einen übermäßigen Anstieg des Hämoglobins und des Hämatokrits zur Folge haben, was zu lebensbedrohlichen kardiovaskulären Komplikationen führen kann</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden</li><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung mit Epoetin theta unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li><li>wenn ein Patient aufgrund der Anwendung von Epoetin theta eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin theta zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>besondere Patientengruppen<ul><li>eingeschränkte Leberfunktion und Sichelzellenanämie<ul><li>wegen der begrenzten Erfahrung keine Aussagen zur Wirksamkeit und Sicherheit von Epoetin theta bei Patienten mit eingeschränkter Leberfunktion und homozygoter Sichelzellanämie möglich</li></ul></li><li>ältere Patienten &gt; 75 Jahre<ul><li>in klinischen Studien hatten Patienten über 75 Jahren eine höhere Inzidenz schwerwiegender und schwerer unerwünschter Ereignisse, unabhängig von einem Kausalzusammenhang mit der Epoetin-theta-Therapie</li><li>Todesfälle waren in dieser Patientengruppe häufiger als bei jüngeren Patienten</li></ul></li></ul></li><li>Laborkontrollen<ul><li>es wird empfohlen, regelmäßig das Hämoglobin zu messen, ein komplettes Blutbild zu erstellen und Thrombozytenzählungen durchzuführen</li></ul></li><li>Symptomatische Anämie infolge chronischer Niereninsuffizienz<ul><li>bei nephrosklerotischen Patienten, die noch keine Dialyse erhalten, ist die Anwendung von Epoetin theta individuell abzuwägen, da die Möglichkeit einer beschleunigten Progression der Niereninsuffizienz nicht sicher ausgeschlossen werden kann</li><li>während der Hämodialyse können Patienten unter Behandlung mit Epoetin theta eine intensivere gerinnungshemmende Therapie benötigen, um einem Verschluss des arterio-venösen Shunts vorzubeugen</li><li>bei Patienten mit chronischer Niereninsuffizienz sollte die Hämoglobin-Konzentration in der Erhaltungsphase die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschreiten<ul><li>in klinischen Studien wurde ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse unter einer Epoetin-Therapie beobachtet, wenn Hämoglobin-Zielspiegel &gt; 12 g/dl (7,45mmol/l) angestrebt wurde</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobin-Konzentration über den für die Symptomkontrolle einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li></ul></li><li>Vorsicht bei der Steigerung der Epoetin-theta-Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li></ul></li><li>Symptomatische Anämie bei Krebspatienten mit nicht-myeloischen malignen Erkrankungen, die eine Chemotherapie erhalten<ul><li>Einfluß auf Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, die primär die Bildung von Erythrozyten anregen</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen exprimiert werden</li><li>wie bei allen Wachstumsfaktoren gibt es Bedenken, dass Epoetine das Wachstum von Tumoren anregen könnten</li></ul></li><li>in mehreren kontrollierten Studien zeigten Epoetine keine Verbesserung des Gesamtüberlebens oder eine Verminderung des Risikos einer Tumorprogression bei Patienten mit Anämie infolge einer Krebserkrankung.</li><li>in kontrollierten klinischen Studien zeigte die Anwendung von Epoetinen:<ul><li>eine verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Halstumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobin- Zielspiegel oberhalb von 14 g/dl (8,69mmol/l) angestrebt wurde</li><li>ein verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobin-Zielwert zwischen 12 - 14 g/dl (7,45 - 8,69mmol/l) angestrebt wurde</li><li>ein erhöhtes Sterberisiko bei Patienten mit aktiven malignen Erkrankungen, dieweder eine Chemotherapie noch eine Strahlentherapie erhielten, wenn ein Hämoglobin-Zielwert von 12g/dl (7,45mmol/l) angestrebt wurde</li></ul></li><li>Epoetine sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Krebspatienten bevorzugt mit Bluttransfusionen behandelt werden</li><li>Entscheidung zur Anwendung rekombinanter Erythropoietine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des einzelnen Patienten beruhen, unter Berücksichtigung der jeweils spezifischen klinischen Umstände<ul><li>Faktoren wie Tumorart und -stadium, Schweregrad der Anämie, Lebenserwartung, Behandlungsumfeld des Patienten und Wunsch des Patienten sollten in diese Beurteilung mit einbezogen werden</li></ul></li></ul></li></ul><p>Epoetin zeta</p><ul><li>allgemein<ul><li>Blutdruck <ul><li>bei allen Patienten, die Epoetin zeta erhalten, sollte der Blutdruck überwacht und ggf. behandelt werden</li><li>Epoetin zeta sollte mit Vorsicht angewendet werden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte blutdrucksenkende Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar<ul><li>sollte die Behandlung mit Epoetin zeta beendet werden</li></ul></li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurften<ul><li>traten unter der Behandlung mit Epoetin zeta auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie, Krampfneigung<ul><li>Epoetin zeta sollte bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS- Infektionen und Hirnmetastasen, vorsichtig angewendet werden</li></ul></li><li>Patienten mit chronischer Leberinsuffizienz <ul><li>Anwendung mit Vorsicht</li><li>Sicherheit von Epoetin zeta bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignisse (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden <ul><li>dazu gehören venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>darüber hinaus wurde über zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken) berichtet</li></ul></li><li>das berichtete Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin zeta abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li></ul></li><li>Hämoglobinspiegel<ul><li>bei allen Patienten ist der Hämoglobinspiegel engmaschig zu ü berwachen, da potentiell ein erhöhtes Risiko fü r thromboembolische Ereignisse und tödliche Verläufe besteht, wenn Patienten behandelt wurden, deren Hämoglobinkonzentration über dem Konzentrationsbereich liegt, für den das Arzneimittel indiziert ist</li></ul></li><li>während der Behandlung mit Erythropoetin kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>bei fortgesetzter Behandlung rückläufig</li><li>empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Ursachen einer Anämie müssen vor Beginn einer Behandlung mit Epoetin zeta sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden<ul><li>Eisen-, Folsäure- oder Vitamin B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikationen</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarksfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen gehen gleichzeitig mit der Zunahme des gepackten Zellvolumens die Ferritinwerte im Serum zurück</li><li>für das Erzielen eines optimalen Ansprechens auf Erythropoetin müssen ausreichende Eisenreserven sichergestellt werden:<ul><li>für Patienten mit chronischer Niereninsuffizienz, deren Serum-Ferritinkonzentration unter 100 ng/ml liegt, wird eine orale Eisensubstitution empfohlen, z.B. 200 - 300 mg / Tag (100-200 mg / Tag bei pädiatrischen Patienten)</li><li>für alle Tumorpatienten, bei denen die Transferrinsättigung weniger als 20% beträgt, wird eine orale Eisensubstitution mit 200 - 300 mg / Tag empfohlen</li><li>für Patienten in einem autologen Blutspendeprogramm sollte mit der oralen Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) bereits einige Wochen vor der Eigenblutspende begonnen werden, um vor der Einleitung der Behandlung mit Epoetin zeta sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten</li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin zeta eine orale Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) erhalten<ul><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin zeta begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>sehr selten wurde die Entwicklung oder Exazerbation einer Porphyrie bei mit Epoetin zeta behandelten Patienten beobachtet</li><li>Epoetin zeta sollte bei Patienten mit Porphyrie mit Vorsicht angewendet werden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens- Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung von Epoetin eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Wirkstoffen (ESAs) zu verbessern, soll der Handelsname und die Chargennummer des angewendeten ESAs eindeutig in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung<ul><li>Patienten sollen nur unter geeigneter Überwachung von einer ESA auf ein anderes umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>übber das Auftreten einer antikörpervermittelten Erythroblastopenie (PRCA) wurde nach monate- bis jahrelanger Behandlung mit Epoetin berichtet</li><li>ebenfalls traten Fälle bei Patienten mit Hepatitis C auf, die mit Interferon und Ribavirin behandelt wurden, während ESAs begleitend eingesetzt wurden</li><li>Epoetin zeta ist für die Behandlung einer Anämie, die mit Hepatitis C assoziiert ist, nicht zugelassen</li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl pro Monat) mit erhöhtem Bedarf an Transfusionen<ul><li>sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen (z.B. Eisen-, Folsäure- oder Vitamin-B12-Mangel, Aluminiumintoxikation, Infektionen oder Entzündungen, Blutverlust, Hämolyse und Knochenmarkfibrose jeglicher Genese) abgeklärt werden</li></ul></li><li>bei einer paradoxen Verminderung der Hämoglobinwerte und der Entwicklung einer schweren Anämie, die mit einem erniedrigten Retikulozytenwert einhergeht<ul><li>sollte die Behandlung mit Epoetin zeta sofort abgesetzt und die Anti-Erythropoetin- Antikörper bestimmt werden</li><li>eine Untersuchung des Knochenmarks zur Diagnose einer PRCA sollte ebenfalls erwogen werden</li></ul></li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere ESA-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anä mie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>Hämoglobin-Bestimmungen<ul><li>bei Patienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und 2 g/dl (1,25 mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko der Zunahme eines Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von über 12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht, wenn die Epoetin-Dosis bei Patienten mit chronischer Niereninsuffizienz gesteigert wird<ul><li>da hohe kumulierte Epoetin-Dosen mit einem gesteigerten Mortalitätsrisiko, schwerwiegenden kardiovaskulären und zerebrovaskulären Ereignissen assoziiert sein können</li><li>bei Patienten, die ein schlechtes Ansprechen des Hämoglobinwertes auf Epoetin zeigen, sollten alternative Gründe für das schlechte Anprechen bedacht werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, denen Epoetin zeta subkutan gegeben wird, sollten regelmäßig Kontrollen hinsichtlich eines Wirkverlustes, definiert als ein Nicht- oder vermindertes Ansprechen auf eine Behandlung mit Epoetin zeta bei Patienten, die zuvor auf eine solche Therapie angesprochen haben, durchgeführt werden<ul><li>dies zeigt sich durch eine anhaltende Verminderung der Hämoglobinwerte trotz Steigerung der Epoetin-zeta- Dosis</li></ul></li><li>bei Verabreichung von Epoetin in längeren Dosisintervallen (länger als 1-mal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shuntthrombosen <ul><li>Shuntthrombosen sind insbesondere bei Hämodialysepatienten aufgetreten, die eine Tendenz zur Hypotonie aufweisen oder deren arteriovenöse Fisteln Komplikationen aufweisen (z.B. Stenosen, Aneurysmen, etc.)</li><li>eine frühzeitige Shuntkorrektur und Thromboseprophylaxe durch die Gabe von z.B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>Hyperkaliämie<ul><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden</li><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder ansteigendem Kaliumwert sollte zusätzlich zu einer geeigneten Behandlung der Hyperkaliämie erwogen werden, die Epoetin-zeta-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>im Verlauf der Therapie mit Erythropoetin ist aufgrund des erhöhten Hämatokrits häufig eine Steigerung der Heparindosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialysesystems möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie mit Epoetin zeta bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie, Auswirkungen auf das Tumorwachstum <ul><li>Hämoglobin-Bestimmungen<ul><li>bei Tumorpatienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>Epoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>we bei allen Wachstumsfaktoren sollte bedacht werden, dass Epoetine das Wachstum von Tumoren jeglicher Art stimulieren können</li><li>eine Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war die Anwendung von Epoetin zeta und anderen ESAs mit einer verkürzten Zeit bis zur Tumorprogression oder mit einer verkürzten Gesamtüberlebenszeit<br />verbunden <ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn eine Hämoglobin-Zielkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>kürzeres Gesamtüberleben und erhöhte Sterblichkeit, die auf eine Krankheitsprogression nach 4 Monaten bei Patienten mit metastasierendem Brustkrebs, die eine Chemotherapie erhielten, zurückgeführt wurde, wenn eine Hämoglobin-Zielkonzentration von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l) angestrebt wurde<ul><li>ESAs für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-zeta plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15 % erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 g/dl und 12 g/dl (6,2 - 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden<ul><li>einer Entscheidung zur Gabe von rekombinantem Erythropoetin sollte eine Nutzen-Risiko Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Erkrankungsprognose</li><li>Umfeld, in dem der Patient behandelt wird</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen, sollte für die Beurteilung der Therapieeffizienz von Epoetin zeta eine 2- bis 3-wöchige Verzögerung zwischen ESA-Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle speziellen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbesondere routinemäßiger Volumenersatz, sollten berü cksichtigt werden</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff vorgesehen ist<ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrunde liegenden kardiovaskulären Erkrankung<ul><li>daher sollte insbesondere bei diesen Patienten eine angemessene Thromboseprophylaxe erfolgen</li></ul></li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit Disposition für tiefe Venenthrombosen ergriffen werden</li><li>Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l)<ul><li>Epoetin zeta sollte nicht bei diesen Patienten eingesetzt werden</li><li>bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l) kann nicht ausgeschlossen werden, dass eine Epoetin zeta-Behandlung mit einem erhöhten Risiko<br />postoperativer Thrombose-/Gefäßereignissen einhergeht</li></ul></li></ul></li></ul>",
		replaced:    `<h2><a>hinweise und vorsichtsmaßnahmen<\a>, Wirkstoffprofil <br />(kumulative Wirkstoffangaben ohne Berücksichtigung des individuellen Zulassungsstatus)</h2> Erythropoietin - invasiv<br /><br /><p>Epoetin alfa</p><ul><li>allgemein<ul><li>Hypertonie<ul><li>bei allen Patienten sollte der Blutdruck engmaschig überwacht und ggf. behandelt werden</li><li>mit Vorsicht anwenden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte antihypertensive Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar, sollte die Behandlung beendet werden</li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurfte<ul><li>traten unter der Behandlung mit Epoetin alfa auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie<ul><li>bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS-Infektionen und Hirnmetastasen, vorsichtig anwenden</li></ul></li><li>chronische Leberinsuffizienz<ul><li>bei Patienten mit chronischer Leberinsuffizienz mit Vorsicht anwenden</li><li>Sicherheit von Epoetin alfa ist bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignissen (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden<ul><li>venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken)</li></ul></li><li>berichtetes Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin alfa abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li><li>bei allen Patienten sollten die Hämoglobinwerte engmaschig überwacht werden, da ein potentiell erhöhtes Risiko für thromboembolische Ereignisse und Mortalität besteht, wenn der Hämoglobin-Wert der Patienten oberhalb des für die jeweilige Indikation gültigen Zielbereichs liegt</li><li>während der Behandlung kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>dieser ist bei fortgesetzter Behandlung rückläufig</li></ul></li><li>es wird empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Anämieursachen sollten abgeklärt und müssen vor Beginn einer Behandlung mit Epoetin alfa sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden, z.B.<ul><li>Eisenmangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Folsäuremangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Hämolyse</li><li>Blutverlust</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen wird der Serumferritinwert simultan zum Anstieg des Hämatokritwertes abfallen; um ein optimales Ansprechen zu erzielen, müssen ausreichende Eisenspeicher gewährleistet sein, falls nötig substituieren<ul><li>orale Eisensubstitution für Patienten mit chronischer Niereninsuffizienz empfohlen, deren Serumferritinspiegel unter 100 ng/ml liegt, z.B.<ul><li>200 - 300 mg Fe2+ / Tag für Erwachsene</li><li>100 - 200 mg Fe2+ / Tag für Kinder und Jugendliche</li></ul></li><li>orale Eisensubstitution für alle Tumorpatienten, deren Transferrinsättigung unter 20% liegt<ul><li>200 - 300 mg Fe2+ / Tag</li></ul></li><li>orale Eisensubstitution für Patienten in einem autologen Blutspendeprogramm bereits einige Wochen vor der Eigenblutspende beginnen, um vor der Einleitung der Behandlung mit Epoetin alfa sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten<ul><li>200 mg Fe2+ / Tag</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin alfa eine orale Eisen-II-Substitution erhalten<ul><li>200 mg Fe2+ / Tag</li><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin alfa begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>in sehr seltenen Fällen eine Porphyrie oder Verschlimmerung einer Porphyrie beobachtet</li><li>bei diesen Patienten mit Vorsicht anwenden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen, einschließlich Stevens Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet</li><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li><li>zum Zeitpunkt der Verschreibung sollten Patienten auf Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen <a>hinweisen<\a>, auftreten, Behandlung unverzüglich beenden und alternative Behandlung in Betracht ziehen</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>Behandlung darf zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Substanzen (ESAs) zu verbessern, soll der Handelsname des verordneten ESAs deutlich in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung <ul><li>Patienten sollen nur unter geeigneter Überwachung von einem ESA auf einen anderen umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>über das Auftreten einer antikörpervermittelten PRCA nach monate- bis jahrelanger Behandlung mit subkutanem Erythropoetin berichtet<ul><li>v.a. bei Patienten mit chronischer Niereninsuffizienz</li><li>Fallberichte auch für Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten<ul><li>für diese Indikation besteht keine Zulassung</li></ul></li></ul></li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl bzw. 0,62 - 1,25 mmol/l pro Monat) mit erhöhtem Bedarf an Transfusionen, sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen untersucht werden, z.B.<ul><li>Eisenmangel</li><li>Folsäuremangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>bei paradoxer Abnahme des Hämoglobinwerts und Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozytenzahlen Absetzen der Behandlung und Durchführung einer Untersuchung auf Anti-Erythropoetin-Antikörper; auch Knochenmarkuntersuchung zur Diagnose einer PRCA erwägen</li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere Erythropoetin-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anämie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>regelmäßig Hämoglobin-Bestimmungen durchführen, bis ein stabiler Hämoglobin-Wert erreicht wird<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und sollte 2 g/dl (1,25mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko einer Zunahme des Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von &gt;12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative<br />Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, die mit subkutaner Gabe von Epoetin alfa behandelt werden, müssen regelmäßig auf Wirkverlust hin überwacht werden, der als keine oder verringerte Reaktion auf die Epoetin alfa-Behandlung bei Patienten definiert wird, die zuvor auf solch eine Therapie angesprochen haben<ul><li>zeigt sich durch andauernden Hämoglobinabfall trotz einer Erhöhung der Dosis von Epoetin alfa</li></ul></li><li>bei Verabreichung von Epoetin alfa in längeren Dosierungsintervallen (länger als einmal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-alfa-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shunt-Thrombosen sind bei Hämodialysepatienten, insbesondere bei Patienten mit Neigung zu Hypotonie oder bei Komplikationen an den arteriovenösen Fisteln aufgetreten (z.B. Stenosen, Aneurysmen etc.)<ul><li>frühzeitige Shuntrevision und Thromboseprophylaxe durch Gabe von z. B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden<ul><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder steigendem Kaliumwert sollte erwogen werden, die Epoetin alfa-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>aufgrund des Anstiegs des Hämatokrits im Rahmen der Epoetin-alfa-Behandlung ist häufig eine Erhöhung der Heparin-Dosis während der Dialyse notwendig<ul><li>ein Verschluss des Dialysesystems ist bei nicht optimaler Heparinisierung möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie<ul><li>regelmäßig Hämoglobin-Bestimmungen durchühren, bis ein stabiler Hämoglobin-Wert erreicht ist<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Erythropoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>es sollte bedacht werden, dass Epoetine das Wachstum von Tumoren stimulieren können</li><li>Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war Anwendung von Epoetin alfa und anderen ESAs mit verkürzter Zeit bis zur Tumorprogression oder mit verkürzter Gesamtüberlebenszeit verbunden<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürzte Gesamtüberlebenszeit und erhöhte Zahl an Todesfällen, zurückzuführen<br />auf eine Tumorprogression nach 4 Monaten, bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen Hämoglobinwerte von 12 bis 14 g/dl (7,5 bis 8,7 mmol/l) angestrebt wurden</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l), angestrebt wurde<ul><li>ESAs sind für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-alfa plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15% erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 und 12 g/dl (6,2 bis 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Tumorpatienten bevorzugt durch Bluttransfusionen behandelt werden</li></ul></li><li>Entscheidung für die Anwendung rekombinanter Erythropoetine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des jeweiligen Patienten beruhen, die auch den spezifischen klinischen Kontext berücksichtigen sollte<ul><li>in diese Beurteilung einzubeziehende Faktoren sind<ul><li>Tumortyp und -stadium</li><li>Anämiegrad</li><li>Lebenserwartung</li><li>Umgebung, in der der Patient behandelt wird</li><li>Präferenzen des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen sollte für die Beurteilung der Therapieeffizienz von eine 2 - 3-wöchige Verzögerung zwischen Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle spezifischen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbes. routinemäßiger Volumenersatz, sind zu berücksichtigen</li></ul></li><li>Patienten, bei denen ein großer elektiver orthopädischer Eingriff vorgesehen ist <ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrundeliegenden kardiovaskulären Erkrankung<ul><li>sollten eine angemessene Thromboseprophylaxe erhalten</li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit einer Prädisposition für tiefe Venenthrombosen (TVT) ergriffen werden</li></ul></li><li>darüber hinaus kann nicht ausgeschlossen werden, dass bei Behandlung mit Epoetin alfa für Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl ein erhöhtes Risiko für postoperative thrombotische/vaskuläre Ereignisse bestehen kann</li><li>aus diesem Grund sollte es bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl nicht angewendet werden</li></ul></li><li>Anwendung kann bei Dopingkontrollen zu positiven Ergebnissen führen<ul><li>Missbrauch durch Gesunde (z.B. für Dopingzwecke) kann zu übermäßigem Anstieg des Hämoglobinwertes führen; mit Risiko lebensbedrohlicher Komplikationen des Herz-Kreislauf-Systems (Thromboserisiko durch Hämokonzentration bei Polyglobulie) verbunden</li></ul></li></ul><p>Epoetin beta</p><ul><li>allgemein<ul><li>mit Vorsicht anwenden bei <ul><li>refraktärer Anämie mit Blastenexzess in Transformation</li><li>epileptiformen Erkrankungen</li><li>Thrombozytose</li><li>chronischer Leberinsuffizienz</li></ul></li><li>Folsäure- und Vitamin-B12-Mangel sollten ausgeschlossen sein, da hierdurch die Wirksamkeit vermindert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>um eine wirksame Erythropoiese sicherzustellen, sollte bei allen Patienten vor und während der Behandlung der Eisenwert bestimmt werden und ggf. eine Eisenersatztherapie gemäß den therapeutischen Richtlinien durchgeführt werden</li><li>Wirksamkeit von Epoetin beta kann durch eine Aluminiumüberladung infolge der Therapie der chronischen Niereninsuffizienz abgeschwächt werden</li><li>Prädialyse-Patienten mit Nephrosklerose <ul><li>über die Anwendung von Epoetin beta sollte bei Prädialyse-Patienten mit Nephrosklerose individuell entschieden werden, da eine beschleunigte Progression der Niereninsuffizienz bei diesen Patienten nicht mit Sicherheit ausgeschlossen werden kann</li></ul></li></ul></li><li>Aregenerative Anämie (pure red cell aplasia, PRCA)<ul><li>in Verbindung mit einer Erythropoietin-Therapie wurde über eine durch neutralisierende Antikörper gegen Erythropoietin hervorgerufene aregenerative Anämie (pure red cell aplasia) berichtet</li><li>für diese Antikörper konnte eine Kreuzreaktivität mit allen erythropoietischen Proteinen gezeigt werden</li><li>Patienten, bei denen der Verdacht auf neutralisierende Antikörper gegen Erythropoietin besteht oder bei denen diese nachgewiesen wurden, sollen deshalb nicht auf Epoetin beta umgestellt werden</li></ul></li><li>PRCA bei Patienten mit Hepatitis C<ul><li>eine paradoxe Hämoglobin-Abnahme und die Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozyten-Zahlen sollten zum Abbruch der Behandlung mit Epoetin und zur Durchführung von Tests auf Antikörper gegen Erythropoietin veranlassen</li><li>über Fälle berichtet bei Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten</li><li>Epoetine sind nicht zugelassen für die Behandlung einer Anämie in Verbindung mit Hepatitis C</li></ul></li><li>Hypertonie<ul><li>bei Patienten mit chronischer Nierenerkrankung kann eine Erhöhung des Blutdrucks erfolgen bzw. die Verstärkung einer bereits bestehenden Hypertension auftreten, insbes. in Fällen eines schnellen Hämatokrit-Anstiegs<ul><li>diese Blutdruckerhöhungen lassen sich medikamentös behandeln</li></ul></li><li>wenn der Blutdruckanstieg nicht durch medikamentöse Therapie kontrolliert werden kann, wird eine vorübergehende Unterbrechung der Therapie mit Epoetin beta empfohlen</li><li>insbes. zu Beginn der Behandlung wird darüber hinaus eine regelmäßige Blutdruckkontrolle empfohlen, auch zwischen den Dialysen</li><li>Blutdruckentgleisungen mit Enzephalopathie-ähnlicher Symptomatik können auftreten, die eine sofortige ärztliche Untersuchung und intensivmedizinische Maßnahmen erfordern<ul><li>plötzlich auftretende stechende migräneartige Kopfschmerzen sind als mögliches Warnzeichen besonders zu beachten</li></ul></li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen <a>hinweisen<\a>, auftreten, sollte die Behandlung mit Epoetin unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>chronische Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung kann es insbesondere bei intravenöser Anwendung zu einem mäßiggradigen dosisabhängigen Anstieg der Thrombozytenzahl innerhalb des Normbereiches kommen, der im Verlauf der Therapie wieder zurückgeht</li><li>empfohlen, die Thrombozytenzahl in den ersten 8 Wochen der Therapie regelmäßig zu kontrollieren</li></ul></li><li>Hämoglobinkonzentration<ul><li>bei Patienten mit chronischer Nierenerkrankung sollte in der Erhaltungsphase die Hämoglobinkonzentration die empfohlene Obergrenze der angestrebten Hämoglobinkonzentration nicht überschreiten</li><li>in klinischen Studien wurde ein erhöhtes Sterberisiko und ein erhöhtes Risiko für schwerwiegende kardiovaskuläre Ereignisse einschl. Hirnschlag beobachtet, wenn Erythropoiese-stimulierende Substanzen (ESA) verabreicht wurden, um ein Hämoglobin &gt; 12 g/dl (7,5 mmol/l) zu erzielen</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobinkonzentration über den für die Kontrolle der Symptome einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li><li>bei Frühgeborenen kann die Thrombozytenzahl insbes. in den ersten zwei Lebenswochen leicht ansteigen, deshalb sollte die Thrombozytenzahl regelmäßig bestimmt werden</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, welche in erster Linie die Produktion von roten Blutkörperchen stimulieren</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedenster Tumorzellen gebildet werden<ul><li>wie bei allen Wachstumsfaktoren besteht der Verdacht, dass auch Epoetine das Wachstum von Tumoren stimulieren könnten</li></ul></li><li>in mehreren kontrollierten Studien mit Patienten, die eine Anämie im Zusammenhang mit einer Krebserkrankung aufwiesen, wurde keine Verbesserung des Gesamtüberlebens bzw. keine Verringerung des Risikos der Tumorprogression durch Epoetine gezeigt</li><li>in kontrollierten klinischen Studien hat die Anwendung von Epoetin beta und anderen Erythropoiese-stimulierenden Substanzen (ESA) Folgendes gezeigt:<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobinzielwert &gt; 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobinzielbereich von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko, wenn bei Patienten mit aktiven malignen Erkrankungen, die weder eine Chemotherapie noch eine Strahlentherapie erhielten, ein Hämoglobinzielwert von 12 g/dl (7,5 mmol/l) angestrebt wurde</li><li>ESA sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden</li><li>einer Entscheidung zur Gabe von rekombinantem Erythropoietin sollte eine Nutzen-Risiko-Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko-Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Prognose der Erkrankung</li><li>Umfeld, in dem der Patient behandelt</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>Laborwerte<ul><li>es kann zu einem Anstieg des Blutdrucks kommen, der medikamentös behandelt werden kann<ul><li>insbes. zu Beginn der Behandlung sollte deshalb der Blutdruck bei Krebspatienten kontrolliert werden</li></ul></li><li>Thrombozytenzahl und Hämoglobinkonzentration sollten bei Krebspatienten in regelmäßigen Abständen überprüft werden</li><li>bei Patienten in einem Eigenblutspendeprogramm kann es zu einem Anstieg der Thrombozytenzahl, in der Regel innerhalb des Normbereiches, kommen<ul><li>daher empfohlen, die Thrombozytenzahl mind. 1mal / Woche zu kontrollieren</li><li>falls Steigerung &gt; 150 x 10<sup>9</sup>/l oder liegt die Thrombozytenzahl über dem Normbereich, sollte die Therapie mit Epoetin beta abgebrochen werden</li></ul></li><li>Serum-Kalium- und -Phosphat-Spiegel sollten während der Behandlung mit Epoetin beta regelmäßig überprüft werden<ul><li>bei einigen urämischen Patienten wurde unter einer Behandlung mit Epoetin beta über erhöhte Kaliumspiegel berichtet, ohne dass ein kausaler Zusammenhang gezeigt wurde</li><li>sollten erhöhte oder ansteigende Kaliumspiegel beobachtet werden, dann sollte eine Unterbrechung der Behandlung mit Epoetin beta in Betracht gezogen werden, bis die Hyperkalämie korrigiert ist</li></ul></li></ul></li><li>Frühgeborene, Retinopathie<ul><li>bei Frühgeborenen konnte das potentielle Risiko, dass Erythropoietin eine Retinopathie auslöst, nicht ausgeschlossen werden<ul><li>deshalb Vorsicht bei der Anwendung</li></ul></li><li>bei der Entscheidung, ob ein Frühgeborenes behandelt wird, sollten die möglichen Nutzen und Risiken dieser Behandlung und verfügbarer alternativer Therapiemöglichkeiten gegeneinander abgewogen werden</li></ul></li><li>Patienten mit chronischer Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung ist im Verlauf der Therapie mit Epoetin beta aufgrund des gesteigerten Hämatokritwertes häufig eine Erhöhung der Heparin-Dosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialyse-Systems möglich</li></ul></li><li>frühzeitige Shunt-Revision und eine Thrombose-Prophylaxe, z.B. durch Gabe von Acetylsalicylsäure, sollte bei Patienten mit chronischer Nierenerkrankung und mit dem Risiko einer Shunt-Thrombose in Betracht gezogen werden</li></ul></li><li>Eigenblutspende<ul><li>für den Einsatz von Epoetin beta im Rahmen der Eigenblutspende müssen die offiziellen Richtlinien zur Blutspende berücksichtigt werden, insbesondere:<ul><li>nur Patienten mit einem Hämatokritwert &gt;= 33 % (Hämoglobinwert &gt;= 11 g/dl [6,83 mmol/l]) dürfen Blut spenden</li><li>Patienten mit einem Körpergewicht &lt; 50 kg sollten besonders sorgfältig beobachtet werden</li><li>das Volumen einer einzelnen Blutspende sollte nicht größer als ca. 12% des Blutvolumens des Patienten sein</li></ul></li><li>Behandlung sollte auf Patienten beschränkt werden, bei denen die Vermeidung von Fremdbluttransfusionen nach deren Nutzen-Risiko-Abschätzung als besonders wichtig angesehen wird</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Verwendung bei Gesunden kann zu einer übermäßigen Steigerung des Hämatokritwertes führen</li><li>in diesem Zusammenhang wurde über lebensbedrohliche Komplikationen des Herz-Kreislauf-Systems berichtet</li></ul></li></ul><p>Epoetin theta</p><ul><li>allgemein<ul><li>Eisensupplementierung wird bei allen Patienten mit Ferritin-Serumwerten &lt; 100 µg/l oder einer Transferrin-Sättigung &lt; 20% empfohlen<ul><li>um eine effektive Erythropoiese zu gewährleisten, muss bei allen Patienten vor und während der Behandlung der Eisenstatus bestimmt werden</li></ul></li><li>Nicht-Ansprechen auf die Behandlung mit Epoetin theta sollte Anlass für eine Suche nach ursächlichen Faktoren sein<ul><li>Eisen-, Folsäure- und Vitamin-B12-Mangelzustände verringern die Wirksamkeit von Epoetinen und müssen daher korrigiert werden</li><li>die folgenden Faktoren können eine Ansprechen der Erythropoiese beeinträchtigen <ul><li>interkurrente Infektionen</li><li>entzündliche oder traumatische Vorfälle</li><li>okkulter Blutverlust</li><li>Hämolyse</li><li>Aluminium-Intoxikation</li><li>hämatologische Grunderkrankung</li><li>Knochenmarkfibrose</li></ul></li><li>Beurteilung sollte auch die Retikulozytenzahlen umfassen</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>wenn typische Ursachen für ein Nicht-Ansprechen ausgeschlossen wurden und bei dem Patienten ein plötzlicher Abfall des Hämoglobins zusammen mit einer Retikulopenie auftritt, sind Untersuchungen auf Anti-Erythropoietin-Antikörper und eine Knochenmarkuntersuchung zur Diagnosestellung einer Erythroblastopenie in Betracht zu ziehen</li><li>Unterbrechung der Behandlung mit Epoetin theta ist in Betracht zu ziehen</li><li>unter Erythropoietin-Therapie wurden Fälle von PRCA beschrieben, die durch neutralisierende Anti-Erythropoietin-Antikörper verursacht wurden</li><li>da für diese Antikörper Kreuzreaktivität mit allen Epoetinen gezeigt wurde, sollten Patienten mit Verdacht oder Nachweis auf neutralisierende Antikörper gegen Erythropoietin nicht auf Epoetin theta umgestellt werden</li><li>bei einer paradoxen Abnahme des Hämoglobinwerts und Ausbildung einer schweren Anämie in Zusammenhang mit einer niedrigen Retikulozytenzahl ist die Behandlung mit Epoetin abzubrechen und einen Test auf Anti-Erythropoetin-Antikörper durchzuführen<ul><li>entsprechende Fälle sind bei Hepatitis C-Patienten berichtet worden, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetin erhielten</li><li>Epoetine sind nicht für die Behandlung einer Hepatitis C-assoziierten Anämie bestimmt</li></ul></li></ul></li><li>Hypertonie<ul><li>während einer Epoetin-theta-Therapie kann es zu einer Erhöhung des Blutdrucks bzw. einer Verstärkung einer bereits bestehenden Hypertonie kommen, insbes. zu Beginn der Behandlung</li><li>auf eine engmaschige Überwachung und Blutdruckkontrollen achten<ul><li>Blutdruck muss vor Beginn der Behandlung und während der Behandlung adäquat eingestellt werden, um akute Komplikationen wie eine hypertensive Krise mit Enzephalopathie-ähnlichen Symptomen (z.B. Kopfschmerzen, Verwirrtheit, Sprachstörungen, Gangstörungen) und assoziierten Komplikationen (Krampfanfälle, Schlaganfall) zu vermeiden, die auch vereinzelt bei Patienten mit ansonsten normalem oder niedrigem Blutdruck auftreten kann</li><li>beim Auftreten dieser Reaktionen ist eine umgehende ärztliche Behandlung und intensivmedizinische Überwachung erforderlich</li></ul></li><li>auf plötzlich auftretende, heftige, migräneartige Kopfschmerzen als mögliches Warnsignal sollte besonders geachtet werden</li><li>Blutdruckerhöhungen <ul><li>können eine Behandlung mit blutdrucksenkenden Arzneimitteln oder eine Dosiserhöhung einer bereits bestehenden antihypertensiven Behandlung notwendig machen</li><li>zusätzlich Reduktion der angewendeten Epoetin theta-Dosis erwägen</li><li>falls die Blutdruckwerte weiterhin erhöht bleiben, kann eine vorübergehende Unterbrechung der Therapie mit Epoetin theta erforderlich sein</li></ul></li><li>nach erfolgreicher Blutdruckeinstellung durch eine Intensivierung der Behandlung sollte die Epoetin-theta-Therapie in reduzierter Dosierung wieder aufgenommen werden</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Anwendung von Epoetin theta bei gesunden Personen kann einen übermäßigen Anstieg des Hämoglobins und des Hämatokrits zur Folge haben, was zu lebensbedrohlichen kardiovaskulären Komplikationen führen kann</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden</li><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung mit Epoetin theta unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li><li>wenn ein Patient aufgrund der Anwendung von Epoetin theta eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin theta zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>besondere Patientengruppen<ul><li>eingeschränkte Leberfunktion und Sichelzellenanämie<ul><li>wegen der begrenzten Erfahrung keine Aussagen zur Wirksamkeit und Sicherheit von Epoetin theta bei Patienten mit eingeschränkter Leberfunktion und homozygoter Sichelzellanämie möglich</li></ul></li><li>ältere Patienten &gt; 75 Jahre<ul><li>in klinischen Studien hatten Patienten über 75 Jahren eine höhere Inzidenz schwerwiegender und schwerer unerwünschter Ereignisse, unabhängig von einem Kausalzusammenhang mit der Epoetin-theta-Therapie</li><li>Todesfälle waren in dieser Patientengruppe häufiger als bei jüngeren Patienten</li></ul></li></ul></li><li>Laborkontrollen<ul><li>es wird empfohlen, regelmäßig das Hämoglobin zu messen, ein komplettes Blutbild zu erstellen und Thrombozytenzählungen durchzuführen</li></ul></li><li>Symptomatische Anämie infolge chronischer Niereninsuffizienz<ul><li>bei nephrosklerotischen Patienten, die noch keine Dialyse erhalten, ist die Anwendung von Epoetin theta individuell abzuwägen, da die Möglichkeit einer beschleunigten Progression der Niereninsuffizienz nicht sicher ausgeschlossen werden kann</li><li>während der Hämodialyse können Patienten unter Behandlung mit Epoetin theta eine intensivere gerinnungshemmende Therapie benötigen, um einem Verschluss des arterio-venösen Shunts vorzubeugen</li><li>bei Patienten mit chronischer Niereninsuffizienz sollte die Hämoglobin-Konzentration in der Erhaltungsphase die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschreiten<ul><li>in klinischen Studien wurde ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse unter einer Epoetin-Therapie beobachtet, wenn Hämoglobin-Zielspiegel &gt; 12 g/dl (7,45mmol/l) angestrebt wurde</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobin-Konzentration über den für die Symptomkontrolle einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li></ul></li><li>Vorsicht bei der Steigerung der Epoetin-theta-Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li></ul></li><li>Symptomatische Anämie bei Krebspatienten mit nicht-myeloischen malignen Erkrankungen, die eine Chemotherapie erhalten<ul><li>Einfluß auf Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, die primär die Bildung von Erythrozyten anregen</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen exprimiert werden</li><li>wie bei allen Wachstumsfaktoren gibt es Bedenken, dass Epoetine das Wachstum von Tumoren anregen könnten</li></ul></li><li>in mehreren kontrollierten Studien zeigten Epoetine keine Verbesserung des Gesamtüberlebens oder eine Verminderung des Risikos einer Tumorprogression bei Patienten mit Anämie infolge einer Krebserkrankung.</li><li>in kontrollierten klinischen Studien zeigte die Anwendung von Epoetinen:<ul><li>eine verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Halstumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobin- Zielspiegel oberhalb von 14 g/dl (8,69mmol/l) angestrebt wurde</li><li>ein verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobin-Zielwert zwischen 12 - 14 g/dl (7,45 - 8,69mmol/l) angestrebt wurde</li><li>ein erhöhtes Sterberisiko bei Patienten mit aktiven malignen Erkrankungen, dieweder eine Chemotherapie noch eine Strahlentherapie erhielten, wenn ein Hämoglobin-Zielwert von 12g/dl (7,45mmol/l) angestrebt wurde</li></ul></li><li>Epoetine sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Krebspatienten bevorzugt mit Bluttransfusionen behandelt werden</li><li>Entscheidung zur Anwendung rekombinanter Erythropoietine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des einzelnen Patienten beruhen, unter Berücksichtigung der jeweils spezifischen klinischen Umstände<ul><li>Faktoren wie Tumorart und -stadium, Schweregrad der Anämie, Lebenserwartung, Behandlungsumfeld des Patienten und Wunsch des Patienten sollten in diese Beurteilung mit einbezogen werden</li></ul></li></ul></li></ul><p>Epoetin zeta</p><ul><li>allgemein<ul><li>Blutdruck <ul><li>bei allen Patienten, die Epoetin zeta erhalten, sollte der Blutdruck überwacht und ggf. behandelt werden</li><li>Epoetin zeta sollte mit Vorsicht angewendet werden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte blutdrucksenkende Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar<ul><li>sollte die Behandlung mit Epoetin zeta beendet werden</li></ul></li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurften<ul><li>traten unter der Behandlung mit Epoetin zeta auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie, Krampfneigung<ul><li>Epoetin zeta sollte bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS- Infektionen und Hirnmetastasen, vorsichtig angewendet werden</li></ul></li><li>Patienten mit chronischer Leberinsuffizienz <ul><li>Anwendung mit Vorsicht</li><li>Sicherheit von Epoetin zeta bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignisse (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden <ul><li>dazu gehören venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>darüber hinaus wurde über zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken) berichtet</li></ul></li><li>das berichtete Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin zeta abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li></ul></li><li>Hämoglobinspiegel<ul><li>bei allen Patienten ist der Hämoglobinspiegel engmaschig zu ü berwachen, da potentiell ein erhöhtes Risiko fü r thromboembolische Ereignisse und tödliche Verläufe besteht, wenn Patienten behandelt wurden, deren Hämoglobinkonzentration über dem Konzentrationsbereich liegt, für den das Arzneimittel indiziert ist</li></ul></li><li>während der Behandlung mit Erythropoetin kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>bei fortgesetzter Behandlung rückläufig</li><li>empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Ursachen einer Anämie müssen vor Beginn einer Behandlung mit Epoetin zeta sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden<ul><li>Eisen-, Folsäure- oder Vitamin B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikationen</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarksfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen gehen gleichzeitig mit der Zunahme des gepackten Zellvolumens die Ferritinwerte im Serum zurück</li><li>für das Erzielen eines optimalen Ansprechens auf Erythropoetin müssen ausreichende Eisenreserven sichergestellt werden:<ul><li>für Patienten mit chronischer Niereninsuffizienz, deren Serum-Ferritinkonzentration unter 100 ng/ml liegt, wird eine orale Eisensubstitution empfohlen, z.B. 200 - 300 mg / Tag (100-200 mg / Tag bei pädiatrischen Patienten)</li><li>für alle Tumorpatienten, bei denen die Transferrinsättigung weniger als 20% beträgt, wird eine orale Eisensubstitution mit 200 - 300 mg / Tag empfohlen</li><li>für Patienten in einem autologen Blutspendeprogramm sollte mit der oralen Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) bereits einige Wochen vor der Eigenblutspende begonnen werden, um vor der Einleitung der Behandlung mit Epoetin zeta sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten</li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin zeta eine orale Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) erhalten<ul><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin zeta begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>sehr selten wurde die Entwicklung oder Exazerbation einer Porphyrie bei mit Epoetin zeta behandelten Patienten beobachtet</li><li>Epoetin zeta sollte bei Patienten mit Porphyrie mit Vorsicht angewendet werden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens- Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung von Epoetin eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Wirkstoffen (ESAs) zu verbessern, soll der Handelsname und die Chargennummer des angewendeten ESAs eindeutig in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung<ul><li>Patienten sollen nur unter geeigneter Überwachung von einer ESA auf ein anderes umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>übber das Auftreten einer antikörpervermittelten Erythroblastopenie (PRCA) wurde nach monate- bis jahrelanger Behandlung mit Epoetin berichtet</li><li>ebenfalls traten Fälle bei Patienten mit Hepatitis C auf, die mit Interferon und Ribavirin behandelt wurden, während ESAs begleitend eingesetzt wurden</li><li>Epoetin zeta ist für die Behandlung einer Anämie, die mit Hepatitis C assoziiert ist, nicht zugelassen</li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl pro Monat) mit erhöhtem Bedarf an Transfusionen<ul><li>sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen (z.B. Eisen-, Folsäure- oder Vitamin-B12-Mangel, Aluminiumintoxikation, Infektionen oder Entzündungen, Blutverlust, Hämolyse und Knochenmarkfibrose jeglicher Genese) abgeklärt werden</li></ul></li><li>bei einer paradoxen Verminderung der Hämoglobinwerte und der Entwicklung einer schweren Anämie, die mit einem erniedrigten Retikulozytenwert einhergeht<ul><li>sollte die Behandlung mit Epoetin zeta sofort abgesetzt und die Anti-Erythropoetin- Antikörper bestimmt werden</li><li>eine Untersuchung des Knochenmarks zur Diagnose einer PRCA sollte ebenfalls erwogen werden</li></ul></li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere ESA-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anä mie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>Hämoglobin-Bestimmungen<ul><li>bei Patienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und 2 g/dl (1,25 mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko der Zunahme eines Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von über 12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht, wenn die Epoetin-Dosis bei Patienten mit chronischer Niereninsuffizienz gesteigert wird<ul><li>da hohe kumulierte Epoetin-Dosen mit einem gesteigerten Mortalitätsrisiko, schwerwiegenden kardiovaskulären und zerebrovaskulären Ereignissen assoziiert sein können</li><li>bei Patienten, die ein schlechtes Ansprechen des Hämoglobinwertes auf Epoetin zeigen, sollten alternative Gründe für das schlechte Anprechen bedacht werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, denen Epoetin zeta subkutan gegeben wird, sollten regelmäßig Kontrollen hinsichtlich eines Wirkverlustes, definiert als ein Nicht- oder vermindertes Ansprechen auf eine Behandlung mit Epoetin zeta bei Patienten, die zuvor auf eine solche Therapie angesprochen haben, durchgeführt werden<ul><li>dies zeigt sich durch eine anhaltende Verminderung der Hämoglobinwerte trotz Steigerung der Epoetin-zeta- Dosis</li></ul></li><li>bei Verabreichung von Epoetin in längeren Dosisintervallen (länger als 1-mal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shuntthrombosen <ul><li>Shuntthrombosen sind insbesondere bei Hämodialysepatienten aufgetreten, die eine Tendenz zur Hypotonie aufweisen oder deren arteriovenöse Fisteln Komplikationen aufweisen (z.B. Stenosen, Aneurysmen, etc.)</li><li>eine frühzeitige Shuntkorrektur und Thromboseprophylaxe durch die Gabe von z.B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>Hyperkaliämie<ul><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden</li><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder ansteigendem Kaliumwert sollte zusätzlich zu einer geeigneten Behandlung der Hyperkaliämie erwogen werden, die Epoetin-zeta-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>im Verlauf der Therapie mit Erythropoetin ist aufgrund des erhöhten Hämatokrits häufig eine Steigerung der Heparindosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialysesystems möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie mit Epoetin zeta bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie, Auswirkungen auf das Tumorwachstum <ul><li>Hämoglobin-Bestimmungen<ul><li>bei Tumorpatienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>Epoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>we bei allen Wachstumsfaktoren sollte bedacht werden, dass Epoetine das Wachstum von Tumoren jeglicher Art stimulieren können</li><li>eine Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war die Anwendung von Epoetin zeta und anderen ESAs mit einer verkürzten Zeit bis zur Tumorprogression oder mit einer verkürzten Gesamtüberlebenszeit<br />verbunden <ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn eine Hämoglobin-Zielkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>kürzeres Gesamtüberleben und erhöhte Sterblichkeit, die auf eine Krankheitsprogression nach 4 Monaten bei Patienten mit metastasierendem Brustkrebs, die eine Chemotherapie erhielten, zurückgeführt wurde, wenn eine Hämoglobin-Zielkonzentration von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l) angestrebt wurde<ul><li>ESAs für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-zeta plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15 % erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 g/dl und 12 g/dl (6,2 - 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden<ul><li>einer Entscheidung zur Gabe von rekombinantem Erythropoetin sollte eine Nutzen-Risiko Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Erkrankungsprognose</li><li>Umfeld, in dem der Patient behandelt wird</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen, sollte für die Beurteilung der Therapieeffizienz von Epoetin zeta eine 2- bis 3-wöchige Verzögerung zwischen ESA-Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle speziellen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbesondere routinemäßiger Volumenersatz, sollten berü cksichtigt werden</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff vorgesehen ist<ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrunde liegenden kardiovaskulären Erkrankung<ul><li>daher sollte insbesondere bei diesen Patienten eine angemessene Thromboseprophylaxe erfolgen</li></ul></li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit Disposition für tiefe Venenthrombosen ergriffen werden</li><li>Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l)<ul><li>Epoetin zeta sollte nicht bei diesen Patienten eingesetzt werden</li><li>bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l) kann nicht ausgeschlossen werden, dass eine Epoetin zeta-Behandlung mit einem erhöhten Risiko<br />postoperativer Thrombose-/Gefäßereignissen einhergeht</li></ul></li></ul></li></ul>`,
		stopAt:      3,
	},
}

func TestAhoCorasick_ReplaceAll(t *testing.T) {
	for _, i2 := range testCasesReplace {
		builder := NewAhoCorasickBuilder(Opts{
			AsciiCaseInsensitive: true,
			MatchOnlyWholeWords:  true,
			MatchKind:            LeftMostLongestMatch,
		})

		ac := builder.Build(i2.patterns)
		r := NewReplacer(ac)
		replaced := r.ReplaceAll(i2.haystack, i2.replaceWith)
		if replaced != i2.replaced {
			t.Errorf("expected %v matches got %v", i2.replaced, replaced)
		}
	}
}

func TestAhoCorasick_ReplaceAllFunc(t *testing.T) {
	for _, i2 := range testCasesReplace {
		builder := NewAhoCorasickBuilder(Opts{
			AsciiCaseInsensitive: true,
			MatchOnlyWholeWords:  true,
			MatchKind:            LeftMostLongestMatch,
		})

		ac := builder.Build(i2.patterns)
		r := NewReplacer(ac)
		replaced := r.ReplaceAllFunc(i2.haystack, func(match Match) (string, bool) {
			return i2.replaceWith[match.pattern], true
		})
		if replaced != i2.replaced {
			t.Errorf("expected %v matches got %v", i2.replaced, replaced)
		}
	}
}

var acsNFA []Replacer

func init() {
	acsNFA = make([]Replacer, len(testCasesReplace))
	for i, t2 := range testCasesReplace {
		builder := NewAhoCorasickBuilder(Opts{
			AsciiCaseInsensitive: true,
			MatchOnlyWholeWords:  true,
			MatchKind:            LeftMostLongestMatch,
		})
		ac := builder.Build(t2.patterns)
		acsNFA[i] = NewReplacer(ac)
	}
}

func BenchmarkAhoCorasick_ReplaceAllNFA(b *testing.B) {
	for i := 0; i < b.N; i++ {
		for i, ac := range acsNFA {
			_ = ac.ReplaceAll(testCasesReplace[i].haystack, testCasesReplace[i].replaceWith)
		}
	}
}

var testCasesReplace = []testCaseReplace{
	{
		patterns:    []string{"bear", "masha"},
		replaceWith: []string{"robocop", "jinx"},
		haystack:    "The Bear and Masha",
		replaced:    "The robocop and jinx",
	},
	{
		patterns:    []string{"bear", "masha"},
		replaceWith: []string{"robocop", "jinx"},
		haystack:    "Bear and Masha",
		replaced:    "robocop and jinx",
	},
	{
		patterns:    []string{"bear", "masha"},
		replaceWith: []string{"robocop", "jinx"},
		haystack:    "The Bear is annoyed at Masha very much",
		replaced:    "The robocop is annoyed at jinx very much",
	},
	{
		patterns:    []string{"hinweise und vorsichtsmaßnahmen", "hinweisen"},
		replaceWith: []string{`<a>hinweise und vorsichtsmaßnahmen<\a>`, `<a>hinweisen<\a>`},
		haystack:    "<h2>Hinweise und Vorsichtsmaßnahmen, Wirkstoffprofil <br />(kumulative Wirkstoffangaben ohne Berücksichtigung des individuellen Zulassungsstatus)</h2> Erythropoietin - invasiv<br /><br /><p>Epoetin alfa</p><ul><li>allgemein<ul><li>Hypertonie<ul><li>bei allen Patienten sollte der Blutdruck engmaschig überwacht und ggf. behandelt werden</li><li>mit Vorsicht anwenden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte antihypertensive Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar, sollte die Behandlung beendet werden</li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurfte<ul><li>traten unter der Behandlung mit Epoetin alfa auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie<ul><li>bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS-Infektionen und Hirnmetastasen, vorsichtig anwenden</li></ul></li><li>chronische Leberinsuffizienz<ul><li>bei Patienten mit chronischer Leberinsuffizienz mit Vorsicht anwenden</li><li>Sicherheit von Epoetin alfa ist bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignissen (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden<ul><li>venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken)</li></ul></li><li>berichtetes Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin alfa abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li><li>bei allen Patienten sollten die Hämoglobinwerte engmaschig überwacht werden, da ein potentiell erhöhtes Risiko für thromboembolische Ereignisse und Mortalität besteht, wenn der Hämoglobin-Wert der Patienten oberhalb des für die jeweilige Indikation gültigen Zielbereichs liegt</li><li>während der Behandlung kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>dieser ist bei fortgesetzter Behandlung rückläufig</li></ul></li><li>es wird empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Anämieursachen sollten abgeklärt und müssen vor Beginn einer Behandlung mit Epoetin alfa sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden, z.B.<ul><li>Eisenmangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Folsäuremangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Hämolyse</li><li>Blutverlust</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen wird der Serumferritinwert simultan zum Anstieg des Hämatokritwertes abfallen; um ein optimales Ansprechen zu erzielen, müssen ausreichende Eisenspeicher gewährleistet sein, falls nötig substituieren<ul><li>orale Eisensubstitution für Patienten mit chronischer Niereninsuffizienz empfohlen, deren Serumferritinspiegel unter 100 ng/ml liegt, z.B.<ul><li>200 - 300 mg Fe2+ / Tag für Erwachsene</li><li>100 - 200 mg Fe2+ / Tag für Kinder und Jugendliche</li></ul></li><li>orale Eisensubstitution für alle Tumorpatienten, deren Transferrinsättigung unter 20% liegt<ul><li>200 - 300 mg Fe2+ / Tag</li></ul></li><li>orale Eisensubstitution für Patienten in einem autologen Blutspendeprogramm bereits einige Wochen vor der Eigenblutspende beginnen, um vor der Einleitung der Behandlung mit Epoetin alfa sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten<ul><li>200 mg Fe2+ / Tag</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin alfa eine orale Eisen-II-Substitution erhalten<ul><li>200 mg Fe2+ / Tag</li><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin alfa begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>in sehr seltenen Fällen eine Porphyrie oder Verschlimmerung einer Porphyrie beobachtet</li><li>bei diesen Patienten mit Vorsicht anwenden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen, einschließlich Stevens Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet</li><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li><li>zum Zeitpunkt der Verschreibung sollten Patienten auf Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, Behandlung unverzüglich beenden und alternative Behandlung in Betracht ziehen</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>Behandlung darf zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Substanzen (ESAs) zu verbessern, soll der Handelsname des verordneten ESAs deutlich in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung <ul><li>Patienten sollen nur unter geeigneter Überwachung von einem ESA auf einen anderen umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>über das Auftreten einer antikörpervermittelten PRCA nach monate- bis jahrelanger Behandlung mit subkutanem Erythropoetin berichtet<ul><li>v.a. bei Patienten mit chronischer Niereninsuffizienz</li><li>Fallberichte auch für Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten<ul><li>für diese Indikation besteht keine Zulassung</li></ul></li></ul></li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl bzw. 0,62 - 1,25 mmol/l pro Monat) mit erhöhtem Bedarf an Transfusionen, sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen untersucht werden, z.B.<ul><li>Eisenmangel</li><li>Folsäuremangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>bei paradoxer Abnahme des Hämoglobinwerts und Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozytenzahlen Absetzen der Behandlung und Durchführung einer Untersuchung auf Anti-Erythropoetin-Antikörper; auch Knochenmarkuntersuchung zur Diagnose einer PRCA erwägen</li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere Erythropoetin-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anämie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>regelmäßig Hämoglobin-Bestimmungen durchführen, bis ein stabiler Hämoglobin-Wert erreicht wird<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und sollte 2 g/dl (1,25mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko einer Zunahme des Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von &gt;12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative<br />Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, die mit subkutaner Gabe von Epoetin alfa behandelt werden, müssen regelmäßig auf Wirkverlust hin überwacht werden, der als keine oder verringerte Reaktion auf die Epoetin alfa-Behandlung bei Patienten definiert wird, die zuvor auf solch eine Therapie angesprochen haben<ul><li>zeigt sich durch andauernden Hämoglobinabfall trotz einer Erhöhung der Dosis von Epoetin alfa</li></ul></li><li>bei Verabreichung von Epoetin alfa in längeren Dosierungsintervallen (länger als einmal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-alfa-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shunt-Thrombosen sind bei Hämodialysepatienten, insbesondere bei Patienten mit Neigung zu Hypotonie oder bei Komplikationen an den arteriovenösen Fisteln aufgetreten (z.B. Stenosen, Aneurysmen etc.)<ul><li>frühzeitige Shuntrevision und Thromboseprophylaxe durch Gabe von z. B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden<ul><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder steigendem Kaliumwert sollte erwogen werden, die Epoetin alfa-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>aufgrund des Anstiegs des Hämatokrits im Rahmen der Epoetin-alfa-Behandlung ist häufig eine Erhöhung der Heparin-Dosis während der Dialyse notwendig<ul><li>ein Verschluss des Dialysesystems ist bei nicht optimaler Heparinisierung möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie<ul><li>regelmäßig Hämoglobin-Bestimmungen durchühren, bis ein stabiler Hämoglobin-Wert erreicht ist<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Erythropoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>es sollte bedacht werden, dass Epoetine das Wachstum von Tumoren stimulieren können</li><li>Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war Anwendung von Epoetin alfa und anderen ESAs mit verkürzter Zeit bis zur Tumorprogression oder mit verkürzter Gesamtüberlebenszeit verbunden<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürzte Gesamtüberlebenszeit und erhöhte Zahl an Todesfällen, zurückzuführen<br />auf eine Tumorprogression nach 4 Monaten, bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen Hämoglobinwerte von 12 bis 14 g/dl (7,5 bis 8,7 mmol/l) angestrebt wurden</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l), angestrebt wurde<ul><li>ESAs sind für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-alfa plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15% erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 und 12 g/dl (6,2 bis 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Tumorpatienten bevorzugt durch Bluttransfusionen behandelt werden</li></ul></li><li>Entscheidung für die Anwendung rekombinanter Erythropoetine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des jeweiligen Patienten beruhen, die auch den spezifischen klinischen Kontext berücksichtigen sollte<ul><li>in diese Beurteilung einzubeziehende Faktoren sind<ul><li>Tumortyp und -stadium</li><li>Anämiegrad</li><li>Lebenserwartung</li><li>Umgebung, in der der Patient behandelt wird</li><li>Präferenzen des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen sollte für die Beurteilung der Therapieeffizienz von eine 2 - 3-wöchige Verzögerung zwischen Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle spezifischen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbes. routinemäßiger Volumenersatz, sind zu berücksichtigen</li></ul></li><li>Patienten, bei denen ein großer elektiver orthopädischer Eingriff vorgesehen ist <ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrundeliegenden kardiovaskulären Erkrankung<ul><li>sollten eine angemessene Thromboseprophylaxe erhalten</li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit einer Prädisposition für tiefe Venenthrombosen (TVT) ergriffen werden</li></ul></li><li>darüber hinaus kann nicht ausgeschlossen werden, dass bei Behandlung mit Epoetin alfa für Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl ein erhöhtes Risiko für postoperative thrombotische/vaskuläre Ereignisse bestehen kann</li><li>aus diesem Grund sollte es bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl nicht angewendet werden</li></ul></li><li>Anwendung kann bei Dopingkontrollen zu positiven Ergebnissen führen<ul><li>Missbrauch durch Gesunde (z.B. für Dopingzwecke) kann zu übermäßigem Anstieg des Hämoglobinwertes führen; mit Risiko lebensbedrohlicher Komplikationen des Herz-Kreislauf-Systems (Thromboserisiko durch Hämokonzentration bei Polyglobulie) verbunden</li></ul></li></ul><p>Epoetin beta</p><ul><li>allgemein<ul><li>mit Vorsicht anwenden bei <ul><li>refraktärer Anämie mit Blastenexzess in Transformation</li><li>epileptiformen Erkrankungen</li><li>Thrombozytose</li><li>chronischer Leberinsuffizienz</li></ul></li><li>Folsäure- und Vitamin-B12-Mangel sollten ausgeschlossen sein, da hierdurch die Wirksamkeit vermindert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>um eine wirksame Erythropoiese sicherzustellen, sollte bei allen Patienten vor und während der Behandlung der Eisenwert bestimmt werden und ggf. eine Eisenersatztherapie gemäß den therapeutischen Richtlinien durchgeführt werden</li><li>Wirksamkeit von Epoetin beta kann durch eine Aluminiumüberladung infolge der Therapie der chronischen Niereninsuffizienz abgeschwächt werden</li><li>Prädialyse-Patienten mit Nephrosklerose <ul><li>über die Anwendung von Epoetin beta sollte bei Prädialyse-Patienten mit Nephrosklerose individuell entschieden werden, da eine beschleunigte Progression der Niereninsuffizienz bei diesen Patienten nicht mit Sicherheit ausgeschlossen werden kann</li></ul></li></ul></li><li>Aregenerative Anämie (pure red cell aplasia, PRCA)<ul><li>in Verbindung mit einer Erythropoietin-Therapie wurde über eine durch neutralisierende Antikörper gegen Erythropoietin hervorgerufene aregenerative Anämie (pure red cell aplasia) berichtet</li><li>für diese Antikörper konnte eine Kreuzreaktivität mit allen erythropoietischen Proteinen gezeigt werden</li><li>Patienten, bei denen der Verdacht auf neutralisierende Antikörper gegen Erythropoietin besteht oder bei denen diese nachgewiesen wurden, sollen deshalb nicht auf Epoetin beta umgestellt werden</li></ul></li><li>PRCA bei Patienten mit Hepatitis C<ul><li>eine paradoxe Hämoglobin-Abnahme und die Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozyten-Zahlen sollten zum Abbruch der Behandlung mit Epoetin und zur Durchführung von Tests auf Antikörper gegen Erythropoietin veranlassen</li><li>über Fälle berichtet bei Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten</li><li>Epoetine sind nicht zugelassen für die Behandlung einer Anämie in Verbindung mit Hepatitis C</li></ul></li><li>Hypertonie<ul><li>bei Patienten mit chronischer Nierenerkrankung kann eine Erhöhung des Blutdrucks erfolgen bzw. die Verstärkung einer bereits bestehenden Hypertension auftreten, insbes. in Fällen eines schnellen Hämatokrit-Anstiegs<ul><li>diese Blutdruckerhöhungen lassen sich medikamentös behandeln</li></ul></li><li>wenn der Blutdruckanstieg nicht durch medikamentöse Therapie kontrolliert werden kann, wird eine vorübergehende Unterbrechung der Therapie mit Epoetin beta empfohlen</li><li>insbes. zu Beginn der Behandlung wird darüber hinaus eine regelmäßige Blutdruckkontrolle empfohlen, auch zwischen den Dialysen</li><li>Blutdruckentgleisungen mit Enzephalopathie-ähnlicher Symptomatik können auftreten, die eine sofortige ärztliche Untersuchung und intensivmedizinische Maßnahmen erfordern<ul><li>plötzlich auftretende stechende migräneartige Kopfschmerzen sind als mögliches Warnzeichen besonders zu beachten</li></ul></li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung mit Epoetin unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>chronische Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung kann es insbesondere bei intravenöser Anwendung zu einem mäßiggradigen dosisabhängigen Anstieg der Thrombozytenzahl innerhalb des Normbereiches kommen, der im Verlauf der Therapie wieder zurückgeht</li><li>empfohlen, die Thrombozytenzahl in den ersten 8 Wochen der Therapie regelmäßig zu kontrollieren</li></ul></li><li>Hämoglobinkonzentration<ul><li>bei Patienten mit chronischer Nierenerkrankung sollte in der Erhaltungsphase die Hämoglobinkonzentration die empfohlene Obergrenze der angestrebten Hämoglobinkonzentration nicht überschreiten</li><li>in klinischen Studien wurde ein erhöhtes Sterberisiko und ein erhöhtes Risiko für schwerwiegende kardiovaskuläre Ereignisse einschl. Hirnschlag beobachtet, wenn Erythropoiese-stimulierende Substanzen (ESA) verabreicht wurden, um ein Hämoglobin &gt; 12 g/dl (7,5 mmol/l) zu erzielen</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobinkonzentration über den für die Kontrolle der Symptome einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li><li>bei Frühgeborenen kann die Thrombozytenzahl insbes. in den ersten zwei Lebenswochen leicht ansteigen, deshalb sollte die Thrombozytenzahl regelmäßig bestimmt werden</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, welche in erster Linie die Produktion von roten Blutkörperchen stimulieren</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedenster Tumorzellen gebildet werden<ul><li>wie bei allen Wachstumsfaktoren besteht der Verdacht, dass auch Epoetine das Wachstum von Tumoren stimulieren könnten</li></ul></li><li>in mehreren kontrollierten Studien mit Patienten, die eine Anämie im Zusammenhang mit einer Krebserkrankung aufwiesen, wurde keine Verbesserung des Gesamtüberlebens bzw. keine Verringerung des Risikos der Tumorprogression durch Epoetine gezeigt</li><li>in kontrollierten klinischen Studien hat die Anwendung von Epoetin beta und anderen Erythropoiese-stimulierenden Substanzen (ESA) Folgendes gezeigt:<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobinzielwert &gt; 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobinzielbereich von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko, wenn bei Patienten mit aktiven malignen Erkrankungen, die weder eine Chemotherapie noch eine Strahlentherapie erhielten, ein Hämoglobinzielwert von 12 g/dl (7,5 mmol/l) angestrebt wurde</li><li>ESA sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden</li><li>einer Entscheidung zur Gabe von rekombinantem Erythropoietin sollte eine Nutzen-Risiko-Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko-Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Prognose der Erkrankung</li><li>Umfeld, in dem der Patient behandelt</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>Laborwerte<ul><li>es kann zu einem Anstieg des Blutdrucks kommen, der medikamentös behandelt werden kann<ul><li>insbes. zu Beginn der Behandlung sollte deshalb der Blutdruck bei Krebspatienten kontrolliert werden</li></ul></li><li>Thrombozytenzahl und Hämoglobinkonzentration sollten bei Krebspatienten in regelmäßigen Abständen überprüft werden</li><li>bei Patienten in einem Eigenblutspendeprogramm kann es zu einem Anstieg der Thrombozytenzahl, in der Regel innerhalb des Normbereiches, kommen<ul><li>daher empfohlen, die Thrombozytenzahl mind. 1mal / Woche zu kontrollieren</li><li>falls Steigerung &gt; 150 x 10<sup>9</sup>/l oder liegt die Thrombozytenzahl über dem Normbereich, sollte die Therapie mit Epoetin beta abgebrochen werden</li></ul></li><li>Serum-Kalium- und -Phosphat-Spiegel sollten während der Behandlung mit Epoetin beta regelmäßig überprüft werden<ul><li>bei einigen urämischen Patienten wurde unter einer Behandlung mit Epoetin beta über erhöhte Kaliumspiegel berichtet, ohne dass ein kausaler Zusammenhang gezeigt wurde</li><li>sollten erhöhte oder ansteigende Kaliumspiegel beobachtet werden, dann sollte eine Unterbrechung der Behandlung mit Epoetin beta in Betracht gezogen werden, bis die Hyperkalämie korrigiert ist</li></ul></li></ul></li><li>Frühgeborene, Retinopathie<ul><li>bei Frühgeborenen konnte das potentielle Risiko, dass Erythropoietin eine Retinopathie auslöst, nicht ausgeschlossen werden<ul><li>deshalb Vorsicht bei der Anwendung</li></ul></li><li>bei der Entscheidung, ob ein Frühgeborenes behandelt wird, sollten die möglichen Nutzen und Risiken dieser Behandlung und verfügbarer alternativer Therapiemöglichkeiten gegeneinander abgewogen werden</li></ul></li><li>Patienten mit chronischer Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung ist im Verlauf der Therapie mit Epoetin beta aufgrund des gesteigerten Hämatokritwertes häufig eine Erhöhung der Heparin-Dosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialyse-Systems möglich</li></ul></li><li>frühzeitige Shunt-Revision und eine Thrombose-Prophylaxe, z.B. durch Gabe von Acetylsalicylsäure, sollte bei Patienten mit chronischer Nierenerkrankung und mit dem Risiko einer Shunt-Thrombose in Betracht gezogen werden</li></ul></li><li>Eigenblutspende<ul><li>für den Einsatz von Epoetin beta im Rahmen der Eigenblutspende müssen die offiziellen Richtlinien zur Blutspende berücksichtigt werden, insbesondere:<ul><li>nur Patienten mit einem Hämatokritwert &gt;= 33 % (Hämoglobinwert &gt;= 11 g/dl [6,83 mmol/l]) dürfen Blut spenden</li><li>Patienten mit einem Körpergewicht &lt; 50 kg sollten besonders sorgfältig beobachtet werden</li><li>das Volumen einer einzelnen Blutspende sollte nicht größer als ca. 12% des Blutvolumens des Patienten sein</li></ul></li><li>Behandlung sollte auf Patienten beschränkt werden, bei denen die Vermeidung von Fremdbluttransfusionen nach deren Nutzen-Risiko-Abschätzung als besonders wichtig angesehen wird</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Verwendung bei Gesunden kann zu einer übermäßigen Steigerung des Hämatokritwertes führen</li><li>in diesem Zusammenhang wurde über lebensbedrohliche Komplikationen des Herz-Kreislauf-Systems berichtet</li></ul></li></ul><p>Epoetin theta</p><ul><li>allgemein<ul><li>Eisensupplementierung wird bei allen Patienten mit Ferritin-Serumwerten &lt; 100 µg/l oder einer Transferrin-Sättigung &lt; 20% empfohlen<ul><li>um eine effektive Erythropoiese zu gewährleisten, muss bei allen Patienten vor und während der Behandlung der Eisenstatus bestimmt werden</li></ul></li><li>Nicht-Ansprechen auf die Behandlung mit Epoetin theta sollte Anlass für eine Suche nach ursächlichen Faktoren sein<ul><li>Eisen-, Folsäure- und Vitamin-B12-Mangelzustände verringern die Wirksamkeit von Epoetinen und müssen daher korrigiert werden</li><li>die folgenden Faktoren können eine Ansprechen der Erythropoiese beeinträchtigen <ul><li>interkurrente Infektionen</li><li>entzündliche oder traumatische Vorfälle</li><li>okkulter Blutverlust</li><li>Hämolyse</li><li>Aluminium-Intoxikation</li><li>hämatologische Grunderkrankung</li><li>Knochenmarkfibrose</li></ul></li><li>Beurteilung sollte auch die Retikulozytenzahlen umfassen</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>wenn typische Ursachen für ein Nicht-Ansprechen ausgeschlossen wurden und bei dem Patienten ein plötzlicher Abfall des Hämoglobins zusammen mit einer Retikulopenie auftritt, sind Untersuchungen auf Anti-Erythropoietin-Antikörper und eine Knochenmarkuntersuchung zur Diagnosestellung einer Erythroblastopenie in Betracht zu ziehen</li><li>Unterbrechung der Behandlung mit Epoetin theta ist in Betracht zu ziehen</li><li>unter Erythropoietin-Therapie wurden Fälle von PRCA beschrieben, die durch neutralisierende Anti-Erythropoietin-Antikörper verursacht wurden</li><li>da für diese Antikörper Kreuzreaktivität mit allen Epoetinen gezeigt wurde, sollten Patienten mit Verdacht oder Nachweis auf neutralisierende Antikörper gegen Erythropoietin nicht auf Epoetin theta umgestellt werden</li><li>bei einer paradoxen Abnahme des Hämoglobinwerts und Ausbildung einer schweren Anämie in Zusammenhang mit einer niedrigen Retikulozytenzahl ist die Behandlung mit Epoetin abzubrechen und einen Test auf Anti-Erythropoetin-Antikörper durchzuführen<ul><li>entsprechende Fälle sind bei Hepatitis C-Patienten berichtet worden, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetin erhielten</li><li>Epoetine sind nicht für die Behandlung einer Hepatitis C-assoziierten Anämie bestimmt</li></ul></li></ul></li><li>Hypertonie<ul><li>während einer Epoetin-theta-Therapie kann es zu einer Erhöhung des Blutdrucks bzw. einer Verstärkung einer bereits bestehenden Hypertonie kommen, insbes. zu Beginn der Behandlung</li><li>auf eine engmaschige Überwachung und Blutdruckkontrollen achten<ul><li>Blutdruck muss vor Beginn der Behandlung und während der Behandlung adäquat eingestellt werden, um akute Komplikationen wie eine hypertensive Krise mit Enzephalopathie-ähnlichen Symptomen (z.B. Kopfschmerzen, Verwirrtheit, Sprachstörungen, Gangstörungen) und assoziierten Komplikationen (Krampfanfälle, Schlaganfall) zu vermeiden, die auch vereinzelt bei Patienten mit ansonsten normalem oder niedrigem Blutdruck auftreten kann</li><li>beim Auftreten dieser Reaktionen ist eine umgehende ärztliche Behandlung und intensivmedizinische Überwachung erforderlich</li></ul></li><li>auf plötzlich auftretende, heftige, migräneartige Kopfschmerzen als mögliches Warnsignal sollte besonders geachtet werden</li><li>Blutdruckerhöhungen <ul><li>können eine Behandlung mit blutdrucksenkenden Arzneimitteln oder eine Dosiserhöhung einer bereits bestehenden antihypertensiven Behandlung notwendig machen</li><li>zusätzlich Reduktion der angewendeten Epoetin theta-Dosis erwägen</li><li>falls die Blutdruckwerte weiterhin erhöht bleiben, kann eine vorübergehende Unterbrechung der Therapie mit Epoetin theta erforderlich sein</li></ul></li><li>nach erfolgreicher Blutdruckeinstellung durch eine Intensivierung der Behandlung sollte die Epoetin-theta-Therapie in reduzierter Dosierung wieder aufgenommen werden</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Anwendung von Epoetin theta bei gesunden Personen kann einen übermäßigen Anstieg des Hämoglobins und des Hämatokrits zur Folge haben, was zu lebensbedrohlichen kardiovaskulären Komplikationen führen kann</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden</li><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung mit Epoetin theta unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li><li>wenn ein Patient aufgrund der Anwendung von Epoetin theta eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin theta zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>besondere Patientengruppen<ul><li>eingeschränkte Leberfunktion und Sichelzellenanämie<ul><li>wegen der begrenzten Erfahrung keine Aussagen zur Wirksamkeit und Sicherheit von Epoetin theta bei Patienten mit eingeschränkter Leberfunktion und homozygoter Sichelzellanämie möglich</li></ul></li><li>ältere Patienten &gt; 75 Jahre<ul><li>in klinischen Studien hatten Patienten über 75 Jahren eine höhere Inzidenz schwerwiegender und schwerer unerwünschter Ereignisse, unabhängig von einem Kausalzusammenhang mit der Epoetin-theta-Therapie</li><li>Todesfälle waren in dieser Patientengruppe häufiger als bei jüngeren Patienten</li></ul></li></ul></li><li>Laborkontrollen<ul><li>es wird empfohlen, regelmäßig das Hämoglobin zu messen, ein komplettes Blutbild zu erstellen und Thrombozytenzählungen durchzuführen</li></ul></li><li>Symptomatische Anämie infolge chronischer Niereninsuffizienz<ul><li>bei nephrosklerotischen Patienten, die noch keine Dialyse erhalten, ist die Anwendung von Epoetin theta individuell abzuwägen, da die Möglichkeit einer beschleunigten Progression der Niereninsuffizienz nicht sicher ausgeschlossen werden kann</li><li>während der Hämodialyse können Patienten unter Behandlung mit Epoetin theta eine intensivere gerinnungshemmende Therapie benötigen, um einem Verschluss des arterio-venösen Shunts vorzubeugen</li><li>bei Patienten mit chronischer Niereninsuffizienz sollte die Hämoglobin-Konzentration in der Erhaltungsphase die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschreiten<ul><li>in klinischen Studien wurde ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse unter einer Epoetin-Therapie beobachtet, wenn Hämoglobin-Zielspiegel &gt; 12 g/dl (7,45mmol/l) angestrebt wurde</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobin-Konzentration über den für die Symptomkontrolle einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li></ul></li><li>Vorsicht bei der Steigerung der Epoetin-theta-Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li></ul></li><li>Symptomatische Anämie bei Krebspatienten mit nicht-myeloischen malignen Erkrankungen, die eine Chemotherapie erhalten<ul><li>Einfluß auf Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, die primär die Bildung von Erythrozyten anregen</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen exprimiert werden</li><li>wie bei allen Wachstumsfaktoren gibt es Bedenken, dass Epoetine das Wachstum von Tumoren anregen könnten</li></ul></li><li>in mehreren kontrollierten Studien zeigten Epoetine keine Verbesserung des Gesamtüberlebens oder eine Verminderung des Risikos einer Tumorprogression bei Patienten mit Anämie infolge einer Krebserkrankung.</li><li>in kontrollierten klinischen Studien zeigte die Anwendung von Epoetinen:<ul><li>eine verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Halstumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobin- Zielspiegel oberhalb von 14 g/dl (8,69mmol/l) angestrebt wurde</li><li>ein verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobin-Zielwert zwischen 12 - 14 g/dl (7,45 - 8,69mmol/l) angestrebt wurde</li><li>ein erhöhtes Sterberisiko bei Patienten mit aktiven malignen Erkrankungen, dieweder eine Chemotherapie noch eine Strahlentherapie erhielten, wenn ein Hämoglobin-Zielwert von 12g/dl (7,45mmol/l) angestrebt wurde</li></ul></li><li>Epoetine sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Krebspatienten bevorzugt mit Bluttransfusionen behandelt werden</li><li>Entscheidung zur Anwendung rekombinanter Erythropoietine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des einzelnen Patienten beruhen, unter Berücksichtigung der jeweils spezifischen klinischen Umstände<ul><li>Faktoren wie Tumorart und -stadium, Schweregrad der Anämie, Lebenserwartung, Behandlungsumfeld des Patienten und Wunsch des Patienten sollten in diese Beurteilung mit einbezogen werden</li></ul></li></ul></li></ul><p>Epoetin zeta</p><ul><li>allgemein<ul><li>Blutdruck <ul><li>bei allen Patienten, die Epoetin zeta erhalten, sollte der Blutdruck überwacht und ggf. behandelt werden</li><li>Epoetin zeta sollte mit Vorsicht angewendet werden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte blutdrucksenkende Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar<ul><li>sollte die Behandlung mit Epoetin zeta beendet werden</li></ul></li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurften<ul><li>traten unter der Behandlung mit Epoetin zeta auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie, Krampfneigung<ul><li>Epoetin zeta sollte bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS- Infektionen und Hirnmetastasen, vorsichtig angewendet werden</li></ul></li><li>Patienten mit chronischer Leberinsuffizienz <ul><li>Anwendung mit Vorsicht</li><li>Sicherheit von Epoetin zeta bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignisse (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden <ul><li>dazu gehören venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>darüber hinaus wurde über zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken) berichtet</li></ul></li><li>das berichtete Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin zeta abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li></ul></li><li>Hämoglobinspiegel<ul><li>bei allen Patienten ist der Hämoglobinspiegel engmaschig zu ü berwachen, da potentiell ein erhöhtes Risiko fü r thromboembolische Ereignisse und tödliche Verläufe besteht, wenn Patienten behandelt wurden, deren Hämoglobinkonzentration über dem Konzentrationsbereich liegt, für den das Arzneimittel indiziert ist</li></ul></li><li>während der Behandlung mit Erythropoetin kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>bei fortgesetzter Behandlung rückläufig</li><li>empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Ursachen einer Anämie müssen vor Beginn einer Behandlung mit Epoetin zeta sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden<ul><li>Eisen-, Folsäure- oder Vitamin B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikationen</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarksfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen gehen gleichzeitig mit der Zunahme des gepackten Zellvolumens die Ferritinwerte im Serum zurück</li><li>für das Erzielen eines optimalen Ansprechens auf Erythropoetin müssen ausreichende Eisenreserven sichergestellt werden:<ul><li>für Patienten mit chronischer Niereninsuffizienz, deren Serum-Ferritinkonzentration unter 100 ng/ml liegt, wird eine orale Eisensubstitution empfohlen, z.B. 200 - 300 mg / Tag (100-200 mg / Tag bei pädiatrischen Patienten)</li><li>für alle Tumorpatienten, bei denen die Transferrinsättigung weniger als 20% beträgt, wird eine orale Eisensubstitution mit 200 - 300 mg / Tag empfohlen</li><li>für Patienten in einem autologen Blutspendeprogramm sollte mit der oralen Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) bereits einige Wochen vor der Eigenblutspende begonnen werden, um vor der Einleitung der Behandlung mit Epoetin zeta sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten</li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin zeta eine orale Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) erhalten<ul><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin zeta begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>sehr selten wurde die Entwicklung oder Exazerbation einer Porphyrie bei mit Epoetin zeta behandelten Patienten beobachtet</li><li>Epoetin zeta sollte bei Patienten mit Porphyrie mit Vorsicht angewendet werden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens- Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung von Epoetin eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Wirkstoffen (ESAs) zu verbessern, soll der Handelsname und die Chargennummer des angewendeten ESAs eindeutig in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung<ul><li>Patienten sollen nur unter geeigneter Überwachung von einer ESA auf ein anderes umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>übber das Auftreten einer antikörpervermittelten Erythroblastopenie (PRCA) wurde nach monate- bis jahrelanger Behandlung mit Epoetin berichtet</li><li>ebenfalls traten Fälle bei Patienten mit Hepatitis C auf, die mit Interferon und Ribavirin behandelt wurden, während ESAs begleitend eingesetzt wurden</li><li>Epoetin zeta ist für die Behandlung einer Anämie, die mit Hepatitis C assoziiert ist, nicht zugelassen</li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl pro Monat) mit erhöhtem Bedarf an Transfusionen<ul><li>sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen (z.B. Eisen-, Folsäure- oder Vitamin-B12-Mangel, Aluminiumintoxikation, Infektionen oder Entzündungen, Blutverlust, Hämolyse und Knochenmarkfibrose jeglicher Genese) abgeklärt werden</li></ul></li><li>bei einer paradoxen Verminderung der Hämoglobinwerte und der Entwicklung einer schweren Anämie, die mit einem erniedrigten Retikulozytenwert einhergeht<ul><li>sollte die Behandlung mit Epoetin zeta sofort abgesetzt und die Anti-Erythropoetin- Antikörper bestimmt werden</li><li>eine Untersuchung des Knochenmarks zur Diagnose einer PRCA sollte ebenfalls erwogen werden</li></ul></li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere ESA-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anä mie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>Hämoglobin-Bestimmungen<ul><li>bei Patienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und 2 g/dl (1,25 mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko der Zunahme eines Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von über 12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht, wenn die Epoetin-Dosis bei Patienten mit chronischer Niereninsuffizienz gesteigert wird<ul><li>da hohe kumulierte Epoetin-Dosen mit einem gesteigerten Mortalitätsrisiko, schwerwiegenden kardiovaskulären und zerebrovaskulären Ereignissen assoziiert sein können</li><li>bei Patienten, die ein schlechtes Ansprechen des Hämoglobinwertes auf Epoetin zeigen, sollten alternative Gründe für das schlechte Anprechen bedacht werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, denen Epoetin zeta subkutan gegeben wird, sollten regelmäßig Kontrollen hinsichtlich eines Wirkverlustes, definiert als ein Nicht- oder vermindertes Ansprechen auf eine Behandlung mit Epoetin zeta bei Patienten, die zuvor auf eine solche Therapie angesprochen haben, durchgeführt werden<ul><li>dies zeigt sich durch eine anhaltende Verminderung der Hämoglobinwerte trotz Steigerung der Epoetin-zeta- Dosis</li></ul></li><li>bei Verabreichung von Epoetin in längeren Dosisintervallen (länger als 1-mal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shuntthrombosen <ul><li>Shuntthrombosen sind insbesondere bei Hämodialysepatienten aufgetreten, die eine Tendenz zur Hypotonie aufweisen oder deren arteriovenöse Fisteln Komplikationen aufweisen (z.B. Stenosen, Aneurysmen, etc.)</li><li>eine frühzeitige Shuntkorrektur und Thromboseprophylaxe durch die Gabe von z.B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>Hyperkaliämie<ul><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden</li><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder ansteigendem Kaliumwert sollte zusätzlich zu einer geeigneten Behandlung der Hyperkaliämie erwogen werden, die Epoetin-zeta-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>im Verlauf der Therapie mit Erythropoetin ist aufgrund des erhöhten Hämatokrits häufig eine Steigerung der Heparindosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialysesystems möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie mit Epoetin zeta bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie, Auswirkungen auf das Tumorwachstum <ul><li>Hämoglobin-Bestimmungen<ul><li>bei Tumorpatienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>Epoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>we bei allen Wachstumsfaktoren sollte bedacht werden, dass Epoetine das Wachstum von Tumoren jeglicher Art stimulieren können</li><li>eine Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war die Anwendung von Epoetin zeta und anderen ESAs mit einer verkürzten Zeit bis zur Tumorprogression oder mit einer verkürzten Gesamtüberlebenszeit<br />verbunden <ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn eine Hämoglobin-Zielkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>kürzeres Gesamtüberleben und erhöhte Sterblichkeit, die auf eine Krankheitsprogression nach 4 Monaten bei Patienten mit metastasierendem Brustkrebs, die eine Chemotherapie erhielten, zurückgeführt wurde, wenn eine Hämoglobin-Zielkonzentration von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l) angestrebt wurde<ul><li>ESAs für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-zeta plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15 % erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 g/dl und 12 g/dl (6,2 - 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden<ul><li>einer Entscheidung zur Gabe von rekombinantem Erythropoetin sollte eine Nutzen-Risiko Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Erkrankungsprognose</li><li>Umfeld, in dem der Patient behandelt wird</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen, sollte für die Beurteilung der Therapieeffizienz von Epoetin zeta eine 2- bis 3-wöchige Verzögerung zwischen ESA-Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle speziellen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbesondere routinemäßiger Volumenersatz, sollten berü cksichtigt werden</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff vorgesehen ist<ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrunde liegenden kardiovaskulären Erkrankung<ul><li>daher sollte insbesondere bei diesen Patienten eine angemessene Thromboseprophylaxe erfolgen</li></ul></li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit Disposition für tiefe Venenthrombosen ergriffen werden</li><li>Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l)<ul><li>Epoetin zeta sollte nicht bei diesen Patienten eingesetzt werden</li><li>bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l) kann nicht ausgeschlossen werden, dass eine Epoetin zeta-Behandlung mit einem erhöhten Risiko<br />postoperativer Thrombose-/Gefäßereignissen einhergeht</li></ul></li></ul></li></ul>",
		replaced:    `<h2><a>hinweise und vorsichtsmaßnahmen<\a>, Wirkstoffprofil <br />(kumulative Wirkstoffangaben ohne Berücksichtigung des individuellen Zulassungsstatus)</h2> Erythropoietin - invasiv<br /><br /><p>Epoetin alfa</p><ul><li>allgemein<ul><li>Hypertonie<ul><li>bei allen Patienten sollte der Blutdruck engmaschig überwacht und ggf. behandelt werden</li><li>mit Vorsicht anwenden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte antihypertensive Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar, sollte die Behandlung beendet werden</li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurfte<ul><li>traten unter der Behandlung mit Epoetin alfa auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie<ul><li>bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS-Infektionen und Hirnmetastasen, vorsichtig anwenden</li></ul></li><li>chronische Leberinsuffizienz<ul><li>bei Patienten mit chronischer Leberinsuffizienz mit Vorsicht anwenden</li><li>Sicherheit von Epoetin alfa ist bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignissen (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden<ul><li>venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken)</li></ul></li><li>berichtetes Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin alfa abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li><li>bei allen Patienten sollten die Hämoglobinwerte engmaschig überwacht werden, da ein potentiell erhöhtes Risiko für thromboembolische Ereignisse und Mortalität besteht, wenn der Hämoglobin-Wert der Patienten oberhalb des für die jeweilige Indikation gültigen Zielbereichs liegt</li><li>während der Behandlung kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>dieser ist bei fortgesetzter Behandlung rückläufig</li></ul></li><li>es wird empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Anämieursachen sollten abgeklärt und müssen vor Beginn einer Behandlung mit Epoetin alfa sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden, z.B.<ul><li>Eisenmangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Folsäuremangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Hämolyse</li><li>Blutverlust</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen wird der Serumferritinwert simultan zum Anstieg des Hämatokritwertes abfallen; um ein optimales Ansprechen zu erzielen, müssen ausreichende Eisenspeicher gewährleistet sein, falls nötig substituieren<ul><li>orale Eisensubstitution für Patienten mit chronischer Niereninsuffizienz empfohlen, deren Serumferritinspiegel unter 100 ng/ml liegt, z.B.<ul><li>200 - 300 mg Fe2+ / Tag für Erwachsene</li><li>100 - 200 mg Fe2+ / Tag für Kinder und Jugendliche</li></ul></li><li>orale Eisensubstitution für alle Tumorpatienten, deren Transferrinsättigung unter 20% liegt<ul><li>200 - 300 mg Fe2+ / Tag</li></ul></li><li>orale Eisensubstitution für Patienten in einem autologen Blutspendeprogramm bereits einige Wochen vor der Eigenblutspende beginnen, um vor der Einleitung der Behandlung mit Epoetin alfa sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten<ul><li>200 mg Fe2+ / Tag</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin alfa eine orale Eisen-II-Substitution erhalten<ul><li>200 mg Fe2+ / Tag</li><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin alfa begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>in sehr seltenen Fällen eine Porphyrie oder Verschlimmerung einer Porphyrie beobachtet</li><li>bei diesen Patienten mit Vorsicht anwenden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen, einschließlich Stevens Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet</li><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li><li>zum Zeitpunkt der Verschreibung sollten Patienten auf Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen <a>hinweisen<\a>, auftreten, Behandlung unverzüglich beenden und alternative Behandlung in Betracht ziehen</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>Behandlung darf zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Substanzen (ESAs) zu verbessern, soll der Handelsname des verordneten ESAs deutlich in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung <ul><li>Patienten sollen nur unter geeigneter Überwachung von einem ESA auf einen anderen umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>über das Auftreten einer antikörpervermittelten PRCA nach monate- bis jahrelanger Behandlung mit subkutanem Erythropoetin berichtet<ul><li>v.a. bei Patienten mit chronischer Niereninsuffizienz</li><li>Fallberichte auch für Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten<ul><li>für diese Indikation besteht keine Zulassung</li></ul></li></ul></li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl bzw. 0,62 - 1,25 mmol/l pro Monat) mit erhöhtem Bedarf an Transfusionen, sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen untersucht werden, z.B.<ul><li>Eisenmangel</li><li>Folsäuremangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>bei paradoxer Abnahme des Hämoglobinwerts und Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozytenzahlen Absetzen der Behandlung und Durchführung einer Untersuchung auf Anti-Erythropoetin-Antikörper; auch Knochenmarkuntersuchung zur Diagnose einer PRCA erwägen</li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere Erythropoetin-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anämie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>regelmäßig Hämoglobin-Bestimmungen durchführen, bis ein stabiler Hämoglobin-Wert erreicht wird<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und sollte 2 g/dl (1,25mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko einer Zunahme des Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von &gt;12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative<br />Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, die mit subkutaner Gabe von Epoetin alfa behandelt werden, müssen regelmäßig auf Wirkverlust hin überwacht werden, der als keine oder verringerte Reaktion auf die Epoetin alfa-Behandlung bei Patienten definiert wird, die zuvor auf solch eine Therapie angesprochen haben<ul><li>zeigt sich durch andauernden Hämoglobinabfall trotz einer Erhöhung der Dosis von Epoetin alfa</li></ul></li><li>bei Verabreichung von Epoetin alfa in längeren Dosierungsintervallen (länger als einmal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-alfa-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shunt-Thrombosen sind bei Hämodialysepatienten, insbesondere bei Patienten mit Neigung zu Hypotonie oder bei Komplikationen an den arteriovenösen Fisteln aufgetreten (z.B. Stenosen, Aneurysmen etc.)<ul><li>frühzeitige Shuntrevision und Thromboseprophylaxe durch Gabe von z. B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden<ul><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder steigendem Kaliumwert sollte erwogen werden, die Epoetin alfa-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>aufgrund des Anstiegs des Hämatokrits im Rahmen der Epoetin-alfa-Behandlung ist häufig eine Erhöhung der Heparin-Dosis während der Dialyse notwendig<ul><li>ein Verschluss des Dialysesystems ist bei nicht optimaler Heparinisierung möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie<ul><li>regelmäßig Hämoglobin-Bestimmungen durchühren, bis ein stabiler Hämoglobin-Wert erreicht ist<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Erythropoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>es sollte bedacht werden, dass Epoetine das Wachstum von Tumoren stimulieren können</li><li>Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war Anwendung von Epoetin alfa und anderen ESAs mit verkürzter Zeit bis zur Tumorprogression oder mit verkürzter Gesamtüberlebenszeit verbunden<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürzte Gesamtüberlebenszeit und erhöhte Zahl an Todesfällen, zurückzuführen<br />auf eine Tumorprogression nach 4 Monaten, bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen Hämoglobinwerte von 12 bis 14 g/dl (7,5 bis 8,7 mmol/l) angestrebt wurden</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l), angestrebt wurde<ul><li>ESAs sind für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-alfa plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15% erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 und 12 g/dl (6,2 bis 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Tumorpatienten bevorzugt durch Bluttransfusionen behandelt werden</li></ul></li><li>Entscheidung für die Anwendung rekombinanter Erythropoetine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des jeweiligen Patienten beruhen, die auch den spezifischen klinischen Kontext berücksichtigen sollte<ul><li>in diese Beurteilung einzubeziehende Faktoren sind<ul><li>Tumortyp und -stadium</li><li>Anämiegrad</li><li>Lebenserwartung</li><li>Umgebung, in der der Patient behandelt wird</li><li>Präferenzen des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen sollte für die Beurteilung der Therapieeffizienz von eine 2 - 3-wöchige Verzögerung zwischen Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle spezifischen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbes. routinemäßiger Volumenersatz, sind zu berücksichtigen</li></ul></li><li>Patienten, bei denen ein großer elektiver orthopädischer Eingriff vorgesehen ist <ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrundeliegenden kardiovaskulären Erkrankung<ul><li>sollten eine angemessene Thromboseprophylaxe erhalten</li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit einer Prädisposition für tiefe Venenthrombosen (TVT) ergriffen werden</li></ul></li><li>darüber hinaus kann nicht ausgeschlossen werden, dass bei Behandlung mit Epoetin alfa für Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl ein erhöhtes Risiko für postoperative thrombotische/vaskuläre Ereignisse bestehen kann</li><li>aus diesem Grund sollte es bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl nicht angewendet werden</li></ul></li><li>Anwendung kann bei Dopingkontrollen zu positiven Ergebnissen führen<ul><li>Missbrauch durch Gesunde (z.B. für Dopingzwecke) kann zu übermäßigem Anstieg des Hämoglobinwertes führen; mit Risiko lebensbedrohlicher Komplikationen des Herz-Kreislauf-Systems (Thromboserisiko durch Hämokonzentration bei Polyglobulie) verbunden</li></ul></li></ul><p>Epoetin beta</p><ul><li>allgemein<ul><li>mit Vorsicht anwenden bei <ul><li>refraktärer Anämie mit Blastenexzess in Transformation</li><li>epileptiformen Erkrankungen</li><li>Thrombozytose</li><li>chronischer Leberinsuffizienz</li></ul></li><li>Folsäure- und Vitamin-B12-Mangel sollten ausgeschlossen sein, da hierdurch die Wirksamkeit vermindert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>um eine wirksame Erythropoiese sicherzustellen, sollte bei allen Patienten vor und während der Behandlung der Eisenwert bestimmt werden und ggf. eine Eisenersatztherapie gemäß den therapeutischen Richtlinien durchgeführt werden</li><li>Wirksamkeit von Epoetin beta kann durch eine Aluminiumüberladung infolge der Therapie der chronischen Niereninsuffizienz abgeschwächt werden</li><li>Prädialyse-Patienten mit Nephrosklerose <ul><li>über die Anwendung von Epoetin beta sollte bei Prädialyse-Patienten mit Nephrosklerose individuell entschieden werden, da eine beschleunigte Progression der Niereninsuffizienz bei diesen Patienten nicht mit Sicherheit ausgeschlossen werden kann</li></ul></li></ul></li><li>Aregenerative Anämie (pure red cell aplasia, PRCA)<ul><li>in Verbindung mit einer Erythropoietin-Therapie wurde über eine durch neutralisierende Antikörper gegen Erythropoietin hervorgerufene aregenerative Anämie (pure red cell aplasia) berichtet</li><li>für diese Antikörper konnte eine Kreuzreaktivität mit allen erythropoietischen Proteinen gezeigt werden</li><li>Patienten, bei denen der Verdacht auf neutralisierende Antikörper gegen Erythropoietin besteht oder bei denen diese nachgewiesen wurden, sollen deshalb nicht auf Epoetin beta umgestellt werden</li></ul></li><li>PRCA bei Patienten mit Hepatitis C<ul><li>eine paradoxe Hämoglobin-Abnahme und die Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozyten-Zahlen sollten zum Abbruch der Behandlung mit Epoetin und zur Durchführung von Tests auf Antikörper gegen Erythropoietin veranlassen</li><li>über Fälle berichtet bei Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten</li><li>Epoetine sind nicht zugelassen für die Behandlung einer Anämie in Verbindung mit Hepatitis C</li></ul></li><li>Hypertonie<ul><li>bei Patienten mit chronischer Nierenerkrankung kann eine Erhöhung des Blutdrucks erfolgen bzw. die Verstärkung einer bereits bestehenden Hypertension auftreten, insbes. in Fällen eines schnellen Hämatokrit-Anstiegs<ul><li>diese Blutdruckerhöhungen lassen sich medikamentös behandeln</li></ul></li><li>wenn der Blutdruckanstieg nicht durch medikamentöse Therapie kontrolliert werden kann, wird eine vorübergehende Unterbrechung der Therapie mit Epoetin beta empfohlen</li><li>insbes. zu Beginn der Behandlung wird darüber hinaus eine regelmäßige Blutdruckkontrolle empfohlen, auch zwischen den Dialysen</li><li>Blutdruckentgleisungen mit Enzephalopathie-ähnlicher Symptomatik können auftreten, die eine sofortige ärztliche Untersuchung und intensivmedizinische Maßnahmen erfordern<ul><li>plötzlich auftretende stechende migräneartige Kopfschmerzen sind als mögliches Warnzeichen besonders zu beachten</li></ul></li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen <a>hinweisen<\a>, auftreten, sollte die Behandlung mit Epoetin unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>chronische Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung kann es insbesondere bei intravenöser Anwendung zu einem mäßiggradigen dosisabhängigen Anstieg der Thrombozytenzahl innerhalb des Normbereiches kommen, der im Verlauf der Therapie wieder zurückgeht</li><li>empfohlen, die Thrombozytenzahl in den ersten 8 Wochen der Therapie regelmäßig zu kontrollieren</li></ul></li><li>Hämoglobinkonzentration<ul><li>bei Patienten mit chronischer Nierenerkrankung sollte in der Erhaltungsphase die Hämoglobinkonzentration die empfohlene Obergrenze der angestrebten Hämoglobinkonzentration nicht überschreiten</li><li>in klinischen Studien wurde ein erhöhtes Sterberisiko und ein erhöhtes Risiko für schwerwiegende kardiovaskuläre Ereignisse einschl. Hirnschlag beobachtet, wenn Erythropoiese-stimulierende Substanzen (ESA) verabreicht wurden, um ein Hämoglobin &gt; 12 g/dl (7,5 mmol/l) zu erzielen</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobinkonzentration über den für die Kontrolle der Symptome einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li><li>bei Frühgeborenen kann die Thrombozytenzahl insbes. in den ersten zwei Lebenswochen leicht ansteigen, deshalb sollte die Thrombozytenzahl regelmäßig bestimmt werden</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, welche in erster Linie die Produktion von roten Blutkörperchen stimulieren</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedenster Tumorzellen gebildet werden<ul><li>wie bei allen Wachstumsfaktoren besteht der Verdacht, dass auch Epoetine das Wachstum von Tumoren stimulieren könnten</li></ul></li><li>in mehreren kontrollierten Studien mit Patienten, die eine Anämie im Zusammenhang mit einer Krebserkrankung aufwiesen, wurde keine Verbesserung des Gesamtüberlebens bzw. keine Verringerung des Risikos der Tumorprogression durch Epoetine gezeigt</li><li>in kontrollierten klinischen Studien hat die Anwendung von Epoetin beta und anderen Erythropoiese-stimulierenden Substanzen (ESA) Folgendes gezeigt:<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobinzielwert &gt; 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobinzielbereich von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko, wenn bei Patienten mit aktiven malignen Erkrankungen, die weder eine Chemotherapie noch eine Strahlentherapie erhielten, ein Hämoglobinzielwert von 12 g/dl (7,5 mmol/l) angestrebt wurde</li><li>ESA sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden</li><li>einer Entscheidung zur Gabe von rekombinantem Erythropoietin sollte eine Nutzen-Risiko-Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko-Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Prognose der Erkrankung</li><li>Umfeld, in dem der Patient behandelt</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>Laborwerte<ul><li>es kann zu einem Anstieg des Blutdrucks kommen, der medikamentös behandelt werden kann<ul><li>insbes. zu Beginn der Behandlung sollte deshalb der Blutdruck bei Krebspatienten kontrolliert werden</li></ul></li><li>Thrombozytenzahl und Hämoglobinkonzentration sollten bei Krebspatienten in regelmäßigen Abständen überprüft werden</li><li>bei Patienten in einem Eigenblutspendeprogramm kann es zu einem Anstieg der Thrombozytenzahl, in der Regel innerhalb des Normbereiches, kommen<ul><li>daher empfohlen, die Thrombozytenzahl mind. 1mal / Woche zu kontrollieren</li><li>falls Steigerung &gt; 150 x 10<sup>9</sup>/l oder liegt die Thrombozytenzahl über dem Normbereich, sollte die Therapie mit Epoetin beta abgebrochen werden</li></ul></li><li>Serum-Kalium- und -Phosphat-Spiegel sollten während der Behandlung mit Epoetin beta regelmäßig überprüft werden<ul><li>bei einigen urämischen Patienten wurde unter einer Behandlung mit Epoetin beta über erhöhte Kaliumspiegel berichtet, ohne dass ein kausaler Zusammenhang gezeigt wurde</li><li>sollten erhöhte oder ansteigende Kaliumspiegel beobachtet werden, dann sollte eine Unterbrechung der Behandlung mit Epoetin beta in Betracht gezogen werden, bis die Hyperkalämie korrigiert ist</li></ul></li></ul></li><li>Frühgeborene, Retinopathie<ul><li>bei Frühgeborenen konnte das potentielle Risiko, dass Erythropoietin eine Retinopathie auslöst, nicht ausgeschlossen werden<ul><li>deshalb Vorsicht bei der Anwendung</li></ul></li><li>bei der Entscheidung, ob ein Frühgeborenes behandelt wird, sollten die möglichen Nutzen und Risiken dieser Behandlung und verfügbarer alternativer Therapiemöglichkeiten gegeneinander abgewogen werden</li></ul></li><li>Patienten mit chronischer Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung ist im Verlauf der Therapie mit Epoetin beta aufgrund des gesteigerten Hämatokritwertes häufig eine Erhöhung der Heparin-Dosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialyse-Systems möglich</li></ul></li><li>frühzeitige Shunt-Revision und eine Thrombose-Prophylaxe, z.B. durch Gabe von Acetylsalicylsäure, sollte bei Patienten mit chronischer Nierenerkrankung und mit dem Risiko einer Shunt-Thrombose in Betracht gezogen werden</li></ul></li><li>Eigenblutspende<ul><li>für den Einsatz von Epoetin beta im Rahmen der Eigenblutspende müssen die offiziellen Richtlinien zur Blutspende berücksichtigt werden, insbesondere:<ul><li>nur Patienten mit einem Hämatokritwert &gt;= 33 % (Hämoglobinwert &gt;= 11 g/dl [6,83 mmol/l]) dürfen Blut spenden</li><li>Patienten mit einem Körpergewicht &lt; 50 kg sollten besonders sorgfältig beobachtet werden</li><li>das Volumen einer einzelnen Blutspende sollte nicht größer als ca. 12% des Blutvolumens des Patienten sein</li></ul></li><li>Behandlung sollte auf Patienten beschränkt werden, bei denen die Vermeidung von Fremdbluttransfusionen nach deren Nutzen-Risiko-Abschätzung als besonders wichtig angesehen wird</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Verwendung bei Gesunden kann zu einer übermäßigen Steigerung des Hämatokritwertes führen</li><li>in diesem Zusammenhang wurde über lebensbedrohliche Komplikationen des Herz-Kreislauf-Systems berichtet</li></ul></li></ul><p>Epoetin theta</p><ul><li>allgemein<ul><li>Eisensupplementierung wird bei allen Patienten mit Ferritin-Serumwerten &lt; 100 µg/l oder einer Transferrin-Sättigung &lt; 20% empfohlen<ul><li>um eine effektive Erythropoiese zu gewährleisten, muss bei allen Patienten vor und während der Behandlung der Eisenstatus bestimmt werden</li></ul></li><li>Nicht-Ansprechen auf die Behandlung mit Epoetin theta sollte Anlass für eine Suche nach ursächlichen Faktoren sein<ul><li>Eisen-, Folsäure- und Vitamin-B12-Mangelzustände verringern die Wirksamkeit von Epoetinen und müssen daher korrigiert werden</li><li>die folgenden Faktoren können eine Ansprechen der Erythropoiese beeinträchtigen <ul><li>interkurrente Infektionen</li><li>entzündliche oder traumatische Vorfälle</li><li>okkulter Blutverlust</li><li>Hämolyse</li><li>Aluminium-Intoxikation</li><li>hämatologische Grunderkrankung</li><li>Knochenmarkfibrose</li></ul></li><li>Beurteilung sollte auch die Retikulozytenzahlen umfassen</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>wenn typische Ursachen für ein Nicht-Ansprechen ausgeschlossen wurden und bei dem Patienten ein plötzlicher Abfall des Hämoglobins zusammen mit einer Retikulopenie auftritt, sind Untersuchungen auf Anti-Erythropoietin-Antikörper und eine Knochenmarkuntersuchung zur Diagnosestellung einer Erythroblastopenie in Betracht zu ziehen</li><li>Unterbrechung der Behandlung mit Epoetin theta ist in Betracht zu ziehen</li><li>unter Erythropoietin-Therapie wurden Fälle von PRCA beschrieben, die durch neutralisierende Anti-Erythropoietin-Antikörper verursacht wurden</li><li>da für diese Antikörper Kreuzreaktivität mit allen Epoetinen gezeigt wurde, sollten Patienten mit Verdacht oder Nachweis auf neutralisierende Antikörper gegen Erythropoietin nicht auf Epoetin theta umgestellt werden</li><li>bei einer paradoxen Abnahme des Hämoglobinwerts und Ausbildung einer schweren Anämie in Zusammenhang mit einer niedrigen Retikulozytenzahl ist die Behandlung mit Epoetin abzubrechen und einen Test auf Anti-Erythropoetin-Antikörper durchzuführen<ul><li>entsprechende Fälle sind bei Hepatitis C-Patienten berichtet worden, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetin erhielten</li><li>Epoetine sind nicht für die Behandlung einer Hepatitis C-assoziierten Anämie bestimmt</li></ul></li></ul></li><li>Hypertonie<ul><li>während einer Epoetin-theta-Therapie kann es zu einer Erhöhung des Blutdrucks bzw. einer Verstärkung einer bereits bestehenden Hypertonie kommen, insbes. zu Beginn der Behandlung</li><li>auf eine engmaschige Überwachung und Blutdruckkontrollen achten<ul><li>Blutdruck muss vor Beginn der Behandlung und während der Behandlung adäquat eingestellt werden, um akute Komplikationen wie eine hypertensive Krise mit Enzephalopathie-ähnlichen Symptomen (z.B. Kopfschmerzen, Verwirrtheit, Sprachstörungen, Gangstörungen) und assoziierten Komplikationen (Krampfanfälle, Schlaganfall) zu vermeiden, die auch vereinzelt bei Patienten mit ansonsten normalem oder niedrigem Blutdruck auftreten kann</li><li>beim Auftreten dieser Reaktionen ist eine umgehende ärztliche Behandlung und intensivmedizinische Überwachung erforderlich</li></ul></li><li>auf plötzlich auftretende, heftige, migräneartige Kopfschmerzen als mögliches Warnsignal sollte besonders geachtet werden</li><li>Blutdruckerhöhungen <ul><li>können eine Behandlung mit blutdrucksenkenden Arzneimitteln oder eine Dosiserhöhung einer bereits bestehenden antihypertensiven Behandlung notwendig machen</li><li>zusätzlich Reduktion der angewendeten Epoetin theta-Dosis erwägen</li><li>falls die Blutdruckwerte weiterhin erhöht bleiben, kann eine vorübergehende Unterbrechung der Therapie mit Epoetin theta erforderlich sein</li></ul></li><li>nach erfolgreicher Blutdruckeinstellung durch eine Intensivierung der Behandlung sollte die Epoetin-theta-Therapie in reduzierter Dosierung wieder aufgenommen werden</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Anwendung von Epoetin theta bei gesunden Personen kann einen übermäßigen Anstieg des Hämoglobins und des Hämatokrits zur Folge haben, was zu lebensbedrohlichen kardiovaskulären Komplikationen führen kann</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden</li><li>wenn Anzeichen und Symptome, die auf diese Reaktionen <a>hinweisen<\a>, auftreten, sollte die Behandlung mit Epoetin theta unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li><li>wenn ein Patient aufgrund der Anwendung von Epoetin theta eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin theta zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>besondere Patientengruppen<ul><li>eingeschränkte Leberfunktion und Sichelzellenanämie<ul><li>wegen der begrenzten Erfahrung keine Aussagen zur Wirksamkeit und Sicherheit von Epoetin theta bei Patienten mit eingeschränkter Leberfunktion und homozygoter Sichelzellanämie möglich</li></ul></li><li>ältere Patienten &gt; 75 Jahre<ul><li>in klinischen Studien hatten Patienten über 75 Jahren eine höhere Inzidenz schwerwiegender und schwerer unerwünschter Ereignisse, unabhängig von einem Kausalzusammenhang mit der Epoetin-theta-Therapie</li><li>Todesfälle waren in dieser Patientengruppe häufiger als bei jüngeren Patienten</li></ul></li></ul></li><li>Laborkontrollen<ul><li>es wird empfohlen, regelmäßig das Hämoglobin zu messen, ein komplettes Blutbild zu erstellen und Thrombozytenzählungen durchzuführen</li></ul></li><li>Symptomatische Anämie infolge chronischer Niereninsuffizienz<ul><li>bei nephrosklerotischen Patienten, die noch keine Dialyse erhalten, ist die Anwendung von Epoetin theta individuell abzuwägen, da die Möglichkeit einer beschleunigten Progression der Niereninsuffizienz nicht sicher ausgeschlossen werden kann</li><li>während der Hämodialyse können Patienten unter Behandlung mit Epoetin theta eine intensivere gerinnungshemmende Therapie benötigen, um einem Verschluss des arterio-venösen Shunts vorzubeugen</li><li>bei Patienten mit chronischer Niereninsuffizienz sollte die Hämoglobin-Konzentration in der Erhaltungsphase die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschreiten<ul><li>in klinischen Studien wurde ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse unter einer Epoetin-Therapie beobachtet, wenn Hämoglobin-Zielspiegel &gt; 12 g/dl (7,45mmol/l) angestrebt wurde</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobin-Konzentration über den für die Symptomkontrolle einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li></ul></li><li>Vorsicht bei der Steigerung der Epoetin-theta-Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li></ul></li><li>Symptomatische Anämie bei Krebspatienten mit nicht-myeloischen malignen Erkrankungen, die eine Chemotherapie erhalten<ul><li>Einfluß auf Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, die primär die Bildung von Erythrozyten anregen</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen exprimiert werden</li><li>wie bei allen Wachstumsfaktoren gibt es Bedenken, dass Epoetine das Wachstum von Tumoren anregen könnten</li></ul></li><li>in mehreren kontrollierten Studien zeigten Epoetine keine Verbesserung des Gesamtüberlebens oder eine Verminderung des Risikos einer Tumorprogression bei Patienten mit Anämie infolge einer Krebserkrankung.</li><li>in kontrollierten klinischen Studien zeigte die Anwendung von Epoetinen:<ul><li>eine verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Halstumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobin- Zielspiegel oberhalb von 14 g/dl (8,69mmol/l) angestrebt wurde</li><li>ein verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobin-Zielwert zwischen 12 - 14 g/dl (7,45 - 8,69mmol/l) angestrebt wurde</li><li>ein erhöhtes Sterberisiko bei Patienten mit aktiven malignen Erkrankungen, dieweder eine Chemotherapie noch eine Strahlentherapie erhielten, wenn ein Hämoglobin-Zielwert von 12g/dl (7,45mmol/l) angestrebt wurde</li></ul></li><li>Epoetine sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Krebspatienten bevorzugt mit Bluttransfusionen behandelt werden</li><li>Entscheidung zur Anwendung rekombinanter Erythropoietine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des einzelnen Patienten beruhen, unter Berücksichtigung der jeweils spezifischen klinischen Umstände<ul><li>Faktoren wie Tumorart und -stadium, Schweregrad der Anämie, Lebenserwartung, Behandlungsumfeld des Patienten und Wunsch des Patienten sollten in diese Beurteilung mit einbezogen werden</li></ul></li></ul></li></ul><p>Epoetin zeta</p><ul><li>allgemein<ul><li>Blutdruck <ul><li>bei allen Patienten, die Epoetin zeta erhalten, sollte der Blutdruck überwacht und ggf. behandelt werden</li><li>Epoetin zeta sollte mit Vorsicht angewendet werden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte blutdrucksenkende Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar<ul><li>sollte die Behandlung mit Epoetin zeta beendet werden</li></ul></li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurften<ul><li>traten unter der Behandlung mit Epoetin zeta auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie, Krampfneigung<ul><li>Epoetin zeta sollte bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS- Infektionen und Hirnmetastasen, vorsichtig angewendet werden</li></ul></li><li>Patienten mit chronischer Leberinsuffizienz <ul><li>Anwendung mit Vorsicht</li><li>Sicherheit von Epoetin zeta bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignisse (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden <ul><li>dazu gehören venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>darüber hinaus wurde über zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken) berichtet</li></ul></li><li>das berichtete Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin zeta abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li></ul></li><li>Hämoglobinspiegel<ul><li>bei allen Patienten ist der Hämoglobinspiegel engmaschig zu ü berwachen, da potentiell ein erhöhtes Risiko fü r thromboembolische Ereignisse und tödliche Verläufe besteht, wenn Patienten behandelt wurden, deren Hämoglobinkonzentration über dem Konzentrationsbereich liegt, für den das Arzneimittel indiziert ist</li></ul></li><li>während der Behandlung mit Erythropoetin kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>bei fortgesetzter Behandlung rückläufig</li><li>empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Ursachen einer Anämie müssen vor Beginn einer Behandlung mit Epoetin zeta sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden<ul><li>Eisen-, Folsäure- oder Vitamin B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikationen</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarksfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen gehen gleichzeitig mit der Zunahme des gepackten Zellvolumens die Ferritinwerte im Serum zurück</li><li>für das Erzielen eines optimalen Ansprechens auf Erythropoetin müssen ausreichende Eisenreserven sichergestellt werden:<ul><li>für Patienten mit chronischer Niereninsuffizienz, deren Serum-Ferritinkonzentration unter 100 ng/ml liegt, wird eine orale Eisensubstitution empfohlen, z.B. 200 - 300 mg / Tag (100-200 mg / Tag bei pädiatrischen Patienten)</li><li>für alle Tumorpatienten, bei denen die Transferrinsättigung weniger als 20% beträgt, wird eine orale Eisensubstitution mit 200 - 300 mg / Tag empfohlen</li><li>für Patienten in einem autologen Blutspendeprogramm sollte mit der oralen Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) bereits einige Wochen vor der Eigenblutspende begonnen werden, um vor der Einleitung der Behandlung mit Epoetin zeta sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten</li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin zeta eine orale Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) erhalten<ul><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin zeta begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>sehr selten wurde die Entwicklung oder Exazerbation einer Porphyrie bei mit Epoetin zeta behandelten Patienten beobachtet</li><li>Epoetin zeta sollte bei Patienten mit Porphyrie mit Vorsicht angewendet werden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens- Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen <a>hinweisen<\a>, auftreten, sollte die Behandlung unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung von Epoetin eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Wirkstoffen (ESAs) zu verbessern, soll der Handelsname und die Chargennummer des angewendeten ESAs eindeutig in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung<ul><li>Patienten sollen nur unter geeigneter Überwachung von einer ESA auf ein anderes umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>übber das Auftreten einer antikörpervermittelten Erythroblastopenie (PRCA) wurde nach monate- bis jahrelanger Behandlung mit Epoetin berichtet</li><li>ebenfalls traten Fälle bei Patienten mit Hepatitis C auf, die mit Interferon und Ribavirin behandelt wurden, während ESAs begleitend eingesetzt wurden</li><li>Epoetin zeta ist für die Behandlung einer Anämie, die mit Hepatitis C assoziiert ist, nicht zugelassen</li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl pro Monat) mit erhöhtem Bedarf an Transfusionen<ul><li>sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen (z.B. Eisen-, Folsäure- oder Vitamin-B12-Mangel, Aluminiumintoxikation, Infektionen oder Entzündungen, Blutverlust, Hämolyse und Knochenmarkfibrose jeglicher Genese) abgeklärt werden</li></ul></li><li>bei einer paradoxen Verminderung der Hämoglobinwerte und der Entwicklung einer schweren Anämie, die mit einem erniedrigten Retikulozytenwert einhergeht<ul><li>sollte die Behandlung mit Epoetin zeta sofort abgesetzt und die Anti-Erythropoetin- Antikörper bestimmt werden</li><li>eine Untersuchung des Knochenmarks zur Diagnose einer PRCA sollte ebenfalls erwogen werden</li></ul></li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere ESA-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anä mie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>Hämoglobin-Bestimmungen<ul><li>bei Patienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und 2 g/dl (1,25 mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko der Zunahme eines Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von über 12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht, wenn die Epoetin-Dosis bei Patienten mit chronischer Niereninsuffizienz gesteigert wird<ul><li>da hohe kumulierte Epoetin-Dosen mit einem gesteigerten Mortalitätsrisiko, schwerwiegenden kardiovaskulären und zerebrovaskulären Ereignissen assoziiert sein können</li><li>bei Patienten, die ein schlechtes Ansprechen des Hämoglobinwertes auf Epoetin zeigen, sollten alternative Gründe für das schlechte Anprechen bedacht werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, denen Epoetin zeta subkutan gegeben wird, sollten regelmäßig Kontrollen hinsichtlich eines Wirkverlustes, definiert als ein Nicht- oder vermindertes Ansprechen auf eine Behandlung mit Epoetin zeta bei Patienten, die zuvor auf eine solche Therapie angesprochen haben, durchgeführt werden<ul><li>dies zeigt sich durch eine anhaltende Verminderung der Hämoglobinwerte trotz Steigerung der Epoetin-zeta- Dosis</li></ul></li><li>bei Verabreichung von Epoetin in längeren Dosisintervallen (länger als 1-mal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shuntthrombosen <ul><li>Shuntthrombosen sind insbesondere bei Hämodialysepatienten aufgetreten, die eine Tendenz zur Hypotonie aufweisen oder deren arteriovenöse Fisteln Komplikationen aufweisen (z.B. Stenosen, Aneurysmen, etc.)</li><li>eine frühzeitige Shuntkorrektur und Thromboseprophylaxe durch die Gabe von z.B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>Hyperkaliämie<ul><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden</li><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder ansteigendem Kaliumwert sollte zusätzlich zu einer geeigneten Behandlung der Hyperkaliämie erwogen werden, die Epoetin-zeta-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>im Verlauf der Therapie mit Erythropoetin ist aufgrund des erhöhten Hämatokrits häufig eine Steigerung der Heparindosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialysesystems möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie mit Epoetin zeta bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie, Auswirkungen auf das Tumorwachstum <ul><li>Hämoglobin-Bestimmungen<ul><li>bei Tumorpatienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>Epoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>we bei allen Wachstumsfaktoren sollte bedacht werden, dass Epoetine das Wachstum von Tumoren jeglicher Art stimulieren können</li><li>eine Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war die Anwendung von Epoetin zeta und anderen ESAs mit einer verkürzten Zeit bis zur Tumorprogression oder mit einer verkürzten Gesamtüberlebenszeit<br />verbunden <ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn eine Hämoglobin-Zielkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>kürzeres Gesamtüberleben und erhöhte Sterblichkeit, die auf eine Krankheitsprogression nach 4 Monaten bei Patienten mit metastasierendem Brustkrebs, die eine Chemotherapie erhielten, zurückgeführt wurde, wenn eine Hämoglobin-Zielkonzentration von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l) angestrebt wurde<ul><li>ESAs für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-zeta plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15 % erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 g/dl und 12 g/dl (6,2 - 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden<ul><li>einer Entscheidung zur Gabe von rekombinantem Erythropoetin sollte eine Nutzen-Risiko Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Erkrankungsprognose</li><li>Umfeld, in dem der Patient behandelt wird</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen, sollte für die Beurteilung der Therapieeffizienz von Epoetin zeta eine 2- bis 3-wöchige Verzögerung zwischen ESA-Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle speziellen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbesondere routinemäßiger Volumenersatz, sollten berü cksichtigt werden</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff vorgesehen ist<ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrunde liegenden kardiovaskulären Erkrankung<ul><li>daher sollte insbesondere bei diesen Patienten eine angemessene Thromboseprophylaxe erfolgen</li></ul></li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit Disposition für tiefe Venenthrombosen ergriffen werden</li><li>Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l)<ul><li>Epoetin zeta sollte nicht bei diesen Patienten eingesetzt werden</li><li>bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l) kann nicht ausgeschlossen werden, dass eine Epoetin zeta-Behandlung mit einem erhöhten Risiko<br />postoperativer Thrombose-/Gefäßereignissen einhergeht</li></ul></li></ul></li></ul>`,
	},
}

type testCaseReplace struct {
	patterns    []string
	haystack    string
	replaceWith []string
	replaced    string
}

func TestAhoCorasick_Iter(t *testing.T) {
	for i, t2 := range leftmostInsensitiveWholeWordTestCases {
		builder := NewAhoCorasickBuilder(Opts{
			AsciiCaseInsensitive: true,
			MatchOnlyWholeWords:  true,
			MatchKind:            LeftMostLongestMatch,
		})

		ac := builder.Build(t2.patterns)
		iter := ac.Iter(t2.haystack)
		matches := make([]Match, 0)

		for next := iter.Next(); next != nil; next = iter.Next() {
			matches = append(matches, *next)
		}

		if len(matches) != len(t2.matches) {
			t.Errorf("test %v expected %v matches got %v", i, len(matches), len(t2.matches))
		}
		for i, m := range matches {
			if m != t2.matches[i] {
				t.Errorf("test %v expected %v matche got %v", i, m, t2.matches[i])
			}
		}
	}
}

func TestAhoCorasick_Parallel(t *testing.T) {
	t2 := leftmostInsensitiveWholeWordTestCases[0]
	builder := NewAhoCorasickBuilder(Opts{
		AsciiCaseInsensitive: true,
		MatchOnlyWholeWords:  true,
		MatchKind:            LeftMostLongestMatch,
	})

	ac := builder.Build(t2.patterns)
	var w sync.WaitGroup

	w.Add(50)
	for i := 0; i < 50; i++ {
		go func() {
			ac := ac
			matches := ac.FindAll(t2.haystack)
			if len(matches) != len(t2.matches) {
				t.Errorf("test %v expected %v matches got %v", 0, len(matches), len(t2.matches))
			}
			for i, m := range matches {
				if m != t2.matches[i] {
					t.Errorf("test %v expected %v matche got %v", i, m, t2.matches[i])
				}
			}
			w.Done()
		}()
	}
	w.Wait()
}

func TestAhoCorasick_IterOverlapping(t *testing.T) {
	builder := NewAhoCorasickBuilder(Opts{
		AsciiCaseInsensitive: true,
		MatchOnlyWholeWords:  true,
		MatchKind:            StandardMatch,
	})

	ac := builder.Build([]string{"hinweise und Vorsichtsmaßnahmen", "Vorsichtsmaßnahmen"})
	haystack := "alle spezifischen Warnhinweise und Vorsichtsmaßnahmen, die mit einem"
	iter := ac.IterOverlapping(haystack)

	matches := make([]Match, 0)
	for next := iter.Next(); next != nil; next = iter.Next() {
		matches = append(matches, *next)
	}
	if len(matches) != 1 {
		t.Errorf("expected 1 match got %v", len(matches))
	}
	if haystack[matches[0].Start():matches[0].End()] != "Vorsichtsmaßnahmen" {
		t.Errorf("expected `Vorsichtsmaßnahmen` match got %v", haystack[matches[0].Start():matches[0].End()])
	}
}

func TestAhoCorasick_LeftmostInsensitiveWholeWord(t *testing.T) {
	for i, t2 := range leftmostInsensitiveWholeWordTestCases {
		builders := []AhoCorasickBuilder{NewAhoCorasickBuilder(Opts{
			AsciiCaseInsensitive: true,
			MatchOnlyWholeWords:  true,
			MatchKind:            LeftMostLongestMatch,
		}), NewAhoCorasickBuilder(Opts{
			AsciiCaseInsensitive: true,
			MatchOnlyWholeWords:  true,
			MatchKind:            LeftMostLongestMatch,
			DFA:                  true,
		})}

		for _, builder := range builders {
			ac := builder.Build(t2.patterns)
			matches := ac.FindAll(t2.haystack)

			if len(matches) != len(t2.matches) {
				t.Errorf("test %v expected %v matches got %v", i, len(matches), len(t2.matches))
			}
			for i, m := range matches {
				if m != t2.matches[i] {
					t.Errorf("test %v expected %v matche got %v", i, m, t2.matches[i])
				}
			}
		}
	}
}

var leftmostInsensitiveWholeWordTestCases = []testCase{
	{
		patterns: []string{"bear", "masha"},
		haystack: "The Bear and Masha",
		matches: []Match{
			{
				pattern: 0,
				len:     4,
				end:     8,
			},
			{
				pattern: 1,
				len:     5,
				end:     18,
			},
		},
	},
	{
		patterns: []string{"hinweise und vorsichtsmaßnahmen", "hinweisen"},
		haystack: "<h2>Hinweise und Vorsichtsmaßnahmen, Wirkstoffprofil <br />(kumulative Wirkstoffangaben ohne Berücksichtigung des individuellen Zulassungsstatus)</h2> Erythropoietin - invasiv<br /><br /><p>Epoetin alfa</p><ul><li>allgemein<ul><li>Hypertonie<ul><li>bei allen Patienten sollte der Blutdruck engmaschig überwacht und ggf. behandelt werden</li><li>mit Vorsicht anwenden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte antihypertensive Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar, sollte die Behandlung beendet werden</li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurfte<ul><li>traten unter der Behandlung mit Epoetin alfa auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie<ul><li>bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS-Infektionen und Hirnmetastasen, vorsichtig anwenden</li></ul></li><li>chronische Leberinsuffizienz<ul><li>bei Patienten mit chronischer Leberinsuffizienz mit Vorsicht anwenden</li><li>Sicherheit von Epoetin alfa ist bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignissen (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden<ul><li>venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken)</li></ul></li><li>berichtetes Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin alfa abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li><li>bei allen Patienten sollten die Hämoglobinwerte engmaschig überwacht werden, da ein potentiell erhöhtes Risiko für thromboembolische Ereignisse und Mortalität besteht, wenn der Hämoglobin-Wert der Patienten oberhalb des für die jeweilige Indikation gültigen Zielbereichs liegt</li><li>während der Behandlung kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>dieser ist bei fortgesetzter Behandlung rückläufig</li></ul></li><li>es wird empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Anämieursachen sollten abgeklärt und müssen vor Beginn einer Behandlung mit Epoetin alfa sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden, z.B.<ul><li>Eisenmangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Folsäuremangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Hämolyse</li><li>Blutverlust</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen wird der Serumferritinwert simultan zum Anstieg des Hämatokritwertes abfallen; um ein optimales Ansprechen zu erzielen, müssen ausreichende Eisenspeicher gewährleistet sein, falls nötig substituieren<ul><li>orale Eisensubstitution für Patienten mit chronischer Niereninsuffizienz empfohlen, deren Serumferritinspiegel unter 100 ng/ml liegt, z.B.<ul><li>200 - 300 mg Fe2+ / Tag für Erwachsene</li><li>100 - 200 mg Fe2+ / Tag für Kinder und Jugendliche</li></ul></li><li>orale Eisensubstitution für alle Tumorpatienten, deren Transferrinsättigung unter 20% liegt<ul><li>200 - 300 mg Fe2+ / Tag</li></ul></li><li>orale Eisensubstitution für Patienten in einem autologen Blutspendeprogramm bereits einige Wochen vor der Eigenblutspende beginnen, um vor der Einleitung der Behandlung mit Epoetin alfa sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten<ul><li>200 mg Fe2+ / Tag</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin alfa eine orale Eisen-II-Substitution erhalten<ul><li>200 mg Fe2+ / Tag</li><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin alfa begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>in sehr seltenen Fällen eine Porphyrie oder Verschlimmerung einer Porphyrie beobachtet</li><li>bei diesen Patienten mit Vorsicht anwenden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen, einschließlich Stevens Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet</li><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li><li>zum Zeitpunkt der Verschreibung sollten Patienten auf Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, Behandlung unverzüglich beenden und alternative Behandlung in Betracht ziehen</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>Behandlung darf zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Substanzen (ESAs) zu verbessern, soll der Handelsname des verordneten ESAs deutlich in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung <ul><li>Patienten sollen nur unter geeigneter Überwachung von einem ESA auf einen anderen umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>über das Auftreten einer antikörpervermittelten PRCA nach monate- bis jahrelanger Behandlung mit subkutanem Erythropoetin berichtet<ul><li>v.a. bei Patienten mit chronischer Niereninsuffizienz</li><li>Fallberichte auch für Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten<ul><li>für diese Indikation besteht keine Zulassung</li></ul></li></ul></li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl bzw. 0,62 - 1,25 mmol/l pro Monat) mit erhöhtem Bedarf an Transfusionen, sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen untersucht werden, z.B.<ul><li>Eisenmangel</li><li>Folsäuremangel</li><li>Vitamin-B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikation</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarkfibrose jeglicher Genese</li></ul></li><li>bei paradoxer Abnahme des Hämoglobinwerts und Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozytenzahlen Absetzen der Behandlung und Durchführung einer Untersuchung auf Anti-Erythropoetin-Antikörper; auch Knochenmarkuntersuchung zur Diagnose einer PRCA erwägen</li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere Erythropoetin-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anämie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>regelmäßig Hämoglobin-Bestimmungen durchführen, bis ein stabiler Hämoglobin-Wert erreicht wird<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und sollte 2 g/dl (1,25mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko einer Zunahme des Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von &gt;12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative<br />Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, die mit subkutaner Gabe von Epoetin alfa behandelt werden, müssen regelmäßig auf Wirkverlust hin überwacht werden, der als keine oder verringerte Reaktion auf die Epoetin alfa-Behandlung bei Patienten definiert wird, die zuvor auf solch eine Therapie angesprochen haben<ul><li>zeigt sich durch andauernden Hämoglobinabfall trotz einer Erhöhung der Dosis von Epoetin alfa</li></ul></li><li>bei Verabreichung von Epoetin alfa in längeren Dosierungsintervallen (länger als einmal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-alfa-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shunt-Thrombosen sind bei Hämodialysepatienten, insbesondere bei Patienten mit Neigung zu Hypotonie oder bei Komplikationen an den arteriovenösen Fisteln aufgetreten (z.B. Stenosen, Aneurysmen etc.)<ul><li>frühzeitige Shuntrevision und Thromboseprophylaxe durch Gabe von z. B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden<ul><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder steigendem Kaliumwert sollte erwogen werden, die Epoetin alfa-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>aufgrund des Anstiegs des Hämatokrits im Rahmen der Epoetin-alfa-Behandlung ist häufig eine Erhöhung der Heparin-Dosis während der Dialyse notwendig<ul><li>ein Verschluss des Dialysesystems ist bei nicht optimaler Heparinisierung möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie<ul><li>regelmäßig Hämoglobin-Bestimmungen durchühren, bis ein stabiler Hämoglobin-Wert erreicht ist<ul><li>danach in regelmäßigen Abständen überprüfen</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Erythropoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>es sollte bedacht werden, dass Epoetine das Wachstum von Tumoren stimulieren können</li><li>Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war Anwendung von Epoetin alfa und anderen ESAs mit verkürzter Zeit bis zur Tumorprogression oder mit verkürzter Gesamtüberlebenszeit verbunden<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürzte Gesamtüberlebenszeit und erhöhte Zahl an Todesfällen, zurückzuführen<br />auf eine Tumorprogression nach 4 Monaten, bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen Hämoglobinwerte von 12 bis 14 g/dl (7,5 bis 8,7 mmol/l) angestrebt wurden</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l), angestrebt wurde<ul><li>ESAs sind für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-alfa plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15% erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 und 12 g/dl (6,2 bis 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Tumorpatienten bevorzugt durch Bluttransfusionen behandelt werden</li></ul></li><li>Entscheidung für die Anwendung rekombinanter Erythropoetine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des jeweiligen Patienten beruhen, die auch den spezifischen klinischen Kontext berücksichtigen sollte<ul><li>in diese Beurteilung einzubeziehende Faktoren sind<ul><li>Tumortyp und -stadium</li><li>Anämiegrad</li><li>Lebenserwartung</li><li>Umgebung, in der der Patient behandelt wird</li><li>Präferenzen des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen sollte für die Beurteilung der Therapieeffizienz von eine 2 - 3-wöchige Verzögerung zwischen Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle spezifischen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbes. routinemäßiger Volumenersatz, sind zu berücksichtigen</li></ul></li><li>Patienten, bei denen ein großer elektiver orthopädischer Eingriff vorgesehen ist <ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrundeliegenden kardiovaskulären Erkrankung<ul><li>sollten eine angemessene Thromboseprophylaxe erhalten</li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit einer Prädisposition für tiefe Venenthrombosen (TVT) ergriffen werden</li></ul></li><li>darüber hinaus kann nicht ausgeschlossen werden, dass bei Behandlung mit Epoetin alfa für Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl ein erhöhtes Risiko für postoperative thrombotische/vaskuläre Ereignisse bestehen kann</li><li>aus diesem Grund sollte es bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl nicht angewendet werden</li></ul></li><li>Anwendung kann bei Dopingkontrollen zu positiven Ergebnissen führen<ul><li>Missbrauch durch Gesunde (z.B. für Dopingzwecke) kann zu übermäßigem Anstieg des Hämoglobinwertes führen; mit Risiko lebensbedrohlicher Komplikationen des Herz-Kreislauf-Systems (Thromboserisiko durch Hämokonzentration bei Polyglobulie) verbunden</li></ul></li></ul><p>Epoetin beta</p><ul><li>allgemein<ul><li>mit Vorsicht anwenden bei <ul><li>refraktärer Anämie mit Blastenexzess in Transformation</li><li>epileptiformen Erkrankungen</li><li>Thrombozytose</li><li>chronischer Leberinsuffizienz</li></ul></li><li>Folsäure- und Vitamin-B12-Mangel sollten ausgeschlossen sein, da hierdurch die Wirksamkeit vermindert wird</li><li>Vorsicht bei der Steigerung der Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li><li>um eine wirksame Erythropoiese sicherzustellen, sollte bei allen Patienten vor und während der Behandlung der Eisenwert bestimmt werden und ggf. eine Eisenersatztherapie gemäß den therapeutischen Richtlinien durchgeführt werden</li><li>Wirksamkeit von Epoetin beta kann durch eine Aluminiumüberladung infolge der Therapie der chronischen Niereninsuffizienz abgeschwächt werden</li><li>Prädialyse-Patienten mit Nephrosklerose <ul><li>über die Anwendung von Epoetin beta sollte bei Prädialyse-Patienten mit Nephrosklerose individuell entschieden werden, da eine beschleunigte Progression der Niereninsuffizienz bei diesen Patienten nicht mit Sicherheit ausgeschlossen werden kann</li></ul></li></ul></li><li>Aregenerative Anämie (pure red cell aplasia, PRCA)<ul><li>in Verbindung mit einer Erythropoietin-Therapie wurde über eine durch neutralisierende Antikörper gegen Erythropoietin hervorgerufene aregenerative Anämie (pure red cell aplasia) berichtet</li><li>für diese Antikörper konnte eine Kreuzreaktivität mit allen erythropoietischen Proteinen gezeigt werden</li><li>Patienten, bei denen der Verdacht auf neutralisierende Antikörper gegen Erythropoietin besteht oder bei denen diese nachgewiesen wurden, sollen deshalb nicht auf Epoetin beta umgestellt werden</li></ul></li><li>PRCA bei Patienten mit Hepatitis C<ul><li>eine paradoxe Hämoglobin-Abnahme und die Entwicklung einer schweren Anämie in Verbindung mit niedrigen Retikulozyten-Zahlen sollten zum Abbruch der Behandlung mit Epoetin und zur Durchführung von Tests auf Antikörper gegen Erythropoietin veranlassen</li><li>über Fälle berichtet bei Patienten mit Hepatitis C, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetine erhielten</li><li>Epoetine sind nicht zugelassen für die Behandlung einer Anämie in Verbindung mit Hepatitis C</li></ul></li><li>Hypertonie<ul><li>bei Patienten mit chronischer Nierenerkrankung kann eine Erhöhung des Blutdrucks erfolgen bzw. die Verstärkung einer bereits bestehenden Hypertension auftreten, insbes. in Fällen eines schnellen Hämatokrit-Anstiegs<ul><li>diese Blutdruckerhöhungen lassen sich medikamentös behandeln</li></ul></li><li>wenn der Blutdruckanstieg nicht durch medikamentöse Therapie kontrolliert werden kann, wird eine vorübergehende Unterbrechung der Therapie mit Epoetin beta empfohlen</li><li>insbes. zu Beginn der Behandlung wird darüber hinaus eine regelmäßige Blutdruckkontrolle empfohlen, auch zwischen den Dialysen</li><li>Blutdruckentgleisungen mit Enzephalopathie-ähnlicher Symptomatik können auftreten, die eine sofortige ärztliche Untersuchung und intensivmedizinische Maßnahmen erfordern<ul><li>plötzlich auftretende stechende migräneartige Kopfschmerzen sind als mögliches Warnzeichen besonders zu beachten</li></ul></li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung mit Epoetin unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>chronische Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung kann es insbesondere bei intravenöser Anwendung zu einem mäßiggradigen dosisabhängigen Anstieg der Thrombozytenzahl innerhalb des Normbereiches kommen, der im Verlauf der Therapie wieder zurückgeht</li><li>empfohlen, die Thrombozytenzahl in den ersten 8 Wochen der Therapie regelmäßig zu kontrollieren</li></ul></li><li>Hämoglobinkonzentration<ul><li>bei Patienten mit chronischer Nierenerkrankung sollte in der Erhaltungsphase die Hämoglobinkonzentration die empfohlene Obergrenze der angestrebten Hämoglobinkonzentration nicht überschreiten</li><li>in klinischen Studien wurde ein erhöhtes Sterberisiko und ein erhöhtes Risiko für schwerwiegende kardiovaskuläre Ereignisse einschl. Hirnschlag beobachtet, wenn Erythropoiese-stimulierende Substanzen (ESA) verabreicht wurden, um ein Hämoglobin &gt; 12 g/dl (7,5 mmol/l) zu erzielen</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobinkonzentration über den für die Kontrolle der Symptome einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li><li>bei Frühgeborenen kann die Thrombozytenzahl insbes. in den ersten zwei Lebenswochen leicht ansteigen, deshalb sollte die Thrombozytenzahl regelmäßig bestimmt werden</li></ul></li><li>Auswirkung auf das Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, welche in erster Linie die Produktion von roten Blutkörperchen stimulieren</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedenster Tumorzellen gebildet werden<ul><li>wie bei allen Wachstumsfaktoren besteht der Verdacht, dass auch Epoetine das Wachstum von Tumoren stimulieren könnten</li></ul></li><li>in mehreren kontrollierten Studien mit Patienten, die eine Anämie im Zusammenhang mit einer Krebserkrankung aufwiesen, wurde keine Verbesserung des Gesamtüberlebens bzw. keine Verringerung des Risikos der Tumorprogression durch Epoetine gezeigt</li><li>in kontrollierten klinischen Studien hat die Anwendung von Epoetin beta und anderen Erythropoiese-stimulierenden Substanzen (ESA) Folgendes gezeigt:<ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobinzielwert &gt; 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobinzielbereich von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko, wenn bei Patienten mit aktiven malignen Erkrankungen, die weder eine Chemotherapie noch eine Strahlentherapie erhielten, ein Hämoglobinzielwert von 12 g/dl (7,5 mmol/l) angestrebt wurde</li><li>ESA sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden</li><li>einer Entscheidung zur Gabe von rekombinantem Erythropoietin sollte eine Nutzen-Risiko-Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko-Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Prognose der Erkrankung</li><li>Umfeld, in dem der Patient behandelt</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>Laborwerte<ul><li>es kann zu einem Anstieg des Blutdrucks kommen, der medikamentös behandelt werden kann<ul><li>insbes. zu Beginn der Behandlung sollte deshalb der Blutdruck bei Krebspatienten kontrolliert werden</li></ul></li><li>Thrombozytenzahl und Hämoglobinkonzentration sollten bei Krebspatienten in regelmäßigen Abständen überprüft werden</li><li>bei Patienten in einem Eigenblutspendeprogramm kann es zu einem Anstieg der Thrombozytenzahl, in der Regel innerhalb des Normbereiches, kommen<ul><li>daher empfohlen, die Thrombozytenzahl mind. 1mal / Woche zu kontrollieren</li><li>falls Steigerung &gt; 150 x 10<sup>9</sup>/l oder liegt die Thrombozytenzahl über dem Normbereich, sollte die Therapie mit Epoetin beta abgebrochen werden</li></ul></li><li>Serum-Kalium- und -Phosphat-Spiegel sollten während der Behandlung mit Epoetin beta regelmäßig überprüft werden<ul><li>bei einigen urämischen Patienten wurde unter einer Behandlung mit Epoetin beta über erhöhte Kaliumspiegel berichtet, ohne dass ein kausaler Zusammenhang gezeigt wurde</li><li>sollten erhöhte oder ansteigende Kaliumspiegel beobachtet werden, dann sollte eine Unterbrechung der Behandlung mit Epoetin beta in Betracht gezogen werden, bis die Hyperkalämie korrigiert ist</li></ul></li></ul></li><li>Frühgeborene, Retinopathie<ul><li>bei Frühgeborenen konnte das potentielle Risiko, dass Erythropoietin eine Retinopathie auslöst, nicht ausgeschlossen werden<ul><li>deshalb Vorsicht bei der Anwendung</li></ul></li><li>bei der Entscheidung, ob ein Frühgeborenes behandelt wird, sollten die möglichen Nutzen und Risiken dieser Behandlung und verfügbarer alternativer Therapiemöglichkeiten gegeneinander abgewogen werden</li></ul></li><li>Patienten mit chronischer Niereninsuffizienz<ul><li>bei Patienten mit chronischer Nierenerkrankung ist im Verlauf der Therapie mit Epoetin beta aufgrund des gesteigerten Hämatokritwertes häufig eine Erhöhung der Heparin-Dosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialyse-Systems möglich</li></ul></li><li>frühzeitige Shunt-Revision und eine Thrombose-Prophylaxe, z.B. durch Gabe von Acetylsalicylsäure, sollte bei Patienten mit chronischer Nierenerkrankung und mit dem Risiko einer Shunt-Thrombose in Betracht gezogen werden</li></ul></li><li>Eigenblutspende<ul><li>für den Einsatz von Epoetin beta im Rahmen der Eigenblutspende müssen die offiziellen Richtlinien zur Blutspende berücksichtigt werden, insbesondere:<ul><li>nur Patienten mit einem Hämatokritwert &gt;= 33 % (Hämoglobinwert &gt;= 11 g/dl [6,83 mmol/l]) dürfen Blut spenden</li><li>Patienten mit einem Körpergewicht &lt; 50 kg sollten besonders sorgfältig beobachtet werden</li><li>das Volumen einer einzelnen Blutspende sollte nicht größer als ca. 12% des Blutvolumens des Patienten sein</li></ul></li><li>Behandlung sollte auf Patienten beschränkt werden, bei denen die Vermeidung von Fremdbluttransfusionen nach deren Nutzen-Risiko-Abschätzung als besonders wichtig angesehen wird</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Verwendung bei Gesunden kann zu einer übermäßigen Steigerung des Hämatokritwertes führen</li><li>in diesem Zusammenhang wurde über lebensbedrohliche Komplikationen des Herz-Kreislauf-Systems berichtet</li></ul></li></ul><p>Epoetin theta</p><ul><li>allgemein<ul><li>Eisensupplementierung wird bei allen Patienten mit Ferritin-Serumwerten &lt; 100 µg/l oder einer Transferrin-Sättigung &lt; 20% empfohlen<ul><li>um eine effektive Erythropoiese zu gewährleisten, muss bei allen Patienten vor und während der Behandlung der Eisenstatus bestimmt werden</li></ul></li><li>Nicht-Ansprechen auf die Behandlung mit Epoetin theta sollte Anlass für eine Suche nach ursächlichen Faktoren sein<ul><li>Eisen-, Folsäure- und Vitamin-B12-Mangelzustände verringern die Wirksamkeit von Epoetinen und müssen daher korrigiert werden</li><li>die folgenden Faktoren können eine Ansprechen der Erythropoiese beeinträchtigen <ul><li>interkurrente Infektionen</li><li>entzündliche oder traumatische Vorfälle</li><li>okkulter Blutverlust</li><li>Hämolyse</li><li>Aluminium-Intoxikation</li><li>hämatologische Grunderkrankung</li><li>Knochenmarkfibrose</li></ul></li><li>Beurteilung sollte auch die Retikulozytenzahlen umfassen</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>wenn typische Ursachen für ein Nicht-Ansprechen ausgeschlossen wurden und bei dem Patienten ein plötzlicher Abfall des Hämoglobins zusammen mit einer Retikulopenie auftritt, sind Untersuchungen auf Anti-Erythropoietin-Antikörper und eine Knochenmarkuntersuchung zur Diagnosestellung einer Erythroblastopenie in Betracht zu ziehen</li><li>Unterbrechung der Behandlung mit Epoetin theta ist in Betracht zu ziehen</li><li>unter Erythropoietin-Therapie wurden Fälle von PRCA beschrieben, die durch neutralisierende Anti-Erythropoietin-Antikörper verursacht wurden</li><li>da für diese Antikörper Kreuzreaktivität mit allen Epoetinen gezeigt wurde, sollten Patienten mit Verdacht oder Nachweis auf neutralisierende Antikörper gegen Erythropoietin nicht auf Epoetin theta umgestellt werden</li><li>bei einer paradoxen Abnahme des Hämoglobinwerts und Ausbildung einer schweren Anämie in Zusammenhang mit einer niedrigen Retikulozytenzahl ist die Behandlung mit Epoetin abzubrechen und einen Test auf Anti-Erythropoetin-Antikörper durchzuführen<ul><li>entsprechende Fälle sind bei Hepatitis C-Patienten berichtet worden, die mit Interferon und Ribavirin behandelt wurden und gleichzeitig Epoetin erhielten</li><li>Epoetine sind nicht für die Behandlung einer Hepatitis C-assoziierten Anämie bestimmt</li></ul></li></ul></li><li>Hypertonie<ul><li>während einer Epoetin-theta-Therapie kann es zu einer Erhöhung des Blutdrucks bzw. einer Verstärkung einer bereits bestehenden Hypertonie kommen, insbes. zu Beginn der Behandlung</li><li>auf eine engmaschige Überwachung und Blutdruckkontrollen achten<ul><li>Blutdruck muss vor Beginn der Behandlung und während der Behandlung adäquat eingestellt werden, um akute Komplikationen wie eine hypertensive Krise mit Enzephalopathie-ähnlichen Symptomen (z.B. Kopfschmerzen, Verwirrtheit, Sprachstörungen, Gangstörungen) und assoziierten Komplikationen (Krampfanfälle, Schlaganfall) zu vermeiden, die auch vereinzelt bei Patienten mit ansonsten normalem oder niedrigem Blutdruck auftreten kann</li><li>beim Auftreten dieser Reaktionen ist eine umgehende ärztliche Behandlung und intensivmedizinische Überwachung erforderlich</li></ul></li><li>auf plötzlich auftretende, heftige, migräneartige Kopfschmerzen als mögliches Warnsignal sollte besonders geachtet werden</li><li>Blutdruckerhöhungen <ul><li>können eine Behandlung mit blutdrucksenkenden Arzneimitteln oder eine Dosiserhöhung einer bereits bestehenden antihypertensiven Behandlung notwendig machen</li><li>zusätzlich Reduktion der angewendeten Epoetin theta-Dosis erwägen</li><li>falls die Blutdruckwerte weiterhin erhöht bleiben, kann eine vorübergehende Unterbrechung der Therapie mit Epoetin theta erforderlich sein</li></ul></li><li>nach erfolgreicher Blutdruckeinstellung durch eine Intensivierung der Behandlung sollte die Epoetin-theta-Therapie in reduzierter Dosierung wieder aufgenommen werden</li></ul></li><li>Missbrauch, Doping<ul><li>missbräuchliche Anwendung von Epoetin theta bei gesunden Personen kann einen übermäßigen Anstieg des Hämoglobins und des Hämatokrits zur Folge haben, was zu lebensbedrohlichen kardiovaskulären Komplikationen führen kann</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens-Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle im Zusammenhang mit langwirkenden Epoetinen beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden</li><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung mit Epoetin theta unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li><li>wenn ein Patient aufgrund der Anwendung von Epoetin theta eine schwere Hautreaktion, wie SJS oder TEN, entwickelt, darf bei dem Patienten die Behandlung mit Epoetin theta zu keinem Zeitpunkt erneut begonnen werden</li></ul></li><li>besondere Patientengruppen<ul><li>eingeschränkte Leberfunktion und Sichelzellenanämie<ul><li>wegen der begrenzten Erfahrung keine Aussagen zur Wirksamkeit und Sicherheit von Epoetin theta bei Patienten mit eingeschränkter Leberfunktion und homozygoter Sichelzellanämie möglich</li></ul></li><li>ältere Patienten &gt; 75 Jahre<ul><li>in klinischen Studien hatten Patienten über 75 Jahren eine höhere Inzidenz schwerwiegender und schwerer unerwünschter Ereignisse, unabhängig von einem Kausalzusammenhang mit der Epoetin-theta-Therapie</li><li>Todesfälle waren in dieser Patientengruppe häufiger als bei jüngeren Patienten</li></ul></li></ul></li><li>Laborkontrollen<ul><li>es wird empfohlen, regelmäßig das Hämoglobin zu messen, ein komplettes Blutbild zu erstellen und Thrombozytenzählungen durchzuführen</li></ul></li><li>Symptomatische Anämie infolge chronischer Niereninsuffizienz<ul><li>bei nephrosklerotischen Patienten, die noch keine Dialyse erhalten, ist die Anwendung von Epoetin theta individuell abzuwägen, da die Möglichkeit einer beschleunigten Progression der Niereninsuffizienz nicht sicher ausgeschlossen werden kann</li><li>während der Hämodialyse können Patienten unter Behandlung mit Epoetin theta eine intensivere gerinnungshemmende Therapie benötigen, um einem Verschluss des arterio-venösen Shunts vorzubeugen</li><li>bei Patienten mit chronischer Niereninsuffizienz sollte die Hämoglobin-Konzentration in der Erhaltungsphase die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschreiten<ul><li>in klinischen Studien wurde ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse unter einer Epoetin-Therapie beobachtet, wenn Hämoglobin-Zielspiegel &gt; 12 g/dl (7,45mmol/l) angestrebt wurde</li><li>in kontrollierten klinischen Studien wurde kein signifikanter Nutzen für die Anwendung von Epoetinen gezeigt, wenn die Hämoglobin-Konzentration über den für die Symptomkontrolle einer Anämie und zur Vermeidung von Bluttransfusionen benötigten Spiegel erhöht wurde</li></ul></li><li>Vorsicht bei der Steigerung der Epoetin-theta-Dosis bei Patienten mit chronischer Niereninsuffizienz, da hohe kumulative Epoetin-Dosen mit einem erhöhten Risiko für Mortalität, schwerwiegende kardiovaskuläre und zerebrovaskuläre Ereignisse einhergehen können<ul><li>bei Patienten mit schlechtem Hämoglobin-Ansprechen auf Epoetine sollten alternative Ursachen für das schlechte Ansprechen in Betracht gezogen werden</li></ul></li></ul></li><li>Symptomatische Anämie bei Krebspatienten mit nicht-myeloischen malignen Erkrankungen, die eine Chemotherapie erhalten<ul><li>Einfluß auf Tumorwachstum<ul><li>Epoetine sind Wachstumsfaktoren, die primär die Bildung von Erythrozyten anregen</li><li>Erythropoietin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen exprimiert werden</li><li>wie bei allen Wachstumsfaktoren gibt es Bedenken, dass Epoetine das Wachstum von Tumoren anregen könnten</li></ul></li><li>in mehreren kontrollierten Studien zeigten Epoetine keine Verbesserung des Gesamtüberlebens oder eine Verminderung des Risikos einer Tumorprogression bei Patienten mit Anämie infolge einer Krebserkrankung.</li><li>in kontrollierten klinischen Studien zeigte die Anwendung von Epoetinen:<ul><li>eine verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Halstumoren, die eine Strahlentherapie erhielten, wenn ein Hämoglobin- Zielspiegel oberhalb von 14 g/dl (8,69mmol/l) angestrebt wurde</li><li>ein verkürztes Gesamtüberleben und eine erhöhte Letalität aufgrund einer Krankheitsprogression nach 4 Monaten bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten, wenn ein Hämoglobin-Zielwert zwischen 12 - 14 g/dl (7,45 - 8,69mmol/l) angestrebt wurde</li><li>ein erhöhtes Sterberisiko bei Patienten mit aktiven malignen Erkrankungen, dieweder eine Chemotherapie noch eine Strahlentherapie erhielten, wenn ein Hämoglobin-Zielwert von 12g/dl (7,45mmol/l) angestrebt wurde</li></ul></li><li>Epoetine sind nicht für die Anwendung bei dieser Patientenpopulation indiziert</li><li>daher sollte in einigen klinischen Situationen die symptomatische Anämie bei Krebspatienten bevorzugt mit Bluttransfusionen behandelt werden</li><li>Entscheidung zur Anwendung rekombinanter Erythropoietine sollte auf einer Nutzen-Risiko-Abwägung unter Beteiligung des einzelnen Patienten beruhen, unter Berücksichtigung der jeweils spezifischen klinischen Umstände<ul><li>Faktoren wie Tumorart und -stadium, Schweregrad der Anämie, Lebenserwartung, Behandlungsumfeld des Patienten und Wunsch des Patienten sollten in diese Beurteilung mit einbezogen werden</li></ul></li></ul></li></ul><p>Epoetin zeta</p><ul><li>allgemein<ul><li>Blutdruck <ul><li>bei allen Patienten, die Epoetin zeta erhalten, sollte der Blutdruck überwacht und ggf. behandelt werden</li><li>Epoetin zeta sollte mit Vorsicht angewendet werden bei unbehandeltem, unzureichend behandeltem oder schlecht einstellbarem Bluthochdruck</li><li>zusätzliche oder verstärkte blutdrucksenkende Therapie kann notwendig sein</li><li>ist der Blutdruck nicht kontrollierbar<ul><li>sollte die Behandlung mit Epoetin zeta beendet werden</li></ul></li></ul></li><li>hypertensive Krisen mit Enzephalopathie und Krampfanfällen, die unverzüglicher ärztlicher Hilfe und einer intensivmedizinischen Betreuung bedurften<ul><li>traten unter der Behandlung mit Epoetin zeta auch bei Patienten auf, die zuvor einen normalen oder niedrigen Blutdruck aufwiesen</li><li>als mögliches Warnsignal sollten insbesondere plötzlich einsetzende, stechende, migräneartige Kopfschmerzen beachtet werden</li></ul></li><li>Epilepsie, Krampfneigung<ul><li>Epoetin zeta sollte bei Patienten mit Epilepsie, Krampfanfällen in der Vorgeschichte oder Allgemeinerkrankungen, die mit einer erhöhten Krampfneigung einhergehen, wie z.B. ZNS- Infektionen und Hirnmetastasen, vorsichtig angewendet werden</li></ul></li><li>Patienten mit chronischer Leberinsuffizienz <ul><li>Anwendung mit Vorsicht</li><li>Sicherheit von Epoetin zeta bei Patienten mit Leberfunktionsstörungen bisher nicht erwiesen</li></ul></li><li>thrombovaskuläre Ereignisse (TVEs)<ul><li>erhöhte Inzidenz von thrombovaskulären Ereignissen (TVEs) bei Patienten beobachtet, die mit ESAs behandelt wurden <ul><li>dazu gehören venöse und arterielle Thrombosen und Embolien (darunter einige mit tödlichem Ausgang) wie z.B. tiefe Venenthrombose, Lungenembolie, Netzhautthrombose und Myokardinfarkt</li><li>darüber hinaus wurde über zerebrovaskuläre Ereignisse (einschließlich Hirninfarkt, Hirnblutung und transitorische ischämische Attacken) berichtet</li></ul></li><li>das berichtete Risiko für diese TVEs sollte sorgfältig gegen den Nutzen einer Behandlung mit Epoetin zeta abgewogen werden<ul><li>gilt insbesondere für Patienten mit vorbestehenden Risikofaktoren für TVEs wie Adipositas und anamnestisch bekannte TVEs (z.B. tiefe Venenthrombose, Lungenembolie oder zerebrovaskulärer Insult)</li></ul></li></ul></li><li>Hämoglobinspiegel<ul><li>bei allen Patienten ist der Hämoglobinspiegel engmaschig zu ü berwachen, da potentiell ein erhöhtes Risiko fü r thromboembolische Ereignisse und tödliche Verläufe besteht, wenn Patienten behandelt wurden, deren Hämoglobinkonzentration über dem Konzentrationsbereich liegt, für den das Arzneimittel indiziert ist</li></ul></li><li>während der Behandlung mit Erythropoetin kann es innerhalb des Normbereiches zu einem leichten dosisabhängigen Anstieg der Thrombozytenzahl kommen<ul><li>bei fortgesetzter Behandlung rückläufig</li><li>empfohlen, die Thrombozytenzahl während der ersten 8 Wochen der Behandlung regelmäßig zu überwachen</li></ul></li><li>alle anderen Ursachen einer Anämie müssen vor Beginn einer Behandlung mit Epoetin zeta sowie vor einer geplanten Dosiserhöhung abgeklärt und behandelt werden<ul><li>Eisen-, Folsäure- oder Vitamin B<sub>12</sub>-Mangel</li><li>Aluminiumintoxikationen</li><li>Infektionen oder Entzündungen</li><li>Blutverlust</li><li>Hämolyse</li><li>Knochenmarksfibrose jeglicher Genese</li></ul></li><li>in den meisten Fällen gehen gleichzeitig mit der Zunahme des gepackten Zellvolumens die Ferritinwerte im Serum zurück</li><li>für das Erzielen eines optimalen Ansprechens auf Erythropoetin müssen ausreichende Eisenreserven sichergestellt werden:<ul><li>für Patienten mit chronischer Niereninsuffizienz, deren Serum-Ferritinkonzentration unter 100 ng/ml liegt, wird eine orale Eisensubstitution empfohlen, z.B. 200 - 300 mg / Tag (100-200 mg / Tag bei pädiatrischen Patienten)</li><li>für alle Tumorpatienten, bei denen die Transferrinsättigung weniger als 20% beträgt, wird eine orale Eisensubstitution mit 200 - 300 mg / Tag empfohlen</li><li>für Patienten in einem autologen Blutspendeprogramm sollte mit der oralen Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) bereits einige Wochen vor der Eigenblutspende begonnen werden, um vor der Einleitung der Behandlung mit Epoetin zeta sowie in deren weiterem Verlauf ausreichend hohe Eisenreserven zu gewährleisten</li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff geplant ist, sollten während der gesamten Dauer der Behandlung mit Epoetin zeta eine orale Eisen-II-Substitution (entsprechend einer Dosierung von 200 mg elementares Eisen / Tag) erhalten<ul><li>wenn möglich, sollte mit der Eisensubstitution bereits vor Einleitung der Behandlung mit Epoetin zeta begonnen werden, um adäquate Eisenreserven zu gewährleisten</li></ul></li></ul></li><li>Porphyrie<ul><li>sehr selten wurde die Entwicklung oder Exazerbation einer Porphyrie bei mit Epoetin zeta behandelten Patienten beobachtet</li><li>Epoetin zeta sollte bei Patienten mit Porphyrie mit Vorsicht angewendet werden</li></ul></li><li>schwere arzneimittelinduzierte Hautreaktionen (SCARs)<ul><li>über schwere arzneimittelinduzierte Hautreaktionen (SCARs), einschließlich Stevens- Johnson-Syndrom (SJS) und toxisch epidermaler Nekrolyse (TEN), die lebensbedrohlich oder tödlich sein können, im Zusammenhang mit Epoetin-Behandlungen berichtet<ul><li>schwerere Fälle wurden im Zusammenhang mit langwirkenden beobachtet</li></ul></li><li>zum Zeitpunkt der Verschreibung sollten die Patienten auf die Anzeichen und Symptome hingewiesen und engmaschig im Hinblick auf Hautreaktionen überwacht werden<ul><li>wenn Anzeichen und Symptome, die auf diese Reaktionen hinweisen, auftreten, sollte die Behandlung unverzüglich beendet und eine alternative Behandlung in Betracht gezogen werden</li></ul></li><li>wenn ein Patient aufgrund der Anwendung von Epoetin eine schwere Hautreaktion, wie SJS oder TEN, entwickelt<ul><li>darf bei dem Patienten die Behandlung mit Epoetin zu keinem Zeitpunkt erneut begonnen werden</li></ul></li></ul></li><li>Rückverfolgbarkeit<ul><li>um die Rückverfolgbarkeit von Erythropoesestimulierenden Wirkstoffen (ESAs) zu verbessern, soll der Handelsname und die Chargennummer des angewendeten ESAs eindeutig in die Patientenakte eingetragen (oder festgelegt) werden</li></ul></li><li>Umstellung<ul><li>Patienten sollen nur unter geeigneter Überwachung von einer ESA auf ein anderes umgestellt werden</li></ul></li></ul></li><li>Erythroblastopenie (PRCA)<ul><li>übber das Auftreten einer antikörpervermittelten Erythroblastopenie (PRCA) wurde nach monate- bis jahrelanger Behandlung mit Epoetin berichtet</li><li>ebenfalls traten Fälle bei Patienten mit Hepatitis C auf, die mit Interferon und Ribavirin behandelt wurden, während ESAs begleitend eingesetzt wurden</li><li>Epoetin zeta ist für die Behandlung einer Anämie, die mit Hepatitis C assoziiert ist, nicht zugelassen</li><li>bei Patienten mit plötzlichem Wirkverlust, definiert als Verminderung der Hämoglobinwerte (1 - 2 g/dl pro Monat) mit erhöhtem Bedarf an Transfusionen<ul><li>sollte der Retikulozytenwert bestimmt und die üblichen Ursachen für ein Nichtansprechen (z.B. Eisen-, Folsäure- oder Vitamin-B12-Mangel, Aluminiumintoxikation, Infektionen oder Entzündungen, Blutverlust, Hämolyse und Knochenmarkfibrose jeglicher Genese) abgeklärt werden</li></ul></li><li>bei einer paradoxen Verminderung der Hämoglobinwerte und der Entwicklung einer schweren Anämie, die mit einem erniedrigten Retikulozytenwert einhergeht<ul><li>sollte die Behandlung mit Epoetin zeta sofort abgesetzt und die Anti-Erythropoetin- Antikörper bestimmt werden</li><li>eine Untersuchung des Knochenmarks zur Diagnose einer PRCA sollte ebenfalls erwogen werden</li></ul></li><li>aufgrund des Risikos einer Kreuzreaktion sollte keine andere ESA-Therapie begonnen werden</li></ul></li><li>Behandlung der symptomatischen Anä mie bei Erwachsenen und Kindern mit chronischer Niereninsuffizienz<ul><li>Hämoglobin-Bestimmungen<ul><li>bei Patienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte der Hämoglobinanstieg ungefähr 1 g/dl (0,62 mmol/l) pro Monat betragen und 2 g/dl (1,25 mmol/l) pro Monat nicht überschreiten<ul><li>um das Risiko der Zunahme eines Bluthochdrucks zu minimieren</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz sollte bei Erhaltungstherapie die empfohlene Obergrenze der Hämoglobin-Zielkonzentration nicht überschritten werden<ul><li>in klinischen Studien wurden ein erhöhtes Mortalitätsrisiko und Risiko für schwerwiegende kardiovaskuläre Ereignisse beobachtet, wenn Erythropoese-stimulierende Wirkstoffe (ESA) mit einer Hämoglobin-Zielkonzentration von über 12 g/dl (7,5 mmol/l) gegeben wurden</li></ul></li><li>kontrollierte klinische Studien haben keinen signifikanten Nutzen gezeigt, der auf die Gabe von Epoetinen zurückzuführen ist, wenn die Hämoglobinkonzentration über den für die Kontrolle der Anämiesymptome und die Vermeidung von Bluttransfusionen erforderliche Konzentration gesteigert wird</li><li>Vorsicht, wenn die Epoetin-Dosis bei Patienten mit chronischer Niereninsuffizienz gesteigert wird<ul><li>da hohe kumulierte Epoetin-Dosen mit einem gesteigerten Mortalitätsrisiko, schwerwiegenden kardiovaskulären und zerebrovaskulären Ereignissen assoziiert sein können</li><li>bei Patienten, die ein schlechtes Ansprechen des Hämoglobinwertes auf Epoetin zeigen, sollten alternative Gründe für das schlechte Anprechen bedacht werden</li></ul></li><li>bei Patienten mit chronischer Niereninsuffizienz, denen Epoetin zeta subkutan gegeben wird, sollten regelmäßig Kontrollen hinsichtlich eines Wirkverlustes, definiert als ein Nicht- oder vermindertes Ansprechen auf eine Behandlung mit Epoetin zeta bei Patienten, die zuvor auf eine solche Therapie angesprochen haben, durchgeführt werden<ul><li>dies zeigt sich durch eine anhaltende Verminderung der Hämoglobinwerte trotz Steigerung der Epoetin-zeta- Dosis</li></ul></li><li>bei Verabreichung von Epoetin in längeren Dosisintervallen (länger als 1-mal wöchentlich) können bei einigen Patienten möglicherweise keine ausreichenden Hämoglobinwerte aufrechterhalten werden, so dass eine Erhöhung der Epoetin-Dosis erforderlich werden kann<ul><li>Hämoglobinwerte sind regelmäßig zu kontrollieren</li></ul></li><li>Shuntthrombosen <ul><li>Shuntthrombosen sind insbesondere bei Hämodialysepatienten aufgetreten, die eine Tendenz zur Hypotonie aufweisen oder deren arteriovenöse Fisteln Komplikationen aufweisen (z.B. Stenosen, Aneurysmen, etc.)</li><li>eine frühzeitige Shuntkorrektur und Thromboseprophylaxe durch die Gabe von z.B. Acetylsalicylsäure wird bei diesen Patienten empfohlen</li></ul></li><li>Hyperkaliämie<ul><li>in Einzelfällen über Hyperkaliämie berichtet, jedoch konnte ein Kausalzusammenhang nicht festgestellt werden</li><li>bei Patienten mit chronischer Niereninsuffizienz sollten die Serumelektrolyte überwacht werden</li><li>bei erhöhtem oder ansteigendem Kaliumwert sollte zusätzlich zu einer geeigneten Behandlung der Hyperkaliämie erwogen werden, die Epoetin-zeta-Therapie bis zur Normalisierung der Kaliumwerte zu unterbrechen</li></ul></li><li>im Verlauf der Therapie mit Erythropoetin ist aufgrund des erhöhten Hämatokrits häufig eine Steigerung der Heparindosis während der Hämodialyse notwendig<ul><li>bei nicht optimaler Heparinisierung ist ein Verschluss des Dialysesystems möglich</li></ul></li><li>nach zur Zeit vorliegenden Erkenntnissen wird durch die Behandlung der Anämie mit Epoetin zeta bei Erwachsenen mit Niereninsuffizienz, die noch nicht dialysepflichtig sind, das Fortschreiten der Niereninsuffizienz nicht beschleunigt</li></ul></li><li>Behandlung von Patienten mit chemotherapieinduzierter Anämie, Auswirkungen auf das Tumorwachstum <ul><li>Hämoglobin-Bestimmungen<ul><li>bei Tumorpatienten, die mit Epoetin zeta behandeltwerden, sollten regelmäßig Hämoglobin-Bestimmungen durchgeführt werden, bis ein stabiler Hämoglobin-Wert erreicht ist</li><li>dieser soll danach in regelmäßigen Abständen überprüft werden</li></ul></li><li>Epoetine sind Wachstumsfaktoren, die in erster Linie die Produktion roter Blutzellen anregen</li><li>Erythropoetin-Rezeptoren können auf der Oberfläche verschiedener Tumorzellen vorhanden sein<ul><li>we bei allen Wachstumsfaktoren sollte bedacht werden, dass Epoetine das Wachstum von Tumoren jeglicher Art stimulieren können</li><li>eine Bedeutung von ESAs bei der Tumorprogression oder bei einer verkürzten progressionsfreien Überlebenszeit kann nicht ausgeschlossen werden</li></ul></li><li>in kontrollierten klinischen Studien war die Anwendung von Epoetin zeta und anderen ESAs mit einer verkürzten Zeit bis zur Tumorprogression oder mit einer verkürzten Gesamtüberlebenszeit<br />verbunden <ul><li>verkürzte Zeit bis zur Tumorprogression bei Patienten mit fortgeschrittenen Kopf-Hals-Tumoren, die eine Strahlentherapie erhielten, wenn eine Hämoglobin-Zielkonzentration über 14 g/dl (8,7 mmol/l) angestrebt wurde</li><li>kürzeres Gesamtüberleben und erhöhte Sterblichkeit, die auf eine Krankheitsprogression nach 4 Monaten bei Patienten mit metastasierendem Brustkrebs, die eine Chemotherapie erhielten, zurückgeführt wurde, wenn eine Hämoglobin-Zielkonzentration von 12 - 14 g/dl (7,5 - 8,7 mmol/l) angestrebt wurde</li><li>erhöhtes Sterberisiko bei Patienten mit einer aktiven malignen Erkrankung, die weder eine Chemo- noch eine Strahlentherapie erhielten und bei denen eine Hämoglobinkonzentration von 12 g/dl (7,5 mmol/l) angestrebt wurde<ul><li>ESAs für die Anwendung in dieser Patientenpopulation nicht indiziert</li></ul></li><li>in einer Primär-Analyse wurde in der Epoetin-zeta plus SOC (standard of care)-Gruppe eine 9%ige Erhöhung des Risikos für PD (documented progressive disease) oder Tod beobachtet und ein um 15 % erhöhtes Risiko, welches bei Patientinnen mit metastasiertem Brustkrebs, die eine Chemotherapie erhielten und bei denen eine Hämoglobinkonzentration zwischen 10 g/dl und 12 g/dl (6,2 - 7,5 mmol/l) angestrebt wurde, statistisch nicht ausgeschlossen werden kann</li></ul></li><li>in Abhängigkeit von der klinischen Situation sollte die Behandlung mit Bluttransfusionen bei Krebspatienten mit Chemotherapie assoziierter Anämie bevorzugt werden<ul><li>einer Entscheidung zur Gabe von rekombinantem Erythropoetin sollte eine Nutzen-Risiko Bewertung unter Einbeziehung des Patienten vorausgehen, die die spezifische klinische Situation berücksichtigt</li><li>Nutzen-Risiko Abwägung sollte die folgenden Faktoren einschließen: <ul><li>Art und Stadium des Tumors</li><li>Schwere der Anämie</li><li>Erkrankungsprognose</li><li>Umfeld, in dem der Patient behandelt wird</li><li>Behandlungspräferenz des Patienten</li></ul></li></ul></li><li>bei Tumorpatienten unter Chemotherapie, die möglicherweise transfundiert werden müssen, sollte für die Beurteilung der Therapieeffizienz von Epoetin zeta eine 2- bis 3-wöchige Verzögerung zwischen ESA-Gabe und der Erythropoetin-Antwort berücksichtigt werden</li></ul></li><li>Patienten, die präoperativ an einem autologen Blutspendeprogramm teilnehmen<ul><li>alle speziellen Warnhinweise und Vorsichtsmaßnahmen, die mit einem autologen Blutspendeprogramm assoziiert sind, insbesondere routinemäßiger Volumenersatz, sollten berü cksichtigt werden</li></ul></li><li>Patienten, bei denen ein großer, elektiver orthopädischer Eingriff vorgesehen ist<ul><li>fremdblutsparende Maßnahmen sollten bei operativen Eingriffen immer zum Einsatz kommen</li><li>Patienten, die sich einem elektiven orthopädischen Eingriff unterziehen, haben ein erhöhtes Risiko für thrombotische und vaskuläre Erkrankungen, besonders bei einer zugrunde liegenden kardiovaskulären Erkrankung<ul><li>daher sollte insbesondere bei diesen Patienten eine angemessene Thromboseprophylaxe erfolgen</li></ul></li><li>zusätzlich sollten spezielle Vorsichtsmaßnahmen bei Patienten mit Disposition für tiefe Venenthrombosen ergriffen werden</li><li>Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l)<ul><li>Epoetin zeta sollte nicht bei diesen Patienten eingesetzt werden</li><li>bei Patienten mit einem Ausgangshämoglobinwert von &gt; 13 g/dl (&gt; 8,1 mmol/l) kann nicht ausgeschlossen werden, dass eine Epoetin zeta-Behandlung mit einem erhöhten Risiko<br />postoperativer Thrombose-/Gefäßereignissen einhergeht</li></ul></li></ul></li></ul>",
		matches: []Match{
			{
				pattern: 0,
				len:     32,
				end:     36,
			},
			{
				pattern: 1,
				len:     9,
				end:     5607,
			},
			{
				pattern: 1,
				len:     9,
				end:     20572,
			},
			{
				pattern: 1,
				len:     9,
				end:     32879,
			},
			{
				pattern: 1,
				len:     9,
				end:     43797,
			},
		},
	},
}

type testCase struct {
	patterns []string
	haystack string
	matches  []Match
}

func BenchmarkAhoCorasick_LeftmostInsensitiveWholeWord(b *testing.B) {
	b.ResetTimer()
	b.StopTimer()
	acs := make([]AhoCorasick, len(leftmostInsensitiveWholeWordTestCases))
	for i, t2 := range leftmostInsensitiveWholeWordTestCases {
		builder := NewAhoCorasickBuilder(Opts{
			AsciiCaseInsensitive: true,
			MatchOnlyWholeWords:  true,
			MatchKind:            LeftMostLongestMatch,
		})
		ac := builder.Build(t2.patterns)
		acs[i] = ac
	}
	b.StartTimer()

	for i := 0; i < b.N; i++ {
		for i, ac := range acs {
			_ = ac.FindAll(leftmostInsensitiveWholeWordTestCases[i].haystack)
		}
	}
}
