/*
 * Copyright (C) 2025 James Hogan <james@albanarts.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 *
 */

/**
 * Test the additional errors that are generated from commands that read from
 * the framebuffer if the number of views in the current read framebuffer is
 * greater than 1.
 */

#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_compat_version = 30;
	config.supports_gl_core_version = 31;
	config.khr_no_error_support = PIGLIT_HAS_ERRORS;

PIGLIT_GL_TEST_CONFIG_END

#define TEX_WIDTH 8
#define TEX_HEIGHT 4

void
piglit_init(int argc, char **argv)
{
	GLuint tex_2d, tex_2da, fbo;
	GLenum fbstatus;
	bool pass = true;
	GLuint *buf;

	piglit_require_extension("GL_OVR_multiview");

	/* generate 2d texture */
	glGenTextures(1, &tex_2d);
	glBindTexture(GL_TEXTURE_2D, tex_2d);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB8, TEX_WIDTH, TEX_HEIGHT, 0,
		     GL_RGB, GL_UNSIGNED_BYTE, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	/* generate 2d array texture */
	glGenTextures(1, &tex_2da);
	glBindTexture(GL_TEXTURE_2D_ARRAY, tex_2da);
	glTexImage3D(GL_TEXTURE_2D_ARRAY, 0, GL_RGB8, TEX_WIDTH, TEX_HEIGHT, 2,
		     0, GL_RGB, GL_UNSIGNED_BYTE, NULL);
	glBindTexture(GL_TEXTURE_2D_ARRAY, 0);

	/* generate FBO */
	glGenFramebuffers(1, &fbo);
	glBindFramebuffer(GL_FRAMEBUFFER, fbo);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		piglit_report_result(PIGLIT_FAIL);

	/*
	 * OVR_multiview:
	 * "INVALID_FRAMEBUFFER_OPERATION is generated by commands that read
	 * from the framebuffer such as BlitFramebuffer, ReadPixels,
	 * CopyTexImage*, and CopyTexSubImage*, if the number of views in the
	 * current read framebuffer is greater than 1."
	 */

	/* attach both layers */
	glFramebufferTextureMultiviewOVR(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0,
					 tex_2da, 0, 0, 2);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		piglit_report_result(PIGLIT_FAIL);
	fbstatus = glCheckFramebufferStatus(GL_FRAMEBUFFER);
	if (fbstatus != GL_FRAMEBUFFER_COMPLETE) {
		printf("Line %u: Expected GL_FRAMEBUFFER_COMPLETE, got %s\n",
		       __LINE__, piglit_get_gl_enum_name(fbstatus));
		piglit_report_result(PIGLIT_FAIL);
	}
	if (!piglit_check_gl_error(GL_NO_ERROR))
		piglit_report_result(PIGLIT_FAIL);

	/* test that glBlitFramebuffer errors */
	glBlitFramebuffer(0, 0, 3, 3, 4, 0, 8, 4, GL_COLOR_BUFFER_BIT,
			  GL_NEAREST);
	pass &= piglit_check_gl_error(GL_INVALID_FRAMEBUFFER_OPERATION);

	/* test that glReadPixels errors */
	buf = malloc(TEX_WIDTH*TEX_HEIGHT*sizeof(*buf));
	glReadPixels(0, 0, TEX_WIDTH, TEX_HEIGHT, GL_RGBA,
		     GL_UNSIGNED_INT_8_8_8_8, buf);
	pass &= piglit_check_gl_error(GL_INVALID_FRAMEBUFFER_OPERATION);

	/* test that glCopyTexImage2D errors */
	glBindTexture(GL_TEXTURE_2D, tex_2d);
	glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGB8, 0, 0, TEX_WIDTH, TEX_HEIGHT,
			 0);
	pass &= piglit_check_gl_error(GL_INVALID_FRAMEBUFFER_OPERATION);

	/* test that glCopyTexSubImage2D errors */
	glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, TEX_WIDTH,
			    TEX_HEIGHT);
	pass &= piglit_check_gl_error(GL_INVALID_FRAMEBUFFER_OPERATION);


	/* Now attach only a single view and try again */
	glFramebufferTextureMultiviewOVR(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0,
					 tex_2da, 0, 0, 1);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		piglit_report_result(PIGLIT_FAIL);
	fbstatus = glCheckFramebufferStatus(GL_FRAMEBUFFER);
	if (fbstatus != GL_FRAMEBUFFER_COMPLETE) {
		printf("Line %u: Expected GL_FRAMEBUFFER_COMPLETE, got %s\n",
		       __LINE__, piglit_get_gl_enum_name(fbstatus));
		piglit_report_result(PIGLIT_FAIL);
	}
	if (!piglit_check_gl_error(GL_NO_ERROR))
		piglit_report_result(PIGLIT_FAIL);

	/* test that glBlitFramebuffer doesn't error */
	glBlitFramebuffer(0, 0, 3, 3, 4, 0, 7, 3, GL_COLOR_BUFFER_BIT,
			  GL_NEAREST);
	pass &= piglit_check_gl_error(GL_NO_ERROR);

	/* test that glReadPixels doesn't error */
	glReadPixels(0, 0, TEX_WIDTH, TEX_HEIGHT, GL_RGBA,
		     GL_UNSIGNED_INT_8_8_8_8, buf);
	pass &= piglit_check_gl_error(GL_NO_ERROR);

	/* test that glCopyTexImage2D doesn't error */
	glBindTexture(GL_TEXTURE_2D, tex_2d);
	glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGB8, 0, 0, TEX_WIDTH, TEX_WIDTH,
			 0);
	pass &= piglit_check_gl_error(GL_NO_ERROR);

	/* test that glCopyTexSubImage2D doesn't error */
	glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, TEX_WIDTH, TEX_WIDTH);
	pass &= piglit_check_gl_error(GL_NO_ERROR);

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}

enum piglit_result
piglit_display(void)
{
	/* Should never be reached */
	return PIGLIT_FAIL;
}
