/* ceilinglamp_high.c
**
**    code for high quality ceiling lamp
**
**    Updated Version foobillard++ started at 12/2010
**    Copyright (C) 2010 - 2013 Holger Schaekel (foobillardplus@go4more.de)
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License Version 2 as
**    published by the Free Software Foundation;
**
**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.
**
**    You should have received a copy of the GNU General Public License
**    along with this program; if not, write to the Free Software
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
*/

#include <stdio.h>
#include <stdlib.h>
#include "ceilinglamp_high.h"

static int ceilinglamp_high_id = -1;

/***********************************************************************
 *        This is a general routine to make a new material active      *
 ***********************************************************************/

void lamp_Material_Parameters(float shininess, float r_diffuseColor, float g_diffuseColor, float b_diffuseColor, float r_specularColor, float g_specularColor, float b_specularColor) {
	GLfloat glfModelSpecular[4];

	// This causes the polygon's ambient & diffuse colors to be derived from the glColor() value.
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, shininess);
	/* Set the specular shading color. */
	glfModelSpecular[0] = r_specularColor;
	glfModelSpecular[1] = g_specularColor;
	glfModelSpecular[2] = b_specularColor;
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, glfModelSpecular);
	glColor3f(r_diffuseColor, g_diffuseColor, b_diffuseColor);
}

/***********************************************************************
 *                          Make an glcallist                          *
 ***********************************************************************/

void init_ceilinglamp_high(void) {
	 if(ceilinglamp_high_id != -1) {
	 	glDeleteLists(ceilinglamp_high_id,1);
	 }
  fprintf(stderr,"Initialize high-res ceiling-lamp graphics\n");
  ceilinglamp_high_id = glGenLists(1);
  glNewList(ceilinglamp_high_id, GL_COMPILE);
  ceilinglamp_high();
  glEndList();
}

/***********************************************************************
 *                         display the lamp                            *
 ***********************************************************************/

void display_ceilinglamp_high(void) {
	 if(ceilinglamp_high_id == -1) {
	 	init_ceilinglamp_high();
	 } else {
	 	glCallList(ceilinglamp_high_id);
	 }
}

/***********************************************************************
 *                           build the lamp                            *
 ***********************************************************************/

void ceilinglamp_high(void) {
	static const GLfloat VertexData[] = {
		-0.139668f, 0.853677f, -0.280424f, -0.139668f, 0.853677f, -0.280424f, -0.139549f, 0.853677f, 0.280381f, 
		-0.139549f, 0.853677f, 0.280381f, -0.136540f, 0.853677f, -0.238809f, 
		-0.136540f, 0.853677f, -0.238809f, -0.136439f, 0.853677f, 0.238765f, 
		-0.136439f, 0.853677f, 0.238765f, -0.135977f, 0.879246f, -0.280148f, 
		-0.135858f, 0.879246f, 0.280104f, -0.135679f, 0.853634f, -0.280126f, 
		-0.135679f, 0.853634f, -0.280126f, -0.135561f, 0.853634f, 0.280081f, 
		-0.135561f, 0.853634f, 0.280081f, -0.132932f, 0.879246f, -0.239634f, 
		-0.132831f, 0.879246f, 0.239588f, -0.132641f, 0.853634f, -0.239700f, 
		-0.132641f, 0.853634f, -0.239700f, -0.132540f, 0.853634f, 0.239655f, 
		-0.132540f, 0.853634f, 0.239655f, -0.132094f, 0.878415f, -0.279858f, 
		-0.131975f, 0.878415f, 0.279812f, -0.130390f, 0.853677f, -0.321111f, 
		-0.130390f, 0.853677f, -0.321111f, -0.130254f, 0.853677f, 0.321065f, 
		-0.130254f, 0.853677f, 0.321065f, -0.129136f, 0.878415f, -0.240501f, 
		-0.129034f, 0.878415f, 0.240454f, -0.126945f, 0.879246f, -0.319760f, 
		-0.126809f, 0.879246f, 0.319712f, -0.126667f, 0.853634f, -0.319651f, 
		-0.126667f, 0.853634f, -0.319651f, -0.126531f, 0.853634f, 0.319603f, 
		-0.126531f, 0.853634f, 0.319603f, -0.125465f, 0.905134f, -0.279363f, 
		-0.125347f, 0.905134f, 0.279314f, -0.123319f, 0.878415f, -0.318338f, 
		-0.123185f, 0.878415f, 0.318288f, -0.122656f, 0.905134f, -0.241982f, 
		-0.122554f, 0.905134f, 0.241932f, -0.121881f, 0.903517f, -0.279095f, 
		-0.121763f, 0.903517f, 0.279045f, -0.121286f, 0.853677f, -0.199966f, 
		-0.121286f, 0.853677f, -0.199966f, -0.121201f, 0.853677f, 0.199915f, 
		-0.121201f, 0.853677f, 0.199915f, -0.119152f, 0.903517f, -0.242782f, 
		-0.119050f, 0.903517f, 0.242731f, -0.118081f, 0.879246f, -0.201817f, 
		-0.117996f, 0.879246f, 0.201765f, -0.117822f, 0.853634f, -0.201966f, 
		-0.117822f, 0.853634f, -0.201966f, -0.117737f, 0.853634f, 0.201914f, 
		-0.117737f, 0.853634f, 0.201914f, -0.117132f, 0.905134f, -0.315911f, 
		-0.116998f, 0.905134f, 0.315859f, -0.114709f, 0.878415f, -0.203765f, 
		-0.114623f, 0.878415f, 0.203711f, -0.113786f, 0.903517f, -0.314599f, 
		-0.113653f, 0.903517f, 0.314545f, -0.109532f, 0.853677f, -0.357257f, 
		-0.109532f, 0.853677f, -0.357257f, -0.109381f, 0.853677f, 0.357201f, 
		-0.109381f, 0.853677f, 0.357201f, -0.108975f, 0.929791f, -0.278130f, 
		-0.108953f, 0.905134f, -0.207089f, -0.108866f, 0.905134f, 0.207034f, 
		-0.108857f, 0.929791f, 0.278075f, -0.106638f, 0.879246f, -0.354950f, 
		-0.106535f, 0.929791f, -0.245665f, -0.106488f, 0.879246f, 0.354893f, 
		-0.106431f, 0.929791f, 0.245608f, -0.106404f, 0.853634f, -0.354764f, 
		-0.106404f, 0.853634f, -0.354764f, -0.106254f, 0.853634f, 0.354707f, 
		-0.106254f, 0.853634f, 0.354707f, -0.105860f, 0.927433f, -0.277898f, 
		-0.105841f, 0.903517f, -0.208887f, -0.105753f, 0.903517f, 0.208830f, 
		-0.105742f, 0.927433f, 0.277841f, -0.103593f, 0.878415f, -0.352522f, 
		-0.103490f, 0.927433f, -0.246360f, -0.103443f, 0.878415f, 0.352464f, 
		-0.103385f, 0.927433f, 0.246302f, -0.101737f, 0.929791f, -0.309873f, 
		-0.101606f, 0.929791f, 0.309814f, -0.0998230f, 1.58893f, -0.0132350f, 
		-0.0998230f, 1.58893f, -0.0132350f, -0.0998230f, 1.61893f, -0.0132350f, 
		-0.0998230f, 1.61893f, -0.0132350f, -0.0998230f, 1.61893f, -0.0132350f, 
		-0.0998200f, 1.58893f, 0.00172600f, -0.0998200f, 1.58893f, 0.00172600f, 
		-0.0998200f, 1.61893f, 0.00172600f, -0.0998200f, 1.61893f, 0.00172600f, 
		-0.0998200f, 1.61893f, 0.00172600f, -0.0988290f, 0.927433f, -0.308732f, 
		-0.0986980f, 0.927433f, 0.308672f, -0.0983950f, 0.905134f, -0.348379f, 
		-0.0982480f, 0.905134f, 0.348319f, -0.0975960f, 1.58893f, -0.0280290f, 
		-0.0975960f, 1.58893f, -0.0280290f, -0.0975960f, 1.61893f, -0.0280290f, 
		-0.0975960f, 1.61893f, -0.0280290f, -0.0975960f, 1.61893f, -0.0280290f, 
		-0.0975870f, 1.58893f, 0.0165200f, -0.0975870f, 1.58893f, 0.0165200f, 
		-0.0975870f, 1.61893f, 0.0165200f, -0.0975870f, 1.61893f, 0.0165200f, 
		-0.0975870f, 1.61893f, 0.0165200f, -0.0955850f, 0.903517f, -0.346139f, 
		-0.0954380f, 0.903517f, 0.346078f, -0.0952600f, 0.853677f, -0.167344f, 
		-0.0952600f, 0.853677f, -0.167344f, -0.0951890f, 0.853677f, 0.167282f, 
		-0.0951890f, 0.853677f, 0.167282f, -0.0946340f, 0.929791f, -0.215360f, 
		-0.0945430f, 0.929791f, 0.215299f, -0.0931890f, 1.58893f, -0.0423260f, 
		-0.0931890f, 1.58893f, -0.0423260f, -0.0931890f, 1.61893f, -0.0423260f, 
		-0.0931890f, 1.61893f, -0.0423260f, -0.0931890f, 1.61893f, -0.0423260f, 
		-0.0931740f, 1.58893f, 0.0308150f, -0.0931740f, 1.58893f, 0.0308150f, 
		-0.0931740f, 1.61893f, 0.0308150f, -0.0931740f, 1.61893f, 0.0308150f, 
		-0.0931740f, 1.61893f, 0.0308150f, -0.0931740f, 1.61893f, 0.0308150f, 
		-0.0927430f, 0.879246f, -0.170058f, -0.0926710f, 0.879246f, 0.169995f, 
		-0.0925400f, 0.853634f, -0.170277f, -0.0925400f, 0.853634f, -0.170277f, 
		-0.0924680f, 0.853634f, 0.170214f, -0.0924680f, 0.853634f, 0.170214f, 
		-0.0919290f, 0.927433f, -0.216923f, -0.0918370f, 0.927433f, 0.216860f, 
		-0.0910765f, 1.58880f, -0.0130759f, -0.0910765f, 1.58880f, -0.0130759f, 
		-0.0910733f, 1.58880f, 0.00156429f, -0.0910733f, 1.58880f, 0.00156429f, 
		-0.0900950f, 0.878415f, -0.172913f, -0.0900210f, 0.878415f, 0.172849f, 
		-0.0892990f, 1.57721f, -0.0124480f, -0.0892960f, 1.57721f, 0.000935000f, 
		-0.0891910f, 1.58880f, -0.0256060f, -0.0891910f, 1.58880f, -0.0256060f, 
		-0.0891820f, 1.58880f, 0.0140930f, -0.0891820f, 1.58880f, 0.0140930f, 
		-0.0873470f, 0.951668f, -0.276514f, -0.0873070f, 1.57721f, -0.0256830f, 
		-0.0872990f, 1.57721f, 0.0141690f, -0.0872300f, 0.951668f, 0.276449f, 
		-0.0867010f, 1.58893f, -0.0558070f, -0.0867010f, 1.58893f, -0.0558070f, 
		-0.0867010f, 1.61893f, -0.0558070f, -0.0867010f, 1.61893f, -0.0558070f, 
		-0.0867010f, 1.61893f, -0.0558070f, -0.0866800f, 1.58893f, 0.0442930f, 
		-0.0866800f, 1.58893f, 0.0442930f, -0.0866800f, 1.61893f, 0.0442930f, 
		-0.0866800f, 1.61893f, 0.0442930f, -0.0866800f, 1.61893f, 0.0442930f, 
		-0.0855750f, 0.905134f, -0.177786f, -0.0855000f, 0.905134f, 0.177720f, 
		-0.0854640f, 0.929791f, -0.338072f, -0.0853920f, 0.951668f, -0.250495f, 
		-0.0853210f, 0.929791f, 0.338006f, -0.0852860f, 0.951668f, 0.250429f, 
		-0.0851720f, 1.57086f, -0.0121400f, -0.0851700f, 1.57086f, 0.000625000f, 
		-0.0848787f, 1.58880f, -0.0395964f, -0.0848787f, 1.58880f, -0.0395964f, 
		-0.0848645f, 1.58880f, 0.0280824f, -0.0848645f, 1.58880f, 0.0280824f, 
		-0.0848460f, 0.948656f, -0.276327f, -0.0847290f, 0.948656f, 0.276261f, 
		-0.0833650f, 1.57721f, -0.0384730f, -0.0833510f, 1.57721f, 0.0269580f, 
		-0.0832720f, 1.57086f, -0.0247630f, -0.0832640f, 1.57086f, 0.0132480f, 
		-0.0831310f, 0.903517f, -0.180421f, -0.0830540f, 0.903517f, 0.180355f, 
		-0.0830220f, 0.927433f, -0.336124f, -0.0829460f, 0.948656f, -0.251054f, 
		-0.0828790f, 0.927433f, 0.336058f, -0.0828400f, 0.948656f, 0.250987f, 
		-0.0815470f, 0.951668f, -0.301954f, -0.0814190f, 0.951668f, 0.301886f, 
		-0.0795120f, 1.57086f, -0.0369620f, -0.0794990f, 1.57086f, 0.0254450f, 
		-0.0793831f, 1.58880f, -0.0510143f, -0.0793831f, 1.58880f, -0.0510143f, 
		-0.0793641f, 1.58880f, 0.0394964f, -0.0793641f, 1.58880f, 0.0394964f, 
		-0.0792110f, 0.948656f, -0.301038f, -0.0790840f, 0.948656f, 0.300969f, 
		-0.0789460f, 0.853677f, -0.385648f, -0.0789460f, 0.853677f, -0.385648f, 
		-0.0787830f, 0.853677f, 0.385580f, -0.0787830f, 0.853677f, 0.385580f, 
		-0.0786960f, 1.56525f, 0.000139000f, -0.0782760f, 1.58893f, -0.0681700f, 
		-0.0782760f, 1.58893f, -0.0681700f, -0.0782760f, 1.61893f, -0.0681700f, 
		-0.0782760f, 1.61893f, -0.0681700f, -0.0782760f, 1.61893f, -0.0681700f, 
		-0.0782490f, 1.58893f, 0.0566530f, -0.0782490f, 1.58893f, 0.0566530f, 
		-0.0782490f, 1.61893f, 0.0566530f, -0.0782490f, 1.61893f, 0.0566530f, 
		-0.0782490f, 1.61893f, 0.0566530f, -0.0775610f, 1.57721f, -0.0505320f, 
		-0.0775420f, 1.57721f, 0.0390150f, -0.0769350f, 1.56525f, 0.0118020f, 
		-0.0768610f, 0.879246f, -0.382590f, -0.0766990f, 0.879246f, 0.382521f, 
		-0.0766930f, 0.853634f, -0.382344f, -0.0766930f, 0.853634f, -0.382344f, 
		-0.0765310f, 0.853634f, 0.382274f, -0.0765310f, 0.853634f, 0.382274f, 
		-0.0758540f, 0.951668f, -0.226208f, -0.0757580f, 0.951668f, 0.226139f, 
		-0.0753522f, 1.56290f, -0.0111641f, -0.0746660f, 0.878415f, -0.379373f, 
		-0.0745060f, 0.878415f, 0.379303f, -0.0743300f, 0.929791f, -0.189910f, 
		-0.0742490f, 0.929791f, 0.189840f, -0.0739760f, 1.57086f, -0.0484640f, 
		-0.0739580f, 1.57086f, 0.0369450f, -0.0736820f, 0.948656f, -0.227463f, 
		-0.0736017f, 1.56290f, -0.0228053f, -0.0735860f, 0.948656f, 0.227393f, 
		-0.0722050f, 0.927433f, -0.192201f, -0.0721240f, 0.927433f, 0.192130f, 
		-0.0711386f, 1.58880f, -0.0631125f, -0.0711386f, 1.58880f, -0.0631125f, 
		-0.0711145f, 1.58880f, 0.0515911f, -0.0711145f, 1.58880f, 0.0515911f, 
		-0.0709210f, 0.905134f, -0.373882f, -0.0707630f, 0.905134f, 0.373811f, 
		-0.0704150f, 1.56289f, -0.0331332f, -0.0704042f, 1.56290f, 0.0216134f, 
		-0.0700240f, 1.57721f, -0.0615920f, -0.0700000f, 1.57721f, 0.0500710f, 
		-0.0688960f, 0.903517f, -0.370913f, -0.0687390f, 0.903517f, 0.370841f, 
		-0.0685050f, 0.951668f, -0.324553f, -0.0683680f, 0.951668f, 0.324480f, 
		-0.0681020f, 1.58893f, -0.0791390f, -0.0681020f, 1.58893f, -0.0791390f, 
		-0.0681020f, 1.61893f, -0.0791390f, -0.0681020f, 1.61893f, -0.0791390f, 
		-0.0681020f, 1.61893f, -0.0791390f, -0.0680710f, 1.58893f, 0.0676180f, 
		-0.0680710f, 1.58893f, 0.0676180f, -0.0680710f, 1.61893f, 0.0676180f, 
		-0.0680710f, 1.61893f, 0.0676180f, -0.0680710f, 1.61893f, 0.0676180f, 
		-0.0667880f, 1.57086f, -0.0590130f, -0.0667650f, 1.57086f, 0.0474900f, 
		-0.0665430f, 0.948656f, -0.322989f, -0.0664070f, 0.948656f, 0.322916f, 
		-0.0659167f, 1.55863f, -0.00131681f, -0.0653110f, 1.56290f, -0.0437424f, 
		-0.0652939f, 1.56290f, 0.0322189f, -0.0642949f, 1.55863f, 0.00942054f, 
		-0.0625222f, 1.58880f, -0.0724025f, -0.0625222f, 1.58880f, -0.0724025f, 
		-0.0624937f, 1.58880f, 0.0608792f, -0.0624937f, 1.58880f, 0.0608792f, 
		-0.0616030f, 0.929791f, -0.360221f, -0.0614500f, 0.929791f, 0.360146f, 
		-0.0614240f, 0.969213f, -0.274577f, -0.0613080f, 0.969213f, 0.274501f, 
		-0.0609220f, 1.57721f, -0.0714050f, -0.0608950f, 1.57721f, 0.0598800f, 
		-0.0607740f, 0.853677f, -0.143843f, -0.0607740f, 0.853677f, -0.143843f, 
		-0.0607130f, 0.853677f, 0.143767f, -0.0607130f, 0.853677f, 0.143767f, 
		-0.0600490f, 0.969213f, -0.256285f, -0.0599410f, 0.969213f, 0.256209f, 
		-0.0598430f, 0.927433f, -0.357641f, -0.0596910f, 0.927433f, 0.357564f, 
		-0.0596560f, 0.965681f, -0.274445f, -0.0595810f, 0.951668f, -0.205812f, 
		-0.0595400f, 0.965681f, 0.274368f, -0.0594940f, 0.951668f, 0.205735f, 
		-0.0592236f, 1.56290f, -0.0526741f, -0.0592033f, 1.56290f, 0.0411480f, 
		-0.0591690f, 0.879246f, -0.147178f, -0.0591070f, 0.879246f, 0.147101f, 
		-0.0590390f, 0.853634f, -0.147447f, -0.0590390f, 0.853634f, -0.147447f, 
		-0.0590016f, 1.55685f, -0.0180731f, -0.0589770f, 0.853634f, 0.147370f, 
		-0.0589770f, 0.853634f, 0.147370f, -0.0583210f, 0.965681f, -0.256680f, 
		-0.0582120f, 0.965681f, 0.256603f, -0.0581070f, 1.57086f, -0.0683720f, 
		-0.0580810f, 1.57086f, 0.0568460f, -0.0578760f, 0.948656f, -0.207651f, 
		-0.0577880f, 0.948656f, 0.207574f, -0.0574800f, 0.878415f, -0.150687f, 
		-0.0574160f, 0.878415f, 0.150609f, -0.0573460f, 0.969213f, -0.292461f, 
		-0.0572220f, 0.969213f, 0.292384f, -0.0564070f, 1.58893f, -0.0884700f, 
		-0.0564070f, 1.58893f, -0.0884700f, -0.0564070f, 1.61893f, -0.0884700f, 
		-0.0564070f, 1.61893f, -0.0884700f, -0.0564070f, 1.61893f, -0.0884700f, 
		-0.0563720f, 1.58893f, 0.0769430f, -0.0563720f, 1.58893f, 0.0769430f, 
		-0.0563720f, 1.61893f, 0.0769430f, -0.0563720f, 1.61893f, 0.0769430f, 
		-0.0563720f, 1.61893f, 0.0769430f, -0.0556960f, 0.965681f, -0.291814f, 
		-0.0556874f, 1.55685f, -0.0288261f, -0.0555720f, 0.965681f, 0.291736f, 
		-0.0548470f, 1.23705f, -0.00410000f, -0.0548470f, 1.23705f, -0.00410000f, 
		-0.0548470f, 1.26705f, -0.00410000f, -0.0548470f, 1.26705f, -0.00410000f, 
		-0.0548450f, 1.23705f, 0.00412000f, -0.0548450f, 1.23705f, 0.00412000f, 
		-0.0548450f, 1.26705f, 0.00412000f, -0.0548450f, 1.26705f, 0.00412000f, 
		-0.0545980f, 0.905134f, -0.156676f, -0.0545310f, 0.905134f, 0.156597f, 
		-0.0536240f, 1.23705f, -0.0122290f, -0.0536240f, 1.23705f, -0.0122290f, 
		-0.0536240f, 1.26705f, -0.0122290f, -0.0536240f, 1.26705f, -0.0122290f, 
		-0.0536180f, 1.23705f, 0.0122480f, -0.0536180f, 1.23705f, 0.0122480f, 
		-0.0536180f, 1.26705f, 0.0122480f, -0.0536180f, 1.26705f, 0.0122480f, 
		-0.0533440f, 0.969213f, -0.239211f, -0.0532430f, 0.969213f, 0.239132f, 
		-0.0530390f, 0.903517f, -0.159914f, -0.0529710f, 0.903517f, 0.159835f, 
		-0.0527425f, 1.55685f, 0.0233890f, -0.0518090f, 0.965681f, -0.240098f, 
		-0.0517070f, 0.965681f, 0.240018f, -0.0512187f, 1.56290f, -0.0613061f, 
		-0.0512020f, 1.23705f, -0.0200850f, -0.0512020f, 1.23705f, -0.0200850f, 
		-0.0512020f, 1.26705f, -0.0200850f, -0.0512020f, 1.26705f, -0.0200850f, 
		-0.0511943f, 1.56290f, 0.0497771f, -0.0511940f, 1.23705f, 0.0201030f, 
		-0.0511940f, 1.23705f, 0.0201030f, -0.0511940f, 1.26705f, 0.0201030f, 
		-0.0511940f, 1.26705f, 0.0201030f, -0.0510772f, 1.58880f, -0.0815330f, 
		-0.0510772f, 1.58880f, -0.0815330f, -0.0510447f, 1.58880f, 0.0700040f, 
		-0.0510447f, 1.58880f, 0.0700040f, -0.0504600f, 1.57721f, -0.0797520f, 
		-0.0504290f, 1.57721f, 0.0682230f, -0.0500978f, 1.26712f, -0.00345483f, 
		-0.0500978f, 1.26712f, -0.00345483f, -0.0500961f, 1.26712f, 0.00347119f, 
		-0.0500961f, 1.26712f, 0.00347119f, -0.0500546f, 1.23699f, -0.00403412f, 
		-0.0500546f, 1.23699f, -0.00403412f, -0.0500524f, 1.23699f, 0.00405074f, 
		-0.0500524f, 1.23699f, 0.00405074f, -0.0493820f, 0.951668f, -0.342304f, 
		-0.0492370f, 0.951668f, 0.342224f, -0.0490650f, 1.22533f, -0.00366800f, 
		-0.0490650f, 1.27877f, -0.00366800f, -0.0490630f, 1.22533f, 0.00368500f, 
		-0.0490630f, 1.27877f, 0.00368500f, -0.0490240f, 1.23699f, -0.0108834f, 
		-0.0490240f, 1.23699f, -0.0108834f, -0.0490190f, 1.23699f, 0.0108989f, 
		-0.0490190f, 1.23699f, 0.0108989f, -0.0488951f, 1.26712f, -0.0114492f, 
		-0.0488951f, 1.26712f, -0.0114492f, -0.0488897f, 1.26712f, 0.0114648f, 
		-0.0488897f, 1.26712f, 0.0114648f, -0.0481780f, 0.969213f, -0.308349f, 
		-0.0481280f, 1.57086f, -0.0763330f, -0.0480990f, 1.57086f, 0.0648030f, 
		-0.0480470f, 0.969213f, 0.308268f, -0.0479680f, 0.948656f, -0.340232f, 
		-0.0478240f, 0.948656f, 0.340150f, -0.0478135f, 1.55466f, 0.00713167f, 
		-0.0476370f, 1.23705f, -0.0274920f, -0.0476370f, 1.23705f, -0.0274920f, 
		-0.0476370f, 1.26705f, -0.0274920f, -0.0476370f, 1.26705f, -0.0274920f, 
		-0.0476260f, 1.23705f, 0.0275080f, -0.0476260f, 1.23705f, 0.0275080f, 
		-0.0476260f, 1.26705f, 0.0275080f, -0.0476260f, 1.26705f, 0.0275080f, 
		-0.0474260f, 0.929791f, -0.171576f, -0.0473530f, 0.929791f, 0.171494f, 
		-0.0472687f, 1.28192f, -0.0107439f, -0.0472653f, 1.28192f, 0.0107604f, 
		-0.0472531f, 1.22218f, -0.0108130f, -0.0472497f, 1.22218f, 0.0108297f, 
		-0.0468550f, 1.26712f, -0.0180676f, -0.0468550f, 1.26712f, -0.0180676f, 
		-0.0468470f, 1.26712f, 0.0180838f, -0.0468470f, 1.26712f, 0.0180838f, 
		-0.0467910f, 0.965681f, -0.307244f, -0.0467698f, 1.55466f, 0.0105124f, 
		-0.0466610f, 0.965681f, 0.307162f, -0.0466432f, 1.23699f, -0.0186077f, 
		-0.0466432f, 1.23699f, -0.0186077f, -0.0466348f, 1.23699f, 0.0186241f, 
		-0.0466348f, 1.23699f, 0.0186241f, -0.0464037f, 1.55685f, 0.0326838f, 
		-0.0460710f, 0.927433f, -0.174391f, -0.0459970f, 0.927433f, 0.174309f, 
		-0.0454121f, 1.21611f, -0.00332261f, -0.0454113f, 1.21611f, 0.00333835f, 
		-0.0454015f, 1.28799f, -0.00346780f, -0.0454004f, 1.28799f, 0.00348287f, 
		-0.0451403f, 1.22218f, -0.0176697f, -0.0451334f, 1.22218f, 0.0176847f, 
		-0.0451144f, 1.28192f, -0.0177352f, -0.0451075f, 1.28192f, 0.0177503f, 
		-0.0437599f, 1.00044f, -0.273532f, -0.0437599f, 1.00044f, -0.273532f, 
		-0.0437586f, 1.00044f, -0.267998f, -0.0437586f, 1.00044f, -0.267998f, 
		-0.0436393f, 1.23699f, -0.0248492f, -0.0436393f, 1.23699f, -0.0248492f, 
		-0.0436279f, 1.23699f, 0.0248637f, -0.0436279f, 1.23699f, 0.0248637f, 
		-0.0434520f, 1.58893f, -0.0959530f, -0.0434520f, 1.58893f, -0.0959530f, 
		-0.0434520f, 1.61893f, -0.0959530f, -0.0434520f, 1.61893f, -0.0959530f, 
		-0.0434520f, 1.61893f, -0.0959530f, -0.0434140f, 1.58893f, 0.0844210f, 
		-0.0434140f, 1.58893f, 0.0844210f, -0.0434140f, 1.61893f, 0.0844210f, 
		-0.0434140f, 1.61893f, 0.0844210f, -0.0434140f, 1.61893f, 0.0844210f, 
		-0.0433491f, 1.26712f, -0.0253521f, -0.0433491f, 1.26712f, -0.0253521f, 
		-0.0433375f, 1.26712f, 0.0253667f, -0.0433375f, 1.26712f, 0.0253667f, 
		-0.0431251f, 1.00045f, 0.267160f, -0.0431251f, 1.00045f, 0.267160f, 
		-0.0431237f, 1.00045f, 0.274201f, -0.0431237f, 1.00045f, 0.274201f, 
		-0.0430450f, 1.55470f, -0.0290642f, -0.0430080f, 1.23705f, -0.0342850f, 
		-0.0430080f, 1.23705f, -0.0342850f, -0.0430080f, 1.26705f, -0.0342850f, 
		-0.0430080f, 1.26705f, -0.0342850f, -0.0429930f, 1.23705f, 0.0342990f, 
		-0.0429930f, 1.23705f, 0.0342990f, -0.0429930f, 1.26705f, 0.0342990f, 
		-0.0429930f, 1.26705f, 0.0342990f, -0.0427686f, 1.56290f, -0.0680483f, 
		-0.0427428f, 1.56290f, 0.0565136f, -0.0422457f, 1.00045f, 0.261317f, 
		-0.0422457f, 1.00045f, 0.261317f, -0.0422416f, 1.00045f, 0.280045f, 
		-0.0422416f, 1.00045f, 0.280045f, -0.0420026f, 1.28192f, -0.0241991f, 
		-0.0419933f, 1.28192f, 0.0242133f, -0.0419671f, 1.22218f, -0.0242616f, 
		-0.0419574f, 1.22218f, 0.0242753f, -0.0419040f, 0.969213f, -0.224872f, 
		-0.0418942f, 1.55466f, -0.0321857f, -0.0418490f, 1.04604f, 7.00000e-006f, 
		-0.0418490f, 1.04604f, 7.00000e-006f, -0.0418213f, 1.55685f, -0.0491716f, 
		-0.0418090f, 0.969213f, 0.224788f, -0.0415320f, 1.04604f, -0.00411300f, 
		-0.0415320f, 1.04604f, -0.00411300f, -0.0415310f, 1.04604f, 0.00412600f, 
		-0.0415310f, 1.04604f, 0.00412600f, -0.0413510f, 0.853677f, -0.403763f, 
		-0.0413510f, 0.853677f, -0.403763f, -0.0411800f, 0.853677f, 0.403678f, 
		-0.0411800f, 0.853677f, 0.403678f, -0.0406990f, 0.965681f, -0.226172f, 
		-0.0406300f, 1.04604f, -0.00754800f, -0.0406300f, 1.04604f, -0.00754800f, 
		-0.0406260f, 1.04604f, 0.00756100f, -0.0406260f, 1.04604f, 0.00756100f, 
		-0.0406030f, 0.965681f, 0.226087f, -0.0403671f, 1.29323f, -0.00906624f, 
		-0.0403599f, 1.29323f, 0.00907699f, -0.0403062f, 1.21088f, -0.00933351f, 
		-0.0402981f, 1.21087f, 0.00934741f, -0.0402860f, 1.00045f, -0.286857f, 
		-0.0402860f, 1.00045f, -0.286857f, -0.0402798f, 1.00045f, -0.254673f, 
		-0.0402798f, 1.00045f, -0.254673f, -0.0402590f, 0.879246f, -0.400226f, 
		-0.0401718f, 1.00045f, 0.254588f, -0.0401718f, 1.00045f, 0.254588f, 
		-0.0401710f, 0.853634f, -0.399941f, -0.0401710f, 0.853634f, -0.399941f, 
		-0.0401651f, 1.00045f, 0.286772f, -0.0401651f, 1.00045f, 0.286772f, 
		-0.0401061f, 1.58880f, -0.0878705f, -0.0401061f, 1.58880f, -0.0878705f, 
		-0.0400900f, 0.879246f, 0.400141f, -0.0400712f, 1.58880f, 0.0763367f, 
		-0.0400712f, 1.58880f, 0.0763367f, -0.0400570f, 0.868091f, -0.268071f, 
		-0.0400570f, 0.868091f, -0.268071f, -0.0400216f, 0.993265f, -0.279901f, 
		-0.0400020f, 0.853634f, 0.399856f, -0.0400020f, 0.853634f, 0.399856f, 
		-0.0399430f, 0.868091f, 0.267986f, -0.0399430f, 0.868091f, 0.267986f, 
		-0.0399023f, 0.993264f, 0.279817f, -0.0397780f, 0.872834f, -0.268071f, 
		-0.0397780f, 0.872834f, -0.268071f, -0.0396640f, 0.872834f, 0.267986f, 
		-0.0396640f, 0.872834f, 0.267986f, -0.0394479f, 1.26712f, -0.0310754f, 
		-0.0394479f, 1.26712f, -0.0310754f, -0.0394348f, 1.26712f, 0.0310887f, 
		-0.0394348f, 1.26712f, 0.0310887f, -0.0392130f, 1.04604f, -0.0103180f, 
		-0.0392130f, 1.04604f, -0.0103180f, -0.0392080f, 1.04604f, 0.0103300f, 
		-0.0392080f, 1.04604f, 0.0103300f, -0.0391110f, 0.878415f, -0.396505f, 
		-0.0390856f, 1.23699f, -0.0315299f, -0.0390856f, 1.23699f, -0.0315299f, 
		-0.0390728f, 1.23699f, 0.0315424f, -0.0390728f, 1.23699f, 0.0315424f, 
		-0.0389990f, 0.858925f, -0.268072f, -0.0389990f, 0.858925f, -0.268072f, 
		-0.0389610f, 0.877414f, -0.268072f, -0.0389610f, 0.877414f, -0.268072f, 
		-0.0389430f, 0.878415f, 0.396420f, -0.0389324f, 0.989733f, -0.268054f, 
		-0.0389318f, 0.989731f, -0.273474f, -0.0388860f, 0.858925f, 0.267986f, 
		-0.0388860f, 0.858925f, 0.267986f, -0.0388710f, 1.57721f, -0.0864460f, 
		-0.0388480f, 0.877414f, 0.267986f, -0.0388480f, 0.877414f, 0.267986f, 
		-0.0388370f, 1.57721f, 0.0749120f, -0.0388180f, 0.989731f, 0.267968f, 
		-0.0388146f, 0.989729f, 0.273390f, -0.0385649f, 1.21088f, -0.0149815f, 
		-0.0385582f, 1.21088f, 0.0149926f, -0.0384649f, 1.29323f, -0.0152360f, 
		-0.0384575f, 1.29323f, 0.0152492f, -0.0382708f, 0.992701f, -0.255581f, 
		-0.0381025f, 0.993261f, 0.285640f, -0.0380200f, 0.951668f, -0.191118f, 
		-0.0379390f, 0.951668f, 0.191032f, -0.0379275f, 1.22218f, -0.0301887f, 
		-0.0379148f, 1.22218f, 0.0302022f, -0.0378903f, 0.989730f, 0.261810f, 
		-0.0378827f, 1.28192f, -0.0302448f, -0.0378704f, 1.28192f, 0.0302577f, 
		-0.0377228f, 1.00045f, -0.292183f, -0.0377228f, 1.00045f, -0.292183f, 
		-0.0377140f, 1.00045f, -0.249348f, -0.0377140f, 1.00045f, -0.249348f, 
		-0.0376380f, 0.881798f, -0.268072f, -0.0376380f, 0.881798f, -0.268072f, 
		-0.0376091f, 1.00045f, 0.249264f, -0.0376091f, 1.00045f, 0.249264f, 
		-0.0375996f, 1.00045f, 0.292095f, -0.0375996f, 1.00045f, 0.292095f, 
		-0.0375250f, 0.881798f, 0.267986f, -0.0375250f, 0.881798f, 0.267986f, 
		-0.0374180f, 1.23705f, -0.0403120f, -0.0374180f, 1.23705f, -0.0403120f, 
		-0.0374180f, 1.26705f, -0.0403120f, -0.0374180f, 1.26705f, -0.0403120f, 
		-0.0374010f, 1.23705f, 0.0403240f, -0.0374010f, 1.23705f, 0.0403240f, 
		-0.0374010f, 1.26705f, 0.0403240f, -0.0374010f, 1.26705f, 0.0403240f, 
		-0.0373540f, 1.04604f, -0.0124400f, -0.0373540f, 1.04604f, -0.0124400f, 
		-0.0373490f, 1.04604f, 0.0124520f, -0.0373490f, 1.04604f, 0.0124520f, 
		-0.0371500f, 0.905134f, -0.390154f, -0.0370750f, 1.57086f, -0.0827180f, 
		-0.0370420f, 1.57086f, 0.0711830f, -0.0369850f, 0.905134f, 0.390068f, 
		-0.0369320f, 0.948656f, -0.193379f, -0.0368500f, 0.948656f, 0.193292f, 
		-0.0364806f, 1.55393f, -0.0225838f, -0.0364806f, 1.55393f, -0.0225838f, 
		-0.0364084f, 0.989730f, -0.284814f, -0.0364036f, 0.989732f, -0.256717f, 
		-0.0362944f, 0.989731f, 0.256631f, -0.0362420f, 1.02511f, 6.00000e-006f, 
		-0.0362420f, 1.02511f, 6.00000e-006f, -0.0362420f, 1.06696f, 6.00000e-006f, 
		-0.0362420f, 1.06696f, 6.00000e-006f, -0.0360952f, 1.20667f, -0.00241683f, 
		-0.0360912f, 1.20667f, 0.00242711f, -0.0360900f, 0.903517f, -0.386720f, 
		-0.0360533f, 1.29743f, -0.00297875f, -0.0360493f, 1.29743f, 0.00299085f, 
		-0.0359880f, 0.850508f, -0.268072f, -0.0359880f, 0.850508f, -0.268072f, 
		-0.0359680f, 1.02527f, -0.00411400f, -0.0359680f, 1.02527f, -0.00411400f, 
		-0.0359680f, 1.06680f, -0.00411400f, -0.0359680f, 1.06680f, -0.00411400f, 
		-0.0359670f, 1.02527f, 0.00412500f, -0.0359670f, 1.02527f, 0.00412500f, 
		-0.0359670f, 1.06680f, 0.00412500f, -0.0359670f, 1.06680f, 0.00412500f, 
		-0.0359270f, 0.903517f, 0.386634f, -0.0359009f, 1.29323f, -0.0205587f, 
		-0.0358913f, 1.29323f, 0.0205703f, -0.0358740f, 0.850508f, 0.267985f, 
		-0.0358740f, 0.850508f, 0.267985f, -0.0358410f, 0.885952f, -0.268072f, 
		-0.0358410f, 0.885952f, -0.268072f, -0.0357620f, 1.21087f, -0.0207985f, 
		-0.0357536f, 1.21087f, 0.0208094f, -0.0357280f, 0.885952f, 0.267985f, 
		-0.0357280f, 0.885952f, 0.267985f, -0.0351870f, 1.02572f, -0.00754900f, 
		-0.0351870f, 1.02572f, -0.00754900f, -0.0351870f, 1.06635f, -0.00754900f, 
		-0.0351870f, 1.06635f, -0.00754900f, -0.0351830f, 1.02572f, 0.00756000f, 
		-0.0351830f, 1.02572f, 0.00756000f, -0.0351830f, 1.06635f, 0.00756000f, 
		-0.0351830f, 1.06635f, 0.00756000f, -0.0351260f, 1.04604f, -0.0139330f, 
		-0.0351260f, 1.04604f, -0.0139330f, -0.0351210f, 1.04604f, 0.0139440f, 
		-0.0351210f, 1.04604f, 0.0139440f, -0.0347330f, 0.969213f, -0.320829f, 
		-0.0347020f, 0.868091f, -0.288072f, -0.0347020f, 0.868091f, -0.288072f, 
		-0.0346940f, 0.868091f, -0.248072f, -0.0346940f, 0.868091f, -0.248072f, 
		-0.0345980f, 0.969213f, 0.320742f, -0.0345890f, 0.868091f, 0.247985f, 
		-0.0345890f, 0.868091f, 0.247985f, -0.0345800f, 0.868091f, 0.287985f, 
		-0.0345800f, 0.868091f, 0.287985f, -0.0344600f, 0.872834f, -0.287933f, 
		-0.0344600f, 0.872834f, -0.287933f, -0.0344520f, 0.872834f, -0.248212f, 
		-0.0344520f, 0.872834f, -0.248212f, -0.0343759f, 1.23699f, -0.0366074f, 
		-0.0343759f, 1.23699f, -0.0366074f, -0.0343603f, 1.23699f, 0.0366193f, 
		-0.0343603f, 1.23699f, 0.0366193f, -0.0343470f, 0.872834f, 0.248125f, 
		-0.0343470f, 0.872834f, 0.248125f, -0.0343380f, 0.872834f, 0.287846f, 
		-0.0343380f, 0.872834f, 0.287846f, -0.0342270f, 1.56525f, 0.0653330f, 
		-0.0340332f, 0.986226f, -0.262568f, -0.0339600f, 1.02643f, -0.0103190f, 
		-0.0339600f, 1.02643f, -0.0103190f, -0.0339600f, 1.06564f, -0.0103190f, 
		-0.0339600f, 1.06564f, -0.0103190f, -0.0339550f, 1.02643f, 0.0103290f, 
		-0.0339550f, 1.02643f, 0.0103290f, -0.0339550f, 1.06564f, 0.0103290f, 
		-0.0339550f, 1.06564f, 0.0103290f, -0.0339499f, 1.26712f, -0.0370029f, 
		-0.0339499f, 1.26712f, -0.0370029f, -0.0339350f, 1.26712f, 0.0370138f, 
		-0.0339350f, 1.26712f, 0.0370138f, -0.0337860f, 0.858925f, -0.287544f, 
		-0.0337860f, 0.858925f, -0.287544f, -0.0337780f, 0.858925f, -0.248601f, 
		-0.0337780f, 0.858925f, -0.248601f, -0.0337530f, 0.877414f, -0.287525f, 
		-0.0337530f, 0.877414f, -0.287525f, -0.0337450f, 0.877414f, -0.248620f, 
		-0.0337450f, 0.877414f, -0.248620f, -0.0337350f, 0.965681f, -0.319364f, 
		-0.0337073f, 0.989731f, -0.290425f, -0.0336730f, 0.858925f, 0.248514f, 
		-0.0336730f, 0.858925f, 0.248514f, -0.0336640f, 0.858925f, 0.287456f, 
		-0.0336640f, 0.858925f, 0.287456f, -0.0336619f, 1.03132f, -0.272398f, 
		-0.0336619f, 1.03132f, -0.272398f, -0.0336619f, 1.03132f, -0.272398f, 
		-0.0336439f, 1.00045f, 0.243445f, -0.0336439f, 1.00045f, 0.243445f, 
		-0.0336390f, 0.877414f, 0.248533f, -0.0336390f, 0.877414f, 0.248533f, 
		-0.0336316f, 1.00045f, 0.297913f, -0.0336316f, 1.00045f, 0.297913f, 
		-0.0336310f, 0.877414f, 0.287437f, -0.0336310f, 0.877414f, 0.287437f, 
		-0.0336010f, 0.889845f, -0.268073f, -0.0336010f, 0.889845f, -0.268073f, 
		-0.0335990f, 0.965681f, 0.319277f, -0.0335939f, 0.989731f, 0.251020f, 
		-0.0335856f, 0.989731f, 0.290337f, -0.0335462f, 1.03132f, 0.272311f, 
		-0.0335462f, 1.03132f, 0.272311f, -0.0335462f, 1.03132f, 0.272311f, 
		-0.0335302f, 1.02272f, -0.274160f, -0.0335302f, 1.02272f, -0.274160f, 
		-0.0334880f, 0.889845f, 0.267985f, -0.0334880f, 0.889845f, 0.267985f, 
		-0.0334730f, 1.22533f, -0.0360620f, -0.0334730f, 1.27877f, -0.0360620f, 
		-0.0334580f, 1.22533f, 0.0360720f, -0.0334580f, 1.27877f, 0.0360720f, 
		-0.0334138f, 1.02272f, 0.274073f, -0.0334138f, 1.02272f, 0.274073f, 
		-0.0331941f, 0.985827f, -0.280548f, -0.0331640f, 1.02272f, 0.273168f, 
		-0.0331640f, 1.02272f, 0.273168f, -0.0331640f, 1.03132f, 0.273168f, 
		-0.0331640f, 1.03132f, 0.273168f, -0.0331640f, 1.03132f, 0.273168f, 
		-0.0330770f, 0.985827f, 0.280460f, -0.0330441f, 1.02272f, -0.277389f, 
		-0.0330441f, 1.02272f, -0.277389f, -0.0330250f, 1.55685f, -0.0561897f, 
		-0.0328297f, 0.985827f, 0.281263f, -0.0326512f, 1.03132f, -0.279112f, 
		-0.0326512f, 1.03132f, -0.279112f, -0.0326512f, 1.03132f, -0.279112f, 
		-0.0326512f, 1.03132f, -0.279112f, -0.0326070f, 0.881798f, -0.286864f, 
		-0.0326070f, 0.881798f, -0.286864f, -0.0326020f, 1.04604f, -0.0148160f, 
		-0.0326020f, 1.04604f, -0.0148160f, -0.0325990f, 0.881798f, -0.249282f, 
		-0.0325990f, 0.881798f, -0.249282f, -0.0325950f, 1.04604f, 0.0148260f, 
		-0.0325950f, 1.04604f, 0.0148260f, -0.0325733f, 1.56289f, -0.0739345f, 
		-0.0325360f, 1.02272f, -0.263353f, -0.0325360f, 1.02272f, -0.263353f, 
		-0.0325360f, 1.03132f, -0.263353f, -0.0325360f, 1.03132f, -0.263353f, 
		-0.0324940f, 0.881798f, 0.249194f, -0.0324940f, 0.881798f, 0.249194f, 
		-0.0324860f, 0.881798f, 0.286776f, -0.0324860f, 0.881798f, 0.286776f, 
		-0.0324357f, 1.21087f, -0.0256804f, -0.0324254f, 1.21087f, 0.0256907f, 
		-0.0323500f, 1.02736f, -0.0124410f, -0.0323500f, 1.02736f, -0.0124410f, 
		-0.0323500f, 1.06471f, -0.0124410f, -0.0323500f, 1.06471f, -0.0124410f, 
		-0.0323450f, 1.02736f, 0.0124510f, -0.0323450f, 1.02736f, 0.0124510f, 
		-0.0323450f, 1.06471f, 0.0124510f, -0.0323450f, 1.06471f, 0.0124510f, 
		-0.0322730f, 0.929791f, -0.374354f, -0.0322624f, 1.29323f, -0.0258967f, 
		-0.0322531f, 1.29323f, 0.0259061f, -0.0321537f, 0.980923f, 0.268780f, 
		-0.0321140f, 0.929791f, 0.374265f, -0.0320470f, 0.980877f, -0.272382f, 
		-0.0315750f, 1.03132f, 0.259214f, -0.0315750f, 1.03132f, 0.259214f, 
		-0.0315750f, 1.03132f, 0.259214f, -0.0315750f, 1.03132f, 0.259214f, 
		-0.0313510f, 0.927433f, -0.371369f, -0.0312630f, 0.843081f, -0.268073f, 
		-0.0312630f, 0.843081f, -0.268073f, -0.0311940f, 0.927433f, 0.371280f, 
		-0.0311780f, 0.850508f, -0.286039f, -0.0311780f, 0.850508f, -0.286039f, 
		-0.0311700f, 0.850508f, -0.250108f, -0.0311700f, 0.850508f, -0.250108f, 
		-0.0311490f, 0.843081f, 0.267985f, -0.0311490f, 0.843081f, 0.267985f, 
		-0.0310640f, 0.850508f, 0.250019f, -0.0310640f, 0.850508f, 0.250019f, 
		-0.0310570f, 0.850508f, 0.285950f, -0.0310570f, 0.850508f, 0.285950f, 
		-0.0310510f, 0.885952f, -0.285966f, -0.0310510f, 0.885952f, -0.285966f, 
		-0.0310430f, 0.885952f, -0.250181f, -0.0310430f, 0.885952f, -0.250181f, 
		-0.0310312f, 1.28799f, -0.0333264f, -0.0310210f, 0.977046f, 0.266306f, 
		-0.0310176f, 1.28799f, 0.0333341f, -0.0309920f, 1.23705f, -0.0454380f, 
		-0.0309920f, 1.23705f, -0.0454380f, -0.0309920f, 1.26705f, -0.0454380f, 
		-0.0309920f, 1.26705f, -0.0454380f, -0.0309730f, 1.23705f, 0.0454480f, 
		-0.0309730f, 1.23705f, 0.0454480f, -0.0309730f, 1.26705f, 0.0454480f, 
		-0.0309730f, 1.26705f, 0.0454480f, -0.0309490f, 0.893445f, -0.268073f, 
		-0.0309490f, 0.893445f, -0.268073f, -0.0309370f, 0.885952f, 0.250092f, 
		-0.0309370f, 0.885952f, 0.250092f, -0.0309300f, 0.885952f, 0.285877f, 
		-0.0309300f, 0.885952f, 0.285877f, -0.0309297f, 1.02272f, 0.257568f, 
		-0.0309297f, 1.02272f, 0.257568f, -0.0309258f, 1.21611f, -0.0334241f, 
		-0.0309110f, 1.21611f, 0.0334329f, -0.0308360f, 0.893445f, 0.267984f, 
		-0.0308360f, 0.893445f, 0.267984f, -0.0308309f, 0.984607f, -0.276027f, 
		-0.0308133f, 0.977046f, -0.275715f, -0.0307621f, 0.984606f, -0.265670f, 
		-0.0307621f, 0.984606f, -0.265670f, -0.0306552f, 0.989731f, -0.294904f, 
		-0.0306490f, 0.984606f, 0.265581f, -0.0306490f, 0.984606f, 0.265581f, 
		-0.0306450f, 0.989731f, -0.246629f, -0.0306073f, 1.20497f, 0.0126558f, 
		-0.0305404f, 0.989731f, 0.246540f, -0.0305291f, 0.989730f, 0.294816f, 
		-0.0305058f, 0.980923f, -0.259425f, -0.0304210f, 1.02848f, -0.0139340f, 
		-0.0304210f, 1.02848f, -0.0139340f, -0.0304210f, 1.06360f, -0.0139340f, 
		-0.0304210f, 1.06360f, -0.0139340f, -0.0304150f, 1.02848f, 0.0139430f, 
		-0.0304150f, 1.02848f, 0.0139430f, -0.0304150f, 1.06360f, 0.0139430f, 
		-0.0304150f, 1.06360f, 0.0139430f, -0.0304130f, 0.977001f, -0.263056f, 
		-0.0298939f, 1.55393f, 0.0210520f, -0.0298939f, 1.55393f, 0.0210520f, 
		-0.0298520f, 1.04604f, -0.0151060f, -0.0298520f, 1.04604f, -0.0151060f, 
		-0.0298460f, 1.04604f, 0.0151150f, -0.0298460f, 1.04604f, 0.0151150f, 
		-0.0297395f, 1.00045f, -0.302333f, -0.0297395f, 1.00045f, -0.302333f, 
		-0.0297260f, 1.00045f, -0.239201f, -0.0297260f, 1.00045f, -0.239201f, 
		-0.0296249f, 1.00045f, 0.239112f, -0.0296249f, 1.00045f, 0.239112f, 
		-0.0296218f, 1.02272f, -0.254716f, -0.0296218f, 1.02272f, -0.254716f, 
		-0.0296114f, 1.00045f, 0.302244f, -0.0296114f, 1.00045f, 0.302244f, 
		-0.0295260f, 1.58893f, -0.101422f, -0.0295260f, 1.58893f, -0.101422f, 
		-0.0295260f, 1.61893f, -0.101422f, -0.0295260f, 1.61893f, -0.101422f, 
		-0.0295260f, 1.61893f, -0.101422f, -0.0295135f, 1.02272f, 0.254626f, 
		-0.0295135f, 1.02272f, 0.254626f, -0.0295068f, 1.02272f, 0.286728f, 
		-0.0295068f, 1.02272f, 0.286728f, -0.0294860f, 1.58893f, 0.0898840f, 
		-0.0294860f, 1.58893f, 0.0898840f, -0.0294860f, 1.61893f, 0.0898840f, 
		-0.0294860f, 1.61893f, 0.0898840f, -0.0294860f, 1.61893f, 0.0898840f, 
		-0.0294759f, 1.00044f, -0.303209f, -0.0294759f, 1.00044f, -0.303209f, 
		-0.0294627f, 1.00044f, -0.238325f, -0.0294627f, 1.00044f, -0.238325f, 
		-0.0291110f, 0.889845f, -0.284846f, -0.0291110f, 0.889845f, -0.284846f, 
		-0.0291030f, 0.889845f, -0.251301f, -0.0291030f, 0.889845f, -0.251301f, 
		-0.0290129f, 1.30063f, -0.00854231f, -0.0290090f, 1.30063f, 0.00855094f, 
		-0.0289970f, 0.889845f, 0.251212f, -0.0289970f, 0.889845f, 0.251212f, 
		-0.0289900f, 0.889845f, 0.284756f, -0.0289900f, 0.889845f, 0.284756f, 
		-0.0289801f, 1.20497f, -0.0160378f, -0.0289741f, 1.20497f, 0.0160445f, 
		-0.0289260f, 0.977001f, 0.281271f, -0.0289130f, 1.02272f, -0.287425f, 
		-0.0289130f, 1.02272f, -0.287425f, -0.0289130f, 1.03132f, -0.287425f, 
		-0.0289130f, 1.03132f, -0.287425f, -0.0289130f, 1.03132f, -0.287425f, 
		-0.0288333f, 0.985827f, -0.251653f, -0.0287379f, 1.03132f, -0.253186f, 
		-0.0287379f, 1.03132f, -0.253186f, -0.0287379f, 1.03132f, -0.253186f, 
		-0.0286303f, 1.03132f, 0.253095f, -0.0286303f, 1.03132f, 0.253095f, 
		-0.0286303f, 1.03132f, 0.253095f, -0.0286229f, 1.03132f, 0.288258f, 
		-0.0286229f, 1.03132f, 0.288258f, -0.0286229f, 1.03132f, 0.288258f, 
		-0.0285364f, 1.26712f, -0.0413229f, -0.0285364f, 1.26712f, -0.0413229f, 
		-0.0285184f, 1.26712f, 0.0413309f, -0.0285184f, 1.26712f, 0.0413309f, 
		-0.0283859f, 1.29913f, -0.0170686f, -0.0283796f, 1.29913f, 0.0170755f, 
		-0.0282340f, 1.02974f, -0.0148170f, -0.0282340f, 1.02974f, -0.0148170f, 
		-0.0282340f, 1.06234f, -0.0148170f, -0.0282340f, 1.06234f, -0.0148170f, 
		-0.0282280f, 1.02974f, 0.0148250f, -0.0282280f, 1.02974f, 0.0148250f, 
		-0.0282280f, 1.06234f, 0.0148250f, -0.0282280f, 1.06234f, 0.0148250f, 
		-0.0282075f, 1.55863f, 0.0539670f, -0.0280572f, 1.23699f, -0.0416497f, 
		-0.0280572f, 1.23699f, -0.0416497f, -0.0280383f, 1.23699f, 0.0416581f, 
		-0.0280383f, 1.23699f, 0.0416581f, -0.0278030f, 0.896718f, 0.267984f, 
		-0.0278030f, 0.896718f, 0.267984f, -0.0277080f, 1.22533f, 0.0406560f, 
		-0.0277080f, 1.27877f, 0.0406560f, -0.0273450f, 1.22218f, -0.0400263f, 
		-0.0273017f, 0.980923f, -0.287211f, -0.0272847f, 1.28192f, -0.0400674f, 
		-0.0271808f, 0.980923f, 0.287120f, -0.0271496f, 1.55685f, 0.0480318f, 
		-0.0271193f, 1.20269f, -0.00493203f, -0.0270850f, 0.843081f, -0.283677f, 
		-0.0270850f, 0.843081f, -0.283677f, -0.0270790f, 0.843081f, -0.252471f, 
		-0.0270790f, 0.843081f, -0.252471f, -0.0270731f, 0.897585f, -0.268074f, 
		-0.0270731f, 0.897585f, -0.268074f, -0.0269720f, 0.843081f, 0.252381f, 
		-0.0269720f, 0.843081f, 0.252381f, -0.0269650f, 0.843081f, 0.283587f, 
		-0.0269650f, 0.843081f, 0.283587f, -0.0269120f, 0.977001f, 0.254326f, 
		-0.0268983f, 1.03132f, -0.250490f, -0.0268983f, 1.03132f, -0.250490f, 
		-0.0268983f, 1.03132f, -0.250490f, -0.0268983f, 1.03132f, -0.250490f, 
		-0.0268415f, 1.20339f, -0.0133094f, -0.0268140f, 0.893445f, -0.283520f, 
		-0.0268140f, 0.893445f, -0.283520f, -0.0268070f, 0.893445f, -0.252628f, 
		-0.0268070f, 0.893445f, -0.252628f, -0.0267833f, 1.03132f, 0.290957f, 
		-0.0267833f, 1.03132f, 0.290957f, -0.0267833f, 1.03132f, 0.290957f, 
		-0.0267478f, 0.984599f, 0.253629f, -0.0267478f, 0.984599f, 0.253629f, 
		-0.0267460f, 0.969213f, -0.214542f, -0.0267000f, 0.893445f, 0.252537f, 
		-0.0267000f, 0.893445f, 0.252537f, -0.0266940f, 0.893445f, 0.283430f, 
		-0.0266940f, 0.893445f, 0.283430f, -0.0266550f, 0.969213f, 0.214451f, 
		-0.0264782f, 1.58880f, -0.0932227f, -0.0264782f, 1.58880f, -0.0932227f, 
		-0.0264414f, 1.55470f, 0.0354083f, -0.0264410f, 1.58880f, 0.0816836f, 
		-0.0264410f, 1.58880f, 0.0816836f, -0.0264193f, 0.989730f, -0.299471f, 
		-0.0264193f, 0.989730f, -0.299471f, -0.0264193f, 0.989730f, -0.299471f, 
		-0.0264140f, 1.57721f, -0.0913380f, -0.0264091f, 0.989732f, -0.242064f, 
		-0.0263770f, 1.57721f, 0.0797990f, -0.0263050f, 0.989730f, 0.241974f, 
		-0.0262921f, 0.989731f, 0.299381f, -0.0259770f, 0.965681f, -0.216139f, 
		-0.0259327f, 0.980923f, 0.250830f, -0.0258850f, 0.965681f, 0.216048f, 
		-0.0258760f, 0.951668f, -0.353631f, -0.0258530f, 1.03111f, -0.0151070f, 
		-0.0258530f, 1.03111f, -0.0151070f, -0.0258530f, 1.06096f, -0.0151070f, 
		-0.0258530f, 1.06096f, -0.0151070f, -0.0258470f, 1.03111f, 0.0151140f, 
		-0.0258470f, 1.03111f, 0.0151140f, -0.0258470f, 1.06096f, 0.0151140f, 
		-0.0258470f, 1.06096f, 0.0151140f, -0.0257963f, 1.02272f, -0.249108f, 
		-0.0257963f, 1.02272f, -0.249108f, -0.0257260f, 0.951668f, 0.353540f, 
		-0.0257021f, 1.21611f, 0.0375847f, -0.0256811f, 1.02272f, 0.292338f, 
		-0.0256811f, 1.02272f, 0.292338f, -0.0255823f, 1.28799f, 0.0376666f, 
		-0.0251930f, 1.57086f, -0.0873840f, -0.0251590f, 1.57086f, 0.0758440f, 
		-0.0251543f, 0.899571f, 0.267983f, -0.0251543f, 0.899571f, 0.267983f, 
		-0.0251360f, 0.948656f, -0.351233f, -0.0250660f, 0.836885f, -0.268074f, 
		-0.0250660f, 0.836885f, -0.268074f, -0.0249870f, 0.948656f, 0.351142f, 
		-0.0249530f, 0.836885f, 0.267983f, -0.0249530f, 0.836885f, 0.267983f, 
		-0.0249386f, 1.04695f, -0.368412f, -0.0249386f, 1.04695f, -0.368412f, 
		-0.0249159f, 1.00796f, -0.268852f, -0.0248606f, 1.31045f, 2.86968e-006f, 
		-0.0248606f, 1.31045f, 2.86968e-006f, -0.0248174f, 1.29743f, -0.0263213f, 
		-0.0248065f, 1.29743f, 0.0263311f, -0.0248026f, 1.00796f, 0.268762f, 
		-0.0247822f, 1.04695f, 0.368330f, -0.0247822f, 1.04695f, 0.368330f, 
		-0.0246710f, 1.31336f, 3.00000e-006f, -0.0246710f, 1.31336f, 3.00000e-006f, 
		-0.0246005f, 1.01045f, 0.264993f, -0.0246005f, 1.01045f, 0.264993f, 
		-0.0246005f, 1.01045f, 0.264993f, -0.0245930f, 1.04695f, 0.371237f, 
		-0.0245930f, 1.04695f, 0.371237f, -0.0245731f, 1.01311f, -0.265149f, 
		-0.0245231f, 1.04696f, -0.372084f, -0.0245231f, 1.04696f, -0.372084f, 
		-0.0245114f, 1.04613f, -0.0198841f, -0.0245114f, 1.04613f, -0.0198841f, 
		-0.0245114f, 1.04613f, -0.0198841f, -0.0245114f, 1.04613f, -0.0198841f, 
		-0.0245114f, 1.04613f, -0.0198841f, -0.0245114f, 1.04613f, -0.0198841f, 
		-0.0245114f, 1.04613f, -0.0198841f, -0.0245114f, 1.04613f, -0.0198841f, 
		-0.0245114f, 1.04613f, -0.0198841f, -0.0244945f, 1.04597f, 0.0202704f, 
		-0.0244945f, 1.04597f, 0.0202704f, -0.0244945f, 1.04597f, 0.0202704f, 
		-0.0244945f, 1.04597f, 0.0202704f, -0.0244945f, 1.04597f, 0.0202704f, 
		-0.0244945f, 1.04597f, 0.0202704f, -0.0244945f, 1.04597f, 0.0202704f, 
		-0.0244945f, 1.04597f, 0.0202704f, -0.0244945f, 1.04597f, 0.0202704f, 
		-0.0244030f, 1.20667f, -0.0267053f, -0.0243941f, 1.20667f, 0.0267128f, 
		-0.0241213f, 1.00045f, 0.234721f, -0.0241213f, 1.00045f, 0.234721f, 
		-0.0241059f, 1.00045f, 0.306632f, -0.0241059f, 1.00045f, 0.306632f, 
		-0.0240740f, 0.896718f, 0.254053f, -0.0240740f, 0.896718f, 0.254053f, 
		-0.0240680f, 0.896718f, 0.281913f, -0.0240680f, 0.896718f, 0.281913f, 
		-0.0239480f, 1.04604f, -0.0243250f, -0.0239480f, 1.04604f, -0.0243250f, 
		-0.0239380f, 1.04604f, 0.0243320f, -0.0239380f, 1.04604f, 0.0243320f, 
		-0.0238740f, 1.23705f, -0.0495500f, -0.0238740f, 1.23705f, -0.0495500f, 
		-0.0238740f, 1.26705f, -0.0495500f, -0.0238740f, 1.26705f, -0.0495500f, 
		-0.0238530f, 1.23705f, 0.0495560f, -0.0238530f, 1.23705f, 0.0495560f, 
		-0.0238530f, 1.26705f, 0.0495560f, -0.0238530f, 1.26705f, 0.0495560f, 
		-0.0238158f, 1.04695f, -0.364729f, -0.0238158f, 1.04695f, -0.364729f, 
		-0.0237389f, 1.30677f, 2.90483e-006f, -0.0237389f, 1.30677f, 2.90483e-006f, 
		-0.0237389f, 1.31559f, 3.08920e-006f, -0.0237389f, 1.31559f, 3.08920e-006f, 
		-0.0237072f, 0.977046f, -0.249513f, -0.0236763f, 0.901703f, -0.268075f, 
		-0.0236763f, 0.901703f, -0.268075f, -0.0236624f, 1.20215f, 0.00803635f, 
		-0.0236624f, 1.20215f, 0.00803635f, -0.0236619f, 1.04695f, 0.364647f, 
		-0.0236619f, 1.04695f, 0.364647f, -0.0236598f, 1.04695f, 0.373465f, 
		-0.0236598f, 1.04695f, 0.373465f, -0.0235850f, 1.02272f, -0.294821f, 
		-0.0235850f, 1.02272f, -0.294821f, -0.0235392f, 1.00506f, -0.265944f, 
		-0.0234559f, 0.897585f, -0.281583f, -0.0234559f, 0.897585f, -0.281583f, 
		-0.0234513f, 1.01837f, -0.269099f, -0.0234513f, 1.01837f, -0.269099f, 
		-0.0234513f, 1.01837f, -0.269099f, -0.0234505f, 0.897585f, -0.254567f, 
		-0.0234505f, 0.897585f, -0.254567f, -0.0234252f, 1.21087f, -0.0341004f, 
		-0.0233372f, 1.01838f, 0.269007f, -0.0232920f, 1.00864f, -0.279916f, 
		-0.0231971f, 1.29323f, -0.0342555f, -0.0231752f, 1.00865f, 0.279822f, 
		-0.0231752f, 1.00865f, 0.279822f, -0.0231752f, 1.00865f, 0.279822f, 
		-0.0231686f, 1.01390f, -0.279835f, -0.0231686f, 1.01390f, -0.279835f, 
		-0.0231686f, 1.01390f, -0.279835f, -0.0230834f, 1.01902f, 0.266024f, 
		-0.0230515f, 1.01390f, 0.279743f, -0.0230515f, 1.01390f, 0.279743f, 
		-0.0230515f, 1.01390f, 0.279743f, -0.0225846f, 1.55399f, -0.0435699f, 
		-0.0225846f, 1.55399f, -0.0435699f, -0.0225639f, 1.01760f, -0.279527f, 
		-0.0225288f, 1.04601f, 0.0170085f, -0.0225145f, 1.56290f, -0.0778864f, 
		-0.0224972f, 1.04604f, -0.0274616f, -0.0224972f, 1.04604f, -0.0274616f, 
		-0.0224856f, 1.04604f, 0.0274677f, -0.0224856f, 1.04604f, 0.0274677f, 
		-0.0224833f, 1.56290f, 0.0663452f, -0.0224453f, 1.01760f, 0.279435f, 
		-0.0222898f, 1.03132f, -0.296023f, -0.0222898f, 1.03132f, -0.296023f, 
		-0.0222898f, 1.03132f, -0.296023f, -0.0222454f, 0.977046f, -0.292046f, 
		-0.0222082f, 0.903956f, 0.267982f, -0.0222082f, 0.903956f, 0.267982f, 
		-0.0221829f, 0.989730f, -0.302851f, -0.0221697f, 0.989732f, -0.238685f, 
		-0.0221229f, 0.977046f, 0.291952f, -0.0220679f, 0.989731f, 0.238593f, 
		-0.0220591f, 1.23699f, -0.0451145f, -0.0220591f, 1.23699f, -0.0451145f, 
		-0.0220542f, 0.989730f, 0.302758f, -0.0220407f, 1.23699f, 0.0451194f, 
		-0.0220407f, 1.23699f, 0.0451194f, -0.0220064f, 0.984615f, -0.248730f, 
		-0.0219018f, 0.984615f, 0.248637f, -0.0218802f, 1.04552f, -0.0151887f, 
		-0.0218802f, 1.04552f, -0.0151887f, -0.0218802f, 1.04552f, -0.0151887f, 
		-0.0218802f, 1.04552f, -0.0151887f, -0.0218710f, 0.980877f, -0.246471f, 
		-0.0217794f, 0.899571f, 0.255377f, -0.0217794f, 0.899571f, 0.255377f, 
		-0.0217739f, 0.899571f, 0.280589f, -0.0217739f, 0.899571f, 0.280589f, 
		-0.0217456f, 1.04696f, -0.375861f, -0.0217456f, 1.04696f, -0.375861f, 
		-0.0217180f, 0.836885f, -0.280580f, -0.0217180f, 0.836885f, -0.280580f, 
		-0.0217130f, 0.836885f, -0.255570f, -0.0217130f, 0.836885f, -0.255570f, 
		-0.0216073f, 1.03452f, -0.368412f, -0.0216073f, 1.03452f, -0.368412f, 
		-0.0216072f, 1.05939f, -0.368412f, -0.0216072f, 1.05939f, -0.368412f, 
		-0.0216050f, 0.836885f, 0.255478f, -0.0216050f, 0.836885f, 0.255478f, 
		-0.0216000f, 0.836885f, 0.280487f, -0.0216000f, 0.836885f, 0.280487f, 
		-0.0215755f, 1.04695f, -0.362442f, -0.0215755f, 1.04695f, -0.362442f, 
		-0.0215358f, 1.26712f, -0.0453667f, -0.0215358f, 1.26712f, -0.0453667f, 
		-0.0215325f, 1.31045f, -0.0124277f, -0.0215325f, 1.31045f, -0.0124277f, 
		-0.0215274f, 1.31045f, 0.0124328f, -0.0215274f, 1.31045f, 0.0124328f, 
		-0.0215174f, 1.26712f, 0.0453712f, -0.0215174f, 1.26712f, 0.0453712f, 
		-0.0214986f, 1.30448f, 2.31897e-006f, -0.0214986f, 1.30448f, 2.31897e-006f, 
		-0.0214985f, 1.31788f, 2.41826e-006f, -0.0214985f, 1.31788f, 2.41826e-006f, 
		-0.0214812f, 1.01499f, -0.258421f, -0.0214812f, 1.01499f, -0.258421f, 
		-0.0214812f, 1.01499f, -0.258421f, -0.0214516f, 1.03452f, 0.368330f, 
		-0.0214516f, 1.03452f, 0.368330f, -0.0214515f, 1.05939f, 0.368330f, 
		-0.0214515f, 1.05939f, 0.368330f, -0.0214226f, 1.04695f, 0.362359f, 
		-0.0214226f, 1.04695f, 0.362359f, -0.0214196f, 1.04695f, 0.375753f, 
		-0.0214196f, 1.04695f, 0.375753f, -0.0213718f, 1.01499f, 0.258328f, 
		-0.0213718f, 1.01499f, 0.258328f, -0.0213718f, 1.01499f, 0.258328f, 
		-0.0213630f, 1.31336f, 0.0123380f, -0.0213630f, 1.31336f, 0.0123380f, 
		-0.0213125f, 1.05916f, -0.372160f, -0.0213125f, 1.05916f, -0.372160f, 
		-0.0212870f, 1.03462f, 0.371236f, -0.0212870f, 1.03462f, 0.371236f, 
		-0.0212870f, 1.05929f, 0.371236f, -0.0212870f, 1.05929f, 0.371236f, 
		-0.0212662f, 1.03467f, -0.372160f, -0.0212662f, 1.03467f, -0.372160f, 
		-0.0212547f, 1.05823f, -0.0202638f, -0.0212547f, 1.05823f, -0.0202638f, 
		-0.0212547f, 1.05823f, -0.0202638f, -0.0212547f, 1.05823f, -0.0202638f, 
		-0.0212547f, 1.05823f, -0.0202638f, -0.0212547f, 1.05823f, -0.0202638f, 
		-0.0212547f, 1.05823f, -0.0202638f, -0.0212547f, 1.05823f, -0.0202638f, 
		-0.0212547f, 1.05823f, -0.0202638f, -0.0212430f, 0.906317f, -0.268075f, 
		-0.0212430f, 0.906317f, -0.268075f, -0.0211868f, 1.03373f, -0.0202636f, 
		-0.0211868f, 1.03373f, -0.0202636f, -0.0211868f, 1.03373f, -0.0202636f, 
		-0.0211868f, 1.03373f, -0.0202636f, -0.0211868f, 1.03373f, -0.0202636f, 
		-0.0211868f, 1.03373f, -0.0202636f, -0.0211868f, 1.03373f, -0.0202636f, 
		-0.0211868f, 1.03373f, -0.0202636f, -0.0211740f, 1.55400f, 0.0326946f, 
		-0.0211740f, 1.55400f, 0.0326946f, -0.0211717f, 1.31412f, -0.0122196f, 
		-0.0211717f, 1.31412f, -0.0122196f, -0.0210930f, 1.03389f, 0.0212321f, 
		-0.0210930f, 1.03389f, 0.0212321f, -0.0210930f, 1.03389f, 0.0212321f, 
		-0.0210930f, 1.03389f, 0.0212321f, -0.0210930f, 1.03389f, 0.0212321f, 
		-0.0210930f, 1.03389f, 0.0212321f, -0.0210930f, 1.03389f, 0.0212321f, 
		-0.0210731f, 1.28192f, -0.0436571f, -0.0210608f, 1.05824f, 0.0212320f, 
		-0.0210608f, 1.05824f, 0.0212320f, -0.0210608f, 1.05824f, 0.0212320f, 
		-0.0210608f, 1.05824f, 0.0212320f, -0.0210608f, 1.05824f, 0.0212320f, 
		-0.0210547f, 1.28192f, 0.0436613f, -0.0210377f, 1.00635f, 0.258176f, 
		-0.0210105f, 1.22218f, -0.0436871f, -0.0209905f, 1.22218f, 0.0436926f, 
		-0.0209351f, 1.02268f, -0.269923f, -0.0209351f, 1.02268f, -0.269923f, 
		-0.0209240f, 1.00979f, 2.00000e-006f, -0.0209240f, 1.00979f, 2.00000e-006f, 
		-0.0209240f, 1.08228f, 2.00000e-006f, -0.0209240f, 1.08228f, 2.00000e-006f, 
		-0.0208940f, 0.853677f, -0.131551f, -0.0208940f, 0.853677f, -0.131551f, 
		-0.0208380f, 0.853677f, 0.131458f, -0.0208380f, 0.853677f, 0.131458f, 
		-0.0207670f, 1.01007f, -0.00411700f, -0.0207670f, 1.01007f, -0.00411700f, 
		-0.0207670f, 1.08200f, -0.00411700f, -0.0207670f, 1.08200f, -0.00411700f, 
		-0.0207650f, 1.01007f, 0.00412200f, -0.0207650f, 1.01007f, 0.00412200f, 
		-0.0207650f, 1.08200f, 0.00412200f, -0.0207650f, 1.08200f, 0.00412200f, 
		-0.0207400f, 1.03407f, -0.0243260f, -0.0207400f, 1.03407f, -0.0243260f, 
		-0.0207400f, 1.05801f, -0.0243260f, -0.0207400f, 1.05801f, -0.0243260f, 
		-0.0207300f, 1.03407f, 0.0243310f, -0.0207300f, 1.03407f, 0.0243310f, 
		-0.0206358f, 1.03509f, -0.364730f, -0.0206358f, 1.03509f, -0.364730f, 
		-0.0206355f, 1.05882f, -0.364730f, -0.0206355f, 1.05882f, -0.364730f, 
		-0.0206043f, 1.20667f, 0.0297306f, -0.0205612f, 1.30677f, -0.0118674f, 
		-0.0205612f, 1.30677f, -0.0118674f, -0.0205561f, 1.31559f, 0.0118714f, 
		-0.0205561f, 1.31559f, 0.0118714f, -0.0205561f, 1.30677f, 0.0118714f, 
		-0.0205561f, 1.30677f, 0.0118714f, -0.0205142f, 0.901703f, -0.279886f, 
		-0.0205142f, 0.901703f, -0.279886f, -0.0205091f, 0.901703f, -0.256265f, 
		-0.0205091f, 0.901703f, -0.256265f, -0.0204811f, 1.05882f, 0.364646f, 
		-0.0204811f, 1.05882f, 0.364646f, -0.0204811f, 1.03509f, 0.364646f, 
		-0.0204811f, 1.03509f, 0.364646f, -0.0204791f, 1.03509f, 0.373465f, 
		-0.0204791f, 1.03509f, 0.373465f, -0.0204791f, 1.05882f, 0.373465f, 
		-0.0204791f, 1.05882f, 0.373465f, -0.0204702f, 1.00046f, -0.274706f, 
		-0.0204702f, 1.00046f, -0.274706f, -0.0204591f, 1.05785f, 0.0254609f, 
		-0.0204591f, 1.05785f, 0.0254609f, -0.0203688f, 0.908777f, 0.267982f, 
		-0.0203688f, 0.908777f, 0.267982f, -0.0203529f, 1.02271f, 0.271786f, 
		-0.0203529f, 1.02271f, 0.271786f, -0.0203430f, 0.879246f, -0.135211f, 
		-0.0203160f, 1.01085f, -0.00755200f, -0.0203160f, 1.01085f, -0.00755200f, 
		-0.0203160f, 1.08122f, -0.00755200f, -0.0203160f, 1.08122f, -0.00755200f, 
		-0.0203120f, 1.01085f, 0.00755700f, -0.0203120f, 1.01085f, 0.00755700f, 
		-0.0203120f, 1.08122f, 0.00755700f, -0.0203120f, 1.08122f, 0.00755700f, 
		-0.0202980f, 0.853634f, -0.135506f, -0.0202980f, 0.853634f, -0.135506f, 
		-0.0202860f, 0.879246f, 0.135117f, -0.0202445f, 1.02272f, -0.274706f, 
		-0.0202445f, 1.02272f, -0.274706f, -0.0202410f, 0.853634f, 0.135413f, 
		-0.0202410f, 0.853634f, 0.135413f, -0.0201378f, 1.29743f, 0.0300492f, 
		-0.0201274f, 1.02272f, 0.274614f, -0.0201274f, 1.02272f, 0.274614f, 
		-0.0200676f, 0.911318f, -0.268076f, -0.0200676f, 0.911318f, -0.268076f, 
		-0.0200676f, 0.911318f, -0.268076f, -0.0200640f, 0.868091f, -0.302717f, 
		-0.0200640f, 0.868091f, -0.302717f, -0.0200570f, 0.948080f, -0.268076f, 
		-0.0200570f, 0.948080f, -0.268076f, -0.0200490f, 0.868091f, -0.233434f, 
		-0.0200490f, 0.868091f, -0.233434f, -0.0200079f, 1.01909f, -0.258242f, 
		-0.0200079f, 1.01909f, -0.258242f, -0.0200079f, 1.01909f, -0.258242f, 
		-0.0199998f, 1.00046f, 0.262791f, -0.0199998f, 1.00046f, 0.262791f, 
		-0.0199998f, 1.00046f, 0.262791f, -0.0199998f, 1.00046f, 0.262791f, 
		-0.0199510f, 0.868091f, 0.233341f, -0.0199510f, 0.868091f, 0.233341f, 
		-0.0199430f, 0.912612f, 0.267982f, -0.0199430f, 0.912612f, 0.267982f, 
		-0.0199430f, 0.912612f, 0.267982f, -0.0199430f, 0.948080f, 0.267982f, 
		-0.0199430f, 0.948080f, 0.267982f, -0.0199360f, 0.868091f, 0.302623f, 
		-0.0199360f, 0.868091f, 0.302623f, -0.0199240f, 0.872834f, -0.302475f, 
		-0.0199240f, 0.872834f, -0.302475f, -0.0199100f, 0.872834f, -0.233676f, 
		-0.0199100f, 0.872834f, -0.233676f, -0.0198120f, 1.00045f, -0.267231f, 
		-0.0198120f, 1.00045f, -0.267231f, -0.0198110f, 0.872834f, 0.233583f, 
		-0.0198110f, 0.872834f, 0.233583f, -0.0197960f, 0.872834f, 0.302381f, 
		-0.0197960f, 0.872834f, 0.302381f, -0.0197798f, 1.00052f, -0.278153f, 
		-0.0197798f, 1.00052f, -0.278153f, -0.0197630f, 0.878415f, -0.139061f, 
		-0.0197433f, 1.30200f, -0.0139346f, -0.0197362f, 1.03132f, -0.298061f, 
		-0.0197362f, 1.03132f, -0.298061f, -0.0197362f, 1.03132f, -0.298061f, 
		-0.0197040f, 0.878415f, 0.138968f, -0.0196990f, 1.00045f, 0.267139f, 
		-0.0196990f, 1.00045f, 0.267139f, -0.0196987f, 1.31685f, -0.0113698f, 
		-0.0196987f, 1.31685f, -0.0113698f, -0.0196618f, 1.00052f, 0.278060f, 
		-0.0196618f, 1.00052f, 0.278060f, -0.0196244f, 1.03132f, 0.243385f, 
		-0.0196244f, 1.03132f, 0.243385f, -0.0196244f, 1.03132f, 0.243385f, 
		-0.0196080f, 1.01208f, -0.0103220f, -0.0196080f, 1.01208f, -0.0103220f, 
		-0.0196080f, 1.07999f, -0.0103220f, -0.0196080f, 1.07999f, -0.0103220f, 
		-0.0196030f, 1.01208f, 0.0103260f, -0.0196030f, 1.01208f, 0.0103260f, 
		-0.0196030f, 1.07999f, 0.0103260f, -0.0196030f, 1.07999f, 0.0103260f, 
		-0.0195876f, 1.30200f, 0.0140288f, -0.0195876f, 1.30200f, 0.0140288f, 
		-0.0195350f, 0.858925f, -0.301801f, -0.0195350f, 0.858925f, -0.301801f, 
		-0.0195265f, 0.951393f, -0.268076f, -0.0195265f, 0.951393f, -0.268076f, 
		-0.0195210f, 0.858925f, -0.234350f, -0.0195210f, 0.858925f, -0.234350f, 
		-0.0195160f, 0.877414f, -0.301768f, -0.0195160f, 0.877414f, -0.301768f, 
		-0.0195020f, 0.877414f, -0.234383f, -0.0195020f, 0.877414f, -0.234383f, 
		-0.0194836f, 1.03479f, -0.0274626f, -0.0194836f, 1.03479f, -0.0274626f, 
		-0.0194836f, 1.05728f, -0.0274626f, -0.0194836f, 1.05728f, -0.0274626f, 
		-0.0194720f, 1.03479f, 0.0274667f, -0.0194720f, 1.03479f, 0.0274667f, 
		-0.0194220f, 0.858925f, 0.234257f, -0.0194220f, 0.858925f, 0.234257f, 
		-0.0194136f, 0.951393f, 0.267982f, -0.0194136f, 0.951393f, 0.267982f, 
		-0.0194070f, 0.858925f, 0.301707f, -0.0194070f, 0.858925f, 0.301707f, 
		-0.0194030f, 0.877414f, 0.234290f, -0.0194030f, 0.877414f, 0.234290f, 
		-0.0193880f, 0.877414f, 0.301674f, -0.0193880f, 0.877414f, 0.301674f, 
		-0.0193880f, 1.00048f, 0.278661f, -0.0193880f, 1.00048f, 0.278661f, 
		-0.0192280f, 0.903956f, 0.256849f, -0.0192280f, 0.903956f, 0.256849f, 
		-0.0192228f, 0.903956f, 0.279114f, -0.0192228f, 0.903956f, 0.279114f, 
		-0.0192029f, 1.05652f, 0.0151935f, -0.0192029f, 1.05652f, 0.0151935f, 
		-0.0191018f, 1.00045f, -0.231858f, -0.0191018f, 1.00045f, -0.231858f, 
		-0.0190654f, 0.984599f, -0.296221f, -0.0190654f, 0.984599f, -0.296221f, 
		-0.0190645f, 1.30200f, -0.0147187f, -0.0190645f, 1.30200f, -0.0147187f, 
		-0.0190048f, 1.00045f, 0.231765f, -0.0190048f, 1.00045f, 0.231765f, 
		-0.0189393f, 0.984599f, 0.296128f, -0.0189393f, 0.984599f, 0.296128f, 
		-0.0188962f, 1.04695f, -0.391596f, -0.0188962f, 1.04695f, -0.391596f, 
		-0.0188962f, 1.04695f, -0.391596f, -0.0188962f, 1.04695f, -0.391596f, 
		-0.0188962f, 1.04695f, -0.391596f, -0.0188962f, 1.04695f, -0.391596f, 
		-0.0188962f, 1.04695f, -0.391596f, -0.0188962f, 1.04695f, -0.391596f, 
		-0.0188962f, 1.04695f, -0.391596f, -0.0188962f, 1.04695f, -0.391596f, 
		-0.0188962f, 1.04695f, -0.391596f, -0.0188962f, 1.04695f, -0.391596f, 
		-0.0188962f, 1.04695f, -0.391596f, -0.0188962f, 1.04695f, -0.391596f, 
		-0.0188906f, 1.04604f, -0.0298596f, -0.0188906f, 1.04604f, -0.0298596f, 
		-0.0188781f, 1.04604f, 0.0298643f, -0.0188781f, 1.04604f, 0.0298643f, 
		-0.0188540f, 0.881798f, -0.300623f, -0.0188540f, 0.881798f, -0.300623f, 
		-0.0188410f, 0.881798f, -0.235529f, -0.0188410f, 0.881798f, -0.235529f, 
		-0.0188347f, 1.05767f, -0.375854f, -0.0188347f, 1.05767f, -0.375854f, 
		-0.0188190f, 1.02272f, -0.243243f, -0.0188190f, 1.02272f, -0.243243f, 
		-0.0188190f, 1.03132f, -0.243243f, -0.0188190f, 1.03132f, -0.243243f, 
		-0.0188190f, 1.03132f, -0.243243f, -0.0188135f, 1.33363f, 2.37289e-006f, 
		-0.0188135f, 1.33363f, 2.37289e-006f, -0.0188135f, 1.33363f, 2.37289e-006f, 
		-0.0188135f, 1.33363f, 2.37289e-006f, -0.0188135f, 1.33363f, 2.37289e-006f, 
		-0.0188135f, 1.33363f, 2.37289e-006f, -0.0188135f, 1.33363f, 2.37289e-006f, 
		-0.0188135f, 1.33363f, 2.37289e-006f, -0.0188135f, 1.33363f, 2.37289e-006f, 
		-0.0188135f, 1.33363f, 2.37289e-006f, -0.0188135f, 1.33363f, 2.37289e-006f, 
		-0.0187740f, 0.905134f, -0.145634f, -0.0187410f, 0.881798f, 0.235435f, 
		-0.0187410f, 0.881798f, 0.235435f, -0.0187360f, 1.03607f, -0.375854f, 
		-0.0187360f, 1.03607f, -0.375854f, -0.0187313f, 1.04695f, 0.389082f, 
		-0.0187313f, 1.04695f, 0.389082f, -0.0187313f, 1.04695f, 0.389082f, 
		-0.0187313f, 1.04695f, 0.389082f, -0.0187313f, 1.04695f, 0.389082f, 
		-0.0187313f, 1.04695f, 0.389082f, -0.0187313f, 1.04695f, 0.389082f, 
		-0.0187313f, 1.04695f, 0.389082f, -0.0187313f, 1.04695f, 0.389082f, 
		-0.0187313f, 1.04695f, 0.389082f, -0.0187313f, 1.04695f, 0.389082f, 
		-0.0187313f, 1.04695f, 0.389082f, -0.0187313f, 1.04695f, 0.389082f, 
		-0.0187313f, 1.04695f, 0.389082f, -0.0187270f, 0.881798f, 0.300529f, 
		-0.0187270f, 0.881798f, 0.300529f, -0.0187120f, 0.905134f, 0.145540f, 
		-0.0187040f, 1.02272f, 0.298201f, -0.0187040f, 1.02272f, 0.298201f, 
		-0.0186952f, 1.05770f, -0.362443f, -0.0186952f, 1.05770f, -0.362443f, 
		-0.0186951f, 1.03621f, -0.362443f, -0.0186951f, 1.03621f, -0.362443f, 
		-0.0186780f, 1.01369f, -0.0124440f, -0.0186780f, 1.01369f, -0.0124440f, 
		-0.0186780f, 1.07838f, -0.0124440f, -0.0186780f, 1.07838f, -0.0124440f, 
		-0.0186730f, 1.01369f, 0.0124480f, -0.0186730f, 1.01369f, 0.0124480f, 
		-0.0186730f, 1.07838f, 0.0124480f, -0.0186730f, 1.07838f, 0.0124480f, 
		-0.0186497f, 1.04695f, -0.388356f, -0.0186497f, 1.04695f, -0.388356f, 
		-0.0186497f, 1.04695f, -0.388356f, -0.0186497f, 1.04695f, -0.388356f, 
		-0.0186497f, 1.04695f, -0.388356f, -0.0186497f, 1.04695f, -0.388356f, 
		-0.0186497f, 1.04695f, -0.388356f, -0.0186497f, 1.04695f, -0.388356f, 
		-0.0186497f, 1.04695f, -0.388356f, -0.0186497f, 1.04695f, -0.388356f, 
		-0.0186213f, 1.30448f, -0.0107477f, -0.0186213f, 1.30448f, -0.0107477f, 
		-0.0186161f, 1.31788f, 0.0107519f, -0.0186161f, 1.31788f, 0.0107519f, 
		-0.0186161f, 1.30448f, 0.0107519f, -0.0186161f, 1.30448f, 0.0107519f, 
		-0.0185672f, 1.33039f, 2.30199e-006f, -0.0185672f, 1.33039f, 2.30199e-006f, 
		-0.0185672f, 1.33039f, 2.30199e-006f, -0.0185672f, 1.33039f, 2.30199e-006f, 
		-0.0185672f, 1.33039f, 2.30199e-006f, -0.0185672f, 1.33039f, 2.30199e-006f, 
		-0.0185672f, 1.33039f, 2.30199e-006f, -0.0185672f, 1.33039f, 2.30199e-006f, 
		-0.0185423f, 1.03621f, 0.362358f, -0.0185423f, 1.03621f, 0.362358f, 
		-0.0185422f, 1.05770f, 0.362358f, -0.0185422f, 1.05770f, 0.362358f, 
		-0.0185392f, 1.03621f, 0.375752f, -0.0185392f, 1.03621f, 0.375752f, 
		-0.0185391f, 1.05770f, 0.375752f, -0.0185391f, 1.05770f, 0.375752f, 
		-0.0185216f, 1.04696f, -0.376982f, -0.0185216f, 1.04696f, -0.376982f, 
		-0.0184842f, 1.04695f, 0.392322f, -0.0184842f, 1.04695f, 0.392322f, 
		-0.0184842f, 1.04695f, 0.392322f, -0.0184842f, 1.04695f, 0.392322f, 
		-0.0184842f, 1.04695f, 0.392322f, -0.0184842f, 1.04695f, 0.392322f, 
		-0.0184842f, 1.04695f, 0.392322f, -0.0184842f, 1.04695f, 0.392322f, 
		-0.0184842f, 1.04695f, 0.392322f, -0.0184842f, 1.04695f, 0.392322f, 
		-0.0184422f, 1.31902f, 2.27073e-006f, -0.0184422f, 1.31902f, 2.27073e-006f, 
		-0.0184087f, 1.05187f, -0.361264f, -0.0184087f, 1.05187f, -0.361264f, 
		-0.0184087f, 1.05187f, -0.361264f, -0.0184087f, 1.05187f, -0.361264f, 
		-0.0184087f, 1.05187f, -0.361264f, -0.0184070f, 0.906317f, -0.278669f, 
		-0.0184070f, 0.906317f, -0.278669f, -0.0184026f, 0.906317f, -0.257482f, 
		-0.0184026f, 0.906317f, -0.257482f, -0.0183627f, 1.04696f, 0.376898f, 
		-0.0183627f, 1.04696f, 0.376898f, -0.0183308f, 1.30330f, 0.00491432f, 
		-0.0183308f, 1.30330f, 0.00491432f, -0.0183308f, 1.30330f, 0.00491432f, 
		-0.0183308f, 1.30330f, 0.00491432f, -0.0182763f, 1.02272f, -0.299057f, 
		-0.0182763f, 1.02272f, -0.299057f, -0.0182556f, 1.05187f, 0.361180f, 
		-0.0182556f, 1.05187f, 0.361180f, -0.0182556f, 1.05187f, 0.361180f, 
		-0.0182556f, 1.05187f, 0.361180f, -0.0182390f, 0.903517f, -0.149188f, 
		-0.0182080f, 0.969213f, -0.328791f, -0.0181760f, 0.903517f, 0.149094f, 
		-0.0181644f, 1.02272f, 0.242389f, -0.0181644f, 1.02272f, 0.242389f, 
		-0.0180816f, 1.29323f, -0.0372127f, -0.0180783f, 1.05648f, 0.0289083f, 
		-0.0180783f, 1.05648f, 0.0289083f, -0.0180749f, 1.03610f, -0.0151661f, 
		-0.0180749f, 1.03610f, -0.0151661f, -0.0180690f, 0.969213f, 0.328697f, 
		-0.0180686f, 1.03610f, 0.0151695f, -0.0180686f, 1.03610f, 0.0151695f, 
		-0.0180662f, 1.29323f, 0.0372161f, -0.0180560f, 1.03350f, 0.0151850f, 
		-0.0180560f, 1.03350f, 0.0151850f, -0.0180560f, 1.03350f, 0.0151850f, 
		-0.0180560f, 1.03350f, 0.0151850f, -0.0180388f, 1.05614f, -0.0152385f, 
		-0.0180388f, 1.05614f, -0.0152385f, -0.0180290f, 0.850508f, -0.299193f, 
		-0.0180290f, 0.850508f, -0.299193f, -0.0180160f, 0.850508f, -0.236959f, 
		-0.0180160f, 0.850508f, -0.236959f, -0.0179560f, 0.885952f, -0.299066f, 
		-0.0179560f, 0.885952f, -0.299066f, -0.0179420f, 0.885952f, -0.237086f, 
		-0.0179420f, 0.885952f, -0.237086f, -0.0179220f, 1.00361f, -0.256368f, 
		-0.0179220f, 1.00361f, -0.256368f, -0.0179220f, 1.00361f, -0.256368f, 
		-0.0179150f, 0.850508f, 0.236865f, -0.0179150f, 0.850508f, 0.236865f, 
		-0.0179020f, 0.850508f, 0.299099f, -0.0179020f, 0.850508f, 0.299099f, 
		-0.0178420f, 0.885952f, 0.236991f, -0.0178420f, 0.885952f, 0.236991f, 
		-0.0178383f, 0.993267f, -0.307711f, -0.0178345f, 1.21088f, -0.0373324f, 
		-0.0178290f, 0.885952f, 0.298972f, -0.0178290f, 0.885952f, 0.298972f, 
		-0.0178180f, 1.21088f, 0.0373355f, -0.0178128f, 1.00361f, 0.256273f, 
		-0.0178128f, 1.00361f, 0.256273f, -0.0178128f, 1.00361f, 0.256273f, 
		-0.0177227f, 0.993265f, 0.233735f, -0.0176850f, 0.965681f, -0.327098f, 
		-0.0176390f, 0.832160f, -0.268076f, -0.0176390f, 0.832160f, -0.268076f, 
		-0.0176373f, 1.05672f, 0.0151693f, -0.0176373f, 1.05672f, 0.0151693f, 
		-0.0176373f, 1.05672f, 0.0151693f, -0.0176373f, 1.05672f, 0.0151693f, 
		-0.0176343f, 0.908777f, 0.257769f, -0.0176343f, 0.908777f, 0.257769f, 
		-0.0176304f, 0.908777f, 0.278194f, -0.0176304f, 0.908777f, 0.278194f, 
		-0.0175650f, 1.01562f, -0.0139370f, -0.0175650f, 1.01562f, -0.0139370f, 
		-0.0175650f, 1.07645f, -0.0139370f, -0.0175650f, 1.07645f, -0.0139370f, 
		-0.0175590f, 1.01562f, 0.0139400f, -0.0175590f, 1.01562f, 0.0139400f, 
		-0.0175590f, 1.07645f, 0.0139400f, -0.0175590f, 1.07645f, 0.0139400f, 
		-0.0175470f, 0.965681f, 0.327003f, -0.0175260f, 0.832160f, 0.267982f, 
		-0.0175260f, 0.832160f, 0.267982f, -0.0173884f, 0.911318f, -0.278081f, 
		-0.0173884f, 0.911318f, -0.278081f, -0.0173884f, 0.911318f, -0.278081f, 
		-0.0173843f, 0.911318f, -0.258071f, -0.0173843f, 0.911318f, -0.258071f, 
		-0.0173843f, 0.911318f, -0.258071f, -0.0173790f, 0.948080f, -0.278076f, 
		-0.0173790f, 0.948080f, -0.278076f, -0.0173750f, 0.948080f, -0.258076f, 
		-0.0173750f, 0.948080f, -0.258076f, -0.0173376f, 1.00044f, -0.259823f, 
		-0.0173376f, 1.00044f, -0.259823f, -0.0172809f, 1.20210f, 0.0167930f, 
		-0.0172809f, 1.20210f, 0.0167930f, -0.0172809f, 1.20210f, 0.0167930f, 
		-0.0172660f, 0.912612f, 0.257982f, -0.0172660f, 0.912612f, 0.257982f, 
		-0.0172660f, 0.912612f, 0.257982f, -0.0172660f, 0.948080f, 0.257982f, 
		-0.0172660f, 0.948080f, 0.257982f, -0.0172620f, 0.912612f, 0.277982f, 
		-0.0172620f, 0.912612f, 0.277982f, -0.0172620f, 0.912612f, 0.277982f, 
		-0.0172512f, 0.949218f, 0.277976f, -0.0172512f, 0.949218f, 0.277976f, 
		-0.0172505f, 0.955074f, -0.268076f, -0.0172505f, 0.955074f, -0.268076f, 
		-0.0171376f, 0.955074f, 0.267982f, -0.0171376f, 0.955074f, 0.267982f, 
		-0.0170583f, 1.00796f, -0.289011f, -0.0170583f, 1.00796f, -0.289011f, 
		-0.0170583f, 1.00796f, -0.289011f, -0.0170583f, 1.00796f, -0.289011f, 
		-0.0170583f, 1.00796f, -0.289011f, -0.0169355f, 1.00796f, 0.288917f, 
		-0.0169355f, 1.00796f, 0.288917f, -0.0169355f, 1.00796f, 0.288917f, 
		-0.0169355f, 1.00796f, 0.288917f, -0.0169355f, 1.00796f, 0.288917f, 
		-0.0169201f, 0.951393f, -0.277812f, -0.0169201f, 0.951393f, -0.277812f, 
		-0.0169161f, 0.951393f, -0.258341f, -0.0169161f, 0.951393f, -0.258341f, 
		-0.0168955f, 1.02256f, 0.256662f, -0.0168955f, 1.02256f, 0.256662f, 
		-0.0168955f, 1.02256f, 0.256662f, -0.0168955f, 1.02256f, 0.256662f, 
		-0.0168899f, 1.31897f, -0.00887997f, -0.0168899f, 1.31897f, -0.00887997f, 
		-0.0168350f, 0.889845f, -0.297127f, -0.0168350f, 0.889845f, -0.297127f, 
		-0.0168230f, 0.889845f, -0.239026f, -0.0168230f, 0.889845f, -0.239026f, 
		-0.0168072f, 0.951393f, 0.258247f, -0.0168072f, 0.951393f, 0.258247f, 
		-0.0167753f, 0.989732f, -0.235573f, -0.0167220f, 0.889845f, 0.238931f, 
		-0.0167220f, 0.889845f, 0.238931f, -0.0167090f, 0.889845f, 0.297032f, 
		-0.0167090f, 0.889845f, 0.297032f, -0.0166608f, 0.989732f, 0.305873f, 
		-0.0165998f, 1.00565f, -0.252775f, -0.0165998f, 1.00565f, -0.252775f, 
		-0.0165998f, 1.00565f, -0.252775f, -0.0165998f, 1.00565f, -0.252775f, 
		-0.0165998f, 1.00565f, -0.252775f, -0.0165187f, 0.977046f, 0.243357f, 
		-0.0165140f, 1.04696f, -0.386277f, -0.0165140f, 1.04696f, -0.386277f, 
		-0.0165140f, 1.04696f, -0.386277f, -0.0165140f, 1.04696f, -0.386277f, 
		-0.0165140f, 1.04696f, -0.386277f, -0.0165140f, 1.04696f, -0.386277f, 
		-0.0164698f, 1.04695f, -0.394638f, -0.0164698f, 1.04695f, -0.394638f, 
		-0.0164698f, 1.04695f, -0.394638f, -0.0164698f, 1.04695f, -0.394638f, 
		-0.0164698f, 1.04695f, -0.394638f, -0.0164698f, 1.04695f, -0.394638f, 
		-0.0164698f, 1.04695f, -0.394638f, -0.0164324f, 1.32831f, 1.27218e-006f, 
		-0.0164324f, 1.32831f, 1.27218e-006f, -0.0164324f, 1.32831f, 1.27218e-006f, 
		-0.0164324f, 1.32831f, 1.27218e-006f, -0.0164159f, 0.984602f, 0.297110f, 
		-0.0164159f, 0.984602f, 0.297110f, -0.0163864f, 1.33668f, 1.46429e-006f, 
		-0.0163864f, 1.33668f, 1.46429e-006f, -0.0163864f, 1.33668f, 1.46429e-006f, 
		-0.0163864f, 1.33668f, 1.46429e-006f, -0.0163864f, 1.33668f, 1.46429e-006f, 
		-0.0163864f, 1.33668f, 1.46429e-006f, -0.0163757f, 1.05636f, -0.389166f, 
		-0.0163757f, 1.05636f, -0.389166f, -0.0163757f, 1.05636f, -0.389166f, 
		-0.0163757f, 1.05636f, -0.389166f, -0.0163757f, 1.05636f, -0.389166f, 
		-0.0163757f, 1.05636f, -0.389166f, -0.0163757f, 1.05636f, -0.389166f, 
		-0.0163757f, 1.05636f, -0.389166f, -0.0163757f, 1.05636f, -0.389166f, 
		-0.0163757f, 1.05636f, -0.389166f, -0.0163757f, 1.05636f, -0.389166f, 
		-0.0163757f, 1.05636f, -0.389166f, -0.0163757f, 1.05636f, -0.389166f, 
		-0.0163757f, 1.05636f, -0.389166f, -0.0163755f, 1.03755f, -0.389166f, 
		-0.0163755f, 1.03755f, -0.389166f, -0.0163755f, 1.03755f, -0.389166f, 
		-0.0163755f, 1.03755f, -0.389166f, -0.0163755f, 1.03755f, -0.389166f, 
		-0.0163755f, 1.03755f, -0.389166f, -0.0163755f, 1.03755f, -0.389166f, 
		-0.0163755f, 1.03755f, -0.389166f, -0.0163755f, 1.03755f, -0.389166f, 
		-0.0163755f, 1.03755f, -0.389166f, -0.0163755f, 1.03755f, -0.389166f, 
		-0.0163755f, 1.03755f, -0.389166f, -0.0163755f, 1.03755f, -0.389166f, 
		-0.0163611f, 1.05548f, -0.0298607f, -0.0163611f, 1.05548f, -0.0298607f, 
		-0.0163611f, 1.03659f, -0.0298607f, -0.0163611f, 1.03659f, -0.0298607f, 
		-0.0163507f, 1.04695f, 0.386192f, -0.0163507f, 1.04695f, 0.386192f, 
		-0.0163507f, 1.04695f, 0.386192f, -0.0163507f, 1.04695f, 0.386192f, 
		-0.0163507f, 1.04695f, 0.386192f, -0.0163507f, 1.04695f, 0.386192f, 
		-0.0163507f, 1.04695f, 0.386192f, -0.0163507f, 1.04695f, 0.386192f, 
		-0.0163507f, 1.04695f, 0.386192f, -0.0163507f, 1.04695f, 0.386192f, 
		-0.0163507f, 1.04695f, 0.386192f, -0.0163507f, 1.04695f, 0.386192f, 
		-0.0163480f, 1.03659f, 0.0298632f, -0.0163480f, 1.03659f, 0.0298632f, 
		-0.0163130f, 0.929791f, -0.161986f, -0.0163027f, 1.04695f, 0.394553f, 
		-0.0163027f, 1.04695f, 0.394553f, -0.0163027f, 1.04695f, 0.394553f, 
		-0.0163027f, 1.04695f, 0.394553f, -0.0163027f, 1.04695f, 0.394553f, 
		-0.0163027f, 1.04695f, 0.394553f, -0.0163020f, 1.01780f, -0.0148190f, 
		-0.0163020f, 1.01780f, -0.0148190f, -0.0163020f, 1.07427f, -0.0148190f, 
		-0.0163020f, 1.07427f, -0.0148190f, -0.0162960f, 1.01780f, 0.0148220f, 
		-0.0162960f, 1.01780f, 0.0148220f, -0.0162960f, 1.07427f, 0.0148220f, 
		-0.0162960f, 1.07427f, 0.0148220f, -0.0162950f, 1.33363f, -0.00940543f, 
		-0.0162950f, 1.33363f, -0.00940543f, -0.0162950f, 1.33363f, -0.00940543f, 
		-0.0162950f, 1.33363f, -0.00940543f, -0.0162950f, 1.33363f, -0.00940543f, 
		-0.0162950f, 1.33363f, -0.00940543f, -0.0162950f, 1.33363f, -0.00940543f, 
		-0.0162950f, 1.33363f, -0.00940543f, -0.0162950f, 1.33363f, -0.00940543f, 
		-0.0162950f, 1.33363f, -0.00940543f, -0.0162950f, 1.33363f, -0.00940543f, 
		-0.0162909f, 1.33120f, 0.00940848f, -0.0162909f, 1.33120f, 0.00940848f, 
		-0.0162909f, 1.33120f, 0.00940848f, -0.0162909f, 1.33120f, 0.00940848f, 
		-0.0162909f, 1.33120f, 0.00940848f, -0.0162909f, 1.33120f, 0.00940848f, 
		-0.0162909f, 1.33120f, 0.00940848f, -0.0162909f, 1.33120f, 0.00940848f, 
		-0.0162909f, 1.33120f, 0.00940848f, -0.0162909f, 1.33120f, 0.00940848f, 
		-0.0162909f, 1.33120f, 0.00940848f, -0.0162909f, 1.33120f, 0.00940848f, 
		-0.0162909f, 1.33120f, 0.00940848f, -0.0162909f, 1.33120f, 0.00940848f, 
		-0.0162440f, 0.929791f, 0.161891f, -0.0162230f, 1.23705f, -0.0525550f, 
		-0.0162230f, 1.23705f, -0.0525550f, -0.0162230f, 1.26705f, -0.0525550f, 
		-0.0162230f, 1.26705f, -0.0525550f, -0.0162101f, 1.03755f, 0.391511f, 
		-0.0162101f, 1.03755f, 0.391511f, -0.0162101f, 1.03755f, 0.391511f, 
		-0.0162101f, 1.03755f, 0.391511f, -0.0162101f, 1.03755f, 0.391511f, 
		-0.0162101f, 1.03755f, 0.391511f, -0.0162101f, 1.03755f, 0.391511f, 
		-0.0162101f, 1.03755f, 0.391511f, -0.0162101f, 1.03755f, 0.391511f, 
		-0.0162101f, 1.03755f, 0.391511f, -0.0162101f, 1.03755f, 0.391511f, 
		-0.0162101f, 1.03755f, 0.391511f, -0.0162101f, 1.03755f, 0.391511f, 
		-0.0162101f, 1.03755f, 0.391511f, -0.0162098f, 1.05636f, 0.391512f, 
		-0.0162098f, 1.05636f, 0.391512f, -0.0162098f, 1.05636f, 0.391512f, 
		-0.0162098f, 1.05636f, 0.391512f, -0.0162098f, 1.05636f, 0.391512f, 
		-0.0162098f, 1.05636f, 0.391512f, -0.0162098f, 1.05636f, 0.391512f, 
		-0.0162098f, 1.05636f, 0.391512f, -0.0162098f, 1.05636f, 0.391512f, 
		-0.0162098f, 1.05636f, 0.391512f, -0.0162098f, 1.05636f, 0.391512f, 
		-0.0162098f, 1.05636f, 0.391512f, -0.0162098f, 1.05636f, 0.391512f, 
		-0.0162098f, 1.05636f, 0.391512f, -0.0162000f, 1.23705f, 0.0525580f, 
		-0.0162000f, 1.23705f, 0.0525580f, -0.0162000f, 1.26705f, 0.0525580f, 
		-0.0162000f, 1.26705f, 0.0525580f, -0.0161626f, 1.05624f, -0.392407f, 
		-0.0161626f, 1.05624f, -0.392407f, -0.0161626f, 1.05624f, -0.392407f, 
		-0.0161626f, 1.05624f, -0.392407f, -0.0161626f, 1.05624f, -0.392407f, 
		-0.0161626f, 1.05624f, -0.392407f, -0.0161626f, 1.05624f, -0.392407f, 
		-0.0161626f, 1.05624f, -0.392407f, -0.0161626f, 1.05624f, -0.392407f, 
		-0.0161626f, 1.05624f, -0.392407f, -0.0161624f, 1.03767f, -0.392407f, 
		-0.0161624f, 1.03767f, -0.392407f, -0.0161624f, 1.03767f, -0.392407f, 
		-0.0161624f, 1.03767f, -0.392407f, -0.0161624f, 1.03767f, -0.392407f, 
		-0.0161624f, 1.03767f, -0.392407f, -0.0160813f, 1.33039f, -0.00928251f, 
		-0.0160813f, 1.33039f, -0.00928251f, -0.0160813f, 1.33039f, -0.00928251f, 
		-0.0160813f, 1.33039f, -0.00928251f, -0.0160813f, 1.33039f, -0.00928251f, 
		-0.0160813f, 1.33039f, -0.00928251f, -0.0160813f, 1.33039f, -0.00928251f, 
		-0.0160813f, 1.33039f, -0.00928251f, -0.0160773f, 1.33444f, 0.00928510f, 
		-0.0160773f, 1.33444f, 0.00928510f, -0.0160773f, 1.33444f, 0.00928510f, 
		-0.0160773f, 1.33444f, 0.00928510f, -0.0160773f, 1.33444f, 0.00928510f, 
		-0.0160773f, 1.33444f, 0.00928510f, -0.0160773f, 1.33444f, 0.00928510f, 
		-0.0160773f, 1.33444f, 0.00928510f, -0.0160773f, 1.33444f, 0.00928510f, 
		-0.0160773f, 1.33444f, 0.00928510f, -0.0160512f, 1.05618f, -0.376983f, 
		-0.0160512f, 1.05618f, -0.376983f, -0.0160509f, 1.03773f, -0.376983f, 
		-0.0160509f, 1.03773f, -0.376983f, -0.0159977f, 1.03767f, 0.388271f, 
		-0.0159977f, 1.03767f, 0.388271f, -0.0159977f, 1.03767f, 0.388271f, 
		-0.0159977f, 1.03767f, 0.388271f, -0.0159977f, 1.03767f, 0.388271f, 
		-0.0159977f, 1.03767f, 0.388271f, -0.0159977f, 1.03767f, 0.388271f, 
		-0.0159977f, 1.03767f, 0.388271f, -0.0159977f, 1.03767f, 0.388271f, 
		-0.0159977f, 1.03767f, 0.388271f, -0.0159973f, 1.05624f, 0.388271f, 
		-0.0159973f, 1.05624f, 0.388271f, -0.0159973f, 1.05624f, 0.388271f, 
		-0.0159973f, 1.05624f, 0.388271f, -0.0159973f, 1.05624f, 0.388271f, 
		-0.0159973f, 1.05624f, 0.388271f, -0.0159973f, 1.05624f, 0.388271f, 
		-0.0159973f, 1.05624f, 0.388271f, -0.0159973f, 1.05624f, 0.388271f, 
		-0.0159973f, 1.05624f, 0.388271f, -0.0159856f, 0.953366f, 0.277245f, 
		-0.0159856f, 0.953366f, 0.277245f, -0.0159697f, 1.31902f, 0.00922271f, 
		-0.0159697f, 1.31902f, 0.00922271f, -0.0159646f, 1.30330f, -0.00921492f, 
		-0.0159646f, 1.30330f, -0.00921492f, -0.0159646f, 1.30330f, -0.00921492f, 
		-0.0159646f, 1.30330f, -0.00921492f, -0.0159627f, 1.30202f, 0.0195305f, 
		-0.0159627f, 1.30202f, 0.0195305f, -0.0159627f, 1.30202f, 0.0195305f, 
		-0.0158916f, 1.05618f, 0.376898f, -0.0158916f, 1.05618f, 0.376898f, 
		-0.0158914f, 1.03773f, 0.376898f, -0.0158914f, 1.03773f, 0.376898f, 
		-0.0158865f, 1.03774f, 0.361181f, -0.0158865f, 1.03774f, 0.361181f, 
		-0.0158865f, 1.03774f, 0.361181f, -0.0158480f, 0.927433f, -0.165075f, 
		-0.0157780f, 0.927433f, 0.164980f, -0.0157365f, 0.984602f, -0.244105f, 
		-0.0157365f, 0.984602f, -0.244105f, -0.0157190f, 1.01918f, 0.287586f, 
		-0.0157190f, 1.01918f, 0.287586f, -0.0157190f, 1.01918f, 0.287586f, 
		-0.0156660f, 0.843081f, -0.295102f, -0.0156660f, 0.843081f, -0.295102f, 
		-0.0156540f, 0.843081f, -0.241051f, -0.0156540f, 0.843081f, -0.241051f, 
		-0.0155520f, 0.843081f, 0.240956f, -0.0155520f, 0.843081f, 0.240956f, 
		-0.0155410f, 0.843081f, 0.295007f, -0.0155410f, 0.843081f, 0.295007f, 
		-0.0155090f, 0.893445f, -0.294830f, -0.0155090f, 0.893445f, -0.294830f, 
		-0.0154970f, 0.893445f, -0.241323f, -0.0154970f, 0.893445f, -0.241323f, 
		-0.0154217f, 1.00049f, -0.285862f, -0.0154217f, 1.00049f, -0.285862f, 
		-0.0154217f, 1.00049f, -0.285862f, -0.0154217f, 1.00049f, -0.285862f, 
		-0.0153950f, 0.893445f, 0.241228f, -0.0153950f, 0.893445f, 0.241228f, 
		-0.0153840f, 0.893445f, 0.294735f, -0.0153840f, 0.893445f, 0.294735f, 
		-0.0153377f, 1.02272f, 0.240756f, -0.0153377f, 1.02272f, 0.240756f, 
		-0.0152850f, 0.832160f, -0.276868f, -0.0152850f, 0.832160f, -0.276868f, 
		-0.0152820f, 0.832160f, -0.259285f, -0.0152820f, 0.832160f, -0.259285f, 
		-0.0152165f, 1.02264f, -0.285671f, -0.0152165f, 1.02264f, -0.285671f, 
		-0.0151720f, 0.832160f, 0.259190f, -0.0151720f, 0.832160f, 0.259190f, 
		-0.0151680f, 0.832160f, 0.276772f, -0.0151680f, 0.832160f, 0.276772f, 
		-0.0150955f, 1.02264f, 0.285576f, -0.0150955f, 1.02264f, 0.285576f, 
		-0.0150897f, 1.26712f, -0.0478973f, -0.0150897f, 1.26712f, -0.0478973f, 
		-0.0150780f, 1.04604f, -0.369999f, -0.0150780f, 1.04604f, -0.369999f, 
		-0.0150691f, 1.26712f, 0.0479008f, -0.0150691f, 1.26712f, 0.0479008f, 
		-0.0149487f, 0.955074f, -0.276674f, -0.0149487f, 0.955074f, -0.276674f, 
		-0.0149452f, 0.955074f, -0.259480f, -0.0149452f, 0.955074f, -0.259480f, 
		-0.0149410f, 1.58893f, -0.104754f, -0.0149410f, 1.58893f, -0.104754f, 
		-0.0149410f, 1.61893f, -0.104754f, -0.0149410f, 1.61893f, -0.104754f, 
		-0.0149410f, 1.61893f, -0.104754f, -0.0149280f, 1.02019f, -0.0151090f, 
		-0.0149280f, 1.02019f, -0.0151090f, -0.0149280f, 1.07189f, -0.0151090f, 
		-0.0149280f, 1.07189f, -0.0151090f, -0.0149220f, 1.04604f, 0.370001f, 
		-0.0149220f, 1.04604f, 0.370001f, -0.0149210f, 1.02019f, 0.0151120f, 
		-0.0149210f, 1.02019f, 0.0151120f, -0.0149210f, 1.07189f, 0.0151120f, 
		-0.0149210f, 1.07189f, 0.0151120f, -0.0149113f, 1.20497f, -0.0295767f, 
		-0.0148990f, 1.58893f, 0.0932100f, -0.0148990f, 1.58893f, 0.0932100f, 
		-0.0148990f, 1.61893f, 0.0932100f, -0.0148990f, 1.61893f, 0.0932100f, 
		-0.0148990f, 1.61893f, 0.0932100f, -0.0148359f, 0.955074f, 0.259384f, 
		-0.0148359f, 0.955074f, 0.259384f, -0.0147683f, 1.02264f, 0.256104f, 
		-0.0147683f, 1.02264f, 0.256104f, -0.0147012f, 1.02261f, -0.286357f, 
		-0.0147012f, 1.02261f, -0.286357f, -0.0146244f, 1.02272f, -0.256274f, 
		-0.0146244f, 1.02272f, -0.256274f, -0.0145653f, 1.04368f, 0.0306760f, 
		-0.0145653f, 1.04368f, 0.0306760f, -0.0145653f, 1.04368f, 0.0306760f, 
		-0.0145653f, 1.04368f, 0.0306760f, -0.0145597f, 1.20208f, -0.0194324f, 
		-0.0145597f, 1.20208f, -0.0194324f, -0.0145343f, 1.23699f, -0.0480687f, 
		-0.0145343f, 1.23699f, -0.0480687f, -0.0145147f, 1.23699f, 0.0480716f, 
		-0.0145147f, 1.23699f, 0.0480716f, -0.0145060f, 1.02272f, -0.240754f, 
		-0.0145060f, 1.02272f, -0.240754f, -0.0145060f, 1.03132f, -0.240754f, 
		-0.0145060f, 1.03132f, -0.240754f, -0.0143315f, 1.22218f, -0.0463099f, 
		-0.0143126f, 1.22218f, 0.0463113f, -0.0143126f, 1.05517f, -0.386277f, 
		-0.0143126f, 1.05517f, -0.386277f, -0.0143126f, 1.05517f, -0.386277f, 
		-0.0143126f, 1.05517f, -0.386277f, -0.0143126f, 1.05517f, -0.386277f, 
		-0.0143126f, 1.05517f, -0.386277f, -0.0143123f, 1.03874f, -0.386277f, 
		-0.0143123f, 1.03874f, -0.386277f, -0.0143123f, 1.03874f, -0.386277f, 
		-0.0143123f, 1.03874f, -0.386277f, -0.0143123f, 1.03874f, -0.386277f, 
		-0.0143123f, 1.03874f, -0.386277f, -0.0142971f, 1.01135f, -0.291633f, 
		-0.0142746f, 1.05515f, -0.394638f, -0.0142746f, 1.05515f, -0.394638f, 
		-0.0142746f, 1.05515f, -0.394638f, -0.0142746f, 1.05515f, -0.394638f, 
		-0.0142746f, 1.05515f, -0.394638f, -0.0142640f, 1.28192f, -0.0463307f, 
		-0.0142434f, 1.28192f, 0.0463324f, -0.0142325f, 1.32831f, -0.00821524f, 
		-0.0142325f, 1.32831f, -0.00821524f, -0.0142325f, 1.32831f, -0.00821524f, 
		-0.0142325f, 1.32831f, -0.00821524f, -0.0142293f, 1.32831f, 0.00821733f, 
		-0.0142293f, 1.32831f, 0.00821733f, -0.0142293f, 1.32831f, 0.00821733f, 
		-0.0142293f, 1.32831f, 0.00821733f, -0.0142293f, 1.32831f, 0.00821733f, 
		-0.0142293f, 1.32831f, 0.00821733f, -0.0142242f, 1.01154f, -0.249988f, 
		-0.0141927f, 1.33668f, -0.00819205f, -0.0141927f, 1.33668f, -0.00819205f, 
		-0.0141927f, 1.33668f, -0.00819205f, -0.0141927f, 1.33668f, -0.00819205f, 
		-0.0141927f, 1.33668f, -0.00819205f, -0.0141927f, 1.33668f, -0.00819205f, 
		-0.0141896f, 1.33668f, 0.00819405f, -0.0141896f, 1.33668f, 0.00819405f, 
		-0.0141896f, 1.33668f, 0.00819405f, -0.0141896f, 1.33668f, 0.00819405f, 
		-0.0141896f, 1.33668f, 0.00819405f, -0.0141896f, 1.33668f, 0.00819405f, 
		-0.0141721f, 1.01135f, 0.291538f, -0.0141498f, 1.03874f, 0.386191f, 
		-0.0141498f, 1.03874f, 0.386191f, -0.0141498f, 1.03874f, 0.386191f, 
		-0.0141498f, 1.03874f, 0.386191f, -0.0141498f, 1.03874f, 0.386191f, 
		-0.0141498f, 1.03874f, 0.386191f, -0.0141496f, 1.05517f, 0.386191f, 
		-0.0141496f, 1.05517f, 0.386191f, -0.0141496f, 1.05517f, 0.386191f, 
		-0.0141496f, 1.05517f, 0.386191f, -0.0141496f, 1.05517f, 0.386191f, 
		-0.0141496f, 1.05517f, 0.386191f, -0.0141260f, 1.58880f, -0.0960445f, 
		-0.0141260f, 1.58880f, -0.0960445f, -0.0141189f, 1.01154f, 0.249893f, 
		-0.0141189f, 1.01154f, 0.249893f, -0.0141189f, 1.01154f, 0.249893f, 
		-0.0141077f, 1.03876f, 0.394552f, -0.0141077f, 1.03876f, 0.394552f, 
		-0.0141077f, 1.03876f, 0.394552f, -0.0141077f, 1.03876f, 0.394552f, 
		-0.0141077f, 1.03876f, 0.394552f, -0.0141077f, 1.03876f, 0.394552f, 
		-0.0141074f, 1.05515f, 0.394552f, -0.0141074f, 1.05515f, 0.394552f, 
		-0.0141074f, 1.05515f, 0.394552f, -0.0141074f, 1.05515f, 0.394552f, 
		-0.0141074f, 1.05515f, 0.394552f, -0.0141074f, 1.05515f, 0.394552f, 
		-0.0140876f, 1.58880f, 0.0844997f, -0.0140876f, 1.58880f, 0.0844997f, 
		-0.0140526f, 1.04604f, -0.0306731f, -0.0140526f, 1.04604f, -0.0306731f, 
		-0.0140526f, 1.04604f, -0.0306731f, -0.0138930f, 1.05065f, -0.383096f, 
		-0.0138930f, 1.05065f, -0.383096f, -0.0138930f, 1.05065f, -0.383096f, 
		-0.0138930f, 1.05065f, -0.383096f, -0.0138930f, 1.05065f, -0.383096f, 
		-0.0138930f, 1.05065f, -0.383096f, -0.0138930f, 1.05065f, -0.383096f, 
		-0.0138930f, 1.05065f, -0.383096f, -0.0138930f, 1.05065f, -0.383096f, 
		-0.0138930f, 1.05065f, -0.383096f, -0.0138930f, 1.05065f, -0.383096f, 
		-0.0138930f, 1.05065f, -0.383096f, -0.0138930f, 1.05065f, -0.383096f, 
		-0.0138930f, 1.05065f, -0.383096f, -0.0138780f, 0.896718f, 0.243853f, 
		-0.0138780f, 0.896718f, 0.243853f, -0.0138680f, 0.896718f, 0.292108f, 
		-0.0138680f, 0.896718f, 0.292108f, -0.0138154f, 1.04695f, -0.379334f, 
		-0.0138154f, 1.04695f, -0.379334f, -0.0138154f, 1.04695f, -0.379334f, 
		-0.0138154f, 1.04695f, -0.379334f, -0.0138154f, 1.04695f, -0.379334f, 
		-0.0138154f, 1.04695f, -0.379334f, -0.0138123f, 1.32513f, -0.00369995f, 
		-0.0138123f, 1.32513f, -0.00369995f, -0.0138123f, 1.32513f, -0.00369995f, 
		-0.0138123f, 1.32513f, -0.00369995f, -0.0138123f, 1.32513f, -0.00369995f, 
		-0.0138123f, 1.32513f, -0.00369995f, -0.0138123f, 1.32513f, -0.00369995f, 
		-0.0138123f, 1.32513f, -0.00369995f, -0.0138123f, 1.32513f, -0.00369995f, 
		-0.0138123f, 1.32513f, -0.00369995f, -0.0138123f, 1.32513f, -0.00369995f, 
		-0.0137454f, 1.03132f, 0.239989f, -0.0137454f, 1.03132f, 0.239989f, 
		-0.0137454f, 1.03132f, 0.239989f, -0.0137454f, 1.03132f, 0.239989f, 
		-0.0137345f, 1.32137f, 1.03241e-006f, -0.0137345f, 1.32137f, 1.03241e-006f, 
		-0.0137345f, 1.32137f, 1.03241e-006f, -0.0137345f, 1.32137f, 1.03241e-006f, 
		-0.0137345f, 1.32137f, 1.03241e-006f, -0.0137345f, 1.32137f, 1.03241e-006f, 
		-0.0137323f, 1.03132f, 0.301359f, -0.0137323f, 1.03132f, 0.301359f, 
		-0.0137323f, 1.03132f, 0.301359f, -0.0137323f, 1.03132f, 0.301359f, 
		-0.0137305f, 1.05066f, 0.383010f, -0.0137305f, 1.05066f, 0.383010f, 
		-0.0137305f, 1.05066f, 0.383010f, -0.0137305f, 1.05066f, 0.383010f, 
		-0.0137305f, 1.05066f, 0.383010f, -0.0137305f, 1.05066f, 0.383010f, 
		-0.0137305f, 1.05066f, 0.383010f, -0.0137305f, 1.05066f, 0.383010f, 
		-0.0137305f, 1.05066f, 0.383010f, -0.0137305f, 1.05066f, 0.383010f, 
		-0.0137305f, 1.05066f, 0.383010f, -0.0136545f, 1.04695f, 0.379248f, 
		-0.0136545f, 1.04695f, 0.379248f, -0.0136545f, 1.04695f, 0.379248f, 
		-0.0136545f, 1.04695f, 0.379248f, -0.0136545f, 1.04695f, 0.379248f, 
		-0.0136545f, 1.04695f, 0.379248f, -0.0136095f, 1.00523f, 0.250830f, 
		-0.0136095f, 1.00523f, 0.250830f, -0.0136095f, 1.00523f, 0.250830f, 
		-0.0136095f, 1.00523f, 0.250830f, -0.0136095f, 1.00523f, 0.250830f, 
		-0.0135699f, 0.897585f, -0.291473f, -0.0135699f, 0.897585f, -0.291473f, 
		-0.0135610f, 0.977001f, -0.241934f, -0.0135601f, 0.897585f, -0.244680f, 
		-0.0135601f, 0.897585f, -0.244680f, -0.0135466f, 1.03766f, -0.394670f, 
		-0.0135466f, 1.03766f, -0.394670f, -0.0135466f, 1.03766f, -0.394670f, 
		-0.0135466f, 1.03766f, -0.394670f, -0.0135466f, 1.03766f, -0.394670f, 
		-0.0135466f, 1.03766f, -0.394670f, -0.0135466f, 1.03766f, -0.394670f, 
		-0.0135466f, 1.03766f, -0.394670f, -0.0135466f, 1.03766f, -0.394670f, 
		-0.0135466f, 1.03766f, -0.394670f, -0.0134966f, 1.03353f, -0.361265f, 
		-0.0134966f, 1.03353f, -0.361265f, -0.0134966f, 1.03353f, -0.361265f, 
		-0.0134966f, 1.03353f, -0.361265f, -0.0134777f, 0.984647f, -0.300778f, 
		-0.0134777f, 0.984647f, -0.300778f, -0.0134777f, 0.984647f, -0.300778f, 
		-0.0134777f, 0.984647f, -0.300778f, -0.0134556f, 1.00044f, -0.312463f, 
		-0.0134556f, 1.00044f, -0.312463f, -0.0133921f, 0.956459f, 0.275746f, 
		-0.0133921f, 0.956459f, 0.275746f, -0.0133660f, 1.57721f, -0.0943190f, 
		-0.0133290f, 1.57721f, 0.0827740f, -0.0133237f, 1.00044f, 0.312366f, 
		-0.0133237f, 1.00044f, 0.312366f, -0.0130850f, 0.951668f, -0.183433f, 
		-0.0130690f, 1.03854f, -0.369999f, -0.0130690f, 1.03854f, -0.369999f, 
		-0.0130690f, 1.05354f, -0.369999f, -0.0130690f, 1.05354f, -0.369999f, 
		-0.0130070f, 0.951668f, 0.183336f, -0.0129609f, 0.985827f, 0.238669f, 
		-0.0129407f, 1.05553f, 0.0306186f, -0.0129407f, 1.05553f, 0.0306186f, 
		-0.0129407f, 1.05553f, 0.0306186f, -0.0129120f, 1.03854f, 0.370001f, 
		-0.0129120f, 1.03854f, 0.370001f, -0.0129120f, 1.05354f, 0.370001f, 
		-0.0129120f, 1.05354f, 0.370001f, -0.0128665f, 0.980923f, -0.299566f, 
		-0.0127779f, 0.957938f, 0.269493f, -0.0127779f, 0.957938f, 0.269493f, 
		-0.0127779f, 0.957938f, 0.269493f, -0.0127490f, 1.57086f, -0.0902270f, 
		-0.0127413f, 0.980923f, 0.299469f, -0.0127130f, 1.57086f, 0.0786820f, 
		-0.0127120f, 0.948656f, -0.185913f, -0.0126330f, 0.948656f, 0.185817f, 
		-0.0126257f, 1.00054f, 0.288226f, -0.0126257f, 1.00054f, 0.288226f, 
		-0.0126208f, 0.958007f, -0.267907f, -0.0126208f, 0.958007f, -0.267907f, 
		-0.0126208f, 0.958007f, -0.267907f, -0.0126208f, 0.958007f, -0.267907f, 
		-0.0126208f, 0.958007f, -0.267907f, -0.0125660f, 0.836885f, -0.289736f, 
		-0.0125660f, 0.836885f, -0.289736f, -0.0125652f, 1.00045f, -0.312254f, 
		-0.0125652f, 1.00045f, -0.312254f, -0.0125570f, 0.836885f, -0.246418f, 
		-0.0125570f, 0.836885f, -0.246418f, -0.0125535f, 0.899571f, 0.246147f, 
		-0.0125535f, 0.899571f, 0.246147f, -0.0125477f, 1.00045f, -0.229288f, 
		-0.0125477f, 1.00045f, -0.229288f, -0.0125445f, 0.899571f, 0.289814f, 
		-0.0125445f, 0.899571f, 0.289814f, -0.0125202f, 1.20210f, 0.0205875f, 
		-0.0125202f, 1.20210f, 0.0205875f, -0.0125085f, 1.02542f, -0.368415f, 
		-0.0125085f, 1.02542f, -0.368415f, -0.0125080f, 1.06848f, -0.368415f, 
		-0.0125080f, 1.06848f, -0.368415f, -0.0124530f, 0.836885f, 0.246322f, 
		-0.0124530f, 0.836885f, 0.246322f, -0.0124504f, 1.00045f, 0.229192f, 
		-0.0124504f, 1.00045f, 0.229192f, -0.0124430f, 0.836885f, 0.289639f, 
		-0.0124430f, 0.836885f, 0.289639f, -0.0124348f, 1.31045f, -0.0215294f, 
		-0.0124348f, 1.31045f, -0.0215294f, -0.0124321f, 1.00045f, 0.312157f, 
		-0.0124321f, 1.00045f, 0.312157f, -0.0124257f, 1.31045f, 0.0215305f, 
		-0.0124257f, 1.31045f, 0.0215305f, -0.0123524f, 1.02542f, 0.368327f, 
		-0.0123524f, 1.02542f, 0.368327f, -0.0123521f, 1.06848f, 0.368327f, 
		-0.0123521f, 1.06848f, 0.368327f, -0.0123474f, 1.02261f, -0.253355f, 
		-0.0123474f, 1.02261f, -0.253355f, -0.0123474f, 1.02261f, -0.253355f, 
		-0.0123474f, 1.02261f, -0.253355f, -0.0123349f, 1.21087f, -0.0394896f, 
		-0.0123310f, 1.31336f, 0.0213660f, -0.0123310f, 1.31336f, 0.0213660f, 
		-0.0123176f, 1.21087f, 0.0394919f, -0.0123009f, 1.02579f, -0.372086f, 
		-0.0123009f, 1.02579f, -0.372086f, -0.0123005f, 1.06812f, -0.372086f, 
		-0.0123005f, 1.06812f, -0.372086f, -0.0122628f, 1.04695f, -0.385763f, 
		-0.0122628f, 1.04695f, -0.385763f, -0.0122628f, 1.04695f, -0.385763f, 
		-0.0122628f, 1.04695f, -0.385763f, -0.0122628f, 1.04695f, -0.385763f, 
		-0.0122570f, 1.02559f, 0.371234f, -0.0122570f, 1.02559f, 0.371234f, 
		-0.0122570f, 1.06832f, 0.371234f, -0.0122570f, 1.06832f, 0.371234f, 
		-0.0122262f, 1.31412f, -0.0211686f, -0.0122262f, 1.31412f, -0.0211686f, 
		-0.0121992f, 1.02480f, -0.0207481f, -0.0121992f, 1.02480f, -0.0207481f, 
		-0.0121992f, 1.02480f, -0.0207481f, -0.0121992f, 1.02480f, -0.0207481f, 
		-0.0121992f, 1.02480f, -0.0207481f, -0.0121992f, 1.02480f, -0.0207481f, 
		-0.0121992f, 1.02480f, -0.0207481f, -0.0121992f, 1.02480f, -0.0207481f, 
		-0.0121992f, 1.02480f, -0.0207481f, -0.0121902f, 1.02480f, 0.0207499f, 
		-0.0121902f, 1.02480f, 0.0207499f, -0.0121902f, 1.02480f, 0.0207499f, 
		-0.0121902f, 1.02480f, 0.0207499f, -0.0121902f, 1.02480f, 0.0207499f, 
		-0.0121902f, 1.02480f, 0.0207499f, -0.0121902f, 1.02480f, 0.0207499f, 
		-0.0121902f, 1.02480f, 0.0207499f, -0.0121684f, 1.06731f, 0.0198874f, 
		-0.0121684f, 1.06731f, 0.0198874f, -0.0121684f, 1.06731f, 0.0198874f, 
		-0.0121684f, 1.06731f, 0.0198874f, -0.0121684f, 1.06731f, 0.0198874f, 
		-0.0121684f, 1.06731f, 0.0198874f, -0.0121684f, 1.06731f, 0.0198874f, 
		-0.0121684f, 1.06731f, 0.0198874f, -0.0121486f, 1.06713f, -0.0212289f, 
		-0.0121486f, 1.06713f, -0.0212289f, -0.0121486f, 1.06713f, -0.0212289f, 
		-0.0121486f, 1.06713f, -0.0212289f, -0.0121078f, 1.04001f, -0.383070f, 
		-0.0121078f, 1.04001f, -0.383070f, -0.0121078f, 1.04001f, -0.383070f, 
		-0.0121078f, 1.04001f, -0.383070f, -0.0121078f, 1.04001f, -0.383070f, 
		-0.0121078f, 1.04001f, -0.383070f, -0.0121078f, 1.04001f, -0.383070f, 
		-0.0121078f, 1.04001f, -0.383070f, -0.0120992f, 1.04695f, 0.385676f, 
		-0.0120992f, 1.04695f, 0.385676f, -0.0120992f, 1.04695f, 0.385676f, 
		-0.0120992f, 1.04695f, 0.385676f, -0.0120992f, 1.04695f, 0.385676f, 
		-0.0120892f, 1.32782f, -0.00323850f, -0.0120892f, 1.32782f, -0.00323850f, 
		-0.0120892f, 1.32782f, -0.00323850f, -0.0120892f, 1.32782f, -0.00323850f, 
		-0.0120892f, 1.32782f, -0.00323850f, -0.0120703f, 1.29323f, -0.0395714f, 
		-0.0120535f, 1.29323f, 0.0395735f, -0.0120259f, 1.32511f, 0.00694445f, 
		-0.0120259f, 1.32511f, 0.00694445f, -0.0120259f, 1.32511f, 0.00694445f, 
		-0.0120259f, 1.32511f, 0.00694445f, -0.0120259f, 1.32511f, 0.00694445f, 
		-0.0120259f, 1.32511f, 0.00694445f, -0.0120259f, 1.32511f, 0.00694445f, 
		-0.0119770f, 1.02530f, -0.0243280f, -0.0119770f, 1.02530f, -0.0243280f, 
		-0.0119770f, 1.06677f, -0.0243280f, -0.0119770f, 1.06677f, -0.0243280f, 
		-0.0119749f, 1.05382f, -0.379335f, -0.0119749f, 1.05382f, -0.379335f, 
		-0.0119749f, 1.05382f, -0.379335f, -0.0119749f, 1.05382f, -0.379335f, 
		-0.0119749f, 1.05382f, -0.379335f, -0.0119749f, 1.05382f, -0.379335f, 
		-0.0119748f, 1.04009f, -0.379335f, -0.0119748f, 1.04009f, -0.379335f, 
		-0.0119748f, 1.04009f, -0.379335f, -0.0119748f, 1.04009f, -0.379335f, 
		-0.0119748f, 1.04009f, -0.379335f, -0.0119748f, 1.04009f, -0.379335f, 
		-0.0119748f, 1.04009f, -0.379335f, -0.0119748f, 1.04009f, -0.379335f, 
		-0.0119748f, 1.04009f, -0.379335f, -0.0119748f, 1.04009f, -0.379335f, 
		-0.0119660f, 1.02530f, 0.0243290f, -0.0119660f, 1.02530f, 0.0243290f, 
		-0.0119660f, 1.06677f, 0.0243290f, -0.0119660f, 1.06677f, 0.0243290f, 
		-0.0119465f, 1.06751f, -0.364732f, -0.0119465f, 1.06751f, -0.364732f, 
		-0.0119464f, 1.02640f, -0.364732f, -0.0119464f, 1.02640f, -0.364732f, 
		-0.0119460f, 1.04001f, 0.382982f, -0.0119460f, 1.04001f, 0.382982f, 
		-0.0119460f, 1.04001f, 0.382982f, -0.0119460f, 1.04001f, 0.382982f, 
		-0.0119460f, 1.04001f, 0.382982f, -0.0119460f, 1.04001f, 0.382982f, 
		-0.0119460f, 1.04001f, 0.382982f, -0.0119460f, 1.04001f, 0.382982f, 
		-0.0118962f, 1.32137f, -0.00686651f, -0.0118962f, 1.32137f, -0.00686651f, 
		-0.0118962f, 1.32137f, -0.00686651f, -0.0118962f, 1.32137f, -0.00686651f, 
		-0.0118962f, 1.32137f, -0.00686651f, -0.0118962f, 1.32137f, -0.00686651f, 
		-0.0118933f, 1.32137f, 0.00686802f, -0.0118933f, 1.32137f, 0.00686802f, 
		-0.0118933f, 1.32137f, 0.00686802f, -0.0118933f, 1.32137f, 0.00686802f, 
		-0.0118933f, 1.32137f, 0.00686802f, -0.0118933f, 1.32137f, 0.00686802f, 
		-0.0118933f, 1.32137f, 0.00686802f, -0.0118933f, 1.32137f, 0.00686802f, 
		-0.0118933f, 1.32137f, 0.00686802f, -0.0118933f, 1.32137f, 0.00686802f, 
		-0.0118734f, 1.30677f, -0.0205581f, -0.0118734f, 1.30677f, -0.0205581f, 
		-0.0118707f, 0.901703f, -0.288532f, -0.0118707f, 0.901703f, -0.288532f, 
		-0.0118655f, 1.31559f, 0.0205591f, -0.0118655f, 1.31559f, 0.0205591f, 
		-0.0118654f, 1.30677f, 0.0205591f, -0.0118654f, 1.30677f, 0.0205591f, 
		-0.0118623f, 0.901703f, -0.247622f, -0.0118623f, 0.901703f, -0.247622f, 
		-0.0118143f, 1.04009f, 0.379248f, -0.0118143f, 1.04009f, 0.379248f, 
		-0.0118143f, 1.04009f, 0.379248f, -0.0118143f, 1.04009f, 0.379248f, 
		-0.0118143f, 1.04009f, 0.379248f, -0.0118143f, 1.04009f, 0.379248f, 
		-0.0118143f, 1.04009f, 0.379248f, -0.0118143f, 1.04009f, 0.379248f, 
		-0.0118143f, 1.04009f, 0.379248f, -0.0118143f, 1.04009f, 0.379248f, 
		-0.0118142f, 1.05382f, 0.379248f, -0.0118142f, 1.05382f, 0.379248f, 
		-0.0118142f, 1.05382f, 0.379248f, -0.0118142f, 1.05382f, 0.379248f, 
		-0.0118142f, 1.05382f, 0.379248f, -0.0118142f, 1.05382f, 0.379248f, 
		-0.0117926f, 1.06751f, 0.364644f, -0.0117926f, 1.06751f, 0.364644f, 
		-0.0117925f, 1.02640f, 0.364644f, -0.0117925f, 1.02640f, 0.364644f, 
		-0.0117904f, 1.02640f, 0.373463f, -0.0117904f, 1.02640f, 0.373463f, 
		-0.0117904f, 1.06751f, 0.373463f, -0.0117904f, 1.06751f, 0.373463f, 
		-0.0117443f, 0.989732f, -0.307948f, -0.0117443f, 0.989732f, -0.307948f, 
		-0.0117443f, 0.989732f, -0.307948f, -0.0117291f, 0.989732f, -0.233592f, 
		-0.0116637f, 1.05006f, -0.377038f, -0.0116637f, 1.05006f, -0.377038f, 
		-0.0116637f, 1.05006f, -0.377038f, -0.0116637f, 1.05006f, -0.377038f, 
		-0.0116637f, 1.05006f, -0.377038f, -0.0116637f, 1.05006f, -0.377038f, 
		-0.0116637f, 1.05006f, -0.377038f, -0.0116300f, 0.989730f, 0.233497f, 
		-0.0116137f, 0.989731f, 0.307851f, -0.0115841f, 1.31908f, -0.00310322f, 
		-0.0115841f, 1.31908f, -0.00310322f, -0.0115841f, 1.31908f, -0.00310322f, 
		-0.0115841f, 1.31908f, -0.00310322f, -0.0115841f, 1.31908f, -0.00310322f, 
		-0.0115841f, 1.31908f, -0.00310322f, -0.0115042f, 1.05006f, 0.376951f, 
		-0.0115042f, 1.05006f, 0.376951f, -0.0115042f, 1.05006f, 0.376951f, 
		-0.0115042f, 1.05006f, 0.376951f, -0.0115042f, 1.05006f, 0.376951f, 
		-0.0115042f, 1.05006f, 0.376951f, -0.0113936f, 1.06472f, -0.0151915f, 
		-0.0113936f, 1.06472f, -0.0151915f, -0.0113531f, 1.31695f, -0.0197142f, 
		-0.0113531f, 1.31695f, -0.0197142f, -0.0112514f, 1.02656f, -0.0274647f, 
		-0.0112514f, 1.02656f, -0.0274647f, -0.0112512f, 1.06551f, -0.0274647f, 
		-0.0112512f, 1.06551f, -0.0274647f, -0.0112399f, 1.06551f, 0.0274646f, 
		-0.0112399f, 1.06551f, 0.0274646f, -0.0112398f, 1.02656f, 0.0274646f, 
		-0.0112398f, 1.02656f, 0.0274646f, -0.0111053f, 1.53200f, -0.00300061f, 
		-0.0111053f, 1.53200f, -0.00300061f, -0.0111053f, 1.53200f, -0.00300061f, 
		-0.0111053f, 1.53200f, -0.00300061f, -0.0111053f, 1.53200f, -0.00300061f, 
		-0.0111052f, 1.33919f, -0.00300065f, -0.0111052f, 1.33919f, -0.00300065f, 
		-0.0111052f, 1.33919f, -0.00300065f, -0.0111052f, 1.33919f, -0.00300065f, 
		-0.0111052f, 1.17999f, -0.00300065f, -0.0111052f, 1.17999f, -0.00300065f, 
		-0.0111052f, 1.17999f, -0.00300065f, -0.0111051f, 1.04914f, 0.418653f, 
		-0.0111051f, 1.04914f, 0.418653f, -0.0111051f, 1.04914f, 0.418653f, 
		-0.0110801f, 0.903956f, 0.248698f, -0.0110801f, 0.903956f, 0.248698f, 
		-0.0110715f, 0.903956f, 0.287262f, -0.0110715f, 0.903956f, 0.287262f, 
		-0.0110389f, 1.56290f, -0.0805093f, -0.0110063f, 1.56290f, 0.0689622f, 
		-0.0109956f, 1.02721f, -0.375026f, -0.0109956f, 1.02721f, -0.375026f, 
		-0.0109954f, 1.06669f, -0.375026f, -0.0109954f, 1.06669f, -0.375026f, 
		-0.0109648f, 1.04913f, -0.421259f, -0.0109648f, 1.04913f, -0.421259f, 
		-0.0109648f, 1.04913f, -0.421259f, -0.0108263f, 1.02834f, -0.362444f, 
		-0.0108263f, 1.02834f, -0.362444f, -0.0108261f, 1.06557f, -0.362444f, 
		-0.0108261f, 1.06557f, -0.362444f, -0.0107817f, 0.980923f, -0.239598f, 
		-0.0107761f, 1.06514f, 0.0153067f, -0.0107761f, 1.06514f, 0.0153067f, 
		-0.0107745f, 1.01796f, -0.249321f, -0.0107745f, 1.01796f, -0.249321f, 
		-0.0107745f, 1.01796f, -0.249321f, -0.0107539f, 1.30448f, -0.0186181f, 
		-0.0107539f, 1.30448f, -0.0186181f, -0.0107459f, 1.30448f, 0.0186192f, 
		-0.0107459f, 1.30448f, 0.0186192f, -0.0107458f, 1.31788f, 0.0186193f, 
		-0.0107458f, 1.31788f, 0.0186193f, -0.0106927f, 1.03132f, 0.302551f, 
		-0.0106927f, 1.03132f, 0.302551f, -0.0106927f, 1.03132f, 0.302551f, 
		-0.0106805f, 0.980923f, 0.239501f, -0.0106729f, 1.06557f, 0.362357f, 
		-0.0106729f, 1.06557f, 0.362357f, -0.0106727f, 1.02834f, 0.362357f, 
		-0.0106727f, 1.02834f, 0.362357f, -0.0106699f, 1.02834f, 0.375750f, 
		-0.0106699f, 1.02834f, 0.375750f, -0.0106698f, 1.01796f, 0.249224f, 
		-0.0106696f, 1.06557f, 0.375750f, -0.0106696f, 1.06557f, 0.375750f, 
		-0.0106536f, 0.906317f, -0.286426f, -0.0106536f, 0.906317f, -0.286426f, 
		-0.0106461f, 0.906317f, -0.249729f, -0.0106461f, 0.906317f, -0.249729f, 
		-0.0106253f, 1.30071f, -0.0280146f, -0.0106037f, 1.08461f, -0.414820f, 
		-0.0106037f, 1.08461f, -0.414820f, -0.0106037f, 1.08461f, -0.414820f, 
		-0.0106037f, 1.08461f, -0.414820f, -0.0105997f, 1.04744f, -0.422099f, 
		-0.0105997f, 1.04744f, -0.422099f, -0.0105997f, 1.04744f, -0.422099f, 
		-0.0105997f, 1.04744f, -0.422099f, -0.0105084f, 1.17740f, 0.00155812f, 
		-0.0105084f, 1.17740f, 0.00155812f, -0.0105084f, 1.33660f, 0.00155821f, 
		-0.0105084f, 1.33660f, 0.00155821f, -0.0105084f, 1.33660f, 0.00155821f, 
		-0.0105084f, 1.33660f, 0.00155821f, -0.0105081f, 1.52940f, 0.00155722f, 
		-0.0105081f, 1.52940f, 0.00155722f, -0.0104283f, 1.08461f, 0.414723f, 
		-0.0104283f, 1.08461f, 0.414723f, -0.0104283f, 1.08461f, 0.414723f, 
		-0.0104283f, 1.08461f, 0.414723f, -0.0104283f, 1.08461f, 0.414723f, 
		-0.0104047f, 1.04845f, 0.421908f, -0.0104047f, 1.04845f, 0.421908f, 
		-0.0104047f, 1.04845f, 0.421908f, -0.0104047f, 1.04845f, 0.421908f, 
		-0.0104047f, 1.04845f, 0.421908f, -0.0103883f, 1.06308f, -0.267997f, 
		-0.0103883f, 1.06308f, -0.267997f, -0.0103883f, 1.06308f, -0.267997f, 
		-0.0103883f, 1.06308f, -0.267997f, -0.0103883f, 1.06308f, -0.267997f, 
		-0.0103883f, 1.06308f, -0.267997f, -0.0103883f, 1.06308f, -0.267997f, 
		-0.0103883f, 1.06308f, -0.267997f, -0.0103363f, 1.29913f, -0.0314692f, 
		-0.0103360f, 1.04103f, -0.377062f, -0.0103360f, 1.04103f, -0.377062f, 
		-0.0103360f, 1.04103f, -0.377062f, -0.0103360f, 1.04103f, -0.377062f, 
		-0.0103360f, 1.04103f, -0.377062f, -0.0103352f, 1.25481f, -0.0569510f, 
		-0.0103352f, 1.25481f, -0.0569510f, -0.0103352f, 1.25481f, -0.0569510f, 
		-0.0103352f, 1.25481f, -0.0569510f, -0.0103352f, 1.25481f, -0.0569510f, 
		-0.0103352f, 1.25481f, -0.0569510f, -0.0103223f, 1.29913f, 0.0314695f, 
		-0.0103198f, 1.25482f, 0.0568018f, -0.0103198f, 1.25482f, 0.0568018f, 
		-0.0103198f, 1.25482f, 0.0568018f, -0.0103198f, 1.25482f, 0.0568018f, 
		-0.0103198f, 1.25482f, 0.0568018f, -0.0103198f, 1.25482f, 0.0568018f, 
		-0.0102751f, 1.06308f, 0.267900f, -0.0102751f, 1.06308f, 0.267900f, 
		-0.0102751f, 1.06308f, 0.267900f, -0.0102751f, 1.06308f, 0.267900f, 
		-0.0102751f, 1.06308f, 0.267900f, -0.0102751f, 1.06308f, 0.267900f, 
		-0.0102751f, 1.06308f, 0.267900f, -0.0102751f, 1.06308f, 0.267900f, 
		-0.0102751f, 1.06308f, 0.267900f, -0.0102751f, 1.06308f, 0.267900f, 
		-0.0102578f, 1.25489f, -0.0670970f, -0.0102578f, 1.25489f, -0.0670970f, 
		-0.0102578f, 1.25489f, -0.0670970f, -0.0102578f, 1.25489f, -0.0670970f, 
		-0.0102578f, 1.25489f, -0.0670970f, -0.0102578f, 1.25489f, -0.0670970f, 
		-0.0102540f, 1.31910f, 0.00592109f, -0.0102540f, 1.31910f, 0.00592109f, 
		-0.0102540f, 1.31910f, 0.00592109f, -0.0102540f, 1.31910f, 0.00592109f, 
		-0.0102540f, 1.31910f, 0.00592109f, -0.0102051f, 1.05624f, -0.0306757f, 
		-0.0102051f, 1.05624f, -0.0306757f, -0.0102051f, 1.05624f, -0.0306757f, 
		-0.0102051f, 1.05624f, -0.0306757f, -0.0102051f, 1.03584f, -0.0306757f, 
		-0.0102051f, 1.03584f, -0.0306757f, -0.0102051f, 1.03584f, -0.0306757f, 
		-0.0102051f, 1.03584f, -0.0306757f, -0.0102051f, 1.03584f, -0.0306757f, 
		-0.0101772f, 1.04103f, 0.376975f, -0.0101772f, 1.04103f, 0.376975f, 
		-0.0101772f, 1.04103f, 0.376975f, -0.0101772f, 1.04103f, 0.376975f, 
		-0.0101772f, 1.04103f, 0.376975f, -0.0101686f, 1.33042f, -0.0743462f, 
		-0.0101686f, 1.33042f, -0.0743462f, -0.0101686f, 1.33042f, -0.0743462f, 
		-0.0101686f, 1.33042f, -0.0743462f, -0.0101686f, 1.33042f, -0.0743462f, 
		-0.0101654f, 1.07317f, 0.273296f, -0.0101654f, 1.07317f, 0.273296f, 
		-0.0101654f, 1.07317f, 0.273296f, -0.0101654f, 1.07317f, 0.273296f, 
		-0.0101654f, 1.07317f, 0.273296f, -0.0101654f, 1.07317f, 0.273296f, 
		-0.0101638f, 1.01837f, -0.291938f, -0.0101598f, 0.908777f, 0.250291f, 
		-0.0101598f, 0.908777f, 0.250291f, -0.0101517f, 0.908777f, 0.285669f, 
		-0.0101517f, 0.908777f, 0.285669f, -0.0101510f, 1.25489f, 0.0668892f, 
		-0.0101510f, 1.25489f, 0.0668892f, -0.0101510f, 1.25489f, 0.0668892f, 
		-0.0101510f, 1.25489f, 0.0668892f, -0.0101510f, 1.25489f, 0.0668892f, 
		-0.0101510f, 1.25489f, 0.0668892f, -0.0101486f, 0.958026f, 0.260636f, 
		-0.0101486f, 0.958026f, 0.260636f, -0.0101486f, 0.958026f, 0.260636f, 
		-0.0101486f, 0.958026f, 0.260636f, -0.0101486f, 0.958026f, 0.260636f, 
		-0.0101093f, 1.55685f, -0.0651894f, -0.0100654f, 0.911318f, -0.285407f, 
		-0.0100654f, 0.911318f, -0.285407f, -0.0100654f, 0.911318f, -0.285407f, 
		-0.0100600f, 0.948080f, -0.285398f, -0.0100600f, 0.948080f, -0.285398f, 
		-0.0100584f, 0.911318f, -0.250748f, -0.0100584f, 0.911318f, -0.250748f, 
		-0.0100584f, 0.911318f, -0.250748f, -0.0100468f, 0.949218f, -0.250768f, 
		-0.0100468f, 0.949218f, -0.250768f, -0.00995559f, 1.07215f, -0.270602f, 
		-0.00995559f, 1.07215f, -0.270602f, -0.00995559f, 1.07215f, -0.270602f, 
		-0.00995559f, 1.07215f, -0.270602f, -0.00994700f, 0.912612f, 0.250659f, 
		-0.00994700f, 0.912612f, 0.250659f, -0.00994700f, 0.912612f, 0.250659f, 
		-0.00994076f, 0.949218f, 0.250670f, -0.00994076f, 0.949218f, 0.250670f, 
		-0.00994000f, 0.912612f, 0.285301f, -0.00994000f, 0.912612f, 0.285301f, 
		-0.00994000f, 0.912612f, 0.285301f, -0.00993869f, 1.06411f, -0.0151680f, 
		-0.00993869f, 1.06411f, -0.0151680f, -0.00993869f, 1.06411f, -0.0151680f, 
		-0.00993373f, 0.949218f, 0.285290f, -0.00993373f, 0.949218f, 0.285290f, 
		-0.00991216f, 1.33277f, 0.0769033f, -0.00991216f, 1.33277f, 0.0769033f, 
		-0.00991216f, 1.33277f, 0.0769033f, -0.00988400f, 1.03132f, -0.302606f, 
		-0.00988400f, 1.03132f, -0.302606f, -0.00988400f, 1.03132f, -0.302606f, 
		-0.00988140f, 1.18213f, 0.00579502f, -0.00988140f, 1.18213f, 0.00579502f, 
		-0.00988140f, 1.18213f, 0.00579502f, -0.00979567f, 0.951393f, -0.284939f, 
		-0.00979567f, 0.951393f, -0.284939f, -0.00978098f, 1.06376f, -0.376851f, 
		-0.00978098f, 1.06376f, -0.376851f, -0.00978086f, 1.03015f, -0.376851f, 
		-0.00978086f, 1.03015f, -0.376851f, -0.00977908f, 1.25207f, -0.0704407f, 
		-0.00977908f, 1.25207f, -0.0704407f, -0.00977908f, 1.25207f, -0.0704407f, 
		-0.00977908f, 1.25207f, -0.0704407f, -0.00977908f, 1.25207f, -0.0704407f, 
		-0.00977908f, 1.25207f, -0.0704407f, -0.00977908f, 1.25207f, -0.0704407f, 
		-0.00977908f, 1.25207f, -0.0704407f, -0.00977908f, 1.25207f, -0.0704407f, 
		-0.00972551f, 1.04905f, -0.415028f, -0.00972551f, 1.04905f, -0.415028f, 
		-0.00972551f, 1.04905f, -0.415028f, -0.00972551f, 1.04905f, -0.415028f, 
		-0.00969329f, 1.05183f, -0.417394f, -0.00969329f, 1.05183f, -0.417394f, 
		-0.00969329f, 1.05183f, -0.417394f, -0.00963158f, 0.957053f, -0.256449f, 
		-0.00963158f, 0.957053f, -0.256449f, -0.00956222f, 1.03054f, -0.390793f, 
		-0.00956222f, 1.03054f, -0.390793f, -0.00953669f, 1.04451f, 0.414450f, 
		-0.00953669f, 1.04451f, 0.414450f, -0.00953669f, 1.04451f, 0.414450f, 
		-0.00953669f, 1.04451f, 0.414450f, -0.00949104f, 1.04342f, -0.415134f, 
		-0.00949104f, 1.04342f, -0.415134f, -0.00949104f, 1.04342f, -0.415134f, 
		-0.00949104f, 1.04342f, -0.415134f, -0.00948941f, 1.06325f, -0.391598f, 
		-0.00948941f, 1.06325f, -0.391598f, -0.00948941f, 1.06325f, -0.391598f, 
		-0.00948941f, 1.06325f, -0.391598f, -0.00948941f, 1.06325f, -0.391598f, 
		-0.00948941f, 1.06325f, -0.391598f, -0.00948941f, 1.06325f, -0.391598f, 
		-0.00948941f, 1.06325f, -0.391598f, -0.00948941f, 1.06325f, -0.391598f, 
		-0.00948941f, 1.06325f, -0.391598f, -0.00948941f, 1.06325f, -0.391598f, 
		-0.00948368f, 1.25205f, -0.0546669f, -0.00948368f, 1.25205f, -0.0546669f, 
		-0.00948368f, 1.25205f, -0.0546669f, -0.00948368f, 1.25205f, -0.0546669f, 
		-0.00948368f, 1.25205f, -0.0546669f, -0.00948368f, 1.25205f, -0.0546669f, 
		-0.00948368f, 1.25205f, -0.0546669f, -0.00948368f, 1.25205f, -0.0546669f, 
		-0.00944916f, 1.02968f, -0.0298616f, -0.00944916f, 1.02968f, -0.0298616f, 
		-0.00944893f, 1.06239f, -0.0298616f, -0.00944893f, 1.06239f, -0.0298616f, 
		-0.00943617f, 1.02968f, 0.0298622f, -0.00943617f, 1.02968f, 0.0298622f, 
		-0.00943581f, 1.06239f, 0.0298622f, -0.00943581f, 1.06239f, 0.0298622f, 
		-0.00941048f, 1.33120f, -0.0162929f, -0.00941048f, 1.33120f, -0.0162929f, 
		-0.00941048f, 1.33120f, -0.0162929f, -0.00941048f, 1.33120f, -0.0162929f, 
		-0.00941048f, 1.33120f, -0.0162929f, -0.00941048f, 1.33120f, -0.0162929f, 
		-0.00941048f, 1.33120f, -0.0162929f, -0.00941048f, 1.33120f, -0.0162929f, 
		-0.00941048f, 1.33120f, -0.0162929f, -0.00941048f, 1.33120f, -0.0162929f, 
		-0.00941048f, 1.33120f, -0.0162929f, -0.00940343f, 1.33363f, 0.0162930f, 
		-0.00940343f, 1.33363f, 0.0162930f, -0.00940343f, 1.33363f, 0.0162930f, 
		-0.00940343f, 1.33363f, 0.0162930f, -0.00940343f, 1.33363f, 0.0162930f, 
		-0.00940343f, 1.33363f, 0.0162930f, -0.00940343f, 1.33363f, 0.0162930f, 
		-0.00940343f, 1.33363f, 0.0162930f, -0.00940343f, 1.33363f, 0.0162930f, 
		-0.00940343f, 1.33363f, 0.0162930f, -0.00940343f, 1.33363f, 0.0162930f, 
		-0.00938433f, 1.02259f, 0.251740f, -0.00938433f, 1.02259f, 0.251740f, 
		-0.00936556f, 1.03087f, -0.388358f, -0.00936556f, 1.03087f, -0.388358f, 
		-0.00936556f, 1.03087f, -0.388358f, -0.00936556f, 1.03087f, -0.388358f, 
		-0.00936556f, 1.03087f, -0.388358f, -0.00936556f, 1.03087f, -0.388358f, 
		-0.00936543f, 1.06303f, -0.388358f, -0.00936543f, 1.06303f, -0.388358f, 
		-0.00936543f, 1.06303f, -0.388358f, -0.00936543f, 1.06303f, -0.388358f, 
		-0.00936543f, 1.06303f, -0.388358f, -0.00936543f, 1.06303f, -0.388358f, 
		-0.00936543f, 1.06303f, -0.388358f, -0.00936543f, 1.06303f, -0.388358f, 
		-0.00935361f, 1.04777f, 0.414610f, -0.00935361f, 1.04777f, 0.414610f, 
		-0.00935361f, 1.04777f, 0.414610f, -0.00932598f, 1.08105f, -0.420194f, 
		-0.00932598f, 1.08105f, -0.420194f, -0.00932598f, 1.08105f, -0.420194f, 
		-0.00932475f, 1.06325f, 0.389080f, -0.00932475f, 1.06325f, 0.389080f, 
		-0.00932475f, 1.06325f, 0.389080f, -0.00932475f, 1.06325f, 0.389080f, 
		-0.00932475f, 1.06325f, 0.389080f, -0.00932475f, 1.06325f, 0.389080f, 
		-0.00932475f, 1.06325f, 0.389080f, -0.00932475f, 1.06325f, 0.389080f, 
		-0.00932475f, 1.06325f, 0.389080f, -0.00932475f, 1.06325f, 0.389080f, 
		-0.00932475f, 1.06325f, 0.389080f, -0.00932412f, 1.03066f, 0.389080f, 
		-0.00932412f, 1.03066f, 0.389080f, -0.00932412f, 1.03066f, 0.389080f, 
		-0.00932412f, 1.03066f, 0.389080f, -0.00932412f, 1.03066f, 0.389080f, 
		-0.00932412f, 1.03066f, 0.389080f, -0.00932412f, 1.03066f, 0.389080f, 
		-0.00932412f, 1.03066f, 0.389080f, -0.00932412f, 1.03066f, 0.389080f, 
		-0.00932412f, 1.03066f, 0.389080f, -0.00932412f, 1.03066f, 0.389080f, 
		-0.00932412f, 1.03066f, 0.389080f, -0.00932412f, 1.03066f, 0.389080f, 
		-0.00932412f, 1.03066f, 0.389080f, -0.00932051f, 0.957955f, -0.277339f, 
		-0.00932051f, 0.957955f, -0.277339f, -0.00932051f, 0.957955f, -0.277339f, 
		-0.00931644f, 0.953366f, -0.252033f, -0.00931644f, 0.953366f, -0.252033f, 
		-0.00929311f, 1.06292f, -0.361268f, -0.00929311f, 1.06292f, -0.361268f, 
		-0.00929311f, 1.06292f, -0.361268f, -0.00928710f, 1.33444f, -0.0160793f, 
		-0.00928710f, 1.33444f, -0.0160793f, -0.00928710f, 1.33444f, -0.0160793f, 
		-0.00928710f, 1.33444f, -0.0160793f, -0.00928710f, 1.33444f, -0.0160793f, 
		-0.00928710f, 1.33444f, -0.0160793f, -0.00928710f, 1.33444f, -0.0160793f, 
		-0.00928710f, 1.33444f, -0.0160793f, -0.00928710f, 1.33444f, -0.0160793f, 
		-0.00928051f, 1.33039f, 0.0160793f, -0.00928051f, 1.33039f, 0.0160793f, 
		-0.00928051f, 1.33039f, 0.0160793f, -0.00928051f, 1.33039f, 0.0160793f, 
		-0.00928051f, 1.33039f, 0.0160793f, -0.00928051f, 1.33039f, 0.0160793f, 
		-0.00928051f, 1.33039f, 0.0160793f, -0.00928051f, 1.33039f, 0.0160793f, 
		-0.00921783f, 1.31902f, 0.0159712f, -0.00921783f, 1.31902f, 0.0159712f, 
		-0.00921693f, 1.31906f, -0.0161164f, -0.00921693f, 1.31906f, -0.0161164f, 
		-0.00921600f, 0.969213f, -0.209139f, -0.00921292f, 1.30330f, 0.0159626f, 
		-0.00921292f, 1.30330f, 0.0159626f, -0.00921292f, 1.30330f, 0.0159626f, 
		-0.00921292f, 1.30330f, 0.0159626f, -0.00920955f, 0.953366f, 0.251935f, 
		-0.00920955f, 0.953366f, 0.251935f, -0.00920569f, 1.18486f, -0.00225703f, 
		-0.00920569f, 1.18486f, -0.00225703f, -0.00920569f, 1.18486f, -0.00225703f, 
		-0.00920569f, 1.18486f, -0.00225703f, -0.00920566f, 1.34406f, -0.00225703f, 
		-0.00920566f, 1.34406f, -0.00225703f, -0.00920566f, 1.34406f, -0.00225703f, 
		-0.00920566f, 1.34406f, -0.00225703f, -0.00920435f, 1.53687f, -0.00225692f, 
		-0.00920435f, 1.53687f, -0.00225692f, -0.00920435f, 1.53687f, -0.00225692f, 
		-0.00920435f, 1.53687f, -0.00225692f, -0.00920295f, 0.953366f, 0.284024f, 
		-0.00920295f, 0.953366f, 0.284024f, -0.00920258f, 1.04291f, -0.423168f, 
		-0.00920258f, 1.04291f, -0.423168f, -0.00920258f, 1.04291f, -0.423168f, 
		-0.00920258f, 1.04291f, -0.423168f, -0.00920073f, 1.06303f, 0.392320f, 
		-0.00920073f, 1.06303f, 0.392320f, -0.00920073f, 1.06303f, 0.392320f, 
		-0.00920073f, 1.06303f, 0.392320f, -0.00920073f, 1.06303f, 0.392320f, 
		-0.00920073f, 1.06303f, 0.392320f, -0.00920073f, 1.06303f, 0.392320f, 
		-0.00920073f, 1.06303f, 0.392320f, -0.00920073f, 1.06303f, 0.392320f, 
		-0.00920027f, 1.03087f, 0.392320f, -0.00920027f, 1.03087f, 0.392320f, 
		-0.00920027f, 1.03087f, 0.392320f, -0.00920027f, 1.03087f, 0.392320f, 
		-0.00920027f, 1.03087f, 0.392320f, -0.00920027f, 1.03087f, 0.392320f, 
		-0.00920027f, 1.03087f, 0.392320f, -0.00920027f, 1.03087f, 0.392320f, 
		-0.00920027f, 1.03087f, 0.392320f, -0.00920027f, 1.03087f, 0.392320f, 
		-0.00919868f, 1.20497f, -0.0318201f, -0.00919538f, 1.32549f, -0.0796986f, 
		-0.00919538f, 1.32549f, -0.0796986f, -0.00919538f, 1.32549f, -0.0796986f, 
		-0.00919538f, 1.32549f, -0.0796986f, -0.00919538f, 1.32549f, -0.0796986f, 
		-0.00916182f, 1.32549f, 0.0795992f, -0.00916182f, 1.32549f, 0.0795992f, 
		-0.00915340f, 1.03869f, -0.385972f, -0.00915340f, 1.03869f, -0.385972f, 
		-0.00915340f, 1.03869f, -0.385972f, -0.00915340f, 1.03869f, -0.385972f, 
		-0.00915340f, 1.03869f, -0.385972f, -0.00915340f, 1.03869f, -0.385972f, 
		-0.00915309f, 1.05521f, -0.385972f, -0.00915309f, 1.05521f, -0.385972f, 
		-0.00915309f, 1.05521f, -0.385972f, -0.00915309f, 1.05521f, -0.385972f, 
		-0.00915309f, 1.05521f, -0.385972f, -0.00915309f, 1.05521f, -0.385972f, 
		-0.00915309f, 1.05521f, -0.385972f, -0.00915309f, 1.05521f, -0.385972f, 
		-0.00915309f, 1.05521f, -0.385972f, -0.00914162f, 1.03098f, 0.376897f, 
		-0.00914162f, 1.03098f, 0.376897f, -0.00914161f, 1.06293f, 0.376897f, 
		-0.00914161f, 1.06293f, 0.376897f, -0.00912800f, 0.969213f, 0.209041f, 
		-0.00912785f, 1.32653f, 0.0718630f, -0.00912785f, 1.32653f, 0.0718630f, 
		-0.00912785f, 1.32653f, 0.0718630f, -0.00912785f, 1.32653f, 0.0718630f, 
		-0.00912785f, 1.32653f, 0.0718630f, -0.00909924f, 1.07658f, -0.268368f, 
		-0.00909924f, 1.07658f, -0.268368f, -0.00909924f, 1.07658f, -0.268368f, 
		-0.00909924f, 1.07658f, -0.268368f, -0.00909924f, 1.07658f, -0.268368f, 
		-0.00909924f, 1.07658f, -0.268368f, -0.00909924f, 1.07658f, -0.268368f, 
		-0.00909924f, 1.07658f, -0.268368f, -0.00907077f, 1.32801f, 0.00825947f, 
		-0.00907077f, 1.32801f, 0.00825947f, -0.00907077f, 1.32801f, 0.00825947f, 
		-0.00907077f, 1.32801f, 0.00825947f, -0.00907077f, 1.32801f, 0.00825947f, 
		-0.00907077f, 1.32801f, 0.00825947f, -0.00907077f, 1.32801f, 0.00825947f, 
		-0.00902752f, 1.25445f, 0.0702946f, -0.00902752f, 1.25445f, 0.0702946f, 
		-0.00902752f, 1.25445f, 0.0702946f, -0.00902752f, 1.25445f, 0.0702946f, 
		-0.00902752f, 1.25445f, 0.0702946f, -0.00902752f, 1.25445f, 0.0702946f, 
		-0.00901331f, 1.03149f, -0.393439f, -0.00901331f, 1.03149f, -0.393439f, 
		-0.00901331f, 1.03149f, -0.393439f, -0.00901331f, 1.03149f, -0.393439f, 
		-0.00901331f, 1.03149f, -0.393439f, -0.00901331f, 1.03149f, -0.393439f, 
		-0.00900387f, 1.02272f, 0.303071f, -0.00900387f, 1.02272f, 0.303071f, 
		-0.00899069f, 1.03870f, 0.385884f, -0.00899069f, 1.03870f, 0.385884f, 
		-0.00899069f, 1.03870f, 0.385884f, -0.00899069f, 1.03870f, 0.385884f, 
		-0.00899069f, 1.03870f, 0.385884f, -0.00899069f, 1.03870f, 0.385884f, 
		-0.00898986f, 1.18474f, 0.00221258f, -0.00898986f, 1.18474f, 0.00221258f, 
		-0.00898986f, 1.18474f, 0.00221258f, -0.00898986f, 1.18474f, 0.00221258f, 
		-0.00898986f, 1.18474f, 0.00221258f, -0.00898958f, 1.34395f, 0.00221209f, 
		-0.00898958f, 1.34395f, 0.00221209f, -0.00898958f, 1.34395f, 0.00221209f, 
		-0.00898958f, 1.34395f, 0.00221209f, -0.00898958f, 1.34395f, 0.00221209f, 
		-0.00898941f, 1.53675f, 0.00221177f, -0.00898941f, 1.53675f, 0.00221177f, 
		-0.00898941f, 1.53675f, 0.00221177f, -0.00898941f, 1.53675f, 0.00221177f, 
		-0.00898941f, 1.53675f, 0.00221177f, -0.00895200f, 0.965681f, -0.210892f, 
		-0.00893837f, 1.02739f, -0.0151498f, -0.00893837f, 1.02739f, -0.0151498f, 
		-0.00893837f, 1.02739f, -0.0151498f, -0.00893837f, 1.02739f, -0.0151498f, 
		-0.00893380f, 1.00046f, 0.288438f, -0.00893380f, 1.00046f, 0.288438f, 
		-0.00893180f, 1.02739f, 0.0151496f, -0.00893180f, 1.02739f, 0.0151496f, 
		-0.00893180f, 1.02739f, 0.0151496f, -0.00886665f, 1.24830f, 0.0656378f, 
		-0.00886665f, 1.24830f, 0.0656378f, -0.00886665f, 1.24830f, 0.0656378f, 
		-0.00886665f, 1.24830f, 0.0656378f, -0.00886665f, 1.24830f, 0.0656378f, 
		-0.00886665f, 1.24830f, 0.0656378f, -0.00886665f, 1.24830f, 0.0656378f, 
		-0.00886665f, 1.24830f, 0.0656378f, -0.00886300f, 0.965681f, 0.210794f, 
		-0.00885100f, 0.832160f, -0.283305f, -0.00885100f, 0.832160f, -0.283305f, 
		-0.00884500f, 0.832160f, -0.252851f, -0.00884500f, 0.832160f, -0.252851f, 
		-0.00883336f, 1.06086f, -0.273123f, -0.00883336f, 1.06086f, -0.273123f, 
		-0.00883336f, 1.06086f, -0.273123f, -0.00883336f, 1.06086f, -0.273123f, 
		-0.00880803f, 1.02264f, 0.290419f, -0.00880803f, 1.02264f, 0.290419f, 
		-0.00880803f, 1.02264f, 0.290419f, -0.00879409f, 1.25445f, 0.0598009f, 
		-0.00879409f, 1.25445f, 0.0598009f, -0.00879409f, 1.25445f, 0.0598009f, 
		-0.00879409f, 1.25445f, 0.0598009f, -0.00879409f, 1.25445f, 0.0598009f, 
		-0.00879409f, 1.25445f, 0.0598009f, -0.00876730f, 1.05580f, 0.385696f, 
		-0.00876730f, 1.05580f, 0.385696f, -0.00876730f, 1.05580f, 0.385696f, 
		-0.00876730f, 1.05580f, 0.385696f, -0.00876730f, 1.05580f, 0.385696f, 
		-0.00876730f, 1.05580f, 0.385696f, -0.00876730f, 1.05580f, 0.385696f, 
		-0.00876730f, 1.05580f, 0.385696f, -0.00876730f, 1.05580f, 0.385696f, 
		-0.00876730f, 1.05580f, 0.385696f, -0.00876730f, 1.05580f, 0.385696f, 
		-0.00876730f, 1.05580f, 0.385696f, -0.00876363f, 1.25440f, 0.0545774f, 
		-0.00876363f, 1.25440f, 0.0545774f, -0.00876363f, 1.25440f, 0.0545774f, 
		-0.00876363f, 1.25440f, 0.0545774f, -0.00876363f, 1.25440f, 0.0545774f, 
		-0.00873800f, 0.832160f, 0.252753f, -0.00873800f, 0.832160f, 0.252753f, 
		-0.00873463f, 1.34207f, 0.00754429f, -0.00873463f, 1.34207f, 0.00754429f, 
		-0.00873463f, 1.34207f, 0.00754429f, -0.00873463f, 1.34207f, 0.00754429f, 
		-0.00873463f, 1.34207f, 0.00754429f, -0.00873463f, 1.34207f, 0.00754429f, 
		-0.00873459f, 1.53488f, 0.00754452f, -0.00873459f, 1.53488f, 0.00754452f, 
		-0.00873459f, 1.53488f, 0.00754452f, -0.00873459f, 1.53488f, 0.00754452f, 
		-0.00873459f, 1.53488f, 0.00754452f, -0.00873459f, 1.53488f, 0.00754452f, 
		-0.00873100f, 0.832160f, 0.283206f, -0.00873100f, 0.832160f, 0.283206f, 
		-0.00871769f, 1.06086f, 0.273024f, -0.00871769f, 1.06086f, 0.273024f, 
		-0.00871769f, 1.06086f, 0.273024f, -0.00871769f, 1.06086f, 0.273024f, 
		-0.00871769f, 1.06086f, 0.273024f, -0.00871769f, 1.06086f, 0.273024f, 
		-0.00866934f, 1.24704f, 0.0692033f, -0.00866934f, 1.24704f, 0.0692033f, 
		-0.00866934f, 1.24704f, 0.0692033f, -0.00866934f, 1.24704f, 0.0692033f, 
		-0.00866934f, 1.24704f, 0.0692033f, -0.00866934f, 1.24704f, 0.0692033f, 
		-0.00866934f, 1.24704f, 0.0692033f, -0.00866934f, 1.24704f, 0.0692033f, 
		-0.00866934f, 1.24704f, 0.0692033f, -0.00866934f, 1.24704f, 0.0692033f, 
		-0.00865716f, 0.955074f, -0.282969f, -0.00865716f, 0.955074f, -0.282969f, 
		-0.00856362f, 1.24708f, 0.0547121f, -0.00856362f, 1.24708f, 0.0547121f, 
		-0.00856362f, 1.24708f, 0.0547121f, -0.00856362f, 1.24708f, 0.0547121f, 
		-0.00856362f, 1.24708f, 0.0547121f, -0.00856362f, 1.24708f, 0.0547121f, 
		-0.00856362f, 1.24708f, 0.0547121f, -0.00856362f, 1.24708f, 0.0547121f, 
		-0.00853363f, 1.04872f, -0.406490f, -0.00853363f, 1.04872f, -0.406490f, 
		-0.00853363f, 1.04872f, -0.406490f, -0.00853363f, 1.04872f, -0.406490f, 
		-0.00853363f, 1.04872f, -0.406490f, -0.00853363f, 1.04872f, -0.406490f, 
		-0.00853363f, 1.04872f, -0.406490f, -0.00853363f, 1.04872f, -0.406490f, 
		-0.00853363f, 1.04872f, -0.406490f, -0.00853363f, 1.04872f, -0.406490f, 
		-0.00853363f, 1.04872f, -0.406490f, -0.00853363f, 1.04872f, -0.406490f, 
		-0.00853363f, 1.04872f, -0.406490f, -0.00852389f, 1.03991f, 0.422851f, 
		-0.00852389f, 1.03991f, 0.422851f, -0.00852389f, 1.03991f, 0.422851f, 
		-0.00852389f, 1.03991f, 0.422851f, -0.00852389f, 1.03991f, 0.422851f, 
		-0.00852212f, 1.06621f, -0.275249f, -0.00852212f, 1.06621f, -0.275249f, 
		-0.00852212f, 1.06621f, -0.275249f, -0.00852212f, 1.06621f, -0.275249f, 
		-0.00852212f, 1.06621f, -0.275249f, -0.00852212f, 1.06621f, -0.275249f, 
		-0.00852212f, 1.06621f, -0.275249f, -0.00852212f, 1.06621f, -0.275249f, 
		-0.00847215f, 1.31683f, -0.0792210f, -0.00847215f, 1.31683f, -0.0792210f, 
		-0.00847215f, 1.31683f, -0.0792210f, -0.00847215f, 1.31683f, -0.0792210f, 
		-0.00843187f, 1.07659f, 0.275562f, -0.00843187f, 1.07659f, 0.275562f, 
		-0.00843187f, 1.07659f, 0.275562f, -0.00843187f, 1.07659f, 0.275562f, 
		-0.00843187f, 1.07659f, 0.275562f, -0.00843187f, 1.07659f, 0.275562f, 
		-0.00843187f, 1.07659f, 0.275562f, -0.00843187f, 1.07659f, 0.275562f, 
		-0.00843187f, 1.07659f, 0.275562f, -0.00843187f, 1.07659f, 0.275562f, 
		-0.00839128f, 1.08284f, -0.403729f, -0.00839128f, 1.08284f, -0.403729f, 
		-0.00838828f, 1.04390f, 0.406348f, -0.00838828f, 1.04390f, 0.406348f, 
		-0.00838828f, 1.04390f, 0.406348f, -0.00838828f, 1.04390f, 0.406348f, 
		-0.00838828f, 1.04390f, 0.406348f, -0.00838828f, 1.04390f, 0.406348f, 
		-0.00836936f, 1.08254f, 0.403840f, -0.00836936f, 1.08254f, 0.403840f, 
		-0.00836936f, 1.08254f, 0.403840f, -0.00836936f, 1.08254f, 0.403840f, 
		-0.00835969f, 1.54515f, 0.00268365f, -0.00835969f, 1.54515f, 0.00268365f, 
		-0.00835969f, 1.54515f, 0.00268365f, -0.00835969f, 1.54515f, 0.00268365f, 
		-0.00835968f, 1.19314f, 0.00268409f, -0.00835968f, 1.19314f, 0.00268409f, 
		-0.00835968f, 1.19314f, 0.00268409f, -0.00835968f, 1.19314f, 0.00268409f, 
		-0.00835967f, 1.35234f, 0.00268356f, -0.00835967f, 1.35234f, 0.00268356f, 
		-0.00835967f, 1.35234f, 0.00268356f, -0.00835967f, 1.35234f, 0.00268356f, 
		-0.00829773f, 1.03272f, -0.386279f, -0.00829773f, 1.03272f, -0.386279f, 
		-0.00829773f, 1.03272f, -0.386279f, -0.00829773f, 1.03272f, -0.386279f, 
		-0.00829773f, 1.03272f, -0.386279f, -0.00829773f, 1.03272f, -0.386279f, 
		-0.00829762f, 1.06119f, -0.386279f, -0.00829762f, 1.06119f, -0.386279f, 
		-0.00829762f, 1.06119f, -0.386279f, -0.00829762f, 1.06119f, -0.386279f, 
		-0.00828880f, 1.31662f, 0.0788251f, -0.00828880f, 1.31662f, 0.0788251f, 
		-0.00827615f, 1.06115f, -0.394639f, -0.00827615f, 1.06115f, -0.394639f, 
		-0.00827615f, 1.06115f, -0.394639f, -0.00827615f, 1.06115f, -0.394639f, 
		-0.00827615f, 1.06115f, -0.394639f, -0.00827498f, 1.54510f, -0.00280414f, 
		-0.00827498f, 1.54510f, -0.00280414f, -0.00827498f, 1.54510f, -0.00280414f, 
		-0.00827498f, 1.54510f, -0.00280414f, -0.00827496f, 1.19309f, -0.00280459f, 
		-0.00827496f, 1.19309f, -0.00280459f, -0.00827496f, 1.19309f, -0.00280459f, 
		-0.00827496f, 1.19309f, -0.00280459f, -0.00827490f, 1.35229f, -0.00280421f, 
		-0.00827490f, 1.35229f, -0.00280421f, -0.00827490f, 1.35229f, -0.00280421f, 
		-0.00827490f, 1.35229f, -0.00280421f, -0.00821933f, 1.32831f, -0.0142313f, 
		-0.00821933f, 1.32831f, -0.0142313f, -0.00821933f, 1.32831f, -0.0142313f, 
		-0.00821933f, 1.32831f, -0.0142313f, -0.00821933f, 1.32831f, -0.0142313f, 
		-0.00821933f, 1.32831f, -0.0142313f, -0.00821933f, 1.32831f, -0.0142313f, 
		-0.00821933f, 1.32831f, -0.0142313f, -0.00821933f, 1.32831f, -0.0142313f, 
		-0.00821933f, 1.32831f, -0.0142313f, -0.00821933f, 1.32831f, -0.0142313f, 
		-0.00821324f, 1.32831f, 0.0142305f, -0.00821324f, 1.32831f, 0.0142305f, 
		-0.00821324f, 1.32831f, 0.0142305f, -0.00821324f, 1.32831f, 0.0142305f, 
		-0.00820900f, 1.23705f, -0.0543860f, -0.00820900f, 1.23705f, -0.0543860f, 
		-0.00820900f, 1.26705f, -0.0543860f, -0.00820900f, 1.26705f, -0.0543860f, 
		-0.00819605f, 1.33668f, -0.0141916f, -0.00819605f, 1.33668f, -0.0141916f, 
		-0.00819605f, 1.33668f, -0.0141916f, -0.00819605f, 1.33668f, -0.0141916f, 
		-0.00819605f, 1.33668f, -0.0141916f, -0.00819605f, 1.33668f, -0.0141916f, 
		-0.00819005f, 1.33668f, 0.0141907f, -0.00819005f, 1.33668f, 0.0141907f, 
		-0.00819005f, 1.33668f, 0.0141907f, -0.00819005f, 1.33668f, 0.0141907f, 
		-0.00819005f, 1.33668f, 0.0141907f, -0.00819005f, 1.33668f, 0.0141907f, 
		-0.00818600f, 1.23705f, 0.0543850f, -0.00818600f, 1.23705f, 0.0543850f, 
		-0.00818600f, 1.26705f, 0.0543850f, -0.00818600f, 1.26705f, 0.0543850f, 
		-0.00817389f, 1.34226f, -0.00721097f, -0.00817389f, 1.34226f, -0.00721097f, 
		-0.00817389f, 1.34226f, -0.00721097f, -0.00817389f, 1.34226f, -0.00721097f, 
		-0.00817389f, 1.34226f, -0.00721097f, -0.00813454f, 1.06119f, 0.386190f, 
		-0.00813454f, 1.06119f, 0.386190f, -0.00813454f, 1.06119f, 0.386190f, 
		-0.00813454f, 1.06119f, 0.386190f, -0.00813454f, 1.06119f, 0.386190f, 
		-0.00813454f, 1.06119f, 0.386190f, -0.00813403f, 1.03272f, 0.386190f, 
		-0.00813403f, 1.03272f, 0.386190f, -0.00813403f, 1.03272f, 0.386190f, 
		-0.00813403f, 1.03272f, 0.386190f, -0.00813403f, 1.03272f, 0.386190f, 
		-0.00813403f, 1.03272f, 0.386190f, -0.00812640f, 1.03903f, 0.416556f, 
		-0.00812640f, 1.03903f, 0.416556f, -0.00812640f, 1.03903f, 0.416556f, 
		-0.00810955f, 1.06115f, 0.394550f, -0.00810955f, 1.06115f, 0.394550f, 
		-0.00810955f, 1.06115f, 0.394550f, -0.00810955f, 1.06115f, 0.394550f, 
		-0.00810955f, 1.06115f, 0.394550f, -0.00810955f, 1.06115f, 0.394550f, 
		-0.00810947f, 1.03276f, 0.394550f, -0.00810947f, 1.03276f, 0.394550f, 
		-0.00810947f, 1.03276f, 0.394550f, -0.00810947f, 1.03276f, 0.394550f, 
		-0.00810947f, 1.03276f, 0.394550f, -0.00810947f, 1.03276f, 0.394550f, 
		-0.00810571f, 1.00048f, 0.251679f, -0.00810571f, 1.00048f, 0.251679f, 
		-0.00810571f, 1.00048f, 0.251679f, -0.00810571f, 1.00048f, 0.251679f, 
		-0.00806184f, 1.09082f, -0.412719f, -0.00806184f, 1.09082f, -0.412719f, 
		-0.00806184f, 1.09082f, -0.412719f, -0.00806184f, 1.09082f, -0.412719f, 
		-0.00806184f, 1.09082f, -0.412719f, -0.00802249f, 1.08746f, 0.421216f, 
		-0.00802249f, 1.08746f, 0.421216f, -0.00802249f, 1.08746f, 0.421216f, 
		-0.00802249f, 1.08746f, 0.421216f, -0.00802249f, 1.08746f, 0.421216f, 
		-0.00802249f, 1.08746f, 0.421216f, -0.00794291f, 1.33326f, -0.0829043f, 
		-0.00794291f, 1.33326f, -0.0829043f, -0.00794291f, 1.33326f, -0.0829043f, 
		-0.00794291f, 1.33326f, -0.0829043f, -0.00794291f, 1.33326f, -0.0829043f, 
		-0.00790774f, 1.33326f, 0.0828056f, -0.00790774f, 1.33326f, 0.0828056f, 
		-0.00790774f, 1.33326f, 0.0828056f, -0.00790774f, 1.33326f, 0.0828056f, 
		-0.00789063f, 1.33921f, 0.00792315f, -0.00789063f, 1.33921f, 0.00792315f, 
		-0.00789056f, 1.18001f, 0.00792313f, -0.00789056f, 1.18001f, 0.00792313f, 
		-0.00789054f, 1.53202f, 0.00792318f, -0.00789054f, 1.53202f, 0.00792318f, 
		-0.00788656f, 1.09082f, 0.412620f, -0.00788656f, 1.09082f, 0.412620f, 
		-0.00788656f, 1.09082f, 0.412620f, -0.00788656f, 1.09082f, 0.412620f, 
		-0.00788656f, 1.09082f, 0.412620f, -0.00778216f, 1.23699f, -0.0496122f, 
		-0.00778216f, 1.23699f, -0.0496122f, -0.00776125f, 1.23699f, 0.0496111f, 
		-0.00776125f, 1.23699f, 0.0496111f, -0.00775527f, 1.17825f, -0.00743426f, 
		-0.00775527f, 1.17825f, -0.00743426f, -0.00775527f, 1.17825f, -0.00743426f, 
		-0.00775527f, 1.17825f, -0.00743426f, -0.00775526f, 1.53026f, -0.00743387f, 
		-0.00775526f, 1.53026f, -0.00743387f, -0.00775526f, 1.53026f, -0.00743387f, 
		-0.00775526f, 1.53026f, -0.00743387f, -0.00766955f, 1.03748f, -0.416913f, 
		-0.00766955f, 1.03748f, -0.416913f, -0.00766955f, 1.03748f, -0.416913f, 
		-0.00766955f, 1.03748f, -0.416913f, -0.00766955f, 1.03748f, -0.416913f, 
		-0.00766955f, 1.03748f, -0.416913f, -0.00759163f, 1.07317f, -0.263382f, 
		-0.00759163f, 1.07317f, -0.263382f, -0.00759163f, 1.07317f, -0.263382f, 
		-0.00759163f, 1.07317f, -0.263382f, -0.00759163f, 1.07317f, -0.263382f, 
		-0.00759163f, 1.07317f, -0.263382f, -0.00759163f, 1.07317f, -0.263382f, 
		-0.00759163f, 1.07317f, -0.263382f, -0.00759163f, 1.07317f, -0.263382f, 
		-0.00759163f, 1.07317f, -0.263382f, -0.00757800f, 1.03305f, -0.370000f, 
		-0.00757800f, 1.03305f, -0.370000f, -0.00757800f, 1.05903f, -0.370000f, 
		-0.00757800f, 1.05903f, -0.370000f, -0.00754644f, 1.24449f, -0.0673957f, 
		-0.00754644f, 1.24449f, -0.0673957f, -0.00754644f, 1.24449f, -0.0673957f, 
		-0.00754644f, 1.24449f, -0.0673957f, -0.00754644f, 1.24449f, -0.0673957f, 
		-0.00754644f, 1.24449f, -0.0673957f, -0.00754644f, 1.24449f, -0.0673957f, 
		-0.00754644f, 1.24449f, -0.0673957f, -0.00754644f, 1.24449f, -0.0673957f, 
		-0.00754644f, 1.24449f, -0.0673957f, -0.00753679f, 1.06250f, -0.278301f, 
		-0.00753679f, 1.06250f, -0.278301f, -0.00753679f, 1.06250f, -0.278301f, 
		-0.00753679f, 1.06250f, -0.278301f, -0.00753679f, 1.06250f, -0.278301f, 
		-0.00753679f, 1.06250f, -0.278301f, -0.00746824f, 1.03864f, -0.421857f, 
		-0.00746824f, 1.03864f, -0.421857f, -0.00746824f, 1.03864f, -0.421857f, 
		-0.00746824f, 1.03864f, -0.421857f, -0.00744586f, 1.07350f, 0.263164f, 
		-0.00744586f, 1.07350f, 0.263164f, -0.00744586f, 1.07350f, 0.263164f, 
		-0.00744586f, 1.07350f, 0.263164f, -0.00744586f, 1.07350f, 0.263164f, 
		-0.00744586f, 1.07350f, 0.263164f, -0.00742417f, 1.30539f, -0.0751302f, 
		-0.00742417f, 1.30539f, -0.0751302f, -0.00742417f, 1.30539f, -0.0751302f, 
		-0.00742200f, 1.03305f, 0.370000f, -0.00742200f, 1.03305f, 0.370000f, 
		-0.00742200f, 1.05903f, 0.370000f, -0.00742200f, 1.05903f, 0.370000f, 
		-0.00741794f, 1.06250f, 0.278203f, -0.00741794f, 1.06250f, 0.278203f, 
		-0.00741794f, 1.06250f, 0.278203f, -0.00741794f, 1.06250f, 0.278203f, 
		-0.00741794f, 1.06250f, 0.278203f, -0.00741794f, 1.06250f, 0.278203f, 
		-0.00741794f, 1.06250f, 0.278203f, -0.00741794f, 1.06250f, 0.278203f, 
		-0.00741794f, 1.06250f, 0.278203f, -0.00741794f, 1.06250f, 0.278203f, 
		-0.00741024f, 1.04422f, -0.394012f, -0.00741024f, 1.04422f, -0.394012f, 
		-0.00741024f, 1.04422f, -0.394012f, -0.00741024f, 1.04422f, -0.394012f, 
		-0.00741024f, 1.04422f, -0.394012f, -0.00741024f, 1.04422f, -0.394012f, 
		-0.00741024f, 1.04422f, -0.394012f, -0.00741024f, 1.04422f, -0.394012f, 
		-0.00740173f, 1.24440f, -0.0564324f, -0.00740173f, 1.24440f, -0.0564324f, 
		-0.00740173f, 1.24440f, -0.0564324f, -0.00740173f, 1.24440f, -0.0564324f, 
		-0.00740173f, 1.24440f, -0.0564324f, -0.00740173f, 1.24440f, -0.0564324f, 
		-0.00733511f, 1.07241f, -0.278039f, -0.00733511f, 1.07241f, -0.278039f, 
		-0.00733511f, 1.07241f, -0.278039f, -0.00733511f, 1.07241f, -0.278039f, 
		-0.00733511f, 1.07241f, -0.278039f, -0.00733511f, 1.07241f, -0.278039f, 
		-0.00733332f, 1.36468f, -0.00195224f, -0.00733332f, 1.36468f, -0.00195224f, 
		-0.00733332f, 1.36468f, -0.00195224f, -0.00733332f, 1.36468f, -0.00195224f, 
		-0.00733332f, 1.36468f, -0.00195224f, -0.00733332f, 1.36468f, -0.00195224f, 
		-0.00733332f, 1.36468f, -0.00195224f, -0.00733332f, 1.36468f, -0.00195224f, 
		-0.00733332f, 1.36468f, -0.00195224f, -0.00733332f, 1.20548f, -0.00195224f, 
		-0.00733332f, 1.20548f, -0.00195224f, -0.00733332f, 1.20548f, -0.00195224f, 
		-0.00733332f, 1.20548f, -0.00195224f, -0.00733332f, 1.20548f, -0.00195224f, 
		-0.00733332f, 1.20548f, -0.00195224f, -0.00733332f, 1.20548f, -0.00195224f, 
		-0.00733332f, 1.20548f, -0.00195224f, -0.00733297f, 1.55749f, -0.00195278f, 
		-0.00733297f, 1.55749f, -0.00195278f, -0.00733297f, 1.55749f, -0.00195278f, 
		-0.00733297f, 1.55749f, -0.00195278f, -0.00733297f, 1.55749f, -0.00195278f, 
		-0.00733297f, 1.55749f, -0.00195278f, -0.00733297f, 1.55749f, -0.00195278f, 
		-0.00733297f, 1.55749f, -0.00195278f, -0.00732499f, 1.08664f, 0.392392f, 
		-0.00732499f, 1.08664f, 0.392392f, -0.00732499f, 1.08664f, 0.392392f, 
		-0.00729780f, 1.25398f, -0.0604750f, -0.00729780f, 1.25398f, -0.0604750f, 
		-0.00729780f, 1.25398f, -0.0604750f, -0.00729780f, 1.25398f, -0.0604750f, 
		-0.00729780f, 1.25398f, -0.0604750f, -0.00729780f, 1.25398f, -0.0604750f, 
		-0.00729780f, 1.25398f, -0.0604750f, -0.00729780f, 1.25398f, -0.0604750f, 
		-0.00729780f, 1.25398f, -0.0604750f, -0.00729780f, 1.25398f, -0.0604750f, 
		-0.00729136f, 1.00753f, 0.246999f, -0.00727064f, 1.28192f, -0.0479271f, 
		-0.00724943f, 1.28192f, 0.0479283f, -0.00724362f, 1.04422f, 0.394011f, 
		-0.00724362f, 1.04422f, 0.394011f, -0.00724362f, 1.04422f, 0.394011f, 
		-0.00724362f, 1.04422f, 0.394011f, -0.00724362f, 1.04422f, 0.394011f, 
		-0.00724362f, 1.04422f, 0.394011f, -0.00720879f, 1.26712f, -0.0496988f, 
		-0.00720879f, 1.26712f, -0.0496988f, -0.00720015f, 0.828613f, -0.266164f, 
		-0.00720015f, 0.828613f, -0.266164f, -0.00719819f, 1.22218f, -0.0479384f, 
		-0.00718720f, 1.26712f, 0.0496975f, -0.00718720f, 1.26712f, 0.0496975f, 
		-0.00718360f, 1.06665f, 0.272589f, -0.00718360f, 1.06665f, 0.272589f, 
		-0.00718360f, 1.06665f, 0.272589f, -0.00718360f, 1.06665f, 0.272589f, 
		-0.00718360f, 1.06665f, 0.272589f, -0.00718360f, 1.06665f, 0.272589f, 
		-0.00718360f, 1.06665f, 0.272589f, -0.00718360f, 1.06665f, 0.272589f, 
		-0.00718360f, 1.06665f, 0.272589f, -0.00718360f, 1.06665f, 0.272589f, 
		-0.00718360f, 1.06665f, 0.272589f, -0.00718360f, 1.06665f, 0.272589f, 
		-0.00718360f, 1.06665f, 0.272589f, -0.00717981f, 1.22218f, 0.0479391f, 
		-0.00717147f, 1.07744f, 0.419646f, -0.00717147f, 1.07744f, 0.419646f, 
		-0.00717147f, 1.07744f, 0.419646f, -0.00717147f, 1.07744f, 0.419646f, 
		-0.00717147f, 1.07744f, 0.419646f, -0.00717147f, 1.07744f, 0.419646f, 
		-0.00713505f, 1.53579f, -0.00735365f, -0.00713505f, 1.53579f, -0.00735365f, 
		-0.00713505f, 1.53579f, -0.00735365f, -0.00713505f, 1.53579f, -0.00735365f, 
		-0.00713505f, 1.53579f, -0.00735365f, -0.00713475f, 1.18378f, -0.00735393f, 
		-0.00713475f, 1.18378f, -0.00735393f, -0.00713475f, 1.18378f, -0.00735393f, 
		-0.00713475f, 1.18378f, -0.00735393f, -0.00713475f, 1.18378f, -0.00735393f, 
		-0.00713048f, 1.37607f, -0.0253992f, -0.00713048f, 1.37607f, -0.0253992f, 
		-0.00713048f, 1.37607f, -0.0253992f, -0.00713042f, 1.48509f, -0.0253992f, 
		-0.00713042f, 1.48509f, -0.0253992f, -0.00713042f, 1.43054f, -0.0253992f, 
		-0.00713042f, 1.43054f, -0.0253992f, -0.00710932f, 1.05913f, -0.381482f, 
		-0.00710932f, 1.05913f, -0.381482f, -0.00710932f, 1.05913f, -0.381482f, 
		-0.00710932f, 1.05913f, -0.381482f, -0.00710932f, 1.05913f, -0.381482f, 
		-0.00710932f, 1.05913f, -0.381482f, -0.00710929f, 1.03478f, -0.381482f, 
		-0.00710929f, 1.03478f, -0.381482f, -0.00710929f, 1.03478f, -0.381482f, 
		-0.00710929f, 1.03478f, -0.381482f, -0.00710929f, 1.03478f, -0.381482f, 
		-0.00710929f, 1.03478f, -0.381482f, -0.00710526f, 1.01552f, -0.294722f, 
		-0.00708749f, 0.828613f, 0.269894f, -0.00708749f, 0.828613f, 0.269894f, 
		-0.00702537f, 1.32352f, 0.0121730f, -0.00702537f, 1.32352f, 0.0121730f, 
		-0.00702537f, 1.32352f, 0.0121730f, -0.00702537f, 1.32352f, 0.0121730f, 
		-0.00702537f, 1.32352f, 0.0121730f, -0.00702537f, 1.32352f, 0.0121730f, 
		-0.00694740f, 1.03478f, 0.381393f, -0.00694740f, 1.03478f, 0.381393f, 
		-0.00694740f, 1.03478f, 0.381393f, -0.00694740f, 1.03478f, 0.381393f, 
		-0.00694740f, 1.03478f, 0.381393f, -0.00694740f, 1.03478f, 0.381393f, 
		-0.00694653f, 1.32511f, -0.0120278f, -0.00694653f, 1.32511f, -0.0120278f, 
		-0.00694653f, 1.32511f, -0.0120278f, -0.00694653f, 1.32511f, -0.0120278f, 
		-0.00694653f, 1.32511f, -0.0120278f, -0.00694653f, 1.32511f, -0.0120278f, 
		-0.00694653f, 1.32511f, -0.0120278f, -0.00694653f, 1.32511f, -0.0120278f, 
		-0.00686996f, 1.32137f, -0.0118953f, -0.00686996f, 1.32137f, -0.0118953f, 
		-0.00686996f, 1.32137f, -0.0118953f, -0.00686996f, 1.32137f, -0.0118953f, 
		-0.00686996f, 1.32137f, -0.0118953f, -0.00686996f, 1.32137f, -0.0118953f, 
		-0.00686996f, 1.32137f, -0.0118953f, -0.00686996f, 1.32137f, -0.0118953f, 
		-0.00686996f, 1.32137f, -0.0118953f, -0.00686996f, 1.32137f, -0.0118953f, 
		-0.00686262f, 1.05898f, 0.382981f, -0.00686262f, 1.05898f, 0.382981f, 
		-0.00686262f, 1.05898f, 0.382981f, -0.00686262f, 1.05898f, 0.382981f, 
		-0.00686262f, 1.05898f, 0.382981f, -0.00686262f, 1.05898f, 0.382981f, 
		-0.00686262f, 1.05898f, 0.382981f, -0.00686262f, 1.05898f, 0.382981f, 
		-0.00682623f, 1.25344f, -0.0650377f, -0.00682623f, 1.25344f, -0.0650377f, 
		-0.00682623f, 1.25344f, -0.0650377f, -0.00682623f, 1.25344f, -0.0650377f, 
		-0.00682623f, 1.25344f, -0.0650377f, -0.00682623f, 1.25344f, -0.0650377f, 
		-0.00679177f, 1.33634f, -0.00760074f, -0.00679177f, 1.33634f, -0.00760074f, 
		-0.00679177f, 1.33634f, -0.00760074f, -0.00678716f, 1.05885f, 0.379247f, 
		-0.00678716f, 1.05885f, 0.379247f, -0.00678716f, 1.05885f, 0.379247f, 
		-0.00678716f, 1.05885f, 0.379247f, -0.00678716f, 1.05885f, 0.379247f, 
		-0.00678716f, 1.05885f, 0.379247f, -0.00678716f, 1.05885f, 0.379247f, 
		-0.00678716f, 1.05885f, 0.379247f, -0.00678716f, 1.05885f, 0.379247f, 
		-0.00678716f, 1.05885f, 0.379247f, -0.00678529f, 1.00045f, -0.227974f, 
		-0.00678529f, 1.00045f, -0.227974f, -0.00674390f, 1.20210f, -0.0231258f, 
		-0.00674390f, 1.20210f, -0.0231258f, -0.00673732f, 1.40285f, 0.0252273f, 
		-0.00673732f, 1.40285f, 0.0252273f, -0.00668948f, 1.00045f, 0.227875f, 
		-0.00668948f, 1.00045f, 0.227875f, -0.00666073f, 1.07658f, -0.277401f, 
		-0.00666073f, 1.07658f, -0.277401f, -0.00666073f, 1.07658f, -0.277401f, 
		-0.00666073f, 1.07658f, -0.277401f, -0.00666073f, 1.07658f, -0.277401f, 
		-0.00666073f, 1.07658f, -0.277401f, -0.00657426f, 1.07658f, 0.264071f, 
		-0.00657426f, 1.07658f, 0.264071f, -0.00657426f, 1.07658f, 0.264071f, 
		-0.00657426f, 1.07658f, 0.264071f, -0.00657426f, 1.07658f, 0.264071f, 
		-0.00657426f, 1.07658f, 0.264071f, -0.00653553f, 1.07122f, 0.272880f, 
		-0.00653553f, 1.07122f, 0.272880f, -0.00653553f, 1.07122f, 0.272880f, 
		-0.00653553f, 1.07122f, 0.272880f, -0.00653553f, 1.07122f, 0.272880f, 
		-0.00653553f, 1.07122f, 0.272880f, -0.00647471f, 1.04986f, -0.394014f, 
		-0.00647471f, 1.04986f, -0.394014f, -0.00647471f, 1.04986f, -0.394014f, 
		-0.00647471f, 1.04986f, -0.394014f, -0.00647464f, 1.17730f, -0.00873725f, 
		-0.00647464f, 1.17730f, -0.00873725f, -0.00647461f, 1.33650f, -0.00873726f, 
		-0.00647461f, 1.33650f, -0.00873726f, -0.00647041f, 1.45616f, 0.0242854f, 
		-0.00647041f, 1.45616f, 0.0242854f, -0.00647041f, 1.45616f, 0.0242854f, 
		-0.00647041f, 1.45616f, 0.0242854f, -0.00647041f, 1.45616f, 0.0242854f, 
		-0.00646117f, 1.17679f, 0.00804845f, -0.00646117f, 1.17679f, 0.00804845f, 
		-0.00646117f, 1.17679f, 0.00804845f, -0.00646117f, 1.17679f, 0.00804845f, 
		-0.00646113f, 1.33599f, 0.00804848f, -0.00646113f, 1.33599f, 0.00804848f, 
		-0.00646113f, 1.33599f, 0.00804848f, -0.00646113f, 1.33599f, 0.00804848f, 
		-0.00646098f, 1.52880f, 0.00804824f, -0.00646098f, 1.52880f, 0.00804824f, 
		-0.00646098f, 1.52880f, 0.00804824f, -0.00646098f, 1.52880f, 0.00804824f, 
		-0.00644152f, 1.08983f, -0.270772f, -0.00644152f, 1.08983f, -0.270772f, 
		-0.00640709f, 1.00340f, -0.293052f, -0.00640709f, 1.00340f, -0.293052f, 
		-0.00640709f, 1.00340f, -0.293052f, -0.00639800f, 1.06626f, -0.272470f, 
		-0.00639800f, 1.06626f, -0.272470f, -0.00639800f, 1.06626f, -0.272470f, 
		-0.00639800f, 1.06626f, -0.272470f, -0.00639800f, 1.06626f, -0.272470f, 
		-0.00639800f, 1.06626f, -0.272470f, -0.00639800f, 1.07141f, -0.272470f, 
		-0.00639800f, 1.07141f, -0.272470f, -0.00639800f, 1.07141f, -0.272470f, 
		-0.00639800f, 1.07141f, -0.272470f, -0.00639800f, 1.07141f, -0.272470f, 
		-0.00639800f, 1.07141f, -0.272470f, -0.00639546f, 1.03829f, 0.422205f, 
		-0.00639546f, 1.03829f, 0.422205f, -0.00639546f, 1.03829f, 0.422205f, 
		-0.00639546f, 1.03829f, 0.422205f, -0.00631850f, 1.20339f, 0.0292828f, 
		-0.00631158f, 1.29323f, -0.0408884f, -0.00630737f, 1.07686f, -0.272446f, 
		-0.00630737f, 1.07686f, -0.272446f, -0.00630737f, 1.07686f, -0.272446f, 
		-0.00630737f, 1.07686f, -0.272446f, -0.00630737f, 1.07686f, -0.272446f, 
		-0.00630737f, 1.07686f, -0.272446f, -0.00630737f, 1.07686f, -0.272446f, 
		-0.00630737f, 1.07686f, -0.272446f, -0.00630705f, 1.04986f, 0.394013f, 
		-0.00630705f, 1.04986f, 0.394013f, -0.00630705f, 1.04986f, 0.394013f, 
		-0.00630705f, 1.04986f, 0.394013f, -0.00629281f, 1.29323f, 0.0408882f, 
		-0.00626503f, 1.25373f, -0.0706809f, -0.00626503f, 1.25373f, -0.0706809f, 
		-0.00626503f, 1.25373f, -0.0706809f, -0.00626503f, 1.25373f, -0.0706809f, 
		-0.00626503f, 1.25373f, -0.0706809f, -0.00626503f, 1.25373f, -0.0706809f, 
		-0.00626503f, 1.25373f, -0.0706809f, -0.00626503f, 1.25373f, -0.0706809f, 
		-0.00626470f, 1.25194f, 0.0847726f, -0.00626470f, 1.25194f, 0.0847726f, 
		-0.00626470f, 1.25194f, 0.0847726f, -0.00626470f, 1.25194f, 0.0847726f, 
		-0.00625090f, 1.48428f, 0.0263414f, -0.00625090f, 1.48428f, 0.0263414f, 
		-0.00625090f, 1.48428f, 0.0263414f, -0.00625090f, 1.48428f, 0.0263414f, 
		-0.00625090f, 1.48428f, 0.0263414f, -0.00624571f, 1.24844f, 0.0602394f, 
		-0.00624571f, 1.24844f, 0.0602394f, -0.00624571f, 1.24844f, 0.0602394f, 
		-0.00624571f, 1.24844f, 0.0602394f, -0.00624571f, 1.24844f, 0.0602394f, 
		-0.00624571f, 1.24844f, 0.0602394f, -0.00624571f, 1.24844f, 0.0602394f, 
		-0.00624571f, 1.24844f, 0.0602394f, -0.00624571f, 1.24844f, 0.0602394f, 
		-0.00624571f, 1.24844f, 0.0602394f, -0.00624571f, 1.24844f, 0.0602394f, 
		-0.00624571f, 1.24844f, 0.0602394f, -0.00623499f, 1.25373f, 0.0705814f, 
		-0.00623499f, 1.25373f, 0.0705814f, -0.00623499f, 1.25373f, 0.0705814f, 
		-0.00623499f, 1.25373f, 0.0705814f, -0.00623499f, 1.25373f, 0.0705814f, 
		-0.00623499f, 1.25373f, 0.0705814f, -0.00623499f, 1.25373f, 0.0705814f, 
		-0.00623499f, 1.25373f, 0.0705814f, -0.00622712f, 1.50999f, 0.0241667f, 
		-0.00622712f, 1.50999f, 0.0241667f, -0.00622712f, 1.50999f, 0.0241667f, 
		-0.00622712f, 1.50999f, 0.0241667f, -0.00622712f, 1.50999f, 0.0241667f, 
		-0.00619173f, 1.07686f, 0.272348f, -0.00619173f, 1.07686f, 0.272348f, 
		-0.00619173f, 1.07686f, 0.272348f, -0.00619173f, 1.07686f, 0.272348f, 
		-0.00619173f, 1.07686f, 0.272348f, -0.00619173f, 1.07686f, 0.272348f, 
		-0.00614304f, 0.993265f, -0.230234f, -0.00613418f, 1.25033f, -0.0854638f, 
		-0.00613418f, 1.25033f, -0.0854638f, -0.00613418f, 1.25033f, -0.0854638f, 
		-0.00613418f, 1.25033f, -0.0854638f, -0.00613418f, 1.25033f, -0.0854638f, 
		-0.00613418f, 1.25033f, -0.0854638f, -0.00613418f, 1.25033f, -0.0854638f, 
		-0.00613418f, 1.25033f, -0.0854638f, -0.00613418f, 1.25033f, -0.0854638f, 
		-0.00613418f, 1.25033f, -0.0854638f, -0.00613076f, 1.09084f, 0.272331f, 
		-0.00613076f, 1.09084f, 0.272331f, -0.00613076f, 1.09084f, 0.272331f, 
		-0.00613076f, 1.09084f, 0.272331f, -0.00613076f, 1.09084f, 0.272331f, 
		-0.00613076f, 1.09084f, 0.272331f, -0.00612305f, 1.31682f, -0.0708451f, 
		-0.00612305f, 1.31682f, -0.0708451f, -0.00612305f, 1.31682f, -0.0708451f, 
		-0.00612305f, 1.31682f, -0.0708451f, -0.00611369f, 1.03992f, -0.406332f, 
		-0.00611369f, 1.03992f, -0.406332f, -0.00611369f, 1.03992f, -0.406332f, 
		-0.00611369f, 1.03992f, -0.406332f, -0.00611369f, 1.03992f, -0.406332f, 
		-0.00611369f, 1.03992f, -0.406332f, -0.00611369f, 1.03992f, -0.406332f, 
		-0.00611369f, 1.03992f, -0.406332f, -0.00609338f, 1.32780f, -0.0105505f, 
		-0.00609338f, 1.32780f, -0.0105505f, -0.00609338f, 1.32780f, -0.0105505f, 
		-0.00609338f, 1.32780f, -0.0105505f, -0.00609338f, 1.32780f, -0.0105505f, 
		-0.00609338f, 1.32780f, -0.0105505f, -0.00609338f, 1.32780f, -0.0105505f, 
		-0.00609338f, 1.32780f, -0.0105505f, -0.00607418f, 1.09066f, -0.403449f, 
		-0.00605400f, 1.40109f, -0.0245159f, -0.00605400f, 1.40109f, -0.0245159f, 
		-0.00605400f, 1.40109f, -0.0245159f, -0.00605400f, 1.40109f, -0.0245159f, 
		-0.00605400f, 1.40109f, -0.0245159f, -0.00605400f, 1.40109f, -0.0245159f, 
		-0.00603810f, 1.21087f, -0.0409297f, -0.00602842f, 0.993265f, 0.311210f, 
		-0.00602123f, 1.21088f, 0.0409292f, -0.00597400f, 1.31634f, 0.0710106f, 
		-0.00597400f, 1.31634f, 0.0710106f, -0.00596763f, 0.977046f, -0.300712f, 
		-0.00595179f, 1.02272f, -0.303896f, -0.00595179f, 1.02272f, -0.303896f, 
		-0.00593630f, 1.02272f, -0.237647f, -0.00593630f, 1.02272f, -0.237647f, 
		-0.00592310f, 1.31910f, -0.0102560f, -0.00592310f, 1.31910f, -0.0102560f, 
		-0.00592310f, 1.31910f, -0.0102560f, -0.00592310f, 1.31910f, -0.0102560f, 
		-0.00592310f, 1.31910f, -0.0102560f, -0.00590024f, 1.25546f, 0.0651296f, 
		-0.00590024f, 1.25546f, 0.0651296f, -0.00590024f, 1.25546f, 0.0651296f, 
		-0.00590024f, 1.25546f, 0.0651296f, -0.00590024f, 1.25546f, 0.0651296f, 
		-0.00590024f, 1.25546f, 0.0651296f, -0.00589729f, 1.05262f, 0.406421f, 
		-0.00589729f, 1.05262f, 0.406421f, -0.00589729f, 1.05262f, 0.406421f, 
		-0.00589729f, 1.05262f, 0.406421f, -0.00589729f, 1.05262f, 0.406421f, 
		-0.00589729f, 1.05262f, 0.406421f, -0.00589729f, 1.05262f, 0.406421f, 
		-0.00589729f, 1.05262f, 0.406421f, -0.00589729f, 1.05262f, 0.406421f, 
		-0.00589729f, 1.05262f, 0.406421f, -0.00586450f, 1.37544f, 0.0252934f, 
		-0.00586450f, 1.37544f, 0.0252934f, -0.00583927f, 1.51064f, -0.0251981f, 
		-0.00583927f, 1.51064f, -0.0251981f, -0.00583927f, 1.51064f, -0.0251981f, 
		-0.00583927f, 1.51064f, -0.0251981f, -0.00583925f, 1.45609f, -0.0251981f, 
		-0.00583925f, 1.45609f, -0.0251981f, -0.00583925f, 1.45609f, -0.0251981f, 
		-0.00583925f, 1.45609f, -0.0251981f, -0.00583346f, 1.00047f, -0.250662f, 
		-0.00583346f, 1.00047f, -0.250662f, -0.00583346f, 1.00047f, -0.250662f, 
		-0.00583346f, 1.00047f, -0.250662f, -0.00582272f, 1.02272f, 0.303797f, 
		-0.00582272f, 1.02272f, 0.303797f, -0.00571829f, 1.05850f, -0.377323f, 
		-0.00571829f, 1.05850f, -0.377323f, -0.00571829f, 1.05850f, -0.377323f, 
		-0.00571829f, 1.05850f, -0.377323f, -0.00571829f, 1.05850f, -0.377323f, 
		-0.00571829f, 1.05850f, -0.377323f, -0.00571829f, 1.05850f, -0.377323f, 
		-0.00571829f, 1.05850f, -0.377323f, -0.00571829f, 1.05850f, -0.377323f, 
		-0.00571829f, 1.05850f, -0.377323f, -0.00571829f, 1.05850f, -0.377323f, 
		-0.00571829f, 1.05850f, -0.377323f, -0.00571829f, 1.05850f, -0.377323f, 
		-0.00571829f, 1.05850f, -0.377323f, -0.00571807f, 1.03540f, -0.377323f, 
		-0.00571807f, 1.03540f, -0.377323f, -0.00571807f, 1.03540f, -0.377323f, 
		-0.00571807f, 1.03540f, -0.377323f, -0.00571807f, 1.03540f, -0.377323f, 
		-0.00571807f, 1.03540f, -0.377323f, -0.00571807f, 1.03540f, -0.377323f, 
		-0.00571807f, 1.03540f, -0.377323f, -0.00571807f, 1.03540f, -0.377323f, 
		-0.00571807f, 1.03540f, -0.377323f, -0.00571154f, 1.42830f, 0.0241843f, 
		-0.00571154f, 1.42830f, 0.0241843f, -0.00571154f, 1.42830f, 0.0241843f, 
		-0.00571154f, 1.42830f, 0.0241843f, -0.00571154f, 1.42830f, 0.0241843f, 
		-0.00571065f, 1.43002f, 0.0261558f, -0.00571065f, 1.43002f, 0.0261558f, 
		-0.00571065f, 1.43002f, 0.0261558f, -0.00571065f, 1.43002f, 0.0261558f, 
		-0.00571065f, 1.43002f, 0.0261558f, -0.00567741f, 1.33619f, -0.0759947f, 
		-0.00567741f, 1.33619f, -0.0759947f, -0.00567741f, 1.33619f, -0.0759947f, 
		-0.00567741f, 1.33619f, -0.0759947f, -0.00567741f, 1.33619f, -0.0759947f, 
		-0.00567741f, 1.33619f, -0.0759947f, -0.00567347f, 0.989731f, -0.309335f, 
		-0.00565152f, 1.06101f, 0.262556f, -0.00565152f, 1.06101f, 0.262556f, 
		-0.00565152f, 1.06101f, 0.262556f, -0.00565152f, 1.06101f, 0.262556f, 
		-0.00565152f, 1.06101f, 0.262556f, -0.00565152f, 1.06101f, 0.262556f, 
		-0.00563594f, 1.31936f, 0.0115491f, -0.00563594f, 1.31936f, 0.0115491f, 
		-0.00563594f, 1.31936f, 0.0115491f, -0.00563594f, 1.31936f, 0.0115491f, 
		-0.00563594f, 1.31936f, 0.0115491f, -0.00563594f, 1.31936f, 0.0115491f, 
		-0.00563594f, 1.31936f, 0.0115491f, -0.00563594f, 1.31936f, 0.0115491f, 
		-0.00563594f, 1.31936f, 0.0115491f, -0.00559300f, 1.07687f, -0.267576f, 
		-0.00559300f, 1.07687f, -0.267576f, -0.00559300f, 1.07687f, -0.267576f, 
		-0.00559300f, 1.07687f, -0.267576f, -0.00555899f, 0.989732f, 0.232108f, 
		-0.00555752f, 1.03540f, 0.377233f, -0.00555752f, 1.03540f, 0.377233f, 
		-0.00555752f, 1.03540f, 0.377233f, -0.00555752f, 1.03540f, 0.377233f, 
		-0.00555752f, 1.03540f, 0.377233f, -0.00555752f, 1.03540f, 0.377233f, 
		-0.00555752f, 1.03540f, 0.377233f, -0.00555752f, 1.03540f, 0.377233f, 
		-0.00555752f, 1.03540f, 0.377233f, -0.00552723f, 1.08798f, -0.423089f, 
		-0.00552723f, 1.08798f, -0.423089f, -0.00552723f, 1.08798f, -0.423089f, 
		-0.00552723f, 1.08798f, -0.423089f, -0.00552723f, 1.08798f, -0.423089f, 
		-0.00552723f, 1.08798f, -0.423089f, -0.00552723f, 1.08798f, -0.423089f, 
		-0.00552446f, 1.03233f, 0.0306743f, -0.00552446f, 1.03233f, 0.0306743f, 
		-0.00552446f, 1.03233f, 0.0306743f, -0.00552446f, 1.03233f, 0.0306743f, 
		-0.00552446f, 1.03233f, 0.0306743f, -0.00551415f, 1.20497f, 0.0326578f, 
		-0.00548654f, 1.07924f, -0.392489f, -0.00547348f, 1.55466f, -0.0549959f, 
		-0.00542237f, 1.30538f, -0.0825271f, -0.00541073f, 1.09091f, 0.403882f, 
		-0.00541073f, 1.09091f, 0.403882f, -0.00541073f, 1.09091f, 0.403882f, 
		-0.00541073f, 1.09091f, 0.403882f, -0.00541073f, 1.09091f, 0.403882f, 
		-0.00541073f, 1.09091f, 0.403882f, -0.00541073f, 1.09091f, 0.403882f, 
		-0.00538740f, 1.09161f, -0.267689f, -0.00538740f, 1.09161f, -0.267689f, 
		-0.00538740f, 1.09161f, -0.267689f, -0.00538740f, 1.09161f, -0.267689f, 
		-0.00538740f, 1.09161f, -0.267689f, -0.00538740f, 1.09161f, -0.267689f, 
		-0.00538740f, 1.09161f, -0.267689f, -0.00538740f, 1.09161f, -0.267689f, 
		-0.00538627f, 1.30539f, 0.0824282f, -0.00535410f, 1.55749f, 0.00537518f, 
		-0.00535410f, 1.55749f, 0.00537518f, -0.00535410f, 1.55749f, 0.00537518f, 
		-0.00535410f, 1.55749f, 0.00537518f, -0.00535410f, 1.55749f, 0.00537518f, 
		-0.00535410f, 1.55749f, 0.00537518f, -0.00535410f, 1.55749f, 0.00537518f, 
		-0.00535410f, 1.55749f, 0.00537518f, -0.00535402f, 1.20548f, 0.00537523f, 
		-0.00535402f, 1.20548f, 0.00537523f, -0.00535402f, 1.20548f, 0.00537523f, 
		-0.00535402f, 1.20548f, 0.00537523f, -0.00535402f, 1.20548f, 0.00537523f, 
		-0.00535402f, 1.20548f, 0.00537523f, -0.00535402f, 1.20548f, 0.00537523f, 
		-0.00535402f, 1.20548f, 0.00537523f, -0.00535402f, 1.36468f, 0.00537524f, 
		-0.00535402f, 1.36468f, 0.00537524f, -0.00535402f, 1.36468f, 0.00537524f, 
		-0.00535402f, 1.36468f, 0.00537524f, -0.00535402f, 1.36468f, 0.00537524f, 
		-0.00535402f, 1.36468f, 0.00537524f, -0.00535402f, 1.36468f, 0.00537524f, 
		-0.00535402f, 1.36468f, 0.00537524f, -0.00533798f, 1.25509f, -0.0855867f, 
		-0.00533798f, 1.25509f, -0.0855867f, -0.00533798f, 1.25509f, -0.0855867f, 
		-0.00533798f, 1.25509f, -0.0855867f, -0.00533798f, 1.25509f, -0.0855867f, 
		-0.00533798f, 1.25509f, -0.0855867f, -0.00533798f, 1.25509f, -0.0855867f, 
		-0.00533798f, 1.25509f, -0.0855867f, -0.00532357f, 1.40333f, -0.0263926f, 
		-0.00532357f, 1.40333f, -0.0263926f, -0.00532357f, 1.40333f, -0.0263926f, 
		-0.00532357f, 1.40333f, -0.0263926f, -0.00532117f, 1.07924f, 0.392391f, 
		-0.00531209f, 1.05507f, -0.421757f, -0.00531209f, 1.05507f, -0.421757f, 
		-0.00531209f, 1.05507f, -0.421757f, -0.00531209f, 1.05507f, -0.421757f, 
		-0.00530620f, 1.32454f, -0.0696525f, -0.00530620f, 1.32454f, -0.0696525f, 
		-0.00530620f, 1.32454f, -0.0696525f, -0.00530620f, 1.32454f, -0.0696525f, 
		-0.00530620f, 1.32454f, -0.0696525f, -0.00530620f, 1.32454f, -0.0696525f, 
		-0.00530620f, 1.32454f, -0.0696525f, -0.00528756f, 0.828613f, -0.273308f, 
		-0.00528756f, 0.828613f, -0.273308f, -0.00527832f, 1.08487f, -0.405011f, 
		-0.00527832f, 1.08487f, -0.405011f, -0.00527832f, 1.08487f, -0.405011f, 
		-0.00527832f, 1.08487f, -0.405011f, -0.00527832f, 1.08487f, -0.405011f, 
		-0.00527832f, 1.08487f, -0.405011f, -0.00525688f, 1.48361f, 0.0242614f, 
		-0.00525688f, 1.48361f, 0.0242614f, -0.00525688f, 1.48361f, 0.0242614f, 
		-0.00525688f, 1.48361f, 0.0242614f, -0.00525688f, 1.48361f, 0.0242614f, 
		-0.00525688f, 1.48361f, 0.0242614f, -0.00525688f, 1.48361f, 0.0242614f, 
		-0.00520781f, 1.33295f, -0.000914334f, -0.00520781f, 1.33295f, -0.000914334f, 
		-0.00520781f, 1.33295f, -0.000914334f, -0.00520781f, 1.33295f, -0.000914334f, 
		-0.00520781f, 1.33295f, -0.000914334f, -0.00519015f, 1.32705f, -0.0862440f, 
		-0.00519015f, 1.32705f, -0.0862440f, -0.00519015f, 1.32705f, -0.0862440f, 
		-0.00517456f, 0.828613f, 0.262749f, -0.00517456f, 0.828613f, 0.262749f, 
		-0.00515333f, 1.32704f, 0.0861441f, -0.00515333f, 1.32704f, 0.0861441f, 
		-0.00515333f, 1.32704f, 0.0861441f, -0.00513256f, 1.07612f, -0.419883f, 
		-0.00513256f, 1.07612f, -0.419883f, -0.00513256f, 1.07612f, -0.419883f, 
		-0.00513256f, 1.07612f, -0.419883f, -0.00512934f, 1.45625f, 0.0261673f, 
		-0.00512934f, 1.45625f, 0.0261673f, -0.00512934f, 1.45625f, 0.0261673f, 
		-0.00512934f, 1.45625f, 0.0261673f, -0.00512934f, 1.45625f, 0.0261673f, 
		-0.00511969f, 1.52592f, -0.00330394f, -0.00511969f, 1.52592f, -0.00330394f, 
		-0.00511969f, 1.52592f, -0.00330394f, -0.00511969f, 1.52592f, -0.00330394f, 
		-0.00511969f, 1.52592f, -0.00330394f, -0.00511969f, 1.52592f, -0.00330394f, 
		-0.00511969f, 1.52592f, -0.00330394f, -0.00510805f, 1.17371f, -0.000896323f, 
		-0.00510805f, 1.17371f, -0.000896323f, -0.00510805f, 1.17371f, -0.000896323f, 
		-0.00510805f, 1.17371f, -0.000896323f, -0.00510805f, 1.17371f, -0.000896323f, 
		-0.00503415f, 1.06095f, -0.262147f, -0.00503415f, 1.06095f, -0.262147f, 
		-0.00503415f, 1.06095f, -0.262147f, -0.00503415f, 1.06095f, -0.262147f, 
		-0.00503415f, 1.06095f, -0.262147f, -0.00503415f, 1.06095f, -0.262147f, 
		-0.00503000f, 1.02272f, -0.303715f, -0.00503000f, 1.02272f, -0.303715f, 
		-0.00503000f, 1.03132f, -0.303715f, -0.00503000f, 1.03132f, -0.303715f, 
		-0.00500127f, 1.09190f, -0.274418f, -0.00500127f, 1.09190f, -0.274418f, 
		-0.00500127f, 1.09190f, -0.274418f, -0.00500127f, 1.09190f, -0.274418f, 
		-0.00500127f, 1.09190f, -0.274418f, -0.00500127f, 1.09190f, -0.274418f, 
		-0.00500127f, 1.09190f, -0.274418f, -0.00500127f, 1.09190f, -0.274418f, 
		-0.00500127f, 1.09190f, -0.274418f, -0.00500127f, 1.09190f, -0.274418f, 
		-0.00498670f, 1.30330f, -0.0183169f, -0.00498670f, 1.30330f, -0.0183169f, 
		-0.00498670f, 1.30330f, -0.0183169f, -0.00498670f, 1.30330f, -0.0183169f, 
		-0.00495867f, 1.06613f, -0.262672f, -0.00495867f, 1.06613f, -0.262672f, 
		-0.00495867f, 1.06613f, -0.262672f, -0.00495867f, 1.06613f, -0.262672f, 
		-0.00495867f, 1.06613f, -0.262672f, -0.00495867f, 1.06613f, -0.262672f, 
		-0.00495867f, 1.06613f, -0.262672f, -0.00495867f, 1.06613f, -0.262672f, 
		-0.00495867f, 1.06613f, -0.262672f, -0.00495867f, 1.06613f, -0.262672f, 
		-0.00495326f, 1.51317f, 0.0257688f, -0.00495326f, 1.51317f, 0.0257688f, 
		-0.00495326f, 1.51317f, 0.0257688f, -0.00495326f, 1.51317f, 0.0257688f, 
		-0.00495326f, 1.51317f, 0.0257688f, -0.00490680f, 1.06527f, 0.361177f, 
		-0.00490680f, 1.06527f, 0.361177f, -0.00490680f, 1.06527f, 0.361177f, 
		-0.00490680f, 1.06527f, 0.361177f, -0.00490597f, 1.02864f, 0.361177f, 
		-0.00490597f, 1.02864f, 0.361177f, -0.00490597f, 1.02864f, 0.361177f, 
		-0.00490597f, 1.02864f, 0.361177f, -0.00490597f, 1.02864f, 0.361177f, 
		-0.00487252f, 1.25568f, 0.0855893f, -0.00487252f, 1.25568f, 0.0855893f, 
		-0.00487252f, 1.25568f, 0.0855893f, -0.00487252f, 1.25568f, 0.0855893f, 
		-0.00487252f, 1.25568f, 0.0855893f, -0.00487252f, 1.25568f, 0.0855893f, 
		-0.00487252f, 1.25568f, 0.0855893f, -0.00487252f, 1.25568f, 0.0855893f, 
		-0.00486376f, 1.06616f, 0.262637f, -0.00486376f, 1.06616f, 0.262637f, 
		-0.00486376f, 1.06616f, 0.262637f, -0.00486376f, 1.06616f, 0.262637f, 
		-0.00486376f, 1.06616f, 0.262637f, -0.00486376f, 1.06616f, 0.262637f, 
		-0.00486376f, 1.06616f, 0.262637f, -0.00486376f, 1.06616f, 0.262637f, 
		-0.00478695f, 1.07640f, -0.411944f, -0.00478695f, 1.07640f, -0.411944f, 
		-0.00478695f, 1.07640f, -0.411944f, -0.00478695f, 1.07640f, -0.411944f, 
		-0.00478695f, 1.07640f, -0.411944f, -0.00478695f, 1.07640f, -0.411944f, 
		-0.00478695f, 1.07640f, -0.411944f, -0.00471846f, 1.07153f, 0.262684f, 
		-0.00471846f, 1.07153f, 0.262684f, -0.00471846f, 1.07153f, 0.262684f, 
		-0.00471846f, 1.07153f, 0.262684f, -0.00469752f, 1.06626f, -0.266131f, 
		-0.00469752f, 1.06626f, -0.266131f, -0.00469752f, 1.06626f, -0.266131f, 
		-0.00469752f, 1.06626f, -0.266131f, -0.00469752f, 1.06626f, -0.266131f, 
		-0.00469752f, 1.06626f, -0.266131f, -0.00469752f, 1.07141f, -0.266131f, 
		-0.00469752f, 1.07141f, -0.266131f, -0.00469752f, 1.07141f, -0.266131f, 
		-0.00469752f, 1.07141f, -0.266131f, -0.00469752f, 1.07141f, -0.266131f, 
		-0.00469752f, 1.07141f, -0.266131f, -0.00469100f, 0.977001f, -0.239200f, 
		-0.00463747f, 1.24747f, -0.0707698f, -0.00463747f, 1.24747f, -0.0707698f, 
		-0.00463747f, 1.24747f, -0.0707698f, -0.00463747f, 1.24747f, -0.0707698f, 
		-0.00463747f, 1.24747f, -0.0707698f, -0.00463747f, 1.24747f, -0.0707698f, 
		-0.00463747f, 1.24747f, -0.0707698f, -0.00463747f, 1.24747f, -0.0707698f, 
		-0.00462777f, 1.25669f, 0.0599733f, -0.00462777f, 1.25669f, 0.0599733f, 
		-0.00462777f, 1.25669f, 0.0599733f, -0.00462777f, 1.25669f, 0.0599733f, 
		-0.00462777f, 1.25669f, 0.0599733f, -0.00462777f, 1.25669f, 0.0599733f, 
		-0.00461898f, 1.40114f, 0.0251628f, -0.00461898f, 1.40114f, 0.0251628f, 
		-0.00461668f, 1.02502f, -0.0151305f, -0.00461668f, 1.02502f, -0.0151305f, 
		-0.00461302f, 1.07640f, 0.411844f, -0.00461302f, 1.07640f, 0.411844f, 
		-0.00461302f, 1.07640f, 0.411844f, -0.00461302f, 1.07640f, 0.411844f, 
		-0.00461302f, 1.07640f, 0.411844f, -0.00461302f, 1.07640f, 0.411844f, 
		-0.00460935f, 1.24747f, 0.0706802f, -0.00460935f, 1.24747f, 0.0706802f, 
		-0.00460935f, 1.24747f, 0.0706802f, -0.00460935f, 1.24747f, 0.0706802f, 
		-0.00460935f, 1.24747f, 0.0706802f, -0.00460935f, 1.24747f, 0.0706802f, 
		-0.00459260f, 1.24684f, 0.0649388f, -0.00459260f, 1.24684f, 0.0649388f, 
		-0.00459260f, 1.24684f, 0.0649388f, -0.00459260f, 1.24684f, 0.0649388f, 
		-0.00459260f, 1.24684f, 0.0649388f, -0.00459260f, 1.24684f, 0.0649388f, 
		-0.00458576f, 1.48297f, -0.0253757f, -0.00458576f, 1.48297f, -0.0253757f, 
		-0.00458576f, 1.48297f, -0.0253757f, -0.00458576f, 1.48297f, -0.0253757f, 
		-0.00458576f, 1.42843f, -0.0253757f, -0.00458576f, 1.42843f, -0.0253757f, 
		-0.00458576f, 1.42843f, -0.0253757f, -0.00458576f, 1.42843f, -0.0253757f, 
		-0.00458569f, 1.37395f, -0.0253756f, -0.00458569f, 1.37395f, -0.0253756f, 
		-0.00458569f, 1.37395f, -0.0253756f, -0.00458569f, 1.37395f, -0.0253756f, 
		-0.00458569f, 1.37395f, -0.0253756f, -0.00454038f, 1.24747f, 0.0840066f, 
		-0.00454038f, 1.24747f, 0.0840066f, -0.00454038f, 1.24747f, 0.0840066f, 
		-0.00454038f, 1.24747f, 0.0840066f, -0.00454038f, 1.24747f, 0.0840066f, 
		-0.00454038f, 1.24747f, 0.0840066f, -0.00452400f, 0.829148f, 0.275916f, 
		-0.00452400f, 0.829148f, 0.275916f, -0.00451931f, 1.07686f, 0.266097f, 
		-0.00451931f, 1.07686f, 0.266097f, -0.00451931f, 1.07686f, 0.266097f, 
		-0.00451931f, 1.07686f, 0.266097f, -0.00451931f, 1.07686f, 0.266097f, 
		-0.00451931f, 1.07686f, 0.266097f, -0.00440578f, 1.05832f, 0.376962f, 
		-0.00440578f, 1.05832f, 0.376962f, -0.00440578f, 1.05832f, 0.376962f, 
		-0.00440578f, 1.05832f, 0.376962f, -0.00440578f, 1.05832f, 0.376962f, 
		-0.00440578f, 1.05832f, 0.376962f, -0.00440578f, 1.05832f, 0.376962f, 
		-0.00440578f, 1.05832f, 0.376962f, -0.00440578f, 1.05832f, 0.376962f, 
		-0.00438762f, 1.09162f, 0.266205f, -0.00438762f, 1.09162f, 0.266205f, 
		-0.00438762f, 1.09162f, 0.266205f, -0.00438762f, 1.09162f, 0.266205f, 
		-0.00438762f, 1.09162f, 0.266205f, -0.00438762f, 1.09162f, 0.266205f, 
		-0.00432057f, 1.08201f, -0.409954f, -0.00432057f, 1.08201f, -0.409954f, 
		-0.00432057f, 1.08201f, -0.409954f, -0.00432057f, 1.08201f, -0.409954f, 
		-0.00432057f, 1.08201f, -0.409954f, -0.00425716f, 0.829042f, -0.259810f, 
		-0.00425716f, 0.829042f, -0.259810f, -0.00425716f, 0.829042f, -0.259810f, 
		-0.00425064f, 1.19306f, 0.00740049f, -0.00425064f, 1.19306f, 0.00740049f, 
		-0.00425064f, 1.19306f, 0.00740049f, -0.00425064f, 1.19306f, 0.00740049f, 
		-0.00425051f, 1.54507f, 0.00740046f, -0.00425051f, 1.54507f, 0.00740046f, 
		-0.00425051f, 1.54507f, 0.00740046f, -0.00425051f, 1.54507f, 0.00740046f, 
		-0.00425044f, 1.35226f, 0.00740042f, -0.00425044f, 1.35226f, 0.00740042f, 
		-0.00425044f, 1.35226f, 0.00740042f, -0.00425044f, 1.35226f, 0.00740042f, 
		-0.00424674f, 0.977136f, 0.300720f, -0.00424613f, 1.32312f, -0.0741587f, 
		-0.00424613f, 1.32312f, -0.0741587f, -0.00424613f, 1.32312f, -0.0741587f, 
		-0.00424613f, 1.32312f, -0.0741587f, -0.00418877f, 1.03132f, -0.237384f, 
		-0.00418877f, 1.03132f, -0.237384f, -0.00418877f, 1.03132f, -0.237384f, 
		-0.00418877f, 1.03132f, -0.237384f, -0.00418745f, 1.34650f, 0.00134576f, 
		-0.00418745f, 1.34650f, 0.00134576f, -0.00418745f, 1.34650f, 0.00134576f, 
		-0.00418745f, 1.34650f, 0.00134576f, -0.00418745f, 1.34650f, 0.00134576f, 
		-0.00418745f, 1.34650f, 0.00134576f, -0.00418745f, 1.34650f, 0.00134576f, 
		-0.00418745f, 1.34650f, 0.00134576f, -0.00418745f, 1.34650f, 0.00134576f, 
		-0.00418735f, 1.18730f, 0.00134573f, -0.00418735f, 1.18730f, 0.00134573f, 
		-0.00418735f, 1.18730f, 0.00134573f, -0.00418735f, 1.18730f, 0.00134573f, 
		-0.00418735f, 1.18730f, 0.00134573f, -0.00418735f, 1.18730f, 0.00134573f, 
		-0.00418735f, 1.18730f, 0.00134573f, -0.00418735f, 1.18730f, 0.00134573f, 
		-0.00418735f, 1.18730f, 0.00134573f, -0.00418735f, 1.18730f, 0.00134573f, 
		-0.00418735f, 1.18730f, 0.00134573f, -0.00418214f, 0.956532f, 0.252628f, 
		-0.00418214f, 0.956532f, 0.252628f, -0.00418214f, 0.956532f, 0.252628f, 
		-0.00418214f, 0.956532f, 0.252628f, -0.00417729f, 0.956532f, 0.283329f, 
		-0.00417729f, 0.956532f, 0.283329f, -0.00417729f, 0.956532f, 0.283329f, 
		-0.00416901f, 1.24347f, -0.0600970f, -0.00416901f, 1.24347f, -0.0600970f, 
		-0.00416901f, 1.24347f, -0.0600970f, -0.00416901f, 1.24347f, -0.0600970f, 
		-0.00416901f, 1.24347f, -0.0600970f, -0.00416901f, 1.24347f, -0.0600970f, 
		-0.00416901f, 1.24347f, -0.0600970f, -0.00416901f, 1.24347f, -0.0600970f, 
		-0.00416901f, 1.24347f, -0.0600970f, -0.00416901f, 1.24347f, -0.0600970f, 
		-0.00415548f, 1.24601f, -0.0650264f, -0.00415548f, 1.24601f, -0.0650264f, 
		-0.00415548f, 1.24601f, -0.0650264f, -0.00415548f, 1.24601f, -0.0650264f, 
		-0.00415548f, 1.24601f, -0.0650264f, -0.00415548f, 1.24601f, -0.0650264f, 
		-0.00415548f, 1.24601f, -0.0650264f, -0.00415548f, 1.24601f, -0.0650264f, 
		-0.00414655f, 1.08201f, 0.409854f, -0.00414655f, 1.08201f, 0.409854f, 
		-0.00414655f, 1.08201f, 0.409854f, -0.00414655f, 1.08201f, 0.409854f, 
		-0.00414655f, 1.08201f, 0.409854f, -0.00413954f, 1.05018f, 0.425531f, 
		-0.00413954f, 1.05018f, 0.425531f, -0.00413954f, 1.05018f, 0.425531f, 
		-0.00413954f, 1.05018f, 0.425531f, -0.00413954f, 1.05018f, 0.425531f, 
		-0.00413954f, 1.05018f, 0.425531f, -0.00413954f, 1.05018f, 0.425531f, 
		-0.00413954f, 1.05018f, 0.425531f, -0.00413954f, 1.05018f, 0.425531f, 
		-0.00413954f, 1.05018f, 0.425531f, -0.00413954f, 1.05018f, 0.425531f, 
		-0.00412880f, 1.02266f, -0.291606f, -0.00412880f, 1.02266f, -0.291606f, 
		-0.00405840f, 1.54317f, 0.00103079f, -0.00405840f, 1.54317f, 0.00103079f, 
		-0.00405840f, 1.54317f, 0.00103079f, -0.00405840f, 1.54317f, 0.00103079f, 
		-0.00405840f, 1.54317f, 0.00103079f, -0.00405840f, 1.54317f, 0.00103079f, 
		-0.00405840f, 1.54317f, 0.00103079f, -0.00405840f, 1.54317f, 0.00103079f, 
		-0.00405840f, 1.54317f, 0.00103079f, -0.00405840f, 1.54317f, 0.00103079f, 
		-0.00404401f, 1.19303f, -0.00753254f, -0.00404401f, 1.19303f, -0.00753254f, 
		-0.00404401f, 1.19303f, -0.00753254f, -0.00404401f, 1.19303f, -0.00753254f, 
		-0.00404241f, 1.54502f, -0.00753179f, -0.00404241f, 1.54502f, -0.00753179f, 
		-0.00404241f, 1.54502f, -0.00753179f, -0.00404241f, 1.54502f, -0.00753179f, 
		-0.00404221f, 1.35221f, -0.00753186f, -0.00404221f, 1.35221f, -0.00753186f, 
		-0.00404221f, 1.35221f, -0.00753186f, -0.00404221f, 1.35221f, -0.00753186f, 
		-0.00402130f, 1.33605f, 0.0728181f, -0.00402130f, 1.33605f, 0.0728181f, 
		-0.00402130f, 1.33605f, 0.0728181f, -0.00402130f, 1.33605f, 0.0728181f, 
		-0.00396230f, 1.03411f, -0.384774f, -0.00396230f, 1.03411f, -0.384774f, 
		-0.00396230f, 1.03411f, -0.384774f, -0.00396230f, 1.03411f, -0.384774f, 
		-0.00396230f, 1.03411f, -0.384774f, -0.00396230f, 1.03411f, -0.384774f, 
		-0.00396230f, 1.03411f, -0.384774f, -0.00396230f, 1.03411f, -0.384774f, 
		-0.00396230f, 1.03411f, -0.384774f, -0.00396230f, 1.03411f, -0.384774f, 
		-0.00396180f, 1.05980f, -0.384774f, -0.00396180f, 1.05980f, -0.384774f, 
		-0.00396180f, 1.05980f, -0.384774f, -0.00396180f, 1.05980f, -0.384774f, 
		-0.00396180f, 1.05980f, -0.384774f, -0.00396180f, 1.05980f, -0.384774f, 
		-0.00396180f, 1.05980f, -0.384774f, -0.00396180f, 1.05980f, -0.384774f, 
		-0.00393415f, 1.31792f, -0.0731447f, -0.00393415f, 1.31792f, -0.0731447f, 
		-0.00393415f, 1.31792f, -0.0731447f, -0.00391809f, 1.24413f, 0.0598527f, 
		-0.00391809f, 1.24413f, 0.0598527f, -0.00391809f, 1.24413f, 0.0598527f, 
		-0.00391809f, 1.24413f, 0.0598527f, -0.00391809f, 1.24413f, 0.0598527f, 
		-0.00391809f, 1.24413f, 0.0598527f, -0.00391809f, 1.24413f, 0.0598527f, 
		-0.00391380f, 1.04414f, -0.412585f, -0.00391380f, 1.04414f, -0.412585f, 
		-0.00391380f, 1.04414f, -0.412585f, -0.00387795f, 1.32681f, 0.0128471f, 
		-0.00387795f, 1.32681f, 0.0128471f, -0.00387795f, 1.32681f, 0.0128471f, 
		-0.00387795f, 1.32681f, 0.0128471f, -0.00387795f, 1.32681f, 0.0128471f, 
		-0.00387795f, 1.32681f, 0.0128471f, -0.00387795f, 1.32681f, 0.0128471f, 
		-0.00387795f, 1.32681f, 0.0128471f, -0.00387669f, 1.04392f, -0.412504f, 
		-0.00387669f, 1.04392f, -0.412504f, -0.00387669f, 1.04392f, -0.412504f, 
		-0.00387489f, 1.32160f, 0.0746580f, -0.00387489f, 1.32160f, 0.0746580f, 
		-0.00387489f, 1.32160f, 0.0746580f, -0.00387489f, 1.32160f, 0.0746580f, 
		-0.00387489f, 1.32160f, 0.0746580f, -0.00383000f, 1.00796f, -0.295419f, 
		-0.00383000f, 1.00796f, -0.295419f, -0.00383000f, 1.00796f, -0.295419f, 
		-0.00383000f, 1.00796f, -0.295419f, -0.00383000f, 1.00796f, -0.295419f, 
		-0.00379915f, 1.03411f, 0.384684f, -0.00379915f, 1.03411f, 0.384684f, 
		-0.00379915f, 1.03411f, 0.384684f, -0.00379915f, 1.03411f, 0.384684f, 
		-0.00379915f, 1.03411f, 0.384684f, -0.00379915f, 1.03411f, 0.384684f, 
		-0.00379915f, 1.03411f, 0.384684f, -0.00379915f, 1.03411f, 0.384684f, 
		-0.00379915f, 1.03411f, 0.384684f, -0.00379915f, 1.03411f, 0.384684f, 
		-0.00378662f, 1.34608f, -0.00226551f, -0.00378662f, 1.34608f, -0.00226551f, 
		-0.00378662f, 1.34608f, -0.00226551f, -0.00378662f, 1.34608f, -0.00226551f, 
		-0.00378662f, 1.34608f, -0.00226551f, -0.00378659f, 1.18688f, -0.00226559f, 
		-0.00378659f, 1.18688f, -0.00226559f, -0.00378659f, 1.18688f, -0.00226559f, 
		-0.00378659f, 1.18688f, -0.00226559f, -0.00378659f, 1.18688f, -0.00226559f, 
		-0.00378634f, 1.53889f, -0.00226531f, -0.00378634f, 1.53889f, -0.00226531f, 
		-0.00378634f, 1.53889f, -0.00226531f, -0.00378634f, 1.53889f, -0.00226531f, 
		-0.00378634f, 1.53889f, -0.00226531f, -0.00374658f, 1.00848f, -0.245767f, 
		-0.00374658f, 1.00848f, -0.245767f, -0.00374658f, 1.00848f, -0.245767f, 
		-0.00374042f, 1.04408f, 0.412506f, -0.00374042f, 1.04408f, 0.412506f, 
		-0.00374042f, 1.04408f, 0.412506f, -0.00374042f, 1.04408f, 0.412506f, 
		-0.00374042f, 1.04408f, 0.412506f, -0.00374042f, 1.04408f, 0.412506f, 
		-0.00370465f, 1.00796f, 0.295318f, -0.00370465f, 1.00796f, 0.295318f, 
		-0.00370465f, 1.00796f, 0.295318f, -0.00370465f, 1.00796f, 0.295318f, 
		-0.00370465f, 1.00796f, 0.295318f, -0.00365322f, 1.00045f, -0.291201f, 
		-0.00365322f, 1.00045f, -0.291201f, -0.00365103f, 1.00844f, 0.295507f, 
		-0.00365103f, 1.00844f, 0.295507f, -0.00365103f, 1.00844f, 0.295507f, 
		-0.00365008f, 1.06665f, 0.277104f, -0.00365008f, 1.06665f, 0.277104f, 
		-0.00365008f, 1.06665f, 0.277104f, -0.00365008f, 1.06665f, 0.277104f, 
		-0.00365008f, 1.06665f, 0.277104f, -0.00365008f, 1.06665f, 0.277104f, 
		-0.00365008f, 1.06665f, 0.277104f, -0.00365008f, 1.06665f, 0.277104f, 
		-0.00365008f, 1.06665f, 0.277104f, -0.00365008f, 1.06665f, 0.277104f, 
		-0.00363074f, 1.26129f, 0.0546645f, -0.00363074f, 1.26129f, 0.0546645f, 
		-0.00363074f, 1.26129f, 0.0546645f, -0.00363074f, 1.26129f, 0.0546645f, 
		-0.00363074f, 1.26129f, 0.0546645f, -0.00363074f, 1.26129f, 0.0546645f, 
		-0.00363074f, 1.26129f, 0.0546645f, -0.00363074f, 1.26129f, 0.0546645f, 
		-0.00360900f, 1.26127f, -0.0547702f, -0.00360900f, 1.26127f, -0.0547702f, 
		-0.00360900f, 1.26127f, -0.0547702f, -0.00360900f, 1.26127f, -0.0547702f, 
		-0.00360900f, 1.26127f, -0.0547702f, -0.00360900f, 1.26127f, -0.0547702f, 
		-0.00360900f, 1.26127f, -0.0547702f, -0.00360900f, 1.26127f, -0.0547702f, 
		-0.00358952f, 1.24619f, -0.0600715f, -0.00358952f, 1.24619f, -0.0600715f, 
		-0.00358952f, 1.24619f, -0.0600715f, -0.00358952f, 1.24619f, -0.0600715f, 
		-0.00358952f, 1.24619f, -0.0600715f, -0.00358952f, 1.24619f, -0.0600715f, 
		-0.00356638f, 1.35037f, 0.00155709f, -0.00356638f, 1.35037f, 0.00155709f, 
		-0.00356638f, 1.35037f, 0.00155709f, -0.00356638f, 1.35037f, 0.00155709f, 
		-0.00356638f, 1.35037f, 0.00155709f, -0.00356638f, 1.35037f, 0.00155709f, 
		-0.00356630f, 1.53969f, 0.00133245f, -0.00356630f, 1.53969f, 0.00133245f, 
		-0.00356630f, 1.53969f, 0.00133245f, -0.00356630f, 1.53969f, 0.00133245f, 
		-0.00356630f, 1.53969f, 0.00133245f, -0.00356630f, 1.53969f, 0.00133245f, 
		-0.00356630f, 1.53969f, 0.00133245f, -0.00356630f, 1.53969f, 0.00133245f, 
		-0.00356630f, 1.53969f, 0.00133245f, -0.00352118f, 1.06626f, 0.264817f, 
		-0.00352118f, 1.06626f, 0.264817f, -0.00352118f, 1.06626f, 0.264817f, 
		-0.00352118f, 1.06626f, 0.264817f, -0.00352118f, 1.06626f, 0.264817f, 
		-0.00352118f, 1.06626f, 0.264817f, -0.00352118f, 1.07141f, 0.264817f, 
		-0.00352118f, 1.07141f, 0.264817f, -0.00352118f, 1.07141f, 0.264817f, 
		-0.00352118f, 1.07141f, 0.264817f, -0.00352118f, 1.07141f, 0.264817f, 
		-0.00352118f, 1.07141f, 0.264817f, -0.00342537f, 1.05750f, 0.416880f, 
		-0.00342537f, 1.05750f, 0.416880f, -0.00342537f, 1.05750f, 0.416880f, 
		-0.00342064f, 0.958053f, -0.256776f, -0.00342064f, 0.958053f, -0.256776f, 
		-0.00342064f, 0.958053f, -0.256776f, -0.00342064f, 0.958053f, -0.256776f, 
		-0.00342064f, 0.958053f, -0.256776f, -0.00342064f, 0.958053f, -0.256776f, 
		-0.00341418f, 1.19107f, 0.00178521f, -0.00341418f, 1.19107f, 0.00178521f, 
		-0.00341418f, 1.19107f, 0.00178521f, -0.00341418f, 1.19107f, 0.00178521f, 
		-0.00341418f, 1.19107f, 0.00178521f, -0.00341418f, 1.19107f, 0.00178521f, 
		-0.00341418f, 1.19107f, 0.00178521f, -0.00341418f, 1.19107f, 0.00178521f, 
		-0.00332245f, 1.32275f, -0.0807647f, -0.00332245f, 1.32275f, -0.0807647f, 
		-0.00332245f, 1.32275f, -0.0807647f, -0.00332245f, 1.32275f, -0.0807647f, 
		-0.00332245f, 1.32275f, -0.0807647f, -0.00332245f, 1.32275f, -0.0807647f, 
		-0.00331523f, 1.05754f, -0.417055f, -0.00331523f, 1.05754f, -0.417055f, 
		-0.00331523f, 1.05754f, -0.417055f, -0.00331523f, 1.05754f, -0.417055f, 
		-0.00331523f, 1.05754f, -0.417055f, -0.00329414f, 0.958052f, 0.279237f, 
		-0.00329414f, 0.958052f, 0.279237f, -0.00329414f, 0.958052f, 0.279237f, 
		-0.00329414f, 0.958052f, 0.279237f, -0.00329414f, 0.958052f, 0.279237f, 
		-0.00328738f, 1.32275f, 0.0806648f, -0.00328738f, 1.32275f, 0.0806648f, 
		-0.00328738f, 1.32275f, 0.0806648f, -0.00328738f, 1.32275f, 0.0806648f, 
		-0.00328738f, 1.32275f, 0.0806648f, -0.00327080f, 1.07664f, -0.262195f, 
		-0.00327080f, 1.07664f, -0.262195f, -0.00327080f, 1.07664f, -0.262195f, 
		-0.00327080f, 1.07664f, -0.262195f, -0.00327080f, 1.07664f, -0.262195f, 
		-0.00327080f, 1.07664f, -0.262195f, -0.00324726f, 1.04397f, -0.408539f, 
		-0.00324726f, 1.04397f, -0.408539f, -0.00324726f, 1.04397f, -0.408539f, 
		-0.00324726f, 1.04397f, -0.408539f, -0.00324726f, 1.04397f, -0.408539f, 
		-0.00324726f, 1.04397f, -0.408539f, -0.00324726f, 1.04397f, -0.408539f, 
		-0.00324726f, 1.04397f, -0.408539f, -0.00321337f, 1.25764f, 0.0706939f, 
		-0.00321337f, 1.25764f, 0.0706939f, -0.00321337f, 1.25764f, 0.0706939f, 
		-0.00321337f, 1.25764f, 0.0706939f, -0.00321337f, 1.25764f, 0.0706939f, 
		-0.00321337f, 1.25764f, 0.0706939f, -0.00321337f, 1.25764f, 0.0706939f, 
		-0.00317041f, 1.07698f, 0.276264f, -0.00317041f, 1.07698f, 0.276264f, 
		-0.00317041f, 1.07698f, 0.276264f, -0.00317041f, 1.07698f, 0.276264f, 
		-0.00317041f, 1.07698f, 0.276264f, -0.00317041f, 1.07698f, 0.276264f, 
		-0.00317041f, 1.07698f, 0.276264f, -0.00315181f, 1.30094f, 0.0293525f, 
		-0.00313463f, 1.08135f, 0.423362f, -0.00313463f, 1.08135f, 0.423362f, 
		-0.00313463f, 1.08135f, 0.423362f, -0.00313463f, 1.08135f, 0.423362f, 
		-0.00313463f, 1.08135f, 0.423362f, -0.00309269f, 1.02256f, 0.291991f, 
		-0.00309269f, 1.02256f, 0.291991f, -0.00309269f, 1.02256f, 0.291991f, 
		-0.00305246f, 0.985827f, -0.236353f, -0.00300008f, 1.04412f, 0.408458f, 
		-0.00300008f, 1.04412f, 0.408458f, -0.00300008f, 1.04412f, 0.408458f, 
		-0.00300008f, 1.04412f, 0.408458f, -0.00300008f, 1.04412f, 0.408458f, 
		-0.00300008f, 1.04412f, 0.408458f, -0.00296425f, 1.26202f, 0.0660688f, 
		-0.00296425f, 1.26202f, 0.0660688f, -0.00296425f, 1.26202f, 0.0660688f, 
		-0.00296425f, 1.26202f, 0.0660688f, -0.00296425f, 1.26202f, 0.0660688f, 
		-0.00296425f, 1.26202f, 0.0660688f, -0.00296425f, 1.26202f, 0.0660688f, 
		-0.00296425f, 1.26202f, 0.0660688f, -0.00295635f, 1.53200f, -0.0111189f, 
		-0.00295635f, 1.53200f, -0.0111189f, -0.00295635f, 1.53200f, -0.0111189f, 
		-0.00294341f, 1.04261f, -0.425704f, -0.00294341f, 1.04261f, -0.425704f, 
		-0.00294341f, 1.04261f, -0.425704f, -0.00294341f, 1.04261f, -0.425704f, 
		-0.00294341f, 1.04261f, -0.425704f, -0.00294341f, 1.04261f, -0.425704f, 
		-0.00294341f, 1.04261f, -0.425704f, -0.00294341f, 1.04261f, -0.425704f, 
		-0.00294341f, 1.04261f, -0.425704f, -0.00293469f, 0.985827f, 0.305091f, 
		-0.00291176f, 1.24181f, 0.0562717f, -0.00291176f, 1.24181f, 0.0562717f, 
		-0.00291176f, 1.24181f, 0.0562717f, -0.00291176f, 1.24181f, 0.0562717f, 
		-0.00291176f, 1.24181f, 0.0562717f, -0.00291176f, 1.24181f, 0.0562717f, 
		-0.00291176f, 1.24181f, 0.0562717f, -0.00291176f, 1.24181f, 0.0562717f, 
		-0.00291176f, 1.24181f, 0.0562717f, -0.00291176f, 1.24181f, 0.0562717f, 
		-0.00291176f, 1.24181f, 0.0562717f, -0.00290944f, 1.18450f, 0.00958746f, 
		-0.00290944f, 1.18450f, 0.00958746f, -0.00290944f, 1.18450f, 0.00958746f, 
		-0.00290944f, 1.18450f, 0.00958746f, -0.00290944f, 1.18450f, 0.00958746f, 
		-0.00288628f, 1.26222f, -0.0669020f, -0.00288628f, 1.26222f, -0.0669020f, 
		-0.00288628f, 1.26222f, -0.0669020f, -0.00288628f, 1.26222f, -0.0669020f, 
		-0.00288628f, 1.26222f, -0.0669020f, -0.00288628f, 1.26222f, -0.0669020f, 
		-0.00287479f, 1.32650f, 0.0665942f, -0.00287479f, 1.32650f, 0.0665942f, 
		-0.00287460f, 1.03669f, 0.414905f, -0.00287460f, 1.03669f, 0.414905f, 
		-0.00285918f, 1.05560f, -0.415137f, -0.00285918f, 1.05560f, -0.415137f, 
		-0.00285918f, 1.05560f, -0.415137f, -0.00285918f, 1.05560f, -0.415137f, 
		-0.00278715f, 1.09484f, -0.417527f, -0.00278715f, 1.09484f, -0.417527f, 
		-0.00278715f, 1.09484f, -0.417527f, -0.00278715f, 1.09484f, -0.417527f, 
		-0.00278525f, 1.05551f, 0.415054f, -0.00278525f, 1.05551f, 0.415054f, 
		-0.00278525f, 1.05551f, 0.415054f, -0.00278525f, 1.05551f, 0.415054f, 
		-0.00274753f, 1.08891f, -0.409955f, -0.00274753f, 1.08891f, -0.409955f, 
		-0.00274753f, 1.08891f, -0.409955f, -0.00274753f, 1.08891f, -0.409955f, 
		-0.00274753f, 1.08891f, -0.409955f, -0.00272380f, 1.24180f, 0.0668443f, 
		-0.00272380f, 1.24180f, 0.0668443f, -0.00272380f, 1.24180f, 0.0668443f, 
		-0.00272380f, 1.24180f, 0.0668443f, -0.00272380f, 1.24180f, 0.0668443f, 
		-0.00272380f, 1.24180f, 0.0668443f, -0.00269580f, 1.07350f, 0.280923f, 
		-0.00269580f, 1.07350f, 0.280923f, -0.00269580f, 1.07350f, 0.280923f, 
		-0.00269580f, 1.07350f, 0.280923f, -0.00269580f, 1.07350f, 0.280923f, 
		-0.00269580f, 1.07350f, 0.280923f, -0.00269580f, 1.07350f, 0.280923f, 
		-0.00269580f, 1.07350f, 0.280923f, -0.00267153f, 1.33231f, -0.0668329f, 
		-0.00267153f, 1.33231f, -0.0668329f, -0.00267153f, 1.33231f, -0.0668329f, 
		-0.00267153f, 1.33231f, -0.0668329f, -0.00267153f, 1.33231f, -0.0668329f, 
		-0.00264979f, 1.34406f, -0.00910199f, -0.00264979f, 1.34406f, -0.00910199f, 
		-0.00264979f, 1.34406f, -0.00910199f, -0.00264979f, 1.34406f, -0.00910199f, 
		-0.00264979f, 1.34406f, -0.00910199f, -0.00264921f, 1.53687f, -0.00910083f, 
		-0.00264921f, 1.53687f, -0.00910083f, -0.00264921f, 1.53687f, -0.00910083f, 
		-0.00264921f, 1.53687f, -0.00910083f, -0.00261025f, 1.09484f, 0.417428f, 
		-0.00261025f, 1.09484f, 0.417428f, -0.00261025f, 1.09484f, 0.417428f, 
		-0.00261025f, 1.09484f, 0.417428f, -0.00261025f, 1.09484f, 0.417428f, 
		-0.00257473f, 1.08891f, 0.409854f, -0.00257473f, 1.08891f, 0.409854f, 
		-0.00257473f, 1.08891f, 0.409854f, -0.00253697f, 1.07687f, -0.276896f, 
		-0.00253697f, 1.07687f, -0.276896f, -0.00253697f, 1.07687f, -0.276896f, 
		-0.00253697f, 1.07687f, -0.276896f, -0.00253697f, 1.07687f, -0.276896f, 
		-0.00253697f, 1.07687f, -0.276896f, -0.00253423f, 1.07687f, -0.264650f, 
		-0.00253423f, 1.07687f, -0.264650f, -0.00253423f, 1.07687f, -0.264650f, 
		-0.00253423f, 1.07687f, -0.264650f, -0.00253423f, 1.07687f, -0.264650f, 
		-0.00253423f, 1.07687f, -0.264650f, -0.00251322f, 1.03510f, -0.417189f, 
		-0.00251322f, 1.03510f, -0.417189f, -0.00251322f, 1.03510f, -0.417189f, 
		-0.00251322f, 1.03510f, -0.417189f, -0.00249331f, 1.25817f, -0.0706829f, 
		-0.00249331f, 1.25817f, -0.0706829f, -0.00249331f, 1.25817f, -0.0706829f, 
		-0.00249331f, 1.25817f, -0.0706829f, -0.00249331f, 1.25817f, -0.0706829f, 
		-0.00249331f, 1.25817f, -0.0706829f, -0.00246391f, 1.05676f, 0.421477f, 
		-0.00246391f, 1.05676f, 0.421477f, -0.00246391f, 1.05676f, 0.421477f, 
		-0.00245265f, 1.08989f, -0.276909f, -0.00245265f, 1.08989f, -0.276909f, 
		-0.00245265f, 1.08989f, -0.276909f, -0.00245265f, 1.08989f, -0.276909f, 
		-0.00245265f, 1.08989f, -0.276909f, -0.00245265f, 1.08989f, -0.276909f, 
		-0.00244326f, 1.07626f, -0.403932f, -0.00244326f, 1.07626f, -0.403932f, 
		-0.00244326f, 1.07626f, -0.403932f, -0.00244326f, 1.07626f, -0.403932f, 
		-0.00244326f, 1.07626f, -0.403932f, -0.00241210f, 1.24591f, -0.0837109f, 
		-0.00241210f, 1.24591f, -0.0837109f, -0.00241210f, 1.24591f, -0.0837109f, 
		-0.00241210f, 1.24591f, -0.0837109f, -0.00241210f, 1.24591f, -0.0837109f, 
		-0.00241210f, 1.24591f, -0.0837109f, -0.00241210f, 1.24591f, -0.0837109f, 
		-0.00241210f, 1.24591f, -0.0837109f, -0.00241079f, 1.09030f, -0.264641f, 
		-0.00241079f, 1.09030f, -0.264641f, -0.00241079f, 1.09030f, -0.264641f, 
		-0.00241079f, 1.09030f, -0.264641f, -0.00241079f, 1.09030f, -0.264641f, 
		-0.00241079f, 1.09030f, -0.264641f, -0.00240459f, 1.05473f, -0.405477f, 
		-0.00240459f, 1.05473f, -0.405477f, -0.00240459f, 1.05473f, -0.405477f, 
		-0.00240459f, 1.05473f, -0.405477f, -0.00240459f, 1.05473f, -0.405477f, 
		-0.00240459f, 1.05473f, -0.405477f, -0.00239501f, 1.05092f, -0.425763f, 
		-0.00239501f, 1.05092f, -0.425763f, -0.00239501f, 1.05092f, -0.425763f, 
		-0.00239501f, 1.05092f, -0.425763f, -0.00239501f, 1.05092f, -0.425763f, 
		-0.00239501f, 1.05092f, -0.425763f, -0.00239501f, 1.05092f, -0.425763f, 
		-0.00239501f, 1.05092f, -0.425763f, -0.00239501f, 1.05092f, -0.425763f, 
		-0.00239501f, 1.05092f, -0.425763f, -0.00239501f, 1.05092f, -0.425763f, 
		-0.00237658f, 1.25819f, 0.0836104f, -0.00237658f, 1.25819f, 0.0836104f, 
		-0.00237658f, 1.25819f, 0.0836104f, -0.00237658f, 1.25819f, 0.0836104f, 
		-0.00237658f, 1.25819f, 0.0836104f, -0.00237658f, 1.25819f, 0.0836104f, 
		-0.00237658f, 1.25819f, 0.0836104f, -0.00237658f, 1.25819f, 0.0836104f, 
		-0.00237658f, 1.25819f, 0.0836104f, -0.00237278f, 1.25817f, -0.0841692f, 
		-0.00237278f, 1.25817f, -0.0841692f, -0.00237278f, 1.25817f, -0.0841692f, 
		-0.00237278f, 1.25817f, -0.0841692f, -0.00237278f, 1.25817f, -0.0841692f, 
		-0.00237278f, 1.25817f, -0.0841692f, -0.00235422f, 1.04170f, 0.425841f, 
		-0.00235422f, 1.04170f, 0.425841f, -0.00235422f, 1.04170f, 0.425841f, 
		-0.00235422f, 1.04170f, 0.425841f, -0.00235422f, 1.04170f, 0.425841f, 
		-0.00235422f, 1.04170f, 0.425841f, -0.00235422f, 1.04170f, 0.425841f, 
		-0.00235422f, 1.04170f, 0.425841f, -0.00235422f, 1.04170f, 0.425841f, 
		-0.00235422f, 1.04170f, 0.425841f, -0.00233972f, 1.31682f, 0.0854064f, 
		-0.00233972f, 1.31682f, 0.0854064f, -0.00233972f, 1.31682f, 0.0854064f, 
		-0.00233972f, 1.31682f, 0.0854064f, -0.00233786f, 1.09016f, 0.276791f, 
		-0.00233786f, 1.09016f, 0.276791f, -0.00233786f, 1.09016f, 0.276791f, 
		-0.00233786f, 1.09016f, 0.276791f, -0.00233786f, 1.09016f, 0.276791f, 
		-0.00233786f, 1.09016f, 0.276791f, -0.00233786f, 1.09016f, 0.276791f, 
		-0.00233786f, 1.09016f, 0.276791f, -0.00233786f, 1.09016f, 0.276791f, 
		-0.00233238f, 1.04909f, -0.408324f, -0.00233238f, 1.04909f, -0.408324f, 
		-0.00233238f, 1.04909f, -0.408324f, -0.00233238f, 1.04909f, -0.408324f, 
		-0.00233238f, 1.04909f, -0.408324f, -0.00233238f, 1.04909f, -0.408324f, 
		-0.00219454f, 1.25925f, -0.0604903f, -0.00219454f, 1.25925f, -0.0604903f, 
		-0.00219454f, 1.25925f, -0.0604903f, -0.00219454f, 1.25925f, -0.0604903f, 
		-0.00219454f, 1.25925f, -0.0604903f, -0.00219454f, 1.25925f, -0.0604903f, 
		-0.00219454f, 1.25925f, -0.0604903f, -0.00219454f, 1.25925f, -0.0604903f, 
		-0.00219454f, 1.25925f, -0.0604903f, -0.00219454f, 1.25925f, -0.0604903f, 
		-0.00219454f, 1.25925f, -0.0604903f, -0.00219454f, 1.25925f, -0.0604903f, 
		-0.00211577f, 1.03775f, 0.406431f, -0.00211577f, 1.03775f, 0.406431f, 
		-0.00211577f, 1.03775f, 0.406431f, -0.00211577f, 1.03775f, 0.406431f, 
		-0.00211577f, 1.03775f, 0.406431f, -0.00211577f, 1.03775f, 0.406431f, 
		-0.00211341f, 0.985827f, 0.236190f, -0.00209988f, 1.36468f, -0.00734203f, 
		-0.00209988f, 1.36468f, -0.00734203f, -0.00209988f, 1.36468f, -0.00734203f, 
		-0.00209988f, 1.36468f, -0.00734203f, -0.00209988f, 1.36468f, -0.00734203f, 
		-0.00209988f, 1.36468f, -0.00734203f, -0.00209988f, 1.36468f, -0.00734203f, 
		-0.00209988f, 1.36468f, -0.00734203f, -0.00209987f, 1.20548f, -0.00734204f, 
		-0.00209987f, 1.20548f, -0.00734204f, -0.00209987f, 1.20548f, -0.00734204f, 
		-0.00209987f, 1.20548f, -0.00734204f, -0.00209987f, 1.20548f, -0.00734204f, 
		-0.00209987f, 1.20548f, -0.00734204f, -0.00209987f, 1.20548f, -0.00734204f, 
		-0.00209987f, 1.20548f, -0.00734204f, -0.00209887f, 1.55749f, -0.00734274f, 
		-0.00209887f, 1.55749f, -0.00734274f, -0.00209887f, 1.55749f, -0.00734274f, 
		-0.00209887f, 1.55749f, -0.00734274f, -0.00209887f, 1.55749f, -0.00734274f, 
		-0.00209887f, 1.55749f, -0.00734274f, -0.00209887f, 1.55749f, -0.00734274f, 
		-0.00209887f, 1.55749f, -0.00734274f, -0.00209597f, 1.24290f, 0.0702758f, 
		-0.00209597f, 1.24290f, 0.0702758f, -0.00209597f, 1.24290f, 0.0702758f, 
		-0.00209597f, 1.24290f, 0.0702758f, -0.00209597f, 1.24290f, 0.0702758f, 
		-0.00209597f, 1.24290f, 0.0702758f, -0.00207154f, 1.05352f, -0.394012f, 
		-0.00207154f, 1.05352f, -0.394012f, -0.00207154f, 1.05352f, -0.394012f, 
		-0.00207154f, 1.05352f, -0.394012f, -0.00207154f, 1.05352f, -0.394012f, 
		-0.00207154f, 1.05352f, -0.394012f, -0.00206947f, 1.19105f, -0.00289979f, 
		-0.00206947f, 1.19105f, -0.00289979f, -0.00206947f, 1.19105f, -0.00289979f, 
		-0.00206947f, 1.19105f, -0.00289979f, -0.00206947f, 1.19105f, -0.00289979f, 
		-0.00206947f, 1.19105f, -0.00289979f, -0.00206947f, 1.19105f, -0.00289979f, 
		-0.00206947f, 1.19105f, -0.00289979f, -0.00206947f, 1.19105f, -0.00289979f, 
		-0.00206947f, 1.19105f, -0.00289979f, -0.00206947f, 1.19105f, -0.00289979f, 
		-0.00204152f, 1.24284f, -0.0704119f, -0.00204152f, 1.24284f, -0.0704119f, 
		-0.00204152f, 1.24284f, -0.0704119f, -0.00204152f, 1.24284f, -0.0704119f, 
		-0.00204152f, 1.24284f, -0.0704119f, -0.00204152f, 1.24284f, -0.0704119f, 
		-0.00204152f, 1.24284f, -0.0704119f, -0.00204152f, 1.24284f, -0.0704119f, 
		-0.00201912f, 1.30538f, -0.0696965f, -0.00200248f, 1.35029f, -0.00310379f, 
		-0.00200248f, 1.35029f, -0.00310379f, -0.00200248f, 1.35029f, -0.00310379f, 
		-0.00200248f, 1.35029f, -0.00310379f, -0.00200248f, 1.35029f, -0.00310379f, 
		-0.00200248f, 1.35029f, -0.00310379f, -0.00200248f, 1.35029f, -0.00310379f, 
		-0.00198306f, 1.06626f, -0.277065f, -0.00198306f, 1.06626f, -0.277065f, 
		-0.00198306f, 1.06626f, -0.277065f, -0.00198306f, 1.06626f, -0.277065f, 
		-0.00198306f, 1.06626f, -0.277065f, -0.00198306f, 1.06626f, -0.277065f, 
		-0.00198306f, 1.06626f, -0.277065f, -0.00198306f, 1.07141f, -0.277065f, 
		-0.00198306f, 1.07141f, -0.277065f, -0.00198306f, 1.07141f, -0.277065f, 
		-0.00198306f, 1.07141f, -0.277065f, -0.00198306f, 1.07141f, -0.277065f, 
		-0.00198306f, 1.07141f, -0.277065f, -0.00195322f, 1.31670f, -0.0853915f, 
		-0.00195322f, 1.31670f, -0.0853915f, -0.00192075f, 1.09207f, 0.392394f, 
		-0.00192075f, 1.09207f, 0.392394f, -0.00192075f, 1.09207f, 0.392394f, 
		-0.00190499f, 1.05352f, 0.394009f, -0.00190499f, 1.05352f, 0.394009f, 
		-0.00190499f, 1.05352f, 0.394009f, -0.00190499f, 1.05352f, 0.394009f, 
		-0.00190499f, 1.05352f, 0.394009f, -0.00190499f, 1.05352f, 0.394009f, 
		-0.00190499f, 1.05352f, 0.394009f, -0.00190499f, 1.05352f, 0.394009f, 
		-0.00186575f, 1.03889f, 0.394012f, -0.00186575f, 1.03889f, 0.394012f, 
		-0.00186575f, 1.03889f, 0.394012f, -0.00186575f, 1.03889f, 0.394012f, 
		-0.00186575f, 1.03889f, 0.394012f, -0.00186575f, 1.03889f, 0.394012f, 
		-0.00185002f, 1.07637f, 0.403703f, -0.00185002f, 1.07637f, 0.403703f, 
		-0.00182713f, 1.17346f, 0.00474915f, -0.00182713f, 1.17346f, 0.00474915f, 
		-0.00182713f, 1.17346f, 0.00474915f, -0.00182713f, 1.17346f, 0.00474915f, 
		-0.00182713f, 1.17346f, 0.00474915f, -0.00182713f, 1.17346f, 0.00474915f, 
		-0.00182713f, 1.17346f, 0.00474915f, -0.00182713f, 1.17346f, 0.00474915f, 
		-0.00182713f, 1.17346f, 0.00474915f, -0.00182713f, 1.17346f, 0.00474915f, 
		-0.00182713f, 1.17346f, 0.00474915f, -0.00182694f, 1.52547f, 0.00474977f, 
		-0.00182694f, 1.52547f, 0.00474977f, -0.00182694f, 1.52547f, 0.00474977f, 
		-0.00182694f, 1.52547f, 0.00474977f, -0.00182694f, 1.52547f, 0.00474977f, 
		-0.00182694f, 1.52547f, 0.00474977f, -0.00182694f, 1.52547f, 0.00474977f, 
		-0.00182694f, 1.52547f, 0.00474977f, -0.00182694f, 1.52547f, 0.00474977f, 
		-0.00182694f, 1.52547f, 0.00474977f, -0.00182694f, 1.52547f, 0.00474977f, 
		-0.00182673f, 1.33266f, 0.00474892f, -0.00182673f, 1.33266f, 0.00474892f, 
		-0.00182673f, 1.33266f, 0.00474892f, -0.00182673f, 1.33266f, 0.00474892f, 
		-0.00182673f, 1.33266f, 0.00474892f, -0.00182673f, 1.33266f, 0.00474892f, 
		-0.00182673f, 1.33266f, 0.00474892f, -0.00182673f, 1.33266f, 0.00474892f, 
		-0.00182673f, 1.33266f, 0.00474892f, -0.00182673f, 1.33266f, 0.00474892f, 
		-0.00182673f, 1.33266f, 0.00474892f, -0.00182673f, 1.33266f, 0.00474892f, 
		-0.00182673f, 1.33266f, 0.00474892f, -0.00176000f, 1.13916f, -0.0910599f, 
		-0.00176000f, 1.13916f, -0.0910599f, -0.00176000f, 1.13916f, -0.0910599f, 
		-0.00176000f, 1.13916f, -0.0910599f, -0.00174224f, 1.03880f, -0.394011f, 
		-0.00174224f, 1.03880f, -0.394011f, -0.00174224f, 1.03880f, -0.394011f, 
		-0.00174224f, 1.03880f, -0.394011f, -0.00174224f, 1.03880f, -0.394011f, 
		-0.00174224f, 1.03880f, -0.394011f, -0.00174224f, 1.03880f, -0.394011f, 
		-0.00174224f, 1.03880f, -0.394011f, -0.00167809f, 1.52592f, -0.00538009f, 
		-0.00167809f, 1.52592f, -0.00538009f, -0.00167809f, 1.52592f, -0.00538009f, 
		-0.00167809f, 1.52592f, -0.00538009f, -0.00167809f, 1.52592f, -0.00538009f, 
		-0.00167809f, 1.52592f, -0.00538009f, -0.00165636f, 1.53535f, 0.0110163f, 
		-0.00165636f, 1.53535f, 0.0110163f, -0.00165636f, 1.53535f, 0.0110163f, 
		-0.00165636f, 1.53535f, 0.0110163f, -0.00165636f, 1.53535f, 0.0110163f, 
		-0.00165636f, 1.53535f, 0.0110163f, -0.00165635f, 1.18334f, 0.0110162f, 
		-0.00165635f, 1.18334f, 0.0110162f, -0.00165635f, 1.18334f, 0.0110162f, 
		-0.00165635f, 1.18334f, 0.0110162f, -0.00165635f, 1.18334f, 0.0110162f, 
		-0.00165635f, 1.18334f, 0.0110162f, -0.00165620f, 1.34254f, 0.0110162f, 
		-0.00165620f, 1.34254f, 0.0110162f, -0.00165620f, 1.34254f, 0.0110162f, 
		-0.00165620f, 1.34254f, 0.0110162f, -0.00165620f, 1.34254f, 0.0110162f, 
		-0.00165620f, 1.34254f, 0.0110162f, -0.00158330f, 1.31801f, 0.0821971f, 
		-0.00158330f, 1.31801f, 0.0821971f, -0.00158330f, 1.31801f, 0.0821971f, 
		-0.00158330f, 1.31801f, 0.0821971f, -0.00158015f, 1.16935f, -0.0774497f, 
		-0.00158015f, 1.16935f, -0.0774497f, -0.00158015f, 1.16935f, -0.0774497f, 
		-0.00154551f, 1.17867f, 0.0767776f, -0.00154551f, 1.17867f, 0.0767776f, 
		-0.00152381f, 1.07168f, -0.261620f, -0.00152381f, 1.07168f, -0.261620f, 
		-0.00152381f, 1.07168f, -0.261620f, -0.00152381f, 1.07168f, -0.261620f, 
		-0.00149668f, 1.14521f, 0.0863309f, -0.00149668f, 1.14521f, 0.0863309f, 
		-0.00149668f, 1.14521f, 0.0863309f, -0.00149668f, 1.14521f, 0.0863309f, 
		-0.00149668f, 1.14521f, 0.0863309f, -0.00146829f, 1.08406f, -0.171439f, 
		-0.00146829f, 1.08406f, -0.171439f, -0.00144536f, 1.31898f, 0.0780024f, 
		-0.00144536f, 1.31898f, 0.0780024f, -0.00144536f, 1.31898f, 0.0780024f, 
		-0.00144536f, 1.31898f, 0.0780024f, -0.00139170f, 1.08408f, 0.169312f, 
		-0.00139170f, 1.08408f, 0.169312f, -0.00138302f, 1.11171f, 0.113275f, 
		-0.00138302f, 1.11171f, 0.113275f, -0.00138302f, 1.11171f, 0.113275f, 
		-0.00136535f, 1.09078f, 0.422821f, -0.00136535f, 1.09078f, 0.422821f, 
		-0.00136535f, 1.09078f, 0.422821f, -0.00136535f, 1.09078f, 0.422821f, 
		-0.00136535f, 1.09078f, 0.422821f, -0.00136535f, 1.09078f, 0.422821f, 
		-0.00136141f, 1.10888f, -0.116230f, -0.00136141f, 1.10888f, -0.116230f, 
		-0.00136141f, 1.10888f, -0.116230f, -0.00136141f, 1.10888f, -0.116230f, 
		-0.00131535f, 1.54301f, -0.00337295f, -0.00131535f, 1.54301f, -0.00337295f, 
		-0.00131535f, 1.54301f, -0.00337295f, -0.00131535f, 1.54301f, -0.00337295f, 
		-0.00131535f, 1.54301f, -0.00337295f, -0.00131535f, 1.54301f, -0.00337295f, 
		-0.00131535f, 1.54301f, -0.00337295f, -0.00122680f, 1.30141f, -0.0275367f, 
		-0.00122153f, 1.08583f, -0.169407f, -0.00122153f, 1.08583f, -0.169407f, 
		-0.00122153f, 1.08583f, -0.169407f, -0.00122153f, 1.08583f, -0.169407f, 
		-0.00120626f, 1.09683f, -0.130474f, -0.00120626f, 1.09683f, -0.130474f, 
		-0.00120626f, 1.09683f, -0.130474f, -0.00120626f, 1.09683f, -0.130474f, 
		-0.00120626f, 1.09683f, -0.130474f, -0.00120626f, 1.09683f, -0.130474f, 
		-0.00120414f, 1.04977f, 0.411783f, -0.00120414f, 1.04977f, 0.411783f, 
		-0.00120414f, 1.04977f, 0.411783f, -0.00120414f, 1.04977f, 0.411783f, 
		-0.00120414f, 1.04977f, 0.411783f, -0.00120414f, 1.04977f, 0.411783f, 
		-0.00120414f, 1.04977f, 0.411783f, -0.00120414f, 1.04977f, 0.411783f, 
		-0.00120414f, 1.04977f, 0.411783f, -0.00120414f, 1.04977f, 0.411783f, 
		-0.00120414f, 1.04977f, 0.411783f, -0.00118320f, 0.984655f, -0.237940f, 
		-0.00118320f, 0.984655f, -0.237940f, -0.00118320f, 0.984655f, -0.237940f, 
		-0.00118320f, 0.984655f, -0.237940f, -0.00115983f, 1.07964f, 0.405571f, 
		-0.00115983f, 1.07964f, 0.405571f, -0.00115983f, 1.07964f, 0.405571f, 
		-0.00115983f, 1.07964f, 0.405571f, -0.00115983f, 1.07964f, 0.405571f, 
		-0.00111012f, 1.40199f, -0.0195420f, -0.00111012f, 1.40199f, -0.0195420f, 
		-0.00111012f, 1.40199f, -0.0195420f, -0.00111012f, 1.40199f, -0.0195420f, 
		-0.00111012f, 1.40199f, -0.0195420f, -0.00106313f, 1.48281f, 0.0198312f, 
		-0.00106313f, 1.48281f, 0.0198312f, -0.00106313f, 1.48281f, 0.0198312f, 
		-0.00106313f, 1.48281f, 0.0198312f, -0.00103963f, 1.31850f, -0.0821985f, 
		-0.00103963f, 1.31850f, -0.0821985f, -0.00103963f, 1.31850f, -0.0821985f, 
		-0.00103963f, 1.31850f, -0.0821985f, -0.00103758f, 1.31898f, -0.0756216f, 
		-0.00103758f, 1.31898f, -0.0756216f, -0.00103758f, 1.31898f, -0.0756216f, 
		-0.00103758f, 1.31898f, -0.0756216f, -0.00102113f, 1.05011f, 0.408396f, 
		-0.00102113f, 1.05011f, 0.408396f, -0.00102113f, 1.05011f, 0.408396f, 
		-0.00102113f, 1.05011f, 0.408396f, -0.00102113f, 1.05011f, 0.408396f, 
		-0.00102113f, 1.05011f, 0.408396f, -0.00102113f, 1.05011f, 0.408396f, 
		-0.00102113f, 1.05011f, 0.408396f, -0.00102113f, 1.05011f, 0.408396f, 
		-0.00102113f, 1.05011f, 0.408396f, -0.00102113f, 1.05011f, 0.408396f, 
		-0.00100809f, 1.36525f, -0.0122164f, -0.00100809f, 1.36525f, -0.0122164f, 
		-0.00100809f, 1.36525f, -0.0122164f, -0.00100809f, 1.36525f, -0.0122164f, 
		-0.00100809f, 1.36525f, -0.0122164f, -0.000997609f, 1.42838f, -0.0196424f, 
		-0.000997609f, 1.42838f, -0.0196424f, -0.000997609f, 1.42838f, -0.0196424f, 
		-0.000997609f, 1.42838f, -0.0196424f, -0.000997609f, 1.42838f, -0.0196424f, 
		-0.000995000f, 1.09336f, 0.139389f, -0.000995000f, 1.09336f, 0.139389f, 
		-0.000995000f, 1.09336f, 0.139389f, -0.000995000f, 1.09336f, 0.139389f, 
		-0.000995000f, 1.09336f, 0.139389f, -0.000992360f, 1.08606f, -0.415658f, 
		-0.000992360f, 1.08606f, -0.415658f, -0.000992360f, 1.08606f, -0.415658f, 
		-0.000944495f, 1.52195f, 0.0130821f, -0.000944495f, 1.52195f, 0.0130821f, 
		-0.000944495f, 1.52195f, 0.0130821f, -0.000944495f, 1.52195f, 0.0130821f, 
		-0.000933822f, 1.03132f, -0.237141f, -0.000933822f, 1.03132f, -0.237141f, 
		-0.000933822f, 1.03132f, -0.237141f, -0.000849705f, 1.48588f, -0.0301638f, 
		-0.000849705f, 1.48588f, -0.0301638f, -0.000849705f, 1.48588f, -0.0301638f, 
		-0.000849705f, 1.48588f, -0.0301638f, -0.000842763f, 1.08311f, -0.415658f, 
		-0.000842763f, 1.08311f, -0.415658f, -0.000842763f, 1.08311f, -0.415658f, 
		-0.000842763f, 1.08311f, -0.415658f, -0.000837508f, 1.43053f, -0.0302214f, 
		-0.000837508f, 1.43053f, -0.0302214f, -0.000837508f, 1.43053f, -0.0302214f, 
		-0.000837508f, 1.43053f, -0.0302214f, -0.000837508f, 1.43053f, -0.0302214f, 
		-0.000837508f, 1.43053f, -0.0302214f, -0.000833489f, 1.03132f, 0.237039f, 
		-0.000833489f, 1.03132f, 0.237039f, -0.000833489f, 1.03132f, 0.237039f, 
		-0.000816361f, 1.08606f, 0.415556f, -0.000816361f, 1.08606f, 0.415556f, 
		-0.000816361f, 1.08606f, 0.415556f, -0.000799375f, 1.08624f, 0.176896f, 
		-0.000799375f, 1.08624f, 0.176896f, -0.000714298f, 1.45708f, -0.0189370f, 
		-0.000714298f, 1.45708f, -0.0189370f, -0.000682819f, 1.50873f, -0.0162152f, 
		-0.000682819f, 1.50873f, -0.0162152f, -0.000682819f, 1.50873f, -0.0162152f, 
		-0.000682819f, 1.50873f, -0.0162152f, -0.000682819f, 1.50873f, -0.0162152f, 
		-0.000682819f, 1.50873f, -0.0162152f, -0.000682819f, 1.50873f, -0.0162152f, 
		-0.000666761f, 1.08311f, 0.415556f, -0.000666761f, 1.08311f, 0.415556f, 
		-0.000666761f, 1.08311f, 0.415556f, -0.000666761f, 1.08311f, 0.415556f, 
		-0.000619034f, 1.25915f, -0.0650319f, -0.000619034f, 1.25915f, -0.0650319f, 
		-0.000619034f, 1.25915f, -0.0650319f, -0.000619034f, 1.25915f, -0.0650319f, 
		-0.000619034f, 1.25915f, -0.0650319f, -0.000619034f, 1.25915f, -0.0650319f, 
		-0.000548909f, 1.07117f, 0.277739f, -0.000548909f, 1.07117f, 0.277739f, 
		-0.000548909f, 1.07117f, 0.277739f, -0.000548909f, 1.07117f, 0.277739f, 
		-0.000548909f, 1.07117f, 0.277739f, -0.000548909f, 1.07117f, 0.277739f, 
		-0.000548909f, 1.07117f, 0.277739f, -0.000548909f, 1.07117f, 0.277739f, 
		-0.000548909f, 1.07117f, 0.277739f, -0.000548909f, 1.07117f, 0.277739f, 
		-0.000543739f, 1.00044f, -0.250084f, -0.000543739f, 1.00044f, -0.250084f, 
		-0.000539498f, 1.36039f, 0.00640950f, -0.000539498f, 1.36039f, 0.00640950f, 
		-0.000506566f, 1.39158f, -0.00915398f, -0.000506566f, 1.39158f, -0.00915398f, 
		-0.000506566f, 1.39158f, -0.00915398f, -0.000506566f, 1.39158f, -0.00915398f, 
		-0.000464854f, 1.52033f, -0.000240345f, -0.000464854f, 1.52033f, -0.000240345f, 
		-0.000464854f, 1.52033f, -0.000240345f, -0.000464854f, 1.52033f, -0.000240345f, 
		-0.000464854f, 1.52033f, -0.000240345f, -0.000464854f, 1.52033f, -0.000240345f, 
		-0.000464854f, 1.52033f, -0.000240345f, -0.000462555f, 1.11651f, -0.104996f, 
		-0.000462555f, 1.11651f, -0.104996f, -0.000462555f, 1.11651f, -0.104996f, 
		-0.000462555f, 1.11651f, -0.104996f, -0.000462555f, 1.11651f, -0.104996f, 
		-0.000462380f, 1.35981f, 0.000169237f, -0.000462380f, 1.35981f, 0.000169237f, 
		-0.000462380f, 1.35981f, 0.000169237f, -0.000462380f, 1.35981f, 0.000169237f, 
		-0.000459877f, 1.15775f, -0.0840565f, -0.000459877f, 1.15775f, -0.0840565f, 
		-0.000436254f, 1.50916f, -0.0242815f, -0.000436254f, 1.50916f, -0.0242815f, 
		-0.000417796f, 1.50013f, -0.0153242f, -0.000417796f, 1.50013f, -0.0153242f, 
		-0.000417796f, 1.50013f, -0.0153242f, -0.000394658f, 1.40152f, -0.0323034f, 
		-0.000394658f, 1.40152f, -0.0323034f, -0.000394658f, 1.40152f, -0.0323034f, 
		-0.000394658f, 1.40152f, -0.0323034f, -0.000394658f, 1.40152f, -0.0323034f, 
		-0.000394658f, 1.40152f, -0.0323034f, -0.000392525f, 1.52210f, -0.0102896f, 
		-0.000392525f, 1.52210f, -0.0102896f, -0.000391319f, 1.00049f, 0.249201f, 
		-0.000391319f, 1.00049f, 0.249201f, -0.000391319f, 1.00049f, 0.249201f, 
		-0.000391319f, 1.00049f, 0.249201f, -0.000380225f, 1.12975f, 0.0942000f, 
		-0.000380225f, 1.12975f, 0.0942000f, -0.000380225f, 1.12975f, 0.0942000f, 
		-0.000330102f, 1.17341f, -0.00531594f, -0.000330102f, 1.17341f, -0.00531594f, 
		-0.000330102f, 1.17341f, -0.00531594f, -0.000330102f, 1.17341f, -0.00531594f, 
		-0.000330102f, 1.17341f, -0.00531594f, -0.000330102f, 1.17341f, -0.00531594f, 
		-0.000330102f, 1.17341f, -0.00531594f, -0.000322118f, 1.31898f, -0.0787568f, 
		-0.000322118f, 1.31898f, -0.0787568f, -0.000322118f, 1.31898f, -0.0787568f, 
		-0.000301129f, 1.26712f, -0.0502177f, -0.000301129f, 1.26712f, -0.0502177f, 
		-0.000296572f, 1.50919f, 0.0167753f, -0.000296572f, 1.50919f, 0.0167753f, 
		-0.000290893f, 1.05077f, -0.412267f, -0.000290893f, 1.05077f, -0.412267f, 
		-0.000290893f, 1.05077f, -0.412267f, -0.000290893f, 1.05077f, -0.412267f, 
		-0.000290893f, 1.05077f, -0.412267f, -0.000290893f, 1.05077f, -0.412267f, 
		-0.000290893f, 1.05077f, -0.412267f, -0.000290893f, 1.05077f, -0.412267f, 
		-0.000290893f, 1.05077f, -0.412267f, -0.000290893f, 1.05077f, -0.412267f, 
		-0.000290893f, 1.05077f, -0.412267f, -0.000279375f, 1.26712f, 0.0502139f, 
		-0.000279375f, 1.26712f, 0.0502139f, -0.000268417f, 0.989730f, -0.309740f, 
		-0.000265579f, 1.04604f, 0.431027f, -0.000265579f, 1.04604f, 0.431027f, 
		-0.000265579f, 1.04604f, 0.431027f, -0.000265579f, 1.04604f, 0.431027f, 
		-0.000265579f, 1.04604f, 0.431027f, -0.000265579f, 1.04604f, 0.431027f, 
		-0.000265579f, 1.04604f, 0.431027f, -0.000258818f, 1.56290f, -0.0813190f, 
		-0.000225811f, 1.56290f, 0.0697685f, -0.000221957f, 1.34133f, 0.0772308f, 
		-0.000221957f, 1.34133f, 0.0772308f, -0.000221957f, 1.34133f, 0.0772308f, 
		-0.000200478f, 1.02264f, -0.249542f, -0.000200478f, 1.02264f, -0.249542f, 
		-0.000145416f, 1.21088f, -0.0413732f, -0.000144927f, 1.32745f, -0.000486075f, 
		-0.000144927f, 1.32745f, -0.000486075f, -0.000144927f, 1.32745f, -0.000486075f, 
		-0.000144927f, 1.32745f, -0.000486075f, -0.000144927f, 1.32745f, -0.000486075f, 
		-0.000144263f, 1.37966f, -0.0163713f, -0.000144263f, 1.37966f, -0.0163713f, 
		-0.000144263f, 1.37966f, -0.0163713f, -0.000144263f, 1.37966f, -0.0163713f, 
		-0.000144263f, 1.37966f, -0.0163713f, -0.000137308f, 0.989731f, 0.309640f, 
		-0.000134946f, 1.01837f, -0.247319f, -0.000131513f, 1.17781f, 0.0109593f, 
		-0.000131513f, 1.17781f, 0.0109593f, -0.000131513f, 1.17781f, 0.0109593f, 
		-0.000131510f, 1.33701f, 0.0109593f, -0.000131510f, 1.33701f, 0.0109593f, 
		-0.000128853f, 1.21088f, 0.0413695f, -0.000125019f, 1.36190f, 0.00969887f, 
		-0.000125019f, 1.36190f, 0.00969887f, -0.000115766f, 1.04633f, -0.431109f, 
		-0.000115766f, 1.04633f, -0.431109f, -0.000115766f, 1.04633f, -0.431109f, 
		-0.000115766f, 1.04633f, -0.431109f, -0.000115766f, 1.04633f, -0.431109f, 
		-0.000115766f, 1.04633f, -0.431109f, -0.000109675f, 1.07956f, -0.405084f, 
		-0.000109675f, 1.07956f, -0.405084f, -0.000109675f, 1.07956f, -0.405084f, 
		-0.000109675f, 1.07956f, -0.405084f, -0.000109675f, 1.07956f, -0.405084f, 
		-9.99597e-005f, 1.31898f, 0.0753732f, -9.99597e-005f, 1.31898f, 0.0753732f, 
		-9.99597e-005f, 1.31898f, 0.0753732f, -9.41938e-005f, 1.02264f, 0.249441f, 
		-9.41938e-005f, 1.02264f, 0.249441f, -9.41938e-005f, 1.02264f, 0.249441f, 
		-9.41938e-005f, 1.02264f, 0.249441f, -9.25590e-005f, 1.48258f, -0.0194418f, 
		-9.25590e-005f, 1.48258f, -0.0194418f, -8.70000e-005f, 0.853677f, -0.409991f, 
		-8.70000e-005f, 0.853677f, -0.409991f, -8.60000e-005f, 0.853634f, -0.405991f, 
		-8.60000e-005f, 0.853634f, -0.405991f, -8.60000e-005f, 0.879246f, -0.406290f, 
		-8.50000e-005f, 0.878415f, -0.402396f, -8.40000e-005f, 0.905134f, -0.395749f, 
		-8.34678e-005f, 1.06334f, -0.394641f, -8.34678e-005f, 1.06334f, -0.394641f, 
		-8.34678e-005f, 1.06334f, -0.394641f, -8.34678e-005f, 1.06334f, -0.394641f, 
		-8.34678e-005f, 1.06334f, -0.394641f, -8.34678e-005f, 1.06334f, -0.394641f, 
		-8.34113e-005f, 1.08484f, -0.428638f, -8.34113e-005f, 1.08484f, -0.428638f, 
		-8.34113e-005f, 1.08484f, -0.428638f, -8.33171e-005f, 1.03100f, -0.394675f, 
		-8.33171e-005f, 1.03100f, -0.394675f, -8.33171e-005f, 1.03100f, -0.394675f, 
		-8.33171e-005f, 1.03100f, -0.394675f, -8.31699e-005f, 1.06577f, -0.391600f, 
		-8.31699e-005f, 1.06577f, -0.391600f, -8.31699e-005f, 1.06577f, -0.391600f, 
		-8.31699e-005f, 1.06577f, -0.391600f, -8.31699e-005f, 1.06577f, -0.391600f, 
		-8.31699e-005f, 1.06577f, -0.391600f, -8.31699e-005f, 1.06577f, -0.391600f, 
		-8.31699e-005f, 1.06577f, -0.391600f, -8.30000e-005f, 0.903517f, -0.392155f, 
		-8.28941e-005f, 1.02858f, -0.392756f, -8.28941e-005f, 1.02858f, -0.392756f, 
		-8.28941e-005f, 1.02858f, -0.392756f, -8.28941e-005f, 1.02858f, -0.392756f, 
		-8.28941e-005f, 1.02858f, -0.392756f, -8.28941e-005f, 1.02858f, -0.392756f, 
		-8.28941e-005f, 1.02858f, -0.392756f, -8.28941e-005f, 1.02858f, -0.392756f, 
		-8.28941e-005f, 1.02858f, -0.392756f, -8.28941e-005f, 1.02858f, -0.392756f, 
		-8.23509e-005f, 1.02814f, -0.389170f, -8.23509e-005f, 1.02814f, -0.389170f, 
		-8.23509e-005f, 1.02814f, -0.389170f, -8.23509e-005f, 1.02814f, -0.389170f, 
		-8.23509e-005f, 1.02814f, -0.389170f, -8.23509e-005f, 1.02814f, -0.389170f, 
		-8.21327e-005f, 1.06339f, -0.386280f, -8.21327e-005f, 1.06339f, -0.386280f, 
		-8.20267e-005f, 1.03052f, -0.386280f, -8.20267e-005f, 1.03052f, -0.386280f, 
		-8.20267e-005f, 1.03052f, -0.386280f, -8.20267e-005f, 1.03052f, -0.386280f, 
		-8.20267e-005f, 1.03052f, -0.386280f, -8.20267e-005f, 1.03052f, -0.386280f, 
		-8.19963e-005f, 1.06552f, -0.388360f, -8.19963e-005f, 1.06552f, -0.388360f, 
		-8.19963e-005f, 1.06552f, -0.388360f, -8.19963e-005f, 1.06552f, -0.388360f, 
		-8.19963e-005f, 1.06552f, -0.388360f, -8.19963e-005f, 1.06552f, -0.388360f, 
		-8.19963e-005f, 1.06552f, -0.388360f, -8.19963e-005f, 1.06552f, -0.388360f, 
		-8.10190e-005f, 1.03289f, -0.381889f, -8.10190e-005f, 1.03289f, -0.381889f, 
		-8.10190e-005f, 1.03289f, -0.381889f, -8.10190e-005f, 1.03289f, -0.381889f, 
		-8.10190e-005f, 1.03289f, -0.381889f, -8.10190e-005f, 1.03289f, -0.381889f, 
		-8.10190e-005f, 1.03289f, -0.381889f, -8.10190e-005f, 1.03289f, -0.381889f, 
		-8.09342e-005f, 1.06102f, -0.381889f, -8.09342e-005f, 1.06102f, -0.381889f, 
		-8.09342e-005f, 1.06102f, -0.381889f, -8.09342e-005f, 1.06102f, -0.381889f, 
		-8.09342e-005f, 1.06102f, -0.381889f, -8.09342e-005f, 1.06102f, -0.381889f, 
		-8.01716e-005f, 1.06540f, -0.376986f, -8.01716e-005f, 1.06540f, -0.376986f, 
		-8.01664e-005f, 1.02851f, -0.376986f, -8.01664e-005f, 1.02851f, -0.376986f, 
		-8.01616e-005f, 1.06033f, -0.378669f, -8.01616e-005f, 1.06033f, -0.378669f, 
		-8.01616e-005f, 1.06033f, -0.378669f, -8.01616e-005f, 1.06033f, -0.378669f, 
		-8.01616e-005f, 1.06033f, -0.378669f, -8.01616e-005f, 1.06033f, -0.378669f, 
		-8.00000e-005f, 0.927433f, -0.376089f, -8.00000e-005f, 0.929791f, -0.379213f, 
		-7.97445e-005f, 1.02546f, -0.375840f, -7.97445e-005f, 1.02546f, -0.375840f, 
		-7.95557e-005f, 1.06845f, -0.375840f, -7.95557e-005f, 1.06845f, -0.375840f, 
		-7.91822e-005f, 1.02037f, 0.293440f, -7.91822e-005f, 1.02037f, 0.293440f, 
		-7.91822e-005f, 1.02037f, 0.293440f, -7.90748e-005f, 1.02322f, -0.373553f, 
		-7.90748e-005f, 1.02322f, -0.373553f, -7.89607e-005f, 1.07069f, -0.373553f, 
		-7.89607e-005f, 1.07069f, -0.373553f, -7.83518e-005f, 1.02209f, -0.369870f, 
		-7.83518e-005f, 1.02209f, -0.369870f, -7.82576e-005f, 1.07181f, -0.369870f, 
		-7.82576e-005f, 1.07181f, -0.369870f, -7.80000e-005f, 1.02228f, -0.366963f, 
		-7.80000e-005f, 1.02228f, -0.366963f, -7.80000e-005f, 1.06104f, -0.370002f, 
		-7.80000e-005f, 1.06104f, -0.370002f, -7.80000e-005f, 1.07163f, -0.366963f, 
		-7.80000e-005f, 1.07163f, -0.366963f, -7.80000e-005f, 1.03104f, -0.370002f, 
		-7.80000e-005f, 1.03104f, -0.370002f, -7.73267e-005f, 1.45702f, 0.0311630f, 
		-7.73267e-005f, 1.45702f, 0.0311630f, -7.73248e-005f, 1.40248f, 0.0311630f, 
		-7.73248e-005f, 1.40248f, 0.0311630f, -7.70676e-005f, 1.06845f, -0.362447f, 
		-7.70676e-005f, 1.06845f, -0.362447f, -7.70488e-005f, 1.02546f, -0.362447f, 
		-7.70488e-005f, 1.02546f, -0.362447f, -7.69280e-005f, 1.07069f, -0.364734f, 
		-7.69280e-005f, 1.07069f, -0.364734f, -7.69022e-005f, 1.02322f, -0.364734f, 
		-7.69022e-005f, 1.02322f, -0.364734f, -7.62386e-005f, 1.52499f, 0.00203832f, 
		-7.62386e-005f, 1.52499f, 0.00203832f, -7.60000e-005f, 0.951668f, -0.357525f, 
		-7.50000e-005f, 0.948656f, -0.355016f, -7.00000e-005f, 0.965681f, -0.329756f, 
		-7.00000e-005f, 0.969213f, -0.331529f, -6.50000e-005f, 0.858925f, -0.307022f, 
		-6.50000e-005f, 0.858925f, -0.307022f, -6.50000e-005f, 0.858925f, -0.307022f, 
		-6.50000e-005f, 0.868091f, -0.308080f, -6.50000e-005f, 0.868091f, -0.308080f, 
		-6.50000e-005f, 0.868091f, -0.308080f, -6.50000e-005f, 0.872834f, -0.307801f, 
		-6.50000e-005f, 0.872834f, -0.307801f, -6.50000e-005f, 0.872834f, -0.307801f, 
		-6.50000e-005f, 0.877414f, -0.306984f, -6.50000e-005f, 0.877414f, -0.306984f, 
		-6.50000e-005f, 0.877414f, -0.306984f, -6.50000e-005f, 0.881798f, -0.305661f, 
		-6.50000e-005f, 0.881798f, -0.305661f, -6.50000e-005f, 0.881798f, -0.305661f, 
		-6.40000e-005f, 0.850508f, -0.304011f, -6.40000e-005f, 0.850508f, -0.304011f, 
		-6.40000e-005f, 0.850508f, -0.304011f, -6.40000e-005f, 0.885952f, -0.303864f, 
		-6.40000e-005f, 0.885952f, -0.303864f, -6.40000e-005f, 0.885952f, -0.303864f, 
		-6.40000e-005f, 0.889845f, -0.301624f, -6.40000e-005f, 0.889845f, -0.301624f, 
		-6.40000e-005f, 0.889845f, -0.301624f, -6.36216e-005f, 0.993267f, -0.229779f, 
		-6.30000e-005f, 0.843081f, -0.299286f, -6.30000e-005f, 0.843081f, -0.299286f, 
		-6.30000e-005f, 0.843081f, -0.299286f, -6.30000e-005f, 0.893445f, -0.298972f, 
		-6.30000e-005f, 0.893445f, -0.298972f, -6.30000e-005f, 0.893445f, -0.298972f, 
		-6.27082e-005f, 0.897585f, -0.295096f, -6.27082e-005f, 0.897585f, -0.295096f, 
		-6.27082e-005f, 0.897585f, -0.295096f, -6.27082e-005f, 0.897585f, -0.295096f, 
		-6.21225e-005f, 1.08979f, -0.405084f, -6.21225e-005f, 1.08979f, -0.405084f, 
		-6.21225e-005f, 1.08979f, -0.405084f, -6.21225e-005f, 1.08979f, -0.405084f, 
		-6.20000e-005f, 0.836885f, -0.293089f, -6.20000e-005f, 0.836885f, -0.293089f, 
		-6.20000e-005f, 0.836885f, -0.293089f, -6.19682e-005f, 0.901703f, -0.291699f, 
		-6.19682e-005f, 0.901703f, -0.291699f, -6.19682e-005f, 0.901703f, -0.291699f, 
		-6.19682e-005f, 0.901703f, -0.291699f, -6.10114e-005f, 0.911318f, -0.288090f, 
		-6.10114e-005f, 0.911318f, -0.288090f, -6.10114e-005f, 0.911318f, -0.288090f, 
		-6.10114e-005f, 0.911318f, -0.288090f, -6.10114e-005f, 0.911318f, -0.288090f, 
		-6.10050e-005f, 0.949218f, -0.288068f, -6.10050e-005f, 0.949218f, -0.288068f, 
		-6.10050e-005f, 0.949218f, -0.288068f, -6.10050e-005f, 0.949218f, -0.288068f, 
		-6.10050e-005f, 0.949218f, -0.288068f, -6.08318e-005f, 0.906317f, -0.289266f, 
		-6.08318e-005f, 0.906317f, -0.289266f, -6.08318e-005f, 0.906317f, -0.289266f, 
		-6.08318e-005f, 0.906317f, -0.289266f, -6.06568e-005f, 0.953366f, -0.286605f, 
		-6.06568e-005f, 0.953366f, -0.286605f, -6.06568e-005f, 0.953366f, -0.286605f, 
		-6.06568e-005f, 0.953366f, -0.286605f, -6.06568e-005f, 0.953366f, -0.286605f, 
		-6.05303e-005f, 1.26180f, -0.0704420f, -6.05303e-005f, 1.26180f, -0.0704420f, 
		-6.05303e-005f, 1.26180f, -0.0704420f, -6.05303e-005f, 1.26180f, -0.0704420f, 
		-6.05303e-005f, 1.26180f, -0.0704420f, -6.05303e-005f, 1.26180f, -0.0704420f, 
		-6.05303e-005f, 1.26180f, -0.0704420f, -6.05303e-005f, 1.26180f, -0.0704420f, 
		-6.05303e-005f, 1.26180f, -0.0704420f, -6.05303e-005f, 1.26180f, -0.0704420f, 
		-6.05303e-005f, 1.26180f, -0.0704420f, -6.05303e-005f, 1.26180f, -0.0704420f, 
		-6.00000e-005f, 0.832160f, -0.285662f, -6.00000e-005f, 0.832160f, -0.285662f, 
		-6.00000e-005f, 0.832160f, -0.285662f, -5.95561e-005f, 1.06095f, -0.280762f, 
		-5.95561e-005f, 1.06095f, -0.280762f, -5.95561e-005f, 1.06095f, -0.280762f, 
		-5.95561e-005f, 1.06095f, -0.280762f, -5.95561e-005f, 1.06095f, -0.280762f, 
		-5.95561e-005f, 1.06095f, -0.280762f, -5.70000e-005f, 0.828091f, -0.268080f, 
		-5.70000e-005f, 0.828091f, -0.268080f, -5.70000e-005f, 0.828091f, -0.268080f, 
		-5.70000e-005f, 0.828091f, -0.268080f, -5.70000e-005f, 0.828091f, -0.268080f, 
		-5.70000e-005f, 0.828091f, -0.268080f, -5.70000e-005f, 0.828091f, -0.268080f, 
		-5.70000e-005f, 0.828091f, -0.268080f, -5.70000e-005f, 0.828091f, -0.268080f, 
		-5.70000e-005f, 0.828091f, -0.268080f, -5.70000e-005f, 0.828091f, -0.268080f, 
		-5.70000e-005f, 0.828091f, -0.268080f, -5.70000e-005f, 0.985109f, -0.269991f, 
		-5.65308e-005f, 1.17750f, 0.0788622f, -5.65308e-005f, 1.17750f, 0.0788622f, 
		-5.65308e-005f, 1.17750f, 0.0788622f, -5.35134e-005f, 0.956459f, -0.252549f, 
		-5.35134e-005f, 0.956459f, -0.252549f, -5.35134e-005f, 0.956459f, -0.252549f, 
		-5.35134e-005f, 0.956459f, -0.252549f, -5.30525e-005f, 0.953366f, -0.249554f, 
		-5.30525e-005f, 0.953366f, -0.249554f, -5.30525e-005f, 0.953366f, -0.249554f, 
		-5.30525e-005f, 0.953366f, -0.249554f, -5.30000e-005f, 0.832160f, -0.250497f, 
		-5.30000e-005f, 0.832160f, -0.250497f, -5.30000e-005f, 0.832160f, -0.250497f, 
		-5.21080e-005f, 0.949218f, -0.248092f, -5.21080e-005f, 0.949218f, -0.248092f, 
		-5.21080e-005f, 0.949218f, -0.248092f, -5.21080e-005f, 0.949218f, -0.248092f, 
		-5.20030e-005f, 0.911318f, -0.248070f, -5.20030e-005f, 0.911318f, -0.248070f, 
		-5.20030e-005f, 0.911318f, -0.248070f, -5.20030e-005f, 0.911318f, -0.248070f, 
		-5.20030e-005f, 0.911318f, -0.248070f, -5.20030e-005f, 0.911318f, -0.248070f, 
		-5.19935e-005f, 0.906317f, -0.246894f, -5.19935e-005f, 0.906317f, -0.246894f, 
		-5.19935e-005f, 0.906317f, -0.246894f, -5.19935e-005f, 0.906317f, -0.246894f, 
		-5.19935e-005f, 0.906317f, -0.246894f, -5.19344e-005f, 0.901703f, -0.244460f, 
		-5.19344e-005f, 0.901703f, -0.244460f, -5.19344e-005f, 0.901703f, -0.244460f, 
		-5.19344e-005f, 0.901703f, -0.244460f, -5.19344e-005f, 0.901703f, -0.244460f, 
		-5.10000e-005f, 0.836885f, -0.243070f, -5.10000e-005f, 0.836885f, -0.243070f, 
		-5.10000e-005f, 0.836885f, -0.243070f, -5.09880e-005f, 0.897585f, -0.241064f, 
		-5.09880e-005f, 0.897585f, -0.241064f, -5.09880e-005f, 0.897585f, -0.241064f, 
		-5.09880e-005f, 0.897585f, -0.241064f, -5.09880e-005f, 0.897585f, -0.241064f, 
		-5.00000e-005f, 0.843081f, -0.236874f, -5.00000e-005f, 0.843081f, -0.236874f, 
		-5.00000e-005f, 0.843081f, -0.236874f, -5.00000e-005f, 0.889845f, -0.234535f, 
		-5.00000e-005f, 0.889845f, -0.234535f, -5.00000e-005f, 0.889845f, -0.234535f, 
		-5.00000e-005f, 0.893445f, -0.237187f, -5.00000e-005f, 0.893445f, -0.237187f, 
		-5.00000e-005f, 0.893445f, -0.237187f, -4.90000e-005f, 0.850508f, -0.232149f, 
		-4.90000e-005f, 0.850508f, -0.232149f, -4.90000e-005f, 0.850508f, -0.232149f, 
		-4.90000e-005f, 0.881798f, -0.230498f, -4.90000e-005f, 0.881798f, -0.230498f, 
		-4.90000e-005f, 0.881798f, -0.230498f, -4.90000e-005f, 0.885952f, -0.232295f, 
		-4.90000e-005f, 0.885952f, -0.232295f, -4.90000e-005f, 0.885952f, -0.232295f, 
		-4.80000e-005f, 0.858925f, -0.229137f, -4.80000e-005f, 0.858925f, -0.229137f, 
		-4.80000e-005f, 0.858925f, -0.229137f, -4.80000e-005f, 0.868091f, -0.228080f, 
		-4.80000e-005f, 0.868091f, -0.228080f, -4.80000e-005f, 0.868091f, -0.228080f, 
		-4.80000e-005f, 0.872834f, -0.228359f, -4.80000e-005f, 0.872834f, -0.228359f, 
		-4.80000e-005f, 0.872834f, -0.228359f, -4.80000e-005f, 0.877414f, -0.229175f, 
		-4.80000e-005f, 0.877414f, -0.229175f, -4.80000e-005f, 0.877414f, -0.229175f, 
		-4.55772e-005f, 1.22218f, -0.0484774f, -4.39217e-005f, 0.981105f, -0.269977f, 
		-4.39217e-005f, 0.981105f, -0.269977f, -4.39217e-005f, 0.981105f, -0.269977f, 
		-4.39217e-005f, 0.981105f, -0.269977f, -4.39217e-005f, 0.981105f, -0.269977f, 
		-4.39217e-005f, 0.981105f, -0.269977f, -4.39217e-005f, 0.981105f, -0.269977f, 
		-4.39217e-005f, 0.981105f, -0.269977f, -4.39217e-005f, 0.981105f, -0.269977f, 
		-3.90434e-005f, 1.02247f, -0.0179449f, -3.90434e-005f, 1.02247f, -0.0179449f, 
		-3.90434e-005f, 1.02247f, -0.0179449f, -3.80980e-005f, 1.39112f, 0.00840436f, 
		-3.80980e-005f, 1.39112f, 0.00840436f, -3.20180e-005f, 1.01837f, 0.247216f, 
		-2.74683e-005f, 1.34168f, -0.0774677f, -2.74683e-005f, 1.34168f, -0.0774677f, 
		-2.74683e-005f, 1.34168f, -0.0774677f, -2.67194e-005f, 1.06838f, -0.0157374f, 
		-2.67194e-005f, 1.06838f, -0.0157374f, -2.67194e-005f, 1.06838f, -0.0157374f, 
		-2.67194e-005f, 1.06838f, -0.0157374f, -2.67194e-005f, 1.06838f, -0.0157374f, 
		-2.67194e-005f, 1.06838f, -0.0157374f, -2.58927e-005f, 1.48270f, 0.0311291f, 
		-2.58927e-005f, 1.48270f, 0.0311291f, -2.56932e-005f, 1.42790f, 0.0311601f, 
		-2.56932e-005f, 1.42790f, 0.0311601f, -2.54484e-005f, 1.22218f, 0.0484733f, 
		-2.20000e-005f, 1.58893f, -0.105875f, -2.20000e-005f, 1.58893f, -0.105875f, 
		-2.20000e-005f, 1.61893f, -0.105875f, -2.20000e-005f, 1.61893f, -0.105875f, 
		-2.20000e-005f, 1.61893f, -0.105875f, -2.20000e-005f, 1.61893f, -0.105875f, 
		-2.00000e-005f, 1.57721f, -0.0953220f, -1.99331e-005f, 1.06838f, 0.0157335f, 
		-1.99331e-005f, 1.06838f, 0.0157335f, -1.99331e-005f, 1.06838f, 0.0157335f, 
		-1.99331e-005f, 1.06838f, 0.0157335f, -1.90000e-005f, 1.57086f, -0.0911840f, 
		-1.20000e-005f, 1.23705f, -0.0550020f, -1.20000e-005f, 1.23705f, -0.0550020f, 
		-1.20000e-005f, 1.26705f, -0.0550020f, -1.20000e-005f, 1.26705f, -0.0550020f, 
		-8.02462e-006f, 1.52396f, -0.00288754f, -8.02462e-006f, 1.52396f, -0.00288754f, 
		-8.02462e-006f, 1.52396f, -0.00288754f, -6.12490e-006f, 1.06853f, -0.0274666f, 
		-6.12490e-006f, 1.06853f, -0.0274666f, -6.09475e-006f, 1.02354f, -0.0274667f, 
		-6.09475e-006f, 1.02354f, -0.0274667f, -6.01997e-006f, 1.02715f, -0.0298637f, 
		-6.01997e-006f, 1.02715f, -0.0298637f, -5.99135e-006f, 1.06492f, -0.0298637f, 
		-5.99135e-006f, 1.06492f, -0.0298637f, -5.65313e-006f, 1.07021f, -0.0228445f, 
		-5.65313e-006f, 1.07021f, -0.0228445f, -5.65313e-006f, 1.07021f, -0.0228445f, 
		-5.65313e-006f, 1.07021f, -0.0228445f, -5.05790e-006f, 1.31412f, -0.0244460f, 
		-5.05790e-006f, 1.31412f, -0.0244460f, -5.02518e-006f, 1.31685f, -0.0227450f, 
		-5.02518e-006f, 1.31685f, -0.0227450f, -5.02456e-006f, 1.31045f, -0.0248626f, 
		-5.02456e-006f, 1.31045f, -0.0248626f, -5.01089e-006f, 1.30677f, -0.0237409f, 
		-5.01089e-006f, 1.30677f, -0.0237409f, -4.38560e-006f, 1.30448f, -0.0215005f, 
		-4.38560e-006f, 1.30448f, -0.0215005f, -4.07550e-006f, 1.33444f, -0.0185691f, 
		-4.07550e-006f, 1.33444f, -0.0185691f, -4.07550e-006f, 1.33444f, -0.0185691f, 
		-4.07550e-006f, 1.33444f, -0.0185691f, -4.07550e-006f, 1.33444f, -0.0185691f, 
		-4.07550e-006f, 1.33444f, -0.0185691f, -4.07550e-006f, 1.33444f, -0.0185691f, 
		-4.07550e-006f, 1.33444f, -0.0185691f, -3.96618e-006f, 1.33120f, -0.0188154f, 
		-3.96618e-006f, 1.33120f, -0.0188154f, -3.96618e-006f, 1.33120f, -0.0188154f, 
		-3.96618e-006f, 1.33120f, -0.0188154f, -3.96618e-006f, 1.33120f, -0.0188154f, 
		-3.96618e-006f, 1.33120f, -0.0188154f, -3.96618e-006f, 1.33120f, -0.0188154f, 
		-3.96618e-006f, 1.33120f, -0.0188154f, -3.74561e-006f, 1.02186f, -0.0228451f, 
		-3.74561e-006f, 1.02186f, -0.0228451f, -3.43789e-006f, 1.33668f, -0.0163884f, 
		-3.43789e-006f, 1.33668f, -0.0163884f, -3.43789e-006f, 1.33668f, -0.0163884f, 
		-3.43789e-006f, 1.33668f, -0.0163884f, -3.43789e-006f, 1.33668f, -0.0163884f, 
		-3.43789e-006f, 1.33668f, -0.0163884f, -3.27640e-006f, 1.32831f, -0.0164344f, 
		-3.27640e-006f, 1.32831f, -0.0164344f, -3.27640e-006f, 1.32831f, -0.0164344f, 
		-3.27640e-006f, 1.32831f, -0.0164344f, -3.27640e-006f, 1.32831f, -0.0164344f, 
		-3.27640e-006f, 1.32831f, -0.0164344f, -3.00000e-006f, 1.00868f, -0.0124480f, 
		-3.00000e-006f, 1.00868f, -0.0124480f, -3.00000e-006f, 1.01091f, -0.0139410f, 
		-3.00000e-006f, 1.01091f, -0.0139410f, -3.00000e-006f, 1.01344f, -0.0148230f, 
		-3.00000e-006f, 1.01344f, -0.0148230f, -3.00000e-006f, 1.01619f, -0.0151130f, 
		-3.00000e-006f, 1.01619f, -0.0151130f, -3.00000e-006f, 1.07589f, -0.0151130f, 
		-3.00000e-006f, 1.07589f, -0.0151130f, -3.00000e-006f, 1.07863f, -0.0148230f, 
		-3.00000e-006f, 1.07863f, -0.0148230f, -3.00000e-006f, 1.08116f, -0.0139410f, 
		-3.00000e-006f, 1.08116f, -0.0139410f, -3.00000e-006f, 1.08339f, -0.0124480f, 
		-3.00000e-006f, 1.08339f, -0.0124480f, -2.82968e-006f, 1.32705f, -0.0127939f, 
		-2.82968e-006f, 1.32705f, -0.0127939f, -2.82968e-006f, 1.32705f, -0.0127939f, 
		-2.82968e-006f, 1.32705f, -0.0127939f, -2.82968e-006f, 1.32705f, -0.0127939f, 
		-2.82968e-006f, 1.32705f, -0.0127939f, -2.00000e-006f, 1.00541f, -0.00755700f, 
		-2.00000e-006f, 1.00541f, -0.00755700f, -2.00000e-006f, 1.00683f, -0.0103260f, 
		-2.00000e-006f, 1.00683f, -0.0103260f, -2.00000e-006f, 1.08525f, -0.0103260f, 
		-2.00000e-006f, 1.08525f, -0.0103260f, -2.00000e-006f, 1.08667f, -0.00755700f, 
		-2.00000e-006f, 1.08667f, -0.00755700f, -1.00000e-006f, 1.00451f, -0.00412100f, 
		-1.00000e-006f, 1.00451f, -0.00412100f, -1.00000e-006f, 1.08757f, -0.00412100f, 
		-1.00000e-006f, 1.08757f, -0.00412100f, -3.33786e-008f, 1.00796f, -0.245840f, 
		-3.33786e-008f, 1.00796f, -0.245840f, -3.33786e-008f, 1.00796f, -0.245840f, 
		0.000000f, 1.00419f, -2.00000e-006f, 0.000000f, 1.00419f, -2.00000e-006f, 
		0.000000f, 1.08789f, -2.00000e-006f, 0.000000f, 1.08789f, -2.00000e-006f, 
		6.36907e-007f, 1.52456f, -0.0100498f, 6.36907e-007f, 1.52456f, -0.0100498f, 
		6.36907e-007f, 1.52456f, -0.0100498f, 6.36907e-007f, 1.52456f, -0.0100498f, 
		6.36907e-007f, 1.52456f, -0.0100498f, 6.36907e-007f, 1.52456f, -0.0100498f, 
		7.26948e-007f, 1.42795f, 0.0193620f, 7.26948e-007f, 1.42795f, 0.0193620f, 
		1.00000e-006f, 1.00451f, 0.00411800f, 1.00000e-006f, 1.00451f, 0.00411800f, 
		1.00000e-006f, 1.08757f, 0.00411800f, 1.00000e-006f, 1.08757f, 0.00411800f, 
		2.00000e-006f, 1.00541f, 0.00755300f, 2.00000e-006f, 1.00541f, 0.00755300f, 
		2.00000e-006f, 1.00683f, 0.0103220f, 2.00000e-006f, 1.00683f, 0.0103220f, 
		2.00000e-006f, 1.08525f, 0.0103220f, 2.00000e-006f, 1.08525f, 0.0103220f, 
		2.00000e-006f, 1.08667f, 0.00755300f, 2.00000e-006f, 1.08667f, 0.00755300f, 
		3.00000e-006f, 1.00868f, 0.0124440f, 3.00000e-006f, 1.00868f, 0.0124440f, 
		3.00000e-006f, 1.01091f, 0.0139370f, 3.00000e-006f, 1.01091f, 0.0139370f, 
		3.00000e-006f, 1.01344f, 0.0148190f, 3.00000e-006f, 1.01344f, 0.0148190f, 
		3.00000e-006f, 1.01619f, 0.0151090f, 3.00000e-006f, 1.01619f, 0.0151090f, 
		3.00000e-006f, 1.07589f, 0.0151090f, 3.00000e-006f, 1.07589f, 0.0151090f, 
		3.00000e-006f, 1.07863f, 0.0148190f, 3.00000e-006f, 1.07863f, 0.0148190f, 
		3.00000e-006f, 1.08116f, 0.0139370f, 3.00000e-006f, 1.08116f, 0.0139370f, 
		3.00000e-006f, 1.08339f, 0.0124440f, 3.00000e-006f, 1.08339f, 0.0124440f, 
		3.00624e-006f, 1.32392f, 0.0140611f, 3.00624e-006f, 1.32392f, 0.0140611f, 
		3.00624e-006f, 1.32392f, 0.0140611f, 3.00624e-006f, 1.32392f, 0.0140611f, 
		3.00624e-006f, 1.32392f, 0.0140611f, 3.00624e-006f, 1.32392f, 0.0140611f, 
		3.03222e-006f, 1.32070f, 0.0133683f, 3.03222e-006f, 1.32070f, 0.0133683f, 
		3.03222e-006f, 1.32070f, 0.0133683f, 3.03222e-006f, 1.32070f, 0.0133683f, 
		3.03222e-006f, 1.32070f, 0.0133683f, 3.24720e-006f, 1.33668f, 0.0163843f, 
		3.24720e-006f, 1.33668f, 0.0163843f, 3.24720e-006f, 1.33668f, 0.0163843f, 
		3.24720e-006f, 1.33668f, 0.0163843f, 3.24720e-006f, 1.33668f, 0.0163843f, 
		3.24720e-006f, 1.33668f, 0.0163843f, 3.24720e-006f, 1.33668f, 0.0163843f, 
		3.46040e-006f, 1.32831f, 0.0164304f, 3.46040e-006f, 1.32831f, 0.0164304f, 
		3.55598e-006f, 1.07021f, 0.0228412f, 3.55598e-006f, 1.07021f, 0.0228412f, 
		3.55598e-006f, 1.07021f, 0.0228412f, 3.55598e-006f, 1.07021f, 0.0228412f, 
		3.96609e-006f, 1.33363f, 0.0188114f, 3.96609e-006f, 1.33363f, 0.0188114f, 
		3.96609e-006f, 1.33363f, 0.0188114f, 3.96609e-006f, 1.33363f, 0.0188114f, 
		3.96609e-006f, 1.33363f, 0.0188114f, 3.96609e-006f, 1.33363f, 0.0188114f, 
		3.96609e-006f, 1.33363f, 0.0188114f, 3.96609e-006f, 1.33363f, 0.0188114f, 
		4.07518e-006f, 1.33039f, 0.0185651f, 4.07518e-006f, 1.33039f, 0.0185651f, 
		4.07518e-006f, 1.33039f, 0.0185651f, 4.07518e-006f, 1.33039f, 0.0185651f, 
		4.07518e-006f, 1.33039f, 0.0185651f, 4.07518e-006f, 1.33039f, 0.0185651f, 
		4.07518e-006f, 1.33039f, 0.0185651f, 4.07518e-006f, 1.33039f, 0.0185651f, 
		4.23250e-006f, 1.31902f, 0.0184402f, 4.23250e-006f, 1.31902f, 0.0184402f, 
		4.34228e-006f, 1.30448f, 0.0214972f, 4.34228e-006f, 1.30448f, 0.0214972f, 
		4.38478e-006f, 1.31788f, 0.0214972f, 4.38478e-006f, 1.31788f, 0.0214972f, 
		4.98299e-006f, 1.31045f, 0.0248587f, 4.98299e-006f, 1.31045f, 0.0248587f, 
		4.98403e-006f, 1.30677f, 0.0237369f, 4.98403e-006f, 1.30677f, 0.0237369f, 
		5.00000e-006f, 1.31336f, 0.0246690f, 5.00000e-006f, 1.31336f, 0.0246690f, 
		5.00995e-006f, 1.31559f, 0.0237368f, 5.00995e-006f, 1.31559f, 0.0237368f, 
		5.66322e-006f, 1.06492f, 0.0298597f, 5.66322e-006f, 1.06492f, 0.0298597f, 
		5.82907e-006f, 1.02715f, 0.0298597f, 5.82907e-006f, 1.02715f, 0.0298597f, 
		5.91340e-006f, 1.02186f, 0.0228419f, 5.91340e-006f, 1.02186f, 0.0228419f, 
		6.10257e-006f, 1.06853f, 0.0274626f, 6.10257e-006f, 1.06853f, 0.0274626f, 
		6.16636e-006f, 1.02354f, 0.0274627f, 6.16636e-006f, 1.02354f, 0.0274627f, 
		6.75665e-006f, 1.26180f, 0.0702987f, 6.75665e-006f, 1.26180f, 0.0702987f, 
		6.75665e-006f, 1.26180f, 0.0702987f, 6.75665e-006f, 1.26180f, 0.0702987f, 
		6.75665e-006f, 1.26180f, 0.0702987f, 6.75665e-006f, 1.26180f, 0.0702987f, 
		6.75665e-006f, 1.26180f, 0.0702987f, 6.75665e-006f, 1.26180f, 0.0702987f, 
		6.97572e-006f, 1.03958f, -0.425396f, 6.97572e-006f, 1.03958f, -0.425396f, 
		6.97572e-006f, 1.03958f, -0.425396f, 6.97572e-006f, 1.03958f, -0.425396f, 
		6.97572e-006f, 1.03958f, -0.425396f, 6.97572e-006f, 1.03958f, -0.425396f, 
		9.00007e-006f, 1.55473f, 0.0423115f, 1.20000e-005f, 1.23705f, 0.0549980f, 
		1.20000e-005f, 1.23705f, 0.0549980f, 1.20000e-005f, 1.26705f, 0.0549980f, 
		1.20000e-005f, 1.26705f, 0.0549980f, 1.27938e-005f, 1.32323f, -0.0141137f, 
		1.27938e-005f, 1.32323f, -0.0141137f, 1.27938e-005f, 1.32323f, -0.0141137f, 
		1.27938e-005f, 1.32323f, -0.0141137f, 1.27938e-005f, 1.32323f, -0.0141137f, 
		1.27938e-005f, 1.32323f, -0.0141137f, 1.70000e-005f, 1.57086f, 0.0796330f, 
		1.80000e-005f, 1.57721f, 0.0837720f, 2.00000e-005f, 1.58893f, 0.0943250f, 
		2.00000e-005f, 1.58893f, 0.0943250f, 2.00000e-005f, 1.58893f, 0.0943250f, 
		2.00000e-005f, 1.61893f, 0.0943250f, 2.00000e-005f, 1.61893f, 0.0943250f, 
		2.00000e-005f, 1.61893f, 0.0943250f, 2.00000e-005f, 1.61893f, 0.0943250f, 
		2.52565e-005f, 1.28192f, -0.0484770f, 2.72043e-005f, 1.02369f, 0.0157355f, 
		2.72043e-005f, 1.02369f, 0.0157355f, 2.72043e-005f, 1.02369f, 0.0157355f, 
		2.72043e-005f, 1.02369f, 0.0157355f, 2.81302e-005f, 1.51062f, 0.0237266f, 
		2.81302e-005f, 1.51062f, 0.0237266f, 3.37200e-005f, 0.993266f, 0.229677f, 
		4.20915e-005f, 0.981105f, 0.269901f, 4.20915e-005f, 0.981105f, 0.269901f, 
		4.20915e-005f, 0.981105f, 0.269901f, 4.20915e-005f, 0.981105f, 0.269901f, 
		4.20915e-005f, 0.981105f, 0.269901f, 4.20915e-005f, 0.981105f, 0.269901f, 
		4.20915e-005f, 0.981105f, 0.269901f, 4.20915e-005f, 0.981105f, 0.269901f, 
		4.53222e-005f, 1.28192f, 0.0484730f, 4.80000e-005f, 0.858925f, 0.229035f, 
		4.80000e-005f, 0.858925f, 0.229035f, 4.80000e-005f, 0.858925f, 0.229035f, 
		4.80000e-005f, 0.868091f, 0.227978f, 4.80000e-005f, 0.868091f, 0.227978f, 
		4.80000e-005f, 0.868091f, 0.227978f, 4.80000e-005f, 0.872834f, 0.228257f, 
		4.80000e-005f, 0.872834f, 0.228257f, 4.80000e-005f, 0.872834f, 0.228257f, 
		4.80000e-005f, 0.877414f, 0.229074f, 4.80000e-005f, 0.877414f, 0.229074f, 
		4.80000e-005f, 0.877414f, 0.229074f, 4.90000e-005f, 0.850508f, 0.232047f, 
		4.90000e-005f, 0.850508f, 0.232047f, 4.90000e-005f, 0.850508f, 0.232047f, 
		4.90000e-005f, 0.881798f, 0.230396f, 4.90000e-005f, 0.881798f, 0.230396f, 
		4.90000e-005f, 0.881798f, 0.230396f, 4.90000e-005f, 0.885952f, 0.232193f, 
		4.90000e-005f, 0.885952f, 0.232193f, 4.90000e-005f, 0.885952f, 0.232193f, 
		5.00000e-005f, 0.843081f, 0.236772f, 5.00000e-005f, 0.843081f, 0.236772f, 
		5.00000e-005f, 0.843081f, 0.236772f, 5.00000e-005f, 0.889845f, 0.234434f, 
		5.00000e-005f, 0.889845f, 0.234434f, 5.00000e-005f, 0.889845f, 0.234434f, 
		5.00000e-005f, 0.893445f, 0.237086f, 5.00000e-005f, 0.893445f, 0.237086f, 
		5.00000e-005f, 0.893445f, 0.237086f, 5.09871e-005f, 0.897585f, 0.240962f, 
		5.09871e-005f, 0.897585f, 0.240962f, 5.09871e-005f, 0.897585f, 0.240962f, 
		5.09871e-005f, 0.897585f, 0.240962f, 5.10000e-005f, 0.836885f, 0.242968f, 
		5.10000e-005f, 0.836885f, 0.242968f, 5.10000e-005f, 0.836885f, 0.242968f, 
		5.19345e-005f, 0.901703f, 0.244358f, 5.19345e-005f, 0.901703f, 0.244358f, 
		5.19345e-005f, 0.901703f, 0.244358f, 5.19345e-005f, 0.901703f, 0.244358f, 
		5.19929e-005f, 0.906317f, 0.246791f, 5.19929e-005f, 0.906317f, 0.246791f, 
		5.19929e-005f, 0.906317f, 0.246791f, 5.19929e-005f, 0.906317f, 0.246791f, 
		5.20036e-005f, 0.911318f, 0.247968f, 5.20036e-005f, 0.911318f, 0.247968f, 
		5.20036e-005f, 0.911318f, 0.247968f, 5.20036e-005f, 0.911318f, 0.247968f, 
		5.20036e-005f, 0.911318f, 0.247968f, 5.21080e-005f, 0.949218f, 0.247990f, 
		5.21080e-005f, 0.949218f, 0.247990f, 5.21080e-005f, 0.949218f, 0.247990f, 
		5.21080e-005f, 0.949218f, 0.247990f, 5.23137e-005f, 1.26154f, 0.0598688f, 
		5.23137e-005f, 1.26154f, 0.0598688f, 5.23137e-005f, 1.26154f, 0.0598688f, 
		5.23137e-005f, 1.26154f, 0.0598688f, 5.23137e-005f, 1.26154f, 0.0598688f, 
		5.23137e-005f, 1.26154f, 0.0598688f, 5.30000e-005f, 0.832160f, 0.250396f, 
		5.30000e-005f, 0.832160f, 0.250396f, 5.30000e-005f, 0.832160f, 0.250396f, 
		5.30182e-005f, 0.953366f, 0.249452f, 5.30182e-005f, 0.953366f, 0.249452f, 
		5.30182e-005f, 0.953366f, 0.249452f, 5.30182e-005f, 0.953366f, 0.249452f, 
		5.41960e-005f, 1.16818f, 0.000357359f, 5.41960e-005f, 1.16818f, 0.000357359f, 
		5.41960e-005f, 1.16818f, 0.000357359f, 5.41960e-005f, 1.16818f, 0.000357359f, 
		5.41960e-005f, 1.16818f, 0.000357359f, 5.41960e-005f, 1.16818f, 0.000357359f, 
		5.70000e-005f, 0.828091f, 0.267978f, 5.70000e-005f, 0.828091f, 0.267978f, 
		5.70000e-005f, 0.828091f, 0.267978f, 5.70000e-005f, 0.828091f, 0.267978f, 
		5.70000e-005f, 0.828091f, 0.267978f, 5.70000e-005f, 0.828091f, 0.267978f, 
		5.70000e-005f, 0.828091f, 0.267978f, 5.70000e-005f, 0.828091f, 0.267978f, 
		5.70000e-005f, 0.828091f, 0.267978f, 5.70000e-005f, 0.828091f, 0.267978f, 
		5.70000e-005f, 0.828091f, 0.267978f, 5.70000e-005f, 0.828091f, 0.267978f, 
		5.70000e-005f, 0.828091f, 0.267978f, 5.70000e-005f, 0.828091f, 0.267978f, 
		5.70000e-005f, 0.985109f, 0.269889f, 5.93160e-005f, 1.06095f, 0.280661f, 
		5.93160e-005f, 1.06095f, 0.280661f, 5.93160e-005f, 1.06095f, 0.280661f, 
		5.93160e-005f, 1.06095f, 0.280661f, 5.93160e-005f, 1.06095f, 0.280661f, 
		5.93160e-005f, 1.06095f, 0.280661f, 6.00000e-005f, 0.832160f, 0.285560f, 
		6.00000e-005f, 0.832160f, 0.285560f, 6.00000e-005f, 0.832160f, 0.285560f, 
		6.07465e-005f, 0.953366f, 0.286504f, 6.07465e-005f, 0.953366f, 0.286504f, 
		6.07465e-005f, 0.953366f, 0.286504f, 6.07465e-005f, 0.953366f, 0.286504f, 
		6.09872e-005f, 0.949218f, 0.287966f, 6.09872e-005f, 0.949218f, 0.287966f, 
		6.09872e-005f, 0.949218f, 0.287966f, 6.09872e-005f, 0.949218f, 0.287966f, 
		6.09990e-005f, 0.908777f, 0.288404f, 6.09990e-005f, 0.908777f, 0.288404f, 
		6.09990e-005f, 0.908777f, 0.288404f, 6.09990e-005f, 0.908777f, 0.288404f, 
		6.10000e-005f, 0.912612f, 0.287978f, 6.10000e-005f, 0.912612f, 0.287978f, 
		6.10000e-005f, 0.912612f, 0.287978f, 6.10000e-005f, 0.912612f, 0.287978f, 
		6.15768e-005f, 0.903956f, 0.290243f, 6.15768e-005f, 0.903956f, 0.290243f, 
		6.15768e-005f, 0.903956f, 0.290243f, 6.15768e-005f, 0.903956f, 0.290243f, 
		6.17567e-005f, 0.899571f, 0.293189f, 6.17567e-005f, 0.899571f, 0.293189f, 
		6.17567e-005f, 0.899571f, 0.293189f, 6.17567e-005f, 0.899571f, 0.293189f, 
		6.20000e-005f, 0.836885f, 0.292987f, 6.20000e-005f, 0.836885f, 0.292987f, 
		6.20000e-005f, 0.836885f, 0.292987f, 6.30000e-005f, 0.843081f, 0.299184f, 
		6.30000e-005f, 0.843081f, 0.299184f, 6.30000e-005f, 0.843081f, 0.299184f, 
		6.30000e-005f, 0.893445f, 0.298870f, 6.30000e-005f, 0.893445f, 0.298870f, 
		6.30000e-005f, 0.893445f, 0.298870f, 6.30000e-005f, 0.896718f, 0.295838f, 
		6.30000e-005f, 0.896718f, 0.295838f, 6.30000e-005f, 0.896718f, 0.295838f, 
		6.40000e-005f, 0.850508f, 0.303909f, 6.40000e-005f, 0.850508f, 0.303909f, 
		6.40000e-005f, 0.850508f, 0.303909f, 6.40000e-005f, 0.885952f, 0.303762f, 
		6.40000e-005f, 0.885952f, 0.303762f, 6.40000e-005f, 0.885952f, 0.303762f, 
		6.40000e-005f, 0.889845f, 0.301522f, 6.40000e-005f, 0.889845f, 0.301522f, 
		6.40000e-005f, 0.889845f, 0.301522f, 6.50000e-005f, 0.858925f, 0.306920f, 
		6.50000e-005f, 0.858925f, 0.306920f, 6.50000e-005f, 0.858925f, 0.306920f, 
		6.50000e-005f, 0.868091f, 0.307978f, 6.50000e-005f, 0.868091f, 0.307978f, 
		6.50000e-005f, 0.868091f, 0.307978f, 6.50000e-005f, 0.872834f, 0.307699f, 
		6.50000e-005f, 0.872834f, 0.307699f, 6.50000e-005f, 0.872834f, 0.307699f, 
		6.50000e-005f, 0.877414f, 0.306882f, 6.50000e-005f, 0.877414f, 0.306882f, 
		6.50000e-005f, 0.877414f, 0.306882f, 6.50000e-005f, 0.881798f, 0.305560f, 
		6.50000e-005f, 0.881798f, 0.305560f, 6.50000e-005f, 0.881798f, 0.305560f, 
		6.61080e-005f, 1.08459f, 0.428550f, 6.61080e-005f, 1.08459f, 0.428550f, 
		6.61080e-005f, 1.08459f, 0.428550f, 6.61080e-005f, 1.08459f, 0.428550f, 
		7.00000e-005f, 0.965681f, 0.329655f, 7.00000e-005f, 0.969213f, 0.331427f, 
		7.50000e-005f, 0.948656f, 0.354914f, 7.60000e-005f, 0.951668f, 0.357423f, 
		7.68898e-005f, 1.07069f, 0.364642f, 7.68898e-005f, 1.07069f, 0.364642f, 
		7.69047e-005f, 1.02322f, 0.364642f, 7.69047e-005f, 1.02322f, 0.364642f, 
		7.70725e-005f, 1.06845f, 0.362354f, 7.70725e-005f, 1.06845f, 0.362354f, 
		7.70835e-005f, 1.02546f, 0.362354f, 7.70835e-005f, 1.02546f, 0.362354f, 
		7.71537e-005f, 1.43144f, 0.0254303f, 7.71537e-005f, 1.43144f, 0.0254303f, 
		7.80000e-005f, 1.03104f, 0.369998f, 7.80000e-005f, 1.03104f, 0.369998f, 
		7.80000e-005f, 1.02228f, 0.366871f, 7.80000e-005f, 1.02228f, 0.366871f, 
		7.80000e-005f, 1.06104f, 0.369998f, 7.80000e-005f, 1.06104f, 0.369998f, 
		7.80000e-005f, 1.07163f, 0.366871f, 7.80000e-005f, 1.07163f, 0.366871f, 
		7.82969e-005f, 1.07181f, 0.369778f, 7.82969e-005f, 1.07181f, 0.369778f, 
		7.83966e-005f, 1.02209f, 0.369778f, 7.83966e-005f, 1.02209f, 0.369778f, 
		7.89608e-005f, 1.07069f, 0.373460f, 7.89608e-005f, 1.07069f, 0.373460f, 
		7.90748e-005f, 1.02322f, 0.373460f, 7.90748e-005f, 1.02322f, 0.373460f, 
		7.95556e-005f, 1.06845f, 0.375748f, 7.95556e-005f, 1.06845f, 0.375748f, 
		7.97444e-005f, 1.02546f, 0.375748f, 7.97444e-005f, 1.02546f, 0.375748f, 
		8.00000e-005f, 0.927433f, 0.375987f, 8.00000e-005f, 0.929791f, 0.379111f, 
		8.00087e-005f, 1.03358f, 0.378577f, 8.00087e-005f, 1.03358f, 0.378577f, 
		8.00087e-005f, 1.03358f, 0.378577f, 8.00087e-005f, 1.03358f, 0.378577f, 
		8.00087e-005f, 1.03358f, 0.378577f, 8.01888e-005f, 1.02851f, 0.376894f, 
		8.01888e-005f, 1.02851f, 0.376894f, 8.01911e-005f, 1.06540f, 0.376894f, 
		8.01911e-005f, 1.06540f, 0.376894f, 8.09624e-005f, 1.06102f, 0.381797f, 
		8.09624e-005f, 1.06102f, 0.381797f, 8.09624e-005f, 1.06102f, 0.381797f, 
		8.09624e-005f, 1.06102f, 0.381797f, 8.09624e-005f, 1.06102f, 0.381797f, 
		8.09624e-005f, 1.06102f, 0.381797f, 8.10009e-005f, 1.03289f, 0.381797f, 
		8.10009e-005f, 1.03289f, 0.381797f, 8.10009e-005f, 1.03289f, 0.381797f, 
		8.10009e-005f, 1.03289f, 0.381797f, 8.10009e-005f, 1.03289f, 0.381797f, 
		8.10009e-005f, 1.03289f, 0.381797f, 8.15311e-005f, 1.33343f, -0.00647334f, 
		8.15311e-005f, 1.33343f, -0.00647334f, 8.15311e-005f, 1.33343f, -0.00647334f, 
		8.15311e-005f, 1.33343f, -0.00647334f, 8.18471e-005f, 1.02839f, 0.388267f, 
		8.18471e-005f, 1.02839f, 0.388267f, 8.18471e-005f, 1.02839f, 0.388267f, 
		8.18471e-005f, 1.02839f, 0.388267f, 8.18471e-005f, 1.02839f, 0.388267f, 
		8.18471e-005f, 1.02839f, 0.388267f, 8.18471e-005f, 1.02839f, 0.388267f, 
		8.18471e-005f, 1.02839f, 0.388267f, 8.20335e-005f, 1.03052f, 0.386188f, 
		8.20335e-005f, 1.03052f, 0.386188f, 8.20335e-005f, 1.03052f, 0.386188f, 
		8.20335e-005f, 1.03052f, 0.386188f, 8.20837e-005f, 1.06339f, 0.386188f, 
		8.20837e-005f, 1.06339f, 0.386188f, 8.20837e-005f, 1.06339f, 0.386188f, 
		8.20837e-005f, 1.06339f, 0.386188f, 8.20837e-005f, 1.06339f, 0.386188f, 
		8.22583e-005f, 1.06577f, 0.389078f, 8.22583e-005f, 1.06577f, 0.389078f, 
		8.22583e-005f, 1.06577f, 0.389078f, 8.22583e-005f, 1.06577f, 0.389078f, 
		8.22583e-005f, 1.06577f, 0.389078f, 8.22583e-005f, 1.06577f, 0.389078f, 
		8.22583e-005f, 1.06577f, 0.389078f, 8.22583e-005f, 1.06577f, 0.389078f, 
		8.22583e-005f, 1.06577f, 0.389078f, 8.22583e-005f, 1.06577f, 0.389078f, 
		8.22583e-005f, 1.06577f, 0.389078f, 8.28395e-005f, 1.02814f, 0.391508f, 
		8.28395e-005f, 1.02814f, 0.391508f, 8.28395e-005f, 1.02814f, 0.391508f, 
		8.28395e-005f, 1.02814f, 0.391508f, 8.28395e-005f, 1.02814f, 0.391508f, 
		8.28395e-005f, 1.02814f, 0.391508f, 8.28395e-005f, 1.02814f, 0.391508f, 
		8.28395e-005f, 1.02814f, 0.391508f, 8.28395e-005f, 1.02814f, 0.391508f, 
		8.28395e-005f, 1.02814f, 0.391508f, 8.28395e-005f, 1.02814f, 0.391508f, 
		8.29460e-005f, 1.06552f, 0.392318f, 8.29460e-005f, 1.06552f, 0.392318f, 
		8.29460e-005f, 1.06552f, 0.392318f, 8.29460e-005f, 1.06552f, 0.392318f, 
		8.29460e-005f, 1.06552f, 0.392318f, 8.29460e-005f, 1.06552f, 0.392318f, 
		8.29460e-005f, 1.06552f, 0.392318f, 8.29460e-005f, 1.06552f, 0.392318f, 
		8.29460e-005f, 1.06552f, 0.392318f, 8.30000e-005f, 0.903517f, 0.392053f, 
		8.33994e-005f, 1.03057f, 0.394549f, 8.33994e-005f, 1.03057f, 0.394549f, 
		8.33994e-005f, 1.03057f, 0.394549f, 8.33994e-005f, 1.03057f, 0.394549f, 
		8.33994e-005f, 1.03057f, 0.394549f, 8.33994e-005f, 1.03057f, 0.394549f, 
		8.34919e-005f, 1.37236f, -0.0112287f, 8.34919e-005f, 1.37236f, -0.0112287f, 
		8.37355e-005f, 1.06334f, 0.394549f, 8.37355e-005f, 1.06334f, 0.394549f, 
		8.37355e-005f, 1.06334f, 0.394549f, 8.37355e-005f, 1.06334f, 0.394549f, 
		8.37355e-005f, 1.06334f, 0.394549f, 8.37355e-005f, 1.06334f, 0.394549f, 
		8.40000e-005f, 0.905134f, 0.395647f, 8.46217e-005f, 1.40203f, 0.0185191f, 
		8.46217e-005f, 1.40203f, 0.0185191f, 8.46301e-005f, 1.45657f, 0.0185190f, 
		8.46301e-005f, 1.45657f, 0.0185190f, 8.46301e-005f, 1.45657f, 0.0185190f, 
		8.50000e-005f, 0.878415f, 0.402294f, 8.60000e-005f, 0.853634f, 0.405890f, 
		8.60000e-005f, 0.853634f, 0.405890f, 8.60000e-005f, 0.879246f, 0.406188f, 
		8.70000e-005f, 0.853677f, 0.409889f, 8.70000e-005f, 0.853677f, 0.409889f, 
		9.64768e-005f, 1.45652f, -0.0311865f, 9.64768e-005f, 1.45652f, -0.0311865f, 
		9.64768e-005f, 1.45652f, -0.0311865f, 9.64768e-005f, 1.45652f, -0.0311865f, 
		0.000103312f, 1.00796f, 0.245738f, 0.000103312f, 1.00796f, 0.245738f, 
		0.000103312f, 1.00796f, 0.245738f, 0.000128592f, 1.29323f, -0.0413734f, 
		0.000131790f, 1.37008f, 0.0157567f, 0.000131790f, 1.37008f, 0.0157567f, 
		0.000132435f, 1.33701f, -0.0109631f, 0.000132435f, 1.33701f, -0.0109631f, 
		0.000132438f, 1.17781f, -0.0109631f, 0.000132438f, 1.17781f, -0.0109631f, 
		0.000146104f, 1.29323f, 0.0413695f, 0.000164966f, 1.36076f, -0.00836382f, 
		0.000164966f, 1.36076f, -0.00836382f, 0.000196181f, 1.36605f, 0.0182183f, 
		0.000196181f, 1.36605f, 0.0182183f, 0.000216923f, 1.00045f, -0.314097f, 
		0.000216923f, 1.00045f, -0.314097f, 0.000236169f, 1.00045f, -0.227449f, 
		0.000236169f, 1.00045f, -0.227449f, 0.000279276f, 1.23699f, -0.0502179f, 
		0.000279276f, 1.23699f, -0.0502179f, 0.000286484f, 1.08986f, 0.404910f, 
		0.000286484f, 1.08986f, 0.404910f, 0.000286484f, 1.08986f, 0.404910f, 
		0.000293250f, 1.09174f, -0.150032f, 0.000293250f, 1.09174f, -0.150032f, 
		0.000293250f, 1.09174f, -0.150032f, 0.000293250f, 1.09174f, -0.150032f, 
		0.000293250f, 1.09174f, -0.150032f, 0.000301781f, 1.23699f, 0.0502137f, 
		0.000301781f, 1.23699f, 0.0502137f, 0.000331526f, 1.18196f, -0.0787976f, 
		0.000331526f, 1.18196f, -0.0787976f, 0.000331526f, 1.18196f, -0.0787976f, 
		0.000331526f, 1.18196f, -0.0787976f, 0.000331526f, 1.18196f, -0.0787976f, 
		0.000331769f, 1.00045f, 0.227348f, 0.000331769f, 1.00045f, 0.227348f, 
		0.000332303f, 1.49205f, 0.00489546f, 0.000332303f, 1.49205f, 0.00489546f, 
		0.000349076f, 1.00045f, 0.313995f, 0.000349076f, 1.00045f, 0.313995f, 
		0.000353750f, 1.09650f, 0.138012f, 0.000353750f, 1.09650f, 0.138012f, 
		0.000353750f, 1.09650f, 0.138012f, 0.000353750f, 1.09650f, 0.138012f, 
		0.000355086f, 1.45453f, 0.0213293f, 0.000355086f, 1.45453f, 0.0213293f, 
		0.000355086f, 1.45453f, 0.0213293f, 0.000355093f, 1.39998f, 0.0213292f, 
		0.000355093f, 1.39998f, 0.0213292f, 0.000422581f, 1.12556f, -0.101490f, 
		0.000422581f, 1.12556f, -0.101490f, 0.000422581f, 1.12556f, -0.101490f, 
		0.000429403f, 1.18480f, -0.0101869f, 0.000429403f, 1.18480f, -0.0101869f, 
		0.000429403f, 1.18480f, -0.0101869f, 0.000429403f, 1.18480f, -0.0101869f, 
		0.000429403f, 1.18480f, -0.0101869f, 0.000429403f, 1.18480f, -0.0101869f, 
		0.000429403f, 1.18480f, -0.0101869f, 0.000429403f, 1.18480f, -0.0101869f, 
		0.000468220f, 1.40191f, -0.0300340f, 0.000468220f, 1.40191f, -0.0300340f, 
		0.000468220f, 1.40191f, -0.0300340f, 0.000468220f, 1.40191f, -0.0300340f, 
		0.000468220f, 1.40191f, -0.0300340f, 0.000468220f, 1.40191f, -0.0300340f, 
		0.000474167f, 1.58880f, -0.0971414f, 0.000474167f, 1.58880f, -0.0971414f, 
		0.000478022f, 1.32806f, 0.00172620f, 0.000478022f, 1.32806f, 0.00172620f, 
		0.000478022f, 1.32806f, 0.00172620f, 0.000488812f, 1.45394f, -0.0208796f, 
		0.000488812f, 1.45394f, -0.0208796f, 0.000493305f, 1.02258f, -0.291936f, 
		0.000493305f, 1.02258f, -0.291936f, 0.000511719f, 1.58880f, 0.0855911f, 
		0.000511719f, 1.58880f, 0.0855911f, 0.000553918f, 1.10896f, 0.112290f, 
		0.000553918f, 1.10896f, 0.112290f, 0.000553918f, 1.10896f, 0.112290f, 
		0.000553918f, 1.10896f, 0.112290f, 0.000576905f, 1.37236f, -0.0161717f, 
		0.000576905f, 1.37236f, -0.0161717f, 0.000576905f, 1.37236f, -0.0161717f, 
		0.000576905f, 1.37236f, -0.0161717f, 0.000576905f, 1.37236f, -0.0161717f, 
		0.000577473f, 1.13806f, 0.0923440f, 0.000577473f, 1.13806f, 0.0923440f, 
		0.000577473f, 1.13806f, 0.0923440f, 0.000578314f, 1.51288f, -0.0114066f, 
		0.000578314f, 1.51288f, -0.0114066f, 0.000580715f, 1.42774f, -0.0318837f, 
		0.000580715f, 1.42774f, -0.0318837f, 0.000580715f, 1.42774f, -0.0318837f, 
		0.000580715f, 1.42774f, -0.0318837f, 0.000589289f, 1.49381f, -0.00846716f, 
		0.000589289f, 1.49381f, -0.00846716f, 0.000589289f, 1.49381f, -0.00846716f, 
		0.000589289f, 1.49381f, -0.00846716f, 0.000639191f, 1.24326f, 0.0545489f, 
		0.000639191f, 1.24326f, 0.0545489f, 0.000639191f, 1.24326f, 0.0545489f, 
		0.000639191f, 1.24326f, 0.0545489f, 0.000639191f, 1.24326f, 0.0545489f, 
		0.000648860f, 1.51307f, -0.0162838f, 0.000648860f, 1.51307f, -0.0162838f, 
		0.000648860f, 1.51307f, -0.0162838f, 0.000648860f, 1.51307f, -0.0162838f, 
		0.000648860f, 1.51307f, -0.0162838f, 0.000648860f, 1.51307f, -0.0162838f, 
		0.000648860f, 1.51307f, -0.0162838f, 0.000704656f, 1.51635f, -0.0205765f, 
		0.000704656f, 1.51635f, -0.0205765f, 0.000782740f, 1.39294f, -0.00215549f, 
		0.000782740f, 1.39294f, -0.00215549f, 0.000782740f, 1.39294f, -0.00215549f, 
		0.000782740f, 1.39294f, -0.00215549f, 0.000820269f, 1.08612f, -0.174019f, 
		0.000820269f, 1.08612f, -0.174019f, 0.000820269f, 1.08612f, -0.174019f, 
		0.000820269f, 1.08612f, -0.174019f, 0.000833524f, 1.02272f, -0.237141f, 
		0.000833524f, 1.02272f, -0.237141f, 0.000847235f, 1.52357f, 0.0117413f, 
		0.000847235f, 1.52357f, 0.0117413f, 0.000847235f, 1.52357f, 0.0117413f, 
		0.000847235f, 1.52357f, 0.0117413f, 0.000933857f, 1.02272f, 0.237039f, 
		0.000933857f, 1.02272f, 0.237039f, 0.000938113f, 1.40073f, -0.0197203f, 
		0.000938113f, 1.40073f, -0.0197203f, 0.000938113f, 1.40073f, -0.0197203f, 
		0.000938113f, 1.40073f, -0.0197203f, 0.000938113f, 1.40073f, -0.0197203f, 
		0.000942357f, 1.36361f, -0.0117708f, 0.000942357f, 1.36361f, -0.0117708f, 
		0.000942357f, 1.36361f, -0.0117708f, 0.000942357f, 1.36361f, -0.0117708f, 
		0.000942357f, 1.36361f, -0.0117708f, 0.000971546f, 1.52031f, 0.000596965f, 
		0.000971546f, 1.52031f, 0.000596965f, 0.000971546f, 1.52031f, 0.000596965f, 
		0.000976912f, 1.48555f, 0.0196775f, 0.000976912f, 1.48555f, 0.0196775f, 
		0.000976912f, 1.48555f, 0.0196775f, 0.000976912f, 1.48555f, 0.0196775f, 
		0.000983456f, 1.51847f, 0.0137148f, 0.000983456f, 1.51847f, 0.0137148f, 
		0.000984660f, 1.08608f, 0.169447f, 0.000984660f, 1.08608f, 0.169447f, 
		0.000984660f, 1.08608f, 0.169447f, 0.00100034f, 1.17935f, 0.0755975f, 
		0.00100034f, 1.17935f, 0.0755975f, 0.00100034f, 1.17935f, 0.0755975f, 
		0.00101933f, 1.18208f, -0.0756660f, 0.00101933f, 1.18208f, -0.0756660f, 
		0.00101933f, 1.18208f, -0.0756660f, 0.00103696f, 1.31850f, 0.0718878f, 
		0.00103696f, 1.31850f, 0.0718878f, 0.00103696f, 1.31850f, 0.0718878f, 
		0.00103696f, 1.31850f, 0.0718878f, 0.00103696f, 1.31850f, 0.0718878f, 
		0.00105400f, 1.09324f, 0.139327f, 0.00105400f, 1.09324f, 0.139327f, 
		0.00105400f, 1.09324f, 0.139327f, 0.00105400f, 1.09324f, 0.139327f, 
		0.00106343f, 1.08347f, -0.172290f, 0.00106343f, 1.08347f, -0.172290f, 
		0.00106343f, 1.08347f, -0.172290f, 0.00109016f, 1.48264f, -0.0314649f, 
		0.00109016f, 1.48264f, -0.0314649f, 0.00109016f, 1.48264f, -0.0314649f, 
		0.00109016f, 1.48264f, -0.0314649f, 0.00111325f, 1.55685f, -0.0660329f, 
		0.00111577f, 1.07179f, -0.280190f, 0.00111577f, 1.07179f, -0.280190f, 
		0.00111577f, 1.07179f, -0.280190f, 0.00111577f, 1.07179f, -0.280190f, 
		0.00112308f, 1.33730f, 0.0811676f, 0.00112308f, 1.33730f, 0.0811676f, 
		0.00112308f, 1.33730f, 0.0811676f, 0.00112308f, 1.33730f, 0.0811676f, 
		0.00112308f, 1.33730f, 0.0811676f, 0.00113729f, 1.55685f, 0.0544819f, 
		0.00116189f, 1.42956f, -0.0197826f, 0.00116189f, 1.42956f, -0.0197826f, 
		0.00116189f, 1.42956f, -0.0197826f, 0.00116189f, 1.42956f, -0.0197826f, 
		0.00116189f, 1.42956f, -0.0197826f, 0.00117461f, 1.24222f, -0.0547866f, 
		0.00117461f, 1.24222f, -0.0547866f, 0.00117461f, 1.24222f, -0.0547866f, 
		0.00117461f, 1.24222f, -0.0547866f, 0.00117461f, 1.24222f, -0.0547866f, 
		0.00117461f, 1.24222f, -0.0547866f, 0.00117461f, 1.24222f, -0.0547866f, 
		0.00117461f, 1.24222f, -0.0547866f, 0.00117461f, 1.24222f, -0.0547866f, 
		0.00117461f, 1.24222f, -0.0547866f, 0.00118475f, 1.08354f, 0.169053f, 
		0.00118475f, 1.08354f, 0.169053f, 0.00118475f, 1.08354f, 0.169053f, 
		0.00120149f, 1.31898f, -0.0190460f, 0.00120149f, 1.31898f, -0.0190460f, 
		0.00122750f, 1.07915f, -0.423262f, 0.00122750f, 1.07915f, -0.423262f, 
		0.00122750f, 1.07915f, -0.423262f, 0.00122750f, 1.07915f, -0.423262f, 
		0.00122750f, 1.07915f, -0.423262f, 0.00122750f, 1.07915f, -0.423262f, 
		0.00129396f, 1.04300f, 0.411618f, 0.00129396f, 1.04300f, 0.411618f, 
		0.00129396f, 1.04300f, 0.411618f, 0.00129396f, 1.04300f, 0.411618f, 
		0.00129396f, 1.04300f, 0.411618f, 0.00129396f, 1.04300f, 0.411618f, 
		0.00129396f, 1.04300f, 0.411618f, 0.00130706f, 1.08933f, -0.147963f, 
		0.00130706f, 1.08933f, -0.147963f, 0.00130706f, 1.08933f, -0.147963f, 
		0.00130817f, 0.977046f, 0.238630f, 0.00135456f, 1.35019f, 0.00323267f, 
		0.00135456f, 1.35019f, 0.00323267f, 0.00135456f, 1.35019f, 0.00323267f, 
		0.00135456f, 1.35019f, 0.00323267f, 0.00135456f, 1.35019f, 0.00323267f, 
		0.00135456f, 1.35019f, 0.00323267f, 0.00135456f, 1.35019f, 0.00323267f, 
		0.00135456f, 1.35019f, 0.00323267f, 0.00135456f, 1.35019f, 0.00323267f, 
		0.00139208f, 1.06654f, 0.277639f, 0.00139208f, 1.06654f, 0.277639f, 
		0.00139208f, 1.06654f, 0.277639f, 0.00139208f, 1.06654f, 0.277639f, 
		0.00139208f, 1.06654f, 0.277639f, 0.00139208f, 1.06654f, 0.277639f, 
		0.00139208f, 1.06654f, 0.277639f, 0.00139208f, 1.06654f, 0.277639f, 
		0.00139208f, 1.06654f, 0.277639f, 0.00139208f, 1.06654f, 0.277639f, 
		0.00143742f, 1.35245f, 0.00855184f, 0.00143742f, 1.35245f, 0.00855184f, 
		0.00143742f, 1.35245f, 0.00855184f, 0.00143742f, 1.35245f, 0.00855184f, 
		0.00144291f, 1.19099f, 0.00317707f, 0.00144291f, 1.19099f, 0.00317707f, 
		0.00144291f, 1.19099f, 0.00317707f, 0.00144291f, 1.19099f, 0.00317707f, 
		0.00144291f, 1.19099f, 0.00317707f, 0.00144291f, 1.19099f, 0.00317707f, 
		0.00144291f, 1.19099f, 0.00317707f, 0.00144291f, 1.19099f, 0.00317707f, 
		0.00144291f, 1.19099f, 0.00317707f, 0.00144291f, 1.19099f, 0.00317707f, 
		0.00148546f, 1.54522f, 0.00856342f, 0.00148546f, 1.54522f, 0.00856342f, 
		0.00148546f, 1.54522f, 0.00856342f, 0.00148546f, 1.54522f, 0.00856342f, 
		0.00148547f, 1.19321f, 0.00856345f, 0.00148547f, 1.19321f, 0.00856345f, 
		0.00148547f, 1.19321f, 0.00856345f, 0.00148547f, 1.19321f, 0.00856345f, 
		0.00150771f, 1.05678f, -0.421680f, 0.00150771f, 1.05678f, -0.421680f, 
		0.00150771f, 1.05678f, -0.421680f, 0.00150771f, 1.05678f, -0.421680f, 
		0.00151389f, 1.08530f, -0.415658f, 0.00151389f, 1.08530f, -0.415658f, 
		0.00151389f, 1.08530f, -0.415658f, 0.00151389f, 1.08530f, -0.415658f, 
		0.00151441f, 1.10499f, -0.121166f, 0.00151441f, 1.10499f, -0.121166f, 
		0.00151441f, 1.10499f, -0.121166f, 0.00151441f, 1.10499f, -0.121166f, 
		0.00151441f, 1.10499f, -0.121166f, 0.00151441f, 1.10499f, -0.121166f, 
		0.00154420f, 1.07857f, 0.422820f, 0.00154420f, 1.07857f, 0.422820f, 
		0.00154420f, 1.07857f, 0.422820f, 0.00154420f, 1.07857f, 0.422820f, 
		0.00154420f, 1.07857f, 0.422820f, 0.00154420f, 1.07857f, 0.422820f, 
		0.00157907f, 1.15110f, 0.0837578f, 0.00157907f, 1.15110f, 0.0837578f, 
		0.00159196f, 1.31898f, -0.0769676f, 0.00159196f, 1.31898f, -0.0769676f, 
		0.00159196f, 1.31898f, -0.0769676f, 0.00160339f, 1.03761f, -0.406369f, 
		0.00160339f, 1.03761f, -0.406369f, 0.00160339f, 1.03761f, -0.406369f, 
		0.00160339f, 1.03761f, -0.406369f, 0.00160339f, 1.03761f, -0.406369f, 
		0.00160339f, 1.03761f, -0.406369f, 0.00160339f, 1.03761f, -0.406369f, 
		0.00160339f, 1.03761f, -0.406369f, 0.00161711f, 1.07664f, 0.279695f, 
		0.00161711f, 1.07664f, 0.279695f, 0.00161711f, 1.07664f, 0.279695f, 
		0.00161711f, 1.07664f, 0.279695f, 0.00161711f, 1.07664f, 0.279695f, 
		0.00161711f, 1.07664f, 0.279695f, 0.00161711f, 1.07664f, 0.279695f, 
		0.00161711f, 1.07664f, 0.279695f, 0.00162734f, 1.33583f, -0.0830226f, 
		0.00162734f, 1.33583f, -0.0830226f, 0.00162734f, 1.33583f, -0.0830226f, 
		0.00162734f, 1.33583f, -0.0830226f, 0.00162734f, 1.33583f, -0.0830226f, 
		0.00162734f, 1.33583f, -0.0830226f, 0.00162734f, 1.33583f, -0.0830226f, 
		0.00163688f, 1.14399f, -0.0868418f, 0.00163688f, 1.14399f, -0.0868418f, 
		0.00163688f, 1.14399f, -0.0868418f, 0.00163688f, 1.14399f, -0.0868418f, 
		0.00163688f, 1.14399f, -0.0868418f, 0.00163688f, 1.14399f, -0.0868418f, 
		0.00163688f, 1.14399f, -0.0868418f, 0.00168989f, 1.08530f, 0.415556f, 
		0.00168989f, 1.08530f, 0.415556f, 0.00168989f, 1.08530f, 0.415556f, 
		0.00168989f, 1.08530f, 0.415556f, 0.00168996f, 1.31898f, 0.0772864f, 
		0.00168996f, 1.31898f, 0.0772864f, 0.00168996f, 1.31898f, 0.0772864f, 
		0.00168996f, 1.31898f, 0.0772864f, 0.00170319f, 1.10437f, 0.121583f, 
		0.00170319f, 1.10437f, 0.121583f, 0.00170319f, 1.10437f, 0.121583f, 
		0.00170319f, 1.10437f, 0.121583f, 0.00170319f, 1.10437f, 0.121583f, 
		0.00170319f, 1.10437f, 0.121583f, 0.00177031f, 1.07631f, -0.280358f, 
		0.00177031f, 1.07631f, -0.280358f, 0.00177031f, 1.07631f, -0.280358f, 
		0.00177031f, 1.07631f, -0.280358f, 0.00177031f, 1.07631f, -0.280358f, 
		0.00186799f, 1.06626f, -0.264482f, 0.00186799f, 1.06626f, -0.264482f, 
		0.00186799f, 1.06626f, -0.264482f, 0.00186799f, 1.06626f, -0.264482f, 
		0.00186799f, 1.06626f, -0.264482f, 0.00186799f, 1.06626f, -0.264482f, 
		0.00186799f, 1.07141f, -0.264482f, 0.00186799f, 1.07141f, -0.264482f, 
		0.00186799f, 1.07141f, -0.264482f, 0.00186799f, 1.07141f, -0.264482f, 
		0.00186799f, 1.07141f, -0.264482f, 0.00186799f, 1.07141f, -0.264482f, 
		0.00192173f, 1.07728f, -0.392495f, 0.00192173f, 1.07728f, -0.392495f, 
		0.00192173f, 1.07728f, -0.392495f, 0.00193730f, 1.25834f, 0.0599727f, 
		0.00193730f, 1.25834f, 0.0599727f, 0.00193730f, 1.25834f, 0.0599727f, 
		0.00193730f, 1.25834f, 0.0599727f, 0.00193730f, 1.25834f, 0.0599727f, 
		0.00193730f, 1.25834f, 0.0599727f, 0.00195406f, 1.33597f, 0.0713234f, 
		0.00195406f, 1.33597f, 0.0713234f, 0.00195406f, 1.33597f, 0.0713234f, 
		0.00195406f, 1.33597f, 0.0713234f, 0.00197839f, 1.30539f, 0.0695840f, 
		0.00197839f, 1.30539f, 0.0695840f, 0.00202233f, 1.30538f, 0.0843891f, 
		0.00206808f, 1.19317f, -0.00854830f, 0.00206808f, 1.19317f, -0.00854830f, 
		0.00206808f, 1.19317f, -0.00854830f, 0.00206808f, 1.19317f, -0.00854830f, 
		0.00206808f, 1.19317f, -0.00854830f, 0.00206808f, 1.19317f, -0.00854830f, 
		0.00206808f, 1.19317f, -0.00854830f, 0.00206808f, 1.19317f, -0.00854830f, 
		0.00207581f, 1.35235f, -0.00854828f, 0.00207581f, 1.35235f, -0.00854828f, 
		0.00207581f, 1.35235f, -0.00854828f, 0.00207581f, 1.35235f, -0.00854828f, 
		0.00207581f, 1.35235f, -0.00854828f, 0.00207581f, 1.35235f, -0.00854828f, 
		0.00207581f, 1.35235f, -0.00854828f, 0.00207581f, 1.35235f, -0.00854828f, 
		0.00207581f, 1.35235f, -0.00854828f, 0.00207586f, 1.54516f, -0.00854808f, 
		0.00207586f, 1.54516f, -0.00854808f, 0.00207586f, 1.54516f, -0.00854808f, 
		0.00207586f, 1.54516f, -0.00854808f, 0.00207586f, 1.54516f, -0.00854808f, 
		0.00207586f, 1.54516f, -0.00854808f, 0.00207586f, 1.54516f, -0.00854808f, 
		0.00207586f, 1.54516f, -0.00854808f, 0.00207586f, 1.54516f, -0.00854808f, 
		0.00209874f, 1.20548f, 0.00733904f, 0.00209874f, 1.20548f, 0.00733904f, 
		0.00209874f, 1.20548f, 0.00733904f, 0.00209874f, 1.20548f, 0.00733904f, 
		0.00209874f, 1.20548f, 0.00733904f, 0.00209874f, 1.20548f, 0.00733904f, 
		0.00209874f, 1.20548f, 0.00733904f, 0.00209874f, 1.20548f, 0.00733904f, 
		0.00209875f, 1.36468f, 0.00733904f, 0.00209875f, 1.36468f, 0.00733904f, 
		0.00209875f, 1.36468f, 0.00733904f, 0.00209875f, 1.36468f, 0.00733904f, 
		0.00209875f, 1.36468f, 0.00733904f, 0.00209875f, 1.36468f, 0.00733904f, 
		0.00209875f, 1.36468f, 0.00733904f, 0.00209875f, 1.36468f, 0.00733904f, 
		0.00209877f, 1.55749f, 0.00733905f, 0.00209877f, 1.55749f, 0.00733905f, 
		0.00209877f, 1.55749f, 0.00733905f, 0.00209877f, 1.55749f, 0.00733905f, 
		0.00209877f, 1.55749f, 0.00733905f, 0.00209877f, 1.55749f, 0.00733905f, 
		0.00209877f, 1.55749f, 0.00733905f, 0.00209877f, 1.55749f, 0.00733905f, 
		0.00210193f, 1.09285f, -0.403450f, 0.00210193f, 1.09285f, -0.403450f, 
		0.00224004f, 1.19138f, -0.00341737f, 0.00224004f, 1.19138f, -0.00341737f, 
		0.00224004f, 1.19138f, -0.00341737f, 0.00224004f, 1.19138f, -0.00341737f, 
		0.00224004f, 1.19138f, -0.00341737f, 0.00224004f, 1.19138f, -0.00341737f, 
		0.00224004f, 1.19138f, -0.00341737f, 0.00224004f, 1.19138f, -0.00341737f, 
		0.00226133f, 1.04337f, -0.408278f, 0.00226133f, 1.04337f, -0.408278f, 
		0.00226133f, 1.04337f, -0.408278f, 0.00226133f, 1.04337f, -0.408278f, 
		0.00226133f, 1.04337f, -0.408278f, 0.00226133f, 1.04337f, -0.408278f, 
		0.00227624f, 1.53974f, 0.00278381f, 0.00227624f, 1.53974f, 0.00278381f, 
		0.00227624f, 1.53974f, 0.00278381f, 0.00227624f, 1.53974f, 0.00278381f, 
		0.00227624f, 1.53974f, 0.00278381f, 0.00227624f, 1.53974f, 0.00278381f, 
		0.00228947f, 1.24608f, 0.0854418f, 0.00228947f, 1.24608f, 0.0854418f, 
		0.00228947f, 1.24608f, 0.0854418f, 0.00228947f, 1.24608f, 0.0854418f, 
		0.00228947f, 1.24608f, 0.0854418f, 0.00228947f, 1.24608f, 0.0854418f, 
		0.00228947f, 1.24608f, 0.0854418f, 0.00237776f, 1.31682f, 0.0686233f, 
		0.00237776f, 1.31682f, 0.0686233f, 0.00237776f, 1.31682f, 0.0686233f, 
		0.00241074f, 1.09030f, 0.264539f, 0.00241074f, 1.09030f, 0.264539f, 
		0.00241074f, 1.09030f, 0.264539f, 0.00241074f, 1.09030f, 0.264539f, 
		0.00241074f, 1.09030f, 0.264539f, 0.00241074f, 1.09030f, 0.264539f, 
		0.00242373f, 1.03412f, -0.377499f, 0.00242373f, 1.03412f, -0.377499f, 
		0.00242373f, 1.03412f, -0.377499f, 0.00242373f, 1.03412f, -0.377499f, 
		0.00242373f, 1.03412f, -0.377499f, 0.00242373f, 1.03412f, -0.377499f, 
		0.00242373f, 1.03412f, -0.377499f, 0.00242373f, 1.03412f, -0.377499f, 
		0.00242373f, 1.03412f, -0.377499f, 0.00242373f, 1.03412f, -0.377499f, 
		0.00243681f, 1.05468f, 0.406123f, 0.00243681f, 1.05468f, 0.406123f, 
		0.00243681f, 1.05468f, 0.406123f, 0.00243681f, 1.05468f, 0.406123f, 
		0.00243681f, 1.05468f, 0.406123f, 0.00243681f, 1.05468f, 0.406123f, 
		0.00243681f, 1.05468f, 0.406123f, 0.00243681f, 1.05468f, 0.406123f, 
		0.00244388f, 1.09309f, 0.403830f, 0.00244388f, 1.09309f, 0.403830f, 
		0.00244388f, 1.09309f, 0.403830f, 0.00244388f, 1.09309f, 0.403830f, 
		0.00246335f, 1.24593f, -0.0706844f, 0.00246335f, 1.24593f, -0.0706844f, 
		0.00246335f, 1.24593f, -0.0706844f, 0.00246335f, 1.24593f, -0.0706844f, 
		0.00246335f, 1.24593f, -0.0706844f, 0.00246335f, 1.24593f, -0.0706844f, 
		0.00246335f, 1.24593f, -0.0706844f, 0.00246335f, 1.24593f, -0.0706844f, 
		0.00246378f, 1.25858f, 0.0649441f, 0.00246378f, 1.25858f, 0.0649441f, 
		0.00246378f, 1.25858f, 0.0649441f, 0.00246378f, 1.25858f, 0.0649441f, 
		0.00246378f, 1.25858f, 0.0649441f, 0.00246378f, 1.25858f, 0.0649441f, 
		0.00246406f, 1.24552f, 0.0649440f, 0.00246406f, 1.24552f, 0.0649440f, 
		0.00246406f, 1.24552f, 0.0649440f, 0.00246406f, 1.24552f, 0.0649440f, 
		0.00246406f, 1.24552f, 0.0649440f, 0.00246406f, 1.24552f, 0.0649440f, 
		0.00246406f, 1.24552f, 0.0649440f, 0.00246406f, 1.24552f, 0.0649440f, 
		0.00247650f, 1.07658f, 0.261631f, 0.00247650f, 1.07658f, 0.261631f, 
		0.00247650f, 1.07658f, 0.261631f, 0.00247650f, 1.07658f, 0.261631f, 
		0.00247650f, 1.07658f, 0.261631f, 0.00247650f, 1.07658f, 0.261631f, 
		0.00249318f, 1.25817f, 0.0705814f, 0.00249318f, 1.25817f, 0.0705814f, 
		0.00249318f, 1.25817f, 0.0705814f, 0.00249318f, 1.25817f, 0.0705814f, 
		0.00249318f, 1.25817f, 0.0705814f, 0.00249318f, 1.25817f, 0.0705814f, 
		0.00249318f, 1.25817f, 0.0705814f, 0.00249318f, 1.25817f, 0.0705814f, 
		0.00249333f, 1.24593f, 0.0705814f, 0.00249333f, 1.24593f, 0.0705814f, 
		0.00249333f, 1.24593f, 0.0705814f, 0.00249333f, 1.24593f, 0.0705814f, 
		0.00249333f, 1.24593f, 0.0705814f, 0.00249333f, 1.24593f, 0.0705814f, 
		0.00250571f, 1.03700f, -0.414711f, 0.00250571f, 1.03700f, -0.414711f, 
		0.00250571f, 1.03700f, -0.414711f, 0.00250571f, 1.03700f, -0.414711f, 
		0.00250571f, 1.03700f, -0.414711f, 0.00251471f, 1.53013f, 0.0110821f, 
		0.00251471f, 1.53013f, 0.0110821f, 0.00251471f, 1.53013f, 0.0110821f, 
		0.00253423f, 1.07687f, 0.264548f, 0.00253423f, 1.07687f, 0.264548f, 
		0.00253423f, 1.07687f, 0.264548f, 0.00253423f, 1.07687f, 0.264548f, 
		0.00253423f, 1.07687f, 0.264548f, 0.00253423f, 1.07687f, 0.264548f, 
		0.00253698f, 1.07687f, 0.276794f, 0.00253698f, 1.07687f, 0.276794f, 
		0.00253698f, 1.07687f, 0.276794f, 0.00253698f, 1.07687f, 0.276794f, 
		0.00253698f, 1.07687f, 0.276794f, 0.00253698f, 1.07687f, 0.276794f, 
		0.00253698f, 1.07687f, 0.276794f, 0.00253698f, 1.07687f, 0.276794f, 
		0.00254497f, 1.03567f, 0.421699f, 0.00254497f, 1.03567f, 0.421699f, 
		0.00254497f, 1.03567f, 0.421699f, 0.00254497f, 1.03567f, 0.421699f, 
		0.00254497f, 1.03567f, 0.421699f, 0.00254497f, 1.03567f, 0.421699f, 
		0.00255203f, 1.31662f, -0.0689863f, 0.00255203f, 1.31662f, -0.0689863f, 
		0.00257393f, 1.08044f, -0.409956f, 0.00257393f, 1.08044f, -0.409956f, 
		0.00257393f, 1.08044f, -0.409956f, 0.00257393f, 1.08044f, -0.409956f, 
		0.00257393f, 1.08044f, -0.409956f, 0.00257899f, 1.18475f, -0.00889403f, 
		0.00257899f, 1.18475f, -0.00889403f, 0.00257899f, 1.18475f, -0.00889403f, 
		0.00257899f, 1.18475f, -0.00889403f, 0.00257899f, 1.18475f, -0.00889403f, 
		0.00257900f, 1.34395f, -0.00889399f, 0.00257900f, 1.34395f, -0.00889399f, 
		0.00257900f, 1.34395f, -0.00889399f, 0.00257900f, 1.34395f, -0.00889399f, 
		0.00257900f, 1.34395f, -0.00889399f, 0.00257919f, 1.53675f, -0.00889367f, 
		0.00257919f, 1.53675f, -0.00889367f, 0.00257919f, 1.53675f, -0.00889367f, 
		0.00257919f, 1.53675f, -0.00889367f, 0.00257919f, 1.53675f, -0.00889367f, 
		0.00260931f, 1.07451f, -0.417530f, 0.00260931f, 1.07451f, -0.417530f, 
		0.00260931f, 1.07451f, -0.417530f, 0.00260931f, 1.07451f, -0.417530f, 
		0.00260931f, 1.07451f, -0.417530f, 0.00262076f, 1.08967f, -0.423335f, 
		0.00262076f, 1.08967f, -0.423335f, 0.00262076f, 1.08967f, -0.423335f, 
		0.00262076f, 1.08967f, -0.423335f, 0.00262076f, 1.08967f, -0.423335f, 
		0.00263311f, 1.06254f, -0.260467f, 0.00263311f, 1.06254f, -0.260467f, 
		0.00263311f, 1.06254f, -0.260467f, 0.00263311f, 1.06254f, -0.260467f, 
		0.00263311f, 1.06254f, -0.260467f, 0.00263311f, 1.06254f, -0.260467f, 
		0.00263311f, 1.06254f, -0.260467f, 0.00263311f, 1.06254f, -0.260467f, 
		0.00264962f, 1.53687f, 0.00909666f, 0.00264962f, 1.53687f, 0.00909666f, 
		0.00264962f, 1.53687f, 0.00909666f, 0.00264962f, 1.53687f, 0.00909666f, 
		0.00265018f, 1.34406f, 0.00909785f, 0.00265018f, 1.34406f, 0.00909785f, 
		0.00265018f, 1.34406f, 0.00909785f, 0.00265018f, 1.34406f, 0.00909785f, 
		0.00265018f, 1.34406f, 0.00909785f, 0.00270591f, 1.07528f, -0.260721f, 
		0.00270591f, 1.07528f, -0.260721f, 0.00270591f, 1.07528f, -0.260721f, 
		0.00270591f, 1.07528f, -0.260721f, 0.00270591f, 1.07528f, -0.260721f, 
		0.00270591f, 1.07528f, -0.260721f, 0.00270591f, 1.07528f, -0.260721f, 
		0.00270591f, 1.07528f, -0.260721f, 0.00274455f, 1.06254f, 0.260364f, 
		0.00274455f, 1.06254f, 0.260364f, 0.00274455f, 1.06254f, 0.260364f, 
		0.00274455f, 1.06254f, 0.260364f, 0.00274455f, 1.06254f, 0.260364f, 
		0.00274455f, 1.06254f, 0.260364f, 0.00274455f, 1.06254f, 0.260364f, 
		0.00274455f, 1.06254f, 0.260364f, 0.00274455f, 1.06254f, 0.260364f, 
		0.00274455f, 1.06254f, 0.260364f, 0.00274808f, 1.32949f, -0.0874157f, 
		0.00274808f, 1.32949f, -0.0874157f, 0.00274808f, 1.32949f, -0.0874157f, 
		0.00274808f, 1.32949f, -0.0874157f, 0.00274808f, 1.32949f, -0.0874157f, 
		0.00274823f, 1.08044f, 0.409853f, 0.00274823f, 1.08044f, 0.409853f, 
		0.00274823f, 1.08044f, 0.409853f, 0.00274823f, 1.08044f, 0.409853f, 
		0.00274823f, 1.08044f, 0.409853f, 0.00278503f, 1.32949f, 0.0873125f, 
		0.00278503f, 1.32949f, 0.0873125f, 0.00278503f, 1.32949f, 0.0873125f, 
		0.00278623f, 1.07451f, 0.417426f, 0.00278623f, 1.07451f, 0.417426f, 
		0.00278623f, 1.07451f, 0.417426f, 0.00278623f, 1.07451f, 0.417426f, 
		0.00278623f, 1.07451f, 0.417426f, 0.00279444f, 1.26228f, -0.0563833f, 
		0.00279444f, 1.26228f, -0.0563833f, 0.00279444f, 1.26228f, -0.0563833f, 
		0.00279444f, 1.26228f, -0.0563833f, 0.00279444f, 1.26228f, -0.0563833f, 
		0.00279444f, 1.26228f, -0.0563833f, 0.00280020f, 1.03783f, -0.423425f, 
		0.00280020f, 1.03783f, -0.423425f, 0.00280020f, 1.03783f, -0.423425f, 
		0.00280020f, 1.03783f, -0.423425f, 0.00280020f, 1.03783f, -0.423425f, 
		0.00281224f, 1.07351f, 0.260419f, 0.00281224f, 1.07351f, 0.260419f, 
		0.00281224f, 1.07351f, 0.260419f, 0.00281224f, 1.07351f, 0.260419f, 
		0.00281224f, 1.07351f, 0.260419f, 0.00281224f, 1.07351f, 0.260419f, 
		0.00284892f, 1.04204f, 0.407588f, 0.00284892f, 1.04204f, 0.407588f, 
		0.00284892f, 1.04204f, 0.407588f, 0.00284892f, 1.04204f, 0.407588f, 
		0.00284892f, 1.04204f, 0.407588f, 0.00284892f, 1.04204f, 0.407588f, 
		0.00284892f, 1.04204f, 0.407588f, 0.00284892f, 1.04204f, 0.407588f, 
		0.00284892f, 1.04204f, 0.407588f, 0.00284892f, 1.04204f, 0.407588f, 
		0.00286029f, 1.24188f, -0.0669047f, 0.00286029f, 1.24188f, -0.0669047f, 
		0.00286029f, 1.24188f, -0.0669047f, 0.00286029f, 1.24188f, -0.0669047f, 
		0.00286029f, 1.24188f, -0.0669047f, 0.00286029f, 1.24188f, -0.0669047f, 
		0.00286029f, 1.24188f, -0.0669047f, 0.00286029f, 1.24188f, -0.0669047f, 
		0.00286029f, 1.24188f, -0.0669047f, 0.00286029f, 1.24188f, -0.0669047f, 
		0.00286508f, 1.03508f, -0.419498f, 0.00286508f, 1.03508f, -0.419498f, 
		0.00286508f, 1.03508f, -0.419498f, 0.00287062f, 1.53202f, -0.0108103f, 
		0.00287062f, 1.53202f, -0.0108103f, 0.00287062f, 1.53202f, -0.0108103f, 
		0.00287062f, 1.53202f, -0.0108103f, 0.00287062f, 1.53202f, -0.0108103f, 
		0.00287069f, 1.33921f, -0.0108103f, 0.00287069f, 1.33921f, -0.0108103f, 
		0.00287069f, 1.33921f, -0.0108103f, 0.00287069f, 1.33921f, -0.0108103f, 
		0.00287069f, 1.33921f, -0.0108103f, 0.00287070f, 1.18001f, -0.0108103f, 
		0.00287070f, 1.18001f, -0.0108103f, 0.00287070f, 1.18001f, -0.0108103f, 
		0.00287070f, 1.18001f, -0.0108103f, 0.00287070f, 1.18001f, -0.0108103f, 
		0.00291440f, 1.03683f, 0.415104f, 0.00291440f, 1.03683f, 0.415104f, 
		0.00291440f, 1.03683f, 0.415104f, 0.00291440f, 1.03683f, 0.415104f, 
		0.00293635f, 1.26229f, 0.0562701f, 0.00293635f, 1.26229f, 0.0562701f, 
		0.00293635f, 1.26229f, 0.0562701f, 0.00293635f, 1.26229f, 0.0562701f, 
		0.00293635f, 1.26229f, 0.0562701f, 0.00293635f, 1.26229f, 0.0562701f, 
		0.00297204f, 1.32298f, -0.0731061f, 0.00297204f, 1.32298f, -0.0731061f, 
		0.00297204f, 1.32298f, -0.0731061f, 0.00297204f, 1.32298f, -0.0731061f, 
		0.00297204f, 1.32298f, -0.0731061f, 0.00297204f, 1.32298f, -0.0731061f, 
		0.00298662f, 1.32274f, 0.0820407f, 0.00298662f, 1.32274f, 0.0820407f, 
		0.00298662f, 1.32274f, 0.0820407f, 0.00298662f, 1.32274f, 0.0820407f, 
		0.00298662f, 1.32274f, 0.0820407f, 0.00303502f, 1.09183f, 0.276000f, 
		0.00303502f, 1.09183f, 0.276000f, 0.00303502f, 1.09183f, 0.276000f, 
		0.00303502f, 1.09183f, 0.276000f, 0.00303502f, 1.09183f, 0.276000f, 
		0.00303502f, 1.09183f, 0.276000f, 0.00303502f, 1.09183f, 0.276000f, 
		0.00303502f, 1.09183f, 0.276000f, 0.00303831f, 1.24669f, -0.0854371f, 
		0.00303831f, 1.24669f, -0.0854371f, 0.00303831f, 1.24669f, -0.0854371f, 
		0.00303831f, 1.24669f, -0.0854371f, 0.00304747f, 1.09169f, -0.265467f, 
		0.00304747f, 1.09169f, -0.265467f, 0.00304747f, 1.09169f, -0.265467f, 
		0.00304747f, 1.09169f, -0.265467f, 0.00305084f, 1.05571f, 0.414904f, 
		0.00305084f, 1.05571f, 0.414904f, 0.00305084f, 1.05571f, 0.414904f, 
		0.00305084f, 1.05571f, 0.414904f, 0.00309355f, 1.04832f, -0.408558f, 
		0.00309355f, 1.04832f, -0.408558f, 0.00309355f, 1.04832f, -0.408558f, 
		0.00309355f, 1.04832f, -0.408558f, 0.00309355f, 1.04832f, -0.408558f, 
		0.00309355f, 1.04832f, -0.408558f, 0.00309355f, 1.04832f, -0.408558f, 
		0.00309355f, 1.04832f, -0.408558f, 0.00309355f, 1.04832f, -0.408558f, 
		0.00309355f, 1.04832f, -0.408558f, 0.00309355f, 1.04832f, -0.408558f, 
		0.00314000f, 1.07687f, -0.265238f, 0.00314000f, 1.07687f, -0.265238f, 
		0.00314000f, 1.07687f, -0.265238f, 0.00314000f, 1.07687f, -0.265238f, 
		0.00314000f, 1.07687f, -0.265238f, 0.00314000f, 1.07687f, -0.265238f, 
		0.00314000f, 1.07687f, -0.265238f, 0.00314000f, 1.07687f, -0.265238f, 
		0.00314054f, 1.04704f, 0.410713f, 0.00314054f, 1.04704f, 0.410713f, 
		0.00314054f, 1.04704f, 0.410713f, 0.00314054f, 1.04704f, 0.410713f, 
		0.00314054f, 1.04704f, 0.410713f, 0.00314054f, 1.04704f, 0.410713f, 
		0.00314054f, 1.04704f, 0.410713f, 0.00314054f, 1.04704f, 0.410713f, 
		0.00314054f, 1.04704f, 0.410713f, 0.00314054f, 1.04704f, 0.410713f, 
		0.00314054f, 1.04704f, 0.410713f, 0.00314123f, 1.18870f, -0.000511089f, 
		0.00314123f, 1.18870f, -0.000511089f, 0.00314123f, 1.18870f, -0.000511089f, 
		0.00314123f, 1.18870f, -0.000511089f, 0.00314123f, 1.18870f, -0.000511089f, 
		0.00314123f, 1.18870f, -0.000511089f, 0.00314123f, 1.18870f, -0.000511089f, 
		0.00314123f, 1.18870f, -0.000511089f, 0.00321236f, 1.25764f, -0.0707845f, 
		0.00321236f, 1.25764f, -0.0707845f, 0.00321236f, 1.25764f, -0.0707845f, 
		0.00321236f, 1.25764f, -0.0707845f, 0.00321236f, 1.25764f, -0.0707845f, 
		0.00321236f, 1.25764f, -0.0707845f, 0.00321236f, 1.25764f, -0.0707845f, 
		0.00322076f, 1.32264f, -0.0818751f, 0.00322076f, 1.32264f, -0.0818751f, 
		0.00322076f, 1.32264f, -0.0818751f, 0.00322076f, 1.32264f, -0.0818751f, 
		0.00322076f, 1.32264f, -0.0818751f, 0.00322076f, 1.32264f, -0.0818751f, 
		0.00330756f, 0.958053f, -0.279384f, 0.00330756f, 0.958053f, -0.279384f, 
		0.00330756f, 0.958053f, -0.279384f, 0.00330756f, 0.958053f, -0.279384f, 
		0.00330756f, 0.958053f, -0.279384f, 0.00330756f, 0.958053f, -0.279384f, 
		0.00331440f, 1.08799f, 0.423360f, 0.00331440f, 1.08799f, 0.423360f, 
		0.00331440f, 1.08799f, 0.423360f, 0.00331440f, 1.08799f, 0.423360f, 
		0.00331440f, 1.08799f, 0.423360f, 0.00331986f, 1.32275f, 0.0733202f, 
		0.00331986f, 1.32275f, 0.0733202f, 0.00331986f, 1.32275f, 0.0733202f, 
		0.00331986f, 1.32275f, 0.0733202f, 0.00331986f, 1.32275f, 0.0733202f, 
		0.00341003f, 0.958054f, 0.256711f, 0.00341003f, 0.958054f, 0.256711f, 
		0.00341003f, 0.958054f, 0.256711f, 0.00341003f, 0.958054f, 0.256711f, 
		0.00341003f, 0.958054f, 0.256711f, 0.00347744f, 1.04341f, -0.412496f, 
		0.00347744f, 1.04341f, -0.412496f, 0.00347744f, 1.04341f, -0.412496f, 
		0.00347744f, 1.04341f, -0.412496f, 0.00347744f, 1.04341f, -0.412496f, 
		0.00347744f, 1.04341f, -0.412496f, 0.00347744f, 1.04341f, -0.412496f, 
		0.00347744f, 1.04341f, -0.412496f, 0.00347744f, 1.04341f, -0.412496f, 
		0.00349848f, 1.52584f, -0.00400864f, 0.00349848f, 1.52584f, -0.00400864f, 
		0.00349848f, 1.52584f, -0.00400864f, 0.00349848f, 1.52584f, -0.00400864f, 
		0.00349848f, 1.52584f, -0.00400864f, 0.00349848f, 1.52584f, -0.00400864f, 
		0.00349848f, 1.52584f, -0.00400864f, 0.00350941f, 1.00044f, -0.291159f, 
		0.00350941f, 1.00044f, -0.291159f, 0.00352071f, 1.54320f, -0.00177941f, 
		0.00352071f, 1.54320f, -0.00177941f, 0.00352071f, 1.54320f, -0.00177941f, 
		0.00352071f, 1.54320f, -0.00177941f, 0.00352071f, 1.54320f, -0.00177941f, 
		0.00352071f, 1.54320f, -0.00177941f, 0.00352071f, 1.54320f, -0.00177941f, 
		0.00356543f, 1.34688f, -0.00133693f, 0.00356543f, 1.34688f, -0.00133693f, 
		0.00356543f, 1.34688f, -0.00133693f, 0.00356543f, 1.34688f, -0.00133693f, 
		0.00356543f, 1.34688f, -0.00133693f, 0.00356543f, 1.34688f, -0.00133693f, 
		0.00356543f, 1.34688f, -0.00133693f, 0.00356543f, 1.34688f, -0.00133693f, 
		0.00356543f, 1.34688f, -0.00133693f, 0.00356543f, 1.34688f, -0.00133693f, 
		0.00356543f, 1.34688f, -0.00133693f, 0.00356631f, 1.53969f, -0.00133640f, 
		0.00356631f, 1.53969f, -0.00133640f, 0.00356631f, 1.53969f, -0.00133640f, 
		0.00356631f, 1.53969f, -0.00133640f, 0.00356631f, 1.53969f, -0.00133640f, 
		0.00356631f, 1.53969f, -0.00133640f, 0.00356631f, 1.53969f, -0.00133640f, 
		0.00356631f, 1.53969f, -0.00133640f, 0.00356631f, 1.53969f, -0.00133640f, 
		0.00358504f, 1.05258f, -0.394016f, 0.00358504f, 1.05258f, -0.394016f, 
		0.00358504f, 1.05258f, -0.394016f, 0.00358504f, 1.05258f, -0.394016f, 
		0.00358972f, 1.24619f, 0.0599698f, 0.00358972f, 1.24619f, 0.0599698f, 
		0.00358972f, 1.24619f, 0.0599698f, 0.00358972f, 1.24619f, 0.0599698f, 
		0.00358972f, 1.24619f, 0.0599698f, 0.00358972f, 1.24619f, 0.0599698f, 
		0.00360101f, 1.03489f, 0.416880f, 0.00360101f, 1.03489f, 0.416880f, 
		0.00360101f, 1.03489f, 0.416880f, 0.00362308f, 1.52832f, -0.00895389f, 
		0.00362308f, 1.52832f, -0.00895389f, 0.00362308f, 1.52832f, -0.00895389f, 
		0.00362308f, 1.52832f, -0.00895389f, 0.00362970f, 1.04824f, 0.427078f, 
		0.00362970f, 1.04824f, 0.427078f, 0.00362970f, 1.04824f, 0.427078f, 
		0.00362970f, 1.04824f, 0.427078f, 0.00363114f, 1.00044f, 0.291055f, 
		0.00363114f, 1.00044f, 0.291055f, 0.00364726f, 1.06024f, 0.378538f, 
		0.00364726f, 1.06024f, 0.378538f, 0.00364726f, 1.06024f, 0.378538f, 
		0.00364726f, 1.06024f, 0.378538f, 0.00364726f, 1.06024f, 0.378538f, 
		0.00364726f, 1.06024f, 0.378538f, 0.00367180f, 1.09463f, -0.414235f, 
		0.00367180f, 1.09463f, -0.414235f, 0.00367180f, 1.09463f, -0.414235f, 
		0.00367904f, 1.04865f, -0.425759f, 0.00367904f, 1.04865f, -0.425759f, 
		0.00367904f, 1.04865f, -0.425759f, 0.00367904f, 1.04865f, -0.425759f, 
		0.00369057f, 1.00954f, -0.295623f, 0.00369930f, 1.54384f, 0.00356509f, 
		0.00369930f, 1.54384f, 0.00356509f, 0.00369930f, 1.54384f, 0.00356509f, 
		0.00369930f, 1.54384f, 0.00356509f, 0.00369930f, 1.54384f, 0.00356509f, 
		0.00369930f, 1.54384f, 0.00356509f, 0.00369930f, 1.54384f, 0.00356509f, 
		0.00369930f, 1.54384f, 0.00356509f, 0.00372748f, 1.03623f, -0.415714f, 
		0.00372748f, 1.03623f, -0.415714f, 0.00375156f, 1.05258f, 0.394010f, 
		0.00375156f, 1.05258f, 0.394010f, 0.00375156f, 1.05258f, 0.394010f, 
		0.00375156f, 1.05258f, 0.394010f, 0.00378661f, 1.34608f, 0.00226146f, 
		0.00378661f, 1.34608f, 0.00226146f, 0.00378661f, 1.34608f, 0.00226146f, 
		0.00378661f, 1.34608f, 0.00226146f, 0.00378661f, 1.34608f, 0.00226146f, 
		0.00380258f, 1.35027f, -0.00176618f, 0.00380258f, 1.35027f, -0.00176618f, 
		0.00380258f, 1.35027f, -0.00176618f, 0.00380258f, 1.35027f, -0.00176618f, 
		0.00380258f, 1.35027f, -0.00176618f, 0.00380258f, 1.35027f, -0.00176618f, 
		0.00380258f, 1.35027f, -0.00176618f, 0.00380258f, 1.35027f, -0.00176618f, 
		0.00380258f, 1.35027f, -0.00176618f, 0.00380258f, 1.35027f, -0.00176618f, 
		0.00381106f, 1.01221f, 0.245745f, 0.00381172f, 1.33498f, 0.0840074f, 
		0.00381172f, 1.33498f, 0.0840074f, 0.00381561f, 1.00954f, 0.295519f, 
		0.00381561f, 1.00954f, 0.295519f, 0.00381561f, 1.00954f, 0.295519f, 
		0.00390299f, 1.05375f, -0.406514f, 0.00390299f, 1.05375f, -0.406514f, 
		0.00390299f, 1.05375f, -0.406514f, 0.00390299f, 1.05375f, -0.406514f, 
		0.00390299f, 1.05375f, -0.406514f, 0.00390299f, 1.05375f, -0.406514f, 
		0.00390299f, 1.05375f, -0.406514f, 0.00396176f, 1.05980f, 0.384682f, 
		0.00396176f, 1.05980f, 0.384682f, 0.00396176f, 1.05980f, 0.384682f, 
		0.00396176f, 1.05980f, 0.384682f, 0.00396176f, 1.05980f, 0.384682f, 
		0.00396176f, 1.05980f, 0.384682f, 0.00396176f, 1.05980f, 0.384682f, 
		0.00396176f, 1.05980f, 0.384682f, 0.00414309f, 0.829042f, -0.276349f, 
		0.00414309f, 0.829042f, -0.276349f, 0.00414309f, 0.829042f, -0.276349f, 
		0.00417611f, 0.956532f, -0.283431f, 0.00417611f, 0.956532f, -0.283431f, 
		0.00417611f, 0.956532f, -0.283431f, 0.00417611f, 0.956532f, -0.283431f, 
		0.00417611f, 0.956532f, -0.283431f, 0.00418739f, 1.18730f, -0.00134975f, 
		0.00418739f, 1.18730f, -0.00134975f, 0.00418739f, 1.18730f, -0.00134975f, 
		0.00418739f, 1.18730f, -0.00134975f, 0.00418739f, 1.18730f, -0.00134975f, 
		0.00418739f, 1.18730f, -0.00134975f, 0.00418739f, 1.18730f, -0.00134975f, 
		0.00418739f, 1.18730f, -0.00134975f, 0.00418739f, 1.18730f, -0.00134975f, 
		0.00418739f, 1.18730f, -0.00134975f, 0.00418878f, 1.02272f, 0.237282f, 
		0.00418878f, 1.02272f, 0.237282f, 0.00419109f, 1.33605f, -0.0730966f, 
		0.00419109f, 1.33605f, -0.0730966f, 0.00419109f, 1.33605f, -0.0730966f, 
		0.00419109f, 1.33605f, -0.0730966f, 0.00419109f, 1.33605f, -0.0730966f, 
		0.00419109f, 1.33605f, -0.0730966f, 0.00419690f, 1.05987f, -0.0306784f, 
		0.00419690f, 1.05987f, -0.0306784f, 0.00419690f, 1.05987f, -0.0306784f, 
		0.00419690f, 1.05987f, -0.0306784f, 0.00419703f, 1.03221f, -0.0306785f, 
		0.00419703f, 1.03221f, -0.0306785f, 0.00419703f, 1.03221f, -0.0306785f, 
		0.00419703f, 1.03221f, -0.0306785f, 0.00421025f, 1.05987f, 0.0306724f, 
		0.00421025f, 1.05987f, 0.0306724f, 0.00421025f, 1.05987f, 0.0306724f, 
		0.00421025f, 1.05987f, 0.0306724f, 0.00425764f, 0.829042f, 0.259708f, 
		0.00425764f, 0.829042f, 0.259708f, 0.00425764f, 0.829042f, 0.259708f, 
		0.00426031f, 0.829042f, 0.276245f, 0.00426031f, 0.829042f, 0.276245f, 
		0.00426031f, 0.829042f, 0.276245f, 0.00431336f, 1.52603f, 0.00410120f, 
		0.00431336f, 1.52603f, 0.00410120f, 0.00431336f, 1.52603f, 0.00410120f, 
		0.00431336f, 1.52603f, 0.00410120f, 0.00431336f, 1.52603f, 0.00410120f, 
		0.00432050f, 1.08733f, 0.409852f, 0.00432050f, 1.08733f, 0.409852f, 
		0.00432050f, 1.08733f, 0.409852f, 0.00432050f, 1.08733f, 0.409852f, 
		0.00432050f, 1.08733f, 0.409852f, 0.00432666f, 1.33231f, -0.00188225f, 
		0.00432666f, 1.33231f, -0.00188225f, 0.00432666f, 1.33231f, -0.00188225f, 
		0.00432666f, 1.33231f, -0.00188225f, 0.00432666f, 1.33231f, -0.00188225f, 
		0.00432666f, 1.33231f, -0.00188225f, 0.00432666f, 1.33231f, -0.00188225f, 
		0.00432666f, 1.33231f, -0.00188225f, 0.00433246f, 1.17312f, -0.00189858f, 
		0.00433246f, 1.17312f, -0.00189858f, 0.00433246f, 1.17312f, -0.00189858f, 
		0.00433246f, 1.17312f, -0.00189858f, 0.00433246f, 1.17312f, -0.00189858f, 
		0.00433246f, 1.17312f, -0.00189858f, 0.00433246f, 1.17312f, -0.00189858f, 
		0.00433246f, 1.17312f, -0.00189858f, 0.00433246f, 1.17312f, -0.00189858f, 
		0.00437033f, 1.45770f, -0.0262823f, 0.00437033f, 1.45770f, -0.0262823f, 
		0.00437033f, 1.45770f, -0.0262823f, 0.00437033f, 1.45770f, -0.0262823f, 
		0.00437033f, 1.45770f, -0.0262823f, 0.00437035f, 1.51224f, -0.0262823f, 
		0.00437035f, 1.51224f, -0.0262823f, 0.00437035f, 1.51224f, -0.0262823f, 
		0.00437035f, 1.51224f, -0.0262823f, 0.00437035f, 1.51224f, -0.0262823f, 
		0.00438506f, 1.06667f, 0.264838f, 0.00438506f, 1.06667f, 0.264838f, 
		0.00438506f, 1.06667f, 0.264838f, 0.00438506f, 1.06667f, 0.264838f, 
		0.00438506f, 1.06667f, 0.264838f, 0.00438506f, 1.06667f, 0.264838f, 
		0.00438506f, 1.06667f, 0.264838f, 0.00438506f, 1.06667f, 0.264838f, 
		0.00438506f, 1.06667f, 0.264838f, 0.00438506f, 1.06667f, 0.264838f, 
		0.00438506f, 1.06667f, 0.264838f, 0.00442578f, 1.25652f, -0.0854402f, 
		0.00442578f, 1.25652f, -0.0854402f, 0.00442578f, 1.25652f, -0.0854402f, 
		0.00442578f, 1.25652f, -0.0854402f, 0.00442578f, 1.25652f, -0.0854402f, 
		0.00442578f, 1.25652f, -0.0854402f, 0.00442578f, 1.25652f, -0.0854402f, 
		0.00442578f, 1.25652f, -0.0854402f, 0.00442578f, 1.25652f, -0.0854402f, 
		0.00442578f, 1.25652f, -0.0854402f, 0.00442578f, 1.25652f, -0.0854402f, 
		0.00449947f, 1.09029f, -0.275355f, 0.00449947f, 1.09029f, -0.275355f, 
		0.00449947f, 1.09029f, -0.275355f, 0.00449947f, 1.09029f, -0.275355f, 
		0.00449947f, 1.09029f, -0.275355f, 0.00449947f, 1.09029f, -0.275355f, 
		0.00451697f, 1.07686f, -0.275349f, 0.00451697f, 1.07686f, -0.275349f, 
		0.00451697f, 1.07686f, -0.275349f, 0.00451697f, 1.07686f, -0.275349f, 
		0.00451697f, 1.07686f, -0.275349f, 0.00451697f, 1.07686f, -0.275349f, 
		0.00460527f, 1.04615f, -0.426201f, 0.00460527f, 1.04615f, -0.426201f, 
		0.00460527f, 1.04615f, -0.426201f, 0.00460527f, 1.04615f, -0.426201f, 
		0.00460527f, 1.04615f, -0.426201f, 0.00460527f, 1.04615f, -0.426201f, 
		0.00461645f, 1.45569f, 0.0251873f, 0.00461645f, 1.45569f, 0.0251873f, 
		0.00461645f, 1.45569f, 0.0251873f, 0.00461645f, 1.45569f, 0.0251873f, 
		0.00462686f, 1.24741f, -0.0652286f, 0.00462686f, 1.24741f, -0.0652286f, 
		0.00462686f, 1.24741f, -0.0652286f, 0.00462686f, 1.24741f, -0.0652286f, 
		0.00462686f, 1.24741f, -0.0652286f, 0.00462686f, 1.24741f, -0.0652286f, 
		0.00462776f, 1.25669f, -0.0600754f, 0.00462776f, 1.25669f, -0.0600754f, 
		0.00462776f, 1.25669f, -0.0600754f, 0.00462776f, 1.25669f, -0.0600754f, 
		0.00462776f, 1.25669f, -0.0600754f, 0.00462776f, 1.25669f, -0.0600754f, 
		0.00464332f, 1.25648f, 0.0848210f, 0.00464332f, 1.25648f, 0.0848210f, 
		0.00464332f, 1.25648f, 0.0848210f, 0.00464332f, 1.25648f, 0.0848210f, 
		0.00464332f, 1.25648f, 0.0848210f, 0.00464332f, 1.25648f, 0.0848210f, 
		0.00464332f, 1.25648f, 0.0848210f, 0.00464332f, 1.25648f, 0.0848210f, 
		0.00464332f, 1.25648f, 0.0848210f, 0.00464332f, 1.25648f, 0.0848210f, 
		0.00464332f, 1.25648f, 0.0848210f, 0.00464332f, 1.25648f, 0.0848210f, 
		0.00469753f, 1.07141f, 0.266029f, 0.00469753f, 1.07141f, 0.266029f, 
		0.00469753f, 1.07141f, 0.266029f, 0.00469753f, 1.07141f, 0.266029f, 
		0.00469753f, 1.07141f, 0.266029f, 0.00469753f, 1.07141f, 0.266029f, 
		0.00475191f, 1.31789f, 0.0782846f, 0.00475191f, 1.31789f, 0.0782846f, 
		0.00475191f, 1.31789f, 0.0782846f, 0.00475191f, 1.31789f, 0.0782846f, 
		0.00475191f, 1.31789f, 0.0782846f, 0.00476177f, 1.07159f, 0.278785f, 
		0.00476177f, 1.07159f, 0.278785f, 0.00476177f, 1.07159f, 0.278785f, 
		0.00476177f, 1.07159f, 0.278785f, 0.00476177f, 1.07159f, 0.278785f, 
		0.00476177f, 1.07159f, 0.278785f, 0.00476177f, 1.07159f, 0.278785f, 
		0.00476177f, 1.07159f, 0.278785f, 0.00478693f, 1.09295f, 0.411842f, 
		0.00478693f, 1.09295f, 0.411842f, 0.00478693f, 1.09295f, 0.411842f, 
		0.00478693f, 1.09295f, 0.411842f, 0.00478693f, 1.09295f, 0.411842f, 
		0.00478693f, 1.09295f, 0.411842f, 0.00479008f, 1.31942f, -0.0121639f, 
		0.00479008f, 1.31942f, -0.0121639f, 0.00479008f, 1.31942f, -0.0121639f, 
		0.00479008f, 1.31942f, -0.0121639f, 0.00479008f, 1.31942f, -0.0121639f, 
		0.00479008f, 1.31942f, -0.0121639f, 0.00479008f, 1.31942f, -0.0121639f, 
		0.00479008f, 1.31942f, -0.0121639f, 0.00480441f, 1.24381f, 0.0598390f, 
		0.00480441f, 1.24381f, 0.0598390f, 0.00480441f, 1.24381f, 0.0598390f, 
		0.00480441f, 1.24381f, 0.0598390f, 0.00480441f, 1.24381f, 0.0598390f, 
		0.00480441f, 1.24381f, 0.0598390f, 0.00480857f, 1.20338f, -0.0295070f, 
		0.00483100f, 0.980877f, 0.238168f, 0.00483881f, 1.37273f, 0.0262231f, 
		0.00483881f, 1.37273f, 0.0262231f, 0.00483881f, 1.37273f, 0.0262231f, 
		0.00483881f, 1.37273f, 0.0262231f, 0.00485821f, 1.06614f, -0.278861f, 
		0.00485821f, 1.06614f, -0.278861f, 0.00485821f, 1.06614f, -0.278861f, 
		0.00485821f, 1.06614f, -0.278861f, 0.00485821f, 1.06614f, -0.278861f, 
		0.00485821f, 1.06614f, -0.278861f, 0.00485821f, 1.06614f, -0.278861f, 
		0.00485821f, 1.06614f, -0.278861f, 0.00490471f, 1.02864f, -0.361269f, 
		0.00490471f, 1.02864f, -0.361269f, 0.00490471f, 1.02864f, -0.361269f, 
		0.00490471f, 1.02864f, -0.361269f, 0.00490471f, 1.02864f, -0.361269f, 
		0.00490592f, 1.06527f, -0.361269f, 0.00490592f, 1.06527f, -0.361269f, 
		0.00490592f, 1.06527f, -0.361269f, 0.00490592f, 1.06527f, -0.361269f, 
		0.00492897f, 1.06083f, 0.262243f, 0.00492897f, 1.06083f, 0.262243f, 
		0.00492897f, 1.06083f, 0.262243f, 0.00492897f, 1.06083f, 0.262243f, 
		0.00492897f, 1.06083f, 0.262243f, 0.00492897f, 1.06083f, 0.262243f, 
		0.00498671f, 1.30330f, 0.0183129f, 0.00498671f, 1.30330f, 0.0183129f, 
		0.00498671f, 1.30330f, 0.0183129f, 0.00498671f, 1.30330f, 0.0183129f, 
		0.00501152f, 1.08774f, -0.410126f, 0.00501152f, 1.08774f, -0.410126f, 
		0.00501152f, 1.08774f, -0.410126f, 0.00501152f, 1.08774f, -0.410126f, 
		0.00501152f, 1.08774f, -0.410126f, 0.00501152f, 1.08774f, -0.410126f, 
		0.00501152f, 1.08774f, -0.410126f, 0.00501152f, 1.08774f, -0.410126f, 
		0.00503000f, 1.02272f, 0.303613f, 0.00503000f, 1.02272f, 0.303613f, 
		0.00503000f, 1.03132f, 0.303613f, 0.00503000f, 1.03132f, 0.303613f, 
		0.00503000f, 1.03132f, 0.303613f, 0.00507544f, 1.37418f, -0.0243443f, 
		0.00507544f, 1.37418f, -0.0243443f, 0.00507544f, 1.37418f, -0.0243443f, 
		0.00507544f, 1.37418f, -0.0243443f, 0.00507544f, 1.37418f, -0.0243443f, 
		0.00507548f, 1.42865f, -0.0243444f, 0.00507548f, 1.42865f, -0.0243444f, 
		0.00507548f, 1.42865f, -0.0243444f, 0.00507548f, 1.42865f, -0.0243444f, 
		0.00507548f, 1.42865f, -0.0243444f, 0.00507548f, 1.48320f, -0.0243444f, 
		0.00507548f, 1.48320f, -0.0243444f, 0.00507548f, 1.48320f, -0.0243444f, 
		0.00507548f, 1.48320f, -0.0243444f, 0.00507548f, 1.48320f, -0.0243444f, 
		0.00510659f, 1.08447f, -0.405013f, 0.00510659f, 1.08447f, -0.405013f, 
		0.00510659f, 1.08447f, -0.405013f, 0.00510659f, 1.08447f, -0.405013f, 
		0.00510659f, 1.08447f, -0.405013f, 0.00510659f, 1.08447f, -0.405013f, 
		0.00517465f, 0.828613f, -0.262851f, 0.00517465f, 0.828613f, -0.262851f, 
		0.00518692f, 1.32705f, 0.0678410f, 0.00518692f, 1.32705f, 0.0678410f, 
		0.00518692f, 1.32705f, 0.0678410f, 0.00518692f, 1.32705f, 0.0678410f, 
		0.00519369f, 1.40264f, -0.0261198f, 0.00519369f, 1.40264f, -0.0261198f, 
		0.00519369f, 1.40264f, -0.0261198f, 0.00519369f, 1.40264f, -0.0261198f, 
		0.00519369f, 1.40264f, -0.0261198f, 0.00520097f, 1.08465f, 0.404981f, 
		0.00520097f, 1.08465f, 0.404981f, 0.00520097f, 1.08465f, 0.404981f, 
		0.00520097f, 1.08465f, 0.404981f, 0.00520097f, 1.08465f, 0.404981f, 
		0.00523679f, 1.42838f, 0.0263720f, 0.00523679f, 1.42838f, 0.0263720f, 
		0.00523679f, 1.42838f, 0.0263720f, 0.00523679f, 1.42838f, 0.0263720f, 
		0.00523679f, 1.42838f, 0.0263720f, 0.00523679f, 1.42838f, 0.0263720f, 
		0.00523683f, 1.48292f, 0.0263720f, 0.00523683f, 1.48292f, 0.0263720f, 
		0.00523683f, 1.48292f, 0.0263720f, 0.00523683f, 1.48292f, 0.0263720f, 
		0.00523683f, 1.48292f, 0.0263720f, 0.00523683f, 1.48292f, 0.0263720f, 
		0.00527449f, 1.31804f, -0.0759117f, 0.00527449f, 1.31804f, -0.0759117f, 
		0.00527449f, 1.31804f, -0.0759117f, 0.00527449f, 1.31804f, -0.0759117f, 
		0.00527449f, 1.31804f, -0.0759117f, 0.00529254f, 1.04084f, -0.394014f, 
		0.00529254f, 1.04084f, -0.394014f, 0.00529254f, 1.04084f, -0.394014f, 
		0.00529254f, 1.04084f, -0.394014f, 0.00529254f, 1.04084f, -0.394014f, 
		0.00529254f, 1.04084f, -0.394014f, 0.00532319f, 1.40333f, 0.0263026f, 
		0.00532319f, 1.40333f, 0.0263026f, 0.00532319f, 1.40333f, 0.0263026f, 
		0.00532319f, 1.40333f, 0.0263026f, 0.00535435f, 1.36468f, -0.00537872f, 
		0.00535435f, 1.36468f, -0.00537872f, 0.00535435f, 1.36468f, -0.00537872f, 
		0.00535435f, 1.36468f, -0.00537872f, 0.00535435f, 1.36468f, -0.00537872f, 
		0.00535435f, 1.36468f, -0.00537872f, 0.00535435f, 1.36468f, -0.00537872f, 
		0.00535435f, 1.36468f, -0.00537872f, 0.00535435f, 1.20548f, -0.00537872f, 
		0.00535435f, 1.20548f, -0.00537872f, 0.00535435f, 1.20548f, -0.00537872f, 
		0.00535435f, 1.20548f, -0.00537872f, 0.00535435f, 1.20548f, -0.00537872f, 
		0.00535435f, 1.20548f, -0.00537872f, 0.00535435f, 1.20548f, -0.00537872f, 
		0.00535435f, 1.20548f, -0.00537872f, 0.00535443f, 1.55749f, -0.00537875f, 
		0.00535443f, 1.55749f, -0.00537875f, 0.00535443f, 1.55749f, -0.00537875f, 
		0.00535443f, 1.55749f, -0.00537875f, 0.00535443f, 1.55749f, -0.00537875f, 
		0.00535443f, 1.55749f, -0.00537875f, 0.00535443f, 1.55749f, -0.00537875f, 
		0.00535443f, 1.55749f, -0.00537875f, 0.00538742f, 1.09161f, 0.267587f, 
		0.00538742f, 1.09161f, 0.267587f, 0.00538742f, 1.09161f, 0.267587f, 
		0.00538742f, 1.09161f, 0.267587f, 0.00538742f, 1.09161f, 0.267587f, 
		0.00538742f, 1.09161f, 0.267587f, 0.00538742f, 1.09161f, 0.267587f, 
		0.00538742f, 1.09161f, 0.267587f, 0.00541554f, 1.04453f, 0.425523f, 
		0.00541554f, 1.04453f, 0.425523f, 0.00541554f, 1.04453f, 0.425523f, 
		0.00541554f, 1.04453f, 0.425523f, 0.00541554f, 1.04453f, 0.425523f, 
		0.00545953f, 1.04084f, 0.394008f, 0.00545953f, 1.04084f, 0.394008f, 
		0.00545953f, 1.04084f, 0.394008f, 0.00545953f, 1.04084f, 0.394008f, 
		0.00545953f, 1.04084f, 0.394008f, 0.00545953f, 1.04084f, 0.394008f, 
		0.00548649f, 1.09011f, 0.392392f, 0.00551465f, 1.29913f, -0.0326614f, 
		0.00552100f, 1.24885f, -0.0706860f, 0.00552100f, 1.24885f, -0.0706860f, 
		0.00552100f, 1.24885f, -0.0706860f, 0.00552100f, 1.24885f, -0.0706860f, 
		0.00555100f, 1.24885f, 0.0705820f, 0.00555100f, 1.24885f, 0.0705820f, 
		0.00555100f, 1.24885f, 0.0705820f, 0.00555100f, 1.24885f, 0.0705820f, 
		0.00565016f, 1.07698f, 0.273898f, 0.00565016f, 1.07698f, 0.273898f, 
		0.00565016f, 1.07698f, 0.273898f, 0.00565016f, 1.07698f, 0.273898f, 
		0.00566289f, 1.07179f, -0.263202f, 0.00566289f, 1.07179f, -0.263202f, 
		0.00566289f, 1.07179f, -0.263202f, 0.00566289f, 1.07179f, -0.263202f, 
		0.00566289f, 1.07179f, -0.263202f, 0.00566289f, 1.07179f, -0.263202f, 
		0.00566289f, 1.07179f, -0.263202f, 0.00566289f, 1.07179f, -0.263202f, 
		0.00569347f, 1.04658f, 0.425811f, 0.00569347f, 1.04658f, 0.425811f, 
		0.00569347f, 1.04658f, 0.425811f, 0.00569347f, 1.04658f, 0.425811f, 
		0.00569347f, 1.04658f, 0.425811f, 0.00569347f, 1.04658f, 0.425811f, 
		0.00574352f, 1.02269f, -0.250585f, 0.00574352f, 1.02269f, -0.250585f, 
		0.00574352f, 1.02269f, -0.250585f, 0.00574352f, 1.02269f, -0.250585f, 
		0.00581009f, 1.05398f, -0.414759f, 0.00581009f, 1.05398f, -0.414759f, 
		0.00581009f, 1.05398f, -0.414759f, 0.00581009f, 1.05398f, -0.414759f, 
		0.00582276f, 1.03132f, -0.303899f, 0.00582276f, 1.03132f, -0.303899f, 
		0.00582276f, 1.03132f, -0.303899f, 0.00582276f, 1.03132f, -0.303899f, 
		0.00586418f, 1.51044f, 0.0253361f, 0.00586418f, 1.51044f, 0.0253361f, 
		0.00586418f, 1.51044f, 0.0253361f, 0.00586418f, 1.51044f, 0.0253361f, 
		0.00592311f, 1.31910f, 0.0102520f, 0.00592311f, 1.31910f, 0.0102520f, 
		0.00592311f, 1.31910f, 0.0102520f, 0.00592311f, 1.31910f, 0.0102520f, 
		0.00592311f, 1.31910f, 0.0102520f, 0.00592311f, 1.31910f, 0.0102520f, 
		0.00593632f, 1.03132f, 0.237545f, 0.00593632f, 1.03132f, 0.237545f, 
		0.00593632f, 1.03132f, 0.237545f, 0.00594310f, 0.989731f, -0.309277f, 
		0.00594808f, 1.52632f, 0.000555611f, 0.00594808f, 1.52632f, 0.000555611f, 
		0.00594808f, 1.52632f, 0.000555611f, 0.00594808f, 1.52632f, 0.000555611f, 
		0.00594808f, 1.52632f, 0.000555611f, 0.00595057f, 1.03132f, 0.303794f, 
		0.00595057f, 1.03132f, 0.303794f, 0.00595057f, 1.03132f, 0.303794f, 
		0.00595057f, 1.03132f, 0.303794f, 0.00595233f, 1.55470f, -0.0543519f, 
		0.00595773f, 0.989732f, -0.232270f, 0.00597107f, 1.31634f, -0.0830789f, 
		0.00600098f, 1.03670f, 0.376971f, 0.00600098f, 1.03670f, 0.376971f, 
		0.00600098f, 1.03670f, 0.376971f, 0.00600098f, 1.03670f, 0.376971f, 
		0.00600098f, 1.03670f, 0.376971f, 0.00600098f, 1.03670f, 0.376971f, 
		0.00601798f, 0.980923f, -0.301643f, 0.00602119f, 1.29323f, -0.0409331f, 
		0.00602154f, 1.07842f, -0.403934f, 0.00602154f, 1.07842f, -0.403934f, 
		0.00602154f, 1.07842f, -0.403934f, 0.00602154f, 1.07842f, -0.403934f, 
		0.00603802f, 1.29323f, 0.0409256f, 0.00605384f, 1.40109f, 0.0244257f, 
		0.00605384f, 1.40109f, 0.0244257f, 0.00605384f, 1.40109f, 0.0244257f, 
		0.00605384f, 1.40109f, 0.0244257f, 0.00605683f, 0.989731f, 0.232167f, 
		0.00607386f, 1.07869f, 0.403347f, 0.00607408f, 0.989729f, 0.309171f, 
		0.00608210f, 1.03975f, 0.406394f, 0.00608210f, 1.03975f, 0.406394f, 
		0.00608210f, 1.03975f, 0.406394f, 0.00608210f, 1.03975f, 0.406394f, 
		0.00608210f, 1.03975f, 0.406394f, 0.00608210f, 1.03975f, 0.406394f, 
		0.00609320f, 1.32780f, 0.0105465f, 0.00609320f, 1.32780f, 0.0105465f, 
		0.00609320f, 1.32780f, 0.0105465f, 0.00609320f, 1.32780f, 0.0105465f, 
		0.00609320f, 1.32780f, 0.0105465f, 0.00609320f, 1.32780f, 0.0105465f, 
		0.00609320f, 1.32780f, 0.0105465f, 0.00609320f, 1.32780f, 0.0105465f, 
		0.00611112f, 1.00045f, -0.313656f, 0.00611112f, 1.00045f, -0.313656f, 
		0.00612866f, 1.00045f, -0.227893f, 0.00612866f, 1.00045f, -0.227893f, 
		0.00614611f, 0.980923f, 0.301539f, 0.00617001f, 1.09117f, -0.269243f, 
		0.00617001f, 1.09117f, -0.269243f, 0.00617001f, 1.09117f, -0.269243f, 
		0.00617001f, 1.09117f, -0.269243f, 0.00617001f, 1.09117f, -0.269243f, 
		0.00617001f, 1.09117f, -0.269243f, 0.00617084f, 1.25039f, -0.0846560f, 
		0.00617084f, 1.25039f, -0.0846560f, 0.00617084f, 1.25039f, -0.0846560f, 
		0.00617084f, 1.25039f, -0.0846560f, 0.00617084f, 1.25039f, -0.0846560f, 
		0.00617084f, 1.25039f, -0.0846560f, 0.00617084f, 1.25039f, -0.0846560f, 
		0.00617084f, 1.25039f, -0.0846560f, 0.00619229f, 1.07125f, -0.274879f, 
		0.00619229f, 1.07125f, -0.274879f, 0.00619229f, 1.07125f, -0.274879f, 
		0.00619229f, 1.07125f, -0.274879f, 0.00619229f, 1.07125f, -0.274879f, 
		0.00619229f, 1.07125f, -0.274879f, 0.00619229f, 1.07125f, -0.274879f, 
		0.00619229f, 1.07125f, -0.274879f, 0.00619229f, 1.07125f, -0.274879f, 
		0.00619229f, 1.07125f, -0.274879f, 0.00619229f, 1.07125f, -0.274879f, 
		0.00619229f, 1.07125f, -0.274879f, 0.00619229f, 1.07125f, -0.274879f, 
		0.00619229f, 1.07125f, -0.274879f, 0.00619229f, 1.07125f, -0.274879f, 
		0.00619229f, 1.07125f, -0.274879f, 0.00619282f, 1.07686f, -0.269099f, 
		0.00619282f, 1.07686f, -0.269099f, 0.00619282f, 1.07686f, -0.269099f, 
		0.00619282f, 1.07686f, -0.269099f, 0.00619282f, 1.07686f, -0.269099f, 
		0.00619282f, 1.07686f, -0.269099f, 0.00620916f, 1.48424f, 0.0241581f, 
		0.00620916f, 1.48424f, 0.0241581f, 0.00620916f, 1.48424f, 0.0241581f, 
		0.00620916f, 1.48424f, 0.0241581f, 0.00620917f, 1.42970f, 0.0241581f, 
		0.00620917f, 1.42970f, 0.0241581f, 0.00620917f, 1.42970f, 0.0241581f, 
		0.00620917f, 1.42970f, 0.0241581f, 0.00620917f, 1.42970f, 0.0241581f, 
		0.00620917f, 1.42970f, 0.0241581f, 0.00620922f, 1.37522f, 0.0241581f, 
		0.00620922f, 1.37522f, 0.0241581f, 0.00620922f, 1.37522f, 0.0241581f, 
		0.00620922f, 1.37522f, 0.0241581f, 0.00622473f, 1.00045f, 0.227788f, 
		0.00622473f, 1.00045f, 0.227788f, 0.00623499f, 1.25373f, -0.0706834f, 
		0.00623499f, 1.25373f, -0.0706834f, 0.00623499f, 1.25373f, -0.0706834f, 
		0.00623499f, 1.25373f, -0.0706834f, 0.00623499f, 1.25373f, -0.0706834f, 
		0.00623499f, 1.25373f, -0.0706834f, 0.00623499f, 1.25373f, -0.0706834f, 
		0.00623499f, 1.25373f, -0.0706834f, 0.00624319f, 1.00045f, 0.313552f, 
		0.00624319f, 1.00045f, 0.313552f, 0.00624523f, 1.31659f, 0.0828352f, 
		0.00624523f, 1.31659f, 0.0828352f, 0.00626503f, 1.25373f, 0.0705788f, 
		0.00626503f, 1.25373f, 0.0705788f, 0.00626503f, 1.25373f, 0.0705788f, 
		0.00626503f, 1.25373f, 0.0705788f, 0.00626503f, 1.25373f, 0.0705788f, 
		0.00626503f, 1.25373f, 0.0705788f, 0.00626503f, 1.25373f, 0.0705788f, 
		0.00626503f, 1.25373f, 0.0705788f, 0.00627195f, 1.25037f, 0.0836844f, 
		0.00627195f, 1.25037f, 0.0836844f, 0.00627195f, 1.25037f, 0.0836844f, 
		0.00627195f, 1.25037f, 0.0836844f, 0.00627195f, 1.25037f, 0.0836844f, 
		0.00627195f, 1.25037f, 0.0836844f, 0.00627195f, 1.25037f, 0.0836844f, 
		0.00627195f, 1.25037f, 0.0836844f, 0.00628318f, 1.06626f, -0.269074f, 
		0.00628318f, 1.06626f, -0.269074f, 0.00628318f, 1.06626f, -0.269074f, 
		0.00628318f, 1.06626f, -0.269074f, 0.00628318f, 1.07141f, -0.269074f, 
		0.00628318f, 1.07141f, -0.269074f, 0.00628318f, 1.07141f, -0.269074f, 
		0.00628318f, 1.07141f, -0.269074f, 0.00628318f, 1.07141f, -0.269074f, 
		0.00628318f, 1.07141f, -0.269074f, 0.00628383f, 1.40115f, -0.0242869f, 
		0.00628383f, 1.40115f, -0.0242869f, 0.00628383f, 1.40115f, -0.0242869f, 
		0.00628383f, 1.40115f, -0.0242869f, 0.00628383f, 1.40115f, -0.0242869f, 
		0.00629341f, 1.21088f, -0.0408927f, 0.00630687f, 1.07686f, 0.268995f, 
		0.00630687f, 1.07686f, 0.268995f, 0.00630687f, 1.07686f, 0.268995f, 
		0.00630687f, 1.07686f, 0.268995f, 0.00630687f, 1.07686f, 0.268995f, 
		0.00630687f, 1.07686f, 0.268995f, 0.00630687f, 1.07686f, 0.268995f, 
		0.00630687f, 1.07686f, 0.268995f, 0.00631021f, 1.06607f, -0.264155f, 
		0.00631021f, 1.06607f, -0.264155f, 0.00631021f, 1.06607f, -0.264155f, 
		0.00631021f, 1.06607f, -0.264155f, 0.00631021f, 1.06607f, -0.264155f, 
		0.00631021f, 1.06607f, -0.264155f, 0.00631021f, 1.06607f, -0.264155f, 
		0.00631021f, 1.06607f, -0.264155f, 0.00631021f, 1.06607f, -0.264155f, 
		0.00631021f, 1.06607f, -0.264155f, 0.00631021f, 1.06607f, -0.264155f, 
		0.00631021f, 1.06607f, -0.264155f, 0.00631157f, 1.21087f, 0.0408846f, 
		0.00631184f, 1.08997f, 0.272345f, 0.00631184f, 1.08997f, 0.272345f, 
		0.00631184f, 1.08997f, 0.272345f, 0.00631184f, 1.08997f, 0.272345f, 
		0.00631184f, 1.08997f, 0.272345f, 0.00631184f, 1.08997f, 0.272345f, 
		0.00631184f, 1.08997f, 0.272345f, 0.00631184f, 1.08997f, 0.272345f, 
		0.00635300f, 1.24853f, -0.0603508f, 0.00635300f, 1.24853f, -0.0603508f, 
		0.00635300f, 1.24853f, -0.0603508f, 0.00635300f, 1.24853f, -0.0603508f, 
		0.00635300f, 1.24853f, -0.0603508f, 0.00635300f, 1.24853f, -0.0603508f, 
		0.00635300f, 1.24853f, -0.0603508f, 0.00635300f, 1.24853f, -0.0603508f, 
		0.00635300f, 1.24853f, -0.0603508f, 0.00635300f, 1.24853f, -0.0603508f, 
		0.00635300f, 1.24853f, -0.0603508f, 0.00635300f, 1.24853f, -0.0603508f, 
		0.00639800f, 1.06626f, 0.272368f, 0.00639800f, 1.06626f, 0.272368f, 
		0.00639800f, 1.06626f, 0.272368f, 0.00639800f, 1.06626f, 0.272368f, 
		0.00639800f, 1.06626f, 0.272368f, 0.00639800f, 1.06626f, 0.272368f, 
		0.00639800f, 1.07141f, 0.272368f, 0.00639800f, 1.07141f, 0.272368f, 
		0.00639800f, 1.07141f, 0.272368f, 0.00639800f, 1.07141f, 0.272368f, 
		0.00639800f, 1.07141f, 0.272368f, 0.00639800f, 1.07141f, 0.272368f, 
		0.00641682f, 1.04030f, -0.406407f, 0.00641682f, 1.04030f, -0.406407f, 
		0.00641682f, 1.04030f, -0.406407f, 0.00641682f, 1.04030f, -0.406407f, 
		0.00641682f, 1.04030f, -0.406407f, 0.00641682f, 1.04030f, -0.406407f, 
		0.00641682f, 1.04030f, -0.406407f, 0.00641682f, 1.04030f, -0.406407f, 
		0.00642141f, 1.42976f, -0.0263583f, 0.00642141f, 1.42976f, -0.0263583f, 
		0.00642141f, 1.42976f, -0.0263583f, 0.00642141f, 1.42976f, -0.0263583f, 
		0.00642141f, 1.42976f, -0.0263583f, 0.00642141f, 1.48430f, -0.0263583f, 
		0.00642141f, 1.48430f, -0.0263583f, 0.00642141f, 1.48430f, -0.0263583f, 
		0.00642141f, 1.48430f, -0.0263583f, 0.00642141f, 1.48430f, -0.0263583f, 
		0.00642148f, 1.37529f, -0.0263583f, 0.00642148f, 1.37529f, -0.0263583f, 
		0.00642148f, 1.37529f, -0.0263583f, 0.00642148f, 1.37529f, -0.0263583f, 
		0.00642148f, 1.37529f, -0.0263583f, 0.00643560f, 1.05464f, -0.415678f, 
		0.00643560f, 1.05464f, -0.415678f, 0.00643560f, 1.05464f, -0.415678f, 
		0.00643560f, 1.05464f, -0.415678f, 0.00643560f, 1.05464f, -0.415678f, 
		0.00646087f, 1.33599f, -0.00805258f, 0.00646087f, 1.33599f, -0.00805258f, 
		0.00646087f, 1.33599f, -0.00805258f, 0.00647436f, 1.33650f, 0.00873375f, 
		0.00647436f, 1.33650f, 0.00873375f, 0.00647440f, 1.17730f, 0.00873374f, 
		0.00647440f, 1.17730f, 0.00873374f, 0.00649388f, 1.35227f, 0.00590706f, 
		0.00649388f, 1.35227f, 0.00590706f, 0.00649388f, 1.35227f, 0.00590706f, 
		0.00649388f, 1.35227f, 0.00590706f, 0.00649388f, 1.35227f, 0.00590706f, 
		0.00649388f, 1.35227f, 0.00590706f, 0.00649388f, 1.35227f, 0.00590706f, 
		0.00649388f, 1.35227f, 0.00590706f, 0.00649388f, 1.35227f, 0.00590706f, 
		0.00649388f, 1.35227f, 0.00590706f, 0.00649398f, 1.19307f, 0.00590705f, 
		0.00649398f, 1.19307f, 0.00590705f, 0.00649398f, 1.19307f, 0.00590705f, 
		0.00649398f, 1.19307f, 0.00590705f, 0.00649398f, 1.19307f, 0.00590705f, 
		0.00649398f, 1.19307f, 0.00590705f, 0.00649398f, 1.19307f, 0.00590705f, 
		0.00649398f, 1.19307f, 0.00590705f, 0.00649398f, 1.19307f, 0.00590705f, 
		0.00649398f, 1.19307f, 0.00590705f, 0.00649425f, 1.54508f, 0.00590689f, 
		0.00649425f, 1.54508f, 0.00590689f, 0.00649425f, 1.54508f, 0.00590689f, 
		0.00649425f, 1.54508f, 0.00590689f, 0.00649425f, 1.54508f, 0.00590689f, 
		0.00649425f, 1.54508f, 0.00590689f, 0.00649425f, 1.54508f, 0.00590689f, 
		0.00649425f, 1.54508f, 0.00590689f, 0.00649425f, 1.54508f, 0.00590689f, 
		0.00649425f, 1.54508f, 0.00590689f, 0.00650570f, 1.02271f, 0.289951f, 
		0.00650570f, 1.02271f, 0.289951f, 0.00651496f, 1.45674f, -0.0246260f, 
		0.00651496f, 1.45674f, -0.0246260f, 0.00651496f, 1.45674f, -0.0246260f, 
		0.00651496f, 1.45674f, -0.0246260f, 0.00651496f, 1.45674f, -0.0246260f, 
		0.00651498f, 1.51129f, -0.0246260f, 0.00651498f, 1.51129f, -0.0246260f, 
		0.00651498f, 1.51129f, -0.0246260f, 0.00651498f, 1.51129f, -0.0246260f, 
		0.00651498f, 1.51129f, -0.0246260f, 0.00651590f, 1.02037f, -0.292573f, 
		0.00651590f, 1.02037f, -0.292573f, 0.00651590f, 1.02037f, -0.292573f, 
		0.00656472f, 1.06601f, 0.277245f, 0.00656472f, 1.06601f, 0.277245f, 
		0.00656472f, 1.06601f, 0.277245f, 0.00656472f, 1.06601f, 0.277245f, 
		0.00656472f, 1.06601f, 0.277245f, 0.00656472f, 1.06601f, 0.277245f, 
		0.00657441f, 1.05410f, 0.422203f, 0.00657441f, 1.05410f, 0.422203f, 
		0.00657441f, 1.05410f, 0.422203f, 0.00660010f, 1.33366f, -0.000786520f, 
		0.00660010f, 1.33366f, -0.000786520f, 0.00660010f, 1.33366f, -0.000786520f, 
		0.00660010f, 1.33366f, -0.000786520f, 0.00660010f, 1.33366f, -0.000786520f, 
		0.00660010f, 1.33366f, -0.000786520f, 0.00660066f, 1.17446f, -0.000786707f, 
		0.00660066f, 1.17446f, -0.000786707f, 0.00660066f, 1.17446f, -0.000786707f, 
		0.00660066f, 1.17446f, -0.000786707f, 0.00660066f, 1.17446f, -0.000786707f, 
		0.00660066f, 1.17446f, -0.000786707f, 0.00668071f, 1.45763f, 0.0253118f, 
		0.00668071f, 1.45763f, 0.0253118f, 0.00668071f, 1.45763f, 0.0253118f, 
		0.00668071f, 1.45763f, 0.0253118f, 0.00669953f, 1.05448f, 0.422888f, 
		0.00669953f, 1.05448f, 0.422888f, 0.00669953f, 1.05448f, 0.422888f, 
		0.00669953f, 1.05448f, 0.422888f, 0.00669953f, 1.05448f, 0.422888f, 
		0.00671852f, 1.03998f, 0.423525f, 0.00671852f, 1.03998f, 0.423525f, 
		0.00671852f, 1.03998f, 0.423525f, 0.00671852f, 1.03998f, 0.423525f, 
		0.00673091f, 1.25980f, -0.0662630f, 0.00673091f, 1.25980f, -0.0662630f, 
		0.00673091f, 1.25980f, -0.0662630f, 0.00673091f, 1.25980f, -0.0662630f, 
		0.00673091f, 1.25980f, -0.0662630f, 0.00673091f, 1.25980f, -0.0662630f, 
		0.00674814f, 1.00045f, 0.289954f, 0.00674814f, 1.00045f, 0.289954f, 
		0.00675183f, 1.24560f, 0.0702966f, 0.00675183f, 1.24560f, 0.0702966f, 
		0.00675183f, 1.24560f, 0.0702966f, 0.00675183f, 1.24560f, 0.0702966f, 
		0.00675183f, 1.24560f, 0.0702966f, 0.00675183f, 1.24560f, 0.0702966f, 
		0.00678691f, 1.03506f, -0.379339f, 0.00678691f, 1.03506f, -0.379339f, 
		0.00678691f, 1.03506f, -0.379339f, 0.00678691f, 1.03506f, -0.379339f, 
		0.00678691f, 1.03506f, -0.379339f, 0.00678691f, 1.03506f, -0.379339f, 
		0.00678691f, 1.03506f, -0.379339f, 0.00678691f, 1.03506f, -0.379339f, 
		0.00678691f, 1.03506f, -0.379339f, 0.00678691f, 1.03506f, -0.379339f, 
		0.00678715f, 1.05885f, -0.379339f, 0.00678715f, 1.05885f, -0.379339f, 
		0.00678715f, 1.05885f, -0.379339f, 0.00678715f, 1.05885f, -0.379339f, 
		0.00678715f, 1.05885f, -0.379339f, 0.00678715f, 1.05885f, -0.379339f, 
		0.00678715f, 1.05885f, -0.379339f, 0.00678715f, 1.05885f, -0.379339f, 
		0.00679916f, 1.25344f, -0.0650409f, 0.00679916f, 1.25344f, -0.0650409f, 
		0.00679916f, 1.25344f, -0.0650409f, 0.00679916f, 1.25344f, -0.0650409f, 
		0.00679916f, 1.25344f, -0.0650409f, 0.00679916f, 1.25344f, -0.0650409f, 
		0.00679916f, 1.25344f, -0.0650409f, 0.00679916f, 1.25344f, -0.0650409f, 
		0.00679916f, 1.25344f, -0.0650409f, 0.00679916f, 1.25344f, -0.0650409f, 
		0.00684250f, 1.25205f, 0.0651273f, 0.00684250f, 1.25205f, 0.0651273f, 
		0.00684250f, 1.25205f, 0.0651273f, 0.00684250f, 1.25205f, 0.0651273f, 
		0.00684250f, 1.25205f, 0.0651273f, 0.00684250f, 1.25205f, 0.0651273f, 
		0.00684458f, 1.18486f, 0.00655411f, 0.00684458f, 1.18486f, 0.00655411f, 
		0.00684458f, 1.18486f, 0.00655411f, 0.00684458f, 1.18486f, 0.00655411f, 
		0.00684458f, 1.18486f, 0.00655411f, 0.00685350f, 1.06658f, -0.273733f, 
		0.00685350f, 1.06658f, -0.273733f, 0.00685350f, 1.06658f, -0.273733f, 
		0.00685350f, 1.06658f, -0.273733f, 0.00685350f, 1.06658f, -0.273733f, 
		0.00685350f, 1.06658f, -0.273733f, 0.00685350f, 1.06658f, -0.273733f, 
		0.00685350f, 1.06658f, -0.273733f, 0.00685350f, 1.06658f, -0.273733f, 
		0.00685350f, 1.06658f, -0.273733f, 0.00685350f, 1.06658f, -0.273733f, 
		0.00685350f, 1.06658f, -0.273733f, 0.00685350f, 1.06658f, -0.273733f, 
		0.00686270f, 1.05898f, -0.383074f, 0.00686270f, 1.05898f, -0.383074f, 
		0.00686270f, 1.05898f, -0.383074f, 0.00686270f, 1.05898f, -0.383074f, 
		0.00686270f, 1.05898f, -0.383074f, 0.00686270f, 1.05898f, -0.383074f, 
		0.00686270f, 1.05898f, -0.383074f, 0.00686270f, 1.05898f, -0.383074f, 
		0.00687003f, 1.32137f, 0.0118913f, 0.00687003f, 1.32137f, 0.0118913f, 
		0.00687003f, 1.32137f, 0.0118913f, 0.00687003f, 1.32137f, 0.0118913f, 
		0.00687003f, 1.32137f, 0.0118913f, 0.00687003f, 1.32137f, 0.0118913f, 
		0.00687003f, 1.32137f, 0.0118913f, 0.00687003f, 1.32137f, 0.0118913f, 
		0.00687003f, 1.32137f, 0.0118913f, 0.00687003f, 1.32137f, 0.0118913f, 
		0.00687003f, 1.32137f, 0.0118913f, 0.00688417f, 1.07534f, 0.278517f, 
		0.00688417f, 1.07534f, 0.278517f, 0.00688417f, 1.07534f, 0.278517f, 
		0.00688417f, 1.07534f, 0.278517f, 0.00694647f, 1.32511f, 0.0120239f, 
		0.00694647f, 1.32511f, 0.0120239f, 0.00694647f, 1.32511f, 0.0120239f, 
		0.00694647f, 1.32511f, 0.0120239f, 0.00694647f, 1.32511f, 0.0120239f, 
		0.00694647f, 1.32511f, 0.0120239f, 0.00694647f, 1.32511f, 0.0120239f, 
		0.00694647f, 1.32511f, 0.0120239f, 0.00694647f, 1.32511f, 0.0120239f, 
		0.00694764f, 1.03506f, 0.379244f, 0.00694764f, 1.03506f, 0.379244f, 
		0.00694764f, 1.03506f, 0.379244f, 0.00694764f, 1.03506f, 0.379244f, 
		0.00694764f, 1.03506f, 0.379244f, 0.00694764f, 1.03506f, 0.379244f, 
		0.00694764f, 1.03506f, 0.379244f, 0.00694764f, 1.03506f, 0.379244f, 
		0.00694764f, 1.03506f, 0.379244f, 0.00695947f, 1.32536f, -0.0120219f, 
		0.00695947f, 1.32536f, -0.0120219f, 0.00695947f, 1.32536f, -0.0120219f, 
		0.00695947f, 1.32536f, -0.0120219f, 0.00695947f, 1.32536f, -0.0120219f, 
		0.00695947f, 1.32536f, -0.0120219f, 0.00708734f, 0.828613f, -0.269996f, 
		0.00708734f, 0.828613f, -0.269996f, 0.00710938f, 1.05913f, 0.381390f, 
		0.00710938f, 1.05913f, 0.381390f, 0.00710938f, 1.05913f, 0.381390f, 
		0.00710938f, 1.05913f, 0.381390f, 0.00710938f, 1.05913f, 0.381390f, 
		0.00710938f, 1.05913f, 0.381390f, 0.00713506f, 1.18378f, 0.00735007f, 
		0.00713506f, 1.18378f, 0.00735007f, 0.00713506f, 1.18378f, 0.00735007f, 
		0.00713506f, 1.18378f, 0.00735007f, 0.00713506f, 1.18378f, 0.00735007f, 
		0.00713506f, 1.34298f, 0.00735008f, 0.00713506f, 1.34298f, 0.00735008f, 
		0.00713506f, 1.34298f, 0.00735008f, 0.00713506f, 1.34298f, 0.00735008f, 
		0.00713506f, 1.34298f, 0.00735008f, 0.00713527f, 1.53579f, 0.00734999f, 
		0.00713527f, 1.53579f, 0.00734999f, 0.00713527f, 1.53579f, 0.00734999f, 
		0.00713527f, 1.53579f, 0.00734999f, 0.00713527f, 1.53579f, 0.00734999f, 
		0.00718025f, 1.28192f, -0.0479426f, 0.00718064f, 1.00801f, -0.294365f, 
		0.00718064f, 1.00801f, -0.294365f, 0.00718064f, 1.00801f, -0.294365f, 
		0.00718064f, 1.00801f, -0.294365f, 0.00718064f, 1.00801f, -0.294365f, 
		0.00718662f, 1.23699f, -0.0497017f, 0.00718662f, 1.23699f, -0.0497017f, 
		0.00719948f, 1.28192f, 0.0479359f, 0.00720047f, 0.828613f, 0.266062f, 
		0.00720047f, 0.828613f, 0.266062f, 0.00720855f, 1.23699f, 0.0496946f, 
		0.00720855f, 1.23699f, 0.0496946f, 0.00724293f, 1.04817f, -0.394015f, 
		0.00724293f, 1.04817f, -0.394015f, 0.00724293f, 1.04817f, -0.394015f, 
		0.00724293f, 1.04817f, -0.394015f, 0.00724293f, 1.04817f, -0.394015f, 
		0.00724293f, 1.04817f, -0.394015f, 0.00724907f, 1.22218f, -0.0479326f, 
		0.00725935f, 1.25397f, 0.0603606f, 0.00725935f, 1.25397f, 0.0603606f, 
		0.00725935f, 1.25397f, 0.0603606f, 0.00725935f, 1.25397f, 0.0603606f, 
		0.00725935f, 1.25397f, 0.0603606f, 0.00725935f, 1.25397f, 0.0603606f, 
		0.00725935f, 1.25397f, 0.0603606f, 0.00725935f, 1.25397f, 0.0603606f, 
		0.00725935f, 1.25397f, 0.0603606f, 0.00725935f, 1.25397f, 0.0603606f, 
		0.00726967f, 1.22218f, 0.0479254f, 0.00732494f, 1.08271f, -0.392493f, 
		0.00733329f, 1.20548f, 0.00194875f, 0.00733329f, 1.20548f, 0.00194875f, 
		0.00733329f, 1.20548f, 0.00194875f, 0.00733329f, 1.20548f, 0.00194875f, 
		0.00733329f, 1.20548f, 0.00194875f, 0.00733329f, 1.20548f, 0.00194875f, 
		0.00733329f, 1.20548f, 0.00194875f, 0.00733329f, 1.20548f, 0.00194875f, 
		0.00733329f, 1.36468f, 0.00194875f, 0.00733329f, 1.36468f, 0.00194875f, 
		0.00733329f, 1.36468f, 0.00194875f, 0.00733329f, 1.36468f, 0.00194875f, 
		0.00733329f, 1.36468f, 0.00194875f, 0.00733329f, 1.36468f, 0.00194875f, 
		0.00733329f, 1.36468f, 0.00194875f, 0.00733329f, 1.36468f, 0.00194875f, 
		0.00733331f, 1.55749f, 0.00194877f, 0.00733331f, 1.55749f, 0.00194877f, 
		0.00733331f, 1.55749f, 0.00194877f, 0.00733331f, 1.55749f, 0.00194877f, 
		0.00733331f, 1.55749f, 0.00194877f, 0.00733331f, 1.55749f, 0.00194877f, 
		0.00733331f, 1.55749f, 0.00194877f, 0.00733331f, 1.55749f, 0.00194877f, 
		0.00734988f, 1.06256f, -0.278425f, 0.00734988f, 1.06256f, -0.278425f, 
		0.00734988f, 1.06256f, -0.278425f, 0.00734988f, 1.06256f, -0.278425f, 
		0.00734988f, 1.06256f, -0.278425f, 0.00734988f, 1.06256f, -0.278425f, 
		0.00735023f, 1.09190f, 0.419642f, 0.00735023f, 1.09190f, 0.419642f, 
		0.00735023f, 1.09190f, 0.419642f, 0.00735023f, 1.09190f, 0.419642f, 
		0.00735023f, 1.09190f, 0.419642f, 0.00735023f, 1.09190f, 0.419642f, 
		0.00741017f, 1.04817f, 0.394008f, 0.00741017f, 1.04817f, 0.394008f, 
		0.00741017f, 1.04817f, 0.394008f, 0.00741017f, 1.04817f, 0.394008f, 
		0.00741017f, 1.04817f, 0.394008f, 0.00741017f, 1.04817f, 0.394008f, 
		0.00741017f, 1.04817f, 0.394008f, 0.00741017f, 1.04817f, 0.394008f, 
		0.00741998f, 1.24440f, 0.0562693f, 0.00741998f, 1.24440f, 0.0562693f, 
		0.00741998f, 1.24440f, 0.0562693f, 0.00741998f, 1.24440f, 0.0562693f, 
		0.00741998f, 1.24440f, 0.0562693f, 0.00741998f, 1.24440f, 0.0562693f, 
		0.00742200f, 1.03305f, -0.370003f, 0.00742200f, 1.03305f, -0.370003f, 
		0.00742200f, 1.05903f, -0.370003f, 0.00742200f, 1.05903f, -0.370003f, 
		0.00743099f, 1.24432f, -0.0577308f, 0.00743099f, 1.24432f, -0.0577308f, 
		0.00743099f, 1.24432f, -0.0577308f, 0.00743099f, 1.24432f, -0.0577308f, 
		0.00743099f, 1.24432f, -0.0577308f, 0.00743099f, 1.24432f, -0.0577308f, 
		0.00743099f, 1.24432f, -0.0577308f, 0.00743099f, 1.24432f, -0.0577308f, 
		0.00743099f, 1.24432f, -0.0577308f, 0.00743099f, 1.24432f, -0.0577308f, 
		0.00744722f, 1.05050f, 0.406382f, 0.00744722f, 1.05050f, 0.406382f, 
		0.00744722f, 1.05050f, 0.406382f, 0.00744722f, 1.05050f, 0.406382f, 
		0.00744722f, 1.05050f, 0.406382f, 0.00744722f, 1.05050f, 0.406382f, 
		0.00744722f, 1.05050f, 0.406382f, 0.00745448f, 1.25966f, 0.0672262f, 
		0.00745448f, 1.25966f, 0.0672262f, 0.00745448f, 1.25966f, 0.0672262f, 
		0.00745448f, 1.25966f, 0.0672262f, 0.00745448f, 1.25966f, 0.0672262f, 
		0.00745448f, 1.25966f, 0.0672262f, 0.00745448f, 1.25966f, 0.0672262f, 
		0.00745448f, 1.25966f, 0.0672262f, 0.00745448f, 1.25966f, 0.0672262f, 
		0.00745448f, 1.25966f, 0.0672262f, 0.00745463f, 1.24444f, 0.0672234f, 
		0.00745463f, 1.24444f, 0.0672234f, 0.00745463f, 1.24444f, 0.0672234f, 
		0.00745463f, 1.24444f, 0.0672234f, 0.00745463f, 1.24444f, 0.0672234f, 
		0.00745463f, 1.24444f, 0.0672234f, 0.00746771f, 1.06256f, 0.278320f, 
		0.00746771f, 1.06256f, 0.278320f, 0.00746771f, 1.06256f, 0.278320f, 
		0.00746771f, 1.06256f, 0.278320f, 0.00746771f, 1.06256f, 0.278320f, 
		0.00746771f, 1.06256f, 0.278320f, 0.00750192f, 1.24583f, -0.0701728f, 
		0.00750192f, 1.24583f, -0.0701728f, 0.00750192f, 1.24583f, -0.0701728f, 
		0.00750192f, 1.24583f, -0.0701728f, 0.00750192f, 1.24583f, -0.0701728f, 
		0.00750192f, 1.24583f, -0.0701728f, 0.00750192f, 1.24583f, -0.0701728f, 
		0.00750192f, 1.24583f, -0.0701728f, 0.00750192f, 1.24583f, -0.0701728f, 
		0.00750192f, 1.24583f, -0.0701728f, 0.00750192f, 1.24583f, -0.0701728f, 
		0.00750192f, 1.24583f, -0.0701728f, 0.00750192f, 1.24583f, -0.0701728f, 
		0.00750192f, 1.24583f, -0.0701728f, 0.00750407f, 1.07667f, -0.265836f, 
		0.00750407f, 1.07667f, -0.265836f, 0.00750407f, 1.07667f, -0.265836f, 
		0.00750407f, 1.07667f, -0.265836f, 0.00750407f, 1.07667f, -0.265836f, 
		0.00752001f, 1.32727f, -0.0693230f, 0.00752001f, 1.32727f, -0.0693230f, 
		0.00752001f, 1.32727f, -0.0693230f, 0.00752001f, 1.32727f, -0.0693230f, 
		0.00752001f, 1.32727f, -0.0693230f, 0.00752001f, 1.32727f, -0.0693230f, 
		0.00752001f, 1.32727f, -0.0693230f, 0.00754964f, 1.07402f, -0.278326f, 
		0.00754964f, 1.07402f, -0.278326f, 0.00754964f, 1.07402f, -0.278326f, 
		0.00754964f, 1.07402f, -0.278326f, 0.00754964f, 1.07402f, -0.278326f, 
		0.00754964f, 1.07402f, -0.278326f, 0.00754964f, 1.07402f, -0.278326f, 
		0.00754964f, 1.07402f, -0.278326f, 0.00754964f, 1.07402f, -0.278326f, 
		0.00754964f, 1.07402f, -0.278326f, 0.00757526f, 1.24634f, -0.0656158f, 
		0.00757526f, 1.24634f, -0.0656158f, 0.00757526f, 1.24634f, -0.0656158f, 
		0.00757526f, 1.24634f, -0.0656158f, 0.00757640f, 1.53558f, -0.00789636f, 
		0.00757640f, 1.53558f, -0.00789636f, 0.00757640f, 1.53558f, -0.00789636f, 
		0.00757640f, 1.53558f, -0.00789636f, 0.00757640f, 1.53558f, -0.00789636f, 
		0.00757640f, 1.53558f, -0.00789636f, 0.00757640f, 1.53558f, -0.00789636f, 
		0.00757800f, 1.03305f, 0.369997f, 0.00757800f, 1.03305f, 0.369997f, 
		0.00757800f, 1.05903f, 0.369997f, 0.00757800f, 1.05903f, 0.369997f, 
		0.00767422f, 1.17716f, -0.00746630f, 0.00767422f, 1.17716f, -0.00746630f, 
		0.00767422f, 1.17716f, -0.00746630f, 0.00767422f, 1.17716f, -0.00746630f, 
		0.00767422f, 1.17716f, -0.00746630f, 0.00771125f, 0.956459f, -0.254631f, 
		0.00771125f, 0.956459f, -0.254631f, 0.00775519f, 1.53026f, 0.00743022f, 
		0.00775519f, 1.53026f, 0.00743022f, 0.00775519f, 1.53026f, 0.00743022f, 
		0.00775519f, 1.53026f, 0.00743022f, 0.00775519f, 1.53026f, 0.00743022f, 
		0.00775522f, 1.33745f, 0.00743013f, 0.00775522f, 1.33745f, 0.00743013f, 
		0.00775522f, 1.33745f, 0.00743013f, 0.00775522f, 1.33745f, 0.00743013f, 
		0.00775522f, 1.33745f, 0.00743013f, 0.00775528f, 1.17825f, 0.00743006f, 
		0.00775528f, 1.17825f, 0.00743006f, 0.00775528f, 1.17825f, 0.00743006f, 
		0.00775528f, 1.17825f, 0.00743006f, 0.00775528f, 1.17825f, 0.00743006f, 
		0.00776129f, 1.26712f, -0.0496151f, 0.00776129f, 1.26712f, -0.0496151f, 
		0.00776279f, 1.07153f, 0.265234f, 0.00776279f, 1.07153f, 0.265234f, 
		0.00776279f, 1.07153f, 0.265234f, 0.00776279f, 1.07153f, 0.265234f, 
		0.00776279f, 1.07153f, 0.265234f, 0.00776279f, 1.07153f, 0.265234f, 
		0.00776279f, 1.07153f, 0.265234f, 0.00776279f, 1.07153f, 0.265234f, 
		0.00778246f, 1.26712f, 0.0496080f, 0.00778246f, 1.26712f, 0.0496080f, 
		0.00787564f, 1.55685f, 0.0539750f, 0.00788666f, 1.07853f, -0.412722f, 
		0.00788666f, 1.07853f, -0.412722f, 0.00788666f, 1.07853f, -0.412722f, 
		0.00788666f, 1.07853f, -0.412722f, 0.00788666f, 1.07853f, -0.412722f, 
		0.00792052f, 1.03260f, 0.0306482f, 0.00792052f, 1.03260f, 0.0306482f, 
		0.00792052f, 1.03260f, 0.0306482f, 0.00792052f, 1.03260f, 0.0306482f, 
		0.00792052f, 1.03260f, 0.0306482f, 0.00799500f, 0.829148f, 0.272559f, 
		0.00799500f, 0.829148f, 0.272559f, 0.00801114f, 1.05259f, -0.423126f, 
		0.00801114f, 1.05259f, -0.423126f, 0.00801114f, 1.05259f, -0.423126f, 
		0.00801114f, 1.05259f, -0.423126f, 0.00801114f, 1.05259f, -0.423126f, 
		0.00801114f, 1.05259f, -0.423126f, 0.00802222f, 1.08188f, -0.421318f, 
		0.00802222f, 1.08188f, -0.421318f, 0.00802222f, 1.08188f, -0.421318f, 
		0.00802222f, 1.08188f, -0.421318f, 0.00802222f, 1.08188f, -0.421318f, 
		0.00803180f, 0.985827f, -0.304363f, 0.00806191f, 1.07853f, 0.412617f, 
		0.00806191f, 1.07853f, 0.412617f, 0.00806191f, 1.07853f, 0.412617f, 
		0.00806191f, 1.07853f, 0.412617f, 0.00806191f, 1.07853f, 0.412617f, 
		0.00808926f, 1.08714f, -0.403704f, 0.00808926f, 1.08714f, -0.403704f, 
		0.00810926f, 1.06115f, -0.394643f, 0.00810926f, 1.06115f, -0.394643f, 
		0.00810926f, 1.06115f, -0.394643f, 0.00810926f, 1.06115f, -0.394643f, 
		0.00810926f, 1.06115f, -0.394643f, 0.00813387f, 1.03272f, -0.386282f, 
		0.00813387f, 1.03272f, -0.386282f, 0.00813387f, 1.03272f, -0.386282f, 
		0.00813387f, 1.03272f, -0.386282f, 0.00813387f, 1.03272f, -0.386282f, 
		0.00813387f, 1.03272f, -0.386282f, 0.00813428f, 1.06119f, -0.386282f, 
		0.00813428f, 1.06119f, -0.386282f, 0.00813428f, 1.06119f, -0.386282f, 
		0.00813428f, 1.06119f, -0.386282f, 0.00816054f, 1.31634f, -0.0749015f, 
		0.00818600f, 1.23705f, -0.0543890f, 0.00818600f, 1.23705f, -0.0543890f, 
		0.00818600f, 1.26705f, -0.0543890f, 0.00818600f, 1.26705f, -0.0543890f, 
		0.00819019f, 1.33668f, -0.0141946f, 0.00819019f, 1.33668f, -0.0141946f, 
		0.00819019f, 1.33668f, -0.0141946f, 0.00819019f, 1.33668f, -0.0141946f, 
		0.00819019f, 1.33668f, -0.0141946f, 0.00819019f, 1.33668f, -0.0141946f, 
		0.00819096f, 1.31648f, 0.0747984f, 0.00819096f, 1.31648f, 0.0747984f, 
		0.00819637f, 1.33668f, 0.0141874f, 0.00819637f, 1.33668f, 0.0141874f, 
		0.00819637f, 1.33668f, 0.0141874f, 0.00819637f, 1.33668f, 0.0141874f, 
		0.00819637f, 1.33668f, 0.0141874f, 0.00819637f, 1.33668f, 0.0141874f, 
		0.00819637f, 1.33668f, 0.0141874f, 0.00819897f, 1.08188f, 0.421214f, 
		0.00819897f, 1.08188f, 0.421214f, 0.00819897f, 1.08188f, 0.421214f, 
		0.00819897f, 1.08188f, 0.421214f, 0.00819897f, 1.08188f, 0.421214f, 
		0.00819897f, 1.08188f, 0.421214f, 0.00820327f, 0.980923f, -0.238838f, 
		0.00820900f, 1.23705f, 0.0543820f, 0.00820900f, 1.23705f, 0.0543820f, 
		0.00820900f, 1.26705f, 0.0543820f, 0.00820900f, 1.26705f, 0.0543820f, 
		0.00821310f, 1.32831f, -0.0142345f, 0.00821310f, 1.32831f, -0.0142345f, 
		0.00821310f, 1.32831f, -0.0142345f, 0.00821310f, 1.32831f, -0.0142345f, 
		0.00821310f, 1.32831f, -0.0142345f, 0.00821906f, 1.32831f, 0.0142275f, 
		0.00821906f, 1.32831f, 0.0142275f, 0.00821906f, 1.32831f, 0.0142275f, 
		0.00821906f, 1.32831f, 0.0142275f, 0.00827555f, 1.06115f, 0.394547f, 
		0.00827555f, 1.06115f, 0.394547f, 0.00827555f, 1.06115f, 0.394547f, 
		0.00827555f, 1.06115f, 0.394547f, 0.00827555f, 1.06115f, 0.394547f, 
		0.00827555f, 1.06115f, 0.394547f, 0.00827646f, 1.03276f, 0.394547f, 
		0.00827646f, 1.03276f, 0.394547f, 0.00827646f, 1.03276f, 0.394547f, 
		0.00827646f, 1.03276f, 0.394547f, 0.00827646f, 1.03276f, 0.394547f, 
		0.00827646f, 1.03276f, 0.394547f, 0.00829772f, 1.03272f, 0.386187f, 
		0.00829772f, 1.03272f, 0.386187f, 0.00829772f, 1.03272f, 0.386187f, 
		0.00829772f, 1.03272f, 0.386187f, 0.00829785f, 1.06119f, 0.386187f, 
		0.00829785f, 1.06119f, 0.386187f, 0.00829785f, 1.06119f, 0.386187f, 
		0.00829785f, 1.06119f, 0.386187f, 0.00830311f, 1.05336f, 0.416552f, 
		0.00830311f, 1.05336f, 0.416552f, 0.00830311f, 1.05336f, 0.416552f, 
		0.00830311f, 1.05336f, 0.416552f, 0.00830311f, 1.05336f, 0.416552f, 
		0.00833183f, 1.25674f, -0.0599625f, 0.00833183f, 1.25674f, -0.0599625f, 
		0.00833183f, 1.25674f, -0.0599625f, 0.00833183f, 1.25674f, -0.0599625f, 
		0.00833183f, 1.25674f, -0.0599625f, 0.00833183f, 1.25674f, -0.0599625f, 
		0.00833183f, 1.25674f, -0.0599625f, 0.00833183f, 1.25674f, -0.0599625f, 
		0.00833183f, 1.25674f, -0.0599625f, 0.00833183f, 1.25674f, -0.0599625f, 
		0.00839139f, 1.08650f, 0.403628f, 0.00839139f, 1.08650f, 0.403628f, 
		0.00840021f, 1.05543f, -0.377043f, 0.00840021f, 1.05543f, -0.377043f, 
		0.00840021f, 1.05543f, -0.377043f, 0.00840021f, 1.05543f, -0.377043f, 
		0.00840021f, 1.05543f, -0.377043f, 0.00840021f, 1.05543f, -0.377043f, 
		0.00841765f, 1.04848f, -0.406274f, 0.00841765f, 1.04848f, -0.406274f, 
		0.00841765f, 1.04848f, -0.406274f, 0.00841765f, 1.04848f, -0.406274f, 
		0.00841765f, 1.04848f, -0.406274f, 0.00841765f, 1.04848f, -0.406274f, 
		0.00850064f, 1.19320f, -0.00228997f, 0.00850064f, 1.19320f, -0.00228997f, 
		0.00850064f, 1.19320f, -0.00228997f, 0.00850064f, 1.19320f, -0.00228997f, 
		0.00850064f, 1.19320f, -0.00228997f, 0.00850064f, 1.19320f, -0.00228997f, 
		0.00850064f, 1.19320f, -0.00228997f, 0.00850064f, 1.19320f, -0.00228997f, 
		0.00850189f, 1.35243f, -0.00229043f, 0.00850189f, 1.35243f, -0.00229043f, 
		0.00850189f, 1.35243f, -0.00229043f, 0.00850189f, 1.35243f, -0.00229043f, 
		0.00850189f, 1.35243f, -0.00229043f, 0.00850189f, 1.35243f, -0.00229043f, 
		0.00850189f, 1.35243f, -0.00229043f, 0.00850189f, 1.35243f, -0.00229043f, 
		0.00850192f, 1.54524f, -0.00229048f, 0.00850192f, 1.54524f, -0.00229048f, 
		0.00850192f, 1.54524f, -0.00229048f, 0.00850192f, 1.54524f, -0.00229048f, 
		0.00850192f, 1.54524f, -0.00229048f, 0.00850192f, 1.54524f, -0.00229048f, 
		0.00850192f, 1.54524f, -0.00229048f, 0.00850192f, 1.54524f, -0.00229048f, 
		0.00855935f, 1.05543f, 0.376947f, 0.00855935f, 1.05543f, 0.376947f, 
		0.00855935f, 1.05543f, 0.376947f, 0.00855935f, 1.05543f, 0.376947f, 
		0.00855935f, 1.05543f, 0.376947f, 0.00855935f, 1.05543f, 0.376947f, 
		0.00855935f, 1.05543f, 0.376947f, 0.00855935f, 1.05543f, 0.376947f, 
		0.00862943f, 1.08963f, -0.419399f, 0.00862943f, 1.08963f, -0.419399f, 
		0.00862943f, 1.08963f, -0.419399f, 0.00865722f, 0.955074f, 0.282867f, 
		0.00865722f, 0.955074f, 0.282867f, 0.00867351f, 1.04439f, 0.406176f, 
		0.00867351f, 1.04439f, 0.406176f, 0.00867351f, 1.04439f, 0.406176f, 
		0.00867351f, 1.04439f, 0.406176f, 0.00873100f, 0.832160f, -0.283308f, 
		0.00873100f, 0.832160f, -0.283308f, 0.00873476f, 1.34207f, -0.00754798f, 
		0.00873476f, 1.34207f, -0.00754798f, 0.00873476f, 1.34207f, -0.00754798f, 
		0.00873476f, 1.34207f, -0.00754798f, 0.00873476f, 1.34207f, -0.00754798f, 
		0.00873476f, 1.34207f, -0.00754798f, 0.00873800f, 0.832160f, -0.252855f, 
		0.00873800f, 0.832160f, -0.252855f, 0.00876747f, 1.05580f, -0.385788f, 
		0.00876747f, 1.05580f, -0.385788f, 0.00876747f, 1.05580f, -0.385788f, 
		0.00876747f, 1.05580f, -0.385788f, 0.00876747f, 1.05580f, -0.385788f, 
		0.00876747f, 1.05580f, -0.385788f, 0.00876747f, 1.05580f, -0.385788f, 
		0.00876747f, 1.05580f, -0.385788f, 0.00876747f, 1.05580f, -0.385788f, 
		0.00876747f, 1.05580f, -0.385788f, 0.00876747f, 1.05580f, -0.385788f, 
		0.00876747f, 1.05580f, -0.385788f, 0.00876759f, 1.03811f, -0.385788f, 
		0.00876759f, 1.03811f, -0.385788f, 0.00876759f, 1.03811f, -0.385788f, 
		0.00876759f, 1.03811f, -0.385788f, 0.00876759f, 1.03811f, -0.385788f, 
		0.00876759f, 1.03811f, -0.385788f, 0.00876759f, 1.03811f, -0.385788f, 
		0.00876759f, 1.03811f, -0.385788f, 0.00883339f, 1.06086f, 0.273021f, 
		0.00883339f, 1.06086f, 0.273021f, 0.00883339f, 1.06086f, 0.273021f, 
		0.00883339f, 1.06086f, 0.273021f, 0.00883339f, 1.06086f, 0.273021f, 
		0.00883339f, 1.06086f, 0.273021f, 0.00884500f, 0.832160f, 0.252749f, 
		0.00884500f, 0.832160f, 0.252749f, 0.00884688f, 1.03149f, -0.393443f, 
		0.00884688f, 1.03149f, -0.393443f, 0.00884688f, 1.03149f, -0.393443f, 
		0.00884688f, 1.03149f, -0.393443f, 0.00884688f, 1.03149f, -0.393443f, 
		0.00884688f, 1.03149f, -0.393443f, 0.00885100f, 0.832160f, 0.283203f, 
		0.00885100f, 0.832160f, 0.283203f, 0.00885727f, 0.984647f, 0.302340f, 
		0.00885727f, 0.984647f, 0.302340f, 0.00885727f, 0.984647f, 0.302340f, 
		0.00885727f, 0.984647f, 0.302340f, 0.00885727f, 0.984647f, 0.302340f, 
		0.00886300f, 0.965681f, -0.210896f, 0.00888390f, 1.06609f, 0.268342f, 
		0.00888390f, 1.06609f, 0.268342f, 0.00888390f, 1.06609f, 0.268342f, 
		0.00888390f, 1.06609f, 0.268342f, 0.00888390f, 1.06609f, 0.268342f, 
		0.00888390f, 1.06609f, 0.268342f, 0.00893082f, 1.03811f, 0.385692f, 
		0.00893082f, 1.03811f, 0.385692f, 0.00893082f, 1.03811f, 0.385692f, 
		0.00893082f, 1.03811f, 0.385692f, 0.00893082f, 1.03811f, 0.385692f, 
		0.00893082f, 1.03811f, 0.385692f, 0.00893082f, 1.03811f, 0.385692f, 
		0.00893082f, 1.03811f, 0.385692f, 0.00893854f, 1.06469f, 0.0151460f, 
		0.00893854f, 1.06469f, 0.0151460f, 0.00893854f, 1.06469f, 0.0151460f, 
		0.00893854f, 1.06469f, 0.0151460f, 0.00893854f, 1.06469f, 0.0151460f, 
		0.00895200f, 0.965681f, 0.210790f, 0.00895989f, 1.20211f, -0.0226039f, 
		0.00898986f, 1.53675f, -0.00221570f, 0.00898986f, 1.53675f, -0.00221570f, 
		0.00898986f, 1.53675f, -0.00221570f, 0.00898986f, 1.53675f, -0.00221570f, 
		0.00899004f, 1.34394f, -0.00221603f, 0.00899004f, 1.34394f, -0.00221603f, 
		0.00899004f, 1.34394f, -0.00221603f, 0.00899004f, 1.34394f, -0.00221603f, 
		0.00899007f, 1.18474f, -0.00221606f, 0.00899007f, 1.18474f, -0.00221606f, 
		0.00899007f, 1.18474f, -0.00221606f, 0.00899007f, 1.18474f, -0.00221606f, 
		0.00900402f, 1.03132f, -0.303173f, 0.00900402f, 1.03132f, -0.303173f, 
		0.00900402f, 1.03132f, -0.303173f, 0.00900402f, 1.03132f, -0.303173f, 
		0.00902685f, 1.04083f, -0.421843f, 0.00902685f, 1.04083f, -0.421843f, 
		0.00902685f, 1.04083f, -0.421843f, 0.00902685f, 1.04083f, -0.421843f, 
		0.00902685f, 1.04083f, -0.421843f, 0.00905289f, 1.04296f, -0.415011f, 
		0.00905289f, 1.04296f, -0.415011f, 0.00905289f, 1.04296f, -0.415011f, 
		0.00905289f, 1.04296f, -0.415011f, 0.00905289f, 1.04296f, -0.415011f, 
		0.00906661f, 1.25553f, -0.0701612f, 0.00906661f, 1.25553f, -0.0701612f, 
		0.00906661f, 1.25553f, -0.0701612f, 0.00906661f, 1.25553f, -0.0701612f, 
		0.00906661f, 1.25553f, -0.0701612f, 0.00906661f, 1.25553f, -0.0701612f, 
		0.00906661f, 1.25553f, -0.0701612f, 0.00906661f, 1.25553f, -0.0701612f, 
		0.00906661f, 1.25553f, -0.0701612f, 0.00906661f, 1.25553f, -0.0701612f, 
		0.00906661f, 1.25553f, -0.0701612f, 0.00906661f, 1.25553f, -0.0701612f, 
		0.00907086f, 1.32801f, -0.00826328f, 0.00907086f, 1.32801f, -0.00826328f, 
		0.00907086f, 1.32801f, -0.00826328f, 0.00907086f, 1.32801f, -0.00826328f, 
		0.00907086f, 1.32801f, -0.00826328f, 0.00907086f, 1.32801f, -0.00826328f, 
		0.00907086f, 1.32801f, -0.00826328f, 0.00907086f, 1.32801f, -0.00826328f, 
		0.00907086f, 1.32801f, -0.00826328f, 0.00911216f, 1.52833f, -0.00329066f, 
		0.00911216f, 1.52833f, -0.00329066f, 0.00911216f, 1.52833f, -0.00329066f, 
		0.00911216f, 1.52833f, -0.00329066f, 0.00911500f, 0.977001f, -0.299748f, 
		0.00912700f, 0.969213f, -0.209143f, 0.00914162f, 1.06293f, -0.376988f, 
		0.00914162f, 1.06293f, -0.376988f, 0.00914163f, 1.03098f, -0.376988f, 
		0.00914163f, 1.03098f, -0.376988f, 0.00915470f, 1.05521f, 0.385880f, 
		0.00915470f, 1.05521f, 0.385880f, 0.00915470f, 1.05521f, 0.385880f, 
		0.00915470f, 1.05521f, 0.385880f, 0.00915470f, 1.05521f, 0.385880f, 
		0.00915470f, 1.05521f, 0.385880f, 0.00915470f, 1.05521f, 0.385880f, 
		0.00915470f, 1.05521f, 0.385880f, 0.00915470f, 1.05521f, 0.385880f, 
		0.00916053f, 1.32653f, 0.0821220f, 0.00916053f, 1.32653f, 0.0821220f, 
		0.00916053f, 1.32653f, 0.0821220f, 0.00916053f, 1.32653f, 0.0821220f, 
		0.00916053f, 1.32653f, 0.0821220f, 0.00916053f, 1.32653f, 0.0821220f, 
		0.00916603f, 1.30072f, 0.0284521f, 0.00918580f, 1.29913f, -0.0318235f, 
		0.00919423f, 1.32549f, 0.0743851f, 0.00919423f, 1.32549f, 0.0743851f, 
		0.00920114f, 1.03087f, -0.388362f, 0.00920114f, 1.03087f, -0.388362f, 
		0.00920114f, 1.03087f, -0.388362f, 0.00920114f, 1.03087f, -0.388362f, 
		0.00920114f, 1.03087f, -0.388362f, 0.00920114f, 1.03087f, -0.388362f, 
		0.00920128f, 1.06303f, -0.388362f, 0.00920128f, 1.06303f, -0.388362f, 
		0.00920128f, 1.06303f, -0.388362f, 0.00920128f, 1.06303f, -0.388362f, 
		0.00920128f, 1.06303f, -0.388362f, 0.00920128f, 1.06303f, -0.388362f, 
		0.00920128f, 1.06303f, -0.388362f, 0.00920128f, 1.06303f, -0.388362f, 
		0.00920296f, 0.953366f, -0.284126f, 0.00920296f, 0.953366f, -0.284126f, 
		0.00920496f, 1.53687f, 0.00225435f, 0.00920496f, 1.53687f, 0.00225435f, 
		0.00920496f, 1.53687f, 0.00225435f, 0.00920496f, 1.53687f, 0.00225435f, 
		0.00920627f, 1.34406f, 0.00225446f, 0.00920627f, 1.34406f, 0.00225446f, 
		0.00920627f, 1.34406f, 0.00225446f, 0.00920627f, 1.34406f, 0.00225446f, 
		0.00920955f, 0.953366f, -0.252037f, 0.00920955f, 0.953366f, -0.252037f, 
		0.00921292f, 1.30330f, -0.0159666f, 0.00921292f, 1.30330f, -0.0159666f, 
		0.00921292f, 1.30330f, -0.0159666f, 0.00921292f, 1.30330f, -0.0159666f, 
		0.00921600f, 0.969213f, 0.209037f, 0.00921785f, 1.31902f, -0.0159752f, 
		0.00921785f, 1.31902f, -0.0159752f, 0.00922471f, 1.31902f, 0.0159677f, 
		0.00922471f, 1.31902f, 0.0159677f, 0.00923435f, 1.32549f, -0.0794281f, 
		0.00923435f, 1.32549f, -0.0794281f, 0.00923435f, 1.32549f, -0.0794281f, 
		0.00924100f, 0.977001f, 0.299642f, 0.00927470f, 1.07651f, -0.272001f, 
		0.00927470f, 1.07651f, -0.272001f, 0.00927470f, 1.07651f, -0.272001f, 
		0.00927470f, 1.07651f, -0.272001f, 0.00928050f, 1.33444f, -0.0160832f, 
		0.00928050f, 1.33444f, -0.0160832f, 0.00928050f, 1.33444f, -0.0160832f, 
		0.00928050f, 1.33444f, -0.0160832f, 0.00928050f, 1.33444f, -0.0160832f, 
		0.00928050f, 1.33444f, -0.0160832f, 0.00928050f, 1.33444f, -0.0160832f, 
		0.00928050f, 1.33444f, -0.0160832f, 0.00928050f, 1.33444f, -0.0160832f, 
		0.00928710f, 1.33039f, 0.0160753f, 0.00928710f, 1.33039f, 0.0160753f, 
		0.00928710f, 1.33039f, 0.0160753f, 0.00928710f, 1.33039f, 0.0160753f, 
		0.00928710f, 1.33039f, 0.0160753f, 0.00928710f, 1.33039f, 0.0160753f, 
		0.00928710f, 1.33039f, 0.0160753f, 0.00928710f, 1.33039f, 0.0160753f, 
		0.00929303f, 1.06292f, 0.361176f, 0.00929303f, 1.06292f, 0.361176f, 
		0.00929303f, 1.06292f, 0.361176f, 0.00930089f, 1.06293f, 0.376892f, 
		0.00930089f, 1.06293f, 0.376892f, 0.00930092f, 1.03098f, 0.376892f, 
		0.00930092f, 1.03098f, 0.376892f, 0.00931644f, 0.953366f, 0.251931f, 
		0.00931644f, 0.953366f, 0.251931f, 0.00932051f, 0.957955f, 0.277237f, 
		0.00932051f, 0.957955f, 0.277237f, 0.00932051f, 0.957955f, 0.277237f, 
		0.00932381f, 1.06325f, -0.391602f, 0.00932381f, 1.06325f, -0.391602f, 
		0.00932381f, 1.06325f, -0.391602f, 0.00932381f, 1.06325f, -0.391602f, 
		0.00932381f, 1.06325f, -0.391602f, 0.00932381f, 1.06325f, -0.391602f, 
		0.00932381f, 1.06325f, -0.391602f, 0.00932381f, 1.06325f, -0.391602f, 
		0.00932381f, 1.06325f, -0.391602f, 0.00932381f, 1.06325f, -0.391602f, 
		0.00932381f, 1.06325f, -0.391602f, 0.00936568f, 1.03087f, 0.388265f, 
		0.00936568f, 1.03087f, 0.388265f, 0.00936568f, 1.03087f, 0.388265f, 
		0.00936568f, 1.03087f, 0.388265f, 0.00936568f, 1.03087f, 0.388265f, 
		0.00936568f, 1.03087f, 0.388265f, 0.00936568f, 1.03087f, 0.388265f, 
		0.00936568f, 1.03087f, 0.388265f, 0.00936581f, 1.06303f, 0.388265f, 
		0.00936581f, 1.06303f, 0.388265f, 0.00936581f, 1.06303f, 0.388265f, 
		0.00936581f, 1.06303f, 0.388265f, 0.00936581f, 1.06303f, 0.388265f, 
		0.00936581f, 1.06303f, 0.388265f, 0.00936581f, 1.06303f, 0.388265f, 
		0.00936581f, 1.06303f, 0.388265f, 0.00936848f, 1.07613f, 0.267124f, 
		0.00936848f, 1.07613f, 0.267124f, 0.00936848f, 1.07613f, 0.267124f, 
		0.00936848f, 1.07613f, 0.267124f, 0.00936848f, 1.07613f, 0.267124f, 
		0.00936848f, 1.07613f, 0.267124f, 0.00936848f, 1.07613f, 0.267124f, 
		0.00936848f, 1.07613f, 0.267124f, 0.00939134f, 1.25205f, 0.0653047f, 
		0.00939134f, 1.25205f, 0.0653047f, 0.00939134f, 1.25205f, 0.0653047f, 
		0.00939134f, 1.25205f, 0.0653047f, 0.00939684f, 1.03054f, -0.390797f, 
		0.00939684f, 1.03054f, -0.390797f, 0.00940343f, 1.33120f, -0.0162970f, 
		0.00940343f, 1.33120f, -0.0162970f, 0.00940343f, 1.33120f, -0.0162970f, 
		0.00940343f, 1.33120f, -0.0162970f, 0.00940343f, 1.33120f, -0.0162970f, 
		0.00940343f, 1.33120f, -0.0162970f, 0.00940343f, 1.33120f, -0.0162970f, 
		0.00940343f, 1.33120f, -0.0162970f, 0.00940343f, 1.33120f, -0.0162970f, 
		0.00940343f, 1.33120f, -0.0162970f, 0.00940343f, 1.33120f, -0.0162970f, 
		0.00941048f, 1.33363f, 0.0162889f, 0.00941048f, 1.33363f, 0.0162889f, 
		0.00941048f, 1.33363f, 0.0162889f, 0.00941048f, 1.33363f, 0.0162889f, 
		0.00941048f, 1.33363f, 0.0162889f, 0.00941048f, 1.33363f, 0.0162889f, 
		0.00941048f, 1.33363f, 0.0162889f, 0.00941048f, 1.33363f, 0.0162889f, 
		0.00941048f, 1.33363f, 0.0162889f, 0.00941048f, 1.33363f, 0.0162889f, 
		0.00941048f, 1.33363f, 0.0162889f, 0.00943574f, 1.06239f, -0.0298657f, 
		0.00943574f, 1.06239f, -0.0298657f, 0.00943598f, 1.02968f, -0.0298657f, 
		0.00943598f, 1.02968f, -0.0298657f, 0.00944897f, 1.06239f, 0.0298576f, 
		0.00944897f, 1.06239f, 0.0298576f, 0.00946043f, 1.25205f, -0.0546708f, 
		0.00946043f, 1.25205f, -0.0546708f, 0.00946043f, 1.25205f, -0.0546708f, 
		0.00946043f, 1.25205f, -0.0546708f, 0.00946043f, 1.25205f, -0.0546708f, 
		0.00946043f, 1.25205f, -0.0546708f, 0.00946043f, 1.25205f, -0.0546708f, 
		0.00946043f, 1.25205f, -0.0546708f, 0.00946043f, 1.25205f, -0.0546708f, 
		0.00946043f, 1.25205f, -0.0546708f, 0.00947647f, 1.06083f, -0.270147f, 
		0.00947647f, 1.06083f, -0.270147f, 0.00947647f, 1.06083f, -0.270147f, 
		0.00947647f, 1.06083f, -0.270147f, 0.00947647f, 1.06083f, -0.270147f, 
		0.00947647f, 1.06083f, -0.270147f, 0.00947647f, 1.06083f, -0.270147f, 
		0.00947647f, 1.06083f, -0.270147f, 0.00947647f, 1.06083f, -0.270147f, 
		0.00947647f, 1.06083f, -0.270147f, 0.00947647f, 1.06083f, -0.270147f, 
		0.00947647f, 1.06083f, -0.270147f, 0.00948879f, 1.18447f, -0.00323877f, 
		0.00948879f, 1.18447f, -0.00323877f, 0.00948879f, 1.18447f, -0.00323877f, 
		0.00948879f, 1.18447f, -0.00323877f, 0.00948879f, 1.18447f, -0.00323877f, 
		0.00948944f, 1.06325f, 0.391506f, 0.00948944f, 1.06325f, 0.391506f, 
		0.00948944f, 1.06325f, 0.391506f, 0.00948944f, 1.06325f, 0.391506f, 
		0.00948944f, 1.06325f, 0.391506f, 0.00948944f, 1.06325f, 0.391506f, 
		0.00948944f, 1.06325f, 0.391506f, 0.00948944f, 1.06325f, 0.391506f, 
		0.00948944f, 1.06325f, 0.391506f, 0.00948944f, 1.06325f, 0.391506f, 
		0.00948944f, 1.06325f, 0.391506f, 0.00948994f, 1.03066f, 0.391506f, 
		0.00948994f, 1.03066f, 0.391506f, 0.00948994f, 1.03066f, 0.391506f, 
		0.00948994f, 1.03066f, 0.391506f, 0.00948994f, 1.03066f, 0.391506f, 
		0.00948994f, 1.03066f, 0.391506f, 0.00948994f, 1.03066f, 0.391506f, 
		0.00948994f, 1.03066f, 0.391506f, 0.00948994f, 1.03066f, 0.391506f, 
		0.00948994f, 1.03066f, 0.391506f, 0.00948994f, 1.03066f, 0.391506f, 
		0.00953007f, 1.05184f, -0.420511f, 0.00953007f, 1.05184f, -0.420511f, 
		0.00953007f, 1.05184f, -0.420511f, 0.00954958f, 1.25167f, 0.0545632f, 
		0.00954958f, 1.25167f, 0.0545632f, 0.00954958f, 1.25167f, 0.0545632f, 
		0.00954958f, 1.25167f, 0.0545632f, 0.00954958f, 1.25167f, 0.0545632f, 
		0.00954958f, 1.25167f, 0.0545632f, 0.00954958f, 1.25167f, 0.0545632f, 
		0.00963158f, 0.957053f, 0.256347f, 0.00963158f, 0.957053f, 0.256347f, 
		0.00969327f, 1.04057f, 0.417390f, 0.00969327f, 1.04057f, 0.417390f, 
		0.00969327f, 1.04057f, 0.417390f, 0.00971061f, 1.04788f, 0.414446f, 
		0.00971061f, 1.04788f, 0.414446f, 0.00971061f, 1.04788f, 0.414446f, 
		0.00971061f, 1.04788f, 0.414446f, 0.00974422f, 1.04314f, -0.415422f, 
		0.00974422f, 1.04314f, -0.415422f, 0.00974422f, 1.04314f, -0.415422f, 
		0.00974422f, 1.04314f, -0.415422f, 0.00976900f, 1.03132f, -0.238940f, 
		0.00976900f, 1.03132f, -0.238940f, 0.00976900f, 1.03132f, -0.238940f, 
		0.00979559f, 0.951393f, 0.284837f, 0.00979559f, 0.951393f, 0.284837f, 
		0.00981393f, 1.33284f, -0.0798298f, 0.00981393f, 1.33284f, -0.0798298f, 
		0.00981393f, 1.33284f, -0.0798298f, 0.00981393f, 1.33284f, -0.0798298f, 
		0.00985295f, 1.25365f, 0.0698440f, 0.00985295f, 1.25365f, 0.0698440f, 
		0.00985295f, 1.25365f, 0.0698440f, 0.00985295f, 1.25365f, 0.0698440f, 
		0.00985295f, 1.25365f, 0.0698440f, 0.00985295f, 1.25365f, 0.0698440f, 
		0.00985295f, 1.25365f, 0.0698440f, 0.00985295f, 1.25365f, 0.0698440f, 
		0.00989613f, 1.33284f, 0.0744417f, 0.00989613f, 1.33284f, 0.0744417f, 
		0.00989613f, 1.33284f, 0.0744417f, 0.00989613f, 1.33284f, 0.0744417f, 
		0.00989613f, 1.33284f, 0.0744417f, 0.00989613f, 1.33284f, 0.0744417f, 
		0.00992616f, 1.02256f, -0.289897f, 0.00992616f, 1.02256f, -0.289897f, 
		0.00992616f, 1.02256f, -0.289897f, 0.00992616f, 1.02256f, -0.289897f, 
		0.00993373f, 0.949218f, -0.285391f, 0.00993373f, 0.949218f, -0.285391f, 
		0.00994000f, 0.912612f, -0.285402f, 0.00994000f, 0.912612f, -0.285402f, 
		0.00994000f, 0.912612f, -0.285402f, 0.00994076f, 0.949218f, -0.250772f, 
		0.00994076f, 0.949218f, -0.250772f, 0.00994700f, 0.912612f, -0.250761f, 
		0.00994700f, 0.912612f, -0.250761f, 0.00994700f, 0.912612f, -0.250761f, 
		0.00997243f, 0.984594f, 0.240899f, 0.00997243f, 0.984594f, 0.240899f, 
		0.0100294f, 1.03684f, -0.383101f, 0.0100294f, 1.03684f, -0.383101f, 
		0.0100294f, 1.03684f, -0.383101f, 0.0100294f, 1.03684f, -0.383101f, 
		0.0100294f, 1.03684f, -0.383101f, 0.0100294f, 1.03684f, -0.383101f, 
		0.0100294f, 1.03684f, -0.383101f, 0.0100294f, 1.03684f, -0.383101f, 
		0.0100294f, 1.03684f, -0.383101f, 0.0100294f, 1.03684f, -0.383101f, 
		0.0100294f, 1.03684f, -0.383101f, 0.0100463f, 1.07304f, -0.267934f, 
		0.0100463f, 1.07304f, -0.267934f, 0.0100463f, 1.07304f, -0.267934f, 
		0.0100463f, 1.07304f, -0.267934f, 0.0100463f, 1.07304f, -0.267934f, 
		0.0100463f, 1.07304f, -0.267934f, 0.0100463f, 1.07304f, -0.267934f, 
		0.0100463f, 1.07304f, -0.267934f, 0.0100463f, 1.07304f, -0.267934f, 
		0.0100463f, 1.07304f, -0.267934f, 0.0100468f, 0.949218f, 0.250666f, 
		0.0100468f, 0.949218f, 0.250666f, 0.0100575f, 1.02037f, 0.250216f, 
		0.0100575f, 1.02037f, 0.250216f, 0.0100575f, 1.02037f, 0.250216f, 
		0.0100584f, 0.911318f, 0.250646f, 0.0100584f, 0.911318f, 0.250646f, 
		0.0100584f, 0.911318f, 0.250646f, 0.0100600f, 0.948080f, 0.285296f, 
		0.0100600f, 0.948080f, 0.285296f, 0.0100654f, 0.911318f, 0.285305f, 
		0.0100654f, 0.911318f, 0.285305f, 0.0100654f, 0.911318f, 0.285305f, 
		0.0100968f, 1.06407f, -0.0152447f, 0.0100968f, 1.06407f, -0.0152447f, 
		0.0100968f, 1.06407f, -0.0152447f, 0.0101484f, 0.958026f, -0.260738f, 
		0.0101484f, 0.958026f, -0.260738f, 0.0101484f, 0.958026f, -0.260738f, 
		0.0101484f, 0.958026f, -0.260738f, 0.0101484f, 0.958026f, -0.260738f, 
		0.0101519f, 0.908777f, -0.285771f, 0.0101519f, 0.908777f, -0.285771f, 
		0.0101598f, 0.908777f, -0.250393f, 0.0101598f, 0.908777f, -0.250393f, 
		0.0101906f, 1.03684f, 0.383004f, 0.0101906f, 1.03684f, 0.383004f, 
		0.0101906f, 1.03684f, 0.383004f, 0.0101906f, 1.03684f, 0.383004f, 
		0.0101906f, 1.03684f, 0.383004f, 0.0101906f, 1.03684f, 0.383004f, 
		0.0101906f, 1.03684f, 0.383004f, 0.0101906f, 1.03684f, 0.383004f, 
		0.0101906f, 1.03684f, 0.383004f, 0.0101906f, 1.03684f, 0.383004f, 
		0.0101906f, 1.03684f, 0.383004f, 0.0102780f, 1.06309f, -0.268011f, 
		0.0102780f, 1.06309f, -0.268011f, 0.0102780f, 1.06309f, -0.268011f, 
		0.0102780f, 1.06309f, -0.268011f, 0.0102780f, 1.06309f, -0.268011f, 
		0.0102780f, 1.06309f, -0.268011f, 0.0102780f, 1.06309f, -0.268011f, 
		0.0102780f, 1.06309f, -0.268011f, 0.0103105f, 1.25482f, -0.0569555f, 
		0.0103105f, 1.25482f, -0.0569555f, 0.0103105f, 1.25482f, -0.0569555f, 
		0.0103105f, 1.25482f, -0.0569555f, 0.0103105f, 1.25482f, -0.0569555f, 
		0.0103105f, 1.25482f, -0.0569555f, 0.0103108f, 1.07351f, 0.273422f, 
		0.0103108f, 1.07351f, 0.273422f, 0.0103108f, 1.07351f, 0.273422f, 
		0.0103108f, 1.07351f, 0.273422f, 0.0103108f, 1.07351f, 0.273422f, 
		0.0103108f, 1.07351f, 0.273422f, 0.0103108f, 1.07351f, 0.273422f, 
		0.0103108f, 1.07351f, 0.273422f, 0.0103140f, 1.01838f, 0.291763f, 
		0.0103140f, 1.01838f, 0.291763f, 0.0103140f, 1.01838f, 0.291763f, 
		0.0103222f, 1.20497f, -0.0314734f, 0.0103354f, 1.20497f, 0.0314650f, 
		0.0103426f, 1.02264f, -0.251889f, 0.0103426f, 1.02264f, -0.251889f, 
		0.0103426f, 1.02264f, -0.251889f, 0.0103426f, 1.02264f, -0.251889f, 
		0.0103470f, 1.25481f, 0.0567977f, 0.0103470f, 1.25481f, 0.0567977f, 
		0.0103470f, 1.25481f, 0.0567977f, 0.0103470f, 1.25481f, 0.0567977f, 
		0.0103470f, 1.25481f, 0.0567977f, 0.0103470f, 1.25481f, 0.0567977f, 
		0.0103499f, 1.25206f, -0.0674497f, 0.0103499f, 1.25206f, -0.0674497f, 
		0.0103499f, 1.25206f, -0.0674497f, 0.0103499f, 1.25206f, -0.0674497f, 
		0.0103499f, 1.25206f, -0.0674497f, 0.0103499f, 1.25206f, -0.0674497f, 
		0.0103647f, 0.977046f, -0.240494f, 0.0103913f, 1.06309f, 0.267905f, 
		0.0103913f, 1.06309f, 0.267905f, 0.0103913f, 1.06309f, 0.267905f, 
		0.0103913f, 1.06309f, 0.267905f, 0.0103913f, 1.06309f, 0.267905f, 
		0.0103913f, 1.06309f, 0.267905f, 0.0104050f, 1.04394f, -0.421911f, 
		0.0104050f, 1.04394f, -0.421911f, 0.0104050f, 1.04394f, -0.421911f, 
		0.0104050f, 1.04394f, -0.421911f, 0.0104050f, 1.04394f, -0.421911f, 
		0.0104282f, 1.08473f, -0.414824f, 0.0104282f, 1.08473f, -0.414824f, 
		0.0104473f, 1.02795f, 0.0289022f, 0.0104473f, 1.02795f, 0.0289022f, 
		0.0105285f, 1.01259f, -0.247952f, 0.0105285f, 1.01259f, -0.247952f, 
		0.0105285f, 1.01259f, -0.247952f, 0.0106037f, 1.08473f, 0.414719f, 
		0.0106037f, 1.08473f, 0.414719f, 0.0106037f, 1.08473f, 0.414719f, 
		0.0106037f, 1.08473f, 0.414719f, 0.0106037f, 1.08473f, 0.414719f, 
		0.0106461f, 0.906317f, 0.249627f, 0.0106461f, 0.906317f, 0.249627f, 
		0.0106536f, 0.906317f, 0.286323f, 0.0106536f, 0.906317f, 0.286323f, 
		0.0106697f, 1.06557f, -0.375842f, 0.0106697f, 1.06557f, -0.375842f, 
		0.0106699f, 1.02834f, -0.375842f, 0.0106699f, 1.02834f, -0.375842f, 
		0.0106726f, 1.06557f, -0.362449f, 0.0106726f, 1.06557f, -0.362449f, 
		0.0106728f, 1.02834f, -0.362449f, 0.0106728f, 1.02834f, -0.362449f, 
		0.0106808f, 1.00509f, -0.249324f, 0.0106808f, 1.00509f, -0.249324f, 
		0.0106808f, 1.00509f, -0.249324f, 0.0106928f, 1.02272f, -0.302653f, 
		0.0106928f, 1.02272f, -0.302653f, 0.0106986f, 1.20347f, 0.0282858f, 
		0.0107455f, 1.30448f, -0.0186233f, 0.0107455f, 1.30448f, -0.0186233f, 
		0.0107456f, 1.31788f, -0.0186233f, 0.0107456f, 1.31788f, -0.0186233f, 
		0.0107539f, 1.31788f, 0.0186141f, 0.0107539f, 1.31788f, 0.0186141f, 
		0.0107540f, 1.30448f, 0.0186140f, 0.0107540f, 1.30448f, 0.0186140f, 
		0.0107637f, 1.02692f, -0.0153118f, 0.0107637f, 1.02692f, -0.0153118f, 
		0.0107953f, 1.53202f, -0.00292099f, 0.0107953f, 1.53202f, -0.00292099f, 
		0.0107953f, 1.53202f, -0.00292099f, 0.0107953f, 1.53202f, -0.00292099f, 
		0.0107953f, 1.53202f, -0.00292099f, 0.0107954f, 1.33921f, -0.00292094f, 
		0.0107954f, 1.33921f, -0.00292094f, 0.0107954f, 1.33921f, -0.00292094f, 
		0.0107954f, 1.33921f, -0.00292094f, 0.0107954f, 1.33921f, -0.00292094f, 
		0.0107954f, 1.18001f, -0.00292097f, 0.0107954f, 1.18001f, -0.00292097f, 
		0.0107954f, 1.18001f, -0.00292097f, 0.0107954f, 1.18001f, -0.00292097f, 
		0.0107954f, 1.18001f, -0.00292097f, 0.0108262f, 1.02834f, 0.362352f, 
		0.0108262f, 1.02834f, 0.362352f, 0.0108264f, 1.06557f, 0.362352f, 
		0.0108264f, 1.06557f, 0.362352f, 0.0108287f, 1.06557f, 0.375746f, 
		0.0108287f, 1.06557f, 0.375746f, 0.0108287f, 1.02834f, 0.375746f, 
		0.0108287f, 1.02834f, 0.375746f, 0.0108338f, 1.00796f, 0.293155f, 
		0.0108338f, 1.00796f, 0.293155f, 0.0108338f, 1.00796f, 0.293155f, 
		0.0110627f, 1.02262f, 0.288975f, 0.0110627f, 1.02262f, 0.288975f, 
		0.0110627f, 1.02262f, 0.288975f, 0.0110627f, 1.02262f, 0.288975f, 
		0.0110716f, 0.903956f, -0.287364f, 0.0110716f, 0.903956f, -0.287364f, 
		0.0110802f, 0.903956f, -0.248800f, 0.0110802f, 0.903956f, -0.248800f, 
		0.0111052f, 1.33919f, 0.00299631f, 0.0111052f, 1.33919f, 0.00299631f, 
		0.0111052f, 1.33919f, 0.00299631f, 0.0111052f, 1.33919f, 0.00299631f, 
		0.0111052f, 1.53200f, 0.00299627f, 0.0111052f, 1.53200f, 0.00299627f, 
		0.0111052f, 1.53200f, 0.00299627f, 0.0111053f, 1.17999f, 0.00299632f, 
		0.0111053f, 1.17999f, 0.00299632f, 0.0111053f, 1.17999f, 0.00299632f, 
		0.0111053f, 1.17999f, 0.00299632f, 0.0111520f, 1.04737f, 0.416137f, 
		0.0111520f, 1.04737f, 0.416137f, 0.0111520f, 1.04737f, 0.416137f, 
		0.0111520f, 1.04737f, 0.416137f, 0.0111568f, 1.06492f, 0.0153022f, 
		0.0111568f, 1.06492f, 0.0153022f, 0.0111950f, 1.04890f, -0.418512f, 
		0.0111950f, 1.04890f, -0.418512f, 0.0111950f, 1.04890f, -0.418512f, 
		0.0112398f, 1.06551f, -0.0274687f, 0.0112398f, 1.06551f, -0.0274687f, 
		0.0112398f, 1.02656f, -0.0274687f, 0.0112398f, 1.02656f, -0.0274687f, 
		0.0112422f, 0.989729f, -0.233480f, 0.0112515f, 1.06551f, 0.0274607f, 
		0.0112515f, 1.06551f, 0.0274607f, 0.0112748f, 1.00053f, 0.252667f, 
		0.0112748f, 1.00053f, 0.252667f, 0.0112820f, 1.04325f, 0.418649f, 
		0.0112820f, 1.04325f, 0.418649f, 0.0112820f, 1.04325f, 0.418649f, 
		0.0112820f, 1.04325f, 0.418649f, 0.0112820f, 1.04325f, 0.418649f, 
		0.0113420f, 0.989731f, 0.233372f, 0.0113425f, 1.31968f, -0.00649321f, 
		0.0113425f, 1.31968f, -0.00649321f, 0.0113425f, 1.31968f, -0.00649321f, 
		0.0113425f, 1.31968f, -0.00649321f, 0.0113425f, 1.31968f, -0.00649321f, 
		0.0113425f, 1.31968f, -0.00649321f, 0.0113425f, 1.31968f, -0.00649321f, 
		0.0113425f, 1.31968f, -0.00649321f, 0.0113789f, 1.02733f, 0.0151882f, 
		0.0113789f, 1.02733f, 0.0151882f, 0.0114812f, 1.56290f, -0.0804413f, 
		0.0115038f, 1.04385f, -0.377043f, 0.0115038f, 1.04385f, -0.377043f, 
		0.0115038f, 1.04385f, -0.377043f, 0.0115038f, 1.04385f, -0.377043f, 
		0.0115038f, 1.04385f, -0.377043f, 0.0115038f, 1.04385f, -0.377043f, 
		0.0115133f, 1.56290f, 0.0688845f, 0.0115841f, 1.31908f, 0.00309935f, 
		0.0115841f, 1.31908f, 0.00309935f, 0.0115841f, 1.31908f, 0.00309935f, 
		0.0115841f, 1.31908f, 0.00309935f, 0.0115841f, 1.31908f, 0.00309935f, 
		0.0115841f, 1.31908f, 0.00309935f, 0.0115841f, 1.31908f, 0.00309935f, 
		0.0115841f, 1.31908f, 0.00309935f, 0.0115841f, 1.31908f, 0.00309935f, 
		0.0116636f, 1.04385f, 0.376946f, 0.0116636f, 1.04385f, 0.376946f, 
		0.0116636f, 1.04385f, 0.376946f, 0.0116636f, 1.04385f, 0.376946f, 
		0.0116636f, 1.04385f, 0.376946f, 0.0116636f, 1.04385f, 0.376946f, 
		0.0117497f, 0.984598f, -0.242181f, 0.0117497f, 0.984598f, -0.242181f, 
		0.0117725f, 1.02261f, 0.252861f, 0.0117725f, 1.02261f, 0.252861f, 
		0.0117725f, 1.02261f, 0.252861f, 0.0117725f, 1.02261f, 0.252861f, 
		0.0117725f, 1.02261f, 0.252861f, 0.0117904f, 1.02640f, -0.373555f, 
		0.0117904f, 1.02640f, -0.373555f, 0.0117904f, 1.06751f, -0.373555f, 
		0.0117904f, 1.06751f, -0.373555f, 0.0117923f, 1.06751f, -0.364737f, 
		0.0117923f, 1.06751f, -0.364737f, 0.0117925f, 1.02640f, -0.364737f, 
		0.0117925f, 1.02640f, -0.364737f, 0.0118143f, 1.04009f, -0.379340f, 
		0.0118143f, 1.04009f, -0.379340f, 0.0118143f, 1.04009f, -0.379340f, 
		0.0118143f, 1.04009f, -0.379340f, 0.0118143f, 1.04009f, -0.379340f, 
		0.0118143f, 1.04009f, -0.379340f, 0.0118143f, 1.05382f, -0.379340f, 
		0.0118143f, 1.05382f, -0.379340f, 0.0118143f, 1.05382f, -0.379340f, 
		0.0118143f, 1.05382f, -0.379340f, 0.0118143f, 1.05382f, -0.379340f, 
		0.0118143f, 1.05382f, -0.379340f, 0.0118207f, 1.02557f, 0.0254543f, 
		0.0118207f, 1.02557f, 0.0254543f, 0.0118624f, 0.901703f, 0.247520f, 
		0.0118624f, 0.901703f, 0.247520f, 0.0118654f, 1.31559f, -0.0205632f, 
		0.0118654f, 1.31559f, -0.0205632f, 0.0118655f, 1.30677f, -0.0205631f, 
		0.0118655f, 1.30677f, -0.0205631f, 0.0118706f, 0.901703f, 0.288431f, 
		0.0118706f, 0.901703f, 0.288431f, 0.0118734f, 1.31559f, 0.0205541f, 
		0.0118734f, 1.31559f, 0.0205541f, 0.0118734f, 1.30677f, 0.0205541f, 
		0.0118734f, 1.30677f, 0.0205541f, 0.0118962f, 1.32137f, 0.00686293f, 
		0.0118962f, 1.32137f, 0.00686293f, 0.0118962f, 1.32137f, 0.00686293f, 
		0.0118962f, 1.32137f, 0.00686293f, 0.0118962f, 1.32137f, 0.00686293f, 
		0.0118962f, 1.32137f, 0.00686293f, 0.0119464f, 1.02640f, 0.364639f, 
		0.0119464f, 1.02640f, 0.364639f, 0.0119468f, 1.06751f, 0.364639f, 
		0.0119468f, 1.06751f, 0.364639f, 0.0119483f, 1.02640f, 0.373458f, 
		0.0119483f, 1.02640f, 0.373458f, 0.0119485f, 1.06751f, 0.373458f, 
		0.0119485f, 1.06751f, 0.373458f, 0.0119660f, 1.02530f, -0.0243330f, 
		0.0119660f, 1.02530f, -0.0243330f, 0.0119660f, 1.06677f, -0.0243330f, 
		0.0119660f, 1.06677f, -0.0243330f, 0.0119748f, 1.04009f, 0.379243f, 
		0.0119748f, 1.04009f, 0.379243f, 0.0119748f, 1.04009f, 0.379243f, 
		0.0119748f, 1.04009f, 0.379243f, 0.0119748f, 1.04009f, 0.379243f, 
		0.0119748f, 1.04009f, 0.379243f, 0.0119749f, 1.05382f, 0.379243f, 
		0.0119749f, 1.05382f, 0.379243f, 0.0119749f, 1.05382f, 0.379243f, 
		0.0119749f, 1.05382f, 0.379243f, 0.0119749f, 1.05382f, 0.379243f, 
		0.0119749f, 1.05382f, 0.379243f, 0.0119770f, 1.06677f, 0.0243240f, 
		0.0119770f, 1.06677f, 0.0243240f, 0.0119985f, 1.00773f, 0.248522f, 
		0.0119985f, 1.00773f, 0.248522f, 0.0119985f, 1.00773f, 0.248522f, 
		0.0120024f, 0.993266f, -0.309953f, 0.0120537f, 1.21087f, -0.0395773f, 
		0.0120720f, 1.21088f, 0.0395699f, 0.0120892f, 1.32782f, 0.00323442f, 
		0.0120892f, 1.32782f, 0.00323442f, 0.0120892f, 1.32782f, 0.00323442f, 
		0.0120892f, 1.32782f, 0.00323442f, 0.0120892f, 1.32782f, 0.00323442f, 
		0.0120995f, 1.04695f, -0.385768f, 0.0120995f, 1.04695f, -0.385768f, 
		0.0120995f, 1.04695f, -0.385768f, 0.0120995f, 1.04695f, -0.385768f, 
		0.0120995f, 1.04695f, -0.385768f, 0.0121034f, 1.32470f, -0.00699266f, 
		0.0121034f, 1.32470f, -0.00699266f, 0.0121034f, 1.32470f, -0.00699266f, 
		0.0121034f, 1.32470f, -0.00699266f, 0.0121034f, 1.32470f, -0.00699266f, 
		0.0121034f, 1.32470f, -0.00699266f, 0.0121034f, 1.32470f, -0.00699266f, 
		0.0121034f, 1.32470f, -0.00699266f, 0.0121345f, 0.993266f, 0.309846f, 
		0.0121575f, 1.05306f, -0.0306788f, 0.0121575f, 1.05306f, -0.0306788f, 
		0.0121575f, 1.05306f, -0.0306788f, 0.0121669f, 1.00051f, -0.253425f, 
		0.0121669f, 1.00051f, -0.253425f, 0.0121675f, 1.02477f, -0.0198915f, 
		0.0121675f, 1.02477f, -0.0198915f, 0.0121675f, 1.02477f, -0.0198915f, 
		0.0121675f, 1.02477f, -0.0198915f, 0.0121675f, 1.02477f, -0.0198915f, 
		0.0121675f, 1.02477f, -0.0198915f, 0.0121675f, 1.02477f, -0.0198915f, 
		0.0121709f, 1.05306f, 0.0306693f, 0.0121709f, 1.05306f, 0.0306693f, 
		0.0121709f, 1.05306f, 0.0306693f, 0.0121762f, 1.02477f, 0.0198819f, 
		0.0121762f, 1.02477f, 0.0198819f, 0.0121762f, 1.02477f, 0.0198819f, 
		0.0121762f, 1.02477f, 0.0198819f, 0.0121762f, 1.02477f, 0.0198819f, 
		0.0121762f, 1.02477f, 0.0198819f, 0.0121762f, 1.02477f, 0.0198819f, 
		0.0121762f, 1.02477f, 0.0198819f, 0.0121762f, 1.02477f, 0.0198819f, 
		0.0121762f, 1.02477f, 0.0198819f, 0.0121860f, 1.06729f, -0.0202709f, 
		0.0121860f, 1.06729f, -0.0202709f, 0.0121860f, 1.06729f, -0.0202709f, 
		0.0121860f, 1.06729f, -0.0202709f, 0.0121860f, 1.06729f, -0.0202709f, 
		0.0121860f, 1.06729f, -0.0202709f, 0.0121860f, 1.06729f, -0.0202709f, 
		0.0121860f, 1.06729f, -0.0202709f, 0.0122036f, 1.06710f, 0.0212247f, 
		0.0122036f, 1.06710f, 0.0212247f, 0.0122036f, 1.06710f, 0.0212247f, 
		0.0122036f, 1.06710f, 0.0212247f, 0.0122570f, 1.02559f, -0.371326f, 
		0.0122570f, 1.02559f, -0.371326f, 0.0122570f, 1.06832f, -0.371326f, 
		0.0122570f, 1.06832f, -0.371326f, 0.0122626f, 1.04696f, 0.385671f, 
		0.0122626f, 1.04696f, 0.385671f, 0.0122626f, 1.04696f, 0.385671f, 
		0.0122626f, 1.04696f, 0.385671f, 0.0122626f, 1.04696f, 0.385671f, 
		0.0123176f, 1.29323f, -0.0394958f, 0.0123310f, 1.31336f, -0.0213700f, 
		0.0123310f, 1.31336f, -0.0213700f, 0.0123346f, 1.29322f, 0.0394886f, 
		0.0123400f, 1.31336f, 0.0213610f, 0.0123400f, 1.31336f, 0.0213610f, 
		0.0123520f, 1.06848f, -0.368420f, 0.0123520f, 1.06848f, -0.368420f, 
		0.0123522f, 1.02542f, -0.368420f, 0.0123522f, 1.02542f, -0.368420f, 
		0.0124130f, 1.02559f, 0.366868f, 0.0124130f, 1.02559f, 0.366868f, 
		0.0124130f, 1.06832f, 0.366868f, 0.0124130f, 1.06832f, 0.366868f, 
		0.0124259f, 1.31045f, -0.0215343f, 0.0124259f, 1.31045f, -0.0215343f, 
		0.0124349f, 1.31045f, 0.0215254f, 0.0124349f, 1.31045f, 0.0215254f, 
		0.0124430f, 0.836885f, -0.289741f, 0.0124430f, 0.836885f, -0.289741f, 
		0.0124530f, 0.836885f, -0.246423f, 0.0124530f, 0.836885f, -0.246423f, 
		0.0125083f, 1.02542f, 0.369775f, 0.0125083f, 1.02542f, 0.369775f, 
		0.0125086f, 1.06848f, 0.369775f, 0.0125086f, 1.06848f, 0.369775f, 
		0.0125444f, 0.899571f, -0.289915f, 0.0125444f, 0.899571f, -0.289915f, 
		0.0125535f, 0.899571f, -0.246249f, 0.0125535f, 0.899571f, -0.246249f, 
		0.0125570f, 0.836885f, 0.246316f, 0.0125570f, 0.836885f, 0.246316f, 
		0.0125660f, 0.836885f, 0.289634f, 0.0125660f, 0.836885f, 0.289634f, 
		0.0126208f, 0.958007f, 0.267805f, 0.0126208f, 0.958007f, 0.267805f, 
		0.0126208f, 0.958007f, 0.267805f, 0.0126208f, 0.958007f, 0.267805f, 
		0.0126208f, 0.958007f, 0.267805f, 0.0126330f, 0.948656f, -0.185919f, 
		0.0127100f, 1.57086f, -0.0902330f, 0.0127120f, 0.948656f, 0.185812f, 
		0.0127460f, 1.57086f, 0.0786770f, 0.0127471f, 1.55466f, 0.0420793f, 
		0.0127779f, 0.957938f, -0.269595f, 0.0127779f, 0.957938f, -0.269595f, 
		0.0127779f, 0.957938f, -0.269595f, 0.0128275f, 1.00346f, -0.290074f, 
		0.0128275f, 1.00346f, -0.290074f, 0.0128275f, 1.00346f, -0.290074f, 
		0.0129120f, 1.03854f, -0.370005f, 0.0129120f, 1.03854f, -0.370005f, 
		0.0129120f, 1.05354f, -0.370005f, 0.0129120f, 1.05354f, -0.370005f, 
		0.0129510f, 1.00346f, 0.289967f, 0.0129510f, 1.00346f, 0.289967f, 
		0.0129510f, 1.00346f, 0.289967f, 0.0129756f, 1.00045f, -0.312091f, 
		0.0129756f, 1.00045f, -0.312091f, 0.0129915f, 1.00045f, -0.229460f, 
		0.0129915f, 1.00045f, -0.229460f, 0.0130070f, 0.951668f, -0.183438f, 
		0.0130690f, 1.03854f, 0.369995f, 0.0130690f, 1.03854f, 0.369995f, 
		0.0130690f, 1.05354f, 0.369995f, 0.0130690f, 1.05354f, 0.369995f, 
		0.0130850f, 0.951668f, 0.183331f, 0.0130893f, 1.00045f, 0.229353f, 
		0.0130893f, 1.00045f, 0.229353f, 0.0131074f, 1.00045f, 0.311984f, 
		0.0131074f, 1.00045f, 0.311984f, 0.0131092f, 1.58880f, -0.0961978f, 
		0.0131092f, 1.58880f, -0.0961978f, 0.0131470f, 1.58880f, 0.0846408f, 
		0.0131470f, 1.58880f, 0.0846408f, 0.0133260f, 1.57721f, -0.0943250f, 
		0.0133640f, 1.57721f, 0.0827690f, 0.0133789f, 1.03766f, -0.394676f, 
		0.0133789f, 1.03766f, -0.394676f, 0.0133789f, 1.03766f, -0.394676f, 
		0.0133789f, 1.03766f, -0.394676f, 0.0133789f, 1.03766f, -0.394676f, 
		0.0133789f, 1.03766f, -0.394676f, 0.0133789f, 1.03766f, -0.394676f, 
		0.0133789f, 1.03766f, -0.394676f, 0.0133789f, 1.03766f, -0.394676f, 
		0.0133789f, 1.03766f, -0.394676f, 0.0133920f, 0.956459f, -0.275848f, 
		0.0133920f, 0.956459f, -0.275848f, 0.0134967f, 1.03353f, 0.361173f, 
		0.0134967f, 1.03353f, 0.361173f, 0.0134967f, 1.03353f, 0.361173f, 
		0.0134967f, 1.03353f, 0.361173f, 0.0135601f, 0.897585f, 0.244578f, 
		0.0135601f, 0.897585f, 0.244578f, 0.0135610f, 0.977001f, 0.241832f, 
		0.0135695f, 0.897585f, 0.291371f, 0.0135695f, 0.897585f, 0.291371f, 
		0.0136545f, 1.04695f, -0.379340f, 0.0136545f, 1.04695f, -0.379340f, 
		0.0136545f, 1.04695f, -0.379340f, 0.0136545f, 1.04695f, -0.379340f, 
		0.0136545f, 1.04695f, -0.379340f, 0.0136545f, 1.04695f, -0.379340f, 
		0.0137305f, 1.05066f, -0.383102f, 0.0137305f, 1.05066f, -0.383102f, 
		0.0137305f, 1.05066f, -0.383102f, 0.0137305f, 1.05066f, -0.383102f, 
		0.0137305f, 1.05066f, -0.383102f, 0.0137305f, 1.05066f, -0.383102f, 
		0.0137305f, 1.05066f, -0.383102f, 0.0137305f, 1.05066f, -0.383102f, 
		0.0137305f, 1.05066f, -0.383102f, 0.0137305f, 1.05066f, -0.383102f, 
		0.0137305f, 1.05066f, -0.383102f, 0.0137322f, 1.02272f, -0.301461f, 
		0.0137322f, 1.02272f, -0.301461f, 0.0137345f, 1.32137f, -5.07720e-006f, 
		0.0137345f, 1.32137f, -5.07720e-006f, 0.0137345f, 1.32137f, -5.07720e-006f, 
		0.0137345f, 1.32137f, -5.07720e-006f, 0.0137345f, 1.32137f, -5.07720e-006f, 
		0.0137345f, 1.32137f, -5.07720e-006f, 0.0137454f, 1.02272f, -0.240091f, 
		0.0137454f, 1.02272f, -0.240091f, 0.0138123f, 1.32513f, 0.00369626f, 
		0.0138123f, 1.32513f, 0.00369626f, 0.0138123f, 1.32513f, 0.00369626f, 
		0.0138123f, 1.32513f, 0.00369626f, 0.0138123f, 1.32513f, 0.00369626f, 
		0.0138123f, 1.32513f, 0.00369626f, 0.0138123f, 1.32513f, 0.00369626f, 
		0.0138123f, 1.32513f, 0.00369626f, 0.0138123f, 1.32513f, 0.00369626f, 
		0.0138123f, 1.32513f, 0.00369626f, 0.0138123f, 1.32513f, 0.00369626f, 
		0.0138151f, 1.04696f, 0.379242f, 0.0138151f, 1.04696f, 0.379242f, 
		0.0138151f, 1.04696f, 0.379242f, 0.0138151f, 1.04696f, 0.379242f, 
		0.0138151f, 1.04696f, 0.379242f, 0.0138151f, 1.04696f, 0.379242f, 
		0.0138617f, 1.02272f, 0.301353f, 0.0138617f, 1.02272f, 0.301353f, 
		0.0138680f, 0.896718f, -0.292210f, 0.0138680f, 0.896718f, -0.292210f, 
		0.0138780f, 0.896718f, -0.243955f, 0.0138780f, 0.896718f, -0.243955f, 
		0.0138931f, 1.05065f, 0.383004f, 0.0138931f, 1.05065f, 0.383004f, 
		0.0138931f, 1.05065f, 0.383004f, 0.0138931f, 1.05065f, 0.383004f, 
		0.0138931f, 1.05065f, 0.383004f, 0.0138931f, 1.05065f, 0.383004f, 
		0.0138931f, 1.05065f, 0.383004f, 0.0138931f, 1.05065f, 0.383004f, 
		0.0138931f, 1.05065f, 0.383004f, 0.0138931f, 1.05065f, 0.383004f, 
		0.0138931f, 1.05065f, 0.383004f, 0.0138931f, 1.05065f, 0.383004f, 
		0.0138931f, 1.05065f, 0.383004f, 0.0138931f, 1.05065f, 0.383004f, 
		0.0139263f, 1.04230f, -0.0306805f, 0.0139263f, 1.04230f, -0.0306805f, 
		0.0139263f, 1.04230f, -0.0306805f, 0.0139263f, 1.04230f, -0.0306805f, 
		0.0139263f, 1.04230f, -0.0306805f, 0.0139393f, 1.04230f, 0.0306711f, 
		0.0139393f, 1.04230f, 0.0306711f, 0.0139393f, 1.04230f, 0.0306711f, 
		0.0139393f, 1.04230f, 0.0306711f, 0.0141075f, 1.05515f, -0.394645f, 
		0.0141075f, 1.05515f, -0.394645f, 0.0141075f, 1.05515f, -0.394645f, 
		0.0141075f, 1.05515f, -0.394645f, 0.0141075f, 1.05515f, -0.394645f, 
		0.0141154f, 1.01154f, -0.249992f, 0.0141154f, 1.01154f, -0.249992f, 
		0.0141154f, 1.01154f, -0.249992f, 0.0141492f, 1.05517f, -0.386283f, 
		0.0141492f, 1.05517f, -0.386283f, 0.0141492f, 1.05517f, -0.386283f, 
		0.0141492f, 1.05517f, -0.386283f, 0.0141492f, 1.05517f, -0.386283f, 
		0.0141492f, 1.05517f, -0.386283f, 0.0141496f, 1.03874f, -0.386283f, 
		0.0141496f, 1.03874f, -0.386283f, 0.0141496f, 1.03874f, -0.386283f, 
		0.0141496f, 1.03874f, -0.386283f, 0.0141496f, 1.03874f, -0.386283f, 
		0.0141496f, 1.03874f, -0.386283f, 0.0141894f, 1.33668f, -0.00819833f, 
		0.0141894f, 1.33668f, -0.00819833f, 0.0141894f, 1.33668f, -0.00819833f, 
		0.0141894f, 1.33668f, -0.00819833f, 0.0141894f, 1.33668f, -0.00819833f, 
		0.0141894f, 1.33668f, -0.00819833f, 0.0141926f, 1.33668f, 0.00818823f, 
		0.0141926f, 1.33668f, 0.00818823f, 0.0141926f, 1.33668f, 0.00818823f, 
		0.0141926f, 1.33668f, 0.00818823f, 0.0141926f, 1.33668f, 0.00818823f, 
		0.0141926f, 1.33668f, 0.00818823f, 0.0142214f, 1.01154f, 0.249884f, 
		0.0142214f, 1.01154f, 0.249884f, 0.0142214f, 1.01154f, 0.249884f, 
		0.0142214f, 1.01154f, 0.249884f, 0.0142214f, 1.01154f, 0.249884f, 
		0.0142295f, 1.32831f, -0.00822103f, 0.0142295f, 1.32831f, -0.00822103f, 
		0.0142295f, 1.32831f, -0.00822103f, 0.0142295f, 1.32831f, -0.00822103f, 
		0.0142295f, 1.32831f, -0.00822103f, 0.0142295f, 1.32831f, -0.00822103f, 
		0.0142325f, 1.32831f, 0.00821116f, 0.0142325f, 1.32831f, 0.00821116f, 
		0.0142325f, 1.32831f, 0.00821116f, 0.0142325f, 1.32831f, 0.00821116f, 
		0.0142325f, 1.32831f, 0.00821116f, 0.0142325f, 1.32831f, 0.00821116f, 
		0.0142325f, 1.32831f, 0.00821116f, 0.0142325f, 1.32831f, 0.00821116f, 
		0.0142435f, 1.22218f, -0.0463366f, 0.0142640f, 1.22218f, 0.0463268f, 
		0.0142746f, 1.03876f, 0.394546f, 0.0142746f, 1.03876f, 0.394546f, 
		0.0142746f, 1.03876f, 0.394546f, 0.0142746f, 1.03876f, 0.394546f, 
		0.0142746f, 1.03876f, 0.394546f, 0.0142746f, 1.03876f, 0.394546f, 
		0.0142748f, 1.05515f, 0.394546f, 0.0142748f, 1.05515f, 0.394546f, 
		0.0142748f, 1.05515f, 0.394546f, 0.0142748f, 1.05515f, 0.394546f, 
		0.0142748f, 1.05515f, 0.394546f, 0.0142748f, 1.05515f, 0.394546f, 
		0.0143124f, 1.03874f, 0.386185f, 0.0143124f, 1.03874f, 0.386185f, 
		0.0143124f, 1.03874f, 0.386185f, 0.0143124f, 1.03874f, 0.386185f, 
		0.0143124f, 1.03874f, 0.386185f, 0.0143124f, 1.03874f, 0.386185f, 
		0.0143126f, 1.28192f, -0.0463151f, 0.0143126f, 1.05517f, 0.386185f, 
		0.0143126f, 1.05517f, 0.386185f, 0.0143315f, 1.28192f, 0.0463056f, 
		0.0145060f, 1.02272f, 0.240652f, 0.0145060f, 1.02272f, 0.240652f, 
		0.0145060f, 1.03132f, 0.240652f, 0.0145060f, 1.03132f, 0.240652f, 
		0.0145060f, 1.03132f, 0.240652f, 0.0145147f, 1.26712f, -0.0480756f, 
		0.0145147f, 1.26712f, -0.0480756f, 0.0145346f, 1.26712f, 0.0480658f, 
		0.0145346f, 1.26712f, 0.0480658f, 0.0145532f, 1.30202f, -0.0194335f, 
		0.0145532f, 1.30202f, -0.0194335f, 0.0147249f, 1.01221f, -0.291302f, 
		0.0148358f, 0.955074f, -0.259486f, 0.0148358f, 0.955074f, -0.259486f, 
		0.0148497f, 1.01220f, 0.291193f, 0.0148497f, 1.01220f, 0.291193f, 
		0.0148497f, 1.01220f, 0.291193f, 0.0148970f, 1.58893f, -0.104760f, 
		0.0148970f, 1.58893f, -0.104760f, 0.0148970f, 1.61893f, -0.104760f, 
		0.0148970f, 1.61893f, -0.104760f, 0.0148970f, 1.61893f, -0.104760f, 
		0.0149113f, 1.29913f, 0.0295726f, 0.0149210f, 1.02019f, -0.0151160f, 
		0.0149210f, 1.02019f, -0.0151160f, 0.0149210f, 1.07189f, -0.0151160f, 
		0.0149210f, 1.07189f, -0.0151160f, 0.0149220f, 1.04604f, -0.370005f, 
		0.0149220f, 1.04604f, -0.370005f, 0.0149280f, 1.02019f, 0.0151060f, 
		0.0149280f, 1.02019f, 0.0151060f, 0.0149280f, 1.07189f, 0.0151060f, 
		0.0149280f, 1.07189f, 0.0151060f, 0.0149390f, 1.58893f, 0.0932030f, 
		0.0149390f, 1.58893f, 0.0932030f, 0.0149390f, 1.61893f, 0.0932030f, 
		0.0149390f, 1.61893f, 0.0932030f, 0.0149390f, 1.61893f, 0.0932030f, 
		0.0149452f, 0.955074f, 0.259378f, 0.0149452f, 0.955074f, 0.259378f, 
		0.0149487f, 0.955074f, 0.276572f, 0.0149487f, 0.955074f, 0.276572f, 
		0.0150690f, 1.23699f, -0.0479048f, 0.0150690f, 1.23699f, -0.0479048f, 
		0.0150780f, 1.04604f, 0.369995f, 0.0150780f, 1.04604f, 0.369995f, 
		0.0150893f, 1.23699f, 0.0478946f, 0.0150893f, 1.23699f, 0.0478946f, 
		0.0151680f, 0.832160f, -0.276874f, 0.0151680f, 0.832160f, -0.276874f, 
		0.0151720f, 0.832160f, -0.259292f, 0.0151720f, 0.832160f, -0.259292f, 
		0.0152820f, 0.832160f, 0.259184f, 0.0152820f, 0.832160f, 0.259184f, 
		0.0152850f, 0.832160f, 0.276766f, 0.0152850f, 0.832160f, 0.276766f, 
		0.0153314f, 1.00046f, 0.256530f, 0.0153314f, 1.00046f, 0.256530f, 
		0.0153840f, 0.893445f, -0.294837f, 0.0153840f, 0.893445f, -0.294837f, 
		0.0153950f, 0.893445f, -0.241330f, 0.0153950f, 0.893445f, -0.241330f, 
		0.0154760f, 1.00046f, -0.283776f, 0.0154760f, 1.00046f, -0.283776f, 
		0.0154970f, 0.893445f, 0.241221f, 0.0154970f, 0.893445f, 0.241221f, 
		0.0155090f, 0.893445f, 0.294728f, 0.0155090f, 0.893445f, 0.294728f, 
		0.0155410f, 0.843081f, -0.295108f, 0.0155410f, 0.843081f, -0.295108f, 
		0.0155520f, 0.843081f, -0.241058f, 0.0155520f, 0.843081f, -0.241058f, 
		0.0156540f, 0.843081f, 0.240949f, 0.0156540f, 0.843081f, 0.240949f, 
		0.0156660f, 0.843081f, 0.295000f, 0.0156660f, 0.843081f, 0.295000f, 
		0.0157780f, 0.927433f, -0.165082f, 0.0158480f, 0.927433f, 0.164974f, 
		0.0158865f, 1.03774f, -0.361273f, 0.0158865f, 1.03774f, -0.361273f, 
		0.0158865f, 1.03774f, -0.361273f, 0.0158913f, 1.03773f, -0.376990f, 
		0.0158913f, 1.03773f, -0.376990f, 0.0158916f, 1.05618f, -0.376990f, 
		0.0158916f, 1.05618f, -0.376990f, 0.0159646f, 1.30330f, 0.00921092f, 
		0.0159646f, 1.30330f, 0.00921092f, 0.0159646f, 1.30330f, 0.00921092f, 
		0.0159646f, 1.30330f, 0.00921092f, 0.0159697f, 1.31902f, -0.00922674f, 
		0.0159697f, 1.31902f, -0.00922674f, 0.0159732f, 1.31902f, 0.00921583f, 
		0.0159732f, 1.31902f, 0.00921583f, 0.0159856f, 0.953366f, -0.277347f, 
		0.0159856f, 0.953366f, -0.277347f, 0.0159964f, 1.05624f, -0.392414f, 
		0.0159964f, 1.05624f, -0.392414f, 0.0159964f, 1.05624f, -0.392414f, 
		0.0159964f, 1.05624f, -0.392414f, 0.0159964f, 1.05624f, -0.392414f, 
		0.0159964f, 1.05624f, -0.392414f, 0.0159964f, 1.05624f, -0.392414f, 
		0.0159964f, 1.05624f, -0.392414f, 0.0159964f, 1.05624f, -0.392414f, 
		0.0159964f, 1.05624f, -0.392414f, 0.0159964f, 1.03767f, -0.392414f, 
		0.0159964f, 1.03767f, -0.392414f, 0.0159964f, 1.03767f, -0.392414f, 
		0.0159964f, 1.03767f, -0.392414f, 0.0159964f, 1.03767f, -0.392414f, 
		0.0159964f, 1.03767f, -0.392414f, 0.0160508f, 1.03773f, 0.376891f, 
		0.0160508f, 1.03773f, 0.376891f, 0.0160511f, 1.05618f, 0.376891f, 
		0.0160511f, 1.05618f, 0.376891f, 0.0160773f, 1.33039f, -0.00928910f, 
		0.0160773f, 1.33039f, -0.00928910f, 0.0160773f, 1.33039f, -0.00928910f, 
		0.0160773f, 1.33039f, -0.00928910f, 0.0160773f, 1.33039f, -0.00928910f, 
		0.0160773f, 1.33039f, -0.00928910f, 0.0160773f, 1.33039f, -0.00928910f, 
		0.0160773f, 1.33039f, -0.00928910f, 0.0160773f, 1.33039f, -0.00928910f, 
		0.0160773f, 1.33039f, -0.00928910f, 0.0160812f, 1.33444f, 0.00927850f, 
		0.0160812f, 1.33444f, 0.00927850f, 0.0160812f, 1.33444f, 0.00927850f, 
		0.0160812f, 1.33444f, 0.00927850f, 0.0160812f, 1.33444f, 0.00927850f, 
		0.0160812f, 1.33444f, 0.00927850f, 0.0160812f, 1.33444f, 0.00927850f, 
		0.0160812f, 1.33444f, 0.00927850f, 0.0160812f, 1.33444f, 0.00927850f, 
		0.0161622f, 1.05624f, 0.388264f, 0.0161622f, 1.05624f, 0.388264f, 
		0.0161622f, 1.05624f, 0.388264f, 0.0161622f, 1.05624f, 0.388264f, 
		0.0161622f, 1.05624f, 0.388264f, 0.0161622f, 1.05624f, 0.388264f, 
		0.0161622f, 1.05624f, 0.388264f, 0.0161622f, 1.05624f, 0.388264f, 
		0.0161623f, 1.03767f, 0.392315f, 0.0161623f, 1.03767f, 0.392315f, 
		0.0161623f, 1.03767f, 0.392315f, 0.0161623f, 1.03767f, 0.392315f, 
		0.0161623f, 1.03767f, 0.392315f, 0.0161623f, 1.03767f, 0.392315f, 
		0.0161623f, 1.03767f, 0.392315f, 0.0161623f, 1.03767f, 0.392315f, 
		0.0161623f, 1.03767f, 0.392315f, 0.0161623f, 1.03767f, 0.392315f, 
		0.0162000f, 1.23705f, -0.0525620f, 0.0162000f, 1.23705f, -0.0525620f, 
		0.0162000f, 1.26705f, -0.0525620f, 0.0162000f, 1.26705f, -0.0525620f, 
		0.0162105f, 1.05636f, -0.389174f, 0.0162105f, 1.05636f, -0.389174f, 
		0.0162105f, 1.05636f, -0.389174f, 0.0162105f, 1.05636f, -0.389174f, 
		0.0162105f, 1.05636f, -0.389174f, 0.0162105f, 1.05636f, -0.389174f, 
		0.0162105f, 1.05636f, -0.389174f, 0.0162105f, 1.05636f, -0.389174f, 
		0.0162105f, 1.05636f, -0.389174f, 0.0162105f, 1.05636f, -0.389174f, 
		0.0162105f, 1.05636f, -0.389174f, 0.0162105f, 1.05636f, -0.389174f, 
		0.0162105f, 1.05636f, -0.389174f, 0.0162105f, 1.05636f, -0.389174f, 
		0.0162107f, 1.03755f, -0.389174f, 0.0162107f, 1.03755f, -0.389174f, 
		0.0162107f, 1.03755f, -0.389174f, 0.0162107f, 1.03755f, -0.389174f, 
		0.0162107f, 1.03755f, -0.389174f, 0.0162107f, 1.03755f, -0.389174f, 
		0.0162107f, 1.03755f, -0.389174f, 0.0162107f, 1.03755f, -0.389174f, 
		0.0162107f, 1.03755f, -0.389174f, 0.0162107f, 1.03755f, -0.389174f, 
		0.0162107f, 1.03755f, -0.389174f, 0.0162107f, 1.03755f, -0.389174f, 
		0.0162107f, 1.03755f, -0.389174f, 0.0162230f, 1.23705f, 0.0525510f, 
		0.0162230f, 1.23705f, 0.0525510f, 0.0162230f, 1.26705f, 0.0525510f, 
		0.0162230f, 1.26705f, 0.0525510f, 0.0162440f, 0.929791f, -0.161993f, 
		0.0162909f, 1.33363f, -0.00941248f, 0.0162909f, 1.33363f, -0.00941248f, 
		0.0162909f, 1.33363f, -0.00941248f, 0.0162909f, 1.33363f, -0.00941248f, 
		0.0162909f, 1.33363f, -0.00941248f, 0.0162909f, 1.33363f, -0.00941248f, 
		0.0162909f, 1.33363f, -0.00941248f, 0.0162909f, 1.33363f, -0.00941248f, 
		0.0162909f, 1.33363f, -0.00941248f, 0.0162909f, 1.33363f, -0.00941248f, 
		0.0162909f, 1.33363f, -0.00941248f, 0.0162909f, 1.33363f, -0.00941248f, 
		0.0162909f, 1.33363f, -0.00941248f, 0.0162909f, 1.33363f, -0.00941248f, 
		0.0162950f, 1.33120f, 0.00940143f, 0.0162950f, 1.33120f, 0.00940143f, 
		0.0162950f, 1.33120f, 0.00940143f, 0.0162950f, 1.33120f, 0.00940143f, 
		0.0162950f, 1.33120f, 0.00940143f, 0.0162950f, 1.33120f, 0.00940143f, 
		0.0162950f, 1.33120f, 0.00940143f, 0.0162950f, 1.33120f, 0.00940143f, 
		0.0162950f, 1.33120f, 0.00940143f, 0.0162950f, 1.33120f, 0.00940143f, 
		0.0162950f, 1.33120f, 0.00940143f, 0.0162960f, 1.01780f, -0.0148260f, 
		0.0162960f, 1.01780f, -0.0148260f, 0.0162960f, 1.07427f, -0.0148260f, 
		0.0162960f, 1.07427f, -0.0148260f, 0.0163020f, 1.01780f, 0.0148150f, 
		0.0163020f, 1.01780f, 0.0148150f, 0.0163020f, 1.07427f, 0.0148150f, 
		0.0163020f, 1.07427f, 0.0148150f, 0.0163026f, 1.04696f, -0.394644f, 
		0.0163026f, 1.04696f, -0.394644f, 0.0163026f, 1.04696f, -0.394644f, 
		0.0163026f, 1.04696f, -0.394644f, 0.0163026f, 1.04696f, -0.394644f, 
		0.0163026f, 1.04696f, -0.394644f, 0.0163026f, 1.04696f, -0.394644f, 
		0.0163130f, 0.929791f, 0.161884f, 0.0163481f, 1.03659f, -0.0298672f, 
		0.0163481f, 1.03659f, -0.0298672f, 0.0163481f, 1.05548f, -0.0298672f, 
		0.0163481f, 1.05548f, -0.0298672f, 0.0163506f, 1.04695f, -0.386284f, 
		0.0163506f, 1.04695f, -0.386284f, 0.0163506f, 1.04695f, -0.386284f, 
		0.0163506f, 1.04695f, -0.386284f, 0.0163506f, 1.04695f, -0.386284f, 
		0.0163506f, 1.04695f, -0.386284f, 0.0163610f, 1.05548f, 0.0298566f, 
		0.0163610f, 1.05548f, 0.0298566f, 0.0163612f, 1.03659f, 0.0298566f, 
		0.0163612f, 1.03659f, 0.0298566f, 0.0163755f, 1.05636f, 0.391505f, 
		0.0163755f, 1.05636f, 0.391505f, 0.0163755f, 1.05636f, 0.391505f, 
		0.0163755f, 1.05636f, 0.391505f, 0.0163755f, 1.05636f, 0.391505f, 
		0.0163755f, 1.05636f, 0.391505f, 0.0163755f, 1.05636f, 0.391505f, 
		0.0163755f, 1.05636f, 0.391505f, 0.0163756f, 1.03755f, 0.389074f, 
		0.0163756f, 1.03755f, 0.389074f, 0.0163756f, 1.03755f, 0.389074f, 
		0.0163756f, 1.03755f, 0.389074f, 0.0163756f, 1.03755f, 0.389074f, 
		0.0163756f, 1.03755f, 0.389074f, 0.0163756f, 1.03755f, 0.389074f, 
		0.0163756f, 1.03755f, 0.389074f, 0.0163756f, 1.03755f, 0.389074f, 
		0.0163756f, 1.03755f, 0.389074f, 0.0163756f, 1.03755f, 0.389074f, 
		0.0163756f, 1.03755f, 0.389074f, 0.0163756f, 1.03755f, 0.389074f, 
		0.0163756f, 1.03755f, 0.389074f, 0.0163863f, 1.33668f, -4.95991e-006f, 
		0.0163863f, 1.33668f, -4.95991e-006f, 0.0163863f, 1.33668f, -4.95991e-006f, 
		0.0163863f, 1.33668f, -4.95991e-006f, 0.0163863f, 1.33668f, -4.95991e-006f, 
		0.0163863f, 1.33668f, -4.95991e-006f, 0.0164324f, 1.32831f, -5.23088e-006f, 
		0.0164324f, 1.32831f, -5.23088e-006f, 0.0164324f, 1.32831f, -5.23088e-006f, 
		0.0164324f, 1.32831f, -5.23088e-006f, 0.0164324f, 1.32831f, -5.23088e-006f, 
		0.0164324f, 1.32831f, -5.23088e-006f, 0.0164700f, 1.04696f, 0.394545f, 
		0.0164700f, 1.04696f, 0.394545f, 0.0164700f, 1.04696f, 0.394545f, 
		0.0164700f, 1.04696f, 0.394545f, 0.0164700f, 1.04696f, 0.394545f, 
		0.0164700f, 1.04696f, 0.394545f, 0.0165140f, 1.04696f, 0.386185f, 
		0.0165140f, 1.04696f, 0.386185f, 0.0165140f, 1.04696f, 0.386185f, 
		0.0165140f, 1.04696f, 0.386185f, 0.0166871f, 1.00838f, -0.289311f, 
		0.0166871f, 1.00838f, -0.289311f, 0.0166871f, 1.00838f, -0.289311f, 
		0.0166871f, 1.00838f, -0.289311f, 0.0166871f, 1.00838f, -0.289311f, 
		0.0167090f, 0.889845f, -0.297134f, 0.0167090f, 0.889845f, -0.297134f, 
		0.0167220f, 0.889845f, -0.239033f, 0.0167220f, 0.889845f, -0.239033f, 
		0.0167615f, 1.02265f, -0.284024f, 0.0167615f, 1.02265f, -0.284024f, 
		0.0167615f, 1.02265f, -0.284024f, 0.0167615f, 1.02265f, -0.284024f, 
		0.0168072f, 0.951393f, -0.258349f, 0.0168072f, 0.951393f, -0.258349f, 
		0.0168230f, 0.889845f, 0.238924f, 0.0168230f, 0.889845f, 0.238924f, 
		0.0168350f, 0.889845f, 0.297025f, 0.0168350f, 0.889845f, 0.297025f, 
		0.0169161f, 0.951393f, 0.258239f, 0.0169161f, 0.951393f, 0.258239f, 
		0.0169201f, 0.951393f, 0.277710f, 0.0169201f, 0.951393f, 0.277710f, 
		0.0169795f, 1.00056f, -0.256915f, 0.0169795f, 1.00056f, -0.256915f, 
		0.0169795f, 1.00056f, -0.256915f, 0.0169795f, 1.00056f, -0.256915f, 
		0.0170255f, 0.989731f, -0.305799f, 0.0170399f, 0.989730f, -0.235756f, 
		0.0171229f, 0.980923f, 0.242549f, 0.0171376f, 0.955074f, -0.268084f, 
		0.0171376f, 0.955074f, -0.268084f, 0.0171402f, 0.989730f, 0.235648f, 
		0.0171552f, 0.989730f, 0.305689f, 0.0171552f, 0.989730f, 0.305689f, 
		0.0171552f, 0.989730f, 0.305689f, 0.0172505f, 0.955074f, 0.267974f, 
		0.0172505f, 0.955074f, 0.267974f, 0.0172512f, 0.949218f, -0.278077f, 
		0.0172512f, 0.949218f, -0.278077f, 0.0172620f, 0.912612f, -0.278083f, 
		0.0172620f, 0.912612f, -0.278083f, 0.0172620f, 0.912612f, -0.278083f, 
		0.0172660f, 0.912612f, -0.258083f, 0.0172660f, 0.912612f, -0.258083f, 
		0.0172660f, 0.912612f, -0.258083f, 0.0172660f, 0.948080f, -0.258083f, 
		0.0172660f, 0.948080f, -0.258083f, 0.0172770f, 1.30200f, -0.0167961f, 
		0.0172934f, 1.20208f, 0.0170407f, 0.0172934f, 1.20208f, 0.0170407f, 
		0.0173750f, 0.948080f, 0.257974f, 0.0173750f, 0.948080f, 0.257974f, 
		0.0173790f, 0.948080f, 0.277974f, 0.0173790f, 0.948080f, 0.277974f, 
		0.0173843f, 0.911318f, 0.257969f, 0.0173843f, 0.911318f, 0.257969f, 
		0.0173843f, 0.911318f, 0.257969f, 0.0173884f, 0.911318f, 0.277979f, 
		0.0173884f, 0.911318f, 0.277979f, 0.0173884f, 0.911318f, 0.277979f, 
		0.0175260f, 0.832160f, -0.268083f, 0.0175260f, 0.832160f, -0.268083f, 
		0.0175470f, 0.965681f, -0.327105f, 0.0175590f, 1.01562f, -0.0139440f, 
		0.0175590f, 1.01562f, -0.0139440f, 0.0175590f, 1.07645f, -0.0139440f, 
		0.0175590f, 1.07645f, -0.0139440f, 0.0175650f, 1.01562f, 0.0139330f, 
		0.0175650f, 1.01562f, 0.0139330f, 0.0175650f, 1.07645f, 0.0139330f, 
		0.0175650f, 1.07645f, 0.0139330f, 0.0176303f, 0.908777f, -0.278296f, 
		0.0176303f, 0.908777f, -0.278296f, 0.0176343f, 0.908777f, -0.257871f, 
		0.0176343f, 0.908777f, -0.257871f, 0.0176373f, 1.03535f, -0.0151733f, 
		0.0176373f, 1.03535f, -0.0151733f, 0.0176373f, 1.03535f, -0.0151733f, 
		0.0176373f, 1.03535f, -0.0151733f, 0.0176390f, 0.832160f, 0.267974f, 
		0.0176390f, 0.832160f, 0.267974f, 0.0176441f, 1.03535f, 0.0151621f, 
		0.0176441f, 1.03535f, 0.0151621f, 0.0176850f, 0.965681f, 0.326996f, 
		0.0178179f, 1.29323f, -0.0373394f, 0.0178290f, 0.885952f, -0.299074f, 
		0.0178290f, 0.885952f, -0.299074f, 0.0178347f, 1.29323f, 0.0373280f, 
		0.0178420f, 0.885952f, -0.237093f, 0.0178420f, 0.885952f, -0.237093f, 
		0.0178481f, 1.01952f, -0.256349f, 0.0178481f, 1.01952f, -0.256349f, 
		0.0178481f, 1.01952f, -0.256349f, 0.0178571f, 1.00051f, 0.282204f, 
		0.0178571f, 1.00051f, 0.282204f, 0.0179020f, 0.850508f, -0.299201f, 
		0.0179020f, 0.850508f, -0.299201f, 0.0179150f, 0.850508f, -0.236966f, 
		0.0179150f, 0.850508f, -0.236966f, 0.0179420f, 0.885952f, 0.236984f, 
		0.0179420f, 0.885952f, 0.236984f, 0.0179560f, 0.885952f, 0.298964f, 
		0.0179560f, 0.885952f, 0.298964f, 0.0180080f, 0.980877f, -0.296500f, 
		0.0180160f, 0.850508f, 0.236857f, 0.0180160f, 0.850508f, 0.236857f, 
		0.0180290f, 0.850508f, 0.299091f, 0.0180290f, 0.850508f, 0.299091f, 
		0.0180663f, 1.21088f, -0.0372201f, 0.0180685f, 1.05597f, -0.0151735f, 
		0.0180685f, 1.05597f, -0.0151735f, 0.0180690f, 0.969213f, -0.328799f, 
		0.0180748f, 1.05597f, 0.0151620f, 0.0180748f, 1.05597f, 0.0151620f, 
		0.0180814f, 1.21088f, 0.0372088f, 0.0180917f, 1.31805f, -0.0111664f, 
		0.0180917f, 1.31805f, -0.0111664f, 0.0180970f, 1.31805f, 0.0111549f, 
		0.0180970f, 1.31805f, 0.0111549f, 0.0181644f, 1.03132f, -0.242491f, 
		0.0181644f, 1.03132f, -0.242491f, 0.0181644f, 1.03132f, -0.242491f, 
		0.0181760f, 0.903517f, -0.149196f, 0.0182080f, 0.969213f, 0.328690f, 
		0.0182390f, 0.903517f, 0.149086f, 0.0182554f, 1.05187f, -0.361272f, 
		0.0182554f, 1.05187f, -0.361272f, 0.0182554f, 1.05187f, -0.361272f, 
		0.0182554f, 1.05187f, -0.361272f, 0.0182763f, 1.03132f, 0.298955f, 
		0.0182763f, 1.03132f, 0.298955f, 0.0182763f, 1.03132f, 0.298955f, 
		0.0183308f, 1.30330f, -0.00491828f, 0.0183308f, 1.30330f, -0.00491828f, 
		0.0183308f, 1.30330f, -0.00491828f, 0.0183308f, 1.30330f, -0.00491828f, 
		0.0183624f, 1.04696f, -0.376991f, 0.0183624f, 1.04696f, -0.376991f, 
		0.0183767f, 1.00045f, 0.262577f, 0.0183767f, 1.00045f, 0.262577f, 
		0.0184025f, 0.906317f, 0.257380f, 0.0184025f, 0.906317f, 0.257380f, 
		0.0184070f, 0.906317f, 0.278567f, 0.0184070f, 0.906317f, 0.278567f, 
		0.0184085f, 1.05187f, 0.361172f, 0.0184085f, 1.05187f, 0.361172f, 
		0.0184085f, 1.05187f, 0.361172f, 0.0184085f, 1.05187f, 0.361172f, 
		0.0184085f, 1.05187f, 0.361172f, 0.0184766f, 1.00045f, -0.309933f, 
		0.0184766f, 1.00045f, -0.309933f, 0.0184849f, 1.04695f, -0.388363f, 
		0.0184849f, 1.04695f, -0.388363f, 0.0184849f, 1.04695f, -0.388363f, 
		0.0184849f, 1.04695f, -0.388363f, 0.0184849f, 1.04695f, -0.388363f, 
		0.0184849f, 1.04695f, -0.388363f, 0.0184849f, 1.04695f, -0.388363f, 
		0.0184849f, 1.04695f, -0.388363f, 0.0184849f, 1.04695f, -0.388363f, 
		0.0184849f, 1.04695f, -0.388363f, 0.0184934f, 1.00045f, -0.231621f, 
		0.0184934f, 1.00045f, -0.231621f, 0.0185216f, 1.04696f, 0.376890f, 
		0.0185216f, 1.04696f, 0.376890f, 0.0185391f, 1.03621f, -0.375844f, 
		0.0185391f, 1.03621f, -0.375844f, 0.0185391f, 1.05770f, -0.375844f, 
		0.0185391f, 1.05770f, -0.375844f, 0.0185420f, 1.03621f, -0.362450f, 
		0.0185420f, 1.03621f, -0.362450f, 0.0185422f, 1.05770f, -0.362450f, 
		0.0185422f, 1.05770f, -0.362450f, 0.0185618f, 1.02273f, 0.260143f, 
		0.0185618f, 1.02273f, 0.260143f, 0.0185618f, 1.02273f, 0.260143f, 
		0.0185618f, 1.02273f, 0.260143f, 0.0185672f, 1.33444f, -6.30200e-006f, 
		0.0185672f, 1.33444f, -6.30200e-006f, 0.0185672f, 1.33444f, -6.30200e-006f, 
		0.0185672f, 1.33444f, -6.30200e-006f, 0.0185672f, 1.33444f, -6.30200e-006f, 
		0.0185672f, 1.33444f, -6.30200e-006f, 0.0185672f, 1.33444f, -6.30200e-006f, 
		0.0185672f, 1.33444f, -6.30200e-006f, 0.0185672f, 1.33444f, -6.30200e-006f, 
		0.0185912f, 1.00045f, 0.231511f, 0.0185912f, 1.00045f, 0.231511f, 
		0.0186161f, 1.30448f, -0.0107555f, 0.0186161f, 1.30448f, -0.0107555f, 
		0.0186213f, 1.30448f, 0.0107438f, 0.0186213f, 1.30448f, 0.0107438f, 
		0.0186496f, 1.04695f, 0.388264f, 0.0186496f, 1.04695f, 0.388264f, 
		0.0186496f, 1.04695f, 0.388264f, 0.0186496f, 1.04695f, 0.388264f, 
		0.0186496f, 1.04695f, 0.388264f, 0.0186496f, 1.04695f, 0.388264f, 
		0.0186496f, 1.04695f, 0.388264f, 0.0186496f, 1.04695f, 0.388264f, 
		0.0186730f, 1.01369f, -0.0124520f, 0.0186730f, 1.01369f, -0.0124520f, 
		0.0186730f, 1.07838f, -0.0124520f, 0.0186730f, 1.07838f, -0.0124520f, 
		0.0186780f, 1.01369f, 0.0124400f, 0.0186780f, 1.01369f, 0.0124400f, 
		0.0186790f, 1.07838f, 0.0124400f, 0.0186790f, 1.07838f, 0.0124400f, 
		0.0186834f, 1.05742f, -0.0151973f, 0.0186834f, 1.05742f, -0.0151973f, 
		0.0186951f, 1.05770f, 0.362350f, 0.0186951f, 1.05770f, 0.362350f, 
		0.0186954f, 1.03621f, 0.362350f, 0.0186954f, 1.03621f, 0.362350f, 
		0.0186980f, 1.03620f, 0.375744f, 0.0186980f, 1.03620f, 0.375744f, 
		0.0186982f, 1.05770f, 0.375744f, 0.0186982f, 1.05770f, 0.375744f, 
		0.0187040f, 1.03132f, -0.298303f, 0.0187040f, 1.03132f, -0.298303f, 
		0.0187040f, 1.03132f, -0.298303f, 0.0187120f, 0.905134f, -0.145642f, 
		0.0187270f, 0.881798f, -0.300630f, 0.0187270f, 0.881798f, -0.300630f, 
		0.0187309f, 1.04695f, -0.391604f, 0.0187309f, 1.04695f, -0.391604f, 
		0.0187309f, 1.04695f, -0.391604f, 0.0187309f, 1.04695f, -0.391604f, 
		0.0187309f, 1.04695f, -0.391604f, 0.0187309f, 1.04695f, -0.391604f, 
		0.0187309f, 1.04695f, -0.391604f, 0.0187309f, 1.04695f, -0.391604f, 
		0.0187309f, 1.04695f, -0.391604f, 0.0187309f, 1.04695f, -0.391604f, 
		0.0187309f, 1.04695f, -0.391604f, 0.0187309f, 1.04695f, -0.391604f, 
		0.0187309f, 1.04695f, -0.391604f, 0.0187410f, 0.881798f, -0.235537f, 
		0.0187410f, 0.881798f, -0.235537f, 0.0187668f, 0.984600f, -0.297233f, 
		0.0187668f, 0.984600f, -0.297233f, 0.0187740f, 0.905134f, 0.145532f, 
		0.0188003f, 0.984598f, -0.295294f, 0.0188003f, 0.984598f, -0.295294f, 
		0.0188003f, 0.984598f, -0.295294f, 0.0188135f, 1.33120f, -6.37292e-006f, 
		0.0188135f, 1.33120f, -6.37292e-006f, 0.0188135f, 1.33120f, -6.37292e-006f, 
		0.0188135f, 1.33120f, -6.37292e-006f, 0.0188135f, 1.33120f, -6.37292e-006f, 
		0.0188135f, 1.33120f, -6.37292e-006f, 0.0188135f, 1.33120f, -6.37292e-006f, 
		0.0188135f, 1.33120f, -6.37292e-006f, 0.0188135f, 1.33120f, -6.37292e-006f, 
		0.0188135f, 1.33120f, -6.37292e-006f, 0.0188135f, 1.33120f, -6.37292e-006f, 
		0.0188190f, 1.02272f, 0.243141f, 0.0188190f, 1.02272f, 0.243141f, 
		0.0188190f, 1.03132f, 0.243141f, 0.0188190f, 1.03132f, 0.243141f, 
		0.0188300f, 1.03132f, 0.298194f, 0.0188300f, 1.03132f, 0.298194f, 
		0.0188410f, 0.881798f, 0.235427f, 0.0188410f, 0.881798f, 0.235427f, 
		0.0188540f, 0.881798f, 0.300521f, 0.0188540f, 0.881798f, 0.300521f, 
		0.0188784f, 1.04604f, -0.0298683f, 0.0188784f, 1.04604f, -0.0298683f, 
		0.0188795f, 0.984601f, 0.244211f, 0.0188795f, 0.984601f, 0.244211f, 
		0.0188905f, 1.04604f, 0.0298556f, 0.0188905f, 1.04604f, 0.0298556f, 
		0.0188962f, 1.04695f, 0.391504f, 0.0188962f, 1.04695f, 0.391504f, 
		0.0188962f, 1.04695f, 0.391504f, 0.0188962f, 1.04695f, 0.391504f, 
		0.0188962f, 1.04695f, 0.391504f, 0.0188962f, 1.04695f, 0.391504f, 
		0.0188962f, 1.04695f, 0.391504f, 0.0188962f, 1.04695f, 0.391504f, 
		0.0188962f, 1.04695f, 0.391504f, 0.0188962f, 1.04695f, 0.391504f, 
		0.0188962f, 1.04695f, 0.391504f, 0.0189765f, 1.20210f, 0.0156803f, 
		0.0189765f, 1.20210f, 0.0156803f, 0.0190383f, 1.01641f, -0.255607f, 
		0.0190383f, 1.01641f, -0.255607f, 0.0190383f, 1.01641f, -0.255607f, 
		0.0192029f, 1.03555f, -0.0151975f, 0.0192029f, 1.03555f, -0.0151975f, 
		0.0192227f, 0.903956f, -0.279216f, 0.0192227f, 0.903956f, -0.279216f, 
		0.0192280f, 0.903956f, -0.256951f, 0.0192280f, 0.903956f, -0.256951f, 
		0.0193880f, 0.877414f, -0.301776f, 0.0193880f, 0.877414f, -0.301776f, 
		0.0194030f, 0.877414f, -0.234392f, 0.0194030f, 0.877414f, -0.234392f, 
		0.0194070f, 0.858926f, -0.301809f, 0.0194070f, 0.858926f, -0.301809f, 
		0.0194077f, 1.01499f, 0.286070f, 0.0194077f, 1.01499f, 0.286070f, 
		0.0194077f, 1.01499f, 0.286070f, 0.0194136f, 0.951393f, -0.268084f, 
		0.0194136f, 0.951393f, -0.268084f, 0.0194220f, 0.858926f, -0.234359f, 
		0.0194220f, 0.858926f, -0.234359f, 0.0194720f, 1.05728f, -0.0274707f, 
		0.0194720f, 1.05728f, -0.0274707f, 0.0194721f, 1.03479f, -0.0274707f, 
		0.0194721f, 1.03479f, -0.0274707f, 0.0194835f, 1.05728f, 0.0274586f, 
		0.0194835f, 1.05728f, 0.0274586f, 0.0194836f, 1.03479f, 0.0274586f, 
		0.0194836f, 1.03479f, 0.0274586f, 0.0195020f, 0.877414f, 0.234282f, 
		0.0195020f, 0.877414f, 0.234282f, 0.0195141f, 1.30201f, 0.0150003f, 
		0.0195141f, 1.30201f, 0.0150003f, 0.0195141f, 1.30201f, 0.0150003f, 
		0.0195160f, 0.877414f, 0.301666f, 0.0195160f, 0.877414f, 0.301666f, 
		0.0195210f, 0.858926f, 0.234249f, 0.0195210f, 0.858926f, 0.234249f, 
		0.0195265f, 0.951393f, 0.267974f, 0.0195265f, 0.951393f, 0.267974f, 
		0.0195350f, 0.858926f, 0.301699f, 0.0195350f, 0.858926f, 0.301699f, 
		0.0195420f, 1.31892f, -9.91612e-005f, 0.0195420f, 1.31892f, -9.91612e-005f, 
		0.0196030f, 1.01208f, -0.0103300f, 0.0196030f, 1.01208f, -0.0103300f, 
		0.0196030f, 1.07999f, -0.0103300f, 0.0196030f, 1.07999f, -0.0103300f, 
		0.0196080f, 1.01208f, 0.0103180f, 0.0196080f, 1.01208f, 0.0103180f, 
		0.0196080f, 1.07999f, 0.0103180f, 0.0196080f, 1.07999f, 0.0103180f, 
		0.0196244f, 1.02272f, -0.243487f, 0.0196244f, 1.02272f, -0.243487f, 
		0.0197040f, 0.878415f, -0.139070f, 0.0197361f, 1.02272f, 0.297959f, 
		0.0197361f, 1.02272f, 0.297959f, 0.0197630f, 0.878415f, 0.138960f, 
		0.0197960f, 0.872835f, -0.302483f, 0.0197960f, 0.872835f, -0.302483f, 
		0.0198110f, 0.872835f, -0.233685f, 0.0198110f, 0.872835f, -0.233685f, 
		0.0198243f, 1.00046f, -0.269684f, 0.0198243f, 1.00046f, -0.269684f, 
		0.0199100f, 0.872835f, 0.233574f, 0.0199100f, 0.872835f, 0.233574f, 
		0.0199240f, 0.872835f, 0.302373f, 0.0199240f, 0.872835f, 0.302373f, 
		0.0199360f, 0.868091f, -0.302725f, 0.0199360f, 0.868091f, -0.302725f, 
		0.0199430f, 0.912612f, -0.268084f, 0.0199430f, 0.912612f, -0.268084f, 
		0.0199430f, 0.912612f, -0.268084f, 0.0199430f, 0.948080f, -0.268084f, 
		0.0199430f, 0.948080f, -0.268084f, 0.0199510f, 0.868091f, -0.233443f, 
		0.0199510f, 0.868091f, -0.233443f, 0.0199862f, 1.02259f, 0.278810f, 
		0.0199862f, 1.02259f, 0.278810f, 0.0200490f, 0.868091f, 0.233333f, 
		0.0200490f, 0.868091f, 0.233333f, 0.0200570f, 0.948080f, 0.267974f, 
		0.0200570f, 0.948080f, 0.267974f, 0.0200640f, 0.868091f, 0.302615f, 
		0.0200640f, 0.868091f, 0.302615f, 0.0200676f, 0.911318f, 0.267974f, 
		0.0200676f, 0.911318f, 0.267974f, 0.0200676f, 0.911318f, 0.267974f, 
		0.0202410f, 0.853634f, -0.135514f, 0.0202410f, 0.853634f, -0.135514f, 
		0.0202782f, 0.977046f, -0.293764f, 0.0202860f, 0.879246f, -0.135219f, 
		0.0202941f, 1.01837f, -0.282432f, 0.0202941f, 1.01837f, -0.282432f, 
		0.0202941f, 1.01837f, -0.282432f, 0.0202980f, 0.853634f, 0.135404f, 
		0.0202980f, 0.853634f, 0.135404f, 0.0203120f, 1.01085f, -0.00756100f, 
		0.0203120f, 1.01085f, -0.00756100f, 0.0203120f, 1.08122f, -0.00756100f, 
		0.0203120f, 1.08122f, -0.00756100f, 0.0203160f, 1.01085f, 0.00754800f, 
		0.0203160f, 1.01085f, 0.00754800f, 0.0203160f, 1.08122f, 0.00754800f, 
		0.0203160f, 1.08122f, 0.00754800f, 0.0203329f, 1.00052f, -0.278518f, 
		0.0203329f, 1.00052f, -0.278518f, 0.0203329f, 1.00052f, -0.278518f, 
		0.0203329f, 1.00052f, -0.278518f, 0.0203430f, 0.879246f, 0.135109f, 
		0.0203638f, 1.00048f, -0.265780f, 0.0203638f, 1.00048f, -0.265780f, 
		0.0203688f, 0.908777f, -0.268084f, 0.0203688f, 0.908777f, -0.268084f, 
		0.0204013f, 0.977046f, 0.293655f, 0.0204455f, 1.02272f, -0.270353f, 
		0.0204455f, 1.02272f, -0.270353f, 0.0204455f, 1.02272f, -0.270353f, 
		0.0204455f, 1.02272f, -0.270353f, 0.0204517f, 1.00052f, 0.278406f, 
		0.0204517f, 1.00052f, 0.278406f, 0.0204517f, 1.00052f, 0.278406f, 
		0.0204517f, 1.00052f, 0.278406f, 0.0204759f, 1.00048f, 0.265668f, 
		0.0204759f, 1.00048f, 0.265668f, 0.0204790f, 1.03509f, -0.373557f, 
		0.0204790f, 1.03509f, -0.373557f, 0.0204791f, 1.05882f, -0.373557f, 
		0.0204791f, 1.05882f, -0.373557f, 0.0204810f, 1.03509f, -0.364739f, 
		0.0204810f, 1.03509f, -0.364739f, 0.0204812f, 1.05882f, -0.364739f, 
		0.0204812f, 1.05882f, -0.364739f, 0.0205091f, 0.901703f, 0.256164f, 
		0.0205091f, 0.901703f, 0.256164f, 0.0205142f, 0.901703f, 0.279784f, 
		0.0205142f, 0.901703f, 0.279784f, 0.0205561f, 1.30677f, -0.0118754f, 
		0.0205561f, 1.30677f, -0.0118754f, 0.0205561f, 1.31559f, -0.0118754f, 
		0.0205561f, 1.31559f, -0.0118754f, 0.0205611f, 1.30677f, 0.0118635f, 
		0.0205611f, 1.30677f, 0.0118635f, 0.0205611f, 1.31559f, 0.0118634f, 
		0.0205611f, 1.31559f, 0.0118634f, 0.0206354f, 1.05882f, 0.364638f, 
		0.0206354f, 1.05882f, 0.364638f, 0.0206359f, 1.03509f, 0.364638f, 
		0.0206359f, 1.03509f, 0.364638f, 0.0206375f, 1.05882f, 0.373456f, 
		0.0206375f, 1.05882f, 0.373456f, 0.0206375f, 1.03509f, 0.373456f, 
		0.0206375f, 1.03509f, 0.373456f, 0.0207300f, 1.03407f, -0.0243350f, 
		0.0207300f, 1.03407f, -0.0243350f, 0.0207300f, 1.05801f, -0.0243350f, 
		0.0207300f, 1.05801f, -0.0243350f, 0.0207400f, 1.03407f, 0.0243220f, 
		0.0207400f, 1.03407f, 0.0243220f, 0.0207400f, 1.05801f, 0.0243220f, 
		0.0207400f, 1.05801f, 0.0243220f, 0.0207650f, 1.01007f, -0.00412600f, 
		0.0207650f, 1.01007f, -0.00412600f, 0.0207650f, 1.08200f, -0.00412600f, 
		0.0207650f, 1.08200f, -0.00412600f, 0.0207670f, 1.01007f, 0.00411300f, 
		0.0207670f, 1.01007f, 0.00411300f, 0.0207670f, 1.08200f, 0.00411300f, 
		0.0207670f, 1.08200f, 0.00411300f, 0.0208380f, 0.853677f, -0.131559f, 
		0.0208380f, 0.853677f, -0.131559f, 0.0208691f, 1.02274f, 0.273490f, 
		0.0208691f, 1.02274f, 0.273490f, 0.0208940f, 0.853677f, 0.131449f, 
		0.0208940f, 0.853677f, 0.131449f, 0.0209240f, 1.00979f, -6.00000e-006f, 
		0.0209240f, 1.00979f, -6.00000e-006f, 0.0209240f, 1.08228f, -6.00000e-006f, 
		0.0209240f, 1.08228f, -6.00000e-006f, 0.0209904f, 1.28192f, -0.0436965f, 
		0.0210024f, 1.01760f, 0.258652f, 0.0210101f, 1.28192f, 0.0436834f, 
		0.0210548f, 1.22218f, -0.0436655f, 0.0210608f, 1.03384f, -0.0212360f, 
		0.0210608f, 1.03384f, -0.0212360f, 0.0210608f, 1.03384f, -0.0212360f, 
		0.0210608f, 1.03384f, -0.0212360f, 0.0210608f, 1.03384f, -0.0212360f, 
		0.0210608f, 1.03384f, -0.0212360f, 0.0210734f, 1.22218f, 0.0436532f, 
		0.0210930f, 1.05818f, -0.0212361f, 0.0210930f, 1.05818f, -0.0212361f, 
		0.0210930f, 1.05818f, -0.0212361f, 0.0210930f, 1.05818f, -0.0212361f, 
		0.0210930f, 1.05818f, -0.0212361f, 0.0210930f, 1.05818f, -0.0212361f, 
		0.0210930f, 1.05818f, -0.0212361f, 0.0211270f, 0.977001f, -0.247170f, 
		0.0211867f, 1.05834f, 0.0202596f, 0.0211867f, 1.05834f, 0.0202596f, 
		0.0211867f, 1.05834f, 0.0202596f, 0.0211867f, 1.05834f, 0.0202596f, 
		0.0211867f, 1.05834f, 0.0202596f, 0.0211867f, 1.05834f, 0.0202596f, 
		0.0211867f, 1.05834f, 0.0202596f, 0.0212430f, 0.906317f, 0.267973f, 
		0.0212430f, 0.906317f, 0.267973f, 0.0212474f, 1.02269f, -0.272971f, 
		0.0212474f, 1.02269f, -0.272971f, 0.0212474f, 1.02269f, -0.272971f, 
		0.0212474f, 1.02269f, -0.272971f, 0.0212545f, 1.03385f, 0.0202592f, 
		0.0212545f, 1.03385f, 0.0202592f, 0.0212545f, 1.03385f, 0.0202592f, 
		0.0212545f, 1.03385f, 0.0202592f, 0.0212545f, 1.03385f, 0.0202592f, 
		0.0212545f, 1.03385f, 0.0202592f, 0.0212545f, 1.03385f, 0.0202592f, 
		0.0212545f, 1.03385f, 0.0202592f, 0.0212545f, 1.03385f, 0.0202592f, 
		0.0212545f, 1.03385f, 0.0202592f, 0.0212870f, 1.03462f, -0.371328f, 
		0.0212870f, 1.03462f, -0.371328f, 0.0212870f, 1.05929f, -0.371328f, 
		0.0212870f, 1.05929f, -0.371328f, 0.0213630f, 1.31336f, -0.0123420f, 
		0.0213630f, 1.31336f, -0.0123420f, 0.0213680f, 1.31336f, 0.0123290f, 
		0.0213680f, 1.31336f, 0.0123290f, 0.0214195f, 1.04696f, -0.375845f, 
		0.0214195f, 1.04696f, -0.375845f, 0.0214226f, 1.04696f, -0.362451f, 
		0.0214226f, 1.04696f, -0.362451f, 0.0214430f, 1.03462f, 0.366866f, 
		0.0214430f, 1.03462f, 0.366866f, 0.0214440f, 1.05929f, 0.371227f, 
		0.0214440f, 1.05929f, 0.371227f, 0.0214516f, 1.05938f, -0.368422f, 
		0.0214516f, 1.05938f, -0.368422f, 0.0214517f, 1.03452f, -0.368422f, 
		0.0214517f, 1.03452f, -0.368422f, 0.0214881f, 1.01917f, 0.261940f, 
		0.0214881f, 1.01917f, 0.261940f, 0.0214881f, 1.01917f, 0.261940f, 
		0.0214992f, 1.30448f, -6.32542e-006f, 0.0214992f, 1.30448f, -6.32542e-006f, 
		0.0215174f, 1.23699f, -0.0453752f, 0.0215174f, 1.23699f, -0.0453752f, 
		0.0215274f, 1.31045f, -0.0124369f, 0.0215274f, 1.31045f, -0.0124369f, 
		0.0215323f, 1.31045f, 0.0124239f, 0.0215323f, 1.31045f, 0.0124239f, 
		0.0215364f, 1.23699f, 0.0453624f, 0.0215364f, 1.23699f, 0.0453624f, 
		0.0215755f, 1.04695f, 0.362350f, 0.0215755f, 1.04695f, 0.362350f, 
		0.0215785f, 1.04696f, 0.375744f, 0.0215785f, 1.04696f, 0.375744f, 
		0.0216000f, 0.836885f, -0.280589f, 0.0216000f, 0.836885f, -0.280589f, 
		0.0216050f, 0.836885f, -0.255580f, 0.0216050f, 0.836885f, -0.255580f, 
		0.0216074f, 1.05938f, 0.368320f, 0.0216074f, 1.05938f, 0.368320f, 
		0.0216077f, 1.03452f, 0.369773f, 0.0216077f, 1.03452f, 0.369773f, 
		0.0216654f, 1.00864f, 0.258160f, 0.0217130f, 0.836885f, 0.255469f, 
		0.0217130f, 0.836885f, 0.255469f, 0.0217180f, 0.836885f, 0.280478f, 
		0.0217180f, 0.836885f, 0.280478f, 0.0217198f, 0.989730f, -0.303088f, 
		0.0217339f, 0.989731f, -0.238467f, 0.0217740f, 0.899571f, -0.280690f, 
		0.0217740f, 0.899571f, -0.280690f, 0.0217794f, 0.899571f, -0.255478f, 
		0.0217794f, 0.899571f, -0.255478f, 0.0218348f, 0.989731f, 0.238355f, 
		0.0218479f, 0.989730f, 0.302977f, 0.0218808f, 1.04656f, 0.0151850f, 
		0.0218808f, 1.04656f, 0.0151850f, 0.0220196f, 1.56290f, -0.0780388f, 
		0.0220407f, 1.26712f, -0.0451234f, 0.0220407f, 1.26712f, -0.0451234f, 
		0.0220495f, 1.56290f, 0.0664779f, 0.0220592f, 1.26712f, 0.0451105f, 
		0.0220592f, 1.26712f, 0.0451105f, 0.0220822f, 1.20497f, -0.0246871f, 
		0.0221366f, 1.20211f, -0.0110690f, 0.0221366f, 1.20211f, -0.0110690f, 
		0.0222082f, 0.903956f, -0.268084f, 0.0222082f, 0.903956f, -0.268084f, 
		0.0224243f, 1.30063f, -0.0202945f, 0.0224856f, 1.04604f, -0.0274717f, 
		0.0224856f, 1.04604f, -0.0274717f, 0.0224970f, 1.04604f, 0.0274582f, 
		0.0224970f, 1.04604f, 0.0274582f, 0.0225288f, 1.04606f, -0.0170125f, 
		0.0225980f, 1.02272f, -0.295200f, 0.0225980f, 1.02272f, -0.295200f, 
		0.0225980f, 1.03132f, -0.295200f, 0.0225980f, 1.03132f, -0.295200f, 
		0.0229182f, 0.980923f, 0.292608f, 0.0229429f, 1.01837f, 0.275804f, 
		0.0231731f, 1.00796f, -0.261717f, 0.0231731f, 1.00796f, -0.261717f, 
		0.0231731f, 1.00796f, -0.261717f, 0.0231731f, 1.00796f, -0.261717f, 
		0.0231731f, 1.00796f, -0.261717f, 0.0231823f, 1.21087f, -0.0342652f, 
		0.0231989f, 1.21088f, 0.0342528f, 0.0232838f, 1.00796f, 0.261606f, 
		0.0232838f, 1.00796f, 0.261606f, 0.0232838f, 1.00796f, 0.261606f, 
		0.0232838f, 1.00796f, 0.261606f, 0.0233248f, 1.01838f, -0.268941f, 
		0.0234039f, 1.01221f, -0.279991f, 0.0234111f, 1.29323f, -0.0341092f, 
		0.0234262f, 1.29323f, 0.0340976f, 0.0234505f, 0.897585f, 0.254465f, 
		0.0234505f, 0.897585f, 0.254465f, 0.0234559f, 0.897585f, 0.281481f, 
		0.0234559f, 0.897585f, 0.281481f, 0.0234713f, 1.03132f, -0.246725f, 
		0.0234713f, 1.03132f, -0.246725f, 0.0234713f, 1.03132f, -0.246725f, 
		0.0235025f, 1.01760f, -0.265472f, 0.0235025f, 1.01760f, -0.265472f, 
		0.0235025f, 1.01760f, -0.265472f, 0.0235224f, 1.01221f, 0.279880f, 
		0.0235375f, 1.30200f, 0.00763358f, 0.0235375f, 1.30200f, 0.00763358f, 
		0.0236598f, 1.04696f, -0.373558f, 0.0236598f, 1.04696f, -0.373558f, 
		0.0236619f, 1.04696f, -0.364739f, 0.0236619f, 1.04696f, -0.364739f, 
		0.0236764f, 0.901703f, 0.267973f, 0.0236764f, 0.901703f, 0.267973f, 
		0.0237388f, 1.30677f, -7.02993e-006f, 0.0237388f, 1.30677f, -7.02993e-006f, 
		0.0237674f, 1.31567f, -4.60607e-005f, 0.0237674f, 1.31567f, -4.60607e-005f, 
		0.0238158f, 1.04695f, 0.364637f, 0.0238158f, 1.04695f, 0.364637f, 
		0.0238179f, 1.04695f, 0.373455f, 0.0238179f, 1.04695f, 0.373455f, 
		0.0238530f, 1.23705f, -0.0495600f, 0.0238530f, 1.23705f, -0.0495600f, 
		0.0238530f, 1.26705f, -0.0495600f, 0.0238530f, 1.26705f, -0.0495600f, 
		0.0238740f, 1.23705f, 0.0495460f, 0.0238740f, 1.23705f, 0.0495460f, 
		0.0238740f, 1.26705f, 0.0495460f, 0.0238740f, 1.26705f, 0.0495460f, 
		0.0239380f, 1.04604f, -0.0243360f, 0.0239380f, 1.04604f, -0.0243360f, 
		0.0239480f, 1.04604f, 0.0243220f, 0.0239480f, 1.04604f, 0.0243220f, 
		0.0240680f, 0.896718f, -0.282015f, 0.0240680f, 0.896718f, -0.282015f, 
		0.0240740f, 0.896718f, -0.254155f, 0.0240740f, 0.896718f, -0.254155f, 
		0.0242380f, 1.00796f, -0.276377f, 0.0242380f, 1.00796f, -0.276377f, 
		0.0242380f, 1.00796f, -0.276377f, 0.0242817f, 1.00865f, -0.265197f, 
		0.0243113f, 1.00044f, 0.307131f, 0.0243113f, 1.00044f, 0.307131f, 
		0.0243156f, 0.980923f, -0.248903f, 0.0243546f, 1.00796f, 0.276266f, 
		0.0243546f, 1.00796f, 0.276266f, 0.0243546f, 1.00796f, 0.276266f, 
		0.0243546f, 1.00796f, 0.276266f, 0.0243546f, 1.00796f, 0.276266f, 
		0.0243945f, 1.29743f, -0.0267171f, 0.0244051f, 1.29743f, 0.0267027f, 
		0.0244945f, 1.04610f, -0.0202744f, 0.0244945f, 1.04610f, -0.0202744f, 
		0.0244945f, 1.04610f, -0.0202744f, 0.0244945f, 1.04610f, -0.0202744f, 
		0.0244945f, 1.04610f, -0.0202744f, 0.0244945f, 1.04610f, -0.0202744f, 
		0.0244945f, 1.04610f, -0.0202744f, 0.0245040f, 1.55393f, -0.0375997f, 
		0.0245040f, 1.55393f, -0.0375997f, 0.0245114f, 1.04594f, 0.0198801f, 
		0.0245114f, 1.04594f, 0.0198801f, 0.0245114f, 1.04594f, 0.0198801f, 
		0.0245114f, 1.04594f, 0.0198801f, 0.0245114f, 1.04594f, 0.0198801f, 
		0.0245114f, 1.04594f, 0.0198801f, 0.0245114f, 1.04594f, 0.0198801f, 
		0.0245114f, 1.04594f, 0.0198801f, 0.0245114f, 1.04594f, 0.0198801f, 
		0.0245746f, 1.00045f, -0.306415f, 0.0245746f, 1.00045f, -0.306415f, 
		0.0245899f, 1.00045f, -0.235142f, 0.0245899f, 1.00045f, -0.235142f, 
		0.0245930f, 1.04695f, -0.371329f, 0.0245930f, 1.04695f, -0.371329f, 
		0.0246710f, 1.31336f, -7.00000e-006f, 0.0246710f, 1.31336f, -7.00000e-006f, 
		0.0246892f, 1.00045f, 0.235030f, 0.0246892f, 1.00045f, 0.235030f, 
		0.0247500f, 1.04695f, 0.371226f, 0.0247500f, 1.04695f, 0.371226f, 
		0.0247822f, 1.04695f, -0.368422f, 0.0247822f, 1.04695f, -0.368422f, 
		0.0248063f, 1.20667f, -0.0263347f, 0.0248173f, 1.20667f, 0.0263195f, 
		0.0248606f, 1.31045f, -7.13691e-006f, 0.0248606f, 1.31045f, -7.13691e-006f, 
		0.0249386f, 1.04695f, 0.368319f, 0.0249386f, 1.04695f, 0.368319f, 
		0.0249530f, 0.836885f, -0.268085f, 0.0249530f, 0.836885f, -0.268085f, 
		0.0249870f, 0.948656f, -0.351244f, 0.0249877f, 0.985827f, -0.246974f, 
		0.0250660f, 0.836885f, 0.267973f, 0.0250660f, 0.836885f, 0.267973f, 
		0.0251171f, 1.55685f, -0.0605601f, 0.0251360f, 0.948656f, 0.351132f, 
		0.0251544f, 0.899571f, -0.268085f, 0.0251544f, 0.899571f, -0.268085f, 
		0.0251560f, 1.57086f, -0.0873950f, 0.0251910f, 1.57086f, 0.0758340f, 
		0.0252770f, 0.977046f, 0.251378f, 0.0253861f, 1.01338f, 0.268779f, 
		0.0253861f, 1.01338f, 0.268779f, 0.0253861f, 1.01338f, 0.268779f, 
		0.0255574f, 0.985827f, -0.247588f, 0.0257260f, 0.951668f, -0.353641f, 
		0.0257950f, 1.03132f, 0.249005f, 0.0257950f, 1.03132f, 0.249005f, 
		0.0257950f, 1.03132f, 0.249005f, 0.0257950f, 1.03132f, 0.249005f, 
		0.0258470f, 1.03111f, -0.0151180f, 0.0258470f, 1.03111f, -0.0151180f, 
		0.0258470f, 1.06096f, -0.0151180f, 0.0258470f, 1.06096f, -0.0151180f, 
		0.0258518f, 0.984601f, 0.290397f, 0.0258518f, 0.984601f, 0.290397f, 
		0.0258530f, 1.03111f, 0.0151030f, 0.0258530f, 1.03111f, 0.0151030f, 
		0.0258530f, 1.06096f, 0.0151030f, 0.0258530f, 1.06096f, 0.0151030f, 
		0.0258760f, 0.951668f, 0.353529f, 0.0258850f, 0.965681f, -0.216150f, 
		0.0259770f, 0.965681f, 0.216037f, 0.0263750f, 1.57721f, -0.0913490f, 
		0.0264110f, 1.57721f, 0.0797880f, 0.0265897f, 0.989731f, -0.299207f, 
		0.0266149f, 1.55398f, -0.0406876f, 0.0266149f, 1.55398f, -0.0406876f, 
		0.0266550f, 0.969213f, -0.214553f, 0.0266940f, 0.893445f, -0.283532f, 
		0.0266940f, 0.893445f, -0.283532f, 0.0267000f, 0.893445f, -0.252639f, 
		0.0267000f, 0.893445f, -0.252639f, 0.0267021f, 0.984609f, 0.254438f, 
		0.0267021f, 0.984609f, 0.254438f, 0.0267041f, 0.989730f, 0.242238f, 
		0.0267159f, 0.989730f, 0.299094f, 0.0267460f, 0.969213f, 0.214440f, 
		0.0268070f, 0.893445f, 0.252526f, 0.0268070f, 0.893445f, 0.252526f, 
		0.0268140f, 0.893445f, 0.283418f, 0.0268140f, 0.893445f, 0.283418f, 
		0.0268971f, 1.02272f, 0.250387f, 0.0268971f, 1.02272f, 0.250387f, 
		0.0269650f, 0.843081f, -0.283689f, 0.0269650f, 0.843081f, -0.283689f, 
		0.0269720f, 0.843081f, -0.252482f, 0.0269720f, 0.843081f, -0.252482f, 
		0.0270731f, 0.897585f, 0.267972f, 0.0270731f, 0.897585f, 0.267972f, 
		0.0270790f, 0.843081f, 0.252369f, 0.0270790f, 0.843081f, 0.252369f, 
		0.0270850f, 0.843081f, 0.283575f, 0.0270850f, 0.843081f, 0.283575f, 
		0.0272692f, 1.22218f, -0.0400792f, 0.0272850f, 1.22218f, 0.0400634f, 
		0.0273278f, 1.28192f, -0.0400392f, 0.0273454f, 1.28192f, 0.0400223f, 
		0.0273829f, 1.58880f, -0.0929426f, 0.0273829f, 1.58880f, -0.0929426f, 
		0.0274205f, 1.58880f, 0.0813801f, 0.0274205f, 1.58880f, 0.0813801f, 
		0.0278030f, 0.896718f, -0.268086f, 0.0278030f, 0.896718f, -0.268086f, 
		0.0278350f, 0.980877f, 0.253844f, 0.0278429f, 0.993264f, -0.240740f, 
		0.0279125f, 1.20316f, -0.00961926f, 0.0280288f, 1.30070f, -0.0107840f, 
		0.0280383f, 1.26712f, -0.0416621f, 0.0280383f, 1.26712f, -0.0416621f, 
		0.0280568f, 1.26712f, 0.0416461f, 0.0280568f, 1.26712f, 0.0416461f, 
		0.0282280f, 1.02974f, -0.0148290f, 0.0282280f, 1.02974f, -0.0148290f, 
		0.0282280f, 1.06234f, -0.0148290f, 0.0282280f, 1.06234f, -0.0148290f, 
		0.0282340f, 1.02974f, 0.0148130f, 0.0282340f, 1.02974f, 0.0148130f, 
		0.0282340f, 1.06234f, 0.0148130f, 0.0282340f, 1.06234f, 0.0148130f, 
		0.0283859f, 1.20497f, 0.0170647f, 0.0284776f, 0.980923f, -0.284976f, 
		0.0285184f, 1.23699f, -0.0413349f, 0.0285184f, 1.23699f, -0.0413349f, 
		0.0285360f, 1.23699f, 0.0413191f, 0.0285360f, 1.23699f, 0.0413191f, 
		0.0286228f, 1.02272f, -0.288361f, 0.0286228f, 1.02272f, -0.288361f, 
		0.0286300f, 1.02272f, -0.253197f, 0.0286300f, 1.02272f, -0.253197f, 
		0.0286377f, 1.20366f, 0.0102611f, 0.0286608f, 1.55470f, -0.0454445f, 
		0.0286769f, 1.55470f, 0.0338818f, 0.0287381f, 1.02272f, 0.253084f, 
		0.0287381f, 1.02272f, 0.253084f, 0.0287459f, 1.02272f, 0.288244f, 
		0.0287459f, 1.02272f, 0.288244f, 0.0289130f, 1.02272f, 0.287323f, 
		0.0289130f, 1.02272f, 0.287323f, 0.0289130f, 1.03132f, 0.287323f, 
		0.0289130f, 1.03132f, 0.287323f, 0.0289802f, 1.29913f, 0.0160337f, 
		0.0289900f, 0.889845f, -0.284858f, 0.0289900f, 0.889845f, -0.284858f, 
		0.0289970f, 0.889845f, -0.251314f, 0.0289970f, 0.889845f, -0.251314f, 
		0.0290376f, 1.55393f, 0.0201729f, 0.0290376f, 1.55393f, 0.0201729f, 
		0.0291030f, 0.889845f, 0.251200f, 0.0291030f, 0.889845f, 0.251200f, 
		0.0291110f, 0.889845f, 0.284744f, 0.0291110f, 0.889845f, 0.284744f, 
		0.0291210f, 1.55863f, 0.0535254f, 0.0291961f, 1.00045f, -0.302731f, 
		0.0291961f, 1.00045f, -0.302731f, 0.0292097f, 1.00045f, -0.238828f, 
		0.0292097f, 1.00045f, -0.238828f, 0.0293096f, 1.00045f, 0.238713f, 
		0.0293096f, 1.00045f, 0.238713f, 0.0294840f, 1.58893f, -0.101434f, 
		0.0294840f, 1.58893f, -0.101434f, 0.0294840f, 1.61893f, -0.101434f, 
		0.0294840f, 1.61893f, -0.101434f, 0.0294840f, 1.61893f, -0.101434f, 
		0.0295025f, 0.977046f, -0.259748f, 0.0295068f, 1.03132f, -0.286830f, 
		0.0295068f, 1.03132f, -0.286830f, 0.0295068f, 1.03132f, -0.286830f, 
		0.0295240f, 1.58893f, 0.0898710f, 0.0295240f, 1.58893f, 0.0898710f, 
		0.0295240f, 1.61893f, 0.0898710f, 0.0295240f, 1.61893f, 0.0898710f, 
		0.0295240f, 1.61893f, 0.0898710f, 0.0296220f, 1.03132f, 0.254614f, 
		0.0296220f, 1.03132f, 0.254614f, 0.0296220f, 1.03132f, 0.254614f, 
		0.0296292f, 1.03132f, 0.286714f, 0.0296292f, 1.03132f, 0.286714f, 
		0.0296292f, 1.03132f, 0.286714f, 0.0298460f, 1.04604f, -0.0151190f, 
		0.0298460f, 1.04604f, -0.0151190f, 0.0298520f, 1.04604f, 0.0151020f, 
		0.0298520f, 1.04604f, 0.0151020f, 0.0299210f, 0.980877f, 0.281603f, 
		0.0301365f, 0.977046f, -0.278180f, 0.0302180f, 1.00044f, 0.302418f, 
		0.0302180f, 1.00044f, 0.302418f, 0.0302536f, 0.977046f, 0.278067f, 
		0.0302780f, 0.989730f, -0.295233f, 0.0303923f, 0.989731f, 0.246210f, 
		0.0304030f, 0.989731f, 0.295119f, 0.0304130f, 0.977001f, 0.262954f, 
		0.0304150f, 1.02848f, -0.0139470f, 0.0304150f, 1.02848f, -0.0139470f, 
		0.0304150f, 1.06360f, -0.0139470f, 0.0304150f, 1.06360f, -0.0139470f, 
		0.0304210f, 1.02848f, 0.0139300f, 0.0304210f, 1.02848f, 0.0139300f, 
		0.0304210f, 1.06360f, 0.0139300f, 0.0304210f, 1.06360f, 0.0139300f, 
		0.0306127f, 1.29913f, 0.0126432f, 0.0308360f, 0.893445f, -0.268086f, 
		0.0308360f, 0.893445f, -0.268086f, 0.0309111f, 1.28799f, -0.0334368f, 
		0.0309244f, 1.03132f, -0.283889f, 0.0309244f, 1.03132f, -0.283889f, 
		0.0309244f, 1.03132f, -0.283889f, 0.0309244f, 1.03132f, -0.283889f, 
		0.0309263f, 1.28799f, 0.0334201f, 0.0309298f, 1.03132f, -0.257670f, 
		0.0309298f, 1.03132f, -0.257670f, 0.0309298f, 1.03132f, -0.257670f, 
		0.0309300f, 0.885952f, -0.285979f, 0.0309300f, 0.885952f, -0.285979f, 
		0.0309370f, 0.885952f, -0.250194f, 0.0309370f, 0.885952f, -0.250194f, 
		0.0309490f, 0.893445f, 0.267971f, 0.0309490f, 0.893445f, 0.267971f, 
		0.0309730f, 1.23705f, -0.0454520f, 0.0309730f, 1.23705f, -0.0454520f, 
		0.0309730f, 1.26705f, -0.0454520f, 0.0309730f, 1.26705f, -0.0454520f, 
		0.0309920f, 1.23705f, 0.0454350f, 0.0309920f, 1.23705f, 0.0454350f, 
		0.0309920f, 1.26705f, 0.0454350f, 0.0309920f, 1.26705f, 0.0454350f, 
		0.0310071f, 0.984607f, -0.268982f, 0.0310071f, 0.984607f, -0.268982f, 
		0.0310172f, 1.21611f, -0.0333382f, 0.0310312f, 1.21611f, 0.0333227f, 
		0.0310430f, 0.885952f, 0.250079f, 0.0310430f, 0.885952f, 0.250079f, 
		0.0310510f, 0.885952f, 0.285864f, 0.0310510f, 0.885952f, 0.285864f, 
		0.0310570f, 0.850508f, -0.286052f, 0.0310570f, 0.850508f, -0.286052f, 
		0.0310640f, 0.850508f, -0.250121f, 0.0310640f, 0.850508f, -0.250121f, 
		0.0311490f, 0.843081f, -0.268086f, 0.0311490f, 0.843081f, -0.268086f, 
		0.0311555f, 0.984593f, -0.264164f, 0.0311555f, 0.984593f, -0.264164f, 
		0.0311555f, 0.984593f, -0.264164f, 0.0311700f, 0.850508f, 0.250006f, 
		0.0311700f, 0.850508f, 0.250006f, 0.0311780f, 0.850508f, 0.285937f, 
		0.0311780f, 0.850508f, 0.285937f, 0.0311797f, 0.984606f, 0.270200f, 
		0.0311797f, 0.984606f, 0.270200f, 0.0311940f, 0.927433f, -0.371382f, 
		0.0312476f, 0.985827f, -0.285391f, 0.0312630f, 0.843081f, 0.267971f, 
		0.0312630f, 0.843081f, 0.267971f, 0.0313510f, 0.927433f, 0.371267f, 
		0.0313694f, 0.985827f, 0.285275f, 0.0315703f, 1.02272f, -0.282244f, 
		0.0315703f, 1.02272f, -0.282244f, 0.0315750f, 1.02272f, -0.259316f, 
		0.0315750f, 1.02272f, -0.259316f, 0.0319600f, 0.980923f, -0.266297f, 
		0.0319881f, 0.993266f, -0.245209f, 0.0320736f, 0.980923f, 0.266185f, 
		0.0321140f, 0.929791f, -0.374367f, 0.0321521f, 1.20497f, -0.00795731f, 
		0.0322532f, 1.21087f, -0.0259101f, 0.0322624f, 1.21087f, 0.0258933f, 
		0.0322730f, 0.929791f, 0.374252f, 0.0323450f, 1.02736f, -0.0124550f, 
		0.0323450f, 1.02736f, -0.0124550f, 0.0323450f, 1.06471f, -0.0124550f, 
		0.0323450f, 1.06471f, -0.0124550f, 0.0323500f, 1.02736f, 0.0124370f, 
		0.0323500f, 1.02736f, 0.0124370f, 0.0323500f, 1.06471f, 0.0124370f, 
		0.0323500f, 1.06471f, 0.0124370f, 0.0324167f, 1.29913f, -0.00679824f, 
		0.0324251f, 1.29323f, -0.0256946f, 0.0324358f, 1.29323f, 0.0256761f, 
		0.0324860f, 0.881798f, -0.286878f, 0.0324860f, 0.881798f, -0.286878f, 
		0.0324940f, 0.881798f, -0.249296f, 0.0324940f, 0.881798f, -0.249296f, 
		0.0325950f, 1.04604f, -0.0148300f, 0.0325950f, 1.04604f, -0.0148300f, 
		0.0325990f, 0.881798f, 0.249180f, 0.0325990f, 0.881798f, 0.249180f, 
		0.0326020f, 1.04604f, 0.0148120f, 0.0326020f, 1.04604f, 0.0148120f, 
		0.0326070f, 0.881798f, 0.286762f, 0.0326070f, 0.881798f, 0.286762f, 
		0.0328052f, 1.55393f, 0.0173990f, 0.0328052f, 1.55393f, 0.0173990f, 
		0.0329293f, 1.03132f, -0.264158f, 0.0329293f, 1.03132f, -0.264158f, 
		0.0329293f, 1.03132f, -0.264158f, 0.0329783f, 1.56290f, -0.0737394f, 
		0.0331640f, 1.02272f, -0.273270f, 0.0331640f, 1.02272f, -0.273270f, 
		0.0331640f, 1.03132f, -0.273270f, 0.0331640f, 1.03132f, -0.273270f, 
		0.0331640f, 1.03132f, -0.273270f, 0.0331650f, 1.02272f, -0.268290f, 
		0.0331650f, 1.02272f, -0.268290f, 0.0334580f, 1.22533f, -0.0360760f, 
		0.0334580f, 1.27877f, -0.0360760f, 0.0334730f, 1.22533f, 0.0360580f, 
		0.0334730f, 1.27877f, 0.0360580f, 0.0334880f, 0.889845f, -0.268087f, 
		0.0334880f, 0.889845f, -0.268087f, 0.0335288f, 1.03132f, 0.267268f, 
		0.0335288f, 1.03132f, 0.267268f, 0.0335288f, 1.03132f, 0.267268f, 
		0.0335302f, 1.03132f, 0.274059f, 0.0335302f, 1.03132f, 0.274059f, 
		0.0335302f, 1.03132f, 0.274059f, 0.0335302f, 1.03132f, 0.274059f, 
		0.0335990f, 0.965681f, -0.319379f, 0.0336010f, 0.889845f, 0.267971f, 
		0.0336010f, 0.889845f, 0.267971f, 0.0336310f, 0.877414f, -0.287539f, 
		0.0336310f, 0.877414f, -0.287539f, 0.0336390f, 0.877414f, -0.248635f, 
		0.0336390f, 0.877414f, -0.248635f, 0.0336613f, 1.02272f, 0.269031f, 
		0.0336613f, 1.02272f, 0.269031f, 0.0336618f, 1.02272f, 0.272296f, 
		0.0336618f, 1.02272f, 0.272296f, 0.0336640f, 0.858926f, -0.287558f, 
		0.0336640f, 0.858926f, -0.287558f, 0.0336730f, 0.858926f, -0.248616f, 
		0.0336730f, 0.858926f, -0.248616f, 0.0337340f, 0.965681f, 0.319263f, 
		0.0337450f, 0.877414f, 0.248519f, 0.0337450f, 0.877414f, 0.248519f, 
		0.0337530f, 0.877414f, 0.287423f, 0.0337530f, 0.877414f, 0.287423f, 
		0.0337780f, 0.858926f, 0.248500f, 0.0337780f, 0.858926f, 0.248500f, 
		0.0337860f, 0.858926f, 0.287442f, 0.0337860f, 0.858926f, 0.287442f, 
		0.0337877f, 0.989732f, -0.290090f, 0.0337963f, 0.989732f, -0.251471f, 
		0.0339012f, 0.989730f, 0.251356f, 0.0339098f, 0.989731f, 0.289972f, 
		0.0339350f, 1.23699f, -0.0370177f, 0.0339350f, 1.23699f, -0.0370177f, 
		0.0339499f, 1.23699f, 0.0369988f, 0.0339499f, 1.23699f, 0.0369988f, 
		0.0339550f, 1.02643f, -0.0103330f, 0.0339550f, 1.02643f, -0.0103330f, 
		0.0339550f, 1.06564f, -0.0103330f, 0.0339550f, 1.06564f, -0.0103330f, 
		0.0339600f, 1.02643f, 0.0103150f, 0.0339600f, 1.02643f, 0.0103150f, 
		0.0339600f, 1.06564f, 0.0103150f, 0.0339600f, 1.06564f, 0.0103150f, 
		0.0339861f, 1.00045f, -0.297571f, 0.0339861f, 1.00045f, -0.297571f, 
		0.0341003f, 1.00045f, 0.243874f, 0.0341003f, 1.00045f, 0.243874f, 
		0.0342540f, 1.56525f, 0.0653190f, 0.0343380f, 0.872835f, -0.287947f, 
		0.0343380f, 0.872835f, -0.287947f, 0.0343470f, 0.872835f, -0.248227f, 
		0.0343470f, 0.872835f, -0.248227f, 0.0343603f, 1.26712f, -0.0366233f, 
		0.0343603f, 1.26712f, -0.0366233f, 0.0343759f, 1.26712f, 0.0366034f, 
		0.0343759f, 1.26712f, 0.0366034f, 0.0344520f, 0.872835f, 0.248110f, 
		0.0344520f, 0.872835f, 0.248110f, 0.0344600f, 0.872835f, 0.287831f, 
		0.0344600f, 0.872835f, 0.287831f, 0.0345800f, 0.868091f, -0.288087f, 
		0.0345800f, 0.868091f, -0.288087f, 0.0345890f, 0.868091f, -0.248087f, 
		0.0345890f, 0.868091f, -0.248087f, 0.0345980f, 0.969213f, -0.320844f, 
		0.0346940f, 0.868091f, 0.247971f, 0.0346940f, 0.868091f, 0.247971f, 
		0.0347020f, 0.868091f, 0.287971f, 0.0347020f, 0.868091f, 0.287971f, 
		0.0347330f, 0.969213f, 0.320727f, 0.0348651f, 1.55685f, -0.0549334f, 
		0.0348759f, 0.986226f, 0.267622f, 0.0351210f, 1.04604f, -0.0139480f, 
		0.0351210f, 1.04604f, -0.0139480f, 0.0351260f, 1.04604f, 0.0139290f, 
		0.0351260f, 1.04604f, 0.0139290f, 0.0351830f, 1.02572f, -0.00756400f, 
		0.0351830f, 1.02572f, -0.00756400f, 0.0351830f, 1.06635f, -0.00756400f, 
		0.0351830f, 1.06635f, -0.00756400f, 0.0351870f, 1.02572f, 0.00754500f, 
		0.0351870f, 1.02572f, 0.00754500f, 0.0351870f, 1.06635f, 0.00754500f, 
		0.0351870f, 1.06635f, 0.00754500f, 0.0357280f, 0.885952f, -0.268087f, 
		0.0357280f, 0.885952f, -0.268087f, 0.0357539f, 1.29323f, -0.0208134f, 
		0.0357619f, 1.29323f, 0.0207943f, 0.0358410f, 0.885952f, 0.267970f, 
		0.0358410f, 0.885952f, 0.267970f, 0.0358740f, 0.850508f, -0.268087f, 
		0.0358740f, 0.850508f, -0.268087f, 0.0358915f, 1.21087f, -0.0205749f, 
		0.0359007f, 1.21087f, 0.0205550f, 0.0359270f, 0.903517f, -0.386735f, 
		0.0359670f, 1.02527f, -0.00412900f, 0.0359670f, 1.02527f, -0.00412900f, 
		0.0359670f, 1.06680f, -0.00412900f, 0.0359670f, 1.06680f, -0.00412900f, 
		0.0359680f, 1.02527f, 0.00411000f, 0.0359680f, 1.02527f, 0.00411000f, 
		0.0359680f, 1.06680f, 0.00411000f, 0.0359680f, 1.06680f, 0.00411000f, 
		0.0359880f, 0.850508f, 0.267970f, 0.0359880f, 0.850508f, 0.267970f, 
		0.0360528f, 1.20667f, 0.00297481f, 0.0360900f, 0.903517f, 0.386618f, 
		0.0360956f, 1.29743f, 0.00241296f, 0.0361465f, 0.989731f, -0.256356f, 
		0.0362420f, 1.02511f, -1.00000e-005f, 0.0362420f, 1.02511f, -1.00000e-005f, 
		0.0362420f, 1.06696f, -1.00000e-005f, 0.0362420f, 1.06696f, -1.00000e-005f, 
		0.0362560f, 0.989732f, 0.256239f, 0.0368178f, 1.55863f, 0.0490801f, 
		0.0368500f, 0.948656f, -0.193394f, 0.0369320f, 0.948656f, 0.193277f, 
		0.0369850f, 0.905134f, -0.390170f, 0.0370400f, 1.57086f, -0.0827340f, 
		0.0370720f, 1.57086f, 0.0711670f, 0.0371500f, 0.905134f, 0.390053f, 
		0.0373155f, 1.00045f, -0.292688f, 0.0373155f, 1.00045f, -0.292688f, 
		0.0373252f, 1.00045f, -0.248873f, 0.0373252f, 1.00045f, -0.248873f, 
		0.0373490f, 1.04604f, -0.0124560f, 0.0373490f, 1.04604f, -0.0124560f, 
		0.0373540f, 1.04604f, 0.0124360f, 0.0373540f, 1.04604f, 0.0124360f, 
		0.0374010f, 1.23705f, -0.0403280f, 0.0374010f, 1.23705f, -0.0403280f, 
		0.0374010f, 1.26705f, -0.0403280f, 0.0374010f, 1.26705f, -0.0403280f, 
		0.0374180f, 1.23705f, 0.0403080f, 0.0374180f, 1.23705f, 0.0403080f, 
		0.0374180f, 1.26705f, 0.0403080f, 0.0374180f, 1.26705f, 0.0403080f, 
		0.0374310f, 1.00045f, 0.248756f, 0.0374310f, 1.00045f, 0.248756f, 
		0.0374407f, 1.00045f, 0.292570f, 0.0374407f, 1.00045f, 0.292570f, 
		0.0375250f, 0.881798f, -0.268088f, 0.0375250f, 0.881798f, -0.268088f, 
		0.0376380f, 0.881798f, 0.267970f, 0.0376380f, 0.881798f, 0.267970f, 
		0.0378706f, 1.22218f, -0.0302618f, 0.0378841f, 1.22218f, 0.0302411f, 
		0.0379147f, 1.28192f, -0.0302060f, 0.0379277f, 1.28192f, 0.0301863f, 
		0.0379390f, 0.951668f, -0.191134f, 0.0379766f, 0.989730f, -0.279254f, 
		0.0379766f, 0.989730f, -0.279254f, 0.0379766f, 0.989730f, -0.279254f, 
		0.0379766f, 0.989730f, -0.279254f, 0.0379766f, 0.989730f, -0.279254f, 
		0.0379801f, 0.989730f, -0.262307f, 0.0380200f, 0.951668f, 0.191016f, 
		0.0380910f, 0.993262f, -0.285772f, 0.0380922f, 0.989732f, 0.262188f, 
		0.0380922f, 0.989732f, 0.262188f, 0.0380922f, 0.989732f, 0.262188f, 
		0.0380971f, 0.989733f, 0.279138f, 0.0380971f, 0.989733f, 0.279138f, 
		0.0380971f, 0.989733f, 0.279138f, 0.0381209f, 1.00044f, -0.249326f, 
		0.0381209f, 1.00044f, -0.249326f, 0.0382149f, 0.993265f, 0.285654f, 
		0.0384577f, 1.21088f, -0.0152531f, 0.0384642f, 1.21088f, 0.0152322f, 
		0.0385579f, 1.29323f, -0.0149967f, 0.0385648f, 1.29323f, 0.0149775f, 
		0.0387838f, 0.989728f, -0.273895f, 0.0387878f, 0.989731f, -0.267667f, 
		0.0388350f, 1.57721f, -0.0864620f, 0.0388480f, 0.877414f, -0.268088f, 
		0.0388480f, 0.877414f, -0.268088f, 0.0388690f, 1.57721f, 0.0748950f, 
		0.0388860f, 0.858926f, -0.268088f, 0.0388860f, 0.858926f, -0.268088f, 
		0.0389014f, 0.989731f, 0.273777f, 0.0389430f, 0.878415f, -0.396522f, 
		0.0389610f, 0.877414f, 0.267970f, 0.0389610f, 0.877414f, 0.267970f, 
		0.0389990f, 0.858926f, 0.267970f, 0.0389990f, 0.858926f, 0.267970f, 
		0.0390727f, 1.26712f, -0.0315465f, 0.0390727f, 1.26712f, -0.0315465f, 
		0.0390855f, 1.26712f, 0.0315259f, 0.0390855f, 1.26712f, 0.0315259f, 
		0.0391110f, 0.878415f, 0.396403f, 0.0391788f, 1.58880f, -0.0883164f, 
		0.0391788f, 1.58880f, -0.0883164f, 0.0392090f, 1.04604f, -0.0103340f, 
		0.0392090f, 1.04604f, -0.0103340f, 0.0392130f, 1.04604f, 0.0103140f, 
		0.0392130f, 1.04604f, 0.0103140f, 0.0392147f, 1.58880f, 0.0767492f, 
		0.0392147f, 1.58880f, 0.0767492f, 0.0394347f, 1.23699f, -0.0310927f, 
		0.0394347f, 1.23699f, -0.0310927f, 0.0394480f, 1.23699f, 0.0310711f, 
		0.0394480f, 1.23699f, 0.0310711f, 0.0396640f, 0.872835f, -0.268088f, 
		0.0396640f, 0.872835f, -0.268088f, 0.0397780f, 0.872835f, 0.267969f, 
		0.0397780f, 0.872835f, 0.267969f, 0.0399430f, 0.868091f, -0.268088f, 
		0.0399430f, 0.868091f, -0.268088f, 0.0400020f, 0.853634f, -0.399958f, 
		0.0400020f, 0.853634f, -0.399958f, 0.0400570f, 0.868091f, 0.267969f, 
		0.0400570f, 0.868091f, 0.267969f, 0.0400900f, 0.879246f, -0.400243f, 
		0.0401710f, 0.853634f, 0.399839f, 0.0401710f, 0.853634f, 0.399839f, 
		0.0402590f, 0.879246f, 0.400124f, 0.0402980f, 1.29323f, -0.00935156f, 
		0.0403061f, 1.29322f, 0.00932955f, 0.0403600f, 1.21087f, -0.00908105f, 
		0.0403672f, 1.21088f, 0.00906222f, 0.0403724f, 1.00045f, -0.286346f, 
		0.0403724f, 1.00045f, -0.286346f, 0.0404933f, 1.00045f, 0.286226f, 
		0.0404933f, 1.00045f, 0.286226f, 0.0406030f, 0.965681f, -0.226189f, 
		0.0406270f, 1.04604f, -0.00756500f, 0.0406270f, 1.04604f, -0.00756500f, 
		0.0406300f, 1.04604f, 0.00754400f, 0.0406300f, 1.04604f, 0.00754400f, 
		0.0406980f, 0.965681f, 0.226070f, 0.0409339f, 0.993265f, 0.267584f, 
		0.0411800f, 0.853677f, -0.403780f, 0.0411800f, 0.853677f, -0.403780f, 
		0.0412451f, 1.21088f, -0.00323918f, 0.0412659f, 1.29322f, -0.00296469f, 
		0.0413510f, 0.853677f, 0.403661f, 0.0413510f, 0.853677f, 0.403661f, 
		0.0415310f, 1.04604f, -0.00413000f, 0.0415310f, 1.04604f, -0.00413000f, 
		0.0415320f, 1.04604f, 0.00410900f, 0.0415320f, 1.04604f, 0.00410900f, 
		0.0418090f, 0.969213f, -0.224890f, 0.0418490f, 1.04604f, -1.10000e-005f, 
		0.0418490f, 1.04604f, -1.10000e-005f, 0.0419040f, 0.969213f, 0.224770f, 
		0.0419581f, 1.28192f, -0.0242785f, 0.0419669f, 1.28192f, 0.0242576f, 
		0.0419936f, 1.22218f, -0.0242179f, 0.0420027f, 1.22218f, 0.0241952f, 
		0.0421156f, 1.00045f, -0.280698f, 0.0421156f, 1.00045f, -0.280698f, 
		0.0421194f, 1.00045f, -0.260866f, 0.0421194f, 1.00045f, -0.260866f, 
		0.0422290f, 1.00045f, 0.260744f, 0.0422290f, 1.00045f, 0.260744f, 
		0.0422340f, 1.00045f, 0.280580f, 0.0422340f, 1.00045f, 0.280580f, 
		0.0423407f, 1.56290f, -0.0683366f, 0.0428566f, 1.00044f, 0.261411f, 
		0.0428566f, 1.00044f, 0.261411f, 0.0429930f, 1.23705f, -0.0343030f, 
		0.0429930f, 1.23705f, -0.0343030f, 0.0429930f, 1.26705f, -0.0343030f, 
		0.0429930f, 1.26705f, -0.0343030f, 0.0430080f, 1.23705f, 0.0342810f, 
		0.0430080f, 1.23705f, 0.0342810f, 0.0430080f, 1.26705f, 0.0342810f, 
		0.0430080f, 1.26705f, 0.0342810f, 0.0431661f, 1.00045f, -0.273737f, 
		0.0431661f, 1.00045f, -0.273737f, 0.0431674f, 1.00045f, -0.267827f, 
		0.0431674f, 1.00045f, -0.267827f, 0.0432818f, 1.00045f, 0.273617f, 
		0.0432818f, 1.00045f, 0.273617f, 0.0433375f, 1.23699f, -0.0253707f, 
		0.0433375f, 1.23699f, -0.0253707f, 0.0433491f, 1.23699f, 0.0253480f, 
		0.0433491f, 1.23699f, 0.0253480f, 0.0434110f, 1.58893f, -0.0959710f, 
		0.0434110f, 1.58893f, -0.0959710f, 0.0434110f, 1.61893f, -0.0959710f, 
		0.0434110f, 1.61893f, -0.0959710f, 0.0434110f, 1.61893f, -0.0959710f, 
		0.0434500f, 1.58893f, 0.0844020f, 0.0434500f, 1.58893f, 0.0844020f, 
		0.0434500f, 1.61893f, 0.0844020f, 0.0434500f, 1.61893f, 0.0844020f, 
		0.0434500f, 1.61893f, 0.0844020f, 0.0436281f, 1.26712f, -0.0248675f, 
		0.0436281f, 1.26712f, -0.0248675f, 0.0436394f, 1.26712f, 0.0248452f, 
		0.0436394f, 1.26712f, 0.0248452f, 0.0444680f, 1.56525f, 0.0594190f, 
		0.0447580f, 1.55473f, -0.0233530f, 0.0447655f, 1.55473f, 0.0117835f, 
		0.0451077f, 1.22218f, -0.0177543f, 0.0451136f, 1.22218f, 0.0177322f, 
		0.0451332f, 1.28192f, -0.0176886f, 0.0451403f, 1.28192f, 0.0176654f, 
		0.0453627f, 1.55466f, 0.0140928f, 0.0454017f, 1.21611f, 0.00346432f, 
		0.0454121f, 1.28799f, 0.00331847f, 0.0459970f, 0.927433f, -0.174410f, 
		0.0460710f, 0.927433f, 0.174289f, 0.0466348f, 1.26712f, -0.0186281f, 
		0.0466348f, 1.26712f, -0.0186281f, 0.0466433f, 1.26712f, 0.0186035f, 
		0.0466433f, 1.26712f, 0.0186035f, 0.0466610f, 0.965681f, -0.307264f, 
		0.0467910f, 0.965681f, 0.307143f, 0.0468470f, 1.23699f, -0.0180878f, 
		0.0468470f, 1.23699f, -0.0180878f, 0.0468549f, 1.23699f, 0.0180638f, 
		0.0468549f, 1.23699f, 0.0180638f, 0.0472493f, 1.28192f, -0.0108333f, 
		0.0472529f, 1.28192f, 0.0108091f, 0.0472653f, 1.22218f, -0.0107634f, 
		0.0472689f, 1.22218f, 0.0107400f, 0.0473530f, 0.929791f, -0.171596f, 
		0.0473921f, 1.55470f, -0.0179832f, 0.0474260f, 0.929791f, 0.171474f, 
		0.0476260f, 1.23705f, -0.0275120f, 0.0476260f, 1.23705f, -0.0275120f, 
		0.0476260f, 1.26705f, -0.0275120f, 0.0476260f, 1.26705f, -0.0275120f, 
		0.0476370f, 1.23705f, 0.0274880f, 0.0476370f, 1.23705f, 0.0274880f, 
		0.0476370f, 1.26705f, 0.0274880f, 0.0476370f, 1.26705f, 0.0274880f, 
		0.0478191f, 1.55685f, 0.0309055f, 0.0478240f, 0.948656f, -0.340252f, 
		0.0479680f, 0.948656f, 0.340130f, 0.0480470f, 0.969213f, -0.308370f, 
		0.0480960f, 1.57086f, -0.0763530f, 0.0481260f, 1.57086f, 0.0647820f, 
		0.0481780f, 0.969213f, 0.308248f, 0.0483353f, 1.22218f, -0.00367092f, 
		0.0483403f, 1.28192f, -0.00359877f, 0.0488897f, 1.23699f, -0.0114689f, 
		0.0488897f, 1.23699f, -0.0114689f, 0.0488951f, 1.23699f, 0.0114452f, 
		0.0488951f, 1.23699f, 0.0114452f, 0.0490190f, 1.26712f, -0.0109031f, 
		0.0490190f, 1.26712f, -0.0109031f, 0.0490240f, 1.26712f, 0.0108793f, 
		0.0490240f, 1.26712f, 0.0108793f, 0.0490650f, 1.22533f, 0.00366500f, 
		0.0490650f, 1.27877f, 0.00366500f, 0.0492370f, 0.951668f, -0.342325f, 
		0.0493820f, 0.951668f, 0.342203f, 0.0500525f, 1.26712f, -0.00405477f, 
		0.0500525f, 1.26712f, -0.00405477f, 0.0500546f, 1.26712f, 0.00402999f, 
		0.0500546f, 1.26712f, 0.00402999f, 0.0500961f, 1.23699f, -0.00347522f, 
		0.0500961f, 1.23699f, -0.00347522f, 0.0500978f, 1.23699f, 0.00345063f, 
		0.0500978f, 1.23699f, 0.00345063f, 0.0504270f, 1.57721f, -0.0797730f, 
		0.0504580f, 1.57721f, 0.0682010f, 0.0511940f, 1.23705f, -0.0201070f, 
		0.0511940f, 1.23705f, -0.0201070f, 0.0511940f, 1.26705f, -0.0201070f, 
		0.0511940f, 1.26705f, -0.0201070f, 0.0512020f, 1.23705f, 0.0200810f, 
		0.0512020f, 1.23705f, 0.0200810f, 0.0512020f, 1.26705f, 0.0200810f, 
		0.0512020f, 1.26705f, 0.0200810f, 0.0515457f, 1.56289f, -0.0609965f, 
		0.0515699f, 1.56290f, 0.0494262f, 0.0516141f, 1.55685f, -0.0368904f, 
		0.0516269f, 1.55685f, 0.0253174f, 0.0517070f, 0.965681f, -0.240120f, 
		0.0518090f, 0.965681f, 0.239996f, 0.0518590f, 1.58880f, -0.0809988f, 
		0.0518590f, 1.58880f, -0.0809988f, 0.0518913f, 1.58880f, 0.0694265f, 
		0.0518913f, 1.58880f, 0.0694265f, 0.0529710f, 0.903517f, -0.159937f, 
		0.0530390f, 0.903517f, 0.159812f, 0.0532430f, 0.969213f, -0.239234f, 
		0.0533440f, 0.969213f, 0.239109f, 0.0536180f, 1.23705f, -0.0122520f, 
		0.0536180f, 1.23705f, -0.0122520f, 0.0536180f, 1.26705f, -0.0122520f, 
		0.0536180f, 1.26705f, -0.0122520f, 0.0536240f, 1.23705f, 0.0122250f, 
		0.0536240f, 1.23705f, 0.0122250f, 0.0536240f, 1.26705f, 0.0122250f, 
		0.0536240f, 1.26705f, 0.0122250f, 0.0545310f, 0.905134f, -0.156699f, 
		0.0545970f, 0.905134f, 0.156574f, 0.0548450f, 1.23705f, -0.00412400f, 
		0.0548450f, 1.23705f, -0.00412400f, 0.0548450f, 1.26705f, -0.00412400f, 
		0.0548450f, 1.26705f, -0.00412400f, 0.0548470f, 1.23705f, 0.00409700f, 
		0.0548470f, 1.23705f, 0.00409700f, 0.0548470f, 1.26705f, 0.00409700f, 
		0.0548470f, 1.26705f, 0.00409700f, 0.0555720f, 0.965681f, -0.291838f, 
		0.0556960f, 0.965681f, 0.291712f, 0.0563700f, 1.58893f, -0.0884940f, 
		0.0563700f, 1.58893f, -0.0884940f, 0.0563700f, 1.61893f, -0.0884940f, 
		0.0563700f, 1.61893f, -0.0884940f, 0.0563700f, 1.61893f, -0.0884940f, 
		0.0564050f, 1.58893f, 0.0769190f, 0.0564050f, 1.58893f, 0.0769190f, 
		0.0564050f, 1.61893f, 0.0769190f, 0.0564050f, 1.61893f, 0.0769190f, 
		0.0564050f, 1.61893f, 0.0769190f, 0.0564980f, 1.55685f, -0.0267528f, 
		0.0572220f, 0.969213f, -0.292486f, 0.0573460f, 0.969213f, 0.292360f, 
		0.0574160f, 0.878415f, -0.150711f, 0.0574800f, 0.878415f, 0.150585f, 
		0.0577880f, 0.948656f, -0.207676f, 0.0578760f, 0.948656f, 0.207550f, 
		0.0580780f, 1.57086f, -0.0683970f, 0.0581050f, 1.57086f, 0.0568210f, 
		0.0582120f, 0.965681f, -0.256705f, 0.0583210f, 0.965681f, 0.256578f, 
		0.0589000f, 1.56290f, -0.0530770f, 0.0589212f, 1.56290f, 0.0415019f, 
		0.0589770f, 0.853634f, -0.147472f, 0.0589770f, 0.853634f, -0.147472f, 
		0.0590390f, 0.853634f, 0.147345f, 0.0590390f, 0.853634f, 0.147345f, 
		0.0591070f, 0.879246f, -0.147203f, 0.0591690f, 0.879246f, 0.147076f, 
		0.0594940f, 0.951668f, -0.205837f, 0.0595400f, 0.965681f, -0.274470f, 
		0.0595810f, 0.951668f, 0.205710f, 0.0596560f, 0.965681f, 0.274343f, 
		0.0596910f, 0.927433f, -0.357666f, 0.0598430f, 0.927433f, 0.357539f, 
		0.0599410f, 0.969213f, -0.256310f, 0.0600490f, 0.969213f, 0.256183f, 
		0.0601720f, 1.55685f, -0.00240722f, 0.0607130f, 0.853677f, -0.143869f, 
		0.0607130f, 0.853677f, -0.143869f, 0.0607740f, 0.853677f, 0.143741f, 
		0.0607740f, 0.853677f, 0.143741f, 0.0608920f, 1.57721f, -0.0714310f, 
		0.0609200f, 1.57721f, 0.0598540f, 0.0613080f, 0.969213f, -0.274603f, 
		0.0614240f, 0.969213f, 0.274475f, 0.0614500f, 0.929791f, -0.360247f, 
		0.0616030f, 0.929791f, 0.360120f, 0.0617667f, 1.58880f, -0.0731006f, 
		0.0617667f, 1.58880f, -0.0731006f, 0.0617960f, 1.58880f, 0.0615239f, 
		0.0617960f, 1.58880f, 0.0615239f, 0.0655337f, 1.56290f, -0.0433516f, 
		0.0655494f, 1.56290f, 0.0317725f, 0.0658437f, 1.55863f, -0.000360750f, 
		0.0664070f, 0.948656f, -0.323018f, 0.0665430f, 0.948656f, 0.322888f, 
		0.0667630f, 1.57086f, -0.0590410f, 0.0667850f, 1.57086f, 0.0474620f, 
		0.0680690f, 1.58893f, -0.0791680f, 0.0680690f, 1.58893f, -0.0791680f, 
		0.0680690f, 1.61893f, -0.0791680f, 0.0680690f, 1.61893f, -0.0791680f, 
		0.0680690f, 1.61893f, -0.0791680f, 0.0681000f, 1.58893f, 0.0675890f, 
		0.0681000f, 1.58893f, 0.0675890f, 0.0681000f, 1.61893f, 0.0675890f, 
		0.0681000f, 1.61893f, 0.0675890f, 0.0681000f, 1.61893f, 0.0675890f, 
		0.0683680f, 0.951668f, -0.324582f, 0.0685050f, 0.951668f, 0.324451f, 
		0.0687390f, 0.903517f, -0.370942f, 0.0688960f, 0.903517f, 0.370811f, 
		0.0699980f, 1.57721f, -0.0616220f, 0.0700210f, 1.57721f, 0.0500410f, 
		0.0702262f, 1.56290f, -0.0336139f, 0.0702383f, 1.56290f, 0.0220339f, 
		0.0707630f, 0.905134f, -0.373912f, 0.0709210f, 0.905134f, 0.373781f, 
		0.0717271f, 1.58880f, -0.0623703f, 0.0717271f, 1.58880f, -0.0623703f, 
		0.0717511f, 1.58880f, 0.0507892f, 0.0717511f, 1.58880f, 0.0507892f, 
		0.0721240f, 0.927433f, -0.192232f, 0.0722050f, 0.927433f, 0.192099f, 
		0.0735860f, 0.948656f, -0.227494f, 0.0736820f, 0.948656f, 0.227361f, 
		0.0736985f, 1.56290f, -0.0223655f, 0.0737037f, 1.56289f, 0.0107825f, 
		0.0739560f, 1.57086f, -0.0484950f, 0.0739740f, 1.57086f, 0.0369130f, 
		0.0742490f, 0.929791f, -0.189942f, 0.0743300f, 0.929791f, 0.189809f, 
		0.0745060f, 0.878415f, -0.379405f, 0.0746660f, 0.878415f, 0.379271f, 
		0.0753111f, 1.56290f, -0.0116768f, 0.0757580f, 0.951668f, -0.226241f, 
		0.0758540f, 0.951668f, 0.226107f, 0.0765310f, 0.853634f, -0.382376f, 
		0.0765310f, 0.853634f, -0.382376f, 0.0766930f, 0.853634f, 0.382242f, 
		0.0766930f, 0.853634f, 0.382242f, 0.0766990f, 0.879246f, -0.382623f, 
		0.0768610f, 0.879246f, 0.382489f, 0.0775390f, 1.57721f, -0.0505650f, 
		0.0775580f, 1.57721f, 0.0389820f, 0.0782470f, 1.58893f, -0.0682030f, 
		0.0782470f, 1.58893f, -0.0682030f, 0.0782470f, 1.61893f, -0.0682030f, 
		0.0782470f, 1.61893f, -0.0682030f, 0.0782470f, 1.61893f, -0.0682030f, 
		0.0782730f, 1.58893f, 0.0566190f, 0.0782730f, 1.58893f, 0.0566190f, 
		0.0782730f, 1.61893f, 0.0566190f, 0.0782730f, 1.61893f, 0.0566190f, 
		0.0782730f, 1.61893f, 0.0566190f, 0.0786960f, 1.56525f, 0.000105000f, 
		0.0787830f, 0.853677f, -0.385681f, 0.0787830f, 0.853677f, -0.385681f, 
		0.0788671f, 1.58880f, -0.0519028f, 0.0788671f, 1.58880f, -0.0519028f, 
		0.0788870f, 1.58880f, 0.0403195f, 0.0788870f, 1.58880f, 0.0403195f, 
		0.0789460f, 0.853677f, 0.385546f, 0.0789460f, 0.853677f, 0.385546f, 
		0.0790840f, 0.948656f, -0.301071f, 0.0792110f, 0.948656f, 0.300936f, 
		0.0794970f, 1.57086f, -0.0369950f, 0.0795100f, 1.57086f, 0.0254110f, 
		0.0814190f, 0.951668f, -0.301988f, 0.0815470f, 0.951668f, 0.301852f, 
		0.0828400f, 0.948656f, -0.251089f, 0.0828790f, 0.927433f, -0.336160f, 
		0.0829460f, 0.948656f, 0.250952f, 0.0830220f, 0.927433f, 0.336023f, 
		0.0830540f, 0.903517f, -0.180456f, 0.0831310f, 0.903517f, 0.180319f, 
		0.0832620f, 1.57086f, -0.0247980f, 0.0832700f, 1.57086f, 0.0132120f, 
		0.0833490f, 1.57721f, -0.0385080f, 0.0833630f, 1.57721f, 0.0269220f, 
		0.0847290f, 0.948656f, -0.276363f, 0.0848460f, 0.948656f, 0.276225f, 
		0.0851670f, 1.57086f, -0.0121760f, 0.0851700f, 1.57086f, 0.000589000f, 
		0.0852225f, 1.58880f, -0.0387130f, 0.0852225f, 1.58880f, -0.0387130f, 
		0.0852363f, 1.58880f, 0.0271269f, 0.0852363f, 1.58880f, 0.0271269f, 
		0.0852860f, 0.951668f, -0.250531f, 0.0853210f, 0.929791f, -0.338108f, 
		0.0853920f, 0.951668f, 0.250393f, 0.0854640f, 0.929791f, 0.337970f, 
		0.0855000f, 0.905134f, -0.177822f, 0.0855750f, 0.905134f, 0.177684f, 
		0.0866770f, 1.58893f, -0.0558440f, 0.0866770f, 1.58893f, -0.0558440f, 
		0.0866770f, 1.61893f, -0.0558440f, 0.0866770f, 1.61893f, -0.0558440f, 
		0.0866770f, 1.61893f, -0.0558440f, 0.0866990f, 1.58893f, 0.0442560f, 
		0.0866990f, 1.58893f, 0.0442560f, 0.0866990f, 1.61893f, 0.0442560f, 
		0.0866990f, 1.61893f, 0.0442560f, 0.0866990f, 1.61893f, 0.0442560f, 
		0.0872300f, 0.951668f, -0.276551f, 0.0872960f, 1.57721f, -0.0257200f, 
		0.0873050f, 1.57721f, 0.0141320f, 0.0873470f, 0.951668f, 0.276412f, 
		0.0889602f, 1.58880f, -0.0266058f, 0.0889602f, 1.58880f, -0.0266058f, 
		0.0889689f, 1.58880f, 0.0150179f, 0.0889689f, 1.58880f, 0.0150179f, 
		0.0892940f, 1.57721f, -0.0124860f, 0.0892970f, 1.57721f, 0.000898000f, 
		0.0900210f, 0.878415f, -0.172951f, 0.0900950f, 0.878415f, 0.172811f, 
		0.0911453f, 1.58880f, -0.0121298f, 0.0911453f, 1.58880f, -0.0121298f, 
		0.0911481f, 1.58880f, 0.000540522f, 0.0911481f, 1.58880f, 0.000540522f, 
		0.0918370f, 0.927433f, -0.216962f, 0.0919290f, 0.927433f, 0.216821f, 
		0.0924680f, 0.853634f, -0.170316f, 0.0924680f, 0.853634f, -0.170316f, 
		0.0925400f, 0.853634f, 0.170175f, 0.0925400f, 0.853634f, 0.170175f, 
		0.0926710f, 0.879246f, -0.170097f, 0.0927430f, 0.879246f, 0.169956f, 
		0.0931710f, 1.61893f, -0.0423660f, 0.0931710f, 1.61893f, -0.0423660f, 
		0.0931710f, 1.61893f, -0.0423660f, 0.0931710f, 1.61893f, -0.0423660f, 
		0.0931720f, 1.58893f, -0.0423660f, 0.0931720f, 1.58893f, -0.0423660f, 
		0.0931870f, 1.58893f, 0.0307760f, 0.0931870f, 1.58893f, 0.0307760f, 
		0.0931870f, 1.61893f, 0.0307760f, 0.0931870f, 1.61893f, 0.0307760f, 
		0.0931870f, 1.61893f, 0.0307760f, 0.0945430f, 0.929791f, -0.215401f, 
		0.0946340f, 0.929791f, 0.215259f, 0.0951890f, 0.853677f, -0.167384f, 
		0.0951890f, 0.853677f, -0.167384f, 0.0952600f, 0.853677f, 0.167242f, 
		0.0952600f, 0.853677f, 0.167242f, 0.0954380f, 0.903517f, -0.346179f, 
		0.0955850f, 0.903517f, 0.346037f, 0.0975840f, 1.58893f, -0.0280700f, 
		0.0975840f, 1.58893f, -0.0280700f, 0.0975840f, 1.61893f, -0.0280700f, 
		0.0975840f, 1.61893f, -0.0280700f, 0.0975840f, 1.61893f, -0.0280700f, 
		0.0975940f, 1.58893f, 0.0164780f, 0.0975940f, 1.58893f, 0.0164780f, 
		0.0975940f, 1.61893f, 0.0164780f, 0.0975940f, 1.61893f, 0.0164780f, 
		0.0975940f, 1.61893f, 0.0164780f, 0.0982480f, 0.905134f, -0.348421f, 
		0.0983950f, 0.905134f, 0.348278f, 0.0986980f, 0.927433f, -0.308774f, 
		0.0988290f, 0.927433f, 0.308630f, 0.0998170f, 1.58893f, -0.0132770f, 
		0.0998170f, 1.58893f, -0.0132770f, 0.0998170f, 1.61893f, -0.0132770f, 
		0.0998170f, 1.61893f, -0.0132770f, 0.0998170f, 1.61893f, -0.0132770f, 
		0.0998200f, 1.58893f, 0.00168400f, 0.0998200f, 1.58893f, 0.00168400f, 
		0.0998200f, 1.61893f, 0.00168400f, 0.0998200f, 1.61893f, 0.00168400f, 
		0.0998200f, 1.61893f, 0.00168400f, 0.101606f, 0.929791f, -0.309916f, 
		0.101737f, 0.929791f, 0.309771f, 0.103385f, 0.927433f, -0.246404f, 
		0.103443f, 0.878415f, -0.352566f, 0.103490f, 0.927433f, 0.246259f, 
		0.103593f, 0.878415f, 0.352420f, 0.105742f, 0.927433f, -0.277942f, 
		0.105753f, 0.903517f, -0.208932f, 0.105841f, 0.903517f, 0.208785f, 
		0.105860f, 0.927433f, 0.277796f, 0.106254f, 0.853634f, -0.354809f, 
		0.106254f, 0.853634f, -0.354809f, 0.106404f, 0.853634f, 0.354662f, 
		0.106404f, 0.853634f, 0.354662f, 0.106431f, 0.929791f, -0.245710f, 
		0.106488f, 0.879246f, -0.354995f, 0.106535f, 0.929791f, 0.245563f, 
		0.106638f, 0.879246f, 0.354848f, 0.108857f, 0.929791f, -0.278176f, 
		0.108866f, 0.905134f, -0.207135f, 0.108953f, 0.905134f, 0.206987f, 
		0.108975f, 0.929791f, 0.278029f, 0.109381f, 0.853677f, -0.357303f, 
		0.109381f, 0.853677f, -0.357303f, 0.109532f, 0.853677f, 0.357155f, 
		0.109532f, 0.853677f, 0.357155f, 0.113653f, 0.903517f, -0.314647f, 
		0.113786f, 0.903517f, 0.314497f, 0.114623f, 0.878415f, -0.203813f, 
		0.114709f, 0.878415f, 0.203663f, 0.116998f, 0.905134f, -0.315961f, 
		0.117132f, 0.905134f, 0.315809f, 0.117737f, 0.853634f, -0.202016f, 
		0.117737f, 0.853634f, -0.202016f, 0.117822f, 0.853634f, 0.201864f, 
		0.117822f, 0.853634f, 0.201864f, 0.117996f, 0.879246f, -0.201867f, 
		0.118081f, 0.879246f, 0.201715f, 0.119050f, 0.903517f, -0.242832f, 
		0.119152f, 0.903517f, 0.242680f, 0.121201f, 0.853677f, -0.200017f, 
		0.121201f, 0.853677f, -0.200017f, 0.121286f, 0.853677f, 0.199864f, 
		0.121286f, 0.853677f, 0.199864f, 0.121763f, 0.903517f, -0.279146f, 
		0.121881f, 0.903517f, 0.278993f, 0.122554f, 0.905134f, -0.242033f, 
		0.122656f, 0.905134f, 0.241880f, 0.123185f, 0.878415f, -0.318390f, 
		0.123319f, 0.878415f, 0.318236f, 0.125347f, 0.905134f, -0.279416f, 
		0.125465f, 0.905134f, 0.279261f, 0.126531f, 0.853634f, -0.319705f, 
		0.126531f, 0.853634f, -0.319705f, 0.126667f, 0.853634f, 0.319549f, 
		0.126667f, 0.853634f, 0.319549f, 0.126809f, 0.879246f, -0.319814f, 
		0.126945f, 0.879246f, 0.319658f, 0.129034f, 0.878415f, -0.240556f, 
		0.129136f, 0.878415f, 0.240399f, 0.130254f, 0.853677f, -0.321167f, 
		0.130254f, 0.853677f, -0.321167f, 0.130390f, 0.853677f, 0.321010f, 
		0.130390f, 0.853677f, 0.321010f, 0.131975f, 0.878415f, -0.279914f, 
		0.132094f, 0.878415f, 0.279756f, 0.132540f, 0.853634f, -0.239756f, 
		0.132540f, 0.853634f, -0.239756f, 0.132641f, 0.853634f, 0.239598f, 
		0.132641f, 0.853634f, 0.239598f, 0.132831f, 0.879246f, -0.239690f, 
		0.132932f, 0.879246f, 0.239532f, 0.135561f, 0.853634f, -0.280183f, 
		0.135561f, 0.853634f, -0.280183f, 0.135679f, 0.853634f, 0.280024f, 
		0.135679f, 0.853634f, 0.280024f, 0.135858f, 0.879246f, -0.280206f, 
		0.135977f, 0.879246f, 0.280046f, 0.136439f, 0.853677f, -0.238867f, 
		0.136439f, 0.853677f, -0.238867f, 0.136540f, 0.853677f, 0.238708f, 
		0.136540f, 0.853677f, 0.238708f, 0.139549f, 0.853677f, -0.280483f, 
		0.139549f, 0.853677f, -0.280483f, 0.139668f, 0.853677f, 0.280322f, 
		0.139668f, 0.853677f, 0.280322f
	};
	static const GLfloat NormalData[] = {
		-0.989416f, 0.143084f, -0.0241326f, -0.0106736f, -0.999942f, -0.00133894f, -0.989406f, 0.143081f, 0.0245583f, 
		-0.0106740f, -0.999942f, 0.00134351f, -0.952574f, 0.143080f, 0.268573f, 
		-0.0105942f, -0.999942f, 0.00186679f, -0.952690f, 0.143062f, -0.268170f, 
		-0.0105962f, -0.999942f, -0.00186243f, -0.962213f, 0.262196f, -0.0734805f, 
		-0.962182f, 0.262194f, 0.0738959f, -0.0107538f, -0.999942f, -0.000262184f, 
		0.981968f, -0.143408f, 0.123181f, -0.0107554f, -0.999942f, 0.000266919f, 
		0.981914f, -0.143419f, -0.123593f, -0.941123f, 0.262196f, 0.213401f, 
		-0.941216f, 0.262193f, -0.212996f, -0.0103539f, -0.999942f, 0.00291915f, 
		0.974650f, -0.143400f, -0.171738f, -0.0103560f, -0.999942f, -0.00291505f, 
		0.974719f, -0.143429f, 0.171322f, 0.962316f, -0.262681f, 0.0703287f, 
		0.962284f, -0.262693f, -0.0707283f, -0.938349f, 0.143066f, -0.314696f, 
		-0.00980584f, -0.999942f, -0.00442594f, -0.938213f, 0.143084f, 0.315093f, 
		-0.00980285f, -0.999942f, 0.00442960f, 0.940289f, -0.262690f, -0.216450f, 
		0.940385f, -0.262683f, 0.216039f, -0.897811f, 0.262186f, -0.353828f, 
		-0.897660f, 0.262187f, 0.354208f, -0.0101999f, -0.999942f, -0.00342094f, 
		0.902034f, -0.143430f, 0.407140f, -0.0101979f, -0.999942f, 0.00342495f, 
		0.901859f, -0.143410f, -0.407535f, -0.880257f, 0.469577f, -0.0681538f, 
		-0.880229f, 0.469575f, 0.0685337f, 0.898834f, -0.262682f, 0.350851f, 
		0.898684f, -0.262677f, -0.351237f, -0.861239f, 0.469578f, 0.194327f, 
		-0.861321f, 0.469581f, -0.193959f, 0.880268f, -0.470223f, 0.0633899f, 
		0.880244f, -0.470218f, -0.0637691f, -0.831092f, 0.143085f, 0.537413f, 
		-0.00957317f, -0.999942f, 0.00490637f, -0.831320f, 0.143071f, -0.537064f, 
		-0.00957616f, -0.999942f, -0.00490277f, 0.859850f, -0.470216f, -0.198885f, 
		0.859932f, -0.470221f, 0.198516f, -0.836414f, 0.262195f, 0.481316f, 
		-0.836619f, 0.262187f, -0.480965f, -0.00903277f, -0.999942f, 0.00584096f, 
		0.880727f, -0.143407f, -0.451391f, -0.00903612f, -0.999942f, -0.00583759f, 
		0.880921f, -0.143410f, 0.451011f, -0.821056f, 0.469583f, -0.324589f, 
		-0.820924f, 0.469576f, 0.324934f, 0.834719f, -0.262686f, -0.483984f, 
		0.834921f, -0.262688f, 0.483634f, 0.822477f, -0.470218f, 0.320042f, 
		0.822340f, -0.470222f, -0.320387f, -0.803901f, 0.143069f, -0.577300f, 
		-0.00806502f, -0.999942f, -0.00711924f, -0.803660f, 0.143075f, 0.577634f, 
		-0.00806157f, -0.999942f, 0.00712216f, -0.771402f, 0.633451f, -0.0606464f, 
		-0.765701f, 0.469576f, 0.439546f, -0.765882f, 0.469587f, -0.439218f, 
		-0.771376f, 0.633451f, 0.0609755f, -0.753617f, 0.262200f, -0.602754f, 
		-0.755011f, 0.633447f, 0.169420f, -0.753371f, 0.262193f, 0.603065f, 
		-0.755078f, 0.633453f, -0.169100f, -0.00873808f, -0.999942f, -0.00627488f, 
		0.741947f, -0.143428f, 0.654937f, -0.00873500f, -0.999942f, 0.00627843f, 
		0.741675f, -0.143430f, -0.655245f, 0.771362f, -0.634047f, 0.0546377f, 
		0.763029f, -0.470214f, -0.443492f, 0.763203f, -0.470229f, 0.443177f, 
		0.771334f, -0.634053f, -0.0549623f, 0.755484f, -0.262686f, 0.600200f, 
		0.753191f, -0.634052f, -0.175162f, 0.755233f, -0.262682f, -0.600518f, 
		0.753262f, -0.634056f, 0.174840f, -0.719254f, 0.633451f, -0.285330f, 
		-0.719132f, 0.633454f, 0.285632f, 0.0153587f, 0.999880f, 0.00196478f, 
		0.999702f, 0.000000f, 0.0243966f, 0.992222f, 0.000000f, 0.124480f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, -1.64983e-006f, -1.00000f, 1.88830e-005f, 
		-0.0153575f, -0.999880f, 0.00197101f, -0.999692f, 0.000000f, 0.0247994f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, 1.64979e-006f, 1.00000f, -1.88827e-005f, 
		-0.992172f, 0.000000f, 0.124882f, 0.720982f, -0.634056f, 0.279568f, 
		0.720866f, -0.634054f, -0.279870f, -0.688899f, 0.469584f, -0.552186f, 
		-0.688671f, 0.469581f, 0.552472f, -0.0152567f, -0.999880f, -0.00264322f, 
		-0.962591f, 0.000000f, -0.270957f, -7.99379e-006f, -1.00000f, 3.50573e-005f, 
		1.64979e-006f, 1.00000f, -1.88827e-005f, -0.984901f, 0.000000f, -0.173120f, 
		0.0152546f, 0.999880f, -0.00264950f, 0.962476f, 0.000000f, -0.271365f, 
		0.984829f, 0.000000f, -0.173529f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, 0.691605f, -0.470212f, 0.548255f, 
		0.691369f, -0.470215f, -0.548550f, -0.635760f, 0.143090f, 0.758508f, 
		-0.00770127f, -0.999942f, 0.00750972f, -0.636080f, 0.143082f, -0.758241f, 
		-0.00770476f, -0.999942f, -0.00750677f, -0.671529f, 0.633451f, 0.384433f, 
		-0.671688f, 0.633455f, -0.384148f, 0.0140980f, 0.999880f, 0.00640471f, 
		0.948099f, 0.000000f, 0.317977f, 0.911459f, 0.000000f, 0.411391f, 
		7.99359e-006f, 1.00000f, -3.50564e-005f, -1.64983e-006f, -1.00000f, 1.88830e-005f, 
		-0.0140951f, -0.999880f, 0.00641083f, -0.947965f, 0.000000f, 0.318376f, 
		3.58729e-006f, 1.00000f, -9.35137e-007f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, -0.911280f, 0.000000f, 0.411787f, 
		-0.657382f, 0.262193f, 0.706473f, -0.657679f, 0.262191f, -0.706197f, 
		-0.00690964f, -0.999942f, 0.00824378f, 0.708552f, -0.143407f, -0.690933f, 
		-0.00691343f, -0.999942f, -0.00824135f, 0.708837f, -0.143420f, 0.690638f, 
		0.668098f, -0.634056f, -0.389382f, 0.668265f, -0.634052f, 0.389101f, 
		0.0155700f, 0.999879f, 0.000445478f, 0.980086f, 0.153106f, 0.126450f, 
		-0.0155702f, -0.999879f, 0.000451989f, -0.980030f, -0.153123f, 0.126867f, 
		0.654979f, -0.262679f, -0.708521f, 0.655269f, -0.262696f, 0.708247f, 
		0.930709f, 0.359637f, 0.0666492f, -0.930688f, -0.359618f, 0.0670461f, 
		-0.0150125f, -0.999879f, -0.00415738f, -0.973887f, -0.153124f, -0.167625f, 
		0.0150043f, 0.999879f, -0.00416310f, 0.973823f, 0.153074f, -0.168040f, 
		-0.635722f, 0.770213f, -0.0512814f, -0.909095f, -0.359636f, -0.210257f, 
		0.909010f, 0.359631f, -0.210635f, -0.635701f, 0.770212f, 0.0515533f, 
		-0.0137999f, -0.999880f, -0.00702285f, -0.839958f, 0.000000f, -0.542651f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		-0.890124f, 0.000000f, -0.455718f, 0.0137873f, 0.999880f, -0.00702488f, 
		0.839731f, 0.000000f, -0.543003f, 0.889927f, 0.000000f, -0.456104f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-0.602122f, 0.469571f, 0.645718f, -0.602392f, 0.469581f, -0.645459f, 
		-0.603196f, 0.633448f, -0.484664f, -0.622593f, 0.770215f, 0.138371f, 
		-0.602991f, 0.633454f, 0.484910f, -0.622654f, 0.770212f, -0.138113f, 
		0.740558f, 0.670154f, 0.0496777f, -0.747749f, -0.661784f, 0.0539741f, 
		0.0147480f, 0.999879f, 0.00501553f, 0.899272f, 0.153104f, 0.409719f, 
		-0.0147457f, -0.999879f, 0.00502174f, -0.899090f, -0.153131f, 0.410106f, 
		0.635733f, -0.770670f, 0.0437219f, 0.635719f, -0.770666f, -0.0439918f, 
		0.869717f, 0.359642f, 0.338008f, -0.869567f, -0.359656f, 0.338379f, 
		-0.725553f, -0.666203f, -0.172468f, 0.730341f, 0.661814f, -0.169127f, 
		0.598405f, -0.470218f, -0.648696f, 0.598676f, -0.470221f, 0.648444f, 
		0.606548f, -0.634062f, 0.479651f, 0.620377f, -0.770669f, -0.145606f, 
		0.606341f, -0.634055f, -0.479922f, 0.620442f, -0.770666f, 0.145345f, 
		-0.592359f, 0.770214f, -0.236390f, -0.592261f, 0.770213f, 0.236639f, 
		0.693013f, 0.670163f, 0.265734f, -0.692913f, -0.670141f, 0.266049f, 
		-0.0131199f, -0.999879f, -0.00839782f, -0.881213f, -0.153108f, -0.447238f, 
		0.0131084f, 0.999879f, -0.00839859f, 0.881037f, 0.153050f, -0.447605f, 
		0.594600f, -0.770666f, 0.229183f, 0.594507f, -0.770667f, -0.229422f, 
		-0.598017f, 0.143080f, -0.788609f, -0.00560869f, -0.999942f, -0.00918064f, 
		-0.597682f, 0.143085f, 0.788862f, -0.00560397f, -0.999942f, 0.00918188f, 
		-0.572552f, -0.819503f, 0.0244741f, 0.0115831f, 0.999880f, 0.0102751f, 
		0.812264f, 0.000000f, 0.583290f, 0.749705f, 0.000000f, 0.661773f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-0.0115745f, -0.999880f, 0.0102777f, -0.812010f, -9.70202e-006f, 0.583643f, 
		-4.98821e-006f, -1.00000f, 1.95894e-006f, 3.58729e-006f, 1.00000f, -9.35137e-007f, 
		-0.749426f, 0.000000f, 0.662088f, -0.806726f, -0.359654f, -0.468874f, 
		0.806534f, 0.359641f, -0.469214f, 0.554255f, 0.819512f, -0.145611f, 
		-0.542475f, 0.262204f, -0.798104f, -0.542136f, 0.262194f, 0.798338f, 
		-0.00650058f, -0.999942f, -0.00857244f, 0.515941f, -0.143442f, 0.844529f, 
		-0.00649605f, -0.999942f, 0.00857368f, 0.515585f, -0.143409f, -0.844752f, 
		-0.554146f, 0.770213f, 0.315744f, -0.554282f, 0.770212f, -0.315508f, 
		0.521556f, 0.853080f, 0.0152734f, 0.545008f, -0.262695f, 0.796214f, 
		0.544678f, -0.262680f, -0.796445f, -0.528381f, 0.633453f, 0.565288f, 
		-0.528620f, 0.633456f, -0.565062f, -0.640026f, -0.670142f, -0.375866f, 
		0.642310f, 0.666222f, -0.378926f, 0.549908f, -0.770664f, -0.321991f, 
		-0.500293f, -0.856747f, -0.125269f, 0.550040f, -0.770667f, 0.321759f, 
		0.523648f, -0.634054f, -0.569007f, 0.523891f, -0.634049f, 0.568788f, 
		0.0126145f, 0.999879f, 0.00913967f, 0.738541f, 0.153098f, 0.656596f, 
		-0.0126095f, -0.999879f, 0.00914626f, -0.738267f, -0.153073f, 0.656910f, 
		-0.495541f, 0.469576f, -0.730710f, -0.495230f, 0.469577f, 0.730921f, 
		0.464123f, 0.869255f, 0.170251f, -0.493800f, -0.853086f, 0.168540f, 
		0.731443f, 0.359625f, 0.579363f, -0.731198f, -0.359639f, 0.579663f, 
		0.499274f, -0.470206f, 0.727758f, 0.498972f, -0.470219f, -0.727956f, 
		-0.496362f, 0.770214f, -0.400492f, -0.496201f, 0.770212f, 0.400696f, 
		-0.0111170f, -0.999880f, -0.0107787f, -0.642688f, 0.000000f, -0.766128f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		-0.716247f, 0.000000f, -0.697847f, 0.0111097f, 0.999880f, -0.0107791f, 
		0.642368f, 0.000000f, -0.766396f, 0.715952f, 0.000000f, -0.698149f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		0.583902f, 0.670145f, 0.458218f, -0.583708f, -0.670153f, 0.458454f, 
		0.500637f, -0.770669f, 0.394249f, 0.500469f, -0.770667f, -0.394466f, 
		-0.338515f, -0.940599f, 0.0261112f, -0.446019f, -0.850326f, -0.279309f, 
		0.441012f, 0.856762f, -0.267333f, 0.335186f, 0.938758f, -0.0799034f, 
		-0.0100620f, -0.999879f, -0.0118920f, -0.710228f, -0.153127f, -0.687116f, 
		0.0100577f, 0.999879f, -0.0118949f, 0.709940f, 0.153103f, -0.687419f, 
		-0.433549f, 0.633447f, -0.640921f, -0.433268f, 0.633453f, 0.641106f, 
		-0.464165f, 0.884957f, -0.0374340f, -0.483215f, 0.873899f, 0.0529577f, 
		-0.632682f, -0.359641f, -0.685836f, 0.632389f, 0.359640f, -0.686108f, 
		-0.383936f, 0.143082f, 0.912206f, -0.00514552f, -0.999942f, 0.00944623f, 
		-0.384324f, 0.143059f, -0.912047f, -0.00514989f, -0.999942f, -0.00944485f, 
		-0.477318f, 0.873899f, 0.0920263f, -0.455153f, 0.885247f, -0.0957745f, 
		0.438218f, -0.634056f, 0.637132f, 0.437947f, -0.634062f, -0.637313f, 
		0.485220f, -0.874172f, 0.0196247f, -0.436461f, 0.770217f, 0.465045f, 
		0.465135f, -0.884887f, -0.0249958f, -0.436665f, 0.770215f, -0.464859f, 
		0.423066f, 0.850340f, 0.312950f, -0.393191f, -0.869258f, 0.299650f, 
		-0.419936f, 0.262192f, 0.868855f, -0.420306f, 0.262194f, -0.868676f, 
		-0.00417297f, -0.999942f, 0.00991468f, 0.473412f, -0.143410f, -0.869088f, 
		0.268926f, 0.961172f, 0.0618700f, -0.00417751f, -0.999942f, -0.00991364f, 
		0.473780f, -0.143416f, 0.868887f, 0.452893f, -0.885207f, -0.106283f, 
		0.469500f, -0.874173f, 0.124065f, -0.500813f, -0.670132f, -0.547822f, 
		0.500570f, 0.670149f, -0.548024f, 0.430561f, -0.770666f, -0.469777f, 
		0.430767f, -0.770663f, 0.469593f, 0.417041f, -0.262672f, -0.870103f, 
		0.417404f, -0.262688f, 0.869925f, -0.430363f, 0.884634f, -0.179472f, 
		-0.430015f, 0.885248f, 0.177264f, 0.00803980f, 0.999880f, 0.0132327f, 
		0.604227f, 0.000000f, 0.796812f, 0.521325f, 0.000000f, 0.853358f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-0.00803033f, -0.999880f, 0.0132275f, -0.603914f, 0.000000f, 0.797050f, 
		-4.98821e-006f, -1.00000f, 1.95894e-006f, 3.58729e-006f, 1.00000f, -9.35137e-007f, 
		-0.520975f, 0.000000f, 0.853572f, 0.435349f, -0.885497f, 0.162377f, 
		-0.253894f, -0.960078f, -0.117419f, 0.433998f, -0.885208f, -0.167489f, 
		0.0134503f, 0.999908f, 0.00172286f, 0.999703f, 0.000000f, 0.0243712f, 
		0.992225f, 0.000000f, 0.124460f, 0.0135484f, -0.999908f, 0.000293026f, 
		-0.999692f, 0.000000f, 0.0248219f, -0.0134483f, -0.999908f, 0.00172833f, 
		-0.0135558f, 0.999908f, 0.000301139f, -0.992172f, 0.000000f, 0.124876f, 
		-0.385043f, 0.469571f, 0.794509f, -0.385373f, 0.469573f, -0.794348f, 
		-0.962591f, 0.000000f, -0.270957f, -0.0133607f, -0.999908f, -0.00231205f, 
		-0.0130590f, 0.999908f, -0.00371481f, -0.984906f, 0.000000f, -0.173091f, 
		0.0133430f, 0.999908f, -0.00231692f, 0.962483f, 0.000000f, -0.271342f, 
		0.984829f, 0.000000f, -0.173529f, 0.0130249f, -0.999908f, -0.00370690f, 
		-0.408642f, 0.884635f, 0.224574f, -0.429083f, 0.873898f, -0.228452f, 
		0.380611f, -0.470219f, -0.796259f, 0.380949f, -0.470221f, 0.796097f, 
		-0.231538f, -0.962613f, 0.140593f, 0.398297f, -0.884886f, -0.241529f, 
		0.401543f, -0.885208f, 0.234881f, -0.328205f, -0.869255f, -0.369699f, 
		0.0123448f, 0.999908f, 0.00561090f, 0.948099f, 0.000000f, 0.317977f, 
		0.911459f, 0.000000f, 0.411391f, 0.0128682f, -0.999908f, 0.00427806f, 
		0.337126f, 0.853084f, -0.398239f, -0.947968f, 0.000000f, 0.318365f, 
		-0.0123417f, -0.999908f, 0.00561596f, -0.0128644f, 0.999908f, 0.00428291f, 
		-0.911273f, 0.000000f, 0.411803f, 0.00935961f, 0.999879f, 0.0124513f, 
		0.512199f, 0.153101f, 0.845111f, -0.00934940f, -0.999879f, 0.0124477f, 
		-0.511846f, -0.153066f, 0.845331f, 0.528171f, 0.359636f, 0.769218f, 
		-0.527855f, -0.359604f, 0.769451f, 0.0135496f, -0.999907f, 0.00163470f, 
		0.995338f, -0.0958520f, 0.0106925f, -0.0135474f, 0.999907f, 0.00164289f, 
		-0.995329f, 0.0958944f, 0.0111168f, 0.0136325f, 0.999907f, 0.000392224f, 
		0.985873f, 0.0958777f, 0.137339f, -0.0136408f, -0.999907f, 0.000400451f, 
		-0.985817f, -0.0958543f, 0.137753f, -0.356275f, 0.770213f, -0.528999f, 
		-0.356046f, 0.770211f, 0.529156f, 0.976845f, 0.209206f, 0.0447934f, 
		0.972648f, -0.209220f, 0.100906f, -0.976831f, -0.209188f, 0.0451917f, 
		-0.972619f, 0.209179f, 0.101279f, -0.0131540f, -0.999907f, -0.00363852f, 
		-0.978707f, -0.105464f, -0.176099f, 0.0131264f, 0.999907f, -0.00363945f, 
		0.978638f, 0.105380f, -0.176529f, -0.0134458f, 0.999907f, -0.00242661f, 
		-0.955155f, 0.115788f, -0.272531f, 0.0134139f, -0.999907f, -0.00242522f, 
		0.955047f, -0.115670f, -0.272958f, -0.369555f, 0.873897f, -0.315806f, 
		0.422897f, 0.670134f, 0.609983f, -0.422643f, -0.670148f, 0.610143f, 
		-0.369424f, 0.873898f, 0.315957f, 0.362191f, -0.770671f, 0.524294f, 
		0.361963f, -0.770665f, -0.524460f, 0.128441f, 0.991064f, -0.0359829f, 
		-0.839958f, 0.000000f, -0.542651f, -0.0120869f, -0.999908f, -0.00614846f, 
		-0.0113703f, 0.999908f, -0.00738821f, -0.890114f, 0.000000f, -0.455738f, 
		0.0120599f, 0.999908f, -0.00614082f, 0.839731f, 0.000000f, -0.543003f, 
		0.889941f, 0.000000f, -0.456076f, 0.0113663f, -0.999908f, -0.00739187f, 
		-0.338278f, 0.633449f, 0.695924f, -0.338574f, 0.633451f, -0.695779f, 
		-0.922050f, 0.335223f, -0.193517f, 0.921968f, -0.335232f, -0.193890f, 
		-0.926631f, -0.304593f, -0.220404f, 0.926537f, 0.304590f, -0.220801f, 
		0.0124648f, -0.999907f, 0.00555826f, 0.941522f, -0.105461f, 0.320022f, 
		-0.0124616f, 0.999907f, 0.00556349f, -0.941401f, 0.105424f, 0.320392f, 
		0.389851f, -0.874168f, 0.289564f, -0.144280f, -0.989273f, 0.0228597f, 
		0.389726f, -0.874172f, -0.289719f, 0.0129203f, 0.999907f, 0.00439833f, 
		0.903860f, 0.115730f, 0.411878f, -0.0129167f, -0.999907f, 0.00440276f, 
		-0.903697f, -0.115697f, 0.412245f, 0.217576f, 0.959967f, -0.176417f, 
		0.332665f, -0.634054f, -0.698076f, 0.332967f, -0.634048f, 0.697937f, 
		0.836033f, 0.545288f, 0.0609061f, -0.836011f, -0.545282f, 0.0612617f, 
		0.835803f, -0.545282f, 0.0640343f, -0.835774f, 0.545286f, 0.0643812f, 
		0.882910f, 0.335216f, 0.328785f, -0.882781f, -0.335197f, 0.329151f, 
		0.883428f, -0.304591f, 0.356062f, -0.883283f, 0.304578f, 0.356431f, 
		0.00106354f, -0.999999f, -0.000772877f, 0.940897f, 0.338606f, -0.00770618f, 
		-0.000354214f, 1.00000f, 2.39544e-005f, -0.941871f, -0.335845f, 0.00935421f, 
		-0.0114969f, -0.999907f, -0.00735407f, -0.885744f, -0.115763f, -0.449508f, 
		0.0114800f, 0.999907f, -0.00734792f, 0.885570f, 0.115662f, -0.449875f, 
		-0.00744602f, -0.999880f, -0.0135764f, -0.388315f, 0.000000f, -0.921527f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		-0.478736f, 0.000000f, -0.877959f, 0.00743972f, 0.999880f, -0.0135790f, 
		0.387923f, 0.000000f, -0.921692f, 0.478356f, 0.000000f, -0.878166f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-0.0121203f, 0.999907f, -0.00627517f, -0.837443f, 0.105469f, -0.536252f, 
		0.0121086f, -0.999907f, -0.00627267f, 0.837234f, -0.105426f, -0.536586f, 
		0.000847719f, -1.00000f, 0.000340024f, 0.919693f, 0.374587f, 0.117686f, 
		-0.000263614f, 1.00000f, 2.02544e-005f, -0.929880f, -0.352058f, 0.106664f, 
		0.128078f, 0.989696f, 0.0640126f, 0.0101418f, 0.999908f, 0.00899980f, 
		0.812250f, 0.000000f, 0.583309f, 0.749686f, 0.000000f, 0.661794f, 
		0.0110339f, -0.999908f, 0.00788029f, -0.811997f, 0.000000f, 0.583662f, 
		-0.0101385f, -0.999908f, 0.00900451f, -0.0110315f, 0.999908f, 0.00788554f, 
		-0.749426f, 0.000000f, 0.662088f, 0.287571f, 0.869251f, 0.402126f, 
		-0.311850f, -0.850338f, 0.423882f, -0.000265137f, 1.00000f, -5.84109e-005f, 
		-0.913575f, -0.374629f, -0.158221f, 0.00122990f, -0.999999f, -0.000277210f, 
		0.933683f, 0.330068f, -0.138895f, -0.829373f, 0.304632f, -0.468337f, 
		0.829183f, -0.304593f, -0.468698f, -0.807726f, -0.335233f, -0.484972f, 
		0.807528f, 0.335236f, -0.485299f, -0.318674f, 0.884957f, 0.339555f, 
		-0.115198f, -0.990651f, -0.0730718f, -1.00000f, -6.11561e-006f, 0.000129741f, 
		1.00000f, 6.76533e-006f, -0.000129829f, 0.202848f, 0.960783f, 0.189075f, 
		-0.322120f, 0.885247f, -0.335524f, -0.986109f, 0.00124923f, -0.166095f, 
		0.986109f, 0.00126186f, 0.166097f, -0.986019f, -0.00126706f, 0.166629f, 
		0.986019f, -0.00125443f, -0.166628f, -0.339005f, 0.143098f, -0.929838f, 
		-0.00265295f, -0.999942f, -0.0104243f, -0.338605f, 0.143086f, 0.929986f, 
		-0.00264901f, -0.999942f, 0.0104274f, 0.317881f, -0.874171f, -0.367119f, 
		-0.934093f, 0.00324971f, -0.357015f, 0.934094f, 0.00326106f, 0.357013f, 
		-0.933870f, -0.00325730f, 0.357599f, 0.933869f, -0.00324596f, -0.357600f, 
		0.309562f, -0.884885f, 0.348065f, -0.698183f, 0.700679f, -0.146935f, 
		0.698094f, -0.700711f, -0.147205f, -0.711504f, -0.678581f, -0.182456f, 
		0.721703f, 0.667927f, -0.181710f, -0.00314125f, 0.999994f, 0.00143156f, 
		-0.848802f, -0.358364f, -0.388730f, -1.79126e-005f, -1.00000f, -0.000383613f, 
		0.848639f, 0.358367f, -0.389081f, -0.283131f, 0.262202f, -0.922544f, 
		0.000538845f, -1.00000f, 0.000215350f, 0.844132f, 0.374620f, 0.383537f, 
		-0.00368477f, -0.999942f, -0.0101067f, 0.244084f, -0.143412f, 0.959091f, 
		-0.00314614f, 0.999994f, -0.00148572f, -0.849481f, -0.330182f, 0.411535f, 
		-0.00610958f, -0.999879f, -0.0143292f, -0.476150f, -0.153121f, -0.865930f, 
		-0.282738f, 0.262189f, 0.922668f, 0.00610301f, 0.999879f, -0.0143310f, 
		0.475776f, 0.153099f, -0.866140f, -0.999606f, -0.0280631f, 0.000409216f, 
		0.999606f, 0.0280806f, 9.80971e-006f, 0.849221f, 0.479820f, 0.220445f, 
		-0.00368040f, -0.999942f, 0.0101083f, 0.243687f, -0.143416f, -0.959192f, 
		-0.999607f, -0.0280350f, 2.20704e-005f, 0.999607f, 0.0280410f, -0.000423213f, 
		0.852658f, 0.473197f, -0.221494f, -0.993087f, 0.117384f, -0.000376584f, 
		0.993085f, -0.117391f, -0.000780232f, 0.993093f, -0.117332f, 0.000350943f, 
		-0.993091f, 0.117346f, 0.000782067f, 0.0102717f, -0.999907f, 0.00898522f, 
		0.808859f, -0.115739f, 0.576499f, -0.0102686f, 0.999907f, 0.00898952f, 
		-0.808616f, 0.115712f, 0.576846f, -0.827230f, 0.00592783f, -0.561833f, 
		0.827230f, 0.00593794f, 0.561831f, -0.827107f, -0.00592551f, 0.562013f, 
		0.827106f, -0.00591541f, -0.562015f, 0.286110f, -0.262695f, 0.921484f, 
		0.0110487f, 0.999907f, 0.00801009f, 0.742307f, 0.115731f, 0.659990f, 
		-0.0110457f, -0.999907f, 0.00801578f, -0.747902f, -0.105411f, 0.655386f, 
		-0.973795f, -0.227414f, 0.00238057f, 0.973794f, 0.227422f, 0.00198582f, 
		-0.972549f, 0.232696f, -0.000945086f, 0.972551f, -0.232685f, -0.00133600f, 
		0.285715f, -0.262679f, -0.921611f, 0.764629f, 0.643455f, -0.0361573f, 
		-0.766812f, -0.641415f, -0.0242075f, -0.973798f, -0.227406f, -0.00196735f, 
		0.973791f, 0.227431f, -0.00239911f, -0.402427f, -0.359628f, -0.841855f, 
		0.972567f, -0.232621f, 0.000929645f, -0.972573f, 0.232594f, 0.00133613f, 
		0.402066f, 0.359627f, -0.842028f, 0.805215f, 0.592460f, 0.0249129f, 
		-0.833867f, -0.550155f, 0.0446757f, 0.676044f, 0.692479f, 0.251867f, 
		-0.675914f, -0.692512f, 0.252128f, 0.686719f, -0.667883f, 0.286964f, 
		-0.686577f, 0.667901f, 0.287263f, -0.787672f, -0.505563f, 0.352107f, 
		-0.829103f, -0.473289f, 0.297635f, -0.279991f, 0.770217f, 0.573036f, 
		-0.280238f, 0.770214f, -0.572920f, 0.739558f, 0.333263f, 0.584799f, 
		-0.750066f, -0.304588f, 0.587049f, -0.733258f, -0.653416f, -0.188095f, 
		0.734964f, -0.333249f, 0.590570f, -0.726179f, 0.335219f, 0.600243f, 
		0.00187303f, -0.999998f, 0.000954376f, 0.845372f, 0.325327f, 0.423685f, 
		-0.000589135f, 1.00000f, 5.59668e-005f, -0.845476f, -0.296559f, 0.444097f, 
		-0.939111f, 0.343611f, -0.00146375f, 0.939110f, -0.343610f, -0.00186232f, 
		-0.000842307f, 1.00000f, -2.76080e-005f, -0.826366f, -0.374574f, -0.420492f, 
		0.00290690f, -0.999996f, 0.000190027f, 0.828245f, 0.351939f, -0.436062f, 
		0.939123f, -0.343579f, 0.00147796f, -0.939114f, 0.343601f, 0.00186660f, 
		-0.642688f, 0.000000f, -0.766128f, -0.00973586f, -0.999908f, -0.00943654f, 
		-0.00868680f, 0.999908f, -0.0104101f, -0.716247f, 0.000000f, -0.697847f, 
		0.00972775f, 0.999908f, -0.00943388f, 0.642351f, 0.000000f, -0.766411f, 
		0.715941f, 0.000000f, -0.698161f, 0.00868307f, -0.999908f, -0.0104152f, 
		-0.659589f, 0.00853629f, -0.751578f, 0.659588f, 0.00854544f, 0.751579f, 
		-0.659437f, -0.00855537f, 0.751711f, 0.659438f, -0.00854622f, -0.751710f, 
		-0.258152f, 0.469571f, -0.844311f, -0.317080f, -0.670140f, -0.671098f, 
		0.323130f, 0.661798f, -0.676469f, -0.257778f, 0.469580f, 0.844420f, 
		0.272967f, -0.770666f, -0.575815f, 0.273208f, -0.770665f, 0.575702f, 
		-0.0315422f, -0.999398f, -0.0144643f, -0.000598144f, 0.999999f, 0.00103652f, 
		-0.719304f, -0.645763f, -0.256111f, 0.724397f, 0.640406f, -0.255206f, 
		0.714450f, 0.649444f, 0.260354f, -0.866025f, -0.500000f, 0.000145549f, 
		0.866024f, 0.500002f, -0.000104909f, -0.866024f, 0.500002f, 0.000104909f, 
		0.866025f, -0.500000f, -0.000145549f, 0.560947f, 0.826838f, 0.0409530f, 
		-0.538187f, -0.841939f, 0.0386519f, 0.262587f, -0.470212f, 0.842585f, 
		0.571146f, -0.819150f, 0.0527900f, -0.571089f, 0.819174f, 0.0530321f, 
		-0.898061f, -0.439849f, 0.00435047f, 0.898078f, 0.439819f, 0.00395676f, 
		-0.854630f, -0.491938f, -0.166148f, 0.853382f, 0.494117f, 0.166094f, 
		-0.853384f, 0.494112f, -0.166100f, 0.854625f, -0.491933f, 0.166185f, 
		-0.853285f, -0.494118f, 0.166592f, 0.854547f, 0.491940f, -0.166566f, 
		-0.854541f, 0.491940f, 0.166597f, 0.853286f, -0.494113f, -0.166597f, 
		0.262224f, -0.470216f, -0.842695f, -0.646690f, 0.677281f, -0.350831f, 
		0.646520f, -0.677314f, -0.351081f, -0.898049f, -0.439878f, -0.00398106f, 
		0.898063f, 0.439845f, -0.00434466f, -0.893692f, 0.448678f, -0.00196136f, 
		0.893687f, -0.448684f, -0.00236286f, -0.610667f, -0.702549f, -0.365390f, 
		0.618371f, 0.692533f, -0.371505f, -0.893686f, 0.448686f, 0.00233103f, 
		0.893672f, -0.448716f, 0.00198043f, -0.810594f, -0.464189f, -0.357023f, 
		0.807337f, 0.469818f, 0.357042f, -0.807323f, 0.469814f, -0.357077f, 
		0.810599f, -0.464200f, 0.356998f, -0.807142f, -0.469793f, 0.357514f, 
		0.810382f, 0.464153f, -0.357552f, -0.810386f, 0.464164f, 0.357527f, 
		0.807128f, -0.469789f, -0.357550f, -0.446619f, 0.0100591f, -0.894668f, 
		0.446618f, 0.0100661f, 0.894668f, -0.446200f, -0.0100545f, 0.894877f, 
		0.446200f, -0.0100614f, -0.894877f, -0.255597f, 0.885248f, -0.388595f, 
		-0.865892f, -0.0280550f, -0.499443f, 0.865691f, 0.0280346f, 0.499793f, 
		-0.865477f, -0.0280498f, 0.500162f, 0.865677f, 0.0280287f, -0.499817f, 
		-0.255434f, 0.885249f, 0.388700f, -0.865695f, -0.0280046f, -0.499789f, 
		0.865898f, 0.0280385f, 0.499434f, -0.865683f, -0.0280767f, 0.499804f, 
		0.865476f, 0.0280368f, -0.500166f, -0.859854f, 0.117366f, -0.496866f, 
		0.860431f, -0.117324f, 0.495876f, -0.860223f, 0.117362f, 0.496228f, 
		0.859641f, -0.117393f, -0.497228f, -0.00881742f, -0.999907f, -0.0104152f, 
		-0.722347f, -0.0958663f, -0.684853f, 0.00881606f, 0.999907f, -0.0104181f, 
		0.722058f, 0.0958867f, -0.685156f, -0.860434f, 0.117420f, -0.495849f, 
		0.859846f, -0.117437f, 0.496863f, 0.860220f, -0.117363f, -0.496234f, 
		-0.859639f, 0.117374f, 0.497236f, 0.231738f, 0.819503f, -0.524130f, 
		-0.557003f, -0.822250f, 0.116843f, -0.719388f, -0.408424f, -0.561846f, 
		0.713464f, 0.418693f, 0.561842f, -0.713480f, 0.418707f, -0.561811f, 
		0.719394f, -0.408423f, 0.561839f, -0.713320f, -0.418724f, 0.562000f, 
		0.719225f, 0.408471f, -0.562021f, -0.719231f, 0.408470f, 0.562013f, 
		0.713337f, -0.418738f, -0.561969f, -0.00973106f, 0.999907f, -0.00956706f, 
		-0.629284f, 0.0958621f, -0.771241f, 0.00972729f, -0.999907f, -0.00957160f, 
		0.628958f, -0.0959179f, -0.771499f, -0.844527f, -0.227399f, -0.484834f, 
		0.842351f, 0.227393f, 0.488607f, -0.842135f, -0.227429f, 0.488963f, 
		0.844319f, 0.227465f, -0.485164f, -0.841800f, 0.232628f, -0.487090f, 
		0.842933f, -0.232637f, 0.485123f, -0.842723f, 0.232693f, 0.485460f, 
		0.841593f, -0.232662f, -0.487432f, 0.268246f, -0.885496f, 0.379396f, 
		0.714151f, 0.550117f, 0.432851f, -0.842334f, -0.227453f, -0.488608f, 
		0.844522f, 0.227446f, 0.484820f, -0.844323f, -0.227382f, 0.485197f, 
		0.842135f, 0.227415f, -0.488970f, 0.998643f, -0.0162136f, -0.0494909f, 
		-3.37460e-006f, 1.00000f, 1.48004e-005f, 2.28523e-006f, -1.00000f, -3.05493e-006f, 
		0.000434703f, -1.00000f, 0.000346588f, 0.693586f, 0.374608f, 0.615312f, 
		-0.842940f, 0.232611f, -0.485124f, 0.841798f, -0.232628f, 0.487094f, 
		-0.00249919f, 0.999995f, 0.00199429f, -0.693315f, -0.374622f, 0.615608f, 
		0.842738f, -0.232564f, -0.485496f, -0.841593f, 0.232602f, 0.487461f, 
		-0.837222f, 0.546858f, -0.00243135f, 0.837209f, -0.546876f, -0.00281743f, 
		0.266661f, -0.885937f, -0.379483f, -0.692073f, -0.575384f, -0.435854f, 
		0.713995f, 0.550081f, -0.433154f, 0.999176f, -0.0191401f, 0.0358031f, 
		-2.68247e-006f, 1.00000f, -1.00611e-005f, 8.16089e-006f, -1.00000f, 2.51535e-006f, 
		0.00259964f, 0.999996f, -0.00134668f, 0.988851f, 0.00282400f, 0.148883f, 
		-0.837207f, 0.546879f, 0.00279300f, 0.837224f, -0.546854f, 0.00245403f, 
		-0.644395f, -0.209172f, -0.735529f, -0.685613f, 0.209176f, -0.697266f, 
		0.644059f, 0.209202f, -0.735814f, 0.685325f, -0.209196f, -0.697544f, 
		0.982604f, 0.0191494f, -0.184723f, 0.00395153f, 0.999988f, -0.00275565f, 
		0.501649f, 0.846579f, 0.177911f, -0.999871f, -0.0159676f, 0.00190898f, 
		0.000330754f, -1.00000f, -1.40849e-005f, -0.988922f, 0.0159649f, 0.147577f, 
		3.58729e-006f, 1.00000f, -9.35137e-007f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		0.503840f, 0.845277f, -0.177913f, -0.988864f, -0.00282285f, -0.148797f, 
		0.000231705f, -1.00000f, 5.40393e-006f, -0.142855f, -0.959843f, -0.241442f, 
		-0.502752f, -0.853955f, 0.134167f, -0.939825f, 0.0162119f, -0.341272f, 
		0.000000f, 1.00000f, -5.78099e-015f, 6.17174e-006f, -1.00000f, 7.15075e-006f, 
		-8.78725e-006f, 1.00000f, -1.52122e-005f, -0.812576f, 0.343573f, -0.470826f, 
		0.814233f, -0.343583f, 0.467948f, -0.218789f, 0.0100877f, -0.975720f, 
		0.218789f, 0.0100912f, 0.975720f, -0.814015f, 0.343672f, 0.468262f, 
		0.812359f, -0.343636f, -0.471156f, -0.218357f, -0.0100882f, 0.975817f, 
		0.218357f, -0.0100917f, -0.975817f, -0.211176f, -0.850334f, -0.482014f, 
		-0.955879f, -0.0159631f, 0.293328f, 0.000278244f, -1.00000f, -7.67649e-005f, 
		-0.988420f, 0.0159670f, 0.150901f, 1.09577e-005f, 1.00000f, 1.37334e-005f, 
		0.812577f, -0.343563f, 0.470831f, -0.814232f, 0.343583f, -0.467949f, 
		-0.812385f, 0.343533f, 0.471185f, 0.814038f, -0.343561f, -0.468303f, 
		0.568809f, 0.696860f, 0.436855f, -0.591872f, -0.667896f, 0.451223f, 
		-0.575535f, -0.322330f, -0.751574f, 0.566971f, 0.337110f, 0.751598f, 
		-0.566979f, 0.337111f, -0.751593f, 0.575540f, -0.322328f, 0.751571f, 
		-0.566749f, -0.337172f, 0.751739f, 0.575298f, 0.322357f, -0.751743f, 
		-0.575303f, 0.322355f, 0.751740f, 0.566741f, -0.337171f, -0.751744f, 
		-0.225377f, 0.633444f, -0.740239f, 0.570075f, -0.661004f, 0.487943f, 
		-0.556110f, 0.692507f, 0.459539f, -0.282431f, 0.959281f, 0.00353130f, 
		-0.225053f, 0.633445f, 0.740337f, -0.270021f, 0.962207f, -0.0352883f, 
		-0.955655f, 0.00281782f, -0.294474f, -2.53304e-006f, -1.00000f, -9.77034e-007f, 
		-2.53304e-006f, -1.00000f, -9.77034e-007f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		0.230949f, -0.634053f, 0.737997f, -0.779799f, -0.626001f, 0.00604834f, 
		0.779787f, 0.626018f, 0.00572843f, 0.230637f, -0.634063f, -0.738086f, 
		-0.779923f, -0.439841f, -0.445264f, 0.775755f, 0.439855f, 0.452473f, 
		-0.775568f, -0.439853f, 0.452794f, 0.779736f, 0.439837f, -0.445595f, 
		-0.779822f, -0.625975f, -0.00571562f, 0.779808f, 0.625990f, -0.00604915f, 
		-0.775750f, -0.439872f, -0.452465f, 0.779912f, 0.439854f, 0.445271f, 
		-0.779739f, -0.439831f, 0.445595f, 0.775584f, 0.439824f, -0.452796f, 
		-0.772986f, 0.448664f, -0.448547f, 0.775139f, -0.448667f, 0.444811f, 
		-0.774942f, 0.448675f, 0.445146f, 0.772762f, -0.448728f, -0.448867f, 
		-0.571437f, 0.545257f, -0.613315f, 0.282936f, -0.958155f, 0.0434222f, 
		0.571178f, -0.545266f, -0.613548f, 0.00703321f, 0.999908f, 0.0115842f, 
		0.604227f, 0.000000f, 0.796812f, 0.521344f, 0.000000f, 0.853347f, 
		0.00822204f, -0.999908f, 0.0107835f, -0.603914f, 0.000000f, 0.797050f, 
		-0.00703362f, -0.999908f, 0.0115921f, -0.00821651f, 0.999908f, 0.0107858f, 
		-0.520975f, 0.000000f, 0.853572f, -0.768792f, 0.639491f, -0.00304615f, 
		0.768792f, -0.639490f, -0.00339920f, 0.772952f, -0.448720f, 0.448548f, 
		-0.775120f, 0.448708f, -0.444803f, -0.772761f, 0.448728f, 0.448870f, 
		0.774921f, -0.448734f, -0.445124f, -0.878493f, -0.0162250f, -0.477480f, 
		-0.00301059f, -0.999995f, -0.000436445f, -0.569153f, -0.545260f, -0.615432f, 
		0.568879f, 0.545270f, -0.615677f, -0.770478f, 0.637459f, 0.00323216f, 
		0.770460f, -0.637482f, 0.00292767f, -0.274786f, -0.960683f, -0.0397516f, 
		0.281884f, -0.958427f, 0.0442594f, 3.70840e-005f, -1.00000f, 0.000331659f, 
		0.284699f, 0.957166f, -0.0527249f, -0.639595f, -0.595084f, -0.486613f, 
		-9.03534e-005f, -0.999999f, 0.00120948f, 0.286203f, 0.956700f, 0.0530436f, 
		0.641162f, 0.594030f, -0.485840f, 0.418269f, 0.892872f, -0.166825f, 
		0.648195f, 0.591938f, 0.479013f, -0.650674f, -0.590476f, 0.477453f, 
		-0.274044f, 0.958065f, 0.0837335f, -0.391802f, -0.214470f, -0.894703f, 
		0.381806f, 0.231937f, 0.894667f, -0.381806f, 0.231933f, -0.894668f, 
		0.391800f, -0.214464f, 0.894706f, -0.381340f, -0.231901f, 0.894875f, 
		0.391426f, 0.214491f, -0.894863f, -0.391428f, 0.214497f, 0.894861f, 
		0.381340f, -0.231905f, -0.894874f, 0.261407f, -0.962298f, -0.0751636f, 
		7.58953e-005f, -1.00000f, 7.03909e-005f, 0.0250766f, 0.999308f, -0.0274811f, 
		-0.0651468f, 0.00549280f, -0.997861f, 0.0569763f, 0.00398395f, 0.998368f, 
		-0.0659657f, -0.00538450f, 0.997807f, 0.0647445f, -0.00549359f, -0.997887f, 
		0.00141953f, -0.999998f, 0.00166832f, 0.655246f, 0.374272f, 0.656181f, 
		-0.000623872f, 1.00000f, 0.000675701f, -0.667022f, -0.386701f, 0.636823f, 
		-0.000955393f, 0.999999f, -0.00102958f, -0.665679f, -0.374610f, -0.645398f, 
		0.000185725f, 1.00000f, -4.85344e-005f, 0.921046f, 0.0162250f, -0.389116f, 
		0.00275744f, -0.999991f, -0.00319831f, 0.654975f, 0.374244f, -0.656467f, 
		0.00378215f, 0.999880f, 0.0150152f, 0.342524f, 0.000000f, 0.939509f, 
		0.246629f, 0.000000f, 0.969110f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, 0.915924f, 0.0191483f, 0.400894f, 
		0.0118055f, 0.999906f, 0.00692426f, -0.906618f, -3.08004e-006f, 0.421952f, 
		-0.00397491f, -0.999992f, 0.000841138f, -0.00377603f, -0.999880f, 0.0150167f, 
		-0.342142f, 0.000000f, 0.939648f, -4.98821e-006f, -1.00000f, 1.95894e-006f, 
		3.58729e-006f, 1.00000f, -9.35137e-007f, -0.246225f, 0.000000f, 0.969213f, 
		-0.000510353f, 0.999998f, -0.00190204f, -0.577948f, -0.366555f, -0.729118f, 
		-9.66145e-005f, -1.00000f, 9.52291e-005f, 0.577644f, 0.366519f, -0.729377f, 
		-0.723818f, 0.546888f, -0.420714f, 0.726448f, -0.546884f, 0.416162f, 
		-0.726268f, 0.546879f, 0.416484f, 0.723657f, -0.546848f, -0.421044f, 
		-0.372144f, 0.922979f, -0.0980776f, 0.373082f, -0.922356f, -0.100351f, 
		-0.726440f, 0.546881f, -0.416181f, 0.723834f, -0.546853f, 0.420733f, 
		-0.723642f, 0.546875f, 0.421035f, 0.726280f, -0.546853f, -0.416497f, 
		0.393758f, 0.888896f, 0.234134f, -0.388762f, -0.892052f, 0.230450f, 
		0.258333f, -0.962285f, -0.0852734f, 0.00453802f, 0.999988f, 0.00173286f, 
		0.827168f, 0.0159634f, 0.561727f, 0.900203f, -0.0159682f, 0.435177f, 
		1.09577e-005f, 1.00000f, 1.37334e-005f, -1.28024e-005f, -1.00000f, -8.72476e-006f, 
		-0.390754f, -0.871431f, 0.296512f, 0.826077f, -0.00281880f, -0.563550f, 
		-8.78725e-006f, 1.00000f, -1.52122e-005f, 6.17174e-006f, -1.00000f, 7.15075e-006f, 
		0.805392f, -0.0191419f, 0.592433f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		8.16089e-006f, -1.00000f, 2.51535e-006f, -0.805417f, 0.0363073f, 0.591595f, 
		-2.53304e-006f, -1.00000f, -9.77034e-007f, 3.58729e-006f, 1.00000f, -9.35137e-007f, 
		0.00716158f, -0.999907f, 0.0116071f, 0.602996f, -0.115761f, 0.789301f, 
		-0.00716200f, 0.999907f, 0.0116164f, -0.611884f, 0.0958645f, 0.785117f, 
		0.374207f, -0.896292f, 0.237970f, -0.387936f, 0.897164f, 0.211193f, 
		-0.194556f, -0.100572f, -0.975722f, 0.184405f, 0.117999f, 0.975741f, 
		-0.184403f, 0.117996f, -0.975742f, 0.194555f, -0.100569f, 0.975722f, 
		-0.184036f, -0.118034f, 0.975807f, 0.194143f, 0.100582f, -0.975803f, 
		-0.194144f, 0.100585f, 0.975803f, 0.184039f, -0.118038f, -0.975806f, 
		0.146681f, 0.940364f, -0.306919f, 0.00819500f, 0.999907f, 0.0109138f, 
		0.514811f, 0.115729f, 0.849456f, -0.00819227f, -0.999907f, 0.0109143f, 
		-0.506938f, -0.0958719f, 0.856635f, -0.733150f, 0.680067f, 0.000190391f, 
		0.733164f, -0.680052f, -0.000118870f, -0.573703f, -0.209204f, 0.791895f, 
		-0.527209f, 0.209214f, 0.823578f, 0.534336f, 0.333286f, 0.776792f, 
		-0.230661f, 0.959010f, -0.164607f, 0.528224f, -0.333282f, 0.780963f, 
		-0.231364f, 0.959281f, 0.162019f, -0.124919f, -0.964001f, 0.234729f, 
		0.271686f, 0.961878f, 0.0312678f, -0.678345f, -0.626016f, -0.384645f, 
		0.672464f, 0.626008f, 0.394850f, -0.672311f, -0.625988f, 0.395142f, 
		0.678197f, 0.625984f, -0.384959f, -0.751064f, 0.660227f, 0.00194944f, 
		0.751076f, -0.660214f, 0.00163041f, -0.672488f, -0.625976f, -0.394858f, 
		0.678360f, 0.625994f, 0.384654f, -0.678189f, -0.626008f, 0.384932f, 
		0.672318f, 0.625984f, -0.395137f, 0.227730f, -0.962290f, 0.148787f, 
		-0.833600f, -2.31384e-006f, 0.552369f, 2.28523e-006f, -1.00000f, -3.05493e-006f, 
		2.28523e-006f, -1.00000f, -3.05493e-006f, -2.28478e-006f, 1.00000f, 3.05527e-006f, 
		-0.335616f, -0.925792f, -0.173983f, -0.664243f, 0.639521f, -0.387032f, 
		0.667465f, -0.639533f, 0.381430f, -0.667329f, 0.639488f, 0.381742f, 
		0.664088f, -0.639504f, -0.387326f, 0.845711f, -0.0191431f, -0.533297f, 
		1.09577e-005f, 1.00000f, 1.37334e-005f, 8.16089e-006f, -1.00000f, 2.51535e-006f, 
		-0.000202061f, 1.00000f, 0.000235571f, -0.250466f, -0.957200f, -0.145031f, 
		-0.225654f, 0.873899f, 0.430559f, -0.668885f, 0.637421f, -0.382474f, 
		0.665791f, -0.637467f, 0.387759f, -0.665652f, 0.637438f, 0.388046f, 
		0.668710f, -0.637454f, -0.382726f, -0.225835f, 0.873900f, -0.430462f, 
		0.00527374f, 0.999879f, 0.0146573f, 0.240342f, 0.153125f, 0.958534f, 
		0.0789151f, 0.987967f, -0.133016f, -0.00526694f, -0.999879f, 0.0146543f, 
		-0.239950f, -0.153129f, 0.958632f, 0.578400f, 0.391405f, 0.715721f, 
		0.538633f, 0.582412f, 0.608827f, 0.490001f, 0.668236f, 0.559785f, 
		0.277985f, 0.359634f, 0.890723f, -0.566299f, -0.548862f, 0.614862f, 
		-0.277609f, -0.359645f, 0.890836f, -0.524765f, -0.590440f, -0.613190f, 
		0.503667f, 0.640214f, -0.580039f, 0.199272f, -0.885208f, -0.420354f, 
		-0.235350f, 0.959281f, -0.156177f, 0.195726f, -0.874171f, 0.444427f, 
		-0.184514f, 0.770218f, -0.610507f, -0.0500838f, -0.0239135f, -0.998459f, 
		0.0548166f, 0.0377591f, 0.997782f, -0.0487800f, 0.0315656f, -0.998311f, 
		0.0602086f, -0.0284328f, 0.997781f, -0.0465731f, -0.0318856f, 0.998406f, 
		0.0496696f, 0.0238998f, -0.998480f, -0.0543685f, 0.0254603f, 0.998196f, 
		0.0454395f, -0.0312594f, -0.998478f, -0.723723f, -2.31384e-006f, 0.690091f, 
		-0.00459520f, -0.999988f, 0.00178534f, -0.184250f, 0.770213f, 0.610593f, 
		-0.473354f, -0.545291f, 0.691805f, 0.707769f, 0.0191423f, -0.706185f, 
		0.00431593f, 0.999990f, -0.00140132f, -0.470765f, 0.545286f, 0.693573f, 
		0.224323f, 0.670137f, 0.707528f, -0.224013f, -0.670142f, 0.707621f, 
		-0.783707f, 0.621118f, -0.00401643f, 0.783693f, -0.621133f, -0.00434539f, 
		0.191550f, -0.770669f, 0.607765f, -0.625315f, -0.780336f, 0.00757203f, 
		0.625323f, 0.780332f, 0.00732288f, 0.191303f, -0.770667f, -0.607845f, 
		-0.625336f, -0.780322f, -0.00730943f, 0.625343f, 0.780313f, -0.00756959f, 
		-0.995608f, -0.00105948f, 0.0936191f, 0.995506f, -0.000783279f, -0.0946918f, 
		-0.969433f, -0.224278f, 0.0994980f, -0.994201f, -0.107477f, -0.00365223f, 
		0.993336f, 0.115239f, -0.00195704f, -0.339213f, 0.868656f, -0.361071f, 
		0.376482f, -0.836200f, -0.398786f, -0.969466f, -0.224314f, -0.0990951f, 
		-0.993352f, -0.00174581f, -0.115108f, 0.993351f, -0.00176741f, 0.115116f, 
		-0.979128f, 0.196245f, 0.0528711f, 0.971523f, -0.236920f, 0.00346400f, 
		0.990219f, 0.138924f, 0.0128785f, 0.970502f, -0.0566198f, 0.234349f, 
		0.981117f, 0.0285055f, 0.191305f, -0.973381f, 0.00320338f, 0.229169f, 
		0.973379f, 0.00318509f, -0.229178f, 0.977417f, -0.105617f, -0.183035f, 
		0.929657f, 0.00901800f, 0.368317f, -0.948499f, 0.0541114f, -0.312123f, 
		-0.734081f, 0.00135091f, 0.679061f, -0.856025f, 0.0712290f, 0.512004f, 
		-0.913274f, -0.113882f, 0.391104f, -0.989745f, -0.130772f, 0.0574821f, 
		-0.998152f, -0.00797055f, 0.0602397f, 0.990657f, -0.132572f, -0.0319792f, 
		0.973215f, -0.136512f, -0.184978f, 0.950150f, 0.0924244f, -0.297779f, 
		0.999813f, -0.00331181f, -0.0190354f, -0.989582f, 0.142885f, 0.0176461f, 
		-0.977982f, 0.0431709f, -0.204173f, -0.921855f, -0.0793835f, -0.379316f, 
		-0.845904f, -0.0111365f, -0.533219f, -0.995518f, 0.0571844f, -0.0753189f, 
		0.916334f, -0.146390f, 0.372696f, 0.988326f, -0.0659212f, 0.137356f, 
		0.960052f, 0.0504154f, 0.275244f, 0.989454f, 0.141790f, 0.0295966f, 
		-0.382378f, -0.820061f, -0.425778f, 0.362600f, 0.842764f, -0.397832f, 
		0.000281465f, -1.00000f, 0.000408842f, 0.481417f, 0.374576f, 0.792421f, 
		0.000252188f, 0.999998f, 0.00191888f, -0.481068f, -0.374605f, 0.792619f, 
		-0.635021f, 0.680067f, -0.366411f, 0.635014f, -0.680036f, 0.366481f, 
		-0.634830f, 0.680071f, 0.366736f, 0.634880f, -0.680046f, -0.366694f, 
		-0.960374f, 0.00298442f, -0.278700f, 0.960775f, 0.00306428f, 0.277313f, 
		-0.966330f, -0.0544651f, 0.251474f, 0.958104f, -0.00263640f, -0.286409f, 
		-0.388330f, 0.000000f, -0.921520f, -0.00652267f, -0.999908f, -0.0118876f, 
		-0.00522821f, 0.999908f, -0.0125031f, -0.478728f, 0.000000f, -0.877963f, 
		0.00651770f, 0.999908f, -0.0118912f, 0.387923f, 0.000000f, -0.921692f, 
		0.478341f, 0.000000f, -0.878174f, 0.00522279f, -0.999908f, -0.0125055f, 
		-0.859139f, -0.0105869f, 0.511632f, 0.859134f, -0.0106106f, -0.511640f, 
		-0.859262f, -0.511429f, -0.0104117f, 0.859254f, 0.511435f, -0.0107828f, 
		-0.852281f, 0.523063f, 0.00468515f, 0.817275f, -0.576232f, 0.00431817f, 
		0.209212f, -0.958162f, -0.195336f, -0.832166f, 0.554504f, 0.00500482f, 
		0.832160f, -0.554517f, 0.00465389f, 0.00156276f, 0.999998f, 0.00121921f, 
		-0.172670f, -0.984286f, 0.0369739f, -0.859404f, -0.0105836f, -0.511188f, 
		0.859406f, -0.0106022f, 0.511184f, -0.832655f, 0.00903572f, 0.553718f, 
		0.832657f, 0.00902048f, -0.553715f, -0.757058f, -0.0191502f, -0.653067f, 
		-0.00603548f, -0.999980f, -0.00180721f, 0.899377f, 0.395957f, -0.185308f, 
		-0.651410f, 0.660224f, -0.373857f, 0.649630f, -0.660219f, 0.376950f, 
		-0.935391f, 0.346094f, 0.0725482f, -0.844265f, 0.512507f, 0.156697f, 
		-0.890557f, 0.441772f, 0.108382f, -0.649486f, 0.660207f, 0.377220f, 
		0.651264f, -0.660210f, -0.374136f, 0.444231f, 0.660993f, 0.604771f, 
		-0.902401f, 0.419899f, -0.0967369f, 0.924635f, 0.193242f, 0.328189f, 
		0.384604f, -0.706995f, 0.593497f, 0.860622f, 0.141850f, -0.489090f, 
		0.905888f, 0.247768f, -0.343478f, 0.906636f, 0.313787f, -0.282043f, 
		-0.908338f, 0.210373f, -0.361477f, -0.925869f, 0.128534f, -0.355310f, 
		-0.937795f, 0.0506979f, -0.343469f, 0.894840f, -0.427377f, 0.128879f, 
		-0.889992f, 0.309188f, 0.335137f, -0.940873f, 0.0340912f, 0.337039f, 
		-0.928202f, 0.115533f, 0.353685f, -0.00135389f, -0.999999f, 0.000168848f, 
		0.0389940f, 0.997162f, 0.0643977f, 0.883170f, -0.317990f, 0.344810f, 
		0.757026f, 0.0660316f, 0.650040f, 0.173269f, 0.850324f, 0.496918f, 
		-0.757534f, 0.0158224f, -0.652604f, 0.757533f, 0.0158394f, 0.652605f, 
		0.729291f, -0.00752047f, -0.684162f, -0.789989f, -0.00818198f, 0.613066f, 
		-0.156063f, -0.869253f, 0.469089f, 0.895401f, -0.317786f, -0.311880f, 
		-0.612512f, -0.0281129f, -0.789962f, 6.17174e-006f, -1.00000f, 7.15075e-006f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.209194f, -0.959380f, 0.189280f, 
		0.887740f, -0.460321f, -0.00463172f, -0.887720f, 0.460364f, -0.00424449f, 
		-0.442911f, -0.623365f, -0.644396f, 0.394935f, 0.698745f, -0.596475f, 
		0.206785f, -0.958554f, -0.195996f, 0.441787f, 0.649761f, 0.618575f, 
		-0.00535693f, -0.999907f, -0.0125526f, -0.472553f, -0.105415f, -0.874975f, 
		-0.486329f, -0.575418f, 0.657555f, 0.00534609f, 0.999907f, -0.0125474f, 
		0.472155f, 0.105438f, -0.875187f, 0.258248f, 0.938101f, -0.230811f, 
		0.226506f, 0.948737f, 0.220441f, 0.00642817f, 0.00128400f, 0.999979f, 
		0.799716f, -0.111811f, -0.589876f, 0.738329f, 0.0151408f, -0.674271f, 
		0.685217f, 0.106717f, -0.720479f, -0.195946f, 0.962207f, 0.189109f, 
		-0.676705f, 0.621125f, -0.395316f, 0.680867f, -0.621135f, 0.388087f, 
		-0.680704f, 0.621130f, 0.388381f, 0.676507f, -0.621175f, -0.395574f, 
		0.587444f, 0.0205136f, 0.809005f, -0.620650f, -0.0144687f, -0.783955f, 
		-0.545332f, -0.780331f, -0.306098f, 0.537895f, 0.780331f, 0.318986f, 
		-0.537764f, -0.780326f, 0.319220f, 0.545203f, 0.780330f, -0.306330f, 
		0.863014f, 0.495848f, -0.0966523f, -0.860719f, -0.499815f, 0.0966827f, 
		-0.863192f, 0.495530f, 0.0966945f, 0.860904f, -0.499503f, -0.0966478f, 
		-0.537894f, -0.780326f, -0.319000f, 0.545348f, 0.780320f, 0.306100f, 
		0.537738f, 0.780343f, -0.319221f, -0.545199f, -0.780331f, 0.306337f, 
		-0.598982f, -0.0235249f, 0.800417f, 0.545414f, -0.0337761f, -0.837486f, 
		-0.00647679f, 0.999907f, -0.0120124f, -0.382429f, 0.115738f, -0.916708f, 
		0.861824f, 0.0993227f, 0.497387f, -0.861710f, -0.0914503f, -0.499093f, 
		-0.861029f, -0.115246f, 0.495326f, 0.859271f, 0.115252f, -0.498368f, 
		0.00647179f, -0.999907f, -0.0120159f, 0.382056f, -0.115706f, -0.916867f, 
		-0.599217f, -0.800244f, -0.0234089f, 0.564815f, 0.824580f, -0.0324424f, 
		-0.569287f, 0.822040f, 0.0127618f, 0.507231f, -0.861803f, 0.00348825f, 
		-0.814816f, 0.382055f, 0.436014f, -0.858418f, 0.0338210f, 0.511834f, 
		-0.856041f, 0.162608f, 0.490664f, -0.859404f, -0.498182f, -0.115064f, 
		0.861154f, 0.495138f, 0.115122f, -0.861148f, 0.495146f, -0.115128f, 
		0.859398f, -0.498191f, 0.115072f, -0.599354f, -0.0235483f, -0.800138f, 
		0.564955f, -0.0323246f, 0.824488f, -0.544282f, 0.0163178f, 0.838744f, 
		0.544289f, 0.0163031f, -0.838739f, -0.857475f, 0.0413581f, -0.512861f, 
		-0.835848f, 0.245182f, -0.491166f, -0.860505f, 0.154311f, -0.485509f, 
		0.844514f, -0.228955f, -0.484124f, -0.841305f, 0.228992f, 0.489662f, 
		-0.773851f, 0.490308f, -0.400939f, 0.811330f, -0.456208f, 0.365536f, 
		-0.844565f, -0.483949f, 0.229136f, 0.841335f, 0.489492f, -0.229244f, 
		-0.841332f, 0.489502f, 0.229236f, 0.844562f, -0.483959f, -0.229127f, 
		0.798919f, 0.473068f, 0.371397f, -0.811872f, -0.424399f, -0.400936f, 
		-0.700086f, 0.573197f, 0.425823f, -0.829197f, 0.534903f, 0.162207f, 
		-0.844497f, 0.429157f, 0.320388f, -0.919070f, 0.390293f, 0.0545951f, 
		0.787251f, -0.616586f, 0.00760616f, 0.814430f, -0.535961f, -0.222371f, 
		0.824448f, -0.556838f, -0.101082f, 0.821811f, -0.409604f, -0.396044f, 
		0.713610f, -0.433189f, -0.550552f, -0.936433f, 0.350823f, 0.00396894f, 
		0.936438f, -0.350816f, 0.00357524f, -0.835967f, -0.388561f, 0.387529f, 
		-0.857143f, -0.512746f, 0.0489526f, -0.740548f, -0.551556f, 0.383894f, 
		-0.811421f, -0.567834f, -0.138419f, -0.791652f, -0.608520f, 0.0546760f, 
		0.841655f, 0.449924f, -0.298638f, 0.622697f, 0.398825f, -0.673192f, 
		0.922511f, 0.385509f, -0.0188715f, -0.0437049f, -0.996795f, 0.0670033f, 
		0.000151544f, 1.00000f, -0.000146609f, -0.836401f, 0.329687f, -0.437882f, 
		0.821998f, -0.304195f, 0.481441f, -0.886885f, -0.456415f, -0.0715560f, 
		0.848061f, 0.451662f, 0.277118f, 0.852714f, 0.367258f, 0.371485f, 
		0.792222f, 0.545465f, 0.273591f, 0.826882f, 0.545329f, 0.137415f, 
		0.799379f, 0.599764f, 0.0357318f, 0.868178f, 0.477838f, 0.133933f, 
		-0.423598f, 0.335194f, -0.841552f, -0.838331f, 0.541386f, -0.0640490f, 
		0.773637f, -0.551135f, 0.312627f, 0.876093f, -0.481871f, 0.0161365f, 
		0.908929f, -0.365157f, 0.201267f, 0.915837f, -0.401045f, 0.0201229f, 
		0.423245f, -0.335207f, -0.841725f, 0.817344f, 0.168814f, 0.550864f, 
		-0.405428f, -0.304577f, -0.861894f, 0.405063f, 0.304566f, -0.862069f, 
		0.00444429f, 0.999989f, -0.00147689f, 0.812407f, -0.580520f, -0.0546977f, 
		-0.499993f, -0.866030f, 0.000110403f, 0.499996f, 0.866028f, -8.75728e-005f, 
		-0.499996f, 0.866028f, 8.44827e-005f, 0.499993f, -0.866029f, -0.000113483f, 
		-0.0980175f, 0.143088f, 0.984844f, -0.00213284f, -0.999942f, 0.0105439f, 
		-0.0984201f, 0.143063f, -0.984808f, -0.00213731f, -0.999942f, -0.0105431f, 
		-0.494165f, -0.853338f, -0.166175f, 0.491993f, 0.854592f, 0.166177f, 
		-0.491983f, 0.854584f, -0.166250f, 0.494166f, -0.853323f, 0.166249f, 
		-0.491908f, -0.854587f, 0.166454f, 0.494085f, 0.853340f, -0.166407f, 
		-0.494087f, 0.853324f, 0.166481f, 0.491895f, -0.854580f, -0.166532f, 
		-0.833849f, -0.476436f, -0.278755f, 0.829125f, 0.482247f, 0.282823f, 
		-0.829603f, 0.483394f, -0.279444f, 0.833272f, -0.476411f, 0.280519f, 
		-0.829904f, -0.482106f, 0.280774f, 0.832923f, 0.477487f, -0.279725f, 
		-0.738720f, -0.438796f, 0.511616f, 0.749336f, 0.420454f, -0.511580f, 
		0.738725f, -0.438806f, -0.511600f, -0.749333f, 0.420462f, 0.511578f, 
		-0.309275f, -0.839262f, 0.447199f, 0.749549f, 0.511367f, 0.420333f, 
		-0.738954f, -0.511399f, -0.438655f, 0.725713f, -0.553397f, -0.408768f, 
		-0.716691f, 0.553393f, 0.424394f, -0.749366f, -0.511404f, 0.420614f, 
		0.738766f, 0.511410f, -0.438959f, -0.723175f, 0.554509f, -0.411750f, 
		0.718345f, -0.554489f, 0.420146f, -0.718173f, 0.554493f, 0.420435f, 
		0.723027f, -0.554479f, -0.412049f, -0.749560f, 0.420451f, -0.511255f, 
		0.738986f, -0.438810f, 0.511219f, -0.738987f, -0.438800f, -0.511227f, 
		0.749561f, 0.420441f, 0.511260f, -0.725601f, -0.408577f, 0.553686f, 
		0.716623f, 0.424215f, -0.553618f, -0.716615f, 0.424221f, 0.553623f, 
		0.725591f, -0.408582f, -0.553695f, 0.000234271f, -0.999999f, -0.00133673f, 
		0.822703f, 0.556728f, 0.114952f, -0.821244f, 0.430037f, 0.375003f, 
		0.783015f, -0.497286f, -0.373622f, -0.971872f, 0.235500f, -0.00231757f, 
		0.971863f, -0.235533f, -0.00269730f, 0.00265777f, 0.999992f, -0.00297567f, 
		0.755470f, -0.651603f, -0.0684014f, -0.145185f, 0.262192f, 0.954032f, 
		-0.469899f, -0.807212f, -0.357217f, 0.464272f, 0.810500f, 0.357128f, 
		-0.464278f, 0.810500f, -0.357122f, 0.469904f, -0.807210f, 0.357214f, 
		-0.464110f, -0.810478f, 0.357391f, 0.469732f, 0.807190f, -0.357486f, 
		-0.469731f, 0.807192f, 0.357483f, 0.464116f, -0.810477f, -0.357384f, 
		-0.00106537f, -0.999942f, 0.0107046f, 0.196225f, -0.143400f, -0.970017f, 
		-0.145585f, 0.262184f, -0.953973f, -0.00438612f, -0.999990f, 9.97693e-005f, 
		-0.807270f, 0.577832f, -0.120105f, -0.00106978f, -0.999942f, -0.0107041f, 
		0.196623f, -0.143406f, 0.969935f, -0.300667f, 0.846634f, 0.439101f, 
		-0.00218633f, -0.999998f, -0.000311407f, -0.717014f, 0.667074f, 0.202248f, 
		-0.993314f, 0.115437f, 0.00131936f, 0.993315f, -0.115436f, 0.000935133f, 
		0.993315f, -0.115436f, 0.000935133f, -0.500164f, -0.0280449f, -0.865476f, 
		0.499817f, 0.0280296f, 0.865677f, 0.996839f, -0.0794440f, -0.000722995f, 
		-0.996839f, 0.0794525f, -0.000345882f, -0.499423f, -0.0280553f, 0.865904f, 
		0.499777f, 0.0280220f, -0.865701f, 0.712307f, -0.580935f, -0.393870f, 
		0.765386f, -0.386580f, -0.514528f, 0.781026f, -0.278500f, -0.558960f, 
		0.000798173f, -1.00000f, 0.000523129f, 0.869065f, 0.455887f, 0.192076f, 
		0.793059f, 0.547633f, 0.266749f, 0.658485f, 0.595558f, 0.460117f, 
		-0.499807f, -0.0280411f, -0.865683f, 0.500161f, 0.0280012f, 0.865480f, 
		-0.998473f, 0.0552499f, -5.92860e-005f, 0.998469f, -0.0553098f, -0.000627837f, 
		0.998469f, -0.0553098f, -0.000627837f, -0.996849f, 0.0793066f, -0.00173337f, 
		0.996402f, -0.0633225f, 0.0563284f, 0.499455f, 0.0280384f, -0.865886f, 
		-0.499798f, -0.0280486f, 0.865688f, -0.496229f, 0.117361f, -0.860223f, 
		0.497230f, -0.117383f, 0.859642f, -0.496851f, 0.117341f, 0.859866f, 
		0.495856f, -0.117325f, -0.860443f, -0.00192731f, 0.999997f, 0.00147474f, 
		-0.707804f, -0.674547f, 0.209760f, -0.497223f, 0.117409f, -0.859642f, 
		0.496231f, -0.117388f, 0.860218f, 0.496864f, -0.117385f, -0.859853f, 
		-0.495865f, 0.117385f, 0.860429f, -0.00254037f, 0.999997f, 0.000208925f, 
		-0.816013f, -0.468802f, -0.338152f, 0.142045f, -0.262683f, -0.954370f, 
		0.110342f, -0.991759f, 0.0651054f, 0.634031f, -8.88567e-007f, 0.773308f, 
		-2.28478e-006f, 1.00000f, 3.05527e-006f, 6.17174e-006f, -1.00000f, 7.15075e-006f, 
		0.142444f, -0.262685f, 0.954309f, -0.00165300f, 0.999998f, -0.00108515f, 
		-0.719836f, -0.662064f, -0.208584f, -0.622586f, 0.716654f, -0.314315f, 
		0.626656f, -0.686394f, 0.369007f, -0.00206082f, 0.999998f, 0.000211401f, 
		-0.828492f, -0.451341f, 0.331501f, -0.651298f, 0.0191404f, -0.758581f, 
		-1.45169e-005f, -1.00000f, 9.90107e-006f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		-0.418820f, -0.713370f, -0.561866f, 0.408516f, 0.719233f, 0.561978f, 
		-0.408548f, 0.719273f, -0.561902f, 0.418839f, -0.713412f, 0.561798f, 
		-0.408324f, -0.719282f, 0.562054f, 0.418599f, 0.713378f, -0.562021f, 
		-0.418615f, 0.713419f, 0.561956f, 0.408357f, -0.719323f, -0.561978f, 
		-0.000318588f, -1.00000f, -0.000880796f, -0.120228f, 0.988184f, 0.0950674f, 
		-0.488968f, -0.227416f, -0.842136f, 0.485174f, 0.227430f, 0.844324f, 
		0.936897f, -0.349552f, -0.00607355f, -0.936903f, 0.349544f, -0.00569839f, 
		-0.484819f, -0.227429f, 0.844528f, 0.488600f, 0.227435f, -0.842344f, 
		-0.485476f, 0.232624f, -0.842733f, 0.487437f, -0.232603f, 0.841606f, 
		-0.487087f, 0.232621f, 0.841804f, 0.485115f, -0.232638f, -0.842937f, 
		-0.663979f, -0.365002f, -0.652615f, 0.648163f, 0.392370f, 0.652634f, 
		-0.648169f, 0.392365f, -0.652631f, 0.663981f, -0.364995f, 0.652616f, 
		-0.647840f, -0.392403f, 0.652934f, 0.663667f, 0.365014f, -0.652926f, 
		-0.485184f, -0.227442f, -0.844314f, 0.488973f, 0.227462f, 0.842120f, 
		-0.925721f, 0.378205f, 0.00116159f, 0.948793f, -0.315865f, 0.00468796f, 
		0.484828f, 0.227409f, -0.844528f, -0.488612f, -0.227406f, 0.842345f, 
		-0.487450f, 0.232602f, -0.841599f, 0.485491f, -0.232599f, 0.842731f, 
		0.487089f, -0.232562f, -0.841819f, -0.485135f, 0.232563f, 0.842946f, 
		0.00185510f, -0.999998f, 0.00100777f, 0.851688f, 0.436010f, -0.290729f, 
		-0.766672f, 0.460358f, -0.447531f, 0.771103f, -0.460358f, 0.439854f, 
		-0.770917f, 0.460364f, 0.440173f, 0.766484f, -0.460330f, -0.447882f, 
		-0.00932520f, -0.000131310f, 0.999956f, -0.757581f, 0.391248f, -0.522491f, 
		-0.000435174f, 1.00000f, 0.000897485f, -0.445531f, -0.374592f, 0.813131f, 
		0.000920438f, 0.999999f, -0.00125603f, -0.205785f, -0.942948f, -0.261727f, 
		-0.000396636f, -1.00000f, 0.000446638f, -0.0992070f, 0.990387f, -0.0963883f, 
		-0.000648551f, 0.999999f, -0.00132722f, -0.473939f, -0.330038f, -0.816368f, 
		-0.000598481f, 1.00000f, -0.000400714f, -0.190271f, -0.947083f, 0.258515f, 
		-0.990614f, 0.0370871f, -0.131558f, -0.975202f, -0.0115415f, -0.221017f, 
		-0.927293f, 0.00933020f, -0.374220f, -0.990208f, -0.0508325f, -0.130015f, 
		-0.950165f, -0.0298360f, -0.310317f, -0.924475f, -0.0784578f, -0.373083f, 
		-0.831511f, -0.123296f, -0.541652f, 0.845033f, -0.120908f, 0.520865f, 
		0.930409f, -0.0767759f, 0.358392f, 0.954206f, -0.0286864f, 0.297771f, 
		0.990427f, -0.0439697f, 0.130848f, 0.933113f, 0.0103218f, 0.359434f, 
		0.975125f, -0.00577921f, 0.221580f, 0.990614f, 0.0370907f, 0.131558f, 
		-0.362101f, 0.0158916f, -0.932003f, 0.362101f, 0.0159014f, 0.932003f, 
		0.363196f, -0.0104026f, -0.931655f, -0.396401f, -0.0163394f, 0.917932f, 
		-0.468278f, 0.343595f, -0.814038f, 0.471160f, -0.343612f, 0.812366f, 
		-0.470822f, 0.343685f, 0.812531f, 0.467935f, -0.343663f, -0.814207f, 
		-0.454487f, 0.270399f, -0.848720f, 0.538260f, -0.292481f, 0.790399f, 
		0.00231072f, 0.999996f, -0.00137062f, 0.621921f, 0.0159671f, -0.782917f, 
		0.501206f, -0.0159646f, -0.865181f, 1.09577e-005f, 1.00000f, 1.37334e-005f, 
		2.28523e-006f, -1.00000f, -3.05493e-006f, -0.982557f, 0.145362f, 0.115984f, 
		-0.946862f, 0.321616f, 0.00407510f, -0.915497f, 0.397717f, 0.0607121f, 
		-0.967345f, 0.251302f, -0.0330323f, -0.915979f, 0.398421f, -0.0473518f, 
		0.845791f, -0.521412f, -0.112996f, 0.900394f, -0.410450f, -0.144299f, 
		0.960707f, -0.276519f, -0.0240784f, 0.994008f, -0.0999243f, -0.0443125f, 
		0.941826f, -0.335715f, 0.0160937f, 0.994026f, -0.0999161f, 0.0439206f, 
		-0.133749f, 0.469570f, 0.872706f, -0.471186f, 0.343556f, -0.812375f, 
		0.468292f, -0.343570f, 0.814041f, 0.480622f, 0.323766f, 0.814971f, 
		-0.476434f, -0.308272f, -0.823395f, -0.837690f, 0.119898f, -0.532823f, 
		-0.927101f, 0.0761624f, -0.366991f, -0.951890f, 0.0280432f, -0.305155f, 
		-0.990085f, 0.0435084f, -0.133561f, -0.929742f, -0.0112108f, -0.368042f, 
		-0.974004f, 0.00602731f, -0.226449f, -0.990274f, -0.0363277f, -0.134303f, 
		0.990273f, -0.0363549f, 0.134303f, 0.974004f, 0.00600004f, 0.226449f, 
		0.929744f, -0.0112376f, 0.368036f, 0.990086f, 0.0434808f, 0.133562f, 
		0.951891f, 0.0280167f, 0.305155f, 0.927102f, 0.0761355f, 0.366996f, 
		0.837693f, 0.119873f, 0.532824f, 0.470832f, -0.343571f, -0.812574f, 
		-0.467946f, 0.343550f, 0.814248f, -0.134122f, 0.469578f, -0.872644f, 
		-0.604845f, -0.0160788f, 0.796181f, -0.00615092f, -0.999938f, 0.00931878f, 
		0.507014f, -0.320002f, -0.800335f, -0.505137f, 0.254423f, 0.824685f, 
		-0.455452f, -0.302035f, 0.837460f, 0.530485f, 0.279169f, -0.800406f, 
		-0.337291f, -0.566845f, -0.751613f, 0.322508f, 0.575417f, 0.751587f, 
		-0.322510f, 0.575420f, -0.751585f, 0.337294f, -0.566854f, 0.751604f, 
		-0.322173f, -0.575404f, 0.751741f, 0.337006f, 0.566891f, -0.751706f, 
		-0.337002f, 0.566882f, 0.751714f, 0.322172f, -0.575402f, -0.751744f, 
		-0.879769f, 0.0419799f, 0.473545f, -0.819989f, 0.115498f, 0.560605f, 
		-0.909647f, 0.112871f, 0.399753f, -0.825447f, -0.0127059f, 0.564336f, 
		-0.915839f, 0.00250530f, 0.401538f, 0.915847f, 0.00250588f, -0.401521f, 
		0.825452f, -0.0127135f, -0.564329f, 0.909645f, 0.112867f, -0.399759f, 
		0.819995f, 0.115490f, -0.560599f, 0.879772f, 0.0419743f, -0.473539f, 
		-0.454016f, -0.846299f, -0.278653f, 0.468705f, 0.846125f, 0.253748f, 
		-0.463313f, 0.838625f, 0.286443f, 0.479592f, -0.838648f, -0.258189f, 
		0.507127f, 0.800220f, -0.320110f, -0.505275f, -0.824563f, 0.254545f, 
		-0.818811f, -0.560047f, 0.126077f, -0.940818f, -0.337492f, 0.0310022f, 
		-0.960891f, -0.248033f, -0.123158f, 0.948729f, 0.306986f, -0.0753143f, 
		0.810501f, 0.585147f, -0.0262911f, 0.899765f, 0.423608f, 0.104781f, 
		-0.878345f, -0.474091f, -0.0612209f, 0.805758f, 0.582490f, 0.107046f, 
		-0.454059f, -0.278692f, -0.846263f, 0.468813f, 0.253861f, 0.846031f, 
		-0.489892f, 0.243821f, -0.836993f, 0.507258f, -0.319947f, 0.800202f, 
		-0.479522f, -0.258148f, 0.838700f, 0.463191f, 0.286340f, -0.838727f, 
		-0.463200f, 0.286350f, 0.838719f, 0.479535f, -0.258167f, -0.838687f, 
		-0.172219f, 0.0154983f, -0.984937f, 0.204938f, 0.0189523f, 0.978591f, 
		-0.919882f, -0.00216864f, 0.392190f, -0.834863f, 0.0118262f, 0.550331f, 
		-0.913569f, -0.114013f, 0.390375f, -0.829285f, -0.116056f, 0.546643f, 
		-0.886001f, -0.0425238f, 0.461728f, 0.885998f, -0.0425416f, -0.461733f, 
		0.829275f, -0.116072f, -0.546654f, 0.913565f, -0.114033f, -0.390379f, 
		0.834865f, 0.0118101f, -0.550329f, 0.919879f, -0.00218508f, -0.392196f, 
		-0.217043f, 0.975880f, 0.0234577f, 0.148591f, -0.988889f, -0.00439178f, 
		-0.183929f, 0.0276378f, 0.982551f, -0.183929f, 0.0276378f, 0.982551f, 
		0.161541f, -0.0642177f, -0.984774f, 0.161541f, -0.0642177f, -0.984774f, 
		0.161541f, -0.0642177f, -0.984774f, -0.812951f, 0.350828f, -0.464790f, 
		0.809174f, -0.350869f, 0.471304f, -0.808989f, 0.350831f, 0.471651f, 
		0.812758f, -0.350837f, -0.465121f, -0.182583f, 0.0159858f, 0.983060f, 
		0.216413f, 0.0189447f, -0.976118f, 0.189270f, 0.978944f, -0.0764565f, 
		-0.202125f, -0.978945f, 0.0285068f, 0.189270f, 0.978944f, -0.0764565f, 
		-0.202125f, -0.978945f, 0.0285068f, 0.00254596f, 0.999985f, 0.00492881f, 
		0.488466f, -8.88568e-007f, 0.872583f, 0.173696f, -0.0682374f, 0.982432f, 
		-0.197219f, 0.0288712f, -0.979934f, -0.197219f, 0.0288712f, -0.979934f, 
		0.173696f, -0.0682374f, 0.982432f, 0.128995f, -0.470220f, -0.873071f, 
		-0.127436f, 0.873901f, -0.469102f, 0.129375f, -0.470212f, 0.873019f, 
		-0.468591f, -0.0191498f, -0.883207f, -0.00681777f, -0.999959f, -0.00589164f, 
		-0.324273f, 0.692496f, -0.644435f, 0.488286f, -0.286136f, -0.824441f, 
		-0.488286f, 0.286136f, 0.824441f, -0.00839789f, -0.00251884f, -0.999962f, 
		-0.643640f, -0.361531f, 0.674554f, -0.127232f, 0.873901f, 0.469159f, 
		0.00495776f, 0.00157817f, -0.999986f, 0.628271f, 0.380265f, 0.678730f, 
		0.323991f, -0.692496f, -0.644577f, -0.0231082f, -0.0241526f, 0.999441f, 
		-0.666201f, -0.346532f, -0.660372f, -0.666180f, -0.561278f, -0.491092f, 
		-0.687362f, -0.438482f, -0.579022f, -0.00958622f, 0.00492911f, -0.999942f, 
		-0.667071f, 0.276857f, 0.691640f, -0.452797f, -0.439827f, -0.775582f, 
		0.445598f, 0.439836f, 0.779735f, -0.445265f, -0.439855f, 0.779914f, 
		0.452461f, 0.439868f, -0.775754f, -0.445118f, 0.448704f, -0.774941f, 
		0.448869f, -0.448677f, 0.772791f, -0.448554f, 0.448713f, 0.772953f, 
		0.444807f, -0.448717f, -0.775113f, -0.596261f, -0.632357f, 0.494568f, 
		-0.770781f, -0.321979f, 0.549751f, -0.719961f, -0.454709f, 0.524306f, 
		-0.445600f, -0.439856f, -0.779722f, 0.452794f, 0.439854f, 0.775568f, 
		0.445276f, 0.439806f, -0.779936f, -0.452474f, -0.439803f, 0.775784f, 
		0.445143f, -0.448697f, 0.774931f, -0.448869f, 0.448736f, -0.772757f, 
		0.405525f, 0.494153f, 0.769001f, -0.308221f, -0.667882f, -0.677446f, 
		0.448527f, -0.448723f, -0.772963f, -0.444782f, 0.448760f, 0.775102f, 
		0.300677f, 0.678571f, -0.670175f, -0.616876f, -0.615811f, -0.490145f, 
		-0.774455f, -0.317899f, -0.546955f, -0.729888f, -0.443955f, -0.519776f, 
		-0.358817f, -0.473191f, -0.804575f, 0.158993f, -0.874168f, 0.458858f, 
		-0.429977f, -0.902766f, 0.0115081f, 0.453590f, 0.890793f, 0.0272689f, 
		0.00299511f, -0.00296712f, -0.999991f, 0.415309f, -0.370733f, 0.830708f, 
		0.700069f, -0.298497f, 0.648694f, 0.602724f, -0.344110f, 0.719938f, 
		-0.840519f, 0.235515f, -0.487914f, 0.843025f, -0.235530f, 0.483565f, 
		-0.842836f, 0.235444f, 0.483935f, 0.840323f, -0.235403f, -0.488306f, 
		-0.232096f, -0.381501f, -0.894756f, 0.214672f, 0.391608f, 0.894739f, 
		-0.214668f, 0.391604f, -0.894742f, 0.232094f, -0.381498f, 0.894758f, 
		-0.214292f, -0.391618f, 0.894826f, 0.231703f, 0.381496f, -0.894860f, 
		-0.231705f, 0.381499f, 0.894858f, 0.214297f, -0.391622f, -0.894823f, 
		0.158508f, -0.874471f, -0.458449f, -0.453592f, -0.890793f, -0.0272446f, 
		0.429965f, 0.902772f, -0.0115274f, -0.860875f, 0.115475f, -0.495540f, 
		0.859772f, -0.115448f, 0.497458f, 0.859772f, -0.115448f, 0.497458f, 
		0.860681f, -0.115463f, -0.495880f, -0.859469f, 0.115519f, 0.497964f, 
		0.860681f, -0.115463f, -0.495880f, 0.863644f, -0.0793441f, 0.497819f, 
		-0.863120f, 0.0793955f, -0.498719f, 0.864138f, -0.0795223f, -0.496932f, 
		-0.891055f, 0.0635377f, 0.449426f, -0.000174581f, -1.00000f, -0.000198401f, 
		0.622286f, 0.672274f, -0.401008f, 0.000218008f, -1.00000f, -8.17324e-005f, 
		0.111588f, 0.988652f, -0.100580f, 0.000218008f, -1.00000f, -8.17324e-005f, 
		-0.864598f, 0.0552930f, -0.499413f, 0.864957f, -0.0552143f, 0.498799f, 
		0.864957f, -0.0552143f, 0.498799f, -0.891274f, 0.0633440f, -0.449019f, 
		0.864379f, -0.0793148f, 0.496546f, -0.864645f, 0.0552776f, 0.499334f, 
		0.864478f, -0.0553873f, -0.499610f, 0.864478f, -0.0553873f, -0.499610f, 
		-0.856495f, 0.139473f, 0.496954f, 0.872541f, -0.167601f, -0.458893f, 
		0.700361f, -0.713724f, -0.00958053f, -0.692253f, 0.721605f, -0.00854423f, 
		-0.682530f, 0.730813f, 0.00809498f, 0.740208f, -0.672351f, 0.00603829f, 
		-0.619563f, -0.247507f, -0.744904f, -0.665204f, -0.0485373f, -0.745082f, 
		-0.606842f, -0.490982f, -0.625043f, -0.652329f, -0.164050f, -0.739969f, 
		-0.711246f, 0.0613590f, -0.700260f, -0.658937f, -0.228913f, 0.716520f, 
		-0.566810f, -0.0547854f, 0.822025f, -0.630226f, -0.472478f, 0.616101f, 
		-0.667876f, -0.115501f, 0.735256f, -0.695182f, -0.322865f, 0.642246f, 
		0.814429f, -0.349508f, 0.463195f, -0.808563f, 0.349474f, -0.473386f, 
		0.801107f, -0.378246f, -0.463851f, -0.824001f, 0.315952f, 0.470315f, 
		0.00540606f, 0.999980f, 0.00337277f, 0.715439f, -0.487413f, 0.500575f, 
		0.716075f, -0.263954f, 0.646192f, 0.730965f, -0.358014f, 0.580961f, 
		-0.230194f, 0.963859f, -0.134111f, 0.266577f, -0.951155f, 0.155695f, 
		-0.416474f, 0.546872f, -0.726279f, 0.421017f, -0.546874f, 0.723652f, 
		-0.420741f, 0.546859f, 0.723824f, 0.416195f, -0.546855f, -0.726451f, 
		-0.824223f, 0.315879f, -0.469975f, 0.801308f, -0.378227f, 0.463519f, 
		-0.319881f, -0.640604f, 0.698071f, 0.416465f, -0.546934f, 0.726237f, 
		-0.421017f, 0.546914f, -0.723622f, -0.416181f, 0.546847f, 0.726465f, 
		0.420712f, -0.546875f, -0.723830f, 0.313158f, 0.643253f, -0.698682f, 
		0.762721f, 0.108935f, -0.637487f, 0.668908f, 0.281411f, -0.688019f, 
		0.705094f, 0.189802f, -0.683240f, 0.628483f, 0.330459f, -0.704135f, 
		0.492162f, 0.608450f, -0.622547f, 0.136255f, -0.959104f, 0.248102f, 
		-0.354957f, 0.00281413f, 0.934878f, -0.434933f, -0.0340247f, 0.899820f, 
		-0.594573f, -0.0730783f, 0.800714f, 0.594568f, -0.0730914f, -0.800716f, 
		0.434934f, -0.0340338f, -0.899819f, 0.354963f, 0.00280687f, -0.934876f, 
		-0.759663f, 0.0423784f, -0.648935f, 0.169232f, 0.0514646f, 0.984232f, 
		-0.000482681f, 0.00219609f, -0.999997f, -0.000482681f, 0.00219609f, -0.999997f, 
		0.169232f, 0.0514646f, 0.984232f, 0.741030f, 0.0569493f, 0.669052f, 
		0.401559f, 0.0301706f, 0.915336f, 0.599335f, 0.798635f, -0.0545859f, 
		0.427282f, 0.904086f, -0.00765195f, -0.417455f, -0.908212f, -0.0297039f, 
		0.506051f, 0.861072f, -0.0496759f, 0.000166196f, -1.00000f, 0.000130044f, 
		0.151459f, 0.964675f, -0.215550f, -0.759157f, 0.648731f, 0.0531952f, 
		0.000000f, -1.00000f, -1.28608e-019f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.000000f, -1.00000f, -1.28608e-019f, 
		0.779644f, -0.622356f, 0.0694807f, -0.665255f, 0.522581f, 0.533240f, 
		-0.764656f, 0.529446f, 0.367407f, -0.810216f, 0.500180f, 0.305566f, 
		-0.835640f, 0.532697f, 0.133936f, -0.810632f, 0.455104f, 0.368451f, 
		-0.840410f, 0.492194f, 0.226839f, -0.875718f, 0.463658f, 0.134684f, 
		0.839385f, -0.526588f, -0.134679f, 0.846433f, -0.481762f, -0.226839f, 
		0.799401f, -0.474548f, -0.368459f, 0.879138f, -0.457358f, -0.133942f, 
		0.838243f, -0.451637f, -0.305570f, 0.840805f, -0.397565f, -0.367409f, 
		0.785150f, -0.314956f, -0.533237f, -0.785530f, -0.407137f, 0.466028f, 
		-0.836844f, -0.393764f, 0.380319f, -0.878676f, -0.452939f, 0.150913f, 
		-0.796505f, -0.469240f, 0.381304f, -0.836189f, -0.447475f, 0.317102f, 
		-0.839218f, -0.522215f, 0.151672f, 0.875272f, 0.459243f, -0.151642f, 
		0.839074f, 0.487842f, -0.240758f, 0.806872f, 0.451084f, -0.381419f, 
		0.835509f, 0.528336f, -0.150950f, 0.808376f, 0.495936f, -0.317137f, 
		0.762092f, 0.524017f, -0.380292f, 0.658516f, 0.514624f, -0.549107f, 
		-0.308422f, 0.204134f, -0.929088f, 0.366005f, -0.180556f, 0.912929f, 
		-0.366003f, -0.180560f, -0.912929f, 0.308423f, 0.204138f, 0.929087f, 
		-0.267024f, -0.0374270f, -0.962963f, -0.381501f, 0.0127345f, -0.924281f, 
		-0.619123f, -0.0274607f, -0.784813f, -0.381263f, -0.0668464f, -0.922047f, 
		-0.618556f, -0.0486584f, -0.784232f, -0.457445f, -0.0198218f, -0.889017f, 
		0.452337f, -0.0174996f, 0.891675f, 0.618554f, -0.0486757f, 0.784234f, 
		0.374783f, -0.0642033f, 0.924887f, 0.619356f, -0.0214795f, 0.784817f, 
		0.374917f, 0.0155992f, 0.926927f, 0.258419f, -0.0340516f, 0.965433f, 
		-0.340321f, -0.225669f, 0.912828f, 0.332381f, 0.164511f, -0.928687f, 
		-0.118129f, 0.633452f, 0.764712f, -0.735356f, 0.0828911f, 0.672593f, 
		-0.000233403f, 1.85072e-005f, -1.00000f, 0.000230372f, 2.37378e-005f, 1.00000f, 
		0.000233404f, 1.84863e-005f, 1.00000f, -0.000233403f, 1.85072e-005f, -1.00000f, 
		0.735363f, 0.0828747f, -0.672587f, -0.118225f, -0.184227f, -0.975748f, 
		0.100719f, 0.194265f, 0.975765f, -0.100717f, 0.194262f, -0.975766f, 
		0.118222f, -0.184223f, 0.975749f, -0.100379f, -0.194373f, 0.975778f, 
		0.117864f, 0.184343f, -0.975769f, -0.117867f, 0.184348f, 0.975768f, 
		0.100381f, -0.194377f, -0.975777f, -0.883011f, 0.0999120f, -0.458596f, 
		-0.823885f, 0.335733f, -0.456615f, -0.838911f, 0.0999161f, -0.535018f, 
		-0.820174f, 0.276528f, -0.500847f, -0.707864f, 0.410464f, -0.574847f, 
		-0.676194f, 0.521438f, -0.520446f, 0.817125f, -0.398407f, 0.416628f, 
		0.854467f, -0.251273f, 0.454696f, 0.762712f, -0.397711f, 0.509996f, 
		0.818184f, -0.321595f, 0.476604f, 0.793197f, -0.145327f, 0.591370f, 
		-0.665289f, -0.533041f, 0.522741f, -0.764633f, -0.367213f, 0.529615f, 
		-0.810178f, -0.305366f, 0.500363f, -0.835543f, -0.133754f, 0.532896f, 
		-0.810614f, -0.368259f, 0.455292f, -0.840352f, -0.226630f, 0.492389f, 
		-0.875646f, -0.134493f, 0.463849f, 0.839295f, 0.134497f, -0.526778f, 
		0.846387f, 0.226629f, -0.481943f, 0.799395f, 0.368257f, -0.474714f, 
		0.879072f, 0.133749f, -0.457541f, 0.838220f, 0.305365f, -0.451819f, 
		0.840803f, 0.367211f, -0.397751f, 0.785223f, 0.533029f, -0.315128f, 
		-0.118449f, 0.633450f, -0.764664f, 0.00330379f, 0.999908f, 0.0131235f, 
		0.342560f, 0.000000f, 0.939496f, 0.246651f, 0.000000f, 0.969104f, 
		0.00467020f, -0.999908f, 0.0127111f, -0.876402f, -0.463176f, 0.131861f, 
		-0.841521f, -0.492544f, 0.221907f, -0.813128f, -0.457570f, 0.359795f, 
		-0.835716f, -0.533273f, 0.131147f, -0.811921f, -0.501909f, 0.298113f, 
		-0.767240f, -0.531649f, 0.358736f, -0.671149f, -0.527148f, 0.521224f, 
		0.792057f, 0.317742f, -0.521235f, 0.844015f, 0.398676f, -0.358742f, 
		0.840608f, 0.452229f, -0.298105f, 0.879679f, 0.457126f, -0.131151f, 
		0.802811f, 0.475458f, -0.359769f, 0.847301f, 0.482533f, -0.221908f, 
		0.839319f, 0.527405f, -0.131858f, -0.839310f, 0.527414f, 0.131887f, 
		-0.847288f, 0.482543f, 0.221935f, -0.802798f, 0.475463f, 0.359793f, 
		-0.879666f, 0.457142f, 0.131180f, -0.840596f, 0.452238f, 0.298127f, 
		-0.844005f, 0.398686f, 0.358755f, -0.792044f, 0.317750f, 0.521250f, 
		0.671139f, -0.527142f, -0.521242f, 0.767226f, -0.531649f, -0.358767f, 
		0.811908f, -0.501914f, -0.298141f, 0.835705f, -0.533283f, -0.131176f, 
		0.813117f, -0.457575f, -0.359814f, 0.841508f, -0.492554f, -0.221934f, 
		0.876390f, -0.463190f, -0.131890f, -0.342123f, 0.000000f, 0.939655f, 
		-0.00330051f, -0.999908f, 0.0131438f, -0.00467255f, 0.999908f, 0.0127292f, 
		-0.246225f, 0.000000f, 0.969213f, -0.797888f, 0.458087f, -0.391830f, 
		-0.717329f, 0.427731f, -0.549986f, -0.848336f, 0.358091f, -0.389997f, 
		-0.776420f, 0.314191f, -0.546311f, -0.788746f, 0.406217f, -0.461376f, 
		0.746200f, -0.479868f, 0.461424f, 0.660349f, -0.515195f, 0.546364f, 
		0.734301f, -0.555554f, 0.390080f, 0.729114f, -0.407238f, 0.550046f, 
		0.795696f, -0.461849f, 0.391873f, -0.753577f, -0.433390f, -0.494262f, 
		-0.759574f, -0.573678f, -0.306498f, -0.762812f, -0.497147f, -0.413477f, 
		0.757787f, 0.395008f, 0.519353f, 0.778346f, 0.500602f, 0.378913f, 
		0.767783f, 0.455564f, 0.450523f, -0.831744f, -0.423642f, -0.358787f, 
		-0.688947f, -0.585156f, -0.427721f, -0.784180f, -0.307036f, -0.539250f, 
		0.893891f, 0.248051f, 0.373403f, -0.751443f, -0.582510f, -0.309864f, 
		0.799480f, 0.337506f, 0.496912f, 0.646285f, 0.560069f, 0.518303f, 
		0.791437f, 0.474102f, 0.385816f, -0.797685f, 0.392030f, 0.458269f, 
		-0.717126f, 0.550140f, 0.427874f, -0.848167f, 0.390198f, 0.358272f, 
		-0.776242f, 0.546461f, 0.314370f, -0.788559f, 0.461545f, 0.406387f, 
		0.746011f, -0.461546f, -0.480044f, 0.660149f, -0.546463f, -0.515347f, 
		0.734106f, -0.390217f, -0.555714f, 0.728935f, -0.550141f, -0.407430f, 
		0.795516f, -0.392016f, -0.462036f, -0.150963f, 0.0848070f, -0.984895f, 
		0.169203f, -0.0764437f, 0.982612f, 0.125118f, 0.0686509f, 0.989764f, 
		-0.169447f, -0.0747370f, -0.982701f, -0.783116f, -0.403577f, -0.473133f, 
		-0.768170f, -0.310049f, -0.560165f, -0.844384f, -0.357105f, -0.399363f, 
		-0.708798f, -0.423796f, -0.563917f, -0.794559f, -0.455789f, -0.401163f, 
		0.792060f, 0.460120f, 0.401161f, 0.721489f, 0.401812f, 0.563915f, 
		0.731500f, 0.552608f, 0.399415f, 0.652659f, 0.510100f, 0.560209f, 
		0.741125f, 0.476299f, 0.473151f, -0.741118f, 0.476313f, -0.473149f, 
		-0.652651f, 0.510110f, -0.560209f, -0.731491f, 0.552622f, -0.399412f, 
		-0.721473f, 0.401822f, -0.563929f, -0.792055f, 0.460144f, -0.401143f, 
		0.794550f, -0.455809f, 0.401159f, 0.708788f, -0.423813f, 0.563917f, 
		0.844382f, -0.357130f, 0.399345f, 0.768152f, -0.310066f, 0.560180f, 
		0.783106f, -0.403597f, 0.473132f, -0.747254f, 0.496870f, 0.441284f, 
		0.703209f, -0.579363f, -0.412111f, -0.150980f, 0.983016f, 0.104329f, 
		0.175378f, -0.981350f, -0.0787079f, -0.138898f, -0.985973f, -0.0925499f, 
		-0.138898f, -0.985973f, -0.0925499f, 0.149472f, 0.986018f, 0.0736669f, 
		0.149472f, 0.986018f, 0.0736669f, 0.000123304f, 1.00000f, -0.000241587f, 
		0.000123304f, 1.00000f, -0.000241587f, 0.0947721f, -0.986788f, -0.131405f, 
		-0.177970f, 0.124743f, 0.976097f, 0.165588f, -0.0774673f, -0.983148f, 
		-0.175222f, -0.0786800f, 0.981380f, 0.150889f, 0.104349f, -0.983028f, 
		-0.139085f, -0.0925449f, -0.985947f, -0.139085f, -0.0925449f, -0.985947f, 
		0.179525f, 0.0884316f, 0.979771f, 0.112127f, -0.634051f, -0.765119f, 
		0.112446f, -0.634054f, 0.765069f, 0.000255334f, 1.00000f, 0.000142264f, 
		-0.119238f, -0.972079f, 0.202103f, -0.590427f, 0.704896f, 0.393086f, 
		-0.619224f, 0.268082f, 0.738034f, -0.592895f, 0.412024f, 0.691890f, 
		-0.395137f, -0.626009f, -0.672294f, 0.384957f, 0.625996f, 0.678186f, 
		-0.384653f, -0.625991f, 0.678364f, 0.394856f, 0.625980f, -0.672487f, 
		-0.384959f, -0.625984f, -0.678197f, 0.395152f, 0.625978f, 0.672315f, 
		0.384652f, 0.626009f, -0.678348f, -0.394849f, -0.626017f, 0.672456f, 
		-0.381726f, 0.639524f, -0.667304f, 0.387319f, -0.639517f, 0.664080f, 
		-0.387046f, 0.639506f, 0.664249f, 0.381468f, -0.639502f, -0.667473f, 
		0.000622384f, -1.00000f, 0.000547895f, 0.518677f, 0.610599f, 0.598450f, 
		0.586479f, 0.525883f, 0.616027f, 0.697074f, 0.383888f, 0.605573f, 
		-0.387578f, 0.638129f, -0.665263f, 0.382706f, -0.637457f, 0.668719f, 
		-0.382415f, 0.637474f, 0.668869f, 0.387758f, -0.637450f, -0.665809f, 
		0.530946f, 0.0191497f, 0.847189f, 0.00384623f, 0.999948f, 0.00942103f, 
		-0.406539f, -0.890792f, -0.203019f, 0.366702f, 0.902768f, 0.224809f, 
		-0.379089f, -0.890794f, 0.250554f, 0.387983f, 0.899117f, -0.202629f, 
		0.00390792f, 0.999987f, 0.00333025f, 0.613061f, -0.477505f, 0.629400f, 
		-0.366706f, -0.902766f, -0.224811f, 0.406538f, 0.890791f, 0.203022f, 
		0.379066f, 0.890798f, -0.250577f, -0.385865f, -0.899970f, 0.202884f, 
		0.00565700f, 0.999978f, -0.00335758f, 0.686622f, -0.450983f, -0.570232f, 
		0.00341637f, -0.999907f, 0.0131852f, 0.343581f, -0.115698f, 0.931969f, 
		-0.996034f, -0.0889764f, 0.000210424f, -0.000216180f, -1.55112e-005f, -1.00000f, 
		-0.00341625f, 0.999907f, 0.0132035f, -0.336433f, 0.105456f, 0.935784f, 
		0.664113f, -0.666234f, 0.339244f, -0.605512f, 0.698639f, -0.381128f, 
		0.618842f, -0.701182f, -0.354088f, -0.677139f, 0.643971f, 0.356067f, 
		-0.00311338f, -0.999880f, -0.0151679f, -0.0994389f, 0.000000f, -0.995044f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		-0.198675f, 0.000000f, -0.980065f, -0.0373849f, -0.0531752f, -0.997885f, 
		0.0232973f, 0.0494705f, 0.998504f, -0.0242326f, 0.0518895f, -0.998359f, 
		0.0428749f, -0.0564286f, 0.997486f, -0.996037f, 0.0889364f, 0.000210425f, 
		0.000214056f, 1.53741e-005f, 1.00000f, -0.0281812f, -0.0599733f, 0.997802f, 
		0.0369814f, 0.0532148f, -0.997898f, -0.0436244f, 0.0605687f, 0.997210f, 
		0.0275715f, -0.0590913f, -0.997872f, 0.236394f, 0.884425f, 0.402381f, 
		0.00310674f, 0.999880f, -0.0151684f, 0.0990094f, 0.000000f, -0.995086f, 
		0.198256f, 0.000000f, -0.980150f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -0.624427f, 0.690549f, -0.365011f, 
		0.566488f, -0.746141f, 0.349807f, -0.00723675f, -0.999973f, -0.00122185f, 
		-0.687074f, 0.525425f, -0.501855f, -0.00776872f, -0.999959f, -0.00466626f, 
		-0.640616f, 0.422197f, -0.641374f, 0.00241489f, 0.999997f, -0.000913587f, 
		0.507641f, -0.668754f, -0.543202f, -0.0868368f, -0.0318718f, 0.995713f, 
		-0.0868368f, -0.0318718f, 0.995713f, 0.0825014f, 0.0190340f, -0.996409f, 
		0.0825014f, 0.0190340f, -0.996409f, 0.0845377f, 0.990071f, 0.112309f, 
		1.87974e-005f, -1.00000f, 0.000460748f, 0.00460967f, 0.999907f, 0.0128276f, 
		0.241547f, 0.115653f, 0.963472f, -0.00460858f, -0.999907f, 0.0128458f, 
		-0.241148f, -0.115710f, 0.963566f, -0.366679f, -0.0159685f, 0.930211f, 
		-0.00368627f, -0.999964f, 0.00769338f, -0.498317f, 0.0159608f, 0.866848f, 
		-2.28478e-006f, 1.00000f, 3.05527e-006f, 0.281630f, 0.333236f, 0.899799f, 
		-0.293075f, -0.335240f, 0.895389f, -0.284378f, 0.0917201f, 0.954315f, 
		-0.672327f, 0.329288f, 0.662983f, -0.457265f, 0.227771f, 0.859668f, 
		0.519906f, -0.336683f, -0.785075f, 0.194003f, -0.168047f, -0.966500f, 
		0.374412f, -0.247962f, -0.893493f, -0.519924f, -0.336681f, 0.785064f, 
		-0.374427f, -0.247960f, 0.893488f, -0.194008f, -0.168045f, 0.966500f, 
		0.457272f, 0.227772f, -0.859664f, 0.284371f, 0.0917118f, -0.954318f, 
		0.672333f, 0.329288f, -0.662977f, 0.640659f, 0.00699564f, 0.767794f, 
		-0.595625f, 0.439516f, -0.672351f, 0.000482284f, -0.00140154f, 0.999999f, 
		-0.000554843f, 0.000384271f, -1.00000f, 0.000482284f, -0.00140154f, 0.999999f, 
		0.678554f, -0.295993f, 0.672274f, 0.274583f, -0.333245f, 0.901971f, 
		-0.272045f, 0.304617f, 0.912798f, -0.366294f, -0.904090f, -0.220115f, 
		-0.491879f, -0.798644f, -0.346731f, -0.413527f, -0.861079f, -0.295869f, 
		0.387347f, 0.902612f, 0.187759f, -0.672459f, -0.662764f, 0.329457f, 
		-0.457469f, -0.859528f, 0.227892f, 0.370716f, 0.895990f, -0.244480f, 
		0.621373f, 0.662764f, -0.417899f, 0.187650f, 0.968723f, -0.162366f, 
		-0.284609f, -0.954241f, 0.0917723f, -0.503564f, -0.0126942f, 0.863865f, 
		-0.710069f, 0.622357f, -0.329352f, 0.000000f, -1.00000f, -1.28608e-019f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, -1.00000f, -1.28608e-019f, 0.631012f, -0.648759f, 0.425365f, 
		-0.595455f, 0.672414f, 0.439650f, 0.000000f, -1.00000f, -1.28608e-019f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, -1.00000f, -1.28608e-019f, 0.678355f, -0.672409f, -0.296145f, 
		0.624790f, -0.0581689f, -0.778623f, -0.299444f, -0.183040f, -0.936392f, 
		-0.351826f, -0.253729f, -0.901022f, -0.583198f, -0.432031f, -0.687917f, 
		0.665847f, 0.288907f, 0.687881f, 0.356069f, 0.194671f, 0.913957f, 
		0.450147f, 0.200460f, 0.870163f, -0.374036f, 0.207127f, -0.903989f, 
		-0.463240f, 0.210112f, -0.860965f, -0.546472f, 0.252532f, -0.798496f, 
		0.481403f, -0.363660f, 0.797498f, 0.364206f, -0.258517f, 0.894719f, 
		0.314890f, -0.189417f, 0.930035f, -0.00161458f, -0.999879f, -0.0154939f, 
		-0.199761f, -0.153121f, -0.967807f, -0.502283f, -0.0156359f, -0.864562f, 
		-0.514374f, 0.120382f, -0.849074f, -0.520393f, 0.0435231f, -0.852817f, 
		-0.683820f, -0.333661f, 0.648892f, -0.000205909f, 2.14019e-005f, -1.00000f, 
		0.000222053f, -4.26918e-005f, 1.00000f, 0.000215746f, -3.90510e-005f, 1.00000f, 
		-0.000205909f, 2.14019e-005f, -1.00000f, 0.630781f, 0.425475f, -0.648911f, 
		-0.630765f, 0.425471f, 0.648930f, -0.000215972f, -3.88223e-005f, -1.00000f, 
		0.000205309f, 2.03513e-005f, 1.00000f, 0.000205309f, 2.03513e-005f, 1.00000f, 
		-0.000221224f, -4.18546e-005f, -1.00000f, 0.683809f, -0.333668f, -0.648900f, 
		0.00160629f, 0.999879f, -0.0154887f, 0.199344f, 0.153096f, -0.967896f, 
		-0.0941563f, 0.00675119f, -0.995535f, 0.0785000f, 0.00560691f, 0.996898f, 
		0.0785000f, 0.00560691f, 0.996898f, -0.808569f, 0.543524f, -0.225386f, 
		-0.769076f, 0.465071f, -0.438442f, -0.800084f, 0.594551f, -0.0798386f, 
		-0.967706f, 0.251903f, -0.00949995f, -0.751406f, 0.363713f, -0.550546f, 
		-0.978564f, -0.106105f, -0.176507f, -0.879846f, 0.0434786f, -0.473265f, 
		0.800084f, -0.594551f, 0.0798386f, 0.729426f, -0.591292f, 0.343963f, 
		0.808569f, -0.543524f, 0.225386f, 0.871806f, -0.235449f, 0.429557f, 
		0.978083f, -0.197375f, 0.0663098f, 0.975663f, 0.0207877f, 0.218287f, 
		0.993776f, 0.0849957f, 0.0720068f, -0.399778f, 0.679434f, -0.615261f, 
		0.365972f, -0.680747f, 0.634545f, -0.366433f, 0.680017f, 0.635063f, 
		0.366477f, -0.680030f, -0.635023f, -0.949009f, -0.0237466f, 0.314354f, 
		-0.835434f, 0.110565f, 0.538354f, -0.985019f, 0.0141139f, 0.171866f, 
		0.908440f, 0.0167922f, -0.417679f, 0.999811f, 0.00125693f, -0.0193982f, 
		0.830948f, -0.0458602f, -0.554456f, -0.989496f, 0.118253f, 0.0831482f, 
		-0.914252f, 0.387390f, -0.118622f, -0.991793f, 0.0183601f, -0.126529f, 
		-0.812183f, 0.243327f, -0.530237f, -0.785521f, 0.451702f, -0.422991f, 
		0.866709f, -0.498656f, 0.0125698f, 0.873548f, -0.455392f, -0.171852f, 
		0.954219f, -0.291104f, -0.0687285f, 0.922352f, -0.0183547f, 0.385914f, 
		0.786699f, -0.383954f, 0.483408f, 0.815603f, -0.118255f, 0.566399f, 
		0.331717f, -0.0191399f, 0.943185f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		-2.68247e-006f, 1.00000f, -1.00611e-005f, 8.16089e-006f, -1.00000f, 2.51535e-006f, 
		-0.999814f, -0.0192104f, 0.00147379f, -0.831007f, -0.554385f, -0.0456534f, 
		-0.908481f, -0.417581f, 0.0169898f, 0.985046f, 0.171731f, 0.0139010f, 
		0.835543f, 0.538233f, 0.110328f, 0.949050f, 0.314216f, -0.0239411f, 
		-0.406044f, 0.0573117f, 0.912055f, -2.53304e-006f, -1.00000f, -9.77034e-007f, 
		-2.53304e-006f, -1.00000f, -9.77034e-007f, 3.58729e-006f, 1.00000f, -9.35137e-007f, 
		-0.819379f, 0.572304f, 0.0329439f, -0.826274f, 0.531622f, 0.186144f, 
		-0.946627f, 0.315416f, 0.0664149f, -0.770077f, 0.391896f, 0.503388f, 
		-0.978502f, -0.106079f, 0.176863f, -0.889452f, -0.107418f, 0.444226f, 
		0.849561f, -0.478979f, -0.220964f, 0.897969f, -0.242357f, -0.367308f, 
		0.978515f, -0.171825f, -0.113950f, 0.975111f, 0.0182559f, -0.220966f, 
		0.993752f, 0.0849567f, -0.0723854f, -0.949138f, -0.0237283f, -0.313964f, 
		-0.835668f, 0.110557f, -0.537993f, -0.985091f, 0.0141353f, -0.171453f, 
		0.908651f, 0.0167769f, 0.417220f, 0.999819f, 0.00124266f, 0.0189920f, 
		0.831237f, -0.0458622f, 0.554024f, 0.675977f, 0.0881187f, 0.731635f, 
		0.588722f, 0.304315f, 0.748865f, 0.542150f, 0.382423f, 0.748216f, 
		0.433240f, 0.640568f, 0.634015f, 0.613606f, 0.185818f, 0.767437f, 
		-0.377219f, 0.660218f, -0.649475f, 0.374130f, -0.660242f, 0.651235f, 
		0.111884f, -0.961469f, -0.251115f, -0.373866f, 0.660188f, 0.651442f, 
		0.376960f, -0.660205f, -0.649639f, -0.554004f, -0.313189f, -0.771358f, 
		0.136792f, 0.0848979f, 0.986955f, -0.287105f, -0.161634f, -0.944164f, 
		0.136792f, 0.0848979f, 0.986955f, -0.138056f, -0.0722655f, -0.987784f, 
		-0.138056f, -0.0722655f, -0.987784f, -0.138056f, -0.0722655f, -0.987784f, 
		0.467272f, 0.358292f, 0.808259f, 0.136792f, 0.0848979f, 0.986955f, 
		0.205109f, 0.156383f, 0.966165f, 0.140631f, 0.113727f, -0.983508f, 
		-0.113448f, -0.140696f, 0.983531f, -0.113448f, -0.140696f, 0.983531f, 
		0.140631f, 0.113727f, -0.983508f, 0.186398f, 0.869316f, 0.457762f, 
		-0.000428722f, -1.00000f, -0.000650552f, 0.0775278f, 0.970672f, 0.227563f, 
		0.142054f, 0.926453f, 0.348575f, -0.000353814f, -0.999999f, 0.00117054f, 
		0.381910f, 0.308925f, 0.871040f, -0.492793f, 0.807368f, 0.324519f, 
		0.458964f, -0.838899f, -0.292576f, -0.136410f, -0.359637f, -0.923068f, 
		0.136019f, 0.359619f, -0.923132f, -0.000945107f, -0.999996f, -0.00256469f, 
		0.350565f, 0.338617f, -0.873180f, -0.0986492f, 0.770217f, 0.630107f, 
		-0.818105f, -0.575068f, 0.000173243f, -0.000285120f, -9.25294e-005f, -1.00000f, 
		-0.907076f, 0.420966f, 0.000191999f, -0.000180036f, 2.98304e-005f, -1.00000f, 
		-0.0989111f, 0.770215f, -0.630068f, -0.168009f, -0.857850f, -0.485661f, 
		-0.111185f, 0.0731389f, 0.991105f, 0.116111f, -0.0918404f, -0.988981f, 
		-0.111185f, 0.0731389f, 0.991105f, -0.907077f, -0.420965f, 0.000191999f, 
		0.000284772f, 9.58618e-005f, 1.00000f, -0.818116f, 0.575053f, 0.000173245f, 
		0.000177118f, -3.01984e-005f, 1.00000f, -0.0999920f, 0.959280f, -0.264165f, 
		-0.232733f, 0.969878f, 0.0719181f, -0.232733f, 0.969878f, 0.0719181f, 
		0.315960f, -0.948700f, -0.0117679f, -0.105199f, -0.670131f, -0.734750f, 
		-0.0998826f, 0.959281f, 0.264203f, 0.104229f, 0.666198f, -0.738455f, 
		0.0911088f, -0.770670f, -0.630688f, 0.0913662f, -0.770670f, 0.630651f, 
		0.00114184f, -0.999998f, -0.00170450f, 0.529059f, 0.422244f, -0.736075f, 
		0.211734f, -0.976779f, -0.0327410f, 0.211734f, -0.976779f, -0.0327410f, 
		-0.248740f, 0.968570f, 0.000597854f, -0.248740f, 0.968570f, 0.000597854f, 
		0.211734f, -0.976779f, -0.0327410f, -0.319218f, -0.780335f, -0.537752f, 
		0.306322f, 0.780339f, 0.545196f, -0.000105180f, 1.00000f, -0.000142337f, 
		-0.241002f, -0.374273f, -0.895454f, -0.306095f, -0.780323f, 0.545346f, 
		0.318996f, 0.780325f, -0.537898f, -0.385103f, 0.633453f, -0.671143f, 
		0.396884f, -0.612975f, 0.683187f, 1.03186e-006f, -1.00000f, 0.000168499f, 
		0.254997f, 0.351399f, -0.900830f, -0.395309f, 0.621149f, 0.676687f, 
		0.388085f, -0.621135f, -0.680868f, -0.000626473f, 1.00000f, -0.000429779f, 
		-0.0909741f, -0.982291f, 0.163795f, -0.498254f, -0.858487f, 0.121418f, 
		0.469282f, 0.876919f, -0.103863f, -0.469568f, 0.876876f, 0.102927f, 
		0.498272f, -0.858486f, -0.121356f, -0.306348f, -0.780320f, -0.545209f, 
		0.319232f, 0.780331f, 0.537750f, 0.000977834f, -0.999999f, 0.00117580f, 
		0.240845f, 0.351895f, 0.904524f, 0.306068f, 0.780357f, -0.545313f, 
		-0.318983f, -0.780346f, 0.537875f, -0.496711f, -0.0914638f, -0.863083f, 
		0.499451f, 0.0914836f, 0.861498f, -0.000123117f, 1.00000f, 0.000404020f, 
		-0.226089f, -0.374592f, 0.899202f, -0.498005f, -0.115215f, 0.859486f, 
		0.494980f, 0.115222f, -0.861231f, -0.496513f, -0.857167f, -0.136891f, 
		0.468470f, 0.875093f, 0.121438f, -0.468494f, 0.875081f, -0.121433f, 
		0.496533f, -0.857162f, 0.136850f, -0.00308486f, -0.999986f, 0.00427190f, 
		-0.558449f, 0.641243f, 0.526252f, -0.521822f, 0.588562f, 0.617492f, 
		-0.480404f, 0.526213f, 0.701649f, 0.219711f, 0.702133f, 0.677301f, 
		-0.483783f, 0.228943f, 0.844713f, 0.489310f, -0.228950f, -0.841521f, 
		-0.224318f, -0.692517f, 0.685640f, -0.478083f, -0.829201f, -0.289588f, 
		0.456851f, 0.804445f, 0.379678f, -0.424774f, 0.843996f, -0.327472f, 
		0.478417f, -0.828626f, 0.290684f, -0.656366f, 0.228869f, -0.718890f, 
		-0.933126f, -0.349050f, -0.0862559f, -0.0998837f, -0.00322442f, 0.994994f, 
		0.804797f, 0.0650557f, 0.589974f, 0.0998833f, -0.00322725f, -0.994994f, 
		-0.488360f, -0.846260f, 0.212951f, 0.521728f, 0.812236f, -0.260908f, 
		-0.521732f, 0.812234f, 0.260905f, 0.488373f, -0.846259f, -0.212926f, 
		-0.477681f, 0.324655f, -0.816346f, 0.468149f, -0.324660f, 0.821847f, 
		-0.417125f, -0.769402f, 0.483763f, -0.496018f, -0.786133f, 0.368730f, 
		-0.471182f, -0.874532f, 0.114809f, -0.399081f, -0.840496f, 0.366470f, 
		-0.465536f, -0.838204f, 0.284060f, 0.389404f, 0.735721f, -0.554148f, 
		-0.387381f, -0.916903f, 0.0960417f, 0.451293f, 0.892230f, -0.0161217f, 
		0.479120f, 0.866624f, -0.139307f, -0.425930f, -0.796450f, -0.429244f, 
		-0.522490f, -0.839855f, -0.147133f, 0.507708f, 0.774628f, 0.377073f, 
		0.510692f, 0.827088f, 0.234776f, -0.539108f, -0.841743f, -0.0288131f, 
		0.411599f, 0.830442f, 0.375435f, 0.476277f, 0.829405f, 0.291974f, 
		0.386802f, 0.912715f, 0.131660f, -0.515339f, 0.797646f, -0.313347f, 
		-0.441454f, 0.877953f, -0.185251f, 0.248561f, 0.468108f, 0.847993f, 
		0.417122f, -0.540950f, 0.730330f, 0.484135f, -0.873976f, -0.0421940f, 
		0.544983f, -0.778363f, 0.311682f, 0.538472f, -0.839240f, 0.0756603f, 
		-0.480368f, 0.843319f, -0.240956f, -0.535935f, 0.814477f, 0.222262f, 
		-0.494943f, 0.853461f, 0.163203f, -0.467945f, 0.879936f, 0.0820931f, 
		0.438185f, -0.885242f, -0.156014f, -0.782649f, -0.468305f, -0.410062f, 
		-0.823636f, -0.543818f, -0.160892f, -0.833984f, -0.486168f, -0.260980f, 
		0.823701f, 0.488822f, 0.287349f, 0.827708f, 0.436826f, 0.352253f, 
		0.908376f, 0.340893f, 0.242167f, 0.764055f, 0.543099f, 0.348230f, 
		0.824280f, 0.526863f, 0.207312f, -0.642985f, 0.240443f, 0.727157f, 
		-0.933115f, -0.348995f, 0.0865983f, -0.100276f, -0.0107430f, -0.994902f, 
		0.801840f, 0.0611568f, -0.594400f, 0.112287f, -0.000158733f, 0.993676f, 
		-0.794342f, 0.602413f, -0.0782242f, 0.838819f, -0.501424f, -0.212030f, 
		0.662092f, -0.513396f, 0.545947f, -0.0990114f, -0.994392f, -0.0371639f, 
		0.0940281f, 0.995508f, 0.0110627f, 0.192721f, -0.669444f, 0.717428f, 
		-0.198336f, 0.678006f, 0.707793f, -0.774314f, 0.163894f, 0.611209f, 
		-0.752197f, 0.394339f, 0.527917f, -0.898320f, 0.300349f, 0.320644f, 
		-0.815362f, 0.398424f, 0.420051f, 0.782479f, -0.410217f, -0.468454f, 
		0.823487f, -0.161039f, -0.544000f, 0.833837f, -0.261131f, -0.486340f, 
		-0.479857f, -0.833553f, -0.273727f, 0.511228f, 0.799650f, 0.314970f, 
		-0.513194f, 0.801739f, -0.306344f, 0.480971f, -0.835715f, 0.265043f, 
		-0.774928f, 0.417637f, 0.474411f, -0.840855f, 0.483077f, 0.244131f, 
		-0.800677f, 0.480463f, 0.357872f, 0.840855f, -0.483077f, -0.244131f, 
		0.774928f, -0.417637f, -0.474411f, 0.800677f, -0.480463f, -0.357872f, 
		-0.682977f, -0.392498f, 0.616026f, -0.769431f, -0.378290f, 0.514657f, 
		-0.796088f, -0.490719f, 0.354174f, -0.780624f, -0.443060f, 0.440822f, 
		-0.772482f, -0.573166f, 0.273408f, 0.922401f, 0.382703f, 0.0521091f, 
		0.881653f, 0.438908f, -0.173344f, 0.847520f, 0.461815f, -0.261603f, 
		0.795289f, 0.489596f, -0.357508f, 0.713044f, 0.555706f, -0.427503f, 
		-0.511067f, -0.799645f, 0.315244f, 0.480575f, 0.833829f, -0.271619f, 
		-0.478875f, 0.835055f, 0.270855f, 0.473752f, -0.836094f, -0.276597f, 
		0.420145f, -0.739723f, -0.525631f, -0.438710f, 0.738857f, 0.511492f, 
		0.438700f, 0.738872f, -0.511478f, -0.420134f, -0.739754f, 0.525596f, 
		-0.782493f, -0.468280f, 0.410389f, -0.823575f, -0.543810f, 0.161229f, 
		-0.833889f, -0.486157f, 0.261307f, 0.823593f, 0.488813f, -0.287672f, 
		0.827579f, 0.436801f, -0.352587f, 0.908296f, 0.340881f, -0.242484f, 
		0.763914f, 0.543099f, -0.348537f, 0.824203f, 0.526852f, -0.207644f, 
		-0.810153f, -0.314221f, -0.494891f, -0.778917f, -0.538240f, -0.321848f, 
		-0.860244f, -0.171733f, -0.480091f, 0.778483f, 0.417591f, 0.468596f, 
		0.865358f, 0.0192217f, 0.500786f, 0.742672f, 0.554392f, 0.375617f, 
		-0.712930f, -0.427503f, 0.555853f, -0.847456f, -0.261548f, 0.461965f, 
		-0.795209f, -0.357485f, 0.489742f, -0.881584f, -0.173241f, 0.439088f, 
		-0.922322f, 0.0522635f, 0.382870f, 0.772382f, 0.273352f, -0.573328f, 
		0.780560f, 0.440782f, -0.443212f, 0.795997f, 0.354147f, -0.490886f, 
		0.769391f, 0.514630f, -0.378409f, 0.682921f, 0.616018f, -0.392607f, 
		0.438958f, 0.511409f, 0.738767f, -0.420612f, -0.511405f, -0.749367f, 
		-0.420420f, 0.554508f, -0.718170f, 0.412048f, -0.554497f, 0.723015f, 
		0.424110f, -0.553345f, -0.716896f, -0.408498f, 0.553350f, 0.725902f, 
		0.420330f, 0.511371f, -0.749548f, -0.438662f, -0.511381f, 0.738962f, 
		-0.411752f, 0.554524f, 0.723163f, 0.420126f, -0.554486f, -0.718359f, 
		-0.683284f, -0.392536f, -0.615660f, -0.769678f, -0.378306f, -0.514276f, 
		-0.796252f, -0.490734f, -0.353782f, -0.780820f, -0.443078f, -0.440456f, 
		-0.772573f, -0.573202f, -0.273076f, 0.922390f, 0.382681f, -0.0524511f, 
		0.881734f, 0.438894f, 0.172964f, 0.847636f, 0.461829f, 0.261202f, 
		0.795445f, 0.489624f, 0.357120f, 0.713233f, 0.555760f, 0.427118f, 
		-0.865230f, 0.501016f, -0.0189818f, -0.731415f, 0.373068f, -0.570835f, 
		-0.770956f, 0.467925f, -0.432057f, 0.860171f, -0.480319f, 0.171463f, 
		0.779017f, -0.322110f, 0.537939f, 0.810124f, -0.495117f, 0.313940f, 
		-0.438915f, 0.738809f, -0.511385f, 0.420331f, -0.739705f, 0.525507f, 
		-0.420317f, -0.739736f, -0.525475f, 0.438905f, 0.738826f, 0.511369f, 
		-0.424094f, -0.716737f, 0.553563f, 0.407316f, 0.736576f, -0.539953f, 
		-0.407331f, 0.736578f, 0.539939f, 0.424104f, -0.716719f, -0.553579f, 
		-0.213536f, -0.702157f, -0.679248f, -0.253234f, -0.644433f, -0.721511f, 
		-0.281753f, -0.579444f, -0.764761f, 0.271483f, 0.546283f, -0.792383f, 
		-0.0239733f, 0.0104484f, -0.999658f, -0.378270f, 0.0509921f, 0.924290f, 
		-0.723576f, 0.0924128f, 0.684030f, 0.196673f, -0.0473337f, -0.979326f, 
		0.707005f, -0.184645f, -0.682679f, 0.376114f, -0.0957441f, -0.921613f, 
		-0.192123f, 0.0453534f, 0.980322f, 0.263421f, 0.550090f, 0.792471f, 
		-0.243170f, -0.652928f, 0.717324f, -0.351349f, -0.924844f, -0.145661f, 
		-0.670455f, -0.688101f, -0.277500f, 0.190893f, 0.980694f, 0.0424249f, 
		0.719424f, 0.688102f, 0.0945775f, -0.186990f, -0.980659f, -0.0578071f, 
		0.377162f, 0.924870f, 0.0486300f, 0.189167f, -0.0580898f, 0.980225f, 
		-0.374423f, 0.0511673f, -0.925845f, -0.714339f, 0.0977097f, -0.692945f, 
		0.652651f, -0.273157f, 0.706705f, 0.341537f, -0.142680f, 0.928975f, 
		-0.187305f, 0.0455251f, -0.981246f, 0.0171843f, -0.00102494f, 0.999852f, 
		0.527843f, -0.669734f, -0.522340f, -0.353517f, 0.758454f, -0.547516f, 
		0.324290f, -0.732863f, 0.598120f, -0.392521f, -0.647937f, -0.652767f, 
		0.363292f, 0.674648f, 0.642549f, -0.363284f, 0.674606f, -0.642597f, 
		0.392506f, -0.647944f, 0.652769f, -0.371104f, 0.627981f, 0.684048f, 
		0.385959f, -0.698835f, -0.602216f, -0.363056f, -0.674681f, 0.642648f, 
		0.392255f, 0.648006f, -0.652859f, -0.671234f, 0.340769f, -0.658271f, 
		-0.693368f, -0.669624f, -0.266165f, -0.976487f, 0.0168973f, 0.214915f, 
		-0.812374f, -0.578036f, -0.0769603f, -0.950177f, -0.250094f, 0.186052f, 
		-0.670336f, 0.332693f, -0.663299f, -0.976485f, 0.0168154f, 0.214930f, 
		-0.807701f, -0.585563f, 0.0688050f, -0.921287f, -0.378663f, 0.0885760f, 
		-0.598082f, 0.381931f, -0.704575f, -0.933040f, 0.357182f, 0.0430989f, 
		-0.777723f, -0.626797f, 0.0476618f, -0.605869f, 0.412377f, -0.680344f, 
		-0.674086f, 0.397736f, 0.622426f, -0.957148f, -0.286606f, -0.0415311f, 
		-0.435512f, 0.477683f, -0.762986f, 0.448112f, -0.443997f, 0.775927f, 
		-0.447540f, 0.460362f, 0.766665f, 0.439860f, -0.460372f, -0.771091f, 
		-0.0629207f, -0.869249f, -0.490353f, 0.0659539f, 0.856762f, -0.511478f, 
		-0.354805f, -0.614702f, -0.704453f, 0.310956f, 0.619301f, 0.720953f, 
		-0.355458f, 0.613615f, -0.705072f, 0.355458f, -0.613615f, 0.705072f, 
		-0.768854f, 0.618944f, 0.160538f, -0.488760f, 0.431898f, -0.758009f, 
		-0.966585f, -0.238271f, 0.0945534f, -0.279047f, -0.530675f, 0.800323f, 
		0.301737f, 0.455334f, -0.837631f, 0.253611f, -0.468373f, -0.846350f, 
		-0.278391f, 0.453752f, 0.846527f, -0.102622f, 0.958326f, 0.266609f, 
		-0.0236999f, 0.00695224f, 0.999695f, -0.441597f, 0.751758f, -0.489746f, 
		0.440337f, -0.424652f, -0.791059f, 0.482736f, -0.229714f, -0.845102f, 
		0.455535f, -0.336707f, -0.824085f, 0.320401f, 0.799882f, 0.507476f, 
		-0.254802f, -0.824366f, -0.505466f, 0.253665f, 0.846297f, -0.468440f, 
		-0.278490f, -0.846462f, 0.453813f, 0.286257f, -0.838636f, -0.463408f, 
		-0.257993f, 0.838632f, 0.479725f, 0.410469f, -0.0162222f, -0.911730f, 
		-1.73730e-005f, 1.00000f, -2.61493e-006f, 8.16089e-006f, -1.00000f, 2.51535e-006f, 
		-0.109259f, 0.959010f, -0.261462f, -0.320459f, 0.507459f, -0.799869f, 
		0.243997f, -0.489570f, 0.837130f, -0.254824f, -0.505463f, -0.824361f, 
		0.320449f, 0.507451f, 0.799879f, -0.286187f, -0.463376f, 0.838677f, 
		0.257942f, 0.479705f, -0.838659f, 0.392943f, -0.263684f, 0.880946f, 
		-0.257958f, 0.479705f, 0.838654f, 0.286192f, -0.463388f, -0.838669f, 
		-0.471641f, 0.350847f, -0.808988f, 0.465107f, -0.350835f, 0.812767f, 
		-0.464754f, 0.350852f, 0.812961f, 0.471295f, -0.350875f, -0.809177f, 
		0.135356f, -0.919152f, 0.369916f, -0.636664f, -0.110268f, 0.763217f, 
		-0.849634f, 0.378155f, -0.367587f, -0.752373f, -0.270114f, 0.600811f, 
		-0.788781f, -0.393250f, 0.472417f, 0.597002f, -0.200818f, 0.776699f, 
		0.654971f, -0.479892f, 0.583709f, 0.497870f, -0.0685901f, 0.864535f, 
		0.984915f, 0.160718f, -0.0641207f, 0.983632f, -0.170128f, 0.0593690f, 
		0.488023f, 0.869568f, -0.0754053f, 0.968105f, -0.0115304f, 0.250280f, 
		0.995990f, -0.0662517f, 0.0601255f, 0.976141f, -0.192840f, -0.0998121f, 
		0.436834f, 0.895350f, -0.0867481f, 0.983637f, -0.170092f, 0.0593884f, 
		0.498160f, 0.863762f, -0.0758378f, -0.636960f, -0.110262f, -0.762970f, 
		-0.946843f, 0.146038f, 0.286637f, -0.746135f, -0.261918f, -0.612112f, 
		-0.920372f, -0.331854f, -0.206854f, -0.987668f, -0.0896403f, 0.128365f, 
		0.717086f, -0.676837f, 0.166374f, 0.555920f, -0.0961748f, -0.825654f, 
		0.694633f, 0.0222019f, -0.719022f, 0.903730f, 0.263720f, -0.337229f, 
		0.963058f, 0.263762f, -0.0543050f, -0.963187f, -0.0531593f, -0.263523f, 
		-0.537862f, -0.820660f, 0.192928f, -0.720538f, 0.0802779f, 0.688753f, 
		-0.607876f, 0.774115f, 0.176728f, 0.963187f, 0.0531593f, 0.263523f, 
		0.607876f, -0.774115f, -0.176728f, 0.537862f, 0.820660f, -0.192928f, 
		0.720538f, -0.0802779f, -0.688753f, -0.130726f, 0.879364f, -0.457853f, 
		0.160041f, 0.0317074f, -0.986601f, -0.700047f, -0.320332f, 0.638218f, 
		0.357775f, 0.152809f, -0.921220f, 0.677350f, 0.274841f, -0.682392f, 
		0.0339413f, 0.0142520f, 0.999322f, -0.914169f, -0.272439f, -0.300121f, 
		-0.601113f, 0.390861f, 0.697059f, -0.572068f, 0.641137f, -0.511548f, 
		0.871438f, 0.267338f, 0.411249f, 0.468441f, -0.679066f, 0.565183f, 
		0.601113f, -0.390861f, -0.697059f, 0.142656f, -0.885930f, -0.441336f, 
		-0.863896f, -0.304126f, 0.401486f, -0.772534f, 0.243296f, -0.586514f, 
		-0.579415f, 0.720999f, 0.380052f, 0.852121f, 0.321630f, -0.412849f, 
		0.772534f, -0.243296f, 0.586514f, 0.575991f, -0.722063f, -0.383223f, 
		-0.946039f, 0.175618f, 0.272338f, -0.903208f, 0.409541f, 0.128416f, 
		-0.565832f, -0.805187f, -0.177502f, -0.782845f, 0.620600f, 0.0448272f, 
		-0.726723f, 0.121882f, -0.676032f, 0.903208f, -0.409541f, -0.128416f, 
		0.946039f, -0.175618f, -0.272338f, 0.782845f, -0.620600f, -0.0448272f, 
		0.565832f, 0.805187f, 0.177502f, 0.726723f, -0.121882f, 0.676032f, 
		-0.964343f, -0.253542f, 0.0758914f, -0.583133f, 0.591732f, -0.556605f, 
		-0.458953f, 0.0845635f, 0.884427f, 0.964343f, 0.253542f, -0.0758914f, 
		0.427607f, -0.178532f, -0.886159f, 0.568377f, -0.586866f, 0.576659f, 
		-0.0324833f, 0.999441f, 0.00785373f, -0.677307f, -0.682415f, 0.274890f, 
		-0.357862f, -0.921185f, 0.152814f, 0.699944f, 0.638304f, -0.320385f, 
		-0.160239f, -0.986568f, 0.0317235f, 0.0693862f, -0.0546517f, 0.996092f, 
		-0.0509938f, 0.0668705f, -0.996458f, -0.0509938f, 0.0668705f, -0.996458f, 
		0.0693862f, -0.0546517f, 0.996092f, -0.0607188f, -0.0478065f, -0.997009f, 
		0.0509745f, 0.0668760f, 0.996458f, -0.0607188f, -0.0478065f, -0.997009f, 
		-0.0607188f, -0.0478065f, -0.997009f, 0.0509745f, 0.0668760f, 0.996458f, 
		0.160463f, 0.0316687f, 0.986534f, -0.700453f, -0.320398f, -0.637739f, 
		0.358221f, 0.152828f, 0.921044f, 0.677733f, 0.274894f, 0.681990f, 
		0.0322385f, 0.00783874f, -0.999449f, -0.829046f, 0.0513298f, 0.556820f, 
		-0.781784f, 0.448436f, 0.433265f, -0.945238f, -0.160426f, 0.284232f, 
		-0.996753f, 0.0785853f, 0.0175505f, -0.875605f, 0.475253f, 0.0863124f, 
		-0.708309f, -0.103712f, 0.698242f, -0.884955f, 0.426435f, -0.187104f, 
		0.708309f, 0.103712f, -0.698242f, -0.475819f, -0.867784f, 0.143342f, 
		0.813674f, -0.545619f, 0.200584f, 0.510913f, 0.858587f, -0.0423784f, 
		-0.384883f, 0.313592f, -0.868058f, -0.480684f, 0.254670f, -0.839098f, 
		0.487852f, -0.213314f, 0.846462f, -0.487954f, 0.235379f, 0.840534f, 
		0.483583f, -0.235415f, -0.843046f, -0.979040f, -0.141539f, 0.146447f, 
		-0.243507f, 0.134945f, -0.960466f, -0.604208f, 0.702311f, 0.376420f, 
		0.979040f, 0.141539f, -0.146447f, 0.243507f, -0.134945f, 0.960466f, 
		0.653052f, -0.693369f, -0.304571f, 0.141825f, -0.982783f, 0.118416f, 
		0.141825f, -0.982783f, 0.118416f, -0.219842f, 0.965487f, -0.139658f, 
		-0.219842f, 0.965487f, -0.139658f, 0.141825f, -0.982783f, 0.118416f, 
		0.0581795f, 0.961134f, 0.269884f, -0.497797f, 0.115464f, -0.859573f, 
		0.495710f, -0.115528f, 0.860770f, 0.495710f, -0.115528f, 0.860770f, 
		0.547350f, -0.0635207f, 0.834490f, -0.500291f, 0.0794561f, -0.862204f, 
		0.497252f, -0.115562f, -0.859876f, -0.495476f, 0.115609f, 0.860894f, 
		0.497252f, -0.115562f, -0.859876f, 0.493235f, -0.139584f, -0.858624f, 
		-0.526171f, 0.167674f, 0.833684f, -0.981969f, 0.189034f, 0.00172963f, 
		0.978383f, -0.185630f, 0.0911537f, 0.291673f, 0.948477f, -0.123769f, 
		-0.291673f, -0.948477f, 0.123769f, -0.497557f, 0.0553979f, -0.865660f, 
		0.543319f, -0.0442021f, 0.838362f, 0.543319f, -0.0442021f, 0.838362f, 
		-0.526549f, 0.167598f, -0.833461f, 0.493753f, -0.139524f, 0.858336f, 
		-0.499265f, 0.0553632f, 0.864679f, 0.498817f, -0.0553251f, -0.864940f, 
		0.498817f, -0.0553251f, -0.864940f, -0.00103513f, 0.00929151f, -0.999956f, 
		-0.488090f, 0.676167f, 0.551875f, 0.00403171f, 0.122966f, 0.992403f, 
		-0.490517f, 0.169110f, 0.854865f, 0.494717f, -0.169123f, -0.852440f, 
		-0.407821f, 0.908531f, 0.0908472f, -0.987198f, -0.0450682f, 0.153001f, 
		-0.662991f, 0.418204f, -0.620925f, -0.281391f, 0.0152700f, -0.959472f, 
		-1.45169e-005f, -1.00000f, 9.90107e-006f, 0.000000f, 1.00000f, -5.78099e-015f, 
		-0.860676f, -0.454222f, 0.230043f, -0.490909f, 0.0525545f, 0.869624f, 
		-0.686199f, 0.694174f, 0.217381f, 0.478805f, -0.315874f, 0.819127f, 
		-0.462199f, 0.378233f, -0.802067f, -0.135833f, 0.227143f, -0.964342f, 
		0.152424f, -0.267041f, 0.951555f, -0.151455f, -0.266799f, -0.951777f, 
		0.136387f, 0.227626f, 0.964150f, -0.964343f, -0.253542f, 0.0758914f, 
		-0.411358f, 0.410992f, -0.813554f, -0.242137f, -0.0782708f, -0.967080f, 
		-0.317090f, 0.0376867f, -0.947646f, 0.255592f, 0.140689f, 0.956493f, 
		0.964343f, 0.253542f, -0.0758914f, 0.343099f, -0.0112058f, 0.939232f, 
		0.461301f, -0.467687f, 0.753970f, 0.580831f, -0.589857f, 0.560985f, 
		-0.768854f, 0.618944f, 0.160538f, -0.410133f, 0.119006f, 0.904228f, 
		-0.609225f, 0.0395843f, 0.792008f, -0.812511f, -0.231639f, 0.534948f, 
		0.668101f, -0.733436f, 0.125351f, 0.441784f, -0.249758f, -0.861654f, 
		0.985298f, -0.0218461f, -0.169443f, 0.291634f, -0.838584f, -0.460138f, 
		-0.342827f, 0.867754f, 0.359823f, -0.495321f, -0.861491f, -0.111760f, 
		0.495321f, 0.861491f, 0.111760f, -0.485000f, 0.278310f, -0.829046f, 
		-0.461083f, 0.128767f, -0.877965f, -0.866794f, -0.473963f, -0.155006f, 
		-0.395211f, 0.0252344f, -0.918244f, 0.420263f, 0.0196229f, -0.907190f, 
		0.929781f, 0.342891f, -0.133916f, 0.843767f, 0.411904f, -0.344082f, 
		0.546163f, 0.204857f, -0.812244f, -0.458794f, 0.882918f, -0.0998199f, 
		-0.456839f, 0.823792f, -0.335656f, -0.535201f, 0.838808f, -0.0998091f, 
		-0.501062f, 0.820070f, -0.276446f, -0.575069f, 0.707735f, -0.410374f, 
		-0.520680f, 0.676078f, -0.521355f, 0.416881f, -0.817023f, 0.398352f, 
		0.454937f, -0.854357f, 0.251211f, 0.510239f, -0.762596f, 0.397622f, 
		0.476843f, -0.818070f, 0.321532f, 0.591561f, -0.793070f, 0.145245f, 
		0.0596735f, 0.0710025f, -0.995690f, -0.962453f, -0.269464f, 0.0327519f, 
		-0.0596735f, -0.0710025f, 0.995690f, -0.601113f, 0.390861f, 0.697059f, 
		0.962453f, 0.269464f, -0.0327519f, 0.601113f, -0.390861f, -0.697059f, 
		-0.0596735f, -0.0710025f, 0.995690f, 0.0596735f, 0.0710025f, -0.995690f, 
		-0.200097f, -0.302439f, -0.931929f, 0.180720f, 0.365777f, 0.912988f, 
		-0.180699f, 0.365769f, -0.912996f, 0.200084f, -0.302439f, 0.931932f, 
		-0.167144f, -0.327091f, 0.930094f, 0.231248f, 0.345910f, -0.909324f, 
		0.198733f, -0.384725f, -0.901383f, -0.176526f, 0.300076f, 0.937439f, 
		-0.515878f, -0.533398f, -0.670340f, -0.572097f, -0.420663f, -0.704093f, 
		-0.499356f, -0.283982f, -0.818534f, -0.534859f, -0.102672f, -0.838680f, 
		-0.454366f, -0.344449f, -0.821527f, -0.459233f, -0.102668f, -0.882363f, 
		0.590807f, 0.148889f, 0.792956f, 0.475844f, 0.329083f, 0.815645f, 
		0.508692f, 0.406511f, 0.758934f, 0.455146f, 0.257216f, 0.852457f, 
		0.415145f, 0.407242f, 0.813516f, -0.458596f, 0.0999119f, 0.883010f, 
		-0.456615f, 0.335733f, 0.823885f, -0.535018f, 0.0999159f, 0.838911f, 
		-0.500847f, 0.276528f, 0.820174f, -0.574847f, 0.410464f, 0.707864f, 
		-0.520446f, 0.521438f, 0.676194f, 0.416628f, -0.398407f, -0.817125f, 
		0.454697f, -0.251273f, -0.854466f, 0.509996f, -0.397711f, -0.762712f, 
		0.476604f, -0.321595f, -0.818184f, 0.591371f, -0.145327f, -0.793197f, 
		-0.00462446f, -0.999964f, -0.00708532f, -0.305341f, 0.504247f, -0.807776f, 
		-0.542585f, -0.717116f, 0.437431f, -0.395256f, -0.712324f, 0.579972f, 
		-0.467776f, -0.727537f, 0.501872f, 0.419072f, 0.698580f, -0.579970f, 
		0.349614f, 0.828544f, -0.437361f, 0.396017f, 0.768983f, -0.501833f, 
		-0.309879f, 0.751413f, 0.582541f, -0.358863f, 0.831700f, 0.423665f, 
		-0.427740f, 0.688874f, 0.585228f, -0.539334f, 0.784102f, 0.307088f, 
		0.373521f, -0.893819f, -0.248131f, 0.496997f, -0.799374f, -0.337631f, 
		0.518306f, -0.646156f, -0.560215f, 0.385855f, -0.791345f, -0.474224f, 
		0.737672f, -0.662992f, 0.127601f, 0.978231f, 0.165301f, 0.125455f, 
		0.415355f, -0.0518343f, 0.908181f, -0.862235f, 0.302866f, -0.405983f, 
		-0.799212f, -0.591528f, 0.106563f, -0.237019f, -0.260926f, -0.935809f, 
		-0.415058f, 0.813581f, -0.407201f, -0.455034f, 0.852528f, -0.257176f, 
		-0.508627f, 0.759014f, -0.406444f, -0.475746f, 0.815717f, -0.329045f, 
		-0.590673f, 0.793065f, -0.148840f, 0.459063f, -0.882457f, 0.102624f, 
		0.454245f, -0.821624f, 0.344377f, 0.534683f, -0.838800f, 0.102611f, 
		0.499230f, -0.818633f, 0.283917f, 0.572034f, -0.704200f, 0.420571f, 
		0.515842f, -0.670449f, 0.533297f, -0.522713f, -0.665390f, -0.532942f, 
		-0.529526f, -0.764729f, -0.367140f, -0.500240f, -0.810272f, -0.305317f, 
		-0.532715f, -0.835663f, -0.133728f, -0.455174f, -0.810705f, -0.368204f, 
		-0.492226f, -0.840450f, -0.226619f, -0.463675f, -0.875739f, -0.134484f, 
		0.526614f, 0.839402f, 0.134470f, 0.481803f, 0.846468f, 0.226621f, 
		0.474622f, 0.799468f, 0.368218f, 0.457377f, 0.879158f, 0.133743f, 
		0.451693f, 0.838295f, 0.305345f, 0.397628f, 0.840871f, 0.367191f, 
		0.315058f, 0.785255f, 0.533022f, -0.164409f, 0.962417f, -0.216155f, 
		-0.164409f, 0.962417f, -0.216155f, 0.242649f, -0.940404f, 0.238248f, 
		0.433885f, -0.498452f, -0.750526f, -0.429495f, 0.557670f, 0.710309f, 
		-0.0921154f, 0.138712f, 0.986039f, 0.0882488f, -0.179041f, -0.979876f, 
		-0.0921154f, 0.138712f, 0.986039f, -0.541268f, 0.299624f, -0.785655f, 
		-0.432018f, 0.569597f, -0.699227f, -0.359981f, 0.412308f, -0.836909f, 
		-0.314428f, 0.566904f, -0.761417f, 0.451901f, -0.400410f, 0.797155f, 
		0.523295f, -0.546147f, 0.654129f, 0.499564f, -0.329941f, 0.800984f, 
		0.389755f, -0.462235f, 0.796511f, 0.428573f, -0.329691f, 0.841207f, 
		-0.309865f, -0.582515f, 0.751438f, -0.358789f, -0.423648f, 0.831741f, 
		-0.427722f, -0.585161f, 0.688942f, -0.539250f, -0.307036f, 0.784180f, 
		0.373405f, 0.248052f, -0.893890f, 0.496912f, 0.337506f, -0.799480f, 
		0.518305f, 0.560069f, -0.646284f, 0.385818f, 0.474102f, -0.791436f, 
		-0.0918591f, 0.980067f, 0.176154f, 0.134493f, -0.971405f, -0.195665f, 
		-0.100641f, 0.980198f, -0.170541f, 0.0990110f, -0.971137f, 0.217000f, 
		-0.0769337f, 0.885247f, 0.458714f, 0.0736746f, 0.986059f, -0.149197f, 
		-0.0923071f, -0.986021f, 0.138718f, 0.0736746f, 0.986059f, -0.149197f, 
		-0.0923071f, -0.986021f, 0.138718f, -0.450418f, 0.513253f, -0.730544f, 
		0.418330f, -0.484148f, 0.768506f, -0.414589f, 0.909292f, -0.0361047f, 
		-0.595497f, 0.343033f, -0.726438f, -0.780045f, 0.609472f, 0.141685f, 
		-0.636457f, 0.771254f, -0.00944170f, -0.416724f, 0.908449f, -0.0325787f, 
		-0.673486f, 0.361020f, -0.645043f, -0.637340f, 0.770344f, 0.0191697f, 
		-0.772358f, 0.614208f, 0.161903f, -0.673429f, 0.361167f, -0.645021f, 
		-0.642101f, 0.766511f, 0.0129511f, -0.771180f, 0.615626f, 0.162129f, 
		-0.424620f, 0.904427f, -0.0413398f, -0.403378f, 0.514400f, 0.756755f, 
		0.471535f, -0.480837f, -0.739223f, -0.417878f, -0.0973521f, -0.903272f, 
		-0.962155f, -0.272502f, -0.000528624f, -0.386308f, -0.598077f, -0.702190f, 
		-0.436112f, -0.276350f, -0.856409f, -0.428327f, 0.841298f, 0.329777f, 
		-0.389451f, 0.796577f, 0.462378f, -0.499322f, 0.801097f, 0.330033f, 
		-0.523026f, 0.654193f, 0.546329f, -0.451633f, 0.797244f, 0.400536f, 
		0.314134f, -0.761470f, -0.566995f, 0.359731f, -0.837004f, -0.412332f, 
		0.431768f, -0.699282f, -0.569721f, 0.541050f, -0.785778f, -0.299695f, 
		-0.458009f, -0.797817f, 0.392066f, -0.427618f, -0.717177f, 0.550272f, 
		-0.358014f, -0.848254f, 0.390245f, -0.314069f, -0.776289f, 0.546567f, 
		-0.406125f, -0.788639f, 0.461639f, 0.479768f, 0.746109f, -0.461674f, 
		0.515073f, 0.660220f, -0.546635f, 0.555480f, 0.734240f, -0.390299f, 
		0.407106f, 0.729004f, -0.550290f, 0.461780f, 0.795622f, -0.392103f, 
		-0.116173f, -0.893629f, -0.433510f, -0.617580f, -0.785627f, -0.0372090f, 
		-0.858891f, -0.510041f, 0.0465296f, -0.967374f, -0.180345f, -0.177942f, 
		-0.443580f, -0.874016f, -0.198324f, -0.920788f, -0.0124262f, -0.389864f, 
		-0.441679f, -0.880330f, 0.173029f, -0.955988f, -0.0597141f, 0.287266f, 
		-0.0234704f, -0.00477517f, 0.999713f, -0.815134f, -0.441743f, -0.374726f, 
		0.0446697f, -0.000143416f, -0.999002f, -0.0234704f, -0.00477517f, 0.999713f, 
		0.0446697f, -0.000143416f, -0.999002f, 0.702059f, 0.532029f, 0.473348f, 
		-0.669588f, 0.739508f, 0.0691298f, -0.236354f, 0.160796f, -0.958270f, 
		0.0583965f, -0.0343627f, 0.997702f, -0.0771291f, 0.0299199f, -0.996572f, 
		0.669588f, -0.739508f, -0.0691298f, 0.349719f, -0.256235f, 0.901133f, 
		0.0234686f, -0.00478169f, -0.999713f, -0.0771291f, 0.0299199f, -0.996572f, 
		-0.0446655f, -0.000137869f, 0.999002f, -0.134399f, -0.195740f, 0.971403f, 
		0.0917347f, 0.176236f, -0.980064f, -0.0777951f, 0.165743f, 0.983096f, 
		0.125038f, -0.179006f, -0.975870f, -0.0793973f, 0.884635f, -0.459475f, 
		-0.784096f, -0.620312f, 0.0201593f, -0.551696f, -0.819753f, -0.153743f, 
		-0.908739f, -0.406532f, 0.0944750f, -0.425220f, -0.845511f, -0.322955f, 
		-0.628793f, 0.404199f, -0.664261f, -0.110489f, 0.993209f, 0.0364512f, 
		-0.977480f, 0.191641f, -0.0883554f, -0.947144f, 0.308721f, 0.0872290f, 
		0.118152f, -0.992844f, -0.0173288f, -0.828255f, 0.313463f, 0.464472f, 
		0.984137f, -0.174682f, -0.0310051f, 0.915387f, -0.329822f, -0.230833f, 
		0.828255f, -0.313463f, -0.464472f, -0.710284f, 0.497744f, 0.497743f, 
		0.0768960f, -0.996686f, -0.0265331f, 0.620073f, -0.682095f, -0.387630f, 
		0.766075f, -0.514779f, -0.384878f, 0.0225195f, 0.999712f, -0.00834213f, 
		-0.0355601f, -0.999327f, -0.00896852f, 0.0225195f, 0.999712f, -0.00834213f, 
		0.182481f, -0.0308755f, -0.982724f, -0.949177f, -0.0902088f, 0.301539f, 
		-0.603185f, 0.741326f, 0.294284f, -0.182481f, 0.0308755f, 0.982724f, 
		0.949177f, 0.0902088f, -0.301539f, 0.612535f, -0.697984f, -0.370970f, 
		-0.417354f, -0.710755f, -0.566253f, -0.373295f, -0.645595f, -0.666226f, 
		-0.291309f, -0.500108f, -0.815494f, 0.291309f, 0.500108f, 0.815494f, 
		0.373295f, 0.645595f, 0.666226f, 0.417354f, 0.710755f, 0.566253f, 
		0.222223f, 0.00282032f, -0.974992f, 0.00288860f, 0.999978f, -0.00603299f, 
		-0.0238977f, -0.00479381f, -0.999703f, -0.814996f, -0.441696f, 0.375082f, 
		0.0450918f, -0.000142602f, 0.998983f, -0.0238977f, -0.00479381f, -0.999703f, 
		0.0450918f, -0.000142602f, 0.998983f, 0.701854f, 0.531990f, -0.473696f, 
		0.978722f, -0.197156f, -0.0568632f, 0.904418f, -0.344398f, -0.251831f, 
		0.470092f, -0.881974f, -0.0336865f, 0.677572f, -0.559009f, -0.477918f, 
		0.521903f, -0.833313f, -0.182228f, 0.976141f, -0.192840f, -0.0998121f, 
		0.898562f, -0.338523f, -0.279265f, 0.473032f, -0.880544f, -0.0297117f, 
		0.677559f, -0.559044f, -0.477894f, 0.524352f, -0.832406f, -0.179319f, 
		0.978726f, -0.197141f, -0.0568434f, 0.904434f, -0.344372f, -0.251809f, 
		0.475735f, -0.878940f, -0.0337859f, 0.679844f, -0.554909f, -0.479466f, 
		0.526138f, -0.830611f, -0.182385f, 0.0645837f, -0.884806f, -0.461462f, 
		-0.00210080f, -0.00414231f, -0.999989f, -0.246858f, -0.657350f, 0.712005f, 
		0.166825f, -0.334669f, 0.927451f, -0.138207f, -0.563180f, 0.814695f, 
		0.000771500f, 0.999999f, 0.00104260f, -0.345659f, -0.592729f, 0.727456f, 
		0.00165562f, 0.00410150f, -0.999990f, -0.248250f, -0.466728f, 0.848845f, 
		0.252619f, 0.653718f, 0.713328f, -0.878996f, -0.462643f, -0.115441f, 
		-0.981836f, -0.187515f, -0.0289095f, -0.992205f, 0.103980f, 0.0686882f, 
		-0.238554f, -0.102511f, -0.965704f, 0.878996f, 0.462643f, 0.115441f, 
		0.981836f, 0.187515f, 0.0289095f, 0.238554f, 0.102511f, 0.965704f, 
		0.992205f, -0.103980f, -0.0686882f, 0.0616203f, -0.885495f, 0.460545f, 
		-0.220171f, -0.904731f, -0.364672f, 0.203184f, 0.890794f, 0.406451f, 
		-0.221182f, -0.898633f, 0.378864f, 0.256643f, 0.885165f, -0.388095f, 
		-0.962540f, -0.0678824f, -0.262506f, -0.220995f, -0.974313f, -0.0433166f, 
		0.962540f, 0.0678824f, 0.262506f, 0.220995f, 0.974313f, 0.0433166f, 
		0.000877578f, 0.999976f, -0.00691892f, -0.0390806f, -0.850100f, -0.525168f, 
		0.379618f, -0.275821f, -0.883070f, -0.852539f, -0.314692f, 0.417308f, 
		-0.579274f, 0.718760f, 0.384480f, -0.0871956f, 0.0856749f, 0.992500f, 
		0.856185f, 0.334504f, -0.393768f, 0.0871956f, -0.0856749f, -0.992500f, 
		0.578598f, -0.721652f, -0.380055f, -0.289152f, 0.424957f, 0.857789f, 
		-0.0920092f, 0.139252f, 0.985973f, 0.0598321f, -0.0397043f, -0.997419f, 
		-0.740910f, 0.457889f, 0.491315f, -0.366199f, 0.409164f, 0.835753f, 
		-0.0612243f, 0.00299972f, -0.998120f, 0.0779035f, -0.142733f, -0.986691f, 
		0.289152f, -0.424957f, -0.857789f, -0.0758735f, 0.0359388f, 0.996470f, 
		0.740910f, -0.457889f, -0.491315f, 0.368001f, -0.401754f, -0.838551f, 
		0.0426267f, -0.00723922f, 0.999065f, -0.000995999f, -0.00146700f, -0.999998f, 
		-0.772534f, 0.243296f, -0.586514f, 0.772534f, -0.243296f, 0.586514f, 
		-0.00140691f, 0.00528065f, 0.999985f, -0.000995999f, -0.00146700f, -0.999998f, 
		-0.203191f, -0.890796f, -0.406442f, 0.220172f, 0.904728f, 0.364679f, 
		-0.612516f, 0.739492f, 0.279242f, -0.915669f, 0.329373f, 0.230356f, 
		-0.976485f, 0.0168154f, 0.214930f, -0.343247f, 0.736879f, 0.582402f, 
		-0.300048f, 0.412697f, 0.860030f, -0.302323f, 0.123761f, 0.945137f, 
		-0.612267f, 0.736644f, 0.287200f, -0.915225f, 0.330411f, 0.230635f, 
		-0.976487f, 0.0168973f, 0.214915f, -0.342782f, 0.724173f, 0.598392f, 
		-0.261339f, 0.404011f, 0.876628f, -0.250091f, 0.122831f, 0.960399f, 
		0.218115f, 0.900520f, -0.376151f, -0.214555f, -0.902182f, 0.374212f, 
		0.180556f, 0.982892f, -0.0363717f, -0.962402f, -0.0678556f, 0.263019f, 
		-0.180556f, -0.982892f, 0.0363717f, 0.962402f, 0.0678556f, -0.263019f, 
		-0.180556f, -0.982892f, 0.0363717f, 0.180556f, 0.982892f, -0.0363717f, 
		0.315574f, 0.201853f, -0.927183f, -0.801047f, -0.596859f, 0.0456337f, 
		-0.642928f, -0.762491f, 0.0724688f, -0.937386f, -0.347063f, 0.0292160f, 
		-0.979040f, -0.141539f, 0.146447f, 0.642928f, 0.762491f, -0.0724688f, 
		0.801047f, 0.596859f, -0.0456337f, -0.315574f, -0.201853f, 0.927183f, 
		0.979040f, 0.141539f, -0.146447f, 0.937386f, 0.347063f, -0.0292160f, 
		0.371459f, -0.689955f, 0.621273f, -0.318300f, 0.719310f, -0.617477f, 
		0.0625187f, 0.128215f, 0.989774f, -0.0625187f, -0.128215f, -0.989774f, 
		-0.803675f, -0.414200f, 0.427252f, -0.821858f, -0.0328915f, -0.568742f, 
		-0.0625187f, -0.128215f, -0.989774f, 0.786427f, 0.489052f, -0.377307f, 
		0.0625187f, 0.128215f, 0.989774f, 0.821858f, 0.0328915f, 0.568742f, 
		-0.296697f, 0.224833f, -0.928128f, -0.813696f, 0.573178f, 0.0967778f, 
		-0.226317f, 0.343372f, -0.911524f, -0.920447f, 0.383632f, 0.0748565f, 
		-0.996096f, -0.0530449f, 0.0705548f, -0.296697f, 0.224833f, -0.928128f, 
		-0.326184f, -0.496316f, -0.804534f, 0.262659f, -0.152614f, 0.952743f, 
		0.882945f, -0.457858f, -0.103797f, -0.280351f, -0.153572f, -0.947533f, 
		0.972400f, -0.222768f, -0.0693765f, 0.262659f, -0.152614f, 0.952743f, 
		0.996096f, 0.0530449f, -0.0705548f, -0.575623f, 0.0325662f, 0.817067f, 
		-0.866794f, -0.473963f, -0.155006f, -0.730443f, -0.681039f, -0.0513734f, 
		-0.407253f, -0.894932f, 0.182325f, -0.184364f, -0.904113f, 0.385473f, 
		-0.229143f, -0.493001f, -0.839312f, -0.963625f, -0.0384246f, -0.264482f, 
		-0.192550f, 0.979823f, 0.0535814f, 0.229143f, 0.493001f, 0.839312f, 
		0.176842f, -0.562710f, 0.807517f, 0.963625f, 0.0384246f, 0.264482f, 
		0.192550f, -0.979823f, -0.0535814f, -0.176842f, 0.562710f, -0.807517f, 
		-0.996029f, -0.0887701f, 0.00679182f, -0.962991f, -0.0532352f, -0.264223f, 
		-0.854489f, -0.0793899f, -0.513367f, -0.316925f, 0.943176f, -0.0998857f, 
		-0.203545f, 0.851055f, 0.484020f, -0.164355f, 0.968598f, 0.186562f, 
		-0.708309f, -0.103712f, 0.698242f, -0.843855f, 0.518782f, -0.137018f, 
		0.165579f, -0.954993f, -0.246112f, 0.203545f, -0.851055f, -0.484020f, 
		0.113920f, -0.990179f, -0.0810409f, 0.708309f, 0.103712f, -0.698242f, 
		0.884045f, -0.432206f, 0.177938f, -0.124210f, 0.991028f, 0.0493470f, 
		-0.742116f, -0.669877f, -0.0229941f, -0.319420f, -0.0777631f, -0.944417f, 
		-0.973915f, 0.219316f, -0.0582290f, -0.699395f, -0.710081f, -0.0814388f, 
		0.958215f, -0.274146f, 0.0816536f, 0.352238f, 0.0372004f, -0.935171f, 
		-0.352238f, -0.0372004f, 0.935171f, 0.699395f, 0.710081f, 0.0814388f, 
		-0.996262f, 0.0398840f, -0.0766308f, -0.968650f, -0.244397f, -0.0445790f, 
		-0.854395f, -0.513550f, -0.0792221f, -0.295587f, -0.173640f, 0.939403f, 
		-0.372280f, -0.918561f, 0.132865f, -0.943993f, 0.102317f, 0.313700f, 
		0.372280f, 0.918561f, -0.132865f, 0.943993f, -0.102317f, -0.313700f, 
		-0.943985f, 0.102302f, 0.313731f, -0.360842f, -0.922416f, 0.137628f, 
		0.360842f, 0.922416f, -0.137628f, 0.943985f, -0.102302f, -0.313731f, 
		-0.943990f, 0.102289f, 0.313720f, 0.349917f, 0.927575f, -0.131008f, 
		-0.349917f, -0.927575f, 0.131008f, 0.943990f, -0.102289f, -0.313720f, 
		-0.200001f, -0.448858f, 0.870934f, -0.194073f, -0.354359f, 0.914749f, 
		-0.288715f, -0.665638f, 0.688164f, 0.213071f, 0.343371f, -0.914711f, 
		0.431869f, 0.582947f, -0.688231f, 0.291218f, 0.396067f, -0.870818f, 
		-0.291210f, 0.396061f, 0.870823f, -0.431858f, 0.582940f, 0.688244f, 
		-0.213069f, 0.343372f, 0.914712f, 0.202332f, -0.426428f, -0.881601f, 
		-0.741996f, 0.0231764f, 0.670003f, -0.245160f, 0.965234f, 0.0906640f, 
		-0.329627f, 0.709975f, -0.622320f, -0.000682226f, 0.000935505f, -0.999999f, 
		-0.000682226f, 0.000935505f, -0.999999f, 0.000724287f, -0.000879852f, 0.999999f, 
		0.425607f, -0.630888f, 0.648721f, -0.336346f, -0.936113f, -0.102781f, 
		-0.867307f, 0.107652f, -0.485994f, 0.926830f, -0.0954756f, 0.363140f, 
		0.336346f, 0.936113f, 0.102781f, -0.867418f, 0.107780f, -0.485767f, 
		0.926940f, -0.0954578f, 0.362865f, 0.337212f, 0.934301f, 0.115627f, 
		-0.337212f, -0.934301f, -0.115627f, -0.867314f, 0.107622f, -0.485988f, 
		0.332368f, 0.936200f, 0.114289f, -0.303188f, -0.948624f, -0.0904969f, 
		0.926828f, -0.0954529f, 0.363152f, -0.0675648f, -0.993660f, -0.0898603f, 
		-0.200461f, -0.955113f, -0.218116f, -0.284284f, -0.767008f, -0.575222f, 
		-0.131440f, -0.913478f, -0.385073f, -0.265430f, -0.766210f, -0.585208f, 
		0.233836f, 0.894020f, 0.382164f, 0.351430f, 0.784381f, 0.511120f, 
		0.234059f, 0.927607f, 0.291139f, 0.318758f, 0.784770f, 0.531535f, 
		0.164651f, 0.929114f, 0.331116f, 0.147237f, 0.967711f, 0.204587f, 
		-0.291335f, -0.870735f, 0.396162f, -0.431902f, -0.688099f, 0.583078f, 
		0.202431f, 0.881562f, -0.426462f, -0.213206f, -0.914656f, 0.343435f, 
		-0.0994389f, 0.000000f, -0.995044f, -0.00272877f, -0.999908f, -0.0132821f, 
		-0.00131040f, 0.999908f, -0.0134881f, -0.198686f, 0.000000f, -0.980063f, 
		-0.439650f, 0.672415f, -0.595455f, 0.000000f, -1.00000f, -1.28608e-019f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, -1.00000f, -1.28608e-019f, 0.301751f, -0.697651f, 0.649791f, 
		-0.329352f, 0.622357f, 0.710069f, 0.000000f, -1.00000f, -1.28608e-019f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, -1.00000f, -1.28608e-019f, 0.425365f, -0.648759f, -0.631012f, 
		0.00271809f, 0.999908f, -0.0132675f, 0.0990449f, 0.000000f, -0.995083f, 
		0.198273f, 0.000000f, -0.980147f, 0.00130792f, -0.999908f, -0.0134793f, 
		0.260567f, -0.176676f, 0.949152f, 0.264127f, -0.440787f, 0.857872f, 
		0.334245f, -0.745978f, 0.576018f, 0.968105f, -0.0115304f, 0.250280f, 
		0.367583f, -0.864731f, 0.342233f, -0.157653f, 0.190123f, -0.969019f, 
		-0.354441f, 0.513692f, -0.781340f, -0.241297f, 0.372609f, -0.896068f, 
		0.0643426f, -0.0986838f, 0.993037f, 0.283832f, -0.579502f, 0.763949f, 
		0.236445f, -0.491934f, 0.837911f, -0.0914066f, -0.282630f, -0.954864f, 
		-0.329422f, -0.672497f, -0.662743f, -0.227741f, -0.456446f, -0.860111f, 
		0.203959f, 0.217583f, 0.954494f, 0.284537f, 0.423622f, 0.859990f, 
		0.417917f, 0.621444f, 0.662686f, 0.963058f, 0.263762f, -0.0543050f, 
		0.249961f, 0.961895f, 0.110799f, 0.351417f, 0.380903f, 0.855231f, 
		-0.301434f, 0.649817f, 0.697765f, -0.000174088f, 3.79640e-006f, -1.00000f, 
		0.000200110f, 7.27183e-007f, 1.00000f, 0.000197078f, -4.52493e-006f, 1.00000f, 
		-0.000174088f, 3.79640e-006f, -1.00000f, 0.439375f, -0.595441f, -0.672607f, 
		-0.439376f, -0.595450f, 0.672598f, -0.000200669f, -4.86687e-014f, -1.00000f, 
		0.000173476f, 2.75406e-006f, 1.00000f, 0.000173476f, 2.75406e-006f, 1.00000f, 
		-0.000200669f, -4.86687e-014f, -1.00000f, 0.295810f, 0.678370f, -0.672540f, 
		-0.00126002f, 0.999999f, -0.000841795f, -0.314844f, -0.677029f, -0.665211f, 
		-0.274550f, -0.473864f, -0.836705f, -0.326870f, -0.594602f, -0.734578f, 
		-0.280242f, 0.590594f, 0.756745f, -0.174042f, 0.776780f, 0.605246f, 
		-0.422342f, 0.275373f, 0.863595f, -0.527415f, 0.0423635f, 0.848551f, 
		-0.790591f, 0.513343f, -0.333832f, -0.723974f, 0.597918f, 0.344028f, 
		-0.663601f, 0.512655f, 0.544810f, -0.817930f, 0.432469f, 0.379422f, 
		-0.478603f, 0.211564f, 0.852162f, -0.868824f, 0.0380377f, 0.493658f, 
		-0.631362f, -0.0737511f, 0.771973f, -0.937153f, 0.212255f, -0.276935f, 
		-0.767136f, 0.607311f, -0.206581f, -0.920788f, -0.0124262f, -0.389864f, 
		-0.125647f, 0.425889f, -0.896009f, -0.554249f, 0.815884f, -0.164747f, 
		-0.307039f, 0.855165f, 0.417637f, -0.901670f, -0.0324232f, 0.431208f, 
		-0.164693f, 0.552053f, 0.817382f, -0.267054f, 0.739735f, 0.617635f, 
		0.796830f, -0.380836f, -0.469070f, 0.330165f, 0.641057f, -0.692847f, 
		0.796841f, -0.380772f, -0.469103f, 0.330143f, 0.641069f, -0.692847f, 
		0.796876f, -0.380753f, -0.469058f, 0.330129f, 0.641083f, -0.692841f, 
		-0.272667f, 0.570548f, -0.774679f, -0.165493f, 0.753183f, -0.636653f, 
		-0.428175f, 0.260839f, -0.865234f, -0.527750f, 0.0423014f, -0.848346f, 
		-0.817152f, 0.541229f, 0.198328f, -0.00141560f, -0.999907f, -0.0135640f, 
		-0.200795f, -0.115748f, -0.972771f, 0.00140766f, 0.999907f, -0.0135491f, 
		0.200423f, 0.115655f, -0.972859f, 0.367103f, 0.533347f, 0.762087f, 
		0.247039f, 0.285326f, 0.926046f, 0.342434f, 0.763822f, 0.547096f, 
		0.953189f, -0.111071f, 0.281237f, 0.375729f, 0.527453f, 0.761985f, 
		0.246438f, 0.287917f, 0.925404f, 0.355211f, 0.756770f, 0.548747f, 
		0.953191f, -0.110995f, 0.281258f, -0.640534f, -0.334450f, 0.691274f, 
		-0.897187f, -0.255126f, 0.360507f, -0.966585f, -0.238271f, 0.0945534f, 
		-0.320716f, -0.591271f, 0.739960f, -0.175766f, -0.739595f, 0.649696f, 
		-0.353023f, -0.782769f, -0.512492f, -0.828255f, 0.313463f, 0.464472f, 
		-0.294350f, -0.928146f, 0.227824f, -0.682256f, 0.246527f, 0.688296f, 
		-0.254173f, 0.0209487f, 0.966932f, -0.397594f, 0.215044f, 0.892006f, 
		0.828255f, -0.313463f, -0.464472f, 0.682256f, -0.246527f, -0.688296f, 
		0.294350f, 0.928146f, -0.227824f, 0.254173f, -0.0209487f, -0.966932f, 
		0.397594f, -0.215044f, -0.892006f, -0.420974f, -0.907073f, 8.87458e-005f, 
		-0.000296154f, -0.000111640f, -1.00000f, -0.575062f, 0.818110f, 0.000121558f, 
		-0.000236825f, 2.94740e-006f, -1.00000f, -0.369368f, -0.686350f, 0.626491f, 
		-0.239409f, -0.873024f, 0.424868f, -0.529231f, -0.445851f, -0.721895f, 
		-0.964343f, -0.253542f, 0.0758914f, 0.239409f, 0.873024f, -0.424868f, 
		0.433065f, 0.567354f, -0.700403f, 0.529231f, 0.445851f, 0.721895f, 
		-0.377280f, -0.434251f, 0.817977f, 0.964343f, 0.253542f, -0.0758914f, 
		0.441109f, 0.354185f, -0.824606f, -0.299886f, -0.885390f, -0.355181f, 
		-0.963187f, -0.0531593f, -0.263523f, 0.131775f, -0.608830f, 0.782279f, 
		0.299886f, 0.885390f, 0.355181f, 0.963187f, 0.0531593f, 0.263523f, 
		-0.131775f, 0.608830f, -0.782279f, 0.356422f, 0.408216f, 0.840430f, 
		0.851590f, 0.512868f, -0.108446f, 0.296339f, 0.585369f, 0.754669f, 
		0.159215f, 0.733822f, 0.660420f, -0.597269f, -0.780226f, -0.185787f, 
		-0.914793f, 0.328661f, -0.234810f, -0.254529f, 0.187356f, -0.948743f, 
		0.884045f, -0.432206f, 0.177938f, 0.597269f, 0.780226f, 0.185787f, 
		0.256906f, -0.110939f, 0.960048f, -0.861408f, -0.0893277f, 0.499997f, 
		-0.957215f, -0.128723f, 0.259170f, -0.996029f, -0.0887701f, 0.00679182f, 
		-0.575045f, -0.818122f, 0.000121554f, 0.000298121f, 0.000118983f, 1.00000f, 
		-0.420953f, 0.907082f, 8.87414e-005f, 0.000236810f, -2.94965e-006f, 1.00000f, 
		-0.418899f, 0.567643f, 0.708734f, -0.299704f, -0.885418f, 0.355263f, 
		-0.605560f, 0.539589f, 0.584928f, -0.946039f, 0.175618f, 0.272338f, 
		-0.865007f, 0.356468f, 0.353119f, 0.418899f, -0.567643f, -0.708734f, 
		0.605560f, -0.539589f, -0.584928f, 0.299704f, 0.885418f, -0.355263f, 
		0.865007f, -0.356468f, -0.353119f, 0.946039f, -0.175618f, -0.272338f, 
		-0.996096f, -0.0530449f, 0.0705548f, -0.948741f, -0.294875f, 0.113755f, 
		-0.856577f, -0.508550f, 0.0874844f, 0.996096f, 0.0530449f, -0.0705548f, 
		0.948741f, 0.294875f, -0.113755f, 0.856577f, 0.508550f, -0.0874844f, 
		0.000181486f, 0.000367397f, 1.00000f, -0.000161100f, -0.000267401f, -1.00000f, 
		-0.250408f, -0.967816f, 0.0250483f, -0.249293f, -0.281657f, 0.926565f, 
		-0.871454f, -0.267279f, -0.411254f, 0.250408f, 0.967816f, -0.0250483f, 
		0.249293f, 0.281657f, -0.926565f, 0.912821f, 0.272270f, 0.304346f, 
		-0.923861f, 0.260146f, -0.280722f, 0.184918f, 0.966967f, 0.175444f, 
		0.284519f, -0.0971470f, 0.953735f, 0.926490f, -0.198675f, 0.319601f, 
		-0.184918f, -0.966967f, -0.175444f, -0.284519f, 0.0971470f, -0.953735f, 
		-0.919517f, 0.0961327f, -0.381113f, -0.977474f, 0.130170f, -0.166135f, 
		-0.984336f, 0.127600f, 0.121659f, 0.867485f, -0.106365f, 0.485959f, 
		2.73113e-006f, -1.00000f, 1.01046e-005f, 0.941879f, -0.131832f, 0.309004f, 
		0.984336f, -0.127600f, -0.121659f, -0.000172246f, 1.00000f, 4.65538e-005f, 
		0.000000f, -1.00000f, -1.28608e-019f, -0.919518f, 0.0961490f, -0.381106f, 
		-0.977472f, 0.130185f, -0.166131f, -0.984334f, 0.127608f, 0.121664f, 
		0.867579f, -0.106498f, 0.485763f, 0.941917f, -0.131933f, 0.308847f, 
		0.984334f, -0.127608f, -0.121664f, -0.000172246f, 1.00000f, 4.65538e-005f, 
		-4.98821e-006f, -1.00000f, 1.95894e-006f, -0.919508f, 0.0961554f, -0.381129f, 
		-0.977464f, 0.130197f, -0.166169f, -0.984335f, 0.127635f, 0.121629f, 
		0.867479f, -0.106391f, 0.485965f, 0.941869f, -0.131860f, 0.309024f, 
		0.984335f, -0.127635f, -0.121629f, -0.000172246f, 1.00000f, 4.65538e-005f, 
		-5.83946e-006f, -1.00000f, 1.58196e-006f, -0.845868f, 0.530390f, -0.0565083f, 
		-0.952883f, 0.284229f, -0.105958f, -0.996262f, 0.0398840f, -0.0766308f, 
		-0.841987f, -0.321167f, -0.433485f, -0.841987f, -0.321167f, -0.433485f, 
		-0.659476f, 0.664334f, -0.351784f, -0.659476f, 0.664334f, -0.351784f, 
		0.915572f, 0.355019f, 0.188918f, -0.0884940f, 0.00948286f, 0.996032f, 
		0.915572f, 0.355019f, 0.188918f, 0.626436f, -0.682124f, 0.377206f, 
		0.626436f, -0.682124f, 0.377206f, 0.0884940f, -0.00948286f, -0.996032f, 
		0.257961f, 0.173925f, 0.950372f, -0.144370f, 0.333240f, -0.931723f, 
		0.143965f, -0.333190f, -0.931804f, -0.982761f, 0.163204f, -0.0868650f, 
		-0.701838f, -0.707387f, -0.0838249f, 0.974672f, -0.219493f, 0.0428659f, 
		0.701838f, 0.707387f, 0.0838249f, -0.000182126f, 0.000108221f, -1.00000f, 
		0.000165675f, -0.000112933f, 1.00000f, -0.00264847f, 0.999907f, -0.0133883f, 
		-0.0952689f, 0.115749f, -0.988699f, -0.220949f, -0.971488f, 0.0859826f, 
		0.225308f, 0.973824f, -0.0300498f, -0.137067f, -0.333219f, -0.932833f, 
		0.00264331f, -0.999907f, -0.0133731f, 0.0948303f, -0.115663f, -0.988751f, 
		-0.788223f, 0.525676f, 0.319951f, -0.787493f, 0.398975f, 0.469759f, 
		-0.862925f, 0.469627f, 0.186576f, -0.707703f, 0.705530f, 0.0372081f, 
		-0.322135f, 0.938950f, -0.120837f, -0.911537f, 0.365752f, -0.187951f, 
		-0.919271f, 0.0768351f, -0.386054f, 0.738820f, -0.399780f, -0.542513f, 
		0.755866f, -0.505638f, -0.415929f, 0.834843f, -0.472856f, -0.281859f, 
		0.865641f, -0.484886f, 0.124704f, 0.322135f, -0.938950f, 0.120837f, 
		0.703783f, -0.710204f, -0.0173417f, 0.136685f, 0.333175f, -0.932905f, 
		-0.776413f, -0.623846f, 0.0894392f, -0.871769f, -0.313737f, 0.376281f, 
		-0.614910f, -0.307387f, 0.726223f, -0.463236f, -0.871856f, 0.158998f, 
		-0.218851f, -0.828817f, 0.514944f, -0.320748f, -0.538571f, 0.779142f, 
		0.248863f, -0.206917f, 0.946178f, 0.238243f, -0.475966f, 0.846580f, 
		0.308767f, -0.766635f, 0.562969f, 0.955744f, -0.163623f, 0.244501f, 
		0.338380f, -0.886021f, 0.316964f, 0.248852f, -0.206945f, 0.946175f, 
		0.238227f, -0.475995f, 0.846568f, 0.308765f, -0.766653f, 0.562945f, 
		0.955734f, -0.163670f, 0.244509f, 0.338374f, -0.886035f, 0.316931f, 
		-0.0372638f, -0.0336628f, 0.998738f, -0.0199789f, -0.0350929f, -0.999184f, 
		-0.950012f, 0.0517574f, -0.307894f, -0.0151462f, -0.00710386f, 0.999860f, 
		0.0151462f, 0.00710386f, -0.999860f, -0.0151462f, -0.00710386f, 0.999860f, 
		0.0151462f, 0.00710386f, -0.999860f, -0.452191f, 0.884498f, 0.114831f, 
		-0.501478f, 0.865102f, 0.0108861f, -0.583562f, 0.775277f, -0.241660f, 
		0.583562f, -0.775277f, 0.241660f, 0.548475f, -0.835564f, 0.0317319f, 
		0.501164f, -0.860709f, -0.0895205f, -0.587140f, -0.808891f, -0.0310116f, 
		-0.486980f, -0.862512f, 0.137566f, -0.523364f, -0.851812f, 0.0224870f, 
		0.587140f, 0.808891f, 0.0310116f, 0.463206f, 0.882481f, -0.0816572f, 
		0.419062f, 0.889055f, -0.184305f, 0.241648f, -0.217837f, 0.945597f, 
		0.217931f, 0.971865f, -0.0893526f, -0.223539f, -0.974092f, 0.0342835f, 
		-0.440614f, -0.152637f, 0.884625f, -0.479549f, -0.0552754f, 0.875772f, 
		-0.523173f, -0.0185027f, 0.852026f, 0.486803f, 0.137472f, -0.862626f, 
		0.586915f, -0.0311383f, -0.809050f, 0.523160f, 0.0224027f, -0.851940f, 
		-0.587103f, -0.808911f, 0.0311976f, -0.523352f, -0.851824f, -0.0223135f, 
		-0.487012f, -0.862521f, -0.137397f, 0.523364f, 0.851910f, 0.0183972f, 
		0.479750f, 0.875670f, 0.0551549f, 0.440831f, 0.884537f, 0.152522f, 
		-0.547912f, 0.311360f, -0.776432f, -0.468867f, 0.193322f, -0.861853f, 
		-0.511824f, 0.253508f, -0.820835f, 0.540702f, -0.392435f, 0.744067f, 
		0.559698f, -0.274707f, 0.781840f, 0.437698f, -0.390852f, 0.809725f, 
		0.508513f, -0.323047f, 0.798157f, 0.400686f, -0.204870f, 0.893017f, 
		-0.337366f, -0.670317f, -0.660953f, -0.433310f, -0.565240f, -0.701959f, 
		-0.583011f, -0.248689f, -0.773468f, -0.342886f, -0.562728f, -0.752174f, 
		-0.333682f, -0.338311f, -0.879888f, 0.609783f, -0.0521719f, 0.790850f, 
		0.626777f, 0.129451f, 0.768371f, 0.404565f, 0.410546f, 0.817178f, 
		0.473433f, 0.307636f, 0.825361f, 0.286855f, 0.488540f, 0.824041f, 
		-0.369548f, 0.846169f, 0.383969f, -0.156379f, 0.763651f, 0.626404f, 
		-0.632716f, 0.773209f, 0.0426382f, -0.533194f, 0.826241f, 0.181740f, 
		0.156379f, -0.763651f, -0.626404f, 0.464301f, -0.818010f, -0.339536f, 
		0.651092f, -0.755943f, -0.0680429f, 0.570807f, -0.802531f, -0.173563f, 
		-0.921947f, -0.385372f, 0.0387516f, -0.668759f, 0.727148f, -0.154974f, 
		-0.482585f, 0.0942713f, 0.870761f, 0.491922f, -0.0107789f, -0.870573f, 
		0.939482f, 0.337785f, 0.0572184f, 0.684422f, -0.701194f, 0.199734f, 
		0.218575f, 0.301644f, 0.928028f, 0.368859f, 0.921219f, 0.123685f, 
		0.911789f, 0.188837f, 0.364667f, -0.406473f, 0.854480f, -0.323488f, 
		-0.440987f, 0.888224f, -0.128798f, -0.431435f, 0.811280f, -0.394572f, 
		-0.471022f, 0.850971f, -0.232351f, -0.607113f, 0.792884f, 0.0524370f, 
		0.337844f, -0.915946f, 0.216575f, 0.417096f, -0.806336f, 0.419349f, 
		0.397552f, -0.857485f, 0.326605f, 0.435311f, -0.753923f, 0.492041f, 
		0.342788f, -0.731358f, 0.589586f, -0.000158518f, 1.00000f, 0.000991674f, 
		-0.211825f, -0.330062f, 0.919885f, 0.000536645f, 1.00000f, 0.000171517f, 
		-0.0436812f, -0.986796f, -0.155964f, -0.0213237f, 0.211974f, -0.977043f, 
		-0.0247420f, -0.0681921f, 0.997365f, -0.00106932f, 0.999997f, -0.00217842f, 
		-0.175878f, -0.351880f, -0.919373f, -0.0914375f, 0.990398f, -0.103686f, 
		-0.947420f, 0.192304f, -0.255764f, 0.0914375f, -0.990398f, 0.103686f, 
		0.319105f, -0.193394f, 0.927777f, 0.925096f, -0.251454f, 0.284551f, 
		-0.319105f, 0.193394f, -0.927777f, -0.884955f, 0.426435f, -0.187104f, 
		-0.245087f, 0.337688f, -0.908790f, 0.0998575f, -0.993440f, 0.0557270f, 
		0.938214f, -0.310871f, 0.152030f, -0.0569723f, 0.995336f, -0.0778518f, 
		0.244494f, -0.334791f, 0.910021f, -0.701152f, 0.191853f, 0.686715f, 
		-0.926834f, 0.153076f, -0.342851f, 0.629460f, -0.0396726f, -0.776019f, 
		-0.314942f, -0.942777f, 0.109471f, 0.325121f, 0.929038f, -0.176592f, 
		0.917430f, -0.0769965f, 0.390376f, -0.862280f, 0.497095f, 0.0967938f, 
		0.812192f, -0.574035f, -0.104058f, 2.78352e-005f, 4.62222e-005f, 1.00000f, 
		8.62780e-005f, -4.26807e-005f, -1.00000f, -0.476411f, 0.277224f, -0.834374f, 
		-0.391363f, -0.860652f, -0.325750f, -0.509857f, 0.217561f, -0.832294f, 
		-0.392266f, -0.860018f, -0.326336f, 0.813595f, 0.0242731f, -0.580925f, 
		-0.948784f, -0.00497388f, 0.315885f, -0.0490114f, -0.995450f, 0.0817152f, 
		0.0801598f, -0.0213459f, -0.996553f, -0.113326f, 0.992973f, 0.0340862f, 
		-0.706085f, -0.680041f, 0.197458f, -0.590602f, -0.743499f, 0.313686f, 
		-0.455611f, 0.0323087f, 0.889592f, -0.310335f, -0.851147f, 0.423369f, 
		-0.734504f, -0.641049f, 0.222622f, -0.612590f, -0.715623f, 0.335584f, 
		-0.406389f, -0.0763962f, 0.910501f, -0.310154f, -0.851078f, 0.423640f, 
		-0.735988f, -0.650395f, 0.187904f, -0.608838f, -0.724699f, 0.322687f, 
		-0.314769f, -0.0391084f, 0.948362f, -0.316659f, -0.852481f, 0.415937f, 
		-0.982031f, 0.188536f, -0.00838520f, 0.988478f, -0.149632f, -0.0228537f, 
		0.218797f, 0.582604f, 0.782752f, 0.233894f, 0.412998f, 0.880185f, 
		0.244001f, 0.282831f, 0.927615f, -0.721099f, 0.000000f, -0.692832f, 
		-0.0242742f, 0.999705f, -0.000143457f, -0.965856f, 0.000000f, 0.259079f, 
		0.721099f, 0.000000f, 0.692832f, 0.965856f, 0.000000f, -0.259080f, 
		0.0242742f, -0.999705f, 0.000143457f, -0.721099f, 0.000000f, -0.692832f, 
		-0.965856f, 0.000000f, 0.259079f, 0.200542f, 0.976722f, 0.0761412f, 
		0.721099f, 0.000000f, 0.692832f, 0.965856f, 0.000000f, -0.259080f, 
		-0.200542f, -0.976722f, -0.0761412f, 0.599222f, 0.441000f, -0.668171f, 
		0.780718f, 0.291820f, -0.552558f, 0.368632f, 0.437411f, -0.820233f, 
		0.283253f, 0.958238f, 0.0393280f, 0.0762697f, 0.915989f, -0.393888f, 
		-0.119831f, 0.696852f, -0.707133f, -0.105392f, 0.994011f, -0.0288871f, 
		-0.862650f, 0.00862163f, -0.505728f, -0.990609f, -0.00633429f, -0.136582f, 
		-0.998683f, -0.00163268f, 0.0512789f, 0.105392f, -0.994011f, 0.0288871f, 
		0.817330f, -0.00317609f, 0.576161f, 0.909466f, 0.00524628f, 0.415744f, 
		0.994396f, -0.00335677f, 0.105669f, -0.902765f, 0.420015f, -0.0927487f, 
		0.865440f, -0.494350f, 0.0814348f, -0.000393302f, 5.19196e-005f, -1.00000f, 
		0.000496432f, -2.91304e-005f, 1.00000f, 0.116000f, -0.702102f, -0.702564f, 
		-0.998963f, -0.0454976f, 0.00161662f, -0.940033f, 0.341059f, 0.00396056f, 
		-0.854891f, 0.518666f, -0.0120876f, -0.0895594f, 0.0458105f, -0.994927f, 
		0.998963f, 0.0454976f, -0.00161662f, 0.979518f, -0.201283f, -0.00546251f, 
		0.907170f, -0.420387f, 0.0178005f, 0.180416f, -0.139087f, 0.973707f, 
		-0.212997f, -0.201708f, 0.956005f, -0.995448f, -0.0952148f, 0.00417095f, 
		0.212997f, 0.201708f, -0.956005f, 0.999977f, 0.00617920f, 0.00286961f, 
		-0.826336f, 0.0482351f, 0.561108f, 0.897826f, -0.0316196f, 0.439215f, 
		-0.906031f, -0.0348444f, -0.421775f, -0.114282f, -0.960413f, 0.254058f, 
		0.114282f, 0.960413f, -0.254058f, -0.222482f, -0.944103f, -0.243251f, 
		-0.929680f, -0.295707f, 0.219663f, -0.861511f, -0.400239f, 0.312423f, 
		-0.929680f, -0.295707f, 0.219663f, -0.985989f, -0.157814f, 0.0540394f, 
		0.201211f, 0.954271f, 0.221089f, 0.0288954f, 0.0201607f, -0.999379f, 
		0.930912f, 0.226472f, -0.286555f, 0.864893f, 0.315752f, -0.390205f, 
		0.998797f, -0.0171363f, -0.0459527f, 0.0288954f, 0.0201607f, -0.999379f, 
		-0.0288954f, -0.0201607f, 0.999379f, 0.0982715f, -0.0151174f, -0.995045f, 
		-0.998910f, 0.0465280f, 0.00375089f, -0.925846f, 0.377896f, 0.00197223f, 
		-0.829833f, 0.557839f, 0.0138732f, -0.0982715f, 0.0151174f, 0.995045f, 
		0.995740f, 0.0922065f, 0.000151126f, 0.995450f, -0.0952598f, 0.00225074f, 
		0.876170f, -0.481712f, -0.0167255f, -0.826336f, 0.0482351f, 0.561108f, 
		0.757143f, 0.0255108f, -0.652751f, -0.906031f, -0.0348444f, -0.421775f, 
		-0.0711042f, -0.425268f, 0.902270f, 0.0711042f, 0.425268f, -0.902270f, 
		-0.791831f, 0.00417175f, 0.610726f, -0.963987f, 0.00719270f, -0.265854f, 
		0.776940f, -0.0148634f, -0.629400f, 0.115607f, -0.993259f, 0.00845628f, 
		0.966008f, -0.00391220f, 0.258482f, -0.136249f, 0.990618f, -0.0105645f, 
		-0.105355f, -0.473155f, 0.874657f, -0.884334f, -0.466854f, -0.000591301f, 
		-0.964015f, -0.265308f, -0.0169020f, -0.998963f, -0.0454976f, 0.00161662f, 
		-0.0422971f, -0.149890f, -0.987798f, 0.0422971f, 0.149890f, 0.987798f, 
		0.884334f, 0.466854f, 0.000591301f, 0.964015f, 0.265308f, 0.0169020f, 
		0.998963f, 0.0454976f, -0.00161662f, -0.0422971f, -0.149890f, -0.987798f, 
		0.0422971f, 0.149890f, 0.987798f, -0.776149f, 0.0131075f, 0.630413f, 
		-0.189423f, 0.958202f, 0.214399f, -0.961944f, 0.00387717f, -0.273220f, 
		0.761450f, -0.00408315f, -0.648211f, 0.189423f, -0.958202f, -0.214399f, 
		0.964096f, -0.00724398f, 0.265456f, -0.475084f, -0.0824639f, 0.876068f, 
		-0.692532f, -0.0551904f, 0.719273f, -0.861408f, -0.0893277f, 0.499997f, 
		-0.213482f, 0.945345f, 0.246472f, -0.856577f, -0.508550f, 0.0874844f, 
		-0.704814f, -0.706262f, 0.0665735f, -0.498589f, -0.861210f, 0.0986275f, 
		-0.303431f, -0.280909f, -0.910505f, 0.856577f, 0.508550f, -0.0874844f, 
		0.704814f, 0.706262f, -0.0665735f, 0.303431f, 0.280909f, 0.910505f, 
		0.498589f, 0.861210f, -0.0986275f, -0.535637f, 0.532468f, -0.655416f, 
		-0.419758f, 0.688350f, -0.591588f, -0.0177329f, 0.985613f, -0.168087f, 
		-0.0342631f, -0.999410f, -0.00230846f, 0.465893f, -0.534390f, 0.705245f, 
		0.0223539f, -0.983780f, 0.177979f, 0.322311f, -0.735875f, 0.595486f, 
		0.0397267f, 0.999106f, 0.0144321f, -0.273842f, 0.215452f, -0.937332f, 
		-0.00822883f, 0.322937f, 0.946385f, 0.915589f, 0.0368613f, 0.400422f, 
		-0.950012f, 0.0517574f, -0.307894f, 0.00308536f, 0.707652f, 0.706555f, 
		-0.00308536f, -0.707652f, -0.706555f, -0.0112325f, 0.496270f, 0.868096f, 
		-0.0860154f, -0.660950f, -0.745484f, 0.104678f, 0.479807f, -0.871107f, 
		0.0833210f, 0.669405f, -0.738210f, -0.270355f, -0.0712296f, -0.960122f, 
		-0.221688f, 0.951490f, -0.213357f, 0.0689887f, -0.959109f, 0.274499f, 
		-0.230022f, -0.0497985f, -0.971911f, -0.00254963f, -0.999952f, -0.00948358f, 
		-7.10611e-005f, 0.999980f, -0.00630007f, 0.222371f, 0.00282461f, -0.974958f, 
		0.122798f, 0.986405f, 0.109206f, -0.386049f, -0.639993f, -0.664361f, 
		0.0136966f, -0.999722f, 0.0191741f, 0.210547f, 0.922660f, 0.323060f, 
		0.410498f, 0.685358f, 0.601479f, -0.985857f, 0.146758f, 0.0809224f, 
		-0.343553f, 0.929158f, 0.136519f, -0.126242f, 0.0891390f, -0.987986f, 
		0.985857f, -0.146758f, -0.0809224f, 0.126242f, -0.0891390f, 0.987986f, 
		0.296528f, -0.951639f, -0.0803324f, -0.448475f, 0.890202f, -0.0800654f, 
		-0.673092f, 0.736431f, -0.0679479f, -0.834865f, 0.542514f, -0.0931632f, 
		-0.162103f, 0.0854418f, 0.983068f, 0.448475f, -0.890202f, 0.0800654f, 
		0.790479f, -0.608737f, 0.0676976f, 0.220008f, -0.301622f, -0.927697f, 
		0.903641f, -0.420359f, 0.0820445f, -0.0720132f, 0.332374f, 0.940394f, 
		0.0737316f, 0.446857f, 0.891562f, -0.0624530f, -0.322247f, -0.944593f, 
		0.0624530f, 0.322247f, 0.944593f, -0.0151462f, -0.00710386f, 0.999860f, 
		0.0151462f, 0.00710386f, -0.999860f, 0.0695662f, -0.823173f, -0.563513f, 
		-0.0695662f, 0.823173f, 0.563513f, -0.0151462f, -0.00710386f, 0.999860f, 
		0.0151462f, 0.00710386f, -0.999860f, 0.0695748f, -0.823188f, -0.563490f, 
		-0.0695748f, 0.823188f, 0.563490f, -0.000693237f, 0.999999f, 0.000822999f, 
		-0.263868f, -0.680538f, 0.683551f, -0.239791f, -0.459112f, 0.855404f, 
		-0.288597f, -0.592753f, 0.751901f, -0.136551f, -0.0365001f, 0.989960f, 
		0.00483310f, -0.999939f, 0.00997247f, -0.452919f, 0.846977f, 0.278379f, 
		-0.0581640f, -0.120978f, -0.990950f, -0.249530f, 0.349179f, 0.903221f, 
		-0.506673f, 0.731828f, 0.455753f, 0.0631981f, -0.291910f, -0.954355f, 
		0.0308490f, -0.168172f, -0.985275f, 0.0307517f, 0.0593326f, 0.997764f, 
		0.385689f, -0.887068f, -0.253680f, 0.275959f, -0.865253f, -0.418549f, 
		0.446354f, -0.802627f, -0.395673f, 0.179401f, -0.362993f, -0.914359f, 
		-0.0685221f, 0.122429f, 0.990109f, 0.0308490f, -0.168172f, -0.985275f, 
		-0.0685221f, 0.122429f, 0.990109f, -0.0330163f, -0.0442762f, -0.998474f, 
		-0.420916f, -0.814908f, 0.398441f, 0.0782136f, 0.383748f, 0.920119f, 
		-0.0616362f, -0.243887f, -0.967843f, 0.484057f, 0.775083f, -0.406123f, 
		0.618862f, 0.618962f, -0.483627f, 0.0424438f, 0.109571f, 0.993072f, 
		-0.0548431f, -0.189199f, -0.980406f, 0.406753f, 0.873691f, -0.266865f, 
		0.0623697f, 0.242801f, 0.968069f, 0.955159f, 0.00111103f, -0.296092f, 
		-0.999333f, -0.0273136f, 0.0242457f, -0.125416f, -0.748039f, 0.651697f, 
		-0.0891380f, 0.751039f, -0.654213f, 0.100909f, -0.873128f, -0.476933f, 
		0.999577f, -0.0221640f, 0.0188556f, -0.948784f, -0.00497388f, 0.315885f, 
		-0.125416f, -0.748039f, 0.651697f, 0.114127f, 0.830155f, 0.545727f, 
		-0.0891380f, 0.751039f, -0.654213f, -0.354103f, 0.840208f, 0.410684f, 
		-0.346894f, 0.708622f, 0.614427f, -0.491340f, 0.725978f, 0.481187f, 
		-0.651434f, 0.745380f, -0.141572f, -0.513925f, 0.842887f, -0.159447f, 
		-0.598915f, 0.720149f, -0.350265f, 0.121335f, 0.558802f, 0.820377f, 
		0.133698f, 0.974600f, 0.179667f, -0.133698f, -0.974600f, -0.179667f, 
		-0.514029f, 0.0921186f, -0.852812f, -0.133698f, -0.974600f, -0.179667f, 
		0.514029f, -0.0921186f, 0.852812f, 0.133698f, 0.974600f, 0.179667f, 
		-0.391268f, -0.557007f, 0.732566f, -0.512306f, -0.410659f, 0.754255f, 
		0.194699f, 0.593841f, -0.780670f, -0.0498350f, 0.985160f, -0.164242f, 
		0.450239f, 0.361566f, -0.816428f, 0.350637f, 0.423879f, -0.835093f, 
		-0.0414529f, 0.998783f, -0.0267207f, 0.0203327f, -0.996324f, 0.0832219f, 
		-0.0868883f, -0.982679f, 0.163686f, -0.0963135f, 0.993401f, 0.0622690f, 
		-0.836054f, 0.0126188f, 0.548502f, 0.0845690f, -0.994848f, -0.0559022f, 
		0.911928f, -0.000647259f, -0.410350f, -0.106864f, -0.550073f, -0.828251f, 
		-0.0200891f, 0.0832884f, 0.996323f, 0.0871806f, 0.163794f, 0.982635f, 
		-0.450525f, -0.816407f, -0.361259f, 0.0495732f, -0.164349f, -0.985156f, 
		-0.350932f, -0.835104f, -0.423614f, -0.195010f, -0.780795f, -0.593574f, 
		0.391581f, 0.732603f, 0.556738f, 0.0412078f, -0.0268222f, -0.998791f, 
		0.512602f, 0.754224f, 0.410349f, -0.382538f, 0.707148f, -0.594648f, 
		-0.382538f, 0.707148f, -0.594648f, -0.603718f, 0.599432f, -0.525553f, 
		-0.819512f, 0.445770f, -0.360123f, -0.297148f, -0.258434f, -0.919192f, 
		-0.470577f, -0.379357f, -0.796647f, -0.578124f, -0.428871f, -0.694149f, 
		0.0351210f, 0.0719999f, -0.996786f, 0.0351210f, 0.0719999f, -0.996786f, 
		-0.0305766f, -0.0789305f, 0.996411f, -0.0305766f, -0.0789305f, 0.996411f, 
		0.0351210f, 0.0719999f, -0.996786f, -0.0837698f, -0.888940f, 0.450298f, 
		-0.520956f, -0.849577f, 0.0826038f, -0.00727465f, -0.992153f, -0.124818f, 
		-0.854489f, -0.0793899f, -0.513367f, -0.304862f, 0.951162f, 0.0484795f, 
		-0.691350f, 0.721723f, -0.0339190f, -0.890401f, 0.436396f, 0.129399f, 
		-0.926363f, 0.271943f, 0.260573f, -0.844885f, -0.375563f, -0.380949f, 
		-0.154282f, 0.224423f, 0.962201f, -0.0706969f, 0.987329f, 0.142068f, 
		-0.970461f, 0.175520f, 0.165525f, -0.938277f, 0.139082f, 0.316690f, 
		-0.910288f, -0.00857408f, 0.413887f, -0.0836710f, 0.955533f, 0.282763f, 
		0.943740f, -0.293086f, -0.153149f, 0.870372f, -0.157223f, -0.466619f, 
		0.804449f, -0.0126254f, -0.593888f, 0.0836710f, -0.955533f, -0.282763f, 
		-0.520923f, -0.0824836f, 0.849609f, -0.875862f, 0.110928f, 0.469640f, 
		-0.599244f, 0.129076f, 0.790093f, -0.420521f, 0.100740f, 0.901673f, 
		0.875862f, -0.110928f, -0.469640f, 0.599244f, -0.129076f, -0.790093f, 
		0.420511f, -0.100736f, -0.901678f, -5.79053e-006f, 1.00000f, 1.62417e-006f, 
		0.000177978f, -1.00000f, -4.81786e-005f, -0.875840f, 0.110923f, 0.469682f, 
		-0.599225f, 0.129066f, 0.790108f, -0.420521f, 0.100740f, 0.901673f, 
		0.875840f, -0.110923f, -0.469682f, 0.599225f, -0.129066f, -0.790108f, 
		0.420511f, -0.100736f, -0.901678f, -5.79053e-006f, 1.00000f, 1.62417e-006f, 
		0.000177978f, -1.00000f, -4.81786e-005f, -0.875848f, 0.110906f, 0.469671f, 
		-0.599333f, 0.129133f, 0.790016f, -0.420677f, 0.100830f, 0.901590f, 
		0.875848f, -0.110906f, -0.469671f, 0.599333f, -0.129133f, -0.790016f, 
		0.420677f, -0.100830f, -0.901590f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000177978f, -1.00000f, -4.81786e-005f, 0.00140339f, -0.0766349f, 0.997058f, 
		-0.894669f, 0.446491f, -0.0146250f, -0.00140339f, 0.0766349f, -0.997058f, 
		0.00140339f, -0.0766349f, 0.997058f, -0.00140339f, 0.0766349f, -0.997058f, 
		0.853408f, -0.521051f, 0.0141625f, -0.00140339f, 0.0766349f, -0.997058f, 
		0.00140339f, -0.0766349f, 0.997058f, 0.915589f, 0.0368613f, 0.400422f, 
		-0.950012f, 0.0517574f, -0.307894f, -0.0221090f, -0.355133f, -0.934554f, 
		0.0218820f, 0.637313f, 0.770294f, -0.854395f, -0.513550f, -0.0792221f, 
		0.381408f, -0.921916f, -0.0678127f, 0.500459f, -0.550594f, 0.668122f, 
		0.541638f, -0.677502f, 0.497613f, 0.537401f, -0.839520f, 0.0800426f, 
		-0.118053f, -0.923737f, 0.364380f, -0.813815f, -0.257003f, 0.521205f, 
		-0.905610f, -0.415289f, 0.0860595f, -0.475620f, -0.848370f, 0.232494f, 
		-0.824504f, -0.496620f, 0.271222f, 0.0468055f, -0.734583f, 0.676903f, 
		0.0982032f, -0.560724f, 0.822159f, -0.166726f, -0.975356f, -0.144508f, 
		0.127522f, 0.975539f, 0.179060f, -0.773400f, 0.503670f, -0.384928f, 
		-0.683695f, 0.724275f, -0.0893744f, -0.539444f, 0.247445f, -0.804842f, 
		-0.731058f, 0.376538f, -0.569010f, -0.269271f, 0.0685424f, -0.960622f, 
		-0.691027f, -0.675549f, 0.257129f, 0.897826f, -0.0316196f, 0.439215f, 
		0.757143f, 0.0255108f, -0.652751f, -0.906031f, -0.0348444f, -0.421775f, 
		-0.114282f, -0.960413f, 0.254058f, 0.114282f, 0.960413f, -0.254058f, 
		-0.0432002f, 0.866071f, 0.498051f, 0.0432002f, -0.866071f, -0.498051f, 
		-0.349716f, -0.911336f, -0.217176f, -0.111173f, -0.178341f, -0.977668f, 
		-0.664693f, -0.734900f, 0.134555f, -0.505871f, -0.860787f, -0.0560413f, 
		-0.664693f, -0.734900f, 0.134555f, -0.840766f, -0.0850932f, -0.534670f, 
		-0.165501f, -0.759497f, -0.629105f, -0.202621f, 0.471723f, -0.858151f, 
		-0.127522f, -0.975538f, -0.179063f, 0.166727f, 0.975356f, 0.144510f, 
		-0.840549f, -0.0851328f, 0.535005f, -0.218205f, 0.476105f, 0.851887f, 
		-0.172676f, -0.756274f, 0.631057f, -0.516163f, -0.854884f, 0.0524345f, 
		-0.197656f, -0.453582f, -0.869020f, -0.360195f, -0.910441f, -0.203363f, 
		-0.228326f, -0.664354f, -0.711688f, 0.955159f, 0.00111103f, -0.296092f, 
		-0.814641f, 0.0178040f, 0.579692f, 0.0897375f, -0.158809f, 0.983223f, 
		-0.0490114f, -0.995450f, 0.0817152f, -0.113326f, 0.992973f, 0.0340862f, 
		-0.240026f, -0.710907f, -0.661060f, -0.314342f, -0.790879f, -0.525071f, 
		-0.240026f, -0.710907f, -0.661060f, -0.361211f, -0.759812f, -0.540567f, 
		-0.706323f, -0.707807f, -0.0108052f, -0.572129f, -0.819661f, 0.0287155f, 
		-0.686979f, -0.701935f, 0.188007f, -0.348834f, -0.911639f, -0.217323f, 
		-0.119855f, -0.324729f, -0.938182f, -0.669501f, -0.730153f, 0.136544f, 
		-0.509735f, -0.858539f, -0.0555155f, -0.669501f, -0.730153f, 0.136544f, 
		0.0875548f, 0.991506f, -0.0961813f, -0.496733f, 0.0946441f, 0.862727f, 
		-0.0875548f, -0.991506f, 0.0961813f, 0.0875548f, 0.991506f, -0.0961813f, 
		-0.0875548f, -0.991506f, 0.0961813f, 0.496733f, -0.0946441f, -0.862727f, 
		0.000623153f, 0.999990f, 0.00444283f, 0.0766026f, 0.0159655f, 0.996934f, 
		0.221028f, -0.0159640f, 0.975137f, 2.28523e-006f, -1.00000f, -3.05493e-006f, 
		-0.813573f, 0.00705165f, -0.581420f, -0.871033f, 0.150932f, -0.467462f, 
		-0.932774f, 0.302140f, -0.196582f, -0.0779902f, -0.976800f, 0.199448f, 
		0.0779902f, 0.976800f, -0.199448f, 0.842668f, -0.00259883f, 0.538428f, 
		0.932774f, -0.302140f, 0.196582f, 0.899664f, -0.197174f, 0.389522f, 
		0.0779902f, 0.976800f, -0.199448f, -0.0779902f, -0.976800f, 0.199448f, 
		0.0768856f, 0.978905f, 0.189299f, -0.0289161f, -0.978936f, -0.202111f, 
		-0.0289161f, -0.978936f, -0.202111f, 0.0768856f, 0.978905f, 0.189299f, 
		0.124365f, -0.960045f, -0.250694f, 0.0405825f, -0.681280f, -0.730897f, 
		-0.496733f, 0.0946441f, 0.862727f, -0.318191f, 0.316300f, 0.893705f, 
		-0.175500f, 0.978517f, 0.108188f, -0.0405825f, 0.681280f, 0.730897f, 
		-0.124365f, 0.960045f, 0.250694f, 0.496733f, -0.0946441f, -0.862727f, 
		0.175500f, -0.978517f, -0.108188f, 0.318191f, -0.316300f, -0.893705f, 
		-0.826336f, 0.0482351f, 0.561108f, 0.897826f, -0.0316196f, 0.439215f, 
		0.757143f, 0.0255108f, -0.652751f, -0.0711042f, -0.425268f, 0.902270f, 
		0.0711042f, 0.425268f, -0.902270f, -0.0688094f, 0.173427f, -0.982440f, 
		0.0295543f, -0.196953f, 0.979967f, 0.0295543f, -0.196953f, 0.979967f, 
		-0.0688094f, 0.173427f, -0.982440f, 0.0648149f, 0.161721f, 0.984706f, 
		0.0648149f, 0.161721f, 0.984706f, -0.0282443f, -0.184002f, -0.982520f, 
		0.0648149f, 0.161721f, 0.984706f, -0.0282443f, -0.184002f, -0.982520f, 
		-0.0470955f, -0.343427f, -0.937998f, -0.0505345f, -0.0530305f, 0.997313f, 
		0.0309882f, -0.180722f, -0.983046f, -0.847844f, 0.530187f, 0.00783264f, 
		0.0470955f, 0.343427f, 0.937998f, -0.0309882f, 0.180722f, 0.983046f, 
		0.0505345f, 0.0530305f, -0.997313f, 0.825350f, -0.564381f, -0.0164745f, 
		-0.0715731f, -0.913916f, 0.399543f, -0.146195f, 0.955112f, -0.257658f, 
		-0.514029f, 0.0921186f, -0.852812f, 0.0715731f, 0.913916f, -0.399543f, 
		-0.230806f, 0.969392f, -0.0837153f, 0.146195f, -0.955112f, 0.257658f, 
		0.230806f, -0.969392f, 0.0837153f, 0.514029f, -0.0921186f, 0.852812f, 
		-0.592253f, -0.405828f, 0.696090f, -0.693294f, -0.664832f, 0.278105f, 
		-0.640518f, -0.762578f, 0.0906205f, -0.111131f, -0.564295f, 0.818059f, 
		-0.467845f, -0.856019f, 0.219891f, 0.0882032f, -0.634241f, 0.768087f, 
		0.0100913f, -0.779437f, 0.626399f, -0.256542f, -0.121492f, -0.958867f, 
		-0.212794f, -0.960182f, -0.181021f, 0.212794f, 0.960182f, 0.181021f, 
		0.256542f, 0.121492f, 0.958867f, -0.965856f, 0.000000f, 0.259079f, 
		-0.0436613f, 0.998731f, 0.0251042f, -0.243507f, 0.000000f, 0.969899f, 
		0.965856f, 0.000000f, -0.259080f, 0.243507f, 0.000000f, -0.969899f, 
		0.0436613f, -0.998731f, -0.0251042f, -0.965856f, 0.000000f, 0.259079f, 
		0.205534f, 0.966183f, -0.155714f, -0.243507f, 0.000000f, 0.969899f, 
		0.965856f, 0.000000f, -0.259080f, 0.243507f, 0.000000f, -0.969899f, 
		-0.205534f, -0.966183f, 0.155714f, 0.0225024f, -0.962234f, -0.271294f, 
		-0.510028f, -0.860011f, 0.0158772f, -0.716193f, -0.697280f, 0.0294679f, 
		-0.884334f, -0.466854f, -0.000591301f, -0.109050f, -0.100058f, -0.988988f, 
		0.510028f, 0.860011f, -0.0158772f, 0.716193f, 0.697280f, -0.0294679f, 
		0.884334f, 0.466854f, 0.000591301f, 0.109050f, 0.100058f, 0.988988f, 
		-0.963768f, 0.182735f, -0.194317f, -0.294933f, 0.953769f, 0.0577892f, 
		-0.0605165f, 0.0666098f, 0.995942f, 0.963768f, -0.182735f, 0.194317f, 
		0.0605165f, -0.0666098f, -0.995942f, 0.240168f, -0.956008f, -0.168428f, 
		-0.0511708f, -0.0257096f, -0.998359f, 0.000287820f, -0.267755f, 0.963487f, 
		0.00219755f, 0.0103405f, 0.999944f, 0.294962f, 0.652648f, -0.697888f, 
		-0.586613f, -0.395546f, -0.706703f, -0.677639f, -0.649132f, -0.345590f, 
		-0.111433f, -0.564223f, -0.818068f, -0.360392f, -0.895218f, -0.262111f, 
		0.0879151f, -0.634174f, -0.768176f, -0.00645502f, -0.789040f, -0.614307f, 
		0.187944f, 0.128312f, -0.973762f, -0.212613f, -0.977136f, 0.000855723f, 
		-0.951684f, -0.306738f, -0.0144914f, 0.205929f, 0.978559f, -0.00400306f, 
		-0.187944f, -0.128312f, 0.973762f, 0.967826f, 0.251573f, -0.00494636f, 
		-0.202309f, -0.956899f, -0.208366f, -0.947550f, -0.149943f, 0.282253f, 
		-0.177128f, -0.275733f, -0.944774f, 0.178125f, 0.948903f, 0.260489f, 
		0.177128f, 0.275733f, 0.944774f, 0.947550f, 0.149943f, -0.282253f, 
		-0.0151462f, -0.00710386f, 0.999860f, 0.0151462f, 0.00710386f, -0.999860f, 
		-0.0682501f, 0.892846f, 0.445161f, 0.0682501f, -0.892846f, -0.445161f, 
		-0.0151462f, -0.00710386f, 0.999860f, 0.0151462f, 0.00710386f, -0.999860f, 
		-0.0682501f, 0.892846f, 0.445161f, 0.0682501f, -0.892846f, -0.445161f, 
		-0.0372638f, -0.0336628f, 0.998738f, 0.915589f, 0.0368613f, 0.400422f, 
		-0.0199789f, -0.0350929f, -0.999184f, -0.0682743f, 0.892740f, 0.445369f, 
		0.0682743f, -0.892740f, -0.445369f, -0.206639f, -0.978409f, 0.00413220f, 
		-0.932799f, -0.360366f, 0.00473781f, -0.248291f, -0.255689f, 0.934331f, 
		0.199323f, 0.979933f, -0.000786234f, 0.248291f, 0.255689f, -0.934331f, 
		0.952925f, 0.302502f, 0.0206649f, 0.0958593f, 0.971003f, -0.219009f, 
		-0.118098f, -0.970641f, 0.209542f, -0.965962f, 0.0105077f, -0.258469f, 
		-0.225443f, 0.00330022f, -0.974251f, 0.0539053f, -0.995382f, 0.0794228f, 
		0.964096f, -0.00724398f, 0.265456f, -0.113208f, 0.991596f, -0.0626157f, 
		0.214544f, -0.00904845f, 0.976672f, -0.0186050f, -0.0728070f, 0.997172f, 
		-0.0683700f, 0.136564f, -0.988269f, -0.409729f, 0.656868f, -0.632966f, 
		-0.292114f, 0.611473f, -0.735371f, -0.148906f, 0.311325f, -0.938565f, 
		0.0809098f, -0.146764f, 0.985857f, 0.371530f, -0.740398f, 0.560158f, 
		0.224796f, -0.654019f, 0.722306f, -0.140230f, 0.200054f, -0.969698f, 
		-0.963987f, 0.00719270f, -0.265854f, -0.236300f, 0.00920580f, -0.971637f, 
		-0.0248528f, 0.990800f, -0.133036f, 0.962122f, -0.0105753f, 0.272413f, 
		0.0248528f, -0.990800f, 0.133036f, 0.226556f, -0.00311315f, 0.973993f, 
		-0.929216f, 0.211964f, -0.302703f, -0.545472f, -0.248928f, 0.800310f, 
		-0.711966f, -0.467217f, 0.524226f, -0.213180f, -0.934121f, -0.286308f, 
		-0.463651f, -0.117369f, 0.878210f, -0.0561935f, -0.971530f, 0.230155f, 
		0.103394f, 0.970721f, -0.216819f, 0.103394f, 0.970721f, -0.216819f, 
		-0.503434f, 0.0953025f, 0.858761f, -0.164188f, -0.931025f, 0.325937f, 
		0.0917802f, 0.967904f, -0.233961f, 0.503434f, -0.0953025f, -0.858761f, 
		-0.119318f, -0.955316f, 0.270433f, -0.503447f, 0.0953030f, 0.858754f, 
		0.0905580f, 0.972997f, -0.212311f, 0.503447f, -0.0953030f, -0.858754f, 
		0.0869510f, 0.970446f, -0.225107f, -0.503673f, 0.0952476f, 0.858627f, 
		-0.166949f, -0.934206f, 0.315257f, 0.503673f, -0.0952476f, -0.858627f, 
		0.0639163f, -0.959375f, -0.274797f, -0.138443f, 0.181518f, 0.973594f, 
		-0.0605240f, -0.955967f, 0.287165f, -0.968612f, -0.215729f, -0.123497f, 
		-0.968612f, -0.215729f, -0.123497f, 0.0473329f, 0.0307954f, -0.998404f, 
		1.28466e-006f, -1.00000f, 3.59230e-006f, 2.28523e-006f, -1.00000f, -3.05493e-006f, 
		-1.73730e-005f, 1.00000f, -2.61493e-006f, 0.0524592f, -0.851037f, -0.522479f, 
		0.226906f, 0.151043f, -0.962133f, -0.933900f, 0.166834f, -0.316223f, 
		-0.322325f, 3.52166e-005f, 0.946629f, 0.387385f, -0.822710f, 0.416031f, 
		-0.235486f, 0.967809f, 0.0888433f, 0.313406f, -0.949614f, -0.00304546f, 
		0.978259f, -0.163648f, 0.127391f, 0.175250f, -0.911634f, -0.371768f, 
		0.0731162f, -0.841913f, -0.534637f, -0.207280f, 0.898869f, -0.386096f, 
		-0.233326f, 0.972393f, 0.00315733f, -0.936631f, 0.223223f, -0.269989f, 
		-0.275930f, -0.0551890f, 0.959592f, 0.372853f, -0.813829f, 0.445717f, 
		-0.212246f, 0.970924f, 0.110714f, 0.328850f, -0.944364f, 0.00582078f, 
		0.967973f, -0.213576f, 0.131960f, 0.191823f, -0.904422f, -0.381083f, 
		0.275930f, 0.0551890f, -0.959592f, 0.103379f, -0.822366f, 0.559489f, 
		0.103379f, -0.822366f, 0.559489f, -0.149952f, 0.825459f, -0.544181f, 
		0.103379f, -0.822366f, 0.559489f, -0.122705f, 0.864195f, 0.487966f, 
		0.163288f, -0.827894f, -0.536590f, -0.122705f, 0.864195f, 0.487966f, 
		-0.0569004f, -0.983227f, -0.173282f, -0.756945f, -0.247142f, -0.604942f, 
		-0.115561f, 0.395860f, 0.911011f, -0.0376499f, 0.893393f, 0.447695f, 
		0.0569004f, 0.983227f, 0.173282f, -0.0916753f, 0.162318f, 0.982471f, 
		0.759682f, 0.245952f, 0.601989f, 0.115561f, -0.395860f, -0.911011f, 
		0.0376499f, -0.893393f, -0.447695f, 0.0916753f, -0.162318f, -0.982471f, 
		-0.194255f, -0.486201f, 0.851982f, -0.934385f, -0.335686f, 0.119332f, 
		-0.157848f, 0.987280f, -0.0190483f, 0.157664f, 0.365088f, -0.917525f, 
		0.923715f, 0.381839f, -0.0307984f, 0.0674159f, -0.943940f, 0.323161f, 
		0.194907f, -0.976714f, 0.0896728f, -0.249348f, -0.332563f, 0.909521f, 
		-0.858730f, 0.195755f, 0.473564f, -0.727845f, -0.205008f, -0.654380f, 
		-0.564659f, -0.177752f, -0.805956f, -0.207346f, -0.909683f, 0.359839f, 
		-0.463970f, -0.117387f, -0.878039f, -0.0932643f, 0.717060f, 0.690743f, 
		-0.518027f, 0.548226f, 0.656579f, -0.698507f, 0.147308f, 0.700277f, 
		-0.529611f, 0.0657398f, 0.845690f, -0.689773f, -0.0345338f, 0.723202f, 
		0.164816f, -0.668316f, -0.725389f, 0.265928f, -0.509858f, -0.818124f, 
		0.699256f, 0.121462f, -0.704477f, 0.539652f, -0.236375f, -0.808024f, 
		-0.0501516f, 0.545032f, 0.836914f, 0.0885794f, 0.694942f, 0.713589f, 
		0.000372402f, 0.999989f, 0.00466627f, 0.0909632f, -0.646918f, 0.757114f, 
		-0.231448f, -0.972257f, 0.0338694f, 0.210473f, 0.933651f, -0.289823f, 
		-0.839963f, -0.163552f, -0.517410f, -0.273151f, -0.0128866f, 0.961885f, 
		-0.231448f, -0.972257f, 0.0338694f, 0.792953f, -0.339467f, 0.505953f, 
		0.211864f, 0.976900f, -0.0279180f, 0.987228f, 0.131783f, -0.0895170f, 
		0.199604f, 0.972079f, -0.123370f, 0.273151f, 0.0128866f, -0.961885f, 
		-0.478731f, 0.0881130f, -0.873529f, 0.106735f, 0.936307f, 0.334571f, 
		0.565999f, -0.0925887f, 0.819190f, -0.175426f, -0.929939f, -0.323171f, 
		-0.224192f, -0.935265f, -0.273891f, -0.478418f, 0.0877878f, -0.873733f, 
		0.115869f, 0.966105f, 0.230685f, 0.565804f, -0.0923894f, 0.819348f, 
		-0.478418f, 0.0877878f, -0.873733f, 0.0794748f, 0.940435f, 0.330553f, 
		-0.256108f, -0.935818f, -0.242183f, 0.565821f, -0.0923932f, 0.819335f, 
		-0.420440f, 0.525050f, -0.739968f, -0.416517f, 0.900993f, 0.121351f, 
		-0.847494f, 0.198224f, 0.492403f, -0.420440f, 0.525050f, -0.739968f, 
		-0.118094f, 0.532562f, 0.838112f, -0.0649337f, 0.907746f, 0.414463f, 
		-0.158541f, 0.654592f, 0.739171f, -0.499538f, -0.827472f, -0.256423f, 
		-0.333475f, -0.880239f, -0.337601f, 0.158541f, -0.654592f, -0.739171f, 
		0.0649337f, -0.907747f, -0.414463f, 0.118094f, -0.532562f, -0.838112f, 
		0.499538f, 0.827472f, 0.256423f, 0.333475f, 0.880239f, 0.337601f, 
		-0.396145f, 0.873380f, -0.283332f, -0.554655f, 0.815412f, -0.165714f, 
		-0.109670f, -0.620631f, 0.776396f, -0.0142671f, -0.769610f, 0.638355f, 
		0.396145f, -0.873380f, 0.283332f, 0.554655f, -0.815412f, 0.165714f, 
		0.0142671f, 0.769610f, -0.638355f, 0.109670f, 0.620631f, -0.776396f, 
		-0.947292f, -0.230616f, -0.222381f, -0.150372f, 0.972485f, 0.177935f, 
		0.0708728f, -0.104305f, 0.992017f, -0.0307647f, -0.842301f, 0.538128f, 
		-0.875506f, -0.478842f, 0.0648021f, -0.00367290f, 0.0700900f, -0.997534f, 
		0.391929f, 0.818067f, -0.420902f, 0.543112f, 0.714614f, -0.440858f, 
		0.743757f, 0.573041f, -0.344164f, 0.00367290f, -0.0700900f, 0.997534f, 
		0.869335f, -0.466936f, -0.161949f, 0.242421f, 0.304053f, -0.921295f, 
		0.102563f, 0.994283f, 0.0297074f, -0.251381f, 0.391469f, 0.885189f, 
		-0.499293f, 0.256569f, 0.827574f, -0.0935336f, -0.714586f, -0.693267f, 
		0.132527f, -0.562785f, -0.815910f, 0.499293f, -0.256569f, -0.827574f, 
		0.251381f, -0.391469f, -0.885189f, -0.132527f, 0.562785f, 0.815910f, 
		0.0935336f, 0.714586f, 0.693267f, -0.945318f, 0.322361f, -0.0495711f, 
		-0.0895848f, -0.995978f, 0.00135674f, -0.282213f, -0.234002f, 0.930376f, 
		-0.230660f, 0.419771f, -0.877832f, -0.611016f, -0.766161f, -0.199139f, 
		-0.793819f, -0.605662f, 0.0549937f, -0.793819f, -0.605662f, 0.0549937f, 
		-0.346552f, -0.835903f, -0.425639f, -0.261414f, -0.321888f, -0.909973f, 
		-0.152948f, 0.171509f, -0.973238f, -0.163015f, -0.0970035f, -0.981843f, 
		-0.175910f, -0.191822f, -0.965536f, -0.0968151f, 0.00355661f, -0.995296f, 
		-0.118450f, 0.532525f, -0.838085f, -0.0651196f, 0.907762f, -0.414400f, 
		-0.158861f, 0.654584f, -0.739109f, -0.499428f, -0.827467f, 0.256654f, 
		-0.333328f, -0.880240f, 0.337743f, 0.158861f, -0.654584f, 0.739109f, 
		0.0651196f, -0.907762f, 0.414400f, 0.118450f, -0.532525f, 0.838085f, 
		0.499428f, 0.827467f, -0.256654f, 0.333328f, 0.880240f, -0.337743f, 
		0.324043f, -0.896334f, 0.302626f, -0.148165f, 0.253019f, 0.956048f, 
		0.778199f, -0.591278f, 0.211652f, 0.467043f, -0.823843f, 0.321176f, 
		0.945019f, -0.243193f, 0.218625f, 0.298560f, -0.912164f, 0.280747f, 
		-0.122018f, 0.198318f, 0.972513f, 0.770564f, -0.606385f, 0.196284f, 
		0.449530f, -0.843418f, 0.294227f, 0.940983f, -0.274715f, 0.197693f, 
		0.310035f, -0.905007f, 0.291274f, -0.154082f, 0.341578f, 0.927137f, 
		0.628457f, -0.718278f, 0.298526f, 0.430283f, -0.830896f, 0.352802f, 
		0.987228f, 0.131783f, -0.0895170f, 0.0891516f, 0.118311f, -0.988966f, 
		0.124646f, 0.312834f, -0.941593f, 0.0393576f, 0.0233177f, -0.998953f, 
		-0.908818f, 0.384371f, -0.162203f, -0.219111f, -0.974403f, -0.0502944f, 
		-0.294494f, -0.213146f, -0.931580f, 0.908818f, -0.384371f, 0.162203f, 
		0.330976f, 0.321842f, 0.887059f, 0.219111f, 0.974403f, 0.0502944f, 
		-0.112639f, -0.534005f, 0.837945f, -0.249463f, -0.260707f, 0.932631f, 
		-0.282215f, 0.0336353f, 0.958761f, -0.0503152f, -0.365961f, 0.929269f, 
		-0.143567f, 0.0377488f, 0.988920f, 0.000207228f, 1.00000f, -0.000771035f, 
		-0.134149f, -0.608927f, -0.781800f, 0.167751f, 0.457837f, -0.873066f, 
		0.192308f, 0.124540f, -0.973400f, 0.196936f, 0.0241574f, -0.980119f, 
		-0.432617f, 0.343551f, 0.833556f, -0.342404f, 0.391280f, 0.854201f, 
		-0.260558f, 0.302213f, 0.916939f, -0.485941f, 0.418362f, 0.767355f, 
		-0.432617f, 0.343551f, 0.833556f, -0.578904f, 0.387098f, 0.717653f, 
		0.135381f, -0.290626f, -0.947211f, 0.373423f, -0.420355f, -0.826957f, 
		0.423532f, -0.526290f, -0.737319f, 0.423532f, -0.526290f, -0.737319f, 
		0.0419507f, 0.0717058f, -0.996543f, -0.662158f, 0.500256f, -0.557934f, 
		-0.389545f, 0.891010f, 0.233146f, -0.0338515f, -0.0575879f, 0.997766f, 
		0.0419507f, 0.0717058f, -0.996543f, 0.662158f, -0.500256f, 0.557934f, 
		-0.0338515f, -0.0575879f, 0.997766f, 0.458607f, -0.840707f, -0.287909f, 
		0.0495838f, 0.0740891f, 0.996018f, -0.0495838f, -0.0740891f, -0.996018f, 
		-0.601113f, 0.390861f, 0.697059f, -0.407913f, 0.824840f, -0.391467f, 
		0.0495838f, 0.0740891f, 0.996018f, 0.601113f, -0.390861f, -0.697059f, 
		-0.0495838f, -0.0740891f, -0.996018f, 0.467489f, -0.758155f, 0.454594f, 
		-0.921947f, -0.385372f, 0.0387516f, -0.194716f, 0.977586f, -0.0800696f, 
		0.898770f, 0.422099f, -0.118509f, -0.0283610f, -0.00333582f, 0.999592f, 
		0.0283610f, 0.00333582f, -0.999592f, 0.229375f, -0.973283f, 0.0103528f, 
		0.323004f, 0.182196f, -0.928694f, -0.933900f, 0.166834f, -0.316223f, 
		-0.322325f, 3.52166e-005f, 0.946629f, 0.245964f, 0.961234f, -0.124623f, 
		-0.316110f, -0.947645f, 0.0451945f, 0.945019f, -0.243193f, 0.218625f, 
		0.0675008f, -0.916874f, -0.393428f, -0.839963f, -0.163552f, -0.517410f, 
		-0.241254f, 0.0498710f, 0.969180f, 0.162713f, -0.943307f, -0.289302f, 
		-0.222297f, 0.971167f, 0.0861297f, 0.260166f, -0.948153f, -0.182538f, 
		0.987228f, 0.131783f, -0.0895170f, 0.162713f, -0.943307f, -0.289302f, 
		0.241254f, -0.0498710f, -0.969180f, -0.00274039f, 0.000000f, 0.999996f, 
		-0.904596f, 0.000000f, -0.426269f, -0.0230200f, 0.999305f, -0.0293259f, 
		0.0686226f, 0.123322f, -0.989991f, 0.917430f, -0.0769965f, 0.390376f, 
		0.0230200f, -0.999305f, 0.0293259f, -0.0730290f, -0.117172f, 0.990423f, 
		-0.919271f, 0.0768351f, -0.386054f, 0.145877f, 0.000000f, -0.989303f, 
		-0.00213879f, -0.997878f, -0.0650716f, 0.00213879f, 0.997878f, 0.0650716f, 
		0.936694f, 0.000000f, 0.350149f, -0.413842f, 0.450239f, -0.791214f, 
		-0.377043f, 0.722246f, 0.579827f, 0.273201f, 0.961678f, 0.0231719f, 
		0.0165340f, -0.986483f, -0.163029f, 0.0165340f, -0.986483f, -0.163029f, 
		0.0165340f, -0.986483f, -0.163029f, -0.0305335f, 0.989275f, 0.142841f, 
		0.0165340f, -0.986483f, -0.163029f, -0.0305335f, 0.989275f, 0.142841f, 
		0.230775f, 0.686499f, -0.689537f, -0.936631f, 0.223223f, -0.269989f, 
		-0.275930f, -0.0551890f, 0.959592f, 0.940983f, -0.274715f, 0.197693f, 
		0.252842f, 0.957908f, -0.135956f, -0.289750f, -0.948389f, 0.128858f, 
		0.267508f, 0.916779f, -0.296572f, 0.268771f, 0.281863f, -0.921040f, 
		-0.866987f, -0.419806f, -0.268508f, -0.866987f, -0.419806f, -0.268508f, 
		-0.945087f, -0.269034f, -0.185554f, -0.623670f, -0.717059f, -0.311228f, 
		-0.340992f, -0.875580f, -0.342176f, -0.158653f, 0.238065f, -0.958204f, 
		-0.768854f, 0.618944f, 0.160538f, -0.216151f, 0.766128f, -0.605249f, 
		-0.153437f, 0.524145f, 0.837692f, 0.212001f, 0.912844f, -0.348958f, 
		-0.0164563f, 0.890780f, -0.454137f, 0.0682399f, -0.987196f, -0.144179f, 
		-0.0466271f, 0.990595f, 0.128639f, -0.0466271f, 0.990595f, 0.128639f, 
		0.0682399f, -0.987196f, -0.144179f, -0.0466271f, 0.990595f, 0.128639f, 
		-0.547191f, -0.816472f, 0.184273f, -0.743412f, -0.606568f, 0.281804f, 
		-0.585706f, -0.760875f, 0.279317f, -0.207944f, 0.232192f, 0.950182f, 
		-0.452999f, -0.780071f, 0.431604f, 0.00419251f, -0.987661f, -0.156549f, 
		-0.605300f, 0.560162f, 0.565536f, -0.245030f, -0.0264332f, 0.969155f, 
		0.0314503f, 0.981163f, 0.190605f, 0.605300f, -0.560162f, -0.565536f, 
		0.245030f, 0.0264332f, -0.969155f, -0.973998f, 0.167828f, 0.152192f, 
		-0.110332f, -0.993385f, 0.0318187f, 0.879777f, -0.196712f, 0.432779f, 
		-0.225289f, -0.200204f, -0.953501f, 0.932824f, -0.339467f, 0.120837f, 
		0.315925f, 0.198169f, 0.927858f, 0.107830f, 0.993924f, 0.0220935f, 
		0.162757f, 0.258275f, 0.952263f, 0.0936975f, -0.124940f, -0.987730f, 
		-0.791928f, 0.610076f, 0.0256354f, -0.0931603f, 0.995006f, -0.0358305f, 
		-0.0936975f, 0.124940f, 0.987730f, 0.748613f, -0.662996f, -0.00387730f, 
		0.564885f, -0.825109f, 0.0100009f, 0.422230f, -0.906425f, 0.0107847f, 
		-0.460342f, -0.00641137f, 0.887718f, -0.589200f, -0.00276092f, 0.807982f, 
		-0.776149f, 0.0131075f, 0.630413f, 0.0929956f, 0.0339128f, -0.995089f, 
		0.791877f, -0.0256331f, -0.610142f, 0.0563330f, -0.991557f, -0.116798f, 
		-0.0563330f, 0.991557f, 0.116798f, -0.0247599f, 0.920496f, 0.389966f, 
		-0.378871f, -0.212071f, 0.900823f, -0.763654f, 0.00848830f, 0.645569f, 
		-0.518245f, -0.248884f, 0.818217f, -0.379176f, -0.346635f, 0.857945f, 
		-0.392727f, -0.486123f, 0.780673f, 0.000552650f, -0.999968f, 0.00803811f, 
		-0.355913f, 0.863742f, 0.356758f, 0.000326785f, 0.278728f, 0.960370f, 
		-0.0238906f, -0.835298f, 0.549278f, -0.932147f, 0.350284f, -0.0916721f, 
		-0.270837f, -0.962318f, 0.0243196f, 0.932147f, -0.350284f, 0.0916721f, 
		-0.230134f, -0.171973f, 0.957843f, 0.270837f, 0.962318f, -0.0243196f, 
		0.179609f, 0.266717f, -0.946891f, -0.244164f, -0.845487f, 0.474906f, 
		-0.0354339f, -0.554827f, 0.831211f, -0.781964f, 0.540722f, 0.310083f, 
		-0.0388160f, 0.320310f, -0.946517f, 0.0354339f, 0.554827f, -0.831211f, 
		0.244164f, 0.845487f, -0.474906f, 0.731674f, -0.610529f, -0.303163f, 
		0.0388160f, -0.320310f, 0.946517f, -0.668639f, 0.320168f, -0.671129f, 
		-0.407730f, -0.662624f, -0.628240f, 0.232607f, 0.176616f, -0.956400f, 
		-0.108651f, 0.764811f, 0.635028f, 0.267094f, 0.291408f, 0.918554f, 
		-0.582749f, -0.249583f, -0.773377f, -0.653006f, -0.178098f, -0.736114f, 
		-0.430917f, -0.338024f, -0.836690f, -0.582749f, -0.249583f, -0.773377f, 
		0.645376f, 0.0460417f, 0.762476f, 0.427211f, 0.233810f, 0.873398f, 
		0.146326f, 0.445443f, 0.883272f, 0.0150239f, 0.854883f, -0.518603f, 
		-0.214799f, -0.482043f, -0.849409f, -0.175292f, -0.911123f, 0.372997f, 
		-0.794632f, -0.551388f, 0.254030f, -0.587427f, -0.712409f, 0.383931f, 
		-0.159457f, 0.983019f, -0.0908148f, -0.240508f, 0.959824f, 0.144550f, 
		0.182188f, 0.936254f, -0.300393f, 0.214799f, 0.482043f, 0.849409f, 
		0.391929f, 0.818067f, -0.420902f, 0.240508f, -0.959824f, -0.144550f, 
		0.159457f, -0.983019f, 0.0908148f, -0.455611f, 0.0323087f, 0.889592f, 
		-0.226025f, 0.113692f, 0.967464f, -0.210226f, 0.915411f, 0.343260f, 
		0.301988f, 0.285651f, 0.909509f, 0.0658965f, 0.208671f, 0.975763f, 
		-0.238823f, -0.968279f, -0.0734795f, -0.585935f, 0.603565f, -0.540730f, 
		0.238823f, 0.968279f, 0.0734795f, -0.123148f, 0.447651f, 0.885688f, 
		0.123148f, -0.447651f, -0.885688f, 0.580831f, -0.589857f, 0.560985f, 
		-0.0848999f, -0.829046f, -0.552698f, -0.237927f, 0.456638f, -0.857247f, 
		-0.549653f, -0.833752f, -0.0523336f, -0.149683f, -0.398458f, -0.904890f, 
		0.149034f, -0.929148f, -0.338339f, 0.0819628f, -0.676164f, -0.732178f, 
		0.221962f, -0.965909f, -0.133236f, 0.173501f, -0.433327f, -0.884378f, 
		-0.215241f, 0.852434f, 0.476474f, -0.237245f, 0.775258f, -0.585398f, 
		-0.484398f, 0.659580f, -0.574728f, 0.0593229f, 0.772731f, -0.631955f, 
		-0.230056f, -0.802274f, 0.550846f, -0.0156508f, -0.706449f, 0.707590f, 
		0.756695f, -0.647743f, 0.0885578f, 0.0899577f, -0.474648f, 0.875567f, 
		-0.317220f, 0.566803f, 0.760332f, -0.425025f, 0.745610f, 0.513244f, 
		-0.929216f, 0.211964f, -0.302703f, 0.138460f, 0.958368f, -0.249717f, 
		-0.163709f, 0.461841f, 0.871724f, -0.642928f, -0.762491f, 0.0724688f, 
		-0.250648f, 0.914045f, -0.318901f, 0.642928f, 0.762491f, -0.0724688f, 
		0.0552943f, 0.499544f, 0.864522f, -0.0552943f, -0.499544f, -0.864522f, 
		0.233309f, -0.937264f, 0.259043f, -0.287247f, 0.664443f, 0.689931f, 
		-0.708309f, -0.103712f, 0.698242f, -0.245003f, 0.00779219f, -0.969491f, 
		0.287247f, -0.664443f, -0.689931f, 0.708309f, 0.103712f, -0.698242f, 
		-0.204581f, -0.849651f, 0.486045f, 0.261223f, 0.842775f, -0.470629f, 
		0.245003f, -0.00779219f, 0.969491f, -0.118196f, 0.768158f, 0.629256f, 
		-0.709519f, 0.321458f, 0.627095f, -0.549385f, 0.573291f, 0.607876f, 
		0.0518262f, -0.356176f, 0.932981f, 0.171614f, 0.751424f, 0.637112f, 
		-0.401795f, 0.298550f, -0.865695f, -0.0630285f, 0.939956f, -0.335426f, 
		-0.570716f, 0.268768f, -0.775917f, -0.000348588f, 0.848364f, -0.529414f, 
		0.0267214f, 0.480573f, -0.876547f, -0.671234f, 0.340769f, -0.658271f, 
		-0.0751258f, 0.938571f, -0.336809f, -0.0180028f, 0.853747f, -0.520376f, 
		0.169757f, 0.664876f, -0.727408f, 0.0546350f, 0.965470f, -0.254720f, 
		-0.607654f, 0.667243f, 0.430747f, -0.390244f, 0.763242f, 0.514947f, 
		0.0802341f, 0.995945f, 0.0406848f, -0.0315624f, 0.892666f, 0.449612f, 
		-0.858730f, 0.195755f, 0.473564f, 0.170705f, 0.944134f, 0.281904f, 
		-0.179842f, 0.377711f, -0.908290f, -0.0260542f, 0.991094f, -0.130587f, 
		-0.731023f, -0.00735854f, -0.682314f, 0.0260542f, -0.991094f, 0.130587f, 
		-0.216236f, 0.00124455f, 0.976340f, 0.762927f, -0.00428541f, 0.646470f, 
		0.214136f, -0.00237127f, -0.976801f, -0.0597642f, 0.994065f, 0.0909062f, 
		-0.0980563f, 0.310482f, -0.945508f, -0.691243f, 0.00588379f, 0.722599f, 
		0.0178878f, -0.996185f, -0.0854192f, 0.126601f, 0.00698841f, 0.991929f, 
		0.702734f, -0.0106484f, -0.711373f, 0.549566f, 0.603569f, -0.577652f, 
		0.414805f, 0.574741f, -0.705414f, -0.0408775f, 0.927970f, 0.370405f, 
		0.221212f, 0.469882f, -0.854562f, -0.0928320f, -0.995682f, -0.000530400f, 
		-0.762685f, 0.646732f, -0.00701046f, 0.164406f, 0.985824f, -0.0334898f, 
		-0.106568f, 0.121619f, -0.986840f, 0.740882f, -0.671630f, -0.00267726f, 
		0.178948f, -0.184399f, 0.966424f, -0.249640f, -0.961049f, 0.118592f, 
		0.156771f, -0.581378f, -0.798388f, 0.737672f, -0.662992f, 0.127601f, 
		-0.697473f, 0.00379105f, -0.716601f, -0.0817304f, -0.815186f, 0.573403f, 
		-0.218333f, 0.00239749f, 0.975872f, 0.737529f, 0.0130645f, 0.675189f, 
		0.216173f, -0.00122861f, -0.976354f, 0.0817304f, 0.815186f, -0.573403f, 
		-0.520956f, -0.849577f, 0.0826038f, -0.276428f, -0.959445f, 0.0552600f, 
		0.217896f, -0.964734f, -0.147680f, 0.441505f, -0.843971f, -0.304609f, 
		-0.0923502f, -0.173552f, -0.980485f, -0.305371f, -0.952216f, -0.00583535f, 
		-0.510028f, -0.860011f, 0.0158772f, 0.0441969f, 0.998869f, -0.0175563f, 
		0.196088f, 0.500432f, 0.843278f, 0.305371f, 0.952216f, 0.00583535f, 
		0.510028f, 0.860011f, -0.0158772f, -0.0441969f, -0.998869f, 0.0175563f, 
		-0.196088f, -0.500432f, -0.843278f, -0.124316f, -0.00729343f, -0.992216f, 
		-0.701722f, 0.0106023f, 0.712372f, 0.150058f, -0.00202215f, 0.988675f, 
		-0.141347f, 0.851189f, 0.505469f, 0.714100f, -0.00610856f, -0.700017f, 
		0.141347f, -0.851189f, -0.505469f, -0.113724f, 0.276763f, -0.954185f, 
		-0.704842f, 0.703026f, 0.0946201f, -0.168677f, -0.980734f, -0.0985360f, 
		0.113724f, -0.276763f, 0.954185f, 0.670467f, -0.738109f, -0.0752865f, 
		0.165294f, 0.981441f, 0.0972204f, -0.462493f, 0.297864f, -0.835091f, 
		-0.618090f, 0.0716938f, -0.782831f, -0.206552f, -0.647323f, 0.733695f, 
		-0.0206121f, -0.614387f, 0.788736f, 0.0779581f, -0.510049f, 0.856605f, 
		0.393143f, -0.348806f, 0.850749f, 0.644615f, -0.163903f, 0.746731f, 
		0.539938f, -0.299382f, 0.786662f, 0.00860814f, 0.717366f, -0.696643f, 
		-0.155364f, 0.535034f, -0.830422f, 0.00860814f, 0.717366f, -0.696643f, 
		-0.0127866f, -0.612127f, -0.790656f, -0.0656178f, -0.997568f, -0.0235125f, 
		-0.571660f, 0.820392f, -0.0127345f, -0.756432f, 0.654072f, 5.96151e-005f, 
		-0.215916f, 0.976382f, 0.00764112f, 0.0656178f, 0.997568f, 0.0235125f, 
		0.0127866f, 0.612127f, 0.790656f, 0.422230f, -0.906425f, 0.0107847f, 
		0.209019f, -0.977651f, -0.0225826f, -0.159728f, -0.986771f, 0.0277353f, 
		-0.743653f, 0.668542f, 0.00566690f, 0.236330f, 0.971671f, 0.00182005f, 
		-0.0538254f, 0.460423f, -0.886066f, 0.718244f, -0.695760f, 0.00662803f, 
		0.0538254f, -0.460423f, 0.886066f, -0.125098f, 0.647242f, -0.751949f, 
		-0.125098f, 0.647242f, -0.751949f, -0.419094f, -0.196644f, 0.886392f, 
		-0.674117f, -0.115252f, 0.729578f, 0.160510f, -0.709153f, 0.686541f, 
		0.0408781f, -0.614869f, 0.787569f, 0.160510f, -0.709153f, 0.686541f, 
		0.653792f, 0.147840f, -0.742091f, 0.550225f, 0.286805f, -0.784216f, 
		0.153469f, 0.479285f, -0.864137f, -0.520923f, -0.0824836f, 0.849609f, 
		-0.263892f, -0.0496555f, 0.963273f, 0.0186243f, -0.0815063f, 0.996499f, 
		-0.0944952f, 0.940893f, 0.325256f, -0.264271f, 0.0132895f, 0.964357f, 
		-0.460342f, -0.00641137f, 0.887718f, -0.234023f, 0.911410f, 0.338475f, 
		-0.0190507f, -0.00719017f, -0.999793f, 0.264946f, -0.00672265f, -0.964240f, 
		0.776940f, -0.0148634f, -0.629400f, 0.589917f, 0.00150371f, -0.807462f, 
		0.234023f, -0.911410f, -0.338475f, 0.0190507f, 0.00719017f, 0.999793f, 
		-0.945318f, 0.322361f, -0.0495711f, -0.184799f, 0.370989f, -0.910064f, 
		-0.141026f, -0.831942f, -0.536641f, 0.902800f, -0.177966f, 0.391510f, 
		0.287491f, -0.0915382f, 0.953399f, 0.144271f, 0.880968f, 0.450645f, 
		-0.0590159f, -0.987064f, 0.149069f, -0.0314470f, 0.0776404f, 0.996485f, 
		0.0214272f, -0.817612f, 0.575371f, -0.532436f, 0.732465f, -0.424271f, 
		-0.532436f, 0.732465f, -0.424271f, -0.292555f, 0.851234f, -0.435674f, 
		-0.0214272f, 0.817612f, -0.575371f, 0.0314470f, -0.0776404f, -0.996485f, 
		0.0590159f, 0.987064f, -0.149069f, 0.582681f, -0.755679f, 0.299052f, 
		0.582681f, -0.755679f, 0.299052f, 0.333833f, -0.876893f, 0.345853f, 
		-0.245900f, 0.965111f, 0.0899647f, -0.696938f, -0.712762f, -0.0790511f, 
		0.235141f, -0.968359f, -0.0836015f, -0.0699667f, -0.333459f, 0.940165f, 
		0.696938f, 0.712762f, 0.0790511f, 0.131342f, 0.317155f, -0.939235f, 
		0.0646991f, 0.841048f, 0.537078f, -0.380124f, 0.123999f, -0.916586f, 
		-0.565610f, 0.0936304f, -0.819341f, -0.0441037f, -0.112009f, 0.992728f, 
		0.229903f, -0.124835f, 0.965174f, 0.461663f, -0.0853191f, 0.882943f, 
		0.000128221f, 1.00000f, 0.000120307f, 4.94320e-005f, -1.00000f, -0.000165703f, 
		0.0441037f, 0.112009f, -0.992728f, -0.380291f, 0.124243f, -0.916484f, 
		-0.565807f, 0.0938518f, -0.819179f, -0.0440596f, -0.112208f, 0.992707f, 
		0.229922f, -0.125108f, 0.965134f, 0.461682f, -0.0855945f, 0.882906f, 
		0.000128221f, 1.00000f, 0.000120307f, 4.94320e-005f, -1.00000f, -0.000165703f, 
		0.0440596f, 0.112208f, -0.992707f, -0.380084f, 0.123990f, -0.916604f, 
		-0.565592f, 0.0936293f, -0.819353f, -0.0441754f, -0.111967f, 0.992730f, 
		0.229842f, -0.124817f, 0.965191f, 0.461629f, -0.0853115f, 0.882961f, 
		0.000128221f, 1.00000f, 0.000120307f, 4.94320e-005f, -1.00000f, -0.000165703f, 
		0.0441754f, 0.111967f, -0.992730f, 0.0883686f, 0.175997f, -0.980416f, 
		-0.539206f, -0.767883f, 0.345851f, -0.282137f, 0.927078f, -0.246831f, 
		0.539206f, 0.767883f, -0.345851f, -0.0883686f, -0.175997f, 0.980416f, 
		0.250979f, -0.910875f, 0.327591f, -0.672727f, 0.737038f, 0.0649063f, 
		-0.165531f, -0.981418f, -0.0970451f, 0.635807f, -0.765447f, -0.0991983f, 
		-0.000328151f, 6.98384e-005f, -1.00000f, 0.000328151f, -6.98384e-005f, 1.00000f, 
		0.162334f, 0.981774f, 0.0988299f, -0.152745f, -0.893113f, -0.423105f, 
		-0.936631f, 0.223223f, -0.269989f, -0.0274196f, 0.576990f, 0.816291f, 
		-0.0271165f, 0.342762f, 0.939031f, 0.180050f, 0.880099f, 0.439327f, 
		0.905725f, -0.333441f, 0.261688f, 0.203251f, 0.945458f, 0.254554f, 
		0.309529f, 0.948259f, 0.0706889f, -0.152745f, -0.893113f, -0.423105f, 
		-0.152745f, -0.893113f, -0.423105f, 0.449567f, 0.867135f, -0.214399f, 
		-0.246432f, -0.965942f, 0.0789085f, -0.181176f, -0.251999f, -0.950616f, 
		0.0950197f, 0.380535f, 0.919872f, -0.187899f, 0.533977f, 0.824356f, 
		0.246432f, 0.965942f, -0.0789085f, 0.181176f, 0.251999f, 0.950616f, 
		0.187899f, -0.533977f, -0.824356f, -0.0950197f, -0.380535f, -0.919872f, 
		-0.475084f, -0.0824639f, 0.876068f, -0.101554f, -0.815254f, -0.570129f, 
		-0.933900f, 0.166834f, -0.316223f, 0.176717f, 0.947700f, 0.265773f, 
		-0.185491f, 0.195745f, 0.962952f, 0.896888f, -0.318907f, 0.306414f, 
		0.00899385f, -0.447824f, -0.894077f, -0.174768f, -0.899325f, -0.400837f, 
		-0.591391f, -0.174444f, -0.787290f, -0.135026f, 0.236139f, 0.962292f, 
		0.0968512f, 0.191097f, 0.976781f, 0.508400f, 0.263766f, 0.819730f, 
		0.00773000f, -0.998825f, 0.0478448f, -0.00773000f, 0.998825f, -0.0478448f, 
		0.352864f, 0.000000f, -0.935675f, -0.721099f, 0.000000f, -0.692832f, 
		0.0929467f, 0.980961f, 0.170516f, -0.258374f, 0.000000f, 0.966045f, 
		0.721099f, 0.000000f, 0.692832f, 0.303194f, -0.0992587f, -0.947745f, 
		-0.0929467f, -0.980961f, -0.170516f, -0.686893f, 0.0267538f, -0.726266f, 
		-0.0814913f, 0.877075f, -0.473390f, -0.262803f, 0.958238f, -0.112759f, 
		-0.502400f, 0.862129f, -0.0657877f, -0.00765825f, 0.996299f, -0.0856169f, 
		-0.252566f, 0.961401f, -0.109176f, -0.448475f, 0.890202f, -0.0800654f, 
		0.0404736f, -0.995864f, 0.0813421f, 0.252566f, -0.961401f, 0.109176f, 
		0.448475f, -0.890202f, 0.0800654f, 0.000320079f, 0.000259840f, 1.00000f, 
		-0.000320079f, -0.000259840f, -1.00000f, -0.0404736f, 0.995864f, -0.0813421f, 
		-0.256966f, 0.962879f, 0.0826621f, -0.699395f, -0.710081f, -0.0814388f, 
		0.245894f, -0.965041f, -0.0907344f, 0.699395f, 0.710081f, 0.0814388f, 
		0.000183515f, 0.000129174f, 1.00000f, -0.000131337f, -0.000249641f, -1.00000f, 
		-0.686750f, -0.726397f, 0.0268528f, -0.0915567f, -0.480418f, 0.872248f, 
		-0.206618f, 0.633982f, -0.745235f, -0.206618f, 0.633982f, -0.745235f, 
		-0.460625f, -0.780003f, 0.423579f, -0.460625f, -0.780003f, 0.423579f, 
		-0.250098f, -0.849938f, 0.463742f, 0.359827f, -0.694510f, 0.623041f, 
		0.587194f, -0.571485f, 0.573243f, 0.963870f, 0.225221f, 0.142236f, 
		0.471329f, 0.850843f, -0.232194f, 0.280246f, 0.753013f, -0.595343f, 
		-0.0590476f, 0.682734f, -0.728277f, -0.182223f, 0.613831f, -0.768119f, 
		-0.0680299f, 0.649400f, -0.757398f, -0.353571f, -0.821076f, 0.448132f, 
		-0.674195f, -0.699885f, 0.235842f, -0.0882006f, -0.833776f, 0.545013f, 
		0.0572512f, -0.768448f, 0.637346f, 0.972007f, 0.233647f, 0.0247374f, 
		0.445977f, 0.813221f, -0.373866f, 0.196261f, 0.786546f, -0.585515f, 
		0.300378f, 0.740351f, -0.601376f, 0.0346356f, 0.679634f, -0.732733f, 
		-0.158911f, 0.461767f, -0.872650f, -0.0644927f, 0.653990f, -0.753749f, 
		-0.0836617f, 0.686533f, -0.722269f, -0.451527f, -0.787350f, 0.419766f, 
		-0.451527f, -0.787350f, 0.419766f, -0.243186f, -0.854071f, 0.459808f, 
		0.338480f, -0.724351f, 0.600622f, 0.556614f, -0.623612f, 0.548898f, 
		0.911194f, 0.398622f, -0.104045f, 0.560086f, 0.786295f, -0.260853f, 
		0.642193f, 0.689892f, -0.334120f, 0.381637f, 0.744549f, -0.547722f, 
		0.0835301f, 0.683799f, -0.724873f, -0.725496f, 0.286471f, -0.625771f, 
		-0.383002f, -0.439982f, 0.812235f, 0.350227f, 0.508703f, -0.786487f, 
		-0.964463f, -0.176304f, 0.196795f, 0.00565751f, 0.995569f, -0.0938591f, 
		-0.498589f, -0.861210f, 0.0986275f, -0.253257f, -0.958547f, 0.130573f, 
		-0.00565751f, -0.995569f, 0.0938591f, 0.498589f, 0.861210f, -0.0986275f, 
		-0.000371239f, -0.000208087f, -1.00000f, 0.253257f, 0.958547f, -0.130573f, 
		0.000466303f, 0.000140694f, 1.00000f, -0.0677054f, 0.661642f, 0.746757f, 
		-0.0677054f, 0.661642f, 0.746757f, 0.0837273f, 0.748880f, 0.657395f, 
		0.972007f, 0.233647f, 0.0247374f, 0.457888f, 0.877580f, 0.142100f, 
		0.382503f, 0.839247f, 0.386466f, -0.100052f, 0.359196f, -0.927884f, 
		-0.220515f, 0.255417f, -0.941348f, 0.105006f, -0.872616f, -0.476986f, 
		0.369890f, -0.784867f, -0.497158f, 0.236262f, -0.820574f, -0.520421f, 
		0.0506975f, 0.534060f, -0.843925f, -0.100040f, 0.359224f, -0.927874f, 
		-0.220526f, 0.255435f, -0.941340f, 0.0973032f, -0.891857f, -0.441728f, 
		0.355047f, -0.798055f, -0.486877f, 0.230579f, -0.838007f, -0.494548f, 
		0.0507127f, 0.534089f, -0.843906f, -0.100040f, 0.359224f, -0.927874f, 
		-0.220526f, 0.255435f, -0.941340f, 0.0732269f, -0.900573f, -0.428494f, 
		0.330226f, -0.809379f, -0.485651f, 0.212768f, -0.847655f, -0.486015f, 
		0.0507127f, 0.534089f, -0.843906f, -0.821314f, -0.570064f, -0.0216910f, 
		-0.207944f, 0.232192f, 0.950182f, -0.0998440f, 0.982932f, 0.154519f, 
		0.469106f, -0.427446f, 0.772806f, 0.556622f, -0.225546f, 0.799563f, 
		-0.841530f, 0.103942f, -0.530117f, -0.519053f, -0.185789f, 0.834306f, 
		-0.817488f, 0.111944f, 0.564962f, -0.588371f, -0.0915174f, -0.803395f, 
		-0.0342039f, -0.203910f, 0.978392f, 0.171452f, 0.964352f, -0.201566f, 
		0.0342039f, 0.203910f, -0.978392f, -0.171452f, -0.964352f, 0.201566f, 
		-0.918091f, 0.186544f, 0.349728f, -0.168956f, -0.333022f, -0.927658f, 
		-0.500220f, 0.497971f, 0.708382f, -0.233186f, 0.624861f, 0.745099f, 
		-0.991572f, -0.0288765f, 0.126299f, -0.332023f, -0.928789f, 0.164658f, 
		-0.995099f, 0.0968502f, -0.0199515f, -1.96055e-005f, 1.00000f, 8.06487e-005f, 
		0.222609f, 0.00810996f, 0.974874f, -0.814928f, -0.00447732f, 0.579545f, 
		-0.890207f, 0.00461189f, -0.455532f, -0.167973f, -0.964694f, -0.202856f, 
		-0.992315f, 0.123113f, 0.0123967f, 0.614853f, 0.690335f, 0.381305f, 
		-0.873656f, -0.359069f, -0.328320f, -0.669985f, 0.525955f, 0.523920f, 
		0.224942f, 0.685248f, 0.692702f, -0.303973f, 0.535824f, 0.787714f, 
		-0.191172f, 0.472238f, 0.860491f, -0.714119f, 0.444183f, 0.541050f, 
		0.0521971f, 0.710222f, 0.702040f, -0.0190680f, 0.611817f, 0.790769f, 
		0.155837f, 0.742953f, -0.650949f, 0.00172468f, 0.662124f, -0.749392f, 
		-0.970740f, -0.178407f, 0.160731f, 0.382176f, 0.726294f, -0.571349f, 
		-0.103560f, -0.944081f, -0.313027f, -0.133710f, -0.907005f, -0.399329f, 
		-0.839963f, -0.163552f, -0.517410f, -0.236803f, 0.230387f, 0.943847f, 
		-0.0662742f, -0.972837f, -0.221803f, 0.792953f, -0.339467f, 0.505953f, 
		0.314911f, -0.114706f, -0.942164f, -0.0503114f, 0.966956f, -0.249931f, 
		-0.907965f, 0.408778f, -0.0921938f, -0.548543f, 0.823282f, -0.145971f, 
		-0.985053f, 0.169864f, -0.0285894f, -0.330996f, 0.930336f, -0.157852f, 
		-0.893346f, -0.408351f, 0.187570f, 0.493141f, -0.720819f, 0.487065f, 
		-0.996169f, -0.0472915f, 0.0735615f, -0.265622f, 0.927765f, -0.262100f, 
		-0.984450f, 0.156068f, -0.0806238f, -0.722294f, -0.617645f, 0.311136f, 
		-0.402734f, 0.395250f, -0.825580f, -0.932147f, 0.350284f, -0.0916721f, 
		-0.851248f, 0.390012f, -0.351094f, -0.226101f, 0.354359f, -0.907363f, 
		-0.417334f, -0.668600f, 0.615472f, -0.475491f, -0.847589f, 0.235586f, 
		-0.0445956f, -0.428326f, 0.902523f, 0.932147f, -0.350284f, 0.0916721f, 
		-0.226101f, 0.354359f, -0.907363f, 0.548818f, 0.828554f, 0.110892f, 
		0.0382791f, -0.315469f, 0.948163f, 0.0314224f, 0.863704f, -0.503018f, 
		0.0239103f, 0.923144f, -0.383711f, 0.0297338f, 0.967792f, -0.249991f, 
		-3.98892e-005f, -1.00000f, 0.000985855f, -0.844885f, -0.375563f, -0.380949f, 
		-0.138610f, -0.662788f, 0.735867f, -0.0743646f, -0.447423f, 0.891225f, 
		-0.0326007f, -0.286267f, 0.957595f, 0.554738f, -0.747103f, -0.366200f, 
		-0.0888121f, -0.00340998f, -0.996043f, 0.0888121f, 0.00340998f, 0.996043f, 
		0.528135f, 0.846037f, 0.0727678f, -0.297759f, 0.953206f, -0.0523196f, 
		-0.524792f, -0.851227f, 0.00239245f, 0.00375773f, 0.0530937f, 0.998582f, 
		-0.133439f, 0.0436863f, -0.990094f, 0.798204f, -0.597189f, -0.0789723f, 
		-0.953379f, 0.292340f, 0.0748652f, -0.878131f, -0.368929f, -0.304595f, 
		-0.158653f, 0.238065f, -0.958204f, 0.573047f, -0.541554f, -0.615090f, 
		-0.0299668f, 0.954811f, -0.295699f, -8.06244e-005f, 1.00000f, 1.96246e-005f, 
		-0.537614f, -0.00835144f, 0.843150f, -0.974917f, 0.00625314f, -0.222482f, 
		0.455632f, 0.00655729f, 0.890144f, -0.949248f, 0.313857f, -0.0205293f, 
		-0.548818f, -0.828554f, -0.110892f, 0.231169f, -0.314127f, -0.920807f, 
		0.949248f, -0.313857f, 0.0205293f, 0.336475f, -0.183740f, -0.923593f, 
		-0.162895f, 0.0660577f, 0.984430f, 0.118525f, 0.252356f, 0.960348f, 
		-0.0156852f, 0.0789010f, 0.996759f, 0.470584f, 0.0553594f, -0.880617f, 
		0.357153f, 0.840253f, 0.407942f, 0.414421f, 0.645511f, 0.641537f, 
		-0.765342f, -0.464037f, -0.446006f, -0.402301f, -0.653814f, -0.640844f, 
		0.0124056f, 0.276366f, 0.960972f, -0.596088f, -0.489057f, -0.636790f, 
		0.568837f, 0.762034f, 0.309402f, -0.0763506f, -0.00253474f, -0.997078f, 
		0.0763506f, 0.00253474f, 0.997078f, 0.480708f, -0.870231f, 0.107785f, 
		-0.909444f, -0.404601f, -0.0959621f, -0.462640f, 0.870138f, 0.169776f, 
		-0.973689f, -0.169866f, -0.151906f, -0.906817f, 0.375322f, 0.191874f, 
		-0.0623763f, -0.913967f, -0.400966f, -0.981627f, 0.177865f, 0.0690834f, 
		0.647353f, 0.749437f, 0.138847f, 4.71261e-005f, -0.000118851f, -1.00000f, 
		-0.999020f, 0.0442441f, -0.00158457f, 0.0785066f, 0.996913f, -0.000836292f, 
		0.611327f, -0.788841f, -0.0633194f, -0.0680918f, 0.680055f, 0.729992f, 
		0.0680918f, -0.680055f, -0.729992f, -0.644099f, 0.762590f, 0.0599500f, 
		-0.0745828f, 0.00281888f, 0.997211f, 2.28523e-006f, -1.00000f, -3.05493e-006f, 
		1.09577e-005f, 1.00000f, 1.37334e-005f, 0.543394f, -0.0117250f, 0.839396f, 
		-0.543394f, 0.0117250f, -0.839396f, 0.869920f, 0.485268f, 0.0880576f, 
		-0.869920f, -0.485268f, -0.0880576f, 4.71261e-005f, -0.000118851f, -1.00000f, 
		-0.224792f, -0.974402f, 0.00321405f, -0.998714f, -0.0506506f, -0.00236104f, 
		0.680426f, -0.732805f, -0.00423647f, 0.792442f, 0.0401360f, 0.608625f, 
		-0.962648f, -0.232542f, 0.138683f, -0.884730f, -0.198093f, -0.421915f, 
		0.855753f, 0.504006f, -0.116894f, -0.975091f, -0.154381f, -0.159259f, 
		-0.462640f, 0.870138f, 0.169776f, -0.123962f, 0.0162172f, -0.992154f, 
		8.16089e-006f, -1.00000f, 2.51535e-006f, -1.73730e-005f, 1.00000f, -2.61493e-006f, 
		2.06423e-005f, -4.94704e-005f, 1.00000f, -0.994103f, 0.108434f, 0.00145364f, 
		0.290720f, 0.956808f, 0.000742352f, -0.977819f, 0.208752f, 0.0170930f, 
		0.358816f, 0.932113f, 0.0491593f, 0.936621f, 0.168822f, 0.306986f, 
		-0.936621f, -0.168822f, -0.306986f, -0.975731f, 0.211051f, 0.0583674f, 
		0.922638f, -0.278825f, 0.266451f, -0.973803f, 0.182999f, -0.134981f, 
		0.922638f, -0.278825f, 0.266451f, -0.975731f, 0.211051f, 0.0583674f, 
		-0.950083f, 0.182234f, 0.253245f, 0.250611f, -0.0921058f, -0.963696f, 
		2.06423e-005f, -4.94704e-005f, 1.00000f, -0.208616f, -0.977992f, -0.00326297f, 
		-0.998713f, -0.0506516f, 0.00273940f, 0.680427f, -0.732805f, 0.00383993f, 
		-0.480439f, -0.873750f, 0.0757596f, -0.683156f, 0.700534f, -0.206276f, 
		0.480439f, 0.873750f, -0.0757596f, -0.115420f, 0.373301f, 0.920502f, 
		0.115420f, -0.373301f, -0.920502f, 0.656365f, -0.713764f, 0.244389f, 
		-0.0367779f, 0.0947395f, 0.994823f, -0.373824f, 0.127440f, 0.918703f, 
		-0.590075f, -0.230778f, -0.773662f, 0.384411f, 0.0138543f, -0.923058f, 
		0.690274f, -0.232345f, -0.685228f, 0.524284f, -0.139617f, -0.840020f, 
		0.107788f, 0.928892f, -0.354318f, 0.442923f, 0.0787032f, 0.893099f, 
		0.666445f, 0.142352f, 0.731838f, -0.185516f, -0.905763f, 0.381022f, 
		0.000852328f, -0.999999f, -0.00123163f, 0.0712742f, 0.641727f, -0.763614f, 
		0.976313f, -0.212421f, 0.0411201f, -0.976313f, 0.212421f, -0.0411201f, 
		-0.813640f, 0.580203f, 0.0367837f, 0.813640f, -0.580203f, -0.0367837f, 
		0.965661f, 0.155413f, -0.208198f, -0.965661f, -0.155413f, 0.208198f, 
		-0.0743374f, -0.525736f, -0.847394f, -0.807657f, -0.506809f, 0.301388f, 
		0.138083f, -0.670943f, 0.728539f, -0.273626f, 0.669013f, 0.691050f, 
		0.972007f, 0.233647f, 0.0247374f, 0.788538f, -0.611504f, -0.0653450f, 
		0.369958f, 0.695876f, -0.615538f, -0.307589f, -0.665525f, 0.680048f, 
		-0.904964f, -0.314937f, 0.286103f, 0.913033f, -0.271070f, 0.304781f, 
		-0.755704f, -0.416964f, 0.505028f, 0.0124081f, -0.675963f, 0.736831f, 
		-0.568837f, -0.762034f, -0.309402f, 0.568837f, 0.762034f, 0.309402f, 
		0.989945f, -0.141396f, -0.00407581f, -0.989945f, 0.141396f, 0.00407581f, 
		-0.846311f, 0.199637f, -0.493865f, 0.352354f, 0.236942f, -0.905376f, 
		0.910328f, -0.294530f, 0.290783f, -0.910328f, 0.294530f, -0.290783f, 
		-0.992761f, -0.112405f, 0.0423078f, 0.993945f, 0.0556352f, -0.0947537f, 
		0.250611f, -0.0921058f, -0.963696f, -0.297759f, 0.953206f, -0.0523196f, 
		-0.524792f, -0.851227f, 0.00239245f, 0.648052f, -0.752759f, -0.115679f, 
		0.935990f, -0.0292007f, -0.350813f, -0.931626f, -0.0401372f, 0.361195f, 
		-0.727747f, 0.0380311f, -0.684790f, -0.823723f, 0.244609f, -0.511514f, 
		0.823723f, -0.244609f, 0.511514f, -9.89098e-005f, -0.999999f, 0.00137457f, 
		0.0333077f, 0.549671f, 0.834717f, 0.109167f, 0.693956f, 0.711694f, 
		-0.0497176f, 0.462309f, 0.885324f, 0.107221f, -0.444856f, -0.889161f, 
		0.975383f, -0.148808f, -0.162738f, -0.893601f, -0.301684f, -0.332362f, 
		-0.341610f, 0.588483f, 0.732796f, -0.491836f, 0.555975f, 0.670066f, 
		-0.341610f, 0.588483f, 0.732796f, -0.0696496f, 0.780017f, 0.621870f, 
		0.963870f, 0.225221f, 0.142236f, 0.443056f, 0.885101f, 0.142467f, 
		0.372603f, 0.843200f, 0.387531f, -8.06244e-005f, 1.00000f, 1.96246e-005f, 
		-0.826956f, 0.00204117f, -0.562263f, 0.682865f, 0.00347549f, -0.730536f, 
		-0.000617538f, -0.999907f, 0.0136171f, 0.0536221f, -0.115673f, 0.991839f, 
		0.960236f, -0.197458f, -0.197378f, -0.960236f, 0.197458f, 0.197378f, 
		-0.854804f, 0.455808f, -0.248092f, -0.310294f, -0.299044f, -0.902380f, 
		-0.791368f, -0.419002f, 0.445168f, 0.0422547f, -0.394382f, -0.917975f, 
		-0.128866f, -0.738843f, -0.661441f, 0.313634f, -0.322290f, -0.893176f, 
		0.932824f, -0.339467f, 0.120837f, 0.791368f, 0.419002f, -0.445168f, 
		-0.0759243f, 0.340987f, 0.936997f, -0.0553109f, 0.672608f, 0.737929f, 
		-0.0312831f, 0.477609f, 0.878015f, 0.000626131f, 0.999907f, 0.0136329f, 
		-0.0531902f, 0.115704f, 0.991859f, -0.0181333f, -0.550127f, -0.834884f, 
		-0.308086f, 0.787175f, -0.534265f, -0.700575f, -0.0384875f, -0.712540f, 
		-0.860174f, -0.163438f, 0.483103f, 0.463472f, -0.762650f, 0.451175f, 
		-0.0962828f, -0.826346f, -0.554871f, 0.860174f, 0.163438f, -0.483103f, 
		0.299959f, 0.851431f, 0.430220f, 0.0110836f, 0.869250f, 0.494249f, 
		-0.0188711f, -0.850323f, 0.525923f, -0.164611f, 0.699412f, -0.695504f, 
		-0.847494f, 0.198224f, 0.492403f, 0.938074f, 0.344910f, 0.0324612f, 
		0.00169077f, 0.999991f, -0.00377478f, 0.0313248f, -0.413701f, -0.909874f, 
		0.0268085f, 0.673533f, 0.738670f, -0.111173f, -0.178341f, -0.977668f, 
		-0.694480f, -0.609037f, 0.383108f, 0.654039f, -0.420589f, 0.628759f, 
		-0.646434f, 0.688387f, 0.329008f, 0.911194f, 0.398622f, -0.104045f, 
		0.992719f, 0.120408f, 0.00322846f, -0.981944f, -0.124000f, -0.142865f, 
		0.0124056f, 0.276366f, 0.960972f, 0.992719f, 0.120408f, 0.00322846f, 
		-0.981944f, -0.124000f, -0.142865f, -0.0155597f, -0.558640f, 0.829264f, 
		-0.0284775f, 0.333201f, 0.942426f, -0.419377f, 0.0119966f, 0.907733f, 
		0.324797f, 0.317578f, 0.890871f, 0.114307f, -0.869721f, 0.480125f, 
		-0.0936506f, 0.128011f, 0.987341f, 0.114080f, -0.869535f, 0.480515f, 
		-0.0135564f, -0.696217f, 0.717703f, 0.989945f, -0.141396f, -0.00407581f, 
		-0.989945f, 0.141396f, 0.00407581f, -0.370303f, 0.844543f, 0.386812f, 
		-0.486417f, -0.754526f, 0.440555f, -0.762499f, -0.273026f, -0.586559f, 
		0.828573f, -0.352760f, -0.434773f, 0.900056f, -0.0563373f, 0.432118f, 
		0.293332f, 0.783518f, -0.547773f, -0.234569f, -0.199030f, -0.951506f, 
		-0.691027f, -0.675549f, 0.257129f, -0.213180f, -0.934121f, -0.286308f, 
		0.573091f, -0.600281f, -0.557879f, 0.664282f, -0.708641f, 0.237817f, 
		-1.96055e-005f, 1.00000f, 8.06487e-005f, 0.730244f, 0.00465713f, -0.683171f, 
		-0.692798f, -8.04640e-005f, -0.721131f, -0.00221107f, 0.999974f, 0.00691843f, 
		-0.0334510f, -0.700738f, 0.712634f, 0.00486270f, -0.607930f, 0.793976f, 
		0.0855969f, -0.472545f, 0.877140f, 0.908847f, 0.252242f, 0.332221f, 
		-0.908847f, -0.252242f, -0.332221f, -0.0498699f, 0.143085f, -0.988453f, 
		0.000537542f, -0.999942f, -0.0107437f, -0.000541995f, -0.999942f, -0.0107430f, 
		-0.0494492f, -0.143397f, 0.988429f, 0.00136873f, 0.262194f, -0.965014f, 
		0.00178453f, -0.262685f, 0.964880f, 0.00218523f, 0.469571f, -0.882892f, 
		0.000335269f, -0.00103769f, 0.999999f, -8.84706e-005f, 0.00104288f, -0.999999f, 
		0.000335269f, -0.00103769f, 0.999999f, 0.0697575f, -0.779630f, 0.622343f, 
		0.0694965f, 0.779627f, -0.622376f, -8.84706e-005f, 0.00104288f, -0.999999f, 
		-0.178757f, 0.771197f, -0.610983f, -0.729886f, -0.550699f, -0.404965f, 
		0.925307f, -0.125054f, -0.358005f, -0.000154691f, -0.271951f, -0.962311f, 
		0.000255834f, 0.271950f, 0.962311f, -0.000154691f, -0.271951f, -0.962311f, 
		0.000255834f, 0.271950f, 0.962311f, -0.0335444f, 0.974732f, -0.220846f, 
		-0.124221f, 0.873200f, -0.471265f, -0.124013f, -0.873192f, 0.471335f, 
		-0.0334455f, -0.974732f, 0.220860f, 0.0683841f, -0.926555f, 0.369892f, 
		0.125010f, -0.986535f, 0.105454f, 0.124957f, 0.986538f, -0.105492f, 
		0.0682240f, 0.926552f, -0.369929f, 0.00256507f, -0.470217f, 0.882547f, 
		-0.0479796f, 0.853575f, 0.518757f, -0.0300353f, 0.925946f, 0.376459f, 
		-0.0301907f, -0.925946f, -0.376446f, -0.0481971f, -0.853572f, -0.518741f, 
		0.0300353f, -0.925946f, -0.376459f, 0.0479796f, -0.853575f, -0.518757f, 
		-9.52478e-005f, -0.889824f, -0.456303f, 0.0481971f, 0.853572f, 0.518741f, 
		9.52425e-005f, 0.889824f, 0.456303f, 0.0301907f, 0.925946f, 0.376446f, 
		-0.0379275f, -0.876579f, 0.479761f, -0.0279280f, -0.948316f, 0.316097f, 
		5.51292e-006f, -0.999830f, 0.0184198f, -5.51292e-006f, 0.999830f, -0.0184198f, 
		-0.0280618f, 0.948317f, -0.316081f, -0.0381280f, 0.876582f, -0.479740f, 
		-0.0383816f, -0.461440f, -0.886341f, -0.0329266f, 0.404024f, 0.914156f, 
		0.0631520f, -0.272849f, 0.959982f, 0.0511958f, -0.747098f, 0.662739f, 
		0.0316984f, -0.500236f, 0.865309f, 0.0484007f, 0.232370f, -0.971422f, 
		0.0273957f, 0.436490f, -0.899292f, 0.0509281f, 0.747087f, -0.662773f, 
		-0.0722855f, 0.867643f, 0.491904f, -0.118740f, 0.961354f, 0.248395f, 
		-0.118849f, -0.961347f, -0.248367f, -0.0724876f, -0.867645f, -0.491872f, 
		0.0239042f, -0.933887f, -0.356767f, 0.117555f, -0.804561f, -0.582119f, 
		0.117800f, 0.804576f, 0.582049f, 0.0240531f, 0.933887f, 0.356759f, 
		-0.290446f, 0.930155f, 0.224617f, -0.151558f, 0.970784f, 0.186032f, 
		-0.0756911f, 0.994712f, 0.0694228f, -0.00831162f, -0.994707f, 0.102416f, 
		0.0331896f, -0.999169f, -0.0236444f, 0.290446f, -0.930155f, -0.224617f, 
		-0.0322304f, 0.996261f, -0.0801636f, 0.0782416f, 0.984835f, -0.154848f, 
		-0.0618676f, -0.977952f, 0.199454f, 0.0177687f, -0.999762f, -0.0126262f, 
		0.0515074f, -0.990393f, -0.128331f, 0.00177137f, 0.994681f, 0.102989f, 
		0.0333734f, 0.998916f, -0.0324493f, 0.0618676f, 0.977952f, -0.199454f, 
		-0.00466506f, -0.163133f, 0.986593f, 0.0201740f, 0.234918f, -0.971806f, 
		-0.00216379f, -0.191810f, -0.981430f, -0.0201707f, 0.234907f, 0.971809f, 
		-0.0918932f, 0.971912f, 0.216664f, -0.147293f, -0.251421f, -0.956605f, 
		0.0163846f, -0.977509f, -0.210254f, 0.182188f, 0.717503f, 0.672307f, 
		0.142035f, 0.525250f, 0.839010f, -0.0168856f, 0.926803f, 0.375168f, 
		0.00316327f, -0.634053f, 0.773283f, 0.00283942f, 0.633449f, -0.773779f, 
		-0.000725121f, -0.503137f, -0.864207f, -0.0155070f, 0.565267f, 0.824762f, 
		0.0155125f, 0.565242f, -0.824779f, 0.000725978f, -0.503118f, 0.864217f, 
		-0.185813f, -0.609836f, -0.770438f, -0.0153138f, -0.420435f, -0.907193f, 
		-0.00600798f, -0.241447f, -0.970395f, -0.00103006f, -0.836802f, -0.547505f, 
		-0.00489860f, 0.870278f, 0.492537f, 0.00490273f, 0.870272f, -0.492547f, 
		0.00103051f, -0.836794f, 0.547517f, 0.0338784f, -0.988546f, -0.147071f, 
		0.000613988f, 0.994285f, 0.106759f, -0.0338860f, -0.988536f, 0.147132f, 
		-0.000610131f, 0.994277f, -0.106832f, -0.0496630f, -0.969118f, 0.241544f, 
		-0.0497490f, 0.969119f, -0.241525f, -0.0889366f, 0.996037f, 1.87488e-005f, 
		-0.000200669f, -4.86687e-014f, -1.00000f, 0.0496650f, -0.969123f, -0.241524f, 
		0.0497606f, 0.969125f, 0.241496f, 0.0889785f, -0.996034f, -1.87576e-005f, 
		-0.000215972f, -3.88223e-005f, -1.00000f, 0.989897f, 0.141582f, 0.00761918f, 
		-0.989897f, -0.141582f, -0.00761918f, 0.989892f, 0.141615f, 0.00761449f, 
		-0.989892f, -0.141615f, -0.00761449f, -0.0326607f, -0.564723f, -0.824634f, 
		-0.0232449f, 0.599022f, 0.800395f, 0.0232469f, 0.599035f, -0.800385f, 
		0.0341266f, -0.545478f, 0.837430f, -0.0155550f, -0.841581f, -0.539907f, 
		-0.0153346f, 0.841559f, 0.539948f, 0.0155430f, -0.841633f, 0.539827f, 
		0.0153242f, 0.841618f, -0.539855f, 0.960236f, -0.197458f, -0.197378f, 
		-0.960236f, 0.197458f, 0.197378f, 0.00364346f, 0.770218f, -0.637770f, 
		0.00390909f, -0.770671f, 0.637221f, 0.0100244f, -0.884883f, 0.465706f, 
		0.00771385f, 0.885247f, -0.465057f, -0.00237824f, -0.227408f, -0.973797f, 
		-0.00198725f, 0.227424f, 0.973794f, -0.00237824f, -0.227408f, -0.973797f, 
		-0.000408664f, -0.0280705f, -0.999606f, -9.10792e-006f, 0.0280739f, 0.999606f, 
		-0.000408664f, -0.0280705f, -0.999606f, 0.000376100f, 0.117390f, -0.993086f, 
		0.000778549f, -0.117396f, 0.993085f, 0.000376100f, 0.117390f, -0.993086f, 
		0.000944228f, 0.232669f, -0.972555f, 0.00134091f, -0.232686f, 0.972551f, 
		0.000944228f, 0.232669f, -0.972555f, 0.00148163f, 0.343633f, -0.939103f, 
		0.00185153f, -0.343578f, 0.939122f, 0.00148163f, 0.343633f, -0.939103f, 
		-0.00434347f, -0.439842f, -0.898065f, -0.00397108f, 0.439813f, 0.898081f, 
		-0.00434347f, -0.439842f, -0.898065f, 0.00197242f, 0.448656f, -0.893702f, 
		0.00234659f, -0.448705f, 0.893677f, 0.00197242f, 0.448656f, -0.893702f, 
		0.00245070f, 0.546875f, -0.837211f, 0.00280537f, -0.546848f, 0.837227f, 
		0.00245070f, 0.546875f, -0.837211f, 0.0246505f, 0.487519f, -0.872764f, 
		-0.00604004f, -0.626003f, -0.779797f, -0.00573581f, 0.626017f, 0.779788f, 
		-0.00604004f, -0.626003f, -0.779797f, 0.00305324f, 0.639484f, -0.768799f, 
		0.00350467f, -0.638829f, 0.769341f, 0.00305324f, 0.639484f, -0.768799f, 
		0.0273554f, 0.662194f, -0.748833f, 0.0273554f, 0.662194f, -0.748833f, 
		-0.00100537f, -0.667470f, 0.744636f, 0.0273554f, 0.662194f, -0.748833f, 
		-0.683695f, 0.724275f, -0.0893744f, 0.242509f, 0.259584f, -0.934776f, 
		0.654944f, 0.642109f, 0.398427f, 0.138460f, 0.958368f, -0.249717f, 
		-0.00756282f, -0.780339f, -0.625311f, -0.00732641f, 0.780334f, 0.625320f, 
		-0.00756282f, -0.780339f, -0.625311f, -0.000464955f, 0.540333f, -0.841451f, 
		-0.000464955f, 0.540333f, -0.841451f, -0.00436654f, -0.568758f, 0.822493f, 
		-0.000464955f, 0.540333f, -0.841451f, 0.000508176f, 0.0953209f, -0.995446f, 
		0.000508176f, 0.0953209f, -0.995446f, -0.0396778f, -0.114565f, 0.992623f, 
		-0.0396778f, -0.114565f, 0.992623f, 0.000508176f, 0.0953209f, -0.995446f, 
		0.00188474f, 0.139630f, -0.990202f, 0.00188474f, 0.139630f, -0.990202f, 
		0.0392573f, -0.167743f, 0.985049f, 0.00188474f, 0.139630f, -0.990202f, 
		0.00188474f, 0.139630f, -0.990202f, 0.000543714f, 0.331346f, -0.943509f, 
		0.000543714f, 0.331346f, -0.943509f, -0.00430553f, -0.371197f, 0.928544f, 
		0.000543714f, 0.331346f, -0.943509f, 0.0227914f, 0.485603f, -0.873882f, 
		0.0227914f, 0.485603f, -0.873882f, 0.0238607f, -0.515023f, 0.856844f, 
		0.0227914f, 0.485603f, -0.873882f, 0.0227914f, 0.485603f, -0.873882f, 
		-0.394897f, -0.892958f, 0.216063f, -0.0463106f, -0.313423f, 0.948484f, 
		-0.367562f, -0.756105f, 0.541482f, -0.479649f, 0.485618f, -0.730830f, 
		-0.263166f, 0.370171f, -0.890908f, 0.394897f, 0.892958f, -0.216063f, 
		-0.00817035f, 0.165997f, -0.986093f, 0.367562f, 0.756105f, -0.541482f, 
		0.132472f, 0.331804f, -0.934001f, 0.0755090f, -0.181565f, 0.980476f, 
		0.190849f, -0.279861f, 0.940880f, 0.483550f, -0.456346f, 0.746945f, 
		-0.0252163f, -0.891041f, -0.453222f, -0.0173411f, 0.902112f, 0.431153f, 
		-0.0252163f, -0.891041f, -0.453222f, 0.00747506f, 0.991298f, 0.131428f, 
		0.0243265f, -0.424546f, 0.905080f, -0.00747506f, -0.991298f, -0.131428f, 
		0.00747506f, 0.991298f, 0.131428f, -0.00747506f, -0.991298f, -0.131428f, 
		-0.0243265f, 0.424545f, -0.905080f, 9.10910e-007f, -1.00000f, -4.04544e-006f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 9.10910e-007f, -1.00000f, -4.04544e-006f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 9.10910e-007f, -1.00000f, -4.04544e-006f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 9.10910e-007f, -1.00000f, -4.04544e-006f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		9.10910e-007f, -1.00000f, -4.04544e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		9.10910e-007f, -1.00000f, -4.04544e-006f, -0.00596389f, 0.999982f, -0.000737357f, 
		0.222609f, 0.00810996f, 0.974874f, -0.781191f, 0.102550f, 0.615812f, 
		0.944863f, 0.0762480f, 0.318465f, -0.0196614f, -0.807578f, -0.589432f, 
		-0.0196614f, -0.807578f, -0.589432f, -0.0196614f, -0.807578f, -0.589432f, 
		-0.0298873f, 0.849658f, 0.526486f, -0.00383542f, -0.540687f, -0.841215f, 
		-0.00383542f, -0.540687f, -0.841215f, -0.00383542f, -0.540687f, -0.841215f, 
		-0.00961303f, 0.527297f, 0.849627f, 0.0252346f, -0.891048f, 0.453207f, 
		0.0173176f, 0.902111f, -0.431156f, 0.0173176f, 0.902111f, -0.431156f, 
		-0.00276015f, -0.169126f, -0.985591f, -0.00276015f, -0.169126f, -0.985591f, 
		-0.00276015f, -0.169126f, -0.985591f, -0.00219531f, 0.169184f, 0.985582f, 
		-0.000760025f, -0.0953664f, -0.995442f, -0.000760025f, -0.0953664f, -0.995442f, 
		-0.000760025f, -0.0953664f, -0.995442f, -0.000760025f, -0.0953664f, -0.995442f, 
		0.0400866f, 0.114545f, 0.992609f, -0.000760025f, -0.0953664f, -0.995442f, 
		-0.000952674f, -0.331400f, -0.943490f, -0.000952674f, -0.331400f, -0.943490f, 
		-0.000952674f, -0.331400f, -0.943490f, 0.00469732f, 0.371274f, 0.928511f, 
		-0.000952674f, -0.331400f, -0.943490f, 0.000101574f, -0.540330f, -0.841453f, 
		0.000101574f, -0.540330f, -0.841453f, 0.000101574f, -0.540330f, -0.841453f, 
		0.00472577f, 0.568734f, 0.822508f, 0.000101574f, -0.540330f, -0.841453f, 
		0.00757481f, -0.780311f, 0.625346f, 0.00730783f, 0.780321f, -0.625336f, 
		0.00730783f, 0.780321f, -0.625336f, -0.0276855f, -0.662153f, -0.748857f, 
		-0.0276855f, -0.662153f, -0.748857f, -0.0276855f, -0.662153f, -0.748857f, 
		0.00131650f, 0.667452f, 0.744651f, -0.0276855f, -0.662153f, -0.748857f, 
		0.00605399f, -0.625990f, 0.779807f, 0.00571242f, 0.625972f, -0.779824f, 
		0.00571242f, 0.625972f, -0.779824f, -0.00279360f, -0.546877f, -0.837208f, 
		-0.00245399f, 0.546877f, 0.837209f, -0.00279360f, -0.546877f, -0.837208f, 
		-0.00339998f, -0.639536f, -0.768754f, -0.00317421f, 0.638834f, 0.769338f, 
		-0.00339998f, -0.639536f, -0.768754f, 0.00435493f, -0.439844f, 0.898064f, 
		0.00397147f, 0.439878f, -0.898049f, 0.00397147f, 0.439878f, -0.898049f, 
		-0.00186717f, -0.343618f, -0.939108f, -0.00146325f, 0.343612f, 0.939110f, 
		-0.00186717f, -0.343618f, -0.939108f, -0.00233387f, -0.448679f, -0.893690f, 
		-0.00197672f, 0.448717f, 0.893672f, -0.00233387f, -0.448679f, -0.893690f, 
		0.00195732f, 0.227400f, -0.973800f, 0.00240931f, -0.227433f, 0.973791f, 
		0.00195732f, 0.227400f, -0.973800f, -2.48673e-005f, 0.0280466f, -0.999607f, 
		0.000426856f, -0.0280290f, 0.999607f, -2.48673e-005f, 0.0280466f, -0.999607f, 
		-0.000797049f, -0.117318f, -0.993094f, -0.000344384f, 0.117319f, 0.993094f, 
		-0.000797049f, -0.117318f, -0.993094f, -0.00135785f, -0.232623f, -0.972566f, 
		-0.000911476f, 0.232594f, 0.972573f, -0.00135785f, -0.232623f, -0.972566f, 
		0.00388569f, 0.333203f, 0.942847f, 0.00315930f, -0.999952f, -0.00928754f, 
		0.00315930f, -0.999952f, -0.00928754f, 0.00315930f, -0.999952f, -0.00928754f, 
		0.00315930f, -0.999952f, -0.00928754f, 0.00315930f, -0.999952f, -0.00928754f, 
		0.00315930f, -0.999952f, -0.00928754f, 0.00315930f, -0.999952f, -0.00928754f, 
		0.00315930f, -0.999952f, -0.00928754f, 0.00315930f, -0.999952f, -0.00928754f, 
		0.190991f, -0.200498f, 0.960897f, -0.100297f, -0.957866f, 0.269134f, 
		-0.0118796f, 0.920494f, -0.390575f, -0.813640f, 0.580203f, 0.0367837f, 
		0.813640f, -0.580203f, -0.0367837f, 0.00839179f, 0.350251f, -0.936618f, 
		-0.241547f, 0.475881f, 0.845690f, -0.607542f, 0.450012f, -0.654509f, 
		0.856092f, 0.462180f, -0.231293f, -0.00146263f, -0.00802184f, 0.999967f, 
		-0.0847837f, -0.868245f, -0.488838f, 0.101190f, -0.948995f, -0.298611f, 
		-0.00786744f, -0.902465f, -0.430691f, -0.00146263f, -0.00802184f, 0.999967f, 
		-0.0149245f, 0.906792f, 0.421314f, 0.798204f, -0.597189f, -0.0789723f, 
		-0.798204f, 0.597189f, 0.0789723f, 0.999670f, -0.0255803f, 0.00214195f, 
		-0.999670f, 0.0255803f, -0.00214195f, -0.00348730f, -0.333206f, 0.942848f, 
		-0.000811550f, 0.999880f, 0.0154628f, 0.0503804f, 0.000000f, 0.998730f, 
		-0.0499732f, 0.000000f, 0.998751f, -4.98821e-006f, -1.00000f, 1.95894e-006f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		0.00308034f, 0.359617f, 0.933095f, 0.0640370f, 0.163507f, 0.984462f, 
		0.0640370f, 0.163507f, 0.984462f, -0.118724f, -0.934237f, 0.336313f, 
		0.0166676f, 0.938548f, -0.344745f, 0.00581228f, 0.670127f, 0.742224f, 
		-0.000710032f, 0.999908f, 0.0135238f, 0.0503942f, 0.000000f, 0.998729f, 
		-0.0499732f, 0.000000f, 0.998751f, 0.000717478f, -0.999908f, 0.0135144f, 
		0.641396f, -0.764682f, -0.0622370f, -0.997106f, -0.0756447f, -0.00763954f, 
		0.997106f, 0.0756447f, 0.00763954f, 0.0121170f, -0.775138f, 0.631675f, 
		0.0118779f, 0.775122f, -0.631701f, -0.0120983f, -0.775203f, -0.631597f, 
		-0.0118597f, 0.775191f, 0.631615f, -0.0263840f, -0.406366f, -0.913329f, 
		-0.0223709f, 0.362474f, 0.931725f, 0.0263802f, -0.406353f, 0.913335f, 
		0.0223663f, 0.362466f, -0.931728f, -0.0706777f, -0.979151f, 0.190441f, 
		0.00184450f, -0.997002f, 0.0773562f, -0.0296408f, -0.991348f, 0.127872f, 
		-0.0395761f, 0.991009f, -0.127810f, -0.0386858f, -0.329163f, 0.943480f, 
		-0.00633432f, 0.304203f, -0.952586f, -0.0131522f, -0.751920f, 0.659123f, 
		-0.00503948f, 0.685973f, -0.727610f, -0.000235060f, -0.0993347f, -0.995054f, 
		0.00178317f, 0.0914610f, 0.995807f, 0.0104135f, -0.511428f, -0.859263f, 
		0.0107804f, 0.511429f, 0.859258f, 0.0326003f, 0.824625f, 0.564740f, 
		0.0232816f, -0.800380f, -0.599041f, -0.0709935f, -0.478691f, 0.875109f, 
		-0.119389f, -0.242892f, 0.962678f, -0.119810f, 0.242896f, -0.962625f, 
		-0.0713729f, 0.478708f, -0.875068f, 0.0256043f, 0.347669f, -0.937268f, 
		0.118372f, 0.566580f, -0.815460f, 0.118725f, -0.566550f, 0.815429f, 
		0.0260092f, -0.347651f, 0.937263f, -0.0322694f, 0.226855f, 0.973394f, 
		-0.123493f, 0.481982f, 0.867435f, -0.123855f, -0.481971f, -0.867389f, 
		-0.0326822f, -0.226845f, -0.973382f, 0.0688484f, -0.378862f, -0.922889f, 
		0.124396f, -0.109018f, -0.986225f, 0.124822f, 0.109023f, 0.986171f, 
		0.0692392f, 0.378880f, 0.922852f, 0.0330292f, -0.985570f, -0.166013f, 
		0.0353770f, 0.985494f, 0.165981f, -0.0638914f, 0.697665f, -0.713569f, 
		0.000000f, -1.00000f, -1.28608e-019f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, -1.00000f, -1.28608e-019f, -5.79053e-006f, 1.00000f, 1.62417e-006f, 
		-0.0635989f, -0.697697f, 0.713564f, -0.0526206f, -0.962915f, -0.264626f, 
		0.0455327f, -0.766941f, -0.640101f, 0.0686131f, -0.634441f, -0.769920f, 
		0.00176594f, -0.996876f, -0.0789642f, 0.0689192f, 0.634420f, 0.769910f, 
		-0.00326225f, 0.999150f, 0.0410997f, -0.00870410f, -0.659522f, -0.751634f, 
		-0.00838013f, 0.659565f, 0.751601f, -0.0102391f, -0.446403f, -0.894774f, 
		-0.00987654f, 0.446346f, 0.894806f, -0.0102808f, -0.218576f, -0.975766f, 
		-0.00987924f, 0.218576f, 0.975770f, -0.00488795f, -0.0544075f, -0.998507f, 
		-0.00537258f, 0.0638694f, 0.997944f, 0.00527865f, 0.0650637f, -0.997867f, 
		-0.00452231f, -0.0810667f, 0.996698f, 0.00987810f, 0.218575f, -0.975770f, 
		0.0102797f, -0.218575f, 0.975766f, 0.00987659f, 0.446344f, -0.894807f, 
		0.0102392f, -0.446401f, 0.894774f, 0.00838236f, 0.659561f, -0.751604f, 
		0.00870633f, -0.659519f, 0.751638f, 0.0126414f, 0.789238f, -0.613958f, 
		-5.57887e-005f, 0.493637f, -0.869668f, 0.00543933f, 0.623479f, -0.781821f, 
		-0.0126414f, -0.789238f, 0.613958f, -0.00543934f, -0.623479f, 0.781821f, 
		5.57848e-005f, -0.493637f, 0.869668f, -0.00331384f, -0.933991f, -0.357281f, 
		-0.00319123f, 0.933963f, 0.357355f, -0.00603759f, -0.827125f, -0.561985f, 
		-0.00582201f, 0.827128f, 0.561984f, 0.00582273f, 0.827124f, -0.561989f, 
		0.00605024f, -0.827147f, 0.561953f, 0.00317239f, 0.933979f, -0.357314f, 
		0.00332480f, -0.933975f, 0.357322f, -0.00128286f, -0.986075f, -0.166298f, 
		-0.00121426f, 0.986083f, 0.166251f, 0.00120903f, 0.986077f, -0.166285f, 
		0.00128198f, -0.986076f, 0.166289f, 0.0450059f, -0.437811f, 0.897940f, 
		0.0280290f, -0.212501f, 0.976759f, 0.00163311f, -0.100108f, 0.994975f, 
		-5.87835e-007f, -1.00000f, -6.20707e-006f, -5.72366e-007f, 1.00000f, 6.20707e-006f, 
		5.72365e-007f, 1.00000f, -6.20578e-006f, 5.82678e-007f, -1.00000f, 6.20449e-006f, 
		0.766874f, -0.639261f, 0.0569981f, -0.820735f, 0.558158f, -0.121880f, 
		-0.981692f, 0.112653f, -0.153588f, 0.981692f, -0.112653f, 0.153588f, 
		0.875753f, -0.480486f, 0.0467995f, -0.929800f, 0.357658f, -0.0869015f, 
		0.998955f, -0.00970502f, -0.0446558f, -0.998955f, 0.00970502f, 0.0446558f, 
		0.00128722f, -0.986079f, 0.166273f, 0.00120992f, 0.986072f, -0.166312f, 
		-0.00120905f, 0.986074f, 0.166302f, -0.00128286f, -0.986075f, -0.166298f, 
		0.00334555f, -0.933951f, 0.357385f, 0.00316337f, 0.933986f, -0.357295f, 
		0.00605024f, -0.827147f, 0.561953f, 0.00581006f, 0.827102f, -0.562021f, 
		-0.00582201f, 0.827128f, 0.561984f, -0.00605024f, -0.827147f, -0.561953f, 
		-0.00317430f, 0.933970f, 0.357336f, -0.00332670f, -0.933967f, -0.357344f, 
		0.00870633f, -0.659519f, 0.751638f, 0.00838236f, 0.659561f, -0.751604f, 
		0.0102392f, -0.446401f, 0.894774f, 0.00987659f, 0.446344f, -0.894807f, 
		0.0102797f, -0.218575f, 0.975766f, 0.00987427f, 0.218635f, -0.975757f, 
		-0.00420862f, -0.0812136f, 0.996688f, 0.00446641f, 0.0544348f, -0.998507f, 
		0.00352309f, 0.0841899f, 0.996444f, -0.00487951f, -0.0544669f, -0.998504f, 
		-0.00987314f, 0.218634f, 0.975757f, -0.0102808f, -0.218576f, -0.975766f, 
		-0.00987654f, 0.446346f, 0.894806f, -0.0102358f, -0.446393f, -0.894779f, 
		-0.00838735f, 0.659555f, 0.751610f, -0.00870410f, -0.659522f, -0.751634f, 
		-0.0621051f, -0.199475f, -0.977933f, 0.0175707f, 0.0126110f, -0.999766f, 
		0.0513351f, 0.128315f, -0.990404f, 0.00195059f, -0.102975f, 0.994682f, 
		0.0335815f, 0.0324633f, 0.998909f, 0.0621051f, 0.199475f, 0.977933f, 
		-0.147164f, 0.956690f, -0.251170f, 0.0162414f, 0.210249f, -0.977513f, 
		-0.00180942f, -0.459143f, 0.888360f, 0.132292f, -0.756325f, 0.640680f, 
		-0.0123135f, -0.332082f, 0.943170f, 0.000000f, -1.00000f, -1.28608e-019f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, -1.00000f, -1.28608e-019f, 0.0694798f, -0.622351f, -0.779648f, 
		0.0697934f, 0.622326f, 0.779640f, -5.79053e-006f, 1.00000f, 1.62417e-006f, 
		-0.0383188f, 0.886354f, -0.461421f, -0.0337138f, -0.911476f, 0.409969f, 
		-0.00341335f, -0.997183f, -0.0749250f, 0.0282933f, -0.991652f, -0.125797f, 
		0.0693210f, -0.979673f, -0.188245f, 0.0409135f, 0.991222f, 0.125721f, 
		-0.0332657f, 0.220865f, 0.974737f, -0.123920f, 0.471317f, 0.873215f, 
		-0.124289f, -0.471304f, -0.873170f, -0.0336806f, -0.220854f, -0.974725f, 
		0.0681273f, -0.369888f, -0.926575f, 0.124790f, -0.105504f, -0.986558f, 
		0.125217f, 0.105508f, 0.986503f, 0.0685232f, 0.369909f, 0.926538f, 
		-0.0722142f, -0.491908f, 0.867647f, -0.118593f, -0.248380f, 0.961376f, 
		-0.119013f, 0.248385f, -0.961322f, -0.0725911f, 0.491926f, -0.867606f, 
		0.0237838f, 0.356760f, -0.933893f, 0.117523f, 0.582109f, -0.804573f, 
		0.117873f, -0.582079f, 0.804544f, 0.0241878f, -0.356740f, 0.933890f, 
		0.0178133f, 0.972231f, 0.233343f, 0.0137767f, -0.980765f, -0.194705f, 
		-0.0325943f, 0.824606f, -0.564768f, -0.0232800f, -0.800364f, 0.599063f, 
		0.0161897f, -0.838620f, -0.544477f, 0.0164294f, 0.838635f, 0.544446f, 
		-0.00195713f, 0.115241f, -0.993336f, -0.00155829f, -0.115258f, 0.993334f, 
		-0.0107738f, 0.511403f, -0.859274f, -0.0104077f, -0.511398f, 0.859281f, 
		0.00339451f, 0.228983f, 0.973425f, 0.00300274f, -0.229001f, -0.973422f, 
		0.00919795f, 0.553387f, 0.832873f, 0.00884668f, -0.553382f, -0.832880f, 
		-0.0226961f, -0.360662f, -0.932420f, -0.0260046f, 0.406418f, 0.913317f, 
		0.00591066f, -0.359523f, 0.933118f, 0.0200827f, 0.393426f, -0.919137f, 
		-0.0372145f, -0.990248f, 0.134255f, 0.00308813f, 0.996363f, -0.0851567f, 
		-0.0121132f, -0.775165f, -0.631643f, -0.0118754f, 0.775135f, 0.631684f, 
		0.00508341f, -0.747693f, 0.664025f, 0.00682113f, 0.807062f, -0.590427f, 
		-0.231286f, -0.966337f, 0.112694f, -0.131652f, -0.280512f, -0.950779f, 
		-0.604208f, 0.702311f, 0.376420f, -0.510383f, 0.597908f, 0.618074f, 
		0.231286f, 0.966337f, -0.112694f, 0.0503523f, 0.378031f, 0.924423f, 
		0.182161f, 0.286005f, 0.940754f, 0.568299f, -0.698362f, -0.435117f, 
		-0.402041f, -0.653649f, -0.641176f, -0.260261f, -0.686161f, -0.679299f, 
		-0.402041f, -0.653649f, -0.641176f, 0.312361f, -0.521296f, -0.794154f, 
		0.594295f, -0.365614f, -0.716337f, 0.594295f, -0.365614f, -0.716337f, 
		-0.0137086f, 0.993475f, -0.113227f, -0.0499732f, 0.000000f, 0.998751f, 
		0.000718668f, -0.999908f, 0.0135405f, -0.000712822f, 0.999908f, 0.0135406f, 
		0.0503942f, 0.000000f, 0.998729f, -0.0344466f, -0.246711f, -0.968477f, 
		-0.0234281f, -0.0111996f, -0.999663f, 0.00144348f, 0.134489f, -0.990914f, 
		-0.00144348f, -0.134489f, 0.990914f, 0.0234281f, 0.0111996f, 0.999663f, 
		0.0344466f, 0.246711f, 0.968477f, -0.00550046f, -0.670163f, 0.742193f, 
		-0.00269785f, -0.359656f, 0.933081f, 0.000818174f, -0.999880f, 0.0154623f, 
		-0.0499623f, 0.000000f, 0.998751f, -0.0499732f, 0.000000f, 0.998751f, 
		-4.98821e-006f, -1.00000f, 1.95894e-006f, 3.58729e-006f, 1.00000f, -9.35137e-007f, 
		0.0503942f, 0.000000f, 0.998729f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		-0.00347509f, -0.333232f, 0.942838f, -0.0657525f, -0.159664f, 0.984979f, 
		-0.0657525f, -0.159664f, 0.984979f, -0.0295642f, -0.897987f, -0.439028f, 
		0.107142f, 0.929409f, 0.353157f, -0.0680918f, 0.680055f, 0.729992f, 
		0.0680918f, -0.680055f, -0.729992f, 0.0261807f, 0.473154f, 0.880591f, 
		-0.000147517f, -0.999999f, 0.00166210f, -0.000147517f, -0.999999f, 0.00166210f, 
		-0.000147517f, -0.999999f, 0.00166210f, -0.000147517f, -0.999999f, 0.00166210f, 
		-0.000147517f, -0.999999f, 0.00166210f, -0.000147517f, -0.999999f, 0.00166210f, 
		-0.000147517f, -0.999999f, 0.00166210f, -0.000147517f, -0.999999f, 0.00166210f, 
		0.00387479f, 0.333228f, 0.942838f, 0.00238438f, 0.227409f, 0.973796f, 
		0.00198175f, -0.227425f, -0.973794f, 0.00238438f, 0.227409f, 0.973796f, 
		-8.12741e-006f, -0.0280815f, -0.999606f, 0.000402704f, 0.0280289f, 0.999607f, 
		0.000402704f, 0.0280289f, 0.999607f, -0.000802227f, 0.117431f, -0.993081f, 
		-0.000356809f, -0.117347f, 0.993091f, -0.000356809f, -0.117347f, 0.993091f, 
		-0.00134132f, 0.232583f, -0.972576f, -0.000923839f, -0.232634f, 0.972564f, 
		-0.000923839f, -0.232634f, 0.972564f, 0.00436574f, 0.439867f, 0.898053f, 
		0.00396634f, -0.439844f, -0.898065f, 0.00436574f, 0.439867f, 0.898053f, 
		-0.00185937f, 0.343551f, -0.939132f, -0.00146567f, -0.343514f, 0.939147f, 
		-0.00146567f, -0.343514f, 0.939147f, -0.00198492f, -0.448775f, 0.893643f, 
		-0.00234203f, 0.448736f, -0.893661f, -0.00198492f, -0.448775f, 0.893643f, 
		0.00606333f, 0.625977f, 0.779818f, 0.00570587f, -0.625983f, -0.779816f, 
		0.00606333f, 0.625977f, 0.779818f, -0.00246229f, -0.546928f, 0.837176f, 
		-0.00246229f, -0.546928f, 0.837176f, -0.00280190f, 0.546928f, -0.837175f, 
		-0.00316809f, -0.638796f, 0.769370f, -0.00316809f, -0.638796f, 0.769370f, 
		-0.00339198f, 0.639497f, -0.768786f, -0.0276813f, 0.662145f, -0.748864f, 
		0.00131890f, -0.667442f, 0.744661f, 0.00131890f, -0.667442f, 0.744661f, 
		0.00131890f, -0.667442f, 0.744661f, 0.00759792f, 0.780334f, 0.625317f, 
		0.00730025f, -0.780330f, -0.625326f, 0.00759792f, 0.780334f, 0.625317f, 
		0.000104543f, 0.540322f, -0.841458f, 0.00472335f, -0.568729f, 0.822512f, 
		0.00472335f, -0.568729f, 0.822512f, 0.00472335f, -0.568729f, 0.822512f, 
		-0.000948886f, 0.331368f, -0.943501f, 0.00469964f, -0.371242f, 0.928524f, 
		0.00469964f, -0.371242f, 0.928524f, 0.00469964f, -0.371242f, 0.928524f, 
		-0.000760914f, 0.0953680f, -0.995442f, 0.0400865f, -0.114546f, 0.992609f, 
		0.0400865f, -0.114546f, 0.992609f, 0.0400865f, -0.114546f, 0.992609f, 
		0.0400865f, -0.114546f, 0.992609f, -0.00219488f, -0.169184f, 0.985582f, 
		-0.00276054f, 0.169128f, -0.985590f, -0.00219488f, -0.169184f, 0.985582f, 
		-0.00219488f, -0.169184f, 0.985582f, -0.556874f, -0.581141f, -0.593436f, 
		-0.459228f, 0.840357f, 0.287941f, 0.571582f, 0.581459f, 0.578964f, 
		-0.0685229f, 0.0448733f, 0.996640f, 0.0685229f, -0.0448733f, -0.996640f, 
		0.390578f, -0.890357f, -0.233908f, 0.0252493f, 0.891049f, 0.453205f, 
		0.0173207f, -0.902115f, -0.431148f, 0.0252493f, 0.891049f, 0.453205f, 
		-0.0381875f, -0.483143f, 0.874708f, -0.0283702f, 0.533128f, -0.845559f, 
		-0.0381875f, -0.483143f, 0.874708f, -0.0381875f, -0.483143f, 0.874708f, 
		-0.119855f, -0.324729f, -0.938182f, -0.532424f, 0.421192f, -0.734249f, 
		-0.715398f, -0.579651f, 0.390142f, -0.518046f, 0.683617f, 0.514098f, 
		0.692144f, -0.299658f, 0.656614f, 0.963870f, 0.225221f, 0.142236f, 
		-0.0150912f, 0.999664f, -0.0210614f, -0.0150912f, 0.999664f, -0.0210614f, 
		0.0150912f, -0.999664f, 0.0210614f, -0.0150912f, 0.999664f, -0.0210614f, 
		0.0150912f, -0.999664f, 0.0210614f, -0.0150912f, 0.999664f, -0.0210614f, 
		-0.0150912f, 0.999664f, -0.0210614f, 0.0150912f, -0.999664f, 0.0210614f, 
		-0.0150912f, 0.999664f, -0.0210614f, 0.0150912f, -0.999664f, 0.0210614f, 
		0.0150912f, -0.999664f, 0.0210614f, 0.0150912f, -0.999664f, 0.0210614f, 
		-0.0150912f, 0.999664f, -0.0210614f, 0.0150912f, -0.999664f, 0.0210614f, 
		0.0105339f, 0.999928f, -0.00572603f, -0.00393101f, -0.988602f, 0.150503f, 
		-0.0110293f, -0.431837f, -0.901884f, 0.00393102f, 0.988602f, -0.150503f, 
		-0.00393101f, -0.988602f, 0.150503f, 0.00393102f, 0.988602f, -0.150503f, 
		-0.0339634f, 0.453410f, 0.890655f, -0.0210590f, 0.898944f, -0.437558f, 
		-0.0250645f, -0.891046f, 0.453221f, -0.0250645f, -0.891046f, 0.453221f, 
		0.0238518f, 0.515009f, 0.856853f, 0.0238518f, 0.515009f, 0.856853f, 
		0.0227834f, -0.485626f, -0.873870f, 0.0238518f, 0.515009f, 0.856853f, 
		0.0391484f, 0.167667f, 0.985066f, 0.0391484f, 0.167667f, 0.985066f, 
		0.00168892f, -0.139504f, -0.990220f, 0.0391484f, 0.167667f, 0.985066f, 
		0.00118174f, 0.213306f, 0.976985f, 0.00118174f, 0.213306f, 0.976985f, 
		-0.00367713f, -0.254641f, -0.967029f, 0.00118174f, 0.213306f, 0.976985f, 
		-0.0511091f, 0.0443102f, 0.997710f, -0.00220863f, -0.0553601f, -0.998464f, 
		-0.0511091f, 0.0443102f, 0.997710f, -0.00220863f, -0.0553601f, -0.998464f, 
		0.000257411f, 0.443959f, 0.896047f, 0.000257411f, 0.443959f, 0.896047f, 
		-0.00470281f, -0.477713f, -0.878503f, 0.000257411f, 0.443959f, 0.896047f, 
		-0.00176941f, 0.613013f, 0.790071f, -0.00176941f, 0.613013f, 0.790071f, 
		-0.00239071f, -0.633429f, -0.773797f, -0.00176941f, 0.613013f, 0.790071f, 
		-0.00756252f, 0.780338f, -0.625312f, -0.00731642f, -0.780342f, 0.625310f, 
		-0.00731642f, -0.780342f, 0.625310f, -0.00605813f, 0.626012f, -0.779790f, 
		-0.00572242f, -0.626003f, 0.779799f, -0.00572242f, -0.626003f, 0.779799f, 
		0.00323198f, 0.637457f, 0.770479f, 0.00323198f, 0.637457f, 0.770479f, 
		0.00269259f, -0.638114f, -0.769937f, 0.000627737f, 0.680738f, 0.732526f, 
		0.000627737f, 0.680738f, 0.732526f, 0.0382843f, -0.679468f, -0.732706f, 
		-0.00435262f, 0.439821f, -0.898075f, -0.00396832f, -0.439819f, 0.898078f, 
		-0.00396832f, -0.439819f, 0.898078f, 0.00196862f, -0.448747f, -0.893657f, 
		0.00235494f, 0.448767f, 0.893646f, 0.00235494f, 0.448767f, 0.893646f, 
		0.00280549f, 0.546847f, 0.837228f, 0.00280549f, 0.546847f, 0.837228f, 
		0.00245046f, -0.546875f, -0.837211f, -0.00197985f, -0.227417f, 0.973796f, 
		-0.00238563f, 0.227416f, -0.973795f, -0.00197985f, -0.227417f, 0.973796f, 
		-8.17391e-006f, -0.0280725f, 0.999606f, -0.000409588f, 0.0280699f, -0.999606f, 
		-8.17391e-006f, -0.0280725f, 0.999606f, 0.000775727f, 0.117390f, 0.993086f, 
		0.000775727f, 0.117390f, 0.993086f, 0.000379380f, -0.117400f, -0.993085f, 
		0.00133730f, 0.232589f, 0.972574f, 0.000937002f, -0.232595f, -0.972573f, 
		0.00133730f, 0.232589f, 0.972574f, 0.00146819f, -0.343601f, -0.939115f, 
		0.00187091f, 0.343608f, 0.939111f, 0.00187091f, 0.343608f, 0.939111f, 
		0.330006f, 0.681085f, 0.653620f, -0.884122f, 0.190410f, 0.426700f, 
		-0.337355f, -0.691162f, 0.639130f, 0.883508f, -0.190414f, 0.427967f, 
		0.00777652f, -0.884535f, -0.466410f, 0.00790389f, 0.885248f, 0.465053f, 
		0.00363992f, -0.770666f, -0.637229f, 0.00390617f, 0.770216f, 0.637771f, 
		-0.0155509f, 0.841613f, -0.539858f, -0.00891013f, -0.832807f, 0.553491f, 
		0.00890086f, -0.832867f, -0.553402f, 0.0155379f, 0.841669f, 0.539770f, 
		-0.0326434f, 0.564728f, -0.824631f, -0.0232093f, -0.599030f, 0.800390f, 
		0.0232111f, -0.599051f, -0.800375f, 0.0341105f, 0.545518f, 0.837404f, 
		0.999670f, -0.0255803f, 0.00214195f, -0.999670f, 0.0255803f, -0.00214195f, 
		-0.0889785f, -0.996034f, 1.87576e-005f, 0.000212424f, 3.64300e-005f, 1.00000f, 
		-0.000176518f, -0.977081f, -0.212867f, -0.0457007f, 0.964316f, 0.260780f, 
		0.0889366f, 0.996037f, -1.87488e-005f, 0.000200110f, 7.27183e-007f, 1.00000f, 
		0.0457130f, 0.964320f, -0.260763f, 0.000173409f, -0.977087f, 0.212842f, 
		-0.0318063f, 0.992066f, 0.121625f, -0.00141863f, -0.990558f, -0.137084f, 
		0.00141630f, -0.990566f, 0.137029f, 0.0318072f, 0.992073f, -0.121567f, 
		0.0106996f, 0.859281f, 0.511391f, 0.00583086f, -0.850791f, -0.525472f, 
		-0.00582706f, -0.850794f, 0.525467f, -0.0106978f, 0.859281f, -0.511392f, 
		0.0164563f, 0.544531f, 0.838579f, 0.0161267f, -0.544482f, -0.838618f, 
		-0.0161206f, -0.544506f, 0.838602f, -0.0164490f, 0.544558f, -0.838562f, 
		0.00283664f, -0.634055f, -0.773283f, 0.00315263f, 0.633447f, 0.773780f, 
		-0.132004f, 0.640849f, 0.756233f, 0.00208490f, 0.888444f, 0.458980f, 
		0.146922f, -0.251465f, -0.956650f, -0.0165107f, -0.977552f, -0.210048f, 
		0.0125787f, 0.943226f, 0.331913f, -0.0136194f, -0.194782f, 0.980752f, 
		-0.0179752f, 0.233405f, -0.972214f, 0.0190840f, 0.217243f, 0.975931f, 
		0.0154189f, -0.182257f, -0.983130f, 0.0234766f, 0.996449f, -0.0808581f, 
		0.0235913f, 0.986139f, 0.164233f, -0.0235913f, -0.986139f, -0.164233f, 
		0.0127403f, -0.998590f, -0.0515263f, -0.0234766f, -0.996449f, 0.0808581f, 
		-0.0127403f, 0.998590f, 0.0515263f, -0.293775f, 0.871612f, -0.392415f, 
		-0.0626775f, 0.994150f, -0.0879646f, -0.0398759f, -0.995014f, -0.0914154f, 
		0.00584630f, -0.999439f, 0.0329862f, 0.293775f, -0.871612f, 0.392415f, 
		-0.0170908f, 0.998271f, 0.0562405f, -0.389735f, 0.606780f, 0.692766f, 
		-0.389735f, 0.606780f, 0.692766f, 0.974726f, 0.138450f, 0.175332f, 
		0.369848f, 0.920615f, 0.125226f, -0.104866f, 0.899725f, 0.423672f, 
		-0.107069f, 0.805723f, 0.582534f, -0.126020f, -0.818824f, -0.560041f, 
		-0.0308802f, -0.940820f, -0.337497f, 0.0612857f, -0.878334f, -0.474103f, 
		0.123298f, -0.960871f, -0.248042f, 0.0751801f, 0.948734f, 0.307004f, 
		0.0262343f, 0.810498f, 0.585153f, 0.0380180f, -0.461390f, -0.886383f, 
		0.00916377f, 0.335354f, 0.942047f, 0.0827845f, 0.720891f, 0.688087f, 
		0.0443479f, 0.419300f, 0.906764f, -0.0273802f, -0.436498f, 0.899288f, 
		-0.0509189f, -0.747075f, 0.662787f, -0.0483877f, -0.232384f, 0.971420f, 
		0.00307145f, 0.0411085f, -0.999150f, -0.0688901f, 0.769918f, -0.634413f, 
		-0.0175487f, 0.938638f, -0.344458f, -0.0437131f, 0.993752f, -0.102693f, 
		-0.114917f, -0.982148f, 0.148928f, -0.00399119f, -0.944286f, 0.329103f, 
		-0.0607800f, -0.911612f, 0.406533f, 0.0323274f, -0.965806f, 0.257243f, 
		0.0475195f, -0.912085f, 0.407239f, 0.111291f, 0.838500f, -0.533415f, 
		0.143111f, 0.895841f, -0.420701f, 0.0229163f, 0.958548f, -0.284009f, 
		0.0436248f, 0.993755f, -0.102702f, -0.0163722f, 0.941828f, -0.335696f, 
		-0.0441607f, 0.994021f, -0.0998625f, -0.115696f, -0.982603f, 0.145280f, 
		-0.00377347f, -0.946890f, 0.321537f, -0.0603969f, -0.915545f, 0.397655f, 
		0.0333188f, -0.967356f, 0.251223f, 0.0476594f, -0.916004f, 0.398327f, 
		0.112683f, 0.845835f, -0.521407f, 0.143992f, 0.900444f, -0.410446f, 
		0.0238023f, 0.960718f, -0.276505f, 0.0440883f, 0.994022f, -0.0998776f, 
		-0.106825f, 0.904762f, 0.412304f, -0.108475f, 0.816622f, 0.566888f, 
		0.00732515f, -0.916332f, -0.400352f, -0.126065f, -0.828203f, -0.546066f, 
		-0.0320199f, -0.943484f, -0.329869f, 0.0607995f, -0.884693f, -0.462193f, 
		0.0495870f, -0.942798f, -0.329656f, 0.0757767f, 0.951080f, 0.299508f, 
		0.0244525f, 0.821591f, 0.569553f, 0.00218455f, -0.470214f, -0.882550f, 
		-0.0528762f, -0.759152f, 0.648762f, -0.000182375f, -1.09119e-005f, -1.00000f, 
		0.000219059f, 1.06077e-005f, 1.00000f, -0.000182375f, -1.09119e-005f, -1.00000f, 
		0.000219059f, 1.06077e-005f, 1.00000f, -0.0697445f, 0.779670f, -0.622295f, 
		0.989991f, 0.100637f, 0.0989459f, -0.989991f, -0.100637f, -0.0989459f, 
		-0.000216527f, 3.79684e-006f, -1.00000f, 0.000182307f, -1.06113e-005f, 1.00000f, 
		-0.000216527f, 3.79684e-006f, -1.00000f, 0.0635709f, -0.713618f, -0.697645f, 
		0.0829920f, 0.735564f, 0.672352f, 0.000182307f, -1.06113e-005f, 1.00000f, 
		0.00255489f, 0.469572f, 0.882891f, 0.996361f, 0.0741662f, -0.0420131f, 
		-0.996361f, -0.0741662f, 0.0420131f, 0.996864f, 0.0581042f, -0.0537128f, 
		-0.995170f, -0.0943151f, 0.0272361f, -0.133439f, 0.0436863f, -0.990094f, 
		0.00137376f, -0.262692f, -0.964879f, -0.000537550f, -0.999942f, 0.0107455f, 
		-0.0498724f, -0.143428f, -0.988403f, 0.00177818f, 0.262197f, 0.965013f, 
		-0.0494576f, 0.143078f, 0.988475f, 0.000542031f, -0.999942f, 0.0107448f, 
		0.942227f, 0.331655f, 0.0470432f, -0.942227f, -0.331655f, -0.0470432f, 
		0.543394f, -0.0117250f, 0.839396f, -0.543394f, 0.0117250f, -0.839396f, 
		0.0143540f, -0.454557f, -0.890602f, 0.00880854f, -0.223090f, -0.974758f, 
		-0.000110639f, -0.0997791f, -0.995010f, -0.0129164f, -0.697263f, 0.716699f, 
		0.999402f, -0.00617129f, -0.0340284f, -0.999402f, 0.00617129f, 0.0340284f, 
		0.0936821f, 0.127998f, -0.987340f, -0.0279200f, -0.898211f, -0.438676f, 
		0.0538534f, 0.107972f, -0.992694f, -0.000653469f, -0.893051f, -0.449956f, 
		0.0272935f, 0.671604f, 0.740407f, 0.976313f, -0.212421f, 0.0411201f, 
		-0.976313f, 0.212421f, -0.0411201f, 0.976313f, -0.212421f, 0.0411201f, 
		-0.976313f, 0.212421f, -0.0411201f, 4.57019e-007f, 1.00000f, -0.000171735f, 
		0.0364817f, -0.351977f, -0.935297f, 1.03186e-006f, -1.00000f, 0.000168499f, 
		-0.0728076f, 0.330036f, -0.941156f, 0.000627883f, 0.999907f, 0.0136065f, 
		-0.0531667f, 0.115681f, 0.991862f, 0.127818f, 0.269842f, 0.954384f, 
		0.708537f, 0.664289f, -0.238108f, 0.170705f, 0.944134f, 0.281904f, 
		-0.988285f, 0.111039f, -0.104702f, 0.0884528f, 0.932547f, -0.350047f, 
		-0.147297f, 0.937639f, -0.314859f, -0.420814f, 0.876368f, -0.234296f, 
		0.946284f, 0.303830f, -0.110608f, -0.000620301f, -0.999907f, 0.0136329f, 
		0.0536014f, -0.115687f, 0.991838f, -0.548787f, 0.120364f, -0.827252f, 
		0.782705f, 0.0825656f, -0.616891f, 0.0768019f, 0.0825580f, -0.993622f, 
		0.914560f, 0.0266620f, -0.403570f, -0.787876f, 0.0538506f, -0.613476f, 
		-1.23968e-005f, -0.999998f, 0.00217616f, -0.0723864f, 0.329989f, 0.941205f, 
		-0.977507f, -0.205835f, -0.0459595f, 0.977507f, 0.205835f, 0.0459595f, 
		9.11606e-005f, 1.00000f, 0.000302878f, 0.0368672f, -0.351941f, 0.935296f, 
		0.875683f, 0.423041f, 0.232843f, 0.923552f, 0.339974f, 0.177398f, 
		-0.838333f, 0.476182f, 0.265419f, 0.974256f, 0.202700f, 0.0986810f, 
		0.996864f, 0.0581042f, -0.0537128f, 0.00375773f, 0.0530937f, 0.998582f, 
		-0.995170f, -0.0943151f, 0.0272361f, 0.996361f, 0.0741662f, -0.0420131f, 
		-0.996361f, -0.0741662f, 0.0420131f, 0.0344085f, 0.646912f, -0.761788f, 
		-0.528166f, 0.572033f, -0.627551f, -0.168214f, 0.641002f, -0.748879f, 
		0.00363896f, 0.948796f, -0.315868f, -0.307641f, 0.270568f, -0.912223f, 
		-0.163282f, 0.881267f, -0.443517f, -0.481229f, 0.494946f, -0.723496f, 
		-0.518661f, 0.300619f, -0.800387f, 0.169934f, 0.863937f, -0.474063f, 
		0.489540f, 0.452401f, -0.745442f, 0.252466f, 0.179600f, -0.950792f, 
		0.528135f, 0.846037f, 0.0727678f, -0.297759f, 0.953206f, -0.0523196f, 
		0.648052f, -0.752759f, -0.115679f, 0.935990f, -0.0292007f, -0.350813f, 
		0.792442f, 0.0401360f, 0.608625f, -0.931626f, -0.0401372f, 0.361195f, 
		0.000719077f, 0.999879f, 0.0155608f, -0.0528755f, 0.153116f, 0.986793f, 
		-0.552385f, 0.246862f, -0.796197f, -0.667344f, 0.729993f, 0.147518f, 
		0.711187f, 0.599089f, -0.367839f, 0.936621f, 0.168822f, 0.306986f, 
		-0.936621f, -0.168822f, -0.306986f, -0.00100686f, -0.999936f, -0.0112447f, 
		-0.0165920f, 0.356835f, -0.934020f, -0.000712575f, -0.999879f, 0.0155610f, 
		0.0532936f, -0.153086f, 0.986775f, 0.888788f, -0.349749f, -0.296197f, 
		-0.815563f, -0.440784f, -0.374921f, 0.731342f, -0.552816f, -0.399416f, 
		0.274687f, -0.829038f, -0.487076f, 0.988929f, 0.100804f, 0.108890f, 
		-0.943268f, -0.101959f, -0.315990f, -0.943268f, -0.101959f, -0.315990f, 
		0.988929f, 0.100804f, 0.108890f, -0.175701f, -0.447981f, -0.876608f, 
		-0.814515f, 0.170024f, 0.554668f, 0.0796169f, 0.654901f, 0.751509f, 
		0.995066f, -0.0801334f, -0.0584932f, -0.956035f, 0.293241f, -0.00246816f, 
		0.956035f, -0.293241f, 0.00246816f, 0.792442f, 0.0401360f, 0.608625f, 
		0.937259f, 0.248370f, -0.244657f, 0.480708f, -0.870231f, 0.107785f, 
		-0.884730f, -0.198093f, -0.421915f, -0.992761f, -0.112405f, 0.0423078f, 
		-0.970445f, 0.00753588f, 0.241204f, -0.917741f, 0.118631f, 0.379049f, 
		0.993945f, 0.0556352f, -0.0947537f, -0.231634f, 0.924558f, 0.302554f, 
		-0.0108706f, 0.158117f, 0.987361f, -0.0108706f, 0.158117f, 0.987361f, 
		0.0128263f, -0.201471f, -0.979410f, 0.231634f, -0.924558f, -0.302554f, 
		-0.955442f, 0.293583f, 0.0306631f, 0.969484f, -0.223969f, 0.0996925f, 
		-0.922638f, 0.278825f, -0.266451f, 0.922638f, -0.278825f, 0.266451f, 
		-0.906755f, 0.283100f, 0.312490f, 0.973453f, -0.197407f, -0.115840f, 
		0.250611f, -0.0921058f, -0.963696f, 0.981692f, -0.112653f, 0.153588f, 
		-0.981692f, 0.112653f, -0.153588f, -0.813640f, 0.580203f, 0.0367837f, 
		0.813640f, -0.580203f, -0.0367837f, 0.965661f, 0.155413f, -0.208198f, 
		-0.965661f, -0.155413f, 0.208198f, -0.362829f, 0.926144f, -0.103019f, 
		-0.147548f, 0.986402f, -0.0723894f, 0.0785066f, 0.996913f, -0.000836292f, 
		0.984768f, 0.171506f, -0.0285955f, 0.123964f, -0.0162240f, 0.992154f, 
		0.000886769f, -0.999988f, 0.00479372f, 0.672260f, -0.737967f, -0.0589196f, 
		-0.0680918f, 0.680055f, 0.729992f, 0.0680918f, -0.680055f, -0.729992f, 
		-0.644099f, 0.762590f, 0.0599500f, 0.0745845f, -0.00282304f, -0.997211f, 
		0.000179820f, -0.999960f, -0.00899737f, -0.0763506f, -0.00253474f, -0.997078f, 
		0.0763506f, 0.00253474f, 0.997078f, 0.528135f, 0.846037f, 0.0727678f, 
		-0.524792f, -0.851227f, 0.00239245f, 0.648052f, -0.752759f, -0.115679f, 
		0.0124056f, 0.276366f, 0.960972f, -0.402301f, -0.653814f, -0.640844f, 
		0.568837f, 0.762034f, 0.309402f, -0.961409f, 0.273896f, -0.0259553f, 
		0.993372f, -0.0881340f, 0.0737849f, -0.470783f, 0.383887f, -0.794351f, 
		-0.558969f, 0.666017f, 0.493937f, 0.369958f, 0.695876f, -0.615538f, 
		0.00375773f, 0.0530937f, 0.998582f, -0.133439f, 0.0436863f, -0.990094f, 
		0.939601f, -0.326886f, -0.101466f, -0.798204f, 0.597189f, 0.0789723f, 
		0.960236f, -0.197458f, -0.197378f, -0.960236f, 0.197458f, 0.197378f, 
		-0.877476f, 0.449721f, 0.166695f, 0.393498f, 0.916306f, 0.0744448f, 
		0.996932f, 0.0782518f, 0.00155591f, 0.730244f, 0.00465713f, -0.683171f, 
		-0.344036f, -0.180376f, -0.921468f, 0.939466f, 0.0587646f, 0.337566f, 
		-0.537614f, -0.00835144f, 0.843150f, 0.975755f, 0.0366696f, -0.215773f, 
		0.510763f, -0.110497f, 0.852591f, -0.00957236f, 0.913227f, -0.407338f, 
		-0.221150f, 0.654339f, -0.723141f, -0.597538f, -0.779187f, -0.189248f, 
		0.820710f, -0.271432f, -0.502752f, 0.0503172f, 0.967685f, -0.247090f, 
		0.161243f, -0.888810f, -0.428973f, 0.953783f, 0.263540f, 0.144380f, 
		0.0607491f, -0.913504f, -0.402269f, -0.118147f, -0.921821f, -0.369172f, 
		-0.186043f, -0.975297f, 0.119095f, 0.996443f, 0.0833273f, -0.0125790f, 
		0.680426f, -0.732805f, -0.00423647f, 0.543394f, -0.0117250f, 0.839396f, 
		-0.543394f, 0.0117250f, -0.839396f, 0.869920f, 0.485268f, 0.0880576f, 
		-0.869920f, -0.485268f, -0.0880576f, 0.0167868f, -0.956576f, -0.290999f, 
		-0.0988923f, 0.952363f, 0.288487f, -0.0290772f, 0.0412964f, 0.998724f, 
		0.0988923f, -0.952363f, -0.288487f, 0.0290772f, -0.0412964f, -0.998724f, 
		-0.847494f, 0.198224f, 0.492403f, -0.349384f, 0.881456f, 0.317753f, 
		0.668916f, 0.742430f, 0.0367276f, 0.474889f, 0.879321f, 0.0357092f, 
		0.780801f, 0.616400f, 0.101985f, 0.00494981f, 0.962948f, -0.269641f, 
		-0.0638770f, -0.00165909f, -0.997956f, 0.0638770f, 0.00165909f, 0.997956f, 
		0.855753f, 0.504006f, -0.116894f, 0.480708f, -0.870231f, 0.107785f, 
		-0.462640f, 0.870138f, 0.169776f, -0.0622296f, -0.0746319f, 0.995268f, 
		-0.131743f, 0.115916f, -0.984483f, 0.0569004f, 0.983227f, 0.173282f, 
		0.0622296f, 0.0746319f, -0.995268f, 0.0533401f, -0.152584f, 0.986850f, 
		-0.0533401f, 0.152584f, -0.986850f, 0.0533401f, -0.152584f, 0.986850f, 
		-0.0533401f, 0.152584f, -0.986850f, 0.131743f, -0.115916f, 0.984483f, 
		-0.0569004f, -0.983227f, -0.173282f, -0.220619f, -0.963109f, -0.154107f, 
		0.990921f, 0.131985f, 0.0256217f, 0.680427f, -0.732805f, 0.00383993f, 
		0.00367624f, 0.945403f, -0.325882f, -0.0188924f, -0.968335f, 0.248937f, 
		-0.224060f, -0.476921f, -0.849908f, -0.578124f, -0.428871f, -0.694149f, 
		0.0414968f, -0.611392f, -0.790239f, 0.486996f, -0.458233f, -0.743544f, 
		0.139258f, -0.352053f, -0.925562f, 0.737081f, -0.415717f, -0.532814f, 
		-0.862728f, 0.303472f, -0.404482f, -0.0755129f, 0.475929f, 0.876236f, 
		-0.270837f, -0.962318f, 0.0243196f, 0.225901f, -0.425259f, -0.876427f, 
		0.862728f, -0.303472f, 0.404482f, -0.0755129f, 0.475929f, 0.876236f, 
		0.270837f, 0.962318f, -0.0243196f, 0.663997f, -0.647901f, 0.373273f, 
		-0.382211f, -0.890895f, 0.245401f, 0.969947f, 0.228536f, -0.0835134f, 
		-0.0194517f, -0.958425f, 0.284681f, 0.257514f, 0.222308f, -0.940354f, 
		-0.824083f, -0.390843f, -0.410036f, -0.309691f, 0.941048f, -0.136087f, 
		-0.322325f, 3.52166e-005f, 0.946629f, 0.0262621f, -0.889411f, 0.456354f, 
		0.187892f, -0.881727f, 0.432728f, 0.187892f, -0.881727f, 0.432728f, 
		0.295732f, -0.831664f, 0.469977f, 0.898114f, -0.00714133f, 0.439704f, 
		0.0261977f, 0.304100f, 0.952280f, 0.0261977f, 0.304100f, 0.952280f, 
		-0.664175f, -0.116542f, -0.738437f, -0.366373f, -0.318500f, -0.874259f, 
		-0.105372f, -0.993314f, -0.0471633f, 0.0258515f, -0.303401f, -0.952512f, 
		-0.138427f, -0.296762f, -0.944865f, 0.105372f, 0.993314f, 0.0471633f, 
		0.346966f, 0.361470f, 0.865421f, 0.173221f, 0.313708f, 0.933585f, 
		-0.0140749f, 0.995470f, -0.0940337f, 0.00354548f, 0.110078f, 0.993917f, 
		0.232930f, -0.918309f, 0.320082f, -0.00354549f, -0.110078f, -0.993917f, 
		-0.164470f, 0.823915f, -0.542322f, 0.0838377f, 0.474398f, -0.876309f, 
		-0.886368f, -0.443349f, -0.133391f, -0.358217f, 0.880313f, -0.311015f, 
		-0.275930f, -0.0551890f, 0.959592f, 0.130327f, -0.920128f, 0.369296f, 
		0.849491f, 0.523353f, 0.0668283f, 0.849491f, 0.523353f, 0.0668283f, 
		0.130327f, -0.920128f, 0.369296f, 0.268771f, 0.281863f, -0.921040f, 
		0.0592518f, -0.955266f, 0.289751f, 0.00539838f, 0.110771f, 0.993831f, 
		-0.0116543f, 0.996994f, -0.0766002f, -0.00540174f, -0.110763f, -0.993832f, 
		-0.0116687f, 0.996991f, -0.0766297f, 0.00539838f, 0.110771f, 0.993831f, 
		0.0604787f, -0.922717f, 0.380704f, -0.00540174f, -0.110763f, -0.993832f, 
		-0.338958f, -0.939063f, -0.0571700f, -0.144646f, -0.988519f, -0.0436683f, 
		-0.169245f, -0.524509f, 0.834414f, 0.0662291f, -0.988697f, -0.134506f, 
		4.71261e-005f, -0.000118851f, -1.00000f, 0.290720f, 0.956805f, -0.00241824f, 
		0.995650f, 0.0931346f, 0.00254138f, 0.680426f, -0.732805f, -0.00423647f, 
		0.949497f, -0.246430f, 0.194236f, -0.528166f, 0.572033f, -0.627551f, 
		0.493141f, -0.720819f, 0.487065f, 0.666853f, -0.596676f, 0.446413f, 
		0.430753f, 0.813181f, -0.391394f, 0.997023f, -0.00444497f, 0.0769825f, 
		-0.0539347f, -0.713402f, 0.698676f, 0.0177278f, -0.614769f, 0.788508f, 
		-0.227280f, -0.688995f, 0.688208f, 0.304018f, -0.535690f, 0.787787f, 
		0.191471f, -0.472250f, 0.860418f, 0.713776f, -0.444149f, 0.541530f, 
		-0.0747432f, -0.366471f, -0.927422f, 0.980642f, 0.0353706f, 0.192587f, 
		-8.06244e-005f, 1.00000f, 1.96246e-005f, 0.869038f, -0.00138635f, -0.494743f, 
		0.455632f, 0.00655729f, 0.890144f, -0.168193f, 0.981404f, -0.0925113f, 
		-0.383551f, 0.915089f, -0.124502f, -0.00565751f, -0.995569f, 0.0938591f, 
		0.168193f, -0.981404f, 0.0925113f, 0.383551f, -0.915089f, 0.124502f, 
		0.0205622f, -0.352314f, -0.935656f, 0.000787455f, 0.323430f, 0.946252f, 
		0.00565751f, 0.995569f, -0.0938591f, -0.0862654f, 0.953284f, 0.289495f, 
		-0.232209f, -0.0605775f, -0.970778f, 0.00905329f, -0.958523f, -0.284872f, 
		0.0862654f, -0.953284f, -0.289495f, -0.0619169f, -0.983908f, -0.167606f, 
		0.0619169f, 0.983908f, 0.167606f, -0.00905329f, 0.958523f, 0.284872f, 
		0.232209f, 0.0605775f, 0.970778f, -0.147763f, 0.532293f, -0.833565f, 
		-0.176493f, 0.639063f, -0.748631f, -0.217890f, 0.808303f, -0.546965f, 
		-0.336651f, 0.615581f, -0.712549f, 0.312128f, 0.687360f, -0.655829f, 
		0.419662f, 0.821367f, -0.386316f, 0.614319f, 0.695900f, -0.371935f, 
		0.964804f, 0.0209217f, -0.262135f, 0.771785f, -0.260796f, 0.579943f, 
		0.506885f, 0.333844f, -0.794743f, 0.271570f, 0.376223f, -0.885836f, 
		-0.383002f, -0.439982f, 0.812235f, -0.712621f, 0.464719f, -0.525555f, 
		0.928003f, -0.158511f, 0.337172f, 2.06423e-005f, -4.94704e-005f, 1.00000f, 
		0.197127f, 0.980376f, 0.00181153f, 0.996880f, 0.0788934f, -0.00260635f, 
		0.680427f, -0.732805f, 0.00383993f, -1.96055e-005f, 1.00000f, 8.06487e-005f, 
		0.222609f, 0.00810996f, 0.974874f, 0.730244f, 0.00465713f, -0.683171f, 
		0.952746f, -0.00837882f, 0.303654f, -0.233186f, 0.624861f, 0.745099f, 
		0.0658227f, 0.720838f, 0.689971f, 0.966395f, -0.208905f, -0.149801f, 
		0.609629f, 0.611409f, 0.504510f, 0.836068f, -0.443726f, -0.322641f, 
		0.936310f, 0.316005f, 0.153179f, -0.175311f, -0.949988f, 0.258435f, 
		0.0920063f, -0.0398623f, 0.994960f, 0.0920063f, -0.0398623f, 0.994960f, 
		0.175311f, 0.949988f, -0.258435f, -0.0920063f, 0.0398623f, -0.994960f, 
		-0.720858f, 0.000000f, -0.693083f, -0.243507f, 0.000000f, 0.969899f, 
		0.0191561f, 0.996445f, 0.0820410f, 0.720858f, 0.000000f, 0.693083f, 
		0.243507f, 0.000000f, -0.969899f, -0.0191561f, -0.996445f, -0.0820410f, 
		-0.720858f, 0.000000f, -0.693083f, -0.0353487f, 0.994493f, -0.0986664f, 
		-0.243507f, 0.000000f, 0.969899f, 0.720858f, 0.000000f, 0.693083f, 
		0.243507f, 0.000000f, -0.969899f, 0.0353487f, -0.994493f, 0.0986664f, 
		-0.00204776f, -0.996001f, 0.0893148f, 0.254325f, -0.958440f, 0.129275f, 
		0.492242f, -0.865842f, 0.0895318f, -0.635999f, -0.764564f, 0.104626f, 
		-0.243609f, 0.969636f, -0.0214704f, -0.0481782f, 0.00403615f, 0.998831f, 
		0.732277f, 0.670146f, 0.121139f, 0.0481782f, -0.00403615f, -0.998831f, 
		0.296528f, -0.951639f, -0.0803324f, 0.0590218f, 0.545582f, -0.835976f, 
		0.395799f, 0.913878f, -0.0903921f, 0.938074f, 0.344910f, 0.0324612f, 
		0.0590218f, 0.545582f, -0.835976f, -0.270355f, -0.0712296f, -0.960122f, 
		0.723329f, -0.0828670f, -0.685514f, 0.0186243f, -0.0815063f, 0.996499f, 
		0.0440596f, 0.112208f, -0.992707f, -0.491993f, -0.0916132f, 0.865765f, 
		-0.282224f, -0.0783042f, 0.956147f, -0.144518f, 0.932529f, 0.330915f, 
		-0.0440596f, -0.112208f, 0.992707f, 0.491993f, 0.0916132f, -0.865765f, 
		0.104177f, -0.931611f, -0.348208f, 0.282224f, 0.0783042f, -0.956147f, 
		0.0441037f, 0.112009f, -0.992728f, -0.849678f, 0.0837012f, 0.520616f, 
		-0.695021f, -0.00596288f, 0.718965f, 1.06585e-005f, 0.988490f, 0.151286f, 
		-0.282393f, -0.0782644f, 0.956101f, -0.0441037f, -0.112009f, 0.992728f, 
		0.492193f, 0.0916391f, -0.865649f, 0.282393f, 0.0782644f, -0.956101f, 
		0.129390f, -0.926673f, -0.352895f, 0.141249f, -0.925969f, -0.350186f, 
		0.0441754f, 0.111967f, -0.992730f, -0.912645f, -0.401067f, 0.0788935f, 
		-0.492235f, -0.0916134f, 0.865628f, -0.282436f, -0.0782305f, 0.956091f, 
		-0.0441754f, -0.111967f, 0.992730f, -0.176450f, 0.972129f, 0.154371f, 
		0.492235f, 0.0916134f, -0.865628f, 0.282436f, 0.0782305f, -0.956091f, 
		-0.340006f, -0.126843f, -0.931830f, -0.595887f, -0.119374f, -0.794146f, 
		0.115792f, 0.0929169f, 0.988918f, 0.340006f, 0.126843f, 0.931830f, 
		0.595887f, 0.119374f, 0.794146f, -0.115795f, -0.0929110f, -0.988918f, 
		-0.000129404f, 1.00000f, -0.000136225f, -4.25207e-005f, -1.00000f, 0.000179956f, 
		-0.338185f, -0.126216f, -0.932577f, -0.593771f, -0.118422f, -0.795872f, 
		0.113142f, 0.0919494f, 0.989315f, 0.338185f, 0.126216f, 0.932577f, 
		0.593771f, 0.118422f, 0.795872f, -0.113142f, -0.0919494f, -0.989315f, 
		-0.000129404f, 1.00000f, -0.000136225f, -4.25207e-005f, -1.00000f, 0.000179956f, 
		-0.340002f, -0.126855f, -0.931830f, -0.595885f, -0.119386f, -0.794146f, 
		0.115792f, 0.0929169f, 0.988918f, 0.340002f, 0.126855f, 0.931830f, 
		0.595885f, 0.119386f, 0.794146f, -0.115795f, -0.0929110f, -0.988918f, 
		-0.000129404f, 1.00000f, -0.000136225f, -4.25207e-005f, -1.00000f, 0.000179956f, 
		-0.629388f, 0.661887f, -0.407157f, 0.0950936f, 0.283865f, -0.954137f, 
		-0.864236f, -0.474027f, -0.168503f, -0.198549f, 0.957633f, 0.208609f, 
		0.0278607f, 0.823619f, 0.566458f, -0.161636f, 0.925842f, 0.341601f, 
		0.886368f, 0.443349f, 0.133391f, 0.257466f, -0.953527f, -0.156514f, 
		0.206720f, -0.940209f, -0.270690f, 0.111806f, -0.883340f, -0.455204f, 
		-0.951147f, 0.144498f, -0.272835f, -0.256924f, 0.254033f, 0.932447f, 
		-0.0895848f, -0.995978f, 0.00135674f, 0.951147f, -0.144498f, 0.272835f, 
		0.177694f, -0.254748f, -0.950541f, 0.102563f, 0.994283f, 0.0297074f, 
		-0.223813f, -0.937387f, -0.266857f, -0.976983f, 0.143830f, -0.157535f, 
		-0.273151f, -0.0128866f, 0.961885f, -0.223813f, -0.937387f, -0.266857f, 
		0.888046f, -0.362184f, 0.283189f, 0.273151f, 0.0128866f, -0.961885f, 
		-0.200654f, -0.979634f, 0.00740847f, 0.0255528f, -0.106774f, 0.993955f, 
		-0.736479f, 0.676244f, -0.0171178f, 0.205929f, 0.978559f, -0.00400306f, 
		-0.0255528f, 0.106774f, -0.993955f, 0.0255528f, -0.106774f, 0.993955f, 
		0.716207f, -0.697361f, 0.0271054f, -0.270355f, -0.0712296f, -0.960122f, 
		0.723329f, -0.0828670f, -0.685514f, 0.0926796f, 0.900025f, -0.425871f, 
		-0.0643350f, -0.891748f, 0.447935f, -0.741666f, 0.00739008f, 0.670729f, 
		-0.225443f, 0.00330022f, -0.974251f, 0.0643350f, 0.891748f, -0.447935f, 
		0.714098f, 0.00611300f, -0.700020f, 0.238525f, 0.00282274f, 0.971132f, 
		0.0296830f, 0.280981f, 0.959254f, -0.0962387f, 0.0631136f, -0.993355f, 
		0.182499f, -0.114832f, 0.976477f, 0.102074f, 0.530030f, 0.841813f, 
		-0.0479982f, -0.320271f, -0.946109f, 0.0606735f, -0.969893f, 0.235851f, 
		-0.0962387f, 0.0631136f, -0.993355f, 0.207213f, -0.222151f, 0.952739f, 
		-0.109817f, -0.107243f, -0.988149f, 0.0381734f, 0.970659f, -0.237411f, 
		-0.0404736f, 0.995864f, -0.0813421f, -0.416498f, -0.902300f, 0.111282f, 
		-0.0621465f, -0.514986f, -0.854943f, -0.202626f, -0.975992f, 0.0798899f, 
		0.0404736f, -0.995864f, 0.0813421f, 0.548755f, 0.828328f, -0.112876f, 
		0.368230f, 0.925225f, -0.0914594f, 0.0156997f, 0.0804606f, 0.996634f, 
		-0.00765825f, 0.996299f, -0.0856169f, 0.0640275f, 0.278435f, 0.958319f, 
		0.521061f, 0.849510f, -0.0826286f, 0.270588f, 0.961255f, -0.0526454f, 
		-0.0441969f, -0.998869f, 0.0175563f, -0.556467f, 0.830169f, 0.0341280f, 
		-0.379741f, 0.925082f, 0.00451558f, 0.0441969f, 0.998869f, -0.0175563f, 
		-0.0626055f, 0.135948f, 0.988736f, 0.430361f, -0.902168f, -0.0297012f, 
		0.219447f, -0.975585f, 0.00880859f, 0.0626055f, -0.135948f, -0.988736f, 
		-0.735397f, -0.671163f, -0.0934452f, -0.294933f, 0.953769f, 0.0577892f, 
		0.826745f, 0.550821f, -0.114407f, 0.140987f, 0.289406f, -0.946766f, 
		-0.140987f, -0.289406f, 0.946766f, 0.349640f, -0.936844f, 0.00864293f, 
		0.00745248f, -0.871799f, -0.489807f, -0.824190f, 0.555369f, -0.110793f, 
		-0.0256142f, 0.632674f, 0.773995f, 0.0731651f, 0.905850f, 0.417233f, 
		-0.139337f, 0.352971f, 0.925201f, 0.824190f, -0.555369f, 0.110793f, 
		0.139337f, -0.352971f, -0.925201f, 0.0800871f, -0.527726f, -0.845631f, 
		-0.597642f, -0.349750f, 0.721456f, -0.244462f, 0.335112f, -0.909911f, 
		-0.0307139f, -0.994180f, 0.103256f, 0.597642f, 0.349750f, -0.721456f, 
		0.0842728f, 0.990704f, -0.106792f, 0.245594f, -0.332203f, 0.910672f, 
		-0.152119f, -0.165689f, -0.974375f, -0.322919f, -0.946422f, -0.00300512f, 
		-0.519771f, -0.853988f, -0.0232936f, 0.0656178f, 0.997568f, 0.0235125f, 
		0.322919f, 0.946422f, 0.00300512f, 0.519771f, 0.853988f, 0.0232936f, 
		0.216361f, 0.222294f, 0.950670f, -0.0656178f, -0.997568f, -0.0235125f, 
		-0.0359577f, 0.111657f, -0.993096f, -0.206639f, -0.978409f, 0.00413220f, 
		-0.714024f, 0.699414f, -0.0314565f, 0.212523f, 0.977129f, -0.00721966f, 
		0.0359577f, -0.111657f, 0.993096f, 0.691307f, -0.722361f, 0.0169765f, 
		0.0522869f, -0.394297f, 0.917494f, 0.00800218f, -0.564653f, 0.825290f, 
		-0.175766f, -0.739595f, 0.649696f, 0.216904f, -0.513082f, 0.830482f, 
		0.537849f, -0.561572f, 0.628772f, -0.250091f, 0.122831f, 0.960399f, 
		0.647213f, 0.203222f, 0.734722f, 0.226598f, -0.776577f, 0.587862f, 
		-0.762575f, -0.00659323f, 0.646866f, -0.214549f, -0.00260589f, -0.976710f, 
		0.738820f, -0.00405679f, -0.673891f, -0.0923181f, -0.989640f, 0.109956f, 
		0.0222233f, 0.994273f, -0.104536f, 0.226556f, -0.00311315f, 0.973993f, 
		0.0190507f, 0.00719017f, 0.999793f, -0.545902f, -0.0391025f, -0.836936f, 
		-0.397152f, -0.0210548f, -0.917511f, -0.0190507f, -0.00719017f, -0.999793f, 
		-0.0311149f, -0.995784f, -0.0862886f, 0.425884f, 0.0338012f, 0.904146f, 
		0.0311149f, 0.995784f, 0.0862886f, 0.255558f, 0.0108453f, 0.966733f, 
		-0.258205f, 0.553169f, -0.792044f, -0.450393f, 0.472836f, -0.757346f, 
		-0.158462f, 0.973790f, 0.163167f, -0.0268043f, 0.550830f, -0.834187f, 
		0.249961f, 0.961895f, 0.110799f, -0.00411009f, 0.992417f, 0.122844f, 
		-0.208937f, 0.0232034f, 0.977654f, 0.130848f, 0.939098f, 0.317763f, 
		-0.213180f, -0.934121f, -0.286308f, 0.329238f, -0.508834f, 0.795418f, 
		0.688184f, -0.659623f, 0.302160f, 0.840885f, -0.541109f, 0.0106424f, 
		0.176678f, -0.384526f, 0.906049f, -0.606035f, -0.531043f, 0.592212f, 
		-0.351911f, -0.756201f, 0.551651f, 0.180797f, -0.666418f, 0.723326f, 
		-0.00894075f, -0.830692f, 0.556660f, -0.0732698f, -0.920004f, 0.384999f, 
		-0.607826f, -0.527079f, 0.593915f, -0.334191f, -0.747765f, 0.573729f, 
		0.215896f, -0.642097f, 0.735595f, 0.0194741f, -0.815122f, 0.578962f, 
		-0.0440910f, -0.907808f, 0.417062f, -0.607837f, -0.527077f, 0.593906f, 
		-0.344447f, -0.751703f, 0.562405f, 0.180807f, -0.666394f, 0.723345f, 
		0.000446198f, -0.822987f, 0.568059f, -0.0615889f, -0.913859f, 0.401334f, 
		-0.00474157f, -0.999546f, -0.0297463f, 0.0938579f, -0.887206f, -0.451725f, 
		-0.0231074f, -0.955792f, 0.293133f, 0.421143f, -0.733503f, -0.533491f, 
		0.595114f, -0.667090f, -0.448141f, 0.0664576f, 0.797835f, -0.599201f, 
		0.423061f, 0.0591419f, -0.904169f, 0.778237f, 0.0194625f, -0.627669f, 
		0.635121f, -0.00310355f, -0.772406f, 0.0664576f, 0.797835f, -0.599201f, 
		-0.355528f, 0.844637f, -0.400236f, -0.669803f, 0.353345f, -0.653078f, 
		-0.534809f, 0.621955f, -0.571971f, 0.355528f, -0.844637f, 0.400236f, 
		-0.0704805f, 0.409141f, 0.909745f, 0.534809f, -0.621955f, 0.571971f, 
		0.669803f, -0.353345f, 0.653078f, 0.0704805f, -0.409141f, -0.909745f, 
		-0.00209713f, 0.837639f, 0.546221f, -0.210824f, 0.608579f, 0.764973f, 
		0.0583491f, 0.935163f, 0.349380f, 0.663122f, 0.283208f, 0.692866f, 
		-0.0186448f, 0.846427f, 0.532178f, -0.0265070f, 0.487262f, 0.872854f, 
		0.401830f, 0.298478f, 0.865704f, 0.0393690f, 0.939415f, 0.340513f, 
		0.570743f, 0.268677f, 0.775929f, -0.713258f, -0.105513f, -0.692914f, 
		-0.713258f, -0.105513f, -0.692914f, -0.120460f, -0.923910f, -0.363153f, 
		-0.108515f, -0.143388f, 0.983699f, -0.108515f, -0.143388f, 0.983699f, 
		0.120460f, 0.923910f, 0.363153f, 0.713258f, 0.105513f, 0.692914f, 
		0.108515f, 0.143388f, -0.983699f, -0.380807f, -0.605569f, 0.698765f, 
		-0.508339f, -0.452713f, 0.732558f, -0.661785f, -0.288301f, 0.692043f, 
		-0.0323655f, -0.758213f, -0.651203f, -0.261562f, 0.0552277f, -0.963605f, 
		0.380807f, 0.605569f, -0.698765f, 0.661785f, 0.288301f, -0.692043f, 
		0.508339f, 0.452713f, -0.732558f, 0.261562f, -0.0552277f, 0.963605f, 
		0.0323655f, 0.758213f, 0.651203f, 0.307519f, -0.585890f, -0.749777f, 
		0.104361f, 0.578392f, -0.809056f, 0.482122f, -0.441781f, -0.756564f, 
		0.652483f, -0.0199946f, -0.757539f, 0.370348f, 0.383284f, -0.846130f, 
		-0.207346f, -0.909683f, 0.359839f, 0.293692f, -0.502238f, -0.813328f, 
		0.369310f, -0.726910f, -0.578975f, 0.934174f, -0.212996f, 0.286273f, 
		0.179485f, -0.377778f, -0.908333f, -0.281490f, 0.533080f, 0.797865f, 
		0.209277f, -0.598232f, 0.773512f, 0.680765f, 0.280190f, 0.676796f, 
		-0.0802112f, -0.995947f, 0.0407028f, 0.0317268f, -0.892655f, 0.449623f, 
		-0.0547714f, -0.965452f, -0.254759f, 0.607753f, -0.667271f, 0.430564f, 
		0.390421f, -0.763227f, 0.514836f, -0.703999f, -0.710153f, 0.00826704f, 
		-0.356055f, -0.375466f, -0.855716f, -0.0214272f, 0.817612f, -0.575371f, 
		0.356055f, 0.375466f, 0.855716f, 0.703999f, 0.710153f, -0.00826704f, 
		0.0214272f, -0.817612f, 0.575371f, -0.255399f, 0.905618f, 0.338566f, 
		-0.285309f, 0.564149f, 0.774813f, -0.479711f, 0.747503f, 0.459474f, 
		-0.0644884f, 0.574280f, 0.816115f, -0.00166591f, 0.742197f, 0.670180f, 
		-0.0938559f, 0.883619f, 0.458702f, -0.660332f, -0.199666f, 0.723944f, 
		-0.256979f, 0.108385f, -0.960320f, 0.0938559f, -0.883619f, -0.458702f, 
		0.660332f, 0.199666f, -0.723944f, 0.254541f, -0.187130f, 0.948784f, 
		-0.827794f, 0.331537f, -0.452594f, -0.339554f, 0.269584f, -0.901126f, 
		-0.691789f, 0.276538f, -0.667050f, -0.154465f, 0.215867f, -0.964128f, 
		-0.320291f, -0.942160f, 0.0987296f, 0.862728f, -0.303472f, 0.404482f, 
		0.185296f, -0.189532f, 0.964232f, 0.373001f, 0.476070f, -0.796384f, 
		0.320291f, 0.942160f, -0.0987296f, 0.0563244f, 0.278409f, -0.958810f, 
		-0.147575f, -0.741192f, 0.654871f, -0.239409f, -0.873024f, 0.424868f, 
		-0.675360f, 0.734032f, -0.0713160f, -0.257814f, 0.777232f, 0.573970f, 
		0.0746657f, 0.514981f, -0.853943f, 0.239409f, 0.873024f, -0.424868f, 
		0.147575f, 0.741192f, -0.654871f, 0.675360f, -0.734032f, 0.0713160f, 
		-0.0746657f, -0.514981f, 0.853943f, 0.257814f, -0.777232f, -0.573970f, 
		-0.122255f, -0.924182f, 0.361857f, 0.670223f, -0.698694f, 0.250256f, 
		0.0260745f, -0.788784f, -0.614117f, -0.669956f, -0.313539f, 0.672943f, 
		-0.206895f, 0.664775f, 0.717822f, -0.0790889f, 0.522768f, 0.848798f, 
		0.235232f, 0.0666190f, 0.969653f, 0.248863f, -0.206917f, 0.946178f, 
		-0.335539f, 0.653238f, 0.678744f, -0.669943f, -0.313599f, 0.672928f, 
		-0.183413f, 0.534604f, 0.824960f, 0.223018f, 0.0789242f, 0.971614f, 
		0.260567f, -0.176676f, 0.949152f, -0.304828f, 0.673448f, 0.673460f, 
		-0.669954f, -0.313600f, 0.672917f, -0.159970f, 0.537938f, 0.827667f, 
		0.235562f, 0.0649549f, 0.969686f, 0.248852f, -0.206945f, 0.946175f, 
		-0.158462f, 0.973790f, 0.163167f, -0.0147507f, 0.902583f, 0.430263f, 
		-0.106964f, 0.465472f, 0.878575f, -0.0422923f, 0.613370f, 0.788662f, 
		-0.561122f, -0.596632f, -0.573736f, -0.341849f, -0.352911f, 0.870972f, 
		-0.166613f, 0.983465f, 0.0709646f, 0.341849f, 0.352911f, -0.870972f, 
		0.557028f, 0.581007f, 0.593423f, 0.166613f, -0.983465f, -0.0709646f, 
		0.0708728f, -0.104305f, 0.992017f, 0.0404560f, -0.870468f, 0.490560f, 
		0.908273f, 0.418263f, 0.00984054f, 0.664431f, -0.745855f, 0.0472356f, 
		0.100564f, -0.343696f, 0.933681f, 0.125480f, -0.741363f, 0.659269f, 
		-0.207944f, 0.232192f, 0.950182f, 0.196427f, -0.628757f, 0.752384f, 
		0.196427f, -0.628757f, 0.752384f, 0.523215f, -0.851992f, 0.0188637f, 
		0.947916f, 0.316406f, 0.0366420f, -0.125684f, 0.982748f, 0.135683f, 
		-0.325532f, -0.943181f, -0.0666239f, 0.0640737f, 0.986951f, 0.147726f, 
		-0.468419f, -0.0312609f, -0.882953f, 0.125684f, -0.982748f, -0.135683f, 
		0.325532f, 0.943181f, 0.0666239f, 0.549407f, 0.0366355f, 0.834751f, 
		-0.0640737f, -0.986951f, -0.147726f, -0.477606f, 0.878170f, 0.0266237f, 
		-0.0258392f, 0.147865f, 0.988670f, 0.557934f, -0.829262f, -0.0321630f, 
		0.0258392f, -0.147865f, -0.988670f, -0.603676f, -0.0135773f, -0.797114f, 
		0.364299f, 0.0101701f, 0.931227f, -0.0237293f, 0.985391f, 0.168644f, 
		0.0237293f, -0.985391f, -0.168644f, 0.301402f, 0.554018f, -0.776029f, 
		0.121067f, 0.465457f, -0.876751f, 0.533634f, 0.751549f, -0.387827f, 
		0.529106f, 0.841088f, -0.112334f, -0.951147f, 0.144498f, -0.272835f, 
		0.181848f, 0.125809f, -0.975245f, -0.612027f, -0.366994f, 0.700527f, 
		-0.392273f, -0.264180f, 0.881096f, -0.217208f, -0.186501f, 0.958143f, 
		-0.141026f, -0.831942f, -0.536641f, 0.951147f, -0.144498f, 0.272835f, 
		0.781152f, 0.281128f, -0.557465f, 0.394504f, 0.135340f, -0.908873f, 
		0.670044f, 0.158288f, -0.725249f, 0.144271f, 0.880968f, 0.450645f, 
		0.0455397f, 0.991493f, 0.121937f, -0.307581f, 0.154274f, -0.938932f, 
		-0.115859f, 0.200450f, -0.972829f, -0.500208f, -0.0164852f, -0.865749f, 
		-0.0507832f, -0.997934f, -0.0393499f, 0.472610f, 0.0165502f, 0.881116f, 
		0.593276f, 0.0136230f, 0.804884f, 0.778023f, 0.0110509f, 0.628139f, 
		-0.473567f, 0.331507f, 0.815988f, -0.827794f, 0.331537f, -0.452594f, 
		-0.308933f, -0.689105f, -0.655511f, -0.475491f, -0.847589f, 0.235586f, 
		-0.514438f, -0.846650f, -0.136150f, 0.0777671f, -0.502002f, -0.861363f, 
		0.304646f, -0.460999f, -0.833469f, 0.862728f, -0.303472f, 0.404482f, 
		0.384630f, 0.418908f, -0.822542f, 0.384630f, 0.418908f, -0.822542f, 
		0.548818f, 0.828554f, 0.110892f, -0.181051f, -0.871866f, -0.455050f, 
		-0.294562f, 0.745619f, -0.597734f, -0.454372f, -0.801544f, -0.388681f, 
		-0.176152f, -0.597434f, 0.782332f, -0.787402f, -0.589504f, -0.180228f, 
		0.783028f, 0.609183f, -0.125551f, -0.181051f, -0.871866f, -0.455050f, 
		0.783028f, 0.609183f, -0.125551f, -0.159728f, -0.986771f, 0.0277353f, 
		-0.395192f, -0.918506f, 0.0130539f, -0.540218f, -0.841293f, 0.0197549f, 
		-0.101953f, -0.125085f, 0.986894f, 0.0913929f, 0.993451f, -0.0685779f, 
		0.791236f, 0.611225f, 0.0186622f, 0.101953f, 0.125085f, -0.986894f, 
		-0.158653f, 0.238065f, -0.958204f, 0.383044f, -0.717766f, -0.581454f, 
		0.279151f, -0.654809f, -0.702354f, 0.908273f, 0.418263f, 0.00984054f, 
		0.514253f, -0.846808f, 0.135869f, 0.578582f, -0.805566f, -0.127698f, 
		0.0685767f, 0.987004f, -0.145327f, -0.0471417f, -0.990399f, 0.129949f, 
		0.0685767f, 0.987004f, -0.145327f, 0.0685767f, 0.987004f, -0.145327f, 
		-0.0471417f, -0.990399f, 0.129949f, 0.0685767f, 0.987004f, -0.145327f, 
		0.389970f, 0.482024f, 0.784587f, 0.379224f, 0.212073f, 0.900674f, 
		0.763332f, -0.00848544f, 0.645951f, 0.515926f, 0.246256f, 0.820474f, 
		0.376853f, 0.343370f, 0.860278f, -0.230660f, 0.419771f, -0.877832f, 
		0.665551f, -0.550165f, -0.504342f, 0.947916f, 0.316406f, 0.0366420f, 
		0.439109f, -0.825982f, -0.353464f, 0.331163f, -0.894520f, -0.300274f, 
		-0.0310119f, -0.988576f, 0.147495f, -0.0310119f, -0.988576f, 0.147495f, 
		0.0326687f, 0.988637f, -0.146730f, -0.0310119f, -0.988576f, 0.147495f, 
		0.0326687f, 0.988637f, -0.146730f, -0.951147f, 0.144498f, -0.272835f, 
		-0.791368f, -0.419002f, 0.445168f, -0.0687155f, -0.997212f, -0.0291071f, 
		0.287091f, -0.116424f, 0.950802f, 0.951147f, -0.144498f, 0.272835f, 
		0.791368f, 0.419002f, -0.445168f, 0.287091f, -0.116424f, 0.950802f, 
		-0.256183f, 0.211398f, -0.943229f, 0.0972877f, 0.994556f, 0.0373187f, 
		0.191877f, -0.861957f, -0.469268f, 0.104210f, -0.792723f, -0.600608f, 
		-0.462576f, 0.686986f, 0.560422f, -0.462576f, 0.686986f, 0.560422f, 
		0.421241f, -0.871455f, -0.251240f, 0.558122f, -0.812549f, -0.168116f, 
		0.337037f, -0.850992f, -0.402763f, 0.000240439f, -1.00000f, 0.000906522f, 
		-0.121009f, 0.679871f, 0.723279f, 0.247301f, -0.965424f, -0.0824496f, 
		-0.995824f, -0.0431320f, -0.0804673f, -0.314911f, 0.114706f, 0.942164f, 
		0.247301f, -0.965424f, -0.0824496f, 0.0487566f, 0.969054f, 0.241988f, 
		0.999370f, 0.00862993f, -0.0344359f, 0.314911f, -0.114706f, -0.942164f, 
		-0.187239f, -0.668498f, -0.719758f, -0.296133f, -0.954524f, -0.0344881f, 
		-0.993739f, 0.0793567f, 0.0786495f, -0.0271522f, -0.874792f, 0.483737f, 
		0.195330f, 0.980370f, 0.0268334f, -0.157065f, -0.928123f, 0.337519f, 
		-0.185491f, 0.195745f, 0.962952f, 0.898114f, -0.00714133f, 0.439704f, 
		0.204039f, 0.975530f, -0.0819095f, 0.00899385f, -0.447824f, -0.894077f, 
		0.242072f, 0.954764f, -0.172702f, -0.256956f, -0.927685f, -0.270878f, 
		-0.234826f, -0.972019f, -0.00604553f, -0.953326f, -0.0499208f, -0.297786f, 
		-0.234826f, -0.972019f, -0.00604553f, -0.216902f, -0.962867f, 0.160748f, 
		-0.236803f, 0.230387f, 0.943847f, 0.974759f, -0.182594f, 0.128469f, 
		0.314911f, -0.114706f, -0.942164f, 0.210623f, 0.974387f, -0.0787941f, 
		-0.593360f, -0.799504f, -0.0933644f, -0.000350119f, -0.000255988f, -1.00000f, 
		0.000464169f, 0.000245549f, 1.00000f, 0.387318f, 0.916895f, 0.0963735f, 
		-0.202309f, -0.956899f, -0.208366f, -0.870325f, 0.492373f, -0.0101676f, 
		0.223872f, 0.957622f, 0.181221f, -0.00398039f, 0.0641885f, -0.997930f, 
		0.00398039f, -0.0641885f, 0.997930f, 0.870325f, -0.492373f, 0.0101676f, 
		-0.121366f, -0.876404f, 0.466033f, 0.412273f, -0.446983f, -0.793875f, 
		0.700903f, -0.685664f, 0.196471f, 0.0560799f, -0.854564f, -0.516310f, 
		0.519111f, -0.414570f, -0.747433f, 0.237687f, -0.763244f, -0.600803f, 
		0.514560f, -0.157333f, -0.842896f, -0.742655f, -0.0786627f, -0.665038f, 
		-0.742655f, -0.0786627f, -0.665038f, -0.0590243f, -0.664909f, -0.744588f, 
		-0.0590243f, -0.664909f, -0.744588f, -0.000669235f, -0.999998f, -0.00168495f, 
		-0.127613f, 0.603469f, -0.787109f, -0.227842f, -0.863693f, 0.449580f, 
		-0.357084f, -0.589137f, 0.724851f, -0.245051f, -0.914673f, 0.321440f, 
		0.227842f, 0.863693f, -0.449580f, 0.357084f, 0.589137f, -0.724851f, 
		0.245051f, 0.914673f, -0.321440f, 0.823077f, 0.400387f, 0.402784f, 
		-0.0847306f, 0.707390f, 0.701726f, 0.297273f, 0.823655f, -0.482930f, 
		0.762098f, -0.330732f, -0.556618f, 0.762098f, -0.330732f, -0.556618f, 
		0.233868f, 0.624137f, -0.745492f, 0.233868f, 0.624137f, -0.745492f, 
		-0.159424f, 0.140361f, 0.977181f, 0.110762f, -0.963279f, 0.244592f, 
		0.113526f, 0.816806f, -0.565633f, -0.946204f, 0.310156f, 0.0922054f, 
		-0.774459f, 0.627439f, 0.0808266f, -0.998531f, -0.0386086f, 0.0380254f, 
		-0.303679f, -0.0755797f, 0.949772f, 0.974759f, -0.182594f, 0.128469f, 
		0.303679f, 0.0755797f, -0.949772f, -0.573882f, 0.807555f, 0.136067f, 
		-0.0621628f, 0.425696f, -0.902728f, -0.547209f, -0.828731f, 0.117332f, 
		7.62364e-005f, 9.71014e-005f, 1.00000f, -1.99664e-005f, 1.88126e-005f, -1.00000f, 
		0.497408f, 0.862238f, -0.0955526f, -0.294307f, -0.907122f, -0.300854f, 
		0.124186f, 0.297262f, -0.946685f, -0.797966f, -0.571542f, -0.191287f, 
		-0.422798f, -0.840388f, -0.339100f, -0.970100f, -0.166228f, -0.176848f, 
		-0.970100f, -0.166228f, -0.176848f, -0.998390f, -0.0546436f, -0.0152152f, 
		-0.975262f, 0.120658f, 0.185217f, -0.132016f, 0.970430f, 0.202085f, 
		-0.221941f, 0.137548f, 0.965310f, 0.395110f, -0.916238f, 0.0662977f, 
		0.898114f, -0.00714133f, 0.439704f, 0.333389f, -0.942710f, -0.0122485f, 
		0.221941f, -0.137548f, -0.965310f, 0.243648f, -0.965451f, -0.0924136f, 
		-0.0752686f, -0.0701564f, 0.994692f, -0.204287f, 0.546552f, 0.812125f, 
		0.716938f, 0.537407f, 0.444065f, -0.255438f, -0.0248715f, -0.966505f, 
		-0.209655f, 0.132275f, -0.968787f, -0.226138f, 0.0575742f, -0.972392f, 
		-0.0788277f, 0.326868f, -0.941777f, -0.381861f, -0.919187f, -0.0963183f, 
		-0.0368209f, -0.317230f, 0.947634f, -0.0368209f, -0.317230f, 0.947634f, 
		-0.0788277f, 0.326868f, -0.941777f, 0.746156f, 0.134336f, -0.652078f, 
		0.594829f, 0.798497f, 0.0926332f, -0.0142596f, 0.769614f, 0.638350f, 
		-0.109668f, 0.620629f, 0.776397f, -0.554670f, -0.815397f, -0.165738f, 
		-0.396154f, -0.873368f, -0.283355f, 0.109668f, -0.620629f, -0.776397f, 
		0.0142596f, -0.769614f, -0.638350f, 0.554670f, 0.815397f, 0.165738f, 
		0.396154f, 0.873368f, 0.283355f, 0.0561919f, -0.971528f, -0.230163f, 
		-0.103397f, 0.970718f, 0.216829f, 0.0561919f, -0.971528f, -0.230163f, 
		0.163334f, 0.827279f, -0.537523f, 0.163334f, 0.827279f, -0.537523f, 
		-0.123058f, -0.863620f, 0.488895f, 0.163334f, 0.827279f, -0.537523f, 
		0.163334f, 0.827279f, -0.537523f, -0.227245f, -0.670723f, -0.706039f, 
		0.295256f, 0.941609f, -0.161856f, -0.839803f, -0.503443f, -0.203166f, 
		-0.0856857f, -0.815237f, 0.572754f, 0.220374f, 0.973003f, -0.0685562f, 
		-0.0856857f, -0.815237f, 0.572754f, -0.0271165f, 0.342762f, 0.939031f, 
		0.569548f, 0.796176f, -0.204254f, 0.397426f, 0.907316f, -0.137222f, 
		0.451454f, 0.864704f, -0.220171f, -0.0252872f, 0.0162252f, 0.999549f, 
		-0.00180888f, 0.999993f, 0.00322412f, 0.134504f, 0.817573f, 0.559896f, 
		0.0275879f, 0.677519f, 0.734987f, 0.443207f, 0.896018f, 0.0268260f, 
		0.735391f, 0.677160f, 0.0255847f, 0.768695f, 0.532289f, 0.354649f, 
		0.243595f, 0.670660f, 0.700626f, 0.0340721f, 0.0819798f, -0.996051f, 
		-0.0117957f, -0.0836244f, 0.996428f, 0.0340721f, 0.0819798f, -0.996051f, 
		-0.0117957f, -0.0836244f, 0.996428f, 0.0127403f, -0.0816832f, -0.996577f, 
		-0.0322808f, 0.0769196f, 0.996515f, 0.0127403f, -0.0816832f, -0.996577f, 
		-0.0322808f, 0.0769196f, 0.996515f, -0.0338812f, -0.0813464f, -0.996110f, 
		0.0118279f, 0.0822020f, 0.996545f, 0.0118279f, 0.0822020f, 0.996545f, 
		-0.0338812f, -0.0813464f, -0.996110f, -0.0562037f, 0.971528f, 0.230161f, 
		0.103403f, -0.970720f, -0.216820f, -0.0562037f, 0.971528f, 0.230161f, 
		-0.110663f, 0.968041f, -0.225055f, 0.0568742f, -0.968279f, 0.243312f, 
		0.0568742f, -0.968279f, 0.243312f, 0.420145f, -0.804341f, 0.420136f, 
		0.409269f, -0.748179f, 0.522232f, 0.500418f, -0.794217f, 0.344674f, 
		0.559260f, -0.790635f, 0.249247f, 0.647128f, -0.744107f, 0.165922f, 
		0.934174f, -0.212996f, 0.286273f, 0.550409f, 0.246888f, -0.797556f, 
		0.722292f, 0.461826f, -0.514791f, 0.211975f, 0.929155f, 0.302882f, 
		0.463696f, 0.117361f, -0.878187f, -0.111173f, -0.178341f, -0.977668f, 
		-0.628623f, 0.662135f, 0.407934f, 0.387937f, -0.859087f, 0.333879f, 
		0.596825f, -0.679401f, 0.426866f, -0.628623f, 0.662135f, 0.407934f, 
		0.549520f, -0.772245f, 0.318851f, 0.630219f, -0.758685f, 0.164991f, 
		-0.00927709f, -0.889928f, -0.456007f, 0.0323143f, -0.881461f, -0.471150f, 
		0.0241288f, -0.626776f, -0.778826f, -0.544121f, 0.565822f, 0.619498f, 
		-0.544121f, 0.565822f, 0.619498f, 0.388115f, -0.859210f, 0.333354f, 
		0.619927f, -0.647538f, 0.443154f, 0.556727f, -0.756966f, 0.342136f, 
		0.539814f, -0.841730f, 0.00957027f, 0.197023f, -0.803035f, -0.562420f, 
		-0.599929f, 0.0387508f, 0.799114f, -0.814910f, -0.0582579f, -0.576652f, 
		0.623767f, 0.0479167f, -0.780140f, 0.0695748f, -0.823188f, -0.563490f, 
		-0.0695748f, 0.823188f, 0.563490f, -0.599940f, 0.0387519f, 0.799106f, 
		-0.814910f, -0.0582579f, -0.576652f, 0.623780f, 0.0479157f, -0.780130f, 
		0.0695662f, -0.823173f, -0.563513f, -0.0695662f, 0.823173f, 0.563513f, 
		-0.0730290f, -0.117172f, 0.990423f, -0.318041f, -0.388414f, 0.864861f, 
		-0.387036f, -0.473556f, 0.791169f, -0.547161f, -0.404248f, 0.732938f, 
		-0.765605f, -0.304811f, 0.566515f, -0.0286732f, -0.999565f, -0.00683764f, 
		0.351125f, 0.482338f, -0.802534f, 0.351125f, 0.482338f, -0.802534f, 
		0.633657f, 0.353288f, -0.688235f, 0.836429f, 0.230402f, -0.497293f, 
		0.0286732f, 0.999565f, 0.00683764f, -0.540218f, -0.841293f, 0.0197549f, 
		0.0589799f, 0.0738626f, -0.995523f, -0.730094f, -0.682250f, 0.0387035f, 
		-0.894142f, -0.447085f, 0.0250165f, 0.398311f, 0.917091f, -0.0171204f, 
		0.164406f, 0.985824f, -0.0334898f, 0.732507f, 0.680537f, -0.0173894f, 
		-0.0589799f, -0.0738626f, 0.995523f, 0.894142f, 0.447085f, -0.0250165f, 
		0.0589799f, 0.0738626f, -0.995523f, -0.0589799f, -0.0738626f, 0.995523f, 
		-0.964574f, -0.00115738f, 0.263809f, -0.117218f, -0.915209f, 0.385554f, 
		-0.216236f, 0.00124455f, 0.976340f, 0.965273f, 2.87344e-005f, -0.261244f, 
		0.218208f, -8.59470e-005f, -0.975902f, 0.117218f, 0.915209f, -0.385554f, 
		0.119700f, 0.984317f, -0.129585f, -0.965232f, -5.06398e-005f, 0.261395f, 
		-0.119700f, -0.984317f, 0.129585f, -0.214137f, 9.15954e-005f, 0.976804f, 
		0.965882f, 0.00116092f, -0.258979f, 0.216173f, -0.00122861f, -0.976354f, 
		-0.437030f, 0.640691f, 0.631284f, -0.281296f, 0.357995f, 0.890344f, 
		-0.438831f, 0.379827f, 0.814346f, -0.489410f, -0.327183f, 0.808350f, 
		-0.459499f, -0.483531f, 0.745022f, -0.459499f, -0.483531f, 0.745022f, 
		-0.906140f, -0.0388501f, -0.421190f, -0.00435899f, -0.0595208f, 0.998218f, 
		-0.0490114f, -0.995450f, 0.0817152f, 0.0510682f, 0.00978344f, -0.998647f, 
		-0.913578f, 0.321449f, 0.249090f, -0.0803034f, 0.136224f, -0.987418f, 
		-0.194716f, 0.977586f, -0.0800696f, 0.913578f, -0.321449f, -0.249090f, 
		0.158980f, -0.972828f, 0.168321f, 0.0803034f, -0.136224f, 0.987418f, 
		-0.480439f, -0.873750f, 0.0757596f, -0.939175f, -0.188920f, -0.286810f, 
		-0.0307205f, 0.0188343f, 0.999351f, 0.480439f, 0.873750f, -0.0757596f, 
		0.0307205f, -0.0188343f, -0.999351f, 0.939175f, 0.188920f, 0.286810f, 
		-0.0470955f, -0.343427f, -0.937998f, -0.519771f, -0.853988f, -0.0232936f, 
		-0.722565f, -0.690175f, -0.0394787f, -0.191212f, -0.216648f, -0.957341f, 
		-0.888489f, -0.458725f, -0.0125667f, -0.266517f, 0.00279138f, -0.963826f, 
		0.519771f, 0.853988f, 0.0232936f, 0.0470955f, 0.343427f, 0.937998f, 
		0.722565f, 0.690175f, 0.0394787f, 0.888489f, 0.458725f, 0.0125667f, 
		0.191212f, 0.216648f, 0.957341f, 0.266517f, -0.00279138f, 0.963826f, 
		-0.141947f, -0.232146f, 0.962268f, -0.965973f, -0.000574806f, 0.258643f, 
		0.0686226f, 0.123322f, -0.989991f, 0.965855f, 0.00138381f, -0.259079f, 
		0.0255110f, -0.998236f, -0.0536071f, -0.0255110f, 0.998236f, 0.0536071f, 
		0.469106f, -0.427446f, 0.772806f, 0.513590f, 0.854720f, 0.0753637f, 
		0.311996f, 0.946666f, 0.0805075f, 0.820710f, -0.271432f, -0.502752f, 
		0.759958f, 0.645821f, 0.0733391f, -0.268525f, -0.962387f, -0.0412911f, 
		0.0407465f, 0.985685f, -0.163599f, -0.510226f, 0.229294f, -0.828911f, 
		-0.150331f, 0.956687f, -0.249301f, 0.268525f, 0.962387f, 0.0412911f, 
		0.150331f, -0.956687f, 0.249301f, -0.0407465f, -0.985685f, 0.163599f, 
		0.510226f, -0.229294f, 0.828911f, -0.0882271f, 0.634182f, -0.768133f, 
		0.00618942f, 0.789044f, -0.614306f, 0.586355f, 0.395468f, -0.706961f, 
		0.677522f, 0.649079f, -0.345918f, 0.111144f, 0.564241f, -0.818095f, 
		0.360272f, 0.895209f, -0.262306f, -0.196914f, 0.404105f, 0.893266f, 
		-0.488645f, 0.158090f, 0.858040f, 0.0279370f, -0.445084f, -0.895053f, 
		0.306402f, -0.365829f, -0.878798f, 0.553722f, -0.166742f, -0.815836f, 
		0.00303201f, 0.998709f, -0.0507001f, -0.0256194f, -0.996976f, 0.0733708f, 
		-0.0279370f, 0.445084f, 0.895053f, -0.803643f, 0.222501f, -0.551952f, 
		-0.00205119f, 0.0536712f, -0.998557f, -0.111312f, 0.950164f, -0.291202f, 
		0.782033f, -0.234218f, 0.577552f, 0.00205119f, -0.0536712f, 0.998557f, 
		0.111312f, -0.950164f, 0.291202f, -0.0350247f, 0.922303f, 0.384877f, 
		0.0255679f, 0.962207f, -0.271115f, 0.0850861f, 0.607032f, 0.790109f, 
		-0.0850861f, -0.607032f, -0.790109f, 0.883744f, -0.00820995f, 0.467900f, 
		-0.883744f, 0.00820995f, -0.467900f, -0.229143f, -0.493001f, -0.839312f, 
		-0.788100f, -0.505808f, 0.350794f, 0.00457924f, -0.996351f, 0.0852301f, 
		0.229143f, 0.493001f, 0.839312f, -0.0333644f, -0.316199f, 0.948106f, 
		0.858783f, 0.403695f, -0.315470f, 0.0333644f, 0.316199f, -0.948106f, 
		-0.00457924f, 0.996351f, -0.0852301f, -0.0282203f, 0.184009f, -0.982519f, 
		0.0647855f, -0.161732f, 0.984706f, 0.0647855f, -0.161732f, 0.984706f, 
		0.0647855f, -0.161732f, 0.984706f, -0.0282203f, 0.184009f, -0.982519f, 
		0.0295400f, 0.196973f, 0.979964f, -0.0687862f, -0.173428f, -0.982441f, 
		0.0295400f, 0.196973f, 0.979964f, -0.0687862f, -0.173428f, -0.982441f, 
		-0.694408f, -0.105308f, 0.711834f, 0.105297f, -0.978438f, -0.177684f, 
		-0.105297f, 0.978438f, 0.177684f, 0.694408f, 0.105308f, -0.711834f, 
		0.105297f, -0.978438f, -0.177684f, -0.105297f, 0.978438f, 0.177684f, 
		0.0289151f, -0.978937f, 0.202105f, -0.0768865f, 0.978907f, -0.189288f, 
		0.0289151f, -0.978937f, 0.202105f, -0.0768865f, 0.978907f, -0.189288f, 
		0.924264f, -0.313588f, -0.217713f, 0.420305f, 0.140780f, 0.896396f, 
		0.707460f, 0.334098f, 0.622799f, 0.634732f, 0.473973f, 0.610299f, 
		0.634732f, 0.473973f, 0.610299f, 0.740182f, 0.433989f, 0.513599f, 
		0.138460f, 0.958368f, -0.249717f, 0.522806f, 0.467475f, 0.712840f, 
		-0.221029f, 0.0159663f, -0.975136f, -0.000664209f, 0.999987f, -0.00513108f, 
		-0.0766034f, -0.0159646f, -0.996934f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		8.16089e-006f, -1.00000f, 2.51535e-006f, -0.0682743f, 0.892740f, 0.445369f, 
		0.0682743f, -0.892740f, -0.445369f, 0.840514f, -0.0388099f, 0.540398f, 
		-0.845495f, 0.0367465f, 0.532717f, -0.822770f, -0.0319835f, -0.567474f, 
		-0.0682501f, 0.892846f, 0.445161f, 0.0682501f, -0.892846f, -0.445161f, 
		0.840256f, -0.0376341f, 0.540882f, -0.599929f, 0.0387508f, 0.799114f, 
		-0.814910f, -0.0582579f, -0.576652f, -0.0682501f, 0.892846f, 0.445161f, 
		0.0682501f, -0.892846f, -0.445161f, 0.840256f, -0.0376341f, 0.540882f, 
		-0.599940f, 0.0387519f, 0.799106f, -0.814910f, -0.0582579f, -0.576652f, 
		0.711889f, -0.363113f, -0.601135f, 0.840885f, -0.541109f, 0.0106424f, 
		0.356262f, -0.175795f, -0.917700f, 0.784396f, -0.588763f, -0.195143f, 
		0.253073f, -0.0158285f, -0.967318f, 0.654944f, 0.642109f, 0.398427f, 
		-0.127532f, 0.975537f, -0.179059f, 0.166730f, -0.975355f, 0.144511f, 
		0.701823f, 0.206381f, -0.681800f, 0.524150f, 0.287030f, -0.801798f, 
		0.296989f, -0.798217f, -0.524068f, 0.301419f, 0.439446f, -0.846188f, 
		-0.845495f, 0.0367465f, 0.532717f, -0.822770f, -0.0319835f, -0.567474f, 
		0.873782f, 0.0434465f, -0.484373f, -0.00308536f, -0.707652f, -0.706555f, 
		0.00308536f, 0.707652f, 0.706555f, 0.554738f, -0.747103f, -0.366200f, 
		0.531294f, -0.0979091f, 0.841511f, 0.755194f, -0.206229f, 0.622215f, 
		0.335591f, -0.0478376f, 0.940792f, 0.708537f, 0.664289f, -0.238108f, 
		-0.125416f, -0.748039f, 0.651697f, 0.114127f, 0.830155f, 0.545727f, 
		0.100909f, -0.873128f, -0.476933f, 0.921914f, -0.0315152f, 0.386110f, 
		-0.906140f, -0.0388501f, -0.421190f, -0.0983729f, 0.0408947f, 0.994309f, 
		-0.114282f, -0.960413f, 0.254058f, 0.114282f, 0.960413f, -0.254058f, 
		-0.0432002f, 0.866071f, 0.498051f, 0.0432002f, -0.866071f, -0.498051f, 
		0.916860f, -0.00587500f, 0.399165f, -0.916860f, 0.00587500f, -0.399165f, 
		0.573047f, -0.541554f, -0.615090f, 0.656100f, 0.751447f, 0.0697115f, 
		0.472141f, 0.880233f, 0.0476711f, 0.327082f, 0.943426f, 0.0544457f, 
		0.271269f, -0.377001f, 0.885598f, -0.383551f, 0.915089f, -0.124502f, 
		-0.961703f, 0.255831f, -0.0983757f, -0.000121976f, 7.84956e-005f, -1.00000f, 
		0.383551f, -0.915089f, 0.124502f, 0.000129526f, 5.07625e-005f, 1.00000f, 
		0.964639f, -0.247036f, 0.0918954f, 0.925308f, -0.0156321f, 0.378893f, 
		-0.925308f, 0.0156321f, -0.378893f, 0.0421647f, 0.649187f, -0.759459f, 
		-0.0168592f, -0.400850f, 0.915989f, -0.857352f, -0.0926484f, 0.506325f, 
		-0.696206f, -0.130557f, 0.705870f, -0.492193f, -0.0916391f, 0.865649f, 
		-0.000173744f, -1.00000f, 5.23816e-005f, 0.857349f, 0.0926523f, -0.506328f, 
		0.696206f, 0.130557f, -0.705870f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		0.492193f, 0.0916391f, -0.865649f, -0.857271f, -0.0922289f, 0.506537f, 
		-0.696121f, -0.130341f, 0.705994f, -0.491993f, -0.0916132f, 0.865765f, 
		-0.000173744f, -1.00000f, 5.23816e-005f, 0.857271f, 0.0922289f, -0.506537f, 
		0.696121f, 0.130341f, -0.705994f, 3.58729e-006f, 1.00000f, -9.35137e-007f, 
		0.491993f, 0.0916132f, -0.865765f, -0.857352f, -0.0926484f, 0.506325f, 
		-0.696229f, -0.130543f, 0.705849f, -0.492235f, -0.0916134f, 0.865628f, 
		-0.000173744f, -1.00000f, 5.23816e-005f, 0.857349f, 0.0926523f, -0.506328f, 
		0.696229f, 0.130543f, -0.705849f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		0.492235f, 0.0916134f, -0.865628f, -0.0353357f, -0.969578f, 0.242219f, 
		-0.890599f, -0.0121175f, 0.454628f, -0.325532f, -0.943181f, -0.0666239f, 
		0.244486f, 0.962664f, -0.116206f, 0.0353357f, 0.969578f, -0.242219f, 
		0.850250f, 0.0125917f, -0.526228f, 0.325532f, 0.943181f, 0.0666239f, 
		-0.244486f, -0.962664f, 0.116206f, -0.409526f, 0.591824f, -0.694286f, 
		-0.508954f, 0.450043f, -0.733776f, -0.720694f, 0.135172f, -0.679948f, 
		-0.686014f, 0.243577f, -0.685605f, -0.792860f, 0.0653677f, -0.605888f, 
		-0.245900f, 0.965111f, 0.0899647f, -0.963928f, 0.256489f, 0.0711106f, 
		0.256603f, -0.961551f, -0.0978511f, -0.000421314f, -1.74077e-005f, -1.00000f, 
		0.000472786f, 0.000154452f, 1.00000f, 0.917790f, -0.374042f, -0.133245f, 
		0.521061f, 0.849510f, -0.0826286f, -0.113592f, -0.891336f, 0.438882f, 
		-0.888639f, 0.458516f, 0.00914391f, -0.153397f, 0.0668319f, 0.985902f, 
		0.851255f, -0.524726f, -0.00526451f, 0.167720f, -0.0547503f, -0.984313f, 
		-0.115461f, 0.0444448f, -0.992317f, -0.888639f, 0.458586f, -0.00432570f, 
		0.150033f, -0.0368427f, 0.987994f, 0.843864f, -0.536525f, 0.00578573f, 
		-0.887718f, -0.00980765f, -0.460282f, -0.272989f, -0.952825f, -0.132674f, 
		0.847031f, 0.00758061f, 0.531489f, 0.272989f, 0.952825f, 0.132674f, 
		-0.713859f, -0.113556f, -0.691021f, 0.0404373f, -0.995958f, 0.0801973f, 
		-0.0404373f, 0.995958f, -0.0801973f, 0.198644f, -0.450295f, 0.870503f, 
		0.713859f, 0.113556f, 0.691021f, -0.0404373f, 0.995958f, -0.0801973f, 
		-0.198644f, 0.450295f, -0.870503f, 0.0404373f, -0.995958f, 0.0801973f, 
		0.438319f, -0.241826f, 0.865677f, 0.515233f, -0.426196f, 0.743567f, 
		0.618573f, -0.293938f, 0.728675f, 0.789200f, 0.00350287f, 0.614126f, 
		0.228127f, 0.350493f, 0.908357f, 0.213631f, 0.640010f, 0.738071f, 
		0.00172158f, -0.999997f, 0.00156108f, 0.234850f, 0.480578f, 0.844920f, 
		0.282419f, 0.618692f, 0.733116f, 0.264357f, 0.720228f, 0.641394f, 
		0.381249f, 0.269028f, 0.884462f, 0.482676f, 0.147838f, 0.863231f, 
		0.265290f, 0.283421f, 0.921571f, 0.554936f, 0.831857f, 0.00773095f, 
		0.149526f, 0.0175110f, -0.988603f, 1.73729e-005f, -1.00000f, 2.61491e-006f, 
		0.000000f, 1.00000f, -5.78099e-015f, -2.28478e-006f, 1.00000f, 3.05527e-006f, 
		0.916860f, -0.00587500f, 0.399165f, -0.916860f, 0.00587500f, -0.399165f, 
		-0.0711042f, -0.425268f, 0.902270f, 0.0711042f, 0.425268f, -0.902270f, 
		-0.205284f, 0.923587f, -0.323799f, -0.397273f, 0.686086f, -0.609475f, 
		-0.126322f, 0.986360f, -0.105532f, -0.000669430f, -0.992718f, 0.120463f, 
		0.306102f, -0.731034f, 0.609828f, 0.153989f, -0.937081f, 0.313317f, 
		-0.222371f, -0.00281921f, 0.974958f, 3.58729e-006f, 1.00000f, -9.35137e-007f, 
		8.16089e-006f, -1.00000f, 2.51535e-006f, -0.135747f, 0.658965f, 0.739823f, 
		-0.413783f, 0.653710f, -0.633598f, -0.571057f, 0.663901f, -0.482834f, 
		-0.667045f, 0.744858f, 0.0153935f, -0.739987f, 0.653540f, 0.159072f, 
		-0.739987f, 0.653540f, 0.159072f, 0.136945f, 0.0364886f, 0.989906f, 
		-1.73730e-005f, 1.00000f, -2.61493e-006f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		8.16089e-006f, -1.00000f, 2.51535e-006f, -0.0164971f, 0.988804f, 0.148307f, 
		0.142331f, -0.550037f, 0.822922f, 0.341254f, 0.891762f, -0.297164f, 
		0.000885017f, 0.120555f, 0.992706f, -0.153763f, 0.313451f, 0.937073f, 
		0.126170f, -0.105621f, -0.986370f, -0.305869f, 0.610015f, 0.730976f, 
		0.397096f, -0.609626f, -0.686055f, 0.205106f, -0.323916f, -0.923585f, 
		0.0617405f, 0.958326f, -0.278927f, 0.0857126f, 0.660947f, -0.745521f, 
		0.492242f, -0.865842f, 0.0895318f, 0.710766f, -0.701310f, 0.0545586f, 
		0.872203f, -0.482343f, 0.0812857f, 0.260302f, -0.131275f, -0.956561f, 
		-0.0838897f, -0.673520f, -0.734393f, 0.883744f, -0.00820995f, 0.467900f, 
		-0.883744f, 0.00820995f, -0.467900f, -0.0144935f, 0.323919f, -0.945974f, 
		0.0500982f, -0.631260f, 0.773951f, 0.134622f, -0.659865f, -0.739226f, 
		0.311450f, -0.308550f, 0.898775f, -0.153670f, 0.537878f, -0.828898f, 
		-0.234790f, 0.967169f, 0.0972511f, -0.914443f, 0.373451f, 0.155975f, 
		0.245894f, -0.965041f, -0.0907344f, 0.868288f, -0.488408f, -0.0867936f, 
		0.214741f, -0.0744211f, 0.973832f, -0.0991790f, 0.317473f, -0.943066f, 
		-0.424174f, -0.535089f, -0.730586f, -0.00107953f, -0.981515f, -0.191384f, 
		-0.287743f, -0.730548f, -0.619277f, -0.0409710f, 0.999136f, -0.00697310f, 
		0.510244f, 0.534551f, 0.673726f, 0.0354897f, -0.999357f, -0.00516155f, 
		0.395339f, 0.686033f, 0.610791f, -0.00363309f, 0.983380f, 0.181524f, 
		7.46783e-005f, -1.00000f, -0.000160924f, -0.0724347f, 0.358658f, 0.930654f, 
		-0.000363313f, 0.999998f, 0.00185961f, 0.103314f, -0.351906f, 0.930317f, 
		0.0663028f, 0.959282f, 0.274559f, -0.771116f, -0.00761158f, -0.636649f, 
		-0.965232f, -5.06398e-005f, 0.261395f, -0.120648f, -0.985351f, 0.120529f, 
		0.778023f, 0.0110509f, 0.628139f, 0.964657f, -0.00110345f, -0.263507f, 
		0.120648f, 0.985351f, -0.120529f, -0.997917f, -0.0615612f, 0.0192675f, 
		-0.286883f, 0.0407918f, 0.957097f, -0.938776f, 0.343664f, 0.0243865f, 
		-0.850359f, 0.526153f, 0.00722137f, 0.997917f, 0.0615612f, -0.0192675f, 
		0.981797f, -0.188115f, -0.0262196f, 0.286883f, -0.0407918f, -0.957097f, 
		0.912710f, -0.408599f, -0.00252854f, -0.258876f, 0.952581f, 0.159912f, 
		0.0151890f, -0.999856f, -0.00759593f, -0.510652f, 0.829707f, 0.225436f, 
		-0.990584f, -0.135556f, 0.0191973f, -0.422084f, 0.819597f, 0.387434f, 
		-0.120739f, 0.983953f, 0.131372f, -0.371912f, 0.350469f, 0.859565f, 
		-0.307325f, 0.0882314f, 0.947505f, -0.0151890f, 0.999856f, 0.00759593f, 
		0.258876f, -0.952581f, -0.159913f, 0.983758f, -0.174161f, -0.0434560f, 
		0.994603f, 0.0858254f, 0.0582922f, 0.706603f, -0.681704f, -0.189717f, 
		0.392233f, -0.897523f, -0.201510f, 0.249745f, -0.197280f, -0.948002f, 
		0.120739f, -0.983953f, -0.131372f, 0.0891139f, 0.995989f, 0.00805879f, 
		-0.777788f, -0.0110403f, -0.628430f, -0.965882f, 0.00105610f, 0.258979f, 
		-0.0891139f, -0.995989f, -0.00805879f, 0.784420f, 0.00746774f, 0.620184f, 
		0.965273f, 2.87344e-005f, -0.261244f, -0.0432002f, 0.866071f, 0.498051f, 
		0.0432002f, -0.866071f, -0.498051f, 0.916860f, -0.00587500f, 0.399165f, 
		-0.916860f, 0.00587500f, -0.399165f, 0.114127f, 0.830155f, 0.545727f, 
		-0.0891380f, 0.751039f, -0.654213f, 0.100909f, -0.873128f, -0.476933f, 
		0.921914f, -0.0315152f, 0.386110f, -0.906140f, -0.0388501f, -0.421190f, 
		0.0218838f, 0.0408949f, -0.998924f, 0.0850861f, 0.607032f, 0.790109f, 
		-0.0850861f, -0.607032f, -0.790109f, 0.833602f, -0.000708099f, 0.552365f, 
		-0.833602f, 0.000708099f, -0.552365f, 0.000554996f, 0.999993f, -0.00364879f, 
		0.0895437f, -0.374590f, -0.922857f, -0.894142f, -0.447085f, 0.0250165f, 
		-0.966181f, -0.257778f, 0.00669583f, -0.997917f, -0.0615612f, 0.0192675f, 
		-0.180703f, -0.00535915f, 0.983523f, 0.894142f, 0.447085f, -0.0250165f, 
		0.966181f, 0.257778f, -0.00669583f, 0.997917f, 0.0615612f, -0.0192675f, 
		0.166283f, 0.0242881f, -0.985779f, 2.26994e-005f, -1.00000f, 0.000172227f, 
		-0.100935f, 0.324857f, -0.940362f, 0.287436f, 0.0244466f, 0.957488f, 
		0.255768f, 0.937647f, 0.235373f, -0.190817f, -0.0414097f, -0.980752f, 
		-0.888489f, -0.458725f, -0.0125667f, -0.964157f, -0.265290f, 0.00465507f, 
		-0.998574f, -0.0518474f, -0.0127284f, 0.888489f, 0.458725f, 0.0125667f, 
		0.274432f, 0.171547f, 0.946181f, 0.964157f, 0.265290f, -0.00465507f, 
		0.998574f, 0.0518474f, 0.0127284f, -0.998574f, -0.0518474f, -0.0127284f, 
		-0.445712f, 0.0475472f, -0.893913f, -0.933510f, 0.357731f, -0.0242288f, 
		-0.842754f, 0.538226f, -0.00882311f, 0.998574f, 0.0518474f, 0.0127284f, 
		0.978133f, -0.206755f, 0.0225407f, 0.445712f, -0.0475472f, 0.893913f, 
		0.904256f, -0.426991f, -0.000200422f, 0.105941f, -0.993664f, -0.0375220f, 
		-0.850149f, 0.000000f, -0.526542f, -0.105941f, 0.993664f, 0.0375220f, 
		0.956100f, 0.0604537f, 0.286737f, -0.903130f, 0.000000f, -0.429367f, 
		-0.958248f, -0.0480381f, -0.281876f, -0.996353f, -0.0681040f, -0.0514095f, 
		0.112898f, 0.992216f, -0.0525465f, 0.872016f, 0.000000f, 0.489478f, 
		-0.112898f, -0.992216f, 0.0525465f, 0.840514f, -0.0388099f, 0.540398f, 
		-0.845495f, 0.0367465f, 0.532717f, 0.873782f, 0.0434465f, -0.484373f, 
		-0.0157402f, 0.771364f, 0.636199f, 0.0157402f, -0.771364f, -0.636199f, 
		0.119530f, -0.696835f, -0.707202f, -0.851784f, -0.0102828f, 0.523793f, 
		-0.928442f, 0.00126829f, 0.371475f, -0.999758f, -0.00509013f, 0.0213972f, 
		0.904527f, 0.0157073f, -0.426128f, -0.196789f, -0.978563f, 0.0607303f, 
		0.969726f, -0.00147817f, -0.244191f, 0.999758f, 0.00509013f, -0.0213972f, 
		0.172623f, 0.982436f, -0.0708629f, -0.124605f, -0.964140f, -0.234324f, 
		-0.0680107f, -0.873180f, -0.482629f, -0.123855f, -0.988528f, -0.0864401f, 
		-0.701879f, -0.0130262f, -0.712177f, 0.0612848f, -0.994631f, -0.0833882f, 
		-0.250364f, -0.962889f, -0.100807f, 0.0680107f, 0.873180f, 0.482629f, 
		0.124605f, 0.964140f, 0.234324f, 0.701879f, 0.0130262f, 0.712177f, 
		0.123855f, 0.988528f, 0.0864401f, 0.250364f, 0.962889f, 0.100807f, 
		-0.0612848f, 0.994631f, 0.0833882f, -0.119896f, 0.697250f, -0.706730f, 
		-0.999758f, -0.00509013f, 0.0213972f, -0.609190f, -0.778670f, -0.150200f, 
		-0.928803f, -0.0200467f, -0.370031f, -0.844816f, -0.0100837f, -0.534962f, 
		0.999758f, 0.00509013f, -0.0213972f, 0.975476f, 0.0178911f, 0.219378f, 
		0.609190f, 0.778670f, 0.150200f, 0.911095f, 0.00286810f, 0.412186f, 
		-0.997246f, 0.0713338f, -0.0203028f, -0.973215f, 0.108784f, 0.202533f, 
		-0.701548f, 0.352394f, 0.619394f, -0.275361f, 0.160985f, 0.947766f, 
		-0.230577f, 0.962831f, -0.140676f, -0.0717048f, 0.145729f, 0.986723f, 
		0.997246f, -0.0713338f, 0.0203028f, 0.721789f, -0.303493f, -0.622023f, 
		0.938205f, -0.126879f, -0.321982f, 0.393309f, -0.167012f, -0.904110f, 
		0.0717048f, -0.145729f, -0.986723f, 0.194907f, -0.976714f, 0.0896728f, 
		-0.965973f, -0.000574806f, 0.258643f, -0.725137f, 0.000000f, -0.688605f, 
		-0.119026f, -0.992114f, -0.0392853f, 0.966074f, 0.000000f, -0.258266f, 
		0.119026f, 0.992114f, 0.0392853f, 0.666445f, 0.142352f, 0.731838f, 
		-0.965856f, 0.000000f, 0.259079f, -0.664175f, -0.116542f, -0.738437f, 
		0.965965f, 0.000691905f, -0.258673f, 0.164127f, 0.985961f, 0.0307089f, 
		0.611678f, 0.000000f, 0.791107f, -0.164127f, -0.985961f, -0.0307089f, 
		-0.486422f, 0.869870f, -0.0819737f, -0.161309f, 0.292118f, 0.942680f, 
		-0.964608f, 0.246923f, -0.0925243f, 0.486422f, -0.869870f, 0.0819737f, 
		0.966329f, -0.237909f, 0.0980156f, 0.193587f, -0.147541f, -0.969926f, 
		0.349951f, -0.0799327f, -0.933351f, 0.564676f, 0.0126801f, -0.825215f, 
		-0.0682501f, 0.892846f, 0.445161f, 0.0682501f, -0.892846f, -0.445161f, 
		0.840256f, -0.0376341f, 0.540882f, -0.814910f, -0.0582579f, -0.576652f, 
		0.623767f, 0.0479167f, -0.780140f, -0.0682501f, 0.892846f, 0.445161f, 
		0.0682501f, -0.892846f, -0.445161f, 0.840256f, -0.0376341f, 0.540882f, 
		-0.814910f, -0.0582579f, -0.576652f, 0.623780f, 0.0479157f, -0.780130f, 
		-0.0682743f, 0.892740f, 0.445369f, 0.0682743f, -0.892740f, -0.445369f, 
		0.840514f, -0.0388099f, 0.540398f, -0.822770f, -0.0319835f, -0.567474f, 
		0.873782f, 0.0434465f, -0.484373f, -0.670446f, -0.191221f, -0.716894f, 
		-0.500750f, -0.333404f, -0.798806f, -0.338958f, -0.939063f, -0.0571700f, 
		-0.285819f, -0.889613f, -0.356226f, -0.227024f, -0.620849f, -0.750338f, 
		0.844513f, -0.491567f, -0.212507f, 0.406423f, -0.0763756f, -0.910487f, 
		0.0843941f, -0.868615f, -0.488248f, 0.509900f, 0.217614f, 0.832254f, 
		0.364409f, -0.889180f, 0.276704f, 0.458915f, 0.249563f, 0.852710f, 
		0.364190f, -0.889443f, 0.276145f, 0.312852f, 0.358319f, -0.879620f, 
		-0.593771f, -0.118422f, -0.795872f, -0.761367f, -0.0866548f, -0.642504f, 
		-0.971859f, -0.00778933f, -0.235435f, -0.996588f, 0.0781124f, 0.0266489f, 
		-0.0999051f, 0.985932f, -0.134004f, 0.593771f, 0.118422f, 0.795872f, 
		0.761367f, 0.0866548f, 0.642504f, 0.871987f, 0.0942397f, 0.480372f, 
		0.257551f, -0.953331f, 0.157568f, -0.103111f, 0.952836f, -0.285433f, 
		0.106981f, 0.769369f, -0.629782f, -0.595887f, -0.119374f, -0.794146f, 
		-0.762525f, -0.0872821f, -0.641044f, -0.872033f, -0.0942177f, -0.480293f, 
		-0.182489f, 0.976197f, -0.117203f, 0.595887f, 0.119374f, 0.794146f, 
		0.762525f, 0.0872821f, 0.641044f, 0.872033f, 0.0942177f, 0.480293f, 
		0.280779f, -0.949232f, 0.141850f, 0.288749f, -0.942434f, 0.168646f, 
		0.00911022f, 0.971721f, -0.235956f, -0.595885f, -0.119386f, -0.794146f, 
		-0.762492f, -0.0873112f, -0.641079f, -0.871994f, -0.0942601f, -0.480355f, 
		-0.152099f, 0.987129f, -0.0494187f, 0.595885f, 0.119386f, 0.794146f, 
		0.762492f, 0.0873112f, 0.641079f, 0.871994f, 0.0942601f, 0.480355f, 
		-0.100385f, 0.981878f, -0.160742f, 0.00688670f, -0.999959f, 0.00585521f, 
		0.441187f, 0.645353f, 0.623597f, 0.840256f, -0.0376341f, 0.540882f, 
		-0.599929f, 0.0387508f, 0.799114f, 0.623767f, 0.0479167f, -0.780140f, 
		0.0695748f, -0.823188f, -0.563490f, -0.0695748f, 0.823188f, 0.563490f, 
		0.840256f, -0.0376341f, 0.540882f, -0.599940f, 0.0387519f, 0.799106f, 
		0.623780f, 0.0479157f, -0.780130f, 0.0695662f, -0.823173f, -0.563513f, 
		-0.0695662f, 0.823173f, 0.563513f, -0.246162f, -0.614660f, 0.749398f, 
		-0.229721f, -0.461388f, 0.856942f, -0.235324f, -0.321851f, 0.917079f, 
		-0.917677f, -0.315302f, -0.241772f, -0.123897f, -0.991209f, -0.0464225f, 
		-0.164946f, -0.397548f, -0.902634f, 0.895767f, 0.365869f, 0.252469f, 
		0.123897f, 0.991209f, 0.0464225f, 0.104575f, 0.320499f, 0.941459f, 
		-0.858072f, -0.209105f, -0.469029f, -0.285049f, -0.957724f, 0.0388741f, 
		-0.113370f, -0.600974f, -0.791188f, -0.552385f, 0.246862f, -0.796197f, 
		-0.618418f, 0.781768f, -0.0799831f, -0.626804f, 0.773997f, 0.0896901f, 
		-0.644866f, 0.650437f, 0.401346f, -0.567653f, 0.639070f, 0.518999f, 
		-0.624206f, 0.727694f, 0.284304f, -0.603332f, 0.663934f, -0.441793f, 
		-0.603332f, 0.663934f, -0.441793f, -0.664088f, 0.744443f, -0.0692220f, 
		-0.641970f, 0.615387f, 0.457354f, -0.641970f, 0.615387f, 0.457354f, 
		-0.557433f, 0.590963f, 0.583122f, 0.921914f, -0.0315152f, 0.386110f, 
		-0.00435899f, -0.0595208f, 0.998218f, 0.0510682f, 0.00978344f, -0.998647f, 
		-0.113326f, 0.992973f, 0.0340862f, 0.902839f, 0.405463f, 0.143114f, 
		0.911176f, 0.247529f, 0.329375f, 0.701624f, 0.710077f, -0.0592884f, 
		0.529106f, 0.841088f, -0.112334f, -0.0270464f, 0.572296f, 0.819601f, 
		0.0985753f, -0.574365f, 0.812642f, 0.355451f, -0.448595f, 0.820010f, 
		0.785372f, -0.424631f, 0.450421f, 0.619844f, -0.409652f, 0.669312f, 
		-0.394897f, -0.892958f, 0.216063f, -0.907301f, -0.409616f, -0.0949757f, 
		0.394897f, 0.892958f, -0.216063f, -0.289855f, -0.318053f, -0.902677f, 
		0.907301f, 0.409616f, 0.0949757f, 0.289855f, 0.318053f, 0.902677f, 
		0.000361592f, 0.999998f, 0.00188799f, 0.185148f, -0.567337f, 0.802402f, 
		-0.121201f, 0.0725778f, -0.989971f, -0.888266f, 0.323774f, -0.325811f, 
		-0.250648f, 0.914045f, -0.318901f, 0.100817f, -0.0870141f, 0.991093f, 
		0.888266f, -0.323774f, 0.325811f, 0.267217f, -0.879288f, 0.394269f, 
		-0.328840f, 0.936245f, 0.123733f, -0.446014f, 0.891059f, -0.0841740f, 
		-0.484067f, 0.774075f, -0.408028f, -0.541180f, 0.543945f, -0.641286f, 
		-0.116535f, 0.889639f, 0.441545f, 0.116535f, -0.889639f, -0.441545f, 
		0.446014f, -0.891059f, 0.0841740f, 0.328841f, -0.936245f, -0.123733f, 
		0.586020f, -0.585841f, 0.559794f, 0.475422f, -0.831230f, 0.288151f, 
		-0.478807f, -0.788689f, -0.385634f, -0.554208f, -0.824339f, -0.115408f, 
		-0.305668f, -0.668659f, -0.677836f, -0.311963f, -0.518648f, -0.796042f, 
		-0.503290f, -0.827379f, -0.249285f, 0.371397f, 0.772574f, 0.514969f, 
		0.442505f, 0.832969f, 0.332193f, 0.483631f, 0.851521f, 0.202519f, 
		-0.591232f, -0.768299f, 0.245278f, -0.876233f, -0.00558093f, -0.481856f, 
		-0.516010f, -0.101138f, -0.850591f, -0.997246f, 0.0713338f, -0.0203028f, 
		0.591232f, 0.768299f, -0.245278f, -0.364780f, -0.0702412f, -0.928440f, 
		0.876233f, 0.00558094f, 0.481856f, 0.997246f, -0.0713338f, 0.0203028f, 
		0.516010f, 0.101138f, 0.850591f, 0.364780f, 0.0702412f, 0.928440f, 
		-0.792262f, -0.538945f, -0.286112f, -0.0694433f, -1.20720e-006f, 0.997586f, 
		-0.870325f, 0.492373f, -0.0101676f, 0.732277f, 0.670146f, 0.121139f, 
		0.870325f, -0.492373f, 0.0101676f, 0.0694433f, 1.20720e-006f, -0.997586f, 
		0.219680f, 0.947646f, 0.231747f, 0.524145f, 0.211918f, 0.824841f, 
		0.666281f, 0.356015f, 0.655227f, 0.387250f, 0.891918f, 0.233493f, 
		0.739855f, 0.575357f, 0.348682f, 0.127551f, -0.991538f, 0.0241683f, 
		-0.751781f, -0.642793f, 0.147114f, -0.951568f, -0.276521f, 0.134368f, 
		-0.916709f, -0.388473f, 0.0934578f, -0.916709f, -0.388473f, 0.0934578f, 
		-0.353043f, 0.175534f, 0.918993f, -0.143171f, -0.989641f, -0.0105813f, 
		0.750905f, 0.656282f, -0.0737331f, -0.127551f, 0.991538f, -0.0241683f, 
		0.901433f, 0.429046f, -0.0577831f, 0.901433f, 0.429046f, -0.0577831f, 
		0.143171f, 0.989641f, 0.0105813f, 0.303194f, -0.0992587f, -0.947745f, 
		-0.632708f, -0.773213f, 0.0426743f, -0.156358f, -0.763611f, 0.626458f, 
		-0.369533f, -0.846148f, 0.384030f, -0.533180f, -0.826245f, 0.181765f, 
		0.651089f, 0.755944f, -0.0680553f, 0.570799f, 0.802534f, -0.173571f, 
		0.464293f, 0.817999f, -0.339574f, 0.156358f, 0.763611f, -0.626458f, 
		-0.520637f, 0.0143833f, -0.853657f, -0.548619f, 0.148149f, -0.822842f, 
		-0.392432f, 0.471262f, -0.789879f, -0.288637f, 0.545816f, -0.786622f, 
		-0.274778f, 0.776801f, -0.566636f, -0.479575f, 0.237174f, -0.844841f, 
		0.438112f, -0.692160f, 0.573562f, 0.480880f, -0.492999f, 0.725056f, 
		0.456397f, -0.421363f, 0.783680f, 0.445052f, -0.329834f, 0.832549f, 
		0.382229f, -0.242444f, 0.891696f, -0.422930f, 0.131868f, -0.896516f, 
		-0.494295f, -0.843850f, -0.208779f, 0.494295f, 0.843850f, 0.208779f, 
		0.422930f, -0.131868f, 0.896516f, -0.515821f, -0.441783f, -0.734001f, 
		-0.641797f, -0.255523f, -0.723052f, -0.409452f, -0.439357f, -0.799572f, 
		-0.368805f, -0.267056f, -0.890317f, 0.492330f, 0.295646f, 0.818660f, 
		0.640986f, 0.151598f, 0.752433f, 0.525502f, 0.361597f, 0.770127f, 
		0.445720f, 0.243199f, 0.861503f, 0.434877f, 0.361655f, 0.824675f, 
		-0.436189f, 0.892587f, 0.114137f, -0.463008f, 0.775117f, 0.429903f, 
		-0.466587f, 0.706637f, 0.531941f, -0.512608f, 0.616333f, 0.597802f, 
		0.274550f, -0.566828f, -0.776742f, 0.436189f, -0.892587f, -0.114137f, 
		0.372635f, -0.716758f, -0.589407f, 0.407748f, -0.848128f, -0.338261f, 
		-0.453572f, 0.823696f, 0.340290f, -0.386385f, -0.495972f, 0.777637f, 
		-0.502611f, -0.171079f, 0.847416f, -0.531057f, -0.0448564f, 0.846148f, 
		0.386385f, 0.495972f, -0.777637f, 0.491593f, 0.100541f, -0.865002f, 
		0.459155f, 0.241834f, -0.854806f, -0.225307f, 0.973824f, 0.0300673f, 
		0.220941f, -0.971488f, -0.0860027f, -0.508377f, -0.838292f, -0.197027f, 
		-0.494460f, -0.868078f, 0.0441562f, -0.541047f, -0.838147f, -0.0691149f, 
		0.494460f, 0.868078f, -0.0441562f, 0.508377f, 0.838292f, 0.197027f, 
		0.541047f, 0.838147f, 0.0691149f, -0.381893f, -0.0561875f, -0.922497f, 
		-0.383067f, -0.857930f, -0.342369f, -0.942001f, -0.217035f, -0.255990f, 
		-0.868357f, -0.429623f, -0.247752f, -0.609892f, -0.725036f, -0.319928f, 
		-0.381900f, -0.0561953f, -0.922494f, -0.331089f, -0.886897f, -0.322170f, 
		-0.941996f, -0.217048f, -0.255994f, -0.853396f, -0.458286f, -0.248373f, 
		-0.578059f, -0.755119f, -0.309263f, -0.381896f, -0.0562058f, -0.922494f, 
		-0.355146f, -0.859938f, -0.366577f, -0.942011f, -0.216998f, -0.255981f, 
		-0.854652f, -0.455961f, -0.248334f, -0.592322f, -0.730912f, -0.339000f, 
		0.136671f, 0.333212f, -0.932894f, 0.327070f, -0.415443f, -0.848783f, 
		0.256373f, -0.583006f, -0.770958f, 0.271964f, -0.0902319f, -0.958068f, 
		0.279833f, -0.200960f, -0.938780f, 0.223114f, 0.0403445f, -0.973957f, 
		0.00264542f, -0.999907f, -0.0133889f, 0.0948003f, -0.115683f, -0.988752f, 
		-0.137070f, -0.333188f, -0.932844f, -0.224454f, 0.970786f, 0.0848195f, 
		0.237609f, -0.970890f, -0.0302304f, -0.00264627f, 0.999907f, -0.0133721f, 
		-0.0952411f, 0.115652f, -0.988713f, -0.964608f, 0.246923f, -0.0925243f, 
		-0.766673f, -0.635756f, -0.0895947f, -0.000118503f, -0.000283861f, -1.00000f, 
		0.962830f, -0.256132f, 0.0857638f, 0.000101387f, 0.000262907f, 1.00000f, 
		0.760761f, 0.643603f, 0.0837680f, 0.143954f, -0.333183f, -0.931808f, 
		-0.698233f, -0.658837f, -0.280008f, -0.698233f, -0.658837f, -0.280008f, 
		-0.871970f, 0.313970f, -0.375622f, -0.871970f, 0.313970f, -0.375622f, 
		0.570199f, 0.639031f, 0.516248f, 0.570199f, 0.639031f, 0.516248f, 
		0.0423687f, 0.0363551f, -0.998440f, 0.896739f, -0.343051f, 0.279599f, 
		-0.0423687f, -0.0363551f, 0.998440f, 0.896739f, -0.343051f, 0.279599f, 
		-0.144368f, 0.333186f, -0.931743f, 0.872203f, -0.482343f, 0.0812857f, 
		-0.872033f, -0.0942177f, -0.480293f, -0.960766f, -0.131663f, -0.244117f, 
		-0.995148f, -0.0979184f, 0.00964002f, -4.98821e-006f, -1.00000f, 1.95894e-006f, 
		0.872033f, 0.0942177f, 0.480293f, 0.960766f, 0.131663f, 0.244117f, 
		0.000173773f, 1.00000f, -5.24037e-005f, 0.995148f, 0.0979184f, -0.00964002f, 
		-0.871987f, -0.0942397f, -0.480372f, -0.960717f, -0.131882f, -0.244191f, 
		-0.995118f, -0.0982350f, 0.00953608f, 2.73113e-006f, -1.00000f, 1.01046e-005f, 
		0.871987f, 0.0942397f, 0.480372f, 0.960717f, 0.131882f, 0.244191f, 
		0.000173773f, 1.00000f, -5.24037e-005f, 0.995118f, 0.0982350f, -0.00953608f, 
		-0.871994f, -0.0942601f, -0.480355f, -0.960717f, -0.131901f, -0.244182f, 
		-0.995116f, -0.0982492f, 0.00954754f, -5.83946e-006f, -1.00000f, 1.58196e-006f, 
		0.871994f, 0.0942601f, 0.480355f, 0.960717f, 0.131901f, 0.244182f, 
		0.000173773f, 1.00000f, -5.24037e-005f, 0.995116f, 0.0982492f, -0.00954754f, 
		-0.907350f, -0.288590f, 0.305665f, -0.282923f, 0.921863f, 0.264807f, 
		-0.283852f, -0.0811871f, 0.955425f, 0.933021f, 0.233019f, -0.274180f, 
		0.282923f, -0.921863f, -0.264807f, 0.283852f, 0.0811871f, -0.955425f, 
		0.776494f, 0.623797f, 0.0890710f, 0.871969f, 0.313643f, 0.375897f, 
		0.615230f, 0.307313f, 0.725983f, 0.463336f, 0.871858f, 0.158692f, 
		0.219112f, 0.828828f, 0.514814f, 0.321143f, 0.538573f, 0.778978f, 
		-0.974415f, 0.188861f, 0.121844f, -0.991019f, -0.0260878f, 0.131153f, 
		-0.899421f, -0.429055f, 0.0833837f, 0.993313f, -0.0283179f, -0.111923f, 
		0.922782f, 0.370041f, -0.107436f, -0.000243720f, 2.78831e-005f, -1.00000f, 
		0.000342851f, -0.000185590f, 1.00000f, 0.829780f, 0.553235f, -0.0734586f, 
		-0.874159f, 0.255767f, -0.412831f, -0.240508f, 0.959824f, 0.144550f, 
		-0.281229f, 0.296448f, 0.912704f, 0.913396f, -0.272066f, 0.302801f, 
		0.281229f, -0.296448f, -0.912704f, 0.240508f, -0.959824f, -0.144550f, 
		0.575045f, -0.818122f, -0.000121554f, -5.14069e-005f, -0.000312304f, -1.00000f, 
		0.420953f, 0.907082f, -8.87414e-005f, -0.000216197f, 1.55279e-005f, -1.00000f, 
		-0.767095f, 0.301007f, 0.566533f, -0.887688f, 0.209473f, 0.410039f, 
		-0.541695f, 0.489572f, 0.683291f, -0.498433f, 0.429127f, -0.753269f, 
		-0.185828f, 0.743004f, 0.642972f, 0.925963f, -0.232043f, -0.297907f, 
		0.757550f, -0.354401f, -0.548195f, 0.498433f, -0.429127f, 0.753269f, 
		0.427765f, -0.592722f, -0.682421f, 0.185828f, -0.743004f, -0.642972f, 
		-0.789281f, 0.358168f, -0.498749f, -0.316677f, -0.267742f, -0.909961f, 
		-0.828570f, -0.554425f, 0.0779996f, 0.118573f, 0.0132969f, 0.992856f, 
		0.857370f, 0.505360f, -0.0976083f, 0.127194f, 0.210752f, 0.969229f, 
		0.0737316f, 0.446857f, 0.891562f, -0.244164f, -0.845487f, 0.474906f, 
		-0.293419f, -0.695975f, 0.655381f, -0.449201f, -0.501575f, -0.739352f, 
		-0.937047f, -0.288247f, 0.197122f, -0.284286f, -0.462433f, 0.839844f, 
		0.293419f, 0.695975f, -0.655381f, 0.244164f, 0.845487f, -0.474906f, 
		0.449201f, 0.501575f, 0.739352f, 0.937047f, 0.288247f, -0.197122f, 
		0.284286f, 0.462433f, -0.839844f, -0.943226f, 0.290980f, -0.160173f, 
		-0.217566f, 0.895202f, -0.388944f, -0.272623f, 0.489274f, 0.828425f, 
		0.943226f, -0.290980f, 0.160173f, 0.272623f, -0.489274f, -0.828425f, 
		0.250979f, -0.910875f, 0.327591f, -0.938246f, -0.223731f, -0.263891f, 
		-0.304620f, 0.881571f, -0.360609f, -0.215227f, -0.341207f, -0.915017f, 
		0.952380f, 0.133541f, 0.274116f, 0.304620f, -0.881571f, 0.360609f, 
		0.145193f, 0.377843f, 0.914414f, -0.300907f, 0.109932f, 0.947296f, 
		-0.640423f, 0.186252f, 0.745096f, -0.868431f, 0.468030f, 0.163631f, 
		-0.150887f, 0.0973383f, 0.983747f, -0.804393f, 0.591701f, -0.0533034f, 
		-0.181972f, 0.246869f, 0.951810f, -0.187899f, 0.533977f, 0.824356f, 
		0.868431f, -0.468030f, -0.163631f, 0.766939f, -0.234941f, -0.597166f, 
		0.326177f, -0.131234f, -0.936155f, 0.804393f, -0.591701f, 0.0533034f, 
		0.133082f, -0.119985f, -0.983815f, 0.187899f, -0.533977f, -0.824356f, 
		0.174789f, -0.302904f, -0.936855f, -0.0834376f, 0.588988f, -0.803823f, 
		-0.822358f, -0.294334f, -0.486924f, -0.0825993f, -0.988138f, -0.129460f, 
		0.0938765f, 0.982192f, 0.162748f, 0.822358f, 0.294334f, 0.486924f, 
		0.407754f, -0.850922f, 0.331163f, 0.744609f, -0.646416f, 0.166447f, 
		0.957426f, -0.134401f, 0.255483f, 0.801778f, 0.328671f, 0.499126f, 
		0.193730f, -0.726665f, 0.659111f, 0.0982032f, -0.560724f, 0.822159f, 
		0.619291f, 0.496164f, 0.608523f, -0.892337f, 0.375907f, 0.249857f, 
		-0.764730f, 0.545290f, 0.343289f, -0.459021f, -0.776885f, 0.430987f, 
		-0.578086f, 0.732399f, 0.359735f, -0.302633f, 0.0791629f, 0.949814f, 
		0.764730f, -0.545290f, -0.343289f, 0.892337f, -0.375907f, -0.249857f, 
		0.578086f, -0.732399f, -0.359735f, 0.459021f, 0.776885f, -0.430987f, 
		0.302633f, -0.0791629f, -0.949814f, -0.804393f, 0.591701f, -0.0533034f, 
		-0.200797f, 0.157557f, -0.966880f, 0.804393f, -0.591701f, 0.0533034f, 
		0.200797f, -0.157557f, 0.966880f, 0.593583f, 0.745122f, -0.304060f, 
		0.872172f, 0.421239f, -0.248745f, 0.910386f, 0.0214045f, -0.413205f, 
		0.705598f, -0.107355f, -0.700433f, 0.312019f, 0.759833f, -0.570349f, 
		0.229666f, 0.454171f, -0.860803f, 0.411036f, 0.0240727f, -0.911301f, 
		0.420974f, -0.907073f, -8.87458e-005f, 5.14103e-005f, 0.000312291f, 1.00000f, 
		0.575062f, 0.818110f, -0.000121558f, 0.000216180f, -1.55112e-005f, 1.00000f, 
		0.885106f, -0.177907f, -0.430043f, 0.714853f, -0.614626f, -0.333497f, 
		0.705855f, 0.109714f, -0.699808f, 0.512938f, 0.188152f, -0.837552f, 
		0.482441f, -0.820773f, -0.305912f, -0.262666f, -0.851204f, -0.454378f, 
		0.282956f, 0.794284f, 0.537633f, -0.274426f, 0.503409f, -0.819311f, 
		-0.220515f, 0.255417f, -0.941348f, -0.425847f, 0.742375f, -0.517236f, 
		-0.792767f, 0.516924f, -0.322971f, -0.937423f, 0.260418f, -0.231130f, 
		-0.286046f, 0.514750f, -0.808214f, -0.220526f, 0.255435f, -0.941340f, 
		-0.445050f, 0.739641f, -0.504838f, -0.805851f, 0.507130f, -0.305651f, 
		-0.941628f, 0.240304f, -0.235776f, -0.286031f, 0.514778f, -0.808202f, 
		-0.220526f, 0.255435f, -0.941340f, -0.445022f, 0.739674f, -0.504814f, 
		-0.805837f, 0.507165f, -0.305630f, -0.941622f, 0.240326f, -0.235778f, 
		0.00140766f, 0.999907f, -0.0135491f, 0.200423f, 0.115681f, -0.972856f, 
		-0.863436f, 0.0980648f, 0.494835f, -0.115920f, 0.987131f, 0.110162f, 
		-0.0875180f, -0.983596f, -0.157730f, -0.00221936f, -0.993159f, -0.116752f, 
		0.863436f, -0.0980648f, -0.494835f, 0.00221936f, 0.993159f, 0.116752f, 
		0.0875179f, 0.983596f, 0.157730f, 0.115920f, -0.987131f, -0.110162f, 
		-0.00141560f, -0.999907f, -0.0135640f, -0.200824f, -0.115686f, -0.972773f, 
		0.0219285f, -0.955493f, 0.294197f, 0.268886f, -0.576233f, 0.771788f, 
		0.165519f, -0.753240f, 0.636579f, 0.438154f, -0.277540f, 0.854981f, 
		0.543648f, -0.0661150f, 0.836705f, 0.817122f, -0.541277f, -0.198320f, 
		0.0570465f, -0.0977363f, 0.993576f, 0.0570465f, -0.0977363f, 0.993576f, 
		-0.0650740f, 0.112575f, -0.991510f, 0.0570465f, -0.0977363f, 0.993576f, 
		-0.0650740f, 0.112575f, -0.991510f, -0.219936f, 0.970342f, -0.100320f, 
		0.214843f, -0.967578f, 0.132798f, 0.556150f, 0.0167942f, -0.830912f, 
		0.652480f, -0.125107f, -0.747407f, 0.554936f, 0.831857f, 0.00773095f, 
		0.406640f, 0.887810f, -0.215493f, 0.206473f, 0.792871f, -0.573346f, 
		0.203616f, 0.420356f, -0.884218f, 0.663349f, -0.497665f, -0.558836f, 
		0.807326f, -0.459125f, -0.370713f, 0.458293f, -0.190152f, -0.868222f, 
		0.863437f, -0.249612f, -0.438373f, 0.637552f, 0.0470324f, -0.768970f, 
		0.0942055f, -0.850315f, -0.517774f, 0.272363f, -0.570548f, -0.774786f, 
		0.165245f, -0.753155f, -0.636751f, 0.427896f, -0.260890f, -0.865356f, 
		0.527492f, -0.0423351f, -0.848505f, 0.817175f, -0.541247f, 0.198183f, 
		0.972870f, -0.230034f, -0.0246779f, 0.322883f, 0.146815f, -0.934982f, 
		-0.000237283f, -0.000947329f, 1.00000f, -0.329361f, -0.709979f, 0.622456f, 
		0.425337f, 0.630886f, -0.648900f, -0.000237283f, -0.000947329f, 1.00000f, 
		0.000279417f, 0.000901313f, -1.00000f, -0.252643f, 0.546268f, -0.798600f, 
		-0.210249f, 0.463028f, -0.861046f, 0.174726f, -0.296222f, 0.939001f, 
		0.247106f, -0.349490f, 0.903767f, 0.363816f, -0.481199f, 0.797550f, 
		-0.207247f, 0.373797f, -0.904060f, -0.363818f, -0.481202f, -0.797547f, 
		-0.247108f, -0.349493f, -0.903766f, -0.174728f, -0.296226f, -0.939000f, 
		0.212556f, 0.440830f, 0.872060f, 0.480669f, 0.871652f, 0.0958152f, 
		0.198273f, 0.000000f, -0.980147f, 0.00130767f, -0.999908f, -0.0134966f, 
		0.00271809f, 0.999908f, -0.0132675f, 0.0990449f, 0.000000f, -0.995083f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.000000f, -1.00000f, -1.28608e-019f, 
		-0.439394f, -0.672445f, 0.595609f, 0.301443f, 0.697698f, -0.649884f, 
		0.000000f, -1.00000f, -1.28608e-019f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.723329f, -0.0828670f, -0.685514f, 0.384687f, 0.917953f, -0.0968371f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.000000f, -1.00000f, -1.28608e-019f, 
		-0.329677f, -0.622321f, -0.709950f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.425629f, 0.648747f, 0.630847f, 0.000000f, -1.00000f, -1.28608e-019f, 
		-5.79053e-006f, 1.00000f, 1.62417e-006f, 0.724038f, -0.597941f, 0.343853f, 
		0.663718f, -0.512672f, 0.544652f, 0.818000f, -0.432526f, 0.379206f, 
		0.478882f, -0.211582f, 0.852001f, 0.868967f, -0.0381045f, 0.493401f, 
		0.631613f, 0.0737014f, 0.771773f, 0.0595090f, 0.959280f, 0.276116f, 
		-0.00131064f, 0.999908f, -0.0134707f, -0.198686f, 0.000000f, -0.980063f, 
		-0.0994389f, 0.000000f, -0.995044f, -0.00272877f, -0.999908f, -0.0132821f, 
		-0.325244f, 0.634447f, 0.701209f, -0.0369896f, 0.997716f, 0.0565234f, 
		0.417643f, -0.662768f, -0.621541f, 0.284261f, -0.860052f, -0.423682f, 
		0.203694f, -0.954548f, -0.217592f, -0.360795f, 0.800532f, -0.478514f, 
		-0.239755f, 0.901965f, -0.359133f, -0.166084f, 0.936854f, -0.307766f, 
		0.190618f, -0.904778f, 0.380843f, 0.000219059f, 1.06077e-005f, 1.00000f, 
		-0.000200669f, -4.86687e-014f, -1.00000f, -0.333763f, -0.683969f, -0.648683f, 
		0.450256f, 0.640344f, 0.622277f, -0.000200669f, -4.86687e-014f, -1.00000f, 
		0.000219059f, 1.06077e-005f, 1.00000f, 0.000200110f, 7.27183e-007f, 1.00000f, 
		-0.000216527f, 3.79684e-006f, -1.00000f, -0.425649f, 0.630922f, -0.648661f, 
		0.329614f, -0.710007f, 0.622290f, -0.000216527f, 3.79684e-006f, -1.00000f, 
		0.000197078f, -4.52493e-006f, 1.00000f, 0.174349f, -0.296224f, -0.939071f, 
		0.246727f, -0.349466f, -0.903880f, 0.363472f, -0.481154f, -0.797734f, 
		-0.212218f, 0.440831f, 0.872142f, -0.270601f, -0.386872f, 0.881536f, 
		0.194100f, 0.354364f, -0.914741f, 0.200028f, 0.448862f, -0.870926f, 
		0.288710f, 0.665615f, -0.688189f, -0.887031f, -0.407120f, -0.217784f, 
		-0.292639f, -0.387236f, 0.874306f, -0.249640f, -0.961049f, 0.118592f, 
		-0.257990f, -0.886503f, 0.384128f, -0.292648f, -0.637954f, 0.712300f, 
		-0.0307205f, 0.0188343f, 0.999351f, -0.119654f, -0.375051f, 0.919250f, 
		-0.703999f, -0.710153f, 0.00826704f, -0.0912486f, -0.223110f, 0.970513f, 
		-0.781902f, 0.164723f, 0.601245f, 0.119654f, 0.375051f, -0.919250f, 
		0.0307205f, -0.0188343f, -0.999351f, 0.703999f, 0.710153f, -0.00826704f, 
		0.0912486f, 0.223110f, -0.970513f, 0.775370f, -0.157968f, -0.611431f, 
		0.321391f, 0.0861339f, 0.943021f, 0.742174f, 0.669813f, 0.0229848f, 
		-0.513540f, -0.406337f, -0.755756f, -0.260740f, -0.208637f, -0.942595f, 
		-0.194760f, -0.136999f, -0.971236f, 0.399171f, 0.483909f, 0.778778f, 
		0.127692f, 0.175642f, 0.976138f, 0.200136f, 0.243474f, 0.949034f, 
		-0.967395f, 0.237985f, -0.0866617f, -0.382649f, -0.0293086f, 0.923429f, 
		-0.760795f, -0.643440f, -0.0847069f, 0.964639f, -0.247036f, 0.0918954f, 
		0.0559594f, 0.00254637f, -0.998430f, 0.754051f, 0.650790f, 0.0887666f, 
		-0.995148f, -0.0979184f, 0.00964002f, -0.960476f, -0.0687458f, 0.269739f, 
		-0.299085f, 0.944150f, 0.138306f, -0.857271f, -0.0922289f, 0.506537f, 
		0.995148f, 0.0979184f, -0.00964002f, 0.960476f, 0.0687458f, -0.269739f, 
		0.857271f, 0.0922289f, -0.506537f, 0.299085f, -0.944150f, -0.138306f, 
		-0.995118f, -0.0982350f, 0.00953608f, -0.0257256f, 0.999581f, 0.0132785f, 
		-0.960511f, -0.0690613f, 0.269533f, -0.857352f, -0.0926484f, 0.506325f, 
		0.995118f, 0.0982350f, -0.00953608f, 0.960511f, 0.0690613f, -0.269533f, 
		0.857349f, 0.0926523f, -0.506328f, 0.400233f, -0.905963f, -0.138004f, 
		0.364399f, -0.922431f, -0.127804f, -0.995116f, -0.0982492f, 0.00954754f, 
		-0.0257148f, 0.999581f, 0.0132679f, -0.960508f, -0.0690684f, 0.269546f, 
		-0.857352f, -0.0926484f, 0.506325f, 0.995116f, 0.0982492f, -0.00954754f, 
		0.960508f, 0.0690684f, -0.269546f, 0.857349f, 0.0926523f, -0.506328f, 
		-0.190642f, -0.104897f, 0.976039f, 0.116080f, 0.0918474f, -0.988984f, 
		-0.256150f, -0.196006f, 0.946556f, -0.632149f, -0.368523f, 0.681600f, 
		-0.190642f, -0.104897f, 0.976039f, 0.198774f, 0.197919f, -0.959853f, 
		0.634369f, 0.470952f, -0.613009f, 0.116080f, 0.0918474f, -0.988984f, 
		0.778508f, 0.0926307f, -0.620762f, 0.759165f, 0.512865f, 0.400796f, 
		0.320675f, 0.815770f, -0.481339f, -0.318306f, -0.719326f, -0.617456f, 
		0.371464f, 0.689968f, 0.621256f, -0.253395f, 0.0477823f, -0.966182f, 
		-0.977127f, 0.152049f, 0.148673f, 0.976120f, -0.189538f, -0.106135f, 
		0.105660f, -0.0123233f, 0.994326f, -0.256635f, 0.885171f, 0.388087f, 
		0.221178f, -0.898631f, -0.378870f, 0.612368f, 0.736552f, -0.287220f, 
		0.915274f, 0.330277f, -0.230631f, 0.976483f, 0.0167277f, -0.214947f, 
		0.341950f, 0.732047f, -0.589218f, 0.298885f, 0.406276f, -0.863486f, 
		0.302363f, 0.123770f, -0.945123f, 0.220186f, -0.904736f, 0.364651f, 
		-0.203195f, 0.890798f, -0.406436f, -0.366190f, -0.409156f, 0.835761f, 
		-0.740881f, -0.457880f, 0.491367f, -0.0612300f, -0.00295647f, -0.998119f, 
		-0.0920262f, -0.139270f, 0.985969f, -0.289154f, -0.424962f, 0.857786f, 
		0.0598272f, 0.0397445f, -0.997417f, 0.367989f, 0.401757f, -0.838556f, 
		0.740881f, 0.457880f, -0.491367f, 0.289154f, 0.424962f, -0.857786f, 
		0.0778948f, 0.142750f, -0.986689f, 0.0425978f, 0.00719406f, 0.999066f, 
		-0.0758981f, -0.0359804f, 0.996466f, 0.0612300f, -0.00294796f, 0.998119f, 
		-0.223466f, 0.837651f, 0.498402f, -0.907667f, 0.295047f, 0.298476f, 
		0.223466f, -0.837651f, -0.498402f, -0.0426000f, 0.00718817f, -0.999066f, 
		0.801238f, -0.300483f, -0.517424f, 0.0612300f, -0.00294796f, 0.998119f, 
		-0.0426000f, 0.00718817f, -0.999066f, -0.961861f, 0.0595926f, -0.266968f, 
		0.174237f, -0.984162f, 0.0326505f, -0.174237f, 0.984162f, -0.0326505f, 
		-0.174237f, 0.984162f, -0.0326505f, 0.961861f, -0.0595926f, 0.266968f, 
		0.174237f, -0.984162f, 0.0326505f, -0.221176f, 0.898640f, 0.378850f, 
		0.256640f, -0.885168f, -0.388089f, -0.290979f, 0.500137f, 0.815595f, 
		-0.373033f, 0.645625f, 0.666344f, -0.417138f, 0.710777f, 0.566385f, 
		0.290979f, -0.500137f, -0.815595f, 0.417138f, -0.710777f, -0.566385f, 
		0.373033f, -0.645625f, -0.666344f, -0.256799f, 0.885168f, -0.387983f, 
		0.216270f, -0.896184f, 0.387403f, 0.0900236f, -0.924742f, 0.369794f, 
		0.117947f, -0.866431f, 0.485166f, -0.000287733f, 1.00000f, -0.000427883f, 
		-0.000287733f, 1.00000f, -0.000427883f, 0.0447613f, -0.969259f, 0.241937f, 
		-0.0793172f, -0.884884f, -0.459010f, -0.594762f, -0.400337f, 0.697129f, 
		-0.898081f, -0.358807f, -0.254379f, -0.125843f, -0.991420f, -0.0353418f, 
		0.594762f, 0.400337f, -0.697129f, 0.882120f, 0.409176f, 0.233324f, 
		0.125843f, 0.991420f, 0.0353418f, 0.0608469f, -0.00302004f, -0.998143f, 
		-0.223687f, 0.837676f, -0.498259f, -0.907739f, 0.295069f, -0.298236f, 
		0.223687f, -0.837676f, 0.498259f, -0.0422394f, 0.00725102f, 0.999081f, 
		0.801594f, -0.300507f, 0.516858f, -0.0422394f, 0.00725102f, 0.999081f, 
		0.0608469f, -0.00302004f, -0.998143f, -0.00207651f, -0.00410125f, -0.999989f, 
		-0.447949f, -0.721965f, 0.527360f, -0.266517f, -0.676072f, 0.686946f, 
		0.248561f, 0.468108f, 0.847993f, -0.0881661f, -0.681518f, 0.726471f, 
		-0.0886223f, -0.874166f, 0.477472f, 0.0848730f, -0.986376f, -0.140920f, 
		-0.676722f, -0.711684f, -0.188555f, -0.628430f, -0.776668f, -0.0431605f, 
		-0.561599f, -0.627699f, 0.539074f, -0.570121f, -0.798528f, 0.193172f, 
		-0.674630f, -0.711949f, -0.194941f, -0.627220f, -0.777356f, -0.0480932f, 
		-0.561596f, -0.627676f, 0.539104f, -0.569470f, -0.800099f, 0.188535f, 
		-0.788739f, -0.577352f, -0.211081f, -0.696193f, -0.717730f, -0.0133604f, 
		-0.436675f, -0.636125f, 0.636129f, -0.513902f, -0.728692f, 0.452674f, 
		-0.209188f, 1.71977e-006f, 0.977875f, -2.28478e-006f, 1.00000f, 3.05527e-006f, 
		-2.28478e-006f, 1.00000f, 3.05527e-006f, 6.17174e-006f, -1.00000f, 7.15075e-006f, 
		0.869006f, -0.488502f, 0.0787051f, 0.384132f, -0.380610f, -0.841177f, 
		0.926259f, -0.344772f, -0.152240f, 0.815895f, -0.117135f, -0.566211f, 
		0.564906f, -0.258457f, -0.783633f, -0.944374f, 0.296094f, -0.143132f, 
		-0.959499f, 0.120361f, -0.254706f, -0.725678f, -0.168786f, -0.667011f, 
		-0.548390f, -0.0466465f, -0.834921f, -0.398709f, 0.0165047f, -0.916929f, 
		-0.172834f, -0.0899542f, 0.980835f, -0.276837f, -0.266750f, 0.923150f, 
		-0.371811f, -0.507063f, 0.777588f, -0.907301f, -0.409616f, -0.0949757f, 
		-0.357929f, -0.0366279f, 0.933030f, -0.551368f, 0.0879089f, 0.829618f, 
		0.371811f, 0.507063f, -0.777588f, 0.263286f, 0.205276f, -0.942625f, 
		0.180739f, 0.0600837f, -0.981694f, 0.907301f, 0.409616f, 0.0949757f, 
		0.436651f, -0.0685235f, -0.897018f, 0.326073f, 0.0227437f, -0.945071f, 
		-0.471586f, -0.747374f, 0.468015f, -0.671881f, -0.596005f, 0.439721f, 
		-0.781492f, -0.520454f, 0.344088f, 0.0162903f, 0.996251f, -0.0849667f, 
		0.470851f, 0.741263f, -0.478361f, 0.595704f, 0.608786f, -0.523943f, 
		0.0362753f, -0.999334f, 0.00400968f, -0.00465153f, -0.994967f, 0.100099f, 
		-0.0232527f, 0.999641f, 0.0132855f, 0.851479f, -0.140530f, -0.505208f, 
		0.517289f, -0.852644f, -0.0735588f, 0.760893f, -0.403200f, -0.508401f, 
		0.604728f, -0.757989f, -0.244453f, -0.0656085f, -0.962298f, 0.263967f, 
		0.0560911f, 0.873900f, 0.482859f, 0.108900f, 0.154750f, -0.981933f, 
		-0.0917297f, -0.197127f, 0.976077f, -0.101297f, 0.212318f, 0.971936f, 
		0.0918445f, -0.182650f, -0.978879f, 0.0583693f, 0.0343269f, 0.997705f, 
		-0.0771101f, -0.0298790f, -0.996575f, -0.236408f, -0.160754f, -0.958264f, 
		-0.669616f, -0.739495f, 0.0690001f, -0.0771101f, -0.0298790f, -0.996575f, 
		-0.0447052f, 9.74422e-005f, 0.999000f, 0.349779f, 0.256159f, 0.901131f, 
		0.669616f, 0.739495f, -0.0690001f, 0.0234995f, 0.00482798f, -0.999712f, 
		0.396940f, -0.791364f, 0.464953f, 0.753032f, 0.353142f, 0.555188f, 
		0.463605f, -0.883128f, 0.0717961f, 0.452300f, -0.833431f, 0.317518f, 
		0.957524f, 0.171099f, 0.232107f, 0.890852f, 0.216084f, 0.399613f, 
		-0.0915960f, -0.919872f, -0.381374f, 0.105074f, 0.882583f, -0.458264f, 
		0.441684f, -0.880322f, -0.173059f, 0.958804f, -0.0895354f, -0.269589f, 
		-0.395515f, 0.712345f, -0.579769f, -0.542777f, 0.717125f, -0.437179f, 
		-0.467997f, 0.727548f, -0.501650f, 0.349794f, -0.828543f, 0.437220f, 
		0.419318f, -0.698578f, 0.579794f, 0.396229f, -0.768980f, 0.501669f, 
		-0.539483f, -0.784090f, -0.306858f, -0.427996f, -0.688869f, -0.585046f, 
		-0.359041f, -0.831693f, -0.423528f, -0.310135f, -0.751406f, -0.582413f, 
		0.518568f, 0.646182f, 0.559943f, 0.497154f, 0.799386f, 0.337372f, 
		0.386076f, 0.791368f, 0.474006f, 0.373630f, 0.893823f, 0.247954f, 
		-0.403372f, -0.514379f, 0.756773f, 0.471531f, 0.480769f, -0.739270f, 
		0.423943f, 0.904684f, 0.0426377f, 0.673458f, 0.360815f, 0.645187f, 
		0.642280f, 0.765429f, -0.0399357f, 0.767352f, 0.609019f, -0.200664f, 
		0.424684f, 0.904395f, 0.0414013f, 0.673516f, 0.360668f, 0.645209f, 
		0.642225f, 0.766407f, -0.0129172f, 0.771311f, 0.615469f, -0.162106f, 
		-0.416138f, -0.558448f, -0.717611f, 0.426832f, 0.496854f, 0.755613f, 
		-0.0736682f, 0.986060f, 0.149193f, -0.0736682f, 0.986060f, 0.149193f, 
		0.0923132f, -0.986021f, -0.138711f, 0.0923132f, -0.986021f, -0.138711f, 
		0.0665890f, 0.884959f, -0.460883f, 0.104375f, 0.980319f, -0.167573f, 
		-0.134584f, -0.972198f, 0.191619f, 0.102588f, 0.983325f, 0.150156f, 
		-0.0786605f, -0.983015f, -0.165814f, 0.653934f, -0.314207f, -0.688218f, 
		0.343196f, -0.913379f, 0.218987f, 0.981619f, -0.0676849f, 0.178445f, 
		-0.0699974f, -0.961966f, -0.264047f, -0.976886f, -0.213740f, 0.00297731f, 
		-0.235048f, -0.955987f, 0.175617f, 0.976886f, 0.213740f, -0.00297731f, 
		0.235048f, 0.955987f, -0.175617f, -0.314097f, -0.566950f, 0.761519f, 
		-0.359629f, -0.412321f, 0.837053f, -0.431724f, -0.569632f, 0.699381f, 
		-0.540938f, -0.299636f, 0.785878f, 0.428221f, 0.329692f, -0.841386f, 
		0.389399f, 0.462267f, -0.796666f, 0.499217f, 0.329940f, -0.801201f, 
		0.523008f, 0.546169f, -0.654340f, 0.451553f, 0.400429f, -0.797343f, 
		-0.539582f, 0.307027f, -0.783956f, -0.428014f, 0.585126f, -0.688790f, 
		-0.359139f, 0.423634f, -0.831597f, -0.310194f, 0.582469f, -0.751338f, 
		0.518591f, -0.560042f, 0.646078f, 0.497259f, -0.337504f, 0.799265f, 
		0.386172f, -0.474066f, 0.791285f, 0.373789f, -0.248032f, 0.893735f, 
		-0.0921004f, -0.138721f, 0.986040f, -0.0921004f, -0.138721f, 0.986040f, 
		0.0882619f, 0.178999f, -0.979882f, 0.104794f, 0.149736f, 0.983157f, 
		-0.0920713f, -0.197054f, -0.976060f, 0.0788147f, -0.165854f, 0.982996f, 
		-0.109858f, 0.154238f, -0.981907f, -0.415749f, -0.584732f, 0.696592f, 
		0.433885f, 0.498453f, -0.750526f, -0.164104f, -0.962592f, -0.215609f, 
		-0.164104f, -0.962592f, -0.215609f, 0.242359f, 0.940617f, 0.237701f, 
		-0.520460f, -0.676073f, 0.521581f, -0.574897f, -0.707731f, 0.410621f, 
		-0.500945f, -0.820067f, 0.276667f, -0.535160f, -0.838805f, 0.100050f, 
		-0.456696f, -0.823788f, 0.335861f, -0.458751f, -0.882917f, 0.100028f, 
		0.591501f, 0.793069f, -0.145493f, 0.476706f, 0.818072f, -0.321730f, 
		0.510072f, 0.762595f, -0.397839f, 0.454829f, 0.854360f, -0.251396f, 
		0.416710f, 0.817025f, -0.398526f, -0.373524f, 0.893819f, 0.248129f, 
		-0.385859f, 0.791340f, 0.474230f, -0.497003f, 0.799370f, 0.337633f, 
		0.309875f, -0.751388f, -0.582575f, 0.358863f, -0.831686f, -0.423694f, 
		0.427736f, -0.688840f, -0.585270f, 0.539343f, -0.784089f, -0.307107f, 
		-0.518308f, 0.646132f, 0.560241f, -0.587213f, -0.770469f, 0.248110f, 
		-0.443672f, -0.830196f, 0.337534f, -0.300236f, -0.772066f, 0.560154f, 
		-0.492247f, -0.729965f, 0.474178f, 0.495539f, 0.644140f, -0.582687f, 
		0.540647f, 0.726698f, -0.423805f, 0.382450f, 0.714930f, -0.585326f, 
		0.409232f, 0.859204f, -0.307080f, -0.660332f, -0.199666f, 0.723944f, 
		-0.801805f, -0.0579899f, 0.594765f, -0.863436f, 0.0980648f, 0.494835f, 
		-0.354880f, -0.934441f, -0.0296642f, 0.801805f, 0.0579899f, -0.594765f, 
		0.660332f, 0.199666f, -0.723944f, 0.863436f, -0.0980648f, -0.494835f, 
		0.407171f, 0.911008f, 0.0653856f, -0.993907f, -0.0204658f, 0.108304f, 
		-0.211651f, -0.000119944f, 0.977345f, 0.993907f, 0.0204658f, -0.108304f, 
		0.211651f, 0.000119947f, -0.977345f, -0.495281f, 0.861492f, 0.111931f, 
		0.495281f, -0.861492f, -0.111931f, -0.458858f, 0.102701f, 0.882554f, 
		-0.454013f, 0.344463f, 0.821716f, -0.534493f, 0.102704f, 0.838910f, 
		-0.499005f, 0.284003f, 0.818741f, -0.571793f, 0.420693f, 0.704322f, 
		-0.515587f, 0.533420f, 0.670547f, 0.414801f, -0.407275f, -0.813675f, 
		0.454790f, -0.257251f, -0.852636f, 0.508372f, -0.406555f, -0.759126f, 
		0.475501f, -0.329121f, -0.815830f, 0.590471f, -0.148933f, -0.793198f, 
		-0.520744f, -0.521407f, -0.675988f, -0.575155f, -0.410426f, -0.707635f, 
		-0.501200f, -0.276501f, -0.819967f, -0.535387f, -0.0998787f, -0.838681f, 
		-0.456971f, -0.335713f, -0.823696f, -0.458972f, -0.0998759f, -0.882819f, 
		0.591706f, 0.145276f, 0.792956f, 0.476951f, 0.321550f, 0.818000f, 
		0.510320f, 0.397660f, 0.762522f, 0.455055f, 0.251234f, 0.854287f, 
		0.416976f, 0.398365f, 0.816968f, -0.165584f, -0.331311f, 0.928878f, 
		0.227226f, 0.340331f, -0.912438f, -0.227245f, 0.340335f, 0.912432f, 
		0.163227f, -0.325715f, -0.931272f, -0.227601f, -0.340307f, -0.912353f, 
		0.165926f, 0.331265f, 0.928834f, -0.143771f, -0.152832f, -0.977738f, 
		-0.963099f, 0.266836f, -0.0352001f, -0.128326f, 0.113761f, -0.985186f, 
		0.213115f, 0.0542053f, 0.975522f, -0.213115f, -0.0542053f, -0.975522f, 
		-0.213115f, -0.0542053f, -0.975522f, 0.143771f, 0.152832f, 0.977738f, 
		0.963099f, -0.266836f, 0.0352001f, 0.213115f, 0.0542053f, 0.975522f, 
		0.128326f, -0.113761f, 0.985186f, -0.142372f, 0.986973f, -0.0749244f, 
		-0.355528f, 0.844637f, -0.400236f, 0.0678370f, -0.997696f, -0.00109161f, 
		-0.959952f, 0.0899041f, 0.265348f, 0.0678370f, -0.997696f, -0.00109161f, 
		-0.0678370f, 0.997696f, 0.00109161f, 0.142372f, -0.986973f, 0.0749244f, 
		-0.0678370f, 0.997696f, 0.00109161f, 0.355528f, -0.844637f, 0.400236f, 
		-0.0678370f, 0.997696f, 0.00109161f, 0.959952f, -0.0899041f, -0.265348f, 
		0.0678370f, -0.997696f, -0.00109161f, 0.578302f, 0.815210f, -0.0316178f, 
		0.666314f, 0.730304f, 0.150603f, 0.820969f, 0.119093f, -0.558414f, 
		0.527387f, 0.755571f, -0.388555f, 0.720148f, 0.420015f, -0.552245f, 
		-0.499244f, -0.769473f, -0.398329f, -0.512504f, -0.821120f, -0.251201f, 
		-0.405418f, -0.823116f, -0.397638f, -0.470141f, -0.821941f, -0.321528f, 
		-0.391063f, -0.908826f, -0.145276f, 0.535245f, 0.838771f, 0.0998804f, 
		0.485088f, 0.807477f, 0.335666f, 0.458838f, 0.882888f, 0.0998821f, 
		0.459733f, 0.843925f, 0.276470f, 0.325497f, 0.851842f, 0.410387f, 
		0.325317f, 0.788899f, 0.521351f, -0.416886f, 0.817021f, -0.398350f, 
		-0.454934f, 0.854360f, -0.251204f, -0.510248f, 0.762596f, -0.397612f, 
		-0.476849f, 0.818069f, -0.321526f, -0.591579f, 0.793053f, -0.145262f, 
		0.458794f, -0.882916f, 0.0998402f, 0.456851f, -0.823789f, 0.335646f, 
		0.535203f, -0.838803f, 0.0998355f, 0.501074f, -0.820066f, 0.276435f, 
		0.575089f, -0.707736f, 0.410344f, 0.520704f, -0.676083f, 0.521324f, 
		-0.935648f, -0.148127f, -0.320344f, -0.533981f, -0.833724f, 0.140602f, 
		-0.521621f, -0.333666f, 0.785225f, 0.140529f, 0.0584878f, -0.988348f, 
		-0.961421f, 0.272212f, -0.0396167f, 0.140529f, 0.0584878f, -0.988348f, 
		-0.164084f, -0.0671042f, 0.984161f, -0.164084f, -0.0671042f, 0.984161f, 
		0.961421f, -0.272212f, 0.0396167f, 0.140529f, 0.0584878f, -0.988348f, 
		-0.329525f, -0.865810f, 0.376545f, 0.269201f, 0.847756f, -0.456990f, 
		-0.509835f, 0.851593f, 0.121895f, -0.783416f, 0.286388f, -0.551580f, 
		-0.509712f, 0.311303f, 0.802050f, 0.528337f, -0.396391f, -0.750822f, 
		0.853432f, 0.512317f, -0.0958398f, 0.612455f, 0.554334f, -0.563571f, 
		0.769839f, 0.518335f, -0.372393f, 0.591901f, -0.210437f, 0.778055f, 
		0.745288f, -0.554349f, 0.370464f, 0.802225f, -0.572979f, 0.167718f, 
		0.518633f, -0.0172270f, 0.854824f, -0.308398f, -0.0152640f, -0.951135f, 
		1.48033e-005f, 1.00000f, -4.56961e-006f, 1.28466e-006f, -1.00000f, 3.59230e-006f, 
		-0.462215f, -0.378218f, -0.802065f, 0.478820f, 0.315858f, 0.819124f, 
		0.652255f, 0.246982f, -0.716633f, 0.623713f, 0.536818f, -0.568163f, 
		0.452948f, 0.782780f, -0.426725f, 0.924598f, 0.213899f, 0.315223f, 
		-0.280557f, -0.228255f, -0.932302f, -0.993907f, -0.0204658f, 0.108304f, 
		-0.943226f, 0.290980f, -0.160173f, -0.993974f, 0.101234f, -0.0420444f, 
		0.268689f, 0.167000f, 0.948640f, 0.993907f, 0.0204658f, -0.108304f, 
		0.993974f, -0.101234f, 0.0420444f, 0.943226f, -0.290980f, 0.160173f, 
		0.995048f, 0.0924265f, -0.0365562f, 0.875713f, 0.428120f, 0.223250f, 
		0.619635f, 0.765352f, 0.174039f, 0.474889f, 0.879321f, 0.0357092f, 
		0.872295f, 0.110258f, -0.476387f, 0.701823f, 0.206381f, -0.681800f, 
		0.00590029f, -0.999961f, -0.00659173f, 0.619659f, 0.567442f, -0.542247f, 
		0.494563f, 0.406200f, -0.768381f, 0.440119f, 0.239761f, -0.865338f, 
		-0.490692f, -0.169053f, 0.854776f, 0.494733f, 0.169064f, -0.852442f, 
		-0.543151f, -0.0443131f, 0.838465f, -0.543151f, -0.0443131f, 0.838465f, 
		0.497203f, 0.0554324f, -0.865862f, -0.526524f, -0.167631f, -0.833470f, 
		0.493734f, 0.139563f, 0.858341f, -0.497540f, -0.0553656f, -0.865672f, 
		-0.497540f, -0.0553656f, -0.865672f, 0.543285f, 0.0442041f, 0.838384f, 
		-0.109792f, 0.955212f, 0.274803f, 0.00153750f, -0.999998f, 0.000892280f, 
		-0.270198f, 0.861698f, 0.429500f, -0.402089f, 0.874602f, 0.270917f, 
		-0.805447f, 0.585347f, 0.0928650f, -0.914541f, 0.245073f, 0.321799f, 
		-0.907796f, 0.376534f, 0.184741f, 0.402089f, -0.874602f, -0.270917f, 
		0.430961f, -0.763923f, -0.480306f, 0.270198f, -0.861698f, -0.429500f, 
		0.805558f, -0.585335f, -0.0919725f, 0.793606f, -0.426423f, -0.433997f, 
		0.898238f, -0.396850f, -0.188888f, -0.792763f, -0.234414f, -0.562651f, 
		-0.907007f, -0.170241f, -0.385170f, -0.550244f, 0.829903f, 0.0921523f, 
		-0.990866f, 0.0970705f, 0.0936079f, -0.764458f, 0.625272f, 0.156970f, 
		0.792763f, 0.234414f, 0.562651f, 0.550244f, -0.829903f, -0.0921523f, 
		0.907007f, 0.170241f, 0.385170f, 0.764458f, -0.625272f, -0.156970f, 
		0.990866f, -0.0970705f, -0.0936079f, 0.493234f, 0.139586f, -0.858625f, 
		-0.526171f, -0.167676f, 0.833684f, -0.354949f, -0.621968f, 0.697973f, 
		-0.436343f, -0.456396f, 0.775440f, -0.464716f, -0.319203f, 0.825922f, 
		0.497252f, 0.115563f, -0.859875f, -0.495476f, -0.115611f, 0.860894f, 
		-0.495476f, -0.115611f, 0.860894f, -0.500158f, -0.0794113f, -0.862285f, 
		0.547352f, 0.0635138f, 0.834489f, -0.497439f, -0.0952217f, -0.862257f, 
		-0.497439f, -0.0952217f, -0.862257f, 0.461841f, 0.114476f, 0.879544f, 
		0.00470324f, 0.00932806f, -0.999945f, 0.00403171f, 0.122966f, 0.992403f, 
		0.434954f, 0.665570f, 0.606492f, 0.138999f, 0.980611f, 0.138134f, 
		-0.225688f, -0.964528f, -0.136931f, 0.138999f, 0.980611f, 0.138134f, 
		-0.225688f, -0.964528f, -0.136931f, 0.138999f, 0.980611f, 0.138134f, 
		-0.487484f, -0.213328f, 0.846670f, 0.480331f, 0.254663f, -0.839302f, 
		-0.480685f, -0.254671f, -0.839098f, 0.487847f, 0.213280f, 0.846473f, 
		-0.270391f, 0.861709f, -0.429356f, -0.402213f, 0.874602f, -0.270732f, 
		-0.805462f, 0.585376f, -0.0925509f, -0.914670f, 0.245163f, -0.321364f, 
		-0.907854f, 0.376596f, -0.184328f, 0.402213f, -0.874602f, 0.270732f, 
		0.431215f, -0.763945f, 0.480043f, 0.270391f, -0.861709f, 0.429356f, 
		0.805550f, -0.585385f, 0.0917229f, 0.793762f, -0.426471f, 0.433665f, 
		0.898290f, -0.396899f, 0.188538f, -0.669803f, 0.353345f, -0.653078f, 
		-0.786491f, -0.609800f, 0.0978525f, -0.903198f, -0.395993f, 0.165599f, 
		-0.946545f, -0.169040f, 0.274733f, 0.669803f, -0.353345f, 0.653078f, 
		0.817755f, 0.543056f, -0.190699f, 0.919441f, 0.245020f, -0.307560f, 
		0.891753f, 0.395917f, -0.219150f, -0.703999f, -0.710153f, 0.00826704f, 
		-0.504676f, -0.452208f, -0.735398f, -0.923014f, 0.230481f, 0.308096f, 
		0.504676f, 0.452208f, 0.735398f, 0.703999f, 0.710153f, -0.00826704f, 
		0.886576f, -0.209400f, -0.412474f, -0.943332f, 0.231571f, 0.237697f, 
		-0.673625f, -0.712523f, -0.196318f, -0.404371f, 0.909794f, -0.0935918f, 
		-0.709808f, 0.358181f, -0.606530f, 0.943332f, -0.231571f, -0.237697f, 
		0.404371f, -0.909794f, 0.0935918f, 0.673625f, 0.712523f, 0.196318f, 
		0.709808f, -0.358181f, 0.606530f, 0.455343f, 0.469107f, 0.756704f, 
		0.275854f, 0.270215f, 0.922436f, 0.363252f, 0.394293f, 0.844145f, 
		0.115634f, -0.899372f, -0.421615f, -0.149753f, 0.890388f, -0.429865f, 
		-0.00278760f, 0.999985f, -0.00462344f, -0.494668f, -0.476113f, -0.727063f, 
		-0.536471f, -0.556351f, -0.634565f, -0.624307f, -0.595400f, -0.505707f, 
		-0.556874f, -0.581141f, -0.593436f, -0.481808f, -0.423398f, 0.767199f, 
		-0.902240f, 0.270603f, -0.335762f, 0.481808f, 0.423398f, -0.767199f, 
		0.542239f, 0.580309f, 0.607633f, 0.873818f, -0.256261f, 0.413246f, 
		-0.907301f, -0.409616f, -0.0949757f, -0.913890f, 0.296332f, 0.277474f, 
		-0.579762f, -0.0761265f, -0.811222f, 0.907301f, 0.409616f, 0.0949757f, 
		0.579762f, 0.0761265f, 0.811222f, 0.913890f, -0.296332f, -0.277474f, 
		-0.105404f, -0.958154f, -0.266142f, -0.640404f, -0.326710f, 0.695085f, 
		-0.538368f, 0.820367f, 0.192764f, -0.945430f, -0.175823f, -0.274313f, 
		0.640404f, 0.326710f, -0.695085f, 0.538368f, -0.820367f, -0.192764f, 
		0.937358f, 0.228701f, 0.262784f, -0.652860f, 0.341445f, 0.676158f, 
		-0.686746f, 0.633253f, 0.356890f, -0.880715f, 0.473454f, -0.0135174f, 
		-0.570641f, 0.0873213f, 0.816544f, -0.944374f, 0.296094f, -0.143132f, 
		0.716180f, 0.325517f, 0.617353f, 0.943464f, -0.207764f, -0.258282f, 
		0.280709f, -0.485451f, 0.827974f, -0.280709f, 0.485451f, -0.827974f, 
		-0.441120f, -0.0862453f, -0.893294f, -0.509285f, -0.0520382f, -0.859023f, 
		-0.269733f, -0.192628f, -0.943471f, 0.636782f, 0.110267f, -0.763118f, 
		0.946894f, -0.146134f, 0.286420f, 0.745939f, 0.261860f, -0.612376f, 
		0.920330f, 0.331738f, -0.207223f, 0.987724f, 0.0895232f, 0.128009f, 
		0.471291f, 0.350857f, -0.809187f, -0.464762f, -0.350838f, 0.812963f, 
		0.460541f, 0.371231f, 0.806281f, -0.471288f, -0.331364f, -0.817365f, 
		0.286190f, 0.463391f, -0.838668f, -0.257951f, -0.479667f, 0.838678f, 
		-0.286186f, 0.463376f, 0.838677f, 0.257943f, -0.479669f, -0.838679f, 
		-0.320447f, -0.507506f, -0.799845f, 0.244000f, 0.489579f, 0.837124f, 
		0.320430f, -0.507487f, 0.799864f, -0.254828f, 0.505476f, -0.824352f, 
		-0.350537f, 0.386076f, -0.853270f, -0.197719f, 0.459637f, -0.865818f, 
		-0.370319f, 0.251959f, -0.894081f, -0.00133174f, 0.999989f, 0.00447860f, 
		-0.377568f, 1.71977e-006f, 0.925982f, 0.139111f, -0.914346f, 0.380288f, 
		0.278476f, -0.846474f, -0.453799f, -0.253660f, 0.846308f, 0.468422f, 
		-0.296357f, -0.818756f, 0.491743f, 0.244810f, 0.864015f, -0.439939f, 
		0.285728f, 0.836710f, 0.467201f, -0.250959f, -0.842133f, -0.477318f, 
		-0.320412f, 0.799886f, -0.507463f, 0.254813f, -0.824359f, 0.505474f, 
		-0.00761461f, 0.0169622f, 0.999827f, -0.329339f, 0.737759f, -0.589277f, 
		-0.851346f, 0.510544f, 0.120639f, -0.970573f, 0.0594708f, -0.233349f, 
		-0.942011f, -0.216998f, -0.255981f, -0.672561f, -0.333704f, 0.660533f, 
		-0.706262f, 0.661783f, 0.251468f, -0.905620f, 0.421889f, 0.0431615f, 
		-0.970659f, 0.0467825f, -0.235867f, -0.941996f, -0.217048f, -0.255994f, 
		-0.735750f, 0.640617f, 0.219731f, -0.672542f, -0.333741f, 0.660533f, 
		-0.905601f, 0.421929f, 0.0431610f, -0.970657f, 0.0468075f, -0.235868f, 
		-0.942001f, -0.217035f, -0.255990f, -0.735706f, 0.640669f, 0.219726f, 
		-0.672551f, -0.333725f, 0.660532f, 0.301735f, -0.455268f, -0.837667f, 
		-0.279048f, 0.530694f, 0.800310f, 0.253581f, 0.468346f, -0.846374f, 
		-0.278375f, -0.453650f, 0.846586f, 0.286540f, 0.463374f, 0.838558f, 
		-0.258314f, -0.479713f, -0.838540f, 0.258303f, -0.479716f, 0.838541f, 
		-0.286539f, 0.463362f, -0.838565f, 0.508905f, -0.297197f, 0.807892f, 
		0.385867f, -0.139598f, 0.911932f, 0.238137f, 0.0101336f, 0.971179f, 
		-0.00400983f, 0.999957f, -0.00839120f, -0.441860f, -0.331199f, -0.833707f, 
		-0.223852f, -0.508125f, -0.831684f, -0.357289f, -0.488969f, -0.795772f, 
		-0.447790f, -0.443971f, 0.776128f, 0.435193f, 0.477683f, -0.763169f, 
		-0.435512f, -0.477707f, -0.762972f, 0.448121f, 0.444011f, 0.775913f, 
		0.670379f, 0.332536f, 0.663334f, 0.976483f, 0.0167277f, -0.214947f, 
		0.798513f, -0.595627f, -0.0872074f, 0.915827f, -0.387931f, -0.103777f, 
		0.663122f, 0.283208f, 0.692866f, 0.647710f, -0.668864f, 0.364818f, 
		0.952337f, 0.171867f, -0.252023f, 0.808530f, 0.377121f, 0.451729f, 
		0.691144f, -0.720584f, 0.0554810f, 0.898077f, -0.398271f, -0.186649f, 
		0.839042f, -0.539973f, -0.0666179f, -0.473645f, 0.0145735f, 0.880595f, 
		-0.891239f, -0.137574f, -0.432165f, -0.468680f, -0.273555f, 0.839944f, 
		-0.371825f, -0.524498f, 0.765929f, 0.0338181f, 0.0395718f, 0.998644f, 
		0.521704f, 0.678641f, -0.516983f, 0.943793f, -0.127652f, -0.304892f, 
		0.682175f, 0.204588f, 0.701984f, 0.709896f, 0.700792f, 0.0702783f, 
		-0.363023f, -0.674630f, 0.642720f, 0.392234f, 0.647980f, -0.652897f, 
		-0.392248f, 0.647970f, 0.652898f, 0.363033f, -0.674650f, -0.642692f, 
		-0.209589f, 0.614341f, -0.760695f, -0.392508f, -0.647947f, -0.652765f, 
		0.363280f, 0.674620f, 0.642586f, 0.00248083f, 0.999994f, -0.00225834f, 
		0.467668f, -0.484767f, -0.739113f, 0.605492f, -0.412330f, -0.680708f, 
		0.785372f, -0.424631f, 0.450421f, 0.822185f, -0.245945f, 0.513345f, 
		0.911176f, 0.247529f, 0.329375f, 0.883894f, 0.130217f, 0.449194f, 
		-0.201735f, 0.588197f, 0.783152f, -0.775606f, 0.193378f, 0.600867f, 
		-0.797801f, 0.349459f, 0.491317f, -0.858591f, 0.475161f, 0.192469f, 
		-0.0775623f, -0.996457f, 0.0325084f, 0.822345f, -0.178180f, -0.540371f, 
		0.0962840f, 0.994722f, -0.0354728f, 0.858591f, -0.475161f, -0.192469f, 
		0.827958f, -0.390026f, -0.402946f, 0.0172432f, -0.000958984f, 0.999851f, 
		0.516317f, -0.621794f, -0.588888f, 0.0970468f, -0.850332f, -0.517221f, 
		-0.612004f, 0.248944f, -0.750651f, -0.307191f, 0.126829f, -0.943159f, 
		0.656525f, -0.0914942f, 0.748735f, 0.334431f, -0.0499930f, 0.941093f, 
		-0.211442f, 0.106021f, -0.971623f, 0.241980f, -0.0208828f, 0.970057f, 
		-0.101165f, 0.853088f, -0.511865f, 0.0750452f, -0.994180f, 0.0772961f, 
		-0.708122f, 0.677370f, -0.199332f, -0.372096f, 0.922026f, -0.106827f, 
		0.0552799f, 0.998360f, 0.0148900f, 0.408184f, -0.896174f, 0.173947f, 
		-0.177099f, 0.982795f, -0.0524360f, 0.186285f, -0.980954f, 0.0550144f, 
		0.351496f, -0.925075f, 0.143830f, 0.675374f, -0.679834f, 0.285827f, 
		-0.178130f, 0.0472273f, 0.982873f, 0.258765f, -0.0136432f, -0.965844f, 
		-0.661253f, 0.263272f, 0.702447f, -0.344549f, 0.137399f, 0.928659f, 
		0.673812f, -0.0832362f, -0.734200f, 0.349174f, -0.0434441f, -0.936050f, 
		0.00141517f, 0.999999f, 0.000225856f, 0.0868239f, -0.971693f, 0.219715f, 
		0.00423532f, -0.999978f, -0.00511840f, 0.313568f, 0.695851f, -0.646116f, 
		0.377987f, 0.502009f, -0.777890f, 0.555940f, 0.532765f, -0.638038f, 
		0.455137f, 0.700986f, -0.549062f, -0.424104f, 0.716711f, 0.553590f, 
		0.407330f, -0.736573f, -0.539947f, -0.407349f, -0.736576f, 0.539929f, 
		0.424112f, 0.716691f, -0.553610f, -0.438918f, -0.738818f, -0.511370f, 
		0.420362f, 0.739727f, 0.525452f, 0.438912f, -0.738842f, 0.511341f, 
		-0.420348f, 0.739760f, -0.525417f, -0.694071f, 0.475826f, -0.540237f, 
		-0.888857f, 0.418011f, -0.187617f, -0.815477f, 0.448942f, -0.365306f, 
		0.631930f, -0.470305f, 0.616018f, 0.822594f, -0.439678f, 0.360586f, 
		0.920925f, -0.381675f, 0.0788796f, -0.759178f, -0.363059f, -0.540219f, 
		-0.806460f, -0.560722f, -0.187653f, -0.796575f, -0.481674f, -0.365319f, 
		0.726957f, 0.314220f, 0.610573f, 0.782801f, 0.468322f, 0.409754f, 
		0.795839f, 0.580975f, 0.170614f, -0.455132f, 0.796423f, -0.398202f, 
		0.495674f, -0.790013f, 0.360812f, 0.420125f, 0.554490f, -0.718356f, 
		-0.411748f, -0.554518f, 0.723169f, -0.429899f, -0.523012f, 0.735966f, 
		0.405133f, 0.573598f, -0.711936f, 0.438654f, -0.511395f, -0.738957f, 
		-0.420329f, 0.511378f, 0.749544f, -0.421123f, -0.540328f, -0.728493f, 
		0.407469f, 0.568725f, 0.714507f, 0.424401f, 0.553369f, 0.716705f, 
		-0.409230f, -0.550795f, -0.727430f, -0.438964f, 0.511395f, -0.738773f, 
		0.420612f, -0.511409f, 0.749364f, -0.778481f, 0.417584f, -0.468605f, 
		-0.865351f, 0.0192035f, -0.500798f, -0.742671f, 0.554392f, -0.375618f, 
		0.810150f, -0.314213f, 0.494901f, 0.778919f, -0.538239f, 0.321845f, 
		0.860241f, -0.171720f, 0.480101f, 0.424109f, -0.716763f, -0.553518f, 
		-0.408481f, 0.725766f, 0.553541f, 0.408490f, 0.725732f, -0.553578f, 
		-0.434077f, -0.721163f, 0.539908f, 0.420559f, -0.749448f, 0.511330f, 
		-0.438911f, 0.738874f, -0.511295f, 0.430633f, 0.733802f, 0.525443f, 
		-0.420574f, -0.749447f, -0.511319f, -0.480848f, 0.835955f, 0.264509f, 
		0.513141f, -0.801954f, -0.305870f, -0.512370f, -0.801048f, 0.309515f, 
		0.480991f, 0.835149f, -0.266784f, -0.693895f, 0.475846f, 0.540444f, 
		-0.888800f, 0.418022f, 0.187862f, -0.815360f, 0.448966f, 0.365537f, 
		0.675738f, -0.487387f, -0.553021f, 0.838996f, -0.445397f, -0.312583f, 
		0.920881f, -0.381705f, -0.0792394f, -0.720367f, -0.489480f, 0.491407f, 
		-0.754852f, -0.610235f, 0.240441f, -0.755988f, -0.539955f, 0.370041f, 
		0.707642f, 0.427686f, -0.562430f, 0.747101f, 0.484130f, -0.455476f, 
		0.754852f, 0.610235f, -0.240441f, -0.480599f, -0.835478f, -0.266460f, 
		0.513210f, 0.801710f, 0.306394f, -0.394636f, 0.467606f, 0.790953f, 
		-0.542279f, 0.0527041f, 0.838544f, -0.519607f, 0.158517f, 0.839572f, 
		0.234541f, -0.473178f, -0.849172f, -0.196323f, 0.704167f, 0.682353f, 
		0.193272f, -0.667439f, 0.719147f, -0.838826f, -0.501412f, 0.212032f, 
		-0.662062f, -0.513440f, -0.545942f, -0.0940279f, 0.995508f, -0.0110634f, 
		0.805193f, 0.588770f, 0.0708056f, 0.0990112f, -0.994392f, 0.0371643f, 
		-0.932666f, 0.350962f, 0.0834234f, -0.642980f, -0.240456f, 0.727157f, 
		-0.112354f, 0.00766284f, -0.993639f, 0.846036f, -0.110732f, -0.521499f, 
		0.112353f, 0.00767061f, 0.993639f, -0.850345f, -0.405697f, 0.335147f, 
		-0.884419f, -0.176491f, 0.432034f, -0.872862f, -0.0440647f, 0.485973f, 
		0.836570f, -0.0128531f, -0.547709f, 0.865377f, 0.0828136f, -0.494232f, 
		0.871782f, 0.170675f, -0.459202f, 0.850935f, 0.431209f, -0.299946f, 
		0.896799f, 0.315917f, -0.309755f, 0.233861f, -0.486925f, 0.841554f, 
		0.0707094f, 0.0337278f, -0.996927f, -0.0788366f, -0.0531427f, 0.995470f, 
		0.0707094f, 0.0337278f, -0.996927f, 0.00108875f, 0.999998f, 0.00136268f, 
		0.494726f, -0.510594f, 0.703235f, -0.522937f, 0.802620f, -0.286945f, 
		-0.447545f, 0.880209f, -0.157907f, 0.347793f, -0.434316f, 0.830909f, 
		0.521051f, -0.726356f, 0.448233f, 0.541127f, -0.839075f, 0.0559888f, 
		0.486592f, -0.871214f, -0.0649098f, -0.485968f, 0.846274f, -0.218300f, 
		-0.0711332f, -0.0336941f, -0.996898f, -0.0711332f, -0.0336941f, -0.996898f, 
		0.0792023f, 0.0531517f, 0.995441f, -0.248250f, -0.466728f, 0.848845f, 
		-0.377221f, 0.532555f, 0.757687f, -0.527513f, 0.782796f, 0.330092f, 
		-0.481162f, 0.875304f, -0.0482382f, -0.525707f, 0.846196f, 0.0870914f, 
		0.435751f, -0.879197f, -0.192700f, 0.350779f, -0.894861f, -0.276005f, 
		0.367982f, -0.922284f, -0.118245f, 0.470278f, -0.836587f, -0.280998f, 
		0.477814f, -0.872018f, -0.106200f, -0.538638f, -0.841274f, -0.0461172f, 
		-0.523893f, -0.833646f, -0.174843f, -0.421913f, -0.773171f, -0.473493f, 
		0.511861f, 0.761989f, 0.396699f, 0.514686f, 0.822236f, 0.242955f, 
		0.416680f, 0.818512f, 0.395493f, 0.480549f, 0.819902f, 0.311181f, 
		0.386096f, 0.908890f, 0.157633f, -0.493922f, -0.784422f, 0.375131f, 
		-0.464095f, -0.835215f, 0.295012f, -0.391822f, -0.910919f, 0.129235f, 
		0.489227f, 0.865656f, -0.106287f, -0.488366f, 0.846257f, 0.212949f, 
		0.521728f, -0.812234f, -0.260911f, -0.521733f, -0.812232f, 0.260911f, 
		0.488379f, 0.846256f, -0.212923f, -0.932671f, 0.350914f, -0.0835734f, 
		-0.656419f, -0.228869f, -0.718841f, -0.111941f, 0.000180135f, 0.993715f, 
		0.848429f, -0.114548f, 0.516765f, 0.0999267f, 0.0107727f, -0.994936f, 
		-0.245952f, -0.660989f, 0.708944f, -0.535012f, -0.196178f, 0.821752f, 
		0.482420f, 0.236889f, -0.843300f, 0.221836f, 0.699553f, 0.679275f, 
		-0.484121f, -0.228974f, -0.844510f, 0.489659f, 0.229013f, 0.841301f, 
		-0.468496f, -0.875079f, -0.121446f, 0.496529f, 0.857161f, 0.136874f, 
		-0.496511f, 0.857166f, -0.136902f, 0.468473f, -0.875090f, 0.121450f, 
		0.489423f, -0.841422f, -0.229074f, -0.483884f, 0.844613f, 0.229095f, 
		0.488646f, 0.846108f, -0.212902f, -0.442489f, -0.857978f, 0.260914f, 
		0.498006f, -0.115208f, -0.859487f, -0.493595f, 0.107433f, 0.863031f, 
		-0.498369f, 0.115249f, -0.859271f, 0.495327f, -0.115238f, 0.861029f, 
		-0.318987f, 0.780331f, 0.537894f, 0.306083f, -0.780342f, -0.545326f, 
		0.319229f, -0.780331f, 0.537752f, -0.306344f, 0.780320f, -0.545210f, 
		0.495173f, -0.861134f, 0.115119f, -0.498199f, 0.859385f, -0.115134f, 
		0.494111f, 0.858559f, 0.136861f, -0.523642f, -0.843232f, -0.121486f, 
		-0.396596f, -0.612993f, 0.683338f, 0.384821f, 0.633468f, -0.671291f, 
		-0.385115f, -0.633429f, -0.671159f, 0.396884f, 0.612970f, 0.683191f, 
		-0.306111f, 0.780324f, 0.545336f, 0.318989f, -0.780339f, -0.537882f, 
		-0.319218f, 0.780334f, -0.537753f, 0.306322f, -0.780338f, 0.545196f, 
		0.211645f, 0.976800f, -0.0326955f, 0.211645f, 0.976800f, -0.0326955f, 
		-0.248640f, -0.968596f, 0.000556020f, -0.248640f, -0.968596f, 0.000556020f, 
		0.211645f, 0.976800f, -0.0326955f, -0.0988353f, -0.770670f, -0.629523f, 
		0.116052f, -0.670125f, -0.733120f, -0.0985660f, -0.770666f, 0.629570f, 
		-0.116371f, 0.670157f, -0.733040f, 0.0414382f, -0.989422f, 0.139025f, 
		-0.233413f, -0.969688f, 0.0722766f, -0.233413f, -0.969688f, 0.0722766f, 
		0.316152f, 0.948636f, -0.0117254f, -0.487557f, 0.552839f, 0.675764f, 
		-0.472266f, 0.470136f, 0.745612f, -0.495067f, 0.310805f, 0.811363f, 
		0.907077f, -0.420965f, -0.000191999f, -0.000162772f, -0.000119418f, -1.00000f, 
		0.818116f, 0.575053f, -0.000173245f, -0.000236825f, 2.94740e-006f, -1.00000f, 
		-0.487846f, 0.552869f, -0.675531f, -0.495408f, 0.310864f, -0.811131f, 
		-0.472584f, 0.470159f, -0.745396f, 4.63208e-005f, 1.00000f, -0.000151442f, 
		0.346614f, -0.330046f, -0.878025f, 4.74983e-005f, -1.00000f, 0.000153792f, 
		-0.319731f, 0.374630f, -0.870301f, 0.0914505f, 0.770218f, 0.631190f, 
		0.818105f, -0.575068f, -0.000173243f, 0.000162749f, 0.000119436f, 1.00000f, 
		0.907076f, 0.420966f, -0.000191999f, 0.000236810f, -2.94965e-006f, 1.00000f, 
		0.0911904f, 0.770217f, -0.631230f, -0.000610270f, -0.999998f, 0.00198194f, 
		-0.319361f, 0.374613f, 0.870444f, 0.000124116f, 1.00000f, 0.000186036f, 
		0.365463f, -0.313576f, 0.876417f, 0.00302425f, -0.999879f, -0.0152875f, 
		0.0943747f, -0.153127f, -0.983690f, -0.00302929f, 0.999879f, -0.0152768f, 
		-0.0947830f, 0.153059f, -0.983661f, 0.141730f, -0.359645f, -0.922263f, 
		-0.142116f, 0.359624f, -0.922211f, 0.136367f, -0.0848952f, -0.987014f, 
		0.136367f, -0.0848952f, -0.987014f, -0.137631f, 0.0722614f, 0.987844f, 
		-0.286683f, 0.161622f, 0.944294f, -0.553641f, 0.313195f, 0.771615f, 
		0.466948f, -0.358306f, -0.808441f, -0.137631f, 0.0722614f, 0.987844f, 
		0.204702f, -0.156385f, -0.966251f, -0.137631f, 0.0722614f, 0.987844f, 
		0.136367f, -0.0848952f, -0.987014f, -0.492769f, -0.807386f, 0.324510f, 
		0.459623f, 0.838345f, -0.293128f, 0.140648f, -0.113747f, -0.983504f, 
		-0.113467f, 0.140709f, 0.983527f, -0.113467f, 0.140709f, 0.983527f, 
		0.140648f, -0.113747f, -0.983504f, -0.373858f, -0.660201f, 0.651433f, 
		0.376957f, 0.660197f, -0.649649f, -0.131346f, -0.962301f, 0.238169f, 
		0.371451f, 0.667479f, 0.645365f, -0.350735f, -0.662164f, -0.662211f, 
		-0.988385f, 0.129905f, -0.0788608f, -0.933824f, 0.0303432f, -0.356442f, 
		-0.786694f, -0.0913168f, -0.610552f, 0.978787f, -0.0823851f, 0.187585f, 
		0.839001f, 0.0651380f, 0.540217f, 0.930704f, -0.0188790f, 0.365286f, 
		-0.897659f, 0.157100f, 0.411738f, -0.976978f, 0.154973f, 0.146618f, 
		-0.955830f, 0.0752264f, 0.284130f, -0.871980f, -0.242864f, 0.425051f, 
		-0.949765f, -0.301747f, 0.0830351f, -0.826271f, -0.531625f, 0.186150f, 
		-0.819378f, -0.572306f, 0.0329510f, 0.965996f, -0.0584524f, -0.251865f, 
		0.980140f, 0.138977f, -0.141461f, 0.934290f, 0.138428f, -0.328541f, 
		0.849556f, 0.478980f, -0.220982f, 0.406042f, -0.0573241f, -0.912055f, 
		0.00372275f, -0.999922f, -0.0119087f, -0.978237f, 0.207463f, -0.00336693f, 
		-0.899421f, 0.436083f, -0.0295527f, -0.945315f, 0.325979f, 0.0108255f, 
		0.899421f, -0.436083f, 0.0295527f, 0.978237f, -0.207463f, 0.00336693f, 
		0.945315f, -0.325979f, -0.0108255f, -0.00265797f, 0.999980f, -0.00570005f, 
		-0.357849f, 0.0498017f, -0.932451f, -0.874211f, -0.485498f, -0.00682402f, 
		-0.881195f, -0.437849f, 0.178282f, -0.957980f, -0.277098f, 0.0741051f, 
		-0.922153f, -0.0177077f, -0.386419f, -0.796154f, -0.372298f, -0.477004f, 
		-0.815589f, -0.118287f, -0.566411f, 0.989886f, 0.139746f, -0.0244490f, 
		0.861805f, 0.460716f, 0.212209f, 0.965767f, 0.00570891f, 0.259349f, 
		0.812170f, 0.243360f, 0.530241f, 0.786796f, 0.451890f, 0.420413f, 
		-0.988352f, 0.129943f, 0.0792128f, -0.933697f, 0.0303669f, 0.356775f, 
		-0.786484f, -0.0913197f, 0.610822f, 0.978707f, -0.0823976f, -0.187998f, 
		0.838836f, 0.0651356f, -0.540474f, 0.930566f, -0.0188886f, -0.365637f, 
		0.406436f, -0.0573227f, 0.911879f, 0.000120865f, -1.00000f, -0.000312509f, 
		-0.365687f, -0.680792f, 0.634661f, 0.399504f, 0.679495f, -0.615372f, 
		-0.399777f, -0.679437f, -0.615259f, 0.365979f, 0.680754f, 0.634534f, 
		-0.898831f, 0.159811f, -0.408123f, -0.977022f, 0.155026f, -0.146268f, 
		-0.956478f, 0.0776637f, -0.281279f, -0.857073f, -0.221313f, -0.465237f, 
		-0.971689f, -0.234469f, -0.0290516f, -0.800093f, -0.594535f, -0.0798670f, 
		-0.729435f, -0.591274f, -0.343977f, -0.808579f, -0.543504f, -0.225396f, 
		0.966560f, -0.0609010f, 0.249103f, 0.981874f, 0.164480f, 0.0941788f, 
		0.914606f, 0.133056f, 0.381826f, 0.808579f, 0.543504f, 0.225396f, 
		0.729435f, 0.591274f, 0.343977f, 0.800093f, 0.594535f, 0.0798670f, 
		0.0786420f, -0.0330484f, -0.996355f, -0.0766559f, 0.0108949f, 0.996998f, 
		-0.0766559f, 0.0108949f, 0.996998f, -0.0766559f, 0.0108949f, 0.996998f, 
		0.0786420f, -0.0330484f, -0.996355f, 0.0921623f, -0.0106460f, 0.995687f, 
		-0.0802956f, 0.0187490f, -0.996595f, 0.0921623f, -0.0106460f, 0.995687f, 
		-0.0802956f, 0.0187490f, -0.996595f, 5.64161e-005f, 0.00136100f, -0.999999f, 
		-0.000115440f, -0.000355761f, 1.00000f, -0.595378f, -0.439502f, 0.672580f, 
		0.678245f, 0.295994f, -0.672586f, 5.64161e-005f, 0.00136100f, -0.999999f, 
		0.462716f, 0.123010f, 0.877931f, 0.559771f, -0.112828f, 0.820930f, 
		0.516002f, 0.0458819f, 0.855357f, -0.310285f, -0.109495f, -0.944317f, 
		-0.554271f, -0.285273f, -0.781923f, -0.470582f, -0.237146f, -0.849891f, 
		0.214491f, 0.176335f, 0.960677f, 0.442603f, 0.294185f, 0.847088f, 
		0.387092f, 0.252725f, 0.886730f, -0.442609f, 0.294178f, -0.847087f, 
		-0.214486f, 0.176330f, -0.960679f, -0.387097f, 0.252719f, -0.886729f, 
		0.310293f, -0.109491f, 0.944315f, 0.554285f, -0.285270f, 0.781914f, 
		0.470593f, -0.237142f, 0.849886f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, -1.00000f, -1.28608e-019f, -0.683879f, -0.648757f, 0.333803f, 
		0.630850f, 0.648745f, -0.425627f, 0.000000f, -1.00000f, -1.28608e-019f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, -1.00000f, -1.28608e-019f, -0.595607f, -0.672446f, -0.439396f, 
		0.649893f, 0.697689f, 0.301445f, 0.000000f, -1.00000f, -1.28608e-019f, 
		0.000000f, 1.00000f, -5.78099e-015f, 0.517909f, -0.0139147f, -0.855323f, 
		0.516769f, 0.129997f, -0.846198f, 0.602097f, -0.0108092f, -0.798350f, 
		0.593279f, 0.123844f, -0.795413f, 0.559362f, 0.0470918f, -0.827584f, 
		-0.582935f, 0.688061f, 0.432156f, -0.347633f, 0.903617f, 0.250254f, 
		-0.294185f, 0.938914f, 0.178596f, 0.355821f, -0.914041f, -0.194731f, 
		0.449905f, -0.870268f, -0.200548f, 0.665607f, -0.688060f, -0.289035f, 
		-0.588997f, 0.763160f, -0.265837f, -0.430525f, 0.878292f, -0.207969f, 
		-0.497467f, 0.835049f, -0.234990f, -0.103321f, 0.992641f, -0.0631480f, 
		-0.228660f, 0.953431f, -0.196681f, 0.250028f, -0.944344f, 0.213777f, 
		0.524193f, -0.782025f, 0.337135f, 0.440666f, -0.849971f, 0.288726f, 
		-0.274198f, 0.333246f, 0.902088f, 0.274584f, -0.333219f, 0.901980f, 
		0.000205309f, 2.03513e-005f, 1.00000f, -0.000215972f, -3.88223e-005f, -1.00000f, 
		-0.678596f, 0.296018f, -0.672221f, 0.595724f, -0.439520f, 0.672261f, 
		-0.000221224f, -4.18546e-005f, -1.00000f, 0.000205309f, 2.03513e-005f, 1.00000f, 
		0.000222053f, -4.26918e-005f, 1.00000f, -0.000205909f, 2.14019e-005f, -1.00000f, 
		-0.640526f, -0.450115f, -0.622192f, 0.710176f, 0.329546f, 0.622134f, 
		-0.000205909f, 2.14019e-005f, -1.00000f, 0.000215746f, -3.90510e-005f, 1.00000f, 
		-0.442184f, 0.294175f, 0.847310f, -0.214120f, 0.176361f, 0.960755f, 
		0.553905f, -0.285262f, -0.782186f, 0.470211f, -0.237155f, -0.850093f, 
		-0.386683f, 0.252720f, 0.886910f, 0.309968f, -0.109589f, -0.944410f, 
		-0.281256f, -0.333257f, 0.899908f, -0.427266f, -0.202738f, 0.881103f, 
		0.343683f, 0.236574f, -0.908798f, 0.281630f, 0.333236f, 0.899799f, 
		0.498314f, -0.0159641f, -0.866849f, 0.00394885f, -0.999963f, -0.00770782f, 
		0.366677f, 0.0159666f, -0.930211f, 8.16089e-006f, -1.00000f, 2.51535e-006f, 
		-2.68247e-006f, 1.00000f, -1.00611e-005f, -0.00460858f, -0.999907f, 0.0128458f, 
		-0.241145f, -0.115729f, 0.963564f, 0.00461663f, 0.999907f, 0.0128432f, 
		0.241541f, 0.115733f, 0.963465f, -5.80669e-005f, 1.00000f, 0.000496505f, 
		-0.0526997f, -0.991936f, 0.115265f, -0.636798f, -0.0208851f, 0.770748f, 
		-0.641259f, -0.677472f, -0.360305f, 0.584393f, 0.734136f, 0.345731f, 
		-0.723965f, 0.00807207f, -0.689789f, -0.568242f, -0.208829f, -0.795922f, 
		-0.663131f, -0.0468447f, -0.747036f, 0.00149585f, -0.999880f, -0.0154125f, 
		0.198273f, 0.000000f, -0.980147f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 0.0990349f, 0.000000f, -0.995084f, 
		0.242509f, 0.885837f, 0.395578f, 0.0275715f, -0.0590913f, -0.997872f, 
		-0.0319480f, 0.0520627f, 0.998133f, 0.0313824f, 0.0486258f, -0.998324f, 
		-0.0282757f, -0.0600954f, 0.997792f, 0.996037f, 0.0889364f, -0.000210425f, 
		-0.000216197f, 1.55279e-005f, -1.00000f, 0.0429598f, -0.0565376f, 0.997476f, 
		-0.0280316f, 0.0591425f, -0.997856f, 0.0293705f, 0.0546101f, 0.998076f, 
		-0.0374330f, -0.0532633f, -0.997879f, -0.00150250f, 0.999880f, -0.0154014f, 
		-0.198686f, 0.000000f, -0.980063f, -0.0994612f, 0.000000f, -0.995041f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-0.677141f, -0.643970f, 0.356067f, 0.618844f, 0.701180f, -0.354088f, 
		0.664113f, 0.666234f, 0.339244f, -0.605512f, -0.698639f, -0.381127f, 
		-0.00341625f, 0.999907f, 0.0132035f, -0.343167f, 0.115726f, 0.932118f, 
		0.996034f, -0.0889764f, -0.000210424f, 0.000216180f, -1.55112e-005f, 1.00000f, 
		0.00342590f, -0.999907f, 0.0132109f, 0.343578f, -0.115739f, 0.931965f, 
		-0.387967f, 0.899119f, 0.202648f, 0.379069f, -0.890797f, -0.250573f, 
		-0.366700f, 0.902770f, -0.224805f, 0.406539f, -0.890789f, 0.203027f, 
		-0.379102f, 0.890791f, 0.250545f, 0.387994f, -0.899113f, -0.202623f, 
		-0.387345f, 0.897596f, -0.210438f, 0.377058f, -0.900522f, 0.216535f, 
		-0.00101808f, -0.999999f, 0.000635188f, -0.601176f, 0.521581f, 0.605426f, 
		-0.382416f, -0.637474f, 0.668869f, 0.387304f, 0.638131f, -0.665421f, 
		-0.387588f, -0.638144f, -0.665243f, 0.382726f, 0.637455f, 0.668709f, 
		0.000247872f, 1.00000f, -0.000235146f, 0.571948f, -0.607491f, -0.551209f, 
		-0.387050f, -0.639475f, 0.664277f, 0.381447f, 0.639504f, -0.667482f, 
		0.387701f, 0.638841f, 0.664508f, -0.381720f, -0.639531f, -0.667301f, 
		-0.394853f, 0.626016f, 0.672454f, 0.384656f, -0.626008f, -0.678346f, 
		0.395143f, -0.625970f, 0.672327f, -0.384943f, 0.625993f, -0.678197f, 
		-0.384654f, 0.625998f, 0.678356f, 0.394855f, -0.625979f, -0.672487f, 
		-0.395136f, 0.626010f, -0.672293f, 0.384957f, -0.625997f, 0.678186f, 
		-0.118384f, -0.634051f, -0.764176f, -0.118058f, -0.634060f, 0.764219f, 
		-0.139103f, 0.0925405f, -0.985945f, 0.179538f, -0.0884671f, 0.979765f, 
		-0.139103f, 0.0925405f, -0.985945f, 0.177926f, -0.124750f, -0.976104f, 
		-0.177443f, 0.0816013f, 0.980742f, 0.196871f, 0.0919267f, -0.976110f, 
		-0.149407f, -0.104475f, 0.983241f, 0.138897f, -0.985973f, 0.0925491f, 
		-0.149468f, 0.986018f, -0.0736675f, 0.138897f, -0.985973f, 0.0925491f, 
		-0.149468f, 0.986018f, -0.0736675f, -0.158379f, -0.981143f, 0.110787f, 
		0.182008f, 0.977425f, -0.107298f, 0.118743f, 0.990418f, 0.0705104f, 
		-0.166679f, -0.978284f, -0.123205f, -0.747272f, -0.496826f, 0.441302f, 
		0.703207f, 0.579362f, -0.412115f, -0.788543f, -0.406222f, 0.461718f, 
		-0.776180f, -0.314200f, 0.546646f, -0.848161f, -0.358098f, 0.390370f, 
		-0.717088f, -0.427719f, 0.550311f, -0.797694f, -0.458108f, 0.392201f, 
		0.795529f, 0.461876f, -0.392179f, 0.728888f, 0.407275f, -0.550317f, 
		0.734133f, 0.555568f, -0.390375f, 0.660125f, 0.515209f, -0.546621f, 
		0.746007f, 0.479893f, -0.461710f, -0.759453f, 0.573689f, 0.306776f, 
		-0.753376f, 0.433418f, 0.494544f, -0.762641f, 0.497167f, 0.413768f, 
		0.778173f, -0.500616f, -0.379250f, 0.757554f, -0.395014f, -0.519688f, 
		0.767583f, -0.455569f, -0.450859f, 0.178342f, -0.124706f, 0.976034f, 
		-0.166483f, 0.0775456f, -0.982990f, 0.197282f, 0.0919078f, 0.976029f, 
		-0.149810f, -0.104436f, -0.983184f, -0.794340f, 0.401405f, 0.455958f, 
		-0.708549f, 0.564117f, 0.423946f, -0.844186f, 0.399613f, 0.357294f, 
		-0.767938f, 0.560387f, 0.310221f, -0.782887f, 0.473367f, 0.403747f, 
		0.740894f, -0.473364f, -0.476448f, 0.652410f, -0.560385f, -0.510225f, 
		0.731273f, -0.399630f, -0.552753f, 0.721251f, -0.564114f, -0.401960f, 
		0.791846f, -0.401385f, -0.460294f, -0.761523f, -0.566943f, -0.314098f, 
		-0.837057f, -0.412314f, -0.359630f, -0.699386f, -0.569625f, -0.431725f, 
		-0.785878f, -0.299636f, -0.540938f, 0.841386f, 0.329693f, 0.428219f, 
		0.796667f, 0.462268f, 0.389398f, 0.801201f, 0.329940f, 0.499217f, 
		0.654341f, 0.546171f, 0.523007f, 0.797344f, 0.400429f, 0.451552f, 
		-0.891923f, -0.377808f, 0.248464f, -0.787471f, -0.371085f, 0.492123f, 
		-0.796726f, -0.487711f, 0.356883f, -0.637740f, -0.504162f, 0.582331f, 
		0.755442f, 0.300326f, -0.582333f, 0.820698f, 0.446195f, -0.356882f, 
		0.715053f, 0.496509f, -0.492116f, 0.773118f, 0.583566f, -0.248472f, 
		-0.746223f, 0.479878f, -0.461376f, -0.660394f, 0.515212f, -0.546293f, 
		-0.734313f, 0.555551f, -0.390061f, -0.729150f, 0.407270f, -0.549974f, 
		-0.795708f, 0.461836f, -0.391863f, 0.797885f, -0.458081f, 0.391845f, 
		0.717376f, -0.427709f, 0.549943f, 0.848340f, -0.358060f, 0.390015f, 
		0.776455f, -0.314183f, 0.546266f, 0.788770f, -0.406197f, 0.461352f, 
		-0.00467255f, 0.999908f, 0.0127292f, -0.246225f, 0.000000f, 0.969213f, 
		-0.342123f, 0.000000f, 0.939655f, -0.00330051f, -0.999908f, 0.0131438f, 
		-0.875767f, -0.463663f, -0.134345f, -0.840499f, -0.492209f, -0.226478f, 
		-0.810786f, -0.455120f, -0.368092f, -0.835680f, -0.532716f, -0.133617f, 
		-0.810343f, -0.500191f, -0.305209f, -0.764820f, -0.529451f, -0.367059f, 
		-0.665531f, -0.522610f, -0.532867f, 0.785384f, 0.314956f, 0.532892f, 
		0.840956f, 0.397569f, 0.367057f, 0.838370f, 0.451639f, 0.305218f, 
		0.879195f, 0.457347f, 0.133599f, 0.799564f, 0.474541f, 0.368115f, 
		0.846534f, 0.481756f, 0.226475f, 0.839430f, 0.526598f, 0.134359f, 
		-0.839274f, 0.522218f, -0.151352f, -0.796681f, 0.469228f, -0.380953f, 
		-0.878746f, 0.452921f, -0.150559f, -0.836330f, 0.447469f, -0.316740f, 
		-0.837011f, 0.393760f, -0.379954f, -0.785745f, 0.407126f, -0.465675f, 
		0.658791f, -0.514644f, 0.548759f, 0.762261f, -0.524016f, 0.379955f, 
		0.808511f, -0.495940f, 0.316786f, 0.835559f, -0.528347f, 0.150632f, 
		0.807035f, -0.451091f, 0.381065f, 0.839174f, -0.487849f, 0.240397f, 
		0.875334f, -0.459239f, 0.151292f, 0.246629f, 0.000000f, 0.969110f, 
		0.00467812f, -0.999908f, 0.0127276f, 0.00330991f, 0.999908f, 0.0131499f, 
		0.342524f, 0.000000f, 0.939509f, 0.112524f, 0.633448f, 0.765560f, 
		-0.671139f, -0.521072f, 0.527311f, -0.767201f, -0.358551f, 0.531830f, 
		-0.811877f, -0.297903f, 0.502105f, -0.835623f, -0.130929f, 0.533472f, 
		-0.813105f, -0.359594f, 0.457769f, -0.841462f, -0.221690f, 0.492743f, 
		-0.876329f, -0.131636f, 0.463379f, 0.839228f, 0.131639f, -0.527606f, 
		0.847247f, 0.221689f, -0.482727f, 0.802781f, 0.359599f, -0.475638f, 
		0.879611f, 0.130924f, -0.457321f, 0.840574f, 0.297908f, -0.452423f, 
		0.844001f, 0.358557f, -0.398872f, 0.792096f, 0.521070f, -0.317915f, 
		-0.882554f, 0.102702f, -0.458858f, -0.821715f, 0.344464f, -0.454013f, 
		-0.838910f, 0.102704f, -0.534493f, -0.818740f, 0.284004f, -0.499005f, 
		-0.704322f, 0.420694f, -0.571793f, -0.670546f, 0.533421f, -0.515588f, 
		0.813675f, -0.407274f, 0.414803f, 0.852636f, -0.257250f, 0.454791f, 
		0.759125f, -0.406553f, 0.508373f, 0.815830f, -0.329120f, 0.475502f, 
		0.793198f, -0.148933f, 0.590471f, 0.100381f, -0.194377f, -0.975777f, 
		-0.117867f, 0.184348f, 0.975768f, 0.117864f, 0.184343f, -0.975769f, 
		-0.100379f, -0.194373f, 0.975778f, 0.118274f, -0.184334f, 0.975721f, 
		-0.100788f, 0.194361f, -0.975738f, 0.100786f, 0.194358f, 0.975739f, 
		-0.118271f, -0.184330f, -0.975723f, 0.168835f, -0.0514345f, -0.984301f, 
		-7.96238e-005f, -0.00216368f, 0.999998f, -0.759316f, -0.0423916f, 0.649339f, 
		0.401119f, -0.0301626f, -0.915529f, 0.740645f, -0.0569590f, -0.669478f, 
		-7.96238e-005f, -0.00216368f, 0.999998f, 0.168835f, -0.0514345f, -0.984301f, 
		0.112197f, 0.633452f, -0.765605f, -0.326141f, 0.162958f, 0.931169f, 
		0.339044f, -0.226540f, -0.913087f, -0.305233f, -0.194721f, 0.932157f, 
		0.322191f, 0.167379f, -0.931760f, -0.594945f, 0.0730804f, -0.800437f, 
		-0.440505f, 0.0363266f, -0.897015f, -0.361969f, 3.71467e-005f, -0.932190f, 
		0.361967f, 4.66679e-005f, 0.932191f, 0.440497f, 0.0363371f, 0.897018f, 
		0.594937f, 0.0730920f, 0.800442f, -0.322551f, -0.167327f, -0.931645f, 
		0.305656f, 0.194691f, 0.932025f, 0.350087f, -0.179762f, 0.919307f, 
		-0.307591f, 0.185435f, -0.933275f, -0.818370f, -0.329111f, -0.471123f, 
		-0.860915f, -0.458361f, -0.220749f, -0.768345f, -0.522430f, -0.369747f, 
		-0.791879f, -0.601504f, -0.105457f, 0.916870f, 0.385011f, 0.105432f, 
		0.836637f, 0.404102f, 0.369783f, 0.827424f, 0.516340f, 0.220820f, 
		0.694250f, 0.544066f, 0.471178f, -0.839395f, 0.526576f, 0.134659f, 
		-0.846444f, 0.481749f, 0.226827f, -0.799409f, 0.474529f, 0.368467f, 
		-0.879146f, 0.457346f, 0.133927f, -0.838251f, 0.451624f, 0.305567f, 
		-0.840807f, 0.397553f, 0.367418f, -0.785145f, 0.314935f, 0.533258f, 
		0.665256f, -0.522559f, -0.533260f, 0.764666f, -0.529427f, -0.367415f, 
		0.810226f, -0.500166f, -0.305562f, 0.835649f, -0.532689f, -0.133916f, 
		0.810636f, -0.455091f, -0.368458f, 0.840419f, -0.492185f, -0.226826f, 
		0.875726f, -0.463647f, -0.134669f, 0.000000f, 1.00000f, -5.78099e-015f, 
		0.000000f, -1.00000f, -1.28608e-019f, -0.713561f, -0.697700f, -0.0636064f, 
		0.735477f, 0.672444f, -0.0830238f, 0.000000f, -1.00000f, -1.28608e-019f, 
		0.000000f, 1.00000f, -5.78099e-015f, -0.323417f, 0.944336f, 0.0602591f, 
		-0.622543f, 0.782005f, 0.0301272f, -0.525999f, 0.849958f, 0.0299277f, 
		0.0949600f, -0.995465f, -0.00558006f, 0.465255f, -0.884532f, 0.0337939f, 
		0.643027f, -0.763164f, 0.0640170f, 0.000230372f, 2.37378e-005f, 1.00000f, 
		-0.000233403f, 1.85072e-005f, -1.00000f, -0.779804f, -0.0696635f, -0.622136f, 
		0.759326f, -0.0530313f, 0.648546f, -0.000233403f, 1.85072e-005f, -1.00000f, 
		0.000233404f, 1.84863e-005f, 1.00000f, -0.594566f, 0.0730703f, 0.800720f, 
		-0.434950f, 0.0340257f, 0.899812f, -0.354987f, -0.00281071f, 0.934867f, 
		0.424868f, 0.0254289f, -0.904898f, 0.723267f, -0.344553f, -0.598471f, 
		0.727798f, 0.0124630f, -0.685679f, 0.671148f, 0.167612f, -0.722126f, 
		0.593002f, -0.299692f, -0.747351f, 0.641165f, -0.0208958f, -0.767119f, 
		-0.416182f, -0.546847f, 0.726465f, 0.420712f, 0.546876f, -0.723829f, 
		-0.421039f, -0.546912f, -0.723611f, 0.416485f, 0.546905f, 0.726248f, 
		-0.00337791f, 0.999991f, 0.00257239f, -0.731677f, -0.369493f, 0.572821f, 
		-0.622951f, -0.536165f, 0.569614f, -0.362520f, -0.696197f, 0.619587f, 
		-0.824208f, -0.315929f, -0.469969f, 0.801288f, 0.378275f, 0.463515f, 
		0.416194f, 0.546866f, -0.726443f, -0.420721f, -0.546882f, 0.723819f, 
		0.421018f, 0.546873f, 0.723653f, -0.416475f, -0.546871f, -0.726279f, 
		-0.824001f, -0.315954f, 0.470315f, 0.801106f, 0.378248f, -0.463851f, 
		0.814429f, 0.349510f, 0.463194f, -0.808563f, -0.349475f, -0.473386f, 
		-0.00269200f, -0.999996f, -0.00127896f, -0.647225f, 0.467995f, -0.601730f, 
		-0.787283f, 0.213468f, -0.578461f, -0.711295f, 0.332058f, -0.619514f, 
		-0.387153f, 0.567593f, 0.726602f, 0.347039f, -0.640880f, 0.684716f, 
		0.142505f, 0.958066f, -0.248602f, -0.682527f, -0.730815f, 0.00809601f, 
		0.740205f, 0.672354f, 0.00603842f, 0.387111f, -0.567659f, -0.726573f, 
		-0.334476f, 0.646743f, -0.685455f, -0.296600f, 0.787774f, -0.539852f, 
		-0.303263f, 0.711062f, -0.634368f, -0.692255f, -0.721603f, -0.00854297f, 
		0.700361f, 0.713724f, -0.00958187f, -0.856493f, -0.139441f, 0.496967f, 
		0.872541f, 0.167549f, -0.458912f, -0.864639f, -0.0553362f, 0.499336f, 
		-0.864639f, -0.0553362f, 0.499336f, 0.864471f, 0.0554098f, -0.499619f, 
		-0.864587f, -0.0552472f, -0.499437f, -0.864587f, -0.0552472f, -0.499437f, 
		0.864954f, 0.0551699f, 0.498809f, -0.891268f, -0.0634075f, -0.449022f, 
		0.864365f, 0.0793686f, 0.496562f, 0.0993878f, 0.987600f, -0.121522f, 
		-0.00198346f, -0.999998f, 0.000694320f, -0.100860f, 0.986686f, -0.127582f, 
		0.864138f, 0.0795223f, -0.496932f, -0.891055f, -0.0635377f, 0.449426f, 
		0.863644f, 0.0793444f, 0.497819f, -0.863119f, -0.0793970f, -0.498719f, 
		0.860680f, 0.115466f, -0.495880f, -0.859471f, -0.115517f, 0.497962f, 
		-0.859471f, -0.115517f, 0.497962f, -0.860875f, -0.115471f, -0.495541f, 
		-0.860875f, -0.115471f, -0.495541f, 0.859770f, 0.115450f, 0.497460f, 
		-0.453591f, 0.890793f, -0.0272554f, 0.429965f, -0.902772f, -0.0115176f, 
		-0.134396f, -0.885208f, 0.445360f, 0.214297f, -0.391622f, -0.894823f, 
		-0.231705f, 0.381499f, 0.894858f, 0.231703f, 0.381496f, -0.894860f, 
		-0.214292f, -0.391618f, 0.894826f, 0.232094f, -0.381498f, 0.894758f, 
		-0.214668f, 0.391604f, -0.894742f, 0.214648f, 0.391580f, 0.894758f, 
		-0.232078f, -0.381471f, -0.894773f, -0.842830f, -0.235486f, 0.483926f, 
		0.840312f, 0.235478f, -0.488289f, -0.840522f, -0.235472f, -0.487930f, 
		0.843036f, 0.235487f, 0.483566f, 0.00299511f, -0.00296712f, -0.999991f, 
		0.415309f, -0.370733f, 0.830708f, 0.700064f, -0.298511f, 0.648693f, 
		0.602724f, -0.344110f, 0.719938f, -0.435611f, 0.899966f, 0.0174227f, 
		0.453591f, -0.890793f, 0.0272645f, -0.00342746f, 0.00296517f, -0.999990f, 
		-0.549421f, 0.335179f, 0.765371f, -0.134583f, -0.885208f, -0.445303f, 
		0.307914f, 0.667895f, -0.677573f, -0.444799f, -0.448678f, 0.775140f, 
		0.448535f, 0.448669f, -0.772989f, -0.303477f, -0.676532f, -0.670974f, 
		-0.448880f, -0.448707f, -0.772767f, 0.445143f, 0.448697f, 0.774931f, 
		0.569714f, 0.671258f, 0.474171f, 0.772047f, 0.349705f, 0.530707f, 
		0.718814f, 0.484106f, 0.498946f, 0.00149363f, 0.999998f, 0.00157474f, 
		0.735265f, -0.448140f, 0.508483f, -0.452480f, 0.439811f, 0.775776f, 
		0.445270f, -0.439821f, -0.779931f, -0.445600f, 0.439836f, -0.779734f, 
		0.452796f, -0.439837f, 0.775577f, 0.444805f, 0.448737f, -0.775103f, 
		-0.448544f, -0.448742f, 0.772942f, -0.445120f, -0.448735f, -0.774923f, 
		0.448869f, 0.448677f, 0.772791f, 0.165573f, 0.962208f, -0.216195f, 
		-0.445283f, 0.439816f, 0.779926f, 0.452476f, -0.439813f, -0.775777f, 
		-0.452802f, 0.439810f, -0.775589f, 0.445603f, -0.439825f, 0.779738f, 
		0.323991f, -0.692496f, -0.644577f, 0.00606802f, -0.000153447f, -0.999982f, 
		0.633696f, 0.370778f, 0.678935f, 0.158956f, 0.873895f, -0.459391f, 
		-0.00541801f, -0.00158291f, -0.999984f, -0.644587f, -0.372693f, 0.667539f, 
		-0.324268f, 0.692479f, -0.644455f, -0.444782f, -0.859703f, 0.251157f, 
		0.467514f, 0.824229f, -0.319495f, 0.444829f, 0.860004f, 0.250042f, 
		-0.490454f, -0.819866f, -0.295422f, 0.468592f, 0.0191422f, 0.883207f, 
		-1.45169e-005f, -1.00000f, 9.90107e-006f, -2.28478e-006f, 1.00000f, 3.05527e-006f, 
		-0.134076f, -0.470215f, -0.872308f, 0.159152f, 0.873901f, 0.459312f, 
		-0.133701f, -0.470213f, 0.872367f, 0.173662f, 0.0682437f, 0.982438f, 
		-0.197205f, -0.0288862f, -0.979937f, 0.173662f, 0.0682437f, 0.982438f, 
		-0.197205f, -0.0288862f, -0.979937f, -0.488468f, 5.44360e-006f, -0.872582f, 
		-2.68247e-006f, 1.00000f, -1.00611e-005f, -2.53304e-006f, -1.00000f, -9.77034e-007f, 
		-0.189270f, 0.978943f, 0.0764634f, 0.202123f, -0.978945f, -0.0285149f, 
		0.202123f, -0.978945f, -0.0285149f, -0.189270f, 0.978943f, 0.0764634f, 
		0.181530f, -0.0156578f, -0.983261f, -0.216386f, -0.0189557f, 0.976124f, 
		0.00247804f, 0.999997f, -0.000670637f, 0.786036f, -0.589680f, -0.185538f, 
		0.812768f, 0.350818f, -0.465118f, -0.808992f, -0.350821f, 0.471653f, 
		-0.812947f, -0.350844f, -0.464786f, 0.809174f, 0.350863f, 0.471310f, 
		0.161537f, 0.0642163f, -0.984775f, 0.161537f, 0.0642163f, -0.984775f, 
		-0.183945f, -0.0276357f, 0.982548f, -0.183945f, -0.0276357f, 0.982548f, 
		0.161537f, 0.0642163f, -0.984775f, 0.000472704f, -0.999999f, 0.00152986f, 
		-0.357563f, 0.374562f, 0.855483f, -0.915976f, -0.00251595f, -0.401225f, 
		-0.825671f, 0.0127223f, -0.564009f, -0.909778f, -0.112855f, -0.399460f, 
		-0.820207f, -0.115476f, -0.560291f, -0.879932f, -0.0419659f, -0.473242f, 
		0.879936f, -0.0419529f, 0.473236f, 0.820220f, -0.115465f, 0.560274f, 
		0.909785f, -0.112840f, 0.399449f, 0.825669f, 0.0127342f, 0.564012f, 
		0.915978f, -0.00250453f, 0.401220f, 0.00118942f, 0.999996f, 0.00248691f, 
		0.357935f, -0.374602f, 0.855311f, 0.181947f, -0.0156644f, 0.983184f, 
		-0.216827f, -0.0189669f, -0.976026f, 0.463183f, -0.286363f, -0.838724f, 
		-0.479509f, 0.258163f, 0.838703f, -0.463195f, -0.286373f, 0.838714f, 
		0.479525f, 0.258186f, -0.838687f, 0.468815f, -0.253865f, 0.846029f, 
		-0.454101f, 0.278726f, -0.846229f, -0.489856f, -0.243802f, -0.837020f, 
		0.507277f, 0.319985f, 0.800174f, -0.00206417f, 0.999996f, 0.00198115f, 
		-0.439754f, -0.712943f, 0.546195f, -0.684682f, -0.641459f, 0.346035f, 
		-0.800187f, -0.517208f, 0.303638f, -0.916303f, -0.400421f, -0.00722785f, 
		-0.828132f, -0.546160f, 0.126122f, -0.943457f, -0.329934f, 0.0321384f, 
		-0.884659f, -0.462266f, -0.0607361f, -0.942799f, -0.329675f, -0.0494417f, 
		0.951039f, 0.299598f, -0.0759301f, 0.821537f, 0.569627f, -0.0245366f, 
		0.904786f, 0.412279f, 0.106716f, 0.816598f, 0.566933f, 0.108423f, 
		0.00428085f, 0.999990f, -0.00139223f, 0.357579f, -0.374646f, -0.855440f, 
		0.505280f, -0.824559f, -0.254547f, -0.507107f, 0.800233f, 0.320108f, 
		-0.468732f, 0.846105f, -0.253766f, 0.454018f, -0.846295f, 0.278661f, 
		-0.948702f, 0.0751046f, 0.307123f, -0.810350f, 0.0261172f, 0.585363f, 
		-0.899651f, -0.104964f, 0.423806f, -0.805580f, -0.107208f, 0.582706f, 
		0.818690f, -0.125904f, -0.560264f, 0.940777f, -0.0307919f, -0.337626f, 
		0.878241f, 0.0614101f, -0.474258f, 0.960844f, 0.123368f, -0.248112f, 
		0.322172f, -0.575402f, -0.751744f, -0.337002f, 0.566882f, 0.751714f, 
		0.337006f, 0.566891f, -0.751706f, -0.322173f, -0.575404f, 0.751741f, 
		0.337294f, -0.566854f, 0.751604f, -0.322510f, 0.575420f, -0.751585f, 
		0.322508f, 0.575417f, 0.751587f, -0.337301f, -0.566856f, -0.751600f, 
		-0.0241877f, -0.0240816f, 0.999417f, -0.691136f, -0.409712f, -0.595372f, 
		0.506992f, 0.319965f, -0.800363f, -0.505135f, -0.254418f, 0.824688f, 
		0.530497f, -0.279173f, -0.800397f, -0.455460f, 0.302017f, 0.837461f, 
		0.463607f, -0.286365f, 0.838489f, -0.479890f, 0.258147f, -0.838490f, 
		0.479905f, 0.258165f, 0.838476f, -0.463607f, -0.286368f, -0.838488f, 
		0.604846f, 0.0160704f, -0.796180f, -1.28024e-005f, -1.00000f, -8.72476e-006f, 
		-2.28478e-006f, 1.00000f, 3.05527e-006f, 0.129423f, 0.469575f, 0.873355f, 
		-0.467933f, -0.343583f, 0.814242f, 0.470825f, 0.343540f, -0.812591f, 
		-0.831218f, 0.123296f, 0.542103f, -0.890729f, 0.153124f, 0.427967f, 
		-0.950039f, 0.0298242f, 0.310704f, -0.990164f, 0.0508103f, 0.130358f, 
		-0.927125f, -0.00934470f, 0.374635f, -0.990569f, -0.0370997f, 0.131895f, 
		0.990569f, -0.0370818f, -0.131899f, 0.975037f, 0.00577291f, -0.221966f, 
		0.932956f, -0.0103160f, -0.359842f, 0.990383f, 0.0439617f, -0.131186f, 
		0.954089f, 0.0286945f, -0.298145f, 0.930254f, 0.0767852f, -0.358791f, 
		0.844766f, 0.120926f, -0.521293f, -0.471186f, -0.343556f, -0.812375f, 
		0.468295f, 0.343603f, 0.814025f, -0.000132255f, -1.00000f, -6.23313e-005f, 
		-0.232870f, 0.931602f, 0.279086f, 0.129061f, 0.469570f, -0.873411f, 
		-0.000660679f, 0.999999f, -0.000945089f, -0.000660679f, 0.999999f, -0.000945089f, 
		0.144196f, -0.972724f, -0.181703f, -0.982107f, 0.148973f, 0.115204f, 
		-0.944270f, 0.329144f, 0.00429280f, -0.911578f, 0.406562f, 0.0610946f, 
		-0.965806f, 0.257280f, -0.0320391f, -0.912077f, 0.407292f, -0.0472136f, 
		0.838472f, -0.533393f, -0.111606f, 0.895801f, -0.420679f, -0.143420f, 
		0.958545f, -0.283994f, -0.0231967f, 0.993743f, -0.102717f, -0.0438552f, 
		0.938648f, -0.344445f, 0.0172665f, 0.993762f, -0.102703f, 0.0434664f, 
		-0.501207f, 0.0159671f, 0.865180f, 0.000610862f, 0.999999f, -0.000902267f, 
		-0.621923f, -0.0159653f, 0.782916f, -2.53304e-006f, -1.00000f, -9.77034e-007f, 
		0.605208f, 0.0160716f, 0.795905f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		0.467948f, 0.343574f, -0.814237f, -0.470838f, -0.343605f, 0.812556f, 
		-0.468296f, -0.343561f, -0.814042f, 0.471171f, 0.343644f, 0.812347f, 
		-0.406848f, -0.0265319f, 0.913111f, 0.368777f, -0.0227014f, -0.929241f, 
		2.34115e-005f, 1.00000f, -3.53424e-006f, 0.209532f, -0.940919f, -0.266024f, 
		-0.369187f, 0.0227001f, -0.929078f, 0.407285f, 0.0265425f, 0.912915f, 
		-0.845930f, 0.112786f, -0.521231f, -0.900507f, 0.144075f, -0.410280f, 
		-0.960748f, 0.0238498f, -0.276394f, -0.994026f, 0.0440747f, -0.0998444f, 
		-0.941869f, -0.0163159f, -0.335584f, -0.994024f, -0.0441532f, -0.0998349f, 
		0.982605f, -0.115741f, 0.145232f, 0.946924f, -0.00385052f, 0.321434f, 
		0.915597f, -0.0604963f, 0.397521f, 0.967383f, 0.0332601f, 0.251124f, 
		0.916053f, 0.0475672f, 0.398226f, 3.04462e-005f, 1.00000f, -0.000507781f, 
		0.103891f, -0.991997f, 0.0717558f, -0.741617f, -0.380452f, -0.552503f, 
		-0.775590f, -0.166851f, -0.608787f, -0.769453f, -0.0268249f, -0.638139f, 
		-0.00985703f, -9.81101e-006f, 0.999951f, -0.759916f, 0.394370f, -0.516720f, 
		-0.770923f, -0.460338f, 0.440189f, 0.766485f, 0.460330f, -0.447880f, 
		-0.766667f, -0.460374f, -0.447523f, 0.771095f, 0.460372f, 0.439853f, 
		-0.485121f, -0.232633f, 0.842935f, 0.487088f, 0.232646f, -0.841797f, 
		-0.487459f, -0.232653f, -0.841580f, 0.485505f, 0.232576f, 0.842730f, 
		-0.488616f, 0.227399f, 0.842344f, 0.484834f, -0.227393f, -0.844529f, 
		0.781973f, -0.0536579f, 0.620999f, 0.757697f, 0.270916f, 0.593717f, 
		0.697513f, 0.466897f, 0.543583f, -0.925724f, -0.378199f, 0.00116484f, 
		0.948790f, 0.315872f, 0.00467966f, -0.485198f, 0.227397f, -0.844319f, 
		0.488973f, -0.227425f, 0.842130f, -0.647840f, -0.392403f, 0.652934f, 
		0.663653f, 0.364995f, -0.652950f, -0.663652f, 0.365005f, 0.652945f, 
		0.647834f, -0.392408f, -0.652937f, -0.663962f, -0.365005f, -0.652630f, 
		0.648148f, 0.392371f, 0.652648f, -0.628114f, 0.371251f, -0.683846f, 
		0.688510f, -0.388012f, 0.612700f, 0.485114f, 0.232659f, -0.842932f, 
		-0.487082f, -0.232630f, 0.841804f, 0.134253f, 0.987737f, 0.0796932f, 
		0.000210382f, -1.00000f, 1.94382e-005f, 0.000210382f, -1.00000f, 1.94382e-005f, 
		0.487444f, 0.232586f, 0.841607f, -0.485474f, -0.232636f, -0.842731f, 
		-0.484850f, 0.227371f, 0.844525f, 0.488621f, -0.227398f, -0.842342f, 
		0.936897f, 0.349552f, -0.00607356f, -0.936902f, -0.349545f, -0.00569809f, 
		-0.488965f, 0.227421f, -0.842136f, 0.485179f, -0.227415f, 0.844324f, 
		0.384587f, 0.922698f, -0.0268453f, -0.301644f, -0.953417f, -0.00252647f, 
		0.408351f, -0.719317f, -0.561989f, -0.418615f, 0.713419f, 0.561956f, 
		0.418618f, 0.713420f, -0.561952f, -0.408357f, -0.719322f, 0.561978f, 
		0.418839f, -0.713412f, 0.561798f, -0.408542f, 0.719268f, -0.561913f, 
		0.408568f, 0.719291f, 0.561865f, -0.418853f, -0.713433f, -0.561761f, 
		0.651298f, -0.0191477f, 0.758581f, 0.00128315f, -0.999997f, 0.00197481f, 
		-0.145578f, -0.262694f, -0.953834f, -0.634033f, 5.44359e-006f, -0.773306f, 
		-0.00246653f, 0.999973f, -0.00689259f, -0.145167f, -0.262683f, 0.953900f, 
		-0.495875f, -0.117372f, 0.860425f, 0.496870f, 0.117417f, -0.859845f, 
		-0.497229f, -0.117379f, -0.859643f, 0.496228f, 0.117401f, 0.860218f, 
		0.00100176f, 0.999999f, -0.000472631f, 0.795543f, -0.603319f, 0.0558422f, 
		-0.496851f, -0.117358f, 0.859864f, 0.495858f, 0.117365f, -0.860436f, 
		0.497221f, 0.117414f, 0.859643f, -0.496226f, -0.117376f, -0.860223f, 
		-0.499803f, 0.0280314f, 0.865685f, 0.499454f, -0.0280491f, -0.865886f, 
		-0.998472f, -0.0552538f, -3.73099e-005f, -0.998472f, -0.0552538f, -3.73099e-005f, 
		0.998470f, 0.0552951f, -0.000646951f, -0.996848f, -0.0793198f, -0.00171488f, 
		0.996404f, 0.0633219f, 0.0563064f, -0.499806f, 0.0280518f, -0.865683f, 
		0.500162f, -0.0280117f, 0.865479f, -0.00770815f, 0.999969f, -0.00128232f, 
		-0.840895f, -0.416231f, -0.345901f, -0.499435f, 0.0281264f, 0.865895f, 
		0.499797f, -0.0281180f, -0.865686f, 0.996839f, 0.0794444f, -0.000723191f, 
		-0.996839f, -0.0794524f, -0.000345511f, 0.499806f, -0.0280508f, 0.865683f, 
		-0.500155f, 0.0280296f, -0.865482f, 0.993314f, 0.115437f, 0.000935541f, 
		-0.993314f, -0.115437f, 0.00131886f, -0.993314f, -0.115437f, 0.00131886f, 
		0.00213741f, -0.999942f, 0.0105430f, -0.0984233f, -0.143426f, -0.984755f, 
		-0.179290f, -0.958422f, 0.221998f, 0.142467f, 0.262202f, 0.954439f, 
		0.861977f, 0.272367f, -0.427566f, 0.790319f, 0.480753f, -0.379832f, 
		0.820900f, 0.393949f, -0.413434f, 0.00213273f, -0.999942f, -0.0105439f, 
		-0.0980011f, -0.143421f, 0.984798f, 0.464113f, -0.810483f, -0.357373f, 
		-0.469731f, 0.807192f, 0.357483f, 0.469732f, 0.807190f, -0.357486f, 
		-0.464116f, -0.810482f, 0.357373f, 0.469904f, -0.807210f, 0.357214f, 
		-0.464276f, 0.810496f, -0.357133f, 0.464272f, 0.810500f, 0.357128f, 
		-0.469899f, -0.807212f, -0.357217f, -0.00170236f, -0.999998f, 0.000905912f, 
		-0.887040f, 0.318714f, 0.334038f, -0.811714f, 0.450458f, 0.371764f, 
		-0.666062f, 0.488193f, 0.563941f, 0.142073f, 0.262187f, -0.954502f, 
		-0.00361044f, -0.999993f, 0.000688887f, -0.884983f, 0.435986f, -0.163467f, 
		-0.971874f, -0.235492f, -0.00231171f, 0.971861f, 0.235538f, -0.00270414f, 
		-0.179372f, -0.958422f, -0.221931f, 0.00189237f, -0.999998f, 0.000687309f, 
		0.584932f, 0.806056f, 0.0901545f, 0.747929f, 0.659809f, -0.0724800f, 
		0.668949f, 0.743107f, 0.0172983f, -0.00180896f, -0.999998f, -0.000761650f, 
		-0.893340f, 0.298577f, -0.335850f, -0.819274f, 0.434504f, -0.374160f, 
		-0.666305f, 0.488216f, -0.563633f, -0.00206010f, -0.999996f, 0.00195172f, 
		-0.893812f, 0.397341f, 0.207895f, 0.716611f, -0.424228f, -0.553624f, 
		-0.725600f, 0.408588f, 0.553679f, -0.716609f, -0.424235f, 0.553621f, 
		0.725598f, 0.408595f, -0.553676f, 0.749565f, -0.420463f, 0.511236f, 
		-0.738968f, 0.438813f, -0.511243f, -0.749565f, -0.420472f, -0.511230f, 
		0.738963f, 0.438820f, 0.511244f, -0.718179f, -0.554487f, 0.420433f, 
		0.723033f, 0.554475f, -0.412044f, -0.723186f, -0.554491f, -0.411755f, 
		0.718344f, 0.554500f, 0.420134f, 0.749355f, -0.511434f, -0.420598f, 
		-0.738764f, 0.511413f, 0.438958f, -0.723496f, -0.553209f, 0.412933f, 
		0.701168f, 0.576994f, -0.418858f, -0.749554f, 0.511360f, -0.420333f, 
		0.738971f, -0.511368f, 0.438663f, -0.708347f, -0.525071f, -0.471747f, 
		0.723188f, 0.554301f, 0.412008f, 0.733699f, 0.430514f, -0.525684f, 
		-0.749323f, -0.420438f, 0.511612f, 0.736459f, -0.407413f, -0.540039f, 
		-0.716623f, 0.424205f, 0.553625f, 0.725864f, 0.408595f, 0.553328f, 
		-0.721259f, -0.434175f, -0.539702f, 0.738965f, -0.438802f, 0.511257f, 
		-0.739849f, 0.420225f, -0.525390f, -0.833412f, 0.477442f, 0.278343f, 
		0.830102f, -0.484146f, -0.276646f, -0.830137f, -0.483113f, 0.278342f, 
		0.832923f, 0.477487f, -0.279725f, -0.828731f, 0.482269f, -0.283941f, 
		0.864214f, -0.435902f, 0.251246f, -0.833855f, -0.476441f, -0.278728f, 
		0.829369f, 0.483279f, 0.280337f, 0.491895f, -0.854580f, -0.166532f, 
		-0.494084f, 0.853328f, 0.166470f, 0.494085f, 0.853328f, -0.166469f, 
		-0.491896f, -0.854580f, 0.166531f, 0.494168f, -0.853320f, 0.166261f, 
		-0.491983f, 0.854584f, -0.166250f, 0.491986f, 0.854583f, 0.166247f, 
		-0.494168f, -0.853320f, -0.166261f, 0.196639f, 0.143082f, 0.969980f, 
		0.00106972f, -0.999942f, 0.0107041f, 0.000996967f, -0.999999f, 0.000314652f, 
		0.725730f, 0.667952f, 0.164791f, 0.196221f, 0.143084f, -0.970064f, 
		0.00106519f, -0.999942f, -0.0107046f, 0.499993f, -0.866029f, -0.000113483f, 
		-0.499996f, 0.866028f, 8.44827e-005f, 0.499996f, 0.866028f, -8.75728e-005f, 
		-0.499993f, -0.866030f, 0.000110403f, 0.405062f, 0.304577f, -0.862066f, 
		-0.784283f, -0.276577f, 0.555343f, -0.405441f, -0.304562f, -0.861893f, 
		0.423245f, -0.335207f, -0.841725f, -0.836087f, 0.542800f, -0.0795393f, 
		0.847131f, -0.486749f, 0.213179f, 0.908064f, -0.361879f, 0.210865f, 
		0.884247f, -0.444686f, 0.142694f, 0.916569f, -0.397881f, 0.0398974f, 
		0.877294f, -0.478602f, 0.0359956f, -0.423609f, 0.335179f, -0.841553f, 
		-0.888716f, -0.452174f, -0.0756511f, 0.850096f, 0.447855f, 0.277061f, 
		0.854866f, 0.362557f, 0.371155f, 0.867393f, 0.478393f, 0.137000f, 
		0.793538f, 0.542080f, 0.276490f, 0.828451f, 0.542124f, 0.140604f, 
		0.801895f, 0.596134f, 0.0398490f, -0.195834f, -0.962284f, -0.188833f, 
		-0.831279f, -0.398074f, 0.387960f, -0.745589f, -0.616068f, 0.254080f, 
		-0.785564f, -0.616052f, 0.0580433f, -0.856225f, -0.513874f, 0.0530361f, 
		0.882652f, 0.460630f, -0.0935153f, 0.862889f, 0.457877f, -0.213943f, 
		0.719896f, 0.443008f, -0.534316f, -0.936433f, -0.350825f, 0.00396021f, 
		0.936440f, 0.350810f, 0.00358123f, -0.00324688f, 0.999995f, -0.000580302f, 
		-0.837075f, -0.535549f, -0.111769f, -0.862745f, -0.503268f, 0.0489047f, 
		-0.912111f, -0.340614f, 0.228112f, -0.745632f, 0.542701f, 0.386664f, 
		-0.840657f, 0.379144f, 0.386710f, -0.898799f, 0.417519f, -0.133559f, 
		-0.918363f, 0.391271f, 0.0592966f, -0.873105f, 0.484663f, 0.0528155f, 
		0.816060f, -0.527426f, -0.236365f, 0.828191f, -0.548067f, -0.117142f, 
		0.819407f, -0.390015f, -0.420073f, 0.708079f, -0.406332f, -0.577510f, 
		0.793807f, -0.608066f, -0.0112421f, 0.841336f, -0.489495f, -0.229236f, 
		-0.844564f, 0.483949f, 0.229139f, -0.841333f, -0.489505f, 0.229223f, 
		0.844562f, 0.483957f, -0.229130f, 0.841964f, 0.224513f, -0.490603f, 
		-0.844510f, -0.228977f, 0.484120f, -0.842003f, -0.225719f, -0.489982f, 
		0.844714f, 0.228940f, 0.483782f, 0.544281f, -0.0162999f, -0.838744f, 
		-0.544277f, -0.0163127f, 0.838747f, 0.564942f, 0.0323282f, 0.824497f, 
		-0.599302f, 0.0235483f, -0.800177f, 0.812185f, -0.521790f, -0.260942f, 
		-0.846199f, 0.488419f, 0.213059f, 0.846154f, 0.488679f, 0.212642f, 
		-0.858040f, -0.442546f, -0.260616f, -0.861146f, -0.495145f, -0.115148f, 
		0.859397f, 0.498191f, 0.115073f, 0.861151f, -0.495139f, 0.115132f, 
		-0.859402f, 0.498184f, -0.115068f, 0.761721f, 0.555195f, -0.333976f, 
		0.893021f, 0.269458f, -0.360426f, 0.847253f, 0.402059f, -0.347148f, 
		0.599264f, -0.800210f, 0.0234021f, -0.564840f, 0.824562f, 0.0324451f, 
		0.00647179f, -0.999907f, -0.0120159f, 0.382056f, -0.115688f, -0.916869f, 
		0.861029f, -0.115238f, -0.495327f, -0.859271f, 0.115248f, 0.498368f, 
		-0.861231f, 0.115227f, -0.494979f, 0.859487f, -0.115209f, 0.498005f, 
		-0.00647679f, 0.999907f, -0.0120124f, -0.382457f, 0.115684f, -0.916703f, 
		0.545438f, 0.0337735f, -0.837471f, -0.599022f, 0.0235246f, 0.800387f, 
		0.544652f, -0.0162993f, 0.838504f, -0.544653f, -0.0163113f, -0.838503f, 
		-0.545203f, 0.780328f, 0.306336f, 0.537741f, -0.780339f, -0.319228f, 
		-0.537897f, 0.780325f, -0.318997f, 0.545339f, -0.780334f, 0.306080f, 
		0.858515f, 0.494081f, -0.137245f, -0.843205f, -0.523616f, 0.121783f, 
		0.875095f, -0.468478f, 0.121395f, -0.857187f, 0.496499f, -0.136819f, 
		-0.843549f, 0.216362f, 0.491541f, -0.537763f, 0.780331f, 0.319207f, 
		0.545203f, -0.780339f, -0.306307f, -0.545334f, 0.780328f, -0.306103f, 
		0.537891f, -0.780332f, 0.318990f, 0.419460f, -0.640885f, -0.642900f, 
		-0.455655f, 0.550090f, -0.699843f, -0.680686f, -0.621153f, 0.388376f, 
		0.676515f, 0.621155f, -0.395594f, -0.676703f, -0.621124f, -0.395321f, 
		0.680865f, 0.621134f, 0.388092f, -0.410625f, 0.650385f, 0.639051f, 
		0.431090f, -0.615221f, 0.660049f, 0.0140116f, 0.00334606f, 0.999896f, 
		0.765591f, 0.00771139f, -0.643281f, -0.136773f, 0.850323f, 0.508178f, 
		0.00534609f, 0.999907f, -0.0125474f, 0.472154f, 0.105452f, -0.875186f, 
		0.141650f, -0.856762f, 0.495876f, -0.00535693f, -0.999907f, -0.0125526f, 
		-0.472553f, -0.105415f, -0.874975f, -0.302284f, 0.885006f, 0.354102f, 
		-0.00213688f, -0.999994f, -0.00259547f, -0.133397f, 0.988644f, 0.0691915f, 
		-0.887722f, -0.460361f, -0.00424338f, 0.887736f, 0.460330f, -0.00463508f, 
		-0.307087f, -0.914964f, 0.261798f, -0.757229f, -0.0158025f, 0.652958f, 
		0.757228f, -0.0158216f, -0.652959f, -0.757483f, 0.0158465f, -0.652663f, 
		0.757482f, 0.0158294f, 0.652664f, 0.757025f, 0.0660129f, 0.650043f, 
		-0.00410242f, 0.999983f, 0.00419322f, -0.731669f, 0.0159694f, 0.681473f, 
		-0.624541f, -0.0159640f, 0.780829f, 2.28523e-006f, -1.00000f, -3.05493e-006f, 
		0.208222f, 0.958326f, 0.195586f, 0.949142f, 0.292587f, 0.116284f, 
		0.906216f, -0.193888f, 0.375739f, 0.789245f, -0.499346f, 0.357417f, 
		0.897306f, 0.0135589f, 0.441201f, 0.878002f, -0.376371f, 0.295731f, 
		0.923179f, -0.0887354f, 0.373988f, -0.400679f, 0.660988f, 0.634469f, 
		0.385592f, -0.705761f, 0.594324f, 0.810195f, -0.340731f, -0.476955f, 
		0.907056f, -0.0930197f, -0.410605f, 0.921125f, -0.247355f, -0.300574f, 
		0.975594f, -0.0309118f, -0.217398f, 0.938661f, 0.338424f, 0.0662215f, 
		-0.944009f, -0.0855589f, 0.318633f, -0.412358f, -0.705789f, 0.576040f, 
		0.444251f, 0.660969f, 0.604781f, -0.649481f, -0.660217f, 0.377211f, 
		0.651256f, 0.660220f, -0.374132f, 0.649638f, 0.660207f, 0.376958f, 
		-0.651405f, -0.660229f, -0.373857f, -0.695240f, -0.0384906f, -0.717747f, 
		-3.37460e-006f, 1.00000f, 1.48004e-005f, -1.45169e-005f, -1.00000f, 9.90107e-006f, 
		-0.837464f, -0.475429f, -0.269484f, -0.891558f, -0.401857f, -0.208893f, 
		-0.983628f, -0.153267f, -0.0947932f, -0.915780f, -0.0293874f, -0.400605f, 
		0.000182313f, 1.00000f, -3.81505e-005f, -0.155748f, -0.986274f, -0.0548204f, 
		0.832666f, -0.00901230f, -0.553702f, -0.832665f, -0.00903334f, 0.553703f, 
		0.859401f, 0.0105981f, 0.511192f, -0.859398f, 0.0105770f, -0.511198f, 
		-0.832169f, -0.554500f, 0.00500573f, 0.832157f, 0.554521f, 0.00465409f, 
		-0.859282f, 0.511389f, 0.0107729f, 0.859284f, -0.511393f, 0.0104069f, 
		0.805026f, 0.593033f, -0.0156558f, -0.787775f, -0.614966f, -0.0350354f, 
		0.859128f, 0.0106128f, -0.511650f, -0.850615f, 0.00594067f, 0.525756f, 
		0.841711f, -0.0154353f, 0.539707f, -0.832917f, -0.00903678f, -0.553324f, 
		0.478341f, 0.000000f, -0.878174f, 0.00522279f, -0.999908f, -0.0125055f, 
		0.00651770f, 0.999908f, -0.0118912f, 0.387923f, 0.000000f, -0.921692f, 
		-0.00522821f, 0.999908f, -0.0125031f, -0.478728f, 0.000000f, -0.877963f, 
		-0.388330f, 0.000000f, -0.921520f, -0.00652267f, -0.999908f, -0.0118876f, 
		-0.959774f, -0.00387441f, 0.280747f, 0.960033f, -0.00388826f, -0.279859f, 
		-0.960371f, 0.00300520f, -0.278708f, 0.960772f, 0.00304399f, 0.277323f, 
		-0.634818f, -0.680086f, 0.366728f, 0.634853f, 0.680082f, -0.366674f, 
		-0.635028f, -0.680054f, -0.366422f, 0.635024f, 0.680023f, 0.366487f, 
		0.865667f, -0.400431f, -0.300459f, 0.965582f, -0.129382f, -0.225636f, 
		0.946594f, -0.224697f, -0.231238f, -0.959369f, 0.195262f, -0.203675f, 
		-2.65951e-005f, -0.999998f, -0.00175066f, -0.462790f, 0.366534f, -0.807142f, 
		0.204707f, 0.959281f, 0.194616f, 0.925426f, -0.357666f, 0.125150f, 
		0.841791f, -0.506537f, 0.186572f, 0.965665f, -0.141003f, 0.218194f, 
		0.952863f, 0.0127052f, 0.303134f, 0.949262f, -0.257010f, 0.181236f, 
		0.362022f, 0.842423f, -0.399078f, -0.362360f, -0.848390f, -0.385914f, 
		-0.856288f, -0.00585708f, -0.516465f, -0.978596f, 0.0441622f, -0.200999f, 
		-0.995579f, 0.0548961f, -0.0762125f, 0.916330f, -0.146414f, 0.372697f, 
		0.988324f, -0.0659448f, 0.137356f, 0.960053f, 0.0503912f, 0.275245f, 
		0.989457f, 0.141768f, 0.0295983f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		0.0261148f, -0.999336f, -0.0254207f, -0.727025f, 0.00319893f, 0.686603f, 
		-0.851699f, 0.0728375f, 0.518945f, -0.997850f, -0.00872417f, 0.0649496f, 
		-0.910822f, -0.112927f, 0.397053f, -0.989745f, -0.130757f, 0.0575061f, 
		0.990654f, -0.132594f, -0.0319900f, 0.973215f, -0.136532f, -0.184967f, 
		0.950162f, 0.0924161f, -0.297743f, 0.999813f, -0.00333173f, -0.0190572f, 
		8.06521e-005f, 1.00000f, -0.000277913f, 0.561734f, -0.374652f, -0.737625f, 
		1.81686e-005f, -0.999994f, -0.00351427f, -0.577850f, 0.325313f, -0.748506f, 
		0.973386f, -0.00317959f, -0.229151f, -0.973386f, -0.00320034f, 0.229148f, 
		-0.975239f, -0.221146f, -0.00197505f, 0.975233f, 0.221160f, -0.00287141f, 
		-0.00111316f, -0.999999f, 0.000693695f, -0.561763f, 0.374600f, 0.737629f, 
		0.964363f, 0.0457665f, 0.260593f, -0.977131f, 0.000128699f, -0.212640f, 
		0.993350f, 0.00176571f, 0.115118f, -0.993351f, 0.00174420f, -0.115109f, 
		0.361151f, -0.846428f, -0.391317f, -0.371390f, 0.840517f, -0.394461f, 
		-0.993336f, 0.115241f, 0.00195678f, 0.993334f, -0.115265f, 0.00155861f, 
		0.992039f, -0.0318292f, -0.121844f, -0.990531f, -0.00139350f, 0.137284f, 
		-0.625333f, 0.780324f, -0.00731138f, 0.625343f, -0.780313f, -0.00756505f, 
		-0.184095f, -0.770667f, 0.610067f, 0.407166f, -0.844261f, 0.348482f, 
		-0.625311f, 0.780339f, 0.00756710f, 0.625322f, -0.780332f, 0.00732450f, 
		-0.101740f, 0.960846f, 0.257729f, -0.184352f, -0.770667f, -0.609989f, 
		-0.783700f, -0.621126f, -0.00400993f, 0.783698f, 0.621127f, -0.00435160f, 
		-0.213237f, 0.670116f, 0.710967f, 0.213421f, -0.666209f, 0.714575f, 
		-0.236906f, -0.958168f, 0.160592f, -0.921200f, -0.373883f, 0.107711f, 
		-0.992933f, 0.108364f, 0.0483779f, -0.994176f, -0.0792003f, 0.0730883f, 
		-0.375368f, 0.845851f, -0.378992f, 0.191465f, 0.770213f, -0.608369f, 
		0.732859f, 0.00281688f, -0.680375f, 1.09577e-005f, 1.00000f, 1.37334e-005f, 
		8.16089e-006f, -1.00000f, 2.51535e-006f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		0.0454395f, -0.0312594f, -0.998478f, -0.0543931f, 0.0254683f, 0.998195f, 
		0.0516286f, 0.0239632f, -0.998379f, -0.0469896f, -0.0318650f, 0.998387f, 
		4.18064e-005f, -1.00000f, -6.31117e-006f, -0.256115f, 0.946005f, -0.198694f, 
		0.0602294f, -0.0284116f, 0.997780f, -0.0459077f, 0.0312734f, -0.998456f, 
		0.0549075f, 0.0378065f, 0.997775f, -0.0500876f, -0.0238831f, -0.998459f, 
		0.191712f, 0.770215f, 0.608289f, -0.225604f, -0.874171f, -0.430033f, 
		-0.204227f, -0.885207f, 0.417971f, -0.272081f, 0.359631f, 0.892546f, 
		0.272474f, -0.359615f, 0.892432f, -0.581808f, 0.564835f, 0.585202f, 
		0.00118790f, -0.999999f, 0.00101890f, -0.0527578f, 0.996559f, 0.0639329f, 
		0.194825f, 0.885249f, 0.422348f, -0.665652f, -0.637439f, 0.388046f, 
		0.668710f, 0.637454f, -0.382726f, -0.668885f, -0.637421f, -0.382474f, 
		0.665791f, 0.637467f, 0.387760f, 0.000199050f, -1.00000f, -3.65723e-005f, 
		-0.269842f, 0.945389f, 0.182826f, 0.581755f, -0.564908f, -0.585184f, 
		-0.576757f, 0.569771f, -0.585416f, 0.195740f, 0.873901f, -0.444953f, 
		0.664107f, 0.639481f, -0.387333f, -0.667343f, -0.639465f, 0.381755f, 
		0.667460f, 0.639539f, 0.381428f, -0.664243f, -0.639521f, -0.387032f, 
		0.852741f, -0.0162294f, -0.522081f, 0.00401423f, -0.999991f, -0.00136576f, 
		-0.678187f, 0.626005f, 0.384942f, 0.672310f, -0.625992f, -0.395138f, 
		-0.672477f, 0.625988f, -0.394858f, 0.678365f, -0.625983f, 0.384663f, 
		0.751076f, 0.660214f, 0.00163001f, -0.751061f, -0.660230f, 0.00195049f, 
		-0.672302f, 0.626002f, 0.395136f, 0.678193f, -0.625995f, -0.384948f, 
		-0.678344f, 0.626017f, -0.384645f, 0.672464f, -0.626005f, 0.394853f, 
		-0.527905f, 0.333246f, 0.781193f, 0.528242f, -0.333253f, 0.780963f, 
		-0.534005f, -0.333254f, 0.777033f, 0.534340f, 0.333238f, 0.776810f, 
		-0.00389928f, 0.999879f, 0.0150815f, -0.341369f, 0.153133f, 0.927371f, 
		0.00390435f, -0.999879f, 0.0150801f, 0.341773f, -0.153110f, 0.927226f, 
		-0.733152f, -0.680064f, 0.000191436f, 0.733160f, 0.680056f, -0.000119696f, 
		0.227907f, 0.962207f, -0.149047f, 0.618130f, -0.473222f, 0.627675f, 
		0.349764f, -0.931055f, -0.103931f, 0.372735f, 0.917726f, -0.137287f, 
		-0.00819227f, -0.999907f, 0.0109143f, -0.514435f, -0.115727f, 0.849684f, 
		0.00819254f, 0.999907f, 0.0109021f, 0.514822f, 0.115763f, 0.849445f, 
		0.184039f, -0.118038f, -0.975806f, -0.194144f, 0.100585f, 0.975803f, 
		0.194143f, 0.100582f, -0.975803f, -0.184036f, -0.118034f, 0.975807f, 
		0.194604f, -0.100606f, 0.975709f, -0.184403f, 0.117996f, -0.975742f, 
		0.184405f, 0.117999f, 0.975741f, -0.194603f, -0.100603f, -0.975709f, 
		0.374741f, -0.895223f, 0.241134f, 0.257883f, 0.959010f, -0.117454f, 
		-0.00716200f, 0.999907f, 0.0116164f, -0.602670f, 0.115697f, 0.789559f, 
		0.00716158f, -0.999907f, 0.0116071f, 0.603003f, -0.115721f, 0.789301f, 
		0.805414f, -0.0363192f, -0.591599f, 0.00553727f, -0.999982f, -0.00212025f, 
		-0.00353695f, 0.999992f, -0.00187993f, -0.854677f, 0.0384939f, -0.517731f, 
		-0.400021f, 0.909047f, -0.116693f, 0.0922903f, -0.989526f, -0.111001f, 
		-0.0870331f, 0.988961f, -0.119921f, -0.826080f, 0.00282291f, 0.563546f, 
		0.000976711f, 0.999999f, -0.000601596f, 0.813419f, -0.0393413f, 0.580346f, 
		-0.00114543f, -0.999999f, -0.000616609f, -0.900201f, 0.0159681f, -0.435181f, 
		-0.00782072f, 0.999953f, -0.00577263f, -0.827168f, -0.0159611f, -0.561728f, 
		-2.53304e-006f, -1.00000f, -9.77034e-007f, 0.399648f, 0.886227f, 0.234272f, 
		-0.723642f, -0.546875f, 0.421035f, 0.726279f, 0.546854f, -0.416496f, 
		-0.726440f, -0.546880f, -0.416181f, 0.723835f, 0.546852f, 0.420733f, 
		-6.02795e-005f, 1.00000f, 0.000976418f, 0.0258581f, -0.999337f, 0.0256271f, 
		0.723661f, 0.546847f, -0.421038f, -0.726271f, -0.546878f, 0.416479f, 
		0.726449f, 0.546883f, 0.416163f, -0.723818f, -0.546888f, -0.420715f, 
		-0.138299f, 0.948622f, -0.284588f, -0.00218070f, -0.999995f, 0.00233797f, 
		-0.593859f, 0.374588f, 0.712050f, 0.00186759f, 0.999996f, 0.00204981f, 
		0.587108f, -0.330089f, 0.739152f, 0.000208314f, 1.00000f, -0.000224269f, 
		0.593809f, -0.374627f, -0.712072f, -0.00533297f, 0.999880f, 0.0145366f, 
		-0.246225f, 0.000000f, 0.969213f, -0.342123f, 0.000000f, 0.939655f, 
		3.58729e-006f, 1.00000f, -9.35137e-007f, -4.98821e-006f, -1.00000f, 1.95894e-006f, 
		-0.272260f, -0.959096f, -0.0775202f, 0.906616f, -2.51528e-006f, -0.421956f, 
		6.17174e-006f, -1.00000f, 7.15075e-006f, -2.28478e-006f, 1.00000f, 3.05527e-006f, 
		0.00534000f, -0.999880f, 0.0145393f, 0.246651f, 0.000000f, 0.969104f, 
		0.342534f, 0.000000f, 0.939505f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, -0.921049f, -0.0162146f, 0.389109f, 
		3.58729e-006f, 1.00000f, -9.35137e-007f, -2.53304e-006f, -1.00000f, -9.77034e-007f, 
		0.921223f, 0.0162153f, 0.388696f, 8.16089e-006f, -1.00000f, 2.51535e-006f, 
		-2.68247e-006f, 1.00000f, -1.00611e-005f, 0.0647445f, -0.00549359f, -0.997887f, 
		-0.0659657f, -0.00538450f, 0.997807f, 0.0569980f, 0.00398120f, 0.998366f, 
		-0.0651468f, 0.00549280f, -0.997861f, 0.258630f, 0.962207f, 0.0852494f, 
		-0.268554f, -0.959370f, 0.0865321f, 0.000193140f, 1.00000f, 0.000200608f, 
		0.678420f, -0.355855f, 0.642739f, -0.268618f, -0.959107f, -0.0892019f, 
		0.641444f, -0.550125f, -0.534707f, -0.611982f, 0.590851f, 0.525712f, 
		0.632964f, -0.598290f, 0.491331f, -0.268101f, -0.962301f, 0.0458061f, 
		0.381340f, -0.231905f, -0.894874f, -0.391428f, 0.214497f, 0.894861f, 
		0.391426f, 0.214491f, -0.894863f, -0.381340f, -0.231901f, 0.894875f, 
		0.391800f, -0.214464f, 0.894706f, -0.381806f, 0.231933f, -0.894668f, 
		0.381791f, 0.231914f, 0.894680f, -0.391802f, -0.214470f, -0.894703f, 
		-0.435656f, -0.884278f, -0.168094f, -0.770478f, -0.637459f, 0.00323212f, 
		0.770460f, 0.637481f, 0.00292777f, 0.568891f, 0.545258f, -0.615675f, 
		-0.900858f, -0.00281759f, 0.434105f, 1.09577e-005f, 1.00000f, 1.37334e-005f, 
		1.09577e-005f, 1.00000f, 1.37334e-005f, 3.37440e-006f, -1.00000f, -1.47995e-005f, 
		-0.569166f, -0.545236f, -0.615442f, 0.878494f, 0.0162183f, 0.477478f, 
		3.37440e-006f, -1.00000f, -1.47995e-005f, -3.37460e-006f, 1.00000f, 1.48004e-005f, 
		-0.772766f, -0.448707f, 0.448883f, 0.774921f, 0.448734f, -0.445124f, 
		-0.775120f, -0.448708f, -0.444803f, 0.772952f, 0.448721f, 0.448548f, 
		0.768797f, 0.639484f, -0.00340229f, -0.768792f, -0.639492f, -0.00304637f, 
		-0.00821651f, 0.999908f, 0.0107858f, -0.520975f, 0.000000f, 0.853572f, 
		-0.603914f, 0.000000f, 0.797050f, -0.00703362f, -0.999908f, 0.0115921f, 
		0.521325f, 0.000000f, 0.853358f, 0.00822204f, -0.999908f, 0.0107835f, 
		0.00703321f, 0.999908f, 0.0115842f, 0.604227f, 0.000000f, 0.796812f, 
		4.44185e-005f, 0.999999f, 0.00118823f, 0.271303f, -0.962138f, 0.0261841f, 
		0.571178f, -0.545266f, -0.613548f, -0.571441f, 0.545238f, -0.613328f, 
		0.772749f, 0.448757f, -0.448860f, -0.774933f, -0.448704f, 0.445132f, 
		-0.772986f, -0.448663f, -0.448547f, 0.775139f, 0.448667f, 0.444811f, 
		-0.779740f, 0.439825f, 0.445600f, 0.775591f, -0.439811f, -0.452798f, 
		-0.775775f, 0.439824f, -0.452468f, 0.779924f, -0.439832f, 0.445271f, 
		-0.779825f, 0.625971f, -0.00572003f, 0.779806f, -0.625992f, -0.00604591f, 
		-0.000256300f, -1.00000f, 9.81828e-005f, -0.000256300f, -1.00000f, 9.81828e-005f, 
		-0.312339f, 0.948972f, -0.0435543f, -0.775586f, 0.439809f, 0.452807f, 
		0.779749f, -0.439804f, -0.445605f, -0.779930f, 0.439824f, -0.445267f, 
		0.775762f, -0.439838f, 0.452478f, -1.08155e-005f, 1.00000f, 0.000144778f, 
		0.282430f, -0.959139f, -0.0169107f, -0.224907f, -0.634053f, 0.739860f, 
		0.486700f, -0.836197f, -0.252779f, -0.779799f, 0.626001f, 0.00604426f, 
		0.779785f, -0.626021f, 0.00573062f, -0.225219f, -0.634049f, -0.739769f, 
		0.469545f, -0.846574f, 0.250680f, -0.00407211f, 0.999992f, 0.000576652f, 
		-0.962735f, -0.0307715f, 0.268691f, 0.955656f, -0.00282193f, 0.294473f, 
		0.000603498f, -1.00000f, -3.21668e-005f, 0.278764f, 0.959282f, 0.0454874f, 
		-0.669495f, 0.479787f, -0.567081f, 0.283218f, 0.958065f, -0.0435822f, 
		0.230789f, 0.633455f, -0.738560f, -0.448279f, 0.882105f, 0.144698f, 
		-0.552466f, 0.696864f, 0.457343f, 0.570100f, -0.660979f, 0.487948f, 
		0.231089f, 0.633451f, 0.738470f, 0.566741f, -0.337171f, -0.751744f, 
		-0.575263f, 0.322352f, 0.751772f, 0.575268f, 0.322350f, -0.751769f, 
		-0.566749f, -0.337172f, 0.751739f, 0.575540f, -0.322328f, 0.751571f, 
		-0.566979f, 0.337111f, -0.751593f, 0.567002f, 0.337121f, 0.751570f, 
		-0.575531f, -0.322307f, -0.751586f, -0.459405f, -0.886575f, 0.0541551f, 
		-0.598807f, -0.667466f, 0.442627f, 0.568809f, 0.696860f, 0.436855f, 
		-0.812373f, -0.343557f, 0.471189f, 0.814038f, 0.343539f, -0.468319f, 
		-0.814232f, -0.343583f, -0.467949f, 0.812577f, 0.343564f, 0.470831f, 
		0.218357f, -0.0100917f, -0.975817f, -0.218357f, -0.0100882f, 0.975817f, 
		0.812375f, 0.343580f, -0.471169f, -0.814024f, -0.343640f, 0.468271f, 
		0.218789f, 0.0100912f, 0.975720f, -0.218789f, 0.0100877f, -0.975720f, 
		0.814233f, 0.343584f, 0.467947f, -0.812576f, -0.343574f, -0.470826f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -0.0326919f, 0.999208f, -0.0227033f, 
		-0.988759f, -0.0497896f, -0.140987f, 0.000000f, 1.00000f, -5.78099e-015f, 
		6.17174e-006f, -1.00000f, 7.15075e-006f, 0.224236f, -0.869264f, -0.440566f, 
		0.988921f, -0.0159675f, -0.147579f, -0.000605196f, -1.00000f, 5.85068e-005f, 
		0.999871f, 0.0159656f, -0.00191020f, 6.17174e-006f, -1.00000f, 7.15075e-006f, 
		-2.28478e-006f, 1.00000f, 3.05527e-006f, -0.00211111f, 0.999998f, -0.000151484f, 
		-0.995947f, 0.0152735f, -0.0886346f, 0.685325f, -0.209196f, -0.697544f, 
		0.644059f, 0.209202f, -0.735814f, -0.685613f, 0.209176f, -0.697266f, 
		-0.644395f, -0.209172f, -0.735529f, -0.837207f, -0.546879f, 0.00279298f, 
		0.837224f, 0.546854f, 0.00245398f, 0.982590f, 0.0191334f, -0.184797f, 
		8.16089e-006f, -1.00000f, 2.51535e-006f, -2.68247e-006f, 1.00000f, -1.00611e-005f, 
		-0.988850f, -0.00281775f, -0.148889f, 3.58729e-006f, 1.00000f, -9.35137e-007f, 
		-2.68247e-006f, 1.00000f, -1.00611e-005f, -2.53304e-006f, -1.00000f, -9.77034e-007f, 
		-0.259618f, -0.874169f, 0.410400f, 0.837204f, 0.546883f, -0.00282025f, 
		-0.837226f, -0.546852f, -0.00242629f, -0.841598f, -0.232585f, 0.487459f, 
		0.842733f, 0.232572f, -0.485501f, -0.842936f, -0.232629f, -0.485122f, 
		0.841795f, 0.232645f, 0.487091f, 0.999177f, -0.0191512f, 0.0357615f, 
		-0.00140631f, -0.999999f, 2.91757e-005f, -0.998643f, 0.0162221f, 0.0494811f, 
		-0.00521681f, 0.999985f, 0.00147737f, -0.844325f, 0.227370f, 0.485199f, 
		0.842138f, -0.227402f, -0.488971f, -0.842337f, 0.227440f, -0.488609f, 
		0.844530f, -0.227421f, 0.484819f, -0.259789f, -0.874170f, -0.410289f, 
		0.841582f, 0.232704f, -0.487430f, -0.842721f, -0.232685f, 0.485467f, 
		0.842929f, 0.232656f, 0.485121f, -0.841796f, -0.232647f, -0.487088f, 
		-0.842139f, 0.227399f, 0.488971f, 0.844324f, -0.227422f, -0.485177f, 
		-0.844527f, 0.227404f, -0.484831f, 0.842348f, -0.227397f, 0.488611f, 
		-0.699653f, 0.608828f, 0.373918f, 0.732123f, -0.550069f, 0.401771f, 
		0.699601f, -0.608931f, -0.373847f, -0.728594f, 0.564873f, -0.387389f, 
		0.00972729f, -0.999907f, -0.00957160f, 0.628962f, -0.0959055f, -0.771498f, 
		-0.00973106f, 0.999907f, -0.00956706f, -0.629287f, 0.0958497f, -0.771240f, 
		0.713301f, -0.418747f, -0.562008f, -0.719216f, 0.408489f, 0.562019f, 
		0.719222f, 0.408488f, -0.562012f, -0.713305f, -0.418744f, 0.562004f, 
		0.719394f, -0.408423f, 0.561839f, -0.713480f, 0.418707f, -0.561811f, 
		0.713480f, 0.418707f, 0.561811f, -0.719409f, -0.408433f, -0.561812f, 
		0.000296177f, 1.00000f, -0.000233432f, 0.753981f, -0.352025f, -0.554609f, 
		-0.000546281f, -1.00000f, 0.000338986f, -0.769523f, 0.358669f, 0.528385f, 
		-0.265292f, 0.819498f, -0.507979f, -0.859641f, -0.117386f, 0.497230f, 
		0.860222f, 0.117374f, -0.496228f, -0.860433f, -0.117432f, -0.495848f, 
		0.859845f, 0.117449f, 0.496863f, 0.00881606f, 0.999907f, -0.0104181f, 
		0.722057f, 0.0958975f, -0.685155f, -0.00881742f, -0.999907f, -0.0104152f, 
		-0.722348f, -0.0958776f, -0.684851f, 0.859642f, 0.117380f, -0.497230f, 
		-0.860219f, -0.117400f, 0.496226f, 0.860427f, 0.117371f, 0.495872f, 
		-0.859847f, -0.117412f, -0.496868f, -0.865683f, 0.0280535f, 0.499806f, 
		0.865479f, -0.0280139f, -0.500161f, -0.865694f, 0.0280172f, -0.499789f, 
		0.865898f, -0.0280505f, 0.499434f, 0.264505f, 0.884957f, -0.383261f, 
		-0.865472f, 0.0281223f, 0.500167f, 0.865674f, -0.0280759f, -0.499820f, 
		-0.865889f, 0.0280509f, -0.499449f, 0.865688f, -0.0280306f, 0.499799f, 
		0.268510f, 0.885248f, 0.379788f, 0.159829f, -0.962887f, -0.217495f, 
		-0.555406f, 0.830644f, 0.0394315f, 0.446200f, -0.0100614f, -0.894877f, 
		-0.446200f, -0.0100545f, 0.894877f, 0.446618f, 0.0100661f, 0.894668f, 
		-0.446619f, 0.0100591f, -0.894668f, 0.807117f, -0.469813f, -0.357543f, 
		-0.810393f, 0.464189f, 0.357479f, 0.810395f, 0.464183f, -0.357484f, 
		-0.807118f, -0.469808f, 0.357548f, 0.810599f, -0.464200f, 0.356998f, 
		-0.807323f, 0.469814f, -0.357077f, 0.807323f, 0.469814f, 0.357077f, 
		-0.810599f, -0.464200f, -0.356998f, -0.893687f, -0.448686f, 0.00233113f, 
		0.893672f, 0.448717f, 0.00198035f, 0.610953f, 0.703155f, -0.363744f, 
		-0.618534f, -0.692518f, -0.371261f, 0.893687f, 0.448684f, -0.00236288f, 
		-0.893688f, -0.448685f, -0.00196022f, -0.898057f, 0.439861f, -0.00398481f, 
		0.898073f, -0.439825f, -0.00434374f, 0.645759f, -0.678016f, -0.351125f, 
		-0.646931f, 0.676455f, -0.351978f, -0.257686f, -0.470221f, 0.844092f, 
		0.853289f, -0.494125f, -0.166547f, -0.854541f, 0.491940f, 0.166597f, 
		0.854543f, 0.491935f, -0.166602f, -0.853290f, -0.494121f, 0.166553f, 
		0.854625f, -0.491933f, 0.166185f, -0.853384f, 0.494112f, -0.166100f, 
		0.853382f, 0.494117f, 0.166094f, -0.854625f, -0.491933f, -0.166185f, 
		-0.898072f, 0.439828f, 0.00435354f, 0.898086f, -0.439802f, 0.00395728f, 
		0.567235f, -0.822016f, 0.0503430f, -0.258042f, -0.470213f, -0.843987f, 
		0.522603f, 0.851567f, 0.0414562f, -0.713020f, 0.632131f, -0.303335f, 
		0.866025f, -0.500000f, -0.000145549f, -0.866024f, 0.500002f, 0.000104909f, 
		0.866024f, 0.500002f, -0.000104909f, -0.866025f, -0.500000f, 0.000145549f, 
		-0.742359f, 0.599586f, 0.298998f, 0.173533f, -0.949603f, 0.261036f, 
		-0.279997f, -0.770668f, -0.572427f, -0.279755f, -0.770666f, 0.572548f, 
		0.262326f, 0.469578f, -0.843019f, 0.326989f, -0.670129f, -0.666338f, 
		-0.327415f, 0.661789f, -0.674414f, 0.262672f, 0.469569f, 0.842916f, 
		-0.00301092f, -0.999994f, 0.00175031f, -0.777342f, 0.374562f, 0.505413f, 
		0.00177384f, 0.999998f, 6.01932e-006f, 0.791769f, -0.351925f, 0.499250f, 
		0.659371f, -0.00856144f, -0.751769f, -0.659372f, -0.00855228f, 0.751768f, 
		0.659588f, 0.00854544f, 0.751579f, -0.659589f, 0.00853629f, -0.751578f, 
		0.715941f, 0.000000f, -0.698161f, 0.00868307f, -0.999908f, -0.0104152f, 
		0.00972775f, 0.999908f, -0.00943388f, 0.642351f, 0.000000f, -0.766411f, 
		-0.00868680f, 0.999908f, -0.0104101f, -0.716247f, 0.000000f, -0.697847f, 
		-0.642688f, 0.000000f, -0.766128f, -0.00973586f, -0.999908f, -0.00943654f, 
		0.000281346f, 1.00000f, -0.000164140f, 0.777328f, -0.374651f, -0.505368f, 
		-0.00291689f, -0.999994f, -0.00186251f, -0.772977f, 0.358716f, -0.523287f, 
		-0.939114f, -0.343601f, 0.00186679f, 0.939123f, 0.343579f, 0.00147772f, 
		0.939114f, 0.343602f, -0.00186515f, -0.939111f, -0.343612f, -0.00146356f, 
		-0.734715f, 0.333215f, 0.590900f, 0.734963f, -0.333229f, 0.590583f, 
		-0.739308f, -0.333237f, 0.585129f, 0.739570f, 0.333231f, 0.584803f, 
		0.273447f, 0.770214f, 0.576192f, -0.777117f, 0.613385f, 0.140883f, 
		-0.849106f, 0.511082f, 0.133468f, -0.571361f, 0.801989f, 0.174240f, 
		-0.662257f, 0.732968f, 0.155478f, -0.719949f, 0.679007f, 0.143610f, 
		0.745453f, -0.648548f, 0.153898f, 0.273200f, 0.770216f, -0.576306f, 
		0.810086f, -0.473277f, -0.346076f, 0.673263f, -0.720169f, -0.167553f, 
		0.728270f, -0.667848f, -0.153629f, 0.785046f, -0.600651f, -0.151397f, 
		-0.748915f, 0.646848f, -0.143924f, -0.810435f, 0.563290f, -0.160931f, 
		-0.913082f, 0.380451f, -0.146760f, -0.00452119f, -0.999989f, 0.000836287f, 
		-0.851035f, 0.338665f, -0.401304f, 0.808892f, -0.465763f, 0.358828f, 
		-0.688506f, 0.660936f, 0.298534f, 0.686696f, -0.667906f, 0.286968f, 
		-0.673039f, -0.696220f, 0.249590f, 0.676044f, 0.692479f, 0.251867f, 
		0.799293f, -0.593501f, -0.0942703f, -0.818730f, 0.564543f, -0.104745f, 
		0.407285f, -0.359620f, -0.839519f, -0.972571f, -0.232602f, 0.00133847f, 
		0.972563f, 0.232638f, 0.000929248f, -0.407643f, 0.359677f, -0.839321f, 
		-0.973801f, 0.227393f, -0.00196646f, 0.973794f, -0.227419f, -0.00239876f, 
		0.799990f, -0.592440f, 0.0950257f, -0.282696f, -0.262687f, 0.922539f, 
		0.972545f, 0.232712f, -0.00133968f, -0.972547f, -0.232705f, -0.000939739f, 
		-0.973798f, 0.227402f, 0.00238630f, 0.973797f, -0.227409f, 0.00197839f, 
		-0.0110457f, -0.999907f, 0.00801578f, -0.742023f, -0.115712f, 0.660313f, 
		0.0110362f, 0.999907f, 0.00799988f, 0.742299f, 0.115691f, 0.660005f, 
		-0.283095f, -0.262689f, -0.922416f, 0.00739768f, -0.999879f, -0.0137193f, 
		0.380147f, -0.153154f, -0.912158f, 0.827054f, -0.00593270f, -0.562091f, 
		-0.827054f, -0.00592258f, 0.562092f, 0.827230f, 0.00592784f, 0.561833f, 
		-0.827230f, 0.00593793f, -0.561831f, -0.00740365f, 0.999878f, -0.0137221f, 
		-0.380525f, 0.153141f, -0.912002f, -0.0102686f, 0.999907f, 0.00898952f, 
		-0.808617f, 0.115695f, 0.576847f, 0.0102586f, -0.999907f, 0.00897462f, 
		0.808877f, -0.115679f, 0.576486f, -0.993088f, -0.117367f, 0.000786552f, 
		0.993092f, 0.117335f, 0.000346511f, -0.993083f, -0.117415f, -0.000372996f, 
		0.993085f, 0.117394f, -0.000782735f, -0.999607f, 0.0280473f, 2.21478e-005f, 
		0.999607f, -0.0280437f, -0.000425396f, 0.00264886f, -0.999942f, -0.0104265f, 
		-0.338594f, -0.143410f, 0.929940f, -0.999605f, 0.0280893f, 0.000417772f, 
		0.999606f, -0.0280837f, -6.03754e-007f, 0.285751f, 0.262190f, -0.921739f, 
		0.00265309f, -0.999942f, 0.0104252f, -0.338992f, -0.143431f, -0.929791f, 
		0.286135f, 0.262188f, 0.921621f, 0.714495f, 0.675732f, -0.181338f, 
		-0.696285f, -0.696842f, -0.172044f, 0.694714f, -0.703812f, -0.148397f, 
		-0.728776f, 0.670582f, -0.138582f, 0.000335938f, 1.00000f, -0.000131777f, 
		0.902513f, -0.330200f, -0.276475f, 0.00125617f, 0.999994f, 0.00311320f, 
		0.898173f, -0.329802f, 0.290714f, -0.318493f, -0.885209f, -0.339068f, 
		0.933901f, -0.00325025f, -0.357516f, -0.933901f, -0.00323892f, 0.357518f, 
		0.934093f, 0.00324971f, 0.357015f, -0.934094f, 0.00326105f, -0.357013f, 
		-0.323965f, -0.884885f, 0.334701f, -0.880313f, 0.473119f, 0.0347446f, 
		0.243697f, 0.143079f, -0.959239f, 0.00368079f, -0.999942f, -0.0101090f, 
		-0.711474f, 0.699695f, 0.0650506f, -0.749843f, -0.660932f, 0.0300780f, 
		0.244102f, 0.143095f, 0.959134f, 0.00368477f, -0.999942f, 0.0101066f, 
		0.986034f, -0.00125959f, -0.166539f, -0.986034f, -0.00124696f, 0.166538f, 
		0.986109f, 0.00124923f, 0.166095f, -0.986109f, 0.00126186f, -0.166097f, 
		0.318361f, 0.873898f, 0.367353f, 1.00000f, -6.10014e-006f, -0.000129743f, 
		-1.00000f, 6.76017e-006f, 0.000129830f, 0.309729f, 0.884634f, -0.348556f, 
		0.807538f, 0.335223f, -0.485292f, -0.807726f, -0.335233f, -0.484972f, 
		0.829190f, -0.304572f, -0.468701f, -0.829373f, 0.304632f, -0.468337f, 
		-0.00281521f, -0.999996f, -0.000760600f, -0.891773f, 0.374604f, 0.253797f, 
		0.000411479f, 1.00000f, 9.52134e-005f, 0.891888f, -0.374588f, 0.253415f, 
		0.000895945f, 0.999999f, -0.000974493f, 0.891776f, -0.374567f, -0.253841f, 
		-0.00285367f, -0.999996f, 0.000818214f, -0.896009f, 0.374212f, -0.239024f, 
		-0.269271f, 0.869248f, 0.414609f, -0.000987900f, -0.999999f, 0.000775440f, 
		-0.945214f, 0.309010f, 0.105278f, -0.0110315f, 0.999908f, 0.00788554f, 
		-0.749426f, 0.000000f, 0.662088f, -0.811997f, 0.000000f, 0.583662f, 
		-0.0101385f, -0.999908f, 0.00900451f, 0.749705f, 0.000000f, 0.661773f, 
		0.0110190f, -0.999908f, 0.00787172f, 0.0101300f, 0.999908f, 0.00898638f, 
		0.812264f, 0.000000f, 0.583290f, 0.000389572f, 1.00000f, -8.47609e-005f, 
		0.926928f, -0.374666f, -0.0207211f, -0.00126060f, -0.999999f, -0.000182929f, 
		-0.926658f, 0.374242f, -0.0353124f, 0.00238338f, 0.999997f, 0.000184994f, 
		0.926965f, -0.374597f, 0.0203286f, 0.0121086f, -0.999907f, -0.00627267f, 
		0.837233f, -0.105407f, -0.536591f, -0.0121203f, 0.999907f, -0.00627517f, 
		-0.837443f, 0.105455f, -0.536253f, 0.00597503f, -0.999880f, -0.0142965f, 
		0.478356f, 0.000000f, -0.878166f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 0.387923f, 0.000000f, -0.921692f, 
		-0.00597809f, 0.999880f, -0.0142900f, -0.478728f, 0.000000f, -0.877963f, 
		-0.388315f, 0.000000f, -0.921527f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, 0.0114800f, 0.999907f, -0.00734792f, 
		0.885575f, 0.115682f, -0.449861f, -0.0114969f, -0.999907f, -0.00735407f, 
		-0.885742f, -0.115779f, -0.449507f, 0.307478f, -0.819502f, 0.483605f, 
		0.126896f, -0.991356f, -0.0333378f, -0.129791f, 0.990192f, -0.0517120f, 
		-0.876248f, 0.333212f, 0.348078f, 0.883420f, -0.304610f, 0.356064f, 
		-0.878942f, -0.333222f, 0.341209f, 0.882910f, 0.335216f, 0.328785f, 
		0.115081f, -0.991543f, 0.0599909f, 0.835795f, -0.545296f, 0.0640293f, 
		0.836033f, 0.545288f, 0.0609061f, -0.338364f, -0.634053f, -0.695332f, 
		-0.338064f, -0.634060f, 0.695471f, -0.0129167f, -0.999907f, 0.00440276f, 
		-0.903695f, -0.115714f, 0.412244f, 0.0129203f, 0.999907f, 0.00439833f, 
		0.903868f, 0.115722f, 0.411862f, -0.358293f, -0.885497f, 0.295840f, 
		-0.358418f, -0.885496f, -0.295693f, -0.0124616f, 0.999907f, 0.00556349f, 
		-0.942733f, 0.115689f, 0.312841f, 0.0124648f, -0.999907f, 0.00555826f, 
		0.941519f, -0.105447f, 0.320035f, 0.926533f, 0.304605f, -0.220801f, 
		-0.918422f, -0.333241f, -0.213194f, 0.921968f, -0.335232f, -0.193890f, 
		-0.920063f, 0.333232f, -0.206011f, 0.333181f, 0.633447f, 0.698381f, 
		-0.124164f, 0.991941f, 0.0252368f, 0.332891f, 0.633446f, -0.698520f, 
		0.889941f, 0.000000f, -0.456076f, 0.0113663f, -0.999908f, -0.00739187f, 
		0.0120599f, 0.999908f, -0.00614082f, 0.839731f, 0.000000f, -0.543003f, 
		-0.0113703f, 0.999908f, -0.00738821f, -0.890114f, 0.000000f, -0.455738f, 
		-0.839958f, 0.000000f, -0.542651f, -0.0120869f, -0.999908f, -0.00614846f, 
		-0.232712f, 0.959767f, -0.157137f, -0.355732f, -0.770668f, 0.528702f, 
		-0.355960f, -0.770667f, -0.528549f, 0.370611f, 0.884635f, -0.282965f, 
		-0.413291f, 0.670150f, 0.616514f, 0.420623f, -0.661786f, 0.620577f, 
		0.390239f, 0.873897f, 0.289859f, -0.939921f, 0.333205f, 0.0743185f, 
		-0.940461f, -0.333237f, 0.0669825f, 0.0134139f, -0.999907f, -0.00242522f, 
		0.955050f, -0.115668f, -0.272948f, -0.0134304f, 0.999907f, -0.00242284f, 
		-0.955157f, 0.115770f, -0.272531f, 0.0131421f, 0.999907f, -0.00364279f, 
		0.978634f, 0.105412f, -0.176532f, -0.0131540f, -0.999907f, -0.00363852f, 
		-0.978892f, -0.115765f, -0.168434f, 0.972648f, -0.209220f, 0.100906f, 
		0.976845f, 0.209206f, 0.0447934f, 0.362272f, 0.770209f, -0.524917f, 
		0.362503f, 0.770218f, 0.524743f, -0.0136408f, -0.999907f, 0.000400451f, 
		-0.985058f, -0.115718f, 0.127553f, 0.0136325f, 0.999907f, 0.000392224f, 
		0.985873f, 0.0958777f, 0.137339f, -0.0135474f, 0.999907f, 0.00164289f, 
		-0.993058f, 0.115721f, 0.0210911f, 0.0135336f, -0.999907f, 0.00163375f, 
		0.995339f, -0.0958410f, 0.0106936f, -0.523057f, 0.359659f, 0.772695f, 
		0.523400f, -0.359654f, 0.772464f, -0.0128644f, 0.999908f, 0.00428291f, 
		-0.911273f, 0.000000f, 0.411803f, -0.947968f, 0.000000f, 0.318365f, 
		-0.0123417f, -0.999908f, 0.00561596f, 0.911459f, 0.000000f, 0.411391f, 
		0.0128682f, -0.999908f, 0.00427806f, 0.0123448f, 0.999908f, 0.00561090f, 
		0.948099f, 0.000000f, 0.317977f, 0.371534f, -0.850343f, -0.372665f, 
		-0.358773f, 0.856758f, -0.370469f, -0.229688f, 0.961224f, 0.152618f, 
		0.251585f, -0.956369f, 0.148534f, -0.408322f, -0.884888f, -0.224158f, 
		-0.428554f, -0.874171f, 0.228401f, -0.00817229f, 0.999879f, 0.0132608f, 
		-0.599562f, 0.153078f, 0.785552f, 0.00818238f, -0.999879f, 0.0132662f, 
		0.599897f, -0.153147f, 0.785283f, -0.385232f, -0.470211f, -0.794039f, 
		-0.384892f, -0.470216f, 0.794201f, 0.398894f, 0.885246f, 0.239213f, 
		0.401857f, 0.884956f, -0.235294f, 0.984829f, 0.000000f, -0.173529f, 
		0.0130249f, -0.999908f, -0.00370690f, 0.0133602f, 0.999908f, -0.00231795f, 
		0.962483f, 0.000000f, -0.271342f, -0.0130432f, 0.999908f, -0.00370835f, 
		-0.984906f, 0.000000f, -0.173091f, -0.962591f, 0.000000f, -0.270957f, 
		-0.0133607f, -0.999908f, -0.00231205f, 0.381096f, 0.469582f, 0.796404f, 
		0.380762f, 0.469574f, -0.796568f, -0.0135558f, 0.999908f, 0.000301139f, 
		-0.992172f, 0.000000f, 0.124876f, -0.999692f, 0.000000f, 0.0248219f, 
		-0.0134483f, -0.999908f, 0.00172833f, 0.992222f, 0.000000f, 0.124480f, 
		0.0135313f, -0.999908f, 0.000294572f, 0.0134503f, 0.999908f, 0.00172286f, 
		0.999703f, 0.000000f, 0.0243712f, -0.445684f, -0.874172f, 0.192843f, 
		-0.445772f, -0.874169f, -0.192652f, -0.00937870f, 0.999880f, 0.0123141f, 
		-0.520965f, 0.000000f, 0.853578f, -0.603914f, 0.000000f, 0.797050f, 
		3.58729e-006f, 1.00000f, -9.35137e-007f, -4.98821e-006f, -1.00000f, 1.95894e-006f, 
		0.00939315f, -0.999880f, 0.0123219f, 0.521325f, 0.000000f, 0.853358f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		0.604248f, 0.000000f, 0.796796f, 0.242701f, -0.964486f, -0.104230f, 
		0.458272f, 0.873897f, -0.162140f, 0.434516f, 0.884958f, 0.167466f, 
		-0.420251f, -0.262688f, -0.868553f, -0.419880f, -0.262672f, 0.868737f, 
		-0.436288f, -0.770663f, -0.464469f, -0.436091f, -0.770666f, 0.464649f, 
		0.508861f, -0.670151f, -0.540332f, -0.512369f, 0.666191f, -0.541911f, 
		-0.476873f, -0.874172f, -0.0917367f, -0.454115f, -0.885207f, 0.100935f, 
		-0.399252f, 0.850333f, 0.342830f, 0.392871f, -0.853074f, 0.343390f, 
		0.00514993f, -0.999942f, 0.00944481f, -0.384309f, -0.143422f, -0.911996f, 
		0.00514553f, -0.999942f, -0.00944622f, -0.383918f, -0.143418f, 0.912161f, 
		0.417473f, 0.262188f, 0.870042f, 0.417097f, 0.262195f, -0.870221f, 
		0.431139f, 0.770212f, 0.469992f, -0.462687f, -0.885498f, 0.0425934f, 
		0.430939f, 0.770213f, -0.470174f, -0.463668f, -0.884885f, -0.0446031f, 
		-0.432988f, -0.634052f, 0.640702f, -0.433258f, -0.634054f, -0.640518f, 
		0.469977f, 0.873898f, 0.124194f, 0.469923f, 0.873899f, -0.124389f, 
		-0.277398f, 0.960752f, -0.00259721f, 0.473808f, 0.143073f, 0.868928f, 
		0.00417762f, -0.999942f, 0.00991363f, 0.473435f, 0.143085f, -0.869129f, 
		0.00417296f, -0.999942f, -0.00991469f, 0.636643f, -0.359604f, -0.682181f, 
		-0.636918f, 0.359645f, -0.681902f, 0.464334f, 0.885247f, -0.0270627f, 
		0.465626f, 0.884633f, 0.0248320f, 0.438220f, 0.633453f, -0.637731f, 
		0.438497f, 0.633452f, 0.637541f, 0.0111070f, -0.999879f, -0.0109204f, 
		0.632119f, -0.153100f, -0.759596f, -0.0111179f, 0.999879f, -0.0109226f, 
		-0.632438f, 0.153163f, -0.759318f, 0.433467f, -0.869253f, -0.237710f, 
		-0.464990f, 0.850334f, -0.246406f, 0.313754f, -0.949125f, 0.0268323f, 
		-0.495762f, -0.770669f, 0.400361f, -0.495931f, -0.770671f, -0.400148f, 
		-0.576647f, 0.670158f, 0.467296f, 0.576868f, -0.670130f, 0.467064f, 
		0.00992090f, -0.999880f, -0.0118937f, 0.715952f, 0.000000f, -0.698149f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		0.642368f, 0.000000f, -0.766396f, -0.00993136f, 0.999880f, -0.0118975f, 
		-0.716247f, 0.000000f, -0.697847f, -0.642688f, 0.000000f, -0.766128f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		0.500900f, 0.770212f, -0.394807f, 0.501066f, 0.770210f, 0.394600f, 
		-0.495039f, -0.470220f, 0.730636f, -0.495349f, -0.470207f, -0.730435f, 
		-0.727601f, 0.359614f, 0.584187f, 0.727841f, -0.359634f, 0.583876f, 
		-0.456035f, 0.869253f, 0.190872f, 0.476029f, -0.856751f, 0.198428f, 
		0.499151f, 0.469586f, -0.728242f, 0.499464f, 0.469578f, 0.728033f, 
		-0.0117100f, 0.999879f, 0.0102583f, -0.804482f, 0.153031f, 0.573924f, 
		0.0117265f, -0.999879f, 0.0102619f, 0.804714f, -0.153086f, 0.573585f, 
		-0.528288f, -0.634055f, -0.564700f, -0.528045f, -0.634058f, 0.564924f, 
		-0.553809f, -0.770667f, -0.315228f, -0.553670f, -0.770669f, 0.315466f, 
		0.513416f, -0.853079f, -0.0930579f, -0.484305f, 0.869261f, -0.0991695f, 
		0.645526f, -0.670137f, -0.366349f, -0.645674f, 0.670148f, -0.366069f, 
		0.524224f, 0.633456f, 0.569142f, 0.523987f, 0.633452f, -0.569364f, 
		-0.542067f, -0.262688f, 0.798222f, -0.542395f, -0.262693f, -0.797997f, 
		-0.523001f, 0.850333f, 0.0583429f, 0.550512f, 0.770212f, 0.322040f, 
		0.550376f, 0.770214f, -0.322270f, 0.00560483f, -0.999942f, -0.00918308f, 
		-0.597661f, -0.143419f, 0.788817f, 0.00560875f, -0.999942f, 0.00918057f, 
		-0.597986f, -0.143442f, -0.788567f, 0.544739f, 0.262189f, -0.796565f, 
		0.545076f, 0.262181f, 0.796338f, 0.809433f, -0.359616f, -0.464214f, 
		-0.809621f, 0.359634f, -0.463872f, -0.0125860f, 0.999880f, 0.00900128f, 
		-0.749426f, 0.000000f, 0.662088f, -0.812010f, -9.70202e-006f, 0.583643f, 
		3.58729e-006f, 1.00000f, -9.35137e-007f, -4.98821e-006f, -1.00000f, 1.95894e-006f, 
		0.0125987f, -0.999880f, 0.00900101f, 0.749705f, 0.000000f, 0.661773f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		0.812250f, 0.000000f, 0.583309f, 0.569805f, -0.819503f, 0.0611312f, 
		0.515606f, 0.143067f, -0.844797f, 0.00649681f, -0.999942f, -0.00857493f, 
		0.0138323f, -0.999879f, -0.00716317f, 0.827930f, -0.153085f, -0.539534f, 
		-0.0138442f, 0.999879f, -0.00716238f, -0.828152f, 0.153131f, -0.539181f, 
		0.515961f, 0.143068f, 0.844580f, 0.00650066f, -0.999942f, 0.00857241f, 
		-0.591753f, -0.770665f, 0.236439f, -0.591848f, -0.770669f, -0.236188f, 
		-0.688779f, 0.670151f, 0.276553f, 0.691743f, -0.666204f, 0.278681f, 
		0.595017f, 0.770215f, -0.229615f, 0.595115f, 0.770213f, 0.229368f, 
		-0.622118f, -0.770668f, -0.137988f, -0.602610f, -0.634046f, 0.484610f, 
		-0.622058f, -0.770668f, 0.138253f, -0.602811f, -0.634052f, -0.484351f, 
		-0.602161f, -0.470218f, -0.645211f, -0.601894f, -0.470224f, 0.645456f, 
		0.724831f, -0.670136f, -0.159806f, -0.724880f, 0.670159f, -0.159485f, 
		-0.867468f, 0.359614f, 0.343767f, 0.867601f, -0.359634f, 0.343413f, 
		-0.635154f, -0.770666f, 0.0515089f, -0.635172f, -0.770669f, -0.0512432f, 
		-0.739688f, 0.670157f, 0.0612542f, 0.747422f, -0.661807f, 0.0580713f, 
		-0.0142157f, 0.999879f, 0.00635284f, -0.937898f, 0.153072f, 0.311312f, 
		0.0142242f, -0.999879f, 0.00634846f, 0.938029f, -0.153067f, 0.310920f, 
		0.620977f, 0.770213f, 0.145466f, 0.606724f, 0.633452f, -0.480234f, 
		0.620914f, 0.770214f, -0.145728f, 0.606934f, 0.633455f, 0.479964f, 
		0.598908f, 0.469584f, 0.648691f, 0.598638f, 0.469580f, -0.648943f, 
		0.0129812f, -0.999880f, -0.00843905f, 0.889927f, 0.000000f, -0.456104f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, -9.12782e-007f, 1.00000f, 4.04419e-006f, 
		0.839731f, 0.000000f, -0.543003f, -0.0129894f, 0.999880f, -0.00843544f, 
		-0.890124f, 0.000000f, -0.455718f, -0.839965f, 0.000000f, -0.542641f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, -1.64983e-006f, -1.00000f, 1.88830e-005f, 
		0.636267f, 0.770211f, -0.0440370f, 0.910296f, -0.359626f, -0.205014f, 
		-0.910379f, 0.359639f, -0.204621f, 0.636282f, 0.770214f, 0.0437614f, 
		0.0153302f, -0.999879f, -0.00276806f, 0.950172f, -0.153117f, -0.271531f, 
		-0.0153475f, 0.999878f, -0.00276533f, -0.950290f, 0.153104f, -0.271124f, 
		-0.930245f, 0.359643f, 0.0728133f, 0.930282f, -0.359626f, 0.0724117f, 
		-0.657587f, -0.262684f, -0.706099f, -0.657295f, -0.262676f, 0.706374f, 
		-0.0154629f, 0.999879f, 0.00188286f, -0.987986f, 0.153105f, 0.0210310f, 
		0.0154647f, -0.999879f, 0.00187349f, 0.987993f, -0.153116f, 0.0206049f, 
		-0.671262f, -0.634053f, -0.383907f, -0.671103f, -0.634049f, 0.384192f, 
		0.00770482f, -0.999942f, 0.00750673f, -0.636050f, -0.143424f, -0.758202f, 
		0.00770129f, -0.999942f, -0.00750972f, -0.635728f, -0.143406f, 0.758475f, 
		0.655370f, 0.262188f, 0.708342f, 0.655072f, 0.262195f, -0.708615f, 
		-0.947969f, -2.06249e-005f, 0.318361f, -4.98821e-006f, -1.00000f, 1.95894e-006f, 
		3.58729e-006f, 1.00000f, -9.35137e-007f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-0.0146893f, 0.999880f, 0.00489344f, -0.911284f, -2.06250e-005f, 0.411779f, 
		0.0146921f, -0.999880f, 0.00488757f, 0.911452f, 0.000000f, 0.411407f, 
		1.09389e-009f, -1.00000f, 5.35907e-006f, 1.64979e-006f, 1.00000f, -1.88827e-005f, 
		0.948102f, 0.000000f, 0.317966f, 0.668687f, 0.633460f, 0.389340f, 
		0.668530f, 0.633452f, -0.389623f, 0.708875f, 0.143083f, 0.690669f, 
		0.00691346f, -0.999942f, 0.00824130f, 0.708584f, 0.143089f, -0.690966f, 
		0.00690965f, -0.999942f, -0.00824376f, -0.688402f, -0.470218f, 0.552266f, 
		-0.688637f, -0.470212f, -0.551977f, 0.0148929f, -0.999880f, -0.00423823f, 
		0.984829f, 0.000000f, -0.173529f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 0.962476f, 0.000000f, -0.271365f, 
		-0.0149114f, 0.999880f, -0.00423869f, -0.984906f, 0.000000f, -0.173091f, 
		-0.962594f, 0.000000f, -0.270947f, 7.99359e-006f, 1.00000f, -3.50564e-005f, 
		-1.64983e-006f, -1.00000f, 1.88830e-005f, 0.691630f, 0.469583f, -0.548763f, 
		0.691865f, 0.469581f, 0.548467f, -0.718672f, -0.634053f, 0.285459f, 
		-0.718793f, -0.634055f, -0.285152f, -0.0154893f, 0.999880f, 0.000348572f, 
		-0.992172f, -1.09229e-005f, 0.124882f, -0.999692f, 0.000000f, 0.0248219f, 
		-9.12782e-007f, 1.00000f, 4.04419e-006f, 1.09389e-009f, -1.00000f, 5.35907e-006f, 
		0.0154814f, -0.999880f, 0.000339699f, 0.992225f, 0.000000f, 0.124460f, 
		-7.99379e-006f, -1.00000f, 3.50573e-005f, 1.64979e-006f, 1.00000f, -1.88827e-005f, 
		0.999703f, 0.000000f, 0.0243818f, 0.721324f, 0.633452f, -0.280053f, 
		0.721443f, 0.633449f, 0.279754f, -0.754598f, -0.634055f, -0.168985f, 
		-0.753256f, -0.262691f, 0.602991f, -0.754530f, -0.634050f, 0.169306f, 
		-0.753516f, -0.262687f, -0.602668f, -0.770886f, -0.634052f, 0.0609345f, 
		-0.765587f, -0.470223f, -0.439052f, -0.765403f, -0.470224f, 0.439371f, 
		-0.770912f, -0.634052f, -0.0606127f, 0.00806197f, -0.999942f, -0.00712270f, 
		-0.803610f, -0.143426f, 0.577615f, 0.00806499f, -0.999942f, 0.00711933f, 
		-0.803861f, -0.143422f, -0.577268f, 0.753744f, 0.633455f, 0.174940f, 
		0.755334f, 0.262195f, -0.600604f, 0.753673f, 0.633450f, -0.175263f, 
		0.755589f, 0.262193f, 0.600284f, 0.771824f, 0.633453f, -0.0549960f, 
		0.763508f, 0.469582f, 0.443337f, 0.763323f, 0.469579f, -0.443659f, 
		0.771851f, 0.633449f, 0.0546708f, 0.741708f, 0.143067f, -0.655287f, 
		0.00873499f, -0.999942f, -0.00627846f, 0.741987f, 0.143068f, 0.654971f, 
		0.00873759f, -0.999942f, 0.00627474f, -0.820599f, -0.470222f, 0.324821f, 
		-0.820745f, -0.470211f, -0.324468f, -0.836500f, -0.262688f, -0.480898f, 
		-0.836300f, -0.262683f, 0.481248f, 0.822658f, 0.469575f, -0.320519f, 
		0.822791f, 0.469583f, 0.320166f, 0.00957644f, -0.999942f, 0.00490297f, 
		-0.831272f, -0.143413f, -0.537047f, 0.00957350f, -0.999942f, -0.00490650f, 
		-0.831051f, -0.143407f, 0.537390f, 0.835044f, 0.262188f, 0.483693f, 
		0.834838f, 0.262198f, -0.484044f, -0.860988f, -0.470222f, -0.193882f, 
		-0.860905f, -0.470222f, 0.194250f, 0.880967f, 0.143062f, 0.451033f, 
		0.00903613f, -0.999942f, 0.00583760f, 0.880776f, 0.143077f, -0.451401f, 
		0.00903282f, -0.999942f, -0.00584090f, -0.879886f, -0.470221f, 0.0685068f, 
		-0.879918f, -0.470214f, -0.0681366f, 0.860265f, 0.469581f, 0.198588f, 
		0.860183f, 0.469577f, -0.198954f, -0.897531f, -0.262682f, 0.354170f, 
		-0.897678f, -0.262688f, -0.353792f, 0.880584f, 0.469576f, -0.0637994f, 
		0.880610f, 0.469578f, 0.0634211f, 0.00980285f, -0.999942f, -0.00442965f, 
		-0.938168f, -0.143407f, 0.315080f, 0.00980523f, -0.999942f, 0.00442561f, 
		-0.938297f, -0.143424f, -0.314687f, 0.898809f, 0.262192f, -0.351279f, 
		0.898957f, 0.262192f, 0.350902f, -0.941085f, -0.262691f, -0.212963f, 
		-0.940995f, -0.262685f, 0.213367f, 0.901908f, 0.143078f, -0.407541f, 
		0.0101979f, -0.999942f, -0.00342494f, 0.902079f, 0.143071f, 0.407167f, 
		0.0101993f, -0.999942f, 0.00342061f, -0.962048f, -0.262686f, 0.0738864f, 
		-0.962077f, -0.262693f, -0.0734796f, 0.0105966f, -0.999942f, 0.00186258f, 
		-0.952640f, -0.143424f, -0.268154f, 0.0105946f, -0.999942f, -0.00186696f, 
		-0.952528f, -0.143405f, 0.268563f, 0.940519f, 0.262182f, 0.216064f, 
		0.940424f, 0.262193f, -0.216467f, 0.0106739f, -0.999942f, -0.00134352f, 
		-0.989355f, -0.143433f, 0.0245577f, 0.0106734f, -0.999942f, 0.00133894f, 
		-0.989371f, -0.143396f, -0.0241291f, 0.962419f, 0.262189f, -0.0707517f, 
		0.962447f, 0.262199f, 0.0703314f, 0.974770f, 0.143073f, 0.171327f, 
		0.0103564f, -0.999942f, 0.00291522f, 0.974696f, 0.143076f, -0.171746f, 
		0.0103543f, -0.999942f, -0.00291932f, 0.981962f, 0.143087f, -0.123598f, 
		0.0107555f, -0.999942f, -0.000266878f, 0.982016f, 0.143080f, 0.123178f, 
		0.0107538f, -0.999942f, 0.000262242f
	};
	static const GLfloat TexCoordData[] = {
		58.0022f, -40.1476f, 58.0022f, -40.1476f, 1.92169f, -40.1476f, 
		1.92169f, -40.1476f, 53.8407f, -40.1476f, 
		53.8407f, -40.1476f, 6.08320f, -40.1476f, 
		6.08320f, -40.1476f, 57.9745f, -37.5907f, 
		1.94935f, -37.5907f, 57.9723f, -40.1519f, 
		57.9723f, -40.1519f, 1.95158f, -40.1519f, 
		1.95158f, -40.1519f, 53.9230f, -37.5907f, 
		6.00084f, -37.5907f, 53.9297f, -40.1519f, 
		53.9297f, -40.1519f, 5.99420f, -40.1519f, 
		5.99420f, -40.1519f, 57.9454f, -37.6738f, 
		1.97845f, -37.6738f, 62.0708f, -40.1476f, 
		62.0708f, -40.1476f, -2.14686f, -40.1476f, 
		-2.14686f, -40.1476f, 54.0097f, -37.6738f, 
		5.91419f, -37.6738f, 61.9355f, -37.5907f, 
		-2.01165f, -37.5907f, 61.9246f, -40.1519f, 
		61.9246f, -40.1519f, -2.00074f, -40.1519f, 
		-2.00074f, -40.1519f, 57.8958f, -35.0019f, 
		2.02812f, -35.0019f, 61.7933f, -37.6738f, 
		-1.86937f, -37.6738f, 54.1576f, -35.0019f, 
		5.76629f, -35.0019f, 57.8689f, -35.1636f, 
		2.05498f, -35.1636f, 49.9560f, -40.1476f, 
		49.9560f, -40.1476f, 9.96791f, -40.1476f, 
		9.96791f, -40.1476f, 54.2376f, -35.1636f, 
		5.68632f, -35.1636f, 50.1410f, -37.5907f, 
		9.78285f, -37.5907f, 50.1560f, -40.1519f, 
		50.1560f, -40.1519f, 9.76792f, -40.1519f, 
		9.76792f, -40.1519f, 61.5504f, -35.0019f, 
		-1.62655f, -35.0019f, 50.3358f, -37.6738f, 
		9.58814f, -37.6738f, 61.4191f, -35.1636f, 
		-1.49524f, -35.1636f, 65.6848f, -40.1476f, 
		65.6848f, -40.1476f, -5.76095f, -40.1476f, 
		-5.76095f, -40.1476f, 57.7722f, -32.5361f, 
		50.6681f, -35.0019f, 9.25581f, -35.0019f, 
		2.15170f, -32.5361f, 65.4541f, -37.5907f, 
		54.5256f, -32.5361f, -5.53019f, -37.5907f, 
		5.39832f, -32.5361f, 65.4355f, -40.1519f, 
		65.4355f, -40.1519f, -5.51157f, -40.1519f, 
		-5.51157f, -40.1519f, 57.7489f, -32.7720f, 
		50.8478f, -35.1636f, 9.07611f, -35.1636f, 
		2.17504f, -32.7720f, 65.2113f, -37.6738f, 
		54.5951f, -32.7720f, -5.28738f, -37.6738f, 
		5.32880f, -32.7720f, 60.9463f, -32.5361f, 
		-1.02240f, -32.5361f, 0.411015f, -1.50000f, 
		0.411015f, -1.50000f, 0.411015f, 1.50000f, 
		0.411015f, 1.50000f, 0.411015f, 1.50000f, 
		-0.411017f, -1.50000f, -0.411017f, -1.50000f, 
		-0.411017f, 1.50000f, -0.411017f, 1.50000f, 
		-0.411017f, 1.50000f, 60.8322f, -32.7720f, 
		-0.908269f, -32.7720f, 64.7969f, -35.0019f, 
		-4.87298f, -35.0019f, 1.22387f, -1.50000f, 
		1.22387f, -1.50000f, 1.22387f, 1.50000f, 
		1.22387f, 1.50000f, 1.22387f, 1.50000f, 
		-1.22387f, -1.50000f, -1.22387f, -1.50000f, 
		-1.22387f, 1.50000f, -1.22387f, 1.50000f, 
		-1.22387f, 1.50000f, 64.5728f, -35.1636f, 
		-4.64889f, -35.1636f, 46.6933f, -40.1476f, 
		46.6933f, -40.1476f, 13.2306f, -40.1476f, 
		13.2306f, -40.1476f, 51.4949f, -32.5361f, 
		8.42899f, -32.5361f, 2.00938f, -1.50000f, 
		2.00938f, -1.50000f, 2.00938f, 1.50000f, 
		2.00938f, 1.50000f, 2.00938f, 1.50000f, 
		-2.00938f, -1.50000f, -2.00938f, -1.50000f, 
		-2.00938f, 1.50000f, -2.00938f, 1.50000f, 
		-2.00938f, 1.50000f, -2.00938f, 1.50000f, 
		46.9646f, -37.5907f, 12.9593f, -37.5907f, 
		46.9865f, -40.1519f, 46.9865f, -40.1519f, 
		12.9374f, -40.1519f, 12.9374f, -40.1519f, 
		51.6511f, -32.7720f, 8.27279f, -32.7720f, 
		0.373650f, -1.50000f, 0.373650f, -1.55667f, 
		-0.373652f, -1.50000f, -0.373652f, -1.55669f, 
		47.2500f, -37.6738f, 12.6739f, -37.6738f, 
		0.367684f, -2.67198f, -0.367685f, -2.67198f, 
		1.11260f, -1.50000f, 1.11260f, -1.55666f, 
		-1.11260f, -1.50000f, -1.11260f, -1.55666f, 
		57.6101f, -30.3485f, 1.09484f, -2.67198f, 
		-1.09484f, -2.67198f, 2.31377f, -30.3485f, 
		2.75000f, -1.50000f, 2.75000f, -1.50000f, 
		2.75000f, 1.50000f, 2.75000f, 1.50000f, 
		2.75000f, 1.50000f, -2.75000f, -1.50000f, 
		-2.75000f, -1.50000f, -2.75000f, 1.50000f, 
		-2.75000f, 1.50000f, -2.75000f, 1.50000f, 
		47.7373f, -35.0019f, 12.1866f, -35.0019f, 
		63.7658f, -32.5361f, 55.0082f, -30.3485f, 
		-3.84195f, -32.5361f, 4.91572f, -30.3485f, 
		0.350691f, -3.30746f, -0.350693f, -3.30746f, 
		1.82670f, -1.50000f, 1.82670f, -1.55665f, 
		-1.82670f, -1.50000f, -1.82670f, -1.55666f, 
		57.5914f, -30.6496f, 2.33252f, -30.6496f, 
		1.79754f, -2.67198f, -1.79754f, -2.67198f, 
		1.04424f, -3.30746f, -1.04424f, -3.30746f, 
		48.0007f, -35.1636f, 11.9232f, -35.1636f, 
		63.5711f, -32.7720f, 55.0640f, -30.6496f, 
		-3.64717f, -32.7720f, 4.85989f, -30.6496f, 
		60.1539f, -30.3485f, -0.230049f, -30.3485f, 
		1.71447f, -3.30746f, -1.71447f, -3.30746f, 
		2.50000f, -1.50000f, 2.50000f, -1.55667f, 
		-2.50000f, -1.50000f, -2.50000f, -1.55666f, 
		60.0623f, -30.6496f, -0.138397f, -30.6496f, 
		68.5233f, -40.1476f, 68.5233f, -40.1476f, 
		-8.59943f, -40.1476f, -8.59943f, -40.1476f, 
		-0.324035f, -3.86822f, 3.42919f, -1.50000f, 
		3.42919f, -1.50000f, 3.42919f, 1.50000f, 
		3.42919f, 1.50000f, 3.42919f, 1.50000f, 
		-3.42919f, -1.50000f, -3.42919f, -1.50000f, 
		-3.42919f, 1.50000f, -3.42919f, 1.50000f, 
		-3.42919f, 1.50000f, 2.46008f, -2.67198f, 
		-2.46008f, -2.67198f, -0.964865f, -3.86822f, 
		68.2175f, -37.5907f, -8.29363f, -37.5907f, 
		68.1929f, -40.1519f, 68.1929f, -40.1519f, 
		-8.26896f, -40.1519f, -8.26896f, -40.1519f, 
		52.5793f, -30.3485f, 7.34459f, -30.3485f, 
		0.309284f, -4.06561f, 67.8958f, -37.6738f, 
		-7.97187f, -37.6738f, 48.9495f, -32.5361f, 
		10.9744f, -32.5361f, 2.34639f, -3.30746f, 
		-2.34639f, -3.30746f, 52.7047f, -30.6496f, 
		0.920959f, -4.06554f, 7.21915f, -30.6496f, 
		49.1785f, -32.7720f, 10.7454f, -32.7720f, 
		3.11745f, -1.50000f, 3.11745f, -1.55666f, 
		-3.11745f, -1.50000f, -3.11745f, -1.55663f, 
		67.3466f, -35.0019f, -7.42271f, -35.0019f, 
		1.51200f, -4.06569f, -1.51202f, -4.06565f, 
		3.06767f, -2.67198f, -3.06767f, -2.67198f, 
		67.0496f, -35.1636f, -7.12575f, -35.1636f, 
		62.4136f, -30.3485f, -2.48972f, -30.3485f, 
		4.03179f, -1.50000f, 4.03179f, -1.50000f, 
		4.03179f, 1.50000f, 4.03179f, 1.50000f, 
		4.03179f, 1.50000f, -4.03179f, -1.50000f, 
		-4.03179f, -1.50000f, -4.03179f, 1.50000f, 
		-4.03179f, 1.50000f, -4.03179f, 1.50000f, 
		2.92590f, -3.30746f, -2.92590f, -3.30746f, 
		62.2572f, -30.6496f, -2.33331f, -30.6496f, 
		-0.270682f, -4.49262f, 2.06936f, -4.06558f, 
		-2.06933f, -4.06562f, -0.805990f, -4.49264f, 
		3.66526f, -1.50000f, 3.66526f, -1.55667f, 
		-3.66526f, -1.50000f, -3.66526f, -1.55663f, 
		65.9803f, -32.5361f, -6.05640f, -32.5361f, 
		57.4159f, -28.5940f, 2.50804f, -28.5940f, 
		3.60673f, -2.67198f, -3.60673f, -2.67198f, 
		44.3424f, -40.1476f, 44.3424f, -40.1476f, 
		15.5815f, -40.1476f, 15.5815f, -40.1476f, 
		55.5866f, -28.5940f, 4.33726f, -28.5940f, 
		65.7222f, -32.7720f, -5.79828f, -32.7720f, 
		57.4026f, -28.9471f, 50.5393f, -30.3485f, 
		2.52128f, -28.9471f, 9.38457f, -30.3485f, 
		2.58042f, -4.06562f, -2.58040f, -4.06565f, 
		44.6759f, -37.5907f, 15.2480f, -37.5907f, 
		44.7028f, -40.1519f, 44.7028f, -40.1519f, 
		0.667851f, -4.69278f, 15.2211f, -40.1519f, 
		15.2211f, -40.1519f, 55.6261f, -28.9471f, 
		4.29781f, -28.9471f, 3.44005f, -3.30746f, 
		-3.44005f, -3.30746f, 50.7232f, -30.6496f, 
		9.20068f, -30.6496f, 45.0267f, -37.6738f, 
		14.8972f, -37.6738f, 59.2042f, -28.5940f, 
		0.719677f, -28.5940f, 4.54431f, -1.50000f, 
		4.54431f, -1.50000f, 4.54431f, 1.50000f, 
		4.54431f, 1.50000f, 4.54431f, 1.50000f, 
		-4.54431f, -1.50000f, -4.54431f, -1.50000f, 
		-4.54431f, 1.50000f, -4.54431f, 1.50000f, 
		-4.54431f, 1.50000f, 59.1395f, -28.9471f, 
		1.25203f, -4.69278f, 0.784444f, -28.9471f, 
		0.411015f, -1.50000f, 0.411015f, -1.50000f, 
		0.411015f, 1.50000f, 0.411015f, 1.50000f, 
		-0.411017f, -1.50000f, -0.411017f, -1.50000f, 
		-0.411017f, 1.50000f, -0.411017f, 1.50000f, 
		45.6256f, -35.0019f, 14.2983f, -35.0019f, 
		1.22387f, -1.50000f, 1.22387f, -1.50000f, 
		1.22387f, 1.50000f, 1.22387f, 1.50000f, 
		-1.22387f, -1.50000f, -1.22387f, -1.50000f, 
		-1.22387f, 1.50000f, -1.22387f, 1.50000f, 
		53.8791f, -28.5940f, 6.04481f, -28.5940f, 
		45.9494f, -35.1636f, 13.9745f, -35.1636f, 
		-1.58203f, -4.69278f, 53.9677f, -28.9471f, 
		5.95617f, -28.9471f, 3.03390f, -4.06557f, 
		2.00938f, -1.50000f, 2.00938f, -1.50000f, 
		2.00938f, 1.50000f, 2.00938f, 1.50000f, 
		-3.03385f, -4.06563f, -2.00938f, -1.50000f, 
		-2.00938f, -1.50000f, -2.00938f, 1.50000f, 
		-2.00938f, 1.50000f, 4.13119f, -1.50000f, 
		4.13119f, -1.55667f, -4.13120f, -1.50000f, 
		-4.13120f, -1.55668f, 4.06523f, -2.67198f, 
		-4.06523f, -2.67198f, 0.373650f, 1.50000f, 
		0.373650f, 1.53446f, -0.373651f, 1.50000f, 
		-0.373651f, 1.53452f, 0.373650f, -1.50000f, 
		0.373650f, -1.53452f, -0.373652f, -1.50000f, 
		-0.373652f, -1.53448f, 64.1883f, -30.3485f, 
		-4.26445f, -30.3485f, 0.367684f, -2.67198f, 
		0.367684f, 2.67198f, -0.367685f, -2.67198f, 
		-0.367685f, 2.67198f, 1.11260f, -1.50000f, 
		1.11260f, -1.53447f, -1.11260f, -1.50000f, 
		-1.11260f, -1.53448f, 1.11260f, 1.50000f, 
		1.11260f, 1.53450f, -1.11260f, 1.50000f, 
		-1.11260f, 1.53446f, 60.7928f, -28.5940f, 
		3.87736f, -3.30746f, -3.87736f, -3.30746f, 
		-0.868918f, -28.5940f, 63.9811f, -30.6496f, 
		-4.05718f, -30.6496f, -0.673468f, -4.92016f, 
		2.75000f, -1.50000f, 2.75000f, -1.50000f, 
		2.75000f, 1.50000f, 2.75000f, 1.50000f, 
		-2.75000f, -1.50000f, -2.75000f, -1.50000f, 
		-2.75000f, 1.50000f, -2.75000f, 1.50000f, 
		47.1155f, -32.5361f, 12.8084f, -32.5361f, 
		1.07075f, 2.97449f, -1.07076f, 2.97435f, 
		1.07076f, -2.97446f, -1.07077f, -2.97427f, 
		1.82670f, 1.50000f, 1.82670f, 1.53448f, 
		-1.82670f, 1.50000f, -1.82670f, 1.53447f, 
		60.6823f, -28.9471f, -0.851746f, -4.92016f, 
		-0.758388f, -28.9471f, 1.82670f, -1.50000f, 
		1.82670f, -1.53446f, -1.82670f, -1.50000f, 
		-1.82670f, -1.53448f, -2.08708f, -4.69278f, 
		47.3969f, -32.7720f, 12.5270f, -32.7720f, 
		0.337983f, -3.57480f, -0.337985f, -3.57478f, 
		0.337986f, 3.57475f, -0.337984f, 3.57480f, 
		1.75802f, -2.97429f, -1.75802f, -2.97427f, 
		1.75802f, 2.97430f, -1.75801f, 2.97432f, 
		-0.422419f, 1.03827f, -0.422017f, 1.00123f, 
		0.422127f, 1.03827f, 0.421723f, 1.00117f, 
		2.50000f, -1.50000f, 2.50000f, -1.53451f, 
		-2.50000f, -1.50000f, -2.50000f, -1.53446f, 
		4.95533f, -1.50000f, 4.95533f, -1.50000f, 
		4.95533f, 1.50000f, 4.95533f, 1.50000f, 
		4.95533f, 1.50000f, -4.95533f, -1.50000f, 
		-4.95533f, -1.50000f, -4.95533f, 1.50000f, 
		-4.95533f, 1.50000f, -4.95533f, 1.50000f, 
		2.50000f, 1.50000f, 2.50000f, 1.53446f, 
		-2.50000f, 1.50000f, -2.50000f, 1.53451f, 
		0.457774f, 1.03827f, 0.457337f, 1.00115f, 
		-0.457776f, 1.03827f, -0.457339f, 1.00115f, 
		1.25804f, -4.92016f, 3.42919f, -1.50000f, 
		3.42919f, -1.50000f, 3.42919f, 1.50000f, 
		3.42919f, 1.50000f, -3.42919f, -1.50000f, 
		-3.42919f, -1.50000f, -3.42919f, 1.50000f, 
		-3.42919f, 1.50000f, 3.41959f, -4.06555f, 
		-3.41950f, -4.06566f, 1.36310f, 1.03827f, 
		1.36180f, 1.00117f, -1.36310f, 1.03827f, 
		-1.36180f, 1.00125f, 2.40593f, 2.97466f, 
		-2.40596f, 2.97449f, 2.40596f, -2.97451f, 
		-2.40596f, -2.97450f, 52.4449f, -28.5940f, 
		1.38845f, -4.92016f, 29.9571f, -20.9116f, 
		29.9571f, -20.9116f, 2.35565f, -4.69278f, 
		7.47896f, -28.5940f, 30.3690f, -20.9116f, 
		30.3690f, -20.9116f, 29.5451f, -20.9116f, 
		29.5451f, -20.9116f, 70.3340f, -40.1476f, 
		70.3340f, -40.1476f, -10.4101f, -40.1476f, 
		-10.4101f, -40.1476f, 52.5749f, -28.9471f, 
		30.7125f, -20.9116f, 30.7125f, -20.9116f, 
		29.2016f, -20.9116f, 29.2016f, -20.9116f, 
		7.34901f, -28.9471f, 0.914981f, 4.09240f, 
		-0.914900f, 4.09276f, 0.914984f, -4.09239f, 
		-0.914901f, -4.09276f, -2.23797f, 1.03827f, 
		-2.23584f, 1.00119f, 2.23797f, 1.03827f, 
		2.23584f, 1.00123f, 69.9803f, -37.5907f, 
		2.23797f, 1.03827f, 2.23584f, 1.00121f, 
		69.9518f, -40.1519f, 69.9518f, -40.1519f, 
		-2.23797f, 1.03827f, -2.23584f, 1.00117f, 
		4.50485f, -1.50000f, 4.50485f, -1.55668f, 
		-10.0564f, -37.5907f, -4.50484f, -1.50000f, 
		-4.50484f, -1.55669f, 0.495526f, 0.750000f, 
		0.855905f, 0.250000f, -1.29157f, 0.140495f, 
		-10.0279f, -40.1519f, -10.0279f, -40.1519f, 
		0.495526f, 0.750000f, 0.855905f, 0.250000f, 
		-1.29157f, 0.140468f, 0.481869f, 0.750000f, 
		0.869561f, 0.250000f, 0.869561f, 0.250000f, 
		0.481869f, 0.750000f, 3.11745f, 1.50000f, 
		3.11745f, 1.53450f, -3.11745f, 1.50000f, 
		-3.11745f, 1.53448f, 30.9894f, -20.9116f, 
		30.9894f, -20.9116f, 28.9247f, -20.9116f, 
		28.9247f, -20.9116f, 69.6082f, -37.6738f, 
		3.11745f, -1.50000f, 3.11745f, -1.53451f, 
		-3.11745f, -1.50000f, -3.11745f, -1.53447f, 
		0.522040f, 0.750000f, 0.829390f, 0.250000f, 
		0.468500f, 0.750000f, 0.882930f, 0.250000f, 
		-9.68431f, -37.6738f, 0.407233f, -0.250075f, 
		-0.407221f, -0.250295f, 0.522040f, 0.750000f, 
		0.829390f, 0.250000f, 4.43291f, -2.67198f, 
		0.882930f, 0.250000f, 0.468500f, 0.750000f, 
		-4.43291f, -2.67198f, 0.407220f, -0.250262f, 
		-0.407201f, -0.250580f, 1.50223f, -4.09242f, 
		-1.50221f, -4.09248f, 1.50223f, 4.09244f, 
		-1.50222f, 4.09247f, 2.11720f, 0.140527f, 
		-2.12043f, 0.140020f, 49.0695f, -30.3485f, 
		10.8544f, -30.3485f, 3.00017f, -2.97457f, 
		-3.00022f, -2.97432f, 1.21255f, -0.250360f, 
		3.00017f, 2.97454f, -3.00022f, 2.97433f, 
		-3.06285f, 1.03827f, -3.05994f, 1.00124f, 
		3.06285f, 1.03827f, 3.05994f, 1.00123f, 
		0.455342f, 0.750000f, 0.896088f, 0.250000f, 
		3.06285f, 1.03827f, 3.05993f, 1.00116f, 
		-3.06285f, 1.03827f, -3.05993f, 1.00111f, 
		0.896088f, 0.250000f, 0.455342f, 0.750000f, 
		4.03179f, -1.50000f, 4.03179f, -1.50000f, 
		4.03179f, 1.50000f, 4.03179f, 1.50000f, 
		-4.03179f, -1.50000f, -4.03179f, -1.50000f, 
		-4.03179f, 1.50000f, -4.03179f, 1.50000f, 
		31.2016f, -20.9116f, 31.2016f, -20.9116f, 
		28.7125f, -20.9116f, 28.7125f, -20.9116f, 
		68.9731f, -35.0019f, 4.22805f, -3.30746f, 
		-4.22805f, -3.30746f, -9.04919f, -35.0019f, 
		49.2955f, -30.6496f, 10.6284f, -30.6496f, 
		0.680589f, -5.00000f, -0.147966f, -5.00000f, 
		-1.99078f, -0.250370f, 1.99087f, -0.250209f, 
		1.99083f, -0.250268f, 29.9571f, -23.0040f, 
		29.9571f, -23.0040f, 29.9571f, -18.8192f, 
		29.9571f, -18.8192f, 0.268878f, -4.50797f, 
		-0.268859f, -4.50813f, 68.6296f, -35.1636f, 
		0.268878f, 4.50796f, -0.268863f, 4.50810f, 
		0.547731f, 0.750000f, 0.803700f, 0.250000f, 
		30.3690f, -22.9882f, 30.3690f, -22.9882f, 
		30.3690f, -18.8350f, 30.3690f, -18.8350f, 
		29.5451f, -22.9882f, 29.5451f, -22.9882f, 
		29.5451f, -18.8350f, 29.5451f, -18.8350f, 
		-8.70575f, -35.1636f, 2.05583f, 4.09263f, 
		-2.05580f, 4.09270f, 0.547731f, 0.750000f, 
		0.803700f, 0.250000f, 0.442334f, 0.750000f, 
		0.909097f, 0.250000f, 2.05581f, -4.09266f, 
		-2.05582f, -4.09266f, 0.442334f, 0.750000f, 
		0.909097f, 0.250000f, 30.7125f, -22.9430f, 
		30.7125f, -22.9430f, 30.7125f, -18.8802f, 
		30.7125f, -18.8802f, 29.2016f, -22.9430f, 
		29.2016f, -22.9430f, 29.2016f, -18.8802f, 
		29.2016f, -18.8802f, 31.3509f, -20.9116f, 
		31.3509f, -20.9116f, 28.5632f, -20.9116f, 
		28.5632f, -20.9116f, 62.0405f, -28.5940f, 
		0.495526f, 0.833333f, 0.855905f, 0.333333f, 
		0.495526f, 0.666667f, 0.855905f, 0.166667f, 
		-2.11659f, -28.5940f, 0.495526f, 0.666667f, 
		0.855905f, 0.166667f, 0.495526f, 0.833333f, 
		0.855905f, 0.333333f, 0.481869f, 0.833333f, 
		0.869561f, 0.333333f, 0.481869f, 0.666667f, 
		0.869561f, 0.166667f, 3.66526f, -1.50000f, 
		3.66526f, -1.53451f, -3.66526f, -1.50000f, 
		-3.66526f, -1.53445f, 0.481869f, 0.666667f, 
		0.869561f, 0.166667f, 0.869561f, 0.333333f, 
		0.481869f, 0.833333f, -3.90666f, -3.86822f, 
		1.09523f, -0.699557f, 30.9894f, -22.8721f, 
		30.9894f, -22.8721f, 30.9894f, -18.9511f, 
		30.9894f, -18.9511f, 28.9247f, -22.8721f, 
		28.9247f, -22.8721f, 28.9247f, -18.9511f, 
		28.9247f, -18.9511f, 3.66526f, 1.50000f, 
		3.66526f, 1.53450f, -3.66526f, 1.50000f, 
		-3.66526f, 1.53449f, 0.522040f, 0.833333f, 
		0.829390f, 0.333333f, 0.522040f, 0.666667f, 
		0.829390f, 0.166667f, 0.468500f, 0.833333f, 
		0.882930f, 0.333333f, 0.468500f, 0.666667f, 
		0.882930f, 0.166667f, 61.8940f, -28.9471f, 
		-2.72461f, -0.250284f, 0.522040f, 0.666667f, 
		0.829390f, 0.166667f, 0.522040f, 0.833333f, 
		0.829390f, 0.333333f, -0.226863f, 4.92753f, 
		0.353529f, 4.92753f, -0.226863f, 4.92753f, 
		3.81931f, 1.03827f, 3.81568f, 1.00121f, 
		0.468500f, 0.666667f, 0.882930f, 0.166667f, 
		-3.81932f, 1.03827f, -3.81567f, 1.00113f, 
		0.882930f, 0.333333f, 0.468500f, 0.833333f, 
		0.429425f, 0.750000f, 0.922005f, 0.250000f, 
		-1.97012f, -28.9471f, 2.72460f, -0.250286f, 
		-2.72463f, -0.250213f, -0.226898f, 4.92753f, 
		-1.05269f, 4.92753f, -0.226898f, 4.92753f, 
		-0.472255f, 3.84315f, -0.472255f, 3.84315f, 
		0.429425f, 0.750000f, 0.922005f, 0.250000f, 
		3.60673f, -2.67198f, 3.60673f, 2.67198f, 
		-3.60673f, -2.67198f, -3.60673f, 2.67198f, 
		-0.472255f, 3.84315f, -0.472255f, 3.84315f, 
		-1.41470f, -0.707128f, -0.353530f, 3.84315f, 
		-0.353530f, 3.84315f, -0.353530f, 4.92753f, 
		-0.353530f, 4.92753f, -0.353530f, 4.92753f, 
		-1.41461f, -0.707052f, -0.921012f, 3.84315f, 
		-0.921012f, 3.84315f, 2.73686f, -4.69278f, 
		-1.45562f, -0.707041f, -1.16090f, 4.92753f, 
		-1.16090f, 4.92753f, -0.353530f, 4.92753f, 
		-1.16090f, 4.92753f, 0.455342f, 0.833333f, 
		0.896088f, 0.333333f, 31.4391f, -20.9116f, 
		31.4391f, -20.9116f, 0.455342f, 0.666667f, 
		0.896088f, 0.166667f, 28.4750f, -20.9116f, 
		28.4750f, -20.9116f, 3.72875f, -4.06569f, 
		1.05269f, 3.84315f, 1.05269f, 3.84315f, 
		1.05269f, 4.92753f, 1.05269f, 4.92753f, 
		0.896088f, 0.166667f, 0.455342f, 0.666667f, 
		0.455342f, 0.833333f, 0.896088f, 0.333333f, 
		2.56358f, -4.09263f, -2.56363f, -4.09254f, 
		31.2016f, -22.7792f, 31.2016f, -22.7792f, 
		31.2016f, -19.0440f, 31.2016f, -19.0440f, 
		28.7125f, -22.7792f, 28.7125f, -22.7792f, 
		28.7125f, -19.0440f, 28.7125f, -19.0440f, 
		67.3929f, -32.5361f, 2.56354f, 4.09269f, 
		-2.56361f, 4.09257f, 3.07733f, -27.4276f, 
		-7.46901f, -32.5361f, 57.1957f, -27.4276f, 
		1.59449f, 4.92753f, 1.05269f, 4.92753f, 
		2.36537f, 4.92753f, 1.59449f, 4.92753f, 
		67.0944f, -32.7720f, 0.573028f, 0.750000f, 
		0.778402f, 0.250000f, -7.17049f, -32.7720f, 
		0.547731f, 0.833333f, 0.803700f, 0.333333f, 
		0.547731f, 0.666667f, 0.803700f, 0.166667f, 
		0.573028f, 0.750000f, 0.778402f, 0.250000f, 
		0.547731f, 0.666667f, 0.803700f, 0.166667f, 
		0.547731f, 0.833333f, 0.803700f, 0.333333f, 
		0.442334f, 0.833333f, 0.909097f, 0.333333f, 
		0.442334f, 0.666667f, 0.909097f, 0.166667f, 
		3.31552f, 3.57452f, 3.32206f, -27.8151f, 
		-3.31545f, 3.57467f, 4.54431f, -1.50000f, 
		4.54431f, -1.50000f, 4.54431f, 1.50000f, 
		4.54431f, 1.50000f, -4.54431f, -1.50000f, 
		-4.54431f, -1.50000f, -4.54431f, 1.50000f, 
		-4.54431f, 1.50000f, 0.416577f, 0.750000f, 
		0.934854f, 0.250000f, 0.909097f, 0.166667f, 
		0.442334f, 0.666667f, 0.442334f, 0.833333f, 
		0.909097f, 0.333333f, 1.82354f, 3.84315f, 
		1.82354f, 3.84315f, 3.31552f, -3.57452f, 
		-3.31545f, -3.57467f, 0.416577f, 0.750000f, 
		0.934854f, 0.250000f, -0.685306f, -0.900427f, 
		57.5199f, -27.8151f, 0.308314f, -0.960301f, 
		0.664191f, -0.901705f, -3.39750f, -0.250325f, 
		1.50729f, -0.960301f, 0.664130f, -0.901707f, 
		3.39752f, -0.250320f, -1.24968f, -4.69278f, 
		3.39750f, -0.250306f, -3.39748f, -0.250425f, 
		55.9163f, -27.4276f, 31.3509f, -22.6678f, 
		31.3509f, -22.6678f, 31.3509f, -19.1554f, 
		31.3509f, -19.1554f, 28.5632f, -22.6678f, 
		28.5632f, -22.6678f, 28.5632f, -19.1554f, 
		28.5632f, -19.1554f, 56.2631f, -27.8151f, 
		-0.440591f, -5.00000f, -1.18521f, -5.00000f, 
		31.4681f, -20.9116f, 31.4681f, -20.9116f, 
		28.4460f, -20.9116f, 28.4460f, -20.9116f, 
		-4.49046f, 1.03827f, -4.48619f, 1.00122f, 
		4.49046f, 1.03827f, 4.48618f, 1.00117f, 
		4.49046f, 1.03827f, 4.48618f, 1.00119f, 
		2.23244f, 3.84315f, 2.23244f, 3.84315f, 
		-4.49046f, 1.03827f, -4.48619f, 1.00124f, 
		5.25565f, -1.50000f, 5.25565f, -1.50000f, 
		5.25565f, 1.50000f, 5.25565f, 1.50000f, 
		5.25565f, 1.50000f, 2.23247f, 3.84315f, 
		2.23247f, 3.84315f, -2.23244f, 3.84315f, 
		-2.23244f, 3.84315f, -5.25565f, -1.50000f, 
		-5.25565f, -1.50000f, -5.25565f, 1.50000f, 
		-5.25565f, 1.50000f, -5.25565f, 1.50000f, 
		-4.46093f, 1.03827f, -4.45669f, 1.00121f, 
		4.46094f, 1.03827f, 4.45669f, 1.00125f, 
		0.429425f, 0.833333f, 0.922005f, 0.333333f, 
		0.429425f, 0.666667f, 0.922005f, 0.166667f, 
		0.805211f, 4.77352f, -0.805203f, 4.77352f, 
		0.429425f, 0.666667f, 0.922005f, 0.166667f, 
		0.429425f, 0.833333f, 0.922005f, 0.333333f, 
		1.58409f, -4.69278f, -1.58394f, -4.69278f, 
		1.83032f, -27.8151f, -2.36537f, 3.84315f, 
		-2.36537f, 3.84315f, -2.36537f, 4.92753f, 
		-2.36537f, 4.92753f, -2.36537f, 4.92753f, 
		2.72497f, -0.707105f, 2.44553f, 4.92753f, 
		1.72833f, 4.92753f, 2.44553f, 4.92753f, 
		2.44556f, 4.92753f, 1.72833f, 4.92753f, 
		2.44556f, 4.92753f, -2.44554f, 4.92753f, 
		-1.72834f, 4.92753f, -2.44554f, 4.92753f, 
		4.13119f, 1.50000f, 4.13119f, 1.53448f, 
		-4.13119f, 1.50000f, -4.13119f, 1.53448f, 
		1.68631f, 4.69278f, -1.68619f, 4.69278f, 
		31.4391f, -22.5415f, 31.4391f, -22.5415f, 
		31.4391f, -19.2817f, 31.4391f, -19.2817f, 
		28.4750f, -22.5415f, 28.4750f, -22.5415f, 
		28.4750f, -19.2817f, 28.4750f, -19.2817f, 
		-3.26342f, -4.49262f, 4.13119f, -1.50000f, 
		4.13119f, -1.53445f, -4.13120f, -1.50000f, 
		-4.13120f, -1.53451f, 0.403754f, 0.750000f, 
		0.947676f, 0.250000f, -4.06523f, -2.67198f, 
		-4.06523f, 2.67198f, 3.97573f, -2.97471f, 
		58.6531f, -27.4276f, 3.97573f, 2.97472f, 
		1.27089f, -27.4276f, -2.91997f, -4.69278f, 
		0.453389f, -4.92016f, 0.573028f, 0.833333f, 
		0.778402f, 0.333333f, 0.573028f, 0.666667f, 
		0.778402f, 0.166667f, 0.400278f, 0.750000f, 
		0.951152f, 0.250000f, 0.573028f, 0.666667f, 
		0.778402f, 0.166667f, 0.573028f, 0.833333f, 
		0.778402f, 0.333333f, 4.52480f, -27.8151f, 
		2.82037f, 4.92753f, 2.36537f, 4.92753f, 
		3.46788f, 4.92753f, 2.82037f, 4.92753f, 
		1.31961f, -4.80778f, 0.416577f, 0.833333f, 
		0.934854f, 0.333333f, 0.416577f, 0.666667f, 
		0.934854f, 0.166667f, -2.82061f, 4.92753f, 
		-3.46788f, 4.92753f, -2.82061f, 4.92753f, 
		0.918056f, -0.960301f, 2.29170f, -0.914330f, 
		51.4116f, -28.5940f, 0.416577f, 0.666667f, 
		0.934854f, 0.166667f, 0.416577f, 0.833333f, 
		0.934854f, 0.333333f, 8.51229f, -28.5940f, 
		4.77786f, -1.50000f, 4.77786f, -1.55666f, 
		-2.22406f, -4.92016f, -4.77786f, -1.50000f, 
		-4.77786f, -1.55666f, -3.99444f, -0.250365f, 
		-3.99444f, -0.250365f, -3.99444f, -0.250365f, 
		4.70157f, -2.67198f, 3.99464f, -0.250149f, 
		-4.70157f, -2.67198f, 3.99451f, -0.250427f, 
		-3.99454f, -0.250292f, 51.5713f, -28.9471f, 
		4.84544f, -27.4276f, 8.35257f, -28.9471f, 
		65.3204f, -30.3485f, 31.4681f, -22.4040f, 
		31.4681f, -22.4040f, 31.4681f, -19.4192f, 
		31.4681f, -19.4192f, 28.4460f, -22.4040f, 
		28.4460f, -22.4040f, 28.4460f, -19.4192f, 
		28.4460f, -19.4192f, 3.01273f, 3.84315f, 
		3.01273f, 3.84315f, -5.39656f, -30.3485f, 
		-3.73693f, -3.57465f, -3.01298f, 3.84315f, 
		-3.01298f, 3.84315f, -3.73694f, 3.57462f, 
		4.48429f, -3.30746f, -4.48429f, -3.30746f, 
		0.392565f, 0.750000f, 0.958865f, 0.250000f, 
		65.0807f, -30.6496f, 0.598212f, 0.750000f, 
		0.753218f, 0.250000f, -5.15684f, -30.6496f, 
		0.598212f, 0.750000f, 0.753218f, 0.250000f, 
		66.7990f, -20.8198f, 66.7990f, -20.8198f, 
		0.252301f, 2.01781f, -0.0722394f, 0.000000f, 
		-0.0722394f, 0.000000f, 2.63744f, 4.50803f, 
		-2.63764f, 4.50786f, 0.252125f, 2.01785f, 
		-0.0722466f, 0.000000f, -0.0722466f, 0.000000f, 
		0.218016f, 0.000000f, 0.218016f, 0.000000f, 
		0.776035f, 2.30437f, 0.776035f, 2.30437f, 
		0.776035f, 2.30437f, 0.218016f, 0.000000f, 
		0.218016f, 0.000000f, 0.772954f, 2.62831f, 
		67.1643f, -20.8198f, 67.1643f, -20.8198f, 
		31.9654f, -20.9116f, 31.9654f, -20.9116f, 
		31.9654f, -20.9116f, 31.9654f, -20.9116f, 
		31.9654f, -20.9116f, 32.0865f, -20.9116f, 
		32.0865f, -20.9116f, 32.0865f, -20.9116f, 
		32.0865f, -20.9116f, 27.9165f, -20.9116f, 
		27.9165f, -20.9116f, 27.9165f, -20.9116f, 
		27.9165f, -20.9116f, 27.9165f, -20.9116f, 
		27.8354f, -20.9116f, 27.8354f, -20.9116f, 
		27.8354f, -20.9116f, 27.8354f, -20.9116f, 
		2.63740f, -4.50807f, -2.63760f, -4.50790f, 
		5.06130f, 1.03827f, 5.05648f, 1.00120f, 
		-5.06130f, 1.03827f, -5.05647f, 1.00116f, 
		0.403754f, 0.666667f, 0.947676f, 0.166667f, 
		0.403754f, 0.833333f, 0.947676f, 0.333333f, 
		32.3899f, -20.9116f, 32.3899f, -20.9116f, 
		27.5242f, -20.9116f, 27.5242f, -20.9116f, 
		4.95533f, -1.50000f, 4.95533f, -1.50000f, 
		4.95533f, 1.50000f, 4.95533f, 1.50000f, 
		-4.95533f, -1.50000f, -4.95533f, -1.50000f, 
		-4.95533f, 1.50000f, -4.95533f, 1.50000f, 
		66.4326f, -20.8198f, 66.4326f, -20.8198f, 
		-0.438561f, 0.000000f, -0.438561f, 0.000000f, 
		0.438561f, 0.000000f, 0.438561f, 0.000000f, 
		54.9389f, -27.8151f, 0.384920f, 0.750000f, 
		0.966510f, 0.250000f, -0.147966f, -5.00000f, 
		-0.669701f, -4.93476f, -0.438555f, 0.000000f, 
		-0.438555f, 0.000000f, 0.438560f, 0.000000f, 
		0.438560f, 0.000000f, -3.34549f, 3.84315f, 
		-3.34549f, 3.84315f, 0.665161f, 1.64873f, 
		0.400278f, 0.833333f, 0.951152f, 0.333333f, 
		0.234532f, 3.25775f, 0.234532f, 3.25775f, 
		0.234532f, 3.25775f, 0.400278f, 0.666667f, 
		0.951152f, 0.166667f, 3.39723f, -4.09261f, 
		0.234631f, 3.25800f, -1.25791f, 2.08964f, 
		3.39720f, 4.09265f, -1.25769f, 2.08987f, 
		-1.25769f, 2.08987f, -1.25769f, 2.08987f, 
		-1.25442f, 2.72001f, -1.25442f, 2.72001f, 
		-1.25442f, 2.72001f, 0.636071f, 3.31239f, 
		-1.25445f, 2.71960f, -1.25445f, 2.71960f, 
		-1.25445f, 2.71960f, 1.89203f, -5.00000f, 
		2.04708f, -4.94481f, -1.20199f, 3.16481f, 
		28.2412f, -20.9116f, 3.95481f, -4.06561f, 
		32.6960f, -20.9116f, 32.6960f, -20.9116f, 
		27.2181f, -20.9116f, 27.2181f, -20.9116f, 
		-3.95481f, -4.06561f, -1.20195f, 3.16483f, 
		-3.51286f, 4.92753f, -2.94957f, 4.92753f, 
		-3.51286f, 4.92753f, 59.1284f, -27.8151f, 
		0.974060f, 0.250000f, 0.377370f, 0.750000f, 
		-4.50229f, -0.250359f, 4.50239f, -0.250206f, 
		0.795584f, -27.8151f, 4.50234f, -0.250268f, 
		4.50485f, -1.50000f, 4.50485f, -1.53447f, 
		-4.50227f, -0.250469f, -4.50484f, -1.50000f, 
		-4.50484f, -1.53452f, 3.15631f, -0.899351f, 
		3.15627f, -0.899348f, 31.4681f, -21.3383f, 
		31.6057f, -21.0578f, 31.6057f, -21.0578f, 
		31.6057f, -21.0578f, 54.6044f, -27.4276f, 
		0.392565f, 0.666667f, 0.958866f, 0.166667f, 
		0.392565f, 0.833333f, 0.958865f, 0.333333f, 
		67.5278f, -20.8198f, 67.5278f, -20.8198f, 
		0.598212f, 0.833333f, 0.753218f, 0.333333f, 
		0.598212f, 0.666667f, 0.753218f, 0.166667f, 
		66.7990f, -22.0599f, 66.7990f, -22.0599f, 
		66.7990f, -19.5797f, 66.7990f, -19.5797f, 
		0.598212f, 0.666667f, 0.753218f, 0.166667f, 
		0.753218f, 0.333333f, 0.598212f, 0.833333f, 
		66.2058f, -20.8198f, 66.2058f, -20.8198f, 
		4.50484f, 1.50000f, 4.50484f, 1.53451f, 
		-0.0722322f, -1.24010f, -0.0722322f, -1.24010f, 
		-0.0722334f, 1.24009f, -0.0722334f, 1.24009f, 
		-4.50485f, 1.50000f, -4.50485f, 1.53445f, 
		-0.665369f, 0.000000f, -0.665369f, 0.000000f, 
		0.665371f, 0.000000f, 0.665371f, 0.000000f, 
		1.69212f, 2.84270f, 1.69212f, 2.84270f, 
		1.69212f, 2.84270f, -0.0722376f, -1.24009f, 
		-0.0722376f, -1.24009f, -0.0722392f, 1.24009f, 
		-0.0722392f, 1.24009f, -0.665365f, 0.000000f, 
		-0.665365f, 0.000000f, 0.665365f, 0.000000f, 
		0.665365f, 0.000000f, 1.69216f, 2.84254f, 
		1.69216f, 2.84254f, 1.69216f, 2.84254f, 
		0.218016f, 1.23355f, 0.218016f, 1.23355f, 
		67.1713f, -19.6014f, 67.1677f, -19.6003f, 
		0.218016f, -1.23355f, 0.218016f, -1.23355f, 
		0.218016f, 1.23355f, 0.218016f, 1.23355f, 
		67.1677f, -22.0393f, 67.1713f, -22.0381f, 
		32.0787f, -19.6994f, 32.0787f, -19.6994f, 
		32.0787f, -19.6994f, 32.0787f, -19.6994f, 
		31.9975f, -19.7141f, 31.9975f, -19.7141f, 
		31.9975f, -19.7141f, 31.9975f, -19.7141f, 
		31.9975f, -19.7141f, 0.369916f, 0.750000f, 
		0.981515f, 0.250000f, 32.0786f, -22.1238f, 
		32.0786f, -22.1238f, 32.0786f, -22.1238f, 
		32.0786f, -22.1238f, 32.0786f, -22.1238f, 
		31.9975f, -22.1091f, 31.9975f, -22.1091f, 
		31.9975f, -22.1091f, -2.02154f, -4.95719f, 
		-1.98000f, -5.00000f, 0.293091f, -1.22012f, 
		0.293091f, -1.22012f, 27.8276f, -22.1244f, 
		27.8355f, -22.1238f, 27.8355f, -22.1238f, 
		27.8355f, -22.1238f, 27.8355f, -22.1238f, 
		27.8355f, -22.1238f, 27.8355f, -22.1238f, 
		4.33547f, 2.97426f, 27.8276f, -19.6988f, 
		27.8354f, -19.6994f, 27.8354f, -19.6994f, 
		27.8354f, -19.6994f, 27.8354f, -19.6994f, 
		-4.33541f, 2.97444f, 1.69666f, 1.84271f, 
		4.33545f, -2.97430f, -4.33543f, -2.97436f, 
		-0.153413f, 3.84315f, -0.00679774f, 3.70871f, 
		29.9571f, -24.5358f, 29.9571f, -24.5358f, 
		29.9571f, -17.2874f, 29.9571f, -17.2874f, 
		43.1124f, -40.1476f, 43.1124f, -40.1476f, 
		16.8115f, -40.1476f, 16.8115f, -40.1476f, 
		30.3690f, -24.5083f, 30.3690f, -24.5083f, 
		30.3690f, -17.3149f, 30.3690f, -17.3149f, 
		29.5451f, -24.5083f, 29.5451f, -24.5083f, 
		29.5451f, -17.3149f, 29.5451f, -17.3149f, 
		32.3899f, -22.1087f, 32.3899f, -22.1087f, 
		32.3899f, -19.7144f, 32.3899f, -19.7144f, 
		27.5242f, -22.1087f, 27.5242f, -22.1087f, 
		66.4326f, -22.0049f, 66.4326f, -22.0049f, 
		66.4326f, -19.6347f, 66.4326f, -19.6347f, 
		-2.97267f, -4.50807f, -0.438558f, -1.18508f, 
		-0.438558f, -1.18508f, 0.438558f, 1.18508f, 
		0.438558f, 1.18508f, -0.438559f, 1.18508f, 
		-0.438559f, 1.18508f, 0.384920f, 0.833333f, 
		0.966510f, 0.333333f, 0.384921f, 0.666667f, 
		0.966510f, 0.166667f, -0.438555f, 1.18508f, 
		-0.438555f, 1.18508f, -0.438553f, -1.18508f, 
		-0.438553f, -1.18508f, 0.438566f, -1.18507f, 
		0.438566f, -1.18507f, 0.438562f, 1.18508f, 
		0.438562f, 1.18508f, -0.263860f, 1.03643f, 
		-0.431797f, 1.16803f, 27.4156f, -19.7358f, 
		27.4156f, -19.7358f, 0.362526f, 0.750000f, 
		0.988905f, 0.250000f, -0.153427f, 3.84315f, 
		-0.153427f, 3.84315f, 43.4783f, -37.5907f, 
		30.7125f, -24.4301f, 30.7125f, -24.4301f, 
		30.7125f, -17.3931f, 30.7125f, -17.3931f, 
		29.2016f, -24.4301f, 29.2016f, -24.4301f, 
		29.2016f, -17.3931f, 29.2016f, -17.3931f, 
		43.5079f, -40.1519f, 43.5079f, -40.1519f, 
		16.4456f, -37.5907f, -0.617753f, 3.84315f, 
		-0.553206f, 3.79986f, 16.4160f, -40.1519f, 
		16.4160f, -40.1519f, -2.97271f, 4.50804f, 
		-0.618044f, 3.84315f, -0.553552f, 3.79978f, 
		0.355141f, 0.750000f, 0.000000f, 0.250000f, 
		0.996289f, 0.250000f, 0.495526f, 0.916667f, 
		0.855905f, 0.416667f, 0.101929f, 0.250000f, 
		0.249501f, 0.750000f, 0.495526f, 0.583333f, 
		0.855905f, 0.0833330f, 1.69825f, 3.30792f, 
		1.69825f, 3.30792f, 1.69825f, 3.30792f, 
		1.18835f, 1.03643f, 1.14278f, 1.15408f, 
		1.14278f, 1.15408f, 1.14278f, 1.15408f, 
		0.495526f, 0.583333f, 0.855905f, 0.0833330f, 
		0.351430f, 0.750000f, 1.00000f, 0.250000f, 
		0.000000f, 0.250000f, 0.249501f, 0.750000f, 
		0.101929f, 0.250000f, 0.855905f, 0.416667f, 
		0.495526f, 0.916667f, 0.481869f, 0.916667f, 
		0.869561f, 0.416667f, 0.481869f, 0.583333f, 
		0.869561f, 0.0833330f, 0.473503f, 1.03812f, 
		0.473503f, 1.03812f, 0.481869f, 0.583333f, 
		0.869561f, 0.0833330f, 0.869561f, 0.416667f, 
		0.481869f, 0.916667f, -1.32181f, 1.03812f, 
		-1.27021f, 1.26450f, 43.8634f, -37.6738f, 
		1.37983f, 4.94546f, -3.79599f, 4.92753f, 
		-4.26225f, 4.92753f, -3.79599f, 4.92753f, 
		16.0605f, -37.6738f, 0.473245f, 1.03812f, 
		0.473245f, 1.03812f, 0.564118f, -1.13550f, 
		0.564118f, -1.13550f, -1.32201f, 1.03812f, 
		-1.27026f, 1.26451f, 3.79574f, 4.92753f, 
		4.26225f, 4.92753f, 3.79574f, 4.92753f, 
		30.9894f, -24.3073f, 30.9894f, -24.3073f, 
		30.9894f, -17.5159f, 30.9894f, -17.5159f, 
		28.9247f, -24.3073f, 28.9247f, -24.3073f, 
		28.9247f, -17.5159f, 28.9247f, -17.5159f, 
		-0.147966f, 5.00000f, -1.32432f, 4.95869f, 
		0.522040f, 0.916667f, 0.829390f, 0.416667f, 
		0.111579f, 0.250000f, 0.239851f, 0.750000f, 
		0.522040f, 0.583333f, 0.829390f, 0.0833330f, 
		0.468500f, 0.916667f, 0.882930f, 0.416667f, 
		0.468500f, 0.583333f, 0.882930f, 0.0833330f, 
		32.6960f, -22.0316f, 32.6960f, -22.0316f, 
		32.6960f, -19.7916f, 32.6960f, -19.7916f, 
		27.2181f, -22.0316f, 27.2181f, -22.0316f, 
		0.522040f, 0.583333f, 0.829390f, 0.0833330f, 
		0.239851f, 0.750000f, 0.111579f, 0.250000f, 
		0.829390f, 0.416667f, 0.522040f, 0.916667f, 
		0.468500f, 0.583333f, 0.882930f, 0.0833330f, 
		0.882930f, 0.416667f, 0.468500f, 0.916667f, 
		-1.07207f, 1.03643f, -1.09158f, 1.10443f, 
		0.377371f, 0.666667f, 0.974059f, 0.166667f, 
		0.377370f, 0.833333f, 0.974060f, 0.333333f, 
		28.4460f, -20.3420f, 28.3084f, -19.9655f, 
		5.51907f, 1.03827f, 5.51381f, 1.00119f, 
		-3.40882f, -0.960301f, -3.47261f, -0.901000f, 
		1.78392f, 5.00000f, 1.45896f, 4.95560f, 
		5.51907f, 1.03827f, 5.51380f, 1.00109f, 
		-1.50729f, -0.960301f, -3.41215f, -0.905264f, 
		69.1138f, -20.8198f, 69.1138f, -20.8198f, 
		69.1138f, -20.8198f, 69.1138f, -20.8198f, 
		69.1138f, -20.8198f, 69.1138f, -20.8198f, 
		69.1138f, -20.8198f, 69.1138f, -20.8198f, 
		69.1138f, -20.8198f, 69.1138f, -20.8198f, 
		69.1138f, -20.8198f, 69.1138f, -20.8198f, 
		69.1138f, -20.8198f, 69.1138f, -20.8198f, 
		32.9347f, -20.9116f, 32.9347f, -20.9116f, 
		26.9794f, -20.9116f, 26.9794f, -20.9116f, 
		0.455342f, 0.916667f, 0.896088f, 0.416667f, 
		0.455342f, 0.583333f, 0.896088f, 0.0833330f, 
		67.5341f, -19.7443f, 67.5293f, -19.7419f, 
		3.90872f, 3.84315f, 3.90872f, 3.84315f, 
		3.90872f, 4.92753f, 3.90872f, 4.92753f, 
		3.90872f, 4.92753f, 2.24263f, 0.000000f, 
		2.24263f, 0.000000f, 2.24263f, 0.000000f, 
		2.24263f, 0.000000f, 2.24263f, 0.000000f, 
		2.24263f, 0.000000f, 2.24263f, 0.000000f, 
		2.24263f, 0.000000f, 2.24263f, 0.000000f, 
		2.24263f, 0.000000f, 2.24263f, 0.000000f, 
		44.5206f, -35.0019f, 0.455342f, 0.583333f, 
		0.896088f, 0.0833330f, 67.5293f, -21.8977f, 
		67.5341f, -21.8953f, 2.00573f, 0.000000f, 
		2.00573f, 0.000000f, 2.00573f, 0.000000f, 
		2.00573f, 0.000000f, 2.00573f, 0.000000f, 
		2.00573f, 0.000000f, 2.00573f, 0.000000f, 
		2.00573f, 0.000000f, 2.00573f, 0.000000f, 
		2.00573f, 0.000000f, 2.00573f, 0.000000f, 
		2.00573f, 0.000000f, 2.00573f, 0.000000f, 
		2.00573f, 0.000000f, 0.896088f, 0.416667f, 
		0.455342f, 0.916667f, 15.4033f, -35.0019f, 
		-3.90872f, 3.84315f, -3.90872f, 3.84315f, 
		66.2058f, -19.7462f, 66.2058f, -19.7462f, 
		66.2058f, -21.8933f, 66.2058f, -21.8933f, 
		31.2016f, -24.1464f, 31.2016f, -24.1464f, 
		31.2016f, -17.6768f, 31.2016f, -17.6768f, 
		28.7125f, -24.1464f, 28.7125f, -24.1464f, 
		28.7125f, -17.6768f, 28.7125f, -17.6768f, 
		68.7953f, -20.8198f, 68.7953f, -20.8198f, 
		68.7953f, -20.8198f, 68.7953f, -20.8198f, 
		68.7953f, -20.8198f, 68.7953f, -20.8198f, 
		68.7953f, -20.8198f, 68.7953f, -20.8198f, 
		68.7953f, -20.8198f, 68.7953f, -20.8198f, 
		-0.665367f, -1.07355f, -0.665367f, -1.07355f, 
		0.665371f, 1.07355f, 0.665371f, 1.07355f, 
		-0.665370f, 1.07355f, -0.665370f, 1.07355f, 
		1.92413f, 0.000000f, 1.92413f, 0.000000f, 
		1.92413f, 0.000000f, 1.92413f, 0.000000f, 
		1.92413f, 0.000000f, 1.92413f, 0.000000f, 
		1.92413f, 0.000000f, 1.92413f, 0.000000f, 
		-0.665367f, -1.07355f, -0.665367f, -1.07355f, 
		-0.665365f, 1.07355f, -0.665365f, 1.07355f, 
		0.665372f, -1.07355f, 0.665372f, -1.07355f, 
		0.665372f, 1.07355f, 0.665372f, 1.07355f, 
		67.6485f, -20.8198f, 67.6485f, -20.8198f, 
		2.32424f, 0.000000f, 2.32424f, 0.000000f, 
		2.32424f, 0.000000f, 2.32424f, 0.000000f, 
		2.32424f, 0.000000f, 2.32424f, 0.000000f, 
		2.32424f, 0.000000f, 2.32424f, 0.000000f, 
		2.32424f, 0.000000f, 2.32424f, 0.000000f, 
		0.777304f, 0.000000f, 0.777304f, 0.000000f, 
		66.0937f, -20.5896f, 66.0827f, -19.9465f, 
		66.0937f, -20.3593f, 66.0827f, -19.9465f, 
		66.0827f, -20.8198f, 0.369916f, 0.833333f, 
		0.981514f, 0.333333f, 0.369916f, 0.666667f, 
		0.981515f, 0.166667f, 0.777305f, 0.000000f, 
		0.777305f, 0.000000f, -0.788457f, 0.873336f, 
		-0.788457f, 0.000000f, -0.777465f, 0.460513f, 
		-0.777465f, 0.230271f, -3.93460f, 3.84315f, 
		-3.93460f, 3.84315f, -0.788457f, 0.000000f, 
		-0.777463f, 0.230226f, -0.788457f, 0.873336f, 
		-0.777463f, 0.230226f, 44.8760f, -35.1636f, 
		62.8364f, -28.5940f, 15.0479f, -35.1636f, 
		3.93436f, 3.84315f, 3.93436f, 3.84315f, 
		3.70458f, 4.09251f, 27.0754f, -19.8704f, 
		27.0754f, -19.8704f, 31.4681f, -21.4935f, 
		31.5891f, -21.8107f, -2.91249f, -28.5940f, 
		28.4460f, -21.4936f, 28.3251f, -21.8107f, 
		-3.70460f, 4.09249f, 28.4460f, -22.3696f, 
		28.3043f, -22.1788f, 28.3043f, -22.1788f, 
		28.3043f, -22.1788f, 31.4681f, -19.9386f, 
		31.5349f, -19.8910f, 0.547731f, 0.916667f, 
		0.803700f, 0.416667f, 0.547731f, 0.583333f, 
		0.803700f, 0.0833330f, 0.442334f, 0.916667f, 
		0.909097f, 0.416667f, 0.442334f, 0.583333f, 
		0.909097f, 0.0833330f, 1.95785f, 1.48007f, 
		1.95785f, 1.48007f, 1.95785f, 1.48007f, 
		0.547731f, 0.583333f, 0.803700f, 0.0833330f, 
		0.803700f, 0.416667f, 0.547731f, 0.916667f, 
		0.909097f, 0.0833330f, 0.442334f, 0.583333f, 
		-5.22964f, 0.140807f, 3.70464f, -4.09245f, 
		0.909097f, 0.416667f, 0.442334f, 0.916667f, 
		-3.70459f, -4.09250f, 1.95787f, 1.48004f, 
		1.95787f, 1.48004f, 1.95787f, 1.48004f, 
		5.22947f, 0.140508f, 62.6670f, -28.9471f, 
		0.623510f, 0.750000f, 0.727920f, 0.250000f, 
		28.4460f, -19.6523f, 28.3251f, -19.7350f, 
		28.3251f, -19.7350f, 28.3251f, -19.7350f, 
		0.362526f, 0.666667f, 0.988905f, 0.166667f, 
		0.362525f, 0.833333f, 0.988906f, 0.333333f, 
		31.3509f, -23.9534f, 31.3509f, -23.9534f, 
		31.3509f, -17.8698f, 31.3509f, -17.8698f, 
		28.5632f, -23.9534f, 28.5632f, -23.9534f, 
		28.5632f, -17.8698f, 28.5632f, -17.8698f, 
		-2.74309f, -28.9471f, 0.623510f, 0.750000f, 
		0.727920f, 0.250000f, 0.355140f, 0.833333f, 
		0.000000f, 0.333333f, 0.996290f, 0.333333f, 
		0.000000f, 0.166667f, 0.355140f, 0.666667f, 
		0.996290f, 0.166667f, 0.101929f, 0.333333f, 
		0.249501f, 0.833333f, 0.101929f, 0.166667f, 
		0.249501f, 0.666667f, 1.48428f, 1.03643f, 
		1.48428f, 1.03643f, -1.95789f, -5.00000f, 
		-1.58355f, -4.95864f, -0.990000f, -5.00000f, 
		0.351430f, 0.666667f, 1.00000f, 0.166667f, 
		0.000000f, 0.166667f, 0.249501f, 0.666667f, 
		0.101929f, 0.166667f, 0.351430f, 0.833333f, 
		1.00000f, 0.333333f, 0.000000f, 0.333333f, 
		0.246248f, 0.833333f, 0.105182f, 0.333333f, 
		0.124049f, 0.250000f, 0.227381f, 0.750000f, 
		0.227381f, 0.750000f, 0.124049f, 0.250000f, 
		-2.44690f, 2.01809f, -2.44690f, 2.01809f, 
		-2.44690f, 2.01809f, -2.44690f, 2.01809f, 
		-2.44690f, 2.01809f, -2.44697f, 2.01811f, 
		-2.44697f, 2.01811f, -2.44697f, 2.01811f, 
		-2.44697f, 2.01811f, -2.44697f, 2.01811f, 
		0.111579f, 0.333333f, 0.239851f, 0.833333f, 
		0.111579f, 0.166667f, 0.239851f, 0.666667f, 
		1.95239f, 3.84315f, 2.00121f, 3.62618f, 
		2.00121f, 3.62618f, 2.00121f, 3.62618f, 
		0.757398f, -0.957497f, 0.757398f, -0.957497f, 
		0.429425f, 0.916667f, 0.922005f, 0.416667f, 
		0.429425f, 0.583333f, 0.922005f, 0.0833330f, 
		0.239851f, 0.666667f, 0.111579f, 0.166667f, 
		4.90957f, -0.250197f, 0.922005f, 0.0833330f, 
		0.429425f, 0.583333f, 0.429425f, 0.916667f, 
		0.922005f, 0.416667f, -4.90969f, -0.250170f, 
		2.43126f, 1.75380f, 2.43126f, 1.75380f, 
		2.43126f, 1.75380f, 2.43126f, 1.75380f, 
		2.43126f, 1.75380f, 5.58195f, -27.8151f, 
		68.6029f, -20.8198f, 68.6029f, -20.8198f, 
		68.6029f, -20.8198f, 68.6029f, -20.8198f, 
		68.6029f, -20.8198f, 68.6029f, -20.8198f, 
		69.3909f, -20.8198f, 69.4263f, -20.8198f, 
		69.3909f, -20.8198f, 69.4263f, -20.8198f, 
		69.3909f, -20.8198f, 69.3909f, -20.8198f, 
		69.3909f, -20.8198f, 1.73171f, 0.000000f, 
		1.73171f, 0.000000f, 1.73171f, 0.000000f, 
		1.73171f, 0.000000f, -2.06285f, -0.960301f, 
		-3.65890f, -0.915680f, 2.51971f, 0.000000f, 
		2.55514f, 0.000000f, 2.51971f, 0.000000f, 
		2.55514f, 0.000000f, 2.51971f, 0.000000f, 
		2.51971f, 0.000000f, 68.8769f, -19.8849f, 
		68.8769f, -19.8849f, 68.8769f, -19.8849f, 
		68.8769f, -19.8849f, 68.8769f, -19.8849f, 
		68.8769f, -19.8849f, 68.8769f, -19.8849f, 
		68.8769f, -19.8849f, 68.8769f, -19.8849f, 
		68.8769f, -19.8849f, 68.8769f, -19.8849f, 
		68.8769f, -19.8849f, 68.8769f, -19.8849f, 
		68.8769f, -19.8849f, 68.8769f, -21.7547f, 
		68.8769f, -21.7547f, 68.8769f, -21.7547f, 
		68.8769f, -21.7547f, 68.8769f, -21.7547f, 
		68.8769f, -21.7547f, 68.8769f, -21.7547f, 
		68.8769f, -21.7547f, 68.8769f, -21.7547f, 
		68.8769f, -21.7547f, 68.8769f, -21.7547f, 
		68.8769f, -21.7547f, 68.8769f, -21.7547f, 
		32.9347f, -19.9688f, 32.9347f, -19.9688f, 
		32.9347f, -21.8544f, 32.9347f, -21.8544f, 
		1.73171f, 0.000000f, 1.73171f, 0.000000f, 
		1.73171f, 0.000000f, 1.73171f, 0.000000f, 
		1.73171f, 0.000000f, 1.73171f, 0.000000f, 
		1.73172f, 0.000000f, 1.73172f, 0.000000f, 
		1.73172f, 0.000000f, 1.73172f, 0.000000f, 
		1.73172f, 0.000000f, 1.73172f, 0.000000f, 
		26.9794f, -21.8544f, 26.9794f, -21.8544f, 
		46.1558f, -32.5361f, 2.51972f, 0.000000f, 
		2.55514f, 0.000000f, 2.51972f, 0.000000f, 
		2.55514f, 0.000000f, 2.51972f, 0.000000f, 
		2.51972f, 0.000000f, 31.4391f, -23.7347f, 
		31.4391f, -23.7347f, 31.4391f, -18.0885f, 
		31.4391f, -18.0885f, 28.4750f, -23.7347f, 
		28.4750f, -23.7347f, 28.4750f, -18.0885f, 
		28.4750f, -18.0885f, 2.24262f, -0.934882f, 
		2.24262f, -0.934882f, 2.24262f, -0.934882f, 
		2.24262f, -0.934882f, 2.24262f, -0.934882f, 
		2.24262f, -0.934882f, 2.24262f, -0.934882f, 
		2.24262f, -0.934882f, 2.24262f, -0.934882f, 
		2.24262f, -0.934882f, 2.24262f, -0.934882f, 
		2.00575f, 0.934884f, 2.00575f, 0.934884f, 
		2.00575f, 0.934884f, 2.00575f, 0.934884f, 
		2.00575f, 0.934884f, 2.00575f, 0.934884f, 
		2.00575f, 0.934884f, 2.00575f, 0.934884f, 
		2.00575f, 0.934884f, 2.00575f, 0.934884f, 
		2.00575f, 0.934884f, 2.00575f, 0.934884f, 
		2.00575f, 0.934884f, 2.00575f, 0.934884f, 
		13.7681f, -32.5361f, 5.25565f, -1.50000f, 
		5.25565f, -1.50000f, 5.25565f, 1.50000f, 
		5.25565f, 1.50000f, 2.24263f, -0.934882f, 
		2.24263f, -0.934882f, 2.24263f, -0.934882f, 
		2.24263f, -0.934882f, 2.24263f, -0.934882f, 
		2.24263f, -0.934882f, 2.24263f, -0.934882f, 
		2.24263f, -0.934882f, 2.24263f, -0.934882f, 
		2.24263f, -0.934882f, 2.24263f, -0.934882f, 
		2.24263f, -0.934882f, 2.24263f, -0.934882f, 
		2.24263f, -0.934882f, 2.24264f, 0.934880f, 
		2.24264f, 0.934880f, 2.24264f, 0.934880f, 
		2.24264f, 0.934880f, 2.24264f, 0.934880f, 
		2.24264f, 0.934880f, 2.24264f, 0.934880f, 
		2.24264f, 0.934880f, 2.24264f, 0.934880f, 
		2.24264f, 0.934880f, 2.24264f, 0.934880f, 
		2.24264f, 0.934880f, 2.24264f, 0.934880f, 
		2.24264f, 0.934880f, -5.25565f, -1.50000f, 
		-5.25565f, -1.50000f, -5.25565f, 1.50000f, 
		-5.25565f, 1.50000f, 69.1954f, -19.8940f, 
		69.1954f, -19.8940f, 69.1954f, -19.8940f, 
		69.1954f, -19.8940f, 69.1954f, -19.8940f, 
		69.1954f, -19.8940f, 69.1954f, -19.8940f, 
		69.1954f, -19.8940f, 69.1954f, -19.8940f, 
		69.1954f, -19.8940f, 69.1954f, -21.7456f, 
		69.1954f, -21.7456f, 69.1954f, -21.7456f, 
		69.1954f, -21.7456f, 69.1954f, -21.7456f, 
		69.1954f, -21.7456f, 1.92412f, -0.925816f, 
		1.92412f, -0.925816f, 1.92412f, -0.925816f, 
		1.92412f, -0.925817f, 1.92412f, -0.925816f, 
		1.92412f, -0.925817f, 1.92412f, -0.925817f, 
		1.92412f, -0.925817f, 2.32423f, 0.925819f, 
		2.32423f, 0.925819f, 2.32423f, 0.925819f, 
		2.32423f, 0.925819f, 2.32423f, 0.925819f, 
		2.32423f, 0.925820f, 2.32423f, 0.925820f, 
		2.32423f, 0.925820f, 2.32423f, 0.925820f, 
		2.32423f, 0.925820f, 67.6485f, -19.8981f, 
		67.6485f, -19.8981f, 67.6485f, -21.7414f, 
		67.6485f, -21.7414f, 1.92414f, -0.925822f, 
		1.92414f, -0.925822f, 1.92414f, -0.925822f, 
		1.92414f, -0.925822f, 1.92414f, -0.925822f, 
		1.92414f, -0.925822f, 1.92414f, -0.925822f, 
		1.92414f, -0.925822f, 1.92414f, -0.925822f, 
		1.92414f, -0.925822f, 1.92413f, 0.925819f, 
		1.92413f, 0.925819f, 1.92413f, 0.925819f, 
		1.92413f, 0.925819f, 1.92413f, 0.925819f, 
		1.92413f, 0.925819f, 1.92413f, 0.925819f, 
		1.92413f, 0.925819f, 1.92413f, 0.925819f, 
		1.92413f, 0.925819f, 0.233585f, 0.833333f, 
		0.117845f, 0.333333f, 0.777304f, 0.921669f, 
		0.777304f, 0.921669f, -0.777465f, -0.920969f, 
		-0.788457f, -0.873336f, -0.788457f, -0.873336f, 
		-0.777465f, -0.920969f, -1.95789f, 5.00000f, 
		0.147965f, 5.00000f, -1.74993f, 4.94160f, 
		0.777304f, 0.921669f, 0.777304f, 0.921669f, 
		0.777305f, -0.921665f, 0.777305f, -0.921665f, 
		-0.777464f, -0.920976f, -0.788457f, -0.873336f, 
		-0.777464f, -0.920976f, 46.4647f, -32.7720f, 
		13.4592f, -32.7720f, 2.06285f, -0.960301f, 
		3.60528f, -0.931552f, -2.31947f, 3.33060f, 
		-2.31947f, 3.33060f, -2.31947f, 3.33060f, 
		0.573028f, 0.916667f, 0.778402f, 0.416667f, 
		0.573028f, 0.583333f, 0.778402f, 0.0833330f, 
		0.573028f, 0.583333f, 0.778402f, 0.0833330f, 
		0.778402f, 0.416667f, 0.573028f, 0.916667f, 
		0.416577f, 0.916667f, 0.934854f, 0.416667f, 
		0.416577f, 0.583333f, 0.934854f, 0.0833330f, 
		-1.67490f, 1.03643f, -1.96381f, 1.23965f, 
		-1.96381f, 1.23965f, -1.96381f, 1.23965f, 
		0.416577f, 0.583333f, 0.934854f, 0.0833330f, 
		0.416577f, 0.916667f, 0.934854f, 0.416667f, 
		4.16126f, 3.84315f, 4.16126f, 3.84315f, 
		0.623510f, 0.833333f, 0.727920f, 0.333333f, 
		0.623510f, 0.666667f, 0.727920f, 0.166667f, 
		-1.53371f, 3.84315f, -2.00594f, 3.64357f, 
		0.623510f, 0.666667f, 0.727920f, 0.166667f, 
		0.727920f, 0.333333f, 0.623510f, 0.833333f, 
		-1.53365f, 3.84315f, -2.00598f, 3.64358f, 
		4.77786f, 1.50000f, 4.77786f, 1.53452f, 
		66.9570f, -20.9116f, 66.9570f, -20.9116f, 
		-4.77786f, 1.50000f, -4.77786f, 1.53449f, 
		0.124049f, 0.333333f, 0.227381f, 0.833333f, 
		0.124049f, 0.166667f, 0.227381f, 0.666667f, 
		5.43857f, -1.50000f, 5.43857f, -1.50000f, 
		5.43857f, 1.50000f, 5.43857f, 1.50000f, 
		5.43857f, 1.50000f, 31.4681f, -23.4966f, 
		31.4681f, -23.4966f, 31.4681f, -18.3266f, 
		31.4681f, -18.3266f, -7.04295f, -20.9116f, 
		-7.04295f, -20.9116f, 28.4460f, -23.4966f, 
		28.4460f, -23.4966f, 28.4460f, -18.3266f, 
		28.4460f, -18.3266f, 2.92100f, -4.69278f, 
		-5.43857f, -1.50000f, -5.43857f, -1.50000f, 
		-5.43857f, 1.50000f, -5.43857f, 1.50000f, 
		-5.43857f, 1.50000f, 0.227382f, 0.666667f, 
		0.124048f, 0.166667f, 1.50145f, 3.84315f, 
		1.95771f, 3.64137f, -2.48056f, 3.84315f, 
		-2.19450f, 3.56772f, 1.95221f, 3.84315f, 
		1.96531f, 3.83262f, 26.8986f, -21.2558f, 
		26.8886f, -20.9116f, 26.8986f, -21.2558f, 
		26.8886f, -20.2986f, 1.73302f, -4.96294f, 
		1.89203f, -5.00000f, 4.77786f, -1.50000f, 
		4.77786f, -1.53447f, -4.77786f, -1.50000f, 
		-4.77786f, -1.53444f, 4.26225f, 3.84315f, 
		4.26225f, 3.84315f, 4.26225f, 4.92753f, 
		4.26225f, 4.92753f, 4.59817f, -2.97440f, 
		-4.59813f, -2.97452f, 68.6029f, -20.0016f, 
		68.6029f, -20.0016f, 68.6029f, -20.0016f, 
		68.6029f, -20.0016f, 68.6029f, -20.0016f, 
		68.6029f, -20.0016f, 68.6029f, -21.6380f, 
		68.6029f, -21.6380f, 68.6029f, -21.6380f, 
		68.6029f, -21.6380f, 68.6029f, -21.6380f, 
		68.6029f, -21.6380f, -2.86105f, 2.41284f, 
		69.3909f, -20.0059f, 69.4263f, -20.0716f, 
		69.3909f, -20.0059f, 69.3909f, -20.0059f, 
		69.3909f, -20.0059f, 4.59816f, 2.97444f, 
		-4.59812f, 2.97456f, 1.73172f, -0.818195f, 
		1.73172f, -0.818195f, 1.73172f, -0.818195f, 
		1.73172f, -0.818195f, 1.73171f, 0.818193f, 
		1.73171f, 0.818193f, 1.73171f, 0.818193f, 
		1.73171f, 0.818193f, 1.73171f, 0.818193f, 
		1.73171f, 0.818193f, 2.84693f, 2.43434f, 
		2.51972f, -0.813868f, 2.55514f, -0.748168f, 
		2.51972f, -0.813868f, 2.55514f, -0.748168f, 
		2.51972f, -0.813868f, 2.51972f, -0.813868f, 
		2.51971f, 0.813870f, 2.55514f, 0.748168f, 
		2.51971f, 0.813870f, 2.55514f, 0.748168f, 
		2.51971f, 0.813870f, 2.51971f, 0.813870f, 
		-2.86127f, 2.41329f, 1.73171f, -0.818194f, 
		1.73171f, -0.818194f, 1.73171f, -0.818194f, 
		1.73171f, -0.818194f, 1.73171f, -0.818194f, 
		1.73171f, -0.818194f, 1.73171f, 0.818195f, 
		1.73171f, 0.818195f, 1.73171f, 0.818195f, 
		1.73171f, 0.818195f, 1.73171f, 0.818195f, 
		1.73171f, 0.818195f, 4.94415f, -1.50000f, 
		4.94415f, -1.55667f, 2.84691f, 2.43443f, 
		2.84691f, 2.43443f, 2.84691f, 2.43443f, 
		2.51971f, -0.813877f, 2.55514f, -0.748168f, 
		2.51971f, -0.813877f, 2.55514f, -0.748168f, 
		2.51971f, -0.813877f, 2.51971f, -0.813877f, 
		2.51972f, 0.813874f, 2.55514f, 0.748168f, 
		2.51972f, 0.813874f, 2.55514f, 0.748168f, 
		2.51972f, 0.813874f, 2.51972f, 0.813874f, 
		-4.94415f, -1.50000f, -4.94415f, -1.55665f, 
		33.0187f, -20.9116f, 33.0255f, -20.9116f, 
		33.0187f, -20.9116f, 68.2613f, -20.6476f, 
		68.2613f, -20.6476f, 68.2613f, -20.6476f, 
		68.2613f, -20.6476f, 68.2613f, -20.6476f, 
		68.2613f, -20.6476f, 68.2613f, -20.6476f, 
		68.2613f, -20.4753f, 68.2613f, -20.4753f, 
		68.2613f, -20.4753f, 68.2613f, -20.4753f, 
		68.2613f, -20.4753f, 68.2613f, -20.4753f, 
		68.2613f, -20.4753f, 0.403754f, 0.583333f, 
		0.947676f, 0.0833330f, 0.403754f, 0.916667f, 
		0.947676f, 0.416667f, 67.8985f, -20.8198f, 
		67.8985f, -20.8198f, 67.8985f, -20.8198f, 
		67.8985f, -20.8198f, 67.8985f, -20.8198f, 
		67.8985f, -20.8198f, 1.39011f, -0.516991f, 
		1.39011f, -0.516991f, 1.39011f, -0.516991f, 
		1.39011f, -0.516991f, 1.39011f, -0.516991f, 
		1.39010f, -0.344666f, 1.39010f, -0.344666f, 
		1.39010f, -0.344666f, 1.39010f, -0.344666f, 
		1.39010f, -0.344666f, 1.39010f, -0.344666f, 
		4.26803f, 4.92753f, 3.90872f, 4.92753f, 
		4.52057f, 4.92753f, 4.26803f, 4.92753f, 
		1.02728f, 0.000000f, 1.02728f, 0.000000f, 
		1.02728f, 0.000000f, 1.02728f, 0.000000f, 
		1.02728f, 0.000000f, 1.02728f, 0.000000f, 
		-4.26805f, 4.92753f, -4.52057f, 4.92753f, 
		-3.90872f, 4.92753f, -4.26805f, 4.92753f, 
		1.39011f, 0.344619f, 1.39011f, 0.344619f, 
		1.39011f, 0.344619f, 1.39011f, 0.344619f, 
		1.39011f, 0.344619f, 1.39011f, 0.344619f, 
		1.39011f, 0.344619f, 1.39011f, 0.344619f, 
		1.39011f, 0.344619f, 1.39011f, 0.344619f, 
		1.39011f, 0.344619f, 1.02728f, 0.000000f, 
		1.02728f, 0.000000f, 1.02728f, 0.000000f, 
		1.02728f, 0.000000f, 1.02728f, 0.000000f, 
		1.02728f, 0.000000f, 2.72120f, 1.68789f, 
		2.72120f, 1.68789f, 2.72120f, 1.68789f, 
		2.72120f, 1.68789f, 2.72120f, 1.68789f, 
		0.400278f, 0.916667f, 0.951152f, 0.416667f, 
		54.1506f, -27.8151f, 0.400278f, 0.583333f, 
		0.951152f, 0.0833330f, 69.3892f, -21.8055f, 
		69.4263f, -21.5680f, 69.3892f, -21.8055f, 
		69.4263f, -22.1157f, 69.3892f, -21.8055f, 
		69.4263f, -21.5680f, 69.4263f, -22.1157f, 
		69.3891f, -21.7421f, 69.3891f, -21.7421f, 
		69.3891f, -21.7421f, 66.0827f, -22.3324f, 
		66.0937f, -22.2465f, 66.0827f, -21.6931f, 
		66.0937f, -22.0779f, -4.28872f, -0.748282f, 
		-3.02436f, -0.960301f, -4.28872f, -0.748282f, 
		-4.28872f, -0.748282f, -5.77924f, 1.03827f, 
		-5.77374f, 1.00124f, 0.221184f, 0.833333f, 
		0.130246f, 0.333333f, 4.86521f, -2.67198f, 
		-4.86521f, -2.67198f, -5.77924f, 1.03827f, 
		-5.77373f, 1.00120f, 48.3004f, -30.3485f, 
		66.9570f, -21.6616f, 66.9570f, -21.6616f, 
		66.9570f, -20.1616f, 66.9570f, -20.1616f, 
		11.6235f, -30.3485f, 4.52448f, -0.707067f, 
		26.8886f, -19.8498f, 26.9208f, -20.0639f, 
		26.9208f, -20.0639f, -7.04295f, -21.6616f, 
		-7.04295f, -21.6616f, -7.04295f, -20.1616f, 
		-7.04295f, -20.1616f, 59.8696f, -27.4276f, 
		0.213325f, 0.771266f, 0.146977f, 0.333333f, 
		0.138105f, 0.271266f, 4.64037f, -3.30746f, 
		0.0543597f, -27.4276f, -4.64037f, -3.30746f, 
		48.5485f, -30.6496f, 11.3754f, -30.6496f, 
		-2.35297f, 1.03643f, -2.48428f, 1.33412f, 
		0.204453f, 0.750000f, 0.204453f, 0.666667f, 
		0.214309f, 0.747514f, 0.146977f, 0.166667f, 
		0.139056f, 0.239824f, 0.598212f, 0.916667f, 
		0.753218f, 0.416667f, -5.85356f, 1.03827f, 
		-5.84799f, 1.00122f, 0.598212f, 0.583333f, 
		0.753218f, 0.0833330f, 0.392565f, 0.583333f, 
		0.958865f, 0.0833330f, 5.85356f, 1.03827f, 
		5.84799f, 1.00122f, 0.392565f, 0.916667f, 
		0.958865f, 0.416667f, -1.89203f, -5.00000f, 
		-1.93113f, -4.95865f, 66.7990f, -22.9677f, 
		66.7990f, -22.9677f, 66.7990f, -18.6719f, 
		66.7990f, -18.6719f, 0.598212f, 0.583333f, 
		0.753218f, 0.0833330f, 5.85356f, 1.03827f, 
		5.84798f, 1.00119f, 0.753218f, 0.416667f, 
		0.598212f, 0.916667f, -0.0722334f, -2.14791f, 
		-0.0722334f, -2.14791f, -5.85356f, 1.03827f, 
		-5.84798f, 1.00115f, -0.0722322f, 2.14791f, 
		-0.0722322f, 2.14791f, -0.0722398f, -2.14791f, 
		-0.0722398f, -2.14791f, -0.0722406f, 2.14791f, 
		-0.0722406f, 2.14791f, 1.92547f, 3.84315f, 
		2.27457f, 3.60880f, 2.27457f, 3.60880f, 
		2.27457f, 3.60880f, 3.92898f, -4.09263f, 
		0.218016f, 2.13658f, 0.218016f, 2.13658f, 
		-3.92904f, -4.09257f, 67.1643f, -22.9331f, 
		67.1643f, -22.9331f, 67.1643f, -18.7065f, 
		67.1643f, -18.7065f, 68.5089f, -20.8198f, 
		68.5089f, -20.8198f, 68.5644f, -20.8198f, 
		68.5089f, -20.8198f, 68.5644f, -20.8198f, 
		0.218016f, -2.13658f, 0.218016f, -2.13658f, 
		0.218016f, 2.13658f, 0.218016f, 2.13658f, 
		0.293084f, -2.11332f, 0.293084f, -2.11332f, 
		32.0769f, -23.0109f, 32.0769f, -23.0109f, 
		32.0769f, -23.0109f, 32.0769f, -23.0109f, 
		32.0769f, -23.0109f, 32.0409f, -23.0015f, 
		32.0769f, -23.0109f, 32.0409f, -23.0015f, 
		32.0409f, -23.0015f, 27.8732f, -23.0015f, 
		27.8732f, -23.0015f, 27.8373f, -23.0109f, 
		27.8373f, -23.0109f, 27.8732f, -23.0015f, 
		27.8373f, -23.0109f, 27.8373f, -23.0109f, 
		27.8373f, -23.0109f, 27.8276f, -18.8109f, 
		27.8276f, -18.8109f, 27.9487f, -18.8571f, 
		27.9487f, -18.8571f, 27.9487f, -18.8571f, 
		27.9487f, -18.8571f, 27.9487f, -18.8571f, 
		27.8276f, -18.8109f, 32.0787f, -18.8121f, 
		32.0787f, -18.8121f, 32.0787f, -18.8121f, 
		32.0865f, -18.8109f, 68.2613f, -21.5091f, 
		68.2613f, -21.5091f, 68.2613f, -21.5091f, 
		68.2613f, -21.5091f, 68.2613f, -21.5091f, 
		68.2613f, -21.5091f, 68.2613f, -21.5091f, 
		68.2613f, -21.5091f, 1.63774f, 0.000000f, 
		1.63774f, 0.000000f, 1.69322f, 0.000000f, 
		1.63774f, 0.000000f, 1.69322f, 0.000000f, 
		1.63769f, -0.300308f, 1.63769f, -0.300308f, 
		1.63769f, -0.300308f, 1.69322f, -0.276719f, 
		1.69322f, -0.276719f, 3.92899f, 4.09263f, 
		-3.92905f, 4.09256f, 1.39012f, 0.689299f, 
		1.39012f, 0.689299f, 1.39012f, 0.689299f, 
		1.39012f, 0.689299f, 1.39011f, 0.689299f, 
		1.39011f, 0.689299f, 1.39011f, 0.689299f, 
		32.3899f, -22.9851f, 32.3899f, -22.9851f, 
		32.3899f, -18.8381f, 32.3899f, -18.8381f, 
		67.8985f, -20.1423f, 67.8985f, -20.1423f, 
		67.8985f, -20.1423f, 67.8985f, -20.1423f, 
		67.8985f, -20.1423f, 67.8985f, -20.1423f, 
		67.8985f, -21.4972f, 67.8985f, -21.4972f, 
		67.8985f, -21.4972f, 67.8985f, -21.4972f, 
		67.8985f, -21.4972f, 67.8985f, -21.4972f, 
		67.8985f, -21.4972f, 67.8985f, -21.4972f, 
		67.8985f, -21.4972f, 67.8985f, -21.4972f, 
		27.5242f, -22.9851f, 27.5242f, -22.9851f, 
		27.5242f, -18.8381f, 27.5242f, -18.8381f, 
		66.4326f, -18.7672f, 66.4326f, -18.7672f, 
		66.4326f, -22.8724f, 66.4326f, -22.8724f, 
		1.39011f, -0.689299f, 1.39011f, -0.689299f, 
		1.39011f, -0.689299f, 1.39011f, -0.689299f, 
		1.39011f, -0.689299f, 1.39011f, -0.689299f, 
		1.39011f, -0.689299f, 1.39011f, -0.689299f, 
		1.02729f, -0.677446f, 1.02729f, -0.677446f, 
		1.02729f, -0.677446f, 1.02729f, -0.677446f, 
		1.02729f, -0.677446f, 1.02729f, -0.677446f, 
		1.02730f, 0.677448f, 1.02730f, 0.677448f, 
		1.02730f, 0.677448f, 1.02730f, 0.677448f, 
		1.02730f, 0.677448f, 1.02731f, 0.677449f, 
		1.02731f, 0.677449f, 1.02731f, 0.677449f, 
		1.02731f, 0.677449f, 1.02731f, 0.677449f, 
		-0.438559f, -2.05261f, -0.438559f, -2.05261f, 
		0.384920f, 0.916667f, 0.966510f, 0.416667f, 
		0.438557f, 2.05261f, 0.438557f, 2.05261f, 
		-0.438557f, 2.05261f, -0.438557f, 2.05261f, 
		0.384921f, 0.583333f, 0.966510f, 0.0833330f, 
		1.02731f, -0.677451f, 1.02731f, -0.677451f, 
		1.02731f, -0.677451f, 1.02731f, -0.677451f, 
		1.02731f, -0.677451f, 1.02731f, -0.677451f, 
		1.02731f, -0.677451f, 1.02731f, -0.677451f, 
		1.02731f, -0.677451f, 1.02731f, -0.677451f, 
		1.02733f, 0.677453f, 1.02733f, 0.677453f, 
		1.02733f, 0.677453f, 1.02733f, 0.677453f, 
		1.02733f, 0.677453f, 1.02733f, 0.677453f, 
		-0.438556f, 2.05261f, -0.438556f, 2.05261f, 
		-0.438555f, -2.05261f, -0.438555f, -2.05261f, 
		0.438561f, -2.05261f, 0.438561f, -2.05261f, 
		0.438561f, 2.05261f, 0.438561f, 2.05261f, 
		-5.20716f, -0.250196f, -5.20716f, -0.250196f, 
		-5.20716f, -0.250196f, 5.20721f, -0.250180f, 
		67.6587f, -20.5327f, 67.6754f, -20.5254f, 
		67.6754f, -20.5254f, 67.6587f, -20.5327f, 
		67.6754f, -20.5254f, 67.6754f, -20.5254f, 
		67.6587f, -20.5327f, 5.20705f, -0.250376f, 
		-5.20708f, -0.250337f, 0.804193f, -0.294376f, 
		0.804193f, -0.294376f, 0.787533f, -0.287083f, 
		0.804198f, -0.294377f, 0.787537f, -0.287085f, 
		0.804198f, -0.294377f, 0.787533f, 0.287118f, 
		0.804222f, 0.294433f, 0.804222f, 0.294433f, 
		0.804222f, 0.294433f, 0.804222f, 0.294433f, 
		0.787537f, 0.287120f, 31.4681f, -19.5044f, 
		31.6079f, -19.1219f, 0.569196f, -1.96126f, 
		0.567011f, -1.96424f, 32.6960f, -22.8515f, 
		32.6960f, -22.8515f, 32.6960f, -18.9717f, 
		32.6960f, -18.9717f, 27.2181f, -18.9717f, 
		27.2181f, -18.9717f, 27.2181f, -22.8515f, 
		27.2181f, -22.8515f, -0.895523f, 0.270106f, 
		-0.895523f, 0.270106f, -0.895523f, 0.270106f, 
		-0.895523f, 0.270106f, -0.895523f, 0.270106f, 
		-0.895496f, 0.270110f, -0.895496f, 0.270110f, 
		-0.895496f, 0.270110f, -0.895496f, 0.270110f, 
		-0.895496f, 0.270110f, -0.895496f, 0.270110f, 
		-0.895496f, 0.270110f, -0.963847f, 0.265417f, 
		-0.963847f, 0.265417f, -0.963847f, 0.265417f, 
		0.377371f, 0.583333f, 0.974059f, 0.0833330f, 
		0.377371f, 0.916667f, 0.974059f, 0.416667f, 
		4.09252f, -4.06554f, -4.09246f, -4.06561f, 
		67.4518f, -22.7722f, 67.4518f, -22.7722f, 
		67.4518f, -18.8673f, 67.4518f, -18.8673f, 
		-0.787255f, 0.262203f, -0.787255f, 0.262203f, 
		-0.787255f, 0.262203f, 66.2058f, -22.6792f, 
		66.2058f, -22.6792f, 66.2058f, -18.9603f, 
		66.2058f, -18.9603f, 53.9627f, -27.4276f, 
		28.4460f, -19.1865f, 28.3486f, -19.0562f, 
		2.93404f, 3.21763f, 2.93404f, 3.21763f, 
		2.93404f, 3.21763f, -0.665370f, -1.85944f, 
		-0.665370f, -1.85944f, -0.665364f, 1.85945f, 
		-0.665364f, 1.85945f, 0.665365f, 1.85945f, 
		0.665365f, 1.85945f, -4.43384f, 4.92753f, 
		-4.26225f, 4.92753f, -4.43384f, 4.92753f, 
		5.96123f, -27.4276f, -0.665367f, 1.85945f, 
		-0.665367f, 1.85945f, -0.665368f, -1.85944f, 
		-0.665368f, -1.85944f, 0.665375f, -1.85943f, 
		0.665375f, -1.85943f, 2.93407f, 3.21740f, 
		0.665374f, 1.85944f, 0.665374f, 1.85944f, 
		0.369916f, 0.916667f, 0.981515f, 0.416667f, 
		0.369916f, 0.583333f, 0.981515f, 0.0833330f, 
		2.73932f, 4.80775f, -1.10069f, 0.000000f, 
		-1.10069f, 0.000000f, -1.10069f, 0.000000f, 
		-1.10069f, 0.000000f, -0.699331f, 0.0927854f, 
		-0.699331f, 0.0927854f, -0.699331f, 0.0927854f, 
		-0.699331f, 0.0927854f, -0.697608f, -0.103953f, 
		-0.697608f, -0.103953f, -0.697620f, -0.103908f, 
		-0.697620f, -0.103908f, -0.697620f, -0.103908f, 
		-0.697620f, -0.103908f, -0.697563f, -0.103876f, 
		-0.697563f, -0.103876f, -1.10074f, 0.000000f, 
		-1.10074f, 0.000000f, -1.10074f, 0.000000f, 
		-1.10074f, 0.000000f, -1.10074f, 0.000000f, 
		-0.733869f, 0.219981f, -0.733869f, 0.219981f, 
		-0.733869f, 0.219981f, -0.733869f, 0.219981f, 
		-0.733869f, 0.219981f, 0.126176f, -0.0358221f, 
		0.126176f, -0.0358221f, 0.126176f, -0.0358221f, 
		0.126176f, -0.0358221f, 0.126303f, -0.0472450f, 
		0.126303f, -0.0472450f, 0.126303f, -0.0472450f, 
		0.126303f, -0.0472450f, 3.10782f, 4.69278f, 
		67.6587f, -21.3940f, 67.6754f, -21.4086f, 
		67.6754f, -21.4086f, 67.6754f, -21.4086f, 
		67.6587f, -21.3940f, 0.128942f, -0.0400538f, 
		0.128942f, -0.0400538f, 0.128942f, -0.0400538f, 
		0.128835f, -0.0435808f, 0.128835f, -0.0435808f, 
		0.128835f, -0.0435808f, -3.10784f, 4.69278f, 
		0.381290f, -0.0358013f, 0.381290f, -0.0358013f, 
		0.381290f, -0.0358013f, 0.253929f, -0.0472392f, 
		0.253929f, -0.0472392f, 0.253929f, -0.0472392f, 
		0.381346f, -0.0358232f, 0.381346f, -0.0358232f, 
		0.381346f, -0.0358232f, 0.381346f, -0.0358232f, 
		0.381346f, -0.0358232f, 0.253985f, -0.0472424f, 
		0.253985f, -0.0472424f, 0.253985f, -0.0472424f, 
		0.253985f, -0.0472424f, 0.253985f, -0.0472424f, 
		0.133870f, 1.00300f, 0.133870f, 1.00300f, 
		0.133870f, 1.00300f, 0.133858f, 0.999400f, 
		0.133858f, 0.999400f, 0.133858f, 0.999400f, 
		0.787533f, 0.574202f, 0.804189f, 0.588784f, 
		0.804189f, 0.588784f, 0.804194f, 0.588788f, 
		0.787537f, 0.574206f, 33.0255f, -19.8498f, 
		33.0187f, -19.9521f, 33.0255f, -20.2986f, 
		33.0187f, -19.9521f, 33.0187f, -21.8711f, 
		33.0255f, -21.5246f, 33.0255f, -21.5246f, 
		33.0255f, -21.9734f, 33.0187f, -21.8711f, 
		0.787533f, -0.574202f, 0.804255f, -0.588842f, 
		0.804255f, -0.588842f, 0.804255f, -0.588842f, 
		0.787537f, -0.574206f, -0.864239f, 0.249713f, 
		-0.864239f, 0.249713f, -0.864239f, 0.249713f, 
		-0.864239f, 0.249713f, -0.864239f, 0.249713f, 
		-0.242203f, 0.985137f, -0.242203f, 0.985137f, 
		-0.242203f, 0.981809f, -0.242203f, 0.985137f, 
		-0.242203f, 0.981809f, -0.242203f, 0.981809f, 
		-2.91490f, 3.25759f, 0.362526f, 0.583333f, 
		0.988905f, 0.0833330f, 0.362525f, 0.916667f, 
		0.988906f, 0.416667f, 0.257169f, 0.985099f, 
		0.257169f, 0.985099f, 0.257169f, 0.985099f, 
		0.257169f, 0.981773f, 0.257169f, 0.981773f, 
		0.257169f, 0.981773f, 0.204453f, 0.750000f, 
		0.204453f, 0.583333f, 0.209011f, 0.640807f, 
		0.146977f, 0.166667f, 0.142419f, 0.140807f, 
		3.22482f, -4.69278f, 0.355141f, 0.916667f, 
		0.000000f, 0.416667f, 0.996290f, 0.416667f, 
		0.101929f, 0.416667f, 0.249501f, 0.916667f, 
		0.000000f, 0.0833330f, 0.355141f, 0.583333f, 
		0.996290f, 0.0833330f, 0.105183f, 0.0833330f, 
		0.246247f, 0.583333f, 0.0643180f, 0.917941f, 
		0.0643180f, 0.915941f, 0.0643180f, 0.915941f, 
		0.0643180f, 0.917941f, 0.351430f, 0.583333f, 
		1.00000f, 0.0833330f, 0.000000f, 0.0833330f, 
		0.246247f, 0.583333f, 0.105183f, 0.0833330f, 
		0.351430f, 0.916667f, 1.00000f, 0.416667f, 
		0.000000f, 0.416667f, 31.4681f, -19.1214f, 
		31.5891f, -19.0339f, 31.5891f, -19.0339f, 
		0.246248f, 0.916667f, 0.105182f, 0.416667f, 
		-0.669258f, 0.000000f, -0.669258f, 0.000000f, 
		-0.669258f, 0.000000f, -4.52057f, 4.92753f, 
		-4.52057f, 4.92753f, -4.52057f, 4.92753f, 
		-1.06702f, -0.515018f, -1.06702f, -0.515018f, 
		-1.06702f, -0.515018f, 0.111579f, 0.416667f, 
		0.239851f, 0.916667f, 67.6227f, -19.1437f, 
		67.6227f, -19.1437f, 67.6227f, -22.4959f, 
		67.6227f, -22.4959f, 0.0102949f, 1.27049f, 
		0.0102949f, 1.27049f, 0.0102949f, 1.27049f, 
		0.0102949f, 1.27049f, 0.0103235f, 1.26593f, 
		0.0103235f, 1.26593f, 0.0103235f, 1.26593f, 
		0.0103235f, 1.26593f, 0.0103235f, 1.26593f, 
		-1.27760f, 0.274019f, -1.27760f, 0.274019f, 
		-1.27760f, 0.274019f, -1.27760f, 0.274019f, 
		-1.06531f, 0.501408f, -1.06531f, 0.501408f, 
		-1.06531f, 0.501408f, 0.133700f, 0.129391f, 
		0.217730f, 0.629391f, 69.0362f, -22.4590f, 
		69.0362f, -22.4590f, -1.31335f, -0.146752f, 
		-1.31335f, -0.146752f, -1.31335f, -0.146752f, 
		-1.31335f, -0.146752f, -1.26929f, -0.265603f, 
		-1.26929f, -0.265603f, -1.26929f, -0.265603f, 
		-1.26929f, -0.265603f, 69.1138f, -19.2005f, 
		69.1138f, -19.2005f, 69.1138f, -19.2005f, 
		69.1138f, -19.2005f, 69.1138f, -19.2005f, 
		69.1138f, -19.2005f, 69.1138f, -19.2005f, 
		69.1138f, -19.2005f, 69.1138f, -19.2005f, 
		69.1138f, -19.2005f, 69.1138f, -19.2005f, 
		-0.380430f, -0.272928f, -0.0469589f, -0.221615f, 
		-0.0469589f, -0.221615f, -0.0469589f, -0.221615f, 
		0.0679629f, -0.227487f, 0.0679629f, -0.227487f, 
		0.380430f, -0.272928f, 0.0679629f, -0.227487f, 
		32.9347f, -22.5445f, 32.9347f, -22.5445f, 
		32.9347f, -19.2787f, 32.9347f, -19.2787f, 
		26.9794f, -22.5445f, 26.9794f, -22.5445f, 
		26.9794f, -19.2786f, 26.9794f, -19.2786f, 
		2.00575f, -1.61927f, 2.00575f, -1.61927f, 
		2.00575f, -1.61927f, 2.00575f, -1.61927f, 
		2.00575f, -1.61927f, 2.00575f, -1.61927f, 
		2.00575f, -1.61927f, 2.00575f, -1.61927f, 
		2.00575f, -1.61927f, 2.00575f, -1.61927f, 
		2.00575f, -1.61927f, 2.24262f, 1.61926f, 
		2.24262f, 1.61926f, 2.24262f, 1.61926f, 
		2.24262f, 1.61926f, 2.24262f, 1.61926f, 
		2.24262f, 1.61926f, 2.24262f, 1.61926f, 
		2.24262f, 1.61926f, 2.24262f, 1.61926f, 
		2.24262f, 1.61926f, 2.24262f, 1.61926f, 
		2.61637f, 3.84315f, 2.76532f, 3.59888f, 
		68.7953f, -22.4234f, 68.7953f, -22.4234f, 
		68.7953f, -22.4234f, 68.7953f, -22.4234f, 
		68.7953f, -22.4234f, 68.7953f, -22.4234f, 
		68.7953f, -19.2162f, 68.7953f, -19.2162f, 
		68.7953f, -19.2162f, 68.7953f, -19.2162f, 
		68.7953f, -19.2162f, 68.7953f, -19.2162f, 
		68.7953f, -19.2162f, 68.7953f, -19.2162f, 
		-1.29956f, 0.137481f, -1.29956f, 0.137481f, 
		-1.29956f, 0.137481f, -0.635707f, -0.328181f, 
		-0.635707f, -0.328181f, -0.635707f, -0.328181f, 
		2.00573f, 1.61926f, 2.00573f, 1.61926f, 
		2.00573f, 1.61926f, 2.00573f, 1.61926f, 
		2.00573f, 1.61926f, 2.00573f, 1.61926f, 
		2.00573f, 1.61926f, 2.00573f, 1.61926f, 
		2.00573f, 1.61926f, 2.00573f, 1.61926f, 
		2.00573f, 1.61926f, 2.00573f, -1.61926f, 
		2.00573f, -1.61926f, 2.00573f, -1.61926f, 
		2.00573f, -1.61926f, 2.00573f, -1.61926f, 
		2.00573f, -1.61926f, 2.00573f, -1.61926f, 
		2.00573f, -1.61926f, 2.00573f, -1.61926f, 
		2.00573f, -1.61926f, 2.00573f, -1.61926f, 
		2.00573f, -1.61926f, 2.00573f, -1.61926f, 
		2.00573f, -1.61926f, 0.214606f, 0.875728f, 
		0.146977f, 0.416667f, 0.137838f, 0.371695f, 
		0.117846f, 0.0833330f, 0.233585f, 0.583333f, 
		66.0827f, -19.3071f, 66.0937f, -19.2246f, 
		66.0937f, -19.2246f, 2.32423f, -1.60357f, 
		2.32423f, -1.60357f, 2.32423f, -1.60357f, 
		2.32423f, -1.60357f, 2.32423f, -1.60357f, 
		2.32423f, -1.60357f, 2.32423f, -1.60357f, 
		2.32423f, -1.60357f, 2.32423f, -1.60357f, 
		1.92412f, 1.60356f, 1.92412f, 1.60356f, 
		1.92412f, 1.60356f, 1.92412f, 1.60356f, 
		1.92412f, 1.60356f, 1.92412f, 1.60356f, 
		1.92412f, 1.60356f, 1.92412f, 1.60356f, 
		0.777304f, 1.59638f, 0.777304f, 1.59638f, 
		0.776515f, -1.60230f, 0.773419f, -1.60979f, 
		50.8709f, -28.5940f, -0.788457f, 1.51266f, 
		-0.788457f, 1.51266f, -0.777465f, 1.59517f, 
		-0.777465f, 1.59517f, 0.233585f, 0.583333f, 
		0.117846f, 0.0833330f, -1.33007f, 0.218370f, 
		-1.33007f, 0.218370f, -1.33007f, 0.218370f, 
		-1.33007f, 0.218370f, -1.33007f, 0.218370f, 
		-1.33007f, 0.218370f, -1.33007f, 0.218370f, 
		-1.33007f, 0.218370f, -1.33020f, 0.218347f, 
		-1.33020f, 0.218347f, -1.33020f, 0.218347f, 
		-1.33020f, 0.218347f, 0.233585f, 0.916667f, 
		0.117846f, 0.416667f, -0.563812f, -0.314837f, 
		-0.563812f, -0.314837f, -0.563812f, -0.314837f, 
		-0.563812f, -0.314837f, 2.32423f, 1.60357f, 
		2.32423f, 1.60357f, 2.32423f, 1.60357f, 
		2.32423f, 1.60357f, 2.32423f, 1.60357f, 
		2.32423f, 1.60357f, 2.32423f, 1.60357f, 
		2.32423f, 1.60357f, 2.32423f, 1.60357f, 
		2.32423f, -1.60357f, 2.32423f, -1.60357f, 
		2.32423f, -1.60357f, 2.32423f, -1.60357f, 
		2.32423f, -1.60357f, 2.32423f, -1.60357f, 
		2.32423f, -1.60357f, 2.32423f, -1.60357f, 
		2.32423f, -1.60357f, 2.32423f, -1.60357f, 
		3.14263f, -4.69278f, -1.30622f, -0.242155f, 
		-1.30622f, -0.242155f, -1.30622f, -0.242155f, 
		-1.30622f, -0.242155f, -1.30622f, -0.242155f, 
		-1.30631f, -0.241990f, -1.30631f, -0.241990f, 
		68.5235f, -21.5711f, 68.5235f, -21.5711f, 
		68.5235f, -21.5711f, 68.5644f, -21.5299f, 
		68.5644f, -21.5299f, 68.5235f, -21.5711f, 
		68.5235f, -20.0685f, 68.5235f, -20.0685f, 
		68.5235f, -20.0685f, 68.5644f, -20.1096f, 
		68.5235f, -20.0685f, 68.5235f, -20.0685f, 
		68.5644f, -20.1096f, 68.5235f, -20.0685f, 
		68.5644f, -20.1096f, 0.777308f, -1.59635f, 
		0.777308f, -1.59635f, 0.777308f, 1.59635f, 
		0.777308f, 1.59635f, 9.05297f, -28.5940f, 
		-1.17966f, 0.456526f, -1.17966f, 0.456526f, 
		-1.17966f, 0.456526f, -1.17966f, 0.456526f, 
		-1.17966f, 0.456526f, 0.234725f, 1.27221f, 
		0.234725f, 1.27221f, 0.234725f, 1.27221f, 
		0.234565f, 1.26737f, 0.234725f, 1.27221f, 
		0.234565f, 1.26737f, 0.234565f, 1.26737f, 
		0.234565f, 1.26737f, 1.65228f, 0.751240f, 
		1.65228f, 0.751240f, 1.65228f, 0.751240f, 
		1.65228f, 0.751240f, 1.69322f, 0.710098f, 
		1.69322f, 0.710098f, 1.65228f, 0.751240f, 
		0.217914f, 1.26740f, 0.254983f, 1.26399f, 
		0.254983f, 1.26399f, 0.254983f, 1.26399f, 
		0.217914f, 1.26740f, 0.217914f, 1.26740f, 
		69.2933f, -22.3597f, 69.2933f, -22.3597f, 
		69.2933f, -22.3597f, 69.2933f, -22.3597f, 
		69.2933f, -22.3597f, 69.2933f, -22.3597f, 
		-4.50637f, 3.84315f, -4.50637f, 3.84315f, 
		1.65229f, -0.751248f, 1.65229f, -0.751248f, 
		1.65229f, -0.751248f, 1.69322f, -0.710103f, 
		1.69322f, -0.710103f, 1.65229f, -0.751248f, 
		-1.31953f, -0.213525f, -1.31953f, -0.213525f, 
		-1.31953f, -0.213525f, -1.31953f, -0.213525f, 
		-1.31953f, -0.213525f, -1.31956f, -0.213513f, 
		-1.31956f, -0.213513f, -1.31956f, -0.213513f, 
		-1.31956f, -0.213513f, -1.31956f, -0.213513f, 
		-1.31954f, -0.213508f, -1.31954f, -0.213508f, 
		-1.31954f, -0.213508f, -1.31954f, -0.213508f, 
		-1.31954f, -0.213508f, 51.0462f, -28.9471f, 
		31.4681f, -22.7529f, 31.5933f, -22.8122f, 
		31.5933f, -22.8122f, 31.5933f, -22.8122f, 
		-2.45696f, 1.03812f, -2.48663f, 1.06050f, 
		28.4460f, -22.7528f, 28.3208f, -22.8121f, 
		28.3208f, -22.8121f, -0.357258f, 0.882298f, 
		-0.357258f, 0.882298f, -0.357258f, 0.882298f, 
		-0.357258f, 0.882298f, -0.357258f, 0.881796f, 
		-0.357258f, 0.881796f, -0.357258f, 0.881796f, 
		-0.357258f, 0.881796f, 8.87767f, -28.9471f, 
		0.623510f, 0.916667f, 0.727920f, 0.416667f, 
		0.623510f, 0.583333f, 0.727920f, 0.0833330f, 
		-0.337662f, -0.219181f, -0.337662f, -0.219181f, 
		-0.233313f, -0.226374f, -0.233313f, -0.226374f, 
		-2.66448f, 3.84315f, -2.80654f, 3.59563f, 
		-2.80654f, 3.59563f, 0.249849f, 0.231772f, 
		0.249849f, 0.231772f, 0.249849f, 0.231772f, 
		0.230838f, 0.221145f, 0.230838f, 0.221145f, 
		0.230838f, 0.221145f, 1.63771f, 0.820417f, 
		1.63771f, 0.820417f, 1.69322f, 0.755988f, 
		1.63771f, 0.820417f, 1.63771f, 0.820417f, 
		1.69322f, 0.755988f, 1.63771f, 0.820417f, 
		1.63771f, 0.820417f, 1.69322f, 0.755988f, 
		1.63771f, 0.820417f, 1.63771f, 0.820417f, 
		1.69322f, 0.755988f, 0.228148f, -0.221637f, 
		0.228148f, -0.221637f, 0.228148f, -0.225218f, 
		0.228148f, -0.225218f, 0.380430f, -0.272928f, 
		0.623510f, 0.583333f, 0.727920f, 0.0833330f, 
		-1.13350f, -0.629301f, -1.13350f, -0.629301f, 
		-1.13350f, -0.629301f, -1.13350f, -0.629301f, 
		-1.13350f, -0.629301f, -1.13350f, -0.629301f, 
		-1.13348f, -0.629322f, -1.13348f, -0.629322f, 
		-1.13348f, -0.629322f, -1.13348f, -0.629322f, 
		-1.13348f, -0.629322f, -1.13348f, -0.629322f, 
		0.727920f, 0.416667f, 0.623510f, 0.916667f, 
		0.000000f, -0.272928f, -0.111954f, -0.219182f, 
		-0.111954f, -0.219182f, -0.111954f, -0.222903f, 
		-0.380430f, -0.272928f, -0.111954f, -0.222903f, 
		-0.513781f, 1.08657f, -0.488637f, 1.18758f, 
		-0.469040f, 1.26050f, -0.488637f, 1.18758f, 
		-0.488637f, 1.18758f, -0.513781f, 1.08657f, 
		-0.513781f, 1.08657f, -0.469040f, 1.26050f, 
		-0.513781f, 1.08657f, -0.513781f, 1.08657f, 
		0.124048f, 0.416667f, 0.227382f, 0.916667f, 
		-0.380430f, -0.272928f, -0.496702f, -0.156031f, 
		-0.496702f, -0.156031f, -0.496702f, -0.156031f, 
		-0.658925f, -0.272928f, -0.480662f, -0.156444f, 
		-0.480662f, -0.156444f, -0.480662f, -0.156444f, 
		-2.22422f, 0.238942f, -2.22422f, 0.238942f, 
		-2.16146f, 0.270330f, -2.22422f, 0.238942f, 
		-2.22422f, 0.238942f, -2.16146f, 0.270330f, 
		-2.16146f, 0.270014f, -2.18867f, 0.259819f, 
		-2.18867f, 0.259819f, -2.16146f, 0.270330f, 
		-2.18867f, 0.259819f, -2.16146f, 0.270014f, 
		-2.18867f, 0.259819f, -0.683830f, -0.514022f, 
		-0.683830f, -0.514022f, -0.683830f, -0.514022f, 
		-0.683830f, -0.514022f, -0.683830f, -0.514022f, 
		-0.389670f, 0.227507f, -0.389670f, 0.227507f, 
		-0.389670f, 0.227507f, -0.390846f, 0.225202f, 
		-0.390846f, 0.225202f, -0.390846f, 0.225202f, 
		-0.390846f, 0.225202f, -0.389670f, 0.227507f, 
		-2.26011f, -0.215571f, -2.26011f, -0.215571f, 
		-2.26011f, -0.215571f, -2.26011f, -0.215571f, 
		-0.225000f, 1.26425f, -0.225000f, 1.26425f, 
		-0.225000f, 1.26425f, -0.225000f, 1.26425f, 
		-0.289171f, 1.26832f, -0.289171f, 1.26832f, 
		-0.289171f, 1.26832f, -0.289171f, 1.26832f, 
		-0.289171f, 1.26832f, -0.225000f, 1.26425f, 
		-2.29866f, -0.207207f, -2.29866f, -0.207207f, 
		-2.21971f, -0.199969f, -2.21971f, -0.199969f, 
		-2.18502f, -0.194351f, -2.16079f, -0.181517f, 
		-2.16143f, -0.190451f, -2.18502f, -0.194351f, 
		-2.25996f, -0.215548f, -2.25996f, -0.215548f, 
		-2.25996f, -0.215548f, -2.25996f, -0.215548f, 
		-2.16143f, -0.191571f, -2.23194f, -0.327530f, 
		-2.16143f, -0.191571f, -2.23083f, -0.157485f, 
		-2.23199f, -0.327502f, -2.16143f, -0.191571f, 
		-2.16143f, -0.191571f, -2.23081f, -0.157482f, 
		-2.23193f, -0.327451f, -2.16143f, -0.191552f, 
		-2.16143f, -0.191552f, -2.23083f, -0.157485f, 
		68.6029f, -22.2369f, 68.6029f, -22.2369f, 
		68.6029f, -22.2369f, 68.6029f, -22.2369f, 
		68.6029f, -22.2369f, 68.6029f, -22.2369f, 
		68.6029f, -19.4026f, 68.6029f, -19.4026f, 
		68.6029f, -19.4026f, 68.6029f, -19.4026f, 
		-2.29877f, -0.207232f, -2.29877f, -0.207232f, 
		69.3909f, -19.4101f, 69.3909f, -19.4101f, 
		69.4263f, -19.5239f, 69.3909f, -19.4101f, 
		69.3909f, -19.4101f, -2.16143f, 0.379065f, 
		-2.23886f, 0.251065f, -2.18418f, 0.386519f, 
		-2.16143f, 0.379065f, -2.23885f, 0.251118f, 
		-2.18420f, 0.386517f, -2.16143f, 0.379065f, 
		-2.16143f, 0.379065f, -2.23886f, 0.251083f, 
		-2.16143f, 0.379065f, -2.16143f, 0.379065f, 
		-2.18418f, 0.386519f, 1.73171f, -1.41715f, 
		1.73171f, -1.41715f, 1.73171f, -1.41715f, 
		1.73171f, -1.41715f, 1.73171f, -1.41715f, 
		1.73171f, -1.41715f, 1.73171f, -1.41715f, 
		1.73171f, -1.41715f, 1.73171f, -1.41715f, 
		1.73171f, -1.41715f, 1.73171f, -1.41715f, 
		1.73172f, 1.41716f, 1.73172f, 1.41716f, 
		1.73172f, 1.41716f, 1.73172f, 1.41716f, 
		5.43857f, -1.50000f, 5.43857f, -1.50000f, 
		5.43857f, 1.50000f, 5.43857f, 1.50000f, 
		2.51971f, -1.40967f, 2.55514f, -1.29587f, 
		2.51971f, -1.40967f, 2.55514f, -1.29587f, 
		2.51971f, -1.40967f, 2.51971f, -1.40967f, 
		2.51972f, 1.40966f, 2.55514f, 1.29586f, 
		2.51972f, 1.40966f, 2.55514f, 1.29586f, 
		2.51972f, 1.40966f, 2.51972f, 1.40966f, 
		-5.43857f, -1.50000f, -5.43857f, -1.50000f, 
		-5.43857f, 1.50000f, -5.43857f, 1.50000f, 
		-1.16046f, 0.628522f, -1.16046f, 0.628522f, 
		-1.16046f, 0.628522f, -1.16046f, 0.628522f, 
		-1.16046f, 0.628522f, 1.73170f, 1.41713f, 
		1.73170f, 1.41713f, 1.73170f, 1.41713f, 
		1.73170f, 1.41713f, 1.73170f, 1.41713f, 
		1.73170f, 1.41713f, 1.73170f, -1.41713f, 
		1.73170f, -1.41713f, 1.73170f, -1.41713f, 
		1.73170f, -1.41713f, 1.73170f, -1.41713f, 
		1.73170f, -1.41713f, -1.11364f, -0.656741f, 
		-1.11364f, -0.656741f, -1.11364f, -0.656741f, 
		2.51972f, 1.40966f, 2.55514f, 1.29586f, 
		2.51972f, 1.40966f, 2.55514f, 1.29586f, 
		2.51972f, 1.40966f, 2.51972f, 1.40966f, 
		2.51972f, -1.40966f, 2.55514f, -1.29587f, 
		2.51972f, -1.40966f, 2.55514f, -1.29587f, 
		2.51972f, -1.40966f, 2.51972f, -1.40966f, 
		2.54257f, 1.03812f, 2.62943f, 1.15069f, 
		2.62943f, 1.15069f, 2.62943f, 1.15069f, 
		-1.25240f, 0.557371f, -1.25240f, 0.557371f, 
		-1.25240f, 0.557371f, -1.25240f, 0.557371f, 
		-1.25240f, 0.557371f, -0.519418f, 0.256786f, 
		-0.519418f, 0.256786f, -0.519418f, 0.256786f, 
		-0.519418f, 0.256786f, -0.519418f, 0.256786f, 
		-0.519418f, 0.256786f, -0.643940f, -0.513400f, 
		-0.643940f, -0.513400f, -0.643940f, -0.513400f, 
		-0.643940f, -0.513400f, -0.643940f, -0.513400f, 
		-0.643947f, -0.513439f, -0.643947f, -0.513439f, 
		-0.643947f, -0.513439f, -0.643947f, -0.513439f, 
		-0.896958f, -0.715541f, -0.896958f, -0.715541f, 
		-0.896940f, -0.715546f, -0.896940f, -0.715546f, 
		-0.896874f, -0.715543f, -0.896874f, -0.715543f, 
		-1.25250f, 0.557337f, -1.25250f, 0.557337f, 
		-1.25250f, 0.557337f, -1.25250f, 0.557337f, 
		-1.25250f, 0.557337f, 4.94415f, -1.50000f, 
		4.94415f, -1.53448f, -4.94415f, -1.50000f, 
		-4.94415f, -1.53448f, -0.749439f, 0.675539f, 
		-0.749439f, 0.675539f, -0.749439f, 0.675539f, 
		-0.749439f, 0.675539f, -0.749395f, 0.675501f, 
		-0.749395f, 0.675501f, -0.749395f, 0.675501f, 
		-0.749395f, 0.675501f, -1.11122f, -0.745199f, 
		-1.11122f, -0.745199f, -1.11122f, -0.745199f, 
		-1.11122f, -0.745199f, -1.11122f, -0.745199f, 
		-1.11122f, -0.745199f, 0.581589f, 0.974461f, 
		0.581589f, 0.974461f, 0.581589f, 0.974461f, 
		0.581589f, 0.974461f, 0.581589f, 0.974461f, 
		0.677259f, 0.981863f, 0.677259f, 0.981863f, 
		0.677259f, 0.981863f, 0.677259f, 0.981863f, 
		0.677259f, 0.981863f, 66.9570f, -22.2106f, 
		66.9570f, -22.2106f, 66.9570f, -19.6126f, 
		66.9570f, -19.6126f, -0.691199f, 1.01053f, 
		-0.691199f, 1.01053f, -0.691199f, 1.01053f, 
		-0.691199f, 1.01053f, -0.691130f, 1.00684f, 
		-0.691130f, 1.00684f, -0.691130f, 1.00684f, 
		-0.691199f, 1.01053f, -0.691130f, 1.00684f, 
		-0.691130f, 1.00684f, -0.772372f, -0.0898810f, 
		-0.772372f, -0.0898810f, -0.772481f, -0.0971555f, 
		-0.772481f, -0.0971555f, -0.772481f, -0.0971555f, 
		-0.772372f, -0.0898810f, -0.697314f, -0.660251f, 
		-0.697314f, -0.660251f, -0.697314f, -0.660251f, 
		-0.697314f, -0.660251f, 0.691666f, 1.00754f, 
		0.691666f, 1.00754f, 0.691666f, 1.00754f, 
		0.691666f, 1.00367f, 0.691666f, 1.00367f, 
		0.691666f, 1.00367f, -3.17899f, 0.190161f, 
		-3.17899f, 0.190161f, -3.17899f, 0.190161f, 
		-7.04295f, -22.2106f, -7.04295f, -22.2106f, 
		-7.04295f, -19.6126f, -7.04295f, -19.6126f, 
		-0.600892f, -0.0544910f, -0.600892f, -0.0544910f, 
		-0.600892f, -0.0544910f, -0.600892f, -0.0544910f, 
		-0.600892f, -0.0544910f, -0.601268f, -0.0644916f, 
		-0.601268f, -0.0644916f, -0.601268f, -0.0644916f, 
		-0.601268f, -0.0644916f, -0.601268f, -0.0644916f, 
		-3.41628f, -0.188669f, -3.41628f, -0.188669f, 
		-3.41628f, -0.188669f, -3.41982f, -0.188630f, 
		-3.41982f, -0.188630f, -3.41982f, -0.188630f, 
		-3.43379f, -0.188545f, -3.43379f, -0.188545f, 
		-0.781345f, -0.0854359f, -0.781345f, -0.0854359f, 
		-0.781345f, -0.0854359f, -0.780891f, -0.0893513f, 
		-0.780891f, -0.0893513f, -0.780891f, -0.0893513f, 
		-0.651884f, 0.940329f, -0.643602f, 0.937916f, 
		-0.643602f, 0.937916f, -0.643602f, 0.937916f, 
		-0.651884f, 0.940329f, -0.651884f, 0.940329f, 
		-3.41626f, 0.190543f, -3.41626f, 0.190543f, 
		-3.41626f, 0.190543f, -3.41986f, 0.190504f, 
		-3.43379f, 0.190418f, -3.41986f, 0.190504f, 
		-3.41986f, 0.190504f, -3.43379f, 0.190418f, 
		-3.43379f, 0.190418f, -3.41626f, 0.190543f, 
		-3.41626f, 0.190543f, -3.41626f, 0.190543f, 
		-3.41986f, 0.190504f, -3.41986f, 0.190504f, 
		-3.41986f, 0.190504f, -3.43379f, 0.190418f, 
		-3.43379f, 0.190418f, -3.41625f, 0.190593f, 
		-3.41625f, 0.190593f, -3.41625f, 0.190593f, 
		-3.41985f, 0.190554f, -3.41985f, 0.190554f, 
		-3.41985f, 0.190554f, -3.43379f, 0.190468f, 
		-3.43379f, 0.190468f, -3.17899f, 0.190218f, 
		-3.17899f, 0.190218f, -3.17899f, 0.190218f, 
		0.188589f, 0.254911f, 0.159878f, 0.272928f, 
		0.188589f, 0.254911f, 0.159878f, 0.272928f, 
		0.159878f, 0.272928f, 0.188156f, 0.254875f, 
		0.188156f, 0.254875f, 0.159878f, 0.272928f, 
		0.188156f, 0.254875f, 0.188589f, 0.254911f, 
		3.24480f, 1.95711f, 4.75814f, 2.97460f, 
		-4.75828f, 2.97420f, -3.16894f, -0.188658f, 
		-3.16894f, -0.188658f, -3.41490f, -0.188619f, 
		-3.41490f, -0.188619f, -3.43379f, -0.188534f, 
		-3.43379f, -0.188534f, 4.94415f, 1.50000f, 
		4.94415f, 1.53445f, 0.649201f, 0.708334f, 
		0.702230f, 0.208334f, 4.75816f, -2.97453f, 
		-4.94416f, 1.50000f, -4.94416f, 1.53446f, 
		-0.230639f, 0.250176f, -0.159859f, 0.272928f, 
		-0.230639f, 0.250176f, -0.230639f, 0.250176f, 
		-0.230639f, 0.250176f, -0.159859f, 0.272928f, 
		-0.159859f, 0.272928f, -0.159859f, 0.272928f, 
		-0.213658f, 0.255787f, -0.213658f, 0.255787f, 
		-0.159859f, 0.272928f, -0.213658f, 0.255787f, 
		-0.213658f, 0.255787f, -4.75830f, -2.97415f, 
		-0.684529f, -0.618093f, -0.684529f, -0.618093f, 
		-0.684529f, -0.618093f, -0.684529f, -0.618093f, 
		-0.684529f, -0.618093f, -0.684529f, -0.618093f, 
		-1.22716f, 0.672327f, -1.22716f, 0.672327f, 
		-1.22716f, 0.672327f, -1.22716f, 0.672327f, 
		-1.22716f, 0.672327f, -1.22717f, 0.672352f, 
		-1.22717f, 0.672352f, -1.22717f, 0.672352f, 
		-1.22717f, 0.672352f, -1.22717f, 0.672352f, 
		0.627136f, -1.16361f, 0.627136f, -1.16361f, 
		0.627136f, -1.16361f, 0.627094f, -1.16367f, 
		0.627094f, -1.16367f, 0.627094f, -1.16367f, 
		0.627094f, -1.16367f, 68.1052f, -19.6085f, 
		68.1052f, -19.6085f, 68.1052f, -19.6085f, 
		68.1052f, -19.6085f, 68.1052f, -19.6085f, 
		68.1052f, -19.6085f, 68.1052f, -22.0310f, 
		68.1052f, -22.0310f, 68.1052f, -22.0310f, 
		68.1052f, -22.0310f, 68.1052f, -22.0310f, 
		68.1052f, -22.0310f, -3.22659f, 2.88482f, 
		0.702230f, 0.291667f, 0.649201f, 0.791667f, 
		1.23400f, 1.21124f, 1.23400f, 1.21124f, 
		1.23400f, 1.21124f, 1.23400f, 1.21124f, 
		1.23400f, 1.21124f, 1.23400f, 1.21124f, 
		1.23398f, -1.21124f, 1.23398f, -1.21124f, 
		1.23398f, -1.21124f, 1.23398f, -1.21124f, 
		1.23398f, -1.21124f, 1.23398f, -1.21124f, 
		1.39011f, -1.19390f, 1.39011f, -1.19390f, 
		1.39011f, -1.19390f, 1.39011f, -1.19390f, 
		1.39011f, -1.19390f, 1.39011f, -1.19390f, 
		1.39011f, -1.19390f, 1.39011f, -1.19390f, 
		1.02730f, -1.17337f, 1.02730f, -1.17337f, 
		1.02730f, -1.17337f, 1.02730f, -1.17337f, 
		1.02730f, -1.17337f, 1.02730f, -1.17337f, 
		1.02730f, -1.17337f, 1.02730f, -1.17337f, 
		1.02730f, -1.17337f, 1.02730f, -1.17337f, 
		1.39012f, 1.19390f, 1.39012f, 1.19390f, 
		1.39012f, 1.19390f, 1.39012f, 1.19390f, 
		1.39012f, 1.19390f, 1.39012f, 1.19390f, 
		1.39012f, 1.19390f, 1.39012f, 1.19390f, 
		0.159878f, 0.788046f, 0.159878f, 0.788046f, 
		0.155005f, 0.794222f, 0.155005f, 0.794266f, 
		0.159878f, 0.788046f, 0.159878f, 0.788046f, 
		-0.657225f, 0.686512f, -0.657225f, 0.686512f, 
		-0.657225f, 0.686512f, 1.02731f, 1.17338f, 
		1.02731f, 1.17338f, 1.02731f, 1.17338f, 
		1.02731f, 1.17338f, 1.02731f, 1.17338f, 
		1.02731f, 1.17338f, 1.02731f, 1.17338f, 
		1.02731f, 1.17338f, 1.02731f, 1.17338f, 
		1.02731f, 1.17338f, 6.05729f, 1.03827f, 
		6.05152f, 1.00124f, 1.95789f, -5.00000f, 
		2.18859f, -4.95556f, 0.484826f, 1.30225f, 
		0.484826f, 1.30225f, 6.05729f, 1.03827f, 
		6.05151f, 1.00115f, -0.644144f, 1.27214f, 
		-0.644144f, 1.27214f, -0.631333f, 1.26732f, 
		-0.631333f, 1.26732f, -0.631333f, 1.26732f, 
		-0.644144f, 1.27214f, 0.732378f, 1.26548f, 
		0.732378f, 1.26548f, 0.616253f, 1.26731f, 
		0.616253f, 1.26731f, 0.732378f, 1.26548f, 
		0.616253f, 1.26731f, -0.159859f, 0.788046f, 
		-0.159859f, 0.788046f, -0.159859f, 0.788046f, 
		-0.191169f, 0.794253f, -0.202609f, 0.794297f, 
		-0.159859f, 0.788046f, -3.41628f, 0.379313f, 
		-3.41984f, 0.379237f, -3.43379f, 0.379065f, 
		-3.43379f, 0.379065f, -0.694365f, 0.757062f, 
		-0.694365f, 0.757062f, -0.694365f, 0.757062f, 
		-0.694365f, 0.757062f, 0.522626f, 1.25875f, 
		0.522626f, 1.25875f, 0.522626f, 1.25875f, 
		0.522626f, 1.25875f, 0.522626f, 1.25875f, 
		-0.636667f, -0.726812f, -0.636667f, -0.726812f, 
		-0.636667f, -0.726812f, -0.636667f, -0.726812f, 
		-0.636667f, -0.726812f, -0.636667f, -0.726812f, 
		-0.636667f, -0.726812f, -0.636667f, -0.726812f, 
		-0.636656f, -0.726805f, -0.636656f, -0.726805f, 
		-0.636656f, -0.726805f, -0.636656f, -0.726805f, 
		0.000000f, 2.62760f, 0.000000f, 2.62760f, 
		-3.07440f, 1.48531f, -3.07440f, 1.48531f, 
		-3.07440f, 1.48531f, -0.159752f, 0.272928f, 
		-0.159752f, 0.272928f, -0.159752f, 0.272928f, 
		-0.159752f, 0.272928f, -0.159752f, 0.272928f, 
		-0.159752f, 0.272928f, -0.159752f, 0.788046f, 
		-0.159752f, 0.788046f, -0.159752f, 0.788046f, 
		-0.159752f, 0.788046f, -0.159752f, 0.788046f, 
		-0.159752f, 0.788046f, -0.656323f, -0.705854f, 
		-0.656323f, -0.705854f, -0.656323f, -0.705854f, 
		-0.656323f, -0.705854f, -2.85960f, -4.80777f, 
		4.06583f, 4.09253f, -0.159790f, 1.33390f, 
		-0.159790f, 1.33390f, -0.159790f, 1.33390f, 
		-0.159790f, 1.33390f, -0.159790f, 1.33390f, 
		-0.159790f, 1.33390f, -0.159790f, 1.33390f, 
		-0.159790f, 1.33390f, -3.41621f, 0.379314f, 
		-3.41988f, 0.379237f, -3.43379f, 0.379065f, 
		-3.43379f, 0.379065f, -4.06593f, 4.09243f, 
		0.159785f, 1.33390f, 0.159785f, 1.33390f, 
		0.159785f, 1.33390f, 0.159785f, 1.33390f, 
		0.159785f, 1.33390f, 0.159785f, 1.33390f, 
		0.159785f, 1.33390f, 0.159785f, 1.33390f, 
		-0.0222892f, 2.65545f, -0.0222892f, 2.65545f, 
		-0.0538162f, 2.69480f, -0.0538162f, 2.69480f, 
		0.525167f, -1.24473f, 0.525167f, -1.24473f, 
		0.525167f, -1.24473f, 0.525167f, -1.24473f, 
		0.525167f, -1.24473f, -0.128200f, 0.272928f, 
		-0.342298f, 0.255596f, -0.342298f, 0.255596f, 
		-0.128200f, 0.272928f, -0.128200f, 0.272928f, 
		-0.128200f, 0.272928f, -0.437171f, 0.242907f, 
		-0.128200f, 0.272928f, -0.437171f, 0.242907f, 
		-0.128200f, 0.272928f, -0.128200f, 0.272928f, 
		-0.342298f, 0.255596f, 0.159798f, 1.33390f, 
		0.159798f, 1.33390f, 0.159798f, 1.33390f, 
		0.159798f, 1.33390f, 0.159798f, 1.33390f, 
		0.159798f, 1.33390f, 0.159798f, 1.33390f, 
		0.159798f, 1.33390f, 0.553628f, 1.19579f, 
		0.553628f, 1.19579f, 0.553628f, 1.19579f, 
		0.553628f, 1.19579f, 0.553628f, 1.19579f, 
		-0.159831f, 1.33390f, -0.159831f, 1.33390f, 
		-0.159831f, 1.33390f, -0.159831f, 1.33390f, 
		-0.159831f, 1.33390f, -0.159831f, 1.33390f, 
		5.73948f, 0.140570f, -0.201067f, 2.67890f, 
		-0.201067f, 2.67890f, -0.201067f, 2.67890f, 
		-0.319614f, 2.84174f, -0.201067f, 2.67890f, 
		-0.201067f, 2.67890f, -0.201067f, 2.67890f, 
		-0.201067f, 2.67890f, -0.201067f, 2.67890f, 
		-0.369059f, 2.84174f, -0.0713897f, 2.65016f, 
		-0.0713897f, 2.65016f, -0.0713897f, 2.65016f, 
		-0.169924f, 2.66489f, -0.169924f, 2.66489f, 
		-0.169924f, 2.66489f, -2.25708f, 0.550231f, 
		-2.25708f, 0.550231f, -2.25708f, 0.550231f, 
		-2.25708f, 0.550231f, -2.24606f, -0.500728f, 
		-2.24606f, -0.500728f, -2.24606f, -0.500728f, 
		-2.16143f, -0.523064f, -2.22547f, -0.593734f, 
		-2.22547f, -0.593734f, -2.16148f, -0.518878f, 
		-2.22547f, -0.593734f, 1.63768f, -1.04026f, 
		1.63768f, -1.04026f, 1.63768f, -1.04026f, 
		1.69322f, -0.958536f, 1.63768f, -1.04026f, 
		1.63768f, -1.04026f, 1.63768f, -1.04026f, 
		1.69322f, -0.958536f, -2.30200f, 0.548870f, 
		0.515197f, 1.19828f, 0.515197f, 1.19828f, 
		0.515197f, 1.19828f, 0.515197f, 1.19828f, 
		0.515197f, 1.19828f, 0.515197f, 1.19828f, 
		4.06583f, -4.09253f, -5.73943f, 0.140494f, 
		-4.06594f, -4.09242f, -2.30205f, 0.548927f, 
		-2.30205f, 0.548927f, 59.9820f, -27.8151f, 
		-4.60729f, 3.84315f, -4.60729f, 3.84315f, 
		4.60732f, 3.84315f, 4.60732f, 3.84315f, 
		0.787533f, -0.994546f, 0.804190f, -1.01980f, 
		0.787537f, -0.994554f, 0.804194f, -1.01981f, 
		0.804194f, -1.01981f, 0.293233f, 0.788046f, 
		0.293233f, 0.788046f, 0.293233f, 0.788046f, 
		0.293233f, 0.788046f, 0.293233f, 0.788046f, 
		0.293233f, 0.788046f, -2.23307f, 0.569144f, 
		-2.23307f, 0.569144f, -2.23307f, 0.569144f, 
		-2.16148f, 0.551655f, -2.19061f, 0.560837f, 
		-2.19061f, 0.560837f, -2.15881f, 0.552315f, 
		-2.19061f, 0.560837f, -2.16148f, 0.551655f, 
		-2.16148f, 0.551655f, 0.498744f, -1.22673f, 
		0.498744f, -1.22673f, 0.484464f, 1.24206f, 
		0.484464f, 1.24206f, 0.484464f, 1.24206f, 
		0.484464f, 1.24206f, 0.484464f, 1.24206f, 
		0.484464f, 1.24206f, 0.484464f, 1.24206f, 
		0.484464f, 1.24206f, 2.44977f, 1.03812f, 
		2.62385f, 1.15716f, 2.62385f, 1.15716f, 
		2.62385f, 1.15716f, -4.60731f, 3.84315f, 
		-4.60731f, 3.84315f, 67.7748f, -19.6996f, 
		67.6587f, -19.7812f, 67.7748f, -19.6996f, 
		67.7748f, -19.6996f, 67.7757f, -19.6981f, 
		67.7757f, -19.6981f, 67.6587f, -19.7812f, 
		67.7757f, -19.6981f, 67.7757f, -19.6981f, 
		67.7757f, -19.6981f, 67.7757f, -19.6981f, 
		67.6587f, -19.7812f, 67.6587f, -19.7812f, 
		67.7748f, -19.6996f, 67.6587f, -21.8584f, 
		67.7386f, -21.9205f, 67.7386f, -21.9205f, 
		67.7386f, -21.9205f, 67.7401f, -21.9231f, 
		67.7401f, -21.9231f, 67.6587f, -21.8584f, 
		67.6587f, -21.8584f, 67.7401f, -21.9231f, 
		67.6587f, -21.8584f, 0.424632f, -1.31816f, 
		0.424632f, -1.31816f, 0.424632f, -1.31816f, 
		0.424632f, -1.31816f, 0.424632f, -1.31816f, 
		0.476850f, -1.22459f, 0.476850f, -1.22459f, 
		0.476850f, -1.22459f, 0.476850f, -1.22459f, 
		0.476850f, -1.22459f, -0.333914f, 0.0901450f, 
		-0.266168f, 0.0367047f, -0.333914f, 0.0901450f, 
		-0.333914f, 0.0901450f, -0.333914f, 0.0901450f, 
		-0.266168f, 0.0367047f, -5.38829f, -0.250319f, 
		0.658924f, -0.272928f, 0.756435f, -0.140801f, 
		0.756435f, -0.140801f, 0.380430f, -0.272928f, 
		0.821190f, -0.117529f, 0.821190f, -0.117529f, 
		0.903597f, 1.12014f, 0.903597f, 1.12014f, 
		0.904461f, 1.12165f, 0.904461f, 1.12165f, 
		0.904461f, 1.12165f, 0.904461f, 1.12165f, 
		0.787533f, 1.03860f, 0.787537f, 1.03860f, 
		0.903597f, 1.12014f, 0.319629f, 1.33390f, 
		0.319629f, 1.33390f, 0.319629f, 1.33390f, 
		0.319629f, 1.33390f, 5.38847f, -0.250149f, 
		0.787533f, -1.03864f, 0.865920f, -1.09818f, 
		0.867383f, -1.10074f, 0.867383f, -1.10074f, 
		0.867383f, -1.10074f, 0.867383f, -1.10074f, 
		0.865920f, -1.09818f, 0.787537f, -1.03865f, 
		0.865920f, -1.09818f, -0.373020f, 0.279590f, 
		-0.266321f, 0.0947622f, -0.373020f, 0.279590f, 
		-0.373020f, 0.279590f, -0.373020f, 0.279590f, 
		-0.373020f, 0.279590f, -0.266321f, 0.0947622f, 
		26.8886f, -21.5246f, 26.8886f, -21.9734f, 
		26.8987f, -22.2768f, 26.8886f, -21.5246f, 
		26.8987f, -22.2768f, -3.22524f, -4.69278f, 
		-3.17899f, -0.523633f, 2.58306f, -4.92016f, 
		-3.17899f, -0.523564f, -2.23832f, 0.505184f, 
		-2.23832f, 0.505184f, -2.23832f, 0.505184f, 
		-2.23832f, 0.505184f, -2.23832f, 0.505184f, 
		-2.23832f, 0.505184f, -2.23832f, 0.505184f, 
		0.278932f, 2.70276f, 0.278932f, 2.70276f, 
		0.278932f, 2.70276f, 0.278932f, 2.70276f, 
		0.260167f, 2.74008f, 0.260167f, 2.74008f, 
		0.260167f, 2.74008f, 0.260167f, 2.74008f, 
		-3.17899f, -0.523605f, -3.41594f, -0.517851f, 
		-3.41594f, -0.517851f, -3.41594f, -0.517851f, 
		-3.41973f, -0.517740f, -3.41973f, -0.517740f, 
		-3.41973f, -0.517740f, -3.43379f, -0.517504f, 
		-3.43379f, -0.517504f, -3.41594f, -0.517851f, 
		-3.41594f, -0.517851f, -3.41594f, -0.517851f, 
		-3.41973f, -0.517740f, -3.41973f, -0.517740f, 
		-3.41973f, -0.517740f, -3.43379f, -0.517504f, 
		-3.43379f, -0.517504f, -3.41594f, -0.517851f, 
		-3.41594f, -0.517851f, -3.41594f, -0.517851f, 
		-3.41973f, -0.517740f, -3.41973f, -0.517740f, 
		-3.41973f, -0.517740f, -3.43379f, -0.517504f, 
		-3.43379f, -0.517504f, 0.000000f, 2.84174f, 
		0.228091f, 2.72989f, 0.228091f, 2.72989f, 
		0.319614f, 2.84174f, 0.184529f, 2.84174f, 
		0.227043f, 2.73022f, 0.369058f, 2.84174f, 
		0.227043f, 2.73022f, 0.395008f, 1.35700f, 
		0.395008f, 1.35700f, 0.395008f, 1.35700f, 
		0.395008f, 1.35700f, -3.17899f, -0.523605f, 
		-0.697321f, 0.834303f, -0.697321f, 0.834303f, 
		-0.697321f, 0.834303f, -0.697321f, 0.834303f, 
		-1.37626f, 0.629798f, -1.37626f, 0.629798f, 
		-1.37626f, 0.629798f, -1.37626f, 0.629798f, 
		-1.37626f, 0.629798f, -1.37626f, 0.629798f, 
		-1.37626f, 0.629798f, 0.649201f, 0.875001f, 
		0.702230f, 0.375001f, -2.09319f, 0.0201256f, 
		-2.09319f, 0.0201256f, -2.09319f, 0.0201256f, 
		-2.09319f, 0.0201256f, -2.09319f, 0.0201256f, 
		-2.09319f, 0.0201256f, 0.410449f, -1.27157f, 
		0.410449f, -1.27157f, 0.410449f, -1.27157f, 
		0.410449f, -1.27157f, 0.410449f, -1.27157f, 
		0.410449f, -1.27157f, 0.410449f, -1.27157f, 
		-0.331970f, 0.115240f, -0.270146f, 0.162814f, 
		-0.331970f, 0.115240f, -0.331970f, 0.115240f, 
		-0.270146f, 0.162814f, -1.14914f, -0.809996f, 
		-1.14914f, -0.809996f, -1.14914f, -0.809996f, 
		0.649201f, 0.625000f, 0.702230f, 0.125000f, 
		-1.14919f, -0.809969f, -1.14919f, -0.809969f, 
		-1.14919f, -0.809969f, -0.668797f, -0.769466f, 
		-0.668797f, -0.769466f, -0.668797f, -0.769466f, 
		-0.668797f, -0.769466f, 0.425693f, 1.24459f, 
		0.425693f, 1.24459f, 0.425693f, 1.24459f, 
		0.425693f, 1.24459f, 0.425693f, 1.24459f, 
		-0.394068f, 0.307589f, -0.394068f, 0.307589f, 
		-0.270146f, 0.162814f, -0.394068f, 0.307589f, 
		-0.394068f, 0.307589f, -0.394068f, 0.307589f, 
		-0.270146f, 0.162814f, -0.319290f, 0.106456f, 
		-0.258563f, 0.138457f, -0.319290f, 0.106456f, 
		-0.319290f, 0.106456f, -0.258563f, 0.138457f, 
		0.658924f, -0.272928f, 0.714890f, -0.191688f, 
		0.714890f, -0.191688f, 0.626809f, -0.201304f, 
		0.380430f, -0.272928f, 0.626809f, -0.201304f, 
		-4.67791f, 3.84315f, -4.67791f, 3.84315f, 
		-4.67791f, 4.92753f, -4.67791f, 4.92753f, 
		-0.313244f, 2.71534f, -0.313244f, 2.71534f, 
		-0.313244f, 2.71534f, -0.269270f, 2.71562f, 
		-0.319614f, 2.84174f, -0.269270f, 2.71562f, 
		-0.269270f, 2.71562f, -0.269270f, 2.71562f, 
		-0.230874f, 2.80575f, -0.369059f, 2.84174f, 
		-0.788457f, -1.51266f, -0.777465f, -1.77847f, 
		-0.788457f, -1.74667f, -0.775092f, -1.73736f, 
		0.723138f, 0.212442f, 0.723138f, 0.212442f, 
		0.700261f, 0.227738f, 0.700261f, 0.227738f, 
		0.700261f, 0.227738f, 0.700261f, 0.227738f, 
		0.700261f, 0.227738f, 0.723138f, 0.212442f, 
		0.723138f, 0.212442f, 0.723138f, 0.212442f, 
		0.304677f, 1.35311f, 0.304677f, 1.35311f, 
		0.304677f, 1.35311f, 0.304677f, 1.35311f, 
		0.304677f, 1.35311f, -0.788457f, 1.74667f, 
		-0.788457f, 1.51266f, -0.777462f, 1.71681f, 
		-0.773884f, 1.80950f, -0.788457f, -1.51266f, 
		-0.788457f, -1.74667f, -0.777462f, -1.65515f, 
		-0.779835f, -1.69627f, -0.788457f, -1.51266f, 
		0.380794f, 2.69568f, 0.311591f, 2.70304f, 
		0.380794f, 2.69568f, 0.311591f, 2.70304f, 
		0.311591f, 2.70304f, 0.311591f, 2.70304f, 
		0.380794f, 2.69568f, 0.380794f, 2.69568f, 
		0.761754f, 0.227594f, 0.807274f, 0.227813f, 
		0.807274f, 0.227813f, 0.807274f, 0.227813f, 
		0.807274f, 0.227813f, 0.761754f, 0.227594f, 
		0.761754f, 0.227594f, 0.761754f, 0.227594f, 
		-1.33452f, -0.706729f, -1.33452f, -0.706729f, 
		-1.33452f, -0.706729f, -1.33452f, -0.706729f, 
		-1.33452f, -0.706729f, -1.33452f, -0.706729f, 
		-1.33452f, -0.706729f, 0.789326f, 0.828581f, 
		0.789326f, 0.828581f, 0.788406f, 0.832119f, 
		0.788406f, 0.832119f, 0.436637f, 0.272928f, 
		0.436637f, 0.272928f, 0.436637f, 0.272928f, 
		0.436637f, 0.272928f, 0.436637f, 0.272928f, 
		0.436637f, 0.272928f, 0.436637f, 0.788046f, 
		0.436637f, 0.788046f, 0.436637f, 0.788046f, 
		0.436637f, 0.788046f, 0.436637f, 0.788046f, 
		0.436637f, 0.788046f, 53.8770f, -27.8151f, 
		-0.434527f, 1.33390f, -0.434527f, 1.33390f, 
		-0.434527f, 1.33390f, -0.437387f, 1.33377f, 
		-0.434527f, 1.33390f, -0.434527f, 1.33390f, 
		-0.434527f, 1.33390f, -0.437387f, 1.33375f, 
		0.436709f, 0.272928f, 0.436709f, 0.272928f, 
		0.436709f, 0.272928f, 0.436709f, 0.272928f, 
		0.436709f, 0.272928f, 0.436709f, 0.272928f, 
		0.404649f, 1.21982f, 0.404649f, 1.21982f, 
		31.4681f, -22.7977f, 31.5098f, -22.8833f, 
		-1.33450f, -0.706729f, -1.33450f, -0.706729f, 
		-1.33450f, -0.706729f, -1.33450f, -0.706729f, 
		-1.33450f, -0.706729f, -1.33450f, -0.706729f, 
		-0.437535f, 1.33373f, -0.434417f, 1.33390f, 
		-0.434417f, 1.33390f, -0.434417f, 1.33390f, 
		-0.437535f, 1.33376f, -0.434417f, 1.33390f, 
		-0.436658f, 0.788046f, -0.436658f, 0.788046f, 
		-0.494357f, 0.794269f, -0.436658f, 0.788046f, 
		-0.494357f, 0.794313f, -0.436658f, 0.788046f, 
		0.382564f, -1.33729f, 0.382564f, -1.33729f, 
		0.382564f, -1.33729f, 0.382564f, -1.33729f, 
		0.382564f, -1.33729f, 0.382564f, -1.33729f, 
		0.382564f, -1.33729f, 0.382564f, -1.33729f, 
		0.382507f, -1.33730f, 0.382507f, -1.33730f, 
		0.382507f, -1.33730f, 0.382507f, -1.33730f, 
		0.382507f, -1.33730f, -0.370543f, 2.63783f, 
		-0.370543f, 2.63783f, -0.370543f, 2.63783f, 
		-0.423699f, 2.64755f, -0.423699f, 2.64755f, 
		-0.423699f, 2.64755f, 0.702230f, 0.416667f, 
		0.649201f, 0.916667f, 0.436587f, 1.33390f, 
		0.436587f, 1.33390f, 0.436587f, 1.33390f, 
		0.436587f, 1.33390f, 0.436587f, 1.33390f, 
		0.436587f, 1.33390f, 0.787533f, 1.03863f, 
		0.787537f, 1.03864f, 0.806729f, 1.06945f, 
		0.806729f, 1.06945f, 0.806729f, 1.06945f, 
		0.804235f, 1.06508f, 0.804235f, 1.06508f, 
		0.804235f, 1.06508f, 0.806729f, 1.06945f, 
		0.428819f, 2.65933f, 0.428819f, 2.65933f, 
		0.428819f, 2.65933f, 0.407407f, 2.68507f, 
		0.407407f, 2.68507f, 0.407407f, 2.68507f, 
		-1.58200f, -0.224436f, -1.56546f, -0.225290f, 
		-1.58200f, -0.224436f, -1.58200f, -0.224436f, 
		-1.56546f, -0.225290f, 0.649201f, 0.574406f, 
		0.702230f, 1.00000f, 0.702230f, 0.0744064f, 
		-2.21742f, -0.686584f, -2.16164f, -0.655448f, 
		-2.16151f, -0.655160f, -2.18405f, -0.669300f, 
		-2.16164f, -0.655448f, -2.21743f, -0.686592f, 
		-2.16151f, -0.655160f, -2.18406f, -0.669301f, 
		-2.16151f, -0.655160f, -2.21740f, -0.686572f, 
		-2.16164f, -0.655448f, -2.18406f, -0.669301f, 
		-0.0816284f, -27.8151f, -1.57730f, 0.256431f, 
		-1.56818f, 0.257151f, -1.56818f, 0.257151f, 
		-1.57730f, 0.256431f, 4.64399f, 4.92753f, 
		4.64399f, 4.92753f, 4.64399f, 4.92753f, 
		4.52057f, 4.92753f, -1.57150f, -0.114070f, 
		-1.57339f, -0.343814f, -1.57150f, -0.114070f, 
		-1.57150f, -0.114070f, -1.57150f, -0.114070f, 
		-1.56693f, -0.116250f, -1.55815f, -0.346806f, 
		-1.57339f, -0.343814f, -1.55815f, -0.346806f, 
		-1.57150f, -0.114070f, -1.57339f, -0.343813f, 
		-1.56693f, -0.116250f, -1.57150f, -0.114070f, 
		-1.57150f, -0.114070f, -1.57150f, -0.114070f, 
		-1.56693f, -0.116250f, -1.55815f, -0.346806f, 
		-1.57339f, -0.343813f, -1.55815f, -0.346806f, 
		-1.57339f, -0.343813f, 0.133925f, 1.00000f, 
		0.127601f, 1.00000f, 0.221183f, 0.542830f, 
		0.130247f, 0.0428304f, 0.219397f, 0.957150f, 
		0.130246f, 0.457150f, 0.221184f, 0.000000f, 
		-0.688558f, 0.235055f, -0.688558f, 0.235055f, 
		-0.698747f, 0.228680f, -0.698747f, 0.228680f, 
		-0.698747f, 0.228680f, -0.698747f, 0.228680f, 
		-0.698747f, 0.228680f, -0.688558f, 0.235055f, 
		-0.688558f, 0.235055f, -0.688558f, 0.235055f, 
		-0.531224f, 0.794831f, -0.494092f, 0.788046f, 
		-0.494092f, 0.788046f, -0.550817f, 0.798021f, 
		-0.494092f, 0.788046f, -0.494092f, 0.788046f, 
		-0.494092f, 0.788046f, -0.531224f, 0.794831f, 
		-1.58199f, -0.224480f, -1.58199f, -0.224480f, 
		-1.56547f, -0.225333f, -1.58199f, -0.224480f, 
		-1.56547f, -0.225333f, -0.341400f, 0.417140f, 
		-0.341400f, 0.417140f, -0.341400f, 0.417140f, 
		-0.341400f, 0.417140f, -0.262048f, 0.355158f, 
		-0.272172f, 0.308949f, -0.281711f, 0.312729f, 
		-0.272172f, 0.308949f, -0.281711f, 0.312729f, 
		-0.341400f, 0.417140f, -0.262048f, 0.355158f, 
		-2.66450f, 3.84315f, -2.84133f, 3.71652f, 
		-2.05007f, 0.0312971f, -1.97117f, 0.0267163f, 
		-2.03214f, 0.0180538f, -2.03214f, 0.0180538f, 
		-1.97117f, 0.0267163f, -1.97117f, 0.0267163f, 
		-1.97117f, 0.0267163f, -1.97117f, 0.0267163f, 
		-1.97117f, 0.0267163f, -1.97117f, 0.0267163f, 
		-2.23190f, 0.714175f, -2.16148f, 0.705439f, 
		-2.19266f, 0.713088f, -2.16143f, 0.707183f, 
		-2.16143f, 0.705440f, -2.23144f, 0.713439f, 
		-2.16148f, 0.704515f, -2.19189f, 0.711583f, 
		-2.23144f, 0.713439f, -2.16148f, 0.704515f, 
		-2.16143f, 0.705440f, -2.19189f, 0.711583f, 
		-0.266036f, 0.243887f, -0.335285f, 0.339923f, 
		-0.266036f, 0.243887f, -0.335285f, 0.339923f, 
		68.4594f, -22.0225f, 68.4594f, -22.0225f, 
		68.4594f, -22.0225f, 68.4331f, -22.0506f, 
		68.4331f, -22.0506f, 68.4331f, -22.0506f, 
		68.4331f, -22.0506f, 68.4331f, -22.0506f, 
		68.4594f, -22.0225f, 68.4594f, -22.0225f, 
		68.4012f, -19.6040f, 68.4012f, -19.6040f, 
		68.4191f, -19.6211f, 68.4191f, -19.6211f, 
		68.4191f, -19.6211f, 68.4191f, -19.6211f, 
		68.4012f, -19.6040f, 68.4012f, -19.6040f, 
		-2.07619f, 0.319101f, -2.07619f, 0.319101f, 
		-2.07619f, 0.319101f, -0.785400f, 0.225831f, 
		-0.785400f, 0.225831f, -0.777309f, 0.229143f, 
		-0.777309f, 0.229143f, -0.777309f, 0.229143f, 
		-0.777309f, 0.229143f, -0.785400f, 0.225831f, 
		-1.56416f, -0.183469f, -1.55403f, -0.191292f, 
		-1.56416f, -0.183469f, 1.53000f, 1.21575f, 
		1.53000f, 1.21575f, 1.54795f, 1.19869f, 
		1.54795f, 1.19869f, 1.54795f, 1.19869f, 
		1.54795f, 1.19869f, 1.53000f, 1.21575f, 
		1.53000f, 1.21575f, -1.57420f, -0.185415f, 
		-1.57420f, -0.185415f, -1.56416f, -0.193392f, 
		-1.71806f, 0.206811f, -1.71806f, 0.206811f, 
		-1.56818f, 0.257081f, -1.71806f, 0.206811f, 
		-1.56818f, 0.257081f, -3.30304f, 2.01804f, 
		-3.30304f, 2.01804f, -3.30304f, 2.01804f, 
		-3.30304f, 2.01804f, -3.30304f, 2.01804f, 
		1.52998f, -1.21576f, 1.52998f, -1.21576f, 
		1.52998f, -1.21576f, 1.53400f, -1.16660f, 
		1.53400f, -1.16660f, 1.53400f, -1.16660f, 
		1.53400f, -1.16660f, 1.53400f, -1.16660f, 
		1.52998f, -1.21576f, 1.52998f, -1.21576f, 
		-1.55446f, 0.189114f, -1.56498f, 0.182125f, 
		-1.56498f, 0.182125f, -1.55446f, 0.189114f, 
		-1.56498f, 0.182125f, -1.55446f, 0.189120f, 
		-1.56498f, 0.182125f, -1.56498f, 0.182125f, 
		-1.55446f, 0.189120f, -1.56498f, 0.182125f, 
		-1.55446f, 0.189104f, -1.56499f, 0.182111f, 
		-1.56499f, 0.182111f, -1.55446f, 0.189104f, 
		-1.56499f, 0.182111f, 3.34872f, 2.07913f, 
		3.34872f, 2.07913f, 3.34872f, 2.07913f, 
		-1.57713f, -0.177780f, -1.57713f, -0.177780f, 
		-1.56818f, -0.178100f, -1.55895f, -0.177962f, 
		-1.55695f, -0.180890f, -1.55895f, -0.177962f, 
		-3.30292f, 2.01771f, -3.30292f, 2.01771f, 
		-3.30292f, 2.01771f, -3.30292f, 2.01771f, 
		-3.30292f, 2.01771f, -2.58224f, 1.03812f, 
		-2.60447f, 1.05451f, -3.36760f, 2.07045f, 
		-3.36760f, 2.07045f, -3.36760f, 2.07045f, 
		-0.636686f, 0.256986f, -0.636686f, 0.256986f, 
		-0.553613f, 0.272928f, -0.636686f, 0.256986f, 
		-0.553613f, 0.272928f, -0.553613f, 0.272928f, 
		-0.553613f, 0.272928f, -0.636722f, 0.254842f, 
		-0.553613f, 0.272928f, -0.636722f, 0.254842f, 
		0.760860f, -0.272928f, 0.872197f, -0.171496f, 
		0.872197f, -0.171496f, 0.658924f, -0.272928f, 
		0.872197f, -0.171496f, 0.880529f, -0.140169f, 
		0.880529f, -0.140169f, 0.880529f, -0.140169f, 
		0.658924f, -0.272928f, 0.658924f, -0.272928f, 
		0.849902f, -0.148184f, 0.849902f, -0.148184f, 
		0.849902f, -0.148184f, 0.842840f, -0.158094f, 
		0.842840f, -0.158094f, 0.842840f, -0.158094f, 
		-0.498364f, 0.272928f, -0.498364f, 0.272928f, 
		-0.498364f, 0.272928f, -0.498364f, 0.272928f, 
		-0.498364f, 0.272928f, -0.498364f, 0.272928f, 
		-2.00826f, -0.0276627f, -1.97681f, -0.125300f, 
		-1.97681f, -0.125300f, -2.00826f, -0.0276627f, 
		-2.06653f, -0.0234037f, -2.00826f, -0.0276627f, 
		-1.61214f, -0.129283f, -1.61214f, -0.129283f, 
		-1.61214f, -0.129283f, -1.61214f, -0.129283f, 
		-1.56699f, -0.157173f, -1.55815f, -0.346806f, 
		-1.57338f, -0.343814f, -1.55815f, -0.346806f, 
		-1.57338f, -0.343814f, 0.498313f, 0.272928f, 
		0.498313f, 0.272928f, 0.498313f, 0.272928f, 
		0.498313f, 0.272928f, 0.498313f, 0.272928f, 
		0.498313f, 0.272928f, 0.498313f, 0.788046f, 
		0.498313f, 0.788046f, 0.498313f, 0.788046f, 
		0.498313f, 0.788046f, 0.498313f, 0.788046f, 
		0.498313f, 0.788046f, -1.05299f, 0.924260f, 
		-1.05299f, 0.924260f, -1.05299f, 0.924260f, 
		0.146977f, 1.00000f, 0.140348f, 1.00000f, 
		0.204453f, 0.500000f, 0.208677f, 0.546126f, 
		0.142753f, 0.0461258f, 0.146977f, 0.0833330f, 
		-1.96674f, -0.0505569f, -2.00762f, -0.0983189f, 
		-2.00762f, -0.0983189f, -1.96674f, -0.0505569f, 
		-1.96674f, -0.0505569f, -2.04238f, -0.0785803f, 
		-1.96674f, -0.0505569f, -1.96674f, -0.0505569f, 
		-1.55891f, -0.340051f, -1.58382f, -0.342361f, 
		-1.58382f, -0.342361f, -1.55891f, -0.340051f, 
		-1.55891f, -0.340051f, -1.58382f, -0.342361f, 
		-1.07997f, 0.947462f, -1.07997f, 0.947462f, 
		-1.07997f, 0.947462f, -1.07997f, 0.947462f, 
		-1.07997f, 0.947462f, 0.204453f, 0.916667f, 
		0.208548f, 0.953843f, 0.146977f, 0.500000f, 
		0.142882f, 0.453843f, 0.208548f, 0.000000f, 
		-1.55892f, -0.340103f, -1.58381f, -0.342422f, 
		-1.55892f, -0.340103f, -1.58381f, -0.342422f, 
		-1.55892f, -0.340103f, 0.841268f, 1.26875f, 
		0.812991f, 1.26932f, 0.812991f, 1.26932f, 
		0.812991f, 1.26932f, 0.841268f, 1.26875f, 
		0.841268f, 1.26875f, -1.99255f, -0.209796f, 
		-1.99255f, -0.209796f, -2.00957f, -0.159947f, 
		-2.04425f, -0.196132f, -2.00957f, -0.159947f, 
		-2.00957f, -0.159947f, -2.00957f, -0.159947f, 
		-2.00957f, -0.159947f, 0.564172f, 1.33354f, 
		0.553613f, 1.33390f, 0.553613f, 1.33390f, 
		0.564172f, 1.33359f, 0.553613f, 1.33390f, 
		0.553613f, 1.33390f, 0.553613f, 1.33390f, 
		-0.553613f, 1.33390f, -0.553613f, 1.33390f, 
		-0.553613f, 1.33390f, -0.553613f, 1.33390f, 
		-0.553613f, 1.33390f, -0.564184f, 1.33354f, 
		-0.564184f, 1.33358f, -2.75339f, 4.84510f, 
		-0.272066f, -0.251660f, -0.266084f, -0.248831f, 
		-0.272066f, -0.251660f, -0.272066f, -0.251660f, 
		-0.266084f, -0.248831f, -2.48057f, 3.84315f, 
		-2.90055f, 3.50067f, -2.90055f, 3.50067f, 
		4.85373f, -0.706987f, -1.98007f, -0.198106f, 
		-1.98007f, -0.198106f, -1.95748f, -0.167616f, 
		-2.04753f, -0.177084f, -1.95748f, -0.167616f, 
		-1.95748f, -0.167616f, 0.905605f, 0.968363f, 
		0.905605f, 0.968363f, 0.938436f, 0.946021f, 
		0.938436f, 0.946021f, 0.938436f, 0.946021f, 
		0.938436f, 0.946021f, 0.905605f, 0.968363f, 
		0.905605f, 0.968363f, -0.895556f, 1.00396f, 
		-0.895556f, 1.00396f, -0.895556f, 1.00396f, 
		-0.269120f, -0.356777f, -0.282224f, -0.361751f, 
		-0.271090f, -0.311240f, -0.260787f, -0.308051f, 
		-0.271090f, -0.311240f, -0.260787f, -0.308051f, 
		-0.269120f, -0.356777f, -0.282224f, -0.361751f, 
		-0.282224f, -0.361751f, -4.85357f, -0.707078f, 
		-0.929979f, -0.0845766f, -0.929979f, -0.0845766f, 
		-0.929979f, -0.0845766f, -0.929979f, -0.0845766f, 
		-0.934278f, -0.0778004f, -0.934278f, -0.0778004f, 
		-0.934278f, -0.0778004f, -0.934278f, -0.0778004f, 
		-0.934278f, -0.0778004f, -0.929979f, -0.0845766f, 
		-0.929979f, -0.0845766f, -1.28836f, -0.870517f, 
		-1.28836f, -0.870517f, -1.28836f, -0.870517f, 
		-1.28836f, -0.870517f, -1.28836f, -0.870517f, 
		0.890847f, 0.974417f, 0.901244f, 0.992117f, 
		0.901244f, 0.992117f, 0.901244f, 0.992117f, 
		0.890847f, 0.974417f, 0.890847f, 0.974417f, 
		-1.17474f, 0.849118f, -1.17474f, 0.849118f, 
		-1.29665f, -0.857629f, -1.29665f, -0.857629f, 
		-1.26923f, 0.864320f, -1.26923f, 0.864320f, 
		-1.26923f, 0.864320f, -1.26923f, 0.864320f, 
		-0.864172f, 0.927644f, -0.864172f, 0.927644f, 
		-0.864172f, 0.927644f, -0.864172f, 0.927644f, 
		-1.28558f, 0.841371f, -1.28558f, 0.841371f, 
		-1.28558f, 0.841371f, -1.28558f, 0.841371f, 
		-1.56546f, 0.389573f, -1.58205f, 0.381416f, 
		-1.58205f, 0.381416f, -1.58205f, 0.381416f, 
		-1.56546f, 0.389573f, -0.893363f, 0.977617f, 
		-0.903857f, 0.992301f, -0.903857f, 0.992301f, 
		-0.903857f, 0.992301f, -0.893363f, 0.977617f, 
		-0.893363f, 0.977617f, -0.905592f, 0.991775f, 
		-0.905592f, 0.991775f, -0.944108f, 1.00374f, 
		-0.944108f, 1.00374f, -0.944108f, 1.00374f, 
		-0.905592f, 0.991775f, -0.944108f, 1.00374f, 
		-0.905592f, 0.991775f, -0.730456f, 0.836357f, 
		-0.730456f, 0.836357f, -0.730456f, 0.836357f, 
		-0.730456f, 0.836357f, -0.730456f, 0.836357f, 
		-1.33007f, 0.845271f, -1.33007f, 0.845271f, 
		-1.33007f, 0.845271f, -1.33007f, 0.845271f, 
		-1.33007f, 0.845271f, -1.33021f, 0.845177f, 
		-1.33021f, 0.845177f, -1.33021f, 0.845177f, 
		-1.33021f, 0.845177f, -0.864043f, 0.927629f, 
		-0.864043f, 0.927629f, -0.864043f, 0.927629f, 
		-0.864043f, 0.927629f, -0.864043f, 0.927629f, 
		-1.58204f, 0.381385f, -1.58204f, 0.381385f, 
		-1.56546f, 0.389540f, -0.576835f, 1.33390f, 
		-0.576835f, 1.33390f, -0.576835f, 1.33390f, 
		-0.576835f, 1.33390f, -0.576835f, 1.33390f, 
		-0.576835f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, -1.05008f, -0.952723f, 
		-1.05008f, -0.952723f, -1.05008f, -0.952723f, 
		-1.05008f, -0.952723f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, -0.730410f, 0.912288f, 
		-0.730410f, 0.912288f, -0.730410f, 0.912288f, 
		-0.582573f, 2.64415f, -0.569241f, 2.62760f, 
		-0.569241f, 2.62760f, -0.569241f, 2.62760f, 
		-0.582573f, 2.64415f, -0.582573f, 2.64415f, 
		-2.25713f, -0.765499f, -2.25713f, -0.765499f, 
		-2.25713f, -0.765499f, -2.25713f, -0.765499f, 
		-2.25713f, -0.765499f, -0.551025f, 2.63042f, 
		-0.551025f, 2.63042f, -0.551744f, 2.62757f, 
		-0.551744f, 2.62757f, -0.551744f, 2.62757f, 
		-0.551744f, 2.62757f, -0.551025f, 2.63042f, 
		-0.551025f, 2.63042f, 0.590355f, 2.62023f, 
		0.579146f, 2.64873f, 0.579146f, 2.64873f, 
		0.579146f, 2.64873f, 0.590355f, 2.62023f, 
		0.590355f, 2.62023f, -2.39562f, 0.783948f, 
		-2.39562f, 0.783948f, -2.29995f, 0.819531f, 
		-2.29995f, 0.819531f, -2.29995f, 0.819531f, 
		-2.39562f, 0.783948f, -0.324804f, 0.462698f, 
		-0.270289f, 0.355129f, -0.272245f, 0.307374f, 
		-0.281706f, 0.311126f, -0.281706f, 0.311126f, 
		-0.281706f, 0.311126f, -0.272245f, 0.307374f, 
		-0.281706f, 0.311126f, -0.324804f, 0.462698f, 
		-0.324804f, 0.462698f, -0.270289f, 0.355129f, 
		0.566780f, 2.63734f, 0.566780f, 2.63734f, 
		0.586718f, 2.62764f, 0.586718f, 2.62764f, 
		0.586718f, 2.62764f, 0.586718f, 2.62764f, 
		0.586718f, 2.62764f, 0.566780f, 2.63734f, 
		0.566780f, 2.63734f, 0.571886f, 2.62561f, 
		0.549482f, 2.63403f, 0.549482f, 2.63403f, 
		0.549482f, 2.63403f, 0.571886f, 2.62561f, 
		0.571886f, 2.62561f, -0.276348f, -0.358262f, 
		-0.282216f, -0.361763f, -0.312072f, -0.401648f, 
		-0.260787f, -0.308080f, -0.312072f, -0.401648f, 
		-0.312072f, -0.401648f, -0.260787f, -0.308080f, 
		-0.276348f, -0.358262f, -0.282216f, -0.361763f, 
		-0.282216f, -0.361763f, -2.25652f, -0.766244f, 
		-2.25652f, -0.766244f, -2.25652f, -0.766244f, 
		-2.25652f, -0.766244f, -0.536267f, 2.66098f, 
		-0.536267f, 2.66098f, -0.536267f, 2.66098f, 
		-0.535760f, 2.66323f, -0.535760f, 2.66323f, 
		-0.535760f, 2.66323f, -0.535760f, 2.66323f, 
		-0.535760f, 2.66323f, -0.536267f, 2.66098f, 
		-1.86704f, 0.292793f, -2.07209f, 0.275762f, 
		-1.93909f, 0.313593f, -1.93909f, 0.313593f, 
		-1.93909f, 0.313593f, -1.86704f, 0.292793f, 
		0.590808f, 0.272928f, 0.735302f, 0.237786f, 
		0.735302f, 0.237786f, 0.679734f, 0.254336f, 
		0.590808f, 0.272928f, 0.679734f, 0.254336f, 
		0.679734f, 0.254336f, 0.679734f, 0.254336f, 
		0.590808f, 0.272928f, 0.590808f, 0.272928f, 
		0.735302f, 0.237786f, 0.735302f, 0.237786f, 
		-2.25020f, -0.763315f, -2.18510f, -0.723048f, 
		-2.18510f, -0.723048f, -2.16143f, -0.708582f, 
		-2.25020f, -0.763315f, -2.16146f, -0.704565f, 
		4.85679f, -0.707072f, -3.42454f, 0.704673f, 
		-3.42454f, 0.704673f, -3.42310f, 0.704582f, 
		-3.42310f, 0.704582f, -3.42310f, 0.704582f, 
		-3.43379f, 0.704448f, -3.43379f, 0.704448f, 
		-3.42454f, 0.704673f, -3.42454f, 0.704673f, 
		-3.42454f, 0.704673f, -3.42310f, 0.704582f, 
		-3.42310f, 0.704582f, -3.42310f, 0.704582f, 
		-3.43379f, 0.704448f, -3.43379f, 0.704448f, 
		-3.42454f, 0.704673f, -3.42454f, 0.704699f, 
		-3.42454f, 0.704699f, -3.42313f, 0.704608f, 
		-3.42313f, 0.704608f, -3.42313f, 0.704608f, 
		-3.43379f, 0.704474f, -3.43379f, 0.704474f, 
		-3.42454f, 0.704699f, -0.861359f, 1.26777f, 
		-0.840465f, 1.25993f, -0.861359f, 1.26777f, 
		-0.861359f, 1.26777f, -0.840465f, 1.25993f, 
		-0.840465f, 1.25993f, -3.42453f, 0.707264f, 
		-3.41747f, 0.707172f, -3.41747f, 0.707172f, 
		-3.43379f, 0.707045f, -3.43379f, 0.707045f, 
		-3.42453f, 0.707264f, -1.84303f, 0.332419f, 
		-1.92367f, 0.297947f, -2.05903f, 0.308702f, 
		-2.05903f, 0.308702f, -2.05903f, 0.308702f, 
		-2.05903f, 0.308702f, -2.05903f, 0.308702f, 
		-2.05903f, 0.308702f, -2.07596f, 0.275945f, 
		-1.92367f, 0.297947f, -1.84303f, 0.332419f, 
		-0.851536f, 1.27921f, -0.851536f, 1.27921f, 
		-0.904197f, 1.25783f, -0.904197f, 1.25783f, 
		-0.904197f, 1.25783f, -0.904197f, 1.25783f, 
		-0.851536f, 1.27921f, -0.851536f, 1.27921f, 
		-3.26339f, 0.717651f, -1.90469f, 0.343027f, 
		-1.89557f, 0.293979f, -2.05903f, 0.308702f, 
		-1.92022f, 0.315093f, -1.92022f, 0.315093f, 
		-1.89557f, 0.293979f, -2.05927f, 0.329155f, 
		-0.590816f, 0.272928f, -0.590816f, 0.272928f, 
		-0.590816f, 0.272928f, -0.590816f, 0.272928f, 
		-0.590816f, 0.272928f, -0.590816f, 0.272928f, 
		-0.590816f, 0.272928f, -0.590816f, 0.788046f, 
		-0.590816f, 0.788046f, -0.590816f, 0.788046f, 
		-0.590816f, 0.788046f, -0.590816f, 0.788046f, 
		-0.590816f, 0.788046f, -2.29862f, -0.760850f, 
		-2.29862f, -0.760850f, -3.17899f, 0.713785f, 
		-3.17899f, 0.713785f, -3.17899f, 0.713785f, 
		-3.40538f, 0.707444f, -3.40538f, 0.707444f, 
		-3.41747f, 0.707172f, -3.41747f, 0.707172f, 
		-3.41747f, 0.707172f, -3.43379f, 0.707045f, 
		-3.43379f, 0.707045f, -3.40538f, 0.707444f, 
		-3.42148f, -0.707217f, -3.41681f, -0.707414f, 
		-3.41681f, -0.707414f, -3.42148f, -0.707217f, 
		-3.43379f, -0.706958f, -3.43379f, -0.706958f, 
		-2.29857f, -0.760316f, -2.29857f, -0.760316f, 
		-0.271033f, -0.316732f, -0.345366f, -0.453011f, 
		-0.271090f, -0.332805f, -0.268070f, -0.332203f, 
		-0.271090f, -0.332805f, -0.271090f, -0.332805f, 
		-0.268070f, -0.332203f, -0.271033f, -0.316732f, 
		-0.345366f, -0.453011f, -0.345366f, -0.453011f, 
		-0.345366f, -0.453011f, -0.271013f, -0.316742f, 
		-0.345381f, -0.453063f, -0.271090f, -0.332805f, 
		-0.268070f, -0.332203f, -0.271090f, -0.332805f, 
		-0.268070f, -0.332203f, -0.271013f, -0.316742f, 
		-0.345381f, -0.453063f, -0.271013f, -0.316742f, 
		-0.345381f, -0.453063f, -0.345381f, -0.453063f, 
		-0.271010f, -0.316756f, -0.345372f, -0.453045f, 
		-0.345372f, -0.453045f, -0.271090f, -0.332805f, 
		-0.268070f, -0.332203f, -0.271090f, -0.332805f, 
		-0.271090f, -0.332805f, -0.268070f, -0.332203f, 
		-0.271010f, -0.316756f, -0.345372f, -0.453045f, 
		-0.271010f, -0.316756f, -0.345372f, -0.453045f, 
		-0.345372f, -0.453045f, 20.8026f, -11.7126f, 
		20.8026f, -11.7126f, 20.8026f, -11.7126f, 
		20.8026f, -11.7126f, -3.39785f, -0.714264f, 
		-3.42243f, -0.713538f, -3.42243f, -0.713538f, 
		-3.42243f, -0.713538f, -3.39785f, -0.714264f, 
		-3.43379f, -0.713216f, -3.39785f, -0.714264f, 
		-3.43379f, -0.713216f, -0.386134f, 0.522384f, 
		-0.279637f, 0.267345f, -0.386134f, 0.522384f, 
		-0.279637f, 0.267345f, -0.386134f, 0.522384f, 
		-0.386134f, 0.522384f, -1.14016f, -0.919844f, 
		-1.14016f, -0.919844f, -1.14016f, -0.919844f, 
		-1.14016f, -0.919844f, -1.14016f, -0.919844f, 
		-1.14016f, -0.919844f, -1.14019f, -0.919879f, 
		-1.14019f, -0.919879f, -1.14019f, -0.919879f, 
		-1.14019f, -0.919879f, -1.14019f, -0.919879f, 
		-1.14019f, -0.919879f, -1.14019f, -0.919866f, 
		-1.14019f, -0.919866f, -1.14019f, -0.919866f, 
		-1.14019f, -0.919866f, -1.14019f, -0.919866f, 
		-1.14019f, -0.919866f, -2.09195f, -0.418567f, 
		-2.09195f, -0.418567f, -2.09195f, -0.418567f, 
		-2.09195f, -0.418567f, 21.9538f, -8.50122f, 
		21.9538f, -8.50122f, 21.9538f, -8.50122f, 
		22.2027f, -7.65208f, 22.2027f, -7.65208f, 
		0.870086f, 0.852147f, 0.816224f, 0.825642f, 
		0.816224f, 0.825642f, 0.870086f, 0.852147f, 
		21.1759f, -10.9921f, 21.1759f, -10.9921f, 
		21.1759f, -10.9921f, 21.1759f, -10.9921f, 
		21.1759f, -10.9921f, 12.7131f, -17.0430f, 
		12.7131f, -17.0430f, 22.1922f, 6.38280f, 
		22.1922f, 6.38280f, 22.1922f, 6.38280f, 
		22.1922f, 6.38280f, 13.0055f, -16.9601f, 
		13.0055f, -16.9601f, 18.5288f, -14.3158f, 
		18.5288f, -14.3158f, 18.5288f, -14.3158f, 
		-0.261957f, 0.285271f, -0.393418f, 0.559012f, 
		-0.393418f, 0.559012f, -0.261957f, 0.285271f, 
		-0.393418f, 0.559012f, -0.393418f, 0.559012f, 
		18.2704f, -14.5357f, 18.2704f, -14.5357f, 
		18.2704f, -14.5357f, 18.2704f, -14.5357f, 
		-2.08710f, 0.432776f, -1.90469f, 0.343027f, 
		-1.97670f, 0.364002f, -1.89929f, 0.361803f, 
		-1.89929f, 0.361803f, -1.89929f, 0.361803f, 
		-1.97670f, 0.364002f, 2.52357f, 4.92016f, 
		13.0165f, -16.8302f, 13.0165f, -16.8302f, 
		13.0165f, -16.8302f, 13.0165f, -16.8302f, 
		16.8557f, -15.7236f, 16.8557f, -15.7236f, 
		16.8557f, -15.7236f, 16.8557f, -15.7236f, 
		16.8557f, -15.7236f, 16.8557f, -15.7236f, 
		-1.56818f, 0.409391f, -1.58364f, 0.405032f, 
		-1.58364f, 0.405032f, -1.56818f, 0.409391f, 
		-1.55673f, 0.312826f, -1.56191f, 0.308979f, 
		-1.55673f, 0.312826f, -1.56270f, 0.280295f, 
		-1.56818f, 0.381052f, -1.58364f, 0.405032f, 
		-1.55815f, 0.280914f, 4.64907f, -0.771653f, 
		4.64907f, -0.771653f, 4.64907f, -0.771653f, 
		3.94241f, -0.960301f, -2.06454f, -0.417325f, 
		-2.06454f, -0.417325f, -2.06454f, -0.417325f, 
		-2.06454f, -0.417325f, -2.06454f, -0.417325f, 
		0.490807f, -1.22459f, 0.490807f, -1.22459f, 
		0.490807f, -1.22459f, 0.490807f, -1.22459f, 
		0.490807f, -1.22459f, -0.456248f, 1.24365f, 
		-0.456248f, 1.24365f, -0.456248f, 1.24365f, 
		-0.456248f, 1.24365f, -2.06652f, -0.403637f, 
		-2.06652f, -0.403637f, -2.06652f, -0.403637f, 
		-2.06652f, -0.403637f, 22.3505f, 6.38280f, 
		22.3505f, 6.38280f, 22.3505f, 6.38280f, 
		22.3505f, 6.38280f, -2.03173f, 0.384979f, 
		-1.94466f, 0.312395f, -1.94466f, 0.312395f, 
		-1.94466f, 0.312395f, -1.94466f, 0.312395f, 
		-2.08359f, 0.376534f, -2.08359f, 0.376534f, 
		-2.08359f, 0.376534f, -2.08359f, 0.376534f, 
		-2.03173f, 0.384979f, -2.08359f, 0.376534f, 
		31.0837f, 11.0530f, 31.0837f, 11.0530f, 
		31.0837f, 11.0530f, 31.0837f, 11.0530f, 
		31.0837f, 11.0530f, 0.468445f, 1.24026f, 
		0.468445f, 1.24026f, 0.468445f, 1.24026f, 
		0.468445f, 1.24026f, 0.468445f, 1.24026f, 
		16.0180f, -16.1796f, 16.0180f, -16.1796f, 
		16.0180f, -16.1796f, 16.0180f, -16.1796f, 
		16.0180f, -16.1796f, -11.5987f, -16.9300f, 
		-11.5987f, -16.9300f, -11.5987f, -16.9300f, 
		28.7379f, 26.6275f, 28.7379f, 26.6275f, 
		28.7379f, 26.6275f, 28.7379f, 26.6275f, 
		4.67791f, 4.92753f, 4.67791f, 4.92753f, 
		4.67791f, 4.92753f, -0.269713f, 1.36523f, 
		-0.269713f, 1.36523f, -0.269713f, 1.36523f, 
		-0.269713f, 1.36523f, -11.5987f, -17.1436f, 
		-11.5987f, -17.1436f, -11.5987f, -17.1436f, 
		-11.5987f, -17.1436f, -0.354483f, 1.33218f, 
		-0.354483f, 1.33218f, -0.354483f, 1.33218f, 
		-0.354483f, 1.33218f, -0.354483f, 1.33218f, 
		-0.354483f, 1.33218f, 4.67791f, 4.92753f, 
		4.67791f, 4.92753f, 4.67791f, 4.92753f, 
		-11.5987f, -16.9300f, -11.5987f, -16.9300f, 
		-11.5987f, -16.9300f, 12.2672f, -16.8914f, 
		12.2672f, -16.8914f, 0.554852f, -1.18234f, 
		0.554852f, -1.18234f, 31.5520f, 25.3776f, 
		31.5520f, 25.3776f, 31.5760f, 25.3187f, 
		31.5760f, 25.3187f, 31.5760f, 25.3187f, 
		31.5760f, 25.3187f, 31.5760f, 25.3187f, 
		-11.5987f, -17.1436f, -11.5987f, -17.1436f, 
		-11.5987f, -17.1436f, -11.5987f, -17.1436f, 
		0.578939f, 0.788046f, 0.578939f, 0.788046f, 
		0.578939f, 0.788046f, 0.607312f, 0.792546f, 
		0.586660f, 0.789197f, 0.578939f, 0.788046f, 
		-0.573496f, 0.788046f, -0.573496f, 0.788046f, 
		-0.573496f, 0.788046f, -0.573496f, 0.788046f, 
		-0.573496f, 0.788046f, -0.573496f, 0.788046f, 
		-0.596265f, 0.791301f, -0.573496f, 0.788046f, 
		-0.573496f, 0.788046f, -0.596265f, 0.791278f, 
		2.70840f, 1.03643f, 2.89241f, 1.13296f, 
		29.2800f, 10.5881f, 29.2800f, 10.5881f, 
		30.8027f, 13.5278f, 30.8027f, 13.5278f, 
		30.8027f, 13.5278f, 30.8027f, 13.5278f, 
		0.0511938f, 0.0171769f, 0.0511938f, 0.0171769f, 
		0.0511938f, 0.0171769f, 0.101266f, 0.0203303f, 
		0.101266f, 0.0203303f, 0.0511938f, 0.0171769f, 
		0.101266f, 0.0203303f, 19.3861f, -13.7826f, 
		19.3861f, -13.7826f, 19.3861f, -13.7826f, 
		19.3861f, -13.7826f, 19.3861f, -13.7826f, 
		29.9508f, 10.5411f, 29.9508f, 10.5411f, 
		29.9508f, 10.5411f, 29.9508f, 10.5411f, 
		21.5678f, -9.75547f, 21.5678f, -9.75547f, 
		32.4147f, 25.5319f, 32.4147f, 25.5319f, 
		31.3761f, 24.5221f, 31.3761f, 24.5221f, 
		31.3761f, 24.5221f, -0.543775f, -1.25150f, 
		-0.543775f, -1.25150f, -0.543775f, -1.25150f, 
		-0.543775f, -1.25150f, -0.543775f, -1.25150f, 
		-0.543775f, -1.25150f, 30.9307f, 26.5967f, 
		30.9307f, 26.5967f, 2.73935f, 1.03643f, 
		2.83930f, 1.12843f, 2.83930f, 1.12843f, 
		2.83930f, 1.12843f, 20.4504f, -12.4571f, 
		20.4504f, -12.4571f, 20.4504f, -12.4571f, 
		-0.125850f, 0.263085f, -0.346671f, 0.405066f, 
		-0.346671f, 0.405066f, -0.346671f, 0.405066f, 
		-0.125850f, 0.263085f, -0.346671f, 0.405066f, 
		-0.346671f, 0.405066f, 22.1350f, 6.38280f, 
		22.1350f, 6.38280f, 22.1350f, 6.38280f, 
		5.00000f, 1.50000f, 5.00000f, 1.53446f, 
		28.3845f, 25.4205f, 28.3845f, 25.4205f, 
		-1.57096f, 0.306670f, -1.55815f, 0.273897f, 
		-1.56241f, 0.273368f, -1.55709f, 0.386255f, 
		-1.56241f, 0.273368f, -1.55709f, 0.386255f, 
		-1.58308f, 0.379402f, -1.57096f, 0.306670f, 
		-1.56509f, 0.314181f, -1.57096f, 0.306670f, 
		-1.56509f, 0.314181f, -5.00000f, 1.50000f, 
		-5.00000f, 1.53449f, -5.44907f, -0.250457f, 
		0.0610956f, -0.0444379f, 0.0610956f, -0.0444379f, 
		0.0661657f, 0.0196056f, 0.0661657f, 0.0196056f, 
		0.0610956f, -0.0444379f, 0.0610956f, -0.0444379f, 
		0.0661657f, 0.0196056f, 4.13870f, -4.06558f, 
		-4.13873f, -4.06556f, 0.0771493f, -0.00966728f, 
		0.0771493f, -0.00966728f, 0.0771493f, -0.00966728f, 
		2.75118f, 3.84315f, 3.00886f, 3.63893f, 
		4.11183f, -4.09245f, 0.0453271f, 0.0237823f, 
		0.0453271f, 0.0237823f, 0.0453271f, 0.0237823f, 
		0.111369f, 0.0202140f, 0.111369f, 0.0202140f, 
		31.5521f, 12.1262f, 31.5521f, 12.1262f, 
		31.4982f, 12.4545f, 31.4982f, 12.4545f, 
		31.4982f, 12.4545f, -5.44928f, -0.250297f, 
		3.23205f, 3.25755f, -0.726284f, -0.973178f, 
		-0.726284f, -0.973178f, -0.726284f, -0.973178f, 
		-0.726284f, -0.973178f, -0.726284f, -0.973178f, 
		-4.11186f, -4.09243f, 29.0418f, 10.7578f, 
		29.0418f, 10.7578f, 0.0824700f, -0.0222085f, 
		0.0824700f, -0.0222085f, 0.0917081f, 0.0369033f, 
		0.0917081f, 0.0369033f, 0.0824700f, -0.0222085f, 
		0.0917081f, 0.0369033f, -2.06876f, -0.483346f, 
		-2.06876f, -0.483346f, -2.06876f, -0.483346f, 
		-2.06876f, -0.483346f, -2.06876f, -0.483346f, 
		22.3959f, 6.38280f, 22.3959f, 6.38280f, 
		22.3959f, 6.38280f, 2.75114f, 3.84315f, 
		3.00891f, 3.63889f, 3.00891f, 3.63889f, 
		3.00891f, 3.63889f, 0.533117f, 1.19567f, 
		0.533117f, 1.19567f, 70.9560f, -40.1476f, 
		70.9560f, -40.1476f, 70.5560f, -40.1519f, 
		70.5560f, -40.1519f, 70.5859f, -37.5907f, 
		70.1964f, -37.6738f, 69.5318f, -35.0019f, 
		69.4263f, -19.3235f, 69.4263f, -19.3235f, 
		69.3909f, -19.1921f, 69.3909f, -19.1921f, 
		69.3665f, -19.1310f, 69.3665f, -19.1310f, 
		0.0614769f, 0.00859860f, 0.0614769f, 0.00859860f, 
		0.0614769f, 0.00859860f, 69.3857f, -22.4919f, 
		69.4263f, -22.3161f, 69.4263f, -22.3161f, 
		69.4054f, -22.4077f, 69.0284f, -18.9321f, 
		69.0284f, -18.9321f, 69.1138f, -18.9500f, 
		69.1138f, -18.9500f, 69.1138f, -18.9500f, 
		69.1138f, -18.9500f, 69.0284f, -18.9321f, 
		69.0284f, -18.9321f, 69.1724f, -35.1636f, 
		69.2805f, -22.6122f, 69.2805f, -22.6122f, 
		69.2269f, -22.6483f, 69.2269f, -22.6483f, 
		69.2269f, -22.6483f, 69.2269f, -22.6483f, 
		69.2269f, -22.6483f, 69.2805f, -22.6122f, 
		69.2805f, -22.6122f, 69.2805f, -22.6122f, 
		68.8769f, -22.6896f, 68.8769f, -22.6896f, 
		68.8769f, -22.6896f, 68.9623f, -22.7075f, 
		68.9623f, -22.7075f, 68.9623f, -22.7075f, 
		68.6029f, -19.1834f, 68.5827f, -19.2459f, 
		68.6029f, -22.4562f, 68.6029f, -22.4562f, 
		68.6029f, -22.4562f, 68.6277f, -22.5166f, 
		68.6277f, -22.5166f, 68.6277f, -22.5166f, 
		68.7517f, -18.9905f, 68.7517f, -18.9905f, 
		68.7953f, -18.9681f, 68.7953f, -18.9681f, 
		68.7953f, -18.9681f, 68.7953f, -18.9681f, 
		68.7517f, -18.9905f, 68.7517f, -18.9905f, 
		68.2334f, -22.2044f, 68.2334f, -22.2044f, 
		68.2334f, -22.2044f, 68.1449f, -22.2135f, 
		68.1449f, -22.2135f, 68.1449f, -22.2135f, 
		68.2334f, -22.2044f, 68.2334f, -22.2044f, 
		68.1449f, -19.4261f, 68.1449f, -19.4261f, 
		68.1449f, -19.4261f, 68.0711f, -19.4227f, 
		68.0711f, -19.4227f, 68.0711f, -19.4227f, 
		67.6485f, -18.9765f, 67.6437f, -18.9349f, 
		67.6485f, -22.6631f, 67.6533f, -22.6215f, 
		67.7855f, -19.5192f, 67.8285f, -19.4902f, 
		67.8285f, -19.4902f, 67.7855f, -19.5192f, 
		67.7855f, -19.5192f, 67.7855f, -19.5192f, 
		67.5657f, -32.7720f, 67.8781f, -32.5361f, 
		67.5366f, -22.9669f, 67.5582f, -22.9328f, 
		67.5149f, -18.6386f, 67.5366f, -18.6727f, 
		-3.12957f, 3.51091f, -3.12957f, 3.51091f, 
		-3.12957f, 3.51091f, 67.3098f, -23.1899f, 
		67.3440f, -23.1682f, 67.2755f, -18.4278f, 
		67.3098f, -18.4496f, 66.9434f, -23.3000f, 
		67.0026f, -23.2956f, 66.9435f, -18.3396f, 
		66.8736f, -18.3353f, 66.6532f, -23.2869f, 
		66.6532f, -23.2869f, 66.9570f, -19.4116f, 
		66.9570f, -19.4116f, 66.6532f, -18.3527f, 
		66.6532f, -18.3527f, 66.9570f, -22.4116f, 
		66.9570f, -22.4116f, 0.553801f, -1.20156f, 
		0.553801f, -1.20156f, 0.553797f, -1.20158f, 
		0.553797f, -1.20158f, 66.2058f, -18.6727f, 
		66.1842f, -18.7067f, 66.2275f, -23.0009f, 
		66.2058f, -22.9669f, 66.4326f, -18.4496f, 
		66.3984f, -18.4714f, 66.4326f, -23.1899f, 
		66.4669f, -23.2117f, 29.7688f, 26.9008f, 
		29.7688f, 26.9008f, 65.7093f, -30.3485f, 
		65.4585f, -30.6496f, 62.9325f, -28.9471f, 
		63.1098f, -28.5940f, 0.522040f, 1.00000f, 
		0.829390f, 0.500000f, 0.522040f, 0.000000f, 
		0.495526f, 1.00000f, 0.855905f, 0.500000f, 
		0.495526f, 0.000000f, 0.481869f, 1.00000f, 
		0.869561f, 0.500000f, 0.481869f, 0.000000f, 
		0.468500f, 1.00000f, 0.882930f, 0.500000f, 
		0.468500f, 0.000000f, 0.455342f, 1.00000f, 
		0.896088f, 0.500000f, 0.455342f, 0.000000f, 
		0.547731f, 1.00000f, 0.803700f, 0.500000f, 
		0.547731f, 0.000000f, 0.442334f, 1.00000f, 
		0.909097f, 0.500000f, 0.442334f, 0.000000f, 
		0.429425f, 1.00000f, 0.922005f, 0.500000f, 
		0.429425f, 0.000000f, 5.80441f, 0.140725f, 
		0.573028f, 1.00000f, 0.778402f, 0.500000f, 
		0.573028f, 0.000000f, 0.416577f, 1.00000f, 
		0.934854f, 0.500000f, 0.416577f, 0.000000f, 
		0.403754f, 1.00000f, 0.396022f, 1.00000f, 
		0.951152f, 0.500000f, 0.400278f, 0.000000f, 
		-2.06876f, 0.483338f, -2.06876f, 0.483338f, 
		-2.06876f, 0.483338f, -2.06876f, 0.483338f, 
		0.598212f, 1.00000f, 0.753218f, 0.500000f, 
		0.598212f, 0.000000f, 0.388361f, 1.00000f, 
		0.380804f, 1.00000f, 0.966510f, 0.500000f, 
		0.384920f, 0.000000f, 0.358748f, 1.00000f, 
		0.351430f, 1.00000f, 0.000000f, 0.500000f, 
		0.996290f, 0.500000f, 0.355141f, 0.000000f, 
		0.249501f, 1.00000f, 0.242872f, 1.00000f, 
		0.105181f, 0.500000f, 0.246249f, 1.00000f, 
		0.246249f, 0.000000f, 0.373365f, 1.00000f, 
		0.366030f, 1.00000f, 0.981515f, 0.500000f, 
		0.369916f, 0.000000f, 0.236450f, 1.00000f, 
		0.230125f, 1.00000f, 0.117846f, 0.500000f, 
		0.233584f, 1.00000f, 0.233584f, 0.000000f, 
		0.853553f, 1.26082f, 0.853553f, 1.26082f, 
		0.853553f, 1.26082f, 0.880192f, 1.25902f, 
		0.880192f, 1.25902f, 0.880192f, 1.25902f, 
		0.880192f, 1.25902f, 0.880192f, 1.25902f, 
		0.880192f, 1.25902f, 0.853553f, 1.26082f, 
		0.853553f, 1.26082f, 0.853553f, 1.26082f, 
		0.623510f, 1.00000f, 0.727920f, 0.500000f, 
		0.623510f, 0.000000f, -0.658925f, -0.272928f, 
		-0.901138f, -0.177187f, -0.760860f, -0.272928f, 
		-0.901138f, -0.177187f, -0.954870f, -0.0948437f, 
		-0.954870f, -0.0948437f, 0.675715f, 1.00000f, 
		0.675715f, 0.750000f, 0.675715f, 0.833333f, 
		0.675715f, 0.583333f, 0.675715f, 0.666667f, 
		0.675715f, 0.416667f, 0.675715f, 0.500000f, 
		0.675715f, 0.916667f, 0.675715f, 0.250000f, 
		0.675715f, 0.333333f, 0.675715f, 0.0833330f, 
		0.675715f, 0.166667f, 56.9560f, -27.0043f, 
		21.9917f, -7.64877f, 21.9917f, -7.64877f, 
		21.9917f, -7.64877f, 0.133925f, 1.00000f, 
		0.127601f, 1.00000f, 0.130246f, 0.000000f, 
		0.221184f, 0.500000f, 0.114981f, 1.00000f, 
		0.121305f, 1.00000f, 0.117846f, 0.000000f, 
		0.233585f, 0.500000f, 0.623510f, 0.500000f, 
		0.727920f, 1.00000f, 0.727920f, 0.000000f, 
		0.108558f, 1.00000f, 0.105183f, 0.000000f, 
		0.101929f, 1.00000f, 0.246247f, 0.500000f, 
		0.992683f, 1.00000f, 0.996290f, 1.00000f, 
		0.000000f, 0.000000f, 0.000000f, 1.00000f, 
		0.355140f, 0.500000f, 0.996290f, 0.000000f, 
		0.985401f, 1.00000f, 0.978065f, 1.00000f, 
		0.981515f, 1.00000f, 0.369916f, 0.500000f, 
		0.981515f, 0.000000f, 0.970626f, 1.00000f, 
		0.963070f, 1.00000f, 0.966510f, 1.00000f, 
		0.384920f, 0.500000f, 0.966510f, 0.000000f, 
		0.598212f, 0.500000f, 0.753218f, 1.00000f, 
		0.753218f, 0.000000f, 0.955408f, 1.00000f, 
		0.951152f, 1.00000f, 0.947676f, 1.00000f, 
		0.400278f, 0.500000f, 0.951152f, 0.000000f, 
		0.573028f, 0.500000f, 0.778402f, 1.00000f, 
		0.778402f, 0.000000f, 0.922005f, 1.00000f, 
		0.429425f, 0.500000f, 0.922005f, 0.000000f, 
		0.934854f, 1.00000f, 0.416577f, 0.500000f, 
		0.934854f, 0.000000f, 0.547731f, 0.500000f, 
		0.803700f, 1.00000f, 0.803700f, 0.000000f, 
		0.896088f, 1.00000f, 0.455342f, 0.500000f, 
		0.896088f, 0.000000f, 0.909097f, 1.00000f, 
		0.442334f, 0.500000f, 0.909097f, 0.000000f, 
		0.829390f, 1.00000f, 0.522040f, 0.500000f, 
		0.829390f, 0.000000f, 0.855905f, 1.00000f, 
		0.495526f, 0.500000f, 0.855905f, 0.000000f, 
		0.869561f, 1.00000f, 0.481869f, 0.500000f, 
		0.869561f, 0.000000f, 0.882930f, 1.00000f, 
		0.468500f, 0.500000f, 0.882930f, 0.000000f, 
		4.81199f, -2.97431f, 56.9524f, -27.4043f, 
		56.9536f, -27.4043f, 56.9557f, -27.4043f, 
		56.9597f, -27.4043f, 56.9594f, -27.4043f, 
		56.9588f, -27.4043f, 56.9568f, -27.4043f, 
		56.9546f, -27.4043f, 56.9529f, -27.4043f, 
		31.8107f, -23.2392f, 31.8107f, -23.2392f, 
		31.8887f, -23.3024f, 29.1641f, 13.4961f, 
		29.1641f, 13.4961f, 3.23222f, 3.25721f, 
		0.0411242f, -0.0458502f, 0.0411242f, -0.0458502f, 
		0.0411242f, -0.0458502f, 31.4681f, -18.9196f, 
		31.6498f, -18.7145f, 31.6498f, -18.7145f, 
		31.6498f, -18.7145f, 31.8122f, -18.5837f, 
		31.8122f, -18.5837f, 0.491119f, 1.22645f, 
		0.491119f, 1.22645f, 0.504795f, 1.19767f, 
		0.504795f, 1.19767f, -4.81198f, -2.97432f, 
		5.50000f, -1.50000f, 5.50000f, -1.50000f, 
		5.50000f, 1.50000f, 5.50000f, 1.50000f, 
		5.50000f, 1.50000f, 5.50000f, 1.50000f, 
		4.92016f, -2.67198f, 28.4460f, -18.9196f, 
		28.1019f, -18.5837f, 28.1019f, -18.5837f, 
		28.2643f, -18.7146f, 4.69278f, -3.30746f, 
		5.50000f, -1.50000f, 5.50000f, -1.50000f, 
		5.50000f, 1.50000f, 5.50000f, 1.50000f, 
		30.2602f, 26.8422f, 30.2602f, 26.8422f, 
		30.2602f, 26.8422f, 32.6960f, -18.6716f, 
		32.6542f, -18.6365f, 32.6960f, -23.1516f, 
		32.7379f, -23.1165f, 32.9347f, -22.7971f, 
		32.9523f, -22.7433f, 32.9347f, -19.0261f, 
		32.9172f, -18.9723f, 32.2405f, -18.5015f, 
		32.2405f, -18.5015f, 32.2405f, -18.5015f, 
		32.1656f, -18.4935f, 0.331960f, -2.42634f, 
		0.293082f, -2.44025f, 0.592613f, -2.24251f, 
		0.564118f, -2.27100f, -0.0722382f, -2.48019f, 
		-0.00236730f, -2.48452f, -0.438561f, -2.37015f, 
		-0.404319f, -2.39193f, -0.643711f, -2.18115f, 
		-0.665370f, -2.14710f, 2.36785f, -1.82925f, 
		2.36785f, -1.82925f, 2.32424f, -1.85164f, 
		2.32424f, -1.85164f, 2.32424f, -1.85164f, 
		2.32424f, -1.85164f, 2.36785f, -1.82925f, 
		2.36785f, -1.82925f, 2.09113f, -1.88769f, 
		2.09113f, -1.88769f, 2.00572f, -1.86976f, 
		2.00572f, -1.86976f, 2.00572f, -1.86976f, 
		2.00572f, -1.86976f, 2.09113f, -1.88769f, 
		2.09113f, -1.88769f, 32.2406f, -23.3217f, 
		32.3155f, -23.3138f, 2.51971f, -1.62774f, 
		2.55514f, -1.49634f, 2.51971f, -1.62774f, 
		2.53961f, -1.56451f, 2.55514f, -1.49634f, 
		2.53961f, -1.56451f, 1.73171f, -1.63638f, 
		1.73171f, -1.63638f, 1.73171f, -1.63638f, 
		1.73171f, -1.63638f, 1.75648f, -1.69678f, 
		1.75648f, -1.69678f, 31.2016f, -24.6468f, 
		31.2016f, -24.6468f, 31.3509f, -24.4240f, 
		31.3509f, -24.4240f, 31.4391f, -24.1714f, 
		31.4391f, -24.1714f, 31.4681f, -23.8965f, 
		31.4681f, -23.8965f, 31.4681f, -17.9267f, 
		31.4681f, -17.9267f, 31.4391f, -17.6518f, 
		31.4391f, -17.6518f, 31.3509f, -17.3992f, 
		31.3509f, -17.3992f, 31.2016f, -17.1764f, 
		31.2016f, -17.1764f, 1.57944f, -1.27120f, 
		1.57944f, -1.27120f, 1.57944f, -1.27120f, 
		1.61767f, -1.23042f, 1.61767f, -1.23042f, 
		1.61767f, -1.23042f, 30.7125f, -24.9744f, 
		30.7125f, -24.9744f, 30.9894f, -24.8327f, 
		30.9894f, -24.8327f, 30.9894f, -16.9905f, 
		30.9894f, -16.9905f, 30.7125f, -16.8488f, 
		30.7125f, -16.8488f, 30.3690f, -25.0648f, 
		30.3690f, -25.0648f, 30.3690f, -16.7584f, 
		30.3690f, -16.7584f, 3.34100f, 2.01780f, 
		3.34100f, 2.01780f, 3.34100f, 2.01780f, 
		29.9571f, -25.0965f, 29.9571f, -25.0965f, 
		29.9571f, -16.7267f, 29.9571f, -16.7267f, 
		30.9208f, 26.7793f, 30.9208f, 26.7793f, 
		30.9208f, 26.7793f, 30.9208f, 26.7793f, 
		30.9208f, 26.7793f, 30.9208f, 26.7793f, 
		-0.532691f, 1.19009f, -0.532691f, 1.19009f, 
		29.5451f, -25.0648f, 29.5451f, -25.0648f, 
		29.5451f, -16.7584f, 29.5451f, -16.7584f, 
		29.2016f, -24.9744f, 29.2016f, -24.9744f, 
		28.9247f, -24.8327f, 28.9247f, -24.8327f, 
		28.9247f, -16.9905f, 28.9247f, -16.9905f, 
		29.2016f, -16.8488f, 29.2016f, -16.8488f, 
		28.7125f, -24.6468f, 28.7125f, -24.6468f, 
		28.5632f, -24.4240f, 28.5632f, -24.4239f, 
		28.4750f, -24.1714f, 28.4750f, -24.1714f, 
		28.4460f, -23.8965f, 28.4460f, -23.8965f, 
		28.4460f, -17.9267f, 28.4460f, -17.9267f, 
		28.4750f, -17.6518f, 28.4750f, -17.6518f, 
		28.5632f, -17.3992f, 28.5632f, -17.3992f, 
		28.7125f, -17.1764f, 28.7125f, -17.1764f, 
		1.27369f, 1.39373f, 1.27369f, 1.39373f, 
		1.27369f, 1.39373f, 1.19995f, 1.39707f, 
		1.19995f, 1.39707f, 1.19995f, 1.39707f, 
		0.957221f, 1.32960f, 0.957221f, 1.32960f, 
		0.914238f, 1.30057f, 0.914238f, 1.30057f, 
		0.914238f, 1.30057f, 2.55514f, 1.49634f, 
		2.49526f, 1.68882f, 2.55514f, 1.49634f, 
		2.51972f, 1.62774f, 2.51972f, 1.62774f, 
		2.49526f, 1.68882f, 2.49526f, 1.68882f, 
		1.73171f, 1.63639f, 1.71148f, 1.57389f, 
		27.6736f, -18.5015f, 27.6736f, -18.5015f, 
		27.6736f, -18.5015f, 27.5987f, -18.5094f, 
		2.15723f, 1.88769f, 2.15723f, 1.88769f, 
		2.24264f, 1.86976f, 2.24264f, 1.86976f, 
		2.24264f, 1.86976f, 2.24264f, 1.86976f, 
		2.15723f, 1.88769f, 2.15723f, 1.88769f, 
		1.88051f, 1.82925f, 1.88051f, 1.82925f, 
		1.92412f, 1.85164f, 1.92412f, 1.85164f, 
		1.92412f, 1.85164f, 1.92412f, 1.85164f, 
		1.88051f, 1.82925f, 1.88051f, 1.82925f, 
		0.772509f, 1.88489f, 0.777304f, 1.84334f, 
		-0.665364f, 2.14711f, -0.687022f, 2.11306f, 
		0.665364f, 2.14711f, 0.643706f, 2.18115f, 
		-0.0722396f, 2.48019f, -0.131375f, 2.47585f, 
		-0.438557f, 2.37015f, -0.472799f, 2.34837f, 
		0.218016f, 2.46711f, 0.218016f, 2.46711f, 
		0.404318f, 2.39193f, 0.438560f, 2.37015f, 
		26.9794f, -19.0261f, 26.9618f, -19.0799f, 
		26.9794f, -22.7971f, 26.9969f, -22.8509f, 
		27.6735f, -23.3217f, 27.7484f, -23.3296f, 
		27.2181f, -18.6716f, 27.1763f, -18.7067f, 
		27.2181f, -23.1516f, 27.2599f, -23.1867f, 
		0.843031f, 1.26688f, 0.843031f, 1.26688f, 
		0.856217f, 1.26034f, 0.856217f, 1.26034f, 
		0.856217f, 1.26034f, 0.856217f, 1.26034f, 
		0.856217f, 1.26034f, 0.843031f, 1.26688f, 
		-0.405825f, -0.591307f, -0.405825f, -0.591307f, 
		-0.261534f, -0.286671f, -0.405825f, -0.591307f, 
		-0.405825f, -0.591307f, -0.261534f, -0.286671f, 
		-2.64214f, -4.92016f, -5.50000f, -1.50000f, 
		-5.50000f, -1.50000f, -5.50000f, 1.50000f, 
		-5.50000f, 1.50000f, 1.20651f, -1.38475f, 
		1.20651f, -1.38475f, 1.20651f, -1.38475f, 
		1.32961f, -1.38451f, 1.32961f, -1.38451f, 
		1.32961f, -1.38451f, -4.69278f, -3.30746f, 
		-4.92016f, -2.67198f, -5.50000f, -1.50000f, 
		-5.50000f, -1.50000f, -5.50000f, -1.50000f, 
		-5.50000f, 1.50000f, -5.50000f, 1.50000f, 
		-5.50000f, 1.50000f, -5.50000f, 1.50000f, 
		4.81195f, 2.97441f, 28.4460f, -22.9036f, 
		28.1033f, -23.2393f, 28.2644f, -23.1089f, 
		28.1033f, -23.2393f, 27.6125f, 25.6218f, 
		27.6125f, 25.6218f, 5.80435f, 0.140634f, 
		2.97027f, -27.4043f, 2.96416f, -27.4043f, 
		2.96450f, -27.4043f, 2.97153f, -27.4043f, 
		2.96602f, -27.4043f, 2.96707f, -27.4043f, 
		2.96930f, -27.4043f, 2.97104f, -27.4043f, 
		-4.81195f, 2.97439f, 0.829390f, 1.00000f, 
		0.522040f, 0.500000f, 0.829390f, 0.000000f, 
		0.495526f, 0.500000f, 0.855905f, 1.00000f, 
		0.855905f, 0.000000f, 0.481869f, 0.500000f, 
		0.869561f, 1.00000f, 0.869561f, 0.000000f, 
		0.468500f, 0.500000f, 0.882930f, 1.00000f, 
		0.882930f, 0.000000f, 0.803700f, 1.00000f, 
		0.547731f, 0.500000f, 0.803700f, 0.000000f, 
		0.455342f, 0.500000f, 0.896088f, 1.00000f, 
		0.896088f, 0.000000f, 0.909097f, 1.00000f, 
		0.442334f, 0.500000f, 0.909097f, 0.000000f, 
		0.778402f, 1.00000f, 0.573028f, 0.500000f, 
		0.778402f, 0.000000f, 0.922005f, 0.000000f, 
		0.922005f, 1.00000f, 0.429425f, 0.500000f, 
		0.934854f, 0.000000f, 0.934854f, 1.00000f, 
		0.416577f, 0.500000f, 0.400278f, 0.500000f, 
		0.955408f, 1.00000f, 0.951152f, 0.000000f, 
		0.947676f, 1.00000f, 0.753218f, 1.00000f, 
		0.598212f, 0.500000f, 0.753218f, 0.000000f, 
		0.384920f, 0.500000f, 0.963070f, 1.00000f, 
		0.970626f, 1.00000f, 0.966511f, 0.000000f, 
		0.369916f, 0.500000f, 0.978065f, 1.00000f, 
		0.985401f, 1.00000f, 0.981515f, 0.000000f, 
		0.355140f, 0.500000f, 0.992683f, 1.00000f, 
		0.996290f, 0.000000f, 0.000000f, 0.000000f, 
		0.000000f, 1.00000f, 0.108558f, 1.00000f, 
		0.246247f, 0.500000f, 0.105183f, 0.000000f, 
		0.101929f, 1.00000f, 0.823772f, 0.226074f, 
		0.812440f, 0.233356f, 0.812440f, 0.233356f, 
		0.812440f, 0.233356f, 0.823772f, 0.226074f, 
		0.823772f, 0.226074f, 0.727920f, 1.00000f, 
		0.623510f, 0.500000f, 0.727920f, 0.000000f, 
		0.121305f, 1.00000f, 0.233585f, 0.500000f, 
		0.114981f, 1.00000f, 0.117846f, 0.000000f, 
		0.0546171f, -0.0577720f, 0.0803521f, -0.0489615f, 
		0.0546171f, -0.0577720f, 0.0803521f, -0.0489615f, 
		0.0546171f, -0.0577720f, 0.0803521f, -0.0489615f, 
		0.675715f, 0.833333f, 0.675715f, 0.666667f, 
		0.675715f, 0.750000f, 0.675715f, 0.583333f, 
		0.675715f, 0.583333f, 0.675715f, 0.416667f, 
		0.675715f, 0.333333f, 0.675715f, 0.416667f, 
		0.675715f, 0.166667f, 0.675715f, 0.916667f, 
		0.675715f, 0.250000f, 0.675715f, 0.166667f, 
		0.675715f, 0.000000f, 0.675715f, 1.00000f, 
		2.96791f, -27.0043f, -0.658925f, -0.272928f, 
		-0.873720f, -0.179918f, -0.873720f, -0.179918f, 
		-0.817657f, -0.238030f, -0.760860f, -0.272928f, 
		-0.817657f, -0.238030f, 0.727920f, 0.500000f, 
		0.623510f, 0.000000f, 0.623510f, 1.00000f, 
		0.230125f, 1.00000f, 0.236450f, 1.00000f, 
		0.117845f, 0.500000f, 0.233585f, 0.000000f, 
		0.242872f, 1.00000f, 0.249501f, 1.00000f, 
		0.105183f, 0.500000f, 0.246247f, 0.000000f, 
		0.358748f, 1.00000f, 0.366030f, 1.00000f, 
		0.988905f, 0.500000f, 0.362526f, 0.000000f, 
		0.351430f, 1.00000f, 1.00000f, 0.500000f, 
		0.351430f, 0.000000f, 0.000000f, 0.500000f, 
		0.380804f, 1.00000f, 0.373365f, 1.00000f, 
		0.974059f, 0.500000f, 0.377371f, 0.000000f, 
		0.396022f, 1.00000f, 0.388361f, 1.00000f, 
		0.958865f, 0.500000f, 0.392565f, 0.000000f, 
		0.753218f, 0.500000f, 0.598212f, 0.000000f, 
		0.598212f, 1.00000f, 0.778402f, 0.500000f, 
		0.573028f, 0.000000f, 0.573028f, 1.00000f, 
		0.416577f, 1.00000f, 0.416577f, 0.000000f, 
		0.934854f, 0.500000f, 0.403754f, 1.00000f, 
		0.403754f, 0.000000f, 0.947676f, 0.500000f, 
		0.803700f, 0.500000f, 0.547731f, 0.000000f, 
		0.547731f, 1.00000f, 0.909097f, 0.500000f, 
		0.442334f, 1.00000f, 0.442334f, 0.000000f, 
		0.429425f, 1.00000f, 0.429425f, 0.000000f, 
		0.922005f, 0.500000f, 0.522040f, 0.000000f, 
		0.829390f, 0.500000f, 0.522040f, 1.00000f, 
		0.495526f, 0.000000f, 0.855905f, 0.500000f, 
		0.495526f, 1.00000f, 0.481869f, 1.00000f, 
		0.481869f, 0.000000f, 0.869561f, 0.500000f, 
		0.468500f, 1.00000f, 0.882930f, 0.500000f, 
		0.468500f, 0.000000f, 0.896088f, 0.500000f, 
		0.455342f, 1.00000f, 0.455342f, 0.000000f, 
		0.0907118f, 0.0127986f, 0.0907118f, 0.0127986f, 
		0.0907118f, 0.0127986f, 0.0907118f, 0.0127986f, 
		-3.00861f, -28.9471f, -3.18588f, -28.5940f, 
		-5.53458f, -30.6496f, -5.78544f, -30.3485f, 
		-0.404306f, 2.39194f, -0.438548f, 2.37016f, 
		-0.438561f, -2.37015f, -0.472803f, -2.34837f, 
		-0.643709f, 2.18115f, -0.665368f, 2.14710f, 
		-0.665364f, -2.14711f, -0.687023f, -2.11306f, 
		0.0197367f, 1.53064f, 0.0197367f, 1.53064f, 
		-7.04295f, -22.4116f, -7.04295f, -22.4116f, 
		-0.218016f, -2.46711f, -0.218016f, -2.46711f, 
		-7.04295f, -19.4116f, -7.04295f, -19.4116f, 
		-0.218016f, 2.46711f, -0.218016f, 2.46711f, 
		0.131391f, 2.47585f, 0.0722546f, 2.48019f, 
		0.0722441f, -2.48019f, 0.00237256f, -2.48452f, 
		0.472794f, 2.34837f, 0.438552f, 2.37016f, 
		0.438556f, -2.37015f, 0.404314f, -2.39194f, 
		0.687026f, 2.11306f, 0.665367f, 2.14710f, 
		0.665369f, -2.14710f, 0.643710f, -2.18115f, 
		-7.64185f, -32.7720f, -7.95425f, -32.5361f, 
		0.914278f, -1.30060f, 0.914278f, -1.30060f, 
		0.957248f, -1.32962f, 0.957248f, -1.32962f, 
		0.914278f, -1.30060f, 0.777308f, -1.84331f, 
		0.772513f, -1.88487f, 0.782103f, 1.80174f, 
		0.777308f, 1.84330f, 1.36219f, 1.38465f, 
		1.36219f, 1.38465f, 1.27367f, 1.39373f, 
		1.27367f, 1.39373f, 1.27367f, 1.39373f, 
		1.36219f, 1.38465f, 1.19993f, -1.39707f, 
		1.19993f, -1.39707f, 1.27368f, -1.39372f, 
		1.27368f, -1.39372f, 1.27368f, -1.39372f, 
		1.19993f, -1.39707f, -0.279638f, 0.267377f, 
		-0.420132f, 0.588391f, -0.279638f, 0.267377f, 
		-0.420132f, 0.588391f, 1.88051f, -1.82925f, 
		1.88051f, -1.82925f, 1.92411f, -1.85163f, 
		1.92411f, -1.85163f, 1.92411f, -1.85163f, 
		1.92411f, -1.85163f, 1.88051f, -1.82925f, 
		1.88051f, -1.82925f, 1.73171f, -1.63637f, 
		1.71147f, -1.57387f, 1.71147f, -1.57387f, 
		1.71147f, -1.57387f, 1.73170f, 1.63636f, 
		1.73170f, 1.63636f, 1.73170f, 1.63636f, 
		1.75647f, 1.69676f, 1.75647f, 1.69676f, 
		2.09115f, 1.88769f, 2.09115f, 1.88769f, 
		2.00574f, 1.86977f, 2.00574f, 1.86977f, 
		2.00574f, 1.86977f, 2.00574f, 1.86977f, 
		2.00574f, 1.86977f, 2.09115f, 1.88769f, 
		2.09115f, 1.88769f, 2.09115f, 1.88769f, 
		2.09115f, 1.88769f, 2.15721f, -1.88769f, 
		2.15721f, -1.88769f, 2.24261f, -1.86977f, 
		2.24261f, -1.86977f, 2.24261f, -1.86977f, 
		2.24261f, -1.86977f, 2.24261f, -1.86977f, 
		2.15721f, -1.88769f, 2.15721f, -1.88769f, 
		2.15721f, -1.88769f, 2.15721f, -1.88769f, 
		2.36784f, 1.82925f, 2.36784f, 1.82925f, 
		2.32424f, 1.85164f, 2.32424f, 1.85164f, 
		2.32424f, 1.85164f, 2.32424f, 1.85164f, 
		2.32424f, 1.85164f, 2.36784f, 1.82925f, 
		2.36784f, 1.82925f, -9.24848f, -35.1636f, 
		2.51972f, -1.62772f, 2.55514f, -1.49634f, 
		2.51972f, -1.62772f, 2.49526f, -1.68881f, 
		2.55514f, -1.49634f, 2.49526f, -1.68881f, 
		31.1249f, 11.7467f, 31.1249f, 11.7467f, 
		2.55514f, 1.49634f, 2.55514f, 1.49634f, 
		2.51972f, 1.62773f, 2.51972f, 1.62773f, 
		2.53961f, 1.56450f, 2.53961f, 1.56450f, 
		-9.60789f, -35.0019f, -0.571373f, -1.22885f, 
		-0.571373f, -1.22885f, -0.571400f, -1.22881f, 
		-0.571400f, -1.22881f, -0.571400f, -1.22881f, 
		-10.2725f, -37.6738f, -10.6321f, -40.1519f, 
		-10.6321f, -40.1519f, -10.6620f, -37.5907f, 
		-11.0321f, -40.1476f, -11.0321f, -40.1476f, 
		-0.505274f, -1.23584f, -0.505274f, -1.23584f, 
		-0.505274f, -1.23584f, -0.505274f, -1.23584f, 
		3.34105f, 2.01799f, 3.34105f, 2.01799f, 
		3.34105f, 2.01799f, 4.11184f, 4.09245f, 
		28.5015f, 11.5411f, 28.5015f, 11.5411f, 
		-0.726308f, 0.973164f, -0.726308f, 0.973164f, 
		-0.726308f, 0.973164f, -0.726308f, 0.973164f, 
		-4.11186f, 4.09243f, 30.6722f, 10.6057f, 
		30.6722f, 10.6057f, 28.1007f, 11.1460f, 
		28.1007f, 11.1460f, -6.12571f, 1.03827f, 
		-6.11988f, 1.00122f, 6.12571f, 1.03827f, 
		6.11987f, 1.00116f, 5.00000f, -1.50000f, 
		5.00000f, -1.53448f, -2.09318f, 0.476882f, 
		-2.09318f, 0.476882f, -2.09318f, 0.476882f, 
		14.9638f, -16.3414f, 14.9638f, -16.3414f, 
		14.9638f, -16.3414f, 14.9638f, -16.3414f, 
		14.9638f, -16.3414f, -5.00000f, -1.50000f, 
		-5.00000f, -1.53451f, 22.1025f, -7.12053f, 
		22.1025f, -7.12053f, 22.1025f, -7.12053f, 
		22.1025f, -7.12053f, 22.1025f, -7.12053f, 
		6.12571f, 1.03827f, 6.11987f, 1.00116f, 
		29.5133f, 23.7724f, 29.5133f, 23.7724f, 
		-6.12571f, 1.03827f, -6.11988f, 1.00122f, 
		16.1556f, -15.8654f, 16.1556f, -15.8654f, 
		16.1556f, -15.8654f, 16.1556f, -15.8654f, 
		-0.230299f, -1.32646f, -0.230299f, -1.32646f, 
		-0.230299f, -1.32646f, -0.231884f, -1.32468f, 
		-0.231884f, -1.32468f, 19.8029f, -12.9454f, 
		19.8029f, -12.9454f, 19.8029f, -12.9454f, 
		-1.30068f, 0.844102f, -1.30068f, 0.844102f, 
		-1.30068f, 0.844102f, -1.30068f, 0.844102f, 
		-1.30068f, 0.844102f, -1.30068f, 0.844102f, 
		-1.30068f, 0.844102f, -1.30068f, 0.844102f, 
		-0.472680f, -1.24891f, -0.472680f, -1.24891f, 
		-0.472680f, -1.24891f, -0.472680f, -1.24891f, 
		-0.472680f, -1.24891f, -0.472680f, -1.24891f, 
		5.00000f, -1.50000f, 5.00000f, -1.55669f, 
		0.0198509f, -0.143826f, 0.0198509f, -0.143826f, 
		0.0198509f, -0.143826f, 0.335301f, -1.40081f, 
		0.335301f, -1.40081f, -2.70798f, 3.84315f, 
		-2.93951f, 3.60500f, -5.00000f, -1.50000f, 
		-5.00000f, -1.55665f, 18.6252f, -14.4795f, 
		18.6252f, -14.4795f, 18.6252f, -14.4795f, 
		18.6252f, -14.4795f, 31.4664f, 11.8321f, 
		31.4664f, 11.8321f, 31.6197f, 12.0583f, 
		31.5127f, 12.1336f, 31.6197f, 12.0583f, 
		20.6685f, -11.6592f, 20.6685f, -11.6592f, 
		20.6685f, -11.6592f, 31.1479f, 25.7469f, 
		31.1479f, 25.7469f, -0.557986f, 1.20967f, 
		-0.557986f, 1.20967f, -0.557986f, 1.20967f, 
		-0.557986f, 1.20967f, 30.7506f, 23.9447f, 
		30.7506f, 23.9447f, 30.7506f, 23.9447f, 
		30.7506f, 23.9447f, -0.897069f, -0.221994f, 
		-0.897069f, -0.221994f, -0.658925f, -0.272928f, 
		-0.832157f, -0.246514f, -0.832157f, -0.246514f, 
		31.4775f, 25.6651f, 31.4775f, 25.6651f, 
		31.5616f, 25.4323f, 31.5616f, 25.4323f, 
		31.5616f, 25.4323f, 31.5616f, 25.4323f, 
		31.5616f, 25.4323f, 31.9601f, 26.0641f, 
		31.9601f, 26.0641f, 30.1397f, 13.6594f, 
		30.1397f, 13.6594f, 30.1397f, 13.6594f, 
		30.1397f, 13.6594f, 12.5398f, -16.8120f, 
		12.5398f, -16.8120f, 12.5398f, -16.8120f, 
		12.5398f, -16.8120f, 4.67791f, 3.84315f, 
		4.67791f, 3.84315f, 28.8575f, 26.7395f, 
		28.8575f, 26.7395f, 28.8575f, 26.7395f, 
		28.8575f, 26.7395f, 4.67791f, 3.84315f, 
		4.67791f, 3.84315f, 0.416730f, -1.30144f, 
		0.416730f, -1.30144f, 0.416730f, -1.30144f, 
		0.416730f, -1.30144f, 0.416730f, -1.30144f, 
		31.0387f, 10.9703f, 31.0387f, 10.9703f, 
		31.0387f, 10.9703f, 31.0387f, 10.9703f, 
		31.0387f, 10.9703f, 0.0115140f, -0.0673444f, 
		0.0115140f, -0.0673444f, 0.0115140f, -0.0673444f, 
		-0.335523f, 1.33936f, -0.335523f, 1.33936f, 
		-0.335523f, 1.33936f, -0.335523f, 1.33936f, 
		28.6554f, 26.2819f, 28.6554f, 26.2819f, 
		12.9914f, -16.7742f, 12.9914f, -16.7742f, 
		12.9914f, -16.7742f, 22.2645f, -7.55408f, 
		22.2645f, -7.55408f, 22.2645f, -7.55408f, 
		22.2789f, -7.28421f, 22.2789f, -7.28421f, 
		22.2789f, -7.28421f, -2.06658f, 0.403646f, 
		-2.06658f, 0.403646f, -2.06658f, 0.403646f, 
		-2.06658f, 0.403646f, -2.06658f, 0.403646f, 
		16.0242f, -16.1909f, 16.0242f, -16.1909f, 
		16.0242f, -16.1909f, 16.0242f, -16.1909f, 
		12.7168f, -17.0588f, 12.7168f, -17.0588f, 
		12.7168f, -17.0588f, -0.510632f, 1.20456f, 
		-0.510632f, 1.20456f, -0.510632f, 1.20456f, 
		-0.510632f, 1.20456f, 3.27052f, -4.69278f, 
		-0.900637f, 0.877780f, -0.900637f, 0.877780f, 
		-0.974265f, 0.938294f, -0.974265f, 0.938294f, 
		-0.250776f, -0.270082f, -0.271090f, -0.279531f, 
		-0.271090f, -0.279531f, -0.271090f, -0.279531f, 
		-0.250776f, -0.270082f, -3.27052f, -4.69278f, 
		0.443662f, 1.29116f, 0.443662f, 1.29116f, 
		0.443662f, 1.29116f, 0.443662f, 1.29116f, 
		0.443662f, 1.29116f, -0.902753f, -0.135138f, 
		-0.872035f, -0.180114f, -0.872035f, -0.180114f, 
		-0.872035f, -0.180114f, -0.658925f, -0.272928f, 
		-0.872035f, -0.180114f, -0.902753f, -0.135138f, 
		-0.658925f, -0.272928f, -0.902753f, -0.135138f, 
		-0.902753f, -0.135138f, 13.0371f, -16.9586f, 
		13.0371f, -16.9586f, 13.0371f, -16.9586f, 
		0.756737f, -1.91824f, 0.772707f, -1.85053f, 
		-0.329731f, -0.455660f, -0.255314f, -0.335441f, 
		-0.329731f, -0.455660f, -0.329731f, -0.455660f, 
		-0.329731f, -0.455660f, -0.255314f, -0.335441f, 
		-1.62176f, -0.379930f, -1.55812f, -0.354627f, 
		-1.60355f, -0.366729f, -1.56818f, -0.378565f, 
		-1.60355f, -0.366729f, -1.55815f, -0.409391f, 
		-1.62176f, -0.379930f, 15.1557f, -16.5119f, 
		15.1557f, -16.5119f, 15.1557f, -16.5119f, 
		6.04691f, -27.8151f, -1.89010f, -0.335811f, 
		-1.89010f, -0.335811f, -1.89010f, -0.335811f, 
		-1.96411f, -0.341755f, -2.06918f, -0.427460f, 
		-1.96411f, -0.341755f, -2.06918f, -0.427460f, 
		-2.06918f, -0.427460f, -1.96411f, -0.341755f, 
		-0.680765f, 0.260179f, -0.590815f, 0.272928f, 
		-0.590815f, 0.272928f, -0.590815f, 0.272928f, 
		-0.676878f, 0.260825f, -0.590815f, 0.272928f, 
		-0.676878f, 0.260825f, -0.680765f, 0.260179f, 
		-0.590815f, 0.272928f, -0.680765f, 0.260179f, 
		-2.16143f, -0.758131f, -2.50302f, -0.779628f, 
		-2.16143f, -0.758131f, -2.22467f, -0.794829f, 
		-2.04837f, -0.424697f, -2.04837f, -0.424697f, 
		-2.04837f, -0.424697f, -2.04837f, -0.424697f, 
		-1.94027f, -0.336966f, -1.94027f, -0.336966f, 
		-1.94027f, -0.336966f, -2.05830f, -0.402968f, 
		-2.05830f, -0.402968f, -2.04837f, -0.424697f, 
		-2.16143f, -0.758131f, -2.33957f, -0.769937f, 
		-2.16143f, -0.758131f, -2.31108f, -0.844975f, 
		-2.16143f, -0.758131f, -2.33949f, -0.769949f, 
		-2.16143f, -0.758131f, -2.31108f, -0.844975f, 
		-0.721266f, 0.926388f, -0.721266f, 0.926388f, 
		-0.721266f, 0.926388f, -0.721266f, 0.926388f, 
		-11.5987f, -17.0003f, -11.5987f, -17.0003f, 
		-11.5987f, -17.0003f, -11.5987f, -17.0003f, 
		17.7692f, -14.8892f, 17.7692f, -14.8892f, 
		17.7692f, -14.8892f, 17.7692f, -14.8892f, 
		17.7692f, -14.8892f, 17.7692f, -14.8892f, 
		-0.393386f, -0.558979f, -0.393386f, -0.558979f, 
		-0.261960f, -0.285239f, -0.393386f, -0.558979f, 
		-0.393386f, -0.558979f, -0.261960f, -0.285239f, 
		21.5842f, -10.3475f, 21.5842f, -10.3475f, 
		22.2680f, 6.38280f, 22.2680f, 6.38280f, 
		22.2680f, 6.38280f, -2.29854f, -0.799289f, 
		-2.29854f, -0.799289f, -2.30013f, -0.789248f, 
		-2.30013f, -0.789248f, -2.30013f, -0.789248f, 
		-2.16143f, -0.758131f, -2.16143f, -0.758131f, 
		-2.29854f, -0.799289f, -0.923448f, 1.25459f, 
		-0.905142f, 1.26412f, -0.905142f, 1.26412f, 
		-0.905142f, 1.26412f, -0.905142f, 1.26412f, 
		-0.923448f, 1.25459f, -0.923448f, 1.25459f, 
		-0.923448f, 1.25459f, -0.368578f, -0.518988f, 
		-0.368578f, -0.518988f, -0.368578f, -0.518988f, 
		-0.270050f, -0.317360f, -0.368578f, -0.518988f, 
		-0.368578f, -0.518988f, -0.270050f, -0.317360f, 
		21.1831f, -11.1738f, 21.1831f, -11.1738f, 
		21.1831f, -11.1738f, 21.1831f, -11.1738f, 
		21.1831f, -11.1738f, 21.1831f, -11.1738f, 
		21.1831f, -11.1738f, -11.5987f, -17.0003f, 
		-11.5987f, -17.0003f, -11.5987f, -17.0003f, 
		-11.5987f, -17.0003f, 22.2363f, 6.38280f, 
		22.2363f, 6.38280f, 22.2363f, 6.38280f, 
		22.2363f, 6.38280f, 17.8029f, -15.0451f, 
		17.8029f, -15.0451f, 17.8029f, -15.0451f, 
		17.8029f, -15.0451f, 17.8029f, -15.0451f, 
		17.8029f, -15.0451f, -0.894937f, 1.22147f, 
		-0.872169f, 1.26392f, -0.894937f, 1.22147f, 
		-0.867801f, 1.22391f, -0.867801f, 1.22391f, 
		0.590808f, 0.272928f, 0.590808f, 0.272928f, 
		0.590808f, 0.272928f, 0.590808f, 0.272928f, 
		0.590808f, 0.272928f, 0.590808f, 0.272928f, 
		0.590815f, 0.788046f, 0.590815f, 0.788046f, 
		0.590815f, 0.788046f, 0.590815f, 0.788046f, 
		0.590815f, 0.788046f, 0.590815f, 0.788046f, 
		-3.17899f, -0.713769f, -3.17899f, -0.713769f, 
		-3.17899f, -0.713769f, 0.590805f, 0.272928f, 
		0.590805f, 0.272928f, 0.590805f, 0.272928f, 
		0.590805f, 0.272928f, 0.590805f, 0.272928f, 
		0.590805f, 0.272928f, -0.196801f, 0.255234f, 
		-0.364339f, 0.481782f, -0.196801f, 0.255234f, 
		-0.364339f, 0.481782f, -3.26681f, 0.719971f, 
		-3.26681f, 0.719971f, -3.26348f, -0.717646f, 
		-2.21743f, 0.686577f, -2.38709f, 0.746279f, 
		-2.56931f, 0.815368f, -2.16151f, 0.655164f, 
		-2.56931f, 0.815368f, -2.21743f, 0.686577f, 
		-2.16164f, 0.655458f, -2.21743f, 0.686577f, 
		-2.25736f, 0.759378f, -2.16153f, 0.648484f, 
		-2.18389f, 0.669213f, -2.16153f, 0.648484f, 
		-2.56877f, 0.815430f, -2.56877f, 0.815430f, 
		-2.25736f, 0.759378f, -2.25736f, 0.759378f, 
		-2.16170f, 0.655099f, -2.16170f, 0.655089f, 
		-2.25736f, 0.759378f, -2.16153f, 0.648484f, 
		-2.18389f, 0.669213f, -2.56956f, 0.815319f, 
		-2.56956f, 0.815319f, -2.16153f, 0.648484f, 
		-2.25736f, 0.759378f, -2.25736f, 0.759378f, 
		-3.41895f, -0.704614f, -3.41895f, -0.704614f, 
		-3.42041f, -0.704744f, -3.42041f, -0.704744f, 
		-3.42041f, -0.704744f, -3.41895f, -0.704614f, 
		-3.43379f, -0.704480f, -3.43379f, -0.704480f, 
		-3.41895f, -0.704614f, -3.41895f, -0.704614f, 
		-3.42041f, -0.704744f, -3.42041f, -0.704744f, 
		-3.42041f, -0.704744f, -3.41895f, -0.704614f, 
		-3.43379f, -0.704480f, -3.43379f, -0.704480f, 
		-3.41895f, -0.704614f, -3.41895f, -0.704614f, 
		-3.42041f, -0.704744f, -3.42041f, -0.704744f, 
		-3.42041f, -0.704744f, -3.41895f, -0.704614f, 
		-3.43379f, -0.704480f, -3.43379f, -0.704480f, 
		-2.30216f, 0.762989f, -2.30216f, 0.762989f, 
		-2.00289f, 0.263151f, -2.00289f, 0.263151f, 
		-2.00289f, 0.263151f, -2.00289f, 0.263151f, 
		-2.07224f, 0.311680f, -2.07224f, 0.311680f, 
		-2.07224f, 0.311680f, -2.03149f, 0.318889f, 
		-1.88243f, -0.316988f, -2.05071f, -0.339958f, 
		-1.90848f, -0.297067f, -1.90848f, -0.297067f, 
		-2.07236f, -0.275386f, -1.88243f, -0.316988f, 
		-1.55446f, -0.189103f, -1.69070f, -0.240659f, 
		-1.92657f, -0.302858f, -1.92657f, -0.302858f, 
		-1.92657f, -0.302858f, -1.69070f, -0.240659f, 
		-0.524635f, 2.66708f, -0.524635f, 2.66708f, 
		-0.333231f, 2.76962f, -0.333231f, 2.76962f, 
		-0.333231f, 2.76962f, -0.274950f, 2.79251f, 
		-0.524635f, 2.66708f, -2.25732f, 0.760857f, 
		-2.25732f, 0.760857f, -2.25732f, 0.760857f, 
		0.557474f, 2.65613f, 0.557474f, 2.65613f, 
		0.565153f, 2.64017f, 0.565153f, 2.64017f, 
		0.565153f, 2.64017f, 0.557474f, 2.65613f, 
		67.6587f, -21.8143f, 67.7949f, -22.0431f, 
		67.6587f, -21.8143f, 67.8285f, -22.1494f, 
		67.6754f, -21.8396f, 67.8285f, -22.1494f, 
		67.6587f, -21.8143f, 67.6754f, -21.8396f, 
		67.6587f, -21.8143f, 67.7949f, -22.0431f, 
		-2.34611f, 0.756971f, -2.19132f, 0.673383f, 
		-2.15680f, 0.601707f, -2.19132f, 0.673383f, 
		-2.29971f, 0.842333f, -2.34611f, 0.756971f, 
		-2.34611f, 0.756971f, -2.16143f, 0.656561f, 
		-2.25717f, 0.765488f, -2.25717f, 0.765488f, 
		-2.25717f, 0.765488f, -2.25717f, 0.765488f, 
		-0.576821f, 1.33390f, -0.576821f, 1.33390f, 
		-0.576821f, 1.33390f, -0.576821f, 1.33390f, 
		-0.576821f, 1.33390f, -0.576821f, 1.33390f, 
		-0.576821f, 1.33390f, -0.576821f, 1.33390f, 
		0.590806f, 0.788046f, 0.590806f, 0.788046f, 
		0.590806f, 0.788046f, 0.635598f, 0.794660f, 
		0.637336f, 0.797368f, 0.590806f, 0.788046f, 
		-0.590799f, 0.788046f, -0.590799f, 0.788046f, 
		-0.635598f, 0.794710f, -0.590799f, 0.788046f, 
		-0.635598f, 0.794710f, -0.590799f, 0.788046f, 
		-0.634695f, 0.794560f, -0.634695f, 0.794560f, 
		0.823648f, 1.26442f, 0.864025f, 1.26383f, 
		0.823648f, 1.26442f, 0.864025f, 1.26383f, 
		0.864025f, 1.26383f, 0.823648f, 1.26442f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		-0.576821f, 1.33390f, -0.576821f, 1.33390f, 
		-0.576821f, 1.33390f, -0.576821f, 1.33390f, 
		-0.576821f, 1.33390f, -0.576821f, 1.33390f, 
		-1.32109f, -0.858193f, -1.32109f, -0.858193f, 
		-1.32109f, -0.858193f, -1.32109f, -0.858193f, 
		-1.32109f, -0.858193f, -0.795116f, -0.956227f, 
		-0.795116f, -0.956227f, -0.795116f, -0.956227f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		0.576821f, 1.33390f, 0.576821f, 1.33390f, 
		-0.576835f, 1.33390f, -0.576835f, 1.33390f, 
		-0.576835f, 1.33390f, -0.576835f, 1.33390f, 
		-0.576835f, 1.33390f, -0.576835f, 1.33390f, 
		-0.576835f, 1.33390f, -0.576835f, 1.33390f, 
		-0.716784f, -0.900544f, -0.716784f, -0.900544f, 
		-0.716784f, -0.900544f, -0.716784f, -0.900544f, 
		-0.716784f, -0.900544f, -0.716784f, -0.900544f, 
		-2.29481f, 0.761950f, -2.29481f, 0.761950f, 
		-1.58204f, -0.381418f, -1.56545f, -0.389595f, 
		-1.58204f, -0.381418f, -1.58204f, -0.381418f, 
		-1.56545f, -0.389595f, -1.31955f, 0.826454f, 
		-1.31955f, 0.826454f, -1.31955f, 0.826454f, 
		-1.31955f, 0.826454f, -1.31955f, 0.826454f, 
		-1.31955f, 0.826454f, -1.31955f, 0.826454f, 
		-1.31955f, 0.826454f, -1.31955f, 0.826454f, 
		-1.31955f, 0.826454f, -1.31953f, 0.826409f, 
		-1.31953f, 0.826409f, -1.31953f, 0.826409f, 
		-1.31953f, 0.826409f, -1.31953f, 0.826409f, 
		-0.864047f, -0.927653f, -0.864047f, -0.927653f, 
		-0.864047f, -0.927653f, -0.864047f, -0.927653f, 
		-0.864047f, -0.927653f, -0.266098f, 0.248820f, 
		-0.342144f, 0.419160f, -0.266098f, 0.248820f, 
		-0.342144f, 0.419160f, -0.342144f, 0.419160f, 
		0.975362f, -0.0841152f, 0.975362f, -0.0841152f, 
		0.975362f, -0.0841152f, 0.939209f, -0.0697421f, 
		0.939209f, -0.0697421f, 0.939209f, -0.0697421f, 
		0.939209f, -0.0697421f, 0.975362f, -0.0841152f, 
		-1.33020f, -0.845182f, -1.33020f, -0.845182f, 
		-1.33020f, -0.845182f, -1.33020f, -0.845182f, 
		-1.33007f, -0.845277f, -1.33007f, -0.845277f, 
		-1.33007f, -0.845277f, -1.33007f, -0.845277f, 
		-1.33007f, -0.845277f, 0.811824f, 1.25468f, 
		0.889884f, 1.08667f, 0.811824f, 1.25468f, 
		0.920512f, 1.18067f, 0.932144f, 1.13646f, 
		0.920512f, 1.18067f, 0.920512f, 1.18067f, 
		0.876918f, 1.17036f, 0.898276f, -0.0861266f, 
		0.898276f, -0.0861266f, 0.898276f, -0.0861266f, 
		0.897174f, -0.0879035f, 0.897174f, -0.0879035f, 
		0.897174f, -0.0879035f, 0.897174f, -0.0879035f, 
		0.897174f, -0.0879035f, 0.898276f, -0.0861266f, 
		0.898276f, -0.0861266f, -0.945201f, -0.923701f, 
		-0.945201f, -0.923701f, -0.945201f, -0.923701f, 
		-0.945201f, -0.923701f, -0.945201f, -0.923701f, 
		-1.58206f, -0.381382f, -1.56546f, -0.389535f, 
		-1.58206f, -0.381382f, -1.58206f, -0.381382f, 
		-1.56546f, -0.389535f, -0.945123f, -0.923680f, 
		-0.945123f, -0.923680f, -0.945123f, -0.923680f, 
		-0.864153f, -0.927664f, -0.864153f, -0.927664f, 
		-0.864153f, -0.927664f, -0.864153f, -0.927664f, 
		-0.864153f, -0.927664f, 0.971041f, -0.0882259f, 
		0.971041f, -0.0882259f, 0.935231f, -0.0742324f, 
		0.935231f, -0.0742324f, 0.935231f, -0.0742324f, 
		0.971041f, -0.0882259f, -0.540617f, -0.790472f, 
		-0.540617f, -0.790472f, -0.540617f, -0.790472f, 
		-0.540617f, -0.790472f, -0.540617f, -0.790472f, 
		0.948703f, 1.02991f, 0.948703f, 1.02991f, 
		0.944156f, 1.00780f, 0.944156f, 1.00780f, 
		0.944156f, 1.00780f, 0.948703f, 1.02991f, 
		-2.03779f, -0.322907f, -2.03779f, -0.322907f, 
		-2.03779f, -0.322907f, -2.03779f, -0.322907f, 
		-2.02935f, -0.362163f, -2.02935f, -0.362163f, 
		-2.05871f, -0.353502f, -2.05871f, -0.353502f, 
		-2.03779f, -0.322907f, -2.03779f, -0.322907f, 
		-0.908065f, 1.00586f, -0.908065f, 1.00586f, 
		-0.901140f, 0.989426f, -0.901140f, 0.989426f, 
		-0.901140f, 0.989426f, -0.901140f, 0.989426f, 
		-0.901140f, 0.989426f, -0.908065f, 1.00586f, 
		-0.908065f, 1.00586f, -0.908065f, 1.00586f, 
		-0.895504f, -1.00394f, -0.895504f, -1.00394f, 
		-0.895504f, -1.00394f, -0.896904f, 0.976705f, 
		-0.896904f, 0.976705f, -0.896904f, 0.976705f, 
		-0.896904f, 0.976705f, -0.896904f, 0.976705f, 
		-0.896955f, 0.976701f, -0.896955f, 0.976701f, 
		-0.896955f, 0.976701f, -0.896955f, 0.976701f, 
		-0.896955f, 0.976701f, -0.896955f, 0.976701f, 
		-0.896955f, 0.976701f, -0.896955f, 0.976701f, 
		-0.896955f, 0.976701f, -0.896955f, 0.976701f, 
		-1.27904f, -0.850507f, -1.27904f, -0.850507f, 
		-1.27904f, -0.850507f, -1.27904f, -0.850507f, 
		0.894274f, -0.0861408f, 0.894274f, -0.0861408f, 
		0.892397f, -0.0825785f, 0.892397f, -0.0825785f, 
		0.892397f, -0.0825785f, 0.894274f, -0.0861408f, 
		-1.58359f, 0.348888f, -1.56489f, 0.355159f, 
		-1.58359f, 0.348888f, -1.56489f, 0.355159f, 
		-1.58359f, 0.348888f, -1.56489f, 0.355159f, 
		-1.57730f, -0.256483f, -1.57730f, -0.256483f, 
		-1.49926f, -0.462953f, -1.49926f, -0.462953f, 
		-1.57730f, -0.256483f, -0.364320f, 2.75441f, 
		-0.313430f, 2.82997f, -0.369059f, 2.84174f, 
		-0.319079f, 2.75565f, -0.319079f, 2.75565f, 
		-0.220445f, 2.82610f, -0.364320f, 2.75441f, 
		-0.319614f, 2.84174f, -0.484662f, 2.71511f, 
		-0.484662f, 2.71511f, -0.494614f, 2.71631f, 
		-0.494614f, 2.71631f, 0.474854f, 2.72080f, 
		0.439413f, 2.72291f, 0.439413f, 2.72291f, 
		0.474854f, 2.72080f, -1.29665f, 0.857667f, 
		-1.29665f, 0.857667f, -1.29665f, 0.857667f, 
		-1.29665f, 0.857667f, -1.98767f, 0.00819828f, 
		-2.04307f, 0.137590f, -1.98767f, 0.00819828f, 
		-1.98767f, 0.00819828f, -1.98767f, 0.00819828f, 
		-1.98767f, 0.00819828f, -1.98969f, 0.184240f, 
		-1.98969f, 0.184240f, -2.04307f, 0.137590f, 
		-2.04307f, 0.137590f, -1.98969f, 0.184240f, 
		0.553613f, 1.33390f, 0.553613f, 1.33390f, 
		0.553613f, 1.33390f, 0.553613f, 1.33390f, 
		0.553613f, 1.33390f, 0.553613f, 1.33390f, 
		0.553613f, 1.33390f, 0.553613f, 1.33390f, 
		-1.55622f, -0.125323f, -1.94873f, 0.0380635f, 
		-1.94873f, 0.0380635f, -1.55622f, -0.125323f, 
		-1.56681f, -0.122485f, -1.94873f, 0.0380635f, 
		-1.56818f, -0.0492823f, -1.80173f, 0.122409f, 
		-2.03284f, 0.110971f, -1.56818f, 0.112487f, 
		-1.80173f, 0.122409f, -1.55446f, -0.189103f, 
		-1.81004f, -0.147087f, -1.55446f, -0.189103f, 
		-1.55446f, -0.189103f, -1.81004f, -0.147087f, 
		-2.03284f, -0.110683f, -1.97252f, 0.0122049f, 
		-1.97252f, 0.0122049f, 0.553613f, 1.33390f, 
		0.553613f, 1.33390f, 0.553613f, 1.33390f, 
		0.563346f, 1.33357f, 0.553613f, 1.33390f, 
		0.553613f, 1.33390f, 0.563346f, 1.33361f, 
		-1.57599f, -0.238196f, -1.57599f, -0.238196f, 
		-1.49930f, -0.442010f, -1.57599f, -0.238196f, 
		-1.49930f, -0.442010f, -1.49930f, -0.442010f, 
		0.211082f, 1.00000f, 0.204453f, 1.00000f, 
		0.204453f, 1.00000f, 0.146977f, 0.583333f, 
		0.142753f, 0.546128f, 0.208677f, 0.0461282f, 
		-0.266096f, 0.248857f, -0.272076f, 0.251697f, 
		-0.266096f, 0.248857f, -0.272076f, 0.251697f, 
		-0.272076f, 0.251697f, -1.58384f, 0.342414f, 
		-1.58384f, 0.342414f, -1.58384f, 0.342414f, 
		-1.55892f, 0.340106f, -1.55892f, 0.340106f, 
		0.142859f, 0.953848f, 0.204453f, 0.416667f, 
		0.208571f, 0.453848f, 0.142859f, 0.000000f, 
		0.146977f, 0.000000f, -1.57461f, -0.223982f, 
		-1.57461f, -0.223982f, -1.56980f, -0.377210f, 
		-1.56703f, -0.380584f, -1.57160f, -0.203488f, 
		-1.57160f, -0.203488f, -1.56355f, -0.208088f, 
		-1.55815f, -0.226344f, -1.57461f, -0.223982f, 
		-0.271090f, 0.332795f, -0.269968f, 0.332588f, 
		-0.317586f, 0.360812f, -0.275563f, 0.317694f, 
		-0.271090f, 0.332795f, -0.269968f, 0.332588f, 
		-0.271090f, 0.332795f, -2.73527f, 1.03643f, 
		-2.84989f, 1.12347f, -2.04724f, 0.300624f, 
		-1.96082f, 0.153305f, -1.96082f, 0.153305f, 
		-1.96082f, 0.153305f, -1.96082f, 0.153305f, 
		-2.02988f, 0.295658f, -2.02988f, 0.295658f, 
		-1.61218f, 0.129351f, -1.55815f, 0.346805f, 
		-1.57339f, 0.343814f, -1.61218f, 0.129351f, 
		-1.56700f, 0.157251f, -1.55815f, 0.346805f, 
		-1.57339f, 0.343814f, -1.61218f, 0.129351f, 
		-1.56700f, 0.157251f, -1.61218f, 0.129351f, 
		-1.56700f, 0.157251f, -1.61214f, 0.129287f, 
		-1.55815f, 0.346790f, -1.57338f, 0.343798f, 
		-1.61214f, 0.129287f, -1.55815f, 0.346790f, 
		-1.57338f, 0.343798f, -1.61214f, 0.129287f, 
		-1.61214f, 0.129287f, -1.56699f, 0.157177f, 
		-3.41965f, 0.656861f, -3.43379f, 0.656561f, 
		-3.43379f, 0.656561f, -3.41569f, 0.657005f, 
		-0.498354f, 0.272928f, -0.498354f, 0.272928f, 
		-0.498354f, 0.272928f, -0.498354f, 0.272928f, 
		-0.498354f, 0.272928f, -0.498354f, 0.272928f, 
		-1.05293f, -0.924343f, -1.05293f, -0.924343f, 
		-1.05293f, -0.924343f, -0.567022f, 0.811010f, 
		-0.567022f, 0.811010f, -0.567022f, 0.811010f, 
		-0.567022f, 0.811010f, -0.158192f, 0.213375f, 
		-0.283994f, 0.296915f, -0.158192f, 0.213375f, 
		-0.283994f, 0.296915f, -2.73527f, 1.03643f, 
		-2.84972f, 1.12329f, 0.957242f, 1.19585f, 
		0.957242f, 1.19585f, 0.957242f, 1.19585f, 
		0.968346f, 1.20251f, 0.968346f, 1.20251f, 
		0.968346f, 1.20251f, -1.13556f, 0.864797f, 
		-1.13556f, 0.864797f, -1.13556f, 0.864797f, 
		-0.271090f, 0.202993f, -0.264660f, 0.201082f, 
		-0.271090f, 0.202993f, -0.264660f, 0.201082f, 
		-3.41001f, 2.19634f, -2.06674f, -0.265188f, 
		-2.05714f, -0.257274f, -2.05714f, -0.257274f, 
		-2.05714f, -0.257274f, -2.05714f, -0.257274f, 
		-2.03170f, -0.244694f, -2.03170f, -0.244694f, 
		-2.05714f, -0.257274f, -1.18710f, -0.865494f, 
		-1.18710f, -0.865494f, -3.41965f, 0.656861f, 
		-3.43379f, 0.656561f, -3.43379f, 0.656561f, 
		-3.41569f, 0.657005f, -1.55446f, -0.189103f, 
		-1.56498f, -0.182116f, -1.56498f, -0.182116f, 
		-1.55446f, -0.189103f, -1.56498f, -0.182116f, 
		-2.01799f, 0.142916f, -2.01799f, 0.142916f, 
		-2.01799f, 0.142916f, -2.01799f, 0.142916f, 
		-2.01799f, 0.142916f, -2.07794f, 0.0511132f, 
		-2.03233f, 0.0461767f, -2.07794f, 0.0511132f, 
		-2.03233f, 0.0461767f, -2.07794f, 0.0511132f, 
		3.42562f, 2.51382f, -0.471019f, -0.624497f, 
		-0.471019f, -0.624497f, -3.40997f, 2.19602f, 
		-3.40997f, 2.19602f, -3.40997f, 2.19602f, 
		-2.23451f, 0.714568f, -2.18072f, 0.711233f, 
		-2.16147f, 0.706860f, -2.18072f, 0.711233f, 
		-2.16164f, 0.704496f, -2.16164f, 0.704496f, 
		-2.23451f, 0.714568f, 1.54151f, 1.15861f, 
		1.54151f, 1.15861f, 1.53402f, 1.16660f, 
		1.53402f, 1.16660f, 1.53402f, 1.16660f, 
		1.53402f, 1.16660f, 1.54151f, 1.15861f, 
		1.54151f, 1.15861f, 0.649201f, 1.00000f, 
		0.702230f, 0.574406f, 0.649201f, 0.0744060f, 
		0.223829f, 1.00000f, 0.217505f, 1.00000f, 
		0.130246f, 0.563655f, 0.221184f, 1.00000f, 
		0.221184f, 0.0428379f, -1.57150f, 0.114069f, 
		-1.56693f, 0.116250f, -1.57339f, 0.343813f, 
		-1.57150f, 0.114069f, -1.56693f, 0.116250f, 
		-1.55815f, 0.346805f, -1.57339f, 0.343813f, 
		-1.57150f, 0.114069f, -1.56693f, 0.116250f, 
		-1.57150f, 0.114069f, 4.64399f, 3.84315f, 
		4.64399f, 3.84315f, -0.335847f, 0.339020f, 
		-0.266027f, 0.247712f, -0.335847f, 0.339020f, 
		-0.266027f, 0.247712f, -0.335847f, 0.339020f, 
		-0.335847f, 0.339020f, 33.0255f, -19.6856f, 
		33.0187f, -19.6128f, 33.0187f, -19.6128f, 
		33.0255f, -19.8498f, 33.0255f, -21.9734f, 
		33.0255f, -22.1376f, 33.0187f, -22.2104f, 
		33.0187f, -22.2104f, 26.8954f, -19.6127f, 
		26.8886f, -19.8498f, 26.8954f, -19.6127f, 
		26.8886f, -19.6856f, 0.702230f, 0.925585f, 
		0.649201f, 0.425585f, 0.702230f, 0.000000f, 
		0.702230f, 0.574406f, 0.649201f, 0.0744060f, 
		0.649201f, 1.00000f, -0.368093f, -0.340147f, 
		-0.258577f, -0.138527f, -0.368093f, -0.340147f, 
		-0.368093f, -0.340147f, -0.258577f, -0.138527f, 
		-1.58199f, 0.224458f, -1.56546f, 0.225311f, 
		-1.58199f, 0.224458f, -1.58199f, 0.224458f, 
		-1.56546f, 0.225311f, -0.265054f, 0.129740f, 
		-0.280873f, 0.318763f, -0.291815f, 0.143634f, 
		-0.265054f, 0.129740f, -0.275529f, 0.317706f, 
		-0.291815f, 0.143634f, -0.291815f, 0.143634f, 
		-0.291815f, 0.143634f, -0.296893f, 0.142790f, 
		-0.270040f, 0.122494f, -0.275542f, 0.317669f, 
		-0.280873f, 0.318723f, -0.296893f, 0.142790f, 
		-0.270040f, 0.122494f, -0.296893f, 0.142790f, 
		-0.296893f, 0.142790f, -0.296893f, 0.142790f, 
		-0.303242f, 1.32892f, -0.303242f, 1.32892f, 
		-0.303242f, 1.32892f, -0.303242f, 1.32892f, 
		-0.303242f, 1.32892f, -0.303254f, 1.32892f, 
		-0.303254f, 1.32892f, -0.303254f, 1.32892f, 
		-0.303254f, 1.32892f, -0.303254f, 1.32892f, 
		0.436636f, 0.272928f, 0.436636f, 0.272928f, 
		0.562815f, 0.255338f, 0.562815f, 0.255338f, 
		0.436636f, 0.272928f, 0.562815f, 0.255338f, 
		0.562797f, 0.257406f, 0.436636f, 0.272928f, 
		0.562797f, 0.257406f, 0.436636f, 0.272928f, 
		0.562797f, 0.257406f, 0.371763f, 2.68617f, 
		0.184529f, 2.84174f, 0.371763f, 2.68617f, 
		0.371763f, 2.68617f, 0.371591f, 2.68623f, 
		0.371591f, 2.68623f, 0.371591f, 2.68623f, 
		0.319614f, 2.84174f, 0.371591f, 2.68623f, 
		0.371591f, 2.68623f, 0.371763f, 2.68617f, 
		-0.423689f, 2.64760f, -0.423689f