// Copyright (c) Microsoft Corporation.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception

#ifndef DOUBLE_FROM_CHARS_TEST_CASES_HPP
#define DOUBLE_FROM_CHARS_TEST_CASES_HPP

#include <charconv>
#include <stddef.h>
#include <system_error>
using namespace std;

inline constexpr DoubleFromCharsTestCase double_from_chars_test_cases[] = {
    {"1.000000000000a000", chars_format::hex, 18, errc{}, 0x1.000000000000ap0}, // exact
    {"1.000000000000a001", chars_format::hex, 18, errc{}, 0x1.000000000000ap0}, // below midpoint, round down
    {"1.000000000000a800", chars_format::hex, 18, errc{}, 0x1.000000000000ap0}, // midpoint, round down to even
    {"1.000000000000a801", chars_format::hex, 18, errc{}, 0x1.000000000000bp0}, // above midpoint, round up
    {"1.000000000000b000", chars_format::hex, 18, errc{}, 0x1.000000000000bp0}, // exact
    {"1.000000000000b001", chars_format::hex, 18, errc{}, 0x1.000000000000bp0}, // below midpoint, round down
    {"1.000000000000b800", chars_format::hex, 18, errc{}, 0x1.000000000000cp0}, // midpoint, round up to even
    {"1.000000000000b801", chars_format::hex, 18, errc{}, 0x1.000000000000cp0}, // above midpoint, round up

    {"1.00000000000020", chars_format::hex, 16, errc{}, 0x1.0000000000002p0}, // exact
    {"1.00000000000021", chars_format::hex, 16, errc{}, 0x1.0000000000002p0}, // below midpoint, round down
    {"1.00000000000028", chars_format::hex, 16, errc{}, 0x1.0000000000002p0}, // midpoint, round down to even
    {"1.00000000000029", chars_format::hex, 16, errc{}, 0x1.0000000000003p0}, // above midpoint, round up
    {"1.00000000000030", chars_format::hex, 16, errc{}, 0x1.0000000000003p0}, // exact
    {"1.00000000000031", chars_format::hex, 16, errc{}, 0x1.0000000000003p0}, // below midpoint, round down
    {"1.00000000000038", chars_format::hex, 16, errc{}, 0x1.0000000000004p0}, // midpoint, round up to even
    {"1.00000000000039", chars_format::hex, 16, errc{}, 0x1.0000000000004p0}, // above midpoint, round up

    {"1.00000000000000044408920985006261616945266723632812500000", chars_format::general, 58, errc{},
        0x1.0000000000002p0}, // exact
    {"1.00000000000000045796699765787707292474806308746337890625", chars_format::general, 58, errc{},
        0x1.0000000000002p0}, // below midpoint, round down
    {"1.00000000000000055511151231257827021181583404541015624999", chars_format::general, 58, errc{},
        0x1.0000000000002p0}, // below midpoint, round down
    {"1.00000000000000055511151231257827021181583404541015625000", chars_format::general, 58, errc{},
        0x1.0000000000002p0}, // midpoint, round down to even
    {"1.00000000000000055511151231257827021181583404541015625001", chars_format::general, 58, errc{},
        0x1.0000000000003p0}, // above midpoint, round up
    {"1.00000000000000056898930012039272696711122989654541015625", chars_format::general, 58, errc{},
        0x1.0000000000003p0}, // above midpoint, round up
    {"1.00000000000000066613381477509392425417900085449218750000", chars_format::general, 58, errc{},
        0x1.0000000000003p0}, // exact
    {"1.00000000000000068001160258290838100947439670562744140625", chars_format::general, 58, errc{},
        0x1.0000000000003p0}, // below midpoint, round down
    {"1.00000000000000077715611723760957829654216766357421874999", chars_format::general, 58, errc{},
        0x1.0000000000003p0}, // below midpoint, round down
    {"1.00000000000000077715611723760957829654216766357421875000", chars_format::general, 58, errc{},
        0x1.0000000000004p0}, // midpoint, round up to even
    {"1.00000000000000077715611723760957829654216766357421875001", chars_format::general, 58, errc{},
        0x1.0000000000004p0}, // above midpoint, round up
    {"1.00000000000000079103390504542403505183756351470947265625", chars_format::general, 58, errc{},
        0x1.0000000000004p0}, // above midpoint, round up

    // https://www.exploringbinary.com/nondeterministic-floating-point-conversions-in-java/
    {"0.0000008p-1022", chars_format::hex, 15, errc{}, 0x0.0000008p-1022},

    // VSO-838635 "<charconv>: from_chars() mishandles certain subnormals"
    // These values change on half-ulp boundaries:
    // 1 * 2^-1075 ~= 2.47e-324 (half-ulp between zero and min subnormal)
    // 2 * 2^-1075 ~= 4.94e-324 (min subnormal)
    // 3 * 2^-1075 ~= 7.41e-324 (half-ulp between min subnormal and next subnormal)
    // 4 * 2^-1075 ~= 9.88e-324 (next subnormal)
    {"1."
     "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
     "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
     "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
     "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
     "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
     "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
     "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
     "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
     "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111"
     "1111111111111111111e-324",
        chars_format::scientific, 1007, errc::result_out_of_range, 0x0.0000000000000p+0},
    {"2."
     "2222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222"
     "2222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222"
     "2222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222"
     "2222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222"
     "2222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222"
     "2222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222"
     "2222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222"
     "2222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222"
     "2222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222"
     "2222222222222222222e-324",
        chars_format::scientific, 1007, errc::result_out_of_range, 0x0.0000000000000p+0},
    {"3."
     "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
     "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
     "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
     "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
     "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
     "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
     "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
     "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
     "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
     "3333333333333333333e-324",
        chars_format::scientific, 1007, errc{}, 0x0.0000000000001p-1022},
    {"4."
     "4444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
     "4444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
     "4444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
     "4444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
     "4444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
     "4444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
     "4444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
     "4444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
     "4444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
     "4444444444444444444e-324",
        chars_format::scientific, 1007, errc{}, 0x0.0000000000001p-1022},
    {"5."
     "5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555"
     "5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555"
     "5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555"
     "5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555"
     "5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555"
     "5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555"
     "5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555"
     "5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555"
     "5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555"
     "5555555555555555555e-324",
        chars_format::scientific, 1007, errc{}, 0x0.0000000000001p-1022},
    {"6."
     "6666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666"
     "6666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666"
     "6666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666"
     "6666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666"
     "6666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666"
     "6666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666"
     "6666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666"
     "6666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666"
     "6666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666"
     "6666666666666666666e-324",
        chars_format::scientific, 1007, errc{}, 0x0.0000000000001p-1022},
    {"7."
     "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
     "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
     "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
     "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
     "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
     "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
     "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
     "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
     "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
     "7777777777777777777e-324",
        chars_format::scientific, 1007, errc{}, 0x0.0000000000002p-1022},
    {"8."
     "8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888"
     "8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888"
     "8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888"
     "8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888"
     "8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888"
     "8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888"
     "8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888"
     "8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888"
     "8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888"
     "8888888888888888888e-324",
        chars_format::scientific, 1007, errc{}, 0x0.0000000000002p-1022},
    {"9."
     "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999"
     "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999"
     "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999"
     "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999"
     "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999"
     "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999"
     "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999"
     "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999"
     "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999"
     "9999999999999999999e-324",
        chars_format::scientific, 1007, errc{}, 0x0.0000000000002p-1022},

    // VSO-852024: Test cases for round-to-nearest, ties-to-even.
    // Consider the values:
    // A: 0x1.0000000000000p+0 == 1.0000000000000000000000000000000000000000000000000000
    // X:    (1 + 2^-53) * 2^0 == 1.00000000000000011102230246251565404236316680908203125
    // B: 0x1.0000000000001p+0 == 1.0000000000000002220446049250313080847263336181640625
    // X is equidistant from A and B. Because they're tied for being nearest, we need to round to even.
    // That means rounding down to A, because A's least significant hexit 0 is even.
    // However, values between X and B aren't tied - they're simply nearer to B, so they need to round up to B.
    // We need to handle tricky cases like the digits of X, followed by a million 0 digits, followed by a 1 digit.
    // Similarly:
    // E:      0x1.ffffffffffffep+0 == 1.999999999999999555910790149937383830547332763671875
    // Y: (1 + 1 - 3 * 2^-53) * 2^0 == 1.99999999999999966693309261245303787291049957275390625
    // F:      0x1.fffffffffffffp+0 == 1.9999999999999997779553950749686919152736663818359375
    // The hexit E is 14 and even, while F is 15 and odd.

    // just below (0 + 2^-53) * 2^-1022: decremented last digit, then appended three 9 digits
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002470"
     "3282292062327208828439643411068618252990130716238221279284125033775363510437593264991818081799618989828234772"
     "2858865463328355177969898199387398005390939063150356595155702263922908583924491051844359318028499365361525003"
     "1937045767824921936562366986365848075700158576926990370631192827955855133292783433840935197801553124659726357"
     "9574622766465272827220056374006485499977096599470454020828166226237857393450736339007967761930577506740176324"
     "6736009689513405355374585166611342237666786041621596804619144672918403005300575308490487653917113865916462395"
     "2491262365388187963623937328042389101867234849766823508986338858792562830275599565752445550725518931369083625"
     "4779186948667994968324049705821028513185451396213837722826145437693412532098591327667236328124999",
        chars_format::fixed, 1080, errc::result_out_of_range, 0x0.0000000000000p+0},

    // (0 + 2^-53) * 2^-1022 exactly
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002470"
     "3282292062327208828439643411068618252990130716238221279284125033775363510437593264991818081799618989828234772"
     "2858865463328355177969898199387398005390939063150356595155702263922908583924491051844359318028499365361525003"
     "1937045767824921936562366986365848075700158576926990370631192827955855133292783433840935197801553124659726357"
     "9574622766465272827220056374006485499977096599470454020828166226237857393450736339007967761930577506740176324"
     "6736009689513405355374585166611342237666786041621596804619144672918403005300575308490487653917113865916462395"
     "2491262365388187963623937328042389101867234849766823508986338858792562830275599565752445550725518931369083625"
     "4779186948667994968324049705821028513185451396213837722826145437693412532098591327667236328125",
        chars_format::fixed, 1077, errc::result_out_of_range, 0x0.0000000000000p+0},

    // (0 + 2^-53) * 2^-1022 exactly, followed by a thousand 0 digits
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002470"
     "3282292062327208828439643411068618252990130716238221279284125033775363510437593264991818081799618989828234772"
     "2858865463328355177969898199387398005390939063150356595155702263922908583924491051844359318028499365361525003"
     "1937045767824921936562366986365848075700158576926990370631192827955855133292783433840935197801553124659726357"
     "9574622766465272827220056374006485499977096599470454020828166226237857393450736339007967761930577506740176324"
     "6736009689513405355374585166611342237666786041621596804619144672918403005300575308490487653917113865916462395"
     "2491262365388187963623937328042389101867234849766823508986338858792562830275599565752445550725518931369083625"
     "4779186948667994968324049705821028513185451396213837722826145437693412532098591327667236328125000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000",
        chars_format::fixed, 2077, errc::result_out_of_range, 0x0.0000000000000p+0},

    // above (0 + 2^-53) * 2^-1022: appended a thousand 0 digits followed by a 1 digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002470"
     "3282292062327208828439643411068618252990130716238221279284125033775363510437593264991818081799618989828234772"
     "2858865463328355177969898199387398005390939063150356595155702263922908583924491051844359318028499365361525003"
     "1937045767824921936562366986365848075700158576926990370631192827955855133292783433840935197801553124659726357"
     "9574622766465272827220056374006485499977096599470454020828166226237857393450736339007967761930577506740176324"
     "6736009689513405355374585166611342237666786041621596804619144672918403005300575308490487653917113865916462395"
     "2491262365388187963623937328042389101867234849766823508986338858792562830275599565752445550725518931369083625"
     "4779186948667994968324049705821028513185451396213837722826145437693412532098591327667236328125000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "00001",
        chars_format::fixed, 2078, errc{}, 0x0.0000000000001p-1022},

    // above (0 + 2^-53) * 2^-1022: appended a 1 digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002470"
     "3282292062327208828439643411068618252990130716238221279284125033775363510437593264991818081799618989828234772"
     "2858865463328355177969898199387398005390939063150356595155702263922908583924491051844359318028499365361525003"
     "1937045767824921936562366986365848075700158576926990370631192827955855133292783433840935197801553124659726357"
     "9574622766465272827220056374006485499977096599470454020828166226237857393450736339007967761930577506740176324"
     "6736009689513405355374585166611342237666786041621596804619144672918403005300575308490487653917113865916462395"
     "2491262365388187963623937328042389101867234849766823508986338858792562830275599565752445550725518931369083625"
     "47791869486679949683240497058210285131854513962138377228261454376934125320985913276672363281251",
        chars_format::fixed, 1078, errc{}, 0x0.0000000000001p-1022},

    // above (0 + 2^-53) * 2^-1022: incremented last digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002470"
     "3282292062327208828439643411068618252990130716238221279284125033775363510437593264991818081799618989828234772"
     "2858865463328355177969898199387398005390939063150356595155702263922908583924491051844359318028499365361525003"
     "1937045767824921936562366986365848075700158576926990370631192827955855133292783433840935197801553124659726357"
     "9574622766465272827220056374006485499977096599470454020828166226237857393450736339007967761930577506740176324"
     "6736009689513405355374585166611342237666786041621596804619144672918403005300575308490487653917113865916462395"
     "2491262365388187963623937328042389101867234849766823508986338858792562830275599565752445550725518931369083625"
     "4779186948667994968324049705821028513185451396213837722826145437693412532098591327667236328126",
        chars_format::fixed, 1077, errc{}, 0x0.0000000000001p-1022},

    // just below (0 + 1 - 3 * 2^-53) * 2^-1022: decremented last digit, then appended three 9 digits
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072006419"
     "9176395546258779936602667813027328296362349540005779643539444484102225369938322261431279727704724131030539099"
     "2976863718870946851468024222968583977359185141028540361975476844303195813273469348201130421165308554532083149"
     "3676067608324920106709384047261543474082573017216837765643921010648239116172158852475760231303527077156200284"
     "1775343298712758123539074213191978739083589771549597066404661620550578925994422322342444472859570416955675758"
     "5423752417124134805999073137808018133811049489046686648944255834488901008259721496147104204399198556535697531"
     "0055231935448663898095485089604066035268185282450207861510244351362091237759797852153577038777504570568436147"
     "5530270683064113556748943345076587312006145811358486831521563686919762403704226016998291015624999",
        chars_format::fixed, 1080, errc{}, 0x0.ffffffffffffep-1022},

    // (0 + 1 - 3 * 2^-53) * 2^-1022 exactly
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072006419"
     "9176395546258779936602667813027328296362349540005779643539444484102225369938322261431279727704724131030539099"
     "2976863718870946851468024222968583977359185141028540361975476844303195813273469348201130421165308554532083149"
     "3676067608324920106709384047261543474082573017216837765643921010648239116172158852475760231303527077156200284"
     "1775343298712758123539074213191978739083589771549597066404661620550578925994422322342444472859570416955675758"
     "5423752417124134805999073137808018133811049489046686648944255834488901008259721496147104204399198556535697531"
     "0055231935448663898095485089604066035268185282450207861510244351362091237759797852153577038777504570568436147"
     "5530270683064113556748943345076587312006145811358486831521563686919762403704226016998291015625",
        chars_format::fixed, 1077, errc{}, 0x0.ffffffffffffep-1022},

    // (0 + 1 - 3 * 2^-53) * 2^-1022 exactly, followed by a thousand 0 digits
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072006419"
     "9176395546258779936602667813027328296362349540005779643539444484102225369938322261431279727704724131030539099"
     "2976863718870946851468024222968583977359185141028540361975476844303195813273469348201130421165308554532083149"
     "3676067608324920106709384047261543474082573017216837765643921010648239116172158852475760231303527077156200284"
     "1775343298712758123539074213191978739083589771549597066404661620550578925994422322342444472859570416955675758"
     "5423752417124134805999073137808018133811049489046686648944255834488901008259721496147104204399198556535697531"
     "0055231935448663898095485089604066035268185282450207861510244351362091237759797852153577038777504570568436147"
     "5530270683064113556748943345076587312006145811358486831521563686919762403704226016998291015625000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000",
        chars_format::fixed, 2077, errc{}, 0x0.ffffffffffffep-1022},

    // above (0 + 1 - 3 * 2^-53) * 2^-1022: appended a thousand 0 digits followed by a 1 digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072006419"
     "9176395546258779936602667813027328296362349540005779643539444484102225369938322261431279727704724131030539099"
     "2976863718870946851468024222968583977359185141028540361975476844303195813273469348201130421165308554532083149"
     "3676067608324920106709384047261543474082573017216837765643921010648239116172158852475760231303527077156200284"
     "1775343298712758123539074213191978739083589771549597066404661620550578925994422322342444472859570416955675758"
     "5423752417124134805999073137808018133811049489046686648944255834488901008259721496147104204399198556535697531"
     "0055231935448663898095485089604066035268185282450207861510244351362091237759797852153577038777504570568436147"
     "5530270683064113556748943345076587312006145811358486831521563686919762403704226016998291015625000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "00001",
        chars_format::fixed, 2078, errc{}, 0x0.fffffffffffffp-1022},

    // above (0 + 1 - 3 * 2^-53) * 2^-1022: appended a 1 digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072006419"
     "9176395546258779936602667813027328296362349540005779643539444484102225369938322261431279727704724131030539099"
     "2976863718870946851468024222968583977359185141028540361975476844303195813273469348201130421165308554532083149"
     "3676067608324920106709384047261543474082573017216837765643921010648239116172158852475760231303527077156200284"
     "1775343298712758123539074213191978739083589771549597066404661620550578925994422322342444472859570416955675758"
     "5423752417124134805999073137808018133811049489046686648944255834488901008259721496147104204399198556535697531"
     "0055231935448663898095485089604066035268185282450207861510244351362091237759797852153577038777504570568436147"
     "55302706830641135567489433450765873120061458113584868315215636869197624037042260169982910156251",
        chars_format::fixed, 1078, errc{}, 0x0.fffffffffffffp-1022},

    // above (0 + 1 - 3 * 2^-53) * 2^-1022: incremented last digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072006419"
     "9176395546258779936602667813027328296362349540005779643539444484102225369938322261431279727704724131030539099"
     "2976863718870946851468024222968583977359185141028540361975476844303195813273469348201130421165308554532083149"
     "3676067608324920106709384047261543474082573017216837765643921010648239116172158852475760231303527077156200284"
     "1775343298712758123539074213191978739083589771549597066404661620550578925994422322342444472859570416955675758"
     "5423752417124134805999073137808018133811049489046686648944255834488901008259721496147104204399198556535697531"
     "0055231935448663898095485089604066035268185282450207861510244351362091237759797852153577038777504570568436147"
     "5530270683064113556748943345076587312006145811358486831521563686919762403704226016998291015626",
        chars_format::fixed, 1077, errc{}, 0x0.fffffffffffffp-1022},

    // just below (1 + 2^-53) * 2^-1022: decremented last digit, then appended three 9 digits
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072016301"
     "2305563795567615250361241457301801308322872404958664760675944619203679411688695321398552054903200090343478188"
     "4412325572184367563347617020518175998922941393629966742598285899994830148971433555578567693279306015978183162"
     "1424250679624607852958851992724935776883207324924799248168692322471659649343292587839501022509739575795105716"
     "0073834364573849432419299709217920738991976169431413149717326525502008499797367678374315520581880443916381057"
     "2367791175177756227497413804253387084478193655533073867420834526162513029462022730109054820067654020201547112"
     "0020281397001415752591234401773622442737124681517501897455599786532342558862196115163359241679580296044770649"
     "4647018477736093430045142168360701364747951396213837722826145437693412532098591327667236328124999",
        chars_format::fixed, 1080, errc{}, 0x1.0000000000000p-1022},

    // (1 + 2^-53) * 2^-1022 exactly
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072016301"
     "2305563795567615250361241457301801308322872404958664760675944619203679411688695321398552054903200090343478188"
     "4412325572184367563347617020518175998922941393629966742598285899994830148971433555578567693279306015978183162"
     "1424250679624607852958851992724935776883207324924799248168692322471659649343292587839501022509739575795105716"
     "0073834364573849432419299709217920738991976169431413149717326525502008499797367678374315520581880443916381057"
     "2367791175177756227497413804253387084478193655533073867420834526162513029462022730109054820067654020201547112"
     "0020281397001415752591234401773622442737124681517501897455599786532342558862196115163359241679580296044770649"
     "4647018477736093430045142168360701364747951396213837722826145437693412532098591327667236328125",
        chars_format::fixed, 1077, errc{}, 0x1.0000000000000p-1022},

    // (1 + 2^-53) * 2^-1022 exactly, followed by a thousand 0 digits
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072016301"
     "2305563795567615250361241457301801308322872404958664760675944619203679411688695321398552054903200090343478188"
     "4412325572184367563347617020518175998922941393629966742598285899994830148971433555578567693279306015978183162"
     "1424250679624607852958851992724935776883207324924799248168692322471659649343292587839501022509739575795105716"
     "0073834364573849432419299709217920738991976169431413149717326525502008499797367678374315520581880443916381057"
     "2367791175177756227497413804253387084478193655533073867420834526162513029462022730109054820067654020201547112"
     "0020281397001415752591234401773622442737124681517501897455599786532342558862196115163359241679580296044770649"
     "4647018477736093430045142168360701364747951396213837722826145437693412532098591327667236328125000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000",
        chars_format::fixed, 2077, errc{}, 0x1.0000000000000p-1022},

    // above (1 + 2^-53) * 2^-1022: appended a thousand 0 digits followed by a 1 digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072016301"
     "2305563795567615250361241457301801308322872404958664760675944619203679411688695321398552054903200090343478188"
     "4412325572184367563347617020518175998922941393629966742598285899994830148971433555578567693279306015978183162"
     "1424250679624607852958851992724935776883207324924799248168692322471659649343292587839501022509739575795105716"
     "0073834364573849432419299709217920738991976169431413149717326525502008499797367678374315520581880443916381057"
     "2367791175177756227497413804253387084478193655533073867420834526162513029462022730109054820067654020201547112"
     "0020281397001415752591234401773622442737124681517501897455599786532342558862196115163359241679580296044770649"
     "4647018477736093430045142168360701364747951396213837722826145437693412532098591327667236328125000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "00001",
        chars_format::fixed, 2078, errc{}, 0x1.0000000000001p-1022},

    // above (1 + 2^-53) * 2^-1022: appended a 1 digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072016301"
     "2305563795567615250361241457301801308322872404958664760675944619203679411688695321398552054903200090343478188"
     "4412325572184367563347617020518175998922941393629966742598285899994830148971433555578567693279306015978183162"
     "1424250679624607852958851992724935776883207324924799248168692322471659649343292587839501022509739575795105716"
     "0073834364573849432419299709217920738991976169431413149717326525502008499797367678374315520581880443916381057"
     "2367791175177756227497413804253387084478193655533073867420834526162513029462022730109054820067654020201547112"
     "0020281397001415752591234401773622442737124681517501897455599786532342558862196115163359241679580296044770649"
     "46470184777360934300451421683607013647479513962138377228261454376934125320985913276672363281251",
        chars_format::fixed, 1078, errc{}, 0x1.0000000000001p-1022},

    // above (1 + 2^-53) * 2^-1022: incremented last digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585072016301"
     "2305563795567615250361241457301801308322872404958664760675944619203679411688695321398552054903200090343478188"
     "4412325572184367563347617020518175998922941393629966742598285899994830148971433555578567693279306015978183162"
     "1424250679624607852958851992724935776883207324924799248168692322471659649343292587839501022509739575795105716"
     "0073834364573849432419299709217920738991976169431413149717326525502008499797367678374315520581880443916381057"
     "2367791175177756227497413804253387084478193655533073867420834526162513029462022730109054820067654020201547112"
     "0020281397001415752591234401773622442737124681517501897455599786532342558862196115163359241679580296044770649"
     "4647018477736093430045142168360701364747951396213837722826145437693412532098591327667236328126",
        chars_format::fixed, 1077, errc{}, 0x1.0000000000001p-1022},

    // just below (1 + 1 - 3 * 2^-53) * 2^-1022: decremented last digit, then appended three 9 digits
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000044501477170144020250"
     "8199667279499186358524265859260511351695091228726223124931264069530541271189424317838013700808305231545782515"
     "4530323827726959236845743044099361970891187471508150509418060480375117378320411851935338796416115205148741308"
     "3163272520124606023105869053620631175265621765214646643181420505164043632222668006474326056011713528291579642"
     "2274554896821334728738317548403413978098469341510556195293821919814730032341053661708792231510873354131880491"
     "1055533902788485678121901775450062980622457102958163711745945687733011032421168917765671370549738710820782247"
     "7584250967061891687062782163335299376138075114200886249979505279101870966346394401564490729731565935244123171"
     "5398102212132212018470035807616260163568645811358486831521563686919762403704226016998291015624999",
        chars_format::fixed, 1080, errc{}, 0x1.ffffffffffffep-1022},

    // (1 + 1 - 3 * 2^-53) * 2^-1022 exactly
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000044501477170144020250"
     "8199667279499186358524265859260511351695091228726223124931264069530541271189424317838013700808305231545782515"
     "4530323827726959236845743044099361970891187471508150509418060480375117378320411851935338796416115205148741308"
     "3163272520124606023105869053620631175265621765214646643181420505164043632222668006474326056011713528291579642"
     "2274554896821334728738317548403413978098469341510556195293821919814730032341053661708792231510873354131880491"
     "1055533902788485678121901775450062980622457102958163711745945687733011032421168917765671370549738710820782247"
     "7584250967061891687062782163335299376138075114200886249979505279101870966346394401564490729731565935244123171"
     "5398102212132212018470035807616260163568645811358486831521563686919762403704226016998291015625",
        chars_format::fixed, 1077, errc{}, 0x1.ffffffffffffep-1022},

    // (1 + 1 - 3 * 2^-53) * 2^-1022 exactly, followed by a thousand 0 digits
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000044501477170144020250"
     "8199667279499186358524265859260511351695091228726223124931264069530541271189424317838013700808305231545782515"
     "4530323827726959236845743044099361970891187471508150509418060480375117378320411851935338796416115205148741308"
     "3163272520124606023105869053620631175265621765214646643181420505164043632222668006474326056011713528291579642"
     "2274554896821334728738317548403413978098469341510556195293821919814730032341053661708792231510873354131880491"
     "1055533902788485678121901775450062980622457102958163711745945687733011032421168917765671370549738710820782247"
     "7584250967061891687062782163335299376138075114200886249979505279101870966346394401564490729731565935244123171"
     "5398102212132212018470035807616260163568645811358486831521563686919762403704226016998291015625000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000",
        chars_format::fixed, 2077, errc{}, 0x1.ffffffffffffep-1022},

    // above (1 + 1 - 3 * 2^-53) * 2^-1022: appended a thousand 0 digits followed by a 1 digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000044501477170144020250"
     "8199667279499186358524265859260511351695091228726223124931264069530541271189424317838013700808305231545782515"
     "4530323827726959236845743044099361970891187471508150509418060480375117378320411851935338796416115205148741308"
     "3163272520124606023105869053620631175265621765214646643181420505164043632222668006474326056011713528291579642"
     "2274554896821334728738317548403413978098469341510556195293821919814730032341053661708792231510873354131880491"
     "1055533902788485678121901775450062980622457102958163711745945687733011032421168917765671370549738710820782247"
     "7584250967061891687062782163335299376138075114200886249979505279101870966346394401564490729731565935244123171"
     "5398102212132212018470035807616260163568645811358486831521563686919762403704226016998291015625000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "00001",
        chars_format::fixed, 2078, errc{}, 0x1.fffffffffffffp-1022},

    // above (1 + 1 - 3 * 2^-53) * 2^-1022: appended a 1 digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000044501477170144020250"
     "8199667279499186358524265859260511351695091228726223124931264069530541271189424317838013700808305231545782515"
     "4530323827726959236845743044099361970891187471508150509418060480375117378320411851935338796416115205148741308"
     "3163272520124606023105869053620631175265621765214646643181420505164043632222668006474326056011713528291579642"
     "2274554896821334728738317548403413978098469341510556195293821919814730032341053661708792231510873354131880491"
     "1055533902788485678121901775450062980622457102958163711745945687733011032421168917765671370549738710820782247"
     "7584250967061891687062782163335299376138075114200886249979505279101870966346394401564490729731565935244123171"
     "53981022121322120184700358076162601635686458113584868315215636869197624037042260169982910156251",
        chars_format::fixed, 1078, errc{}, 0x1.fffffffffffffp-1022},

    // above (1 + 1 - 3 * 2^-53) * 2^-1022: incremented last digit
    {"0."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000044501477170144020250"
     "8199667279499186358524265859260511351695091228726223124931264069530541271189424317838013700808305231545782515"
     "4530323827726959236845743044099361970891187471508150509418060480375117378320411851935338796416115205148741308"
     "3163272520124606023105869053620631175265621765214646643181420505164043632222668006474326056011713528291579642"
     "2274554896821334728738317548403413978098469341510556195293821919814730032341053661708792231510873354131880491"
     "1055533902788485678121901775450062980622457102958163711745945687733011032421168917765671370549738710820782247"
     "7584250967061891687062782163335299376138075114200886249979505279101870966346394401564490729731565935244123171"
     "5398102212132212018470035807616260163568645811358486831521563686919762403704226016998291015626",
        chars_format::fixed, 1077, errc{}, 0x1.fffffffffffffp-1022},

    // just below (1 + 2^-53) * 2^-33: decremented last digit, then appended three 9 digits
    {"0.00000000011641532182693482737782207114105741986576081359316958696581423282623291015624999", chars_format::fixed,
        91, errc{}, 0x1.0000000000000p-33},

    // (1 + 2^-53) * 2^-33 exactly
    {"0.00000000011641532182693482737782207114105741986576081359316958696581423282623291015625", chars_format::fixed,
        88, errc{}, 0x1.0000000000000p-33},

    // (1 + 2^-53) * 2^-33 exactly, followed by a thousand 0 digits
    {"0."
     "0000000001164153218269348273778220711410574198657608135931695869658142328262329101562500000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000",
        chars_format::fixed, 1088, errc{}, 0x1.0000000000000p-33},

    // above (1 + 2^-53) * 2^-33: appended a thousand 0 digits followed by a 1 digit
    {"0."
     "0000000001164153218269348273778220711410574198657608135931695869658142328262329101562500000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001",
        chars_format::fixed, 1089, errc{}, 0x1.0000000000001p-33},

    // above (1 + 2^-53) * 2^-33: appended a 1 digit
    {"0.000000000116415321826934827377822071141057419865760813593169586965814232826232910156251", chars_format::fixed,
        89, errc{}, 0x1.0000000000001p-33},

    // above (1 + 2^-53) * 2^-33: incremented last digit
    {"0.00000000011641532182693482737782207114105741986576081359316958696581423282623291015626", chars_format::fixed,
        88, errc{}, 0x1.0000000000001p-33},

    // just below (1 + 1 - 3 * 2^-53) * 2^-33: decremented last digit, then appended three 9 digits
    {"0.00000000023283064365386959013215878657682774040271755922049123910255730152130126953124999", chars_format::fixed,
        91, errc{}, 0x1.ffffffffffffep-33},

    // (1 + 1 - 3 * 2^-53) * 2^-33 exactly
    {"0.00000000023283064365386959013215878657682774040271755922049123910255730152130126953125", chars_format::fixed,
        88, errc{}, 0x1.ffffffffffffep-33},

    // (1 + 1 - 3 * 2^-53) * 2^-33 exactly, followed by a thousand 0 digits
    {"0."
     "0000000002328306436538695901321587865768277404027175592204912391025573015213012695312500000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000",
        chars_format::fixed, 1088, errc{}, 0x1.ffffffffffffep-33},

    // above (1 + 1 - 3 * 2^-53) * 2^-33: appended a thousand 0 digits followed by a 1 digit
    {"0."
     "0000000002328306436538695901321587865768277404027175592204912391025573015213012695312500000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001",
        chars_format::fixed, 1089, errc{}, 0x1.fffffffffffffp-33},

    // above (1 + 1 - 3 * 2^-53) * 2^-33: appended a 1 digit
    {"0.000000000232830643653869590132158786576827740402717559220491239102557301521301269531251", chars_format::fixed,
        89, errc{}, 0x1.fffffffffffffp-33},

    // above (1 + 1 - 3 * 2^-53) * 2^-33: incremented last digit
    {"0.00000000023283064365386959013215878657682774040271755922049123910255730152130126953126", chars_format::fixed,
        88, errc{}, 0x1.fffffffffffffp-33},

    // just below (1 + 2^-53) * 2^0: decremented last digit, then appended three 9 digits
    {"1.00000000000000011102230246251565404236316680908203124999", chars_format::fixed, 58, errc{},
        0x1.0000000000000p+0},

    // (1 + 2^-53) * 2^0 exactly
    {"1.00000000000000011102230246251565404236316680908203125", chars_format::fixed, 55, errc{}, 0x1.0000000000000p+0},

    // (1 + 2^-53) * 2^0 exactly, followed by a thousand 0 digits
    {"1."
     "0000000000000001110223024625156540423631668090820312500000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "000000000000000000000000000000000000000000000000000000000000000000000000",
        chars_format::fixed, 1055, errc{}, 0x1.0000000000000p+0},

    // above (1 + 2^-53) * 2^0: appended a thousand 0 digits followed by a 1 digit
    {"1."
     "0000000000000001110223024625156540423631668090820312500000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000001",
        chars_format::fixed, 1056, errc{}, 0x1.0000000000001p+0},

    // above (1 + 2^-53) * 2^0: appended a 1 digit
    {"1.000000000000000111022302462515654042363166809082031251", chars_format::fixed, 56, errc{}, 0x1.0000000000001p+0},

    // above (1 + 2^-53) * 2^0: incremented last digit
    {"1.00000000000000011102230246251565404236316680908203126", chars_format::fixed, 55, errc{}, 0x1.0000000000001p+0},

    // just below (1 + 1 - 3 * 2^-53) * 2^0: decremented last digit, then appended three 9 digits
    {"1.99999999999999966693309261245303787291049957275390624999", chars_format::fixed, 58, errc{},
        0x1.ffffffffffffep+0},

    // (1 + 1 - 3 * 2^-53) * 2^0 exactly
    {"1.99999999999999966693309261245303787291049957275390625", chars_format::fixed, 55, errc{}, 0x1.ffffffffffffep+0},

    // (1 + 1 - 3 * 2^-53) * 2^0 exactly, followed by a thousand 0 digits
    {"1."
     "9999999999999996669330926124530378729104995727539062500000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "000000000000000000000000000000000000000000000000000000000000000000000000",
        chars_format::fixed, 1055, errc{}, 0x1.ffffffffffffep+0},

    // above (1 + 1 - 3 * 2^-53) * 2^0: appended a thousand 0 digits followed by a 1 digit
    {"1."
     "9999999999999996669330926124530378729104995727539062500000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000001",
        chars_format::fixed, 1056, errc{}, 0x1.fffffffffffffp+0},

    // above (1 + 1 - 3 * 2^-53) * 2^0: appended a 1 digit
    {"1.999999999999999666933092612453037872910499572753906251", chars_format::fixed, 56, errc{}, 0x1.fffffffffffffp+0},

    // above (1 + 1 - 3 * 2^-53) * 2^0: incremented last digit
    {"1.99999999999999966693309261245303787291049957275390626", chars_format::fixed, 55, errc{}, 0x1.fffffffffffffp+0},

    // just below (1 + 2^-53) * 2^33: decremented last digit, then appended three 9 digits
    {"8589934592.00000095367431640624999", chars_format::fixed, 34, errc{}, 0x1.0000000000000p+33},

    // (1 + 2^-53) * 2^33 exactly
    {"8589934592.00000095367431640625", chars_format::fixed, 31, errc{}, 0x1.0000000000000p+33},

    // (1 + 2^-53) * 2^33 exactly, followed by a thousand 0 digits
    {"8589934592."
     "0000009536743164062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "000000000000000000000000000000000000000",
        chars_format::fixed, 1031, errc{}, 0x1.0000000000000p+33},

    // above (1 + 2^-53) * 2^33: appended a thousand 0 digits followed by a 1 digit
    {"8589934592."
     "0000009536743164062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000001",
        chars_format::fixed, 1032, errc{}, 0x1.0000000000001p+33},

    // above (1 + 2^-53) * 2^33: appended a 1 digit
    {"8589934592.000000953674316406251", chars_format::fixed, 32, errc{}, 0x1.0000000000001p+33},

    // above (1 + 2^-53) * 2^33: incremented last digit
    {"8589934592.00000095367431640626", chars_format::fixed, 31, errc{}, 0x1.0000000000001p+33},

    // just below (1 + 1 - 3 * 2^-53) * 2^33: decremented last digit, then appended three 9 digits
    {"17179869183.99999713897705078124999", chars_format::fixed, 35, errc{}, 0x1.ffffffffffffep+33},

    // (1 + 1 - 3 * 2^-53) * 2^33 exactly
    {"17179869183.99999713897705078125", chars_format::fixed, 32, errc{}, 0x1.ffffffffffffep+33},

    // (1 + 1 - 3 * 2^-53) * 2^33 exactly, followed by a thousand 0 digits
    {"17179869183."
     "9999971389770507812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "000000000000000000000000000000000000000",
        chars_format::fixed, 1032, errc{}, 0x1.ffffffffffffep+33},

    // above (1 + 1 - 3 * 2^-53) * 2^33: appended a thousand 0 digits followed by a 1 digit
    {"17179869183."
     "9999971389770507812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000001",
        chars_format::fixed, 1033, errc{}, 0x1.fffffffffffffp+33},

    // above (1 + 1 - 3 * 2^-53) * 2^33: appended a 1 digit
    {"17179869183.999997138977050781251", chars_format::fixed, 33, errc{}, 0x1.fffffffffffffp+33},

    // above (1 + 1 - 3 * 2^-53) * 2^33: incremented last digit
    {"17179869183.99999713897705078126", chars_format::fixed, 32, errc{}, 0x1.fffffffffffffp+33},

    // just below (1 + 2^-53) * 2^77: decremented last digit, then appended three 9 digits
    {"151115727451828663615487.999", chars_format::fixed, 28, errc{}, 0x1.0000000000000p+77},

    // (1 + 2^-53) * 2^77 exactly
    {"151115727451828663615488", chars_format::fixed, 24, errc{}, 0x1.0000000000000p+77},

    // (1 + 2^-53) * 2^77 exactly, followed by a thousand 0 digits
    {"151115727451828663615488."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000",
        chars_format::fixed, 1025, errc{}, 0x1.0000000000000p+77},

    // above (1 + 2^-53) * 2^77: appended a thousand 0 digits followed by a 1 digit
    {"151115727451828663615488."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "00000000000000000001",
        chars_format::fixed, 1026, errc{}, 0x1.0000000000001p+77},

    // above (1 + 2^-53) * 2^77: appended a 1 digit
    {"151115727451828663615488.1", chars_format::fixed, 26, errc{}, 0x1.0000000000001p+77},

    // above (1 + 2^-53) * 2^77: incremented last digit
    {"151115727451828663615489", chars_format::fixed, 24, errc{}, 0x1.0000000000001p+77},

    // just below (1 + 1 - 3 * 2^-53) * 2^77: decremented last digit, then appended three 9 digits
    {"302231454903657243344895.999", chars_format::fixed, 28, errc{}, 0x1.ffffffffffffep+77},

    // (1 + 1 - 3 * 2^-53) * 2^77 exactly
    {"302231454903657243344896", chars_format::fixed, 24, errc{}, 0x1.ffffffffffffep+77},

    // (1 + 1 - 3 * 2^-53) * 2^77 exactly, followed by a thousand 0 digits
    {"302231454903657243344896."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000",
        chars_format::fixed, 1025, errc{}, 0x1.ffffffffffffep+77},

    // above (1 + 1 - 3 * 2^-53) * 2^77: appended a thousand 0 digits followed by a 1 digit
    {"302231454903657243344896."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "00000000000000000001",
        chars_format::fixed, 1026, errc{}, 0x1.fffffffffffffp+77},

    // above (1 + 1 - 3 * 2^-53) * 2^77: appended a 1 digit
    {"302231454903657243344896.1", chars_format::fixed, 26, errc{}, 0x1.fffffffffffffp+77},

    // above (1 + 1 - 3 * 2^-53) * 2^77: incremented last digit
    {"302231454903657243344897", chars_format::fixed, 24, errc{}, 0x1.fffffffffffffp+77},

    // just below (1 + 2^-53) * 2^1023: decremented last digit, then appended three 9 digits
    {"8988465674311580536566680721305029496276241413130815897397134275615404541548669375241369800602409693534988440"
     "3114202125541629105369684531108613657287705365884742938136589844238179474556051429647415148697857438797685859"
     "063890851407391008830874765563025951597582513936655578157348020066364210154316532161708031.999",
        chars_format::fixed, 312, errc{}, 0x1.0000000000000p+1023},

    // (1 + 2^-53) * 2^1023 exactly
    {"8988465674311580536566680721305029496276241413130815897397134275615404541548669375241369800602409693534988440"
     "3114202125541629105369684531108613657287705365884742938136589844238179474556051429647415148697857438797685859"
     "063890851407391008830874765563025951597582513936655578157348020066364210154316532161708032",
        chars_format::fixed, 308, errc{}, 0x1.0000000000000p+1023},

    // (1 + 2^-53) * 2^1023 exactly, followed by a thousand 0 digits
    {"8988465674311580536566680721305029496276241413130815897397134275615404541548669375241369800602409693534988440"
     "3114202125541629105369684531108613657287705365884742938136589844238179474556051429647415148697857438797685859"
     "063890851407391008830874765563025951597582513936655578157348020066364210154316532161708032."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000",
        chars_format::fixed, 1309, errc{}, 0x1.0000000000000p+1023},

    // above (1 + 2^-53) * 2^1023: appended a thousand 0 digits followed by a 1 digit
    {"8988465674311580536566680721305029496276241413130815897397134275615404541548669375241369800602409693534988440"
     "3114202125541629105369684531108613657287705365884742938136589844238179474556051429647415148697857438797685859"
     "063890851407391008830874765563025951597582513936655578157348020066364210154316532161708032."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "00000000000000000001",
        chars_format::fixed, 1310, errc{}, 0x1.0000000000001p+1023},

    // above (1 + 2^-53) * 2^1023: appended a 1 digit
    {"8988465674311580536566680721305029496276241413130815897397134275615404541548669375241369800602409693534988440"
     "3114202125541629105369684531108613657287705365884742938136589844238179474556051429647415148697857438797685859"
     "063890851407391008830874765563025951597582513936655578157348020066364210154316532161708032.1",
        chars_format::fixed, 310, errc{}, 0x1.0000000000001p+1023},

    // above (1 + 2^-53) * 2^1023: incremented last digit
    {"8988465674311580536566680721305029496276241413130815897397134275615404541548669375241369800602409693534988440"
     "3114202125541629105369684531108613657287705365884742938136589844238179474556051429647415148697857438797685859"
     "063890851407391008830874765563025951597582513936655578157348020066364210154316532161708033",
        chars_format::fixed, 308, errc{}, 0x1.0000000000001p+1023},

    // just below (1 + 1 - 3 * 2^-53) * 2^1023: decremented last digit, then appended three 9 digits
    {"1797693134862315608353258760581052985162070023416521662616611746258695532672923265745300992879465492467506314"
     "9033587701752208710592698796290627760473556921329019091915239418047621712533496094635638726128664019802903779"
     "9514183602981511756283727771403830521483963923935633133642802139091669457927874464075218943.999",
        chars_format::fixed, 313, errc{}, 0x1.ffffffffffffep+1023},

    // (1 + 1 - 3 * 2^-53) * 2^1023 exactly
    {"1797693134862315608353258760581052985162070023416521662616611746258695532672923265745300992879465492467506314"
     "9033587701752208710592698796290627760473556921329019091915239418047621712533496094635638726128664019802903779"
     "9514183602981511756283727771403830521483963923935633133642802139091669457927874464075218944",
        chars_format::fixed, 309, errc{}, 0x1.ffffffffffffep+1023},

    // (1 + 1 - 3 * 2^-53) * 2^1023 exactly, followed by a thousand 0 digits
    {"1797693134862315608353258760581052985162070023416521662616611746258695532672923265745300992879465492467506314"
     "9033587701752208710592698796290627760473556921329019091915239418047621712533496094635638726128664019802903779"
     "9514183602981511756283727771403830521483963923935633133642802139091669457927874464075218944."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000",
        chars_format::fixed, 1310, errc{}, 0x1.ffffffffffffep+1023},

    // above (1 + 1 - 3 * 2^-53) * 2^1023: appended a thousand 0 digits followed by a 1 digit
    {"1797693134862315608353258760581052985162070023416521662616611746258695532672923265745300992879465492467506314"
     "9033587701752208710592698796290627760473556921329019091915239418047621712533496094635638726128664019802903779"
     "9514183602981511756283727771403830521483963923935633133642802139091669457927874464075218944."
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "00000000000000000001",
        chars_format::fixed, 1311, errc{}, 0x1.fffffffffffffp+1023},

    // above (1 + 1 - 3 * 2^-53) * 2^1023: appended a 1 digit
    {"1797693134862315608353258760581052985162070023416521662616611746258695532672923265745300992879465492467506314"
     "9033587701752208710592698796290627760473556921329019091915239418047621712533496094635638726128664019802903779"
     "9514183602981511756283727771403830521483963923935633133642802139091669457927874464075218944.1",
        chars_format::fixed, 311, errc{}, 0x1.fffffffffffffp+1023},

    // above (1 + 1 - 3 * 2^-53) * 2^1023: incremented last digit
    {"1797693134862315608353258760581052985162070023416521662616611746258695532672923265745300992879465492467506314"
     "9033587701752208710592698796290627760473556921329019091915239418047621712533496094635638726128664019802903779"
     "9514183602981511756283727771403830521483963923935633133642802139091669457927874464075218945",
        chars_format::fixed, 309, errc{}, 0x1.fffffffffffffp+1023},

    // VSO-852024 also affected hexfloats.
    {"0.00000000000008p-1022", chars_format::hex, 22, errc::result_out_of_range, 0x0.0000000000000p+0},
    {"0."
     "0000000000000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000001p-1022",
        chars_format::hex, 1023, errc{}, 0x0.0000000000001p-1022},

    {"0.ffffffffffffe8p-1022", chars_format::hex, 22, errc{}, 0x0.ffffffffffffep-1022},
    {"0."
     "ffffffffffffe800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000001p-1022",
        chars_format::hex, 1023, errc{}, 0x0.fffffffffffffp-1022},

    {"1.00000000000008p+0", chars_format::hex, 19, errc{}, 0x1.0000000000000p+0},
    {"1."
     "0000000000000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000001p+0",
        chars_format::hex, 1020, errc{}, 0x1.0000000000001p+0},

    {"1.ffffffffffffe8p+0", chars_format::hex, 19, errc{}, 0x1.ffffffffffffep+0},
    {"1."
     "ffffffffffffe800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000001p+0",
        chars_format::hex, 1020, errc{}, 0x1.fffffffffffffp+0},

    {"1.00000000000008p+1023", chars_format::hex, 22, errc{}, 0x1.0000000000000p+1023},
    {"1."
     "0000000000000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000001p+1023",
        chars_format::hex, 1023, errc{}, 0x1.0000000000001p+1023},

    {"1.ffffffffffffe8p+1023", chars_format::hex, 22, errc{}, 0x1.ffffffffffffep+1023},
    {"1."
     "ffffffffffffe800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
     "0000000000000000000000000000000001p+1023",
        chars_format::hex, 1023, errc{}, 0x1.fffffffffffffp+1023},

    // VSO-733765 "<charconv>: [Feedback] double std::from_chars behavior on exponent out of range"
    // LWG-3081 "Floating point from_chars API does not distinguish between overflow and underflow"
    // These test cases exercise every overflow/underflow codepath.
    {"1e+1000", chars_format::scientific, 7, errc::result_out_of_range, double_inf},
    {"1e-1000", chars_format::scientific, 7, errc::result_out_of_range, 0.0},
    {"1.fffffffffffff8p+1023", chars_format::hex, 22, errc::result_out_of_range, double_inf},
    {"1e+2000", chars_format::scientific, 7, errc::result_out_of_range, double_inf},
    {"1e-2000", chars_format::scientific, 7, errc::result_out_of_range, 0.0},
    {"1e+9999", chars_format::scientific, 7, errc::result_out_of_range, double_inf},
    {"1e-9999", chars_format::scientific, 7, errc::result_out_of_range, 0.0},
    {"10e+5199", chars_format::scientific, 8, errc::result_out_of_range, double_inf},
    {"0.001e-5199", chars_format::scientific, 11, errc::result_out_of_range, 0.0},
};

#endif // DOUBLE_FROM_CHARS_TEST_CASES_HPP
